!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, Y. Nagai
!      Last updated:    Mar 30, 2019 by M. Shiga
!      Description:     energy and force by aenet
!
!///////////////////////////////////////////////////////////////////////



#ifdef ioutils

!***********************************************************************
      module ioutils_variables
!***********************************************************************

      use iso_c_binding
      implicit none

      interface

         function get_numberoffiles(string) &
     &      bind(c,name='get_numberoffiles_c')
            import
            character string(*)
            integer(c_int):: get_numberoffiles
         end

         subroutine c_get_filenames(string,num,stringptrs) &
     &      bind(c,name='get_filenames_c')
            import
            character string(*)
            integer:: num
            type(c_ptr), dimension(num) :: stringptrs
         end subroutine

         function c_remove(string) bind(c,name='remove')
            import
            character string(*)
            integer(c_int):: c_remove
         end

         subroutine c_remove_all(pattern,string) &
     &      bind(c,name='remove_all')
            use iso_c_binding
            import
            character pattern(*)
            character string(*)
         end

         function c_mkdir(string) bind(c,name='mkdir_c')
            import
            character string(*)
            integer(c_int):: c_mkdir
         end

         function c_directory_exists(string) &
     &                   bind(c,name='directory_exists')
            import
            character string(*)
            integer(c_int):: c_directory_exists
         end

      end interface

      contains

!c         function f_c_string(string, trim)
!c            use, intrinsic :: iso_c_binding, only: c_char, c_null_char
!c            character(len=*), intent(in) :: string
!c            logical, intent(in), optional :: trim
!c
!c            character(kind=c_char, len=:), allocatable :: f_c_string
!c            logical :: trim_
!c
!c            trim_ = .true.
!c            if (present(trim)) trim_ = trim
!c
!c            block
!c               intrinsic trim
!c               if (trim_) then
!c                  f_c_string = trim(string)//c_null_char
!c               else
!c                  f_c_string = string//c_null_char
!c               end if
!c            end block
!c         end function

         subroutine checkdir(string, test)
            character(*),intent(in)::string
            integer, intent(out) :: test
            character(len=:), allocatable :: stringC
            stringC=f_c_string(string)
            test = c_directory_exists( stringC )
         end subroutine

         subroutine mkdir(string)
         character(*),intent(in)::string
         integer::i
         integer*4 status
         integer*4 getcwd
         character*100 dirname
         character(len=:), allocatable :: stringC
         stringC=f_c_string(string)
         status = getcwd( dirname )
         i = c_mkdir( stringC )
         return
         end subroutine

         subroutine remove(string)
         character(*),intent(in)::string
         integer::i                 
         integer*4 status
         integer*4 getcwd
         character*100 dirname
         character(len=:), allocatable :: stringC
         stringC=f_c_string(string)
         status = getcwd( dirname )
         dirname = dirname//"/"
         dirname = dirname//string
         i = c_remove(stringC)
         return
         end subroutine

         subroutine remove_all(string,dir)
         character(*),intent(in)::string
         character(*),intent(in)::dir
         character(len=:), allocatable :: stringC
         character(len=:), allocatable :: stringC2
         stringC=f_c_string(string)
         stringC2=f_c_string(dir)
         call c_remove_all(stringC,stringC2)
         end

         function c_charptr_to_f_charptr(ccp) result(result)
         type(c_ptr),intent(in),value :: ccp
         character(:,c_char),pointer :: result
         interface
            function strlen(p) bind(c)
               import c_ptr,c_size_t
               type(c_ptr),value :: p
               integer(c_size_t) strlen
            end function
         end interface
         result => convert_cptr(ccp,strlen(ccp))
         contains
            function convert_cptr(p,len)
            type(c_ptr),intent(in) :: p
               integer(c_size_t),intent(in) :: len
               character(len,c_char),pointer :: convert_cptr
               call c_f_pointer(p,convert_cptr)
            end function
         end function

         subroutine readdir(string,num,filenames)
         character(*),intent(in)::string
         integer,intent(out)::num
         integer::numall
         integer::ns
         character(len=100),allocatable,intent(inout)::filenames(:)
         character(len=:), allocatable :: stringC
         stringC=f_c_string(string)
         numall = get_numberoffiles(stringC)
         call get_filenames_array(stringC,numall,filenames)
         num = numall -2
         end subroutine

         subroutine get_filenames_array(string,num,filenames)
         character(*),intent(in)::string
         integer,intent(in)::num
         character(len=100), dimension(num), target :: stringarray
         type(c_ptr), dimension(num) :: stringptrs
         character(:,c_char),pointer :: filename
         character(len=100),allocatable,intent(inout)::filenames(:)
         integer::count
         integer::ns
         character(len=:), allocatable :: stringC
         stringC=f_c_string(string)
         do ns = 1,num
            stringarray(ns) = "undef"//c_null_char
            stringptrs(ns) = c_loc(stringarray(ns))
         end do
         call c_get_filenames(stringC,num,stringptrs)
         if (allocated(filenames)) deallocate(filenames)
         allocate(filenames(num-2))
         count = 0
         do ns=1,num
            filename => c_charptr_to_f_charptr(stringptrs(ns))
            if (filename .ne. "." .and. filename .ne. "..") then
               count = count + 1
               filenames(count) = filename
            end if
         end do

      end

!***********************************************************************
      end module
!***********************************************************************




#endif



!#ifdef aenet2
#if defined(aenet2) || defined(aenet_pytorch)
! #ifdef aenet2
!aenet2



!***********************************************************************
      module aenet_variables
!***********************************************************************

!     /*   number of types   */
      integer :: ntype_aenet

!     /*   types   */
      character(len=4), dimension(:), allocatable :: type_aenet

!     /*   types in integer   */
      integer, dimension(:), allocatable :: itype_aenet

!     /*   networks   */
      character(len=80), dimension(:), allocatable :: network_aenet

!     /*   interval of training   */
      integer :: istep_train_aenet

!     /*   interval for saving the trained networks    */
        integer :: istep_save_aenet

!     /*   last step of training   */
      integer :: lstep_train_aenet

!     /*   print interval of xsf trajectory   */
      integer :: iprint_xsf_aenet

!     /*   flag for the communicator of aenet   */
      integer :: iflag_aenet = 0

!     /*   flag for the training of aenet   */
      integer :: jflag_aenet = 0

!     /*   number of xsf files for training   */
      integer :: nxsf_train_aenet

!     /*   minimum number of xsf files required for training   */
      integer :: minxsf_train_aenet

!     /*   option of xsf files - 0: accepted, 1: accepted/rejected   */
      integer :: ioption_xsf_aenet = 0

!     /*   directory name for training   */
      character(len=80) :: dir_train_aenet
 
!     /*   directory name for saving trained networks periodically */
      character(len=80) :: dir_save_aenet

!     /*   existence of ann files - 0: no, 1: yes   */
      integer :: iann_start_aenet = 0

!     /*   delay time: default value is '0.25'  */
      character(len=8) :: delay_aenet = '0.25'

!     //   cut off skin of neighbor list
      real(8) :: skin_aenet

!     //   cut off of neighbor list
      real(8) :: rcut_aenet

!     //   atomic coordinates for neighbor list
      real(8), dimension(3) :: coo_i
      real(8), dimension(:,:), allocatable :: coo_j

!     //   atomic kinds for neighbor list
      integer :: itype_i
      integer, dimension(:), allocatable :: itype_j

!     //   atomic indices for neighbor list
      integer :: index_i
      integer, dimension(:), allocatable :: index_j

!     /*   print interval of heat flux trajectory   */
      integer :: iprint_hfx_aenet

!     /*   neighbor list   */
      integer :: n_j

!     //   energy
      real(8) :: e_i

!     //   force
      real(8), dimension(:,:), allocatable :: f_i

!     //   virial
      real(8) :: vir_i(3,3)

!     //   heat flux
      real(8), dimension(:), allocatable :: hfx, hfy, hfz

!     /*   types in integer - aenet output  */
      integer, dimension(:), allocatable :: jtype_aenet

!     // initialize flag
      integer,save::iset_predict = 0

!***********************************************************************
      end module aenet_variables
!***********************************************************************





!***********************************************************************
      subroutine force_aenet_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, natom, myrank

      use aenet_variables, only : &
     &   istep_train_aenet, iprint_xsf_aenet, dir_train_aenet, &
     &   type_aenet, network_aenet, ntype_aenet, minxsf_train_aenet, &
     &   ioption_xsf_aenet, lstep_train_aenet, delay_aenet, itype_aenet, &
     &   istep_save_aenet, dir_save_aenet, jtype_aenet

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: i, ierr

!     /*   flag for initial visit   */
      integer, save :: iset = 0

!     /*   integers   */
      character(len=80) :: char, symbol

!-----------------------------------------------------------------------
!     /*   read keywords                                              */
!-----------------------------------------------------------------------

      
!     /*   first visit   */
      if ( iset .eq. 0 ) then

!        /*   step interval of training ann   */
         call read_int1_MPI &
     &      ( istep_train_aenet, '<istep_train_aenet>', 19, iounit )

!        /*   step interval for saving the trained networks   */
         call read_int1_MPI &
     &      ( istep_save_aenet, '<istep_save_aenet>', 18, iounit )

!        /*   last step of training ann   */
         call read_int1_MPI &
     &      ( lstep_train_aenet, '<lstep_train_aenet>', 19, iounit )

!        /*   print interval of xsf files  */
         call read_int1_MPI &
     &      ( iprint_xsf_aenet, '<iprint_xsf_aenet>', 18, iounit )

!        /*   print interval of xsf files  */
         call read_int1_MPI &
     &      ( minxsf_train_aenet, '<minxsf_train_aenet>', 20, iounit )

!        /*   directory name of trained networks   */
         call read_char_MPI &
     &      ( dir_train_aenet, 80, '<dir_train_aenet>', 17, iounit )

!        /*   directory for periodically saving trained networks    */
        call read_char_MPI &
     &      ( dir_save_aenet, 80, '<dir_save_aenet>', 16, iounit )

!        /*   option of xsf files  */
         call read_int1_MPI &
     &      ( ioption_xsf_aenet, '<ioption_xsf_aenet>', 19, iounit )

!        /*   read   */
         call read_int1_MPI &
     &      ( ntype_aenet, '<ntype_aenet>', 13, iounit )

!        /*   delay time  */
         call read_char_MPI &
     &      ( delay_aenet, 8, '<delay_aenet>', 13, iounit )

!        /*   memory allocation: types   */
         if ( .not. allocated(type_aenet) ) &
     &      allocate( type_aenet(ntype_aenet) )

!        /*   memory allocation: types   */
         if ( .not. allocated(itype_aenet) ) &
     &      allocate( itype_aenet(natom) )

!        /*   memory allocation: types   */
         if ( .not. allocated(jtype_aenet) ) &
     &      allocate( jtype_aenet(natom) )

!        /*   memory allocation: networks   */
         if ( .not. allocated(network_aenet) ) &
     &      allocate( network_aenet(ntype_aenet) )

!-----------------------------------------------------------------------
!        /*   read keyword from input.dat with the format             */
!-----------------------------------------------------------------------
!
!        <ntype_aenet>
!        number_of_types
!        type_1  network_1
!        type_2  network_2
!        ......  .........
!
!-----------------------------------------------------------------------

!        /*   master rank only   */
         if ( myrank .eq. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'input.dat' )

!           /*   read   */
            call search_tag ( '<ntype_aenet>', 13, iounit, ierr )

!           /*   read number of types  */
            read ( iounit, *, iostat=ierr ) ntype_aenet

!           /*   read types and networks  */
            do i = 1, ntype_aenet
               read ( iounit, *, iostat=ierr ) &
     &            type_aenet(i), network_aenet(i)
            end do

!           /*   file close   */
            close( iounit )

!        /*   master rank only   */
         end if

!        /*   communicate   */
         call my_mpi_bcast_int_0( ierr )

!        /*   stop on error   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine force_aenet_MPI', 26 )

!        /*   communicate   */
         call my_mpi_bcast_char_1 &
     &      ( type_aenet, len(type_aenet), ntype_aenet )

!        /*   communicate   */
         call my_mpi_bcast_char_1 &
     &      ( network_aenet, len(network_aenet), ntype_aenet )

!-----------------------------------------------------------------------

!        /*   master rank only   */
         if ( myrank .eq. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'structure.dat' )

!           /*   read number of types  */
            read ( iounit, *, iostat=ierr )

!           /*   read number of types  */
            read ( iounit, *, iostat=ierr )

!           /*   read atom   */
            do i = 1, natom

               read ( iounit, *, iostat=ierr ) &
     &            symbol, char, char, char, itype_aenet(i)

               if ( symbol(1:2) .ne. type_aenet(itype_aenet(i))(1:2) ) &
     &            write( 6, '(a,i8,a,a,a,a,a)' ) &
     &               'Warning - aenet type of atom ', i, &
     &               ' does not match between ', trim(symbol(1:2)), &
     &               ' and ', trim(type_aenet(itype_aenet(i))(1:2)), '.'

            end do

!           /*   file close   */
            close( iounit )

!        /*   master rank only   */
         end if

!        /*   communicate   */
         call my_mpi_bcast_int_0( ierr )

!        //   error message
         if ( ierr .ne. 0 ) then
            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - type incorrect at structure.dat.'
               write( 6, '(a)' )
            end if
         end if

!        /*   stop on error   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine force_aenet_MPI', 26 )

!        /*   communicate   */
         call my_mpi_bcast_int_1( itype_aenet, natom )

!-----------------------------------------------------------------------

!        /*   second visit   */
         iset = 1

!     /*   first visit   */
      end if

!     /*   start predict   */
      call predict_aenet_MPI

      return
      end

#else
      module aenet_variables
      end module 

!***********************************************************************
      subroutine force_aenet_MPI
!***********************************************************************

      use common_variables, only : myrank

      implicit none

      if ( myrank .eq. 0 ) then

         write( 6, '(a)' ) &
     &      'Error - AENET (parallel) is not linked.'
         write( 6, '(a)' ) &
     &      'Try to compile with -Daenet2 option.'
         write( 6, '(a)' )

      end if

      call error_handling_MPI ( 1, 'subroutine force_aenet_MPI', 26 )

      return
      end

#endif


#if defined(aenet2) || defined(aenet_pytorch)
!#ifdef aenet2


!***********************************************************************
      subroutine analysis_aenet_MPI( ioption )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, au_charge, au_energy, au_length, box, species, &
     &   pot, fx, fy, fz, istep, natom, nbead, iounit, &
     &   iboundary, istep, istep_start, method, myrank, nprocs

      use aenet_variables, only : &
     &   iprint_xsf_aenet, nxsf_train_aenet, minxsf_train_aenet,  &
     &   jflag_aenet, istep_train_aenet, dir_train_aenet, &
     &   ntype_aenet, type_aenet, network_aenet, ioption_xsf_aenet, &
     &   lstep_train_aenet, delay_aenet, istep_save_aenet, &
     &   dir_save_aenet

      use hmc_variables, only : &
     &   x_hmc_last, y_hmc_last, z_hmc_last, pot_hmc_last, &
     &   fx_hmc_last, fy_hmc_last, fz_hmc_last, box_hmc_last

      use rehmc_variables, only : &
     &   x_rehmc_last, y_rehmc_last, z_rehmc_last, pot_rehmc_last, &
     &   fx_rehmc_last, fy_rehmc_last, fz_rehmc_last, box_rehmc_last

      use dual_variables, only : &
     &   justtrained_dual

#ifdef ioutils
      use ioutils_variables, only : &
     &   mkdir, readdir, checkdir
#endif

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: i, j, k, ierr, kstep_aenet, itest, ioption

!     /*   integers   */
      integer :: itssize_aenet = 1000

!     /*   real numbers   */
      real(8) :: xa, ya, za, ax, ay, az, bx, by, bz, cx, cy, cz

!     /*   real numbers   */
      real(8) :: const_1, const_2, const_3

!     /*   high level potential and force   */
      real(8) :: pot_high_j, fxj_high, fyj_high, fzj_high

!     /*   low level potential and force   */
      real(8) :: pot_low_j, fxj_low, fyj_low, fzj_low

!     /*   bead number   */
      character(len=8) :: char_num

!     /*   directory name   */
      character(len=10) :: char_dir = 'structures'

!     /*   step number   */
      integer, save :: istep_aenet = 0

!     /*   flag for initial visit   */
      integer, save :: iset = 0

#ifdef ioutils
      character(len=100), allocatable, save :: filenames(:)
#endif

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( istep .eq. 0 ) then
         continue
      else if ( ioption .ne. 2 ) then
         return
      end if


!-----------------------------------------------------------------------
!     /*   read keywords                                              */
!-----------------------------------------------------------------------

!     /*   first visit   */
      if ( iset .eq. 0 ) then

!        /*   step interval of training ann   */
         call read_int1_MPI &
     &      ( istep_train_aenet, '<istep_train_aenet>', 19, iounit )

!        /*   return by option   */
         if ( istep_train_aenet .le. 0 ) then
            iset = 1
            return
         end if

!        /*   step interval for saving the trained networks   */
         call read_int1_MPI &
     &      ( istep_save_aenet, '<istep_save_aenet>', 18, iounit )

!        /*   last step of training ann   */
         call read_int1_MPI &
     &      ( lstep_train_aenet, '<lstep_train_aenet>', 19, iounit )

!        /*   print interval of xsf files  */
         call read_int1_MPI &
     &      ( iprint_xsf_aenet, '<iprint_xsf_aenet>', 18, iounit )

!        /*   option of xsf files  */
         call read_int1_MPI &
     &      ( ioption_xsf_aenet, '<ioption_xsf_aenet>', 19, iounit )

!        /*   print interval of xsf files  */
         call read_int1_MPI &
     &      ( minxsf_train_aenet, '<minxsf_train_aenet>', 20, iounit )

!        /*   directory name of trained networks   */
         call read_char_MPI &
     &      ( dir_train_aenet, 80, '<dir_train_aenet>', 17, iounit )
     
!        /*   directory for periodically saving trained networks    */
        call read_char_MPI &
     &      ( dir_save_aenet, 80, '<dir_save_aenet>', 16, iounit )

!        /*   read   */
         call read_int1_MPI &
     &      ( ntype_aenet, '<ntype_aenet>', 13, iounit )

!        /*   memory allocation: types   */
         if ( .not. allocated(type_aenet) ) &
     &      allocate( type_aenet(ntype_aenet) )

!        /*   memory allocation: networks   */
         if ( .not. allocated(network_aenet) ) &
     &      allocate( network_aenet(ntype_aenet) )

!-----------------------------------------------------------------------

!        /*   master rank only   */
         if ( myrank .eq. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'input.dat' )

!           /*   read   */
            call search_tag ( '<ntype_aenet>', 13, iounit, ierr )

!           /*   read number of types  */
            read ( iounit, *, iostat=ierr ) ntype_aenet

!           /*   read types and networks  */
            do i = 1, ntype_aenet
               read ( iounit, *, iostat=ierr ) &
     &            type_aenet(i), network_aenet(i)
            end do

!           /*   file close   */
            close( iounit )

!        /*   master rank only   */
         end if

!        /*   communicate   */
         call my_mpi_bcast_int_0( ierr )

!        /*   stop on error   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine force_aenet_MPI', 26 )

!        /*   communicate   */
         call my_mpi_bcast_char_1 &
     &      ( type_aenet, len(type_aenet), ntype_aenet )

!        /*   communicate   */
         call my_mpi_bcast_char_1 &
     &      ( network_aenet, len(network_aenet), ntype_aenet )

!-----------------------------------------------------------------------

!        /*   reset flag   */
         ierr = 0

!        /*   master process   */
         if ( myrank .eq. 0 ) then

!           /*   try five times   */
            do i = 1, 5

!              /*   make directory   */
#ifdef ioutils
!               call mkdir( trim(adjustl(char_dir)) )
               call mkdir( char_dir )

!              /*   test if restart file exists   */
                call checkdir(char_dir, itest)

!              /*   file does not exist   */
               if ( itest .eq. 0 ) then

!                 /*   error: failed   */
                  if ( i .eq. 5 ) then
                     ierr = 1
                     exit
!                 /*   error: try one more   */
                  else
                     call system( 'sleep ' // trim(delay_aenet) )
                     cycle
!                 /*   error   */
                  end if

!              /*   file exists   */
               else

!                 /*   no error   */
                  exit

               end if
#else
               call system( 'mkdir -p ' // trim(adjustl(char_dir)) &
     &                      // ' 2> /dev/null' )
#endif

            end do

!        /*   master process   */
         end if

!        /*   wait   */
         call my_mpi_bcast_int_0( ierr )

!        //   error message
         if ( ( myrank .eq. 0 ) .and. ( ierr .ne. 0 ) ) then
            write( 6, '(a)' )
            write( 6, '(a)' ) 'Error - directory ' // &
     &                         trim(adjustl(char_dir)) // ' not found.'
            write( 6, '(a)' )
         end if

!        /*   stop on error   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine analysis_aenet_MPI', 29 )

!        /*   reset flag   */
         itest = 0

!        /*   master rank only   */
         if ( myrank .eq. 0 ) then

!           /*   test if restart file exists */
            call testfile ( 'aenet.ini', 9, itest, iounit )

!           /*   on error   */
            if ( itest .eq. 1 ) then

!              /*   initial step   */
               istep_aenet = 0

!              /*   reset error flag   */
               ierr = 0

!           /*   read restart file   */
            else

!              /*   open file   */
               open( iounit, file = 'aenet.ini' )

!              /*   read step   */
               read( iounit, *, iostat=ierr ) istep_aenet

!              /*   close file   */
               close( iounit )

!           /*   end of if statement   */
            end if

!        /*   master rank only   */
         end if

!        /*   communicate   */
         call my_mpi_bcast_int_0( ierr )

!        //   error message
         if ( ( myrank .eq. 0 ) .and. ( ierr .ne. 0 ) ) then
            write( 6, '(a)' )
            write( 6, '(a)' ) 'Error - aenet.ini read incorrectly.'
            write( 6, '(a)' )
         end if

!        /*   stop on error   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine analysis_aenet_MPI', 29 )

!        /*   communicate   */
         call my_mpi_bcast_int_0( istep_aenet )

!        /*   reset flag   */
         itest = 0

!        /*   check existence of ann potentials  */
         if ( myrank .eq. 0 ) then
            j = 0
            do i = 1, ntype_aenet
               call testfile ( network_aenet(i), len(network_aenet(i)), &
     &                         itest, iounit )
               if ( itest .eq. 0 ) j = j + 1
            end do
         end if
         call my_mpi_bcast_int_0( j )

!        /*   if they do not exist, generate and train   */
         if ( j .ne. ntype_aenet ) jflag_aenet = 1

!        /*   set complete   */
         iset = 1

!     /*   initial visit   */
      end if

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( istep_train_aenet .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   print xsf file                                             */
!-----------------------------------------------------------------------

!     /*   every iprint_xsf_aenet steps   */
      if ( ( iprint_xsf_aenet .gt. 0 ) .and. &
     &     ( mod(istep,iprint_xsf_aenet) .eq. 0 ) ) then

!        /*   conversion factor   */
         const_1 = au_length * 1.d+10

!        /*   conversion factor   */
         const_2 =  au_charge / au_energy

!        /*   conversion factor   */
         const_3 =  1.d0 / const_2 / const_1

!        /*   loop of beads   */
         do j = 1, nbead

!           /*   skip if this is not my bead   */
            if ( mod( j-1, nprocs ) .ne. myrank ) cycle

!           /*   counter   */
            kstep_aenet = istep_aenet + j

!           /*   counter   */
            call int8_to_char( kstep_aenet, char_num )

!           /*   open sample file   */
            open( iounit, file = char_dir &
     &         // '/' // char_num // '.xsf' )

!           /*   for periodic boundary condition   */
            if ( (iboundary .eq. 1) .or. (iboundary .eq. 2) ) then

!              /*   potential energy in eV   */
               if ( istep .eq. istep_start ) then
                  pot_high_j = pot(j) / const_2
                  pot_low_j = pot(j) / const_2
               else if ( ioption_xsf_aenet .eq. 0 ) then
                  pot_high_j = pot(j) / const_2
                  pot_low_j = pot(j) / const_2
               else if ( method(1:6) .eq. 'PIHMC ' ) then
                  pot_high_j = pot_hmc_last(j) / const_2
                  pot_low_j = pot_hmc_last(j) / const_2
               else if ( method(1:6) .eq. 'REHMC ' ) then
                  pot_high_j = pot_rehmc_last(j) / const_2
                  pot_low_j = pot_rehmc_last(j) / const_2
               else
                  pot_high_j = pot(j) / const_2
                  pot_low_j = pot(j) / const_2
               end if

               write( iounit, '(a,f24.12,a)' ) &
     &            '# total energy = ', pot_high_j, ' eV'

               write( iounit, '(a)' ) 'CRYSTAL'
               write( iounit, '(a)' ) 'PRIMVEC'

!              /*   lattice vectors   */
               if ( istep .eq. istep_start ) then
                  ax = box(1,1) * const_1
                  ay = box(2,1) * const_1
                  az = box(3,1) * const_1
                  bx = box(1,2) * const_1
                  by = box(2,2) * const_1
                  bz = box(3,2) * const_1
                  cx = box(1,3) * const_1
                  cy = box(2,3) * const_1
                  cz = box(3,3) * const_1
               else if ( ioption_xsf_aenet .eq. 0 ) then
                  ax = box(1,1) * const_1
                  ay = box(2,1) * const_1
                  az = box(3,1) * const_1
                  bx = box(1,2) * const_1
                  by = box(2,2) * const_1
                  bz = box(3,2) * const_1
                  cx = box(1,3) * const_1
                  cy = box(2,3) * const_1
                  cz = box(3,3) * const_1
               else if ( method(1:6) .eq. 'PIHMC ' ) then
                  ax = box_hmc_last(1,1) * const_1
                  ay = box_hmc_last(2,1) * const_1
                  az = box_hmc_last(3,1) * const_1
                  bx = box_hmc_last(1,2) * const_1
                  by = box_hmc_last(2,2) * const_1
                  bz = box_hmc_last(3,2) * const_1
                  cx = box_hmc_last(1,3) * const_1
                  cy = box_hmc_last(2,3) * const_1
                  cz = box_hmc_last(3,3) * const_1
               else if ( method(1:6) .eq. 'REHMC ' ) then
                  ax = box_rehmc_last(1,1,j) * const_1
                  ay = box_rehmc_last(2,1,j) * const_1
                  az = box_rehmc_last(3,1,j) * const_1
                  bx = box_rehmc_last(1,2,j) * const_1
                  by = box_rehmc_last(2,2,j) * const_1
                  bz = box_rehmc_last(3,2,j) * const_1
                  cx = box_rehmc_last(1,3,j) * const_1
                  cy = box_rehmc_last(2,3,j) * const_1
                  cz = box_rehmc_last(3,3,j) * const_1
               else
                  ax = box(1,1) * const_1
                  ay = box(2,1) * const_1
                  az = box(3,1) * const_1
                  bx = box(1,2) * const_1
                  by = box(2,2) * const_1
                  bz = box(3,2) * const_1
                  cx = box(1,3) * const_1
                  cy = box(2,3) * const_1
                  cz = box(3,3) * const_1
               end if

!              /*   write three lines   */
               write( iounit, '(3f16.8)' ) ax, ay, az
               write( iounit, '(3f16.8)' ) bx, by, bz
               write( iounit, '(3f16.8)' ) cx, cy, cz

!              /*   write one line   */
               write( iounit, '(a)' ) 'PRIMCOORD'

!              /*   write one line   */
               write( iounit, '(i8,i2)' ) natom, 1

!           /*   for free boundary condition   */
            else if ( iboundary .eq. 0 ) then

!              /*   potential energy in eV   */
               if ( istep .eq. istep_start ) then
                  pot_high_j = pot(j) / const_2
                  pot_low_j = pot(j) / const_2
               else if ( ioption_xsf_aenet .eq. 0 ) then
                  pot_high_j = pot(j) / const_2
                  pot_low_j = pot(j) / const_2
               else if ( method(1:6) .eq. 'PIHMC ' ) then
                  pot_high_j = pot_hmc_last(j) / const_2
                  pot_low_j = pot_hmc_last(j) / const_2
               else if ( method(1:6) .eq. 'REHMC ' ) then
                  pot_high_j = pot_rehmc_last(j) / const_2
                  pot_low_j = pot_rehmc_last(j) / const_2
               else
                  pot_high_j = pot(j) / const_2
                  pot_low_j = pot(j) / const_2
               end if

               write( iounit, '(a,f24.12,a)' ) &
     &            '# total energy = ', pot_high_j, ' eV'

!              /*   write one line   */
               write( iounit, '(a)' ) 'ATOMS'

!           /*   for periodic boundary condition   */
            end if

!           /*   loop of atoms   */
            do i = 1, natom

!              /*   geometry in angstroms   */
               if ( istep .eq. istep_start ) then
                  xa = x(i,j) * const_1
                  ya = y(i,j) * const_1
                  za = z(i,j) * const_1
               else if ( ioption_xsf_aenet .eq. 0 ) then
                  xa = x(i,j) * const_1
                  ya = y(i,j) * const_1
                  za = z(i,j) * const_1
               else if ( method(1:6) .eq. 'PIHMC ' ) then
                  xa = x_hmc_last(i,j) * const_1
                  ya = y_hmc_last(i,j) * const_1
                  za = z_hmc_last(i,j) * const_1
               else if ( method(1:6) .eq. 'REHMC ' ) then
                  xa = x_rehmc_last(i,j) * const_1
                  ya = y_rehmc_last(i,j) * const_1
                  za = z_rehmc_last(i,j) * const_1
               else
                  xa = x(i,j) * const_1
                  ya = y(i,j) * const_1
                  za = z(i,j) * const_1
               end if

!              /*   geometry in eV per angstrom   */
               if ( istep .eq. istep_start ) then
                  fxj_high = fx(i,j) * const_3 * dble(nbead)
                  fyj_high = fy(i,j) * const_3 * dble(nbead)
                  fzj_high = fz(i,j) * const_3 * dble(nbead)
                  fxj_low = fx(i,j) * const_3 * dble(nbead)
                  fyj_low = fy(i,j) * const_3 * dble(nbead)
                  fzj_low = fz(i,j) * const_3 * dble(nbead)
               else if ( ioption_xsf_aenet .eq. 0 ) then
                  fxj_high = fx(i,j) * const_3 * dble(nbead)
                  fyj_high = fy(i,j) * const_3 * dble(nbead)
                  fzj_high = fz(i,j) * const_3 * dble(nbead)
                  fxj_low = fx(i,j) * const_3 * dble(nbead)
                  fyj_low = fy(i,j) * const_3 * dble(nbead)
                  fzj_low = fz(i,j) * const_3 * dble(nbead)
               else if ( method(1:6) .eq. 'PIHMC ' ) then
                  fxj_high = fx_hmc_last(i,j) * const_3 * dble(nbead)
                  fyj_high = fy_hmc_last(i,j) * const_3 * dble(nbead)
                  fzj_high = fz_hmc_last(i,j) * const_3 * dble(nbead)
                  fxj_low = fx_hmc_last(i,j) * const_3 * dble(nbead)
                  fyj_low = fy_hmc_last(i,j) * const_3 * dble(nbead)
                  fzj_low = fz_hmc_last(i,j) * const_3 * dble(nbead)
               else if ( method(1:6) .eq. 'REHMC ' ) then
                  fxj_high = fx_rehmc_last(i,j) * const_3 * dble(nbead)
                  fyj_high = fy_rehmc_last(i,j) * const_3 * dble(nbead)
                  fzj_high = fz_rehmc_last(i,j) * const_3 * dble(nbead)
                  fxj_low = fx_rehmc_last(i,j) * const_3 * dble(nbead)
                  fyj_low = fy_rehmc_last(i,j) * const_3 * dble(nbead)
                  fzj_low = fz_rehmc_last(i,j) * const_3 * dble(nbead)
               else
                  fxj_high = fx(i,j) * const_3 * dble(nbead)
                  fyj_high = fy(i,j) * const_3 * dble(nbead)
                  fzj_high = fz(i,j) * const_3 * dble(nbead)
                  fxj_low = fx(i,j) * const_3 * dble(nbead)
                  fyj_low = fy(i,j) * const_3 * dble(nbead)
                  fzj_low = fz(i,j) * const_3 * dble(nbead)
               end if

!              /*   write one line   */
               write( iounit, '(a4,6f16.8)' ) &
     &            species(i)(1:4), xa, ya, za, &
     &            fxj_high, fyj_high, fzj_high

!           /*   loop of atoms   */
            end do

!           /*   close input file   */
            close( iounit )

!        /*   loop of beads   */
         end do

!        /*   update counter   */
         istep_aenet = istep_aenet + nbead

!        /*   wait   */
         call my_mpi_barrier

!-----------------------------------------------------------------------
!        /*   restart file                                            */
!-----------------------------------------------------------------------

!        /*   master rank only   */
         if ( myrank .eq. 0 ) then

!           /*   write step  */
            open( iounit, file = 'aenet.ini' )
            write( iounit, '(i8)' ) istep_aenet
            close( iounit )

!        /*   master rank only   */
         end if

!     /*   every iprint_xsf_aenet steps   */
      end if

!-----------------------------------------------------------------------
!     /*   count number of xsf files                                  */
!-----------------------------------------------------------------------

!     /*   conditions to generate and train   */
      if ( ( istep_train_aenet .gt. 0 ) .and. &
     &     ( istep .le. lstep_train_aenet ) .and. &
     &     ( mod(istep,istep_train_aenet) .eq. 0 ) ) then

!        /*   wait for all processes   */
         call my_mpi_barrier

!        //   try five times
         do i = 1, 5

!           /*   add number of structure files   */
            if ( myrank .eq. 0 ) then
#ifdef ioutils
!              /*   structure files   */
               call readdir( char_dir, nxsf_train_aenet, &
     &                       filenames )
#else
               call system &
     &            ( 'ls -U1 ' // trim(adjustl(char_dir)) // &
     &              '/*.xsf > test.1.out; ' &
     &           // 'cat test.1.out | wc -l > test.2.out' )
#endif
            end if

!           /*   wait for all processes   */
            call my_mpi_barrier

!           /*   delay   */
            do k = 1, i-1
               call system( 'sleep ' // delay_aenet )
            end do

!           /*   read number of xsf files   */
#ifdef ioutils
            ierr = 0
#else
            if ( myrank .eq. 0 ) then
               open ( iounit, file = 'test.2.out' )
               read ( iounit, *, iostat=ierr ) nxsf_train_aenet
               close( iounit )
            end if
#endif

!           //   communicate
            call my_mpi_bcast_int_0( ierr )

!           //   exit from loop
            if ( ierr .eq. 0 ) then
!               if ( myrank .eq. 0 ) then
!                  write( 6, '(a,i8,a)' )
!     &            'Xsf files are found: ', nxsf_train_aenet, ' files.'
!               end if
               do k = 1, i-1
                  call system( 'sleep ' // delay_aenet )
               end do
               exit
!           //   delay
            else
!               if ( myrank .eq. 0 ) then
!                  write( 6, '(a,i1,a)' )
!     &               'Xsf files are missing, reread ', i, ' times.'
!               end if
               do k = 1, i-1
                  call system( 'sleep ' // delay_aenet )
               end do
            end if

!        //   try five times
         end do

!        //   error message
         if ( ( myrank .eq. 0 ) .and. ( ierr .ne. 0 ) ) then
            write( 6, '(a)' )
            write( 6, '(a)' ) 'Error - Xsf files could not be found.'
            write( 6, '(a)' )
         end if

!        //   stop on error
         call error_handling_MPI &
     &      ( ierr, 'subroutine analysis_aenet_MPI', 29 )

!        /*   wait for all processes   */
         call my_mpi_barrier

!        /*   remove file   */
#ifdef ioutils
#else
         if ( myrank .eq. 0 ) &
     &      call system( 'rm -f test.1.out test.2.out 2> /dev/null' )
#endif

!        /*   wait for all processes   */
         call my_mpi_barrier

!     /*   conditions to generate and train   */
      end if

!     /*   number of xsf files   */
      call my_mpi_bcast_int_0( nxsf_train_aenet )

!-----------------------------------------------------------------------
!     /*   generate and train                                         */
!-----------------------------------------------------------------------

!     /*   conditions to generate and train   */
      if ( ( istep_train_aenet .gt. 0 ) .and. &
     &     ( ( ( istep .le. lstep_train_aenet ) .and. &
     &         ( mod(istep,istep_train_aenet) .eq. 0 ) &
     &      .and. (istep .ne. 0) ) .or.  &
     &         ( jflag_aenet .eq. 1 ) ) .and. &
     &     ( nxsf_train_aenet .ge. minxsf_train_aenet ) ) then


!----------------------------------------------------------------------

!        /*   wait for all processes   */
         call my_mpi_barrier

!        //   initial value
         j = 0

!        /*   generate   */

         call generate_aenet_MPI
#ifdef aenet2
!        /*   try five times   */
         do i = 1, 5

!c           /*   generate   */
!            call generate_aenet_MPI

!           /*   add number of structure files   */
            if ( myrank .eq. 0 ) then
#ifdef ioutils
               call readdir( char_dir, j, filenames )
#else
               call system &
     &            ( 'cat *.train | wc -l > test.2.out 2> /dev/null' )
#endif
            end if

!           /*   wait for all processes   */
            call my_mpi_barrier

!           /*   number of files that match   */
#ifdef ioutils
            ierr = 0
            j = itssize_aenet
#else
            if ( myrank .eq. 0 ) then
               open ( iounit, file = 'test.2.out' )
               read( iounit, *, iostat=ierr ) j
               close( iounit )
            end if
#endif

!           /*   communicate   */
            call my_mpi_bcast_int_0( ierr )
            call my_mpi_bcast_int_0( j )

!           /*   no error   */
            if ( ( ierr .eq. 0 ) .and. ( j .ge. itssize_aenet ) ) then
               if ( myrank .eq. 0 ) then
                  write( 6, '(a,i8,a)' ) &
     &               'Train file is found: ', j, ' lines.'
               end if
               do k = 1, i-1
                  call system( 'sleep ' // delay_aenet )
               end do
               exit
!           /*   delay   */
            else
               if ( myrank .eq. 0 ) then
                  write( 6, '(a,i1,a)' ) &
     &               'Train file is missing, reread ', i, ' times.'
               end if
               if ( i .eq. 5 ) ierr = 1
               call my_mpi_bcast_int_0( ierr )
               do k = 1, i-1
                  call system( 'sleep ' // delay_aenet )
               end do
            end if

!        /*   try five times   */
         end do

!        //   error message
         if ( ( myrank .eq. 0 ) .and. ( ierr .ne. 0 ) ) then
            write( 6, '(a)' )
            write( 6, '(a)' ) 'Error - Train file could not be found.'
            write( 6, '(a)' )
         end if

!        //   stop on error
         call error_handling_MPI &
     &      ( ierr, 'subroutine analysis_aenet_MPI', 29 )

!        /*   wait for all processes   */
         call my_mpi_barrier

!        /*   remove file   */
#ifdef ioutils
#else
         if ( myrank .eq. 0 ) &
     &      call system( 'rm -f test.1.out test.2.out 2> /dev/null' )
#endif

!        /*   wait for all processes   */
         call my_mpi_barrier
#else
         !if ( myrank .eq. 0 ) then
         !   call system( 'touch trainingstart.tmp' )
         !end if
#endif
!----------------------------------------------------------------------

!        /*   train   */
!#ifdef aenet2
#if defined(aenet2) || defined(aenet_pytorch)
         call train_aenet_MPI

!        /*   change flag   */
!#ifdef aenet2
         jflag_aenet = 0
         call reset_flag_predict_aenet_MPI

!        /*   flag just trained   */
         justtrained_dual = 1
!#endif
!#else
!
!        //   error message
!!       if ( ( myrank .eq. 0 ) .and. ( ierr .ne. 0 ) ) then
!           write( 6, '(a)' ) &
!           write( 6, '(a)' ) &
!!             'Error - training is not implemented ' &
!!             // 'with aenet_pytorch flag.'
!           write( 6, '(a)' )
!        end if
#endif

!     /*   conditions to generate and train   */
      end if

      return
      end


      

#if defined(aenet2) || defined(aenet_pytorch)
!***********************************************************************
      subroutine generate_aenet_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!
!     In input.dat the statements after the keyword <generate_aenet>
!     is simply copied to file generate.in without FILES information.
!     The FILES information is then added automatically.
!
!-----------------------------------------------------------------------

!     /*   shared variables   */
      use common_variables, only : &
     &   iounit, iounit_aenet, istep, myrank, mpi_comm_pimd

!     /*   shared variables   */
      use aenet_variables, only : &
     &   iflag_aenet, delay_aenet

#ifdef ioutils
      use ioutils_variables, only : &
     &   readdir, remove_all
#endif

#ifdef aenet_pytorch
      use aenet_generate,only:generate_subroutine
      use aenet_trainbin2ascii,only:trainbin2ascii_subroutine
#endif

!     /*   local variables   */
      implicit none

!     /*   integers   */
      integer :: ierr, jerr, i, j, k

!     /*   integers   */
      integer, save :: iset = 0

!     /*   characters   */
      character(len=80) :: char_line, char_word

!     /*   integers   */
      integer :: iounit_generate = 1050

!     /*   real numbers  */
      real(8) :: time_started, time_ended

!     /*   directory name   */
      character(len=10) :: char_dir = 'structures'

#ifdef ioutils
      character(len=100), allocatable, save :: filenames(:)
#endif


!     /*   mpi   */
      include 'mpif.h'

!-----------------------------------------------------------------------
!     /*   print message                                              */
!-----------------------------------------------------------------------

!     /*   time started   */
      time_started = mpi_wtime()

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

         write( 6, '(a)' )

         write( 6, '(a)' ) &
     &      'AENETAENETAENETAENETAENETAENETAENETAENET' // &
     &      'AENETAENETAENETAENETAENETAENETAENETAEN'

         write( 6, '(a)' ) &
     &      'Generate started.'

         write( 6, '(a)' ) &
     &      'AENETAENETAENETAENETAENETAENETAENETAENET' // &
     &      'AENETAENETAENETAENETAENETAENETAENETAEN'

         write( 6, '(a)' )

!     /*   master rank only   */
      end if

!-----------------------------------------------------------------------
!     /*   set the communicator                                       */
!-----------------------------------------------------------------------

!     /*   if communicator not set   */
      if ( iflag_aenet .ne. 2 ) then

!        /*   set mpi comm   */

#ifdef aenet2
         call set_mpi_aenet( mpi_comm_pimd )

!        /*   set flag   */
         iflag_aenet = 2
#endif

!     /*   if communicator not set   */
      end if

!-----------------------------------------------------------------------
!     /*   read keyword from input.dat                                */
!-----------------------------------------------------------------------

!     /*   first visit   */
      if ( iset .eq. 0 ) then

!        /*   master rank   */
         if ( myrank .eq. 0 ) then

!           /*   open file   */
            open ( iounit, file = 'input.dat' )

!           /*   read   */
            call search_tag ( '<generate_aenet>', 16, iounit, ierr )

!           /*   close file   */
            close( iounit )

!        /*   master rank   */
         end if

!        /*   communicate   */
         call my_mpi_bcast_int_0( ierr )

!        /*   error message   */
         if ( ( myrank .eq. 0 ) .and. ( ierr .ne. 0 ) ) then
            write( 6, '(a)' )
            write( 6, '(a)' ) &
     &        'Error - missing keyword <generate_aenet>.'
            write( 6, '(a)' )
         end if

!        /*   stop on error   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine generate_aenet_MPI', 29 )

!        /*   end of initial setting   */
         iset = 1

!     /*   first visit   */
      end if

!-----------------------------------------------------------------------
!     /*   create input for generate                                  */
!-----------------------------------------------------------------------

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

!        /*   open file   */
         open ( iounit, file = 'input.dat' )

!        /*   open file   */
         open ( iounit_aenet, file = 'generate.in' )

!        /*   read   */
         call search_tag ( '<generate_aenet>', 16, iounit, ierr )

!        /*   line by line   */
         do

!           /*   read line   */
            read ( iounit, '(a)', iostat=jerr ) char_line

!           /*   read correctly   */
            if ( jerr .eq. 0 ) then

!              /*   but found next keyword   */
               if ( char_line(1:1) .eq. '<' ) jerr = 1

!              /*   but found blank line   */
               if ( char_line(1:1) .eq. ' ' ) then
                  char_word = adjustl(char_line)
                  if ( char_word(1:1) .eq. ' ' ) jerr = 1
                  if ( char_word(1:1) .eq. '<' ) jerr = 1
               end if

!           /*   read correctly   */
            end if

!           /*   read incorrectly exit   */
            if ( jerr .ne. 0 ) exit

!           /*   otherwise print line   */
            write( iounit_aenet, '(a)' ) char_line

!        /*   line by line   */
         end do

!        /*   close file   */
         close( iounit )

!        /*   add FILES   */
         write( iounit_aenet, '(a)' ) 'FILES'

!        /*   close file   */
         close( iounit_aenet )

!     /*   master rank only   */
      end if

!     /*   wait for all processes to finish   */
      call my_mpi_barrier

!     //   initial value
      j = 0

!     //   try five times
      do i = 1, 5

!        /*   add number of structure files   */
         if ( myrank .eq. 0 ) then
#ifdef ioutils
            call readdir( char_dir, j, filenames )
#else
            call system &
     &         ( 'ls -U1 ' // trim(adjustl(char_dir)) // &
     &           '/*.xsf > test.1.out; ' // &
     &           'cat test.1.out | wc -l > test.2.out' )
#endif
         end if

!        /*   wait for all processes   */
         call my_mpi_barrier

!        /*   delay   */
         do k = 1, i-1
            call system( 'sleep ' // delay_aenet )
         end do

!        /*   read number of xsf files   */
#ifdef ioutils
         ierr = 0
#else
         if ( myrank .eq. 0 ) then
            open ( iounit, file = 'test.2.out' )
               read ( iounit, *, iostat=ierr ) j
            close( iounit )
         end if
#endif

!        //   communicate
         call my_mpi_bcast_int_0( ierr )
         call my_mpi_bcast_int_0( j )

!        //   exit from loop
         if ( ( ierr .eq. 0 ) .and. ( j .gt. 0 ) ) then
            if ( myrank .eq. 0 ) then
               write( 6, '(a,i8,a)' ) &
     &            'Xsf files are found: ', j, ' files.'
               write( 6, '(a)' )
!               call system
!                  ( 'cat test.2.out >> generate.in; '
!     &           // 'cat test.1.out >> generate.in' )
            end if
            do k = 1, i-1
               call system( 'sleep ' // delay_aenet )
            end do
            exit
!        //   delay
         else
            if ( myrank .eq. 0 ) then
               write( 6, '(a,i1,a)' ) &
     &            'Xsf files are missing, reread ', i, ' times.'
            end if
            if ( i .eq. 5 ) ierr = 1
            call my_mpi_bcast_int_0( ierr )
            do k = 1, i-1
               call system( 'sleep ' // delay_aenet )
            end do
         end if

!     //   try five times
      end do

!     //   error message
      if ( ( myrank .eq. 0 ) .and. ( ierr .ne. 0 ) ) then
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Error - Xsf files could not be found.'
         write( 6, '(a)' )
      end if

!     //   stop on error
      call error_handling_MPI &
     &   ( ierr, 'subroutine generate_aenet_MPI', 29 )

!     //   initialize
      ierr = 0

!     //   master rank
      if ( myrank .eq. 0 ) then

#ifdef ioutils
         open ( iounit_aenet, file = 'generate.in', access='append' )
         write( iounit_aenet, '(i8)' ) j
         do i = 1, j
            write( iounit_aenet, '(a)' ) char_dir//"/"//filenames(i)
         end do
         close( iounit_aenet )
#else
         open ( iounit_aenet, file = 'generate.in', access='append' )
         open ( iounit, file = 'test.1.out' )
         write( iounit_aenet, '(i8)' ) j
         do i = 1, j
            read ( iounit, '(a)', iostat=ierr ) char_line
            write( iounit_aenet, '(a)' ) char_line
         end do
         close( iounit_aenet )
         close( iounit )
#endif

!     //   master rank
      end if

!     //   communicate
      call my_mpi_bcast_int_0( ierr )

!     //   error message
      if ( ( myrank .eq. 0 ) .and. ( ierr .ne. 0 ) ) then
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Error - Xsf files read incorrectly.'
         write( 6, '(a)' )
      end if

!     //   stop on error
      call error_handling_MPI &
     &   ( ierr, 'subroutine generate_aenet_MPI', 29 )

!-----------------------------------------------------------------------
!        /*   remove train files                                      */
!-----------------------------------------------------------------------

!     /*   wait for all processes to finish   */
      call my_mpi_barrier

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then
#ifdef ioutils
         call remove_all( ".+.train$" , "./" )
         call remove_all( ".+.train.scaled$" , "./" )
#else
         call system( 'rm -f test.1.out test.2.out 2> /dev/null' )
         call system( 'rm -f *.train *.train.scaled 2> /dev/null' )
#endif
      end if

!     /*   wait for all processes to finish   */
      call my_mpi_barrier

!     /*   delay   */
#ifdef ioutils
#else
      call system( 'sleep ' // delay_aenet )
#endif

!-----------------------------------------------------------------------
!     /*   execute generate                                           */
!-----------------------------------------------------------------------

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

!        /*   open generate file   */
         open ( iounit_generate, file = 'generate.scr' )

!        /*   print   */
         write( iounit_generate, '(/,a,i8)' ) &
     &      "generate was started at step: ", istep

!     /*   master rank only   */
      end if

!     /*   execution of generate   */
#ifdef aenet2      
      call generate_sub_MPI( 'generate.in', iounit_generate )
#endif
#ifdef aenet_pytorch
      if ( myrank .eq. 0 ) then
!        //   warning: MPI is not supported
         call generate_subroutine( 'generate.in', iounit_generate )
      end if
      call my_mpi_barrier
#endif

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

!        /*   print   */
         write( iounit_generate, '(/,a,i8)' ) &
     &      "generate was finished at step: ", istep

!        /*   close generate file   */
         close( iounit_generate )

!     /*   master rank only   */
      end if

!     /*   wait for all processes to finish   */
      call my_mpi_barrier

!-----------------------------------------------------------------------
!     /*   print message                                              */
!-----------------------------------------------------------------------

!     /*   time started   */
      time_ended = mpi_wtime()

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

         write( 6, '(a)' )

         write( 6, '(a)' ) &
     &      'AENETAENETAENETAENETAENETAENETAENETAENET' // &
     &      'AENETAENETAENETAENETAENETAENETAENETAEN'

         write( 6, '(a,f10.2,a)' ) &
     &      'Generate ended in', &
     &      time_ended-time_started, &
     &      ' seconds.'

         write( 6, '(a)' ) &
     &      'AENETAENETAENETAENETAENETAENETAENETAENET' // &
     &      'AENETAENETAENETAENETAENETAENETAENETAEN'

         write( 6, '(a)' )

!     /*   master rank only   */
      end if

      return
      end


#endif

#if defined(aenet2) || defined(aenet_pytorch)
!***********************************************************************
      subroutine train_aenet_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!
!     In input.dat the statements after the keyword <train_aenet>
!     is simply copied to file train.in.
!
!-----------------------------------------------------------------------

!     /*   shared variables   */
      use common_variables, only : &
     &    iounit, iounit_aenet, istep, mpi_comm_pimd, myrank

!     /*   shared variables   */
      use aenet_variables, only : &
     &   iflag_aenet, dir_train_aenet, ntype_aenet, network_aenet, &
     &   dir_save_aenet, istep_save_aenet

#ifdef ioutils
      use ioutils_variables, only : &
     &   mkdir, remove_all
#endif
#ifdef aenet_pytorch
      use aenet_nnASCII2bin, only : nnASCII2bin_subroutine
#endif 

!     /*   local variables   */
      implicit none

!     /*   integers   */
      integer :: i, ierr, jerr, itest

!     /*   integers   */
      integer, save :: iset = 0

!     /*   characters   */
      character(len=80) :: char_line, char_word
      character(len=8)  :: char_num

!     /*   integers   */
      integer :: iounit_train = 1100

!     /*   real numbers  */
      real(8) :: time_started, time_ended

      LOGICAL :: fileExists
      CHARACTER(LEN=100) :: finished_tagfile

#ifdef aenet_pytorch      
      character(len=80) :: char_line_ascii
      character(len=1024) :: infile, outfile
      logical            :: to_bin, to_ascii
#endif

!     /*   mpi   */
      include 'mpif.h'

!-----------------------------------------------------------------------
!     /*   print message                                              */
!-----------------------------------------------------------------------

!     /*   time started   */
      time_started = mpi_wtime()
#ifdef aenet2
!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

         write( 6, '(a)' )

         write( 6, '(a)' ) &
     &      'AENETAENETAENETAENETAENETAENETAENETAENET' // &
     &      'AENETAENETAENETAENETAENETAENETAENETAEN'

         write( 6, '(a)' ) &
     &      'Train started.'

         write( 6, '(a)' ) &
     &      'AENETAENETAENETAENETAENETAENETAENETAENET' // &
     &      'AENETAENETAENETAENETAENETAENETAENETAEN'
         write( 6, '(a)' )

         write( 6, '(a)' )

!     /*   master rank only   */
      end if
#endif

!-----------------------------------------------------------------------
!     /*   set the communicator                                       */
!-----------------------------------------------------------------------

!     /*   if communicator not set   */
      if ( iflag_aenet .ne. 3 ) then

#ifdef aenet2
!        /*   set mpi comm   */
         call set_mpi_aenet( mpi_comm_pimd )

!        /*   set flag   */
         iflag_aenet = 3

!     /*   if communicator not set   */
#endif
      end if

!-----------------------------------------------------------------------
!     /*   read keyword from input.dat                                */
!-----------------------------------------------------------------------

!     /*   first visit   */
      if ( iset .eq. 0 ) then

!        /*   master rank   */
         if ( myrank .eq. 0 ) then

!           /*   open file   */
            open ( iounit, file = 'input.dat' )

!           /*   read   */
            call search_tag ( '<train_aenet>', 13, iounit, ierr )

!           /*   close file   */
            close( iounit )

!        /*   master rank   */
         end if

!        /*   communicate   */
         call my_mpi_bcast_int_0( ierr )

!        /*   stop on error   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine train_aenet_MPI', 26 )

!        /*   end of initial setting   */
         iset = 1

!     /*   first visit   */
      end if

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

!        /*   open file   */
         open ( iounit, file = 'input.dat' )

!        /*   open file   */
         open ( iounit_aenet, file = 'train.in' )

!        /*   read   */
         call search_tag ( '<train_aenet>', 13, iounit, ierr )

!        /*   line by line   */
         do

!           /*   read line   */
            read ( iounit, '(a)', iostat=jerr ) char_line

!           /*   read correctly   */
            if ( jerr .eq. 0 ) then

!              /*   but found next keyword   */
               if ( char_line(1:1) .eq. '<' ) jerr = 1

!              /*   but found blank line   */
               if ( char_line(1:1) .eq. ' ' ) then
                  char_word = adjustl(char_line)
                  if ( char_word(1:1) .eq. ' ' ) jerr = 1
               end if

!           /*   read correctly   */
            end if

!           /*   read incorrectly exit   */
            if ( jerr .ne. 0 ) exit

!           /*   otherwise print line   */
            write( iounit_aenet, '(a)' ) char_line

!        /*   line by line   */
         end do

!        /*   close file   */
         close( iounit )

!        /*   close file   */
         close( iounit_aenet )

!     /*   master rank only   */
      end if

!-----------------------------------------------------------------------
!        /*   make train directory and remove train files             */
!-----------------------------------------------------------------------

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

!        /*   make directory as needed   */
#ifdef ioutils
         call mkdir( dir_train_aenet )
#else
         call system( 'mkdir -p ' // trim(adjustl(dir_train_aenet)) )
#endif

         if ( istep_save_aenet .gt. 0) then
            call system( 'mkdir -p ' // trim(adjustl(dir_save_aenet)) )
         end if

!        /*   remove files   */
#ifdef ioutils
         call remove_all( "^TEST.*" , "./" )
         call remove_all( "^TRAIN.*" , "./" )
#else
         call system( 'rm -f TEST.* TRAIN.* 2> /dev/null' )
#endif

!     /*   master rank only   */
      end if

!     /*   wait for all processes to finish   */
      call my_mpi_barrier

!-----------------------------------------------------------------------
!        /*   execute train                                           */
!-----------------------------------------------------------------------
#ifdef aenet2
!     /*   print   */
      if ( myrank .eq. 0 ) then

!        /*   open generate file   */
         open ( iounit_train, file = 'train.scr' )

         write( iounit_train, '(/,a,i8)' ) &
     &      "train was started at step: ", istep

      end if

!     /*   execution of generate   */

      call train_sub( 'train.in', iounit_train, dir_train_aenet )


!     /*   print   */
      if ( myrank .eq. 0 ) then

         write( iounit_train, '(/,a,i8)' ) &
     &      "train was finished at step: ", istep
         write( iounit_train, '(a)' ) &
     &      'train directory: ' // trim(adjustl(dir_train_aenet))

!        /*   close generate file   */
         close( iounit_train )

      end if

#endif
#ifdef aenet_pytorch
      finished_tagfile = 'trainingfinished.tmp.ini'
      if ( myrank .eq. 0 ) then
            call system( 'touch trainingstart.tmp.ini' )
            open(101,file="dir_train_aenet.ini")
            write(101,*) trim(adjustl(dir_train_aenet))
            close(101)
      end if
#endif

!     /*   wait for all processes to finish   */
      call my_mpi_barrier


#ifdef aenet_pytorch
      INQUIRE(FILE=finished_tagfile, EXIST=fileExists)
      if (fileExists) then
            write( 6, '(a)' ) 'Training is done. Update potential files '
#endif
!-----------------------------------------------------------------------
!     /*   copy final network file                                    */
!-----------------------------------------------------------------------

!     /*   reset flag   */
      itest = 0

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

         !        /*  loop of atomic types   */
#ifdef aenet_pytorch            
         do i = 1, ntype_aenet
!            /*   subdirectory file  */
            char_line = trim(adjustl(dir_train_aenet)) // '/' // &
     &                  trim(adjustl(network_aenet(i)))
            char_line_ascii = trim(adjustl(char_line))// '.ascii'
            call testfile ( char_line_ascii, len(char_line_ascii), &
     &                      itest, iounit )
     !           /*   detect error   */
            if ( itest .ne. 0 ) then
               write( 6, '(a)' )
               write( 6, '(a)' ) &
     &            'Error - trained network not found: ' &
     &            // trim(adjustl(char_line_ascii))
               write( 6, '(a)' )
               exit
            else 
               to_ascii = .false.
               to_bin = .true.
               infile = trim(adjustl(char_line_ascii))
               outfile = trim(adjustl(char_line))
               call nnASCII2bin_subroutine( &
     &             trim(adjustl(infile)), trim(adjustl(outfile)), &
     &             to_bin, to_ascii)
!               call nnASCII2bin_subroutine( &
!     &             char_line_ascii, char_line, to_bin, to_ascii)
            end if

         end do
#endif

!        /*  loop of atomic types   */
         do i = 1, ntype_aenet

!           /*   subdirectory file  */
            char_line = trim(adjustl(dir_train_aenet)) // '/' // &
     &                  trim(adjustl(network_aenet(i)))


!           /*   check existence of file   */
            call testfile ( char_line, len(char_line), &
     &                      itest, iounit )

!           /*   detect error   */
            if ( itest .ne. 0 ) then
               write( 6, '(a)' )
               write( 6, '(a)' ) &
     &            'Error - trained network not found: ' &
     &            // trim(adjustl(char_line))
               write( 6, '(a)' )
               exit
            else
               write( 6, '(a)' ) &
     &            'trained network overwritten by: ' &
     &            // trim(adjustl(char_line))
            end if

!           /*   main directory file  */
            char_word = trim(adjustl(network_aenet(i)))

!           /*   copy file  */
#ifdef ioutils
            call system( 'cp ' // char_line // ' ' // char_word )
#else
            call system( 'cp ' // char_line // ' ' // char_word )
#endif

!           /* If we reach a step where we wish to    */
!           /* save the trained network we save it.   */
            if( ( istep_save_aenet .gt. 0 ) .and. &
     &          ( mod(istep,istep_save_aenet) .eq. 0 )) then

               call int8_to_char( istep, char_num )

               char_word = trim(adjustl(dir_save_aenet)) // '/' // &
     &                     trim(adjustl(network_aenet(i))) // &
     &                     '_iter_' // trim(adjustl(char_num))

#ifdef ioutils
               call system( 'cp ' // char_line // ' ' // char_word )
#else
               call system( 'cp ' // char_line // ' ' // char_word )
#endif

            end if

!c           /*   copy file  */
!            call system( 'rm -f ' // char_line // '-* 2> /dev/null' )

!        /*  loop of atomic types   */
         end do

!c        /*   subdirectory file  */
!         char_line = trim(adjustl(dir_train_aenet))

!     /*   master rank only   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( itest )

!     /*   error termination   */
      call error_handling_MPI( itest, 'subroutine train_aenet_MPI', 26 )

!c     /*   wait for all processes to finish   */
!      call my_mpi_barrier

!-----------------------------------------------------------------------
!     /*   print message                                              */
!-----------------------------------------------------------------------

!     /*   time started   */
      time_ended = mpi_wtime()

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

         write( 6, '(a)' )

         write( 6, '(a)' ) &
     &      'AENETAENETAENETAENETAENETAENETAENETAENET' // &
     &      'AENETAENETAENETAENETAENETAENETAENETAEN'

         write( 6, '(a,f10.2,a)' ) &
     &      'Train ended in', &
     &      time_ended-time_started, &
     &      ' seconds.'

         write( 6, '(a)' ) &
     &      'AENETAENETAENETAENETAENETAENETAENETAENET' // &
     &      'AENETAENETAENETAENETAENETAENETAENETAEN'

         write( 6, '(a)' )

!     /*   master rank only   */
         call system( &
     &      'rm -f trainingfinished.tmp.ini ' &
     &      // 'trainingstart.tmp.ini 2> /dev/null' )
      end if

#ifdef aenet_pytorch
      endif
#endif

      return
      end


#endif
#endif

