!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    May 21, 2020 by M. Shiga
!      Description:     energy and force from angular dependent method
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine force_adp_setup_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   au_length, au_charge, au_energy, ikind, nkind, natom, iounit, &
     &   myrank

      use mm_variables, only : &
     &   xref_eam, yref_eam, y2ref_eam, srho_eam, dfdrho_eam, rcut_eam, &
     &   ax_adp, ay_adp, az_adp, bxx_adp, bxy_adp, bxz_adp, byy_adp, &
     &   byz_adp, bzz_adp, neam, nref_eam, ikind_eam, iphir_eam, &
     &   irhor_eam, ifrho_eam, iur_adp, iwr_adp

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, ierr, nrhor_eam, nfrho_eam, nphir_eam, &
     &           nur_adp, nwr_adp

      real(8) :: factor

!-----------------------------------------------------------------------
!     /*   number of tables                                           */
!-----------------------------------------------------------------------

!     /*   eam   */
      nrhor_eam = nkind
      nfrho_eam = nkind
      nphir_eam = nkind*(nkind+1)/2

!     /*   adp   */
      nur_adp   = nkind*(nkind+1)/2
      nwr_adp   = nkind*(nkind+1)/2

!     /*   neam = total number of eam and adp reference tables   */
      neam = nrhor_eam + nfrho_eam + nphir_eam + nur_adp + nwr_adp

!-----------------------------------------------------------------------
!     /*   read eam and adp data                                      */
!-----------------------------------------------------------------------

!     /*   master rank   */
      if ( myrank .eq. 0 ) then

!        /*   open file   */
         open (iounit,file = 'eam.dat')

!        /*   tag   */
         call search_tag ( '<nref_eam>', 10, iounit, ierr )

!        /*   number of reference data points in the table   */
         if ( neam .ne. 0 ) read( iounit, *, iostat=ierr ) nref_eam

!        /*   tag   */
         call search_tag ( '<rcut_eam>', 10, iounit, ierr )

!        /*   cut off distance   */
         if ( neam .ne. 0 ) read( iounit, *, iostat=ierr ) rcut_eam

!        /*   angstrom --> bohr   */
         rcut_eam = rcut_eam / au_length * 1.d-10

!        /*   close file   */
         close(iounit)

!        /*   on error, no eam   */
         if ( ierr .ne. 0 ) neam = 0

!     /*   master rank   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   error handling   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_adp_setup_MPI', 30 )

!     /*   communicate   */
      call my_mpi_bcast_int_0( neam )
      call my_mpi_bcast_int_0( nref_eam )
      call my_mpi_bcast_real_0( rcut_eam )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   species number   */
      if ( .not. allocated( ikind_eam ) ) &
     &   allocate( ikind_eam(natom) )

!     /*   sum of electron density rho_eam   */
      if ( .not. allocated( srho_eam ) ) &
     &   allocate( srho_eam(natom) )

!     /*   gradient of embedding potential   */
      if ( .not. allocated( dfdrho_eam ) ) &
     &   allocate( dfdrho_eam(natom) )

!     /*   table number of rho-r, f-rho, phi-r   */
      if ( .not. allocated( irhor_eam ) ) &
     &   allocate( irhor_eam(nkind) )
      if ( .not. allocated( ifrho_eam ) ) &
     &   allocate( ifrho_eam(nkind) )
      if ( .not. allocated( iphir_eam ) ) &
     &   allocate( iphir_eam(nkind,nkind) )

!     /*   table number of u-r, w-r   */
      if ( .not. allocated( iur_adp ) ) &
     &   allocate( iur_adp(nkind,nkind) )
      if ( .not. allocated( iwr_adp ) ) &
     &   allocate( iwr_adp(nkind,nkind) )

!     /*   table rho-r, f-rho, phi-r, u-r, w-r   */
      if ( .not. allocated( xref_eam ) ) &
     &   allocate( xref_eam(nref_eam,neam) )
      if ( .not. allocated( yref_eam ) ) &
     &   allocate( yref_eam(nref_eam,neam) )

!     /*   spline dimensions for rho-r, f-rho, phi-r, u-r, w-r   */
      if ( .not. allocated( y2ref_eam ) ) &
     &   allocate( y2ref_eam(nref_eam,neam) )

!     /*   adp dipole functions  */
      if ( .not. allocated(ax_adp) ) allocate( ax_adp(natom) )
      if ( .not. allocated(ay_adp) ) allocate( ay_adp(natom) )
      if ( .not. allocated(az_adp) ) allocate( az_adp(natom) )

!     /*   adp quadrupole functions  */
      if ( .not. allocated(bxx_adp) ) allocate( bxx_adp(natom) )
      if ( .not. allocated(bxy_adp) ) allocate( bxy_adp(natom) )
      if ( .not. allocated(bxz_adp) ) allocate( bxz_adp(natom) )
      if ( .not. allocated(byy_adp) ) allocate( byy_adp(natom) )
      if ( .not. allocated(byz_adp) ) allocate( byz_adp(natom) )
      if ( .not. allocated(bzz_adp) ) allocate( bzz_adp(natom) )

!-----------------------------------------------------------------------
!     /*   make species number                                        */
!-----------------------------------------------------------------------

      ikind_eam(:) = ikind(:)

!-----------------------------------------------------------------------
!     /*   read eam table:  r[angstrom] - rho                         */
!-----------------------------------------------------------------------

!     /*   master rank   */
      if ( myrank .eq. 0 ) then

!        /*   open file   */
         open (iounit,file = 'eam.dat')

!        /*   tag   */
         call search_tag ( '<rhor_eam>', 10, iounit, ierr )

!        /*   loop of table   */
         do i = 1, nrhor_eam

!           /*   read species for the table   */
            read( iounit, *, iostat=ierr ) k

!           /*   read table for species k   */
            do j = 1, nref_eam
               read( iounit, *, iostat=ierr ) &
     &            xref_eam(j,i), yref_eam(j,i)
            end do

!           /*   check error   */
            if ( ierr .ne. 0 ) exit

!           /*   i-th table is for species k   */
            irhor_eam(k) = i

!           /*   angstrom --> bohr   */
            do j = 1, nref_eam
               xref_eam(j,i) = xref_eam(j,i) / au_length * 1.d-10
            end do

!        /*   loop of table   */
         end do

!        /*   close file   */
         close(iounit)

!     /*   master rank   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   error handling   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_adp_setup_MPI', 30 )

!-----------------------------------------------------------------------
!     /*   read eam table:  rho - f[electron volt]                    */
!-----------------------------------------------------------------------

!     /*   master rank   */
      if ( myrank .eq. 0 ) then

!        /*   open file   */
         open (iounit,file = 'eam.dat')

!        /*   tag   */
         call search_tag ( '<frho_eam>', 10, iounit, ierr )

!        /*   loop of table   */
         do i = nrhor_eam+1, nrhor_eam+nfrho_eam

!           /*   read species for the table   */
            read( iounit, *, iostat=ierr ) k

!           /*   read table for species k   */
            do j = 1, nref_eam
               read( iounit, *, iostat=ierr ) &
     &            xref_eam(j,i), yref_eam(j,i)
            end do

!           /*   check error   */
            if ( ierr .ne. 0 ) exit

!           /*   electron volt --> hartree   */
            do j = 1, nref_eam
               yref_eam(j,i) = yref_eam(j,i)
               yref_eam(j,i) = yref_eam(j,i) * au_charge / au_energy
            end do

!           /*   i-th table is for species k   */
            ifrho_eam(k) = i

!        /*   loop of table   */
         end do

!        /*   close file   */
         close(iounit)

!     /*   master rank   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   error handling   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_adp_setup_MPI', 30 )

!-----------------------------------------------------------------------
!     /*   read eam table:  r[angstrom] - phi[electron volt]          */
!-----------------------------------------------------------------------

!     /*   master rank   */
      if ( myrank .eq. 0 ) then

!        /*   open file   */
         open (iounit,file = 'eam.dat')

!        /*   tag   */
         call search_tag ( '<phir_eam>', 10, iounit, ierr )

!        /*   loop of table   */
         do i = nrhor_eam+nfrho_eam+1, nrhor_eam+nfrho_eam+nphir_eam

!           /*   read species for the table   */
            read( iounit, *, iostat=ierr ) k, l

!           /*   read table for species k-l   */
            do j = 1, nref_eam
               read( iounit, *, iostat=ierr ) &
     &            xref_eam(j,i), yref_eam(j,i)
            end do

!           /*   check error   */
            if ( ierr .ne. 0 ) exit

!           /*   angstrom --> bohr,  electron volt --> hartree   */
            do j = 1, nref_eam
               xref_eam(j,i) = xref_eam(j,i) / au_length * 1.d-10
               yref_eam(j,i) = yref_eam(j,i) * au_charge / au_energy
            end do

!           /*   i-th table is for species pair k-l   */
            iphir_eam(k,l) = i
            iphir_eam(l,k) = i

!        /*   loop of table   */
         end do

!        /*   close file   */
         close(iounit)

!     /*   master rank   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   error handling   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_adp_setup_MPI', 30 )

!-----------------------------------------------------------------------
!     /*   read adp table:  r[angstrom] - u[electron volt]            */
!-----------------------------------------------------------------------

!     /*   master rank   */
      if ( myrank .eq. 0 ) then

!        /*   open file   */
         open (iounit,file = 'eam.dat')

!        /*   tag   */
         call search_tag ( '<ur_adp>', 8, iounit, ierr )

!        /*   starting position   */
         m = nrhor_eam + nfrho_eam + nphir_eam

!        /*   loop of table   */
         do i = m+1, m+nur_adp

!           /*   read species for the table   */
            read( iounit, *, iostat=ierr ) k, l

!           /*   read table for species k-l   */
            do j = 1, nref_eam
               read( iounit, *, iostat=ierr ) &
     &            xref_eam(j,i), yref_eam(j,i)
            end do

!           /*   check error   */
            if ( ierr .ne. 0 ) exit

!           /*   conversion factor   */
            factor = sqrt( au_charge / au_energy ) * au_length / 1.d-10

!           /*   angs -> bohr, sqrt(eV)/angs -> sqrt(hartree)/bohr   */
            do j = 1, nref_eam
               xref_eam(j,i) = xref_eam(j,i) / au_length * 1.d-10
               yref_eam(j,i) = yref_eam(j,i) * factor
            end do

!           /*   i-th table is for species pair k-l   */
            iur_adp(k,l) = i
            iur_adp(l,k) = i

!        /*   loop of table   */
         end do

!        /*   close file   */
         close(iounit)

!     /*   master rank   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   error handling   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_adp_setup_MPI', 30 )

!-----------------------------------------------------------------------
!     /*   read adp table:  r[angstrom] - w[electron volt]            */
!-----------------------------------------------------------------------

!     /*   master rank   */
      if ( myrank .eq. 0 ) then

!        /*   open file   */
         open (iounit,file = 'eam.dat')

!        /*   tag   */
         call search_tag ( '<wr_adp>', 8, iounit, ierr )

!        /*   starting position   */
         m = nrhor_eam + nfrho_eam + nphir_eam + nur_adp

!        /*   loop of table   */
         do i = m+1, m+nwr_adp

!           /*   read species for the table   */
            read( iounit, *, iostat=ierr ) k, l

!           /*   read table for species k-l   */
            do j = 1, nref_eam
               read( iounit, *, iostat=ierr ) &
     &            xref_eam(j,i), yref_eam(j,i)
            end do

!           /*   check error   */
            if ( ierr .ne. 0 ) exit

!           /*   conversion factor   */
            factor = sqrt( au_charge / au_energy ) * au_length / 1.d-10

!           /*   angs -> bohr, sqrt(eV)/angs -> sqrt(hartree)/bohr   */
            do j = 1, nref_eam
               xref_eam(j,i) = xref_eam(j,i) / au_length * 1.d-10
               yref_eam(j,i) = yref_eam(j,i) * factor
            end do

!           /*   i-th table is for species pair k-l   */
            iwr_adp(k,l) = i
            iwr_adp(l,k) = i

!        /*   loop of table   */
         end do

!        /*   close file   */
         close(iounit)

!     /*   master rank   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   error handling   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_adp_setup_MPI', 30 )

!-----------------------------------------------------------------------
!     /*   communication                                              */
!-----------------------------------------------------------------------

      call my_mpi_bcast_real_2( xref_eam, nref_eam, neam )
      call my_mpi_bcast_real_2( yref_eam, nref_eam, neam )
      call my_mpi_bcast_real_2( y2ref_eam, nref_eam, neam )

      call my_mpi_bcast_int_1( irhor_eam, nkind )
      call my_mpi_bcast_int_1( ifrho_eam, nkind )
      call my_mpi_bcast_int_2( iphir_eam, nkind, nkind )
      call my_mpi_bcast_int_2( iur_adp, nkind, nkind )
      call my_mpi_bcast_int_2( iwr_adp, nkind, nkind )

!-----------------------------------------------------------------------
!     /*   initialization of spline                                   */
!-----------------------------------------------------------------------

      do i = 1, neam
         call spline_init_eam &
     &      ( xref_eam(:,i), yref_eam(:,i), y2ref_eam(:,i), nref_eam )
      end do

!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine force_adp_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, vir, pot, au_length, au_charge, volume, &
     &   box, au_energy, natom, nbead, iounit, iboundary, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub

      use mm_variables, only : &
     &   ax_adp, ay_adp, az_adp, bxx_adp, bxy_adp, bxz_adp, byy_adp, &
     &   byz_adp, bzz_adp, rcut_eam, rcut_eam2, bigbox, bigboxinv, &
     &   srho_eam, dfdrho_eam, ikind_eam, neam, nbox_eam, n_list, j_list

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, n, jx, jy, jz, j2

      real(8) :: rij, rij2, rinv, xij, yij, zij, dax, day, daz, u, du, &
     &           a2, w, dw, trbi, trbj, p1, p2, p3, p4, p5, p6, p7, b2, &
     &           onesixth, onethird, fxi, fyi, fzi, ax, ay, az, cij, &
     &           bx, by, bz, cx, cy, cz, absa, absb, absc, aij, bij, &
     &           ur_adp, ur_grad_adp, wr_adp, wr_grad_adp, srho, &
     &           rhor_eam, rhor_grad_eam, frho_eam, frho_grad_eam, &
     &           phir_eam, phir_grad_eam, drhoirdr, drhojrdr, dphirdr, &
     &           dfdrhoi, dfdrhoj

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   for initial access   */
      if ( iset .eq. 0 ) then

!        /*   read eam and adp files   */
         call force_adp_setup_MPI

!        /*   set complete   */
         iset = 1

!     /*   for initial access   */
      end if

!     /*   return if no eam   */
      if ( neam .eq. 0 ) return

!     /*   cut off distance squared   */
      rcut_eam2 = rcut_eam*rcut_eam

!-----------------------------------------------------------------------
!     /*   extention of simulation box in real space sum              */
!-----------------------------------------------------------------------

!     /*   free boundary   */
      if ( iboundary .eq. 0 ) then

!        /*   number of replicated boxes   */
         nbox_eam(1) = 1
         nbox_eam(2) = 1
         nbox_eam(3) = 1

!     /*   periodic boundary   */
      else

!        /*   vector product of lattice vectors b, c   */
         ax = box(2,2)*box(3,3) - box(2,3)*box(3,2)
         ay = box(3,2)*box(1,3) - box(3,3)*box(1,2)
         az = box(1,2)*box(2,3) - box(1,3)*box(2,2)

!        /*   vector product of lattice vectors c, a   */
         bx = box(2,3)*box(3,1) - box(2,1)*box(3,3)
         by = box(3,3)*box(1,1) - box(3,1)*box(1,3)
         bz = box(1,3)*box(2,1) - box(1,1)*box(2,3)

!        /*   vector product of lattice vectors a, b   */
         cx = box(2,1)*box(3,2) - box(2,2)*box(3,1)
         cy = box(3,1)*box(1,2) - box(3,2)*box(1,1)
         cz = box(1,1)*box(2,2) - box(1,2)*box(2,1)

!        /*   distance between parallel planes   */
         absa = volume / sqrt( ax*ax + ay*ay + az*az )
         absb = volume / sqrt( bx*bx + by*by + bz*bz )
         absc = volume / sqrt( cx*cx + cy*cy + cz*cz )

!        /*   number of replicated boxes   */
         nbox_eam(1) = int(2.d0*rcut_eam/absa) + 1
         nbox_eam(2) = int(2.d0*rcut_eam/absb) + 1
         nbox_eam(3) = int(2.d0*rcut_eam/absc) + 1

!     /*   boundary condition   */
      end if

!-----------------------------------------------------------------------
!     /*   constants                                                  */
!-----------------------------------------------------------------------

      onethird = 1.d0 / 3.d0
      onesixth = 1.d0 / 6.d0

!-----------------------------------------------------------------------
!     /*   first loop                                                 */
!-----------------------------------------------------------------------

!     /*   free boundary or minimum image convention   */
      if ( nbox_eam(1)*nbox_eam(2)*nbox_eam(3) .eq. 1 ) then

!        /*   make neighbor list   */
         call force_eam_makelist_MPI

!        /*   loop of beads   */
         do m = 1, nbead

!           /*   bead parallel   */
            if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

!           /*   srho_eam = sum of electron density rho_eam   */
            srho_eam(:) = 0.d0

!           /*   a_adp = sum of density dipoles for adp   */
            ax_adp(:) = 0.d0
            ay_adp(:) = 0.d0
            az_adp(:) = 0.d0

!           /*   b_adp = sum of density quadrupoles for adp   */
            bxx_adp(:) = 0.d0
            bxy_adp(:) = 0.d0
            bxz_adp(:) = 0.d0
            byy_adp(:) = 0.d0
            byz_adp(:) = 0.d0
            bzz_adp(:) = 0.d0

!           /*   loop of atom pairs   */
            do i = 1, natom

!           /*   force parallel   */
            if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!           /*   loop of atom pairs   */
            do n = 1, n_list(i,m)

!              /*   atom in neighbor list   */
               j = j_list(n,i,m)

!              /*   interatomic distance   */
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              /*   apply free or periodic boundary   */
               call pbc_atom_MPI ( xij, yij, zij )

!              /*   interatomic distance squared   */
               rij2 = xij*xij + yij*yij + zij*zij

!              /*   neglect beyond cutoff distance   */
               if ( rij2 .gt. rcut_eam2 ) cycle

!              /*   interatomic distance   */
               rij = sqrt( rij2 )

!              /*   ikind_eam = species number   */
               k = ikind_eam(i)
               l = ikind_eam(j)

!              /*   sum of electron density   */
               srho_eam(i) = srho_eam(i) + rhor_eam(rij,l)
               srho_eam(j) = srho_eam(j) + rhor_eam(rij,k)

!              /*   dipole function   */
               u = ur_adp(rij,k,l)

!              /*   sum of dipoles   */
               ax_adp(i) = ax_adp(i) + u * xij
               ay_adp(i) = ay_adp(i) + u * yij
               az_adp(i) = az_adp(i) + u * zij
               ax_adp(j) = ax_adp(j) - u * xij
               ay_adp(j) = ay_adp(j) - u * yij
               az_adp(j) = az_adp(j) - u * zij

!              /*   quadrupole function   */
               w = wr_adp(rij,k,l)

!              /*   sum of quadrupoles   */
               bxx_adp(i) = bxx_adp(i) + w * xij * xij
               bxy_adp(i) = bxy_adp(i) + w * xij * yij
               bxz_adp(i) = bxz_adp(i) + w * xij * zij
               byy_adp(i) = byy_adp(i) + w * yij * yij
               byz_adp(i) = byz_adp(i) + w * yij * zij
               bzz_adp(i) = bzz_adp(i) + w * zij * zij
               bxx_adp(j) = bxx_adp(j) + w * xij * xij
               bxy_adp(j) = bxy_adp(j) + w * xij * yij
               bxz_adp(j) = bxz_adp(j) + w * xij * zij
               byy_adp(j) = byy_adp(j) + w * yij * yij
               byz_adp(j) = byz_adp(j) + w * yij * zij
               bzz_adp(j) = bzz_adp(j) + w * zij * zij

!           /*   loop of atom pairs   */
            end do

!           /*   loop of atom pairs   */
            end do

!           /*   communication   */
            call my_mpi_allreduce_real_1_sub( srho_eam, natom )
            call my_mpi_allreduce_real_1_sub( ax_adp, natom )
            call my_mpi_allreduce_real_1_sub( ay_adp, natom )
            call my_mpi_allreduce_real_1_sub( az_adp, natom )
            call my_mpi_allreduce_real_1_sub( bxx_adp, natom )
            call my_mpi_allreduce_real_1_sub( bxy_adp, natom )
            call my_mpi_allreduce_real_1_sub( bxz_adp, natom )
            call my_mpi_allreduce_real_1_sub( byy_adp, natom )
            call my_mpi_allreduce_real_1_sub( byz_adp, natom )
            call my_mpi_allreduce_real_1_sub( bzz_adp, natom )

!-----------------------------------------------------------------------
!           /*   second loop                                          */
!-----------------------------------------------------------------------

!           /*   loop of atoms   */
            do i = 1, natom

!              /*   initialize   */
               dfdrho_eam(i) = 0.d0

!              /*   force parallel   */
               if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!              /*   sum of electron density   */
               srho = srho_eam(i)

!              /*   species number   */
               k    = ikind_eam(i)

!-----------------------------------------------------------------------
!              /*   eam embedding potential   */
!-----------------------------------------------------------------------

!              /*   embedding potential   */
               pot(m)  =  pot(m) + frho_eam(srho,k)

!              /*   gradient of embedding potential   */
               dfdrho_eam(i) =  frho_grad_eam(srho,k)

!-----------------------------------------------------------------------
!              /*   adp part   */
!-----------------------------------------------------------------------

!              /*   norm of dipole vector   */
               a2 = ax_adp(i) * ax_adp(i) &
     &            + ay_adp(i) * ay_adp(i) &
     &            + az_adp(i) * az_adp(i)

!              /*   sum of quadrupole matrix   */
               b2 = bxx_adp(i) * bxx_adp(i) &
     &            + bxy_adp(i) * bxy_adp(i) * 2.d0 &
     &            + bxz_adp(i) * bxz_adp(i) * 2.d0 &
     &            + byy_adp(i) * byy_adp(i) &
     &            + byz_adp(i) * byz_adp(i) * 2.d0 &
     &            + bzz_adp(i) * bzz_adp(i)

!              /*   trace of quadrupole matrix   */
               trbi = bxx_adp(i) + byy_adp(i) + bzz_adp(i)

!-----------------------------------------------------------------------
!              /*   adp potential   */
!-----------------------------------------------------------------------

               pot(m) = pot(m) + 0.5d0 * a2 + 0.5d0 * b2 &
     &                         - onesixth * trbi * trbi

!           /*   loop of atoms   */
            end do

!           /*   communication   */
            call my_mpi_allreduce_real_1_sub( dfdrho_eam, natom )

!-----------------------------------------------------------------------
!           /*   third loop                                           */
!-----------------------------------------------------------------------

!           /*   loop of atom pairs   */
            do i = 1, natom

!           /*   force parallel   */
            if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!           /*   loop of atom pairs   */
            do n = 1, n_list(i,m)

!              /*   atom in neighbor list   */
               j = j_list(n,i,m)

!              /*   interatomic distance   */
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              /*   apply free or periodic boundary   */
               call pbc_atom_MPI ( xij, yij, zij )

!              /*   interatomic distance squared   */
               rij2 = xij*xij + yij*yij + zij*zij

!              /*   neglect beyond cutoff distance   */
               if ( rij2 .gt. rcut_eam2 ) cycle

!              /*   interatomic distance   */
               rij = sqrt( rij2 )

!              /*   inverse of interatomic distance   */
               rinv = 1.d0/rij

!              /*   species number   */
               k = ikind_eam(i)
               l = ikind_eam(j)

!              /*   electron density   */
               drhoirdr =  rhor_grad_eam(rij,k)
               drhojrdr =  rhor_grad_eam(rij,l)

!              /*   gradient of pair potential   */
               dphirdr  =  phir_grad_eam(rij,k,l)

!              /*   gradient of embedding potential   */
               dfdrhoi  =  dfdrho_eam(i)
               dfdrhoj  =  dfdrho_eam(j)

!-----------------------------------------------------------------------
!              /*   eam pair potential   */
!-----------------------------------------------------------------------

               pot(m)   =  pot(m) + phir_eam(rij,k,l)

!-----------------------------------------------------------------------
!              /*   eam forces   */
!-----------------------------------------------------------------------

               fxi = - dfdrhoi * drhojrdr * xij * rinv &
     &               - dfdrhoj * drhoirdr * xij * rinv &
     &               - dphirdr * xij * rinv

               fyi = - dfdrhoi * drhojrdr * yij * rinv &
     &               - dfdrhoj * drhoirdr * yij * rinv &
     &               - dphirdr * yij * rinv

               fzi = - dfdrhoi * drhojrdr * zij * rinv &
     &               - dfdrhoj * drhoirdr * zij * rinv &
     &               - dphirdr * zij * rinv

!-----------------------------------------------------------------------
!              /*   adp part   */
!-----------------------------------------------------------------------

               dax = ax_adp(i) - ax_adp(j)
               day = ay_adp(i) - ay_adp(j)
               daz = az_adp(i) - az_adp(j)

               trbi = bxx_adp(i) + byy_adp(i) + bzz_adp(i)
               trbj = bxx_adp(j) + byy_adp(j) + bzz_adp(j)

               u  = ur_adp(rij,k,l)
               du = ur_grad_adp(rij,k,l)

               w  = wr_adp(rij,k,l)
               dw = wr_grad_adp(rij,k,l)

               p1 = du * rinv * ( dax*xij + day*yij + daz*zij )

               p2 = dw * rinv

               p3 = ( bxx_adp(i) + bxx_adp(j) ) * xij * xij &
     &            + ( bxy_adp(i) + bxy_adp(j) ) * xij * yij * 2.d0 &
     &            + ( bxz_adp(i) + bxz_adp(j) ) * xij * zij * 2.d0 &
     &            + ( byy_adp(i) + byy_adp(j) ) * yij * yij &
     &            + ( byz_adp(i) + byz_adp(j) ) * yij * zij * 2.d0 &
     &            + ( bzz_adp(i) + bzz_adp(j) ) * zij * zij

               p4 = - 2.d0 * w * ( bxx_adp(i) + bxx_adp(j) ) * xij &
     &              - 2.d0 * w * ( bxy_adp(i) + bxy_adp(j) ) * yij &
     &              - 2.d0 * w * ( bxz_adp(i) + bxz_adp(j) ) * zij

               p5 = - 2.d0 * w * ( bxy_adp(i) + bxy_adp(j) ) * xij &
     &              - 2.d0 * w * ( byy_adp(i) + byy_adp(j) ) * yij &
     &              - 2.d0 * w * ( byz_adp(i) + byz_adp(j) ) * zij

               p6 = - 2.d0 * w * ( bxz_adp(i) + bxz_adp(j) ) * xij &
     &              - 2.d0 * w * ( byz_adp(i) + byz_adp(j) ) * yij &
     &              - 2.d0 * w * ( bzz_adp(i) + bzz_adp(j) ) * zij

               p7 = onethird * ( trbi + trbj ) * ( dw*rij + 2.d0*w )

!-----------------------------------------------------------------------
!              /*   adp forces   */
!-----------------------------------------------------------------------

               fxi = fxi - p1 * xij - u * dax
               fyi = fyi - p1 * yij - u * day
               fzi = fzi - p1 * zij - u * daz

               fxi = fxi - p2 * p3 * xij + p4
               fyi = fyi - p2 * p3 * yij + p5
               fzi = fzi - p2 * p3 * zij + p6

               fxi = fxi + p7 * xij
               fyi = fyi + p7 * yij
               fzi = fzi + p7 * zij

!-----------------------------------------------------------------------
!              /*   total force and virial   */
!-----------------------------------------------------------------------

               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

               fx(j,m) = fx(j,m) - fxi
               fy(j,m) = fy(j,m) - fyi
               fz(j,m) = fz(j,m) - fzi

               vir(1,1) = vir(1,1) + fxi*xij
               vir(1,2) = vir(1,2) + fxi*yij
               vir(1,3) = vir(1,3) + fxi*zij
               vir(2,1) = vir(2,1) + fyi*xij
               vir(2,2) = vir(2,2) + fyi*yij
               vir(2,3) = vir(2,3) + fyi*zij
               vir(3,1) = vir(3,1) + fzi*xij
               vir(3,2) = vir(3,2) + fzi*yij
               vir(3,3) = vir(3,3) + fzi*zij

!           /*   loop of atom pairs   */
            end do
            end do

!        /*   loop of beads   */
         end do

!-----------------------------------------------------------------------
!     /*   first loop                                                 */
!-----------------------------------------------------------------------

!     /*   periodic boundary   */
      else

!        /*   replicated boxes   */
         bigbox(:,1) = dble(nbox_eam(1))*box(:,1)
         bigbox(:,2) = dble(nbox_eam(2))*box(:,2)
         bigbox(:,3) = dble(nbox_eam(3))*box(:,3)

!        /*   inverse of box matrix   */
         call inv3 ( bigbox, bigboxinv )

!        /*   loop of beads   */
         do m = 1, nbead

!           /*   bead parallel   */
            if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

!           /*   srho_eam = sum of electron density rho_eam   */
            srho_eam(:) = 0.d0

!           /*   a_adp = sum of density dipoles for adp   */
            ax_adp(:) = 0.d0
            ay_adp(:) = 0.d0
            az_adp(:) = 0.d0

!           /*   b_adp = sum of density quadrupoles for adp   */
            bxx_adp(:) = 0.d0
            bxy_adp(:) = 0.d0
            bxz_adp(:) = 0.d0
            byy_adp(:) = 0.d0
            byz_adp(:) = 0.d0
            bzz_adp(:) = 0.d0

!           /*   loop of atom pairs   */
            do i = 1, natom

!           /*   force parallel   */
            if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!           /*   loop of atom pairs   */
            do j = 1, natom

!           /*   loop of replicated boxes   */
            do jx = 0, nbox_eam(1)-1
            do jy = 0, nbox_eam(2)-1
            do jz = 0, nbox_eam(3)-1

!              /*   square of box index   */
               j2 = jx*jx + jy*jy + jz*jz

!              /*   skip same atom   */
               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

!              /*   interatomic distance of i and j in same box   */
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              /*   distance of i and j in different box  */
               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

!              /*   vector in big box   */
               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

!              /*   apply periodic boundary in big box   */
               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

!              /*   distance of nearest i and j   */
               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

!              /*   interatomic distance squared   */
               rij2 = xij*xij + yij*yij + zij*zij

!              /*   neglect beyond cutoff distance   */
               if ( rij2 .gt. rcut_eam2 ) cycle

!              /*   interatomic distance   */
               rij = sqrt( rij2 )

!              /*   ikind_eam = species number   */
               k = ikind_eam(i)
               l = ikind_eam(j)

!              /*   sum of electron density   */
               srho_eam(i) = srho_eam(i) + rhor_eam(rij,l)

!              /*   dipole function   */
               u = ur_adp(rij,k,l)

!              /*   sum of dipoles   */
               ax_adp(i) = ax_adp(i) + u * xij
               ay_adp(i) = ay_adp(i) + u * yij
               az_adp(i) = az_adp(i) + u * zij

!              /*   quadrupole function   */
               w = wr_adp(rij,k,l)

!              /*   sum of quadrupoles   */
               bxx_adp(i) = bxx_adp(i) + w * xij * xij
               bxy_adp(i) = bxy_adp(i) + w * xij * yij
               bxz_adp(i) = bxz_adp(i) + w * xij * zij
               byy_adp(i) = byy_adp(i) + w * yij * yij
               byz_adp(i) = byz_adp(i) + w * yij * zij
               bzz_adp(i) = bzz_adp(i) + w * zij * zij

!           /*   loop of replicated boxes   */
            end do
            end do
            end do

!           /*   loop of atom pairs   */
            end do

!           /*   loop of atom pairs   */
            end do

!           /*   communication   */
            call my_mpi_allreduce_real_1_sub( srho_eam, natom )
            call my_mpi_allreduce_real_1_sub( ax_adp, natom )
            call my_mpi_allreduce_real_1_sub( ay_adp, natom )
            call my_mpi_allreduce_real_1_sub( az_adp, natom )
            call my_mpi_allreduce_real_1_sub( bxx_adp, natom )
            call my_mpi_allreduce_real_1_sub( bxy_adp, natom )
            call my_mpi_allreduce_real_1_sub( bxz_adp, natom )
            call my_mpi_allreduce_real_1_sub( byy_adp, natom )
            call my_mpi_allreduce_real_1_sub( byz_adp, natom )
            call my_mpi_allreduce_real_1_sub( bzz_adp, natom )

!-----------------------------------------------------------------------
!           /*   second loop                                          */
!-----------------------------------------------------------------------

!           /*   loop of atoms   */
            do i = 1, natom

!              /*   initialize   */
               dfdrho_eam(i) = 0.d0

!              /*   force parallel   */
               if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!              /*   sum of electron density   */
               srho = srho_eam(i)

!              /*   species number   */
               k    = ikind_eam(i)

!-----------------------------------------------------------------------
!              /*   eam embedding potential   */
!-----------------------------------------------------------------------

!              /*   embedding potential   */
               pot(m)  =  pot(m) + frho_eam(srho,k)

!              /*   gradient of embedding potential   */
               dfdrho_eam(i) =  frho_grad_eam(srho,k)

!-----------------------------------------------------------------------
!              /*   adp part   */
!-----------------------------------------------------------------------

!              /*   norm of dipole vector   */
               a2 = ax_adp(i) * ax_adp(i) &
     &            + ay_adp(i) * ay_adp(i) &
     &            + az_adp(i) * az_adp(i)

!              /*   sum of quadrupole matrix   */
               b2 = bxx_adp(i) * bxx_adp(i) &
     &            + bxy_adp(i) * bxy_adp(i) * 2.d0 &
     &            + bxz_adp(i) * bxz_adp(i) * 2.d0 &
     &            + byy_adp(i) * byy_adp(i) &
     &            + byz_adp(i) * byz_adp(i) * 2.d0 &
     &            + bzz_adp(i) * bzz_adp(i)

!              /*   trace of quadrupole matrix   */
               trbi = bxx_adp(i) + byy_adp(i) + bzz_adp(i)

!-----------------------------------------------------------------------
!              /*   adp potential   */
!-----------------------------------------------------------------------

               pot(m) = pot(m) + 0.5d0*a2 + 0.5d0*b2 &
     &                         - onesixth * trbi * trbi

!           /*   loop of atoms   */
            end do

!           /*   communication   */
            call my_mpi_allreduce_real_1_sub( dfdrho_eam, natom )

!-----------------------------------------------------------------------
!           /*   third loop                                           */
!-----------------------------------------------------------------------

!           /*   loop of atom pairs   */
            do i = 1, natom

!           /*   force parallel   */
            if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!           /*   loop of atom pairs   */
            do j = 1, natom

!           /*   loop of replicated boxes   */
            do jx = 0, nbox_eam(1)-1
            do jy = 0, nbox_eam(2)-1
            do jz = 0, nbox_eam(3)-1

!              /*   square of box index   */
               j2 = jx*jx + jy*jy + jz*jz

               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

!              /*   interatomic distance of i and j in same box   */
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              /*   distance of i and j in different box  */
               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

!              /*   vector in big box   */
               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

!              /*   apply periodic boundary in big box   */
               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

!              /*   distance of nearest i and j   */
               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

!              /*    interatomic distance squared   */
               rij2 = xij*xij + yij*yij + zij*zij

!              /*   neglect beyond cutoff distance   */
               if ( rij2 .gt. rcut_eam2 ) cycle

!              /*   interatomic distance   */
               rij = sqrt( rij2 )

!              /*   inverse of interatomic distance   */
               rinv = 1.d0/rij

!              /*   species number   */
               k = ikind_eam(i)
               l = ikind_eam(j)

!              /*   electron density   */
               drhoirdr =  rhor_grad_eam(rij,k)
               drhojrdr =  rhor_grad_eam(rij,l)

!              /*   gradient of pair potential   */
               dphirdr  =  phir_grad_eam(rij,k,l)

!              /*   gradient of embedding potential   */
               dfdrhoi  =  dfdrho_eam(i)
               dfdrhoj  =  dfdrho_eam(j)

!-----------------------------------------------------------------------
!              /*   pair potential   */
!-----------------------------------------------------------------------

               pot(m)   =  pot(m) + 0.5d0*phir_eam(rij,k,l)

!-----------------------------------------------------------------------
!              /*   eam forces   */
!-----------------------------------------------------------------------

               fxi = - dfdrhoi * drhojrdr * xij * rinv &
     &               - dfdrhoj * drhoirdr * xij * rinv &
     &               - dphirdr * xij * rinv

               fyi = - dfdrhoi * drhojrdr * yij * rinv &
     &               - dfdrhoj * drhoirdr * yij * rinv &
     &               - dphirdr * yij * rinv

               fzi = - dfdrhoi * drhojrdr * zij * rinv &
     &               - dfdrhoj * drhoirdr * zij * rinv &
     &               - dphirdr * zij * rinv

!-----------------------------------------------------------------------
!              /*   adp part   */
!-----------------------------------------------------------------------

               dax = ax_adp(i) - ax_adp(j)
               day = ay_adp(i) - ay_adp(j)
               daz = az_adp(i) - az_adp(j)

               trbi = bxx_adp(i) + byy_adp(i) + bzz_adp(i)
               trbj = bxx_adp(j) + byy_adp(j) + bzz_adp(j)

               k = ikind_eam(i)
               l = ikind_eam(j)

               u  = ur_adp(rij,k,l)
               du = ur_grad_adp(rij,k,l)

               w  = wr_adp(rij,k,l)
               dw = wr_grad_adp(rij,k,l)

               p1 = du * rinv * ( dax*xij + day*yij + daz*zij )

               p2 = dw * rinv

               p3 = ( bxx_adp(i) + bxx_adp(j) ) * xij * xij &
     &            + ( bxy_adp(i) + bxy_adp(j) ) * xij * yij * 2.d0 &
     &            + ( bxz_adp(i) + bxz_adp(j) ) * xij * zij * 2.d0 &
     &            + ( byy_adp(i) + byy_adp(j) ) * yij * yij &
     &            + ( byz_adp(i) + byz_adp(j) ) * yij * zij * 2.d0 &
     &            + ( bzz_adp(i) + bzz_adp(j) ) * zij * zij

               p4 = - 2.d0 * w * ( bxx_adp(i) + bxx_adp(j) ) * xij &
     &              - 2.d0 * w * ( bxy_adp(i) + bxy_adp(j) ) * yij &
     &              - 2.d0 * w * ( bxz_adp(i) + bxz_adp(j) ) * zij

               p5 = - 2.d0 * w * ( bxy_adp(i) + bxy_adp(j) ) * xij &
     &              - 2.d0 * w * ( byy_adp(i) + byy_adp(j) ) * yij &
     &              - 2.d0 * w * ( byz_adp(i) + byz_adp(j) ) * zij

               p6 = - 2.d0 * w * ( bxz_adp(i) + bxz_adp(j) ) * xij &
     &              - 2.d0 * w * ( byz_adp(i) + byz_adp(j) ) * yij &
     &              - 2.d0 * w * ( bzz_adp(i) + bzz_adp(j) ) * zij

               p7 = onethird * ( trbi + trbj ) * ( dw*rij + 2.d0*w )

!-----------------------------------------------------------------------
!              /*   adp forces   */
!-----------------------------------------------------------------------

               fxi = fxi - p1 * xij - u * dax
               fyi = fyi - p1 * yij - u * day
               fzi = fzi - p1 * zij - u * daz

               fxi = fxi - p2 * p3 * xij + p4
               fyi = fyi - p2 * p3 * yij + p5
               fzi = fzi - p2 * p3 * zij + p6

               fxi = fxi + p7 * xij
               fyi = fyi + p7 * yij
               fzi = fzi + p7 * zij

!-----------------------------------------------------------------------
!              /*   total force and virial   */
!-----------------------------------------------------------------------

               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

               vir(1,1) = vir(1,1) + 0.5d0*fxi*xij
               vir(1,2) = vir(1,2) + 0.5d0*fxi*yij
               vir(1,3) = vir(1,3) + 0.5d0*fxi*zij
               vir(2,1) = vir(2,1) + 0.5d0*fyi*xij
               vir(2,2) = vir(2,2) + 0.5d0*fyi*yij
               vir(2,3) = vir(2,3) + 0.5d0*fyi*zij
               vir(3,1) = vir(3,1) + 0.5d0*fzi*xij
               vir(3,2) = vir(3,2) + 0.5d0*fzi*yij
               vir(3,3) = vir(3,3) + 0.5d0*fzi*zij

!           /*   loop of replicated boxes   */
            end do
            end do
            end do

!           /*   loop of atom pairs   */
            end do

!           /*   loop of atom pairs   */
            end do

!        /*   loop of beads   */
         end do

!     /*   boundary condition   */
      end if

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz, natom, nbead )

!     /*   virial   */
      call my_mpi_allreduce_real_2 ( vir, 3, 3 )

      return
      end





!***********************************************************************
      real(8) function ur_adp( r, k, l )
!***********************************************************************

      use mm_variables, only : &
     &   xref_eam, yref_eam, y2ref_eam, iur_adp, nref_eam

      implicit none

      integer :: i, k, l
      real(8) :: r

!-----------------------------------------------------------------------

!     /*   i-th table is for r-u table for species k and l   */
      i = iur_adp(k,l)

!-----------------------------------------------------------------------

      if ( ( r .lt. xref_eam(1,i) ) .or. &
     &     ( r .gt. xref_eam(nref_eam,i) ) ) then

         ur_adp = 0.d0

      else

         ur_adp = getspline_eam &
     &              ( xref_eam(:,i), yref_eam(:,i), y2ref_eam(:,i), &
     &                nref_eam, r )

      end if

!-----------------------------------------------------------------------

      return
      contains
      include 'getspline_eam_func.F90'
      end





!***********************************************************************
      real(8) function wr_adp( r, k, l )
!***********************************************************************

      use mm_variables, only : &
     &   xref_eam, yref_eam, y2ref_eam, iwr_adp, nref_eam

      implicit none

      integer :: i, k, l
      real(8) :: r

!-----------------------------------------------------------------------

!     /*   i-th table is for r-u table for species k and l   */
      i = iwr_adp(k,l)

!-----------------------------------------------------------------------

      if ( ( r .lt. xref_eam(1,i) ) .or. &
     &     ( r .gt. xref_eam(nref_eam,i) ) ) then

         wr_adp = 0.d0

      else

         wr_adp = getspline_eam &
     &              ( xref_eam(:,i), yref_eam(:,i), y2ref_eam(:,i), &
     &                nref_eam, r )

      end if

!-----------------------------------------------------------------------

      return
      contains
      include 'getspline_eam_func.F90'
      end





!***********************************************************************
      real(8) function ur_grad_adp ( r, k, l )
!***********************************************************************

      use mm_variables, only : &
     &   xref_eam, yref_eam, y2ref_eam, iur_adp, nref_eam

      implicit none

      integer :: i, k, l
      real(8) :: r

!-----------------------------------------------------------------------

!     /*   i-th table is for r-u table for species k   */
      i = iur_adp(k,l)

!-----------------------------------------------------------------------

      if ( ( r .lt. xref_eam(1,i) ) .or. &
     &     ( r .gt. xref_eam(nref_eam,i) ) ) then

         ur_grad_adp = 0.d0

      else

         ur_grad_adp = getspline_grad_eam &
     &                   ( xref_eam(:,i), yref_eam(:,i), y2ref_eam(:,i), &
     &                     nref_eam, r )

      end if

!-----------------------------------------------------------------------

      return
      contains
      include 'getspline_grad_eam_func.F90'
      end





!***********************************************************************
      real(8) function wr_grad_adp ( r, k, l )
!***********************************************************************

      use mm_variables, only : &
     &   xref_eam, yref_eam, y2ref_eam, iwr_adp, nref_eam

      implicit none

      integer :: i, k, l
      real(8) :: r

!-----------------------------------------------------------------------

!     /*   i-th table is for r-w table for species k   */
      i = iwr_adp(k,l)

!-----------------------------------------------------------------------

      if ( ( r .lt. xref_eam(1,i) ) .or. &
     &     ( r .gt. xref_eam(nref_eam,i) ) ) then

         wr_grad_adp = 0.d0

      else

         wr_grad_adp = getspline_grad_eam &
     &                   ( xref_eam(:,i), yref_eam(:,i), y2ref_eam(:,i), &
     &                     nref_eam, r )

      end if

!-----------------------------------------------------------------------

      return
      contains
      include 'getspline_grad_eam_func.F90'
      end
