!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     energy and force from ABINIT-MP calculation
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      module abinitmp_variables
!***********************************************************************

!-----------------------------------------------------------------------
!     general variables
!-----------------------------------------------------------------------

!     /*   current number of MD steps   */
      integer :: pimd_istep

!     /*   current bead   */
      integer :: pimd_ibead

!     /*   ABINIT-MP input file   */
      character(len=1024) :: abinit_mp_input

!     /*   ABINIT-MP output file   */
      character(len=1024) :: abinit_mp_output

!     /*   if 1, all processess generate output file   */
      integer :: abinit_mp_output_all_proc

!     /*   ABINIT-MP output results every n-step   */
      integer :: abinit_mp_output_every_nstep

!     /*   if 1, reuse monomer MO   */
      integer :: abinit_mp_reuse_monomer_mo

!     /*   if 1, reuse diemr MO   */
      integer :: abinit_mp_reuse_dimer_mo

!     /*   if 1, reuse trimer MO   */
      integer :: abinit_mp_reuse_trimer_mo

!-----------------------------------------------------------------------
!     energy, coordinate, and force
!-----------------------------------------------------------------------

!     /*   total energy   */
      real(8) :: total_energy

!     /*   nuclear coordinate   */
      real(8),allocatable :: coord_x(:), coord_y(:), coord_z(:)

!     /*   force   */
      real(8),allocatable :: force_x(:), force_y(:), force_z(:)

!-----------------------------------------------------------------------
!     values for fragments
!-----------------------------------------------------------------------

!     /*   number of fragments   */
      integer :: n_frag

!     /*   fragment structure   */
      type FRAGMENT_TYPE

!       /*   molecular orbital coefficents   */
        real(8), pointer :: MOC(:,:)

!       /*   dipole moment   */
        real(8) :: dipx, dipy, dipz

!       /*   .true. if array is allocated   */
        logical :: flag_allocated = .false.

      end type FRAGMENT_TYPE

!     /*   fragment set structure   */
      type FRAGMENT_SET_TYPE

!       /*   monomer, dimer, and trimer fragments   */
        type(FRAGMENT_TYPE), pointer :: monomers(:)
        type(FRAGMENT_TYPE), pointer :: dimers(:,:)
        type(FRAGMENT_TYPE), pointer :: trimers(:,:,:)

!       /*   .true. if array is allocated   */
        logical :: flag_allocated = .false.

      end type FRAGMENT_SET_TYPE

!     /*   fragment sets for all beads   */
      type(FRAGMENT_SET_TYPE), allocatable, target :: fragment_sets(:)

!     /*   fragment set for current bead   */
      type(FRAGMENT_SET_TYPE), pointer :: fragment_set

!***********************************************************************
      end module abinitmp_variables
!***********************************************************************





!***********************************************************************
      subroutine force_abinit_mp_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pot, fx, fy, fz, vir, myrank, &
     &   nbead, istep, natom, mbox, myrank_main, nprocs_main

      use abinitmp_variables, only : &
     &   coord_x, coord_y, coord_z, total_energy, &
     &   force_x, force_y, force_z, fragment_set, fragment_sets, &
     &   pimd_istep, pimd_ibead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ibead, i, j

      real(8) :: xi, yi, zi

      integer :: ierr = 0

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!     /*   skip if `ibead is not my job'                              */
!-----------------------------------------------------------------------

      if ( mod(ibead-1,nprocs_main) .ne. myrank_main ) cycle

!-----------------------------------------------------------------------
!     /*   prepare for ABINIT-MP execution                            */
!-----------------------------------------------------------------------

!     /*   set istep   */
      pimd_istep = istep

!     /*   set ibead   */
      pimd_ibead = ibead

!     /*   set coordinate   */
      do i = 1, natom
         coord_x(i) = x(i,ibead)
         coord_y(i) = y(i,ibead)
         coord_z(i) = z(i,ibead)
      end do

!     /*   point MO array of current bead   */
      fragment_set => fragment_sets(ibead)

!-----------------------------------------------------------------------
!     /*   ABINIT-MP execution                                        */
!-----------------------------------------------------------------------

      call abinitmp_force(ierr)

!-----------------------------------------------------------------------
!     /*   error termination                                          */
!-----------------------------------------------------------------------

      if( ierr .ne. 0 ) then

         if ( myrank .eq. 0 ) then
            write( 6, '(a)' ) &
     &        'Error - unable to execute ABINIT-MP. See log file.'
            write( 6, '(a)' )
         end if

         call error_handling_MPI &
     &        ( 1, 'subroutine force_abinit_mp_MPI', 30 )

      end if

!-----------------------------------------------------------------------
!     /*   potential energy and forces                                */
!-----------------------------------------------------------------------

!     /*   get total energy   */
      pot(ibead) = total_energy

!     /*   get force   */
      do i = 1, natom
         fx(i,ibead) = force_x(i)
         fy(i,ibead) = force_y(i)
         fz(i,ibead) = force_z(i)
      end do

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1_main ( pot, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2_main ( fx, natom, nbead )
      call my_mpi_allreduce_real_2_main ( fy, natom, nbead )
      call my_mpi_allreduce_real_2_main ( fz, natom, nbead )

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom

         xi = x(i,j)
         yi = y(i,j)
         zi = z(i,j)

         call pbc_unfold_MPI &
     &      ( xi, yi, zi, mbox(1,i,j), mbox(2,i,j), mbox(3,i,j) )

         vir(1,1) = vir(1,1) + fx(i,j)*xi
         vir(1,2) = vir(1,2) + fx(i,j)*yi
         vir(1,3) = vir(1,3) + fx(i,j)*zi
         vir(2,1) = vir(2,1) + fy(i,j)*xi
         vir(2,2) = vir(2,2) + fy(i,j)*yi
         vir(2,3) = vir(2,3) + fy(i,j)*zi
         vir(3,1) = vir(3,1) + fz(i,j)*xi
         vir(3,2) = vir(3,2) + fz(i,j)*yi
         vir(3,3) = vir(3,3) + fz(i,j)*zi

      end do
      end do

      return
      end





!***********************************************************************
      subroutine init_abinit_mp_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : myrank, nbead, iounit

      use abinitmp_variables, only : &
     &   abinit_mp_input, abinit_mp_output,abinit_mp_output_all_proc, &
     &   abinit_mp_output_every_nstep, abinit_mp_reuse_monomer_mo, &
     &   abinit_mp_reuse_dimer_mo, abinit_mp_reuse_trimer_mo, &
     &   fragment_sets

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr = 0

      character(len=1024) :: char_long

!-----------------------------------------------------------------------
!     /*   initialize ABINIT-MP                                       */
!-----------------------------------------------------------------------

!     /*   set parameters   */

      call read_char_MPI ( abinit_mp_input, 1024, &
     &     '<abinit_mp_input>', 17, iounit )

      call read_char_MPI ( abinit_mp_output, 1024, &
     &     '<abinit_mp_output>', 18, iounit )

!c      call read_int1_MPI ( abinit_mp_output_all_proc,
!c     &     '<abinit_mp_output_all_proc>', 27, iounit )
!c
!c      call read_int1_MPI ( abinit_mp_output_every_nstep,
!c     &     '<abinit_mp_output_every_nstep>', 30, iounit )
!c
!c      call read_int1_MPI ( abinit_mp_reuse_monomer_mo,
!c     &     '<abinit_mp_reuse_monomer_mo>', 28, iounit )
!c
!c      call read_int1_MPI ( abinit_mp_reuse_dimer_mo,
!c     &     '<abinit_mp_reuse_dimer_mo>', 26, iounit )
!c
!c      call read_int1_MPI ( abinit_mp_reuse_trimer_mo,
!c     &     '<abinit_mp_reuse_trimer_mo>', 27, iounit )

!     /*   parent process only   */
      if ( myrank .eq. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag ( '<abinit_mp_output>', 18, iounit, ierr )

!        /*   read a line   */
         read ( iounit, *, iostat=ierr ) &
     &      char_long, &
     &      abinit_mp_output_all_proc, &
     &      abinit_mp_output_every_nstep

!        /*   file close   */
         close ( iounit )

!        /*   otherwise, read from default input   */
         if ( ierr .ne. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )

!           /*   search for tag    */
            call search_tag( '<abinit_mp_output>', 18, iounit, ierr )

!           /*   read a line   */
            read ( iounit, *, iostat=ierr ) &
     &         char_long, &
     &         abinit_mp_output_all_proc, &
     &         abinit_mp_output_every_nstep

!           /*   file close   */
            close ( iounit )

!        /*   otherwise, read from default input   */
         end if

!     /*   parent process only   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   error message   */
      if ( ierr .ne. 0 ) then
         if ( myrank .eq. 0 ) then
            write ( 6, '(a)' ) &
     &         'Error - <abinit_mp_output> read incorrectly.'
            write ( 6, '(a)' )
         end if
      end if

!     /*   error termination   */
      call error_handling_MPI &
     &     ( ierr, 'subroutine init_abinit_mp_MPI', 29 )

!     /*   communicate   */
      call my_mpi_bcast_int_0( abinit_mp_output_all_proc )

!     /*   communicate   */
      call my_mpi_bcast_int_0( abinit_mp_output_every_nstep )

!     /*   parent process only   */
      if ( myrank .eq. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag &
     &      ( '<abinit_mp_reuse_mo>', 20, iounit, ierr )

!        /*   read a line   */
         read ( iounit, *, iostat=ierr ) &
     &      abinit_mp_reuse_monomer_mo, &
     &      abinit_mp_reuse_dimer_mo, &
     &      abinit_mp_reuse_trimer_mo

!        /*   file close   */
         close ( iounit )

!        /*   otherwise, read from default input   */
         if ( ierr .ne. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )

!           /*   search for tag    */
            call search_tag &
     &         ( '<abinit_mp_reuse_mo>', 20, iounit, ierr )

!           /*   read a line   */
            read ( iounit, *, iostat=ierr ) &
     &         abinit_mp_reuse_monomer_mo, &
     &         abinit_mp_reuse_dimer_mo, &
     &         abinit_mp_reuse_trimer_mo

!           /*   file close   */
            close ( iounit )

!        /*   otherwise, read from default input   */
         end if

!     /*   parent process only   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   error message   */
      if ( ierr .ne. 0 ) then
         if ( myrank .eq. 0 ) then
            write ( 6, '(a)' ) &
     &         'Error - <abinit_mp_reuse_mo> read incorrectly.'
            write ( 6, '(a)' )
         end if
      end if

!     /*   error termination   */
      call error_handling_MPI &
     &     ( ierr, 'subroutine init_abinit_mp_MPI', 29 )

!     /*   communicate   */
      call my_mpi_bcast_int_0( abinit_mp_reuse_monomer_mo )

!     /*   communicate   */
      call my_mpi_bcast_int_0( abinit_mp_reuse_dimer_mo )

!     /*   communicate   */
      call my_mpi_bcast_int_0( abinit_mp_reuse_trimer_mo )

!-----------------------------------------------------------------------
!     /*   print input parameters                                     */
!-----------------------------------------------------------------------

!     /*   parent process only   */
      if ( myrank .eq. 0 ) then

         write( 6, '(a,a)' ) &
     &      'ABINIT-MP input file: ', trim(abinit_mp_input)

         if      ( abinit_mp_output_all_proc .eq. 0 ) then
            write( 6, '(a,a,a,i0,a)' ) &
     &         'ABINIT-MP log file: ', &
     &         trim(abinit_mp_output), &
     &         ' printed in master process every ', &
     &         abinit_mp_output_every_nstep, ' steps.'
         else if ( abinit_mp_output_all_proc .eq. 1 ) then
            write( 6, '(a,a,a,i0,a)' ) &
     &         'ABINIT-MP log file: ', &
     &         trim(abinit_mp_output), &
     &         ' printed in all processes every ', &
     &         abinit_mp_output_every_nstep, ' steps.'
         else
            ierr = 1
         end if

         if      ( abinit_mp_reuse_monomer_mo .eq. 0 ) then
            write( 6, '(a)' ) &
     &         'ABINIT-MP: monomer molecular orbitals ' // &
     &         'initialized every step.'
         else if ( abinit_mp_reuse_monomer_mo .eq. 1 ) then
            write( 6, '(a)' ) &
     &         'ABINIT-MP: monomer molecular orbitals ' // &
     &         'reused every step.'
         else
            ierr = 1
         end if

         if      ( abinit_mp_reuse_dimer_mo .eq. 0 ) then
            write( 6, '(a)' ) &
     &         'ABINIT-MP: dimer molecular orbitals ' // &
     &         ' initialized every step.'
         else if ( abinit_mp_reuse_dimer_mo .eq. 1 ) then
            write( 6, '(a)' ) &
     &         'ABINIT-MP: dimer molecular orbitals ' // &
     &         ' reused every step.'
         else
            ierr = 1
         end if

         if      ( abinit_mp_reuse_trimer_mo .eq. 0 ) then
            write( 6, '(a)' ) &
     &         'ABINIT-MP: trimer molecular orbitals ' // &
     &         ' initialized every step.'
         else if ( abinit_mp_reuse_trimer_mo .eq. 1 ) then
            write( 6, '(a)' ) &
     &         'ABINIT-MP: trimer molecular orbitals ' // &
     &         ' reused every step.'
         else
            ierr = 1
         end if

         write( 6, '(a)' )

!     /*   parent process only   */
      end if

!-----------------------------------------------------------------------
!     /*   error termination                                          */
!-----------------------------------------------------------------------

      if( ierr .ne. 0 ) then

         if ( myrank .eq. 0 ) then

            write( 6, '(a)' ) &
     &        'Error - ABINIT-MP settings incorrect.'
            write( 6, '(a)' )

         end if

         call error_handling_MPI &
     &        ( 1, 'subroutine init_abinit_mp_MPI', 29 )

      end if

!-----------------------------------------------------------------------
!     /*   allocate array to store fragment MO coefficents            */
!-----------------------------------------------------------------------

      allocate( fragment_sets(nbead) )

!-----------------------------------------------------------------------
!     /*   initialize ABINIT-MP                                       */
!-----------------------------------------------------------------------

      call abinitmp_init(ierr)

!-----------------------------------------------------------------------
!     /*   error termination                                          */
!-----------------------------------------------------------------------

      if( ierr .ne. 0 ) then

         if ( myrank .eq. 0 ) then

            write( 6, '(a)' ) &
     &        'Error - unable to initialize ABINIT-MP. See log file.'
            write( 6, '(a)' )

         end if

         call error_handling_MPI &
     &        ( 1, 'subroutine init_abinit_mp_MPI', 29 )

      end if

      return
      end





!***********************************************************************
      subroutine finalize_abinit_mp_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : nbead, myrank

      use abinitmp_variables, only : &
     &   fragment_set, fragment_sets, n_frag

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr, ibead, ifrag, jfrag, kfrag

!-----------------------------------------------------------------------
!     /*   finalize ABINIT-MP                                       */
!-----------------------------------------------------------------------

!     /*   finalize ABINIT-MP subroutine   */

      ierr = 0

      call abinitmp_finalize(ierr)

      if ( ierr .ne. 0 ) then

         if ( myrank .eq. 0 ) then

            write( 6, '(a)' ) &
     &        'Error - unable to finalize ABINIT-MP. See log file.'
            write( 6, '(a)' )

         end if

         call error_handling_MPI &
     &        ( 1, 'subroutine finalize_abinit_mp_MPI', 33 )

      end if

!     /*   deallocate MOC array   */

      do ibead = 1, nbead

         fragment_set => fragment_sets(ibead)

         if( fragment_set%flag_allocated ) then

!     /*   monomers   */
            do ifrag = 1, n_frag
               if( fragment_set%monomers(ifrag)%flag_allocated ) then
                  deallocate( fragment_set%monomers(ifrag)%MOC )
               end if
            end do
            deallocate( fragment_set%monomers )

!     /*   dimers   */
            do ifrag = 1, n_frag
               do jfrag = 1, n_frag
                  if( fragment_set%dimers(ifrag,jfrag)% &
     &                 flag_allocated ) then
                     deallocate( fragment_set%dimers(ifrag,jfrag)%MOC )
                  end if
               end do
            end do
            deallocate( fragment_set%dimers )

!     /*   trimers   */
            do ifrag = 1, n_frag
               do jfrag = 1, n_frag
                  do kfrag = 1, n_frag
                     if( fragment_set%trimers(ifrag,jfrag,kfrag)% &
     &                    flag_allocated ) then
                        deallocate( fragment_set% &
     &                       trimers(ifrag,jfrag,kfrag)%MOC )
                     end if
                  end do
               end do
            end do
            deallocate( fragment_set%trimers )

         end if

      end do

      deallocate( fragment_sets )

      return
      end





!=======================================================================
!
!     The following dummy subroutines are applied when
!     libabinitmp.a is not available
!
!=======================================================================

#ifndef abinitmp

!***********************************************************************
      subroutine abinitmp_init( ierr )
!***********************************************************************

      use common_variables, only : myrank

      implicit none

      integer, intent(out) :: ierr

      ierr = 1

      if( myrank .eq. 0 ) then

         write( 6, '(a)' ) &
     &      'Error termination - ABINIT-MP is not linked to PIMD.'
         write( 6, '(a)' )
         write( 6, '(a)' ) &
     &      'Recompile pimd.mpi.x with ../lib/libabinitmp.a ' // &
     &      'with the options'
         write( 6, '(a)' )
         write( 6, '(a)' ) &
     &      '  ABINITMP = -Dabinitmp'
         write( 6, '(a)' ) &
     &      '  LIBABINITMP = -L../lib -labinitmp'
         write( 6, '(a)' )

      end if

      call my_mpi_barrier
      call my_mpi_abort

      return
      end

#endif





#ifndef abinitmp

!***********************************************************************
      subroutine abinitmp_force( ierr )
!***********************************************************************

      use common_variables, only : myrank

      implicit none

      integer, intent(out) :: ierr

      ierr = 1

      if( myrank .eq. 0 ) then

         write(6, '(a)') 'Error termination.'
         write(6, '(a)') ''
         write(6, '(a)') 'ipotential=ABINIT-MP is not available because'
         write(6, '(a)') 'ABINIT-MP subroutines were not linked.'
         write(6, '(a)') 'Compile the pimd.mpi.x with the followings.'
         write(6, '(a)') '(You need libabinitmp.a in ../lib)'
         write(6, '(a)') ''
         write(6, '(a)') '  FLAGMP = -Dabinitmp'
         write(6, '(a)') '  LINKMP = -L../lib -labinitmp'
         write(6, '(a)') ''

      end if

      call my_mpi_barrier
      call my_mpi_abort

      return
      end

#endif





#ifndef abinitmp

!***********************************************************************
      subroutine abinitmp_finalize( ierr )
!***********************************************************************

      use common_variables, only : myrank

      implicit none

      integer, intent(out) :: ierr

      ierr = 1

      if( myrank .eq. 0 ) then

         write(6, '(a)') 'Error termination.'
         write(6, '(a)') ''
         write(6, '(a)') &
     &      'ABINIT-MP is not available. To link ABINIT-MP routines,' &
     &     // ' copy libabinitmp.a'
         write(6, '(a)') &
     &      'to ../lib directory, then recompile pimd.mpi.x with the' &
     &      // ' following options'
         write(6, '(a)') &
     &      'included in makefile.'
         write(6, '(a)') ''
         write(6, '(a)') '  FLAGMP = -Dabinitmp'
         write(6, '(a)') '  LINKMP = -L../lib -labinitmp'
         write(6, '(a)') ''

      end if

      call my_mpi_barrier
      call my_mpi_abort

      return
      end

#endif

