!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     energy and force from ABINIT-MP calculation
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine force_abinit_mp5
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, dipx, dipy, dipz, vir, &
     &   au2debye, abinit_mp_exe_command, au_length, &
     &   iounit, iounit_abi, nbead, natom, mbox

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr, ibead, i, j, itest

      real(8) :: xi, yi, zi

      real(8), parameter :: bohr2ang = au_length/1.d-10

      character(len=80):: char_line, char_word(14)

      integer :: flag

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      ierr = 0

      if ( iset .eq. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag &
     &      ( '<abinit_mp_exe_command>', 23, iounit, ierr )

!        /*   read a line   */
         read ( iounit, *, iostat=ierr ) abinit_mp_exe_command

!        /*   file close   */
         close ( iounit )

         if ( ierr .ne. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )

!           /*   search for tag    */
            call search_tag &
     &         ( '<abinit_mp_exe_command>', 23, iounit, ierr )

!           /*   read a line   */
            read ( iounit, *, iostat=ierr ) abinit_mp_exe_command

!           /*   file close   */
            close ( iounit )

         end if

!-----------------------------------------------------------------------
!        /*   confirm abinit-mp command                               */
!-----------------------------------------------------------------------

         call system &
     &      ("echo '0' > test.out")
         call system &
     &      ("sleep 0.1")
         call system &
     &      ("which " // abinit_mp_exe_command // &
     &       " > /dev/null 2>&1 && echo '1' > test.out")

         open ( iounit, file = 'test.out' )

         read ( iounit, * ) itest

         close( iounit )

         if ( itest .eq. 0 ) then

            ierr = 1

            write( 6, '(a)' ) &
     &         'Error - ABINIT-MP command not found: ' // &
     &         trim(abinit_mp_exe_command)
            write( 6, '(a)' )

         else

            ierr = 0

            write( 6, '(a)' ) &
     &         'ABINIT-MP command found: ' // &
     &         trim(abinit_mp_exe_command)
            write( 6, '(a)' )

         end if

         call system('rm -f test.out')

         call error_handling( ierr, 'subroutine force_abinit_mp5', 27 )

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!     /*   make abinit-mp input:  input.ajf                           */
!-----------------------------------------------------------------------

!     /*   open the abinit-mp prototype file   */
      open ( iounit,     file = 'abinit-mp.dat'   )

!     /*   open the abinit-mp input file   */
      open ( iounit_abi, file = 'input.ajf' )

!     /*   do loop end   */
      do

!        /*   read a line   */
         read ( iounit, '(a)', iostat=ierr ) char_line

!        /*   exit at the end of the line   */
         if ( ierr .ne. 0 ) exit

!        /*   write a copy of the line   */
         write( iounit_abi, '(a)' ) trim(char_line)

!        /*   if line matched   */
         if ( char_line(1:4) .eq. '&XYZ' ) then

!           /*   start loop of atoms   */
            do i = 1, natom

!              /*   read a line   */
               read ( iounit, *, iostat=ierr ) &
     &            char_word(1), char_word(2), char_word(3), &
     &            char_word(4), char_word(5), char_word(6), &
     &            char_word(7)

!              /*   bohr to angstrom   */

                xi = x(i,ibead) * bohr2ang
                yi = y(i,ibead) * bohr2ang
                zi = z(i,ibead) * bohr2ang

!              /*   write cartesian coordinates   */

                write( iounit_abi, '(a5,a3,a2,3f22.16,1x,a2)' ) &
     &             char_word(1), char_word(2), char_word(3), &
     &             xi, yi, zi, char_word(7)

!           /*   end loop of atoms   */
            end do

!        /*   if line matched   */
         end if

!     /*   do loop end   */
      end do

!     /*   close file   */
      close( iounit )

!     /*   close file   */
      close( iounit_abi )

!-----------------------------------------------------------------------
!     /*   run abinit-mp                                              */
!-----------------------------------------------------------------------

      call system ( abinit_mp_exe_command // &
     &              ' < input.ajf > output.abi' )

!-----------------------------------------------------------------------
!     /*   read abinit-mp output:  potential                          */
!-----------------------------------------------------------------------

!     /*   open the abinit-mp input file   */
      open ( iounit_abi, file = 'output.abi' )

!     /*   do loop start   */
      do

!        /*   read a line   */
         read ( iounit_abi, '(a)', iostat=ierr ) char_line

!        /*   exit at the end of the line   */
         if ( ierr .ne. 0 ) exit

!        /*   if matched   */
         if ( char_line(9:27) .eq. 'Total      energy =' ) then

!           /*   go back one line   */
            backspace( iounit_abi )

!           /*   read the potential data   */
            read ( iounit_abi, *, iostat=ierr ) &
     &         char_word(1), char_word(2), char_word(3), pot(ibead)

!           /*   exit from the do loop   */
            exit

         end if

!     /*   do loop end   */
      end do

!     /*   close file   */
      close( iounit_abi )

!     /*   print total energy   */
!      write(*,'(f16.8)')  pot(ibead)

!-----------------------------------------------------------------------
!     /*   read abinit-mp output:  potential gradient                 */
!-----------------------------------------------------------------------

!     /*   open the abinit-mp input file   */
      open ( iounit_abi, file = 'output.abi' )

!     /*   zero clear   */

      do i = 1, natom
         fx(i,ibead) = 0.d0
         fy(i,ibead) = 0.d0
         fz(i,ibead) = 0.d0
      end do

!     /*   init flag   */
      flag = 0

!     /*   do loop start   */
      do

!        /*   read a line   */
         read ( iounit_abi, '(a)', iostat=ierr ) char_line

!        /*   exit at the end of the line   */
         if ( ierr .ne. 0 ) exit

!        /*   if matched   */
         if ( char_line(9:16) .eq. 'GRADIENT' ) then

!           /*   if first GRADINET   */
            if ( flag .eq. 0 ) then

               flag = 1

!           /*   if second GRADINET   */
            else

!              /*   loop of atoms   */
               do i = 1, natom

!                 /*   read the potential data   */

                  read ( iounit_abi, *, iostat=ierr ) &
     &               char_word(1), &
     &               fx(i,ibead), fy(i,ibead), fz(i,ibead)

!                 /*   gradient to force   */

                  fx(i,ibead) = - fx(i,ibead)
                  fy(i,ibead) = - fy(i,ibead)
                  fz(i,ibead) = - fz(i,ibead)

!              /*   loop of atoms   */
               end do

!              /*   exit from the do loop   */
               exit

!           /*   if matched   */
            end if

!        /*   if matched   */
         end if

!     /*   do loop end   */
      end do

!     /*   close file   */
      close( iounit_abi )

!     /*   print force   */
!      write(*,*) 'force bead=',ibead
!      do i = 1, natom
!         write(*,'(3f)') fx(i,ibead), fy(i,ibead), fz(i,ibead)
!      end do

!-----------------------------------------------------------------------
!     /*   read abinit-mp output:  dipole moment                      */
!-----------------------------------------------------------------------

!     /*   zero clear   */

      dipx(ibead) = 0.d0
      dipy(ibead) = 0.d0
      dipz(ibead) = 0.d0

!     /*   open the abinit-mp input file   */
      open ( iounit_abi, file = 'output.abi' )

!     /*   do loop start   */
      do

!        /*   read a line   */
         read ( iounit_abi, '(a)', iostat=ierr ) char_line

!        /*   exit at the end of the line   */
         if ( ierr .ne. 0 ) exit

!        /*   if matched   */
         if ( char_line(6:30) .eq. '##  Dipole moment / Debye' ) then

!           /*   read one lines   */
            read ( iounit_abi, '(a)', iostat=ierr ) char_line

!           /*   read the potential data   */
            read ( iounit_abi, *, iostat=ierr ) &
     &         char_word(1), char_word(2), dipx(ibead), &
     &         char_word(3), char_word(4), dipy(ibead), &
     &         char_word(5), char_word(6), dipz(ibead)
!            write(*,'(3f)') dipx(ibead),dipy(ibead),dipz(ibead)

!           /*   change unit:  debye -> au   */
            dipx(ibead) = dipx(ibead) / au2debye
            dipy(ibead) = dipy(ibead) / au2debye
            dipz(ibead) = dipz(ibead) / au2debye

!           /*   exit from the do loop   */
            exit

         end if

!     /*   do loop end   */
      end do

!     /*   close file   */
      close( iounit_abi )

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom

         xi = x(i,j)
         yi = y(i,j)
         zi = z(i,j)

         call pbc_unfold &
     &      ( xi, yi, zi, mbox(1,i,j), mbox(2,i,j), mbox(3,i,j) )

         vir(1,1) = vir(1,1) + fx(i,j)*xi
         vir(1,2) = vir(1,2) + fx(i,j)*yi
         vir(1,3) = vir(1,3) + fx(i,j)*zi
         vir(2,1) = vir(2,1) + fy(i,j)*xi
         vir(2,2) = vir(2,2) + fy(i,j)*yi
         vir(2,3) = vir(2,3) + fy(i,j)*zi
         vir(3,1) = vir(3,1) + fz(i,j)*xi
         vir(3,2) = vir(3,2) + fz(i,j)*yi
         vir(3,3) = vir(3,3) + fz(i,j)*zi

      end do
      end do

      return
      end

