!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     error function and its derivatives
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      real(8) function erf_0(x)
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, ntab, nterm

      real(8) :: ax,  erftab, dtab, dx, factor, gerf, sqrtpi, &
     &           x, xtab, error_function

      parameter ( ntab = 6000, nterm = 5 )

      common /erfs/ erftab(0:ntab,0:nterm), dtab, sqrtpi

      integer, save :: isave = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( isave .eq. 0 ) then

         dtab = 6.d0/dble(ntab)

         do i = 0, ntab
            xtab        = dble(i)*dtab
            erftab(i,0) = error_function(xtab)
            do j = 1, nterm
               erftab(i,j) = gerf(xtab,j)
            end do
         end do

         isave = 1

      end if

!-----------------------------------------------------------------------

      if ( x .eq. 0.d0 ) then

         erf_0 = 0.d0
         return

      end if

      ax     = abs(x)
      factor = x/ax
      i      = nint(ax/dtab)
      xtab   = dble(i)*dtab

!cc      if ( xtab .ge. 6.d0 ) then
      if ( ax .ge. 6.d0 ) then

         erf_0 = 1.d0*factor
         return

      else

         dx  = ax - xtab

!         coeff = 1.d0
!         erf_0 =  erftab(i,0)
!         do j = 1, nterm
!            coeff = coeff*dx/dble(j)
!            erf_0 = erf_0 + coeff*erftab(i,j)
!         end do
!         erf_0 = erf_0*factor

         erf_0 = ( erftab(i,0) &
     &         + dx*(                        erftab(i,1) &
     &         + dx*0.5000000000000000d+00*( erftab(i,2) &
     &         + dx*0.3333333333333333d+00*( erftab(i,3) &
     &         + dx*0.2500000000000000d+00*( erftab(i,4) )))) )*factor

         return

      end if

      return
      end





!***********************************************************************
      real(8) function erf_1(x)
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, ntab, nterm

      real(8) :: ax,  erftab, dtab, dx, gerf, pi, sqrtpi, &
     &           x, xtab, error_function

      parameter ( ntab = 6000, nterm = 5 )

      common /erfs/ erftab(0:ntab,0:nterm), dtab, sqrtpi

      integer, save :: isave = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( isave .eq. 0 ) then

         dtab = 6.d0/dble(ntab)

         do i = 0, ntab
            xtab        = dble(i)*dtab
            erftab(i,0) = error_function(xtab)
            do j = 1, nterm
               erftab(i,j) = gerf(xtab,j)
            end do
         end do

         pi     = acos(-1.d0)
         sqrtpi = sqrt(pi)

         isave = 1

      end if

!-----------------------------------------------------------------------

      if ( x .eq. 0.d0 ) then

         erf_1 = 2.d0/sqrtpi
         return

      end if

      ax   = abs(x)
      i    = nint(ax/dtab)
      xtab = dble(i)*dtab

!cc      if ( xtab .ge. 6.d0 ) then
      if ( ax .ge. 6.d0 ) then

         erf_1 = 0.d0
         return

      else

         dx  = ax - xtab

!         coeff = 1.d0
!         erf_1 =  erftab(i,1)
!         do j = 1, nterm
!            coeff = coeff*dx/dble(j)
!            erf_1 = erf_1 + coeff*erftab(i,j+1)
!         end do

!         erf_1 = 2.d0/sqrtpi*exp(-x*x)
!         return

         erf_1 =  erftab(i,1) &
     &         + dx*(                        erftab(i,2) &
     &         + dx*0.5000000000000000d+00*( erftab(i,3) &
     &         + dx*0.3333333333333333d+00*( erftab(i,4) &
     &         + dx*0.2500000000000000d+00*( erftab(i,5) ))))

         return

      end if

      return
      end





!***********************************************************************
      real(8) function gerf(x,n)
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j, k, n, nmax

      real(8) :: pi, x

      parameter ( nmax = 8 )

      real(8):: error_function

      integer, save :: isave = 0

      real(8), save :: sqrtpi

      integer, save :: l(nmax,nmax)

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( isave .eq. 0 ) then

         do j = 1, nmax
         do k = 1, nmax
            l(k,j) =  0
         end do
         end do

         l(1,1) =  1
         l(2,1) =  0
         l(2,2) = -2

         do k = 3, nmax
            l(k,1) = l(k-1,2)
            do j = 2, k-1
               l(k,j) = j*l(k-1,j+1) - 2*l(k-1,j-1)
            end do
            l(k,k) = - 2*l(k-1,k-1)
         end do

         pi     = acos(-1.d0)
         sqrtpi = sqrt(pi)

         isave = 1

      end if

      gerf = 0.d0

      if ( n .eq. 0 ) then

         gerf = error_function(x)

      else if ( n .le. nmax ) then

         gerf = 2.d0/sqrtpi*l(n,1)*exp(-x*x)

         do j = 2, n
            gerf = gerf + 2.d0/sqrtpi*l(n,j)*x**(j-1)*exp(-x*x)
         end do

      end if

      return
      end





!***********************************************************************
      FUNCTION error_function(x) RESULT(fn_val)
!***********************************************************************

!-----------------------------------------------------------------------
!             EVALUATION OF THE REAL ERROR FUNCTION
! Based upon a Fortran 66 routine in the Naval Surface Warfare Centers
! Mathematics Library (1993 version).
! Adapted by Alan.Miller @ vic.cmis.csiro.au
!-----------------------------------------------------------------------

      IMPLICIT NONE

      REAL (8), INTENT(IN) :: x
      REAL (8)             :: fn_val

!     /*   Local variables   */

      REAL (8), PARAMETER :: &
     &      c = .564189583547756d0, one = 1.0d0, half = 0.5d0, &
     &      zero = 0.0d0

      REAL (8), PARAMETER :: &
     &     a(5) = (/ .771058495001320D-04, -.133733772997339D-02, &
     &               .323076579225834D-01,  .479137145607681D-01, &
     &               .128379167095513D+00 /), &
     &     b(3) = (/ .301048631703895D-02,  .538971687740286D-01, &
     &               .375795757275549D+00 /), &
     &     p(8) = (/ -1.36864857382717D-07, 5.64195517478974D-01, &
     &                7.21175825088309D+00, 4.31622272220567D+01, &
     &                1.52989285046940D+02, 3.39320816734344D+02, &
     &                4.51918953711873D+02, 3.00459261020162D+02 /), &
     &     q(8) = (/  1.00000000000000D+00, 1.27827273196294D+01, &
     &                7.70001529352295D+01, 2.77585444743988D+02, &
     &                6.38980264465631D+02, 9.31354094850610D+02, &
     &                7.90950925327898D+02, 3.00459260956983D+02 /), &
     &     r(5) = (/  2.10144126479064D+00, 2.62370141675169D+01, &
     &                2.13688200555087D+01, 4.65807828718470D+00, &
     &                2.82094791773523D-01 /), &
     &     s(4) = (/  9.41537750555460D+01, 1.87114811799590D+02, &
     &                9.90191814623914D+01, 1.80124575948747D+01 /)

      REAL (8) :: ax, bot, t, top, x2

!-----------------------------------------------------------------------

      ax = ABS(x)

      IF (ax <= half) THEN
         t = x*x
         top = ((((a(1)*t + a(2))*t + a(3))*t + a(4))*t + a(5)) + one
         bot = ((b(1)*t + b(2))*t + b(3))*t + one
         fn_val = x*(top/bot)
         RETURN
      END IF

      IF (ax <= 4.0d0) THEN
         top = ((((((p(1)*ax + p(2))*ax + p(3))*ax + p(4))*ax + p(5))*ax &
     &       + p(6))*ax + p(7))*ax + p(8)
         bot = ((((((q(1)*ax + q(2))*ax + q(3))*ax + q(4))*ax + q(5))*ax &
     &       + q(6))*ax + q(7))*ax + q(8)
         fn_val = half + (half - EXP(-x*x)*top/bot)
      IF (x < zero) fn_val = -fn_val
         RETURN
      END IF

      IF (ax < 5.8d0) THEN
         x2 = x*x
         t = one / x2
         top = (((r(1)*t + r(2))*t + r(3))*t + r(4))*t + r(5)
         bot = (((s(1)*t + s(2))*t + s(3))*t + s(4))*t + one
         fn_val = (c - top/(x2*bot)) / ax
         fn_val = half + (half - EXP(-x2)*fn_val)
         IF (x < zero) fn_val = -fn_val
         RETURN
      END IF

      fn_val = SIGN(one, x)
      RETURN

!***********************************************************************
      END FUNCTION error_function
!***********************************************************************

