!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    June 14, 2020 by M. Shiga
!      Description:     static elastic constant
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine elastic
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, au_length, au_energy, box, boxinv, ux, uy, uz, &
     &   fdiff, fbox, iounit, natom, iboundary, iounit

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize variables   */
      implicit none

!     /*   integers   */
      integer :: i, j, k, l, m

!     /*   real numbers   */
      real(8) :: s1, s2, s3, r1, r2, r3, factor, const, det3
      real(8) :: c(6,6)

!     /*   centroid coordinates   */
      real(8), dimension(natom) :: ux0
      real(8), dimension(natom) :: uy0
      real(8), dimension(natom) :: uz0

!     /*   box   */
      real(8), dimension(3,3)   :: box0
      real(8), dimension(3,3)   :: boxinv0

!     /*   box forces   */
      real(8), dimension(-1:1,3,3) :: gbox

!     /*   conversion   */
      real(8), dimension(3,3)   :: dedh

!     /*   box volume   */
      real(8)                   :: volume0

!-----------------------------------------------------------------------
!     /*   error termination for free boundary                        */
!-----------------------------------------------------------------------

!     /*   free boundary   */
      if ( iboundary .eq. 0 ) then

!        /*   print message   */
         write( 6, '(a)' ) &
     &     'Error - Elastic is valid only for periodic boundary.'

!        /*   print message   */
         write( 6, '(a)' )

!        /*   error termination   */
         call error_handling ( 1, 'subroutine elastic', 18 )

!     /*   free boundary   */
      end if

!-----------------------------------------------------------------------
!     /*   get fbox as analytical derivatives                         */
!-----------------------------------------------------------------------

!     /*   calculate potential   */
      call getforce

!     /*   virial to fbox   */
      call getfbox

!-----------------------------------------------------------------------
!     /*   get fbox at origin                                         */
!-----------------------------------------------------------------------

!     /*   store box matrix   */
      box0(:,:) = box(:,:)

!     /*   store coordinates   */
      ux0(:) = ux(:,1)
      uy0(:) = uy(:,1)
      uz0(:) = uz(:,1)

!     /*   inverse box matrix   */
      call inv3 ( box0, boxinv0 )

!     /*   volume   */
      volume0 = det3( box0 )

!     /*    scaling factor   */
      dedh(1,1) = boxinv0(1,1)
      dedh(1,2) = boxinv0(1,1) + boxinv0(2,2)
      dedh(1,3) = boxinv0(1,1) + boxinv0(3,3)
      dedh(2,1) = boxinv0(2,2) + boxinv0(1,1)
      dedh(2,2) = boxinv0(2,2)
      dedh(2,3) = boxinv0(2,2) + boxinv0(3,3)
      dedh(3,1) = boxinv0(3,3) + boxinv0(1,1)
      dedh(3,2) = boxinv0(3,3) + boxinv0(2,2)
      dedh(3,3) = boxinv0(3,3)

!     /*   convert to gbox   */
      do i = 1, 3
         gbox(0,i,i) = - fbox(i,i) / dedh(i,i)
         do j = i+1, 3
            gbox(0,i,j) = - ( fbox(i,j) + fbox(j,i) ) / dedh(i,j)
            gbox(0,j,i) = - ( fbox(i,j) + fbox(j,i) ) / dedh(j,i)
         end do
      end do

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

!     //   loop of twelve points
      do l = 1, 12

!        /*   original box   */
         box(:,:) = box0(:,:)

!        /*   shift box   */
         if      ( l .eq. 1 ) then
            box(1,1) = box0(1,1) + fdiff / dedh(1,1)
         else if ( l .eq. 2 ) then
            box(1,1) = box0(1,1) - fdiff / dedh(1,1)
         else if ( l .eq. 3 ) then
            box(1,2) = box0(1,2) + fdiff / dedh(1,2)
            box(2,1) = box0(2,1) + fdiff / dedh(2,1)
         else if ( l .eq. 4 ) then
            box(1,2) = box0(1,2) - fdiff / dedh(1,2)
            box(2,1) = box0(2,1) - fdiff / dedh(2,1)
         else if ( l .eq. 5 ) then
            box(1,3) = box0(1,3) + fdiff / dedh(1,3)
            box(3,1) = box0(3,1) + fdiff / dedh(3,1)
         else if ( l .eq. 6 ) then
            box(1,3) = box0(1,3) - fdiff / dedh(1,3)
            box(3,1) = box0(3,1) - fdiff / dedh(3,1)
         else if ( l .eq. 7 ) then
            box(2,2) = box0(2,2) + fdiff / dedh(2,2)
         else if ( l .eq. 8 ) then
            box(2,2) = box0(2,2) - fdiff / dedh(2,2)
         else if ( l .eq. 9 ) then
            box(2,3) = box0(2,3) + fdiff / dedh(2,3)
            box(3,2) = box0(3,2) + fdiff / dedh(3,2)
         else if ( l .eq. 10 ) then
            box(2,3) = box0(2,3) - fdiff / dedh(2,3)
            box(3,2) = box0(3,2) - fdiff / dedh(3,2)
         else if ( l .eq. 11 ) then
            box(3,3) = box0(3,3) + fdiff / dedh(3,3)
         else if ( l .eq. 12 ) then
            box(3,3) = box0(3,3) - fdiff / dedh(3,3)
         end if

!        /*   inverse box matrix   */
         call inv3 ( box, boxinv )

!        /*   loop of atoms   */
         do k = 1, natom

!           /*   original centroid coordinates   */
            r1 = ux0(k)
            r2 = uy0(k)
            r3 = uz0(k)

!           /*   relative coordinates   */
            s1 = boxinv0(1,1)*r1 + boxinv0(1,2)*r2 + boxinv0(1,3)*r3
            s2 = boxinv0(2,1)*r1 + boxinv0(2,2)*r2 + boxinv0(2,3)*r3
            s3 = boxinv0(3,1)*r1 + boxinv0(3,2)*r2 + boxinv0(3,3)*r3

!           /*   shifted centroid coordinates   */
            ux(k,1) = box(1,1)*s1 + box(1,2)*s2 + box(1,3)*s3
            uy(k,1) = box(2,1)*s1 + box(2,2)*s2 + box(2,3)*s3
            uz(k,1) = box(3,1)*s1 + box(3,2)*s2 + box(3,3)*s3

!        /*   loop of atoms   */
         end do

!        /*   normal mode position -> cartesian position   */
         x(:,1) = ux(:,1)
         y(:,1) = uy(:,1)
         z(:,1) = uz(:,1)

!        /*   calculate potential   */
         call getforce

!        /*   virial to fbox   */
         call getfbox

!        /*   shift box   */
         if ( mod(l,2) .eq. 1 ) then

!           /*   convert to gbox   */
            do i = 1, 3
               gbox(+1,i,i) = - fbox(i,i) / dedh(i,i)
               do j = i+1, 3
                  gbox(+1,i,j) = - ( fbox(i,j) + fbox(j,i) ) / dedh(i,j)
                  gbox(+1,j,i) = - ( fbox(i,j) + fbox(j,i) ) / dedh(j,i)
               end do
            end do

         else

!           /*   convert to gbox   */
            do i = 1, 3
               gbox(-1,i,i) = - fbox(i,i) / dedh(i,i)
               do j = i+1, 3
                  gbox(-1,i,j) = - ( fbox(i,j) + fbox(j,i) ) / dedh(i,j)
                  gbox(-1,j,i) = - ( fbox(i,j) + fbox(j,i) ) / dedh(j,i)
               end do
            end do

         end if

!        /*   initialize index   */
         m = 0

!        /*   Voigt index   */
         if      ( l .eq. 2 ) then
            m = 1
         else if ( l .eq. 4 ) then
            m = 6
         else if ( l .eq. 6 ) then
            m = 5
         else if ( l .eq. 8 ) then
            m = 2
         else if ( l .eq. 10 ) then
            m = 4 
         else if ( l .eq. 12 ) then
            m = 3
         end if

!        /*   for odd cycles   */
         if ( m .eq. 0 ) cycle

!        /*   constant factor   */
         const = 1.d0 / ( 2.d0 * fdiff ) / volume0

!        /*   elastic constants   */
         c(m,1) = ( gbox(+1,1,1) - gbox(-1,1,1) ) * const
         c(m,2) = ( gbox(+1,2,2) - gbox(-1,2,2) ) * const
         c(m,3) = ( gbox(+1,3,3) - gbox(-1,3,3) ) * const
         c(m,4) = ( gbox(+1,2,3) - gbox(-1,2,3) ) * const
         c(m,5) = ( gbox(+1,1,3) - gbox(-1,1,3) ) * const
         c(m,6) = ( gbox(+1,1,2) - gbox(-1,1,2) ) * const

!     //   loop of twelve points
      end do

!-----------------------------------------------------------------------
!     /*   print results                                              */
!-----------------------------------------------------------------------

!     /*   [au] -> [GPa]   */
      factor = au_energy / au_length**3 / 10**9

!     /*   elastic constants in GPa   */
      write( 6, '(a)' ) 

      write( 6, '(a)' ) &
     &  '--------------------------------------------------------------' &
     &  // '----------------'
      write( 6, '(a)' )  &
     &  '                      elastic constants Cij in GPa            ' &
     &  // '               '
      write( 6, '(a)' ) &
     &  ' i / j           1           2           3           4        ' &
     &  // '   5           6'
      write( 6, '(a)' ) &
     &  '--------------------------------------------------------------' &
     &  // '----------------'

      do i = 1, 6
         write( 6, '(i6,6f12.2)' ) i, ( c(i,j)*factor, j = 1, 6 )
      end do

      write( 6, '(a)' ) &
     &  '--------------------------------------------------------------' &
     &  // '----------------'

      write( 6, '(a)' )

!-----------------------------------------------------------------------
!     /*   print results                                              */
!-----------------------------------------------------------------------

!     /*   open file   */
      open( iounit, file = 'standard.out' )

!     /*   elastic constants in GPa   */
      write( iounit, '(a)' ) 

      write( iounit, '(a)' )  &
     &  '--------------------------------------------------------------' &
     &  // '----------------'
      write( iounit, '(a)' )  &
     &  '                     elastic constants C(i,j) in GPa          ' &
     &  // '               '
      write( iounit, '(a)' )  &
     &  '     C           1           2           3           4        ' &
     &  // '   5           6'
      write( iounit, '(a)' )  &
     &  '--------------------------------------------------------------' &
     &  // '----------------'

      do i = 1, 6
         write( iounit, '(i6,6f12.2)' ) i, ( c(i,j)*factor, j = 1, 6 )
      end do

      write( iounit, '(a)' )  &
     &  '--------------------------------------------------------------' &
     &  // '----------------'

      write( iounit, '(a)' ) 

!     /*   close file   */
      open( iounit, file = 'standard.out' )

      return
      end

