!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jan 16, 2024 by M. Shiga
!      Description:     Solves One Dimensional Schroedinger Equation
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      module dvr_variables
!***********************************************************************

!     /*   number of grid points   */
      integer :: ndvr

!     /*   maximum value of the grid   */
      real(8) :: dvrmax

!     /*   minimum value of the grid   */
      real(8) :: dvrmin

!     /*   grid spacing   */
      real(8) :: dvrmesh

!     /*   reduced mass   */
      real(8) :: redmass

!     /*   grid points   */
      real(8), dimension(:), allocatable :: xdvr

!     /*   potential energies   */
      real(8), dimension(:), allocatable :: vdvr

!     /*   hamiltonian matrix   */
      real(8), dimension(:,:), allocatable :: hdvr

!     /*   eigenvectors   */
      real(8), dimension(:,:), allocatable :: cdvr

!     /*   eigenvalues   */
      real(8), dimension(:), allocatable :: edvr

!     /*   eigenvalues   */
      real(8), dimension(:,:), allocatable :: cminv

!     /*   potential minimum   */
      real(8) :: vmin_dvr

!     /*   maximum vibrational states   */
      integer :: nvib = 3

!     /*   maximum rotational states   */
      integer :: nrot = 10

!     /*   conversion factor   */
      real(8) :: au_to_cminv = 2.194743d+5

!     /*   maximum frequency   */
      integer :: max_cminv = 5000

!     /*   spectrum  */
      real(8), dimension(:), allocatable :: fcminv

!***********************************************************************
      end module dvr_variables
!***********************************************************************





!***********************************************************************
      subroutine dvrcycle_MPI
!***********************************************************************

!     /*   local variables   */
      implicit none

!     /*   potential calculation   */
      call getforce_MPI

!     /*   solve eigenvalue problem   */
      call main_dvr_MPI

!     /*   write output   */
      call standard_dvr_MPI

      return
      end





!***********************************************************************
      subroutine setup_dvr_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, ux, uy, uz, physmass, natom, nbead, iounit, myrank

      use dvr_variables, only : &
     &   dvrmin, dvrmax, dvrmesh, redmass, hdvr, vdvr, edvr, fcminv, &
     &   cdvr, xdvr, cminv, ndvr, nvib, nrot, max_cminv

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   reset   */
      implicit none

!     /*   integer   */
      integer :: i, ierr

!-----------------------------------------------------------------------
!     /*   number of atoms must be 2                                  */
!-----------------------------------------------------------------------

!     /*   error flag   */
      ierr = 0

!     /*   check error   */
      if ( natom .ne. 2 ) ierr = 1

!     /*   check error   */
      if ( ( myrank .eq. 0 ) .and. ( ierr .ne. 0 ) ) then
         write( 6, '(a)' ) 'Error - set <natom> = 2.'
      end if

!     /*   check error   */
      call error_handling_MPI( ierr, 'subroutine setup_dvr_MPI', 24 )

!-----------------------------------------------------------------------
!     /*   read type of collective variables                          */
!-----------------------------------------------------------------------

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag ( '<dvr>', 5, iounit, ierr )

!        /*   read integer   */
         read ( iounit, *, iostat=ierr ) dvrmin, dvrmax

!        /*   file close   */
         close( iounit )

!     /*   master rank only   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   error message   */
      if ( ( myrank .eq. 0 ) .and. ( ierr .ne. 0 ) ) then
         write( 6, '(a)' ) 'Error - keyword <dvr> is incorrect.'
      end if

!     /*   check error   */
      call error_handling_MPI( ierr, 'subroutine setup_dvr_MPI', 24 )

!     /*   communicate   */
      call my_mpi_bcast_real_0( dvrmin )
      call my_mpi_bcast_real_0( dvrmax )

!-----------------------------------------------------------------------
!     /*   reduced mass                                               */
!-----------------------------------------------------------------------

      redmass = physmass(1) * physmass(2) / ( physmass(1)+physmass(2) )

!-----------------------------------------------------------------------
!     /*   number of grids                                            */
!-----------------------------------------------------------------------

      ndvr = nbead

!-----------------------------------------------------------------------
!     /*   grid size                                                  */
!-----------------------------------------------------------------------

      dvrmesh = ( dvrmax - dvrmin ) / ( ndvr + 1 )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   grid   */
      if ( .not. allocated(xdvr) ) allocate( xdvr(ndvr) )

!     /*   hamiltonian   */
      if ( .not. allocated(hdvr) ) allocate( hdvr(ndvr,ndvr) )

!     /*   eigenstate   */
      if ( .not. allocated(cdvr) ) allocate( cdvr(ndvr,ndvr) )

!     /*   eigenvalue   */
      if ( .not. allocated(edvr) ) allocate( edvr(ndvr) )

!     /*   potential   */
      if ( .not. allocated(vdvr) ) allocate( vdvr(ndvr) )

!     /*   frequency   */
      if ( .not. allocated(cminv) ) allocate( cminv(0:nvib+1,0:nrot+1) )

      if ( .not. allocated(fcminv) ) allocate( fcminv(max_cminv) )

!-----------------------------------------------------------------------
!     /*   generate grids:  one dimensional case                      */
!-----------------------------------------------------------------------

      do i = 1, ndvr
         xdvr(i) = dvrmin + dvrmesh * dble(i)
      end do

!-----------------------------------------------------------------------
!     /*   set geometry                                               */
!-----------------------------------------------------------------------

      x(1,:) = 0.d0
      y(1,:) = 0.d0
      z(1,:) = 0.d0

      do i = 1, ndvr
         x(2,i) = xdvr(i)
      end do
      y(2,:) = 0.d0
      z(2,:) = 0.d0

      ux(:,:) = 0.d0
      uy(:,:) = 0.d0
      uz(:,:) = 0.d0

      return
      end





!***********************************************************************
      subroutine main_dvr_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &    pot, pi

      use dvr_variables, only : &
     &    hdvr, vdvr, edvr, cdvr, vmin_dvr, ndvr, dvrmax, dvrmin, &
     &    xdvr, redmass, au_to_cminv, cminv, nvib, nrot

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integer   */
      integer :: i, j, ix, jx, irot, ivib

!     /*   real numbers   */
      real(8) :: const, pm, sinm, sinp, di, dn

!     /*   real numbers   */
      real(8) :: d(ndvr,ndvr), t(ndvr,ndvr)

!-----------------------------------------------------------------------
!     /*   loop of rotational states                                  */
!-----------------------------------------------------------------------

      do irot = 0, nrot+1

!-----------------------------------------------------------------------
!     /*   potential                                                  */
!-----------------------------------------------------------------------

!     /*   minimum   */
      vmin_dvr = minval( pot(:) )

!     /*   interatomic potential   */
      vdvr(:) = pot(:) - vmin_dvr

!     /*   rotational potential   */
      do i = 1, ndvr
         vdvr(i) = vdvr(i) &
     &           + dble(irot*(irot+1)) / (2.d0*redmass*xdvr(i)*xdvr(i))
      end do

!-----------------------------------------------------------------------
!     /*   constant                                                   */
!-----------------------------------------------------------------------

      const = 1.d0 / (2.d0*redmass) / (dvrmax-dvrmin)**2 * pi**2 / 2.d0

!-----------------------------------------------------------------------
!     /*   kronecker delta                                            */
!-----------------------------------------------------------------------

      d(:,:) = 0.d0

      do i = 1, ndvr
         d(i,i) = 1.d0
      end do

!-----------------------------------------------------------------------
!     /*   generate kinetic energy matrix                             */
!-----------------------------------------------------------------------

      do i = 1, ndvr
      do j = 1, ndvr

         if ( i .eq. j ) then

            di = dble( i )
            dn = dble( ndvr + 1 )

            sinm = sin( pi*di/dn )

            t(i,i) = const * ( (2.d0*dn**2+1.d0)/3.d0 - 1.d0/sinm**2 )

         else

            if ( mod(abs(i-j),2) .eq. 0 ) then
               pm =   1.d0
            else
               pm = - 1.d0
            end if

            sinm = sin( pi * dble(i-j) / dble(2*ndvr+2) )
            sinp = sin( pi * dble(i+j) / dble(2*ndvr+2) )

            t(i,j) = const * pm * ( 1.d0/sinm**2 - 1.d0/sinp**2 )

         end if

      end do
      end do

!-----------------------------------------------------------------------
!     /*   generate hamiltonian matrix:  one dimensional case         */
!-----------------------------------------------------------------------

      i = 0

      do ix = 1, ndvr

         i = i + 1

         j = 0

         do jx = 1, ndvr

            j = j + 1

            hdvr(i,j) =  t(ix,jx) + vdvr(ix)*d(ix,jx)

         end do

      end do

!-----------------------------------------------------------------------
!     /*   diagonalize hamiltonian                                    */
!-----------------------------------------------------------------------

      call ddiag_MPI( hdvr, edvr, cdvr, ndvr )

!-----------------------------------------------------------------------
!     /*   states                                                     */
!-----------------------------------------------------------------------

      do ivib = 0, nvib+1
         cminv(ivib,irot)  = edvr(ivib+1) * au_to_cminv
      end do

!-----------------------------------------------------------------------
!     /*   loop of rotational states                                  */
!-----------------------------------------------------------------------

      end do

      return
      end





!***********************************************************************
      subroutine standard_dvr_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   beta, iounit, iounit_std, myrank

      use dvr_variables, only : &
     &   au_to_cminv, cminv, fcminv, max_cminv, nvib, nrot

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   reset   */
      implicit none

!     /*   integer   */
      integer :: ivib, irot, i

!     /*   boltzmann factor   */
      real(8) :: weight

!     /*   energy differences   */
      real(8) ::  vprp, vprm, vmrp, vmrm, de, vpr0, vmr0, vpavg, vmavg

!-----------------------------------------------------------------------
!     /*   master rank only                                           */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      write( 6, * )
      write( 6, '(a)' ) &
     &   '------------------------------------------------------' // &
     &   '------------------------'
      write( 6, '(a)' ) &
     &   '   (v,r)  E(v,r) E0(v,r)   wt. (+1,+1)  (+1,0) (+1,-1)' // &
     &   ' (-1,+1)  (-1,0) (-1,-1)'
      write( 6, '(a)' ) &
     &   '------------------------------------------------------' // &
     &   '------------------------'

      open ( iounit_std, file ='standard.out' )

      write( iounit_std, * )
      write( iounit_std, '(a)' ) &
     &   '------------------------------------------------------' // &
     &   '------------------------'
      write( iounit_std, '(a)' ) &
     &   '   (v,r)  E(v,r) E0(v,r)   wt. (+1,+1)  (+1,0) (+1,-1)' // &
     &   ' (-1,+1)  (-1,0) (-1,-1)'
      write( iounit_std, '(a)' ) &
     &   '------------------------------------------------------' // &
     &   '------------------------'

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      fcminv(:) = 0.d0

      do ivib = 0, nvib
      do irot = 0, nrot

         de = ( cminv(ivib,irot) - cminv(0,0) ) / au_to_cminv

         weight = dble(2*irot+1) * exp( - beta * de )

         vprp = 0.d0
         vprm = 0.d0
         vpr0 = 0.d0
         vmrp = 0.d0
         vmrm = 0.d0
         vmr0 = 0.d0

         vpavg = 0.d0
         vmavg = 0.d0

         vprp = - cminv(ivib,irot) + cminv(ivib+1,irot+1)
         vpr0 = - cminv(ivib,irot) + cminv(ivib+1,irot)

         if ( irot .ge. 1 ) &
     &      vprm = - cminv(ivib,irot) + cminv(ivib+1,irot-1)

         if ( ivib .ge. 1 ) &
     &      vmrp = - cminv(ivib,irot) + cminv(ivib-1,irot+1)

         if ( ivib .ge. 1 ) &
     &      vmr0 = - cminv(ivib,irot) + cminv(ivib-1,irot)

         if ( ( ivib .ge. 1 ) .and. ( irot .ge. 1 ) ) &
     &      vmrm = - cminv(ivib,irot) + cminv(ivib-1,irot-1)

         vpavg = 0.5d0 * ( vprp + vprm )
         vmavg = 0.5d0 * ( vmrp + vmrm )

         write ( 6, '(2i4,2f8.1,f6.3,6f8.1)' ) &
     &      irot, ivib, cminv(ivib,irot), &
     &      cminv(ivib,irot)-cminv(0,0), weight, &
     &      vprp, vpr0, vprm, vmrp, vmr0, vmrm

         write ( iounit_std, '(2i4,2f8.1,f6.3,6f8.1)' ) &
     &      irot, ivib, cminv(ivib,irot), &
     &      cminv(ivib,irot)-cminv(0,0), weight, &
     &      vprp, vpr0, vprm, vmrp, vmr0, vmrm

!        /*   frequency   */
         i = nint(vpr0)

!        /*   add to spectrum   */
         if ( ( i .ge. 1 ) .and. ( i .le. max_cminv ) ) then
            fcminv(i) = fcminv(i) + weight
         end if

      end do
      end do

      write( 6, * )

      write( iounit_std, * )

      close( iounit_std )

      open( iounit, file = 'dvr.out' )

      do i = 1, max_cminv
         write( iounit, '(i6,f8.4)' ) i, fcminv(i)
      end do

      close( iounit )

      return
      end
