!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     rotational correction to centroid force
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine correct_force
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   itrans_start, irot_start, iboundary, method

      implicit none

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if      ( method(1:5) .eq. 'PIMD ' ) then
         continue
      else if ( method(1:5) .eq. 'PIHMC' ) then
         continue
      else if ( method(1:5) .eq. 'CMD  ' ) then
         continue
      else if ( method(1:5) .eq. 'RPMD ' ) then
         continue
      else if ( method(1:5) .eq. 'TRPMD' ) then
         continue
      else if ( method(1:5) .eq. 'BCMD ' ) then
         continue
      else
         return
      end if

      if ( ( itrans_start .ne. 2 ) .or. ( irot_start .ne. 2 ) ) return

!-----------------------------------------------------------------------
!     /*   Cartesian force  ->  normal mode force                     */
!-----------------------------------------------------------------------

      call nm_trans_force ( 1 )

!-----------------------------------------------------------------------
!     /*   remove net force                                           */
!-----------------------------------------------------------------------

!      if ( itrans_start .eq. 1 ) call subtract_force_cent
      if ( itrans_start .eq. 2 ) call subtract_force_cent
!      if ( itrans_start .eq. 3 ) call subtract_force_modes
      if ( itrans_start .eq. 4 ) call subtract_force_modes
!      if ( itrans_start .eq. 5 ) call subtract_force_all
      if ( itrans_start .eq. 6 ) call subtract_force_all

!-----------------------------------------------------------------------
!     /*   subtract rotation:  only free boundary condition           */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) then

!         if ( irot_start .eq. 1 ) call subtract_torque_cent
         if ( irot_start .eq. 2 ) call subtract_torque_cent
!         if ( irot_start .eq. 3 ) call subtract_torque_modes
         if ( irot_start .eq. 4 ) call subtract_torque_modes
!         if ( irot_start .eq. 5 ) call subtract_torque_mix
         if ( irot_start .eq. 6 ) call subtract_torque_mix

      end if

!-----------------------------------------------------------------------
!     /*   normal mode force  ->  Cartesian force                     */
!-----------------------------------------------------------------------

      call nm_trans_force ( 0 )

      return
      end





!***********************************************************************
      subroutine subtract_force_cent
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fux, fuy, fuz, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j

      real(8) :: sump, sumfx, sumfy, sumfz

!-----------------------------------------------------------------------
!     /*   remove net force                                           */
!-----------------------------------------------------------------------

      sumfx = 0.d0
      sumfy = 0.d0
      sumfz = 0.d0
      sump  = 0.d0

      do j = 1, natom
         sumfx = sumfx + fux(j,1)
         sumfy = sumfy + fuy(j,1)
         sumfz = sumfz + fuz(j,1)
         sump  = sump  + 1.d0
      end do

      sumfx = sumfx/sump
      sumfy = sumfy/sump
      sumfz = sumfz/sump

      do j = 1, natom
         fux(j,1) = fux(j,1) - sumfx
         fuy(j,1) = fuy(j,1) - sumfy
         fuz(j,1) = fuz(j,1) - sumfz
      end do

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_force ( 0 )

      return
      end





!***********************************************************************
      subroutine subtract_torque_cent
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ux, uy, uz, fux, fuy, fuz, xg, yg, zg, fictmass, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i

      real(8) :: fm

      real(8), dimension(3,3) :: a, c

      real(8), dimension(3)   :: b, d, e, f

      real(8) :: tiny = 1.d-14

!-----------------------------------------------------------------------
!     /*   center of mass                                             */
!-----------------------------------------------------------------------

      call center_of_mass

!-----------------------------------------------------------------------
!     /*   centroids only                                             */
!-----------------------------------------------------------------------

!     /*   torque   */

      b(:) = 0.d0

      do i = 1, natom
         fm   = fictmass(i,1)
         b(1) = b(1) + (uy(i,1)-yg(1))*fuz(i,1) &
     &               - (uz(i,1)-zg(1))*fuy(i,1)
         b(2) = b(2) + (uz(i,1)-zg(1))*fux(i,1) &
     &               - (ux(i,1)-xg(1))*fuz(i,1)
         b(3) = b(3) + (ux(i,1)-xg(1))*fuy(i,1) &
     &               - (uy(i,1)-yg(1))*fux(i,1)
      end do

!     /*   moment of inertia   */

      a(:,:) = 0.d0

      do i = 1, natom
         fm   = fictmass(i,1)
         a(1,1) = a(1,1) + fm*(uy(i,1)-yg(1))*(uy(i,1)-yg(1)) &
     &                   + fm*(uz(i,1)-zg(1))*(uz(i,1)-zg(1))
         a(1,2) = a(1,2) - fm*(ux(i,1)-xg(1))*(uy(i,1)-yg(1))
         a(1,3) = a(1,3) - fm*(ux(i,1)-xg(1))*(uz(i,1)-zg(1))
         a(2,1) = a(2,1) - fm*(uy(i,1)-yg(1))*(ux(i,1)-xg(1))
         a(2,2) = a(2,2) + fm*(uz(i,1)-zg(1))*(uz(i,1)-zg(1)) &
     &                   + fm*(ux(i,1)-xg(1))*(ux(i,1)-xg(1))
         a(2,3) = a(2,3) - fm*(uy(i,1)-yg(1))*(uz(i,1)-zg(1))
         a(3,1) = a(3,1) - fm*(uz(i,1)-zg(1))*(ux(i,1)-xg(1))
         a(3,2) = a(3,2) - fm*(uz(i,1)-zg(1))*(uy(i,1)-yg(1))
         a(3,3) = a(3,3) + fm*(ux(i,1)-xg(1))*(ux(i,1)-xg(1)) &
     &                   + fm*(uy(i,1)-yg(1))*(uy(i,1)-yg(1))
      end do

!     /*   principal axis:  diagonalize moment of inertia          */

      call ddiag ( a, e, c, 3 )

!     /*   in principal axis:  torque                              */

      d(1) = c(1,1)*b(1) + c(2,1)*b(2) + c(3,1)*b(3)
      d(2) = c(1,2)*b(1) + c(2,2)*b(2) + c(3,2)*b(3)
      d(3) = c(1,3)*b(1) + c(2,3)*b(2) + c(3,3)*b(3)

!     /*   d = torque divided by moment of inertia                 */

      if ( natom .eq. 1 ) then
         d(1) = 0.d0
         d(2) = 0.d0
         d(3) = 0.d0
      else if ( natom .eq. 2 ) then
         d(1) = 0.d0
         if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
         if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
      else
         if ( e(1) .gt. tiny ) d(1) = d(1)/e(1)
         if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
         if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
      end if

!     /*   d in laboratory frame                                   */

      f(1) = c(1,1)*d(1) + c(1,2)*d(2) + c(1,3)*d(3)
      f(2) = c(2,1)*d(1) + c(2,2)*d(2) + c(2,3)*d(3)
      f(3) = c(3,1)*d(1) + c(3,2)*d(2) + c(3,3)*d(3)

      do i = 1, natom
         fm   = fictmass(i,1)
         fux(i,1) = fux(i,1) - f(2)*(uz(i,1)-zg(1))*fm &
     &                       + f(3)*(uy(i,1)-yg(1))*fm
         fuy(i,1) = fuy(i,1) - f(3)*(ux(i,1)-xg(1))*fm &
     &                       + f(1)*(uz(i,1)-zg(1))*fm
         fuz(i,1) = fuz(i,1) - f(1)*(uy(i,1)-yg(1))*fm &
     &                       + f(2)*(ux(i,1)-xg(1))*fm
      end do

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_force ( 0 )

      return
      end





!***********************************************************************
      subroutine subtract_force_modes
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fux, fuy, fuz, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j, k

      real(8) :: sump, sumfx, sumfy, sumfz

!-----------------------------------------------------------------------
!     /*   remove centroid force                                      */
!-----------------------------------------------------------------------

      sumfx = 0.d0
      sumfy = 0.d0
      sumfz = 0.d0
      sump  = 0.d0

      do j = 1, natom
         sumfx = sumfx + fux(j,1)
         sumfy = sumfy + fuy(j,1)
         sumfz = sumfz + fuz(j,1)
         sump  = sump  + 1.d0
      end do

      sumfx = sumfx/sump
      sumfy = sumfy/sump
      sumfz = sumfz/sump

      do j = 1, natom
         fux(j,1) = fux(j,1) - sumfx
         fuy(j,1) = fuy(j,1) - sumfy
         fuz(j,1) = fuz(j,1) - sumfz
      end do

!-----------------------------------------------------------------------
!     /*   remove noncentroid force                                   */
!-----------------------------------------------------------------------

      sumfx = 0.d0
      sumfy = 0.d0
      sumfz = 0.d0
      sump  = 0.d0

      do k = 2, nbead
      do j = 1, natom
         sumfx = sumfx + fux(j,k)
         sumfy = sumfy + fuy(j,k)
         sumfz = sumfz + fuz(j,k)
         sump  = sump  + 1.d0
      end do
      end do

      sumfx = sumfx/sump
      sumfy = sumfy/sump
      sumfz = sumfz/sump

      do k = 2, nbead
      do j = 1, natom
         fux(j,k) = fux(j,k) - sumfx
         fuy(j,k) = fuy(j,k) - sumfy
         fuz(j,k) = fuz(j,k) - sumfz
      end do
      end do

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_force ( 0 )

      return
      end





!***********************************************************************
      subroutine subtract_torque_modes
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ux, uy, uz, fux, fuy, fuz, xg, yg, zg, fictmass, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, k

      real(8) :: fm

      real(8), dimension(3,3) :: a, c

      real(8), dimension(3)   :: b, d, e, f

      real(8) :: tiny = 1.d-14

!-----------------------------------------------------------------------
!     /*   center of mass                                             */
!-----------------------------------------------------------------------

      call center_of_mass

!-----------------------------------------------------------------------
!     /*   centroids                                                  */
!-----------------------------------------------------------------------

!     /*   torque   */

      b(:) = 0.d0

      do i = 1, natom
         fm   = fictmass(i,1)
         b(1) = b(1) + (uy(i,1)-yg(1))*fuz(i,1) &
     &               - (uz(i,1)-zg(1))*fuy(i,1)
         b(2) = b(2) + (uz(i,1)-zg(1))*fux(i,1) &
     &               - (ux(i,1)-xg(1))*fuz(i,1)
         b(3) = b(3) + (ux(i,1)-xg(1))*fuy(i,1) &
     &               - (uy(i,1)-yg(1))*fux(i,1)
      end do

!     /*   moment of inertia   */

      a(:,:) = 0.d0

      do i = 1, natom
         fm   = fictmass(i,1)
         a(1,1) = a(1,1) + fm*(uy(i,1)-yg(1))*(uy(i,1)-yg(1)) &
     &                   + fm*(uz(i,1)-zg(1))*(uz(i,1)-zg(1))
         a(1,2) = a(1,2) - fm*(ux(i,1)-xg(1))*(uy(i,1)-yg(1))
         a(1,3) = a(1,3) - fm*(ux(i,1)-xg(1))*(uz(i,1)-zg(1))
         a(2,1) = a(2,1) - fm*(uy(i,1)-yg(1))*(ux(i,1)-xg(1))
         a(2,2) = a(2,2) + fm*(uz(i,1)-zg(1))*(uz(i,1)-zg(1)) &
     &                   + fm*(ux(i,1)-xg(1))*(ux(i,1)-xg(1))
         a(2,3) = a(2,3) - fm*(uy(i,1)-yg(1))*(uz(i,1)-zg(1))
         a(3,1) = a(3,1) - fm*(uz(i,1)-zg(1))*(ux(i,1)-xg(1))
         a(3,2) = a(3,2) - fm*(uz(i,1)-zg(1))*(uy(i,1)-yg(1))
         a(3,3) = a(3,3) + fm*(ux(i,1)-xg(1))*(ux(i,1)-xg(1)) &
     &                   + fm*(uy(i,1)-yg(1))*(uy(i,1)-yg(1))
      end do

!     /*   principal axis:  diagonalize moment of inertia          */

      call ddiag ( a, e, c, 3 )

!     /*   in principal axis:  torque                              */

      d(1) = c(1,1)*b(1) + c(2,1)*b(2) + c(3,1)*b(3)
      d(2) = c(1,2)*b(1) + c(2,2)*b(2) + c(3,2)*b(3)
      d(3) = c(1,3)*b(1) + c(2,3)*b(2) + c(3,3)*b(3)

!     /*   d = torque divided by moment of inertia                 */

      if ( natom .eq. 1 ) then
         d(1) = 0.d0
         d(2) = 0.d0
         d(3) = 0.d0
      else if ( natom .eq. 2 ) then
         d(1) = 0.d0
         if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
         if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
      else
         if ( e(1) .gt. tiny ) d(1) = d(1)/e(1)
         if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
         if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
      end if

!     /*   d in laboratory frame                                   */

      f(1) = c(1,1)*d(1) + c(1,2)*d(2) + c(1,3)*d(3)
      f(2) = c(2,1)*d(1) + c(2,2)*d(2) + c(2,3)*d(3)
      f(3) = c(3,1)*d(1) + c(3,2)*d(2) + c(3,3)*d(3)

      do i = 1, natom
         fm   = fictmass(i,1)
         fux(i,1) = fux(i,1) - f(2)*(uz(i,1)-zg(1))*fm &
     &                       + f(3)*(uy(i,1)-yg(1))*fm
         fuy(i,1) = fuy(i,1) - f(3)*(ux(i,1)-xg(1))*fm &
     &                       + f(1)*(uz(i,1)-zg(1))*fm
         fuz(i,1) = fuz(i,1) - f(1)*(uy(i,1)-yg(1))*fm &
     &                       + f(2)*(ux(i,1)-xg(1))*fm
      end do

!-----------------------------------------------------------------------
!     /*   noncentroids                                               */
!-----------------------------------------------------------------------

!     /*   torque   */

      b(:) = 0.d0

      do k = 2, nbead
      do i = 1, natom
         fm   = fictmass(i,k)
         b(1) = b(1) + (uy(i,k))*fuz(i,k) &
     &               - (uz(i,k))*fuy(i,k)
         b(2) = b(2) + (uz(i,k))*fux(i,k) &
     &               - (ux(i,k))*fuz(i,k)
         b(3) = b(3) + (ux(i,k))*fuy(i,k) &
     &               - (uy(i,k))*fux(i,k)
      end do
      end do

!     /*   moment of inertia   */

      a(:,:) = 0.d0

      do k = 2, nbead
      do i = 1, natom
         fm   = fictmass(i,k)
         a(1,1) = a(1,1) + fm*(uy(i,k))*(uy(i,k)) &
     &                   + fm*(uz(i,k))*(uz(i,k))
         a(1,2) = a(1,2) - fm*(ux(i,k))*(uy(i,k))
         a(1,3) = a(1,3) - fm*(ux(i,k))*(uz(i,k))
         a(2,1) = a(2,1) - fm*(uy(i,k))*(ux(i,k))
         a(2,2) = a(2,2) + fm*(uz(i,k))*(uz(i,k)) &
     &                   + fm*(ux(i,k))*(ux(i,k))
         a(2,3) = a(2,3) - fm*(uy(i,k))*(uz(i,k))
         a(3,1) = a(3,1) - fm*(uz(i,k))*(ux(i,k))
         a(3,2) = a(3,2) - fm*(uz(i,k))*(uy(i,k))
         a(3,3) = a(3,3) + fm*(ux(i,k))*(ux(i,k)) &
     &                   + fm*(uy(i,k))*(uy(i,k))
      end do
      end do

!     /*   principal axis:  diagonalize moment of inertia       */

      call ddiag( a, e, c, 3 )

      /*   in principal axis:  torque                           */

      d(1) = c(1,1)*b(1) + c(2,1)*b(2) + c(3,1)*b(3)
      d(2) = c(1,2)*b(1) + c(2,2)*b(2) + c(3,2)*b(3)
      d(3) = c(1,3)*b(1) + c(2,3)*b(2) + c(3,3)*b(3)

!     /*   d = torque divided by moment of inertia              */

      if ( natom .eq. 1 ) then
         d(1) = 0.d0
         d(2) = 0.d0
         d(3) = 0.d0
      else if ( natom .eq. 2 ) then
         d(1) = 0.d0
         if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
         if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
      else
         if ( e(1) .gt. tiny ) d(1) = d(1)/e(1)
         if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
         if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
      end if

!     /*   d in laboratory frame                                */

      f(1) = c(1,1)*d(1) + c(1,2)*d(2) + c(1,3)*d(3)
      f(2) = c(2,1)*d(1) + c(2,2)*d(2) + c(2,3)*d(3)
      f(3) = c(3,1)*d(1) + c(3,2)*d(2) + c(3,3)*d(3)

      do k = 2, nbead
      do i = 1, natom
         fm   = fictmass(i,k)
         fux(i,k) = fux(i,k) - f(2)*(uz(i,k))*fm &
     &                       + f(3)*(uy(i,k))*fm
         fuy(i,k) = fuy(i,k) - f(3)*(ux(i,k))*fm &
     &                       + f(1)*(uz(i,k))*fm
         fuz(i,k) = fuz(i,k) - f(1)*(uy(i,k))*fm &
     &                       + f(2)*(ux(i,k))*fm
      end do
      end do

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_force ( 0 )

      return
      end





!***********************************************************************
      subroutine subtract_force_all
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fux, fuy, fuz, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j, k

      real(8) :: sump, sumfx, sumfy, sumfz

!-----------------------------------------------------------------------
!     /*   remove centroid and noncentroid forces                     */
!-----------------------------------------------------------------------

      do k = 1, nbead

         sumfx = 0.d0
         sumfy = 0.d0
         sumfz = 0.d0
         sump  = 0.d0

         do j = 1, natom
            sumfx = sumfx + fux(j,k)
            sumfy = sumfy + fuy(j,k)
            sumfz = sumfz + fuz(j,k)
            sump  = sump  + 1.d0
         end do

         sumfx = sumfx/sump
         sumfy = sumfy/sump
         sumfz = sumfz/sump

         do j = 1, natom
            fux(j,k) = fux(j,k) - sumfx
            fuy(j,k) = fuy(j,k) - sumfy
            fuz(j,k) = fuz(j,k) - sumfz
         end do

      end do

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_force ( 0 )

      return
      end





!***********************************************************************
      subroutine subtract_torque_all
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ux, uy, uz, fux, fuy, fuz, xg, yg, zg, fictmass, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, k

      real(8) :: fm

      real(8), dimension(3,3) :: a, c

      real(8), dimension(3)   :: b, d, e, f

      real(8) :: tiny = 1.d-14

!-----------------------------------------------------------------------
!     /*   center of mass                                             */
!-----------------------------------------------------------------------

      call center_of_mass

!-----------------------------------------------------------------------
!     /*   centroids and centroids                                    */
!-----------------------------------------------------------------------

     do k = 1, nbead

!        /*   torque   */

         b(:) = 0.d0

          do i = 1, natom
            fm   = fictmass(i,k)
            b(1) = b(1) + (uy(i,k)-yg(i))*fuz(i,k) &
     &                  - (uz(i,k)-zg(i))*fuy(i,k)
            b(2) = b(2) + (uz(i,k)-zg(i))*fux(i,k) &
     &                  - (ux(i,k)-xg(i))*fuz(i,k)
            b(3) = b(3) + (ux(i,k)-xg(i))*fuy(i,k) &
     &                  - (uy(i,k)-yg(i))*fux(i,k)
         end do

!        /*   moment of inertia   */

         a(:,:) = 0.d0

          do i = 1, natom
            fm   = fictmass(i,k)
            a(1,1) = a(1,1) + fm*(uy(i,k)-yg(i))*(uy(i,k)-yg(i)) &
     &                      + fm*(uz(i,k)-zg(i))*(uz(i,k)-zg(i))
            a(1,2) = a(1,2) - fm*(ux(i,k)-xg(i))*(uy(i,k)-yg(i))
            a(1,3) = a(1,3) - fm*(ux(i,k)-xg(i))*(uz(i,k)-zg(i))
            a(2,1) = a(2,1) - fm*(uy(i,k)-yg(i))*(ux(i,k)-xg(i))
            a(2,2) = a(2,2) + fm*(uz(i,k)-zg(i))*(uz(i,k)-zg(i)) &
     &                      + fm*(ux(i,k)-xg(i))*(ux(i,k)-xg(i))
            a(2,3) = a(2,3) - fm*(uy(i,k)-yg(i))*(uz(i,k)-zg(i))
            a(3,1) = a(3,1) - fm*(uz(i,k)-zg(i))*(ux(i,k)-xg(i))
            a(3,2) = a(3,2) - fm*(uz(i,k)-zg(i))*(uy(i,k)-yg(i))
            a(3,3) = a(3,3) + fm*(ux(i,k)-xg(i))*(ux(i,k)-xg(i)) &
     &                      + fm*(uy(i,k)-yg(i))*(uy(i,k)-yg(i))
         end do

!        /*   principal axis:  diagonalize moment of inertia   */

         call ddiag( a, e, c, 3 )

         /*   in principal axis:  torque   */

         d(1) = c(1,1)*b(1) + c(2,1)*b(2) + c(3,1)*b(3)
         d(2) = c(1,2)*b(1) + c(2,2)*b(2) + c(3,2)*b(3)
         d(3) = c(1,3)*b(1) + c(2,3)*b(2) + c(3,3)*b(3)

!        /*   d = torque divided by moment of inertia   */

         if ( natom .eq. 1 ) then
            d(1) = 0.d0
            d(2) = 0.d0
            d(3) = 0.d0
         else if ( natom .eq. 2 ) then
            d(1) = 0.d0
            if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
            if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
         else
            if ( e(1) .gt. tiny ) d(1) = d(1)/e(1)
            if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
            if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
         end if

!        /*   d in laboratory frame   */

         f(1) = c(1,1)*d(1) + c(1,2)*d(2) + c(1,3)*d(3)
         f(2) = c(2,1)*d(1) + c(2,2)*d(2) + c(2,3)*d(3)
         f(3) = c(3,1)*d(1) + c(3,2)*d(2) + c(3,3)*d(3)

         do i = 1, natom
            fm   = fictmass(i,k)
            fux(i,k) = fux(i,k) - f(2)*(uz(i,k)-zg(i))*fm &
     &                          + f(3)*(uy(i,k)-yg(i))*fm
            fuy(i,k) = fuy(i,k) - f(3)*(ux(i,k)-xg(i))*fm &
     &                          + f(1)*(uz(i,k)-zg(i))*fm
            fuz(i,k) = fuz(i,k) - f(1)*(uy(i,k)-yg(i))*fm &
     &                          + f(2)*(ux(i,k)-xg(i))*fm
         end do

      end do

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_force ( 0 )

      return
      end





!***********************************************************************
      subroutine subtract_torque_mix
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ux, uy, uz, fux, fuy, fuz, xg, yg, zg, fictmass, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, k

      real(8) :: fm

      real(8), dimension(3,3) :: a, c

      real(8), dimension(3)   :: b, d, e, f

      real(8) :: tiny = 1.d-14

!-----------------------------------------------------------------------
!     /*   center of mass                                             */
!-----------------------------------------------------------------------

      call center_of_mass

!-----------------------------------------------------------------------
!     /*   centroids and centroids                                    */
!-----------------------------------------------------------------------

     do k = 1, nbead

!        /*   torque   */

         b(:) = 0.d0

          do i = 1, natom
            b(1) = b(1) + (uy(i,1)-yg(1))*fuz(i,k) &
     &                  - (uz(i,1)-zg(1))*fuy(i,k)
            b(2) = b(2) + (uz(i,1)-zg(1))*fux(i,k) &
     &                  - (ux(i,1)-xg(1))*fuz(i,k)
            b(3) = b(3) + (ux(i,1)-xg(1))*fuy(i,k) &
     &                  - (uy(i,1)-yg(1))*fux(i,k)
         end do

!        /*   moment of inertia   */

         a(:,:) = 0.d0

          do i = 1, natom
            fm   = fictmass(i,1)
            a(1,1) = a(1,1) + fm*(uy(i,1)-yg(1))*(uy(i,1)-yg(1)) &
     &                      + fm*(uz(i,1)-zg(1))*(uz(i,1)-zg(1))
            a(1,2) = a(1,2) - fm*(ux(i,1)-xg(1))*(uy(i,1)-yg(1))
            a(1,3) = a(1,3) - fm*(ux(i,1)-xg(1))*(uz(i,1)-zg(1))
            a(2,1) = a(2,1) - fm*(uy(i,1)-yg(1))*(ux(i,1)-xg(1))
            a(2,2) = a(2,2) + fm*(uz(i,1)-zg(1))*(uz(i,1)-zg(1)) &
     &                      + fm*(ux(i,1)-xg(1))*(ux(i,1)-xg(1))
            a(2,3) = a(2,3) - fm*(uy(i,1)-yg(1))*(uz(i,1)-zg(1))
            a(3,1) = a(3,1) - fm*(uz(i,1)-zg(1))*(ux(i,1)-xg(1))
            a(3,2) = a(3,2) - fm*(uz(i,1)-zg(1))*(uy(i,1)-yg(1))
            a(3,3) = a(3,3) + fm*(ux(i,1)-xg(1))*(ux(i,1)-xg(1)) &
     &                      + fm*(uy(i,1)-yg(1))*(uy(i,1)-yg(1))
         end do

!        /*   principal axis:  diagonalize moment of inertia   */

         call ddiag( a, e, c, 3 )

         /*   in principal axis:  torque   */

         d(1) = c(1,1)*b(1) + c(2,1)*b(2) + c(3,1)*b(3)
         d(2) = c(1,2)*b(1) + c(2,2)*b(2) + c(3,2)*b(3)
         d(3) = c(1,3)*b(1) + c(2,3)*b(2) + c(3,3)*b(3)

!        /*   d = torque divided by moment of inertia   */

         if ( natom .eq. 1 ) then
            d(1) = 0.d0
            d(2) = 0.d0
            d(3) = 0.d0
         else if ( natom .eq. 2 ) then
            d(1) = 0.d0
            if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
            if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
         else
            if ( e(1) .gt. tiny ) d(1) = d(1)/e(1)
            if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
            if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
         end if

!        /*   d in laboratory frame   */

         f(1) = c(1,1)*d(1) + c(1,2)*d(2) + c(1,3)*d(3)
         f(2) = c(2,1)*d(1) + c(2,2)*d(2) + c(2,3)*d(3)
         f(3) = c(3,1)*d(1) + c(3,2)*d(2) + c(3,3)*d(3)

         do i = 1, natom
            fm   = fictmass(i,1)
            fux(i,k) = fux(i,k) - f(2)*(uz(i,1)-zg(1))*fm &
     &                          + f(3)*(uy(i,1)-yg(1))*fm
            fuy(i,k) = fuy(i,k) - f(3)*(ux(i,1)-xg(1))*fm &
     &                          + f(1)*(uz(i,1)-zg(1))*fm
            fuz(i,k) = fuz(i,k) - f(1)*(uy(i,1)-yg(1))*fm &
     &                          + f(2)*(ux(i,1)-xg(1))*fm
         end do

      end do

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_force ( 0 )

      return
      end
