!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 25, 2018 by M. Shiga
!      Description:     convert tinker input file to pimd input files
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      module tinker_variables
!***********************************************************************

!     //   initialize variables
      implicit none

!     //   number of atoms
      integer :: natom

!     //   number of atoms
      integer :: nbead = 1

!     //   file unit: input file
      integer :: iounit = 10

!     //   atomic coordinates
      real(8), dimension(:,:), allocatable :: x, y, z

!     //   atomic forces
      real(8), dimension(:,:), allocatable :: fx, fy, fz

!     //   energies
      real(8), dimension(:), allocatable :: pot

!     //   virial
      real(8), dimension(3,3) :: vir

!     //   conversion factor: from bohr to m
      real(8), parameter:: au_length = 0.529177249d-10

!     //   conversion factor: from au to kg
      real(8), parameter:: au_mass   = 9.1093897d-31

!     //   conversion factor: from hartree to Joule
      real(8), parameter:: au_energy = 4.3597482d-18

!     //   conversion factor: from au to second
      real(8), parameter:: au_time   = 0.024188843d-15

!     //   conversion factor: from au to Coulomb
      real(8), parameter:: au_charge = 1.60217646d-19

!     //   conversion factor: Avogadro number
      real(8), parameter:: avogadro  = 6.02214129e+23

!     //   circular constant
      real(8) :: pi

!     //   boundary condition
      integer :: iboundary = 0

!     //   simulation box
      real(8), dimension(3,3) :: box

!     //   inverse of box
      real(8), dimension(3,3) :: boxinv

!     //   lj parameter
      real(8), dimension(3) :: nbox_lj

!     //   number of bond connections
      integer :: nbond = 0

!     //   number of linear bonds
      integer :: nlin = 0

!     //   number of angular bonds
      integer :: nangl = 0

!     //   number of urey-bradley pairs
      integer :: nub = 0

!     //   number of torsion x 3 + improper torsion
      integer :: ndih = 0

!     //   number of torsion bonds
      integer :: ntors = 0

!     //   number of torsion terms ( = ntors x 3 )
      integer :: ntors_3 = 0

!     //   number of improper bonds
      integer :: nimproper = 0

!     //   number of improper torsion bonds
      integer :: nimptors = 0

!     //   number of trigonals
      integer :: ntri = 0

!     //   number of bonded charge pairs
      integer :: nbcp = 0

!     //   number of coulomb interactions
      integer :: ncoulomb = 0

!     //   number of lennard-jones pairs
      integer :: nlj = 0

!     //   bond topology matrix, 1-2
      integer, dimension(:,:), allocatable :: bond

!     //   bond topology matrix, 1-3
      integer, dimension(:,:), allocatable :: bond_13

!     //   bond topology matrix, 1-4
      integer, dimension(:,:), allocatable :: bond_14

!     //   linear bonds: i and j atoms
      integer, dimension(:), allocatable :: i_lin
      integer, dimension(:), allocatable :: j_lin

!     //   1-3: i and j atoms
      integer, dimension(:), allocatable :: i_13
      integer, dimension(:), allocatable :: j_13

!     //   1-4: i and j atoms
      integer, dimension(:), allocatable :: i_14
      integer, dimension(:), allocatable :: j_14

!     //   angular bonds: i, j and k atoms
      integer, dimension(:), allocatable :: i_angl
      integer, dimension(:), allocatable :: j_angl
      integer, dimension(:), allocatable :: k_angl

!     //   urey-bradley bonds: i, j and k atoms
      integer, dimension(:), allocatable :: i_ub
      integer, dimension(:), allocatable :: j_ub
      integer, dimension(:), allocatable :: k_ub

!     //   torsion bonds: i, j, k and l atoms
      integer, dimension(:), allocatable :: i_tors
      integer, dimension(:), allocatable :: j_tors
      integer, dimension(:), allocatable :: k_tors
      integer, dimension(:), allocatable :: l_tors

!     //   trigonals: i, j, k and l atoms
      integer, dimension(:), allocatable :: i_tri
      integer, dimension(:), allocatable :: j_tri
      integer, dimension(:), allocatable :: k_tri
      integer, dimension(:), allocatable :: l_tri

!     //   improper torsion: i, j, k and l atoms
      integer, dimension(:), allocatable :: i_imptors
      integer, dimension(:), allocatable :: j_imptors
      integer, dimension(:), allocatable :: k_imptors
      integer, dimension(:), allocatable :: l_imptors

!     //   all dihedral bonds: i, j, k and l atoms
      integer, dimension(:), allocatable :: i_dih
      integer, dimension(:), allocatable :: j_dih
      integer, dimension(:), allocatable :: k_dih
      integer, dimension(:), allocatable :: l_dih

!     //   improper dihedral bonds: i, j, k and l atoms
      integer, dimension(:), allocatable :: i_improper
      integer, dimension(:), allocatable :: j_improper
      integer, dimension(:), allocatable :: k_improper
      integer, dimension(:), allocatable :: l_improper

!     //   bonded charge pairs: i and j atoms
      integer, dimension(:), allocatable :: i_bcp
      integer, dimension(:), allocatable :: j_bcp

!     //   bonded charge pairs: scaling factor
      real(8), dimension(:), allocatable :: f_bcp

!     //   linear bonds: equilibrium bond lengths
      real(8), dimension(:), allocatable :: eq_lin

!     //   linear bonds: force constants
      real(8), dimension(:), allocatable :: fc_lin

!     //   angular bonds: equilibrium bond angles
      real(8), dimension(:), allocatable :: eq_angl

!     //   angular bonds: force constants
      real(8), dimension(:), allocatable :: fc_angl

!     //   urey-bradley bonds: equilibrium length
      real(8), dimension(:), allocatable :: eq_ub

!     //   urey-bradley bonds: force constants
      real(8), dimension(:), allocatable :: fc_ub

!     //   torsion bonds: rotational energy barriers
      real(8), dimension(:), allocatable :: v_tors

!     //   torsion bonds: phase factor
      integer, dimension(:), allocatable :: mu_tors

!     //   torsion bonds: degeneracy
      integer, dimension(:), allocatable :: nu_tors

!     //   improper torsion: rotational energy barriers
      real(8), dimension(:), allocatable :: v_imptors

!     //   improper torsion: phase factor
      integer, dimension(:), allocatable :: mu_imptors

!     //   improper torsion: degeneracy
      integer, dimension(:), allocatable :: nu_imptors

!     //   dihedral bonds: rotational energy barriers
      real(8), dimension(:), allocatable :: v_dih

!     //   dihedral bonds: phase factor
      integer, dimension(:), allocatable :: mu_dih

!     //   dihedral bonds: degeneracy
      integer, dimension(:), allocatable :: nu_dih

!     //   improper dihedral bonds: force constant
      real(8), dimension(:), allocatable :: fc_improper

!     //   improper dihedral bonds: angle
      real(8), dimension(:), allocatable :: eq_improper

!     //   charge
      real(8), dimension(:), allocatable :: q

!     //   lennard-jones: sigma
      real(8), dimension(:,:), allocatable :: sig

!     //   lennard-jones: epsilon
      real(8), dimension(:,:), allocatable :: eps

!     //   net charge of the system
      real(8) :: qsum

!     //   charges
      real(8), dimension(:), allocatable :: q_param

!     //   lennard-jones: sigma
      real(8), dimension(:), allocatable :: sig_param

!     //   lennard-jones: epsilon
      real(8), dimension(:), allocatable :: eps_param

!     //   lennard-jones: sigma 1-4
      real(8), dimension(:), allocatable :: sig14_param

!     //   lennard-jones: epsilon 1-4
      real(8), dimension(:), allocatable :: eps14_param

!     //   lennard-jones: atom i
      integer, dimension(:), allocatable :: i_lj

!     //   lennard-jones: atom j
      integer, dimension(:), allocatable :: j_lj

!     //   lennard-jones: epsilon
      real(8), dimension(:), allocatable :: eps_lj

!     //   lennard-jones: sigma
      real(8), dimension(:), allocatable :: sig_lj

!     //   file unit: tinker parameter file
      integer :: iounit_tinker = 11

!     //   file unit: pimd input file mm.dat
      integer :: iounit_pimm = 12

!     //   file unit: pimd geometry file centroid.dat
      integer :: iounit_geom = 13

!     //   file unit: centroid.xyz
      integer :: iounit_xyz = 14

!     //   file unit: pimd input file input.dat
      integer :: iounit_piin = 15

!     //   kind of atom
      integer, dimension(:), allocatable :: jkind

!     //   number of atom kinds
      integer :: nkind

!     //   atom kinds
      integer, dimension(:), allocatable :: ikind

!     //   bonds
      integer :: jbond(4)

!     //   parameters for linear bonds
      real(8), dimension(:,:), allocatable :: eq_lin_kind

!     //   parameters for linear bonds
      real(8), dimension(:,:), allocatable :: fc_lin_kind

!     //   lj cutoff parameters (in bohr):
!     //   switching function (swf) is applied to lj interaction.
!     //   swf is 1 up to the radius rin_lj, decreases to 0 as
!     //   approaching the radius rout_lj, and remains 0 onwards.
!     //   this swf is multiplied to the original lj function.

!     //   LJ cutoff distance (inside)
      real(8) :: rin_lj

!     //   LJ cutoff distance (outside)
      real(8) :: rout_lj

!     //   LJ cutoff distance (inside)
      character(len=32) :: char_rin_lj

!     //   LJ cutoff distance (outside)
      character(len=32) :: char_rout_lj

!     //   parameter file
      character(len=80) :: inputfile

!     //   input file
      character(len=80) :: paramfile

!     //   PIMD MM file
      character(len=80) :: pimmfile

!     //   input style
      character(len=80) :: input_style

!     //   xyz file
      character(len=80) :: xyzfile

!     //   pimd input file
      character(len=80) :: piinfile

!     //   tinker keywords:
      character(len=25) :: forcefield  = 'oplsaa.prm    '
      character(len=25) :: vdwindex    = 'CLASS         '
      character(len=25) :: vdwtype     = 'LENNARD-JONES '
      character(len=25) :: radiusrule  = 'ARITHMETIC    '
      character(len=25) :: radiustype  = 'R-MIN         '
      character(len=25) :: radiussize  = 'RADIUS        '
      character(len=25) :: epsilonrule = 'GEOMETRIC     '

!     //   tinker keywords:

      real(8) :: torsionunit = 1.d0
      real(8) :: imptorunit  = 1.d0
      real(8) :: impropunit  = 1.d0
      real(8) :: vdw14scale  = 1.d0
      real(8) :: chg14scale  = 1.d0
      real(8) :: bondunit    = 1.d0
      real(8) :: angleunit   = 1.d0

!     //   scaling factors

      real(8) :: scale_sigma         = 1.d0
      real(8) :: scale_tors          = 1.d0
      real(8) :: scale_imptors       = 1.d0
      real(8) :: scale_improper      = 1.d0
      real(8) :: scale_q_14          = 1.d0
      real(8) :: scale_eps_14        = 1.d0
      real(8) :: scale_sig_14        = 1.d0

!     //    energy decomposition

      real(8) :: e_lin
      real(8) :: e_bond
      real(8) :: e_ub
      real(8) :: e_angl
      real(8) :: e_dih
      real(8) :: e_tors
      real(8) :: e_imptors
      real(8) :: e_improper
      real(8) :: e_cmap
      real(8) :: e_coulomb
      real(8) :: e_coulomb_cor
      real(8) :: e_lj
      real(8) :: e_total
      real(8) :: e_total_cor

!     //   conversion factor: from bohr to angstrom
      real(8) :: bohr2ang

!     //   conversion factor: from hartree to kcal per mol
      real(8) :: har2kcal

!     //   conversion factor: from degree to radian
      real(8) :: deg2rad

!     //   atomic mass in amu
      real(8), dimension(:), allocatable :: amumass_kind

!     //   definition of atomic kind
      character(len=80), dimension(:), allocatable :: def_kind

!     //   bond order of atomic kind
      integer, dimension(:), allocatable :: nbo_kind

!     //   atomic number
      integer, dimension(:), allocatable :: numbre_kind

!     //   symbol of atomic kind
      character(len=3), dimension(:), allocatable :: symbol_kind

!     //   atomic mass in amu
      real(8), dimension(:), allocatable :: amumass

!     //   definition of atomic kind
      character(len=80), dimension(:), allocatable :: def

!     //   bond order of atomic kind
      integer, dimension(:), allocatable :: nbo

!     //   atomic number
      integer, dimension(:), allocatable :: numbre

!     //   atomic species
      integer, dimension(:), allocatable :: ikind_numbre

!     //   symbol of atomic kind
      character(len=3), dimension(:), allocatable :: symbol

!     //   atomic symbol
      character(len=2), dimension(:), allocatable :: symbol_numbre

!     //   default value of electric constant from mm force field
      real(8) :: econst_mm

!     //   electric constant calculated from atomic units
      real(8) :: econst_au

!     //   electric constant ( = econst_mm / econst_au )
      real(8) :: econst_ratio

!     //   number of lj nonbonded pairs
      integer :: nljpair

!     //   lj nonbonded pair: atom i
      integer, dimension(:), allocatable :: i_ljpair

!     //   lj nonbonded pair parameters: epsilon
      real(8), dimension(:), allocatable :: eps_ljpair

!     //   lj nonbonded pair parameters: sigma
      real(8), dimension(:), allocatable :: sig_ljpair

!     //   number of lj bonded pairs
      integer :: nljbond
      integer :: nljbond_book
      integer :: nljbond_correct

!     //   bonded pairs
      integer, dimension(:,:), allocatable :: bond_book

!     //   lj bonded pair: atom i
      integer, dimension(:), allocatable :: i_ljbond

!     //   lj bonded pair: atom j
      integer, dimension(:), allocatable :: j_ljbond

!     //   lj bonded pair parameters: epsilon bonded
      real(8), dimension(:), allocatable :: eps_ljbond

!     //   lj bonded pair parameters: sigma bonded
      real(8), dimension(:), allocatable :: sig_ljbond

!     //   lj bonded pair parameters: epsilon non-bonded
      real(8), dimension(:), allocatable :: eps_ljnonb

!     //   lj bonded pair parameters: sigma non-bonded
      real(8), dimension(:), allocatable :: sig_ljnonb

!     //   number of reference cmaps
      integer :: nref_cmap

!     //   number of cmaps kinds
      integer :: nkind_cmap

!     //   number of cmaps
      integer :: ncmap

!     //   cmap grids
      integer, parameter :: ngrid_cmap = 24
      integer, parameter :: nbuff_cmap = 8
      integer :: mgrid_cmap = ngrid_cmap + nbuff_cmap

!     //   cmap kind
      integer, dimension(:,:), allocatable :: ikind_cmap

!     //   cmap list
      integer, dimension(:,:), allocatable :: i_cmap
      integer, dimension(:,:), allocatable :: j_cmap
      integer, dimension(:,:), allocatable :: k_cmap
      integer, dimension(:,:), allocatable :: l_cmap
      integer, dimension(:), allocatable :: m_cmap

!     //   existence of cmap
      logical, dimension(:), allocatable :: exist_cmap

!     //   cmap grid potential
      real(8), dimension(:,:,:), allocatable :: vref_cmap
      real(8), dimension(:,:,:), allocatable :: vgrid_cmap
      real(8), dimension(:,:,:), allocatable :: v2grid_cmap

!     //   grids
      real(8), dimension(:), allocatable :: xgrid_cmap
      real(8), dimension(:), allocatable :: ygrid_cmap

!***********************************************************************
      end module tinker_variables
!***********************************************************************






!***********************************************************************
      program convert_tinker
!***********************************************************************

!----------------------------------------------------------------------
!     //   definition of variables
!----------------------------------------------------------------------

!     //   variables
      use tinker_variables, only : &
     &   au_length, bohr2ang, har2kcal, au_energy, avogadro, pi, &
     &   deg2rad, econst_au, econst_mm, rin_lj, rout_lj, pot, &
     &   e_bond, e_angl, e_dih, e_tors, e_imptors, e_improper, e_ub, &
     &   e_lj, e_coulomb_cor, e_coulomb, econst_ratio, e_total, e_cmap, &
     &   e_total_cor, forcefield, vdwindex, vdwtype, radiusrule, &
     &   radiustype, radiussize, epsilonrule, bondunit, angleunit, &
     &   torsionunit, impropunit, imptorunit, vdw14scale, chg14scale, &
     &   scale_sigma, scale_eps_14, scale_q_14, scale_tors, &
     &   scale_imptors, scale_improper, inputfile, paramfile, iounit, &
     &   iounit_pimm, iounit_tinker, iounit_xyz, iounit_piin, nlin, &
     &   nangl, nimproper, nlj, nbond, nub, ndih, ntors, ntors_3, &
     &   nimptors, ncoulomb, ncmap, char_rin_lj, char_rout_lj, &
     &   pimmfile, input_style, xyzfile, piinfile, iounit_geom

!     //   local variables
      implicit none

!     //   argument
      integer :: iargc

!----------------------------------------------------------------------
!     //   mathematical and physical constants
!----------------------------------------------------------------------

!     //   circular constant
      pi = acos(-1.d0)

!     //   from bohr to angstrom
      bohr2ang = au_length / 1.d-10

!     //   from hartree to kcal/mol
      har2kcal = au_energy / 1000.d0 * avogadro / 4.184d0

!     //   from degree to radian
      deg2rad = pi / 180.d0

!     //   electric constant calculated from atomic units
      econst_au = har2kcal*bohr2ang

!     //   default value of electric constant from mm force field
      econst_mm = econst_au

!----------------------------------------------------------------------
!     //   open input file
!----------------------------------------------------------------------

!     //   error termination
      if ( iargc() .ne. 8 ) then

!        //   print usage
         write( 6, '(a)' )
         write( 6, '(a)' ) &
     &      "Usage: convert_tinker.x $1 $2 $3 $4 $5 $6 $7 $8"
         write( 6, '(a)' )
         write( 6, '(a)' ) &
     &      "$1:  INPUT - Name of input file:     ex) " // &
     &      "'structure.txyz'"
         write( 6, '(a)' ) &
     &      "$2:  INPUT - Tinker parameter file:  ex) " // &
     &      "'oplsaa.prm'"
         write( 6, '(a)' ) &
     &      "$3:  INPUT - Inner LJ cutoff [bohr]: ex) " // &
     &      "'20.0'"
         write( 6, '(a)' ) &
     &      "$4:  INPUT - Outer LJ cutoff [bohr]: ex) " // &
     &      "'25.0'"
         write( 6, '(a)' ) &
     &      "$5: OUTPUT - MM force field file:    ex) " // &
     &      "'mm.dat'"
         write( 6, '(a)' ) &
     &      "$6: OUTPUT - Geometry file (old):    ex) " // &
     &      "'centroid.dat'"
         write( 6, '(a)' ) &
     &      "$7: OUTPUT - Geometry file (new):    ex) " // &
     &      "'structure.dat'"
         write( 6, '(a)' ) &
     &      "$8: OUTPUT - PIMD input file:        ex) " // &
     &      "'input.dat'"
         write( 6, '(a)' )

!        //   stop
         stop

!     //   error termination
      end if

!     //   get string from the command line
      call getarg( 1, inputfile )

!     //   get string from the command line
      call getarg( 2, paramfile )

!     //   get string from the command line
      call getarg( 3, char_rin_lj )

!     //   get string from the command line
      call getarg( 4, char_rout_lj )

!     //   get string from the command line
      call getarg( 5, pimmfile )

!     //   get string from the command line
      call getarg( 6, input_style )

!     //   get string from the command line
      call getarg( 7, xyzfile )

!     //   get string from the command line
      call getarg( 8, piinfile )

!     //   LJ cut off parameters
      read( char_rin_lj, * ) rin_lj

!     //   LJ cut off parameters
      read( char_rout_lj, * ) rout_lj

!     //   comments
      write( 6, '(a)' )
      write( 6, '(a,a)' )     'Input file:      ', trim(inputfile)
      write( 6, '(a,a)' )     'Parameter file:  ', trim(paramfile)
      write( 6, '(a,a)' )     'PIMD MM file:    ', trim(pimmfile)
      write( 6, '(a,a)' )     'Geometry file:   ', trim(input_style)
      write( 6, '(a,a)' )     'xyz file:        ', trim(xyzfile)
      write( 6, '(a,a)' )     'PIMD input file: ', trim(piinfile)
      write( 6, '(a)' )

!----------------------------------------------------------------------
!     //   open files
!----------------------------------------------------------------------

!     //   input file
      open ( iounit, file = inputfile )

!     //   PIMD MM file
      open ( iounit_pimm, file = pimmfile )

!     //   data file
      open ( iounit_tinker, file = paramfile )

!     //   geometry file
      open ( iounit_geom, file = input_style )

!     //   xyz file
      open ( iounit_xyz, file = xyzfile )

!     //   PIMD input file
      open ( iounit_piin, file = piinfile )

!----------------------------------------------------------------------
!     //   tinker parameter file
!----------------------------------------------------------------------

!     //   comment
      write( 6, '(a)' ) 'Reading tinker file.'

!     //   read tinker file
      call init_tinker

!----------------------------------------------------------------------
!     //   read input file
!----------------------------------------------------------------------

!     //   comment
      write( 6, '(a)' ) 'Reading input file.'

!     //   read input file
      call read_input

!----------------------------------------------------------------------
!     //   print out
!----------------------------------------------------------------------

!     //  comments
      write( 6, '(a)' )
      write( 6, '(a,a)' )    'force field:  ', forcefield
      write( 6, '(a,a)' )    'vdwindex:     ', vdwindex
      write( 6, '(a,a)' )    'vdwtype:      ', vdwtype
      write( 6, '(a,a)' )    'radiusrule:   ', radiusrule
      write( 6, '(a,a)' )    'radiustype:   ', radiustype
      write( 6, '(a,a)' )    'radiussize:   ', radiussize
      write( 6, '(a,a)' )    'epsilonrule:  ', epsilonrule
      write( 6, '(a,f5.2)' ) 'bondunit:     ', bondunit
      write( 6, '(a,f5.2)' ) 'angleunit:    ', angleunit
      write( 6, '(a,f5.2)' ) 'torsionunit:  ', torsionunit
      write( 6, '(a,f5.2)' ) 'impropunit:   ', impropunit
      write( 6, '(a,f5.2)' ) 'imptorunit:   ', imptorunit
      write( 6, '(a,f5.2)' ) 'vdw-14-scale: ', vdw14scale
      write( 6, '(a,f5.2)' ) 'chg-14-scale: ', chg14scale
      write( 6, '(a)' )
      write( 6, '(a)' )      'scaling factors:'
      write( 6, '(a,f5.2)' ) 'sigma:        ', scale_sigma
      write( 6, '(a,f5.2)' ) 'eps_14:       ', scale_eps_14
      write( 6, '(a,f5.2)' ) 'q_14:         ', scale_q_14
      write( 6, '(a,f5.2)' ) 'torsion:      ', scale_tors
      write( 6, '(a,f5.2)' ) 'imp. torsion: ', scale_imptors
      write( 6, '(a,f5.2)' ) 'improper:     ', scale_improper
      write( 6, '(a)' )

      write( 6, '(a,f10.6)' )'econst (au):  ', econst_au
      write( 6, '(a,f10.6)' )'econst (mm):  ', econst_mm
      write( 6, '(a,f10.6)' )'ratio:        ', econst_ratio
      write( 6, '(a)' )

!----------------------------------------------------------------------
!     //   linear bonds
!----------------------------------------------------------------------

!     //   comment
      write( 6, '(a)' ) 'Looking for linear bond parameters.'

!     //   identify linear bonds
      call lin_tinker

!----------------------------------------------------------------------
!     //   angular bonds
!----------------------------------------------------------------------

!     //    comment
      write( 6, '(a)' ) 'Looking for angular bond parameters.'

!     //   identify angular bonds
      call angl_tinker

!----------------------------------------------------------------------
!     //   angular bonds
!----------------------------------------------------------------------

!     //    comment
      write( 6, '(a)' ) 'Looking for urey-bradley parameters.'

!     //   identify angular bonds
      call ub_tinker

!----------------------------------------------------------------------
!     //   total number of linear bonds
!----------------------------------------------------------------------

      nlin = nbond + nub

!----------------------------------------------------------------------
!     //   torsion bonds
!----------------------------------------------------------------------

!     //   comment
      write( 6, '(a)' ) 'Looking for torsion parameters.'

!     //   identify torsion bonds
      call tors_tinker

!----------------------------------------------------------------------
!     //   improper torsion bonds
!----------------------------------------------------------------------

!     //   comment
      write( 6, '(a)' ) 'Looking for improper torsion parameters.'

!     //   identify improper torsion bonds
      call imptors_tinker

!-----------------------------------------------------------------------
!     //   total number of dihedral bonds
!-----------------------------------------------------------------------

      ndih = ntors_3 + nimptors

!----------------------------------------------------------------------
!     //   improper dihedral bonds
!----------------------------------------------------------------------

!     //   comment
      write( 6, '(a)' ) 'Looking for improper dihedral parameters.'

!     //   identify improper dihedral bonds
      call improper_tinker

!----------------------------------------------------------------------
!     //   cmap
!----------------------------------------------------------------------

!     //   comment
      write( 6, '(a)' ) 'Looking for cmap parameters.'

!     //   identify cmap
      call cmap_tinker

!----------------------------------------------------------------------
!     //   charges
!----------------------------------------------------------------------

!     //   comment
      write( 6, '(a)' ) 'Looking for charge parameters.'

!     //   identify charges
      call charge_tinker

!----------------------------------------------------------------------
!     //   bonded charge pairs
!----------------------------------------------------------------------

!     //   comment
      write( 6, '(a)' ) 'Looking for bonded charge pairs.'

!     //   identify bonded charge pairs
      call bcp_tinker

!----------------------------------------------------------------------
!     //   lennard-jones pairs
!----------------------------------------------------------------------

!     //   comment
      write( 6, '(a)' ) 'Looking for Lennard-Jones parameters.'

!     //   identify Lennard-Jones pairs
      call ljpair_tinker

!     //   identify Lennard-Jones pairs
      call lj_tinker

!----------------------------------------------------------------------
!     //   output
!----------------------------------------------------------------------

!     //   create output pimmfile = 'mm.dat'
      call write_output

!     //   comment
      write( 6, '(a,a)' ) 'PIMD MM created:    ', trim(pimmfile)
      write( 6, '(a,a)' ) 'PIMD input created: ', trim(input_style)
      write( 6, * )

!----------------------------------------------------------------------
!     //   check energy decomposition
!----------------------------------------------------------------------

      pot(:)     = 0.d0
      call check_force_mm_bond
      e_bond     = pot(1)

      pot(:)     = 0.d0
      call check_force_mm_ub
      e_ub       = pot(1)

      pot(:)     = 0.d0
      call check_force_mm_angl
      e_angl     = pot(1)

      pot(:)     = 0.d0
      call check_force_mm_tors
      e_tors     = pot(1)

      pot(:)     = 0.d0
      call check_force_mm_imptors
      e_imptors  = pot(1)

      pot(:)     = 0.d0
      call check_force_mm_improper
      e_improper = pot(1)

      pot(:)     = 0.d0
      call check_force_mm_cmap
      e_cmap    = pot(1)

      pot(:)     = 0.d0
      call check_force_mm_lj
      e_lj       = pot(1)

      pot(:)     = 0.d0
      call check_force_mm_coulomb
      e_coulomb  = pot(1)

      e_coulomb_cor = e_coulomb * econst_ratio

!     //   total potential

      e_total = e_bond + e_ub + e_angl + e_tors + e_imptors &
     &        + e_improper + e_cmap + e_lj + e_coulomb

!     //   with electrostatic correction of electric constant

      e_total_cor = e_bond + e_ub + e_angl + e_tors + e_imptors &
     &        + e_improper + e_lj + e_coulomb_cor

!     //   dihedral contribution

      e_dih = e_tors + e_imptors

!     //   print out

      write( 6, '(a)' ) &
     &    '                          hartree    kcal/mol'

      write( 6, '(a,f12.6,f12.4)' ) &
     &    'Total Energy:        ', &
     &    e_total, e_total*har2kcal

      write( 6, '(a,f12.6,f12.4)' ) &
     &    'TE + es correction:  ', &
     &    e_total_cor, e_total_cor*har2kcal
      write( 6, '(a)' )

      write( 6, '(a)' ) &
     &    'Energy Decomposition:     hartree    kcal/mol  counts'

      write( 6, '(a,f12.6,f12.4,i8)' ) &
     &    'Bond Stretching:     ', &
     &    e_bond, e_bond*har2kcal, nbond

      write( 6, '(a,f12.6,f12.4,i8)' ) &
     &    'Angle Bending:       ', &
     &    e_angl, e_angl*har2kcal, nangl

      write( 6, '(a,f12.6,f12.4,i8)' ) &
     &    'Urey-Bradley:        ', &
     &    e_ub, e_ub*har2kcal, nub

      write( 6, '(a,f12.6,f12.4,i8)' ) &
     &    'Improper Dihedral:   ', &
     &    e_improper, e_improper*har2kcal, nimproper

      write( 6, '(a,f12.6,f12.4,i8)' ) &
     &    'Torsional Angle:     ', &
     &    e_tors, e_tors*har2kcal, ntors

      write( 6, '(a,f12.6,f12.4,i8)' ) &
     &    'Improper Torsion:    ', &
     &    e_imptors, e_imptors*har2kcal, nimptors

      write( 6, '(a,f12.6,f12.4,i8)' ) &
     &    'Cmap:                ', &
     &    e_cmap, e_cmap*har2kcal, ncmap

      write( 6, '(a,f12.6,f12.4,i8)' ) &
     &    'Van der Waals:       ', &
     &    e_lj, e_lj*har2kcal, nlj

      write( 6, '(a,f12.6,f12.4,i8)' ) &
     &    'Charge-Charge:       ', &
     &    e_coulomb, e_coulomb*har2kcal, ncoulomb

      write( 6, '(a,f12.6,f12.4,i8)' ) &
     &    'C-C + es correction: ', &
     &    e_coulomb_cor, e_coulomb_cor*har2kcal, ncoulomb

!----------------------------------------------------------------------
!     //   close files
!----------------------------------------------------------------------

!     //   input file
      close ( iounit )

!     //   data file
      close ( iounit_tinker )

!     //   PIMD MM file
      close ( iounit_pimm )

!     //   geometry file
      close ( iounit_geom )

!     //   xyz file
      close ( iounit_xyz )

!     //   PIMD input file
      close ( iounit_piin )

!----------------------------------------------------------------------
!     //   final comment
!----------------------------------------------------------------------

!     //   comment
      write( 6, '(a)' )
      write( 6, '(a)' ) 'Normal termination.'
      write( 6, '(a)' )

!-----------------------------------------------------------------------
!     //   end of program
!-----------------------------------------------------------------------

      stop
      end





!***********************************************************************
      subroutine read_input
!***********************************************************************

!     //   variables
      use tinker_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, eq_lin, fc_lin, eq_angl, &
     &   fc_angl, eq_ub, fc_ub, v_tors, fc_improper, eq_improper, &
     &   v_imptors, v_dih, amumass_kind, q, sig, eps, sig_lj, eps_lj, &
     &   amumass, nbo, def, numbre, nbo_kind, def_kind, nbond, &
     &   jkind, bond, bond_14, bond_13, i_lj, j_lj, natom, iounit, &
     &   symbol_kind, symbol, i_angl, j_angl, k_angl, jbond, i_13, &
     &   j_13, i_ub, j_ub, k_ub, ntors_3, ntors, ntri, i_14, j_14, &
     &   i_tors, j_tors, k_tors, l_tors, mu_tors, nu_tors, nbead, &
     &   i_improper, j_improper, k_improper, l_improper, nangl, &
     &   i_imptors, j_imptors, k_imptors, l_imptors, mu_imptors, &
     &   nu_imptors, i_dih, j_dih, k_dih, l_dih, mu_dih, nu_dih, &
     &   i_tri, j_tri, k_tri, l_tri, i_lin, j_lin, numbre_kind, &
     &   symbol_numbre, ikind_numbre

!     //   local variables
      implicit none

!     //   error flag
      integer :: ierr = 0

!     //   error flag
      integer :: jerr = 0

!     //   integers
      integer :: i, j, k, l, m, ijk, ijkl

!     //   line
      character(len=100) :: char_line

!----------------------------------------------------------------------
!     //   read number of atoms
!----------------------------------------------------------------------

!     //   read number of atoms
      read ( iounit, *, iostat=ierr ) natom

!     //   error handling
      call error_handling( ierr, 'read_input', 10 )

!----------------------------------------------------------------------
!     //   memory allocation
!----------------------------------------------------------------------

!     //   atomic coordinates
      allocate( x(natom,nbead) )
      allocate( y(natom,nbead) )
      allocate( z(natom,nbead) )

!     //   atomic forces
      allocate( fx(natom,nbead) )
      allocate( fy(natom,nbead) )
      allocate( fz(natom,nbead) )

!     //   potential
      allocate( pot(nbead) )

!     //   atom symbols
      allocate( symbol(natom) )

!     //   atom kinds
      allocate( jkind(natom) )

!     //   bond connections
      allocate( bond(natom,natom) )

!     //   1-3 connections
      allocate( bond_13(natom,natom) )

!     //   1-4 connections
      allocate( bond_14(natom,natom) )

!     //   atomic charges
      allocate( q(natom) )

!     //   lennard-jones: sigma
      allocate( sig(natom,natom) )

!     //   lennard-jones: epsilon
      allocate( eps(natom,natom) )

!     //   lennard-jones: atom i
      allocate( i_lj(natom*(natom-1)/2) )

!     //   lennard-jones: atom i
      allocate( j_lj(natom*(natom-1)/2) )

!     //   lennard-jones: sigma
      allocate( sig_lj(natom*(natom-1)/2) )

!     //   lennard-jones: epsilon
      allocate( eps_lj(natom*(natom-1)/2) )

!     //   atomic mass in amu
      allocate( amumass(natom) )

!     //   definition of atomic kind
      allocate( def(natom) )

!     //   bond order
      allocate( nbo(natom) )

!     //   atomic number
      allocate( numbre(natom) )

!     //   atomic symbol
      allocate( symbol_numbre(natom) )

!     //   atomic species
      allocate( ikind_numbre(natom) )

!----------------------------------------------------------------------
!     //   initialization
!----------------------------------------------------------------------

!     //   potential
      pot(:) = 0.d0

!     //   force
      fx(:,:) = 0.d0
      fy(:,:) = 0.d0
      fz(:,:) = 0.d0

!     //   virial
      vir(:,:) = 0.d0

!----------------------------------------------------------------------
!     //   initialize bond topology
!----------------------------------------------------------------------

!     //   1-2 bonds
      bond(:,:) = 0

!     //   1-3 bonds
      bond_13(:,:) = 0

!     //   1-4 bonds
      bond_14(:,:) = 0

!----------------------------------------------------------------------
!     //   read atoms and bonds
!----------------------------------------------------------------------

!     //   add blank line
      write( 6, '(a)' )

!     //   read number of atoms
      rewind ( iounit )

!     //   read number of atoms
      read ( iounit, *, iostat=ierr ) i

!     //   counter reset
      m = 0

!     //   loop of atoms
      do i = 1, natom

!        //   read atom numbers, atom symbols, atom kinds
         read( iounit, '(a)', iostat=jerr ) char_line

!        //   error handling
         if ( jerr .ne. 0 ) ierr = jerr

!        //   initialize bonds
         jbond(:) = 0

!        //   output
!        write( 6, '(a)' ) char_line

!        //   read atom numbers, atom symbols, atom kinds
         read( char_line, *, iostat=jerr ) &
     &      j, symbol(j), x(j,1), y(j,1), z(j,1), jkind(j), jbond(1:4)

!        //   atomic mass in amu
         amumass(j) = amumass_kind(jkind(j))

!        //   definition of atomic kind
         def(j) = def_kind(jkind(j))

!        //   bond order
         nbo(j) = nbo_kind(jkind(j))

!        //   atomic number
         numbre(j) = numbre_kind(jkind(j))

!        //   atomic symbol
         call numbre_to_symbol &
     &      ( numbre(j), symbol_numbre(j), ikind_numbre(j) )

          write( 6, '(a,i8,a,a,a,a,a)' ) &
     &       'Atom ', j, ' :  type ', &
     &       trim(symbol(j)), ',  kind ', &
     &       trim(symbol_kind(jkind(j))), '.'

!        //   loop of bonds relevant to atom j
         do k = 1, 4

!           //   for all bonds
            if ( jbond(k) .ne. 0 ) then

!              //   update counter
               m = m + 1

!              //   partner atom
               l = jbond(k)

!              //   bond topology switched on for j-l pair
               bond(j,l) = 1

!              //   bond topology switched on for j-l pair
               bond(l,j) = 1

!           //   for all bonds
            end if

!        //   loop of bonds relevant to atom j
         end do

!     //   loop of atoms
      end do

!     //   error handling
      call error_handling( ierr, 'read_input', 10 )

!----------------------------------------------------------------------
!     //   count number of bonds
!----------------------------------------------------------------------

!     //   counter
      m = 0

!     //   loop of atom pair
      do i = 1, natom-1
      do j = i+1, natom

!        //  for all bonds
         if ( ( bond(i,j) .ne. 0 ) .or. ( bond(j,i) .ne. 0 ) ) then

!           //  update counter
            m = m + 1

!        //  for all bonds
         end if

!     //   loop of atom pair
      end do
      end do

!     // number of bonds
      nbond = m

!----------------------------------------------------------------------
!     //   angular bonds: count
!----------------------------------------------------------------------

!     //   counter reset
      m = 0

!     //   loop of all atoms
      do i = 1, natom-1

!     //   loop of all atoms
      do k = i+1, natom

!     //   loop of all atoms
      do j = 1, natom

!        //   skip for same atoms
         if ( j .eq. i ) cycle
         if ( j .eq. k ) cycle

!        //   connection
         ijk = bond(i,j) * bond(j,k)

!        //   angular bonds
         if ( ijk .eq. 1 ) then

!           //   update counter
            m = m + 1

!           //   1-3 pair
            bond_13(i,k) = 1
            bond_13(k,i) = 1

!        //   angular bonds
         end if

!     //   loop of all atoms
      end do

!     //   loop of all atoms
      end do

!     //   loop of all atoms
      end do

!     //   number of bond angles
      nangl = m

!----------------------------------------------------------------------
!     //   memory allocation
!----------------------------------------------------------------------

!     //   linear bonds: atom i
      allocate( i_lin(nbond+nangl) )

!     //   linear bonds: atom j
      allocate( j_lin(nbond+nangl) )

!     //   linear bonds: equilibrium bond length
      allocate( eq_lin(nbond+nangl) )

!     //   linear bonds: force constants
      allocate( fc_lin(nbond+nangl) )

!----------------------------------------------------------------------
!     //   linear bonds
!----------------------------------------------------------------------

!     //   counter
      m = 0

!     //   loop of atom pair
      do i = 1, natom-1
      do j = i+1, natom

!        //  for all bonds
         if ( ( bond(i,j) .ne. 0 ) .or. ( bond(j,i) .ne. 0 ) ) then

!           //  update counter
            m = m + 1

!           //  linear bonded pair
            i_lin(m) = i
            j_lin(m) = j

!        //  for all bonds
         end if

!     //   loop of atom pair
      end do
      end do

!----------------------------------------------------------------------
!     //   memory allocation
!----------------------------------------------------------------------

!     //   angular bonds: atom i
      allocate( i_angl(nangl) )

!     //   angular bonds: atom j
      allocate( j_angl(nangl) )

!     //   angular bonds: atom k
      allocate( k_angl(nangl) )

!     //   1-3 bonds: atom i
      allocate( i_13(nangl) )

!     //   1-3 bonds: atom j
      allocate( j_13(nangl) )

!     //   angular bonds: angle
      allocate( eq_angl(nangl) )

!     //   angular bonds: force constants
      allocate( fc_angl(nangl) )

!     //   urey-bradley bonds: atom i
      allocate( i_ub(nangl) )

!     //   urey-bradley bonds: atom j
      allocate( j_ub(nangl) )

!     //   urey-bradley bonds: atom k
      allocate( k_ub(nangl) )

!     //   urey-bradley bonds: equilibrium length
      allocate( eq_ub(nangl) )

!     //   urey-bradley bonds: force constants
      allocate( fc_ub(nangl) )

!----------------------------------------------------------------------
!     //   angular bonds: list
!----------------------------------------------------------------------

!     //   counter reset
      m = 0

!     //   loop of all atoms
      do i = 1, natom-1

!     //   loop of all atoms
      do j = 1, natom

!        //   skip if disconnected
         if ( bond(i,j) .eq. 0 ) cycle

!        //   loop of all atoms
         do k = i+1, natom

!           //   skip if disconnected
            if ( j .eq. i ) cycle
            if ( j .eq. k ) cycle

!           //   connections
            ijk = bond(i,j) * bond(j,k)

!           //   angular bonds
            if ( ijk .eq. 1 ) then

!              //   update counter
               m = m + 1

!              //   atoms
               i_angl(m) = i
               j_angl(m) = j
               k_angl(m) = k

!              //   1-3 pairs
               i_13(m) = i
               j_13(m) = k

!           //   angular bonds
            end if

!        //   loop of all atoms
         end do

!     //   loop of all atoms
      end do

!     //   loop of all atoms
      end do

!----------------------------------------------------------------------
!     //   torsion bonds: count
!----------------------------------------------------------------------

!     //   counter reset
      m = 0

!     //   loop of all atoms
      do i = 1, natom

!     //   loop of all atoms
      do j = 1, natom

!        //   skip for same atoms
         if ( i .eq. j ) cycle

!        //   skip when disconnected
         if ( bond(i,j) .eq. 0 ) cycle

!        //   loop of all atoms
         do k = j+1, natom

!           //   skip for same atoms
            if ( i .eq. k ) cycle

!           //   skip when disconnected
            if ( bond(j,k) .eq. 0 ) cycle

!           //   loop of all atoms
            do l = 1, natom

!              //   skip for same atoms
               if ( i .eq. l ) cycle

!              //   skip for same atoms
               if ( j .eq. l ) cycle

!              //   skip for same atoms
               if ( k .eq. l ) cycle

!              //   atom connections
               ijkl = bond(i,j) * bond(j,k) * bond(k,l)

!              //   if connected
               if ( ijkl .eq. 1 ) then

!                 //   update counter
                  m = m + 1

!                 //   1-4 pairs
                  bond_14(i,l) = 1
                  bond_14(l,i) = 1

!              //   if connected
               end if

!           //   loop of all atoms
            end do

!        //   loop of all atoms
         end do

!     //   loop of all atoms
      end do

!     //   loop of all atoms
      end do

!     //   number of torsions
      ntors = m

!----------------------------------------------------------------------
!     //    trigonals: count
!----------------------------------------------------------------------

!     //   counter reset
      m = 0

!     //   loop of all atoms
      do i = 1, natom

!     //   loop of all atoms
      do j = 1, natom

!        //   skip for same atoms
         if ( i .eq. j ) cycle

!        //   skip when disconnected
         if ( bond(i,j) .eq. 0 ) cycle

!        //   loop of all atoms
         do k = j+1, natom

!           //   skip for same atoms
            if ( i .eq. k ) cycle

!           //   skip when disconnected
            if ( bond(i,k) .eq. 0 ) cycle

!           //   loop of all atoms
            do l = 1, natom

!              //   skip for same atoms
               if ( i .eq. l ) cycle

!              //   skip for same atoms
               if ( j .eq. l ) cycle

!              //   skip for same atoms
               if ( k .eq. l ) cycle

!              //   atom connections
               ijkl = bond(i,j) * bond(i,k) * bond(i,l)

!              //   if connected
               if ( ijkl .eq. 1 ) then

!                 //   update counter
                  m = m + 1

!              //   if connected
               end if

!           //   loop of all atoms
            end do

!        //   loop of all atoms
         end do

!     //   loop of all atoms
      end do

!     //   loop of all atoms
      end do

!     //   number of trigonals
      ntri = m

!----------------------------------------------------------------------
!     //   number of torsion bonds
!----------------------------------------------------------------------

      ntors_3 = ntors * 3

!----------------------------------------------------------------------
!     //   memory allocation
!----------------------------------------------------------------------

!     //   1-4 bonds: i
      allocate( i_14(ntors) )

!     //   1-4 bonds: j
      allocate( j_14(ntors) )

!     //   torsion bonds: atom i
      allocate( i_tors(ntors_3) )

!     //   torsion bonds: atom j
      allocate( j_tors(ntors_3) )

!     //   torsion bonds: atom h
      allocate( k_tors(ntors_3) )

!     //   torsion bonds: atom k
      allocate( l_tors(ntors_3) )

!     //   torsion bonds:
      allocate( v_tors(ntors_3) )

!     //   phase
      allocate( mu_tors(ntors_3) )

!     //   degeneracy
      allocate( nu_tors(ntors_3) )

!----------------------------------------------------------------------
!     //   torsion bonds: list
!----------------------------------------------------------------------

!     //   counter reset
      m = 0

!     //   loop of all atoms
      do i = 1, natom

!     //   loop of all atoms
      do j = 1, natom

!        //   if i and j are similar
         if ( i .eq. j ) cycle

!        //   check bond connection
         if ( bond(i,j) .eq. 0 ) cycle

!        //   loop of all atoms
         do k = j+1, natom

            if ( i .eq. k ) cycle

            if ( bond(j,k) .eq. 0 ) cycle

!           //   loop of all atoms
            do l = 1, natom

!              //   skip for same atoms
               if ( i .eq. l ) cycle

!              //   skip for same atoms
               if ( j .eq. l ) cycle

!              //   skip for same atoms
               if ( k .eq. l ) cycle

!              //   bond connection
               ijkl = bond(i,j) * bond(j,k) * bond(k,l)

!              //   if connected
               if ( ijkl .eq. 1 ) then

!                 //   update counter
                  m = m + 1

!                 //   dihedral bond: atom i
                  i_tors(3*m-2) = i
                  i_tors(3*m-1) = i
                  i_tors(3*m-0) = i

!                 //   dihedral bond: atom j
                  j_tors(3*m-2) = j
                  j_tors(3*m-1) = j
                  j_tors(3*m-0) = j

!                 //   dihedral bond: atom k
                  k_tors(3*m-2) = k
                  k_tors(3*m-1) = k
                  k_tors(3*m-0) = k

!                 //   dihedral bond: atom l
                  l_tors(3*m-2) = l
                  l_tors(3*m-1) = l
                  l_tors(3*m-0) = l

!                 //   1-4 bond: atom i
                  i_14(m) = i

!                 //   1-4 bond: atom l
                  j_14(m) = l

!              //   if connected
               end if

!           //   loop of all atoms
            end do

!        //   loop of all atoms
         end do

!     //   loop of all atoms
      end do

!     //   loop of all atoms
      end do

!----------------------------------------------------------------------
!     //   memory allocation
!----------------------------------------------------------------------

!     //   trigonals: atom i
      allocate( i_tri(ntri) )

!     //   trigonals: atom j
      allocate( j_tri(ntri) )

!     //   trigonals: atom k
      allocate( k_tri(ntri) )

!     //   trigonals: atom l
      allocate( l_tri(ntri) )

!     //   improper bonds: atom i
      allocate( i_improper(ntri) )

!     //   improper bonds: atom j
      allocate( j_improper(ntri) )

!     //   improper bonds: atom k
      allocate( k_improper(ntri) )

!     //   improper bonds: atom l
      allocate( l_improper(ntri) )

!     //   improper bonds: force constant
      allocate( fc_improper(ntri) )

!     //   improper bonds: angle
      allocate( eq_improper(ntri) )

!     //   improper torsion bonds: atom i
      allocate( i_imptors(ntri) )

!     //   improper torsion bonds: atom j
      allocate( j_imptors(ntri) )

!     //   improper torsion bonds: atom k
      allocate( k_imptors(ntri) )

!     //   improper torsion bonds: atom l
      allocate( l_imptors(ntri) )

!     //   improper torsion bonds: energy barrier
      allocate( v_imptors(ntri) )

!     //   improper torsion bonds: phase
      allocate( mu_imptors(ntri) )

!     //   improper torsion bonds: degeneracy
      allocate( nu_imptors(ntri) )

!     //   torsion bonds: atom i
      allocate( i_dih(ntors_3+ntri) )

!     //   torsion bonds: atom j
      allocate( j_dih(ntors_3+ntri) )

!     //   torsion bonds: atom h
      allocate( k_dih(ntors_3+ntri) )

!     //   torsion bonds: atom k
      allocate( l_dih(ntors_3+ntri) )

!     //   torsion bonds:
      allocate( v_dih(ntors_3+ntri) )

!     //   phase
      allocate( mu_dih(ntors_3+ntri) )

!     //   degeneracy
      allocate( nu_dih(ntors_3+ntri) )

!----------------------------------------------------------------------
!     //    trigonals: list
!----------------------------------------------------------------------

!     //   counter reset
      m = 0

!     //   loop of all atoms
      do i = 1, natom

!     //   loop of all atoms
      do j = 1, natom

!        //   skip for same atoms
         if ( i .eq. j ) cycle

!        //   skip when disconnected
         if ( bond(i,j) .eq. 0 ) cycle

!        //   loop of all atoms
         do k = j+1, natom

!           //   skip for same atoms
            if ( i .eq. k ) cycle

!           //   skip when disconnected
            if ( bond(i,k) .eq. 0 ) cycle

!           //   loop of all atoms
            do l = 1, natom

!              //   skip for same atoms
               if ( i .eq. l ) cycle

!              //   skip for same atoms
               if ( j .eq. l ) cycle

!              //   skip for same atoms
               if ( k .eq. l ) cycle

!              //   atom connections
               ijkl = bond(i,j) * bond(i,k) * bond(i,l)

!              //   if connected
               if ( ijkl .eq. 1 ) then

!                 //   update counter
                  m = m + 1

!                 //   trigonal: atom i
                  i_tri(m) = i

!                 //   trigonal: atom j
                  j_tri(m) = j

!                 //   trigonal: atom k
                  k_tri(m) = k

!                 //   trigonal: atom l
                  l_tri(m) = l

!              //   if connected
               end if

!           //   loop of all atoms
            end do

!        //   loop of all atoms
         end do

!     //   loop of all atoms
      end do

!     //   loop of all atoms
      end do

!-----------------------------------------------------------------------
!     //   end of subroutine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine lookup ( char, nchar, iounit, ierr )
!***********************************************************************

      implicit none

      integer :: iounit, ierr, nchar

      character(len=nchar) :: char, char_line

!     //   read from the top line
      rewind( iounit )

!     //   loop of lines
      do

!         //   read a line
          read ( iounit, '(a)', iostat=ierr ) char_line(1:nchar)

!         //   exit when error occurred
          if ( ierr .ne. 0 ) exit

!         //   exit when we find a match
          if ( char_line(1:nchar) .eq. char(1:nchar) ) exit

!     //   loop of lines
      end do

!-----------------------------------------------------------------------
!     //   end of subroutine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine init_tinker
!***********************************************************************

!     //   variables
      use tinker_variables, only : &
     &   scale_sigma, scale_q_14, scale_eps_14, scale_tors, angleunit, &
     &   scale_sig_14, scale_improper, scale_imptors, bondunit, &
     &   def_kind, chg14scale, vdw14scale, econst_ratio, econst_mm, &
     &   econst_au, sig_param, symbol_kind, eps_param, sig14_param, &
     &   eps14_param, q_param, amumass_kind, forcefield, vdwindex, &
     &   vdwtype, radiusrule, radiustype, radiussize, epsilonrule, &
     &   torsionunit, impropunit, nbo_kind, nkind, numbre_kind, &
     &   imptorunit, vdw14scale, chg14scale, iounit_tinker, ikind

!     //   local variables
      implicit none

!     //   integers
      integer :: i

!     //   error flag
      integer :: ierr

!     //   keywords
      character(len=15) :: keyword

!-----------------------------------------------------------------------
!     //   read keywords
!-----------------------------------------------------------------------

!     //   reset input file
      rewind( iounit_tinker )

!     //   loop of lines
      do

!        //   read a line
         read ( iounit_tinker, *, iostat=ierr ) keyword

!        //   error confirmation
         if ( ierr .ne. 0 ) exit

!-----------------------------------------------------------------------

!        //   forcefield
         if ( keyword(1:11) .eq. 'forcefield ' ) then

!           //   one line back
            backspace ( iounit_tinker )

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword, forcefield

!           //   stop on error
            call error_handling( ierr, 'init_tinker', 11 )

!        //   forcefield
         end if

!-----------------------------------------------------------------------

!        //   vdwindex
         if ( keyword(1:9) .eq. 'vdwindex ' ) then

!           //   one line back
            backspace ( iounit_tinker )

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword, vdwindex

!           //   stop on error
            call error_handling( ierr, 'init_tinker', 11 )

!           //   check vdwindex
            if ( vdwindex(1:6) .eq. 'CLASS ' ) then

!              //   error flag
               ierr = 0

!           //   check vdwindex
            else if ( vdwindex(1:5) .eq. 'TYPE ' ) then

!              //   error flag
               ierr = 0

!           //   check vdwindex
            else

!              //   error flag
               ierr = 1

!           //   check vdwindex
            end if

!           //   stop on error
            call error_handling( ierr, 'init_tinker', 11 )

!        //   vdwindex
         end if

!-----------------------------------------------------------------------

!        //   vdwtype
         if ( keyword(1:8) .eq. 'vdwtype ' ) then

!           //   one line back
            backspace ( iounit_tinker )

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword, vdwtype

!           //   stop on error
            call error_handling( ierr, 'init_tinker', 11 )

!           //   check vdwtype
            if ( vdwtype(1:13) .eq. 'LENNARD-JONES' ) then

!              //   error flag
               ierr = 0

!           //   check vdwtype
            else

!              //   error flag
               ierr = 1

!           //   check vdwtype
            end if

!           //   stop on error
            call error_handling( ierr, 'init_tinker', 11 )

!        //   vdwtype
         end if

!-----------------------------------------------------------------------

!        //   radiusrule
         if ( keyword(1:11) .eq. 'radiusrule ' ) then

!           //   one line back
            backspace ( iounit_tinker )

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword, radiusrule

!           //   stop on error
            call error_handling( ierr, 'init_tinker', 11 )

!        //   radiusrule
         end if

!-----------------------------------------------------------------------

!        //   radiustype
         if ( keyword(1:11) .eq. 'radiustype ' ) then

!           //   one line back
            backspace ( iounit_tinker )

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword, radiustype

!           //   stop on error
            call error_handling( ierr, 'init_tinker', 11 )

!        //   radiustype
         end if

!-----------------------------------------------------------------------

!        //   radiussize
         if ( keyword(1:11) .eq. 'radiussize ' ) then

!           //   one line back
            backspace ( iounit_tinker )

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword, radiussize

!           //   stop on error
            call error_handling( ierr, 'init_tinker', 11 )

!        //   radiussize
         end if

!-----------------------------------------------------------------------

!        //   epsilonrule
         if ( keyword(1:12) .eq. 'epsilonrule ' ) then

!           //   one line back
            backspace ( iounit_tinker )

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword, epsilonrule

!           //   stop on error
            call error_handling( ierr, 'init_tinker', 11 )

!        //   epsilonrule
         end if

!-----------------------------------------------------------------------

!        //   bondunit
         if ( keyword(1:9) .eq. 'bondunit ' ) then

!           //   one line back
            backspace ( iounit_tinker )

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword, bondunit

!           //   stop on error
            call error_handling( ierr, 'init_tinker', 11 )

!        //   epsilonrule
         end if

!-----------------------------------------------------------------------

!        //   angleunit
         if ( keyword(1:10) .eq. 'angleunit ' ) then

!           //   one line back
            backspace ( iounit_tinker )

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword, angleunit

!           //   stop on error
            call error_handling( ierr, 'init_tinker', 11 )

!        //   epsilonrule
         end if

!-----------------------------------------------------------------------

!        //   torsionunit
         if ( keyword(1:12) .eq. 'torsionunit ' ) then

!           //   one line back
            backspace ( iounit_tinker )

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword, torsionunit

!           //   stop on error
            call error_handling( ierr, 'init_tinker', 11 )

!        //   torsionunit
         end if

!-----------------------------------------------------------------------

!        //   impropunit
         if ( keyword(1:11) .eq. 'impropunit ' ) then

!           //   one line back
            backspace ( iounit_tinker )

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword, impropunit

!           //   stop on error
            call error_handling( ierr, 'init_tinker', 11 )

!        //   impropunit
         end if

!-----------------------------------------------------------------------

!        //   imptorunit
         if ( keyword(1:11) .eq. 'imptorunit ' ) then

!           //   one line back
            backspace ( iounit_tinker )

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword, imptorunit

!           //   stop on error
            call error_handling( ierr, 'init_tinker', 11 )

!        //   imptorunit
         end if

!-----------------------------------------------------------------------

!        //   vdw-14-scale
         if ( keyword(1:13) .eq. 'vdw-14-scale ' ) then

!           //   one line back
            backspace ( iounit_tinker )

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword, vdw14scale

!           //   warning: defined either as numerator or denominator
            vdw14scale = min( vdw14scale, 1.d0/vdw14scale )

!           //   stop on error
            call error_handling( ierr, 'init_tinker', 11 )

!        //   vdw-14-scale
         end if

!-----------------------------------------------------------------------

!        //   chg-14-scale
         if ( keyword(1:13) .eq. 'chg-14-scale ' ) then

!           //   one line back
            backspace ( iounit_tinker )

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword, chg14scale

!           //   warning: defined either as numerator or denominator
            chg14scale = min( chg14scale, 1.d0/chg14scale )

!           //   stop on error
            call error_handling( ierr, 'init_tinker', 11 )

!        //   chg-14-scale
         end if

!-----------------------------------------------------------------------

!        //   electric constant
         if ( keyword(1:9) .eq. 'electric ' ) then

!           //   one line back
            backspace ( iounit_tinker )

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword, econst_mm

!           //   stop on error
            call error_handling( ierr, 'init_tinker', 11 )

!        //   forcefield
         end if

      end do

!-----------------------------------------------------------------------
!     //   ratio of electric constants from mm and au unit conversion
!-----------------------------------------------------------------------

      econst_ratio = econst_mm / econst_au

!-----------------------------------------------------------------------
!     //   scaling factors
!-----------------------------------------------------------------------

!     //   LJ sigma

      if      ( radiustype(1:13) .eq. 'SIGMA        ' ) then
         scale_sigma = 1.d0
      else if ( radiustype(1:13) .eq. 'R-MIN        ' ) then
         scale_sigma = 2.d0**(-1.d0/6.d0)
      else
         scale_sigma = 1.d0
      end if

      if      ( radiussize(1:13) .eq. 'DIAMETER     ' ) then
         scale_sigma = scale_sigma
      else if ( radiussize(1:13) .eq. 'RADIUS       ' ) then
         scale_sigma = scale_sigma * 2.d0
      else
         scale_sigma = scale_sigma * 2.d0
      end if

!     //   LJ interaction (epsilon) for 1-4 pair
      scale_eps_14 = vdw14scale

!     //   electrostatic interaction for 1-4 pair
      scale_q_14   = chg14scale

!     //   torsion
      scale_tors     = 2.d0 * torsionunit

!     //   torsion
      scale_improper = 2.d0 * impropunit

!     //   improper torsion
      scale_imptors  = 2.d0 * imptorunit

!-----------------------------------------------------------------------
!     //   number of atom kinds
!-----------------------------------------------------------------------

!     //   reset input file
      rewind( iounit_tinker )

!     //   initialize
      nkind = 0

!     //   loop of lines
      do

!        //   read a line
         read ( iounit_tinker, *, iostat=ierr ) keyword

!        //   error confirmation
         if ( ierr .ne. 0 ) exit

!        //   look for keyword
         if ( keyword(1:5) .eq. 'atom ' ) then

!           //   step back one line
            backspace( iounit_tinker )

!           //   read a line again
            read ( iounit_tinker, * ) keyword, i

!           //   number of kinds
            if ( keyword(1:5) .eq. 'atom ' ) nkind = max(i,nkind)

!        //   look for keyword
         end if

!     //   loop of lines
      end do

!-----------------------------------------------------------------------
!     //   print
!-----------------------------------------------------------------------

      write( 6, '(a,i6,a)' ) &
     &   'Number of atom kinds in tinker file:', nkind, '.'

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

!     //   atomic kind
      allocate( ikind(nkind) )

!     //   symbol of atomic kind
      allocate( symbol_kind(nkind) )

!     //   definition of atomic kind
      allocate( def_kind(nkind) )

!     //   atomic number
      allocate( numbre_kind(nkind) )

!     //   atomic mass in amu
      allocate( amumass_kind(nkind) )

!     //   bond order of atomic kind
      allocate( nbo_kind(nkind) )

!-----------------------------------------------------------------------
!     //   read atom kinds
!-----------------------------------------------------------------------

!     //   reset input file
      rewind( iounit_tinker )

!     //   loop of lines
      do

!        //   read a line
         read ( iounit_tinker, *, iostat=ierr ) keyword

!        //   error confirmation
         if ( ierr .ne. 0 ) exit

!        //   look for keyword
         if ( keyword(1:5) .eq. 'atom ' ) then

!           //   step back one line
            backspace( iounit_tinker )

!           //   read a line again
            read ( iounit_tinker, *, iostat=ierr ) &
     &         keyword, i, ikind(i), symbol_kind(i), &
     &         def_kind(i), numbre_kind(i), amumass_kind(i), &
     &         nbo_kind(i)

!           //   error stop
            if ( ierr .ne. 0 ) then
               write( 6, '(a,i6)' ) &
     &            'Error: atom read incorrectly for atom kind', i
               stop
            end if

!        //   look for keyword
         end if

!     //   loop of lines
      end do

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

!     //   lj sigma
      allocate( sig_param(nkind) )

!     //   lj epsilon
      allocate( eps_param(nkind) )

!     //   lj sigma 1-4
      allocate( sig14_param(nkind) )

!     //   lj epsilon 1-4
      allocate( eps14_param(nkind) )

!     //   charge
      allocate( q_param(nkind)   )

!-----------------------------------------------------------------------
!     //   read vdw parameters
!-----------------------------------------------------------------------

!     //   default sigma
      sig_param(:) = 0.d0

!     //   default epsilon
      eps_param(:) = 0.d0

!     //   reset input file
      rewind( iounit_tinker )

!     //   loop of lines
      do

!        //   read a line
         read ( iounit_tinker, *, iostat=ierr ) keyword

!        //   error confirmation
         if ( ierr .ne. 0 ) exit

!        //   look for keyword
         if ( keyword(1:4) .eq. 'vdw ' ) then

!           //   step back one line
            backspace( iounit_tinker )

!           //   read a line again
            read ( iounit_tinker, * )  keyword, i

!           //   detect error
            if ( i .gt. nkind ) &
     &         call error_handling( 1, 'init_tinker', 11 )

!           //   step back one line
            backspace( iounit_tinker )

!           //   read a line again
            read ( iounit_tinker, * ) &
     &         keyword, i, sig_param(i), eps_param(i)

!        //   look for keyword
         end if

!     //   loop of lines
      end do

!-----------------------------------------------------------------------
!     //   vdw14 parameters
!-----------------------------------------------------------------------

!     //   default sigma 1-4
      sig14_param(:) = sig_param(:) * scale_sig_14

!     //   default epsilon 1-4
      eps14_param(:) = eps_param(:) * scale_eps_14

!     //   reset input file
      rewind( iounit_tinker )

!     //   loop of lines
      do

!        //   read a line
         read ( iounit_tinker, *, iostat=ierr ) keyword

!        //   error confirmation
         if ( ierr .ne. 0 ) exit

!        //   look for keyword
         if ( keyword(1:6) .eq. 'vdw14 ' ) then

!           //   step back one line
            backspace( iounit_tinker )

!           //   read a line again
            read ( iounit_tinker, * )  keyword, i

!           //   detect error
            if ( i .gt. nkind ) &
     &         call error_handling( 1, 'init_tinker', 11 )

!           //   step back one line
            backspace( iounit_tinker )

!           //   read a line again
            read ( iounit_tinker, * ) &
     &         keyword, i, sig14_param(i), eps14_param(i)

!        //   look for keyword
         end if

!     //   loop of lines
      end do

!-----------------------------------------------------------------------
!     //   read charge parameters
!-----------------------------------------------------------------------

!     //   default charge
      q_param(:) = 0.d0

!     //   reset input file
      rewind( iounit_tinker )

!     //   loop of lines
      do

!        //   read a line
         read ( iounit_tinker, *, iostat=ierr ) keyword

!        //   error confirmation
         if ( ierr .ne. 0 ) exit

!        //   look for keyword
         if ( keyword(1:7) .eq. 'charge ' ) then

!           //   step back one line
            backspace( iounit_tinker )

!           //   read a line again
            read ( iounit_tinker, * )  keyword, i

!           //   detect error
            if ( i .gt. nkind ) &
     &         call error_handling( 1, 'init_tinker', 11 )

!           //   step back one line
            backspace( iounit_tinker )

!           //   read a line again
            read ( iounit_tinker, * )  keyword, i, q_param(i)

!        //   look for keyword
         end if

!     //   loop of lines
      end do

!-----------------------------------------------------------------------
!     //   end of subroutine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine lin_tinker
!***********************************************************************

!     //   shared variables
      use tinker_variables, only : &
     &   eq_lin, fc_lin, nbond, ikind, jkind, i_lin, j_lin, &
     &   iounit_tinker

!     //   local variables
      implicit none

!     //   error flag
      integer :: ierr

!     //   keywords
      character(len=15) :: keyword

!     //   flag
      integer :: iflag = 0

!     //   real numbers
      real(8) :: r, fc

!     //   integers
      integer :: i, j, ii, jj, m

!-----------------------------------------------------------------------
!     //   linear bond parameters
!-----------------------------------------------------------------------

!     //  loop of linear bonds
      do m = 1, nbond

!        //   atom kinds
         ii = ikind(jkind(i_lin(m)))
         jj = ikind(jkind(j_lin(m)))

!        //   reset input file
         rewind( iounit_tinker )

!        //   turn off flag
         iflag = 0

!        //   loop of lines
         do

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword

!           //   error confirmation
            if ( ierr .ne. 0 ) exit

!           //   look for keyword
            if ( keyword(1:5) .eq. 'bond ' ) then

!              //   step back one line
               backspace( iounit_tinker )

!              //   read a line again
               read ( iounit_tinker, *, iostat=ierr ) &
     &            keyword, i, j, fc, r

!              //   error handling
               if ( ierr .ne. 0 ) exit

!              //   substitute the values if the line matches
               if ( ( i .eq. ii ) .and. ( j .eq. jj ) ) then

!                 //   equilibrium bond length
                  eq_lin(m)  = r

!                 //   force constant
                  fc_lin(m) = fc

!                 //   turn on flag
                  iflag = 1

!                 //   proceed to next bond
                  exit

!              //   substitute the values if the line matches
               else if ( ( j .eq. ii ) .and. ( i .eq. jj ) ) then

!                 //   equilibrium bond length
                  eq_lin(m)  = r

!                 //   force constant
                  fc_lin(m) = fc

!                 //   turn on flag
                  iflag = 1

!                 //   proceed to next bond
                  exit

!              //   substitute the values if the line matches
               end if

!           //   look for keyword
            end if

!        //   loop of lines
         end do

!        //   error termination
         if ( iflag .eq. 0 ) then

            write( 6, '(a)' ) &
     &         'Error termination.'

            write( 6, '(a,2i5,a,2i5,a)' ) &
     &         'Missing bond type: ', ii, jj, ' for atoms ', &
     &         i_lin(m), j_lin(m), '.'

            stop

!        //   error termination
         end if

!     //   loop of linear bonds
      end do

!-----------------------------------------------------------------------
!     //   end of subroutine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine angl_tinker
!***********************************************************************

!     //   shared variables
      use tinker_variables, only : &
     &   eq_angl, fc_angl, nangl, ikind, jkind, i_angl, j_angl, k_angl, &
     &   iounit_tinker

!     //   local variables
      implicit none

!     //   error flag
      integer :: ierr

!     //   keywords
      character(len=15) :: keyword

!     //   flag
      integer :: iflag = 0

!     //   real numbers
      real(8) :: theta, fc

!     //   integers
      integer :: i, j, k, ii, jj, kk, m

!-----------------------------------------------------------------------
!     //   angular bond parameters
!-----------------------------------------------------------------------

!     //  loop of angular bonds
      do m = 1, nangl

!        //   atom kinds
         ii = ikind(jkind(i_angl(m)))
         jj = ikind(jkind(j_angl(m)))
         kk = ikind(jkind(k_angl(m)))

!        //   reset input file
         rewind( iounit_tinker )

!        //   turn off flag
         iflag = 0

!        //   loop of lines
         do

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword

!           //   error confirmation
            if ( ierr .ne. 0 ) exit

!           //   look for keyword
            if ( keyword(1:6) .eq. 'angle ' ) then

!              //   step back one line
               backspace( iounit_tinker )

!              //   read a line again
               read ( iounit_tinker, * ) keyword, i, j, k, fc, theta

!              //   error confirmation
               if ( ierr .ne. 0 ) exit

!              //   substitute the values if the line matches
               if ( ( i .eq. ii ) .and. ( j .eq. jj ) .and. &
     &              ( k .eq. kk ) ) then

!                 //   equilibrium bond angle
                  eq_angl(m)  = theta

!                 //   force constant
                  fc_angl(m) = fc

!                 //   turn on flag
                  iflag = 1

!                 //   proceed to next bond
                  exit

!              //   substitute the values if the line matches
               else if ( ( i .eq. kk ) .and. ( j .eq. jj ) .and. &
     &              ( k .eq. ii ) ) then

!                 //   equilibrium bond angle
                  eq_angl(m)  = theta

!                 //   force constant
                  fc_angl(m) = fc

!                 //   turn on flag
                  iflag = 1

!                 //   proceed to next bond
                  exit

!              //   substitute the values if the line matches
               end if

!           //   look for keyword
            end if

!        //   loop of lines
         end do

!        //   error termination
         if ( iflag .eq. 0 ) then

!           //   comment
            write( 6, '(a)' ) &
     &         'Error termination.'

!           //   comment
            write( 6, '(a,3i5,a,3i5,a)' ) &
     &         'Missing angle type: ', ii, jj, kk, ' for atoms ', &
     &         i_angl(m), j_angl(m), k_angl(m), '.'

!           //   stop
            stop

!        //   error termination
         end if

!     //   loop of angular bonds
      end do

!-----------------------------------------------------------------------
!     //   end of subroutine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine ub_tinker
!***********************************************************************

!     //   shared variables
      use tinker_variables, only : &
     &   eq_ub, fc_ub, eq_lin, fc_lin, nub, ikind, jkind, &
     &   i_ub, j_ub, k_ub, i_angl, j_angl, k_angl, i_lin, j_lin, &
     &   iounit_tinker, nbond, nangl

!     //   local variables
      implicit none

!     //   error flag
      integer :: ierr

!     //   keywords
      character(len=15) :: keyword

!     //   flag
      integer :: iflag = 0

!     //   real numbers
      real(8) :: r, fc

!     //   integers
      integer :: i, j, k, ii, jj, kk, l, m

!-----------------------------------------------------------------------
!     //   urey-bradley bond parameters
!-----------------------------------------------------------------------

!     //   counter
      l = 0

!     //  loop of angular bonds
      do m = 1, nangl

!        //   atom kinds
         ii = ikind(jkind(i_angl(m)))
         jj = ikind(jkind(j_angl(m)))
         kk = ikind(jkind(k_angl(m)))

!        //   reset input file
         rewind( iounit_tinker )

!        //   turn off flag
         iflag = 0

!        //   loop of lines
         do

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword

!           //   error confirmation
            if ( ierr .ne. 0 ) exit

!           //   look for keyword
            if ( keyword(1:9) .eq. 'ureybrad ' ) then

!              //   step back one line
               backspace( iounit_tinker )

!              //   read a line again
               read ( iounit_tinker, * ) keyword, i, j, k, fc, r

!              //   error confirmation
               if ( ierr .ne. 0 ) exit

!              //   substitute the values if the line matches
               if ( ( i .eq. ii ) .and. ( j .eq. jj ) .and. &
     &              ( k .eq. kk ) ) then

!                 //   turn on flag
                  iflag = 1

!                 //   proceed to next bond
                  exit

!              //   substitute the values if the line matches
               else if ( ( i .eq. kk ) .and. ( j .eq. jj ) .and. &
     &              ( k .eq. ii ) ) then

!                 //   turn on flag
                  iflag = 1

!                 //   proceed to next bond
                  exit

!              //   substitute the values if the line matches
               end if

!           //   look for keyword
            end if

!        //   loop of lines
         end do

!        //   urey-bradley
         if ( iflag .eq. 1 ) then

!           //   counter
            l = l + 1

!           //   atoms
            i_ub(l) = i_angl(m)
            j_ub(l) = j_angl(m)
            k_ub(l) = k_angl(m)

!        //   urey-bradley
         end if

!     //   loop of angular bonds
      end do

!     //   counter
      nub = l

!-----------------------------------------------------------------------
!     //   angular bond parameters
!-----------------------------------------------------------------------

!     //  loop of urey-bradley bonds
      do m = 1, nub

!        //   atom kinds
         ii = ikind(jkind(i_ub(m)))
         jj = ikind(jkind(j_ub(m)))
         kk = ikind(jkind(k_ub(m)))

!        //   reset input file
         rewind( iounit_tinker )

!        //   counter
         l = 0

!        //   loop of lines
         do

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword

!           //   error confirmation
            if ( ierr .ne. 0 ) exit

!           //   look for keyword
            if ( keyword(1:9) .eq. 'ureybrad ' ) then

!              //   step back one line
               backspace( iounit_tinker )

!              //   read a line again
               read ( iounit_tinker, * ) keyword, i, j, k, fc, r

!              //   error confirmation
               if ( ierr .ne. 0 ) exit

!              //   substitute the values if the line matches
               if ( ( i .eq. ii ) .and. ( j .eq. jj ) .and. &
     &              ( k .eq. kk ) ) then

!                 //   equilibrium length
                  eq_ub(m)  = r

!                 //   force constant
                  fc_ub(m) = fc

!                 //   proceed to next bond
                  exit

!              //   substitute the values if the line matches
               else if ( ( i .eq. kk ) .and. ( j .eq. jj ) .and. &
     &              ( k .eq. ii ) ) then

!                 //   equilibrium length
                  eq_ub(m)  = r

!                 //   force constant
                  fc_ub(m) = fc

!                 //   proceed to next bond
                  exit

!              //   substitute the values if the line matches
               end if

!           //   look for keyword
            end if

!        //   loop of lines
         end do

!     //  loop of urey-bradley bonds
      end do

!-----------------------------------------------------------------------
!     //   add to linear bonds
!-----------------------------------------------------------------------

!     //  loop of urey-bradley bonds
      do m = 1, nub

!        //   atom i
         i_lin(nbond+m) = i_ub(m)

!        //   atom j
         j_lin(nbond+m) = k_ub(m)

!        //   equilibrium bond length in bohr
         eq_lin(nbond+m) = eq_ub(m)

!        //   force constant in hartree bohr**-2
         fc_lin(nbond+m) = fc_ub(m)

!     //  loop of urey-bradley bonds
      end do

!-----------------------------------------------------------------------
!     //   end of subroutine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine tors_tinker
!***********************************************************************

!     //   shared variables
      use tinker_variables, only : &
     &   pi, scale_tors, v_tors, v_dih, ikind, jkind, i_dih, j_dih, &
     &   k_dih, l_dih, i_tors, j_tors, k_tors, l_tors, mu_tors, &
     &   nu_tors, mu_dih, nu_dih, iounit_tinker, ntors

!     //   local variables
      implicit none

!     //   error flag
      integer :: ierr

!     //   keywords
      character(len=15) :: keyword

!     //   flag
      integer :: iflag = 0

!     //   integers
      integer :: i, j, k, l, m, ii, jj, kk, ll

!     //   real numbers
      real(8) :: v1, v2, v3, phi1, phi2, phi3

!     //   integers
      integer :: nu1, nu2, nu3

!-----------------------------------------------------------------------
!     //   torsion bond parameters
!-----------------------------------------------------------------------

!     //  loop of torsion bonds
      do m = 1, ntors

!        //   atom kinds
         ii = ikind(jkind(i_tors(3*m-2)))
         jj = ikind(jkind(j_tors(3*m-2)))
         kk = ikind(jkind(k_tors(3*m-2)))
         ll = ikind(jkind(l_tors(3*m-2)))

!         write( 6, '(13i5)' ) m,
!     &       i_tors(3*m-2), j_tors(3*m-2),
!     &       k_tors(3*m-2), l_tors(3*m-2),
!     &       jkind(i_tors(3*m-2)), jkind(j_tors(3*m-2)),
!     &       jkind(k_tors(3*m-2)), jkind(l_tors(3*m-2)),
!     &       ii, jj, kk, ll

!        //   reset input file
         rewind( iounit_tinker )

!        //   turn off flag
         iflag = 0

!        //   loop of lines
         do

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword

!           //   error confirmation
            if ( ierr .ne. 0 ) exit

!           //   look for keyword
            if ( keyword(1:8) .eq. 'torsion ' ) then

!              //   step back one line
               backspace( iounit_tinker )

!              //   read a line again
               read ( iounit_tinker, *, iostat=ierr ) &
     &            keyword, i, j, k, l, v1

!              //   on error, zero term
               if ( ierr .ne. 0 ) then

!                 //   step back one line
                  backspace( iounit_tinker )
                  backspace( iounit_tinker )
                  do
                     read ( iounit_tinker, '(a)', iostat=ierr ) &
     &                  keyword
                     if ( keyword(1:8) .eq. 'torsion ' ) then
                        backspace( iounit_tinker )
                        exit
                     else
                        backspace( iounit_tinker )
                        backspace( iounit_tinker )
                     end if
                  end do

!                 //   read a line again
                  read ( iounit_tinker, *, iostat=ierr ) &
     &               keyword, i, j, k, l

                  v1   = 0.d0
                  phi1 = 0.d0
                  nu1  = 0

                  v2   = 0.d0
                  phi2 = 0.d0
                  nu2  = 0

                  v3   = 0.d0
                  phi3 = 0.d0
                  nu3  = 0

!              //   otherwise, read a line again
               else

!                 //   step back one line
                  backspace( iounit_tinker )

!                 //   read a line again
                  read ( iounit_tinker, *, iostat=ierr ) &
     &               keyword, i, j, k, l, v1, phi1, nu1, v2

!                 //   on error, read only one terms
                  if ( ierr .ne. 0 ) then

!                    //   step back one line
                     backspace( iounit_tinker )
                     backspace( iounit_tinker )
                     do
                        read ( iounit_tinker, '(a)', iostat=ierr ) &
     &                     keyword
                        if ( keyword(1:8) .eq. 'torsion ' ) then
                           backspace( iounit_tinker )
                           exit
                        else
                           backspace( iounit_tinker )
                           backspace( iounit_tinker )
                        end if
                     end do

!                    //   read a line again
                     read ( iounit_tinker, *, iostat=ierr ) &
     &               keyword, i, j, k, l, v1, phi1, nu1

                     v2   = 0.d0
                     phi2 = 0.d0
                     nu2  = 0

                     v3   = 0.d0
                     phi3 = 0.d0
                     nu3  = 0

!                 //   otherwise, read a line again
                  else

!                    //   step back one line
                     backspace( iounit_tinker )

!                    //   read a line again
                     read ( iounit_tinker, *, iostat=ierr ) &
     &                  keyword, i, j, k, l, v1, phi1, nu1, v2, &
     &                  phi2, nu2, v3

!                    //   on error, read only two terms
                     if ( ierr .ne. 0 ) then

!                       //   step back one line
                        backspace( iounit_tinker )
                        backspace( iounit_tinker )
                        do
                           read ( iounit_tinker, '(a)', iostat=ierr ) &
     &                        keyword
                           if ( keyword(1:8) .eq. 'torsion ' ) then
                              backspace( iounit_tinker )
                              exit
                           else
                              backspace( iounit_tinker )
                              backspace( iounit_tinker )
                           end if
                        end do

!                       //   read a line again
                        read ( iounit_tinker, *, iostat=ierr ) &
     &                     keyword, i, j, k, l, v1, phi1, nu1, v2, &
     &                     phi2, nu2

                        v3   = 0.d0
                        phi3 = 0.d0
                        nu3  = 0

!                    //   otherwise, read a line again for three terms
                     else

!                       //   step back one line
                        backspace( iounit_tinker )

!                       //   read a line again
                        read ( iounit_tinker, *, iostat=ierr ) &
     &                     keyword, i, j, k, l, v1, phi1, nu1, v2, &
     &                     phi2, nu2, v3, phi3, nu3

!                       //   on error, read only two terms
                        if ( ierr .ne. 0 ) then

!                          //   step back one line
                           backspace( iounit_tinker )
                           backspace( iounit_tinker )
                           do
                              read ( iounit_tinker, '(a)', iostat=ierr ) &
     &                           keyword
                              if ( keyword(1:8) .eq. 'torsion ' ) then
                                 backspace( iounit_tinker )
                                 exit
                              else
                                 backspace( iounit_tinker )
                                 backspace( iounit_tinker )
                              end if
                           end do

!                          //   read a line again
                           read ( iounit_tinker, *, iostat=ierr ) &
     &                        keyword, i, j, k, l, v1, phi1, nu1, v2, &
     &                        phi2, nu2

                           v3   = 0.d0
                           phi3 = 0.d0
                           nu3  = 0

!                       //   on error
                        end if

!                    //   on error
                     end if

!                 //   on error
                  end if

!              //   on error
               end if

!              //   error confirmation
               if ( ierr .ne. 0 ) exit

!              //   substitute the values if the line matches
               if ( ( i .eq. ii ) .and. ( j .eq. jj ) .and. &
     &              ( k .eq. kk ) .and. ( l .eq. ll ) ) then

!                 //   atoms
                  i_dih(3*m-2)   = i_tors(3*m-2)
                  j_dih(3*m-2)   = j_tors(3*m-2)
                  k_dih(3*m-2)   = k_tors(3*m-2)
                  l_dih(3*m-2)   = l_tors(3*m-2)
                  i_dih(3*m-1)   = i_tors(3*m-1)
                  j_dih(3*m-1)   = j_tors(3*m-1)
                  k_dih(3*m-1)   = k_tors(3*m-1)
                  l_dih(3*m-1)   = l_tors(3*m-1)
                  i_dih(3*m-0)   = i_tors(3*m-0)
                  j_dih(3*m-0)   = j_tors(3*m-0)
                  k_dih(3*m-0)   = k_tors(3*m-0)
                  l_dih(3*m-0)   = l_tors(3*m-0)

!                 //   rotational barrier
                  v_tors(3*m-2)  = v1 * scale_tors
                  v_tors(3*m-1)  = v2 * scale_tors
                  v_tors(3*m-0)  = v3 * scale_tors

!                 //   phase
                  mu_tors(3*m-2) = nint( cos(phi1*pi/180.d0) )
                  mu_tors(3*m-1) = nint( cos(phi2*pi/180.d0) )
                  mu_tors(3*m-0) = nint( cos(phi3*pi/180.d0) )

!                 //   degeneracy
                  nu_tors(3*m-2) = nu1
                  nu_tors(3*m-1) = nu2
                  nu_tors(3*m-0) = nu3

!                 //   rotational barrier
                  v_dih(3*m-2)  = v_tors(3*m-2)
                  v_dih(3*m-1)  = v_tors(3*m-1)
                  v_dih(3*m-0)  = v_tors(3*m-0)

!                 //   phase
                  mu_dih(3*m-2) = mu_tors(3*m-2)
                  mu_dih(3*m-1) = mu_tors(3*m-1)
                  mu_dih(3*m-0) = mu_tors(3*m-0)

!                 //   degeneracy
                  nu_dih(3*m-2) = nu_tors(3*m-2)
                  nu_dih(3*m-1) = nu_tors(3*m-1)
                  nu_dih(3*m-0) = nu_tors(3*m-0)

!                 //   turn on flag
                  iflag = 3

!                 //   proceed to next bond
                  exit

!              //   substitute the values if the line matches
               else if ( ( i .eq. 0  ) .and. ( j .eq. jj ) .and. &
     &                    ( k .eq. kk ) .and. ( l .eq. ll ) ) then

!                 //   for better parameter
                  if ( iflag .le. 1 ) then

!                 //   atoms
                  i_dih(3*m-2)   = i_tors(3*m-2)
                  j_dih(3*m-2)   = j_tors(3*m-2)
                  k_dih(3*m-2)   = k_tors(3*m-2)
                  l_dih(3*m-2)   = l_tors(3*m-2)
                  i_dih(3*m-1)   = i_tors(3*m-1)
                  j_dih(3*m-1)   = j_tors(3*m-1)
                  k_dih(3*m-1)   = k_tors(3*m-1)
                  l_dih(3*m-1)   = l_tors(3*m-1)
                  i_dih(3*m-0)   = i_tors(3*m-0)
                  j_dih(3*m-0)   = j_tors(3*m-0)
                  k_dih(3*m-0)   = k_tors(3*m-0)
                  l_dih(3*m-0)   = l_tors(3*m-0)

!                 //   rotational barrier
                  v_tors(3*m-2)  = v1 * scale_tors
                  v_tors(3*m-1)  = v2 * scale_tors
                  v_tors(3*m-0)  = v3 * scale_tors

!                 //   phase
                  mu_tors(3*m-2) = nint( cos(phi1*pi/180.d0) )
                  mu_tors(3*m-1) = nint( cos(phi2*pi/180.d0) )
                  mu_tors(3*m-0) = nint( cos(phi3*pi/180.d0) )

!                 //   degeneracy
                  nu_tors(3*m-2) = nu1
                  nu_tors(3*m-1) = nu2
                  nu_tors(3*m-0) = nu3

!                 //   rotational barrier
                  v_dih(3*m-2)  = v_tors(3*m-2)
                  v_dih(3*m-1)  = v_tors(3*m-1)
                  v_dih(3*m-0)  = v_tors(3*m-0)

!                 //   phase
                  mu_dih(3*m-2) = mu_tors(3*m-2)
                  mu_dih(3*m-1) = mu_tors(3*m-1)
                  mu_dih(3*m-0) = mu_tors(3*m-0)

!                 //   degeneracy
                  nu_dih(3*m-2) = nu_tors(3*m-2)
                  nu_dih(3*m-1) = nu_tors(3*m-1)
                  nu_dih(3*m-0) = nu_tors(3*m-0)

!                 //   turn on flag
                  iflag = 2

!                 //   for better parameter
                  end if

!              //   substitute the values if the line matches
               else if ( ( i .eq. ii ) .and. ( j .eq. jj ) .and. &
     &                    ( k .eq. kk ) .and. ( l .eq. 0  ) ) then

!                 //   for better parameter
                  if ( iflag .le. 1 ) then

!                 //   atoms
                  i_dih(3*m-2)   = i_tors(3*m-2)
                  j_dih(3*m-2)   = j_tors(3*m-2)
                  k_dih(3*m-2)   = k_tors(3*m-2)
                  l_dih(3*m-2)   = l_tors(3*m-2)
                  i_dih(3*m-1)   = i_tors(3*m-1)
                  j_dih(3*m-1)   = j_tors(3*m-1)
                  k_dih(3*m-1)   = k_tors(3*m-1)
                  l_dih(3*m-1)   = l_tors(3*m-1)
                  i_dih(3*m-0)   = i_tors(3*m-0)
                  j_dih(3*m-0)   = j_tors(3*m-0)
                  k_dih(3*m-0)   = k_tors(3*m-0)
                  l_dih(3*m-0)   = l_tors(3*m-0)

!                 //   rotational barrier
                  v_tors(3*m-2)  = v1 * scale_tors
                  v_tors(3*m-1)  = v2 * scale_tors
                  v_tors(3*m-0)  = v3 * scale_tors

!                 //   phase
                  mu_tors(3*m-2) = nint( cos(phi1*pi/180.d0) )
                  mu_tors(3*m-1) = nint( cos(phi2*pi/180.d0) )
                  mu_tors(3*m-0) = nint( cos(phi3*pi/180.d0) )

!                 //   degeneracy
                  nu_tors(3*m-2) = nu1
                  nu_tors(3*m-1) = nu2
                  nu_tors(3*m-0) = nu3

!                 //   rotational barrier
                  v_dih(3*m-2)  = v_tors(3*m-2)
                  v_dih(3*m-1)  = v_tors(3*m-1)
                  v_dih(3*m-0)  = v_tors(3*m-0)

!                 //   phase
                  mu_dih(3*m-2) = mu_tors(3*m-2)
                  mu_dih(3*m-1) = mu_tors(3*m-1)
                  mu_dih(3*m-0) = mu_tors(3*m-0)

!                 //   degeneracy
                  nu_dih(3*m-2) = nu_tors(3*m-2)
                  nu_dih(3*m-1) = nu_tors(3*m-1)
                  nu_dih(3*m-0) = nu_tors(3*m-0)

!                 //   turn on flag
                  iflag = 2

!                 //   for better parameter
                  end if

!              //   substitute the values if the line matches
               else if ( ( 0 .eq. ii ) .and. ( j .eq. jj ) .and. &
     &                    ( k .eq. kk ) .and. ( l .eq. 0  ) ) then

!                 //   for better parameter
                  if ( iflag .le. 0 ) then

!                 //   atoms
                  i_dih(3*m-2)   = i_tors(3*m-2)
                  j_dih(3*m-2)   = j_tors(3*m-2)
                  k_dih(3*m-2)   = k_tors(3*m-2)
                  l_dih(3*m-2)   = l_tors(3*m-2)
                  i_dih(3*m-1)   = i_tors(3*m-1)
                  j_dih(3*m-1)   = j_tors(3*m-1)
                  k_dih(3*m-1)   = k_tors(3*m-1)
                  l_dih(3*m-1)   = l_tors(3*m-1)
                  i_dih(3*m-0)   = i_tors(3*m-0)
                  j_dih(3*m-0)   = j_tors(3*m-0)
                  k_dih(3*m-0)   = k_tors(3*m-0)
                  l_dih(3*m-0)   = l_tors(3*m-0)

!                 //   rotational barrier
                  v_tors(3*m-2)  = v1 * scale_tors
                  v_tors(3*m-1)  = v2 * scale_tors
                  v_tors(3*m-0)  = v3 * scale_tors

!                 //   phase
                  mu_tors(3*m-2) = nint( cos(phi1*pi/180.d0) )
                  mu_tors(3*m-1) = nint( cos(phi2*pi/180.d0) )
                  mu_tors(3*m-0) = nint( cos(phi3*pi/180.d0) )

!                 //   degeneracy
                  nu_tors(3*m-2) = nu1
                  nu_tors(3*m-1) = nu2
                  nu_tors(3*m-0) = nu3

!                 //   rotational barrier
                  v_dih(3*m-2)  = v_tors(3*m-2)
                  v_dih(3*m-1)  = v_tors(3*m-1)
                  v_dih(3*m-0)  = v_tors(3*m-0)

!                 //   phase
                  mu_dih(3*m-2) = mu_tors(3*m-2)
                  mu_dih(3*m-1) = mu_tors(3*m-1)
                  mu_dih(3*m-0) = mu_tors(3*m-0)

!                 //   degeneracy
                  nu_dih(3*m-2) = nu_tors(3*m-2)
                  nu_dih(3*m-1) = nu_tors(3*m-1)
                  nu_dih(3*m-0) = nu_tors(3*m-0)

!                 //   turn on flag
                  iflag = 1

!                 //   for better parameter
                  end if

!              //   substitute the values if the line matches
               else if ( ( i .eq. ll ) .and. ( j .eq. kk ) .and. &
     &                    ( k .eq. jj ) .and. ( l .eq. ii ) ) then

!                 //   atoms
                  i_dih(3*m-2)   = i_tors(3*m-2)
                  j_dih(3*m-2)   = j_tors(3*m-2)
                  k_dih(3*m-2)   = k_tors(3*m-2)
                  l_dih(3*m-2)   = l_tors(3*m-2)
                  i_dih(3*m-1)   = i_tors(3*m-1)
                  j_dih(3*m-1)   = j_tors(3*m-1)
                  k_dih(3*m-1)   = k_tors(3*m-1)
                  l_dih(3*m-1)   = l_tors(3*m-1)
                  i_dih(3*m-0)   = i_tors(3*m-0)
                  j_dih(3*m-0)   = j_tors(3*m-0)
                  k_dih(3*m-0)   = k_tors(3*m-0)
                  l_dih(3*m-0)   = l_tors(3*m-0)

!                 //   rotational barrier
                  v_tors(3*m-2)  = v1 * scale_tors
                  v_tors(3*m-1)  = v2 * scale_tors
                  v_tors(3*m-0)  = v3 * scale_tors

!                 //   phase
                  mu_tors(3*m-2) = nint( cos(phi1*pi/180.d0) )
                  mu_tors(3*m-1) = nint( cos(phi2*pi/180.d0) )
                  mu_tors(3*m-0) = nint( cos(phi3*pi/180.d0) )

!                 //   degeneracy
                  nu_tors(3*m-2) = nu1
                  nu_tors(3*m-1) = nu2
                  nu_tors(3*m-0) = nu3

!                 //   rotational barrier
                  v_dih(3*m-2)  = v_tors(3*m-2)
                  v_dih(3*m-1)  = v_tors(3*m-1)
                  v_dih(3*m-0)  = v_tors(3*m-0)

!                 //   phase
                  mu_dih(3*m-2) = mu_tors(3*m-2)
                  mu_dih(3*m-1) = mu_tors(3*m-1)
                  mu_dih(3*m-0) = mu_tors(3*m-0)

!                 //   degeneracy
                  nu_dih(3*m-2) = nu_tors(3*m-2)
                  nu_dih(3*m-1) = nu_tors(3*m-1)
                  nu_dih(3*m-0) = nu_tors(3*m-0)

!                 //   turn on flag
                  iflag = 3

!                 //   proceed to next bond
                  exit

!              //   substitute the values if the line matches
               else if ( ( i .eq. 0  ) .and. ( j .eq. kk ) .and. &
     &                    ( k .eq. jj ) .and. ( l .eq. ii ) ) then

!                 //   for better parameter
                  if ( iflag .le. 1 ) then

!                 //   atoms
                  i_dih(3*m-2)   = i_tors(3*m-2)
                  j_dih(3*m-2)   = j_tors(3*m-2)
                  k_dih(3*m-2)   = k_tors(3*m-2)
                  l_dih(3*m-2)   = l_tors(3*m-2)
                  i_dih(3*m-1)   = i_tors(3*m-1)
                  j_dih(3*m-1)   = j_tors(3*m-1)
                  k_dih(3*m-1)   = k_tors(3*m-1)
                  l_dih(3*m-1)   = l_tors(3*m-1)
                  i_dih(3*m-0)   = i_tors(3*m-0)
                  j_dih(3*m-0)   = j_tors(3*m-0)
                  k_dih(3*m-0)   = k_tors(3*m-0)
                  l_dih(3*m-0)   = l_tors(3*m-0)

!                 //   rotational barrier
                  v_tors(3*m-2)  = v1 * scale_tors
                  v_tors(3*m-1)  = v2 * scale_tors
                  v_tors(3*m-0)  = v3 * scale_tors

!                 //   phase
                  mu_tors(3*m-2) = nint( cos(phi1*pi/180.d0) )
                  mu_tors(3*m-1) = nint( cos(phi2*pi/180.d0) )
                  mu_tors(3*m-0) = nint( cos(phi3*pi/180.d0) )

!                 //   degeneracy
                  nu_tors(3*m-2) = nu1
                  nu_tors(3*m-1) = nu2
                  nu_tors(3*m-0) = nu3

!                 //   rotational barrier
                  v_dih(3*m-2)  = v_tors(3*m-2)
                  v_dih(3*m-1)  = v_tors(3*m-1)
                  v_dih(3*m-0)  = v_tors(3*m-0)

!                 //   phase
                  mu_dih(3*m-2) = mu_tors(3*m-2)
                  mu_dih(3*m-1) = mu_tors(3*m-1)
                  mu_dih(3*m-0) = mu_tors(3*m-0)

!                 //   degeneracy
                  nu_dih(3*m-2) = nu_tors(3*m-2)
                  nu_dih(3*m-1) = nu_tors(3*m-1)
                  nu_dih(3*m-0) = nu_tors(3*m-0)

!                 //   turn on flag
                  iflag = 2

!                 //   for better parameter
                  end if

!              //   substitute the values if the line matches
               else if ( ( i .eq. ll ) .and. ( j .eq. kk ) .and. &
     &                    ( k .eq. jj ) .and. ( l .eq. 0  ) ) then

!                 //   for better parameter
                  if ( iflag .le. 1 ) then

!                 //   atoms
                  i_dih(3*m-2)   = i_tors(3*m-2)
                  j_dih(3*m-2)   = j_tors(3*m-2)
                  k_dih(3*m-2)   = k_tors(3*m-2)
                  l_dih(3*m-2)   = l_tors(3*m-2)
                  i_dih(3*m-1)   = i_tors(3*m-1)
                  j_dih(3*m-1)   = j_tors(3*m-1)
                  k_dih(3*m-1)   = k_tors(3*m-1)
                  l_dih(3*m-1)   = l_tors(3*m-1)
                  i_dih(3*m-0)   = i_tors(3*m-0)
                  j_dih(3*m-0)   = j_tors(3*m-0)
                  k_dih(3*m-0)   = k_tors(3*m-0)
                  l_dih(3*m-0)   = l_tors(3*m-0)

!                 //   rotational barrier
                  v_tors(3*m-2)  = v1 * scale_tors
                  v_tors(3*m-1)  = v2 * scale_tors
                  v_tors(3*m-0)  = v3 * scale_tors

!                 //   phase
                  mu_tors(3*m-2) = nint( cos(phi1*pi/180.d0) )
                  mu_tors(3*m-1) = nint( cos(phi2*pi/180.d0) )
                  mu_tors(3*m-0) = nint( cos(phi3*pi/180.d0) )

!                 //   degeneracy
                  nu_tors(3*m-2) = nu1
                  nu_tors(3*m-1) = nu2
                  nu_tors(3*m-0) = nu3

!                 //   rotational barrier
                  v_dih(3*m-2)  = v_tors(3*m-2)
                  v_dih(3*m-1)  = v_tors(3*m-1)
                  v_dih(3*m-0)  = v_tors(3*m-0)

!                 //   phase
                  mu_dih(3*m-2) = mu_tors(3*m-2)
                  mu_dih(3*m-1) = mu_tors(3*m-1)
                  mu_dih(3*m-0) = mu_tors(3*m-0)

!                 //   degeneracy
                  nu_dih(3*m-2) = nu_tors(3*m-2)
                  nu_dih(3*m-1) = nu_tors(3*m-1)
                  nu_dih(3*m-0) = nu_tors(3*m-0)

!                 //   turn on flag
                  iflag = 2

!                 //   for better parameter
                  end if

!              //   substitute the values if the line matches
               else if ( ( i .eq. 0  ) .and. ( j .eq. kk ) .and. &
     &                    ( k .eq. jj ) .and. ( l .eq. 0  ) ) then

!                 //   for better parameter
                  if ( iflag .le. 0 ) then

!                 //   atoms
                  i_dih(3*m-2)   = i_tors(3*m-2)
                  j_dih(3*m-2)   = j_tors(3*m-2)
                  k_dih(3*m-2)   = k_tors(3*m-2)
                  l_dih(3*m-2)   = l_tors(3*m-2)
                  i_dih(3*m-1)   = i_tors(3*m-1)
                  j_dih(3*m-1)   = j_tors(3*m-1)
                  k_dih(3*m-1)   = k_tors(3*m-1)
                  l_dih(3*m-1)   = l_tors(3*m-1)
                  i_dih(3*m-0)   = i_tors(3*m-0)
                  j_dih(3*m-0)   = j_tors(3*m-0)
                  k_dih(3*m-0)   = k_tors(3*m-0)
                  l_dih(3*m-0)   = l_tors(3*m-0)

!                 //   rotational barrier
                  v_tors(3*m-2)  = v1 * scale_tors
                  v_tors(3*m-1)  = v2 * scale_tors
                  v_tors(3*m-0)  = v3 * scale_tors

!                 //   phase
                  mu_tors(3*m-2) = nint( cos(phi1*pi/180.d0) )
                  mu_tors(3*m-1) = nint( cos(phi2*pi/180.d0) )
                  mu_tors(3*m-0) = nint( cos(phi3*pi/180.d0) )

!                 //   degeneracy
                  nu_tors(3*m-2) = nu1
                  nu_tors(3*m-1) = nu2
                  nu_tors(3*m-0) = nu3

!                 //   rotational barrier
                  v_dih(3*m-2)  = v_tors(3*m-2)
                  v_dih(3*m-1)  = v_tors(3*m-1)
                  v_dih(3*m-0)  = v_tors(3*m-0)

!                 //   phase
                  mu_dih(3*m-2) = mu_tors(3*m-2)
                  mu_dih(3*m-1) = mu_tors(3*m-1)
                  mu_dih(3*m-0) = mu_tors(3*m-0)

!                 //   degeneracy
                  nu_dih(3*m-2) = nu_tors(3*m-2)
                  nu_dih(3*m-1) = nu_tors(3*m-1)
                  nu_dih(3*m-0) = nu_tors(3*m-0)

!                 //   turn on flag
                  iflag = 1

!                 //   for better parameter
                  end if

!              //   substitute the values if the line matches
               end if

!           //   look for keyword
            end if

!        //   loop of lines
         end do

!        //   error termination
         if ( iflag .eq. 0 ) then

!           //   comment
            write( 6, '(a)' ) &
     &         'Error termination.'

!           //   comment
            write( 6, '(a,4i5,a,4i5,a)' ) &
     &         'Missing torsion type:', ii, jj, kk, ll, ' for atoms ', &
     &         i_tors(3*m-2), j_tors(3*m-2), &
     &         k_tors(3*m-1), l_tors(3*m-2), '.'

!           //   stop
            stop

!        //   error termination
         end if

!     //   loop of torsion bonds
      end do

!-----------------------------------------------------------------------
!     //   end of subroutine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine imptors_tinker
!***********************************************************************

!     //   shared variables
      use tinker_variables, only : &
     &   pi, scale_imptors, v_imptors, v_dih, ikind, jkind, i_dih, &
     &   j_dih, k_dih, l_dih, i_imptors, j_imptors, k_imptors, &
     &   l_imptors, mu_imptors, nu_imptors, mu_dih, nu_dih, &
     &   i_tri, j_tri, k_tri, l_tri, ntri, ntors_3, iounit_tinker, &
     &   nimptors

!     //   local variables
      implicit none

!     //   error flag
      integer :: ierr

!     //   keywords
      character(len=15) :: keyword

!     //   flag
      integer :: iflag = 0

!     //   integers
      integer :: i, j, k, l, m, n, ii, jj, kk, ll, mm

!     //   real numbers
      real(8) :: v1, phi1

!     //   integers
      integer :: nu1

!-----------------------------------------------------------------------
!     //   improper torsion bond parameters
!-----------------------------------------------------------------------

!     //   counter
      n = ntors_3

!     //   counter
      m = 0

!     //  loop of improper torsion bonds
      do mm = 1, ntri

!        //   atom kinds
         ii = ikind(jkind(i_tri(mm)))
         jj = ikind(jkind(j_tri(mm)))
         kk = ikind(jkind(k_tri(mm)))
         ll = ikind(jkind(l_tri(mm)))

!        //   reset input file
         rewind( iounit_tinker )

!        //   turn off flag
         iflag = 0

!        //   loop of lines
         do

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword

!           //   error confirmation
            if ( ierr .ne. 0 ) exit

!           //   look for keyword
            if ( keyword(1:8) .eq. 'imptors ' ) then

!              //   step back one line
               backspace( iounit_tinker )

!              //   read a line again
               read ( iounit_tinker, *, iostat=ierr ) &
     &            keyword, i, j, k, l, v1, phi1, nu1

!              //   error confirmation
               if ( ierr .ne. 0 ) exit

!              //   substitute the values if the line matches
                if ( ( i .eq. jj ) .and. ( j .eq. ll ) .and. &
     &               ( k .eq. ii ) .and. ( l .eq. kk ) ) then

!                 //   counter
                  m = m + 1

!                 //   rotational barrier
                  v_imptors(m)  = v1 * scale_imptors

!                 //   phase
                  mu_imptors(m) = nint( cos(phi1*pi/180.d0) )

!                 //   degeneracy
                  nu_imptors(m) = nu1

!                 //   counter
                  n = n + 1

!                 //   atoms
                  i_imptors(m) = j_tri(mm)
                  j_imptors(m) = l_tri(mm)
                  k_imptors(m) = i_tri(mm)
                  l_imptors(m) = k_tri(mm)

!                 //   atoms
                  i_dih(n) = i_imptors(m)
                  j_dih(n) = j_imptors(m)
                  k_dih(n) = k_imptors(m)
                  l_dih(n) = l_imptors(m)

!                 //   rotational barrier
                  v_dih(n)  = v_imptors(m)

!                 //   phase
                  mu_dih(n) = mu_imptors(m)

!                 //   degeneracy
                  nu_dih(n) = nu_imptors(m)

!                 //   turn on flag
                  iflag = 1

!                 //   proceed to next bond
                  exit

!              //   substitute the values if the line matches
               else if ( ( i .eq. kk ) .and. ( j .eq. ll ) .and. &
     &                   ( k .eq. ii ) .and. ( l .eq. jj ) ) then

!                 //   counter
                  m = m + 1

!                 //   rotational barrier
                  v_imptors(m)  = v1 * scale_imptors

!                 //   phase
                  mu_imptors(m) = nint( cos(phi1*pi/180.d0) )

!                 //   degeneracy
                  nu_imptors(m) = nu1

!                 //   counter
                  n = n + 1

!                 //   atoms
                  i_imptors(m) = k_tri(mm)
                  j_imptors(m) = l_tri(mm)
                  k_imptors(m) = i_tri(mm)
                  l_imptors(m) = j_tri(mm)

!                 //   atoms
                  i_dih(n) = i_imptors(m)
                  j_dih(n) = j_imptors(m)
                  k_dih(n) = k_imptors(m)
                  l_dih(n) = l_imptors(m)

!                 //   rotational barrier
                  v_dih(n)  = v_imptors(m)

!                 //   phase
                  mu_dih(n) = mu_imptors(m)

!                 //   degeneracy
                  nu_dih(n) = nu_imptors(m)

!                 //   turn on flag
                  iflag = 1

!                 //   proceed to next bond
                  exit

!              //   substitute the values if the line matches
               end if

!           //   look for keyword
            end if

!        //   loop of lines
         end do

!     //   loop of improper torsion bonds
      end do

!-----------------------------------------------------------------------
!     //   number of improper torsions
!-----------------------------------------------------------------------

      nimptors = m

!-----------------------------------------------------------------------
!     //   end of subroutine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine improper_tinker
!***********************************************************************

!     //   shared variables
      use tinker_variables, only : &
     &   scale_improper, eq_improper, fc_improper, ikind, jkind, &
     &   i_improper, j_improper, k_improper, l_improper, &
     &   i_tri, j_tri, k_tri, l_tri, ntri, iounit_tinker, nimproper

!     //   local variables
      implicit none

!     //   error flag
      integer :: ierr

!     //   keywords
      character(len=15) :: keyword

!     //   flag
      integer :: iflag = 0

!     //   integers
      integer :: i, j, k, l, m, n, ii, jj, kk, ll

!     //   real numbers
      real(8) :: fc, phi

!-----------------------------------------------------------------------
!     //   improper bond parameters
!-----------------------------------------------------------------------

!     //   counter
      n = 0

!     //  loop of improper bonds
      do m = 1, ntri

!        //   atom kinds
         ii = ikind(jkind(i_tri(m)))
         jj = ikind(jkind(j_tri(m)))
         kk = ikind(jkind(k_tri(m)))
         ll = ikind(jkind(l_tri(m)))

!        //   reset input file
         rewind( iounit_tinker )

!        //   turn off flag
         iflag = 0

!        //   loop of lines
         do

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword

!           //   error confirmation
            if ( ierr .ne. 0 ) exit

!           //   look for keyword
            if ( keyword(1:9) .eq. 'improper ' ) then

!              //   step back one line
               backspace( iounit_tinker )

!              //   read a line again
               read ( iounit_tinker, *, iostat=ierr ) &
     &            keyword, i, j, k, l, fc, phi

!              //   error confirmation
               if ( ierr .ne. 0 ) exit

!              //   substitute the values if the line matches
               if ( ( i .eq. ii ) .and. ( j .eq. jj ) .and. &
     &              ( k .eq. kk ) .and. ( l .eq. ll ) ) then

!                 //   turn on flag
                  iflag = 1

!                 //   proceed to next bond
                  exit

!              //   substitute the values if the line matches
               else if ( ( i .eq. ll ) .and. ( j .eq. kk ) .and. &
     &                   ( k .eq. jj ) .and. ( l .eq. ii ) ) then

!                 //   turn on flag
                  iflag = 1

!                 //   proceed to next bond
                  exit

!              //   substitute the values if the line matches
               end if

!           //   look for keyword
            end if

!        //   loop of lines
         end do

!        //   improper bonds
         if ( iflag .eq. 1 ) then

!           //   counter
            n = n + 1

!           //   atoms
            i_improper(n) = i_tri(m)
            j_improper(n) = j_tri(m)
            k_improper(n) = k_tri(m)
            l_improper(n) = l_tri(m)

!        //   improper bonds
         end if

!     //   loop of improper bonds
      end do

!     //   counter
      nimproper = n

!-----------------------------------------------------------------------
!     //   improper bond parameters
!-----------------------------------------------------------------------

!     //  loop of improper bonds
      do m = 1, nimproper

!        //   atom kinds
         ii = ikind(jkind(i_improper(m)))
         jj = ikind(jkind(j_improper(m)))
         kk = ikind(jkind(k_improper(m)))
         ll = ikind(jkind(l_improper(m)))

!        //   reset input file
         rewind( iounit_tinker )

!        //   loop of lines
         do

!           //   read a line
            read ( iounit_tinker, *, iostat=ierr ) keyword

!           //   error confirmation
            if ( ierr .ne. 0 ) exit

!           //   look for keyword
            if ( keyword(1:9) .eq. 'improper ' ) then

!              //   step back one line
               backspace( iounit_tinker )

!              //   read a line again
               read ( iounit_tinker, *, iostat=ierr ) &
     &            keyword, i, j, k, l, fc, phi

!              //   error confirmation
               if ( ierr .ne. 0 ) exit

!              //   substitute the values if the line matches
               if ( ( i .eq. ii ) .and. ( j .eq. jj ) .and. &
     &              ( k .eq. kk ) .and. ( l .eq. ll ) ) then

!                 //   force constant
                  fc_improper(m) = fc * scale_improper

!                 //   angle
                  eq_improper(m) = phi

!                 //   proceed to next bond
                  exit

!              //   substitute the values if the line matches
               else if ( ( i .eq. ll ) .and. ( j .eq. kk ) .and. &
     &                   ( k .eq. jj ) .and. ( l .eq. ii ) ) then

!                 //   force constant
                  fc_improper(m) = fc * scale_improper

!                 //   angle
                  eq_improper(m) = phi

!                 //   proceed to next bond
                  exit

!              //   substitute the values if the line matches
               end if

!           //   look for keyword
            end if

!        //   loop of lines
         end do

!     //   loop of improper bonds
      end do

!-----------------------------------------------------------------------
!     //   end of subroutine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine bcp_tinker
!***********************************************************************

!     //   variables
      use tinker_variables, only : &
     &   f_bcp, scale_q_14, bond, bond_13, bond_14, i_bcp, j_bcp, &
     &   nbcp, natom, ncoulomb

!     //   local variables
      implicit none

!     //   error flag
      integer :: i, j, k, m

!----------------------------------------------------------------------
!     //   bonded charge pairs
!----------------------------------------------------------------------

!     //   counter reset
      m = 0

!     //   loop of all atoms
      do i = 1, natom-1

!     //   loop of all atoms
      do j = i+1, natom

!        //   type of pair:  for 1-2 bonded pairs
         if      ( bond(i,j) .eq. 1 ) then

!           //   counter update
            m = m + 1

!        //   for 1-3 bonded pairs
         else if ( bond_13(i,j) .eq. 1 ) then

!           //   counter update
            m = m + 1

!        //   for 1-4 bonded pairs
         else if ( bond_14(i,j) .eq. 1 ) then

!           //   counter update
            m = m + 1

!        //   for nonbonded pairs
         else

!           //   do nothing
            cycle

!        //   type of bonded pairs
         end if

!     //   loop of all atoms
      end do

!     //   loop of all atoms
      end do

!     //   total number of bonded pairs
      nbcp = m

!----------------------------------------------------------------------
!     //   bonded charge pairs
!----------------------------------------------------------------------

!     //   atom i
      allocate( i_bcp(nbcp) )

!     //   atom j
      allocate( j_bcp(nbcp) )

!     //   scaling factor
      allocate( f_bcp(nbcp) )

!----------------------------------------------------------------------
!     //   bonded charge pairs
!----------------------------------------------------------------------

!     //   bonded pairs
      m = 0

!     //   bonded pairs
      k = 0

!     //   loop of all atoms
      do i = 1, natom-1

!     //   loop of all atoms
      do j = i+1, natom

!        //   type of pair:  for 1-2 bonded pairs
         if      ( bond(i,j) .eq. 1 ) then

!           //   update bonded pairs
            m = m + 1

!           //   atom i
            i_bcp(m) = i

!           //   atom j
            j_bcp(m) = j

!           //   scaling factor
            f_bcp(m) = 0.d0

!           //   update bonded pairs
            k = k + 1

!        //   for 1-3 bonds
         else if ( bond_13(i,j) .eq. 1 ) then

!           //   update bonded pairs
            m = m + 1

!           //   atom i
            i_bcp(m) = i

!           //   atom j
            j_bcp(m) = j

!           //   scaling factor
            f_bcp(m) = 0.d0

!           //   update bonded pairs
            k = k + 1

!        //   for 1-4 bonded pairs
         else if ( bond_14(i,j) .eq. 1 ) then

!           //   update bonded pairs
            m = m + 1

!           //   atom i
            i_bcp(m) = i

!           //   atom j
            j_bcp(m) = j

!           //   scaling factor
            f_bcp(m) = scale_q_14

!           //   update bonded pairs
            if ( scale_q_14 .eq. 0.d0 ) k = k + 1

!        //   type of pair:  for nonbonded pairs
         else

!           //   do nothing
            cycle

!        //   type of pair:  for 1-2 bonded pairs
         end if

!     //   loop of all atoms
      end do

!     //   loop of all atoms
      end do

!     //   number of coulomb interactions
      ncoulomb = natom*(natom-1)/2 - k

!-----------------------------------------------------------------------
!     //   end of subroutine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine charge_tinker
!***********************************************************************

!     //   variables
      use tinker_variables, only : q, q_param, qsum, jkind, natom

!     //   local variables
      implicit none

!     //   integers
      integer :: i

!-----------------------------------------------------------------------
!     //   charges
!-----------------------------------------------------------------------

      do i = 1, natom
         q(i) = q_param(jkind(i))
      end do

!-----------------------------------------------------------------------
!     //   total charge
!-----------------------------------------------------------------------

      qsum = 0.d0

      do i = 1, natom
         qsum = qsum + q(i)
      end do

!-----------------------------------------------------------------------
!     //   end of subroutine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine lj_tinker
!***********************************************************************

!     //   variables
      use tinker_variables, only : &
     &   eps14_param, sig14_param, eps_lj, sig_lj, eps, sig, vdwindex, &
     &   epsilonrule, scale_sigma, eps_param, sig_param, radiusrule, &
     &   bond, bond_13, bond_14, ikind, jkind, natom, i_lj, j_lj, nlj

!     //   local variables
      implicit none

!     //   integers
      integer :: i, j, m

!     //   real numbers
      real(8) :: sig_ii, sig_jj, eps_ii, eps_jj

!     //   initialize lj pair
      m = 0

!     //   loop of all atoms
      do i = 1, natom-1

!     //   loop of all atoms
      do j = i+1, natom

!        //   type of pairs: for 1-2 bonded pairs
         if ( bond(i,j) .eq. 1 ) then

!           //   epsilon
            eps(i,j) = 0.d0
            eps(j,i) = eps(i,j)

!           //   sigma
            sig(i,j) = 0.d0
            sig(j,i) = sig(i,j)

!        //   for 1-3 bonded pairs
         else if ( bond_13(i,j) .eq. 1 ) then

!           //   epsilon
            eps(i,j) = 0.d0
            eps(j,i) = eps(i,j)

!           //   sigma
            sig(i,j) = 0.d0
            sig(j,i) = sig(i,j)

!        //   for 1-4 bonded pairs
         else if ( bond_14(i,j) .eq. 1 ) then

!           //   update lj pair
            m = m + 1

!           //   epsilon
            if ( vdwindex(1:6) .eq. 'TYPE  ' ) then
               eps_ii = abs(eps14_param(jkind(i)))
               eps_jj = abs(eps14_param(jkind(j)))
            else if( vdwindex(1:6) .eq. 'CLASS ' ) then
               eps_ii = abs(eps14_param(ikind(jkind(i))))
               eps_jj = abs(eps14_param(ikind(jkind(j))))
            else
               eps_ii = abs(eps14_param(jkind(i)))
               eps_jj = abs(eps14_param(jkind(j)))
            end if

            if      ( epsilonrule(1:11) .eq. 'GEOMETRIC  ' ) then
               eps(i,j) = sqrt( eps_ii * eps_jj )
            else if ( epsilonrule(1:11) .eq. 'ARITHMETIC ' ) then
               eps(i,j) = 0.5d0 * ( eps_ii + eps_jj )
            else
               eps(i,j) = sqrt( eps_ii * eps_jj )
            end if

            eps(j,i) = eps(i,j)

!           //   sigma
            if ( vdwindex(1:6) .eq. 'TYPE  ' ) then
               sig_ii = sig14_param(jkind(i))
               sig_jj = sig14_param(jkind(j))
            else if( vdwindex(1:6) .eq. 'CLASS ' ) then
               sig_ii = sig14_param(ikind(jkind(i)))
               sig_jj = sig14_param(ikind(jkind(j)))
            else
               sig_ii = sig14_param(jkind(i))
               sig_jj = sig14_param(jkind(j))
            end if

            if       ( radiusrule(1:11) .eq. 'GEOMETRIC  ' ) then
               sig(i,j) = sqrt( sig_ii * sig_jj )
            else if ( radiusrule(1:11) .eq. 'ARITHMETIC ' ) then
               sig(i,j) = 0.5d0 * ( sig_ii + sig_jj )
            else
               sig(i,j) = 0.5d0 * ( sig_ii + sig_jj )
            end if

            sig(i,j) = sig(i,j) * scale_sigma
            sig(j,i) = sig(i,j)

!           //   update lj pair

            if ( eps(i,j) .eq. 0.d0 ) then
               m = m - 1
            else
               i_lj(m)   = i
               j_lj(m)   = j
               sig_lj(m) = sig(i,j)
               eps_lj(m) = eps(i,j)
            end if

!        //   for nonbonded pairs
         else

!           //   update lj pair
            m = m + 1

!           //   epsilon
            if ( vdwindex(1:6) .eq. 'TYPE  ' ) then
               eps_ii = abs(eps_param(jkind(i)))
               eps_jj = abs(eps_param(jkind(j)))
            else if( vdwindex(1:6) .eq. 'CLASS ' ) then
               eps_ii = abs(eps_param(ikind(jkind(i))))
               eps_jj = abs(eps_param(ikind(jkind(j))))
            else
               eps_ii = abs(eps_param(jkind(i)))
               eps_jj = abs(eps_param(jkind(j)))
            end if

            if      ( epsilonrule(1:11) .eq. 'GEOMETRIC  ' ) then
               eps(i,j) = sqrt( eps_ii * eps_jj )
            else if ( epsilonrule(1:11) .eq. 'ARITHMETIC ' ) then
               eps(i,j) = 0.5d0 * ( eps_ii + eps_jj )
            else
               eps(i,j) = sqrt( eps_ii * eps_jj )
            end if

            eps(j,i) = eps(i,j)

!           //   sigma
            if ( vdwindex(1:6) .eq. 'TYPE  ' ) then
               sig_ii = sig_param(jkind(i))
               sig_jj = sig_param(jkind(j))
            else if( vdwindex(1:6) .eq. 'CLASS ' ) then
               sig_ii = sig_param(ikind(jkind(i)))
               sig_jj = sig_param(ikind(jkind(j)))
            else
               sig_ii = sig_param(jkind(i))
               sig_jj = sig_param(jkind(j))
            end if

            if      ( radiusrule(1:11) .eq. 'GEOMETRIC  ' ) then
               sig(i,j) = sqrt( sig_ii * sig_jj )
            else if ( radiusrule(1:11) .eq. 'ARITHMETIC ' ) then
               sig(i,j) = 0.5d0 * ( sig_ii + sig_jj )
            else
               sig(i,j) = 0.5d0 * ( sig_ii + sig_jj )
            end if

            sig(i,j) = sig(i,j) * scale_sigma
            sig(j,i) = sig(i,j)

!           //   update lj pair
            if ( eps(i,j) .eq. 0.d0 ) then
               m = m - 1
            else
               i_lj(m)   = i
               j_lj(m)   = j
               sig_lj(m) = sig(i,j)
               eps_lj(m) = eps(i,j)
            end if

!        //   type of pairs
         end if

!     //   loop of all atoms
      end do

!     //   loop of all atoms
      end do

!     //   number of lj pairs
      nlj = m

!-----------------------------------------------------------------------
!     //   end of subroutine
!-----------------------------------------------------------------------

      return
      end





!**********************************************************************
      subroutine write_output
!**********************************************************************

#ifdef nextver

!     //   variables
      use tinker_variables, only : &
     &   x, y, z, bohr2ang, eq_lin, fc_lin, eq_ub, fc_ub, bondunit, &
     &   har2kcal, deg2rad, angleunit, fc_angl, eq_angl, vgrid_cmap, &
     &   f_bcp, eq_improper, fc_improper, q, eps_ljpair, sig_ljpair, &
     &   eps_ljbond, sig_ljbond, epsilonrule, radiusrule, har2kcal, &
     &   v_dih, i_dih, j_dih, k_dih, l_dih, i_bcp, j_bcp, ikind_numbre, &
     &   i_angl, j_angl, k_angl, i_lin, j_lin, symbol_numbre, rin_lj, &
     &   rout_lj, qsum, natom, nbond, nub, nlin, nljbond_correct, &
     &   nangl, ntors, nimptors, ndih, nimproper, natom, nbcp, nlj, &
     &   iounit_piin, iounit_pimm, iounit_xyz, mu_dih, nu_dih, &
     &   i_improper, j_improper, k_improper, l_improper, iounit_geom, &
     &   ncmap, nkind_cmap, ngrid_cmap, exist_cmap, i_cmap, j_cmap, &
     &   k_cmap, l_cmap, m_cmap, i_ljbond, j_ljbond, i_ljpair, nljpair

!     //   initialize
      implicit none

!     //   integers
      integer :: i, j, k, l, m, n, m1, n1

!     //   real numbers
      real(8) :: x1, x2, v1, c1, c2, fc, phi

#else

!     //   variables
      use tinker_variables, only : &
     &   x, y, z, bohr2ang, eq_lin, fc_lin, eq_ub, fc_ub, bondunit, &
     &   har2kcal, deg2rad, angleunit, fc_angl, eq_angl, vgrid_cmap, &
     &   f_bcp, eq_improper, fc_improper, q, eps, sig, sig_lj, eps_lj, &
     &   v_dih, har2kcal, i_dih, j_dih, k_dih, l_dih, i_bcp, j_bcp, &
     &   i_angl, j_angl, k_angl, i_lin, j_lin, symbol_numbre, &
     &   rin_lj, rout_lj, qsum, natom, nbond, nub, nlin, ikind_numbre, &
     &   nangl, ntors, nimptors, ndih, nimproper, natom, nbcp, nlj, &
     &   iounit_piin, iounit_pimm, iounit_xyz, mu_dih, nu_dih, &
     &   i_improper, j_improper, k_improper, l_improper, iounit_geom, &
     &   ncmap, nkind_cmap, ngrid_cmap, exist_cmap, i_cmap, j_cmap, &
     &   k_cmap, l_cmap, m_cmap

!     //   initialize
      implicit none

!     //   integers
      integer :: i, j, k, l, m, n, m1, n1, ierr

!     //   real numbers
      real(8) :: x1, x2, v1, c1, c2, sig_ij, eps_ij, fc, phi

#endif

!----------------------------------------------------------------------
!     //   print number of bonds
!----------------------------------------------------------------------

!     //   comment
      write( 6, '(a)' )
      write( 6, '(a)' ) 'Summary:'
      write( 6, '(a)' )

!     //   comment
      write( 6, '(a,f9.2)' )  'lj inner cutoff (bohr) ', rin_lj
      write( 6, '(a,f9.2)' )  'lj outer cutoff (bohr) ', rout_lj

      write( 6, '(a,i9)' )    'number of atoms:       ', natom
      write( 6, '(a,i9)' )    'bond connections:      ', nbond
      write( 6, '(a,i9)' )    'urey-bradley terms:    ', nub
      write( 6, '(a,i9)' )    'number of linear bonds:', nlin
      write( 6, '(a,i9)' )    'number of angles:      ', nangl
      write( 6, '(a,i9)' )    'number of torsions:    ', ntors
      write( 6, '(a,i9)' )    'torsion terms:         ', ntors*3
      write( 6, '(a,i9)' )    'improper torsions:     ', nimptors
      write( 6, '(a,i9)' )    'number of dihedrals:   ', ndih
      write( 6, '(a,i9)' )    'number of impropers:   ', nimproper
      write( 6, '(a,i9)' )    'number of cmap terms:  ', ncmap
      write( 6, '(a,i9)' )    'number of cmap types:  ', nkind_cmap
      write( 6, '(a,i9)' )    'number of charges:     ', natom
      write( 6, '(a,i9)' )    'bonded charge pairs:   ', nbcp
      write( 6, '(a,i9)' )    'number of lj pairs:    ', nlj
      write( 6, '(a,f9.2)' )  'net charge:            ', qsum
      write( 6, '(a,f11.4)' ) 'hartree to kcal/mol: ',   har2kcal

!----------------------------------------------------------------------
!     //   print to centroid.dat
!----------------------------------------------------------------------

!     //   conversion factor from angstrom to bohr
      c1 = 1.d0 / bohr2ang

!     //   conversion
      x(:,:) = x(:,:) * c1
      y(:,:) = y(:,:) * c1
      z(:,:) = z(:,:) * c1

!     //   print geometry
      do i = 1, natom
         write ( iounit_geom, '(3e24.16)' ) x(i,1), y(i,1), z(i,1)
      end do

!----------------------------------------------------------------------
!     //   print to centroid.xyz
!----------------------------------------------------------------------

!     //   print xyz
      write ( iounit_xyz, '(i8)' ) natom

!     //   print xyz
      write ( iounit_xyz, '(a)' ) 'ANGSTROM'

!     //   print xyz
      do i = 1, natom
!         write ( iounit_xyz, '(a,2x,3f10.5)' )
!     &      symbol(i)(1:3), x(i,1)/c1, y(i,1)/c1, z(i,1)/c1
         write ( iounit_xyz, '(a2,1x,3f10.5,i4,1x,a1)' ) &
     &      symbol_numbre(i)(1:2), x(i,1)/c1, y(i,1)/c1, z(i,1)/c1, &
     &      ikind_numbre(i), 'B'
      end do

!     //   print xyz
      write ( iounit_xyz, '(a)' )

!----------------------------------------------------------------------
!     //   print to input.dat
!----------------------------------------------------------------------

!     //   method
      write( iounit_piin, '(a)' ) '<method>'

!     //   method
      write( iounit_piin, '(a)' ) 'STATIC'

!     //   blank line
      write( iounit_piin, '(a)' )

!     //   potential
      write( iounit_piin, '(a)' ) '<ipotential>'

!     //   potential
      write( iounit_piin, '(a)' ) 'MM'

!     //   blank line
      write( iounit_piin, '(a)' )

!     //   geometry file
      write( iounit_piin, '(a)' ) '<input_style>'

!     //   geometry file
      write( iounit_piin, '(a)' ) 'NEW'

!     //   blank line
      write( iounit_piin, '(a)' )

!----------------------------------------------------------------------
!     //   print to sreen
!----------------------------------------------------------------------

!     //   blank line
!      write( 6, '(a)' )

!     //   blank line
!      write( 6, '(a)' )

!     //   blank line
!      write( 6, '(a)' )
!     &   'Copy the following when input_style = OLD is preferred.'

!     //   blank line
!      write( 6, '(a)' )

!     //   geometry file
!      write( 6, '(a)' ) '<input_style>'

!     //   geometry file
!      write( 6, '(a)' ) 'OLD'

!     //   blank line
!      write( 6, '(a)' )

!     //   number of atoms
!      write( 6, '(a)' ) '<natom>'

!     //   number of atoms
!      write( 6, '(i5)' ) natom

!     //   blank line
!      write( 6, '(a)' )

!     //   number of atomic species
!      write( 6, '(a)' ) '<nspec>'

!     //   number of atomic species
!      write( 6, '(i5)' ) natom

!     //   atomic species and geometry

!      do i = 1, natom
!         write ( 6, '(a3,2x,f10.5,a2)' )
!     &      symbol(i)(1:3), amumass(i), ' 1'
!      end do

!     //   blank line
!      write( 6, '(a)' )

!----------------------------------------------------------------------
!     //   print to mm.dat
!----------------------------------------------------------------------

!     //   conversion factor: from angstrom to bohr
      c1 = 1.d0 / bohr2ang

!     //   conversion factor:
!     //   from kcal/mol angstrom**-2 to hartree bohr**-2
      c2 = bohr2ang * bohr2ang / har2kcal * ( 2.d0 * bondunit )

!     //   conversion
      eq_lin(:) = eq_lin(:) * c1

!     //   conversion
      fc_lin(:) = fc_lin(:) * c2

!     //   conversion
      eq_ub(:) = eq_ub(:) * c1

!     //   conversion
      fc_ub(:) = fc_ub(:) * c2

!     //   keyword
      write( iounit_pimm, '(a)' ) '<linear_bonds>'

!     //   number of bonds
      write( iounit_pimm, '(i8)' ) nlin

!     //   loop of linear bonds
      do m = 1, nlin

!        //   atom i
         i = i_lin(m)

!        //   atom j
         j = j_lin(m)

!        //   equilibrium bond length in bohr
         x1 = eq_lin(m)

!        //   force constant in hartree bohr**-2
         x2 = fc_lin(m)

!        //   atoms i j, bond length, force constant
         write( iounit_pimm, '(2i8,2e16.8)' ) i, j, x1, x2

!     //   loop of linear bonds
      end do

!     //   a blank line
      write( iounit_pimm, * )

!----------------------------------------------------------------------
!     //   print to mm.dat
!----------------------------------------------------------------------

!     //   conversion factor:
!     //   from degree to degree
      c1 = 1.d0

!     //   conversion factor:
!     //   from kcal/mol radian**-2 to hartree degree**-2
      c2 = deg2rad * deg2rad / har2kcal * ( 2.d0 * angleunit )

!     //   conversion
      eq_angl(:) = eq_angl(:) * c1

!     //   conversion
      fc_angl(:) = fc_angl(:) * c2

!     //   keyword
      write( iounit_pimm, '(a)' ) '<angular_bonds>'

!     //   number of angular bonds
      write( iounit_pimm, '(i8)' ) nangl

!     //   loop of angular bonds
      do m = 1, nangl

!        //   atom i
         i = i_angl(m)

!        //   atom j
         j = j_angl(m)

!        //   atom k
         k = k_angl(m)

!        //   equilbrium angle in degree
         x1 = eq_angl(m)

!        //   force constant in hartree degree**-2
         x2 = fc_angl(m)

!        //   atoms i j k, equilibrium angle, force constant
         write( iounit_pimm, '(3i8,2e16.8)' ) i, j, k, x1, x2

!     //   loop of angular bonds
      end do

!     //   a blank line
      write( iounit_pimm, * )

!----------------------------------------------------------------------
!     //   print to mm.dat
!----------------------------------------------------------------------

!     //   conversion factor:
!     //   from kcal/mol to hartree
      c1 = 1.d0 / har2kcal

!     //   conversion
      v_dih(:) = v_dih(:) * c1

!     //   keyword
      write( iounit_pimm, '(a)' ) '<dihedral_bonds>'

!     //   dihedral bonds: each described by sum of three terms
      write( iounit_pimm, '(i8)' ) ndih

!     //   loop of dihedral bonds
      do m = 1, ndih

!        //   atom i
         i = i_dih(m)

!        //   atom j
         j = j_dih(m)

!        //   atom k
         k = k_dih(m)

!        //   atom l
         l = l_dih(m)

!        //   1st term: barrier height in hartree
         v1 =  v_dih(m)

!        //   1st term: phase factor
         m1 = mu_dih(m)

!        //   1st term: degeneracy
         n1 = nu_dih(m)

!        //   1st term: atoms i j k l, barrier, phase, degeneracy
         write( iounit_pimm, '(4i8,e16.8,2i3)' ) &
     &      i, j, k, l, v1, n1, m1

!     //   loop of dihedral bonds
      end do

!     //   a blank line
      write( iounit_pimm, * )

!----------------------------------------------------------------------
!     //   print to mm.dat
!----------------------------------------------------------------------

!     //   conversion factor:
!     //   from kcal/mol/radian**2 to hartree/degree**2
      c1 = deg2rad * deg2rad / har2kcal

!     //   conversion
      fc_improper(:) = fc_improper(:) * c1

!     //   keyword
      write( iounit_pimm, '(a)' ) '<improper_bonds>'

!     //   dihedral bonds: each described by sum of three terms
      write( iounit_pimm, '(i8)' ) nimproper

!     //   loop of improper bonds
      do m = 1, nimproper

!        //   atom i
         i = i_improper(m)

!        //   atom j
         j = j_improper(m)

!        //   atom k
         k = k_improper(m)

!        //   atom l
         l = l_improper(m)

!        //   force constant
         fc = fc_improper(m)

!        //   1st term: phase factor
         phi = eq_improper(m)

!        //   1st term: atoms i j k l, barrier, phase, degeneracy
         write( iounit_pimm, '(4i8,2e16.8)' ) i, j, k, l, phi, fc

!     //   loop of improper bonds
      end do

!     //   a blank line
      write( iounit_pimm, * )

!-----------------------------------------------------------------------
!     //   cmap list
!-----------------------------------------------------------------------

!     //   print number of cmaps
      write( iounit_pimm, '(a)' ) '<ncmap>'
      write( iounit_pimm, '(i8)' ) nkind_cmap

!     //   print list
      do m1 = 1, nkind_cmap

!        //   list
         i = i_cmap(1,m1)
         j = j_cmap(1,m1)
         k = k_cmap(1,m1)
         l = l_cmap(1,m1)
         m = l_cmap(2,m1)
         n = m_cmap(m1)

!        //   print atoms
         write( iounit_pimm, '(9i8)' ) i, j, k, l, j, k, l, m, n

!     //   print list
      end do

!     //   a blank line
      write( iounit_pimm, * )

!-----------------------------------------------------------------------
!     //   cmap list
!-----------------------------------------------------------------------

!     //   print number of cmaps
      write( iounit_pimm, '(a)' ) '<nkind_cmap>'
      write( iounit_pimm, '(i8)' ) ncmap

!     //   cmap counter
      m = 0

!     //   loop of cmap reference
      do n = 1, nkind_cmap

!        //   flag
         if ( .not. exist_cmap(n) ) cycle

!        //   cmap counter
         m = m + 1

!        //   print number of cmaps
         write( iounit_pimm, '(i8)' ) m

!        //   read a line again
         do j = 1, ngrid_cmap
!            write( iounit_pimm, '(f16.8)' )
!     &         ( vgrid_cmap(i,j,n)*har2kcal, i = 1, ngrid_cmap )
            write( iounit_pimm, '(f16.8)' ) &
     &         ( vgrid_cmap(j,i,n)*har2kcal, i = 1, ngrid_cmap )
         end do

!     //   loop of cmap reference
      end do

!     //   a blank line
      write( iounit_pimm, * )

!----------------------------------------------------------------------
!     //   print to mm.dat
!----------------------------------------------------------------------

!     //   keyword
      write( iounit_pimm, '(a)' ) '<charges>'

!     //   number of atoms
      write( iounit_pimm, '(i8)' ) natom

!     //   loop of all atoms
      do i = 1, natom

!        //   atom, charge
         write( iounit_pimm, '(i8,e16.8)' ) i, q(i)

!     //   loop of all atoms
      end do

!     //   a blank line
      write( iounit_pimm, * )

!----------------------------------------------------------------------
!     //   print to mm.dat
!----------------------------------------------------------------------

!     //   keyword
      write( iounit_pimm, '(a)' ) '<nbcp>'

!     //   number of bonded charge pairs
      write( iounit_pimm, '(i8)' ) nbcp

!     //   loop of bonded charge pairs
      do m = 1, nbcp

!        //   atoms i j, scaling factor
         write( iounit_pimm, '(2i8,e16.8)' ) &
     &      i_bcp(m), j_bcp(m), f_bcp(m)

!     //   loop of bonded charge pairs
      end do

!     //   a blank line
      write( iounit_pimm, * )

#ifdef nextver

!----------------------------------------------------------------------
!     //   lj pair
!----------------------------------------------------------------------

!     //   print
      write( 6, '(a)' )
      write( 6, '(a)' ) 'Pairwise format for Lennard Jones parameters.'
      write( 6, '(a)' )

!     //   conversion factor from angstrom to bohr
      c1 = 1.d0 / bohr2ang

!     //   conversion factor from kcal/mol to hartree
      c2 = 1.d0 / har2kcal

!     //   keyword
      write( iounit_pimm, '(a)' ) '<ljrule>'

!     //   lj combination rules
      write( iounit_pimm, '(1x,a,1x,a)' ) &
     &   trim(epsilonrule), trim(radiusrule)

!     //   a blank line
      write( iounit_pimm, * )

!     //   keyword
      write( iounit_pimm, '(a)' ) '<ljcutoff>'

!     //   lj cut off parameters
      write( iounit_pimm, '(2e16.8)' ) rin_lj, rout_lj

!     //   a blank line
      write( iounit_pimm, * )

!     //   keyword
      write( iounit_pimm, '(a)' ) '<ljpair>'

!     //   lj atoms, lj pairs with scaling
      write( iounit_pimm, '(2i8)' ) nljpair

!     //   lj parameters of atoms
      do i = 1, nljpair
         write( iounit_pimm, '(i8,2e16.8)' ) &
     &      i_ljpair(i), eps_ljpair(i)*c2, sig_ljpair(i)*c1
      end do

!     //   a blank line
      write( iounit_pimm, * )

!     //   keyword
      write( iounit_pimm, '(a)' ) '<ljbond>'

!     //   lj atoms, lj pairs with scaling
      write( iounit_pimm, '(2i8)' ) nljbond_correct

!     //   lj pairs with scaling
      do m = 1, nljbond_correct
         write( iounit_pimm, '(2i8,2e16.8)' ) &
     &      i_ljbond(m), j_ljbond(m), &
     &      eps_ljbond(m)*c2, sig_ljbond(m)*c1
      end do

!     //   a blank line
      write( iounit_pimm, * )

#else

!----------------------------------------------------------------------
!     //   lennard-jones
!----------------------------------------------------------------------

!     //   print
      write( 6, '(a)' )
      write( 6, '(a)' ) 'Explicit format for Lennard Jones parameters.'

!     //   counter reset
      m = 0

!     //   loop of all atoms
      do i = 1, natom-1

!     //   loop of all atoms
      do j = i+1, natom

!        //   for nonzero interactions only
         if ( eps(i,j) .ne. 0.d0 ) then

!           //   counter update
            m = m + 1

!        //   for nonzero interactions only
         end if

!     //   loop of all atoms
      end do

!     //   loop of all atoms
      end do

!----------------------------------------------------------------------
!     //   check the number of lj interactions
!----------------------------------------------------------------------

      if ( m .eq. nlj ) then

!        //   comment
         write( 6, '(a)' )
         write( 6, '(a)' ) 'Checking LJ interactions: OK.'
         write( 6, '(a)' )

!        //   flag
         ierr = 0

      else

!        //   comment
         write( 6, '(a)' )
         write( 6, '(a)' )  'Checking LJ interactions: FAILED.'
         write( 6, '(a,2i5)' ) 'LJ interactions differ:', m, nlj
         write( 6, '(a)' )

!        //   flag
         ierr = 1

      end if

      call error_handling( ierr, 'write_output', 12 )

!----------------------------------------------------------------------
!     //   print to mm.dat
!----------------------------------------------------------------------

!     //   conversion factor:
!     //   from angstrom to bohr
      c1 = 1.d0 / bohr2ang

!     //   conversion factor:
!     //   from kcal/mol to hartree
      c2 = 1.d0 / har2kcal

!     //   conversion
      sig(:,:) = sig(:,:) * c1

!     //   conversion
      eps(:,:) = eps(:,:) * c2

!     //   conversion
      sig_lj(:) = sig_lj(:) * c1

!     //   conversion
      eps_lj(:) = eps_lj(:) * c2

!     //   keyword
      write( iounit_pimm, '(a)' ) '<lennard-jones>'

!     //   number of lj interactions
      write( iounit_pimm, '(i8)' ) nlj

!     //   lj cut off parameters
      write( iounit_pimm, '(2e16.8)' ) rin_lj, rout_lj

!     //   loop of all atoms
      do i = 1, natom-1

!     //   loop of all atoms
      do j = i+1, natom

!        //   for nonzero interactions only
         if ( eps(i,j) .ne. 0.d0 ) then

!           //   sigma in bohr
            sig_ij = sig(i,j)

!           //   epsilon in hartree
            eps_ij = eps(i,j)

!           //   atoms i j, epsilon, sigma
            write( iounit_pimm, '(2i8,2e16.8)' ) i, j, eps_ij, sig_ij

!        //   for nonzero interactions only
         end if

!     //   loop of all atoms
      end do

!     //   loop of all atoms
      end do

!     //   a blank line
      write( iounit_pimm, * )

#endif

!-----------------------------------------------------------------------
!     //   ewald sum options
!-----------------------------------------------------------------------

!     //   ewald parameters
      write( iounit_pimm, '(a)' ) '<ewald>'
      write( iounit_pimm, '(a)' ) '1.d-8 11.d0 0'

!     //   a blank line
      write( iounit_pimm, * )

!     //   ewald parameters
      write( iounit_pimm, '(a)' ) '<ewald_type>'
      write( iounit_pimm, '(a)' ) 'STANDARD'

!     //   a blank line
      write( iounit_pimm, * )

!     //   ewald parameters
      write( iounit_pimm, '(a)' ) '<pme_mesh_ewald>'
      write( iounit_pimm, '(a)' ) '1.09'

!     //   a blank line
      write( iounit_pimm, * )

!-----------------------------------------------------------------------
!     //   end of subroutine
!-----------------------------------------------------------------------

      return
      end





!**********************************************************************
      subroutine error_handling( ierr, sub_name, len_name )
!**********************************************************************

!     //   length of subroutine name
      integer :: len_name

!     //   subroutine name
      character(len=len_name) :: sub_name

!     //   integer
      integer :: ierr

!     //   test error: if there is an error
      if ( ierr .ne. 0 ) then

!        //   comment
         write( 6, '(a)' ) 'Error termination.'

!        //   comment
         write( 6, '(a,a,a)' ) 'Subroutine ', sub_name, '.'

!        //   stop
         stop

!     //   if there is no error
      else

!        //  do nothing
         continue

!     //   test error
      end if

!-----------------------------------------------------------------------
!     //   end of subroutine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine check_force_mm_bond
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use tinker_variables, only : &
     &   x, y, z, pot, fx, fy, fz, vir, fc_lin, eq_lin, har2kcal, &
     &   bohr2ang, symbol, i_lin, j_lin, nbead, nbond, nlin

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: m, k, i, j

      real(8) :: xij, yij, zij, rij, dr, const, fxi, fyi, fzi

      real(8) :: dpot

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( nlin .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   print out                                                  */
!-----------------------------------------------------------------------

      write( 6, '(a,12x,a,6x,a,5x,a,4x,a,4x,a)' ) &
     &   'Type', 'Atom Names', 'Ideal', 'Actual', 'Energy', 'Fconst'
      write( 6, '(a)' )

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do k = 1, nbond

            i = i_lin(k)
            j = j_lin(k)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom ( xij, yij, zij )

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            dr = ( rij - eq_lin(k) )

            pot(m) = pot(m) + 0.5d0*fc_lin(k)*dr*dr

            const = - fc_lin(k)*dr/rij

            fxi = const*xij
            fyi = const*yij
            fzi = const*zij

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

            dpot = 0.5d0*fc_lin(k)*dr*dr

            write( 6, '(a4,6x,2(i5,a1,a3),4f10.4)' ) &
     &         'Bond', &
     &         i, '-', symbol(i), &
     &         j, '-', symbol(j), &
     &         eq_lin(k)*bohr2ang, &
     &         rij*bohr2ang, &
     &         dpot*har2kcal, &
     &         fc_lin(k)*har2kcal/bohr2ang/bohr2ang

         end do

      end do

      write( 6, '(a)' )

      return
      end






!***********************************************************************
      subroutine check_force_mm_ub
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use tinker_variables, only : &
     &   x, y, z, fx, fy, fz, vir, pot, har2kcal, bohr2ang, symbol, &
     &   fc_lin, eq_lin, i_lin, j_lin, nbead, nlin, nbond

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer ::  m, k, i, j

      real(8) :: xij, yij, zij, rij, dr, const, fxi, fyi, fzi

      real(8) :: dpot

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( nlin .eq. nbond ) return

!-----------------------------------------------------------------------
!     /*   print out                                                  */
!-----------------------------------------------------------------------

      write( 6, '(a,12x,a,6x,a,5x,a,4x,a,4x,a)' ) &
     &   'Type', 'Atom Names', 'Ideal', 'Actual', 'Energy', 'Fconst'
      write( 6, '(a)' )

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do k = nbond+1, nlin

            i = i_lin(k)
            j = j_lin(k)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom ( xij, yij, zij )

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            dr = ( rij - eq_lin(k) )

            pot(m) = pot(m) + 0.5d0*fc_lin(k)*dr*dr

            const = - fc_lin(k)*dr/rij

            fxi = const*xij
            fyi = const*yij
            fzi = const*zij

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

            dpot = 0.5d0*fc_lin(k)*dr*dr

            write( 6, '(a8,2x,2(i5,a1,a3),4f10.4)' ) &
     &         'Ureybrad', &
     &         i, '-', symbol(i), &
     &         j, '-', symbol(j), &
     &         eq_lin(k)*bohr2ang, &
     &         rij*bohr2ang, &
     &         dpot*har2kcal, &
     &         fc_lin(k)*har2kcal/bohr2ang/bohr2ang

         end do

      end do

      write( 6, '(a)' )

      return
      end






!***********************************************************************
      subroutine check_force_mm_angl
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use tinker_variables, only : &
     &   pi, x, y, z, fx, fy, fz, vir, pot, har2kcal, symbol, deg2rad, &
     &   fc_angl, eq_angl, i_angl, j_angl, k_angl, nbead, nangl

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, m, l

      real(8) :: xij, yij, zij, xkj, ykj, zkj, rij2, rkj2, rijk, &
     &           pijk, qijk, bijk, aijk, da, const, &
     &           fxi, fxj, fxk, fyi, fyj, fyk, fzi, fzj, fzk

      real(8) :: tiny_value = 1.d-4

      real(8) :: dpot

!-----------------------------------------------------------------------
!     /*   print out                                                  */
!-----------------------------------------------------------------------

      write( 6, '(a,17x,a,11x,a,4x,a,4x,a,4x,a)' ) &
     &   'Type', 'Atom Names', 'Ideal', 'Actual', 'Energy', 'Fconst'
      write( 6, '(a)' )

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( nangl .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do l = 1, nangl

            i = i_angl(l)
            j = j_angl(l)
            k = k_angl(l)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom ( xij, yij, zij )

            xkj = x(k,m) - x(j,m)
            ykj = y(k,m) - y(j,m)
            zkj = z(k,m) - z(j,m)

            call pbc_atom ( xkj, ykj, zkj )

            rij2 = xij*xij + yij*yij + zij*zij
            rkj2 = xkj*xkj + ykj*ykj + zkj*zkj

            rijk = sqrt( rij2*rkj2 )

            pijk = xij*xkj + yij*ykj + zij*zkj

            qijk  = pijk/rijk

            qijk = max( qijk, -1.d0 )
            qijk = min( qijk,  1.d0 )

            bijk = acos( qijk )

            aijk = bijk*(180.d0/pi)

            da   = aijk - eq_angl(l)

            pot(m) = pot(m) + 0.5d0*fc_angl(l)*da*da

            if ( abs(bijk)    .lt. tiny_value ) cycle
            if ( abs(bijk-pi) .lt. tiny_value ) cycle
            if ( abs(bijk+pi) .lt. tiny_value ) cycle

            const = fc_angl(l)*da /sin(bijk) /rijk *(180.d0/pi)

            fxi = const*( xkj - pijk/rij2*xij )
            fxk = const*( xij - pijk/rkj2*xkj )
            fxj = - fxi - fxk

            fyi = const*( ykj - pijk/rij2*yij )
            fyk = const*( yij - pijk/rkj2*ykj )
            fyj = - fyi - fyk

            fzi = const*( zkj - pijk/rij2*zij )
            fzk = const*( zij - pijk/rkj2*zkj )
            fzj = - fzi - fzk

            fx(i,m) = fx(i,m) + fxi
            fx(j,m) = fx(j,m) + fxj
            fx(k,m) = fx(k,m) + fxk

            fy(i,m) = fy(i,m) + fyi
            fy(j,m) = fy(j,m) + fyj
            fy(k,m) = fy(k,m) + fyk

            fz(i,m) = fz(i,m) + fzi
            fz(j,m) = fz(j,m) + fzj
            fz(k,m) = fz(k,m) + fzk

            vir(1,1) = vir(1,1) + fxi*xij + fxk*xkj
            vir(1,2) = vir(1,2) + fxi*yij + fxk*ykj
            vir(1,3) = vir(1,3) + fxi*zij + fxk*zkj
            vir(2,1) = vir(2,1) + fyi*xij + fyk*xkj
            vir(2,2) = vir(2,2) + fyi*yij + fyk*ykj
            vir(2,3) = vir(2,3) + fyi*zij + fyk*zkj
            vir(3,1) = vir(3,1) + fzi*xij + fzk*xkj
            vir(3,2) = vir(3,2) + fzi*yij + fzk*ykj
            vir(3,3) = vir(3,3) + fzi*zij + fzk*zkj

            dpot = 0.5d0*fc_angl(l)*da*da

            write( 6, '(a5,5x,3(i5,a1,a3),4f10.4)' ) &
     &         'Angle', &
     &         i, '-', symbol(i), &
     &         j, '-', symbol(j), &
     &         k, '-', symbol(k), &
     &         eq_angl(l), &
     &         aijk, &
     &         dpot*har2kcal, &
     &         fc_angl(l)*har2kcal/deg2rad/deg2rad

         end do

      end do

      write( 6, '(a)' )

      return
      end





!***********************************************************************
      subroutine check_force_mm_tors
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use tinker_variables, only : &
     &   x, y, z, fx, fy, fz, vir, pot, har2kcal, symbol, deg2rad, &
     &   v_dih, i_dih, j_dih, k_dih, l_dih, mu_dih, nu_dih, ntors_3, &
     &   nbead, ndih

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, n, mu, nu, ii

      real(8) :: xij, yij, zij, xkj, ykj, zkj, xlj, ylj, zlj, &
     &           xijk, yijk, zijk, xjkl, yjkl, zjkl, rijk2, rjkl2, v, &
     &           rijkl2, rijk2inv, rjkl2inv, rijkl2inv, cos_phi, phi, &
     &           factor, fxi, fyi, fzi, fxj, fyj, fzj, fxk, fyk, fzk, &
     &           fxl, fyl, fzl

      real(8) :: tiny_value = 1.d-4

      real(8), dimension(0:12):: sinfactor

      real(8) :: dpot

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( ndih .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   print out                                                  */
!-----------------------------------------------------------------------

      write( 6, '(a,21x,a,16x,a,4x,a,3x,a,1x,a,1x,a)' ) &
     &   'Type', 'Atom Names', 'Angle', 'Energy', 'Barrier', 'Mu', 'Nu'
      write( 6, '(a)' )

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do n = 1, ntors_3

            i = i_dih(n)
            j = j_dih(n)
            k = k_dih(n)
            l = l_dih(n)

!-----------------------------------------------------------------------

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom ( xij, yij, zij )

            xkj = x(k,m) - x(j,m)
            ykj = y(k,m) - y(j,m)
            zkj = z(k,m) - z(j,m)

            call pbc_atom ( xkj, ykj, zkj )

            xlj = x(l,m) - x(j,m)
            ylj = y(l,m) - y(j,m)
            zlj = z(l,m) - z(j,m)

            call pbc_atom ( xlj, ylj, zlj )

!-----------------------------------------------------------------------

            xijk = yij*zkj - zij*ykj
            yijk = zij*xkj - xij*zkj
            zijk = xij*ykj - yij*xkj

            xjkl = ylj*zkj - zlj*ykj
            yjkl = zlj*xkj - xlj*zkj
            zjkl = xlj*ykj - ylj*xkj

            rijk2  = xijk*xijk + yijk*yijk + zijk*zijk
            rjkl2  = xjkl*xjkl + yjkl*yjkl + zjkl*zjkl

            rijkl2 = sqrt(rijk2*rjkl2)

            if ( abs(rijk2)  .lt. tiny_value ) cycle
            if ( abs(rjkl2)  .lt. tiny_value ) cycle
            if ( abs(rijkl2) .lt. tiny_value ) cycle

            rijk2inv  = 1.d0 / rijk2
            rjkl2inv  = 1.d0 / rjkl2
            rijkl2inv = 1.d0 / rijkl2

!-----------------------------------------------------------------------
!           /*   cos_phi = cos( phi )                                 */
!-----------------------------------------------------------------------

            cos_phi = ( xijk*xjkl + yijk*yjkl + zijk*zjkl ) * rijkl2inv

            cos_phi = max( cos_phi, -1.d0 )
            cos_phi = min( cos_phi,  1.d0 )

            phi = acos(cos_phi)

!-----------------------------------------------------------------------
!           /*   mu     =  cos(delta)  =  +1 or -1                    */
!           /*   delta  =  0 or pi                                    */
!-----------------------------------------------------------------------

            mu = mu_dih(n)
            nu = nu_dih(n)
            v  =  v_dih(n)

!-----------------------------------------------------------------------
!           /*   pot = 0.5 * v * ( cos ( nu*phi - delta ) + 1 )       */
!-----------------------------------------------------------------------

            pot(m)  = pot(m) + 0.5d0 * v * ( 1.d0 + cos(nu*phi)*mu )

!-----------------------------------------------------------------------
!           /*    sinfactor(n) = sin(n*phi) / sin(phi)                */
!-----------------------------------------------------------------------

            if      ( nu .eq. 0 ) then
               sinfactor(0) = 0.d0
            else if ( nu .eq. 1 ) then
               sinfactor(1) = 1.d0
            else if( nu .eq. 2 ) then
               sinfactor(2) = 2.d0*cos_phi
            else if( nu .eq. 3 ) then
               sinfactor(3) = 4.d0*cos_phi*cos_phi - 1.d0
            else if( nu .eq. 4 ) then
               sinfactor(4) = 4.d0*cos_phi*(2.d0*cos_phi*cos_phi - 1.d0)
            else if( nu .eq. 6 ) then
               sinfactor(6) = 2.d0 * ( 4.d0*cos_phi*cos_phi - 1.d0 ) &
     &                   * cos_phi * ( 4.d0*cos_phi*cos_phi - 3.d0 )
            else
               sinfactor(1) = 1.d0
               sinfactor(2) = 2.d0*cos_phi
               do ii = 3, nu
                  sinfactor(ii) = sinfactor(ii-2) &
     &                         - sin((ii-2)*phi)*sin(phi) &
     &                         + cos((ii-2)*phi)*cos(phi) &
     &                         + cos((ii-1)*phi)
               end do
            end if

            factor = - sinfactor(nu) * 0.5d0 * v * mu * nu

            fxi = factor * ( + ( ykj*zjkl - zkj*yjkl ) * rijkl2inv &
     &                    - ( ykj*zijk - zkj*yijk ) * cos_phi*rijk2inv )
            fyi = factor * ( + ( zkj*xjkl - xkj*zjkl ) * rijkl2inv &
     &                    - ( zkj*xijk - xkj*zijk ) * cos_phi*rijk2inv )
            fzi = factor * ( + ( xkj*yjkl - ykj*xjkl ) * rijkl2inv &
     &                    - ( xkj*yijk - ykj*xijk ) * cos_phi*rijk2inv )

            fxl = factor * ( + ( ykj*zijk - zkj*yijk ) * rijkl2inv &
     &                    - ( ykj*zjkl - zkj*yjkl ) * cos_phi*rjkl2inv )
            fyl = factor * ( + ( zkj*xijk - xkj*zijk ) * rijkl2inv &
     &                    - ( zkj*xjkl - xkj*zjkl ) * cos_phi*rjkl2inv )
            fzl = factor * ( + ( xkj*yijk - ykj*xijk ) * rijkl2inv &
     &                    - ( xkj*yjkl - ykj*xjkl ) * cos_phi*rjkl2inv )

            fxk = factor * ( - ( yij*zjkl - zij*yjkl ) * rijkl2inv &
     &                    - ( ylj*zijk - zlj*yijk ) * rijkl2inv &
     &                    + ( yij*zijk - zij*yijk ) * cos_phi*rijk2inv &
     &                    + ( ylj*zjkl - zlj*yjkl ) * cos_phi*rjkl2inv )
            fyk = factor * ( - ( zij*xjkl - xij*zjkl ) * rijkl2inv &
     &                    - ( zlj*xijk - xlj*zijk ) * rijkl2inv &
     &                    + ( zij*xijk - xij*zijk ) * cos_phi*rijk2inv &
     &                    + ( zlj*xjkl - xlj*zjkl ) * cos_phi*rjkl2inv )
            fzk = factor * ( - ( xij*yjkl - yij*xjkl ) * rijkl2inv &
     &                    - ( xlj*yijk - ylj*xijk ) * rijkl2inv &
     &                    + ( xij*yijk - yij*xijk ) * cos_phi*rijk2inv &
     &                    + ( xlj*yjkl - ylj*xjkl ) * cos_phi*rjkl2inv )

            fxj = - ( fxi + fxk + fxl )
            fyj = - ( fyi + fyk + fyl )
            fzj = - ( fzi + fzk + fzl )

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) + fxj
            fy(j,m) = fy(j,m) + fyj
            fz(j,m) = fz(j,m) + fzj

            fx(k,m) = fx(k,m) + fxk
            fy(k,m) = fy(k,m) + fyk
            fz(k,m) = fz(k,m) + fzk

            fx(l,m) = fx(l,m) + fxl
            fy(l,m) = fy(l,m) + fyl
            fz(l,m) = fz(l,m) + fzl

            vir(1,1) = vir(1,1) + fxi*xij + fxk*xkj + fxl*xlj
            vir(1,2) = vir(1,2) + fxi*yij + fxk*ykj + fxl*ylj
            vir(1,3) = vir(1,3) + fxi*zij + fxk*zkj + fxl*zlj
            vir(2,1) = vir(2,1) + fyi*xij + fyk*xkj + fyl*xlj
            vir(2,2) = vir(2,2) + fyi*yij + fyk*ykj + fyl*ylj
            vir(2,3) = vir(2,3) + fyi*zij + fyk*zkj + fyl*zlj
            vir(3,1) = vir(3,1) + fzi*xij + fzk*xkj + fzl*xlj
            vir(3,2) = vir(3,2) + fzi*yij + fzk*ykj + fzl*ylj
            vir(3,3) = vir(3,3) + fzi*zij + fzk*zkj + fzl*zlj

!-----------------------------------------------------------------------
!           /*    print out                                           */
!-----------------------------------------------------------------------

            dpot = 0.5d0 * v * ( 1.d0 + cos(nu*phi)*mu )

            write( 6, '(a7,3x,4(i5,a1,a3),3f10.4,2i3)' ) &
     &         'Torsion', &
     &         i, '-', symbol(i), &
     &         j, '-', symbol(j), &
     &         k, '-', symbol(k), &
     &         l, '-', symbol(l), &
     &         phi/deg2rad, &
     &         dpot*har2kcal, &
     &         v*har2kcal, &
     &         mu, &
     &         nu

         end do

      end do

      write( 6, '(a)' )

      return
      end





!***********************************************************************
      subroutine check_force_mm_imptors
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use tinker_variables, only : &
     &   x, y, z, fx, fy, fz, vir, pot, har2kcal, symbol, deg2rad, &
     &   v_dih, i_dih, j_dih, k_dih, l_dih, mu_dih, nu_dih, ntors_3, &
     &   nbead, ndih

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, n, mu, nu, ii

      real(8) :: xij, yij, zij, xkj, ykj, zkj, xlj, ylj, zlj, &
     &           xijk, yijk, zijk, xjkl, yjkl, zjkl, rijk2, rjkl2, v, &
     &           rijkl2, rijk2inv, rjkl2inv, rijkl2inv, cos_phi, phi, &
     &           factor, fxi, fyi, fzi, fxj, fyj, fzj, fxk, fyk, fzk, &
     &           fxl, fyl, fzl

      real(8) :: tiny_value = 1.d-4

      real(8), dimension(0:12):: sinfactor

      real(8) :: dpot

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( ntors_3 .eq. ndih ) return

!-----------------------------------------------------------------------
!     /*   print out                                                  */
!-----------------------------------------------------------------------

      write( 6, '(a,21x,a,16x,a,4x,a,3x,a,1x,a,1x,a)' ) &
     &   'Type', 'Atom Names', 'Angle', 'Energy', 'Barrier', 'Mu', 'Nu'
      write( 6, '(a)' )

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do n = ntors_3+1, ndih

            i = i_dih(n)
            j = j_dih(n)
            k = k_dih(n)
            l = l_dih(n)

!-----------------------------------------------------------------------

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom ( xij, yij, zij )

            xkj = x(k,m) - x(j,m)
            ykj = y(k,m) - y(j,m)
            zkj = z(k,m) - z(j,m)

            call pbc_atom ( xkj, ykj, zkj )

            xlj = x(l,m) - x(j,m)
            ylj = y(l,m) - y(j,m)
            zlj = z(l,m) - z(j,m)

            call pbc_atom ( xlj, ylj, zlj )

!-----------------------------------------------------------------------

            xijk = yij*zkj - zij*ykj
            yijk = zij*xkj - xij*zkj
            zijk = xij*ykj - yij*xkj

            xjkl = ylj*zkj - zlj*ykj
            yjkl = zlj*xkj - xlj*zkj
            zjkl = xlj*ykj - ylj*xkj

            rijk2  = xijk*xijk + yijk*yijk + zijk*zijk
            rjkl2  = xjkl*xjkl + yjkl*yjkl + zjkl*zjkl

            rijkl2 = sqrt(rijk2*rjkl2)

            if ( abs(rijk2)  .lt. tiny_value ) cycle
            if ( abs(rjkl2)  .lt. tiny_value ) cycle
            if ( abs(rijkl2) .lt. tiny_value ) cycle

            rijk2inv  = 1.d0 / rijk2
            rjkl2inv  = 1.d0 / rjkl2
            rijkl2inv = 1.d0 / rijkl2

!-----------------------------------------------------------------------
!           /*   cos_phi = cos( phi )                                 */
!-----------------------------------------------------------------------

            cos_phi = ( xijk*xjkl + yijk*yjkl + zijk*zjkl ) * rijkl2inv

            cos_phi = max( cos_phi, -1.d0 )
            cos_phi = min( cos_phi,  1.d0 )

            phi = acos(cos_phi)

!-----------------------------------------------------------------------
!           /*   mu     =  cos(delta)  =  +1 or -1                    */
!           /*   delta  =  0 or pi                                    */
!-----------------------------------------------------------------------

            mu = mu_dih(n)
            nu = nu_dih(n)
            v  =  v_dih(n)

!-----------------------------------------------------------------------
!           /*   pot = 0.5 * v * ( cos ( nu*phi - delta ) + 1 )       */
!-----------------------------------------------------------------------

            pot(m)  = pot(m) + 0.5d0 * v * ( 1.d0 + cos(nu*phi)*mu )

!-----------------------------------------------------------------------
!           /*    sinfactor(n) = sin(n*phi) / sin(phi)                */
!-----------------------------------------------------------------------

            if      ( nu .eq. 0 ) then
               sinfactor(0) = 0.d0
            else if ( nu .eq. 1 ) then
               sinfactor(1) = 1.d0
            else if( nu .eq. 2 ) then
               sinfactor(2) = 2.d0*cos_phi
            else if( nu .eq. 3 ) then
               sinfactor(3) = 4.d0*cos_phi*cos_phi - 1.d0
            else if( nu .eq. 4 ) then
               sinfactor(4) = 4.d0*cos_phi*(2.d0*cos_phi*cos_phi - 1.d0)
            else if( nu .eq. 6 ) then
               sinfactor(6) = 2.d0 * ( 4.d0*cos_phi*cos_phi - 1.d0 ) &
     &                   * cos_phi * ( 4.d0*cos_phi*cos_phi - 3.d0 )
            else
               sinfactor(1) = 1.d0
               sinfactor(2) = 2.d0*cos_phi
               do ii = 3, nu
                  sinfactor(ii) = sinfactor(ii-2) &
     &                         - sin((ii-2)*phi)*sin(phi) &
     &                         + cos((ii-2)*phi)*cos(phi) &
     &                         + cos((ii-1)*phi)
               end do
            end if

            factor = - sinfactor(nu) * 0.5d0 * v * mu * nu

            fxi = factor * ( + ( ykj*zjkl - zkj*yjkl ) * rijkl2inv &
     &                    - ( ykj*zijk - zkj*yijk ) * cos_phi*rijk2inv )
            fyi = factor * ( + ( zkj*xjkl - xkj*zjkl ) * rijkl2inv &
     &                    - ( zkj*xijk - xkj*zijk ) * cos_phi*rijk2inv )
            fzi = factor * ( + ( xkj*yjkl - ykj*xjkl ) * rijkl2inv &
     &                    - ( xkj*yijk - ykj*xijk ) * cos_phi*rijk2inv )

            fxl = factor * ( + ( ykj*zijk - zkj*yijk ) * rijkl2inv &
     &                    - ( ykj*zjkl - zkj*yjkl ) * cos_phi*rjkl2inv )
            fyl = factor * ( + ( zkj*xijk - xkj*zijk ) * rijkl2inv &
     &                    - ( zkj*xjkl - xkj*zjkl ) * cos_phi*rjkl2inv )
            fzl = factor * ( + ( xkj*yijk - ykj*xijk ) * rijkl2inv &
     &                    - ( xkj*yjkl - ykj*xjkl ) * cos_phi*rjkl2inv )

            fxk = factor * ( - ( yij*zjkl - zij*yjkl ) * rijkl2inv &
     &                    - ( ylj*zijk - zlj*yijk ) * rijkl2inv &
     &                    + ( yij*zijk - zij*yijk ) * cos_phi*rijk2inv &
     &                    + ( ylj*zjkl - zlj*yjkl ) * cos_phi*rjkl2inv )
            fyk = factor * ( - ( zij*xjkl - xij*zjkl ) * rijkl2inv &
     &                    - ( zlj*xijk - xlj*zijk ) * rijkl2inv &
     &                    + ( zij*xijk - xij*zijk ) * cos_phi*rijk2inv &
     &                    + ( zlj*xjkl - xlj*zjkl ) * cos_phi*rjkl2inv )
            fzk = factor * ( - ( xij*yjkl - yij*xjkl ) * rijkl2inv &
     &                    - ( xlj*yijk - ylj*xijk ) * rijkl2inv &
     &                    + ( xij*yijk - yij*xijk ) * cos_phi*rijk2inv &
     &                    + ( xlj*yjkl - ylj*xjkl ) * cos_phi*rjkl2inv )

            fxj = - ( fxi + fxk + fxl )
            fyj = - ( fyi + fyk + fyl )
            fzj = - ( fzi + fzk + fzl )

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) + fxj
            fy(j,m) = fy(j,m) + fyj
            fz(j,m) = fz(j,m) + fzj

            fx(k,m) = fx(k,m) + fxk
            fy(k,m) = fy(k,m) + fyk
            fz(k,m) = fz(k,m) + fzk

            fx(l,m) = fx(l,m) + fxl
            fy(l,m) = fy(l,m) + fyl
            fz(l,m) = fz(l,m) + fzl

            vir(1,1) = vir(1,1) + fxi*xij + fxk*xkj + fxl*xlj
            vir(1,2) = vir(1,2) + fxi*yij + fxk*ykj + fxl*ylj
            vir(1,3) = vir(1,3) + fxi*zij + fxk*zkj + fxl*zlj
            vir(2,1) = vir(2,1) + fyi*xij + fyk*xkj + fyl*xlj
            vir(2,2) = vir(2,2) + fyi*yij + fyk*ykj + fyl*ylj
            vir(2,3) = vir(2,3) + fyi*zij + fyk*zkj + fyl*zlj
            vir(3,1) = vir(3,1) + fzi*xij + fzk*xkj + fzl*xlj
            vir(3,2) = vir(3,2) + fzi*yij + fzk*ykj + fzl*ylj
            vir(3,3) = vir(3,3) + fzi*zij + fzk*zkj + fzl*zlj

!-----------------------------------------------------------------------
!           /*    print out                                           */
!-----------------------------------------------------------------------

            dpot = 0.5d0 * v * ( 1.d0 + cos(nu*phi)*mu )

            write( 6, '(a7,3x,4(i5,a1,a3),3f10.4,2i3)' ) &
     &         'Imptors', &
     &         i, '-', symbol(i), &
     &         j, '-', symbol(j), &
     &         k, '-', symbol(k), &
     &         l, '-', symbol(l), &
     &         phi/deg2rad, &
     &         dpot*har2kcal, &
     &         v*har2kcal, &
     &         mu, &
     &         nu

         end do

      end do

      write( 6, * )

      return
      end






!***********************************************************************
      subroutine pbc_atom ( xi, yi, zi )
!***********************************************************************

      use tinker_variables, only: iboundary, box, boxinv

      implicit none

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      real(8) :: ai, bi, ci, xi, yi, zi

!-----------------------------------------------------------------------
!     /*   apply boundary condition                                   */
!-----------------------------------------------------------------------

      if      ( iboundary .eq. 1 ) then

         ai = boxinv(1,1)*xi + boxinv(1,2)*yi + boxinv(1,3)*zi
         bi = boxinv(2,1)*xi + boxinv(2,2)*yi + boxinv(2,3)*zi
         ci = boxinv(3,1)*xi + boxinv(3,2)*yi + boxinv(3,3)*zi

!        /*   shift to the range -box/2 < x < +box/2   */

         ai = ai - nint(ai)
         bi = bi - nint(bi)
         ci = ci - nint(ci)

         xi = box(1,1)*ai + box(1,2)*bi + box(1,3)*ci
         yi = box(2,1)*ai + box(2,2)*bi + box(2,3)*ci
         zi = box(3,1)*ai + box(3,2)*bi + box(3,3)*ci

      else if ( iboundary .eq. 2 ) then

         ai = boxinv(1,1)*xi + boxinv(1,2)*yi + boxinv(1,3)*zi
         bi = boxinv(2,1)*xi + boxinv(2,2)*yi + boxinv(2,3)*zi
         ci = boxinv(3,1)*xi + boxinv(3,2)*yi + boxinv(3,3)*zi

!        /*   shift to the range -box/2 < x < +box/2   */

         ai = ai - nint(ai)
         bi = bi - nint(bi)
         ci = ci - nint(ci)

         xi = box(1,1)*ai + box(1,2)*bi + box(1,3)*ci
         yi = box(2,1)*ai + box(2,2)*bi + box(2,3)*ci
         zi = box(3,1)*ai + box(3,2)*bi + box(3,3)*ci

      end if

      return
      end





!***********************************************************************
      subroutine check_force_mm_lj
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use tinker_variables, only : &
     &   x, y, z, fx, fy, fz, vir, pot, har2kcal, symbol, bohr2ang, &
     &   rin_lj, rout_lj, iboundary, eps_lj, sig_lj, i_lj, j_lj, &
     &   nbead, nlj

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, l, m

      real(8) :: rout_lj2, xij, yij, zij, rij2, rij, rinv, epsi, sigm, &
     &           sr, sr2, sr6, sr12, u6, u12, uij, duij, fxi, fyi, fzi, &
     &           swf, dswf, two16

      real(8) :: dpot

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( nlj .eq. 0 ) return

      rout_lj2 = rout_lj*rout_lj

!-----------------------------------------------------------------------
!     /*   print out                                                  */
!-----------------------------------------------------------------------

      two16 = 2.d0**(1.d0/6.d0)

      write( 6, '(a,12x,a,6x,a,5x,a,4x,a,3x,a)' ) &
     &   'Type', 'Atom Names', 'Ideal', 'Actual', 'Energy', 'Epsilon'
      write( 6, '(a)' )

!-----------------------------------------------------------------------
!     /*   main loop : free boundary                                  */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) then

         do m = 1, nbead

         do l = 1, nlj

            i = i_lj(l)
            j = j_lj(l)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            rij2 = xij*xij + yij*yij + zij*zij

            if ( rij2 .gt. rout_lj2 ) cycle

            rij     = sqrt(rij2)

            rinv    = 1.d0/rij

            epsi     = eps_lj(l)
            sigm     = sig_lj(l)

            sr      = sigm*rinv
            sr2     = sr*sr
            sr6     = sr2*sr2*sr2
            sr12    = sr6*sr6

            u6      = - 4.d0*epsi*sr6
            u12     = + 4.d0*epsi*sr12

!           /*   switching function   */
            call getswf( rij, rin_lj, rout_lj, swf, dswf )

!           /*   bare potential   */
            uij     = + u6 + u12

!           /*   bare potential gradient   */
            duij    = ( - 6.d0*u6*rinv - 12.d0*u12*rinv )*swf

!           /*   corrected potential   */
            pot(m)  = pot(m) + uij*swf

!           /*   corrected forces   */

            fxi = - uij*dswf*xij*rinv - duij*xij*rinv
            fyi = - uij*dswf*yij*rinv - duij*yij*rinv
            fzi = - uij*dswf*zij*rinv - duij*zij*rinv

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

!-----------------------------------------------------------------------
!     /*   print out                                                  */
!-----------------------------------------------------------------------

            dpot = uij*swf

            write( 6, '(a6,4x,2(i5,a1,a3),4f10.4)' ) &
     &         'VDW-LJ', &
     &         i, '-', symbol(i), &
     &         j, '-', symbol(j), &
     &         sigm*two16*bohr2ang, &
     &         rij*bohr2ang, &
     &         dpot*har2kcal, &
     &         epsi*har2kcal

         end do

         end do

      end if

      write( 6, '(a)' )

      return
      end





!***********************************************************************
      subroutine getswf ( r, rin, rout, swf, dswf )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8) :: c0, c1, c2, c3, dr, dr3, r, rin, rout, rout2, swf, dswf

!-----------------------------------------------------------------------
!     /*   switching function                                         */
!-----------------------------------------------------------------------

      if ( r .le. rin ) then

         swf  = 1.d0
         dswf = 0.d0

      else if ( r .ge. rout ) then

         swf  = 0.d0
         dswf = 0.d0

      else

         dr = rout - rin

         rout2 = rout*rout

         dr3   = dr*dr*dr

         c0 = + rout*rout2 - 3.d0*rin*rout2
         c1 = + 6.d0*rin*rout
         c2 = - 3.d0*rin - 3.d0*rout
         c3 = + 2.d0

         c0 = c0 /dr3
         c1 = c1 /dr3
         c2 = c2 /dr3
         c3 = c3 /dr3

!        /*   switching function   */
         swf  = c3*r*r*r + c2*r*r + c1*r + c0

!        /*   gradient of switching function   */
         dswf = 3.d0*c3*r*r + 2.d0*c2*r + c1

      end if

      return
      end





!***********************************************************************
      subroutine check_force_mm_coulomb
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use tinker_variables, only : &
     &   x, y, z, fx, fy, fz, pot, bohr2ang, har2kcal, q, symbol, &
     &   f_bcp, iboundary, nbcp, i_bcp, j_bcp, nbead, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, l, m

      real(8) :: xij, yij, zij, rij2, rij, rinv, &
     &           uij, duij, fxi, fyi, fzi

      integer :: ii, jj
      real(8) :: dpot, fbcp

!-----------------------------------------------------------------------
!     /*   print out                                                  */
!-----------------------------------------------------------------------

      write( 6, '(a,11x,a,5x,a,4x,a,9x,a,9x,a)' ) &
     &   'Type', 'Atom Names', 'Distance', 'Energy', 'Charges', 'Scale'
      write( 6, '(a)' )

!-----------------------------------------------------------------------
!     /*   main loop : free boundary                                  */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) then

         do m = 1, nbead

         do i = 1, natom-1
         do j = i+1, natom

            fbcp = 1.d0
            do l = 1, nbcp
               ii = i_bcp(l)
               jj = j_bcp(l)
               if ( ( ii .eq. i ) .and. ( jj .eq. j ) ) then
                  fbcp = f_bcp(l)
                  exit
               end if
            end do

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            rij2 = xij*xij + yij*yij + zij*zij

            rij  = sqrt(rij2)

            rinv = 1.d0 / rij

!           /*   bare potential   */
            uij  = + q(i)*q(j) / rij

!           /*   bare potential gradient   */
            duij = - 1.d0*uij*rinv

!           /*   potential   */
            pot(m)  = pot(m) + uij*fbcp

!           /*   forces   */

            fxi = - duij*xij*rinv
            fyi = - duij*yij*rinv
            fzi = - duij*zij*rinv

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

!-----------------------------------------------------------------------
!     /*   print out                                                  */
!-----------------------------------------------------------------------

            dpot = uij*fbcp

            write( 6, '(a,4x,2(i5,a1,a3),5f10.4)' ) &
     &         'Charge', &
     &         i, '-', symbol(i), &
     &         j, '-', symbol(j), &
     &         rij*bohr2ang, &
     &         dpot*har2kcal, &
     &         q(i), &
     &         q(j), &
     &         fbcp

         end do
         end do

         end do

         write( 6, '(a)' )

      end if

      return
      end





!***********************************************************************
      subroutine check_force_mm_improper
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use tinker_variables, only : &
     &   pi, x, y, z, fx, fy, fz, pot, vir, deg2rad, har2kcal, &
     &   eq_improper, fc_improper, symbol, i_improper, j_improper, &
     &   k_improper, l_improper, nimproper, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, n

      real(8) :: tiny_value = 1.d-4

      real(8) :: xij, yij, zij, xkj, ykj, zkj, xlj, ylj, zlj, &
     &           xijk, yijk, zijk, xjkl, yjkl, zjkl, rijk2, rjkl2, &
     &           rijkl2, rijk2inv, rjkl2inv, rijkl2inv, cos_phi, phi, &
     &           fxi, fyi, fzi, fxj, fyj, fzj, fxk, fyk, fzk, &
     &           fxl, fyl, fzl

      real(8) :: dphi, factor_1, factor_2, factor_3, factor_4, &
     &           px1, py1, pz1, px2, py2, pz2, px3, py3, pz3, &
     &           px4, py4, pz4, px5, py5, pz5, px6, py6, pz6

      real(8) :: ax, ay, az, a1, a2, xkl, ykl, zkl, xki, yki, zki

      real(8) :: daxdxi, daxdyi, daxdzi, daydxi, daydyi, daydzi, &
     &           dazdxi, dazdyi, dazdzi, dadxi, dadyi, dadzi, &
     &           daxdxj, daxdyj, daxdzj, daydxj, daydyj, daydzj, &
     &           dazdxj, dazdyj, dazdzj, dadxj, dadyj, dadzj, &
     &           daxdxl, daxdyl, daxdzl, daydxl, daydyl, daydzl, &
     &           dazdxl, dazdyl, dazdzl, dadxl, dadyl, dadzl

      real(8) :: f1, sin_phi, sign_phi

!      real(8)::  daxdxk, daxdyk, daxdzk, daydxk, daydyk, daydzk,
!     &           dazdxk, dazdyk, dazdzk, dadxk, dadyk, dadzk

      real(8) :: dpot

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( nimproper .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   print out                                                  */
!-----------------------------------------------------------------------

      write( 6, '(a,21x,a,16x,a,4x,a,4x,a,4x,a)' ) &
     &   'Type', 'Atom Names', 'Ideal', 'Actual', 'Energy', 'Fconst'
      write( 6, '(a)' )

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do n = 1, nimproper

            i = i_improper(n)
            j = j_improper(n)
            k = k_improper(n)
            l = l_improper(n)

!-----------------------------------------------------------------------

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom ( xij, yij, zij )

            xkj = x(k,m) - x(j,m)
            ykj = y(k,m) - y(j,m)
            zkj = z(k,m) - z(j,m)

            call pbc_atom ( xkj, ykj, zkj )

            xlj = x(l,m) - x(j,m)
            ylj = y(l,m) - y(j,m)
            zlj = z(l,m) - z(j,m)

            call pbc_atom ( xlj, ylj, zlj )

!-----------------------------------------------------------------------

            xijk = yij*zkj - zij*ykj
            yijk = zij*xkj - xij*zkj
            zijk = xij*ykj - yij*xkj

            xjkl = ylj*zkj - zlj*ykj
            yjkl = zlj*xkj - xlj*zkj
            zjkl = xlj*ykj - ylj*xkj

            rijk2  = xijk*xijk + yijk*yijk + zijk*zijk
            rjkl2  = xjkl*xjkl + yjkl*yjkl + zjkl*zjkl

            rijkl2 = sqrt(rijk2*rjkl2)

            if ( abs(rijk2)  .lt. tiny_value ) cycle
            if ( abs(rjkl2)  .lt. tiny_value ) cycle
            if ( abs(rijkl2) .lt. tiny_value ) cycle

            rijk2inv  = 1.d0 / rijk2
            rjkl2inv  = 1.d0 / rjkl2
            rijkl2inv = 1.d0 / rijkl2

!-----------------------------------------------------------------------

            factor_1 = fc_improper(n) * (180.d0/pi) * (180.d0/pi)

            cos_phi = ( xijk*xjkl + yijk*yjkl + zijk*zjkl ) * rijkl2inv

            cos_phi = max( cos_phi, -1.d0 )
            cos_phi = min( cos_phi,  1.d0 )

            phi = acos(cos_phi)

            sign_phi = ( yijk*zjkl - zijk*yjkl ) * xkj &
     &               + ( zijk*xjkl - xijk*zjkl ) * ykj &
     &               + ( xijk*yjkl - yijk*xjkl ) * zkj

            sign_phi = sign( 1.d0, sign_phi )

            phi = phi * sign_phi

            factor_2  = eq_improper(n) * (pi/180.d0)

            dphi = phi - factor_2

!-----------------------------------------------------------------------

            pot(m)  = pot(m) + 0.5d0 * factor_1 * dphi * dphi

!-----------------------------------------------------------------------

            if ( ( abs(phi)    .gt. tiny_value ) .and. &
     &           ( abs(phi-pi) .gt. tiny_value ) .and. &
     &           ( abs(phi+pi) .gt. tiny_value ) ) then

!-----------------------------------------------------------------------

               factor_3 = sin(phi)

               factor_4 = factor_1 * dphi / factor_3

               px1 = yijk*zij - zijk*yij
               py1 = zijk*xij - xijk*zij
               pz1 = xijk*yij - yijk*xij

               px2 = yjkl*zij - zjkl*yij
               py2 = zjkl*xij - xjkl*zij
               pz2 = xjkl*yij - yjkl*xij

               px3 = yijk*zkj - zijk*ykj
               py3 = zijk*xkj - xijk*zkj
               pz3 = xijk*ykj - yijk*xkj

               px4 = yjkl*zkj - zjkl*ykj
               py4 = zjkl*xkj - xjkl*zkj
               pz4 = xjkl*ykj - yjkl*xkj

               px5 = yijk*zlj - zijk*ylj
               py5 = zijk*xlj - xijk*zlj
               pz5 = xijk*ylj - yijk*xlj

               px6 = yjkl*zlj - zjkl*ylj
               py6 = zjkl*xlj - xjkl*zlj
               pz6 = xjkl*ylj - yjkl*xlj

               fxi = factor_4 * ( - px4*rijkl2inv &
     &                            + px3*rijk2inv*cos_phi )
               fyi = factor_4 * ( - py4*rijkl2inv &
     &                            + py3*rijk2inv*cos_phi )
               fzi = factor_4 * ( - pz4*rijkl2inv &
     &                            + pz3*rijk2inv*cos_phi )

               fxk = factor_4 * ( + px2*rijkl2inv &
     &                            + px5*rijkl2inv &
     &                            - px1*rijk2inv*cos_phi &
     &                            - px6*rjkl2inv*cos_phi )
               fyk = factor_4 * ( + py2*rijkl2inv &
     &                            + py5*rijkl2inv &
     &                            - py1*rijk2inv*cos_phi &
     &                            - py6*rjkl2inv*cos_phi )
               fzk = factor_4 * ( + pz2*rijkl2inv &
     &                            + pz5*rijkl2inv &
     &                            - pz1*rijk2inv*cos_phi &
     &                            - pz6*rjkl2inv*cos_phi )

               fxl = factor_4 * ( - px3*rijkl2inv &
     &                            + px4*rjkl2inv*cos_phi )
               fyl = factor_4 * ( - py3*rijkl2inv &
     &                            + py4*rjkl2inv*cos_phi )
               fzl = factor_4 * ( - pz3*rijkl2inv &
     &                            + pz4*rjkl2inv*cos_phi )

               fxj = - ( fxi + fxk + fxl )
               fyj = - ( fyi + fyk + fyl )
               fzj = - ( fzi + fzk + fzl )

!-----------------------------------------------------------------------

            else

!-----------------------------------------------------------------------

               xki = - xij + xkj
               yki = - yij + ykj
               zki = - zij + zkj

               xkl = - xlj + xkj
               ykl = - ylj + ykj
               zkl = - zlj + zkj

               ax = yijk*zjkl - zijk*yjkl
               ay = zijk*xjkl - xijk*zjkl
               az = xijk*yjkl - yijk*xjkl

               a2 = ax*ax + ay*ay + az*az

               a1 = sqrt( a2 )

               sin_phi = a1 / rijkl2

               sin_phi = max( sin_phi, -1.d0 )
               sin_phi = min( sin_phi,  1.d0 )

               phi = sign_phi * asin( sin_phi )

               if ( cos_phi .lt. 0.d0 ) phi = pi - phi

               daxdxi = - zjkl * zkj - ykj * yjkl
               daxdyi = + yjkl * xkj
               daxdzi = + zjkl * xkj

               daydxi = + xjkl * ykj
               daydyi = - xjkl * xkj - zkj * zjkl
               daydzi = + zjkl * ykj

               dazdxi = + xjkl * zkj
               dazdyi = + yjkl * zkj
               dazdzi = - yjkl * ykj - xkj * xjkl

               daxdxj = - yijk * ykl + zjkl * zki &
     &                  + yjkl * yki - zijk * zkl
               daxdyj = + yijk * xkl - yjkl * xki
               daxdzj = + zijk * xkl - zjkl * xki

               daydxj = + xijk * ykl - xjkl * yki
               daydyj = - zijk * zkl + xjkl * xki &
     &                  + zjkl * zki - xijk * xkl
               daydzj = + zijk * ykl - zjkl * yki

               dazdxj = + xijk * zkl - xjkl * zki
               dazdyj = + yijk * zkl - yjkl * zki
               dazdzj = - xijk * xkl + yjkl * yki &
     &                  + xjkl * xki - yijk * ykl

!               daxdxk = - yjkl * yij + zijk * zlj
!                        + yijk * ylj - zjkl * zij
!               daxdyk = + yjkl * xij - yijk * xlj
!               daxdzk = + zjkl * xij - zijk * xlj
!
!               daydxk = + xjkl * yij - xijk * ylj
!               daydyk = - zjkl * zij + xijk * xlj
!                        + zijk * zlj - xjkl * xij
!               daydzk = + zjkl * yij - zijk * ylj
!
!               dazdxk = + xjkl * zij - xijk * zlj
!               dazdyk = + yjkl * zij - yijk * zlj
!               dazdzk = - xjkl * xij + yijk * ylj
!                        + xijk * xlj - yjkl * yij

               daxdxl = + zijk * zkj + ykj * yijk
               daxdyl = - yijk * xkj
               daxdzl = - zijk * xkj

               daydxl = - xijk * ykj
               daydyl = + xijk * xkj + zkj * zijk
               daydzl = - zijk * ykj

               dazdxl = - xijk * zkj
               dazdyl = - yijk * zkj
               dazdzl = + yijk * ykj + xkj * xijk

               dadxi = ax/a1*daxdxi + ay/a1*daydxi + az/a1*dazdxi
               dadyi = ax/a1*daxdyi + ay/a1*daydyi + az/a1*dazdyi
               dadzi = ax/a1*daxdzi + ay/a1*daydzi + az/a1*dazdzi

               dadxj = ax/a1*daxdxj + ay/a1*daydxj + az/a1*dazdxj
               dadyj = ax/a1*daxdyj + ay/a1*daydyj + az/a1*dazdyj
               dadzj = ax/a1*daxdzj + ay/a1*daydzj + az/a1*dazdzj

!               dadxk = ax/a1*daxdxk + ay/a1*daydxk + az/a1*dazdxk
!               dadyk = ax/a1*daxdyk + ay/a1*daydyk + az/a1*dazdyk
!               dadzk = ax/a1*daxdzk + ay/a1*daydzk + az/a1*dazdzk

               dadxl = ax/a1*daxdxl + ay/a1*daydxl + az/a1*dazdxl
               dadyl = ax/a1*daxdyl + ay/a1*daydyl + az/a1*dazdyl
               dadzl = ax/a1*daxdzl + ay/a1*daydzl + az/a1*dazdzl

               f1 = - sign_phi/cos_phi * (180.d0/pi) * (180.d0/pi) &
     &              * fc_improper(n) * dphi

               fxi = + f1 * ( dadxi / rijkl2 &
     &                + sin_phi * ( + yijk*zkj - zijk*ykj ) * rijk2inv )

               fyi = + f1 * ( dadyi / rijkl2 &
     &                + sin_phi * ( + zijk*xkj - xijk*zkj ) * rijk2inv )

               fzi = + f1 * ( dadzi / rijkl2 &
     &                + sin_phi * ( + xijk*ykj - yijk*xkj ) * rijk2inv )

               fxj = + f1 * ( dadxj / rijkl2 &
     &                + sin_phi * ( - yijk*zki + zijk*yki ) * rijk2inv &
     &                - sin_phi * ( + yjkl*zkl - zjkl*ykl ) * rjkl2inv )

               fyj = + f1 * ( dadyj / rijkl2 &
     &                + sin_phi * ( - zijk*xki + xijk*zki ) * rijk2inv &
     &                - sin_phi * ( + zjkl*xkl - xjkl*zkl ) * rjkl2inv )

               fzj = + f1 * ( dadzj / rijkl2 &
     &                + sin_phi * ( - xijk*yki + yijk*xki ) * rijk2inv &
     &                - sin_phi * ( + xjkl*ykl - yjkl*xkl ) * rjkl2inv )

!              fxk = - f1 * ( dadxk / rijkl2
!     &                - sin_phi * ( - yjkl*zlj + zjkl*ylj ) * rjkl2inv
!     &                + sin_phi * ( + yijk*zij - zijk*yij ) * rijk2inv )
!
!              fyk = - f1 * ( dadyk / rijkl2
!     &                - sin_phi * ( - zjkl*xlj + xjkl*zlj ) * rjkl2inv
!     &                + sin_phi * ( + zijk*xij - xijk*zij ) * rijk2inv )
!
!              fzk = - f1 * ( dadzk / rijkl2
!     &                - sin_phi * ( - xjkl*ylj + yjkl*xlj ) * rjkl2inv
!     &                + sin_phi * ( + xijk*yij - yijk*xij ) * rijk2inv )

               fxl = + f1 * ( dadxl / rijkl2 &
     &                + sin_phi * ( + yjkl*zkj - zjkl*ykj ) * rjkl2inv )

               fyl = + f1 * ( dadyl / rijkl2 &
     &                + sin_phi * ( + zjkl*xkj - xjkl*zkj ) * rjkl2inv )

               fzl = + f1 * ( dadzl / rijkl2 &
     &                + sin_phi * ( + xjkl*ykj - yjkl*xkj ) * rjkl2inv )

               fxk = - fxi - fxj - fxl
               fyk = - fyi - fyj - fyl
               fzk = - fzi - fzj - fzl

!-----------------------------------------------------------------------

            end if

!-----------------------------------------------------------------------

            fx(i,m) = fx(i,m) + fxi
            fx(j,m) = fx(j,m) + fxj
            fx(k,m) = fx(k,m) + fxk
            fx(l,m) = fx(l,m) + fxl

            fy(i,m) = fy(i,m) + fyi
            fy(j,m) = fy(j,m) + fyj
            fy(k,m) = fy(k,m) + fyk
            fy(l,m) = fy(l,m) + fyl

            fz(i,m) = fz(i,m) + fzi
            fz(j,m) = fz(j,m) + fzj
            fz(k,m) = fz(k,m) + fzk
            fz(l,m) = fz(l,m) + fzl

!-----------------------------------------------------------------------

            vir(1,1) = vir(1,1) + fxi*xij + fxk*xkj + fxl*xlj
            vir(1,2) = vir(1,2) + fxi*yij + fxk*ykj + fxl*ylj
            vir(1,3) = vir(1,3) + fxi*zij + fxk*zkj + fxl*zlj
            vir(2,1) = vir(2,1) + fyi*xij + fyk*xkj + fyl*xlj
            vir(2,2) = vir(2,2) + fyi*yij + fyk*ykj + fyl*ylj
            vir(2,3) = vir(2,3) + fyi*zij + fyk*zkj + fyl*zlj
            vir(3,1) = vir(3,1) + fzi*xij + fzk*xkj + fzl*xlj
            vir(3,2) = vir(3,2) + fzi*yij + fzk*ykj + fzl*ylj
            vir(3,3) = vir(3,3) + fzi*zij + fzk*zkj + fzl*zlj

!-----------------------------------------------------------------------
!     /*   print out                                                  */
!-----------------------------------------------------------------------

            dpot = 0.5d0 * factor_1 * dphi * dphi

            write( 6, '(a7,3x,4(i5,a1,a3),4f10.4)' ) &
     &         'Imprdih', &
     &         i, '-', symbol(i), &
     &         j, '-', symbol(j), &
     &         k, '-', symbol(k), &
     &         l, '-', symbol(l), &
     &         eq_improper(n), &
     &         phi/deg2rad, &
     &         dpot*har2kcal, &
     &         fc_improper(n)*har2kcal/deg2rad/deg2rad

         end do

      end do

      write( 6, '(a)' )

      return
      end





!***********************************************************************
      subroutine numbre_to_symbol( i, char, k )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k
      character(len=2) :: char

      integer, dimension(110), save :: iflag
      integer, save :: iset = 1
      integer, save :: icount = 0

      character(len=2), dimension(1:110) :: &
     &     symbol = (/       'H ', 'He', 'Li', 'Be', &
     &                 'B ', 'C ', 'N ', 'O ', 'F ', &
     &                 'Ne', 'Na', 'Mg', 'Al', 'Si', &
     &                 'P ', 'S ', 'Cl', 'Ar', 'K ', &
     &                 'Ca', 'Sc', 'Ti', 'V ', 'Cr', &
     &                 'Mn', 'Fe', 'Co', 'Ni', 'Cu', &
     &                 'Zn', 'Ga', 'Ge', 'As', 'Se', &
     &                 'Br', 'Kr', 'Rb', 'Sr', 'Y ', &
     &                 'Zr', 'Nb', 'Mo', 'Tc', 'Ru', &
     &                 'Rh', 'Pd', 'Ag', 'Cd', 'In', &
     &                 'Sn', 'Sb', 'Te', 'I ', 'Xe', &
     &                 'Cs', 'Ba', 'La', 'Ce', 'Pr', &
     &                 'Nd', 'Pm', 'Sm', 'Eu', 'Gd', &
     &                 'Tb', 'Dy', 'Ho', 'Er', 'Tm', &
     &                 'Yb', 'Lu', 'Hf', 'Ta', 'W ', &
     &                 'Re', 'Os', 'Ir', 'Pt', 'Au', &
     &                 'Hg', 'Tl', 'Pb', 'Bi', 'Po', &
     &                 'At', 'Rn', 'Fr', 'Ra', 'Ac', &
     &                 'Th', 'Pa', 'U ', 'Np', 'Pu', &
     &                 'Am', 'Cm', 'Bk', 'Cf', 'Es', &
     &                 'Fm', 'Md', 'No', 'Lr', 'Rf', &
     &                 'Db', 'Sg', 'Bh', 'Hs', 'Mt', &
     &                 'Ds' /)

!-----------------------------------------------------------------------
!     /*   initial setting                                            */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         iflag(:) = 0

         iset = 1

         icount = 0

      end if

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      k = 0

      do j = 1, 110

         if ( j .eq. i ) then

            char = symbol(j)

            if ( iflag(j) .eq. 0 ) then

               icount = icount + 1

               iflag(j) = icount

            end if

            k = iflag(j)

            exit

         end if

      end do

      return
      end





!***********************************************************************
      subroutine ljpair_tinker
!***********************************************************************

!     //   variables
      use tinker_variables, only : &
     &   eps_ljbond, sig_ljbond, eps_ljpair, sig_ljpair, &
     &   eps14_param, sig14_param, vdwindex, epsilonrule, scale_sigma, &
     &   eps_param, sig_param, radiusrule, eps_ljnonb, sig_ljnonb, &
     &   bond_book, i_ljbond, j_ljbond, nljbond, nbond, nangl, ntors, &
     &   natom, i_lin, j_lin, i_angl, k_angl, ikind, jkind, i_14, j_14, &
     &   i_ljpair, nljpair, nljbond_book, nljbond_correct

!     //   local variables
      implicit none

!     //   integers
      integer :: i, j, m, n

!     //   real numbers
      real(8) :: sig_ii, sig_jj, eps_ii, eps_jj, eps_ij

!     //   real numbers
      real(8) :: tiny = 1.d-8

!----------------------------------------------------------------------
!     //   memory allocations
!----------------------------------------------------------------------

      nljbond = nbond + nangl + ntors

      allocate( i_ljpair(natom) )

      allocate( eps_ljpair(natom) )
      allocate( sig_ljpair(natom) )

      allocate( i_ljbond(nljbond) )
      allocate( j_ljbond(nljbond) )

      allocate( eps_ljbond(nljbond) )
      allocate( sig_ljbond(nljbond) )

      allocate( eps_ljnonb(nljbond) )
      allocate( sig_ljnonb(nljbond) )

!     //   double countings
      nljbond_book = 0

      allocate( bond_book(natom,natom) )

      bond_book(:,:) = 0

!----------------------------------------------------------------------
!     //   atoms
!----------------------------------------------------------------------

      m = 0

      do i = 1, natom

!        //   epsilon non-bonded

         if ( vdwindex(1:6) .eq. 'TYPE  ' ) then
            eps_ii = abs(eps_param(jkind(i)))
         else if( vdwindex(1:6) .eq. 'CLASS ' ) then
            eps_ii = abs(eps_param(ikind(jkind(i))))
         else
            eps_ii = abs(eps_param(jkind(i)))
         end if

         if ( eps_ii .eq. 0.d0 ) cycle

         m = m + 1

         i_ljpair(m) = i

         eps_ljpair(m) = eps_ii

!        //   sigma non-bonded

         if ( vdwindex(1:6) .eq. 'TYPE  ' ) then
            sig_ii = sig_param(jkind(i))
         else if( vdwindex(1:6) .eq. 'CLASS ' ) then
            sig_ii = sig_param(ikind(jkind(i)))
         else
            sig_ii = sig_param(jkind(i))
         end if

         sig_ljpair(m) = sig_ii * scale_sigma

      end do

      nljpair = m

!----------------------------------------------------------------------
!     //   pairs in linear bonds
!----------------------------------------------------------------------

!     //   numbering
      n = 0

      do m = 1, nbond

!        //   atom i
         i = i_lin(m)

!        //   atom j
         j = j_lin(m)

!        //   epsilon non-bonded

         if ( vdwindex(1:6) .eq. 'TYPE  ' ) then
            eps_ii = abs(eps_param(jkind(i)))
            eps_jj = abs(eps_param(jkind(j)))
         else if( vdwindex(1:6) .eq. 'CLASS ' ) then
            eps_ii = abs(eps_param(ikind(jkind(i))))
            eps_jj = abs(eps_param(ikind(jkind(j))))
         else
            eps_ii = abs(eps_param(jkind(i)))
            eps_jj = abs(eps_param(jkind(j)))
         end if

         if      ( epsilonrule(1:11) .eq. 'GEOMETRIC  ' ) then
            eps_ij = sqrt( eps_ii * eps_jj )
         else if ( epsilonrule(1:11) .eq. 'ARITHMETIC ' ) then
            eps_ij = 0.5d0 * ( eps_ii + eps_jj )
         else
            eps_ij = sqrt( eps_ii * eps_jj )
         end if

!        //   avoid double booking
         if ( ( abs(eps_ij) .lt. tiny ) .or. &
     &        ( bond_book(i,j) .ne. 0 ) .or. &
     &        ( bond_book(j,i) .ne. 0 ) ) then

!           //   double countings
            nljbond_book = nljbond_book + 1

!           //   skip
            cycle

!        //   avoid double booking
         end if

!        //   book
         bond_book(i,j) = 1
         bond_book(j,i) = 1

!        //   numbering
         n = n + 1

!        //   atom i
         i_ljbond(n) = i

!        //   atom j
         j_ljbond(n) = j

!        //   epsilon bonded
         eps_ljbond(n) = 0.d0

!        //   epsilon non-bonded
         eps_ljnonb(n) = eps_ij

!        //   sigma bonded

         if ( vdwindex(1:6) .eq. 'TYPE  ' ) then
            sig_ii = sig_param(jkind(i))
            sig_jj = sig_param(jkind(j))
         else if( vdwindex(1:6) .eq. 'CLASS ' ) then
            sig_ii = sig_param(ikind(jkind(i)))
            sig_jj = sig_param(ikind(jkind(j)))
         else
            sig_ii = sig_param(jkind(i))
            sig_jj = sig_param(jkind(j))
         end if

         if       ( radiusrule(1:11) .eq. 'GEOMETRIC  ' ) then
            sig_ljbond(n) = sqrt( sig_ii * sig_jj )
         else if ( radiusrule(1:11) .eq. 'ARITHMETIC ' ) then
            sig_ljbond(n) = 0.5d0 * ( sig_ii + sig_jj )
         else
            sig_ljbond(n) = 0.5d0 * ( sig_ii + sig_jj )
         end if

         sig_ljbond(n) = sig_ljbond(n) * scale_sigma

!        //   sigma non-bonded

         sig_ljnonb(n) = sig_ljbond(n)

      end do

!----------------------------------------------------------------------
!     //   pairs in angular bonds
!----------------------------------------------------------------------

      do m = 1, nangl

!        //   atom i
         i = i_angl(m)

!        //   atom j
         j = k_angl(m)

!        //   epsilon non-bonded

         if ( vdwindex(1:6) .eq. 'TYPE  ' ) then
            eps_ii = abs(eps_param(jkind(i)))
            eps_jj = abs(eps_param(jkind(j)))
         else if( vdwindex(1:6) .eq. 'CLASS ' ) then
            eps_ii = abs(eps_param(ikind(jkind(i))))
            eps_jj = abs(eps_param(ikind(jkind(j))))
         else
            eps_ii = abs(eps_param(jkind(i)))
            eps_jj = abs(eps_param(jkind(j)))
         end if

         if      ( epsilonrule(1:11) .eq. 'GEOMETRIC  ' ) then
            eps_ij = sqrt( eps_ii * eps_jj )
         else if ( epsilonrule(1:11) .eq. 'ARITHMETIC ' ) then
            eps_ij = 0.5d0 * ( eps_ii + eps_jj )
         else
            eps_ij = sqrt( eps_ii * eps_jj )
         end if

!        //   avoid double booking
         if ( ( abs(eps_ij) .lt. tiny ) .or. &
     &        ( bond_book(i,j) .ne. 0 ) .or. &
     &        ( bond_book(j,i) .ne. 0 ) ) then

!           //   double countings
            nljbond_book = nljbond_book + 1

!           //   skip
            cycle

!        //   avoid double booking
         end if

!        //   book
         bond_book(i,j) = 1
         bond_book(j,i) = 1

!        //   numbering
         n = n + 1

!        //   atom i
         i_ljbond(n) = i

!        //   atom j
         j_ljbond(n) = j

!        //   epsilon bonded
         eps_ljbond(n) = 0.d0

!        //   epsilon non-bonded
         eps_ljnonb(n) = eps_ij

!        //   sigma bonded

         if ( vdwindex(1:6) .eq. 'TYPE  ' ) then
            sig_ii = sig_param(jkind(i))
            sig_jj = sig_param(jkind(j))
         else if( vdwindex(1:6) .eq. 'CLASS ' ) then
            sig_ii = sig_param(ikind(jkind(i)))
            sig_jj = sig_param(ikind(jkind(j)))
         else
            sig_ii = sig_param(jkind(i))
            sig_jj = sig_param(jkind(j))
         end if

         if       ( radiusrule(1:11) .eq. 'GEOMETRIC  ' ) then
            sig_ljbond(n) = sqrt( sig_ii * sig_jj )
         else if ( radiusrule(1:11) .eq. 'ARITHMETIC ' ) then
            sig_ljbond(n) = 0.5d0 * ( sig_ii + sig_jj )
         else
            sig_ljbond(n) = 0.5d0 * ( sig_ii + sig_jj )
         end if

         sig_ljbond(n) = sig_ljbond(n) * scale_sigma

!        //   sigma non-bonded

         sig_ljnonb(n) = sig_ljbond(n)

      end do

!----------------------------------------------------------------------
!     //   pairs in torsional bonds
!----------------------------------------------------------------------

      do m = 1, ntors

!        //   atom i
         i = i_14(m)

!        //   atom j
         j = j_14(m)

!        //   epsilon bonded

         if ( vdwindex(1:6) .eq. 'TYPE  ' ) then
            eps_ii = abs(eps14_param(jkind(i)))
            eps_jj = abs(eps14_param(jkind(j)))
         else if( vdwindex(1:6) .eq. 'CLASS ' ) then
            eps_ii = abs(eps14_param(ikind(jkind(i))))
            eps_jj = abs(eps14_param(ikind(jkind(j))))
         else
            eps_ii = abs(eps14_param(jkind(i)))
            eps_jj = abs(eps14_param(jkind(j)))
         end if

         if      ( epsilonrule(1:11) .eq. 'GEOMETRIC  ' ) then
            eps_ij = sqrt( eps_ii * eps_jj )
         else if ( epsilonrule(1:11) .eq. 'ARITHMETIC ' ) then
            eps_ij = 0.5d0 * ( eps_ii + eps_jj )
         else
            eps_ij = sqrt( eps_ii * eps_jj )
         end if

!        //   avoid double booking
         if ( ( abs(eps_ij) .lt. tiny ) .or. &
     &        ( bond_book(i,j) .ne. 0 ) .or. &
     &        ( bond_book(j,i) .ne. 0 ) ) then

!           //   double countings
            nljbond_book = nljbond_book + 1

!           //   skip
            cycle

!        //   avoid double booking
         end if

!        //   book
         bond_book(i,j) = 1
         bond_book(j,i) = 1

!        //   numbering
         n = n + 1

!        //   atom i
         i_ljbond(n) = i

!        //   atom j
         j_ljbond(n) = j

!        //   epsilon bonded
         eps_ljbond(n) = eps_ij

!        //   epsilon non-bonded

         if ( vdwindex(1:6) .eq. 'TYPE  ' ) then
            eps_ii = abs(eps_param(jkind(i)))
            eps_jj = abs(eps_param(jkind(j)))
         else if( vdwindex(1:6) .eq. 'CLASS ' ) then
            eps_ii = abs(eps_param(ikind(jkind(i))))
            eps_jj = abs(eps_param(ikind(jkind(j))))
         else
            eps_ii = abs(eps_param(jkind(i)))
            eps_jj = abs(eps_param(jkind(j)))
         end if

         if      ( epsilonrule(1:11) .eq. 'GEOMETRIC  ' ) then
            eps_ij = sqrt( eps_ii * eps_jj )
         else if ( epsilonrule(1:11) .eq. 'ARITHMETIC ' ) then
            eps_ij = 0.5d0 * ( eps_ii + eps_jj )
         else
            eps_ij = sqrt( eps_ii * eps_jj )
         end if

!        //   epsilon non-bonded
         eps_ljnonb(n) = eps_ij

!        //   sigma bonded

         if ( vdwindex(1:6) .eq. 'TYPE  ' ) then
            sig_ii = sig14_param(jkind(i))
            sig_jj = sig14_param(jkind(j))
         else if( vdwindex(1:6) .eq. 'CLASS ' ) then
            sig_ii = sig14_param(ikind(jkind(i)))
            sig_jj = sig14_param(ikind(jkind(j)))
         else
            sig_ii = sig14_param(jkind(i))
            sig_jj = sig14_param(jkind(j))
         end if

         if       ( radiusrule(1:11) .eq. 'GEOMETRIC  ' ) then
            sig_ljbond(n) = sqrt( sig_ii * sig_jj )
         else if ( radiusrule(1:11) .eq. 'ARITHMETIC ' ) then
            sig_ljbond(n) = 0.5d0 * ( sig_ii + sig_jj )
         else
            sig_ljbond(n) = 0.5d0 * ( sig_ii + sig_jj )
         end if

         sig_ljbond(n) = sig_ljbond(n) * scale_sigma

!        //   sigma non-bonded

         if ( vdwindex(1:6) .eq. 'TYPE  ' ) then
            sig_ii = sig_param(jkind(i))
            sig_jj = sig_param(jkind(j))
         else if( vdwindex(1:6) .eq. 'CLASS ' ) then
            sig_ii = sig_param(ikind(jkind(i)))
            sig_jj = sig_param(ikind(jkind(j)))
         else
            sig_ii = sig_param(jkind(i))
            sig_jj = sig_param(jkind(j))
         end if

         if       ( radiusrule(1:11) .eq. 'GEOMETRIC  ' ) then
            sig_ljnonb(n) = sqrt( sig_ii * sig_jj )
         else if ( radiusrule(1:11) .eq. 'ARITHMETIC ' ) then
            sig_ljnonb(n) = 0.5d0 * ( sig_ii + sig_jj )
         else
            sig_ljnonb(n) = 0.5d0 * ( sig_ii + sig_jj )
         end if

         sig_ljnonb(n) = sig_ljnonb(n) * scale_sigma

      end do

!     //   corrected number of lj bonds
      nljbond_correct = nljbond - nljbond_book

      return
      end





!***********************************************************************
      subroutine cmap_tinker
!***********************************************************************

!     //   variables
      use tinker_variables, only : &
     &   xgrid_cmap, ygrid_cmap, vgrid_cmap, v2grid_cmap, vref_cmap, &
     &   har2kcal, nkind_cmap, nkind_cmap, nbuff_cmap, nref_cmap, &
     &   ikind_cmap, ngrid_cmap, mgrid_cmap, i_cmap, j_cmap, k_cmap, &
     &   exist_cmap, l_cmap, m_cmap, ncmap, ikind, jkind, iounit_tinker, &
     &   i_tors, j_tors, k_tors, l_tors, bond, natom, ntors

!     //   local variables
      implicit none

!     //   integers
      integer :: iref_cmap, list_cmap, icmap

!     //   integers
      integer :: i, j, k, l, m, n, ierr, itmp

!     //   real numbers
      real(8) :: tmp

!     //   keywords
      character(len=15) :: keyword

!-----------------------------------------------------------------------
!     //   read cmap references
!-----------------------------------------------------------------------

!     //   reset input file
      rewind( iounit_tinker )

!     //   initialize
      iref_cmap = 0

!     //   loop of lines
      do

!        //   read a line
         read ( iounit_tinker, *, iostat=ierr ) keyword

!        //   error confirmation
         if ( ierr .ne. 0 ) exit

!        //   look for keyword
         if ( keyword(1:8) .eq. 'tortors ' ) then

!           //   counter
            iref_cmap = iref_cmap + 1

!        //   look for keyword
         end if

!     //   loop of lines
      end do

!     //   number of cmap sets
      nref_cmap = iref_cmap

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

!     //   existence of cmap
      allocate( exist_cmap(nref_cmap) )

!     //   atoms
      allocate( ikind_cmap(5,nref_cmap) )

!     //   existence of cmap
      allocate( vref_cmap(ngrid_cmap+1,ngrid_cmap+1,nref_cmap) )

!-----------------------------------------------------------------------
!     //   cmap counter
!-----------------------------------------------------------------------

!     //   reset input file
      rewind( iounit_tinker )

!     //   initialize
      iref_cmap = 0

!     //   loop of lines
      do

!        //   read a line
         read ( iounit_tinker, *, iostat=ierr ) keyword

!        //   error confirmation
         if ( ierr .ne. 0 ) exit

!        //   look for keyword
         if ( keyword(1:8) .eq. 'tortors ' ) then

!           //   counter
            iref_cmap = iref_cmap + 1

!           //   step back one line
            backspace( iounit_tinker )

!           //   read atom kinds
            read ( iounit_tinker, *, iostat=ierr ) &
     &         keyword, ( ikind_cmap(i,iref_cmap), i = 1, 5 ), &
     &         itmp, itmp

!           //   loop of grids
            do j = 1, ngrid_cmap+1
            do i = 1, ngrid_cmap+1

!              //   read potential data
               read ( iounit_tinker, *, iostat=ierr ) &
     &            tmp, tmp, vref_cmap(i,j,iref_cmap)

!           //   loop of grids
            end do
            end do

!        //   look for keyword
         end if

!     //   loop of lines
      end do

!-----------------------------------------------------------------------
!     //   cmap counter
!-----------------------------------------------------------------------

!     //   cmap counter
      icmap = 0

!     //   flag
      exist_cmap(:) = .false.

!     //   list
      list_cmap = 0

!     //   loop of cmap reference
      do iref_cmap = 1, nref_cmap

!        //  loop of torsion bonds
         do n = 1, ntors

!           //   atom and atom kind
            i = i_tors(3*n-2)
            j = j_tors(3*n-2)
            k = k_tors(3*n-2)
            l = l_tors(3*n-2)

!           //   apply only for specific cmap set
            if ( ikind(jkind(i)) .ne. ikind_cmap(1,iref_cmap) ) cycle
            if ( ikind(jkind(j)) .ne. ikind_cmap(2,iref_cmap) ) cycle
            if ( ikind(jkind(k)) .ne. ikind_cmap(3,iref_cmap) ) cycle
            if ( ikind(jkind(l)) .ne. ikind_cmap(4,iref_cmap) ) cycle

!           //   loop of atoms
            do m = 1, natom

!              //   skip when disconnected
               if ( bond(l,m) .eq. 0 ) cycle

!              //   apply only for specific cmap set
               if ( ikind(jkind(m)) .ne. ikind_cmap(5,iref_cmap) ) cycle

!              //   flag
               exist_cmap(iref_cmap) = .true.

!              //   list
               list_cmap = list_cmap + 1

!           //   loop of atoms
            end do

!        //  loop of torsion bonds
         end do

!        //  loop of torsion bonds
         do n = 1, ntors

!           //   atom and atom kind
            l = i_tors(3*n-2)
            k = j_tors(3*n-2)
            j = k_tors(3*n-2)
            i = l_tors(3*n-2)

!           //   apply only for specific cmap set
            if ( ikind(jkind(i)) .ne. ikind_cmap(1,iref_cmap) ) cycle
            if ( ikind(jkind(j)) .ne. ikind_cmap(2,iref_cmap) ) cycle
            if ( ikind(jkind(k)) .ne. ikind_cmap(3,iref_cmap) ) cycle
            if ( ikind(jkind(l)) .ne. ikind_cmap(4,iref_cmap) ) cycle

!           //   loop of atoms
            do m = 1, natom

!              //   skip when disconnected
               if ( bond(l,m) .eq. 0 ) cycle

!              //   apply only for specific cmap set
               if ( ikind(jkind(m)) .ne. ikind_cmap(5,iref_cmap) ) cycle

!              //   skip when symmetric
               if ( ( ikind(jkind(i)) .eq. ikind(jkind(m)) ) .and. &
     &              ( ikind(jkind(j)) .eq. ikind(jkind(l)) ) ) cycle

!              //   flag
               exist_cmap(iref_cmap) = .true.

!              //   list
               list_cmap = list_cmap + 1

!           //   loop of atoms
            end do

!        //  loop of torsion bonds
         end do

!        //   cmap counter
         if ( exist_cmap(iref_cmap) ) icmap = icmap + 1

!     //   loop of cmap reference
      end do

!     //   number of cmaps
      ncmap = icmap

!     //   number of cmap lists
      nkind_cmap = list_cmap

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

!     //   atomic kinds of cmap set
      allocate( i_cmap(2,nkind_cmap) )
      allocate( j_cmap(2,nkind_cmap) )
      allocate( k_cmap(2,nkind_cmap) )
      allocate( l_cmap(2,nkind_cmap) )

!     //   cmap set
      allocate( m_cmap(nkind_cmap) )

!     //   grid potential
      allocate( vgrid_cmap(mgrid_cmap,mgrid_cmap,nkind_cmap) )

!     //   derivatives
      allocate( v2grid_cmap(mgrid_cmap,mgrid_cmap,nkind_cmap) )

!     //   grids
      allocate( xgrid_cmap(mgrid_cmap) )

!     //   grids
      allocate( ygrid_cmap(mgrid_cmap) )

!-----------------------------------------------------------------------
!     //   cmap list
!-----------------------------------------------------------------------

!     //   cmap counter
      icmap = 0

!     //   list
      list_cmap = 0

!     //   loop of cmap reference
      do iref_cmap = 1, nref_cmap

!        //   flag
         if ( .not. exist_cmap(iref_cmap) ) cycle

!        //   cmap counter
         icmap = icmap + 1

!        //   copy grid potential
         do j = 1, ngrid_cmap
         do i = 1, ngrid_cmap
            vgrid_cmap(i,j,icmap) = vref_cmap(i,j,iref_cmap)
         end do
         end do

!        //  loop of torsion bonds
         do n = 1, ntors

!           //   atom and atom kind
            i = i_tors(3*n-2)
            j = j_tors(3*n-2)
            k = k_tors(3*n-2)
            l = l_tors(3*n-2)

!           //   apply only for specific cmap set
            if ( ikind(jkind(i)) .ne. ikind_cmap(1,iref_cmap) ) cycle
            if ( ikind(jkind(j)) .ne. ikind_cmap(2,iref_cmap) ) cycle
            if ( ikind(jkind(k)) .ne. ikind_cmap(3,iref_cmap) ) cycle
            if ( ikind(jkind(l)) .ne. ikind_cmap(4,iref_cmap) ) cycle

!           //   loop of atoms
            do m = 1, natom

!              //   skip when disconnected
               if ( bond(l,m) .eq. 0 ) cycle

!              //   apply only for specific cmap set
               if ( ikind(jkind(m)) .ne. ikind_cmap(5,iref_cmap) ) cycle

!              //   list
               list_cmap = list_cmap + 1

!              //   list
               i_cmap(1,list_cmap) = i
               j_cmap(1,list_cmap) = j
               k_cmap(1,list_cmap) = k
               l_cmap(1,list_cmap) = l
               i_cmap(2,list_cmap) = j
               j_cmap(2,list_cmap) = k
               k_cmap(2,list_cmap) = l
               l_cmap(2,list_cmap) = m
               m_cmap(list_cmap)   = icmap

!           //   loop of atoms
            end do

!        //  loop of torsion bonds
         end do

!        //  loop of torsion bonds
         do n = 1, ntors

!           //   atom and atom kind
            l = i_tors(3*n-2)
            k = j_tors(3*n-2)
            j = k_tors(3*n-2)
            i = l_tors(3*n-2)

!           //   apply only for specific cmap set
            if ( ikind(jkind(i)) .ne. ikind_cmap(1,iref_cmap) ) cycle
            if ( ikind(jkind(j)) .ne. ikind_cmap(2,iref_cmap) ) cycle
            if ( ikind(jkind(k)) .ne. ikind_cmap(3,iref_cmap) ) cycle
            if ( ikind(jkind(l)) .ne. ikind_cmap(4,iref_cmap) ) cycle

!           //   loop of atoms
            do m = 1, natom

!              //   skip when disconnected
               if ( bond(l,m) .eq. 0 ) cycle

!              //   apply only for specific cmap set
               if ( ikind(jkind(m)) .ne. ikind_cmap(5,iref_cmap) ) cycle

!              //   skip when symmetric
               if ( ( ikind(jkind(i)) .eq. ikind(jkind(m)) ) .and. &
     &              ( ikind(jkind(j)) .eq. ikind(jkind(l)) ) ) cycle

!              //   list
               list_cmap = list_cmap + 1

!              //   list
               i_cmap(1,list_cmap) = i
               j_cmap(1,list_cmap) = j
               k_cmap(1,list_cmap) = k
               l_cmap(1,list_cmap) = l
               i_cmap(2,list_cmap) = j
               j_cmap(2,list_cmap) = k
               k_cmap(2,list_cmap) = l
               l_cmap(2,list_cmap) = m
               m_cmap(list_cmap)   = icmap

!           //   loop of atoms
            end do

!        //  loop of torsion bonds
         end do

!     //   loop of cmap reference
      end do

!-----------------------------------------------------------------------
!     /*   set up cmap grids                                          */
!-----------------------------------------------------------------------

!     /*   change units to au   */
      vgrid_cmap(:,:,:) = vgrid_cmap(:,:,:) / har2kcal

!     /*   loop of cmap grids   */

      do i = 1, mgrid_cmap
         xgrid_cmap(i) = -180.d0 + dble(i-1)/dble(ngrid_cmap)*360.d0
         ygrid_cmap(i) = -180.d0 + dble(i-1)/dble(ngrid_cmap)*360.d0
      end do

!     /*   copy cmap grids to buffer region   */

      do k = 1, nkind_cmap
      do j = 1, ngrid_cmap
      do i = 1, nbuff_cmap
         vgrid_cmap(ngrid_cmap+i,j,k) = vgrid_cmap(i,j,k)
      end do
      end do
      end do

!     /*   copy cmap grids to buffer region   */

      do k = 1, nkind_cmap
      do j = 1, nbuff_cmap
      do i = 1, mgrid_cmap
         vgrid_cmap(i,ngrid_cmap+j,k) = vgrid_cmap(i,j,k)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   prepare for bicubic spline interpolation                   */
!-----------------------------------------------------------------------

      do k = 1, nkind_cmap

         call splie2_cmap( ygrid_cmap(:), vgrid_cmap(:,:,k), &
     &                     v2grid_cmap(:,:,k), mgrid_cmap, mgrid_cmap )

      end do

      return
      end





!***********************************************************************
      subroutine check_force_mm_cmap
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use tinker_variables, only : &
     &   pi, x, y, z, fx, fy, fz, pot, vir, har2kcal, symbol, &
     &   xgrid_cmap, ygrid_cmap, vgrid_cmap, v2grid_cmap, &
     &   i_cmap, j_cmap, k_cmap, l_cmap, ncmap, mgrid_cmap, nbead

      use mm_variables, only :
 
!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, n, i1or2

      real(8) :: xij, yij, zij, xkj, ykj, zkj, xlj, ylj, zlj, xijk, &
     &           yijk, zijk, xjkl, yjkl, zjkl, rijk2, rjkl2, rijkl2, &
     &           rijk2inv, rjkl2inv, rijkl2inv, cos_phi, sign_phi, &
     &           v, phi(2), dvdphi(2), fxi, fyi, fzi, fxj, fyj, fzj, &
     &           fxk, fyk, fzk, fxl, fyl, fzl, factor, sin_phi, f1, &
     &           xki, yki, zki, xkl, ykl, zkl, ax, ay, az, a1, a2, &
     &           daxdxi, daxdyi, daxdzi, daydxi, daydyi, daydzi, &
     &           dazdxi, dazdyi, dazdzi, dadxi, dadyi, dadzi, &
     &           daxdxj, daxdyj, daxdzj, daydxj, daydyj, daydzj, &
     &           dazdxj, dazdyj, dazdzj, dadxj, dadyj, dadzj, &
     &           daxdxl, daxdyl, daxdzl, daydxl, daydyl, daydzl, &
     &           dazdxl, dazdyl, dazdzl, dadxl, dadyl, dadzl, &
     &           phi1, phi2, p1c, p2c, dvdphi1, dvdphi2

      real(8) :: tiny_value = 1.d-4

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( ncmap .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      write( 6, '(a,25x,a,22x,a,5x,a,4x,a)' ) &
     &   'Type', 'Atom Names', 'Angle', 'Angle', 'Energy'
      write( 6, '(a)' )

      do m = 1, nbead

         do n = 1, ncmap

            do i1or2 = 1, 2

               i = i_cmap(i1or2,n)
               j = j_cmap(i1or2,n)
               k = k_cmap(i1or2,n)
               l = l_cmap(i1or2,n)

               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

               call pbc_atom ( xij, yij, zij )

               xkj = x(k,m) - x(j,m)
               ykj = y(k,m) - y(j,m)
               zkj = z(k,m) - z(j,m)

               call pbc_atom ( xkj, ykj, zkj )

               xlj = x(l,m) - x(j,m)
               ylj = y(l,m) - y(j,m)
               zlj = z(l,m) - z(j,m)

               call pbc_atom ( xlj, ylj, zlj )

               xijk = yij*zkj - zij*ykj
               yijk = zij*xkj - xij*zkj
               zijk = xij*ykj - yij*xkj

               xjkl = ylj*zkj - zlj*ykj
               yjkl = zlj*xkj - xlj*zkj
               zjkl = xlj*ykj - ylj*xkj

               rijk2  = xijk*xijk + yijk*yijk + zijk*zijk
               rjkl2  = xjkl*xjkl + yjkl*yjkl + zjkl*zjkl

               rijkl2 = sqrt(rijk2*rjkl2)

               if ( abs(rijk2)  .lt. tiny_value ) cycle
               if ( abs(rjkl2)  .lt. tiny_value ) cycle
               if ( abs(rijkl2) .lt. tiny_value ) cycle

               rijk2inv  = 1.d0 / rijk2
               rjkl2inv  = 1.d0 / rjkl2
               rijkl2inv = 1.d0 / rijkl2

               cos_phi = ( xijk*xjkl + yijk*yjkl + zijk*zjkl ) &
     &                 * rijkl2inv

               cos_phi = max( cos_phi, -1.d0 )
               cos_phi = min( cos_phi,  1.d0 )

               phi(i1or2) = acos(cos_phi)

               sign_phi = ( yijk*zjkl - zijk*yjkl ) * xkj &
     &                  + ( zijk*xjkl - xijk*zjkl ) * ykj &
     &                  + ( xijk*yjkl - yijk*xjkl ) * zkj

               sign_phi = sign( 1.d0, sign_phi )

               phi(i1or2) = phi(i1or2) * sign_phi

            end do

            p1c = 0.5d0 * ( xgrid_cmap(1) + xgrid_cmap(mgrid_cmap) )
            p2c = 0.5d0 * ( ygrid_cmap(1) + ygrid_cmap(mgrid_cmap) )

            phi1 = phi(1) * 180.d0 / pi
            phi2 = phi(2) * 180.d0 / pi

            phi1 = phi1 - nint( (phi1-p1c) / 360.d0 ) * 360.d0
            phi2 = phi2 - nint( (phi2-p2c) / 360.d0 ) * 360.d0

            call splin2_cmap( xgrid_cmap(:), ygrid_cmap(:), &
     &                        vgrid_cmap(:,:,n), v2grid_cmap(:,:,n), &
     &                        mgrid_cmap, mgrid_cmap, &
     &                        phi1, phi2, v, dvdphi1, dvdphi2 )

            dvdphi(1) = dvdphi1 * 180.d0 / pi
            dvdphi(2) = dvdphi2 * 180.d0 / pi

            pot(m)  = pot(m) + v

            do i1or2 = 1, 2

               i = i_cmap(i1or2,n)
               j = j_cmap(i1or2,n)
               k = k_cmap(i1or2,n)
               l = l_cmap(i1or2,n)

               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

               call pbc_atom ( xij, yij, zij )

               xkj = x(k,m) - x(j,m)
               ykj = y(k,m) - y(j,m)
               zkj = z(k,m) - z(j,m)

               call pbc_atom ( xkj, ykj, zkj )

               xlj = x(l,m) - x(j,m)
               ylj = y(l,m) - y(j,m)
               zlj = z(l,m) - z(j,m)

               call pbc_atom ( xlj, ylj, zlj )

               xijk = yij*zkj - zij*ykj
               yijk = zij*xkj - xij*zkj
               zijk = xij*ykj - yij*xkj

               xjkl = ylj*zkj - zlj*ykj
               yjkl = zlj*xkj - xlj*zkj
               zjkl = xlj*ykj - ylj*xkj

               rijk2  = xijk*xijk + yijk*yijk + zijk*zijk
               rjkl2  = xjkl*xjkl + yjkl*yjkl + zjkl*zjkl

               rijkl2 = sqrt(rijk2*rjkl2)

               if ( abs(rijk2)  .lt. tiny_value ) cycle
               if ( abs(rjkl2)  .lt. tiny_value ) cycle
               if ( abs(rijkl2) .lt. tiny_value ) cycle

               rijk2inv  = 1.d0 / rijk2
               rjkl2inv  = 1.d0 / rjkl2
               rijkl2inv = 1.d0 / rijkl2

               cos_phi = ( xijk*xjkl + yijk*yjkl + zijk*zjkl ) &
     &                 * rijkl2inv

               cos_phi = max( cos_phi, -1.d0 )
               cos_phi = min( cos_phi,  1.d0 )

               phi(i1or2) = acos(cos_phi)

               sign_phi = ( yijk*zjkl - zijk*yjkl ) * xkj &
     &                  + ( zijk*xjkl - xijk*zjkl ) * ykj &
     &                  + ( xijk*yjkl - yijk*xjkl ) * zkj

               sign_phi = sign( 1.d0, sign_phi )

               phi(i1or2) = phi(i1or2) * sign_phi

               if ( ( abs(phi(i1or2))    .gt. tiny_value ) .and. &
     &              ( abs(phi(i1or2)-pi) .gt. tiny_value ) .and. &
     &              ( abs(phi(i1or2)+pi) .gt. tiny_value ) ) then

                  sin_phi = sin( phi(i1or2) )

                  factor = dvdphi(i1or2) / sin_phi

                  fxi = factor * ( + ( ykj*zjkl - zkj*yjkl ) * rijkl2inv &
     &                    - ( ykj*zijk - zkj*yijk ) * cos_phi*rijk2inv )
                  fyi = factor * ( + ( zkj*xjkl - xkj*zjkl ) * rijkl2inv &
     &                    - ( zkj*xijk - xkj*zijk ) * cos_phi*rijk2inv )
                  fzi = factor * ( + ( xkj*yjkl - ykj*xjkl ) * rijkl2inv &
     &                    - ( xkj*yijk - ykj*xijk ) * cos_phi*rijk2inv )

                  fxl = factor * ( + ( ykj*zijk - zkj*yijk ) * rijkl2inv &
     &                    - ( ykj*zjkl - zkj*yjkl ) * cos_phi*rjkl2inv )
                  fyl = factor * ( + ( zkj*xijk - xkj*zijk ) * rijkl2inv &
     &                    - ( zkj*xjkl - xkj*zjkl ) * cos_phi*rjkl2inv )
                  fzl = factor * ( + ( xkj*yijk - ykj*xijk ) * rijkl2inv &
     &                    - ( xkj*yjkl - ykj*xjkl ) * cos_phi*rjkl2inv )

                  fxk = factor * ( - ( yij*zjkl - zij*yjkl ) * rijkl2inv &
     &                    - ( ylj*zijk - zlj*yijk ) * rijkl2inv &
     &                    + ( yij*zijk - zij*yijk ) * cos_phi*rijk2inv &
     &                    + ( ylj*zjkl - zlj*yjkl ) * cos_phi*rjkl2inv )
                  fyk = factor * ( - ( zij*xjkl - xij*zjkl ) * rijkl2inv &
     &                    - ( zlj*xijk - xlj*zijk ) * rijkl2inv &
     &                    + ( zij*xijk - xij*zijk ) * cos_phi*rijk2inv &
     &                    + ( zlj*xjkl - xlj*zjkl ) * cos_phi*rjkl2inv )
                  fzk = factor * ( - ( xij*yjkl - yij*xjkl ) * rijkl2inv &
     &                    - ( xlj*yijk - ylj*xijk ) * rijkl2inv &
     &                    + ( xij*yijk - yij*xijk ) * cos_phi*rijk2inv &
     &                    + ( xlj*yjkl - ylj*xjkl ) * cos_phi*rjkl2inv )

                  fxj = - ( fxi + fxk + fxl )
                  fyj = - ( fyi + fyk + fyl )
                  fzj = - ( fzi + fzk + fzl )

               else

                  xki = - xij + xkj
                  yki = - yij + ykj
                  zki = - zij + zkj

                  xkl = - xlj + xkj
                  ykl = - ylj + ykj
                  zkl = - zlj + zkj

                  ax = yijk*zjkl - zijk*yjkl
                  ay = zijk*xjkl - xijk*zjkl
                  az = xijk*yjkl - yijk*xjkl

                  a2 = ax*ax + ay*ay + az*az

                  a1 = sqrt( a2 )

                  sin_phi = a1 / rijkl2

                  sin_phi = max( sin_phi, -1.d0 )
                  sin_phi = min( sin_phi,  1.d0 )

                  phi = sign_phi * asin( sin_phi )

                  if ( cos_phi .lt. 0.d0 ) phi = pi - phi

                  daxdxi = - zjkl * zkj - ykj * yjkl
                  daxdyi = + yjkl * xkj
                  daxdzi = + zjkl * xkj

                  daydxi = + xjkl * ykj
                  daydyi = - xjkl * xkj - zkj * zjkl
                  daydzi = + zjkl * ykj

                  dazdxi = + xjkl * zkj
                  dazdyi = + yjkl * zkj
                  dazdzi = - yjkl * ykj - xkj * xjkl

                  daxdxj = - yijk * ykl + zjkl * zki &
     &                     + yjkl * yki - zijk * zkl
                  daxdyj = + yijk * xkl - yjkl * xki
                  daxdzj = + zijk * xkl - zjkl * xki

                  daydxj = + xijk * ykl - xjkl * yki
                  daydyj = - zijk * zkl + xjkl * xki &
     &                     + zjkl * zki - xijk * xkl
                  daydzj = + zijk * ykl - zjkl * yki

                  dazdxj = + xijk * zkl - xjkl * zki
                  dazdyj = + yijk * zkl - yjkl * zki
                  dazdzj = - xijk * xkl + yjkl * yki &
     &                     + xjkl * xki - yijk * ykl

!                  daxdxk = - yjkl * yij + zijk * zlj
!                           + yijk * ylj - zjkl * zij
!                  daxdyk = + yjkl * xij - yijk * xlj
!                  daxdzk = + zjkl * xij - zijk * xlj
!
!                  daydxk = + xjkl * yij - xijk * ylj
!                  daydyk = - zjkl * zij + xijk * xlj
!                           + zijk * zlj - xjkl * xij
!                  daydzk = + zjkl * yij - zijk * ylj
!
!                  dazdxk = + xjkl * zij - xijk * zlj
!                  dazdyk = + yjkl * zij - yijk * zlj
!                  dazdzk = - xjkl * xij + yijk * ylj
!                           + xijk * xlj - yjkl * yij

                  daxdxl = + zijk * zkj + ykj * yijk
                  daxdyl = - yijk * xkj
                  daxdzl = - zijk * xkj

                  daydxl = - xijk * ykj
                  daydyl = + xijk * xkj + zkj * zijk
                  daydzl = - zijk * ykj

                  dazdxl = - xijk * zkj
                  dazdyl = - yijk * zkj
                  dazdzl = + yijk * ykj + xkj * xijk

                  dadxi = ax/a1*daxdxi + ay/a1*daydxi + az/a1*dazdxi
                  dadyi = ax/a1*daxdyi + ay/a1*daydyi + az/a1*dazdyi
                  dadzi = ax/a1*daxdzi + ay/a1*daydzi + az/a1*dazdzi

                  dadxj = ax/a1*daxdxj + ay/a1*daydxj + az/a1*dazdxj
                  dadyj = ax/a1*daxdyj + ay/a1*daydyj + az/a1*dazdyj
                  dadzj = ax/a1*daxdzj + ay/a1*daydzj + az/a1*dazdzj

!                  dadxk = ax/a1*daxdxk + ay/a1*daydxk + az/a1*dazdxk
!                  dadyk = ax/a1*daxdyk + ay/a1*daydyk + az/a1*dazdyk
!                  dadzk = ax/a1*daxdzk + ay/a1*daydzk + az/a1*dazdzk

                  dadxl = ax/a1*daxdxl + ay/a1*daydxl + az/a1*dazdxl
                  dadyl = ax/a1*daxdyl + ay/a1*daydyl + az/a1*dazdyl
                  dadzl = ax/a1*daxdzl + ay/a1*daydzl + az/a1*dazdzl

                  f1 = - sign_phi / cos_phi * dvdphi(i1or2)

                  fxi = + f1 * ( dadxi / rijkl2 &
     &                + sin_phi * ( + yijk*zkj - zijk*ykj ) * rijk2inv )

                  fyi = + f1 * ( dadyi / rijkl2 &
     &                + sin_phi * ( + zijk*xkj - xijk*zkj ) * rijk2inv )

                  fzi = + f1 * ( dadzi / rijkl2 &
     &                + sin_phi * ( + xijk*ykj - yijk*xkj ) * rijk2inv )

                  fxj = + f1 * ( dadxj / rijkl2 &
     &                + sin_phi * ( - yijk*zki + zijk*yki ) * rijk2inv &
     &                - sin_phi * ( + yjkl*zkl - zjkl*ykl ) * rjkl2inv )

                  fyj = + f1 * ( dadyj / rijkl2 &
     &                + sin_phi * ( - zijk*xki + xijk*zki ) * rijk2inv &
     &                - sin_phi * ( + zjkl*xkl - xjkl*zkl ) * rjkl2inv )

                  fzj = + f1 * ( dadzj / rijkl2 &
     &                + sin_phi * ( - xijk*yki + yijk*xki ) * rijk2inv &
     &                - sin_phi * ( + xjkl*ykl - yjkl*xkl ) * rjkl2inv )

!                 fxk = - f1 * ( dadxk / rijkl2
!     &                - sin_phi * ( - yjkl*zlj + zjkl*ylj ) * rjkl2inv
!     &                + sin_phi * ( + yijk*zij - zijk*yij ) * rijk2inv )
!
!                 fyk = - f1 * ( dadyk / rijkl2
!     &                - sin_phi * ( - zjkl*xlj + xjkl*zlj ) * rjkl2inv
!     &                + sin_phi * ( + zijk*xij - xijk*zij ) * rijk2inv )
!
!                 fzk = - f1 * ( dadzk / rijkl2
!     &                - sin_phi * ( - xjkl*ylj + yjkl*xlj ) * rjkl2inv
!     &                + sin_phi * ( + xijk*yij - yijk*xij ) * rijk2inv )

                  fxl = + f1 * ( dadxl / rijkl2 &
     &                + sin_phi * ( + yjkl*zkj - zjkl*ykj ) * rjkl2inv )

                  fyl = + f1 * ( dadyl / rijkl2 &
     &                + sin_phi * ( + zjkl*xkj - xjkl*zkj ) * rjkl2inv )

                  fzl = + f1 * ( dadzl / rijkl2 &
     &                + sin_phi * ( + xjkl*ykj - yjkl*xkj ) * rjkl2inv )

                  fxk = - fxi - fxj - fxl
                  fyk = - fyi - fyj - fyl
                  fzk = - fzi - fzj - fzl

               end if

               fx(i,m) = fx(i,m) + fxi
               fx(j,m) = fx(j,m) + fxj
               fx(k,m) = fx(k,m) + fxk
               fx(l,m) = fx(l,m) + fxl

               fy(i,m) = fy(i,m) + fyi
               fy(j,m) = fy(j,m) + fyj
               fy(k,m) = fy(k,m) + fyk
               fy(l,m) = fy(l,m) + fyl

               fz(i,m) = fz(i,m) + fzi
               fz(j,m) = fz(j,m) + fzj
               fz(k,m) = fz(k,m) + fzk
               fz(l,m) = fz(l,m) + fzl

               vir(1,1) = vir(1,1) + fxi*xij + fxk*xkj + fxl*xlj
               vir(1,2) = vir(1,2) + fxi*yij + fxk*ykj + fxl*ylj
               vir(1,3) = vir(1,3) + fxi*zij + fxk*zkj + fxl*zlj
               vir(2,1) = vir(2,1) + fyi*xij + fyk*xkj + fyl*xlj
               vir(2,2) = vir(2,2) + fyi*yij + fyk*ykj + fyl*ylj
               vir(2,3) = vir(2,3) + fyi*zij + fyk*zkj + fyl*zlj
               vir(3,1) = vir(3,1) + fzi*xij + fzk*xkj + fzl*xlj
               vir(3,2) = vir(3,2) + fzi*yij + fzk*ykj + fzl*ylj
               vir(3,3) = vir(3,3) + fzi*zij + fzk*zkj + fzl*zlj

            end do

!-----------------------------------------------------------------------
!     /*   print out                                                  */
!-----------------------------------------------------------------------

            write( 6, '(a4,7x,5(i5,a1,a3),3f10.4)' ) &
     &         'Cmap', &
     &         i_cmap(1,n), '-', symbol(i_cmap(1,n)), &
     &         j_cmap(1,n), '-', symbol(j_cmap(1,n)), &
     &         k_cmap(1,n), '-', symbol(k_cmap(1,n)), &
     &         l_cmap(1,n), '-', symbol(l_cmap(1,n)), &
     &         l_cmap(2,n), '-', symbol(l_cmap(2,n)), &
     &         phi1, phi2, v*har2kcal

         end do

      end do

      write( 6, '(a)' )

      return
      end





!***********************************************************************
      subroutine splie2_cmap( x2a, ya, y2a, m, n )
!***********************************************************************

      implicit none

      integer :: j, k, m, n

      real(8) :: x2a(n), ya(m,n), y2a(m,n), ytmp(n), y2tmp(n)

      do j = 1, m

         do k = 1, n
            ytmp(k) = ya(j,k)
         end do

         call spline_cmap( x2a, ytmp, n, y2tmp )

         do k = 1, n
            y2a(j,k) = y2tmp(k)
         end do

      end do

      return
      end





!***********************************************************************
      subroutine splin2_cmap &
     &   ( x1a, x2a, ya, y2a, m, n, x1, x2, y, dydx1, dydx2 )
!***********************************************************************

      implicit none

      integer :: j, k, m, n

      real(8) :: x1a(m), x2a(n), ya(m,n), y2a(m,n)
      real(8) :: ytmp(m), y2tmp(m), yytmp(m), z2tmp(m), zztmp(m)
      real(8) :: x1, x2, y, dydx1, dydx2, dy2dx1dx2

      do j = 1, m

         do k = 1, n
            ytmp(k) = ya(j,k)
            y2tmp(k) = y2a(j,k)
         end do

         call splint_cmap( x2a, ytmp, y2tmp, n, x2, yytmp(j), zztmp(j) )

      end do

      call spline_cmap( x1a, yytmp, m, y2tmp )
      call splint_cmap( x1a, yytmp, y2tmp, m, x1, y, dydx1 )

      call spline_cmap( x1a, zztmp, m, z2tmp )
      call splint_cmap( x1a, zztmp, z2tmp, m, x1, dydx2, dy2dx1dx2 )

      return
      end





!***********************************************************************
      subroutine spline_cmap( x, y, n, y2 )
!***********************************************************************

      implicit none

      integer :: i, k, n

      real(8) :: x(n), y(n), y2(n), u(n), sig, p

      y2(1) = 0.d0
      u(1)  = 0.d0

      do i = 2, n-1

         sig = ( x(i) - x(i-1) ) / ( x(i+1) - x(i-1) )

         p = sig * y2(i-1) + 2.d0

         y2(i) = ( sig - 1.d0 ) / p

         u(i) = ( 6.d0 * ( ( y(i+1) - y(i) ) / ( x(i+1) - x(i) ) &
     &                   - ( y(i) - y(i-1) ) / ( x(i) - x(i-1) ) ) &
     &                   / ( x(i+1) - x(i-1) ) &
     &          - sig * u(i-1) ) / p

      end do

      y2(n) = 0.d0

      do k = n-1, 1, -1
         y2(k) = y2(k) * y2(k+1) + u(k)
      end do

      return
      end





!***********************************************************************
      subroutine splint_cmap( xa, ya, y2a, n, x, y, dydx )
!***********************************************************************

      implicit none

      integer :: khi, klo, n

      real(8) :: xa(n), ya(n), y2a(n), x, y, h, a, b, dydx, hinv

      klo = int( ( x - xa(1) ) / ( xa(n) - xa(1) ) * ( n - 1 ) ) + 1
      khi = klo + 1

      h = xa(khi) - xa(klo)

      hinv = 1.d0 / h

      a = ( xa(khi) - x ) * hinv
      b = ( x - xa(klo) ) * hinv

      y = a * ya(klo) + b * ya(khi) &
     &  + ( (a*a*a-a) * y2a(klo) + (b*b*b-b) * y2a(khi) ) * (h*h) / 6.d0

      dydx = ( - ya(klo) + ya(khi) ) * hinv &
     &     + ( ( 1.d0 - 3.d0*a*a ) * y2a(klo) &
     &       + ( 3.d0*b*b - 1.d0 ) * y2a(khi) ) * h / 6.d0

      return
      end
