!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     convert charmm input file to pimd input files
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      module charmm_variables
!***********************************************************************

!-----------------------------------------------------------------------

!     /*   name of charmm par file   */
      character(len=80) :: par_file

!     /*   name of charmm psf file   */
      character(len=80) :: psf_file

!     /*   name of charmm psf file   */
      character(len=80) :: pdb_file

!-----------------------------------------------------------------------

!     /*   circular constant   */
      real(8), parameter :: pi = 3.141592653589793d0

!     /*   hartree to kcal/mol   */
      real(8), parameter :: unit_to_kcal      = 627.5095d0

!     /*   bohr to angstrom   */
      real(8), parameter :: unit_to_angstrom  = 0.529177249d0

!     /*   degree to radian   */
      real(8), parameter :: unit_to_rad       = 180.d0 / pi

!-----------------------------------------------------------------------

!     /*   symbol of atomic species   */
      character(len=4), dimension(:), allocatable :: symbol_atom

!     /*   symbol of atomic species   */
      character(len=4), dimension(:), allocatable :: symbol_type

!     /*   symbol of molecular residue   */
      character(len=4), dimension(:), allocatable :: symbol_resi

!-----------------------------------------------------------------------

!     /*   number of atoms   */
      integer :: natom

!     /*   number of direct bonds   */
      integer :: nbond = 0

!     /*   number of angular bonds    */
      integer :: nangl = 0

!     /*   number of urey-bradley bonds    */
      integer :: nub = 0

!     /*   number of dihedral bonds    */
      integer :: ndih = 0

!     /*   number of improper bonds    */
      integer :: nimproper = 0

!     /*   number of linear bonds = nbond + nub   */
      integer :: nlin = 0

!     /*   number of lj pairs   */
      integer :: nlj = 0

!     /*   number of non-interacting lj pairs   */
      integer :: nlj14 = 0

!     /*   number of bonded charge pairs   */
      integer :: nbcp = 0

!     /*   number of direct bonds   */
      integer :: nbond_nonzero

!     /*   number of urey-bradley bonds    */
      integer :: nub_nonzero

!     /*   number of linear bonds = nbond + nub   */
      integer :: nlin_nonzero

!     /*   number of angular bonds    */
      integer :: nangl_nonzero

!     /*   number of dihedral bonds    */
      integer :: ndih_nonzero

!     /*   number of improper bonds    */
      integer :: nimproper_nonzero

!     /*   number of lj pairs   */
      integer :: nlj_nonzero

!     /*   number of non-interacting lj pairs   */
      integer :: nlj14_nonzero

!     /*   number of charged atoms   */
      integer :: ncharge

!-----------------------------------------------------------------------

!     /*   atomic charges   */
      real(8), dimension(:),   allocatable :: charge

!-----------------------------------------------------------------------

!     /*   lj: epsilon   */
      real(8), dimension(:),   allocatable :: eps_lj

!     /*   lj: square root of epsilon   */
      real(8), dimension(:),   allocatable :: sqrt_eps_lj

!     /*   lj: half of equilibrium distance   */
      real(8), dimension(:),   allocatable :: rmh_lj

!     /*   lj: square root of epsilon for 1-4 interaction   */
      real(8), dimension(:),   allocatable :: sqrt_eps_lj14

!     /*   lj: half of equilibrium distance for 1-4 interaction   */
      real(8), dimension(:),   allocatable :: rmh_lj14

!-----------------------------------------------------------------------

!     /*   atom i of direct bond   */
      integer, dimension(:),   allocatable :: i_lj14

!     /*   atom j of direct bond   */
      integer, dimension(:),   allocatable :: j_lj14

!     /*   lj: epsilon   */
      real(8), dimension(:),   allocatable :: eps_lj14

!     /*   lj: square root of epsilon   */
      real(8), dimension(:),   allocatable :: sig_lj14

!-----------------------------------------------------------------------

!     /*   lj pair: epsilon   */
      real(8), dimension(:),   allocatable :: eps_lj_pair

!     /*   lj pair: sigma   */
      real(8), dimension(:),   allocatable :: sig_lj_pair

!     /*   lj pair: atom i   */
      integer, dimension(:),   allocatable :: i_lj_pair

!     /*   lj pair: atom j   */
      integer, dimension(:),   allocatable :: j_lj_pair

!-----------------------------------------------------------------------

!     /*   atom i of direct bond   */
      integer, dimension(:),   allocatable :: i_bond

!     /*   atom j of direct bond   */
      integer, dimension(:),   allocatable :: j_bond

!     /*   force constant of direct bond   */
      real(8), dimension(:),   allocatable :: fc_bond

!     /*   equilibrium distance of direct bond   */
      real(8), dimension(:),   allocatable :: eq_bond

!-----------------------------------------------------------------------

!     /*   atom i of bond angle   */
      integer, dimension(:),   allocatable :: i_angl

!     /*   atom j of bond angle   */
      integer, dimension(:),   allocatable :: j_angl

!     /*   atom k of bond angle   */
      integer, dimension(:),   allocatable :: k_angl

!     /*   force constant of bond angle   */
      real(8), dimension(:),   allocatable :: fc_angl

!     /*   equilibrium distance of bond angle   */
      real(8), dimension(:),   allocatable :: eq_angl

!-----------------------------------------------------------------------

!     /*   atom i of urey-bradley   */
      integer, dimension(:),   allocatable :: i_ub

!     /*   atom j of urey-bradley   */
      integer, dimension(:),   allocatable :: j_ub

!     /*   force constant of urey-bradley   */
      real(8), dimension(:),   allocatable :: fc_ub

!     /*   equilibrium distance of urey-bradley   */
      real(8), dimension(:),   allocatable :: eq_ub

!-----------------------------------------------------------------------

!     /*   atom i of dihedral bond   */
      integer, dimension(:),   allocatable :: i_dih

!     /*   atom j of dihedral bond   */
      integer, dimension(:),   allocatable :: j_dih

!     /*   atom k of dihedral bond   */
      integer, dimension(:),   allocatable :: k_dih

!     /*   atom l of dihedral bond   */
      integer, dimension(:),   allocatable :: l_dih

!     /*   barrier height of dihedral bond   */
      real(8), dimension(:),   allocatable :: v_dih

!     /*   phase angle of dihedral bond   */
      integer, dimension(:),   allocatable :: mu_dih

!     /*   multiplicity of dihedral bond   */
      integer, dimension(:),   allocatable :: nu_dih

!-----------------------------------------------------------------------

!     /*   atom i of improper bond   */
      integer, dimension(:),   allocatable :: i_improper

!     /*   atom j of improper bond   */
      integer, dimension(:),   allocatable :: j_improper

!     /*   atom k of improper bond   */
      integer, dimension(:),   allocatable :: k_improper

!     /*   atom l of improper bond   */
      integer, dimension(:),   allocatable :: l_improper

!     /*   force constant of improper bond   */
      real(8), dimension(:),   allocatable :: fc_improper

!     /*   equilbrium distance of improper bond   */
      real(8), dimension(:),   allocatable :: eq_improper

!-----------------------------------------------------------------------

!     /*   number of parameters for direct bonds   */
      integer :: nbond_par

!     /*   number of parameters for angular bonds   */
      integer :: nangl_par

!     /*   number of parameters for urey-bradley   */
      integer :: nub_par

!     /*   number of parameters for dihedral bonds   */
      integer :: ndih_par

!     /*   number of parameters for improper bonds   */
      integer :: nimproper_par

!     /*   number of parameters for lennard-jones   */
      integer :: nlj_par

!     /*   number of parameters for 1-4 lennard-jones   */
      integer :: nlj14_par

!-----------------------------------------------------------------------

!     /*   atomic symbols for direct bonds   */
      character(len=4), dimension(2,300) :: symbol_bond

!     /*   atomic symbols for angular bonds   */
      character(len=4), dimension(3,500) :: symbol_angl

!     /*   atomic symbols for dihedral bonds   */
      character(len=4), dimension(4,700) :: symbol_dih

!     /*   atomic symbols for improper bonds   */
      character(len=4), dimension(4,700) :: symbol_improper

!     /*   atomic symbols for lennard-jones   */
      character(len=4), dimension(300)   :: symbol_lj

!-----------------------------------------------------------------------

!     /*   force constant of direct bond   */
      real(8), dimension(300) :: fc_bond_par

!     /*   equilibrium distance of direct bond   */
      real(8), dimension(300) :: eq_bond_par

!-----------------------------------------------------------------------

!     /*   force constant of angular bond   */
      real(8), dimension(500) :: fc_angl_par

!     /*   equilibrium angle of angular bond   */
      real(8), dimension(500) :: eq_angl_par

!-----------------------------------------------------------------------

!     /*   force constant of urey-bradley   */
      real(8), dimension(500) :: fc_ub_par

!     /*   equilibrium distance of urey-bradley   */
      real(8), dimension(500) :: eq_ub_par

!-----------------------------------------------------------------------

!     /*   barrier height of dihedral bond   */
      real(8), dimension(700) :: v_dih_par

!     /*   phase angle of dihedral bond   */
      real(8), dimension(700) :: delta_dih_par

!     /*   multiplicity of dihedral bond   */
      integer, dimension(700) :: nu_dih_par

!-----------------------------------------------------------------------

!     /*   force constant of improper bond   */
      real(8), dimension(700) :: fc_improper_par

!     /*   equilbrium angle of improper bond   */
      real(8), dimension(700) :: eq_improper_par

!-----------------------------------------------------------------------

!     /*   lj: epsilon   */
      real(8), dimension(300) :: eps_lj_par

!     /*   lj: half of equilibrium distance   */
      real(8), dimension(300) :: rmh_lj_par

!     /*   lj: epsilon for 1-4 interaction   */
      real(8), dimension(300) :: eps_lj14_par

!     /*   lj: half of equilibrium distance for 1-4 interaction   */
      real(8), dimension(300) :: rmh_lj14_par

!-----------------------------------------------------------------------

!     /*   existence of 1-4 bond   */
      logical, dimension(:),   allocatable :: bonded_dih

!-----------------------------------------------------------------------

!     /*   list of 1-2 bond   */
      integer, dimension(:),   allocatable :: nlist_12
      integer, dimension(:,:), allocatable :: list_12

!     /*   list of 1-3 bond   */
      integer, dimension(:),   allocatable :: nlist_13
      integer, dimension(:,:), allocatable :: list_13

!     /*   list of 1-4 bond   */
      integer, dimension(:),   allocatable :: nlist_14
      integer, dimension(:,:), allocatable :: list_14

!-----------------------------------------------------------------------

!     /*   lennard-jones cut off radius (inner, switching function)   */
      real(8) :: rcut_in

!     /*   lennard-jones cut off radius (outer, switching function)   */
      real(8) :: rcut_out

!-----------------------------------------------------------------------

!     /*   scaling factor for 1-4 bond   */
!cc      real(8) :: factor_14 =  0.500d0
      real(8) :: factor_14 =  1.000d0

!-----------------------------------------------------------------------

!     /*   atom i of bonded charge pair   */
      integer, dimension(:),   allocatable :: i_bcp

!     /*   atom j of bonded charge pair   */
      integer, dimension(:),   allocatable :: j_bcp

!     /*   scaling factor of bonded charge pair   */
      real(8), dimension(:),   allocatable :: factor_bcp

!-----------------------------------------------------------------------

!     /*   atomic coordinates   */
      real(8), dimension(:), allocatable :: x
      real(8), dimension(:), allocatable :: y
      real(8), dimension(:), allocatable :: z

!     /*   atomic masses in amu   */
      real(8), dimension(:), allocatable :: physmass_amu

!     /*   atomic symbols   */
      character(len=4), dimension(:), allocatable :: symbol_atom_pdb

!     /*   symbol of molecular residue   */
      character(len=4), dimension(:), allocatable :: symbol_resi_pdb

!     /*   number of atoms   */
      integer :: natom_pdb

!-----------------------------------------------------------------------

!     /*   option   */
      integer :: ioption_sort

!***********************************************************************
      end module charmm_variables
!***********************************************************************






!***********************************************************************
      program convert_charmm
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   local variables   */
      implicit none

!-----------------------------------------------------------------------
!     /*   set up                                                     */
!-----------------------------------------------------------------------

      call setup_convert

!-----------------------------------------------------------------------
!     /*   read pdb file                                              */
!-----------------------------------------------------------------------

      call read_pdb

!-----------------------------------------------------------------------
!     /*   read charmm par file                                       */
!-----------------------------------------------------------------------

      call read_charmm_par

!-----------------------------------------------------------------------
!     /*   read charmm psf file                                       */
!-----------------------------------------------------------------------

      call read_charmm_psf

!-----------------------------------------------------------------------
!     /*   write centroid.dat file                                    */
!-----------------------------------------------------------------------

      call write_centroid

!-----------------------------------------------------------------------
!     /*   write input.dat file                                       */
!-----------------------------------------------------------------------

      call write_input

!-----------------------------------------------------------------------
!     /*   write mm.dat file                                          */
!-----------------------------------------------------------------------

      call write_mm

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      write( 6, '(a)' )
      write( 6, '(a)' ) 'Done.'
      write( 6, '(a)' )

      stop
      end





!***********************************************************************
      subroutine read_charmm_par
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */

      use charmm_variables, only : &
     &   symbol_bond, symbol_angl, symbol_dih, symbol_improper, &
     &   fc_bond_par, eq_bond_par, fc_angl_par, fc_improper_par, &
     &   eq_angl_par, fc_ub_par, v_dih_par, delta_dih_par, &
     &   eq_improper_par, eps_lj_par, rmh_lj_par, symbol_lj, &
     &   eps_lj14_par, rmh_lj14_par, eq_ub_par, fc_ub_par, &
     &   nbond_par, nangl_par, nub_par, ndih_par, nimproper_par, &
     &   nu_dih_par, nlj_par, nlj14_par, par_file

!     /*   local variables   */

      implicit none

      integer :: ierr

      character(len=10) :: char_dummy

      character(len=90) :: string

!-----------------------------------------------------------------------
!     /*   open charmm parameter file                                 */
!-----------------------------------------------------------------------

      open( 1, file = par_file, status = 'old' )

!-----------------------------------------------------------------------
!     /*   skip headers                                               */
!-----------------------------------------------------------------------

      do

!        /*   read line   */
         read( 1, '(a)' ) string

!        /*   find the next keyword   */
         if ( string(1:5) .eq. 'BONDS' ) exit

      end do

!-----------------------------------------------------------------------
!
!     ===  linear bonds  ===
!
!        term:
!
!           v(bond) = fc_bond * ( x - eq_bond )**2
!
!        parameters:
!
!           fc_bond:  kcal/mol/A**2
!           eq_bond:  A
!
!        format:
!
!           atom types(2), fc_bond, eq_bond
!
!-----------------------------------------------------------------------

!     /*   zero clear   */
      nbond_par = 0

!     /*   start loop   */
      do

!        /*   read line   */
         read ( 1, '(a)' ) string

!        /*   adjust to the left   */
         string = adjustl(string)

!        /*   look for the next keyword   */
         if ( string(1:5) .eq. 'ANGLE' ) exit

!        /*   skip comment lines   */
         if ( string(1:1) .eq. '*' ) cycle
         if ( string(1:1) .eq. '!' ) cycle
         if ( string(1:1) .eq. ' ' ) cycle
         if ( string(1:1) .eq. ' ' ) cycle

!        /*   update counter   */
         nbond_par = nbond_par + 1

!        /*   read atom types(2), force constant, distance   */
         read ( string, * ) &
     &      symbol_bond(1,nbond_par), &
     &      symbol_bond(2,nbond_par), &
     &      fc_bond_par(nbond_par), &
     &      eq_bond_par(nbond_par)

!     /*   end of loop   */
      end do

!     /*   print   */

      write( 6, '(a,i6)' )
      write( 6, '(a,i6)' ) &
     &   '   PAR:  types of linear bonds    = ', nbond_par

!-----------------------------------------------------------------------
!
!     ===  angular bonds  ===
!
!        terms:
!
!           v(angle) = fc_angl * ( x - eq_angl )**2
!
!           v(urey-bradley) = fc_ub * ( x - eq_ub )**2
!
!        parameters:
!
!           fc_angl:  kcal/mol/rad**2
!           eq_angl:  degrees
!
!           fc_ub:    kcal/mol/A**2 (urey-bradley)
!           eq_ub:    A
!
!        format:
!
!           atom types(3), fc_angl, eq_angl, fc_ub, eq_ub
!
!-----------------------------------------------------------------------

!     /*   zero clear   */

      nangl_par = 0
      nub_par   = 0

!     /*   start loop   */
      do

!        /*   read line   */
         read ( 1, '(a)' ) string

!        /*   adjust to the left   */
         string = adjustl(string)

!        /*   look for the next keyword   */
         if( string(1:9) .eq. 'DIHEDRALS' ) exit

!        /*   skip comment lines   */
         if ( string(1:1) .eq. '*' ) cycle
         if ( string(1:1) .eq. '!' ) cycle
         if ( string(1:1) .eq. ' ' ) cycle
         if ( string(1:1) .eq. ' ' ) cycle

!        /*   update counter  */
         nangl_par = nangl_par + 1

!        /*   read atom types(3), force constant, angle    */
!        /*   read urey-bradley force constant, distance   */

         read( string, *, iostat=ierr ) &
     &      symbol_angl(1,nangl_par), &
     &      symbol_angl(2,nangl_par), &
     &      symbol_angl(3,nangl_par), &
     &      fc_angl_par(nangl_par), &
     &      eq_angl_par(nangl_par), &
     &      fc_ub_par(nangl_par), &
     &      eq_ub_par(nangl_par)

         if ( ierr .ne. 0 ) nub_par = nub_par + 1

!        /*   on error, urey-bradley term is zero   */

         if ( ierr .ne. 0 ) then

            read( string, * ) &
     &         symbol_angl(1,nangl_par), &
     &         symbol_angl(2,nangl_par), &
     &         symbol_angl(3,nangl_par), &
     &         fc_angl_par(nangl_par), &
     &         eq_angl_par(nangl_par)

            fc_ub_par(nangl_par) = 0.d0
            eq_ub_par(nangl_par)  = 0.d0

         end if

!     /*   end of loop   */
      end do

!     /*   print   */

      write( 6, '(a,i6)' ) &
     &   '   PAR:  types of angular bonds   = ', nangl_par

      write( 6, '(a,i6)' ) &
     &   '   PAR:  types of urey-bradley    = ', nub_par

!-----------------------------------------------------------------------
!
!     ===  dihedral bonds  ===
!
!        term:
!
!           v(dih) = v_dih ( 1 + cos( nu_dih x - delta_dih ) )
!
!        parameters:
!
!           v_dih:      kcal/mol
!           nu_dih:     multiplicity
!           delta_dih:  degrees
!
!        format:
!
!           atom types(4), v_dih, nu_dih, delta_dih
!
!-----------------------------------------------------------------------

!     /*   zero clear   */
      ndih_par = 0

!     /*   start loop   */
      do

!        /*   read line   */
         read ( 1, '(a)' ) string

!        /*   adjust to the left   */
         string = adjustl(string)

!        /*   look for the next keyword   */
         if( string(1:8) .eq. 'IMPROPER' ) exit

!        /*   skip comment lines   */
         if ( string(1:1) .eq. '*' ) cycle
         if ( string(1:1) .eq. '!' ) cycle
         if ( string(1:1) .eq. ' ' ) cycle
         if ( string(1:1) .eq. ' ' ) cycle

!        /*   update counter  */
         ndih_par = ndih_par + 1

!        /*   read atom types(4), height, multiplicity, angle   */

         read( string, * ) &
     &      symbol_dih(1,ndih_par), &
     &      symbol_dih(2,ndih_par), &
     &      symbol_dih(3,ndih_par), &
     &      symbol_dih(4,ndih_par), &
     &      v_dih_par(ndih_par), &
     &      nu_dih_par(ndih_par), &
     &      delta_dih_par(ndih_par)

      end do

!     /*   print   */
      write( 6, '(a,i6)' ) &
     &   '   PAR:  types of dihedral bonds  = ', ndih_par

!-----------------------------------------------------------------------
!
!     ===  improper bonds  ===
!
!        term:
!
!           v(improper) = fc_improper * ( x - eq_improper )**2
!
!        parameters:
!
!           fc_improper:  kcal/mol/rad**2
!           eq_improper:  degrees
!
!        format:
!
!           atom types(4), fc_improper, ignored, eq_improper
!
!        note:
!
!           6th column is ignored.
!
!-----------------------------------------------------------------------

!     /*   zero clear   */
      nimproper_par = 0

!     /*   start loop   */
      do

!        /*   read line   */
         read( 1, '(a)' ) string

!        /*   adjust to the left   */
         string = adjustl(string)

!        /*   look for the next keyword   */
         if ( string(1:7) .eq. 'NONBOND' ) exit

!        /*   skip comment lines   */
         if ( string(1:1) .eq. '*' ) cycle
         if ( string(1:1) .eq. '!' ) cycle
         if ( string(1:1) .eq. ' ' ) cycle
         if ( string(1:1) .eq. ' ' ) cycle

!        /*   update counter  */
         nimproper_par = nimproper_par + 1

!        /*   read atom types(4), height, ignored, angle   */

         read ( string, * ) &
     &      symbol_improper(1,nimproper_par), &
     &      symbol_improper(2,nimproper_par), &
     &      symbol_improper(3,nimproper_par), &
     &      symbol_improper(4,nimproper_par), &
     &      fc_improper_par(nimproper_par), &
     &      char_dummy, &
     &      eq_improper_par(nimproper_par)

!     /*   end of loop   */
      end do

!     /*   print   */
      write( 6, '(a,i6)' ) &
     &   '   PAR:  types of improper bonds  = ', nimproper_par

!-----------------------------------------------------------------------
!
!     ===  lennard-jones  ===
!
!        term:
!
!           v(lj) = eps(i,j) * ( (rmh(i,j)/x)**12 - 2*(rmh(i,j)/x)**6 )
!
!        parameters:
!
!           eps:  kcal/mol,  eps(i,j) = sqrt(eps(i) * eps(j) )
!           rmh:  A,         rmh(i,j) = ( rmh(i) + rmh(j) ) / 2
!
!        format:
!
!           atom, ignored, eps, Rmin/2, ignored, eps(1-4), Rmin/2(1-4)
!
!        note:
!
!           2nd column is ignored.
!           5th column is ignored.
!
!-----------------------------------------------------------------------

!     /*   zero clear   */

      nlj_par   = 0
      nlj14_par = 0

      do

!        /*   read line   */
         read( 1, '(a)' ) string

!        /*   adjust to the left   */
         string = adjustl(string)

!        /*   look for the next keyword   */
         if( string(1:5) .eq. 'cutnb' ) cycle
         if( string(1:5) .eq. 'HBOND' ) exit

!        /*   skip comment lines   */
         if ( string(1:1) .eq. '*' ) cycle
         if ( string(1:1) .eq. '!' ) cycle
         if ( string(1:1) .eq. ' ' ) cycle
         if ( string(1:1) .eq. ' ' ) cycle

!        /*   update counter  */
         nlj_par = nlj_par + 1

!        /*   read atom types(4), height, ignored, angle   */

         read( string, *, iostat=ierr ) &
     &      symbol_lj(nlj_par), &
     &      char_dummy, &
     &      eps_lj_par(nlj_par), &
     &      rmh_lj_par(nlj_par), &
     &      char_dummy, &
     &      eps_lj14_par(nlj_par), &
     &      rmh_lj14_par(nlj_par)

!        /*   update counter  */
         if ( ierr .eq. 0 ) nlj14_par = nlj14_par + 1

!        /*   on error, parameters for 1-4 term are the same   */

         if ( ierr .ne. 0 ) then

            read ( string, * ) &
     &         symbol_lj(nlj_par), &
     &         char_dummy, &
     &         eps_lj_par(nlj_par), &
     &         rmh_lj_par(nlj_par)

            eps_lj14_par(nlj_par) =  eps_lj_par(nlj_par)
            rmh_lj14_par(nlj_par) =  rmh_lj_par(nlj_par)

         end if

      end do

!     /*   print   */

      write( 6, '(a,i6)' ) &
     &   '   PAR:  types of lennard-jones   = ', nlj_par

      write( 6, '(a,i6)' ) &
     &   '   PAR:  types of 1-4 lj          = ', nlj14_par

!-----------------------------------------------------------------------
!     /*   close charmm parameter file                                */
!-----------------------------------------------------------------------

      close( 1 )

      return
      end





!***********************************************************************
      subroutine read_charmm_psf
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */

      use charmm_variables, only : &
     &   symbol_resi, symbol_atom, symbol_type, physmass_amu, charge, &
     &   nlin, nangl, ndih, nimproper, nlj, nbond, nlj14, nub, &
     &   natom, psf_file

!     /*   local variables   */

      implicit none

      integer :: i, ntitle

      character(len=20) :: char_dummy

!-----------------------------------------------------------------------
!     /*   open charmm parameter file                                 */
!-----------------------------------------------------------------------

      open( 1, file = psf_file, status = 'old' )

!-----------------------------------------------------------------------
!     /*   skip headers                                               */
!-----------------------------------------------------------------------

!     /*   skip lines   */

      read ( 1, * )
      read ( 1, * )

!     /*   number of title lines   */
      read ( 1, * ) ntitle

!     /*   skip lines   */

      do i = 1, ntitle
         read ( 1, * )
      end do

      read ( 1, * )

!-----------------------------------------------------------------------
!     /*   number of atoms                                            */
!-----------------------------------------------------------------------

      read ( 1, * ) natom

!-----------------------------------------------------------------------
!     /*    memory allocation                                         */
!-----------------------------------------------------------------------

      allocate( charge(natom)    )
      allocate( symbol_resi(natom) )
      allocate( symbol_atom(natom)  )
      allocate( symbol_type(natom)  )
      allocate( physmass_amu(natom) )

!-----------------------------------------------------------------------
!     /*    read information of atoms                                 */
!-----------------------------------------------------------------------

      do i = 1 , natom

        read( 1, * ) &
     &    char_dummy,    &   !  ignored (atom ID) &
     &    char_dummy,     &  !  ignored (segment name) &
     &    char_dummy,      & !  ignored (residue ID) &
     &    symbol_resi(i),   &!  residue name &
     &    symbol_atom(i),   &!  atom name &
     &    symbol_type(i),   &!  atom type &
     &    charge(i),        &!  charge &
     &    physmass_amu(i),  &!  atomic mass &
     &    char_dummy        !  ignored (unused 0)

      end do

!-----------------------------------------------------------------------
!     /*    this is to fix charmm bug                                 */
!-----------------------------------------------------------------------

      do i = 1 , natom

          if( ( symbol_resi(i) .eq. 'CYS' ) .and. &
     &        ( symbol_atom(i) .eq. 'HG'  ) ) then

             symbol_atom(i) = 'HG1'

          end if

          if( ( symbol_resi(i) .eq. 'SER' ) .and. &
     &        ( symbol_atom(i) .eq. 'HG'  ) ) then

             symbol_atom(i) = 'HG1'

          end if

          if( symbol_resi(i) .eq. 'ILE' ) then

             if( symbol_atom(i) .eq. 'CD1'  ) symbol_atom(i) = 'CD'
             if( symbol_atom(i) .eq. 'HD11' ) symbol_atom(i) = 'HD1'
             if( symbol_atom(i) .eq. 'HD12' ) symbol_atom(i) = 'HD2'
             if( symbol_atom(i) .eq. 'HD13' ) symbol_atom(i) = 'HD3'

          end if

      end do

!-----------------------------------------------------------------------
!     /*   bonds                                                      */
!-----------------------------------------------------------------------

      call makelist_bond

!-----------------------------------------------------------------------
!     /*   make a list of 1-2 bonded pairs                            */
!-----------------------------------------------------------------------

      call makelist_12

!-----------------------------------------------------------------------
!     /*   angular bonds                                              */
!-----------------------------------------------------------------------

      call makelist_angl

!-----------------------------------------------------------------------
!     /*   make a list of 1-3 bonded pairs                            */
!-----------------------------------------------------------------------

      call makelist_13

!-----------------------------------------------------------------------
!     /*   linear bonds and                                           */
!-----------------------------------------------------------------------

      nlin = nbond + nub

!-----------------------------------------------------------------------
!     /*   dihedral bonds                                             */
!-----------------------------------------------------------------------

      call makelist_dih

!-----------------------------------------------------------------------
!     /*   make a list of 1-4 bonded pairs                            */
!-----------------------------------------------------------------------

      call makelist_14

!-----------------------------------------------------------------------
!     /*   improper bonds                                             */
!-----------------------------------------------------------------------

      call makelist_improper

!-----------------------------------------------------------------------
!     /*   lennard-jones                                              */
!-----------------------------------------------------------------------

      call makelist_lj

!-----------------------------------------------------------------------
!     /*   lennard-jones with 1-4 pairs                               */
!-----------------------------------------------------------------------

      call makelist_lj14

!-----------------------------------------------------------------------
!     /*   bonded charge pairs                                        */
!-----------------------------------------------------------------------

      call makelist_nbcp

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      write( 6, * )
      write( 6, '(a,i6)' ) &
     &   '   PSF:  number of atoms          = ', natom
      write( 6, '(a,i6)' ) &
     &   '   PSF:  number of bonds          = ', nbond
      write( 6, '(a,i6)' ) &
     &   '   PSF:  number of urey-bradley   = ', nub
      write( 6, '(a,i6)' ) &
     &   '   PSF:  number of linear bonds   = ', nlin
      write( 6, '(a,i6)' ) &
     &   '   PSF:  number of angular bonds  = ', nangl
      write( 6, '(a,i6)' ) &
     &   '   PSF:  number of dihedral bonds = ', ndih
      write( 6, '(a,i6)' ) &
     &   '   PSF:  number of improper bonds = ', nimproper
      write( 6, '(a,i6)' ) &
     &   '   PSF:  number of lennard-jones  = ', nlj
      write( 6, '(a,i6)' ) &
     &   '   PSF:  number of 1-4 lj         = ', nlj14

!-----------------------------------------------------------------------
!     /*   close charmm parameter file                                */
!-----------------------------------------------------------------------

      close(1)

      return
      end





!***********************************************************************
      subroutine makelist_12
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */

      use charmm_variables, only : &
     &   nbond, nlist_12, list_12, natom, i_bond, j_bond

!     /*   local variables   */

      implicit none

      integer :: i, j, k, nmax

!-----------------------------------------------------------------------
!     /*   make coordination number of each atom                      */
!-----------------------------------------------------------------------

      allocate( nlist_12(natom) )

      nlist_12(:) = 0

      do k = 1, nbond

         i = i_bond(k)
         j = j_bond(k)

         nlist_12(i) = nlist_12(i) + 1
         nlist_12(j) = nlist_12(j) + 1

      end do

!-----------------------------------------------------------------------
!     /*   calculate maximum coordination number                      */
!-----------------------------------------------------------------------

      nmax = 0

      do i = 1, natom
         nmax = max( nlist_12(i), nmax )
      end do

!-----------------------------------------------------------------------
!     /*   make list of coordination                                  */
!-----------------------------------------------------------------------

      allocate( list_12(natom,nmax) )

      list_12(:,:) = 0

      nlist_12(:)   = 0

      do k = 1, nbond

         i = i_bond(k)
         j = j_bond(k)

         nlist_12(i) = nlist_12(i) + 1
         nlist_12(j) = nlist_12(j) + 1

         list_12(i,nlist_12(i)) = j
         list_12(j,nlist_12(j)) = i

      end do

      return
      end





!***********************************************************************
      subroutine makelist_13
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */

      use charmm_variables, only : &
     &   nangl, i_angl, k_angl, nlist_13, list_13, natom

!     /*   local variables   */

      implicit none

      integer :: i, j, k, nmax

!-----------------------------------------------------------------------
!     /*   make coordination number of each atom                      */
!-----------------------------------------------------------------------

      allocate( nlist_13(natom) )

      nlist_13(:) = 0

      do k = 1, nangl

         i = i_angl(k)
         j = k_angl(k)

         nlist_13(i) = nlist_13(i) + 1
         nlist_13(j) = nlist_13(j) + 1

      end do

!-----------------------------------------------------------------------
!     /*   calculate maximum coordination number                      */
!-----------------------------------------------------------------------

      nmax = 0

      do i = 1, natom
         nmax = max( nlist_13(i), nmax )
      end do

!-----------------------------------------------------------------------
!     /*   make list of coordination                                  */
!-----------------------------------------------------------------------

      allocate( list_13(natom,nmax) )

      list_13(:,:) = 0

      nlist_13(:)   = 0

      do k = 1, nangl

         i = i_angl(k)
         j = k_angl(k)

         nlist_13(i) = nlist_13(i) + 1
         nlist_13(j) = nlist_13(j) + 1

         list_13(i,nlist_13(i)) = j
         list_13(j,nlist_13(j)) = i

      end do

      return
      end





!***********************************************************************
      subroutine makelist_14
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */

      use charmm_variables, only : &
     &   ndih, bonded_dih, nlist_14, list_14, i_dih, l_dih, natom

!     /*   local variables   */

      implicit none

      integer :: i, j, k, nmax

!-----------------------------------------------------------------------
!     /*   make coordination number of each atom                      */
!-----------------------------------------------------------------------

      allocate( nlist_14(natom) )

      nlist_14(:) = 0

      do k = 1, ndih

         if( bonded_dih(k) ) cycle

         i = i_dih(k)
         j = l_dih(k)

         nlist_14(i) = nlist_14(i) + 1
         nlist_14(j) = nlist_14(j) + 1

      end do

!-----------------------------------------------------------------------
!     /*   calculate maximum coordination number                      */
!-----------------------------------------------------------------------

      nmax = 0

      do i = 1, natom
         nmax = max( nlist_14(i), nmax )
      end do

!-----------------------------------------------------------------------
!     /*   make list of coordination                                  */
!-----------------------------------------------------------------------

      allocate( list_14(natom,nmax) )

      nlist_14(:) = 0

      list_14(:,:) = 0

      do k = 1, ndih

         if( bonded_dih(k) ) cycle

         i = i_dih(k)
         j = l_dih(k)

         nlist_14(i) = nlist_14(i) + 1
         nlist_14(j) = nlist_14(j) + 1

         list_14(i,nlist_14(i)) = j
         list_14(j,nlist_14(j)) = i

      end do

      return
      end





!***********************************************************************
      subroutine write_mm_default
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */

      use charmm_variables, only : &
     &   fc_bond, fc_ub, eq_bond, eq_ub, fc_angl, eq_angl, v_dih, &
     &   fc_improper, eq_improper, eps_lj14, eps_lj_pair, &
     &   sig_lj_pair, charge, sig_lj14, rcut_in, rcut_out, &
     &   unit_to_angstrom, unit_to_kcal, unit_to_rad, factor_bcp, &
     &   i_lj_pair, j_lj_pair, i_bcp, j_bcp, i_lj14, j_lj14, &
     &   i_bond, j_bond, i_ub, j_ub, i_angl, j_angl, k_angl, &
     &   i_dih, j_dih, k_dih, l_dih, mu_dih, nu_dih, i_improper, &
     &   j_improper, k_improper, l_improper, nimproper_nonzero, &
     &   nbond_nonzero, nub_nonzero, nlin_nonzero, nangl_nonzero, &
     &   nbond, nangl, nub, ndih, nimproper, ndih_nonzero, nlj, &
     &   nlj_nonzero, nlj14_nonzero, ncharge, nlj14, nbcp, natom

!     /*   local variables   */

      implicit none

      integer :: i, k

!-----------------------------------------------------------------------
!     /*   file open                                                  */
!-----------------------------------------------------------------------

      open ( 1, file = 'mm.dat' )

!-----------------------------------------------------------------------
!     /*   linear bonds                                               */
!-----------------------------------------------------------------------

      i = 0

      do k = 1, nbond

         if ( fc_bond(k) .eq. 0 ) cycle

         i = i + 1

      end do

      nbond_nonzero = i

!-----------------------------------------------------------------------

      i = 0

      do k = 1, nub

         if ( fc_ub(k) .eq. 0 ) cycle

         i = i + 1

      end do

      nub_nonzero = i

      nlin_nonzero = nbond_nonzero + nub_nonzero

!-----------------------------------------------------------------------

      write( 1, '(a)'   ) '<linear_bonds>'

      write( 1, '(i8)' )  nlin_nonzero

      do k = 1, nbond

         if ( fc_bond(k) .eq. 0.d0 ) cycle

         fc_bond(k) = fc_bond(k) * 2.d0

         eq_bond(k) = eq_bond(k) / unit_to_angstrom

         fc_bond(k) = fc_bond(k) / unit_to_kcal * unit_to_angstrom**2

         write( 1, '(2i8,2e24.16)' ) &
     &      i_bond(k), j_bond(k), eq_bond(k), fc_bond(k)

      end do

      do k = 1, nub

         if ( fc_ub(k) .eq. 0.d0 ) cycle

         fc_ub(k) = fc_ub(k) * 2.d0

         eq_ub(k) = eq_ub(k) / unit_to_angstrom

         fc_ub(k) = fc_ub(k) / unit_to_kcal * unit_to_angstrom**2

         write( 1, '(2i8,2e24.16)' ) &
     &      i_ub(k), j_ub(k), eq_ub(k), fc_ub(k)

      end do

!-----------------------------------------------------------------------
!     /*   general form of linear bonds                               */
!-----------------------------------------------------------------------

      write( 1, *      )
      write( 1, '(a)'  ) '<genlin_bonds>'

      write( 1, '(i8)' )  0

!-----------------------------------------------------------------------
!     /*   angular bonds                                              */
!-----------------------------------------------------------------------

      i = 0

      do k = 1, nangl

         if ( fc_angl(k) .eq. 0 ) cycle

         i = i + 1

      end do

      nangl_nonzero = i

!-----------------------------------------------------------------------

      write( 1, *      )
      write( 1, '(a)'  ) '<angular_bonds>'

      write( 1, '(i8)' ) nangl_nonzero

      do k = 1, nangl

         if ( fc_angl(k) .eq. 0 ) cycle

         fc_angl(k) = fc_angl(k) * 2.d0

         eq_angl(k) = eq_angl(k)

         fc_angl(k) = fc_angl(k) / unit_to_kcal / unit_to_rad**2

         write( 1, '(3i8,2e24.16)' ) &
     &      i_angl(k), j_angl(k), k_angl(k), eq_angl(k), fc_angl(k)

      end do

!-----------------------------------------------------------------------
!     /*   dihedral bonds                                             */
!-----------------------------------------------------------------------

      i = 0

      do k = 1, ndih

         if ( v_dih(k) .eq. 0 ) cycle

         i = i + 1

      end do

      ndih_nonzero = i

!-----------------------------------------------------------------------

      write( 1, *      )
      write( 1, '(a)'  ) '<dihedral_bonds>'

      write( 1, '(i8)' ) ndih_nonzero

      do k = 1, ndih

         if ( v_dih(k) .eq. 0 ) cycle

         v_dih(k) = v_dih(k) * 2.d0

         v_dih(k) = v_dih(k) / unit_to_kcal

         write( 1, '(4i8,e24.16,2i8)' ) &
     &      i_dih(k), j_dih(k), k_dih(k), l_dih(k), &
     &      v_dih(k), nu_dih(k), mu_dih(k)

      end do

!-----------------------------------------------------------------------
!     /*   improper bonds                                             */
!-----------------------------------------------------------------------

      i = 0

      do k = 1, nimproper

         if ( fc_improper(k) .eq. 0 ) cycle

         i = i + 1

      end do

      nimproper_nonzero = i

!-----------------------------------------------------------------------

      write( 1, *      )
      write( 1, '(a)'  ) '<improper_bonds>'

      write( 1, '(i8)' )  nimproper_nonzero

      do k = 1, nimproper

         if ( fc_improper(k) .eq. 0 ) cycle

         fc_improper(k) = fc_improper(k) * 2.d0

         eq_improper(k) = eq_improper(k)

         fc_improper(k) = fc_improper(k) / unit_to_kcal / unit_to_rad**2

         write( 1 , '(4i8,2e24.16)' ) &
     &      i_improper(k), j_improper(k), k_improper(k), l_improper(k), &
     &      eq_improper(k), fc_improper(k)

      end do

!-----------------------------------------------------------------------
!     /*   lennard-jones                                              */
!-----------------------------------------------------------------------

      i = 0

      do k = 1, nlj

         if ( eps_lj_pair(k) .eq. 0 ) cycle

         i = i + 1

      end do

      nlj_nonzero = i

!-----------------------------------------------------------------------

      i = 0

      do k = 1, nlj14

         if ( eps_lj14(k) .eq. 0 ) cycle

         i = i + 1

      end do

      nlj14_nonzero = i

!-----------------------------------------------------------------------

      write( 1, *      )
      write( 1, '(a)'  ) '<lennard-jones>'

      write( 1, '(i8)' ) nlj_nonzero + nlj14_nonzero

      write( 1, '(2e24.16)' ) rcut_in, rcut_out

      do k = 1, nlj

         if ( eps_lj_pair(k) .eq. 0 ) cycle

         eps_lj_pair(k) = eps_lj_pair(k) / unit_to_kcal

         sig_lj_pair(k) = sig_lj_pair(k) / unit_to_angstrom

         write( 1, '(2i8,2e24.16)' ) &
     &      i_lj_pair(k), j_lj_pair(k), eps_lj_pair(k), sig_lj_pair(k)

      end do

      do k = 1, nlj14

         if ( eps_lj14(k) .eq. 0 ) cycle

         eps_lj14(k) = eps_lj14(k) / unit_to_kcal

         sig_lj14(k) = sig_lj14(k) / unit_to_angstrom

         write( 1, '(2i8,2e24.16)' ) &
     &      i_lj14(k), j_lj14(k), eps_lj14(k), sig_lj14(k)

      end do

!-----------------------------------------------------------------------
!     /*   atomic charges                                             */
!-----------------------------------------------------------------------

      i = 0

      do k = 1, natom

         if ( charge(k) .eq. 0 ) cycle

         i = i + 1

      end do

      ncharge = i

!-----------------------------------------------------------------------

      write( 1, *      )
      write( 1, '(a)'  ) '<charges>'

      write( 1, '(i8)' )  ncharge

      do k = 1, natom

         if ( charge(k) .eq. 0 ) cycle

         write( 1, '(i8,2e24.16)' ) k, charge(k)

      end do

!-----------------------------------------------------------------------
!     /*   bonded charge pairs                                        */
!-----------------------------------------------------------------------

      write( 1, *      )
      write( 1, '(a)'  ) '<nbcp>'

      write( 1, '(i8)' ) nbcp

      do k = 1, nbcp

         write( 1, '(2i8,e24.16)' ) i_bcp(k), j_bcp(k), factor_bcp(k)

      end do

!-----------------------------------------------------------------------
!     /*   file close                                                 */
!-----------------------------------------------------------------------

      close( 1 )

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      write( 6, * )
      write( 6, '(a,i6)' ) &
     &   '    MM:  number of bonds          = ', nbond_nonzero
      write( 6, '(a,i6)' ) &
     &   '    MM:  number of urey-bradley   = ', nub_nonzero
      write( 6, '(a,i6)' ) &
     &   '    MM:  number of linear bonds   = ', nlin_nonzero
      write( 6, '(a,i6)' ) &
     &   '    MM:  number of angular bonds  = ', nangl_nonzero
      write( 6, '(a,i6)' ) &
     &   '    MM:  number of dihedral bonds = ', ndih_nonzero
      write( 6, '(a,i6)' ) &
     &   '    MM:  number of improper bonds = ', nimproper_nonzero
      write( 6, '(a,i6)' ) &
     &   '    MM:  number of lennard-jones  = ', nlj_nonzero
      write( 6, '(a,i6)' ) &
     &   '    MM:  number of 1-4 lj         = ', nlj14_nonzero
      write( 6, '(a,i6)' ) &
     &   '    MM:  number of charged atoms  = ', ncharge
      write( 6, '(a,i6)' ) &
     &   '    MM:  number of bonded charges = ', nbcp

      return
      end





!***********************************************************************
      subroutine makelist_bond
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */

      use charmm_variables, only : &
     &   eq_bond, fc_bond, symbol_bond, eq_bond_par, fc_bond_par, &
     &   i_bond, j_bond, nbond, symbol_type, nbond_par

!     /*   local variables   */

      implicit none

      integer :: i, j

      character(len=4), dimension(4) :: symbol_b

!-----------------------------------------------------------------------
!     /*   linear bonds                                               */
!-----------------------------------------------------------------------

!     /*   skip a line   */
      read ( 1, * )

!     /*   read number of linear bonds   */
      read ( 1, * ) nbond

!     /*   only if linear bond exists   */

      if ( nbond .ne. 0 ) then

!        /*   memory allocation   */

         allocate( i_bond(nbond) )
         allocate( j_bond(nbond) )
         allocate( fc_bond(nbond) )
         allocate( eq_bond(nbond) )

!        /*   read bonded linear atoms   */

         read( 1, * ) ( i_bond(i), j_bond(i), i = 1, nbond )

!        /*   identify bond parameters   */

         do i = 1, nbond     !  combinations of atoms

!           /*   set atom type   */

            symbol_b(1) = symbol_type(i_bond(i))
            symbol_b(2) = symbol_type(j_bond(i))

!           /*   search for bond parameters   */

            do j = 1, nbond_par     !  combinations of symbols

               if ( ( ( symbol_bond(1,j) .eq. symbol_b(1) ) &
     &          .and. ( symbol_bond(2,j) .eq. symbol_b(2) ) ) .or. &
     &              ( ( symbol_bond(1,j) .eq. symbol_b(2) ) &
     &          .and. ( symbol_bond(2,j) .eq. symbol_b(1) ) ) ) then

!                 /*   bond parameter identified   */

                  fc_bond(i) = fc_bond_par(j)
                  eq_bond(i) = eq_bond_par(j)

                  exit

               end if

!              /*   if not found, print error information   */

               if( j .eq. nbond_par ) then

                  write( 6, '(a)' ) &
     &               'ERROR, in read_charmm_psf: linear bond'
                  write( 6, '(a,2a6)' ) &
     &               '       type = ', symbol_b(1), symbol_b(2)
                  write( 6, '(a,2i6)' ) &
     &               '       atom = ', i_bond(i), j_bond(i)

               end if

            end do

         end do

      end if

      return
      end





!***********************************************************************
      subroutine makelist_angl
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */

      use charmm_variables, only : &
     &   eq_angl, fc_angl, symbol_type, eq_angl_par, fc_angl_par, &
     &   eq_ub_par, fc_ub_par, eq_ub, fc_ub, i_ub, j_ub, &
     &   i_angl, j_angl, k_angl, nangl, nub, symbol_angl, nangl_par

!     /*   local variables   */

      implicit none

      integer :: i, j, k

      character(len=4), dimension(4) :: symbol_b

!-----------------------------------------------------------------------
!     /*   angular bonds                                              */
!-----------------------------------------------------------------------

!     /*   skip a line   */
      read ( 1, * )

!     /*   read number of angular bonds   */

      read ( 1, * ) nangl

!     /*   only if angular bond exists   */

      if ( nangl .ne. 0 ) then

!        /*   memory allocation   */

         allocate( i_angl(nangl) )
         allocate( j_angl(nangl) )
         allocate( k_angl(nangl) )

         allocate( fc_angl(nangl) )
         allocate( eq_angl(nangl) )

!        /*   read bonded atoms   */

         read( 1, * ) &
     &      ( i_angl(i), j_angl(i), k_angl(i), i = 1, nangl )

!-----------------------------------------------------------------------
!        /*   identify angular bond parameters                        */
!-----------------------------------------------------------------------

         nub = 0

         do i = 1, nangl     !  combinations of atoms

!           /*   set atom type   */

            symbol_b(1) = symbol_type(i_angl(i))
            symbol_b(2) = symbol_type(j_angl(i))
            symbol_b(3) = symbol_type(k_angl(i))

!           /*   search for bond parameters   */

            do j = 1 , nangl_par     !  combinations of symbols

!              /*  if found   */

               if( symbol_angl(2,j) .eq. symbol_b(2) ) then

                  if( ( ( symbol_angl(1,j) .eq. symbol_b(1) ) .and. &
     &                  ( symbol_angl(3,j) .eq. symbol_b(3) ) ) .or. &
     &                ( ( symbol_angl(1,j) .eq. symbol_b(3) ) .and. &
     &                  ( symbol_angl(3,j) .eq. symbol_b(1) ) ) ) then

!                    /*   bond parameter identified   */

                     fc_angl(i) = fc_angl_par(j)
                     eq_angl(i) = eq_angl_par(j)

!                    /*   count number of urey-bradley pairs   */

                     if( fc_ub_par(j) .ne. 0.d0 ) then

                        nub = nub + 1

                     end if

                     exit

                  end if

               end if

!              /*   if not found, print error information   */

               if( j .eq. nangl_par ) then

                  write( 6, '(a)' ) &
     &               'ERROR, in read_charmm_psf: angular bond'

                  write( 6, '(a,3a6)' ) &
     &               '       type = ', symbol_b(1), symbol_b(2), &
     &                                 symbol_b(3)
                  write( 6, '(a,3i6)' ) &
     &               '       atom = ', i_angl(i), j_angl(i), k_angl(i)

               end if

            end do

         end do

!-----------------------------------------------------------------------
!        /*   identify urey-bradley parameters                        */
!-----------------------------------------------------------------------

!        /*   memory allocation   */

         allocate( i_ub(nub)  )
         allocate( j_ub(nub)  )
         allocate( fc_ub(nub) )
         allocate( eq_ub(nub) )

!        /*   count number of urey-bradley parameters   */

         k = 0

!        /*   identify angular bond parameters   */

         do i = 1, nangl     !  combinations of atoms

!           /*   set atom type   */

            symbol_b(1) = symbol_type(i_angl(i))
            symbol_b(2) = symbol_type(j_angl(i))
            symbol_b(3) = symbol_type(k_angl(i))

!           /*   search for bond parameters   */

            do j = 1 , nangl_par     !  combinations of symbols

!              /*  if found   */

               if( symbol_angl(2,j) .eq. symbol_b(2) ) then

                  if( ( ( symbol_angl(1,j) .eq. symbol_b(1) ) .and. &
     &                  ( symbol_angl(3,j) .eq. symbol_b(3) ) ) .or. &
     &                ( ( symbol_angl(1,j) .eq. symbol_b(3) ) .and. &
     &                  ( symbol_angl(3,j) .eq. symbol_b(1) ) ) ) then

!                    /*   make a list of urey-bradley pairs   */

                     if( fc_ub_par(j) .ne. 0.d0 ) then

                        k = k + 1

                        i_ub(k) = i_angl(i)
                        j_ub(k) = k_angl(i)

                        fc_ub(k) = fc_ub_par(j)
                        eq_ub(k) = eq_ub_par(j)

                     end if

                     exit

                  end if

               end if

            end do

         end do

      end if

      return
      end





!***********************************************************************
      subroutine makelist_dih
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */

      use charmm_variables, only : &
     &   v_dih, pi, v_dih_par, delta_dih_par, nu_dih_par, ndih_par, &
     &   i_angl, j_angl, k_angl,symbol_dih, i_dih, j_dih, k_dih, l_dih, &
     &   mu_dih, nu_dih, ndih, symbol_type, nbond, nangl, bonded_dih

!     /*   local variables   */

      implicit none

      integer :: i, j, k, l, i1, i2, j1, j2, ntmp

      integer, dimension(:), allocatable :: i_tmp, j_tmp, k_tmp, l_tmp

      character(len=4), dimension(4) :: symbol_b, symbol_a

!-----------------------------------------------------------------------
!     /*   dihedral bonds part one                                    */
!-----------------------------------------------------------------------

!     /*   skip a line   */
      read ( 1, * )

!     /*   read number of dihedral bonds   */
      read ( 1, * ) ntmp

!     /*   only if dihedral bond exists   */
      if ( ntmp .ne. 0 ) then

!        /*   reset counter   */
         l = 0

!        /*   memory allocation   */

         allocate( i_tmp(ntmp) )
         allocate( j_tmp(ntmp) )
         allocate( k_tmp(ntmp) )
         allocate( l_tmp(ntmp) )

!        /*   read bonded atoms   */

         read( 1, * ) ( i_tmp(i), j_tmp(i), k_tmp(i), l_tmp(i), &
     &                  i = 1, ntmp )

!        /*   identify dihedral bond parameters   */

         do i = 1, ntmp     !  combinations of atoms

!           /*   set atom type   */

            symbol_b(1) = symbol_type(i_tmp(i))
            symbol_b(2) = symbol_type(j_tmp(i))
            symbol_b(3) = symbol_type(k_tmp(i))
            symbol_b(4) = symbol_type(l_tmp(i))

!           /*   reset counter   */
            k = 0

!           /*   match 1-2-3-4 atoms   */

            do j = 1, ndih_par     !  combinations of symbols

               symbol_a(1) = symbol_dih(1,j)
               symbol_a(2) = symbol_dih(2,j)
               symbol_a(3) = symbol_dih(3,j)
               symbol_a(4) = symbol_dih(4,j)

               if     ( ( symbol_a(2) .eq. symbol_b(2) ) .and. &
     &                  ( symbol_a(3) .eq. symbol_b(3) ) .and. &
     &                  ( symbol_a(1) .eq. symbol_b(1) ) .and. &
     &                  ( symbol_a(4) .eq. symbol_b(4) ) ) then

!                 /*   how many matched   */

                  k = k + 1
                  l = l + 1

               else if( ( symbol_a(2) .eq. symbol_b(3) ) .and. &
     &                  ( symbol_a(3) .eq. symbol_b(2) ) .and. &
     &                  ( symbol_a(1) .eq. symbol_b(4) ) .and. &
     &                  ( symbol_a(4) .eq. symbol_b(1) ) ) then

!                 /*   how many matched   */

                  k = k + 1
                  l = l + 1

               end if

            end do

!           /*   if not found, match 2-3 atoms   */

            if( k .eq. 0 ) then

               do j = 1, ndih_par

                  symbol_a(1) = symbol_dih(1,j)
                  symbol_a(2) = symbol_dih(2,j)
                  symbol_a(3) = symbol_dih(3,j)
                  symbol_a(4) = symbol_dih(4,j)

                  if     ( ( symbol_a(2) .eq. symbol_b(2) ) .and. &
     &                     ( symbol_a(3) .eq. symbol_b(3) ) .and. &
     &                     ( symbol_a(1) .eq. 'X'         ) .and. &
     &                     ( symbol_a(4) .eq. 'X'         ) ) then

!                    /*   how many matched   */

                     k = k + 1
                     l = l + 1

                  else if( ( symbol_a(2) .eq. symbol_b(3) ) .and. &
     &                     ( symbol_a(3) .eq. symbol_b(2) ) .and. &
     &                     ( symbol_a(1) .eq. 'X'         ) .and. &
     &                     ( symbol_a(4) .eq. 'X'         ) ) then

!                    /*   how many matched   */

                     k = k + 1
                     l = l + 1

                  end if

               end do

            end if

!           /*   if not found, error   */

            if( k .eq. 0 ) then

               write( 6, '(a)' ) &
     &            'ERROR, in read_charmm_psf: dihedral bond'

               write( 6, '(a,4a6)' ) &
     &            '       type = ', symbol_b(1), symbol_b(2), &
     &                              symbol_b(3), symbol_b(4)

            end if

         end do

      end if

!-----------------------------------------------------------------------
!     /*   dihedral bonds part two                                    */
!-----------------------------------------------------------------------

!     /*   only if dihedral bond exists   */
      if ( ntmp .ne. 0 ) then

!        /*   number of dihedral bonds   */
         ndih = l

!        /*   memory allocation   */

         allocate( i_dih(ndih)  )
         allocate( j_dih(ndih)  )
         allocate( k_dih(ndih)  )
         allocate( l_dih(ndih)  )

         allocate( v_dih(ndih)  )
         allocate( mu_dih(ndih) )
         allocate( nu_dih(ndih) )

!        /*   reset counter   */

         l = 0

!        /*   identify dihedral bond parameters   */

         do i = 1, ntmp     !  combinations of atoms

!           /*   set atom type   */

            symbol_b(1) = symbol_type(i_tmp(i))
            symbol_b(2) = symbol_type(j_tmp(i))
            symbol_b(3) = symbol_type(k_tmp(i))
            symbol_b(4) = symbol_type(l_tmp(i))

!           /*   reset counter   */

            k = 0

            do j = 1, ndih_par     !  combinations of symbols

               symbol_a(1) = symbol_dih(1,j)
               symbol_a(2) = symbol_dih(2,j)
               symbol_a(3) = symbol_dih(3,j)
               symbol_a(4) = symbol_dih(4,j)

               if     ( ( symbol_a(2) .eq. symbol_b(2) ) .and. &
     &                  ( symbol_a(3) .eq. symbol_b(3) ) .and. &
     &                  ( symbol_a(1) .eq. symbol_b(1) ) .and. &
     &                  ( symbol_a(4) .eq. symbol_b(4) ) ) then

!                 /*   how many matched   */

                  k = k + 1
                  l = l + 1

!                 /*   substitution   */

                  i_dih(l)  = i_tmp(i)
                  j_dih(l)  = j_tmp(i)
                  k_dih(l)  = k_tmp(i)
                  l_dih(l)  = l_tmp(i)

                  v_dih(l)  = v_dih_par(j)
                  mu_dih(l) = nint( cos(delta_dih_par(j)/180.d0*pi) )
                  nu_dih(l) = nu_dih_par(j)

               else if( ( symbol_a(2) .eq. symbol_b(3) ) .and. &
     &                  ( symbol_a(3) .eq. symbol_b(2) ) .and. &
     &                  ( symbol_a(1) .eq. symbol_b(4) ) .and. &
     &                  ( symbol_a(4) .eq. symbol_b(1) ) ) then

!                /*   how many matched   */

                  k = k + 1
                  l = l + 1

!                 /*   substitution   */

                  i_dih(l)  = i_tmp(i)
                  j_dih(l)  = j_tmp(i)
                  k_dih(l)  = k_tmp(i)
                  l_dih(l)  = l_tmp(i)

                  v_dih(l)  = v_dih_par(j)
                  mu_dih(l) = nint( cos(delta_dih_par(j)/180.d0*pi) )
                  nu_dih(l) = nu_dih_par(j)

               end if

            end do

            if( k .eq. 0 ) then

               do j = 1, ndih_par

                  symbol_a(1) = symbol_dih(1,j)
                  symbol_a(2) = symbol_dih(2,j)
                  symbol_a(3) = symbol_dih(3,j)
                  symbol_a(4) = symbol_dih(4,j)

                  if     ( ( symbol_a(2) .eq. symbol_b(2) ) .and. &
     &                     ( symbol_a(3) .eq. symbol_b(3) ) .and. &
     &                     ( symbol_a(1) .eq. 'X'         ) .and. &
     &                     ( symbol_a(4) .eq. 'X'         ) ) then

!                    /*   how many matched   */

                     k = k + 1
                     l = l + 1

!                    /*   substitution   */

                     i_dih(l)  = i_tmp(i)
                     j_dih(l)  = j_tmp(i)
                     k_dih(l)  = k_tmp(i)
                     l_dih(l)  = l_tmp(i)

                     v_dih(l)  = v_dih_par(j)
                     mu_dih(l) = nint( cos(delta_dih_par(j)/180.d0*pi) )
                     nu_dih(l) = nu_dih_par(j)

                  else if( ( symbol_a(2) .eq. symbol_b(3) ) .and. &
     &                     ( symbol_a(3) .eq. symbol_b(2) ) .and. &
     &                     ( symbol_a(1) .eq. 'X'         ) .and. &
     &                     ( symbol_a(4) .eq. 'X'         ) ) then

!                    /*   how many matched   */

                     k = k + 1
                     l = l + 1

!                    /*   substitution   */

                     i_dih(l)  = i_tmp(i)
                     j_dih(l)  = j_tmp(i)
                     k_dih(l)  = k_tmp(i)
                     l_dih(l)  = l_tmp(i)

                     v_dih(l)  = v_dih_par(j)
                     mu_dih(l) = nint( cos(delta_dih_par(j)/180.d0*pi) )
                     nu_dih(l) = nu_dih_par(j)

                  end if

               end do

            end if

         end do

      end if

!-----------------------------------------------------------------------
!     /*   dihedral bonds part three                                  */
!-----------------------------------------------------------------------

!     /*   only if dihedral bond exists   */
      if ( ndih .ne. 0 ) then

!        /*   memory allocation   */
         allocate( bonded_dih(ndih) )

         do i = 1 , ndih

            i1 = i_dih(i)
            i2 = l_dih(i)

            do j = 1 , nbond

               j1 = i_angl(j)
               j2 = j_angl(j)

               if( ( ( i1 .eq. j1 ) .and. (i2 .eq. j2 ) ) .or. &
     &             ( ( i1 .eq. j2 ) .and. (i2 .eq. j1 ) ) ) then

                  bonded_dih(i) = .true.

                  exit

               end if

            end do

            do j = 1 , nangl

               j1 = i_angl(j)
               j2 = k_angl(j)

               if( ( ( i1 .eq. j1 ) .and. (i2 .eq. j2 ) ) .or. &
     &             ( ( i1 .eq. j2 ) .and. (i2 .eq. j1 ) ) ) then

                  bonded_dih(i) = .true.

                  exit

               end if

            end do

            do j = i+1, ndih

               if( j .eq. i ) cycle

               j1 = i_dih(j)
               j2 = l_dih(j)

               if( ( ( i1 .eq. j1 ) .and. (i2 .eq. j2 ) ) .or. &
     &             ( ( i1 .eq. j2 ) .and. (i2 .eq. j1 ) ) ) then

                  bonded_dih(i) = .true.

                  exit

               end if

            end do

         end do

!        /*   memory allocation   */

         deallocate( i_tmp )
         deallocate( j_tmp )
         deallocate( k_tmp )
         deallocate( l_tmp )

      end if

      return
      end





!***********************************************************************
      subroutine makelist_improper
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */

      use charmm_variables, only : &
     &   eq_improper, fc_improper, symbol_type, fc_improper_par, &
     &   eq_improper_par, symbol_improper, nimproper_par, &
     &   i_improper, j_improper, k_improper, l_improper, nimproper

!     /*   local variables   */

      implicit none

      integer :: i, j, k

      character(len=4), dimension(4) :: symbol_b, symbol_a

!-----------------------------------------------------------------------
!     /*   improper bonds                                             */
!-----------------------------------------------------------------------

!     /*   skip a line   */
      read ( 1, * )

!     /*   read number of improper bonds   */
      read ( 1, * ) nimproper

!     /*   only if improper bond exists   */

      if ( nimproper .ne. 0 ) then

!        /*   memory allocation   */

         allocate( i_improper(nimproper) )
         allocate( j_improper(nimproper) )
         allocate( k_improper(nimproper) )
         allocate( l_improper(nimproper) )

!        /*   read bonded atoms   */

         read ( 1, * ) ( i_improper(i), j_improper(i), &
     &                   k_improper(i), l_improper(i), &
     &                   i = 1, nimproper )

!        /*   memory allocation   */

         allocate( fc_improper(nimproper) )
         allocate( eq_improper(nimproper) )

         do i = 1, nimproper     !  combinations of atoms

            symbol_b(1) = symbol_type(i_improper(i))
            symbol_b(2) = symbol_type(j_improper(i))
            symbol_b(3) = symbol_type(k_improper(i))
            symbol_b(4) = symbol_type(l_improper(i))

            do j = 1, nimproper_par     !  combinations of symbols

               symbol_a(1) = symbol_improper(1,j)
               symbol_a(2) = symbol_improper(2,j)
               symbol_a(3) = symbol_improper(3,j)
               symbol_a(4) = symbol_improper(4,j)

               if( ( symbol_a(1) .eq. symbol_b(1) ) .and. &
     &             ( symbol_a(4) .eq. symbol_b(4) ) .and. &
     &             ( symbol_a(2) .eq. symbol_b(2) ) .and. &
     &             ( symbol_a(3) .eq. symbol_b(3) ) ) then

                  fc_improper(i) = fc_improper_par(j)
                  eq_improper(i) = eq_improper_par(j)

                  exit

               else if( ( symbol_a(1) .eq. symbol_b(4) ) .and. &
     &                  ( symbol_a(4) .eq. symbol_b(1) ) .and. &
     &                  ( symbol_a(2) .eq. symbol_b(3) ) .and. &
     &                  ( symbol_a(3) .eq. symbol_b(2) ) ) then

                  fc_improper(i) = fc_improper_par(j)
                  eq_improper(i) = eq_improper_par(j)

                  exit

               end if

               if( j .eq. nimproper_par ) then

                  do k = 1 , nimproper_par

                     symbol_a(1) = symbol_improper(1,k)
                     symbol_a(2) = symbol_improper(2,k)
                     symbol_a(3) = symbol_improper(3,k)
                     symbol_a(4) = symbol_improper(4,k)

                     if( ( symbol_a(1) .eq. symbol_b(1) ) .and. &
     &                   ( symbol_a(4) .eq. symbol_b(4) ) .and. &
     &                   ( symbol_a(2) .eq. 'X'         ) .and. &
     &                   ( symbol_a(3) .eq. 'X'         ) ) then

                         fc_improper(i) = fc_improper_par(k)
                         eq_improper(i) = eq_improper_par(k)

                         exit

                     else if( ( symbol_a(1) .eq. symbol_b(4) ) .and. &
     &                        ( symbol_a(4) .eq. symbol_b(1) ) .and. &
     &                        ( symbol_a(2) .eq. 'X'         ) .and. &
     &                        ( symbol_a(3) .eq. 'X'         ) ) then

                         fc_improper(i) = fc_improper_par(k)
                         eq_improper(i) = eq_improper_par(k)

                         exit

                     end if

                       if( k .eq. nimproper_par ) then

                        write( 6, '(a)' ) &
     &                     'ERROR, in read_charmm_psf: improper bond'

                        write( 6, '(a,4a6)' ) &
     &                     '       type = ', symbol_b(1), symbol_b(2), &
     &                                       symbol_b(3), symbol_b(4)
                        write( 6, '(a,4i6)' ) &
     &               '       atom = ', i_improper(i), j_improper(i), &
     &                                 k_improper(i), l_improper(i)

                     end if

                  end do

               end if

            end do

         end do

      end if

      return
      end





!***********************************************************************
      subroutine makelist_lj
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */

      use charmm_variables, only : &
     &   rmh_lj, eps_lj, rmh_lj_par, eps_lj_par, eps_lj_pair, &
     &   rmh_lj14_par, sig_lj_pair, sqrt_eps_lj, symbol_type, &
     &   eps_lj14_par, sqrt_eps_lj14, rmh_lj14, symbol_lj, &
     &   list_12, list_13, list_14, nlist_14, j_lj_pair, natom, &
     &   nlj, nlj_par, i_lj_pair, nlist_12, nlist_13

!     /*   local variables   */

      implicit none

      integer :: i, j, k, iflag

      real(8) :: const

      character(len=4), dimension(4) :: symbol_b

!-----------------------------------------------------------------------
!     /*   lennard-jones                                              */
!-----------------------------------------------------------------------

      const = 2.d0 ** (-1.d0/6.d0)

!     /*   memory allocation   */

      allocate( rmh_lj(natom)      )
      allocate( eps_lj(natom)      )
      allocate( sqrt_eps_lj(natom) )

      allocate( rmh_lj14(natom)      )
      allocate( sqrt_eps_lj14(natom) )

!     /*  identify lj parameters   */

      do i = 1, natom

!        /*   set atom type   */
         symbol_b(1) = symbol_type(i)

!        /*   search for lj parameters   */

         do j = 1, nlj_par

!           /*   if found   */
            if( symbol_b(1) .eq. symbol_lj(j) ) then

!              /*   take square root of charmm epsilon   */
               sqrt_eps_lj(i) = sqrt( abs( eps_lj_par(j) ) )

!              /*   half of charmm sigma   */
               rmh_lj(i)   = rmh_lj_par(j)

!              /*   take square root of charmm epsilon 1-4   */
               sqrt_eps_lj14(i) = sqrt( abs( eps_lj14_par(j) ) )

!              /*   half of charmm sigma 1-4   */
               rmh_lj14(i) = rmh_lj14_par(j)

               exit

            end if

!           /*   if not found   */

            if( j .eq. nlj_par ) then

               write( 6, '(a)' ) &
     &            'ERROR, in read_charmm_psf: lennard-jones'

               write( 6, '(a,i6)' ) &
     &            '       atom = ', i

            end if

         end do

!     /*  end loop of atoms   */

      end do

!-----------------------------------------------------------------------
!     /*   count number of lj pairs (without 1-4 contribution)        */
!-----------------------------------------------------------------------

      nlj = 0

      do i = 1, natom-1

         do j = i+1, natom

            iflag = 0

            do k = 1, nlist_12(i)
               if ( list_12(i,k) .eq. j ) iflag = 1
            end do

            do k = 1, nlist_13(i)
               if ( list_13(i,k) .eq. j ) iflag = 1
            end do

            do k = 1, nlist_14(i)
               if ( list_14(i,k) .eq. j ) iflag = 1
            end do

            if ( iflag .eq. 1 ) cycle

            nlj = nlj + 1

         end do

         continue

      end do

!-----------------------------------------------------------------------
!     /*   make lj list (without 1-4 pairs)                           */
!-----------------------------------------------------------------------

      allocate( i_lj_pair(nlj) )
      allocate( j_lj_pair(nlj) )
      allocate( eps_lj_pair(nlj) )
      allocate( sig_lj_pair(nlj) )

      nlj = 0

      do i = 1, natom-1

         do j = i+1, natom

            iflag = 0

            do k = 1, nlist_12(i)
               if ( list_12(i,k) .eq. j ) iflag = 1
            end do

            do k = 1, nlist_13(i)
               if ( list_13(i,k) .eq. j ) iflag = 1
            end do

            do k = 1, nlist_14(i)
               if ( list_14(i,k) .eq. j ) iflag = 1
            end do

            if ( iflag .eq. 1 ) cycle

            nlj = nlj + 1

            i_lj_pair(nlj) = i

            j_lj_pair(nlj) = j

            eps_lj_pair(nlj) = sqrt_eps_lj(i) * sqrt_eps_lj(j)

            sig_lj_pair(nlj) = ( rmh_lj(i) + rmh_lj(j) ) * const

         end do

         continue

      end do

      return
      end





!***********************************************************************
      subroutine makelist_lj14
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */

      use charmm_variables, only : &
     &   i_lj14, j_lj14, eps_lj14, sig_lj14, sqrt_eps_lj14, &
     &   rmh_lj14, ndih, bonded_dih, i_dih, l_dih, nlj14

!     /*   local variables   */

      implicit none

      integer :: i, k

      real(8) :: const

!-----------------------------------------------------------------------
!     /*   1-4 pairs                                                  */
!-----------------------------------------------------------------------

      const = 2.d0 ** (-1.d0/6.d0)

      i = 0

      do k = 1, ndih

         if( bonded_dih(k) ) cycle

         i = i + 1

      end do

      nlj14 = i

      allocate( i_lj14(nlj14)   )
      allocate( j_lj14(nlj14)   )
      allocate( eps_lj14(nlj14) )
      allocate( sig_lj14(nlj14) )

      i = 0

      do k = 1, ndih

         if( bonded_dih(k) ) cycle

         i = i + 1

         i_lj14(i) = i_dih(k)
         j_lj14(i) = l_dih(k)

         eps_lj14(i) &
     &      = sqrt_eps_lj14(i_dih(k)) * sqrt_eps_lj14(l_dih(k))

         sig_lj14(i) &
     &      = const * ( rmh_lj14(i_dih(k)) + rmh_lj14(l_dih(k)) )

      end do

      return
      end





!***********************************************************************
      subroutine makelist_nbcp
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */

      use charmm_variables, only : &
     &   list_14, nlist_14, factor_14, factor_bcp, i_bcp, j_bcp, natom, &
     &   list_13, list_12, nlist_12, nlist_13, i_bcp, j_bcp, nbcp

!     /*   local variables   */

      implicit none

      integer :: i, j, k

      real(8) :: factor

!-----------------------------------------------------------------------
!     /*   count the number of bonded charge pairs                    */
!-----------------------------------------------------------------------

      nbcp = 0

      do i = 1, natom-1

         do j = i+1, natom

            factor = 1.d0

            do k = 1, nlist_14(i)
               if ( list_14(i,k) .eq. j ) factor = factor_14
            end do

            do k = 1, nlist_13(i)
               if ( list_13(i,k) .eq. j ) factor = 0.d0
            end do

            do k = 1, nlist_12(i)
               if ( list_12(i,k) .eq. j ) factor = 0.d0
            end do

            if ( factor .ne. 1.d0 ) then

               nbcp = nbcp + 1

            end if

         end do

      end do

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   atom i of bonded charge pair   */
      allocate( i_bcp(nbcp)      )

!     /*   atom j of bonded charge pair   */
      allocate( j_bcp(nbcp)      )

!     /*   scaling factor of bonded charge pair   */
      allocate( factor_bcp(nbcp) )

!-----------------------------------------------------------------------
!     /*   make list of bonded charge pairs                           */
!-----------------------------------------------------------------------

      nbcp = 0

      do i = 1, natom-1

         do j = i+1, natom

            factor = 1.d0

            do k = 1, nlist_14(i)
               if ( list_14(i,k) .eq. j ) factor = factor_14
            end do

            do k = 1, nlist_13(i)
               if ( list_13(i,k) .eq. j ) factor = 0.d0
            end do

            do k = 1, nlist_12(i)
               if ( list_12(i,k) .eq. j ) factor = 0.d0
            end do

            if ( factor .ne. 1.d0 ) then

               nbcp = nbcp + 1

               i_bcp(nbcp) = i

               j_bcp(nbcp) = j

               factor_bcp(nbcp) = factor

            end if

         end do

      end do

      return
      end





!***********************************************************************
      subroutine read_pdb
!***********************************************************************

      use charmm_variables, only : &
     &   x, y, z, symbol_atom_pdb, symbol_resi_pdb, pdb_file, natom_pdb

      implicit none

      integer :: i, ierr

      character(len=80) :: char_line

!-----------------------------------------------------------------------

      open ( 10, file = pdb_file )

      i = 0

      do

         read ( 10, '(a)', iostat=ierr ) char_line

         if ( ierr .ne. 0 ) exit

         if ( char_line(1:6) .eq. 'ATOM  ' ) then

            i = i + 1

         end if

      end do

      close( 10 )

      natom_pdb = i

!-----------------------------------------------------------------------

      allocate( x(natom_pdb) )
      allocate( y(natom_pdb) )
      allocate( z(natom_pdb) )

      allocate( symbol_atom_pdb(natom_pdb) )
      allocate( symbol_resi_pdb(natom_pdb) )

!-----------------------------------------------------------------------

      open ( 10, file =  pdb_file )

      i = 0

      do

         read ( 10, '(a)', iostat=ierr ) char_line

         if ( ierr .ne. 0 ) exit

         if ( char_line(1:6) .eq. 'ATOM  ' ) then

            i = i + 1

            backspace( 10 )

            read ( 10, '(6x,5x,1x,a4,1x,a3,1x,1x,4x,1x,3x,3f8.3)' ) &
     &         symbol_atom_pdb(i), symbol_resi_pdb(i), x(i), y(i), z(i)

            symbol_atom_pdb(i) = adjustl(symbol_atom_pdb(i))
            symbol_resi_pdb(i) = adjustl(symbol_resi_pdb(i))

         end if

      end do

      close( 10 )

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      write( 6, * )
      write( 6, '(a,i6)' ) &
     &   '   PDB:  number of atoms          = ', natom_pdb

      return
      end





!***********************************************************************
      subroutine write_centroid_default
!***********************************************************************

      use charmm_variables, only : &
     &   x, y, z, unit_to_angstrom, symbol_atom_pdb, natom_pdb, &
     &   pdb_file

      implicit none

      integer :: i

!-----------------------------------------------------------------------

      open ( 11, file = 'centroid.xyz' )

      write( 11, '(i6)' ) natom_pdb
      write( 11, '(a)'  ) pdb_file

      do i = 1, natom_pdb

         write( 11, '(a4,1x,3f10.3)' ) &
     &      symbol_atom_pdb(i), x(i), y(i), z(i)

      end do

      close( 11 )

!-----------------------------------------------------------------------

      open ( 12, file = 'centroid.dat' )

      do i = 1, natom_pdb

         x(i) = x(i) / unit_to_angstrom
         y(i) = y(i) / unit_to_angstrom
         z(i) = z(i) / unit_to_angstrom

         write( 12, '(3e24.16)' ) x(i), y(i), z(i)

      end do

      close( 12 )

!-----------------------------------------------------------------------

      return
      end




!***********************************************************************
      subroutine write_input_default
!***********************************************************************

      use charmm_variables, only : &
     &   physmass_amu, symbol_atom_pdb, natom_pdb

      implicit none

      integer :: i

!-----------------------------------------------------------------------

      open ( 13, file = 'input.dat' )

      write( 13, '(a)'  ) '<method>'
      write( 13, '(a)'  ) 'STATIC'
      write( 13, *      )

      write( 13, '(a)'  ) '<natom>'
      write( 13, '(i6)' )  natom_pdb
      write( 13, '(a)'  )

      write( 13, '(a)'  ) '<nspec>'
      write( 13, '(i6)' )  natom_pdb

      do i = 1, natom_pdb
         write( 13, '(a4,1x,f10.4,i2)' ) &
     &      symbol_atom_pdb(i), physmass_amu(i), 1
      end do

      write( 13, *      )
      write( 13, '(a)'  ) '<ipotential>'
      write( 13, '(a)'  ) 'MM'
      write( 13, *      )

      write( 13, '(a)'  ) '<iboundary>'
      write( 13, '(i6)' ) 0
      write( 13, *      )

      close( 13 )

!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine write_mm_renumber
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */

      use charmm_variables, only : &
     &   fc_bond, eq_bond, fc_ub, eq_ub, unit_to_angstrom, v_dih, &
     &   unit_to_kcal, unit_to_rad, factor_bcp, charge, sig_lj14, &
     &   rcut_in, rcut_out, eq_improper, fc_improper, eps_lj_pair, &
     &   sig_lj_pair, fc_angl, eq_angl, eps_lj14, i_angl, j_angl, &
     &   k_angl, i_ub, j_ub, ndih, i_bond, j_bond, i_dih, j_dih, k_dih, &
     &   l_dih, i_improper, j_improper, k_improper, l_improper, natom, &
     &   i_lj_pair, j_lj_pair, i_bcp, j_bcp, nbcp, i_lj14, j_lj14, &
     &   nlj14_nonzero, ndih_nonzero, nimproper, nimproper_nonzero, &
     &   nbond, nub, nbond_nonzero, nub_nonzero, nlin_nonzero, ncharge, &
     &   nangl, nangl_nonzero, nu_dih, mu_dih, nlj, nlj_nonzero, nlj14

!     /*   local variables   */

      implicit none

      integer :: i, k, l, inew, jnew, knew, lnew

!-----------------------------------------------------------------------
!     /*   file open                                                  */
!-----------------------------------------------------------------------

      open ( 1, file = 'mm.dat' )

!-----------------------------------------------------------------------
!     /*   linear bonds                                               */
!-----------------------------------------------------------------------

      i = 0

      do k = 1, nbond

         if ( fc_bond(k) .eq. 0 ) cycle

         i = i + 1

      end do

      nbond_nonzero = i

!-----------------------------------------------------------------------

      i = 0

      do k = 1, nub

         if ( fc_ub(k) .eq. 0 ) cycle

         i = i + 1

      end do

      nub_nonzero = i

      nlin_nonzero = nbond_nonzero + nub_nonzero

!-----------------------------------------------------------------------

      write( 1, '(a)'   ) '<linear_bonds>'

      write( 1, '(i8)' )  nlin_nonzero

      do k = 1, nbond

         if ( fc_bond(k) .eq. 0.d0 ) cycle

         fc_bond(k) = fc_bond(k) * 2.d0

         eq_bond(k) = eq_bond(k) / unit_to_angstrom

         fc_bond(k) = fc_bond(k) / unit_to_kcal * unit_to_angstrom**2

         call renumber( i_bond(k), inew )
         call renumber( j_bond(k), jnew )

         write( 1, '(2i8,2e24.16)' ) &
     &      inew, jnew, eq_bond(k), fc_bond(k)

      end do

      do k = 1, nub

         if ( fc_ub(k) .eq. 0.d0 ) cycle

         fc_ub(k) = fc_ub(k) * 2.d0

         eq_ub(k) = eq_ub(k) / unit_to_angstrom

         fc_ub(k) = fc_ub(k) / unit_to_kcal * unit_to_angstrom**2

         call renumber( i_ub(k), inew )
         call renumber( j_ub(k), jnew )

         write( 1, '(2i8,2e24.16)' ) &
     &      inew, jnew, eq_ub(k), fc_ub(k)

      end do

!-----------------------------------------------------------------------
!     /*   general form of linear bonds                               */
!-----------------------------------------------------------------------

      write( 1, *      )
      write( 1, '(a)'  ) '<genlin_bonds>'

      write( 1, '(i8)' )  0

!-----------------------------------------------------------------------
!     /*   angular bonds                                              */
!-----------------------------------------------------------------------

      i = 0

      do k = 1, nangl

         if ( fc_angl(k) .eq. 0 ) cycle

         i = i + 1

      end do

      nangl_nonzero = i

!-----------------------------------------------------------------------

      write( 1, *      )
      write( 1, '(a)'  ) '<angular_bonds>'

      write( 1, '(i8)' ) nangl_nonzero

      do k = 1, nangl

         if ( fc_angl(k) .eq. 0 ) cycle

         fc_angl(k) = fc_angl(k) * 2.d0

         eq_angl(k) = eq_angl(k)

         fc_angl(k) = fc_angl(k) / unit_to_kcal / unit_to_rad**2

         call renumber( i_angl(k), inew )
         call renumber( j_angl(k), jnew )
         call renumber( k_angl(k), knew )

         write( 1, '(3i8,2e24.16)' ) &
     &      inew, jnew, knew, eq_angl(k), fc_angl(k)

      end do

!-----------------------------------------------------------------------
!     /*   dihedral bonds                                             */
!-----------------------------------------------------------------------

      i = 0

      do k = 1, ndih

         if ( v_dih(k) .eq. 0 ) cycle

         i = i + 1

      end do

      ndih_nonzero = i

!-----------------------------------------------------------------------

      write( 1, *      )
      write( 1, '(a)'  ) '<dihedral_bonds>'

      write( 1, '(i8)' ) ndih_nonzero

      do k = 1, ndih

         if ( v_dih(k) .eq. 0 ) cycle

         v_dih(k) = v_dih(k) * 2.d0

         v_dih(k) = v_dih(k) / unit_to_kcal

         call renumber( i_dih(k), inew )
         call renumber( j_dih(k), jnew )
         call renumber( k_dih(k), knew )
         call renumber( l_dih(k), lnew )

         write( 1, '(4i8,e24.16,2i8)' ) &
     &      inew, jnew, knew, lnew, v_dih(k), nu_dih(k), mu_dih(k)

      end do

!-----------------------------------------------------------------------
!     /*   improper bonds                                             */
!-----------------------------------------------------------------------

      i = 0

      do k = 1, nimproper

         if ( fc_improper(k) .eq. 0 ) cycle

         i = i + 1

      end do

      nimproper_nonzero = i

!-----------------------------------------------------------------------

      write( 1, *      )
      write( 1, '(a)'  ) '<improper_bonds>'

      write( 1, '(i8)' )  nimproper_nonzero

      do k = 1, nimproper

         if ( fc_improper(k) .eq. 0 ) cycle

         fc_improper(k) = fc_improper(k) * 2.d0

         eq_improper(k) = eq_improper(k)

         fc_improper(k) = fc_improper(k) / unit_to_kcal / unit_to_rad**2

         call renumber( i_improper(k), inew )
         call renumber( j_improper(k), jnew )
         call renumber( k_improper(k), knew )
         call renumber( l_improper(k), lnew )

         write( 1 , '(4i8,2e24.16)' ) &
     &      inew, jnew, knew, lnew, eq_improper(k), fc_improper(k)

      end do

!-----------------------------------------------------------------------
!     /*   lennard-jones                                              */
!-----------------------------------------------------------------------

      i = 0

      do k = 1, nlj

         if ( eps_lj_pair(k) .eq. 0 ) cycle

         i = i + 1

      end do

      nlj_nonzero = i

!-----------------------------------------------------------------------

      i = 0

      do k = 1, nlj14

         if ( eps_lj14(k) .eq. 0 ) cycle

         i = i + 1

      end do

      nlj14_nonzero = i

!-----------------------------------------------------------------------

      write( 1, *      )
      write( 1, '(a)'  ) '<lennard-jones>'

      write( 1, '(i8)' ) nlj_nonzero + nlj14_nonzero

      write( 1, '(2e24.16)' ) rcut_in, rcut_out

      do k = 1, nlj

         if ( eps_lj_pair(k) .eq. 0 ) cycle

         eps_lj_pair(k) = eps_lj_pair(k) / unit_to_kcal

         sig_lj_pair(k) = sig_lj_pair(k) / unit_to_angstrom

         call renumber( i_lj_pair(k), inew )
         call renumber( j_lj_pair(k), jnew )

         write( 1, '(2i8,2e24.16)' ) &
     &      inew, jnew, eps_lj_pair(k), sig_lj_pair(k)

      end do

      do k = 1, nlj14

         if ( eps_lj14(k) .eq. 0 ) cycle

         eps_lj14(k) = eps_lj14(k) / unit_to_kcal

         sig_lj14(k) = sig_lj14(k) / unit_to_angstrom

         call renumber( i_lj14(k), inew )
         call renumber( j_lj14(k), jnew )

         write( 1, '(2i8,2e24.16)' ) &
     &      inew , jnew, eps_lj14(k), sig_lj14(k)

      end do

!-----------------------------------------------------------------------
!     /*   atomic charges                                             */
!-----------------------------------------------------------------------

      i = 0

      do k = 1, natom

         if ( charge(k) .eq. 0 ) cycle

         i = i + 1

      end do

      ncharge = i

!-----------------------------------------------------------------------

      write( 1, *      )
      write( 1, '(a)'  ) '<charges>'

      write( 1, '(i8)' )  ncharge

      do l = 1, natom
         do k = 1, natom

            if ( charge(k) .eq. 0 ) cycle

            call renumber( k, knew )

            if ( knew .ne. l ) cycle

            write( 1, '(i8,2e24.16)' ) knew, charge(k)

         end do
      end do

!-----------------------------------------------------------------------
!     /*   bonded charge pairs                                        */
!-----------------------------------------------------------------------

      write( 1, *      )
      write( 1, '(a)'  ) '<nbcp>'

      write( 1, '(i8)' ) nbcp

      do k = 1, nbcp

         call renumber( i_bcp(k), inew )
         call renumber( j_bcp(k), jnew )

         write( 1, '(2i8,e24.16)' ) inew, jnew, factor_bcp(k)

      end do

!-----------------------------------------------------------------------
!     /*   file close                                                 */
!-----------------------------------------------------------------------

      close( 1 )

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      write( 6, * )
      write( 6, '(a,i6)' ) &
     &   '    MM:  number of bonds          = ', nbond_nonzero
      write( 6, '(a,i6)' ) &
     &   '    MM:  number of urey-bradley   = ', nub_nonzero
      write( 6, '(a,i6)' ) &
     &   '    MM:  number of linear bonds   = ', nlin_nonzero
      write( 6, '(a,i6)' ) &
     &   '    MM:  number of angular bonds  = ', nangl_nonzero
      write( 6, '(a,i6)' ) &
     &   '    MM:  number of dihedral bonds = ', ndih_nonzero
      write( 6, '(a,i6)' ) &
     &   '    MM:  number of improper bonds = ', nimproper_nonzero
      write( 6, '(a,i6)' ) &
     &   '    MM:  number of lennard-jones  = ', nlj_nonzero
      write( 6, '(a,i6)' ) &
     &   '    MM:  number of 1-4 lj         = ', nlj14_nonzero
      write( 6, '(a,i6)' ) &
     &   '    MM:  number of charged atoms  = ', ncharge
      write( 6, '(a,i6)' ) &
     &   '    MM:  number of bonded charges = ', nbcp

      return
      end





!***********************************************************************
      subroutine renumber( i, inew )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, inew, natom

      integer, save :: iset = 0

      integer, dimension(:), allocatable, save :: new

!-----------------------------------------------------------------------
!     /*   make list                                                  */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         open( 20, file = 'renumber.dat' )

         read ( 20, * ) natom

         allocate( new(natom) )

         do j = 1, natom

            read ( 20, * ) k, new(k)

         end do

         close( 20 )

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   get from the list                                          */
!-----------------------------------------------------------------------

      inew = new(i)

      return
      end





!***********************************************************************
      subroutine write_centroid_renumber
!***********************************************************************

      use charmm_variables, only : &
     &   x, y, z, unit_to_angstrom, natom_pdb, symbol_atom_pdb, pdb_file

      implicit none

      integer :: i, inew, j

!-----------------------------------------------------------------------

      open ( 11, file = 'centroid.xyz' )

      write( 11, '(i6)' ) natom_pdb
      write( 11, '(a)'  ) pdb_file

      do j = 1, natom_pdb
         do i = 1, natom_pdb

            call renumber( i, inew )

            if ( inew .ne. j ) cycle

            write( 11, '(a2,3f8.3)' ) &
     &         symbol_atom_pdb(i), x(i), y(i), z(i)

         end do
      end do

      close( 11 )

!-----------------------------------------------------------------------

      open ( 12, file = 'centroid.dat' )

      do j = 1, natom_pdb
         do i = 1, natom_pdb

            call renumber( i, inew )

            if ( inew .ne. j ) cycle

            x(i) = x(i) / unit_to_angstrom
            y(i) = y(i) / unit_to_angstrom
            z(i) = z(i) / unit_to_angstrom

            write( 12, '(3e24.16)' ) x(i), y(i), z(i)

         end do
      end do

      close( 12 )

!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine write_input_renumber
!***********************************************************************

      use charmm_variables, only : &
     &   symbol_atom_pdb, physmass_amu, natom_pdb

      implicit none

      integer :: i, inew, j

!-----------------------------------------------------------------------

      open ( 13, file = 'input.dat' )

      write( 13, '(a)'  ) '<method>'
      write( 13, '(a)'  ) 'STATIC'
      write( 13, *      )

      write( 13, '(a)'  ) '<natom>'
      write( 13, '(i6)' )  natom_pdb
      write( 13, '(a)'  )

      write( 13, '(a)'  ) '<nspec>'
      write( 13, '(i6)' )  natom_pdb

      do j = 1, natom_pdb
         do i = 1, natom_pdb

            call renumber( i, inew )

            if ( inew .ne. j ) cycle

            write( 13, '(a4,1x,f12.4,i2)' ) &
     &         symbol_atom_pdb(i), physmass_amu(i), 1

         end do
      end do

      write( 13, '(a)'  ) '<ipotential>'
      write( 13, '(a)'  ) 'MM'
      write( 13, *      )

      write( 13, '(a)'  ) '<iboundary>'
      write( 13, '(i6)' ) 0

      close( 13 )

!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine setup_convert
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */
      use charmm_variables, only : &
     &   rcut_in, rcut_out, pdb_file, psf_file, par_file, ioption_sort

!     /*   local variables   */
      implicit none

!     /*   local variables   */
      character(len=80) :: char_line

!     /*   local variables   */
      integer :: ierr

!-----------------------------------------------------------------------
!     /*   read header                                                */
!-----------------------------------------------------------------------

      write( 6, '(a)' )
      write( 6, '(a)' ) &
     &   'This program converts charmm force field into pimd format.'
      write( 6, '(a)' )

      write( 6, '(a)' ) &
     &   'For this purpose, three files are required:'
      write( 6, '(a)' )
      write( 6, '(a)' ) &
     &   '   - pdb file:   atomic positions.'
      write( 6, '(a)' ) &
     &   '   - psf file:   topology information.'
      write( 6, '(a)' ) &
     &   '   - par file:   charmm parameters.'
      write( 6, '(a)' )
      write( 6, '(a)' ) &
     &   'as well as lennard-jones cut off radius.'
      write( 6, '(a)' )

      write( 6, '(a)' ) &
     &   'Then this program creates:'
      write( 6, '(a)' )
      write( 6, '(a)' ) &
     &   '   - mm.dat:     mm interaction data.'
      write( 6, '(a)' ) &
     &   '   - input.dat:  input example for the simplest case:'
      write( 6, '(a)' ) &
     &   '                 single-point calculation with free boundary.'

      write( 6, '(a)' )
      write( 6, '(a)' ) &
     &   'Additionally, if you want to change the order of atoms'
      write( 6, '(a)' ) &
     &   'from the original charmm par/psf files, a new file called'
      write( 6, '(a)' ) &
     &   '"renumber.dat" should be prepared.  This should be'
      write( 6, '(a)' ) &
     &   'written in the following form with (n+1) lines:'
      write( 6, '(a)' )
      write( 6, '(a)' ) &
     &   '   line 1:       number of atoms (n).'
      write( 6, '(a)' ) &
     &   '   line 2:       old number, new number (of the 2nd atom).'
      write( 6, '(a)' ) &
     &   '   line n+1:     old number, new number (of the n-th atom).'

      write( 6, '(a)' )
      write( 6, '(a)' ) &
     &   'Renumber the atoms?  Select from below:'
      write( 6, '(a)' )
      write( 6, '(a)' ) &
     &   '    n - use same numbering as charmm.'
      write( 6, '(a)' ) &
     &   '    y - renumber using renumber.dat (mandatory for QM/MM).'
      write( 6, '(a)' ) &
     &   '    blank - cancel and exit.'
      write( 6, '(a)' )

      read ( 5, '(a)', iostat=ierr ) char_line

      write( 6, '(a)' )
      write( 6, '(a)' ) char_line
      write( 6, '(a)' )

!-----------------------------------------------------------------------

      if ( char_line(1:1) .eq. 'y' ) then

         ioption_sort = 1

      else if ( char_line(1:1) .eq. 'n' ) then

         ioption_sort = 0

      else

         write( 6, '(a)' ) 'ERROR: invalid input.  stop.'
         stop

      end if

!-----------------------------------------------------------------------
!     /*   read pdb file                                              */
!-----------------------------------------------------------------------

      write( 6, '(a)' )

      write( 6, '(a)' ) 'INPUT (1/4):  the name of pdb file.'

      write( 6, '(a)' )

      read ( 5, '(a)', iostat=ierr )  pdb_file

      write( 6, '(a)' )
      write( 6, '(a)' ) pdb_file
      write( 6, '(a)' )

      if ( ( ierr .ne. 0 ) .or. ( pdb_file(1:1) .eq. ' ' ) ) then
         write( 6, '(a)' ) 'ERROR: invalid input.  stop.'
         stop
      end if

!-----------------------------------------------------------------------
!     /*   read psf file                                              */
!-----------------------------------------------------------------------

      write( 6, '(a)' )

      write( 6, '(a)' ) 'INPUT (2/4):  the name of charmm psf file,'

      write( 6, '(a)' )

      read ( 5, '(a)', iostat=ierr )  psf_file

      write( 6, '(a)' )
      write( 6, '(a)' ) psf_file
      write( 6, '(a)' )

      if ( ( ierr .ne. 0 ) .or. ( psf_file(1:1) .eq. ' ' ) ) then
         write( 6, '(a)' ) 'ERROR: invalid input.  stop.'
         stop
      end if

!-----------------------------------------------------------------------
!     /*   read par file                                              */
!-----------------------------------------------------------------------

      write( 6, '(a)' )

      write( 6, '(a)' ) 'INPUT (3/4):  the name of charmm par file,'

      write( 6, '(a)' )

      read ( 5, '(a)', iostat=ierr )  par_file

      write( 6, '(a)' )
      write( 6, '(a)' ) par_file
      write( 6, '(a)' )

      if ( ( ierr .ne. 0 ) .or. ( par_file(1:1) .eq. ' ' ) ) then
         write( 6, '(a)' ) 'ERROR: invalid input.  stop.'
         stop
      end if

!-----------------------------------------------------------------------
!     /*   read lennard-jones cut off radius                          */
!-----------------------------------------------------------------------

      write( 6, '(a)' )

      write( 6, '(a)' ) &
     &   'INPUT (4/4):  two parameters for lennard-jones cutoff,'
      write( 6, '(a)' ) &
     &   '              rcut_in [bohr] and rcut_out [bohr].'

      write( 6, '(a)' )

      write( 6, '(a)' ) &
     &   '              these parameters are inside radius and'
      write( 6, '(a)' ) &
     &   '              outside radius, respectively, where l-j'
      write( 6, '(a)' ) &
     &   '              interaction is turned off smoothly using'
      write( 6, '(a)' ) &
     &   '              a switching function.'

      write( 6, '(a)' )

      write( 6, '(a)' ) &
     &   '              the values are commonly used for all the'
      write( 6, '(a)' ) &
     &   '              atomic pairs in the system.'
      write( 6, '(a)' )

      read ( 5, *, iostat=ierr )  rcut_in, rcut_out

      write( 6, '(a)' )
      write( 6, '(2e24.16)' ) rcut_in, rcut_out
      write( 6, '(a)' )

      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) 'ERROR: invalid input.  stop.'
         stop
      end if

      return
      end





!***********************************************************************
      subroutine write_mm
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */

      use charmm_variables, only : &
     &   ioption_sort

!     /*   local variables   */

      implicit none

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      if      ( ioption_sort .eq. 0 ) then

         call write_mm_default

      else if ( ioption_sort .eq. 1 ) then

         call write_mm_renumber

      end if

      return
      end





!***********************************************************************
      subroutine write_centroid
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */

      use charmm_variables, only : &
     &    ioption_sort

!     /*   local variables   */

      implicit none

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      if      ( ioption_sort .eq. 0 ) then

         call write_centroid_default

      else if ( ioption_sort .eq. 1 ) then

         call write_centroid_renumber

      end if

      return
      end





!***********************************************************************
      subroutine write_input
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */

      use charmm_variables, only : &
     &   ioption_sort

!     /*   local variables   */

      implicit none

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      if      ( ioption_sort .eq. 0 ) then

         call write_input_default

      else if ( ioption_sort .eq. 1 ) then

         call write_input_renumber

      end if

      return
      end
