!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 17, 2018 by M. Shiga
!      Description:     mpi communication routines
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine my_mpi_finalize_1
!***********************************************************************

      implicit none

      integer :: ierr

      include 'mpif.h'

!     /*   finalize MPI prallelization   */
      call MPI_FINALIZE ( ierr )

      return
      end





!***********************************************************************
      subroutine my_mpi_finalize_2
!***********************************************************************

      use common_variables, only : &
     &   mpi_comm_main, mpi_comm_sub, mpi_group_world, mpi_group_main, &
     &   mpi_group_sub, mpi_group_pimd, mpi_comm_pimd

      implicit none

      integer :: ierr

      include 'mpif.h'

!     /*   finalize MPI prallelization   */

      call MPI_COMM_FREE ( mpi_comm_pimd, ierr )
      call MPI_COMM_FREE ( mpi_comm_main, ierr )
      call MPI_COMM_FREE ( mpi_comm_sub, ierr )

      call MPI_GROUP_FREE ( mpi_group_world, ierr )
      call MPI_GROUP_FREE ( mpi_group_pimd, ierr )
      call MPI_GROUP_FREE ( mpi_group_main, ierr )
      call MPI_GROUP_FREE ( mpi_group_sub, ierr )

      call MPI_FINALIZE ( ierr )

      return
      end





!***********************************************************************
      subroutine my_mpi_finalize_3
!***********************************************************************

      use common_variables, only : &
     &   mpi_comm_main, mpi_comm_sub, mpi_group_world, mpi_group_main, &
     &   mpi_group_sub, mpi_group_pimd, mpi_comm_pimd

      use polymers_variables, only : mpi_comm_top, mpi_group_top

      implicit none

      integer :: ierr

      include 'mpif.h'

!     /*   finalize MPI prallelization   */

      call MPI_COMM_FREE ( mpi_comm_pimd, ierr )
      call MPI_COMM_FREE ( mpi_comm_main, ierr )
      call MPI_COMM_FREE ( mpi_comm_sub, ierr )
      call MPI_COMM_FREE ( mpi_comm_top, ierr )

      call MPI_GROUP_FREE ( mpi_group_world, ierr )
      call MPI_GROUP_FREE ( mpi_group_pimd, ierr )
      call MPI_GROUP_FREE ( mpi_group_main, ierr )
      call MPI_GROUP_FREE ( mpi_group_sub, ierr )
      call MPI_GROUP_FREE ( mpi_group_top, ierr )

      call MPI_FINALIZE ( ierr )

      return
      end





!***********************************************************************
      subroutine my_mpi_abort
!***********************************************************************

      implicit none

      integer :: ierr

      include 'mpif.h'

      call MPI_ABORT ( MPI_COMM_WORLD, 99, ierr )

      return
      end





!***********************************************************************
      subroutine my_mpi_barrier
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: ierr

      include 'mpif.h'

      call MPI_BARRIER ( mpi_comm_pimd, ierr )

      return
      end





!***********************************************************************
      subroutine my_mpi_barrier_world
!***********************************************************************

      implicit none

      integer :: ierr

      include 'mpif.h'

      call MPI_BARRIER ( MPI_COMM_WORLD, ierr )

      return
      end





!***********************************************************************
      subroutine my_mpi_barrier_main
!***********************************************************************

      use common_variables, only : mpi_comm_main

      implicit none

      integer :: ierr

      include 'mpif.h'

      call MPI_BARRIER ( mpi_comm_main, ierr )

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_int_0_world ( i )
!***********************************************************************

      implicit none

      integer               ::  i, ierr
      integer, dimension(1) ::  j

      include 'mpif.h'

      j(1) = i

      call MPI_BCAST ( j, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr )

      i = j(1)

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_int_1_world ( i, n )
!***********************************************************************

      implicit none

      integer :: ierr, n
      integer, dimension(n) :: i

      include 'mpif.h'

      call MPI_BCAST ( i, n, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr )

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_real_0_world ( a )
!***********************************************************************

      implicit none

      integer :: ierr
      real(8) :: a, b(1)

      include 'mpif.h'

      b(1) = a

      call MPI_BCAST ( b, 1, MPI_DOUBLE_PRECISION, &
     &                 0, MPI_COMM_WORLD, ierr )

      a = b(1)

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_real_1_world ( a, n )
!***********************************************************************

      implicit none

      integer :: ierr, n
      real(8) :: a(n)

      include 'mpif.h'

      call MPI_BCAST ( a, n, MPI_DOUBLE_PRECISION, &
     &                 0, MPI_COMM_WORLD, ierr )

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_real_2_world ( a, n1, n2 )
!***********************************************************************

      implicit none

      integer :: i, j, k, n, n1, n2, ierr
      real(8) :: a(n1,n2), b(n1*n2)

      include 'mpif.h'

      k = 0
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         b(k) = a(i,j)
      end do
      end do

      n = n1*n2

      call MPI_BCAST ( b, n, MPI_DOUBLE_PRECISION, &
     &                 0, MPI_COMM_WORLD, ierr )

      k = 0
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         a(i,j) = b(k)
      end do
      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_char_0_world ( c, l )
!***********************************************************************

      implicit none

      integer :: l, ierr
      character(len=l) :: c

      include 'mpif.h'

      call MPI_BCAST ( c, l, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr )

      return
      end




!***********************************************************************
      subroutine my_mpi_bcast_char_1_world ( c, l, n )
!***********************************************************************

      implicit none

      integer :: i, ierr, l, n
      character(len=l), dimension(n) :: c
      character(len=l) :: d

      include 'mpif.h'

      do i = 1, n

         d(1:l) = c(i)(1:l)

         call MPI_BCAST ( d, l, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr )

         c(i)(1:l) = d(1:l)

      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_int_0 ( i )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer               ::  i, ierr
      integer, dimension(1) ::  j

      include 'mpif.h'

      j(1) = i

      call MPI_BCAST ( j, 1, MPI_INTEGER, 0, mpi_comm_pimd, ierr )

      i = j(1)

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_int_1 ( i, n )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: ierr, n
      integer, dimension(n) :: i

      include 'mpif.h'

      call MPI_BCAST ( i, n, MPI_INTEGER, 0, mpi_comm_pimd, ierr )

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_int_2 ( ii, n1, n2 )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: i, j, k, n, n1, n2, ierr
      integer :: ii(n1,n2), jj(n1*n2)

      include 'mpif.h'

      k = 0
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         jj(k) = ii(i,j)
      end do
      end do

      n = n1*n2

      call MPI_BCAST ( jj, n, MPI_INTEGER, 0, mpi_comm_pimd, ierr )

      k = 0
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         ii(i,j) = jj(k)
      end do
      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_int_3 ( ii, n1, n2, n3 )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: i, j, k, l, n, n1, n2, n3, ierr
      integer :: ii(n1,n2,n3), jj(n1*n2*n3)

      include 'mpif.h'

      k = 0
      do l = 1, n3
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         jj(k) = ii(i,j,l)
      end do
      end do
      end do

      n = n1*n2*n3

      call MPI_BCAST ( jj, n, MPI_INTEGER, 0, mpi_comm_pimd, ierr )

      k = 0
      do l = 1, n3
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         ii(i,j,l) = jj(k)
      end do
      end do
      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_real_0 ( a )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: ierr
      real(8) :: a, b(1)

      include 'mpif.h'

      b(1) = a

      call MPI_BCAST ( b, 1, MPI_DOUBLE_PRECISION, &
     &                 0, mpi_comm_pimd, ierr )

      a = b(1)

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_real_1 ( a, n )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: ierr, n
      real(8) :: a(n)

      include 'mpif.h'

      call MPI_BCAST ( a, n, MPI_DOUBLE_PRECISION, &
     &                 0, mpi_comm_pimd, ierr )

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_real_2 ( a, n1, n2 )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: i, j, k, n, n1, n2, ierr
      real(8) :: a(n1,n2), b(n1*n2)

      include 'mpif.h'

      k = 0
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         b(k) = a(i,j)
      end do
      end do

      n = n1*n2

      call MPI_BCAST ( b, n, MPI_DOUBLE_PRECISION, &
     &                 0, mpi_comm_pimd, ierr )

      k = 0
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         a(i,j) = b(k)
      end do
      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_real_3 ( a, n1, n2, n3 )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: i, j, k, l, n, n1, n2, n3, ierr
      real(8) :: a(n1,n2,n3), b(n1*n2*n3)

      include 'mpif.h'

      k = 0
      do l = 1, n3
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         b(k) = a(i,j,l)
      end do
      end do
      end do

      n = n1*n2*n3

      call MPI_BCAST ( b, n, MPI_DOUBLE_PRECISION, &
     &                 0, mpi_comm_pimd, ierr )

      k = 0
      do l = 1, n3
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         a(i,j,l) = b(k)
      end do
      end do
      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_real_01 ( a, n )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: ierr, n, i
      real(8) :: a(0:n), b(n+1)

      include 'mpif.h'

      do i = 0, n
         b(i+1) = a(i)
      end do

      call MPI_BCAST ( b, n+1, MPI_DOUBLE_PRECISION, &
     &                 0, mpi_comm_pimd, ierr )

      do i = 0, n
         a(i) = b(i+1)
      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_real_03 ( a, n1, n2, n3 )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: i, j, k, l, n, n1, n2, n3, ierr
      real(8) :: a(0:n1,n2,n3), b((n1+1)*n2*n3)

      include 'mpif.h'

      k = 0
      do l = 1, n3
      do j = 1, n2
      do i = 0, n1
         k = k + 1
         b(k) = a(i,j,l)
      end do
      end do
      end do

      n = (n1+1)*n2*n3

      call MPI_BCAST ( b, n, MPI_DOUBLE_PRECISION, &
     &                 0, mpi_comm_pimd, ierr )

      k = 0
      do l = 1, n3
      do j = 1, n2
      do i = 0, n1
         k = k + 1
         a(i,j,l) = b(k)
      end do
      end do
      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_real_4 ( a, n1, n2, n3, n4 )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: i, j, k, l, m, n, n1, n2, n3, n4, ierr
      real(8) :: a(n1,n2,n3,n4), b(n1*n2*n3*n4)

      include 'mpif.h'

      k = 0
      do m = 1, n4
      do l = 1, n3
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         b(k) = a(i,j,l,m)
      end do
      end do
      end do
      end do

      n = n1*n2*n3*n4

      call MPI_BCAST ( b, n, MPI_DOUBLE_PRECISION, &
     &                 0, mpi_comm_pimd, ierr )

      k = 0
      do m = 1, n4
      do l = 1, n3
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         a(i,j,l,m) = b(k)
      end do
      end do
      end do
      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_complex_0 ( a )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: ierr
      complex(8) :: a, b(1)

      include 'mpif.h'

      b(1) = a

      call MPI_BCAST ( b, 1, MPI_DOUBLE_COMPLEX, &
     &                 0, mpi_comm_pimd, ierr )

      a = b(1)

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_complex_1 ( a, n )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: ierr, n
      complex(8) :: a(n)

      include 'mpif.h'

      call MPI_BCAST ( a, n, MPI_DOUBLE_COMPLEX, &
     &                 0, mpi_comm_pimd, ierr )

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_complex_2 ( a, n1, n2 )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: i, j, k, n, n1, n2, ierr
      complex(8) :: a(n1,n2), b(n1*n2)

      include 'mpif.h'

      k = 0
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         b(k) = a(i,j)
      end do
      end do

      n = n1*n2

      call MPI_BCAST ( b, n, MPI_DOUBLE_COMPLEX, &
     &                 0, mpi_comm_pimd, ierr )

      k = 0
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         a(i,j) = b(k)
      end do
      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_char_0 ( c, l )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: l, ierr
      character(len=l) :: c

      include 'mpif.h'

      call MPI_BCAST ( c, l, MPI_CHARACTER, 0, mpi_comm_pimd, ierr )

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_char_1 ( c, l, n )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: i, ierr, l, n
      character(len=l), dimension(n) :: c
      character(len=l) :: d

      include 'mpif.h'

      do i = 1, n

         d(1:l) = c(i)(1:l)

         call MPI_BCAST ( d, l, MPI_CHARACTER, 0, mpi_comm_pimd, ierr )

         c(i)(1:l) = d(1:l)

      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_char_2 ( c, l, n1, n2 )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: i, j, ierr, l, n1, n2
      character(len=l), dimension(n1,n2) :: c
      character(len=l) :: d

      include 'mpif.h'

      do j = 1, n2
      do i = 1, n1

         d(1:l) = c(i,j)(1:l)

         call MPI_BCAST ( d, l, MPI_CHARACTER, 0, mpi_comm_pimd, ierr )

         c(i,j)(1:l) = d(1:l)

      end do
      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_allreduce_real_0 ( a )
!***********************************************************************
!=======================================================================
!
!     all-reduce communication of a real number
!
!=======================================================================

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: ierr
      real(8) :: a
      real(8), dimension(1) :: b1, b2

      include 'mpif.h'

      b1(1) = a

      call MPI_ALLREDUCE ( b1, b2, 1, MPI_DOUBLE_PRECISION, &
     &                     MPI_SUM, mpi_comm_pimd, ierr )

!      call MPI_REDUCE ( b1, b2, 1, MPI_DOUBLE_PRECISION,
!     &                  MPI_SUM, 0, mpi_comm_pimd, ierr )
!      call MPI_BCAST ( b2, 1, MPI_DOUBLE_PRECISION,
!     &                 0, mpi_comm_pimd, ierr )

      a = b2(1)

      return
      end





!***********************************************************************
      subroutine my_mpi_allreduce_real_1 ( a, n )
!***********************************************************************
!=======================================================================
!
!     all-reduce communication of one-dimensional real numbers
!
!=======================================================================

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: i, n, ierr
      real(8) :: a(n), b1(n), b2(n)

      include 'mpif.h'

      do i = 1, n
         b1(i) = a(i)
      end do

      call MPI_ALLREDUCE ( b1, b2, n, MPI_DOUBLE_PRECISION, &
     &                     MPI_SUM, mpi_comm_pimd, ierr )

!      call MPI_REDUCE ( b1, b2, n, MPI_DOUBLE_PRECISION,
!     &                  MPI_SUM, 0, mpi_comm_pimd, ierr )
!      call MPI_BCAST ( b2, n, MPI_DOUBLE_PRECISION,
!     &                 0, mpi_comm_pimd, ierr )

      do i = 1, n
         a(i) = b2(i)
      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_allreduce_real_2 ( a, n1, n2 )
!***********************************************************************
!=======================================================================
!
!     all-reduce communication of two-dimensional real numbers
!
!=======================================================================

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: i, j, k, n, n1, n2, ierr
      real(8) :: a(n1,n2), b1(n1*n2), b2(n1*n2)

      include 'mpif.h'

      k = 0
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         b1(k) = a(i,j)
      end do
      end do

      n = n1*n2

      call MPI_ALLREDUCE ( b1, b2, n, MPI_DOUBLE_PRECISION, &
     &                     MPI_SUM, mpi_comm_pimd, ierr )

!      call MPI_REDUCE ( b1, b2, n, MPI_DOUBLE_PRECISION,
!     &                  MPI_SUM, 0, mpi_comm_pimd, ierr )
!      call MPI_BCAST ( b2, n, MPI_DOUBLE_PRECISION,
!     &                 0, mpi_comm_pimd, ierr )

      k = 0
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         a(i,j) = b2(k)
      end do
      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_allreduce_real_3 ( a, n1, n2, n3 )
!***********************************************************************
!=======================================================================
!
!     all-reduce communication of three-dimensional real numbers
!
!=======================================================================

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: i, j, k, l, n, n1, n2, n3, ierr
      real(8) :: a(n1,n2,n3), b1(n1*n2*n3), b2(n1*n2*n3)

      include 'mpif.h'

      l = 0
      do k = 1, n3
      do j = 1, n2
      do i = 1, n1
         l = l + 1
         b1(l) = a(i,j,k)
      end do
      end do
      end do

      n = n1*n2*n3

      call MPI_ALLREDUCE ( b1, b2, n, MPI_DOUBLE_PRECISION, &
     &                     MPI_SUM, mpi_comm_pimd, ierr )

!      call MPI_REDUCE ( b1, b2, n, MPI_DOUBLE_PRECISION,
!     &                  MPI_SUM, 0, mpi_comm_pimd, ierr )
!      call MPI_BCAST ( b2, n, MPI_DOUBLE_PRECISION,
!     &                 0, mpi_comm_pimd, ierr )

      l = 0
      do k = 1, n3
      do j = 1, n2
      do i = 1, n1
         l = l + 1
         a(i,j,k) = b2(l)
      end do
      end do
      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_allreduce_real_4 ( a, n1, n2, n3, n4 )
!***********************************************************************
!=======================================================================
!
!     all-reduce communication of four-dimensional real numbers
!
!=======================================================================

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: i, j, k, l, m, n, n1, n2, n3, n4, ierr
      real(8) :: a(n1,n2,n3,n4), b1(n1*n2*n3*n4), b2(n1*n2*n3*n4)

      include 'mpif.h'

      m = 0
      do l = 1, n4
      do k = 1, n3
      do j = 1, n2
      do i = 1, n1
         m = m + 1
         b1(m) = a(i,j,k,l)
      end do
      end do
      end do
      end do

      n = n1*n2*n3*n4

      call MPI_ALLREDUCE ( b1, b2, n, MPI_DOUBLE_PRECISION, &
     &                     MPI_SUM, mpi_comm_pimd, ierr )

!      call MPI_REDUCE ( b1, b2, n, MPI_DOUBLE_PRECISION,
!     &                  MPI_SUM, 0, mpi_comm_pimd, ierr )
!      call MPI_BCAST ( b2, n, MPI_DOUBLE_PRECISION,
!     &                 0, mpi_comm_pimd, ierr )

      m = 0
      do l = 1, n4
      do k = 1, n3
      do j = 1, n2
      do i = 1, n1
         m = m + 1
         a(i,j,k,l) = b2(m)
      end do
      end do
      end do
      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_allreduce_md
!***********************************************************************
!=======================================================================
!
!     all-reduce communication in molecular dynamics
!
!=======================================================================

      use common_variables, only : mpi_comm_pimd

      use common_variables, only : &
     &   pot, fx, fy, fz, vir, dipx, dipy, dipz, natom, nbead

      implicit none

      integer :: i, j, k, n, ierr
      real(8) :: b1(nbead+3*nbead*natom+3*3+3*nbead)
      real(8) :: b2(nbead+3*nbead*natom+3*3+3*nbead)

      include 'mpif.h'

      k = 0

      do i = 1, nbead
         k = k + 1
         b1(k) = pot(i)
      end do

      do j = 1, nbead
      do i = 1, natom
         k = k + 1
         b1(k) = fx(i,j)
         k = k + 1
         b1(k) = fy(i,j)
         k = k + 1
         b1(k) = fz(i,j)
      end do
      end do

      do j = 1, 3
      do i = 1, 3
         k = k + 1
         b1(k) = vir(i,j)
      end do
      end do

      do i = 1, nbead
         k = k + 1
         b1(k) = dipx(i)
         k = k + 1
         b1(k) = dipy(i)
         k = k + 1
         b1(k) = dipz(i)
      end do

      n = k

      call MPI_ALLREDUCE ( b1, b2, n, MPI_DOUBLE_PRECISION, &
     &                     MPI_SUM, mpi_comm_pimd, ierr )

      k = 0

      do i = 1, nbead
         k = k + 1
         pot(i) = b2(k)
      end do

      do j = 1, nbead
      do i = 1, natom
         k = k + 1
         fx(i,j) = b2(k)
         k = k + 1
         fy(i,j) = b2(k)
         k = k + 1
         fz(i,j) = b2(k)
      end do
      end do

      do j = 1, 3
      do i = 1, 3
         k = k + 1
         vir(i,j) = b2(k)
      end do
      end do

      do i = 1, nbead
         k = k + 1
         dipx(i) = b2(k)
         k = k + 1
         dipy(i) = b2(k)
         k = k + 1
         dipz(i) = b2(k)
      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_box
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : box, boxinv, vbox

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   broadcast box                                              */
!-----------------------------------------------------------------------

      call my_mpi_bcast_real_2 ( box, 3, 3 )
      call my_mpi_bcast_real_2 ( boxinv, 3, 3 )

!-----------------------------------------------------------------------
!     /*   broadcast box velocity                                     */
!-----------------------------------------------------------------------

      call my_mpi_bcast_real_2 ( vbox, 3, 3 )

      return
      end





!***********************************************************************
      subroutine my_mpi_wtime_MPI( comment, iounit, myrank, nprocs )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   comment   */
      character(len=1) :: comment

!     /*   file number   */
      integer :: iounit

!     /*   processor rank   */
      integer :: myrank

!     /*   number of processors   */
      integer :: nprocs

!     /*   integers   */
      integer :: i

!     /*   initial setting flag   */
      integer, save :: iset = 0

!     /*   old time   */
      real(8), save :: time_old

!     /*   new time   */
      real(8), save :: time_new

!     /*   time interval   */
      real(8), save :: time_interval

!     /*   timer   */
      real(8) :: mpi_wtime

!-----------------------------------------------------------------------
!     /*   initial time                                               */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

!        /*   old time   */
         time_old = mpi_wtime()

!        /*   initial setting end  */
         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   new time                                                   */
!-----------------------------------------------------------------------

      time_new = mpi_wtime()

!-----------------------------------------------------------------------
!     /*   time interval                                              */
!-----------------------------------------------------------------------

      time_interval = time_new - time_old

!-----------------------------------------------------------------------
!     /*   print to file                                              */
!-----------------------------------------------------------------------

!     /*   loop of processes   */
      do i = 1, nprocs

!        /*   loop of process ranks   */
         if ( myrank .eq. i-1 ) then

!           /*   open file   */
            open ( iounit, file = 'wtime.out', access = 'append' )

!           /*   write wall clock time   */
            write( iounit, '(a,1x,i4,f10.3)' ) &
     &         comment, myrank, time_interval

!           /*   close file   */
            close( iounit )

!        /*   loop of process ranks   */
         end if

!        /*   synchronize   */
         call my_mpi_barrier

!     /*   loop of processes   */
      end do

!-----------------------------------------------------------------------
!     /*   old time                                                   */
!-----------------------------------------------------------------------

      time_old = time_new

!-----------------------------------------------------------------------
!     /*   synchronize                                                */
!-----------------------------------------------------------------------

      call my_mpi_barrier

      return
      end





!***********************************************************************
      subroutine my_mpi_allreduce_qmmm_a
!***********************************************************************
!=======================================================================
!
!     all-reduce communication in molecular dynamics
!
!=======================================================================

      use common_variables, only : mpi_comm_pimd

      use common_variables, only : &
     &   natom, nbead

      use qmmm_variables, only : &
     &   pot_a, fx_a, fy_a, fz_a, vir_a, dipx_a, dipy_a, dipz_a

      implicit none

      integer :: i, j, k, n, ierr
      real(8) :: b1(nbead+3*nbead*natom+3*3+3*nbead)
      real(8) :: b2(nbead+3*nbead*natom+3*3+3*nbead)

      include 'mpif.h'

      k = 0

      do i = 1, nbead
         k = k + 1
         b1(k) = pot_a(i)
      end do

      do j = 1, nbead
      do i = 1, natom
         k = k + 1
         b1(k) = fx_a(i,j)
         k = k + 1
         b1(k) = fy_a(i,j)
         k = k + 1
         b1(k) = fz_a(i,j)
      end do
      end do

      do j = 1, 3
      do i = 1, 3
         k = k + 1
         b1(k) = vir_a(i,j)
      end do
      end do

      do i = 1, nbead
         k = k + 1
         b1(k) = dipx_a(i)
         k = k + 1
         b1(k) = dipy_a(i)
         k = k + 1
         b1(k) = dipz_a(i)
      end do

      n = k

      call MPI_ALLREDUCE ( b1, b2, n, MPI_DOUBLE_PRECISION, &
     &                     MPI_SUM, mpi_comm_pimd, ierr )

      k = 0

      do i = 1, nbead
         k = k + 1
         pot_a(i) = b2(k)
      end do

      do j = 1, nbead
      do i = 1, natom
         k = k + 1
         fx_a(i,j) = b2(k)
         k = k + 1
         fy_a(i,j) = b2(k)
         k = k + 1
         fz_a(i,j) = b2(k)
      end do
      end do

      do j = 1, 3
      do i = 1, 3
         k = k + 1
         vir_a(i,j) = b2(k)
      end do
      end do

      do i = 1, nbead
         k = k + 1
         dipx_a(i) = b2(k)
         k = k + 1
         dipy_a(i) = b2(k)
         k = k + 1
         dipz_a(i) = b2(k)
      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_allreduce_qmmm_b
!***********************************************************************
!=======================================================================
!
!     all-reduce communication in molecular dynamics
!
!=======================================================================

      use common_variables, only : mpi_comm_pimd

      use common_variables, only : &
     &   natom, nbead

      use qmmm_variables, only : &
     &   pot_b, fx_b, fy_b, fz_b, vir_b, dipx_b, dipy_b, dipz_b

      implicit none

      integer :: i, j, k, n, ierr
      real(8) :: b1(nbead+3*nbead*natom+3*3+3*nbead)
      real(8) :: b2(nbead+3*nbead*natom+3*3+3*nbead)

      include 'mpif.h'

      k = 0

      do i = 1, nbead
         k = k + 1
         b1(k) = pot_b(i)
      end do

      do j = 1, nbead
      do i = 1, natom
         k = k + 1
         b1(k) = fx_b(i,j)
         k = k + 1
         b1(k) = fy_b(i,j)
         k = k + 1
         b1(k) = fz_b(i,j)
      end do
      end do

      do j = 1, 3
      do i = 1, 3
         k = k + 1
         b1(k) = vir_b(i,j)
      end do
      end do

      do i = 1, nbead
         k = k + 1
         b1(k) = dipx_b(i)
         k = k + 1
         b1(k) = dipy_b(i)
         k = k + 1
         b1(k) = dipz_b(i)
      end do

      n = k

      call MPI_ALLREDUCE ( b1, b2, n, MPI_DOUBLE_PRECISION, &
     &                     MPI_SUM, mpi_comm_pimd, ierr )

      k = 0

      do i = 1, nbead
         k = k + 1
         pot_b(i) = b2(k)
      end do

      do j = 1, nbead
      do i = 1, natom
         k = k + 1
         fx_b(i,j) = b2(k)
         k = k + 1
         fy_b(i,j) = b2(k)
         k = k + 1
         fz_b(i,j) = b2(k)
      end do
      end do

      do j = 1, 3
      do i = 1, 3
         k = k + 1
         vir_b(i,j) = b2(k)
      end do
      end do

      do i = 1, nbead
         k = k + 1
         dipx_b(i) = b2(k)
         k = k + 1
         dipy_b(i) = b2(k)
         k = k + 1
         dipz_b(i) = b2(k)
      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_bcast_logical ( i )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      logical               ::  i, ierr
      logical, dimension(1) ::  j

      include 'mpif.h'

      j(1) = i

      call MPI_BCAST ( j, 1, MPI_LOGICAL, 0, mpi_comm_pimd, ierr )

      i = j(1)

      return
      end





!***********************************************************************
      subroutine my_mpi_allreduce_int_2 ( ii, n1, n2 )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: i, j, k, n, n1, n2, ierr
      integer :: ii(n1,n2), jj1(n1*n2), jj2(n1*n2)

      include 'mpif.h'

      k = 0
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         jj1(k) = ii(i,j)
      end do
      end do

      n = n1*n2

      call MPI_ALLREDUCE ( jj1, jj2, n, MPI_INTEGER, &
     &                     MPI_SUM, mpi_comm_pimd, ierr )

      k = 0
      do j = 1, n2
      do i = 1, n1
         k = k + 1
         ii(i,j) = jj2(k)
      end do
      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_allreduce_int_3 ( ii, n1, n2, n3 )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: i, j, k, l, n, n1, n2, n3, ierr
      integer :: ii(n1,n2,n3)
      integer, allocatable :: jj1(:), jj2(:)

      include 'mpif.h'

      allocate(jj1(n1*n2*n3))
      allocate(jj2(n1*n2*n3))

      l = 0
      do k = 1, n3
      do j = 1, n2
      do i = 1, n1
         l = l + 1
         jj1(l) = ii(i,j,k)
      end do
      end do
      end do

      n = n1*n2*n3

      call MPI_ALLREDUCE ( jj1, jj2, n, MPI_INTEGER, &
     &                     MPI_SUM, mpi_comm_pimd, ierr )

      l = 0
      do k = 1, n3
      do j = 1, n2
      do i = 1, n1
         l = l + 1
         ii(i,j,k) = jj2(l)
      end do
      end do
      end do

      deallocate( jj1 )
      deallocate( jj2 )
      
      return
      end





!***********************************************************************
      subroutine my_mpi_allreduce_mnhc
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      use common_variables, only : &
     &   xbath, ybath, zbath, vxbath, vybath, vzbath, &
     &   vux, vuy, vuz, natom, nnhc, nbead, myrank, nprocs

      implicit none

      integer :: i, j, k, l, o, ierr

      real(8) :: b3( 6*natom*nnhc + 3*natom )

      include 'mpif.h'

!-----------------------------------------------------------------------

      o = 6*natom*nnhc + 3*natom

!-----------------------------------------------------------------------

      do k = 2, nbead
        l = 0
!       /* Load transfer array for correct rank */
        if( mod(k-1, nprocs) .eq. myrank ) then
            do j = 1, nnhc
            do i = 1, natom
                l = l + 1
                b3(l) = xbath(i,j,k)
                l = l + 1
                b3(l) = ybath(i,j,k)
                l = l + 1
                b3(l) = zbath(i,j,k)
            enddo
            enddo
            do j = 1, nnhc
            do i = 1, natom
                l = l + 1
                b3(l) = vxbath(i,j,k)
                l = l + 1
                b3(l) = vybath(i,j,k)
                l = l + 1
                b3(l) = vzbath(i,j,k)
            enddo
            enddo
            do i = 1, natom
               l = l + 1
               b3(l) = vux(i,k)
               l = l + 1
               b3(l) = vuy(i,k)
               l = l + 1
               b3(l) = vuz(i,k)
            end do
        end if

        call MPI_BCAST(b3, o, MPI_DOUBLE_PRECISION, &
     &                 mod(k-1, nprocs) , mpi_comm_pimd, ierr )

        l = 0
        do j = 1, nnhc
        do i = 1, natom
            l = l + 1
            xbath(i,j,k) = b3(l)
            l = l + 1
            ybath(i,j,k) = b3(l)
            l = l + 1
            zbath(i,j,k) = b3(l)
         enddo
         enddo
        do j = 1, nnhc
        do i = 1, natom
            l = l + 1
            vxbath(i,j,k) = b3(l)
            l = l + 1
            vybath(i,j,k) = b3(l)
            l = l + 1
            vzbath(i,j,k) = b3(l)
         enddo
         enddo
         do i = 1, natom
            l = l + 1
            vux(i,k) = b3(l)
            l = l + 1
            vuy(i,k) = b3(l)
            l = l + 1
            vuz(i,k) = b3(l)
         end do
      end do

      return
      end





!***********************************************************************
      subroutine my_mpi_allreduce_max_0 ( c )
!***********************************************************************

      use common_variables, only : mpi_comm_pimd

      implicit none

      integer :: ierr
      real(8) :: c, a(1), b(1)

      include 'mpif.h'

      a(1) = c

      call MPI_ALLREDUCE ( a, b, 1, MPI_DOUBLE_PRECISION, &
     &                     MPI_MAX, mpi_comm_pimd, ierr )

      c = b(1)

      return
      end





!***********************************************************************
      subroutine my_mpi_allreduce_max_0_main ( c )
!***********************************************************************

      use common_variables, only : mpi_comm_main

      implicit none

      integer :: ierr
      real(8) :: c, a(1), b(1)

      include 'mpif.h'

      a(1) = c

      call MPI_ALLREDUCE ( a, b, 1, MPI_DOUBLE_PRECISION, &
     &                     MPI_MAX, mpi_comm_main, ierr )

      c = b(1)

      return
      end





!***********************************************************************
      subroutine my_mpi_allreduce_max_0_sub ( c )
!***********************************************************************

      use common_variables, only : mpi_comm_sub

      implicit none

      integer :: ierr
      real(8) :: c, a(1), b(1)

      include 'mpif.h'

      a(1) = c

      call MPI_ALLREDUCE ( a, b, 1, MPI_DOUBLE_PRECISION, &
     &                     MPI_MAX, mpi_comm_sub, ierr )

      c = b(1)

      return
      end





!***********************************************************************
      module mpi_wtime_variables
!***********************************************************************

      real(8), dimension(:), allocatable :: time_start
      real(8), dimension(:), allocatable :: time_spent
      integer, parameter :: ntime = 50

!***********************************************************************
      end module mpi_wtime_variables
!***********************************************************************





!***********************************************************************
      subroutine my_mpi_wtime( i )
!***********************************************************************

      use common_variables, only : &
     &   iounit, myrank

      use mpi_wtime_variables, only : &
     &   time_start, time_spent, ntime

      implicit none
      include 'mpif.h'
      integer, save :: iset = 0
      integer :: i, j
      real(8) :: t
      character(len=8) :: char_8

      if ( iset .eq. 0 ) then
         if( .not. allocated(time_start) ) allocate( time_start(ntime) )
         if( .not. allocated(time_spent) ) allocate( time_spent(ntime) )
         time_spent(:) = 0.d0
         time_start(:) = 0.d0
         iset = 1
      end if

      if ( ( i .ge. 1 ) .and. ( i .le. ntime ) ) then

         if ( time_start(i) .eq. 0.d0 ) then
            time_start(i) = mpi_wtime()
            iset = 1
         else
            t = mpi_wtime()
            time_spent(i) = time_spent(i) + t - time_start(i)
            time_start(i) = 0.d0
         end if

      else

         call int8_to_char( myrank, char_8 )
         open ( iounit, file = 'mpi_wtime.' // char_8 // '.out' )

         t = 0.d0
         do j = 1, ntime
            t = t + time_spent(j)
         end do

         write( iounit, '(a)' ) &
     &      '---------------------------'
         write( iounit, '(a)' ) &
     &      'num    time [s]   ratio [%]'
         write( iounit, '(a)' ) &
     &      '---------------------------'
         do j = 1, ntime
            if ( (j.ne.1) .and. (time_spent(j).eq.0.d0) ) cycle
            write( iounit, '(i3,2f12.4)' ) &
     &         j, time_spent(j), time_spent(j)/t*100.d0
         end do
         write( iounit, '(a)' ) &
     &      '---------------------------'
         write( iounit, '(a,2f12.4)' ) 'tot', t, 100.d0
         write( iounit, '(a)' ) &
     &      '---------------------------'

         close( iounit )

       end if

      return
      end
