!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 24, 2022 by M. Shiga
!      Description:     shared variables module
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      module common_variables
!***********************************************************************

!-----------------------------------------------------------------------
!     pimd execution command (pimd.x)
!-----------------------------------------------------------------------

      character(len=80) :: pimd_command

!-----------------------------------------------------------------------
!     system
!-----------------------------------------------------------------------

!     /*   code   */
      character(len=80) :: code

!     /*   method   */
      character(len=80) :: method

!     /*   QM/MM potential   */
      character(len=80) :: qmmm_potential

!     /*   QM/MM embedding   */
      character(len=80) :: qmmm_embedding

!     /*   ONIOM high precision potential   */
      character(len=80) :: oniom_hi_potential

!     /*   ONIOM low precision potential   */
      character(len=80) :: oniom_lo_potential

!     /*   ensemble   */
      character(len=10) :: ensemble

!     /*   option of initial geometry file   */
      character(len=10) :: input_style

!     /*   number of atoms   */
      integer:: natom

!     /*   number of beads:  only even number is valid here.   */
      integer:: nbead

!     /*   number of beads:  matsubara dynamics   */
      integer:: mbead_matsubara

!     /*   number of atomic kinds   */
      integer:: nkind

!     /*   number of atoms per kind   */
      integer, dimension(:), allocatable :: natom_kind

!     /*   number of atomic species   */
      integer:: mspec

!     /*   time step decompsition for reference harmonic forces   */
      integer:: nref

!     /*   Suzuki-Yoshida decomposition for thermostat   */
      integer:: nys

!     /*   thermostat type   */
      character(len=10) :: bath_type

!     /*   number of "colors" of the Nose-Hoover chains   */
      integer:: ncolor

!     /*   length of Nose-Hoover chain   */
      integer:: nnhc

!     /*   type of potential model   */
      character(len=80) :: ipotential

!     /*   lattice vector to unfold the trajectory   */
      integer, dimension(:,:,:), allocatable :: mbox

!     /*   atomic positions   */

      real(8), dimension(:,:), allocatable, target::   x
      real(8), dimension(:,:), allocatable, target::   y
      real(8), dimension(:,:), allocatable, target::   z

!     /*   atomic positions for ab initio calculation   */

      real(8), dimension(:,:), allocatable::   xs
      real(8), dimension(:,:), allocatable::   ys
      real(8), dimension(:,:), allocatable::   zs

!     /*   atomic positions for geometry optimization   */

      real(8), dimension(:), allocatable::   xnew
      real(8), dimension(:), allocatable::   ynew
      real(8), dimension(:), allocatable::   znew
      real(8), dimension(:), allocatable::   xold
      real(8), dimension(:), allocatable::   yold
      real(8), dimension(:), allocatable::   zold

!     /*   atomic velocities   */

      real(8), dimension(:,:), allocatable, target::   vx
      real(8), dimension(:,:), allocatable, target::   vy
      real(8), dimension(:,:), allocatable, target::   vz

!     /*   atomic forces   */

      real(8), dimension(:,:), allocatable, target::   fx
      real(8), dimension(:,:), allocatable, target::   fy
      real(8), dimension(:,:), allocatable, target::   fz

!     /*   atomic forces for hessian calculations   */

      real(8), dimension(:,:), allocatable :: fxp
      real(8), dimension(:,:), allocatable :: fyp
      real(8), dimension(:,:), allocatable :: fzp
      real(8), dimension(:,:), allocatable :: fxm
      real(8), dimension(:,:), allocatable :: fym
      real(8), dimension(:,:), allocatable :: fzm

!     /*   atomic potentials   */

      real(8), dimension(:), allocatable::   pot

!     /*   hessian   */

      real(8), dimension(:,:,:), allocatable::  hessian

!     /*   atomic masses   */

      real(8), dimension(:), allocatable::   physmass

!     /*   normal mode positions   */

      real(8), dimension(:,:), allocatable::   ux
      real(8), dimension(:,:), allocatable::   uy
      real(8), dimension(:,:), allocatable::   uz

!     /*   normal mode velocities   */

      real(8), dimension(:,:), allocatable::   vux
      real(8), dimension(:,:), allocatable::   vuy
      real(8), dimension(:,:), allocatable::   vuz

!     /*   normal mode momenta      */

      real(8), dimension(:,:), allocatable::   pux
      real(8), dimension(:,:), allocatable::   puy
      real(8), dimension(:,:), allocatable::   puz

!     /*   normal mode forces   */

      real(8), dimension(:,:), allocatable::   fux
      real(8), dimension(:,:), allocatable::   fuy
      real(8), dimension(:,:), allocatable::   fuz

!     /*   normal mode reference forces   */

      real(8), dimension(:,:), allocatable::   fux_ref
      real(8), dimension(:,:), allocatable::   fuy_ref
      real(8), dimension(:,:), allocatable::   fuz_ref

!     /*   normal mode matrices   */

      real(8), dimension(:,:), allocatable::   u
      real(8), dimension(:,:), allocatable::   uinv
      real(8), dimension(:,:), allocatable::   tnm
      real(8), dimension(:,:), allocatable::   tnminv

!     /*   normal mode masses   */

      real(8), dimension(:,:), allocatable::   dnmmass
      real(8), dimension(:,:), allocatable::   fictmass

!     /*   thermostat positions   */

      real(8), dimension(:,:,:), allocatable::   xbath
      real(8), dimension(:,:,:), allocatable::   ybath
      real(8), dimension(:,:,:), allocatable::   zbath
      real(8), dimension(:,:,:), allocatable::   xbath_cent
      real(8), dimension(:,:,:), allocatable::   ybath_cent
      real(8), dimension(:,:,:), allocatable::   zbath_cent
      real(8), dimension(:,:),   allocatable::   rbath_cent
      real(8), dimension(:,:),   allocatable::   rbath_box
      real(8), dimension(:,:,:), allocatable::   xbath_box
      real(8), dimension(:,:,:), allocatable::   ybath_box
      real(8), dimension(:,:,:), allocatable::   zbath_box

      real(8), dimension(:,:,:,:), allocatable:: xbath_cart
      real(8), dimension(:,:,:,:), allocatable:: ybath_cart
      real(8), dimension(:,:,:,:), allocatable:: zbath_cart

!     /*   thermostat velocities   */

      real(8), dimension(:,:,:), allocatable::   vxbath
      real(8), dimension(:,:,:), allocatable::   vybath
      real(8), dimension(:,:,:), allocatable::   vzbath
      real(8), dimension(:,:,:), allocatable::   vxbath_cent
      real(8), dimension(:,:,:), allocatable::   vybath_cent
      real(8), dimension(:,:,:), allocatable::   vzbath_cent
      real(8), dimension(:,:),   allocatable::   vbath_cent
      real(8), dimension(:,:),   allocatable::   vbath_box
      real(8), dimension(:,:,:), allocatable::   vxbath_box
      real(8), dimension(:,:,:), allocatable::   vybath_box
      real(8), dimension(:,:,:), allocatable::   vzbath_box

      real(8), dimension(:,:,:,:), allocatable:: vxbath_cart
      real(8), dimension(:,:,:,:), allocatable:: vybath_cart
      real(8), dimension(:,:,:,:), allocatable:: vzbath_cart

!     /*   thermostat forces   */

      real(8), dimension(:,:,:), allocatable::   fxbath
      real(8), dimension(:,:,:), allocatable::   fybath
      real(8), dimension(:,:,:), allocatable::   fzbath
      real(8), dimension(:,:,:), allocatable::   fxbath_cent
      real(8), dimension(:,:,:), allocatable::   fybath_cent
      real(8), dimension(:,:,:), allocatable::   fzbath_cent
      real(8), dimension(:,:),   allocatable::   fbath_cent
      real(8), dimension(:,:),   allocatable::   fbath_box
      real(8), dimension(:,:,:), allocatable::   fxbath_box
      real(8), dimension(:,:,:), allocatable::   fybath_box
      real(8), dimension(:,:,:), allocatable::   fzbath_box

      real(8), dimension(:,:,:,:), allocatable:: fxbath_cart
      real(8), dimension(:,:,:,:), allocatable:: fybath_cart
      real(8), dimension(:,:,:,:), allocatable:: fzbath_cart

!     /*   thermostat masses   */

      real(8), dimension(:), allocatable::       qmass
      real(8), dimension(:,:), allocatable::     qmass_cent
      real(8), dimension(:,:), allocatable::     qmass_box

      real(8), dimension(:,:), allocatable::     qmass_cart

!     /*   atomic species   */

      character(len=8), dimension(:), allocatable::   spec
      character(len=8), dimension(:), allocatable::   species
      integer, dimension(:), allocatable::            int_spec
      character(len=8), dimension(:), allocatable::   char_spec
      integer, dimension(:), allocatable::            natom_spec
      real(8), dimension(:), allocatable::            physmass_spec
      integer, dimension(:), allocatable::            num_spec

!     /*   cell matrix   */
      real(8), dimension(3,3), target::  box
      real(8), dimension(3,3), target::  boxinv

!     /*   barostat variables   */

!     /*   type of flexible box   */
      character(len=8)::         npt_type
      character(len=8)::         nph_type
      character(len=8)::         ntt_type
      character(len=8)::         nth_type

!     /*   mass of box for Andersen barostat   */
      real(8)::                  volmass

!     /*   mass of box for Martyna barostat   */
      real(8)::                  boxmass(3,3)
      character(len=8)::         box_anis

!     /*   velocity of cubic box (Andersen)   */
      real(8)::                  vvol

!     /*   velocity of cubic box (Martyna)   */
      real(8)::                  vlog

!     /*   reference volume   */
      real(8)::                  volume_ref

!     /*   reference box   */
      real(8), dimension(3,3)::  box_ref
      real(8), dimension(3,3)::  boxinv_ref

!     /*   isotropic virial   */
      real(8)::                  vir_iso

!     /*   internal isotropic pressure   */
      real(8)::                  pres_iso

!     /*   energy of barostat   */
      real(8)::                  ebaro

!     /*   velocity of box for Martyna barostat   */
      real(8), dimension(3,3)::  vbox

!     /*   virial   */
      real(8), dimension(3,3)::  vir
      real(8), dimension(3,3)::  fbox

!     /*   internal pressure   */
      real(8), dimension(3,3)::  pres

!     /*   time derivative of box matrix   */
      real(8), dimension(3,3)::  boxdot

!     /*   metric matrix   */
      real(8), dimension(3,3)::  gbox

!     /*   stress matrix   */
      real(8), dimension(3,3)::  stress

!     /*   strain matrix   */
      real(8), dimension(3,3)::  strain

!     /*   isotropic stress   */
      real(8) ::  stress_iso

!     /*   isotropic strain   */
      real(8) ::  strain_iso

!     /*   sigma matrix   */
      real(8), dimension(3,3)::  sigma_ref

!     /*   constant for nph, npt, nth and ntt simulations   */
      real(8) :: cmtk = 1.d0   !   real(8) :: cmtk = 3.d0

!     /*   potential for nph, npt, nth and ntt simulations   */
      real(8) :: potential_cmtk

      integer:: ipos_start   = 0
      integer:: ivel_start   = 0
      integer:: ibath_start  = 0
      integer:: itrans_start = 0
      integer:: irot_start   = 0
      integer:: ibox_start   = 0

!     /*   temperature parameters   */

      real(8):: beta
      real(8):: temperature

!     /*   pressure paramter   */
      real(8) :: pressure

!     /*   tension parameter   */
      real(8), dimension(3,3)::  tension

!     /*   beadspread parameters   */

      real(8):: beadspread
      real(8):: beta_beadspread

!     /*   timescale parameters   */

      integer:: igamma
      real(8):: gamma1
      real(8):: gamma2
      real(8):: gamma_cent
      real(8):: gamma2_cent
      real(8):: dt
      real(8):: dt_ref
      real(8):: time_bath
      real(8):: omega_bath
      real(8):: time_baro
      real(8):: omega_baro
      real(8):: omega_p
      real(8):: omega_p2
      real(8):: gamma_p2
      real(8):: time_mode
      real(8), allocatable:: ysweight(:)

!     /*   time step parameters   */

      integer:: nstep
      integer:: istep         = 0
      integer:: iref          = 0
      integer:: istep_start   = 0
      integer:: istep_end     = 0

!     /*   thermostat constants   */

      integer:: ndof
      real(8):: gkt
      real(8):: gnkt
      real(8):: gnpkt
      real(8):: gnkt9

!     /*   hybrid monte carlo parameters   */

      integer  ::  istep_hmc
      integer  ::  iorder_hmc

!     /*   type of replica exchange method   */
      character(len=8) :: irem_type

!     /*   properties   */

      real(8):: ekin
      real(8):: qkin
      real(8):: potential
      real(8):: hamiltonian_sys
      real(8):: ebath
      real(8):: ebath_cent
      real(8):: ebath_mode
      real(8):: ebath_box
      real(8):: ebox
      real(8):: hamiltonian
      real(8):: hamiltonian_cor
      real(8):: temp
      real(8), dimension(3,3) :: pint

!     /*  analyses   */

      integer :: iprint_std
      integer :: iprint_rest
      integer :: iprint_input
      integer :: iprint_minfo
      integer :: iprint_charge = -1

!     /*   only for MPI   */

      integer:: nprocs       = 1
      integer:: myrank       = 0
      integer:: nprocs_world = 1
      integer:: myrank_world = 0
      integer:: nprocs_pimd  = 1
      integer:: myrank_pimd  = 0

      integer:: np_beads     = 1
      integer:: np_force     = 1
      integer:: np_cycle     = 1

!     /*   for two-stages MPI   */

      integer :: mpi_group_world = 0
      integer :: mpi_group_main  = 0
      integer :: mpi_group_sub   = 0
      integer :: nprocs_main     = 0
      integer :: nprocs_sub      = 0
      integer :: mpi_comm_main   = 0
      integer :: mpi_comm_sub    = 0
      integer :: myrank_main     = 0
      integer :: myrank_sub      = 0
      integer :: mpi_group_pimd  = 0
      integer :: mpi_comm_pimd   = 0

!     /*   for exit   */

      integer:: iexit        = 0
      integer:: iread_exit

!     /*   boundary condition   */

      integer:: iboundary    = 0
      character(len=10) :: char_boundary
      real(8):: volume

!     /*   atomic charges from ES program   */
      real(8), dimension(:,:,:), allocatable:: q_es

!     /*   molecular charge   */

      real(8):: charge_mol

!     /*   dipole moment   */

      real(8), dimension(:), allocatable:: dipx
      real(8), dimension(:), allocatable:: dipy
      real(8), dimension(:), allocatable:: dipz

!     /*   center of mass   */

      real(8), dimension(:), allocatable:: xg
      real(8), dimension(:), allocatable:: yg
      real(8), dimension(:), allocatable:: zg

!     /*   option for random numbers, random seed   */
      integer:: irandom, iseed

!     /*   finite difference for numerical derivatives   */
      real(8):: fdiff

!     /*   option for free or fixed ends in string method   */
      character(len=10) :: ends_string

!     /*   om method: overdamped or underdamped   */
      character(len=12) :: equation_om

!     /*   atomic species   */
      integer, dimension(:), allocatable :: ikind

!     /*   afed type   */
      character(len=12) :: afed_type

!     /*   tamd type   */
      character(len=8) :: tamd_type

!     /*   logmfd type   */
      character(len=8) :: logmfd_type

!     /*   execution command for scan   */
      character(len=80) :: scan_exe_shell

!     /*   box for each bead   */
      real(8), dimension(:,:,:), allocatable :: box_bead
      real(8), dimension(:,:,:), allocatable :: boxinv_bead
      real(8), dimension(:,:,:), allocatable :: boxdot_bead
      real(8), dimension(:,:,:), allocatable :: vir_bead
      real(8), dimension(:), allocatable     :: volume_bead
      real(8), dimension(:), allocatable     :: vvol_bead
      real(8), dimension(:), allocatable     :: vlog_bead
      real(8), dimension(:,:,:), allocatable :: vbox_bead
      real(8), dimension(:,:,:), allocatable :: pres_bead
      real(8), dimension(:), allocatable     :: pres_bead_iso

!-----------------------------------------------------------------------
!     file numbers
!-----------------------------------------------------------------------

!     /*   input and output   */
      integer, parameter:: iounit        = 10

!     /*   hamiltonian and temperature   */
      integer, parameter:: iounit_std    = 11

!     /*   trajectory in trj format   */
      integer, parameter:: iounit_trj    = 12

!     /*   box and pressure   */
      integer, parameter:: iounit_box    = 13

!     /*   dipole moment                */
      integer, parameter:: iounit_dip    = 14

!     /*   linear and angular momentum   */
      integer, parameter:: iounit_mom    = 15

!     /*   trajectory in xyz format (16, 22 used in VASP)    */
      integer, parameter:: iounit_xyz    = 23

!     /*   stock averages     */
      integer, parameter:: iounit_avg    = 17

!     /*   bond function   */
      integer, parameter:: iounit_bond   = 18

!     /*   average of energy estimator   */
      integer, parameter:: iounit_eavg   = 19

!     /*   average radius of gyration   */
      integer, parameter:: iounit_rgy    = 20

!     /*   for temporary purposes   */
      integer, parameter:: iounit_tmp    = 22

!c     /*   restart file   */
!      integer, parameter:: iounit_rest  = iounit

!c     /*   radial distribution file   */
!      integer, parameter:: iounit_rdf   = iounit

!     /*   qmmm potentials   */
      integer, parameter:: iounit_qmmm   = 31

!     /*   alchemical mixture   */
      integer, parameter:: iounit_alc    = 32

!     /*   g98 input and output      */
      integer, parameter:: iounit_g98    = 33

!     /*   g03 input and output      */
      integer, parameter:: iounit_g03    = 34

!     /*   g09 input and output      */
      integer, parameter:: iounit_g09    = 35

!     /*   g16 input and output      */
      integer, parameter:: iounit_g16    = 35

!     /*   molpro input and output   */
      integer, parameter:: iounit_molpro = 36

!     /*   turbomole input and output   */
      integer, parameter:: iounit_turbo  = 37

!     /*   cpmd input and output      */
      integer, parameter:: iounit_cpmd   = 38

!     /*   mopac input and output     */
      integer, parameter:: iounit_mopac  = 39

!     /*   gamess input and output      */
      integer, parameter:: iounit_gamess = 40

!     /*   dftb input and output      */
      integer, parameter:: iounit_dftb   = 41

!     /*   abinit-mp input and output      */
      integer, parameter:: iounit_abi    = 42

!     /*   orca input and output   */
      integer, parameter:: iounit_orca   = 43

!     /*   ntchem input and output   */
      integer, parameter:: iounit_ntchem = 44

!     /*   cp2k citations file   */
      integer, parameter:: iounit_cp2k   = 45

!     /*   qe input and output   */
      integer, parameter:: iounit_qe     = 46

!     /*   pfp input and output   */
      integer, parameter:: iounit_pfp    = 47

!     /*   surface hopping output   */
      integer, parameter:: iounit_tfs    = 60

!     /*   surface hopping output   */
      integer, parameter:: iounit_mfe    = 62

!     /*   metadynamics output   */
      integer, parameter:: iounit_meta   = 63

!     /*   nonadiabatic coupling   */
      integer, parameter:: iounit_nac    = 64

!     /*   scan   */
      integer, parameter:: iounit_scn    = 65

!     /*   dual level   */
      integer, parameter:: iounit_dual   = 112

!     /*   phonon   */
      integer, parameter:: iounit_phonon = 67

!     /*   constraints   */
      integer, parameter:: iounit_cons   = 68

!     /*   adiabatic free energy dynamics   */
      integer, parameter:: iounit_afed   = 69

!     /*   adiabatic free energy dynamics   */
      integer, parameter:: iounit_logmfd = 70

!     /*   adiabatic free energy dynamics   */
      integer, parameter:: iounit_weight = 71

!     /*   adiabatic free energy dynamics   */
      integer, parameter:: iounit_tamd   = 72

!     /*   artificial neural network   */
      integer, parameter:: iounit_aenet  = 80

!     /*   n2p2 input and output  */
      integer, parameter:: iounit_n2p2  = 81

!     /*   smash input and output   */
      integer, parameter:: iounit_smash  = 95

!     /*   smash intermediate files   */
      integer, parameter:: iounit_smash_i1  = 96
      integer, parameter:: iounit_smash_i2  = 97
      integer, parameter:: iounit_smash_i3  = 98
      integer, parameter:: iounit_smash_i4  = 99

!     /*   minfo file output   */
      integer, parameter:: iounit_minfo = 110
      
!     /*   dcd wrapped trajectory output   */
      integer, parameter:: iounit_dcd   = 111

!     /*  /dev/null output   */
      integer, parameter:: iounit_null  = 999

!-----------------------------------------------------------------------
!     timer
!-----------------------------------------------------------------------

!     /*   wall clock time   */
      character(len=28):: char_date

!-----------------------------------------------------------------------
!     external codes
!-----------------------------------------------------------------------

!     /*   g16 command   */
      character(len=80):: g16_exe_command
      character(len=80):: g16_formchk_command

!     /*   g16 option   */
      integer          :: iexe_oniom_g16
      integer          :: iexe_grad_g16
      integer          :: iexe_dip_g16

!     /*   g09 command   */
      character(len=80):: g09_exe_command
      character(len=80):: g09_formchk_command

!     /*   g09 option   */
      integer          :: iexe_oniom_g09
      integer          :: iexe_grad_g09
      integer          :: iexe_dip_g09

!     /*   g03 command   */
      character(len=80):: g03_exe_command
      character(len=80):: g03_formchk_command

!     /*   g03 option   */
      integer          :: iexe_grad_g03
      integer          :: iexe_dip_g03

!     /*   g98 command   */
      character(len=80):: g98_exe_command
      character(len=80):: g98_formchk_command

!     /*   g98 option   */
      integer          :: iexe_grad_g98
      integer          :: iexe_dip_g98

!     /*  SMASH command   */
      character(len=80):: smash_exe_command

!     /*  SMASH option: calculation of gradient   */
      integer          :: iexe_grad_smash

!     /*  SMASH option: calculation of dipole moment   */
      integer          :: iexe_dip_smash

!     /*  SMASH option: threads used for smash   */
      integer          :: i_threads_smash

!     /*   SMASH option: extrapolation of lcao coefficients   */
      character(len=20) :: smash_guess

!     /*   mopac command   */
      character(len=80):: mopac_command

!     /*   molpro command   */
      character(len=80)                             :: molpro_command
      integer                                       :: nmolpro
      character(len=80), dimension(:), allocatable  :: molpro_valname
      character(len=80), dimension(:), allocatable  :: molpro_keyword
      integer, dimension(:), allocatable            :: molpro_param_1
      integer, dimension(:), allocatable            :: molpro_param_2

!     /*   turbomole command   */
      character(len=80), dimension(2):: turbo_command
      character(len=80), dimension(2):: turbo_control

!     /*   cpmd command   */
      character(len=80):: cpmd_command
      character(len=80):: cpmd_pp_dir

!     /*   mpdyn command   */
      character(len=80):: mpdyn_command

!     /*   gamess command   */
      character(len=80):: gamess_command

!     /*   dftb command   */
      character(len=80):: dftb_exe_command

!     /*   abinit-mp command   */
      character(len=80):: abinit_mp_exe_command

!     /*   orca command   */
      character(len=80):: orca_command

!     /*   character   */
      character(len=80):: char

!     /*   water model   */
      integer :: model_water

!     /*   user execution command   */
      character(len=80) :: user_command

!     /*   user input file (atomic positions)   */
      character(len=80) :: user_input_file

!     /*   user output file (potential, forces, virial)   */
      character(len=80) :: user_output_file

!     /*   xtb command   */
      character(len=300):: xtb_exe_command

!-----------------------------------------------------------------------
!     mathematical and physical constants
!-----------------------------------------------------------------------

!     /*   circular constant   */
      real(8), parameter:: pi = 3.141592653589793d0

!     /*   Boltzmann constant   */
      real(8), parameter:: boltz     = 0.316682968d-5

!     /*   unit conversion factors   */
      real(8), parameter:: au_length  = 0.529177249d-10
      real(8), parameter:: au_mass    = 9.1093897d-31
      real(8), parameter:: au_energy  = 4.3597482d-18
      real(8), parameter:: au_time    = 0.024188843d-15
      real(8), parameter:: au_charge  = 1.60217646d-19
      real(8), parameter:: ry_energy  = 0.5d0*au_energy

!     /*   hartree to kcal per mol   */
      real(8), parameter:: au2kcal    = 627.5095d0

!     //   conversion factor: Avogadro number
      real(8), parameter:: avogadro  = 6.02214129e+23

!     /*   atomic mass unit   */
      real(8), parameter:: amu_mass  = 1.6605402d-27

!     /*   speed of light in SI unit   */
      real(8), parameter:: speedlight_SI = 2.99892458d+8

!     /*   planck constant   */
      real(8), parameter:: hbar      = 1.d0

!     /*   constant   */
      complex(8), parameter:: zin    = (0.d0,1.d0)

!     /*   debye unit   */
      real(8), parameter :: au2debye = 2.5418d0

!     /*   mass of the earth   */
      real(8) :: amu_mass_earth = 5.9742e+24 / amu_mass

!-----------------------------------------------------------------------
!     mass scaling parameters
!-----------------------------------------------------------------------

      character(len=32), dimension(:), allocatable :: atom_change

!-----------------------------------------------------------------------
!     bcmd
!-----------------------------------------------------------------------

!     //   integrator
      character(len=12) :: integrator_bcmd = 'RESPA'

!     //   decoherence time
      real(8) :: tau_bcmd

!     //   decoherence time relative to ( beta hbar )
      real(8) :: scale_bcmd = 1.d0

!     //   kick step interval
      integer :: kickstep_bcmd = 1

!-----------------------------------------------------------------------
!     trpmd
!-----------------------------------------------------------------------

!     //   integrator
      character(len=15) :: integrator_trpmd = 'VANDEN-EIJNDEN'

!     //   decoherence time relative to ( beta hbar )
      real(8) :: scale_trpmd = 0.5d0

!-----------------------------------------------------------------------
!     extensive message passing
!-----------------------------------------------------------------------

      character(len=5) :: xmpi = 'OFF'

!-----------------------------------------------------------------------
!     pimd qtst
!-----------------------------------------------------------------------

      integer :: iatom_qtst = 0

!***********************************************************************
      end module common_variables
!***********************************************************************





!***********************************************************************
      module XMPI_variables
!***********************************************************************

      integer, dimension(:), allocatable :: istart_bead
      integer, dimension(:), allocatable :: iend_bead
      integer, dimension(:), allocatable :: nbead_paral

      integer, dimension(:), allocatable :: istart_atom
      integer, dimension(:), allocatable :: iend_atom
      integer, dimension(:), allocatable :: natom_paral

      integer :: jstart_bead
      integer :: jend_bead
      integer :: jstart_atom
      integer :: jend_atom

      logical :: positions_sync
      logical :: velocities_sync
      logical :: forces_sync
      logical :: charges_sync

!***********************************************************************
      end module XMPI_variables
!***********************************************************************





!***********************************************************************
      module mm_variables
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   mm potential                                               */
!-----------------------------------------------------------------------

!     /*   linear bonds   */

      integer:: nlin = 0

      integer, dimension(:), allocatable ::  i_lin
      integer, dimension(:), allocatable ::  j_lin
      real(8), dimension(:), allocatable ::  eq_lin
      real(8), dimension(:), allocatable ::  fc_lin

!     /*   generalized linear bonds   */

      integer:: ngenlin = 0

      integer, dimension(:), allocatable ::  i_genlin
      integer, dimension(:), allocatable ::  j_genlin
      integer, dimension(:), allocatable ::  n_genlin
      real(8), dimension(:), allocatable ::  eq_genlin
      real(8), dimension(:), allocatable ::  fc_genlin

!     /*   angular bonds   */

      integer:: nangl = 0

      integer, dimension(:), allocatable ::  i_angl
      integer, dimension(:), allocatable ::  j_angl
      integer, dimension(:), allocatable ::  k_angl
      real(8), dimension(:), allocatable ::  eq_angl
      real(8), dimension(:), allocatable ::  fc_angl

!     /*   dihedral bonds   */

      integer:: ndih  = 0

      integer, dimension(:), allocatable ::  i_dih
      integer, dimension(:), allocatable ::  j_dih
      integer, dimension(:), allocatable ::  k_dih
      integer, dimension(:), allocatable ::  l_dih
      real(8), dimension(:), allocatable ::  v_dih
      integer, dimension(:), allocatable ::  nu_dih
      integer, dimension(:), allocatable ::  mu_dih

!     /*   improper bonds   */

      integer:: nimproper  = 0

      integer, dimension(:), allocatable ::  i_improper
      integer, dimension(:), allocatable ::  j_improper
      integer, dimension(:), allocatable ::  k_improper
      integer, dimension(:), allocatable ::  l_improper
      real(8), dimension(:), allocatable ::  eq_improper
      real(8), dimension(:), allocatable ::  fc_improper

!     /*   Lennard-Jones interaction   */

      integer:: nlj = 0

      integer, dimension(:), allocatable ::  i_lj
      integer, dimension(:), allocatable ::  j_lj
      real(8), dimension(:), allocatable ::  eps_lj
      real(8), dimension(:), allocatable ::  sig_lj
      real(8) ::                             rin_lj
      real(8) ::                             rout_lj

      integer, dimension(3) :: nbox_lj

!     /*   Lennard-Jones pair interaction   */

      integer:: nljpair = 0

      real(8), dimension(:), allocatable ::  eps_ljpair
      real(8), dimension(:), allocatable ::  sig_ljpair
      real(8) ::                             rin_ljpair
      real(8) ::                             rout_ljpair

      character(len=12) :: epsrule_ljpair
      character(len=12) :: sigrule_ljpair

      integer, dimension(3) :: nbox_ljpair

!     /*   Buckingham potential  */

      integer:: nbuck = 0

      integer, dimension(:), allocatable ::  i_buck
      integer, dimension(:), allocatable ::  j_buck
      real(8), dimension(:), allocatable ::  a_buck
      real(8), dimension(:), allocatable ::  b_buck
      real(8), dimension(:), allocatable ::  c_buck
      real(8) :: rin_buck
      real(8) :: rout_buck

      integer, dimension(3) :: nbox_buck

!     /*   Coulomb interaction    */

      integer:: ncharge = 0

      real(8), dimension(:), allocatable ::  q
      integer, dimension(:), allocatable ::  i_q

      real(8), dimension(:), allocatable ::  pol
      integer, dimension(:), allocatable ::  list_pol
      integer, dimension(:), allocatable ::  kind_q

      integer :: nkind_q
      integer :: ndampint

      character(len=8), dimension(:,:), allocatable :: dampform_qq
      character(len=8), dimension(:,:), allocatable :: dampform_qp
      character(len=8), dimension(:,:), allocatable :: dampform_pp

      real(8), dimension(:,:,:), allocatable :: damppar_qq
      real(8), dimension(:,:,:), allocatable :: damppar_qp
      real(8), dimension(:,:,:), allocatable :: damppar_pp

      real(8), dimension(:,:), allocatable :: px
      real(8), dimension(:,:), allocatable :: py
      real(8), dimension(:,:), allocatable :: pz

      integer:: npol = 0

!     /*   bonded charge pairs    */

      integer:: nbcp = 0

      integer, dimension(:), allocatable ::  i_bcp
      integer, dimension(:), allocatable ::  j_bcp
      real(8), dimension(:), allocatable ::  factor_bcp

!     /*   ewald   */

      integer, dimension(3)::  lmax_ewald
      integer, dimension(3)::  nbox_ewald

      real(8)::  alpha_ewald
      real(8)::  ratio_ewald
      real(8)::  rcut_ewald
      real(8)::  eps_ewald
      real(8)::  s_ewald

      integer::  ioption_ewald

      complex(8), dimension(:,:), allocatable ::  eigax
      complex(8), dimension(:,:), allocatable ::  eigay
      complex(8), dimension(:,:), allocatable ::  eigaz
      complex(8), dimension(:,:), allocatable ::  eigbx
      complex(8), dimension(:,:), allocatable ::  eigby
      complex(8), dimension(:,:), allocatable ::  eigbz
      complex(8), dimension(:,:), allocatable ::  eigcx
      complex(8), dimension(:,:), allocatable ::  eigcy
      complex(8), dimension(:,:), allocatable ::  eigcz

      real(8), dimension(3,3)::  bigbox
      real(8), dimension(3,3)::  bigboxinv

      integer :: nmol_dip
      integer :: natom_dip_max
      integer, dimension(:), allocatable :: natom_per_dip
      integer, dimension(:,:), allocatable :: list_atom_dip

!     /*   ewald polarizable   */

      integer, dimension(3)::  lmax_ewpol
      integer, dimension(3)::  nbox_ewpol

      real(8)::  alpha_ewpol
      real(8)::  ratio_ewpol
      real(8)::  rcut_ewpol
      real(8)::  eps_ewpol
      real(8)::  s_ewpol

!-----------------------------------------------------------------------
!     /*   eam potential                                              */
!-----------------------------------------------------------------------

      integer:: neam
      integer:: nref_eam

      real(8), dimension(:),   allocatable :: srho_eam
      real(8), dimension(:),   allocatable :: dfdrho_eam
      integer, dimension(:),   allocatable :: ikind_eam

      integer, dimension(:),   allocatable :: irhor_eam
      integer, dimension(:),   allocatable :: ifrho_eam
      integer, dimension(:,:), allocatable :: iphir_eam

      real(8), dimension(:,:), allocatable :: xref_eam
      real(8), dimension(:,:), allocatable :: yref_eam
      real(8), dimension(:,:), allocatable :: y2ref_eam

      real(8) :: rcut_eam
      real(8) :: rcut_eam2

      integer, dimension(3) :: nbox_eam

      integer, dimension(:,:), allocatable :: iur_adp
      integer, dimension(:,:), allocatable :: iwr_adp

      real(8), dimension(:), allocatable :: ax_adp
      real(8), dimension(:), allocatable :: ay_adp
      real(8), dimension(:), allocatable :: az_adp

      real(8), dimension(:), allocatable :: bxx_adp
      real(8), dimension(:), allocatable :: bxy_adp
      real(8), dimension(:), allocatable :: bxz_adp
      real(8), dimension(:), allocatable :: byy_adp
      real(8), dimension(:), allocatable :: byz_adp
      real(8), dimension(:), allocatable :: bzz_adp

!-----------------------------------------------------------------------
!     /*   morse potential                                            */
!-----------------------------------------------------------------------

      integer:: nmorse = 0

      integer, dimension(:), allocatable ::  i_morse
      integer, dimension(:), allocatable ::  j_morse
      real(8), dimension(:), allocatable ::  eq_morse
      real(8), dimension(:), allocatable ::  depth_morse
      real(8), dimension(:), allocatable ::  alpha_morse

!-----------------------------------------------------------------------
!     /*   particle mesh Ewald                                        */
!-----------------------------------------------------------------------

!     //   ewald_flag = 0: standard ewald, 1: particle mesh ewald
      integer :: ewald_flag

!     //   ewald method, STANDARD or PME
      character(len=9) :: ewald_type

!     //   number of FFT meshes per bohr
      real(8) :: pme_mesh_ewald

!     //   number of FFT meshes
      integer, dimension(3) :: nfft_in

!     //   order of b-spline
      integer :: Bsp_order

      integer :: SizeBtheta
      integer :: SizeGridQ
      integer :: MaxGrid

      integer :: NffTable
      integer :: NffWork

      integer, dimension(3) :: Nfft
      integer, dimension(3) :: Nfftdim

      real(8), dimension(:,:), allocatable :: BthetaX
      real(8), dimension(:,:), allocatable :: BthetaY
      real(8), dimension(:,:), allocatable :: BthetaZ

      real(8), dimension(:,:), allocatable :: dBthetaX
      real(8), dimension(:,:), allocatable :: dBthetaY
      real(8), dimension(:,:), allocatable :: dBthetaZ

      real(8), dimension(:,:), allocatable :: ScRs

      real(8), dimension(:), allocatable :: BsplineModuleX
      real(8), dimension(:), allocatable :: BsplineModuleY
      real(8), dimension(:), allocatable :: BsplineModuleZ

      real(8), dimension(:), allocatable :: FFTtable
      real(8), dimension(:), allocatable :: FFTwork

!     //   for parallel runs
      integer, dimension(:), allocatable :: Nscnt
      integer, dimension(:), allocatable :: Ndisp
      integer, dimension(:), allocatable :: Nrenum

!     //   last positions when neighbor list is updated
      real(8), dimension(:,:), allocatable :: x_list
      real(8), dimension(:,:), allocatable :: y_list
      real(8), dimension(:,:), allocatable :: z_list

!     //   cut off radius of neighbor list
      real(8) :: rcut_list

!     //   cut off radius squared of neighbor list
      real(8) :: rcut2_list

!     //   maximum deviation allowed without updating neighbor list
      real(8) :: dmax_list

!     /*   number of replicated boxes   */
      integer, dimension(3) :: nbox_list

!     //   cut off skin of neighbor list
      real(8) :: skin_eam

!     //   number of atoms neighbor list
      integer, dimension(:,:), allocatable :: n_list

!     //   maximum number of atoms neighbor list
      integer :: nmax_list

!     //   atoms in neighbor list
      integer, dimension(:,:,:), allocatable :: j_list

!     //   box in neighbor list
      integer, dimension(:,:,:), allocatable :: jx_list
      integer, dimension(:,:,:), allocatable :: jy_list
      integer, dimension(:,:,:), allocatable :: jz_list

!-----------------------------------------------------------------------
!     /*   cmap                                                       */
!-----------------------------------------------------------------------

      integer :: ncmap
      integer :: nkind_cmap
      integer :: ngrid_cmap = 24
      integer :: nbuff_cmap = 8
      integer :: mgrid_cmap

      integer, dimension(:,:), allocatable :: i_cmap
      integer, dimension(:,:), allocatable :: j_cmap
      integer, dimension(:,:), allocatable :: k_cmap
      integer, dimension(:,:), allocatable :: l_cmap

      integer, dimension(:), allocatable :: ikind_cmap
      real(8), dimension(:), allocatable :: xgrid_cmap
      real(8), dimension(:), allocatable :: ygrid_cmap

      real(8), dimension(:,:,:), allocatable :: vgrid_cmap
      real(8), dimension(:,:,:), allocatable :: v2grid_cmap

!-----------------------------------------------------------------------
!     /*   mm potential - next version                                */
!-----------------------------------------------------------------------

#ifdef nextver

      integer :: nljbond = 0

      integer, dimension(:), allocatable :: i_ljbond
      integer, dimension(:), allocatable :: j_ljbond

      real(8), dimension(:), allocatable :: sig_ljbond
      real(8), dimension(:), allocatable :: eps_ljbond

      real(8), dimension(:), allocatable :: sig_ljnonb
      real(8), dimension(:), allocatable :: eps_ljnonb

      integer, dimension(:), allocatable :: l_ljatom

      integer, dimension(:), allocatable :: i_ljpair

      integer, dimension(:,:), allocatable :: n_list_ljpair
      integer, dimension(:,:,:), allocatable :: j_list_ljpair

      real(8), dimension(:,:), allocatable :: x_list_ljpair
      real(8), dimension(:,:), allocatable :: y_list_ljpair
      real(8), dimension(:,:), allocatable :: z_list_ljpair

      integer :: nmax_list_ljpair 
      real(8) :: dmax_list_ljpair

      real(8) :: skin_ljpair
      real(8) :: rcut_list_ljpair
      real(8) :: rcut2_list_ljpair

      integer, dimension(:,:), allocatable :: n_list_ewald
      integer, dimension(:,:,:), allocatable :: j_list_ewald

      real(8), dimension(:,:), allocatable :: x_list_ewald
      real(8), dimension(:,:), allocatable :: y_list_ewald
      real(8), dimension(:,:), allocatable :: z_list_ewald

      integer :: nmax_list_ewald
      real(8) :: dmax_list_ewald

      real(8) :: skin_ewald
      real(8) :: rcut_list_ewald
      real(8) :: rcut2_list_ewald

#endif

!***********************************************************************
      end module mm_variables
!***********************************************************************





!***********************************************************************
      module qmmm_variables
!***********************************************************************

!     /*   print interval: qm/mm energy decomposition   */
      integer:: iprint_qmmm

!     /*   number of atoms in the primary region   */
      integer:: natom_p

!     /*   number of link atoms   */
      integer:: natom_l

!     /*   number of atoms in the secondary region   */
      integer:: natom_s

!     /*   potential of qm part   */
      real(8), dimension(:), allocatable:: pot_a

!     /*   potential of mm part   */
      real(8), dimension(:), allocatable:: pot_b

!     /*   forces of qm part   */
      real(8), dimension(:,:), allocatable:: fx_a
      real(8), dimension(:,:), allocatable:: fy_a
      real(8), dimension(:,:), allocatable:: fz_a

!     /*   forces of mm part   */
      real(8), dimension(:,:), allocatable:: fx_b
      real(8), dimension(:,:), allocatable:: fy_b
      real(8), dimension(:,:), allocatable:: fz_b

!     /*   virial of qm part   */
      real(8), dimension(:,:), allocatable:: vir_a

!     /*   virial of mm part   */
      real(8), dimension(:,:), allocatable:: vir_b

!     /*   dipole moment of qm part   */
      real(8), dimension(:), allocatable :: dipx_a
      real(8), dimension(:), allocatable :: dipy_a
      real(8), dimension(:), allocatable :: dipz_a

!     /*   dipole moment of mm part   */
      real(8), dimension(:), allocatable :: dipx_b
      real(8), dimension(:), allocatable :: dipy_b
      real(8), dimension(:), allocatable :: dipz_b

!     /*   layer A (qm) or layer B (mm)   */
      character(len=8), dimension(:), allocatable :: layer

!     /*   atom in layer A bonded to link atom   */
      integer, dimension(:), allocatable :: i_link

!     /*   atom in layer B bonded to link atom   */
      integer, dimension(:), allocatable :: j_link

!     /*   bond ratio of link atom   */
      real(8), dimension(:), allocatable :: r_link

!     /*   atomic species of link atom   */
      character(len=8), dimension(:), allocatable :: species_link

!     /*   atomic number of link atom   */
      integer, dimension(:), allocatable :: int_spec_link

!     /*   qm/mm input directory (for mechanical embedding only)   */
      character(len=80) :: qmmm_dat_dir

!     /*   qm/mm scratch directory (for mechanical embedding only)   */
      character(len=80) :: qmmm_scr_dir

!     /*   print interval: oniom energy decomposition   */
      integer:: iprint_oniom

!     /*   input directory   */
      character(len=80) :: oniom_hi_pl_dat_dir
      character(len=80) :: oniom_lo_pl_dat_dir
      character(len=80) :: oniom_lo_ps_dat_dir

!     /*   scratch directory   */
      character(len=80) :: oniom_hi_pl_scr_dir
      character(len=80) :: oniom_lo_pl_scr_dir
      character(len=80) :: oniom_lo_ps_scr_dir

!     /*   thermostat masses attached to centroids   */
      real(8), dimension(:,:), allocatable:: qmass_cent_multi_a
      real(8), dimension(:,:), allocatable:: qmass_cent_multi_b

!     /*   thermostat masses attached to non-centroids   */
      real(8), dimension(:), allocatable:: qmass_multi_a
      real(8), dimension(:), allocatable:: qmass_multi_b

!     /*   multiple time step parameter   */
      integer :: nmulti = 1

!     /*   step sizes   */
      real(8) :: dt_multi
      real(8) :: dt_multi_ref

!     /*   step numbers   */
      integer :: jref = 1
      integer :: kref = 1

!     /*   normal mode forces of qm part   */
      real(8), dimension(:,:), allocatable:: fux_a
      real(8), dimension(:,:), allocatable:: fuy_a
      real(8), dimension(:,:), allocatable:: fuz_a

!     /*   normal mode forces of mm part   */
      real(8), dimension(:,:), allocatable:: fux_b
      real(8), dimension(:,:), allocatable:: fuy_b
      real(8), dimension(:,:), allocatable:: fuz_b

!     /*   average potential of qm part   */
      real(8) :: potential_a

!     /*   average potential of mm part   */
      real(8) :: potential_b

!     /*   option for variable charge   */
      integer :: ivar_qmmm

!***********************************************************************
      end module qmmm_variables
!***********************************************************************





!***********************************************************************
      module cart_variables
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   cartesian potential                                        */
!-----------------------------------------------------------------------

      integer:: ncart = 0

      integer, dimension(:), allocatable ::  i_cart
      integer, dimension(:), allocatable ::  nx_cart
      integer, dimension(:), allocatable ::  ny_cart
      integer, dimension(:), allocatable ::  nz_cart
      real(8), dimension(:), allocatable ::  eqx_cart
      real(8), dimension(:), allocatable ::  eqy_cart
      real(8), dimension(:), allocatable ::  eqz_cart
      real(8), dimension(:), allocatable ::  fc_cart

!***********************************************************************
      end module cart_variables
!***********************************************************************





!***********************************************************************
      module best_variables
!***********************************************************************

!     /*   print interval   */
      integer :: iprint_best

!     /*   option of best in integer   */
      integer :: ioption_best = 0

!     /*   species of solvent atom   */
      integer :: ikind_best = 0

!     /*   solute atom   */
      integer :: iobest = 0

!     /*   solvent atoms in primary and secondary subsystems   */
      integer, dimension(:), allocatable :: ibest

!     /*   number of solute atom (always 1)  */
      integer :: nobest = 1

!     /*   number of solvent atoms in primary subsystem   */
      integer :: npbest = 0

!     /*   number of solvent atoms in secondary subsystem   */
      integer :: nsbest = 0

!     /*   total number of solvent atoms   */
      integer :: nbest = 0

!     /*   energy cutoff for buffer range (input)   */
      real(8) :: eps_best = 1.d-8

!     /*   force constant of bias potential (input)   */
      real(8) :: fc_best = 1.d0

!     /*   exponent of g function = 0.5 beta fc_beta   */
      real(8) :: alpha_best

!     /*   buffer range = sqrt(-log(beta eps_best)/alpha_best))  */
      real(8) :: r_best

!     /*   best index   */
      real(8), dimension(:), allocatable :: s_best

!     /*   index s in sorted order   */
      real(8), dimension(:), allocatable :: s_sort

!     /*   atom in sorted order   */
      integer, dimension(:), allocatable :: num_sort

!     /*   components of index   */
      real(8), dimension(:), allocatable :: sx_best
      real(8), dimension(:), allocatable :: sy_best
      real(8), dimension(:), allocatable :: sz_best

!     /*   bias potential   */
      real(8), dimension(:), allocatable :: pot_best

!     /*   cartesian force   */
      real(8), dimension(:,:), allocatable :: fx_best
      real(8), dimension(:,:), allocatable :: fy_best
      real(8), dimension(:,:), allocatable :: fz_best

!     /*   normal mode force   */
      real(8), dimension(:,:), allocatable :: fux_best
      real(8), dimension(:,:), allocatable :: fuy_best
      real(8), dimension(:,:), allocatable :: fuz_best

!     /*   virial   */
      real(8), dimension(:,:), allocatable :: vir_best

!     /*   derivative of bias potential   */
      real(8), dimension(:), allocatable :: dv_best

!     /*   number of atoms in primary subsystem   */
      integer :: npbest_buf = 0

!     /*   number of atoms in secondary subsystem   */
      integer :: nsbest_buf = 0

!     /*   number of atoms in primary and secondary subsystems   */
      integer :: nbest_buf = 0

!     /*   number of exchange combinations   */
      integer :: nc_buf = 0

!     /*   maximum value   */
      real(8) :: s_best_max

!     /*   maximum value   */
      real(8) :: s_best_min

!     /*   border value   */
      real(8) :: s_best_border

!     /*   upper bound of buffer region  */
      real(8) :: s_best_hi

!     /*   lower bound of buffer region   */
      real(8) :: s_best_lo

!     /*   bias potential   */
      real(8) :: v_best

!     /*   f function   */
      real(8) :: f_best

!     /*   g function   */
      real(8), dimension(:,:), allocatable   :: gbuf

!     /*   derivative of g function   */
      real(8), dimension(:,:,:), allocatable :: dgbuf

!     /*   coordinate   */
      real(8), dimension(:), allocatable     :: sbuf

!     /*   atom number   */
      integer, dimension(:), allocatable     :: ibuf

!     /*   qm atom   */
      integer, dimension(:), allocatable     :: ival

!     /*   qm occupation   */
      integer, dimension(:,:), allocatable   :: iocc

!     /*   atom number   */
      integer, dimension(:,:), allocatable   :: inum

!     /*   derivative of log(g)  */
      real(8), dimension(:), allocatable     :: dloggbuf

!     /*   h function   */
      real(8), dimension(:), allocatable     :: hbuf

!     /*   derivative of h function   */
      real(8), dimension(:,:), allocatable   :: dhbuf

!     /*   maximum number of atoms in buffer region (changed)  */
      integer :: nbest_buf_max = 2

!     /*   maximum combination number (changed)   */
      integer :: nc_buf_max    = 2

!***********************************************************************
      end module best_variables
!***********************************************************************





!***********************************************************************
      module analysis_variables
!***********************************************************************

      integer ::  iprint_bond
      integer ::  iprint_rgy
      integer ::  iprint_eavg
      integer ::  iprint_rdf
      integer ::  iprint_rdf_bead
      integer ::  iprint_rdf_cent
      integer ::  iprint_trj
      integer ::  iprint_xyz
      integer ::  iprint_xsf
      integer ::  iprint_dcd
      integer ::  iprint_dip
      integer ::  iprint_mom
      integer ::  iprint_box
      integer ::  iprint_cons
      integer ::  iprint_str
      integer ::  iprint_pot
      integer ::  iprint_matsubara
      integer ::  iprint_akin
      integer ::  iprint_cavg

      integer ::  iformat_xyz
      integer ::  iformat_trj
      integer, dimension(2) ::  ikind_xyz

      real(8), dimension(:), allocatable::  rbead
      real(8), dimension(:), allocatable::  rbead2
      real(8), dimension(:), allocatable::  rcent
      real(8), dimension(:), allocatable::  rcent2
      real(8), dimension(:), allocatable::  rkubo
      real(8), dimension(:), allocatable::  rkubo2
      real(8), dimension(:), allocatable::  rbead_avg
      real(8), dimension(:), allocatable::  rbead2_avg
      real(8), dimension(:), allocatable::  rcent_avg
      real(8), dimension(:), allocatable::  rcent2_avg
      real(8), dimension(:), allocatable::  rkubo_avg
      real(8), dimension(:), allocatable::  rkubo2_avg

      real(8), dimension(:), allocatable::  rgy
      real(8), dimension(:), allocatable::  rgy2
      real(8), dimension(:), allocatable::  rgy_avg
      real(8), dimension(:), allocatable::  rgy2_avg

      real(8) ::  epot
      real(8) ::  ekinvir
      real(8) ::  ekinpri
      real(8) ::  etot
      real(8) ::  epot_avg
      real(8) ::  ekinvir_avg
      real(8) ::  ekinpri_avg
      real(8) ::  etot_avg

      real(8) ::  eprivir_avg
      real(8) ::  specific_heat

      real(8), dimension(3)                ::  params_rdf
      real(8), dimension(:,:), allocatable ::  rdf
      real(8), dimension(:,:), allocatable ::  rdf_n
      real(8), dimension(:,:), allocatable ::  rdf_g
      real(8), dimension(:,:), allocatable ::  rdf_avg

      real(8), dimension(:,:), allocatable ::  rdf_bead
      real(8), dimension(:,:), allocatable ::  rdf_n_bead
      real(8), dimension(:,:), allocatable ::  rdf_g_bead
      real(8), dimension(:,:), allocatable ::  rdf_avg_bead
      
      real(8), dimension(:,:), allocatable ::  rdf_cent
      real(8), dimension(:,:), allocatable ::  rdf_n_cent
      real(8), dimension(:,:), allocatable ::  rdf_g_cent
      real(8), dimension(:,:), allocatable ::  rdf_avg_cent

      integer, dimension(:,:), allocatable ::  ikindpair
      integer, dimension(:,:), allocatable ::  ikindpair_inv
      integer, dimension(:),   allocatable ::  npair_kindpair

      integer ::  nkindpair

      real(8) ::  temp_tot
      real(8) ::  temp_cent
      real(8) ::  temp_cart
      real(8) ::  p_tot
      real(8) ::  p_cent
      real(8) ::  p_cart
      real(8) ::  a_tot
      real(8) ::  a_cent
      real(8) ::  a_cart
      real(8) ::  t_tot
      real(8) ::  t_cent
      real(8) ::  t_cart

      real(8) ::  volume_avg
      real(8) ::  pres_iso_avg
      real(8), dimension(3,3) ::  box_avg
      real(8), dimension(3,3) ::  pres_avg
      real(8), dimension(3,3) ::  stress_avg
      real(8), dimension(3,3) ::  strain_avg
      real(8) :: stress_iso_avg
      real(8) :: strain_iso_avg

!     /*   average centroid force   */
      real(8), dimension(:), allocatable :: fux_avg
      real(8), dimension(:), allocatable :: fuy_avg
      real(8), dimension(:), allocatable :: fuz_avg

      real(8), dimension(:), allocatable     :: volume_bead_avg
      real(8), dimension(:), allocatable     :: pres_bead_iso_avg
      real(8), dimension(:,:,:), allocatable :: box_bead_avg
      real(8), dimension(:,:,:), allocatable :: pres_bead_avg

      real(8), dimension(:), allocatable :: akin
      real(8), dimension(:), allocatable :: akin_avg

      real(8) :: e_avg
      real(8) :: c_avg
      real(8) :: c1_avg
      real(8) :: c2_avg

!***********************************************************************
      end module analysis_variables
!***********************************************************************





!***********************************************************************
      module multistate_variables
!***********************************************************************

!     /*   number of states:  default value is 1.   */
      integer:: nstate = 1

!     /*   v : diabatic hamiltonian matrix   */
      real(8), dimension(:,:,:), allocatable:: vstate

!     /*   g : derivative of v   */
      real(8), dimension(:,:,:,:), allocatable:: gxstate
      real(8), dimension(:,:,:,:), allocatable:: gystate
      real(8), dimension(:,:,:,:), allocatable:: gzstate

!     /*   d : nonadiabatic coupling matrix elements   */
      real(8), dimension(:,:,:,:), allocatable:: dxstate
      real(8), dimension(:,:,:,:), allocatable:: dystate
      real(8), dimension(:,:,:,:), allocatable:: dzstate

!     /*   d : nonadiabatic coupling matrix elements   */
      real(8), dimension(:,:,:,:), allocatable:: dxstate_old
      real(8), dimension(:,:,:,:), allocatable:: dystate_old
      real(8), dimension(:,:,:,:), allocatable:: dzstate_old

!     /*   d : nonadiabatic coupling matrix elements   */
      real(8), dimension(:,:), allocatable:: dipxstate
      real(8), dimension(:,:), allocatable:: dipystate
      real(8), dimension(:,:), allocatable:: dipzstate

!     /*   c : coefficient   */
      complex(8), dimension(:,:), allocatable::  cstate

!     /*   pop : population   */
      real(8), dimension(:,:), allocatable:: pop_state

!     /*   occ : occupancy   */
      real(8) :: occ_state

!     /*   pop : sum of population   */
      real(8), dimension(:), allocatable:: pop_sum

!     /*   occupied state in TFS method   */
      integer, dimension(:), allocatable:: istate_tfs

!     /*   branching ratio   */
      real(8), dimension(:), allocatable :: branch_ratio

!     /*   initial state   */
      integer:: istate_init

!     /*   print interval   */
      integer:: iprint_tfs

!     /*   print interval   */
      integer:: iprint_mfe

!     /*   print interval   */
      integer:: iprint_nac

!***********************************************************************
      end module multistate_variables
!***********************************************************************





!***********************************************************************
      module hmc_variables
!***********************************************************************

      integer  ::  jstep

      real(8)  ::  potential_hmc
      real(8)  ::  potential_second
      real(8)  ::  potential_fourth
      real(8)  ::  potential_cor
      real(8)  ::  hamiltonian_hmc
      real(8)  ::  e_hmc

!     /*   preserved coordinates   */
      real(8), dimension(:,:), allocatable ::  x_hmc_last
      real(8), dimension(:,:), allocatable ::  y_hmc_last
      real(8), dimension(:,:), allocatable ::  z_hmc_last

!     /*   preserved potential   */
      real(8), dimension(:), allocatable   ::  pot_hmc_last

!     /*   preserved force   */
      real(8), dimension(:,:), allocatable ::  fx_hmc_last
      real(8), dimension(:,:), allocatable ::  fy_hmc_last
      real(8), dimension(:,:), allocatable ::  fz_hmc_last

!     /*   preserved box   */
      real(8), dimension(:,:), allocatable ::  box_hmc_last

      real(8), dimension(:), allocatable    ::  pot_hmc
      real(8), dimension(:), allocatable    ::  pot_second
      real(8), dimension(:), allocatable    ::  pot_fourth

      real(8), dimension(:,:), allocatable  ::  fx_second
      real(8), dimension(:,:), allocatable  ::  fy_second
      real(8), dimension(:,:), allocatable  ::  fz_second

      real(8), dimension(:,:), allocatable  ::  ux_save
      real(8), dimension(:,:), allocatable  ::  uy_save
      real(8), dimension(:,:), allocatable  ::  uz_save

      real(8), dimension(:,:), allocatable  ::  vux_save
      real(8), dimension(:,:), allocatable  ::  vuy_save
      real(8), dimension(:,:), allocatable  ::  vuz_save

      real(8), dimension(:,:), allocatable  ::  fux_save
      real(8), dimension(:,:), allocatable  ::  fuy_save
      real(8), dimension(:,:), allocatable  ::  fuz_save

      real(8), dimension(:,:), allocatable  ::  fux_ref_save
      real(8), dimension(:,:), allocatable  ::  fuy_ref_save
      real(8), dimension(:,:), allocatable  ::  fuz_ref_save

      real(8), dimension(:), allocatable    ::  pot_save
      real(8), dimension(:), allocatable    ::  pot_second_save
      real(8), dimension(:), allocatable    ::  pot_fourth_save
      real(8), dimension(:), allocatable    ::  pot_hmc_save

      real(8)  ::  potential_save
      real(8)  ::  potential_hmc_save
      real(8)  ::  potential_second_save
      real(8)  ::  potential_fourth_save

      real(8)  ::  ekin_save
      real(8)  ::  qkin_save

      real(8)  ::  hamiltonian_save
      real(8)  ::  hamiltonian_hmc_save

      integer, dimension(:,:,:), allocatable :: mbox_save

      real(8)                  ::  vvol_save
      real(8)                  ::  vlog_save
      real(8), dimension(3,3)  ::  vbox_save

      real(8), dimension(3,3)  ::  box_save

      real(8), dimension(3,3)  ::  vir_save
      real(8), dimension(3,3)  ::  pres_save
      real(8)                  ::  pres_iso_save

      real(8), dimension(3,3)  ::  strain_save
      real(8), dimension(3,3)  ::  stress_save

      real(8)  ::  bfactor

      integer  ::  naccept = 0
      integer  ::  nreject = 0
      real(8)  ::  ratio = 0.d0

      real(8) :: ratio_min_hmc
      real(8) :: ratio_max_hmc
      integer :: istep_min_hmc
      integer :: istep_max_hmc
      integer :: istep_mul_hmc
      integer :: istep_adjust_hmc

!***********************************************************************
      end module hmc_variables
!***********************************************************************





!***********************************************************************
      module lbfgs_variables
!***********************************************************************

!     //   number of lbfgs update
      integer, parameter :: nup   = 5

!     //   dimension of the workspace in lbfgs
      integer:: nwork

!     //   dimension of the system
      integer:: ndim

!     //   position
      real(8), dimension(:), allocatable:: pos

!     //   old position
      real(8), dimension(:), allocatable:: pos0

!     //   potential
      real(8):: func

!     //   gradient
      real(8), dimension(:), allocatable:: grad

!     //   diagonal matrix in lbfgs
      real(8), dimension(:), allocatable:: dm

!     //   workspace in lbfgs
      real(8), dimension(:), allocatable:: ws

!     //   accuracy of solution in lbfgs
      real(8):: eps = 0.d0

!     //   machine precision: postol = 1.d-16
      real(8):: postol = 1.d-16

!     //   accuracy of line search in lbfgs: gradtol = 0.9.
      real(8):: gtol   = 0.9d0

!     //   lower bound of line search in lbfgs: stpmin = 1.d-20.
      real(8):: stpmin = 1.d-20

!     //   upper bound of line search in lbfgs: stpmax = 1.d+20.
      real(8):: stpmax = 1.d+20

!     //   scaling factor of initial hessian in lbfgs.
      real(8):: dminit = 0.05d0

!     //   print options in lbfgs
      integer, dimension(2):: iprint = (/ -1, -1 /)

!     //   print options in lbfgs: mp = 6, lp = 6.
      integer:: mp = 6
      integer:: lp = 6

!     //   convergence flag
      integer:: iflag = 0

!     //   convergence for geometrical shift and residual force
      real(8):: dmax, drms, fmax, frms

!-----------------------------------------------------------------------
!     //   important convergence parameters
!-----------------------------------------------------------------------

!     //   convergence parameters for geometrical shift
      real(8):: dmax_tol = 1.d-4
      real(8):: drms_tol = 1.d-5

!     //   convergence parameters for residual force
      real(8):: fmax_tol = 1.d-5
      real(8):: frms_tol = 1.d-6

!***********************************************************************
      end module lbfgs_variables
!***********************************************************************





!***********************************************************************
      module mech_variables
!***********************************************************************

!     /*   options   */
      character(len=8) :: mech_type = 'NONE'

!     /*   zone   */
      integer, dimension(:), allocatable :: itype_zone_mech

!     /*   atoms or species   */
      integer, dimension(:), allocatable :: i_mech
      integer, dimension(:), allocatable :: j_mech
      integer, dimension(:), allocatable :: k_mech
      integer, dimension(:), allocatable :: l_mech

!     /*   coordination number   */
      integer, dimension(:,:), allocatable :: nu_mech
      integer, dimension(:,:), allocatable :: mu_mech
      real(8), dimension(:,:), allocatable :: req_mech

!     /*   force constant   */
      real(8), dimension(:), allocatable :: fc_mech

!     /*   radius   */
      real(8), dimension(:), allocatable :: r_mech

!     /*   mechanical potential   */
      real(8), dimension(:), allocatable :: pot_mech

!     /*   mechanical force   */
      real(8), dimension(:,:), allocatable :: fx_mech
      real(8), dimension(:,:), allocatable :: fy_mech
      real(8), dimension(:,:), allocatable :: fz_mech

!     /*   mechanical virial   */
      real(8), dimension(:,:), allocatable :: vir_mech

!     /*   print interval   */
      integer :: iprint_mech

!     /*   group type mechanical force   */

      integer, parameter :: MAXLINE=256

      type group_mech_type
        character(LEN=MAXLINE) :: label
        integer :: natom_each
        real(8) :: cfx
        real(8) :: cfy
        real(8) :: cfz
        integer :: flag_pot
        integer, dimension(:), allocatable :: id
      end type group_mech_type

      type(group_mech_type), allocatable :: group_mech(:)
      integer :: ngroup_mech

!     /*   zone type mechanical force   */

      character(len=10), dimension(:), allocatable :: zone_mech
      integer :: nzone_mech

!     /*   electrostatic force   */

      real(8), dimension(:), allocatable :: q_mech
      real(8) :: ex_mech, ey_mech, ez_mech

!     /*   freeze   */
      integer :: nfreeze_mech = 0

!     /*   afir   */
      integer :: nafir_mech

!***********************************************************************
      end module mech_variables
!***********************************************************************





!***********************************************************************
      module dual_variables
!***********************************************************************

      real(8), dimension(:), allocatable:: x_lo
      real(8), dimension(:), allocatable:: y_lo
      real(8), dimension(:), allocatable:: z_lo

      real(8), dimension(:), allocatable:: x_hi
      real(8), dimension(:), allocatable:: y_hi
      real(8), dimension(:), allocatable:: z_hi

      real(8), dimension(:), allocatable:: e_lo
      real(8), dimension(:), allocatable:: e_hi

      real(8), dimension(:), allocatable:: fx_lo
      real(8), dimension(:), allocatable:: fy_lo
      real(8), dimension(:), allocatable:: fz_lo

      real(8), dimension(:), allocatable:: fx_hi
      real(8), dimension(:), allocatable:: fy_hi
      real(8), dimension(:), allocatable:: fz_hi

      character(len=80) :: idual_lo
      character(len=80) :: idual_hi

      character(len=80)::  dat_dir_lo
      character(len=80)::  dat_dir_hi

      character(len=80)::  scr_dir_lo
      character(len=80)::  scr_dir_hi

      real(8), dimension(:), allocatable :: dipx_lo
      real(8), dimension(:), allocatable :: dipy_lo
      real(8), dimension(:), allocatable :: dipz_lo

      real(8), dimension(:), allocatable :: dipx_hi
      real(8), dimension(:), allocatable :: dipy_hi
      real(8), dimension(:), allocatable :: dipz_hi

      real(8) :: potential_low
      real(8) :: potential_high

      real(8), dimension(:), allocatable:: pot_low
      real(8), dimension(:), allocatable:: pot_high

      real(8), dimension(:,:), allocatable, target:: x_trial
      real(8), dimension(:,:), allocatable, target:: y_trial
      real(8), dimension(:,:), allocatable, target:: z_trial

      real(8), dimension(:,:), allocatable, target:: fx_low
      real(8), dimension(:,:), allocatable, target:: fy_low
      real(8), dimension(:,:), allocatable, target:: fz_low

      real(8), dimension(:,:), allocatable, target:: fx_high
      real(8), dimension(:,:), allocatable, target:: fy_high
      real(8), dimension(:,:), allocatable, target:: fz_high

      real(8), dimension(:,:), allocatable:: vir_low
      real(8), dimension(:,:), allocatable:: vir_high

      real(8), dimension(:), allocatable:: dipx_low
      real(8), dimension(:), allocatable:: dipy_low
      real(8), dimension(:), allocatable:: dipz_low

      real(8), dimension(:), allocatable:: dipx_high
      real(8), dimension(:), allocatable:: dipy_high
      real(8), dimension(:), allocatable:: dipz_high

      real(8), dimension(:,:,:), allocatable:: vir_bead_low
      real(8), dimension(:,:,:), allocatable:: vir_bead_high

!     /*   print interval   */
      integer:: iprint_dual

!     /*   parameters   */
      real(8), dimension(:), allocatable :: params_dual

!     /*   saved potentials   */
      real(8) :: potential_high_save
      real(8) :: potential_low_save

!     /*   parameters   */
      integer :: justtrained_dual = 0

!***********************************************************************
      end module dual_variables
!***********************************************************************





!***********************************************************************
      module meta_variables
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   basic parameters for metadynamics                          */
!-----------------------------------------------------------------------

!     /*   maximum number of gaussians   */
      integer:: mg_meta

!     /*   number of gaussians   */
      integer:: ng_meta

!     /*   number of collective variables   */
      integer:: nmeta

!-----------------------------------------------------------------------
!     /*   reference step for metadynamics                            */
!-----------------------------------------------------------------------

!     /*   reference step for harmonic potential:  number of steps   */
      integer :: nref_meta

!     /*   reference step for harmonic potential:  step size   */
      real(8) :: dt_ref_meta

!-----------------------------------------------------------------------
!     /*   type of collective variables                               */
!-----------------------------------------------------------------------

!     /*   number of types of collective variables   */
      integer:: ntype_meta = 8

!     /*   type of cv   */
      integer, dimension(:), allocatable:: itype_meta

!     /*   boundary condition of cv   */
      integer, dimension(:), allocatable:: ipbc_meta

!-----------------------------------------------------------------------
!     /*   starting options                                           */
!-----------------------------------------------------------------------

!     /*   starting option for collective variables   */
      integer:: icv_start

!     /*   starting option for hills   */
      integer:: ihills_start

!-----------------------------------------------------------------------
!     /*   collective variables                                       */
!-----------------------------------------------------------------------

!     /*   cv position   */
      real(8), dimension(:,:), allocatable:: rmeta

!     /*   cv velocity   */
      real(8), dimension(:,:), allocatable:: vmeta

!     /*   actual cv position   */
      real(8), dimension(:,:), allocatable:: smeta

!     /*   cv mass   */
      real(8), dimension(:),   allocatable:: fictmass_meta

!     /*   characteristic frequency of cv bath   */
      real(8) :: time_cv_bath

!     /*   characteristic angular frequency of cv   */
      real(8) :: omega_cv_bath

!-----------------------------------------------------------------------
!     /*   thermostats attached to collective variables               */
!-----------------------------------------------------------------------

!     /*   thermostat position attached to cv   */
      real(8), dimension(:,:,:,:), allocatable:: rbath_meta

!     /*   thermostat velocity attached to cv   */
      real(8), dimension(:,:,:,:), allocatable:: vbath_meta

!     /*   thermostat force attached to cv   */
      real(8), dimension(:,:,:,:), allocatable:: fbath_meta

!     /*   thermostat mass attached to cv   */
      real(8), dimension(:),       allocatable:: qmass_meta

!-----------------------------------------------------------------------
!     /*   energy components                                          */
!-----------------------------------------------------------------------

!     /*   kinetic energy of the system   */
      real(8):: ekin_sys

!     /*   kinetic energy of collective variables   */
      real(8):: ekin_meta

!     /*   gaussian potential   */
      real(8):: potential_meta

!     /*   harmonic potential    */
      real(8):: potential_ref_meta

!     /*   energy of thermostat attached to the system   */
      real(8):: ebath_sys

!     /*   energy of thermostat attached to cv   */
      real(8):: ebath_meta

!     /*   potential energy lifted by gaussians   */
      real(8):: pot_lifted

!     /*   harmonic potential, each bead    */
      real(8), dimension(:),   allocatable:: pot_ref_meta

!     /*   gaussian potential, each bead   */
      real(8), dimension(:),   allocatable:: pot_meta

!-----------------------------------------------------------------------
!     /*   force components                                           */
!-----------------------------------------------------------------------

!     /*   cv force: harmonic   */
      real(8), dimension(:,:), allocatable:: fref_meta

!     /*   force constant of harmonic potential   */
      real(8), dimension(:),   allocatable:: fc_meta

!     /*   cv force: gaussian   */
      real(8), dimension(:,:), allocatable:: fmeta

!-----------------------------------------------------------------------
!     /*   hills                                                      */
!-----------------------------------------------------------------------

!     /*   gaussian center   */
      real(8), dimension(:,:), allocatable:: gc_meta

!     /*   latest gaussian center   */
      real(8), dimension(:,:), allocatable:: gc_meta_save

!     /*   gaussian width   */
      real(8), dimension(:,:), allocatable:: gw_meta

!     /*   current value of gaussian width   */
      real(8), dimension(:),   allocatable:: gw_meta_save

!     /*   gaussian width of cv type   */
      real(8), dimension(:),   allocatable:: gw_meta_type

!     /*   gaussian height   */
      real(8), dimension(:),   allocatable:: gh_meta

!     /*   current value of gaussian height   */
      real(8):: gh_meta_save

!-----------------------------------------------------------------------
!     /*   displacement of collective variables                       */
!-----------------------------------------------------------------------

!     /*   maximum displacement of cv   */
      real(8):: dmeta_max = 1.5d0

!     /*   displacement of cv   */
      real(8), dimension(:), allocatable :: dmeta

!-----------------------------------------------------------------------
!     /*   print inverval and option                                  */
!-----------------------------------------------------------------------

!     /*   print interval for reconstruction of pes   */
      integer:: iprint_rec_meta

!     /*   print interval for energies   */
      integer:: iprint_meta

!     /*   print interval for xyz trajectory   */
      integer:: iprint_xyz

!     /*   print interval for collective variables trajectory   */
      integer:: iprint_cv_meta

!     /*   parameters for hills reconstruction   */
      real(8), dimension(:,:), allocatable :: params_rec_meta

!c     /*   dimension of hills reconstruction   */
!      integer :: ndim_rec_meta

!     /*   axes of hills reconstruction   */
      integer, dimension(:), allocatable :: idim_rec_meta(:)

!-----------------------------------------------------------------------
!     /*   options for metadynamics                                   */
!-----------------------------------------------------------------------

!     /*   time scale of cv: minimum time to fall off gaussian hill   */
      real(8) :: time_cv_meta

!     /*   period of harmonic term   */
      real(8) :: time_fc_meta

!     /*   time limit of adding gaussian   */
      real(8) :: time_limit_meta

!     /*   option   */
      integer :: ioption_meta

!     /*   option   */
      integer :: joption_meta

!-----------------------------------------------------------------------
!     /*   information for defining collective variables              */
!-----------------------------------------------------------------------

!     /*   atom i for geometric collective variables   */
      integer, dimension(:), allocatable:: i_meta

!     /*   atom j for geometric collective variables   */
      integer, dimension(:), allocatable:: j_meta

!     /*   atom k for geometric collective variables   */
      integer, dimension(:), allocatable:: k_meta

!     /*   atom l for geometric collective variables   */
      integer, dimension(:), allocatable:: l_meta

!     /*   rational function parameters for coordination number cv  */
      integer, dimension(:,:), allocatable:: nu_meta
      integer, dimension(:,:), allocatable:: mu_meta
      real(8), dimension(:,:), allocatable:: req_meta

!-----------------------------------------------------------------------
!     /*   other variables                                            */
!-----------------------------------------------------------------------

!     /*   time since last gaussian has been added   */
      real(8), dimension(:), allocatable :: time_meta_save

!     /*   angular frequency of harmonic term   */
      real(8) :: omega_meta

!-----------------------------------------------------------------------
!     /*   hills forces                                               */
!-----------------------------------------------------------------------

!     /*   hills forces in cartesian     */
      real(8), dimension(:,:), allocatable:: fux_meta
      real(8), dimension(:,:), allocatable:: fuy_meta
      real(8), dimension(:,:), allocatable:: fuz_meta

!     /*   hills forces in normal modes  */
      real(8), dimension(:,:), allocatable:: fx_meta
      real(8), dimension(:,:), allocatable:: fy_meta
      real(8), dimension(:,:), allocatable:: fz_meta

!-----------------------------------------------------------------------
!     /*   hills reconstruction                                       */
!-----------------------------------------------------------------------

!     /*   hills potential   */
      real(8), dimension(:),     allocatable :: pot_1d
      real(8), dimension(:,:),   allocatable :: pot_2d
      real(8), dimension(:,:,:), allocatable :: pot_3d

!     /*   cut off parameter   */
      real(8) :: cut_rec_3d = 10.d0

!     /*   list of meshes in 3D   */
      integer, dimension(:), allocatable :: j1_rec_3d
      integer, dimension(:), allocatable :: j2_rec_3d
      integer, dimension(:), allocatable :: j3_rec_3d

!     /*   number of meshes in 3D   */
      integer :: n1_rec_3d, n2_rec_3d, n3_rec_3d

!-----------------------------------------------------------------------
!     /*   temperature parameter in well tempered metadynamics        */
!-----------------------------------------------------------------------

      real(8) :: dtemp_meta

!-----------------------------------------------------------------------
!     /*   boundary parameters                                        */
!-----------------------------------------------------------------------

      real(8), dimension(:,:), allocatable :: params_meta

!-----------------------------------------------------------------------
!     /*   combination with hmc                                       */
!-----------------------------------------------------------------------

      real(8), dimension(:), allocatable :: ekin_bead
      real(8), dimension(:), allocatable :: epot_bead

      real(8), dimension(:,:), allocatable :: rmeta_save
      real(8), dimension(:,:), allocatable :: vmeta_save
      real(8), dimension(:,:), allocatable :: smeta_save
      real(8), dimension(:,:), allocatable :: fmeta_save
      real(8), dimension(:,:), allocatable :: fref_meta_save

      real(8), dimension(:), allocatable :: pot_meta_save
      real(8), dimension(:), allocatable :: pot_ref_meta_save
      real(8), dimension(:), allocatable :: ekin_bead_save
      real(8), dimension(:), allocatable :: epot_bead_save

!***********************************************************************
      end module meta_variables
!***********************************************************************





!***********************************************************************
      module rotor_variables
!***********************************************************************

!     //   number of molecules
      integer :: nmol

!     //   number of molecules with 6 dimensional degrees of freedom
      integer :: nmol_6

!     //   number of molecules with 5 dimensional degrees of freedom
      integer :: nmol_5

!     //   number of molecules with 3 dimensional degrees of freedom
      integer :: nmol_3

!     //   number of components
      integer :: ncomp

!     //   number of components with 6 dimensional degrees of freedom
      integer :: ncomp_6

!     //   number of components with 5 dimensional degrees of freedom
      integer :: ncomp_5

!     //   number of components with 3 dimensional degrees of freedom
      integer :: ncomp_3

!     //   maximum number of number of atoms per molecule
      integer :: natom_comp_max

!     //   number of molecules of a component
      integer, dimension(:), allocatable :: nmol_comp

!     //   number of atoms in a molecule
      integer, dimension(:), allocatable :: natom_per_mol

!     //   number of atoms in a component
      integer, dimension(:), allocatable :: natom_per_comp

!     /*   list of atoms in a molecule   */
      integer, dimension(:,:), allocatable :: list_atom_mol

!     //   name of compenent
      character(len=8), dimension(:), allocatable :: comp_spec

!     //   dimension of a compenent
      integer, dimension(:), allocatable :: ndim_comp

!     //   dimension of a molecule
      integer, dimension(:), allocatable :: ndim_mol

!     //   gear parameter: fifth order
      real(8), dimension(:), allocatable :: a5_gear

!     //   gear parameter: fourth order
      real(8), dimension(:), allocatable :: a4_gear

!     //   powers of dt: fifth order
      real(8), dimension(:), allocatable :: dt_gear

!     //   thermostat: fifth order
      real(8), dimension(:), allocatable :: s5

!     //   thermostat velocity
      real(8) :: zeta

!     //   cartesian coordinates
      real(8), dimension(:,:,:), allocatable :: r5

!     //   angular velocity
      real(8), dimension(:,:,:), allocatable :: o4

!     //   quaternion
      real(8), dimension(:,:,:), allocatable :: q4

!     //   velocity of quaternion
      real(8), dimension(:,:), allocatable :: qdot

!     //   coordinates in molecular axis
      real(8), dimension(:,:), allocatable :: x_mol
      real(8), dimension(:,:), allocatable :: y_mol
      real(8), dimension(:,:), allocatable :: z_mol

!     //   translational forces
      real(8), dimension(:), allocatable :: fgx
      real(8), dimension(:), allocatable :: fgy
      real(8), dimension(:), allocatable :: fgz

!     //   coordinates in molecular axis
      real(8), dimension(:,:), allocatable :: x_comp
      real(8), dimension(:,:), allocatable :: y_comp
      real(8), dimension(:,:), allocatable :: z_comp

!     //   moment of inertia
      real(8), dimension(:,:), allocatable :: pmi

!     //   torque
      real(8), dimension(:), allocatable :: fnx
      real(8), dimension(:), allocatable :: fny
      real(8), dimension(:), allocatable :: fnz

!     //   molecular mass
      real(8), dimension(:), allocatable :: physmass_mol

!     //   mass of thermostat
      real(8) :: qmass_rotor

!     //   energy of thermostat
      real(8) :: ebath_rotor

!     //   kinetic energy of molecular translation
      real(8), dimension(:), allocatable :: ekin_tra

!     //   kinetic energy of molecular rotation
      real(8), dimension(:), allocatable :: ekin_rot

!     //   instantaneous temperature of molecular translation
      real(8), dimension(:), allocatable :: temp_tra

!     //   instantaneous temperature of molecular rotation
      real(8), dimension(:), allocatable :: temp_rot

!     //   kinetic energy from vx, vy, vz
      real(8) :: ekin_2

!***********************************************************************
      end module rotor_variables
!***********************************************************************





!***********************************************************************
      module rehmc_variables
!***********************************************************************
!-----------------------------------------------------------------------
!     replica exchange hybrid monte carlo
!-----------------------------------------------------------------------

!     /*   step number of the inner md cycle   */
      integer :: jstep

!     /*   number of times accepted   */
      integer :: naccept_hmc = 0

!     /*   number of times rejected   */
      integer :: nreject_hmc = 0

!     /*   saved hamiltonian of each bead   */
      real(8) :: ratio_hmc = 0.d0

!     /*   instantaneous temperature of each bead   */
      real(8), dimension(:), allocatable :: ekin_bead

!     /*   kinetic energy of each bead   */
      real(8), dimension(:), allocatable :: temp_bead

!     /*   hamiltonian of each bead   */
      real(8), dimension(:), allocatable :: hamiltonian_bead

!     /*   barostat energy of each bead   */
      real(8), dimension(:), allocatable :: ebaro_bead

!     /*   beta times energy difference   */
      real(8), dimension(:), allocatable :: bfactor_bead

!     /*   temperature of each bead  */
      real(8), dimension(:), allocatable :: temperature_bead

!     /*   beta values of each bead  */
      real(8), dimension(:), allocatable :: beta_bead

!     /*   preserved coordinates   */
      real(8), dimension(:,:), allocatable ::  x_rehmc_last
      real(8), dimension(:,:), allocatable ::  y_rehmc_last
      real(8), dimension(:,:), allocatable ::  z_rehmc_last

!     /*   preserved potential   */
      real(8), dimension(:), allocatable   ::  pot_rehmc_last

!     /*   preserved box   */
      real(8), dimension(:,:,:), allocatable ::  box_rehmc_last

!     /*   preserved force   */
      real(8), dimension(:,:), allocatable ::  fx_rehmc_last
      real(8), dimension(:,:), allocatable ::  fy_rehmc_last
      real(8), dimension(:,:), allocatable ::  fz_rehmc_last

!     /*   saved coordinates   */
      real(8), dimension(:,:), allocatable :: x_save
      real(8), dimension(:,:), allocatable :: y_save
      real(8), dimension(:,:), allocatable :: z_save

!     /*   saved coordinates   */
      real(8), dimension(:,:), allocatable :: ux_save
      real(8), dimension(:,:), allocatable :: uy_save
      real(8), dimension(:,:), allocatable :: uz_save

!     /*   saved velocities   */
      real(8), dimension(:,:), allocatable :: vx_save
      real(8), dimension(:,:), allocatable :: vy_save
      real(8), dimension(:,:), allocatable :: vz_save

!     /*   saved forces   */
      real(8), dimension(:,:), allocatable :: fx_save
      real(8), dimension(:,:), allocatable :: fy_save
      real(8), dimension(:,:), allocatable :: fz_save

!     /*   saved potential   */
      real(8), dimension(:), allocatable :: pot_save

!     /*   saved hamiltonian   */
      real(8), dimension(:), allocatable :: hamiltonian_bead_save

!     /*   alchemical mixture potential   */
      real(8), dimension(:,:), allocatable :: pot_alchem_save

!     /*   alchemical mixture force   */
      real(8), dimension(:,:,:), allocatable :: fx_alchem_save
      real(8), dimension(:,:,:), allocatable :: fy_alchem_save
      real(8), dimension(:,:,:), allocatable :: fz_alchem_save

!     /*   saved box position   */
      real(8), dimension(:,:,:), allocatable :: box_bead_save

!     /*   saved box inverse  */
      real(8), dimension(:,:,:), allocatable :: boxinv_bead_save

!     /*   saved box volume   */
      real(8), dimension(:), allocatable     :: volume_bead_save

!     /*   saved box velocity   */
      real(8), dimension(:,:,:), allocatable :: vbox_bead_save

!     /*   saved virial   */
      real(8), dimension(:,:,:), allocatable :: vir_bead_save

!     /*   box info   */
      integer, dimension(:,:,:), allocatable :: mbox_save

!     /*   dt in fs   */
      real(8) :: dt_fs

!     /*   number of times accepted   */
      integer :: naccept_rem = 0

!     /*   number of times rejected   */
      integer :: nreject_rem = 0

!     /*   saved hamiltonian of each bead   */
      real(8) :: ratio_rem = 0.d0

!     /*   beta times energy difference   */
      real(8), dimension(:), allocatable :: bfactor_rem

!     /*   number of times accepted   */
      integer :: naccept_rex = 0

!     /*   number of times rejected   */
      integer :: nreject_rex = 0

!     /*   saved hamiltonian of each bead   */
      real(8) :: ratio_rex = 0.d0

!     /*   beta times energy difference   */
      real(8), dimension(:), allocatable :: bfactor_rex

!     /*   average potential   */
      real(8), dimension(:), allocatable :: pot_avg

!     /*   radial distribution   */
      real(8), dimension(:,:,:), allocatable :: rdf_bead

!     /*   radial distribution   */
      real(8), dimension(:,:,:), allocatable :: rdf_n_bead

!     /*   radial distribution   */
      real(8), dimension(:,:,:), allocatable :: rdf_g_bead

!     /*   average radial distribution   */
      real(8), dimension(:,:,:), allocatable :: rdf_bead_avg

!     /*   bond distance   */
      real(8), dimension(:,:), allocatable :: rdist

!     /*   mean square bond distance   */
      real(8), dimension(:,:), allocatable :: rdist2

!     /*   bond average   */
      real(8), dimension(:,:), allocatable :: rdist_avg

!     /*   mean square bond average   */
      real(8), dimension(:,:), allocatable :: rdist2_avg

      real(8) :: ratio_min_hmc
      real(8) :: ratio_max_hmc
      integer :: istep_min_hmc
      integer :: istep_max_hmc
      integer :: istep_mul_hmc
      integer :: istep_adjust_hmc

!     //   atomic exchange
      integer :: istep_ax_hmc

!-----------------------------------------------------------------------
!     replica exchange monte carlo
!-----------------------------------------------------------------------

!     //   atoms exchanged
      integer :: iatom_mc
      integer :: jatom_mc

!     //   acceptance of main cycle
      integer :: naccept_mc = 0
      integer :: nreject_mc = 0 
      real(8) :: ratio_mc = 0.d0

!     //   acceptance of subcycle
      integer :: naccept_mc_sub = 0
      integer :: nreject_mc_sub = 0
      real(8) :: ratio_mc_sub = 0.d0

!     //   dual: saved low level potential
      real(8), dimension(:), allocatable :: pot_low_save

!     //   dual: saved coordinates
      real(8), dimension(:,:), allocatable :: x_sub_save
      real(8), dimension(:,:), allocatable :: y_sub_save
      real(8), dimension(:,:), allocatable :: z_sub_save

!     //   dual: saved potential
      real(8), dimension(:), allocatable :: pot_sub_save

!     //   dual: number of steps for inner mc cycle
      integer :: nsubstep_ax

!***********************************************************************
      end module rehmc_variables
!***********************************************************************





!***********************************************************************
      module alchem_variables
!***********************************************************************

!     /*   print interval   */
      integer :: iprint_alc

!     /*   potential model of the reactant   */
      character(len=80) :: alchem_potential_a

!     /*   potential model of the reactant   */
      character(len=80) :: alchem_potential_b

!     /*   data directory of the reactant  */
      character(len=80) :: alchem_dat_dir_a

!     /*   data directory of the product   */
      character(len=80) :: alchem_dat_dir_b

!     /*   scratch directory of the reactant  */
      character(len=80) :: alchem_scr_dir_a

!     /*   scratch directory of the product   */
      character(len=80) :: alchem_scr_dir_b

!     /*   potential from the reactant calculation   */
      real(8), dimension(:), allocatable :: pot_a

!     /*   potential from the product calculation   */
      real(8), dimension(:), allocatable :: pot_b

!     /*   mixed potential   */
      real(8), dimension(:,:), allocatable :: pot_alchem

!     /*   mixed force   */
      real(8), dimension(:,:,:), allocatable :: fx_alchem
      real(8), dimension(:,:,:), allocatable :: fy_alchem
      real(8), dimension(:,:,:), allocatable :: fz_alchem

!     /*   force from the reactant calculation   */
      real(8), dimension(:,:), allocatable :: fx_a
      real(8), dimension(:,:), allocatable :: fy_a
      real(8), dimension(:,:), allocatable :: fz_a

!     /*   force from the product calculation   */
      real(8), dimension(:,:), allocatable :: fx_b
      real(8), dimension(:,:), allocatable :: fy_b
      real(8), dimension(:,:), allocatable :: fz_b

!     /*   virial from the product calculation   */
      real(8), dimension(:,:), allocatable :: vir_a

!     /*   virial from the product calculation   */
      real(8), dimension(:,:), allocatable :: vir_b

!     /*   ratio of the alchemical mixture   */
      real(8), dimension(:,:), allocatable :: ratio_alchem

!     /*   average potential   */
      real(8), dimension(:,:), allocatable :: pot_alchem_avg

!***********************************************************************
      end module alchem_variables
!***********************************************************************





!***********************************************************************
      module hess_variables
!***********************************************************************

!     /*   index   */
      integer, dimension(:), allocatable :: ihess_atom
      integer, dimension(:), allocatable :: ihess_xyz
      integer, dimension(:), allocatable :: ihess_pm
      integer :: ihess_natom

!***********************************************************************
      end module hess_variables
!***********************************************************************





!***********************************************************************
      module nma_variables
!***********************************************************************

!     /*   hessian of the first bead   */
      real(8), dimension(:,:), allocatable :: hess

!     /*   reduced mass   */
      real(8), dimension(:), allocatable   :: redmass

!     /*   eigen values   */
      real(8), dimension(:), allocatable   :: eigval

!     /*   eigen vectors   */
      real(8), dimension(:,:), allocatable :: eigvec

!     /*   position of the first bead   */
      real(8), dimension(:), allocatable :: r

!     /*   normal mode of the first bead   */
      real(8), dimension(:), allocatable :: qr

!     /*   print modes   */
      integer :: nstep_modes

!***********************************************************************
      end module nma_variables
!***********************************************************************





!***********************************************************************
      module phonon_variables
!***********************************************************************

!     /*   hessian of the primitive cell   */
      complex(8), dimension(:,:), allocatable :: hess_prim

!     /*   eigen values of the primitive cell   */
      complex(8), dimension(:), allocatable   :: eigval_prim

!     /*   eigen vectors of the primitive cell   */
      complex(8), dimension(:,:), allocatable :: eigvec_prim

!     /*   lattice vectors of the primitive cell   */
      real(8), dimension(3,3) :: box_prim

!     /*   inverse of the primitive cell   */
      real(8), dimension(3,3) :: boxinv_prim

!     /*   number of atoms in the primitive cell   */
      integer :: natom_prim

!     /*   number of primitive cells in a direction   */
      integer :: na_prim

!     /*   number of primitive cells in a direction   */
      integer :: nb_prim

!     /*   number of primitive cells in a direction   */
      integer :: nc_prim

!     /*   number of primitive cells in the box   */
      integer :: nabc

!     /*   cartesian coordinates of atoms in the primitive cell   */
      real(8), dimension(:), allocatable :: x_prim
      real(8), dimension(:), allocatable :: y_prim
      real(8), dimension(:), allocatable :: z_prim

!     /*   atom in the primitive cell   */
      integer, dimension(:), allocatable :: i_prim

!     /*   number of k-points in a, b, c directions   */
      integer, dimension(3) :: lmax_phonon

!     /*   total amount of k-point sampling   */
      integer :: labc_phonon

!     /*   option for phonon calculation   */
      integer :: ioption_phonon

!     /*   option for phonon calculation   */
      integer :: joption_phonon

!     /*   number of k-points for dispersion curve   */
      integer :: ndisp

!     /*   k-points for the dispersion curve   */
      real(8), dimension(:), allocatable :: gx_disp
      real(8), dimension(:), allocatable :: gy_disp
      real(8), dimension(:), allocatable :: gz_disp

!     /*   real values   */
      real(8) :: pdos_min, pdos_max, dpdos

!     /*   real values   */
      real(8), dimension(:), allocatable :: pdos

!     /*   integers   */
      integer :: npdos

!     /*   maximum, minimum values of temperature for phonon   */
      real(8) :: tempmin_phonon, tempmax_phonon, tempstep_phonon

!     /*   temperature and its inverse for phonon calculation   */
      real(8) :: temperature_phonon, beta_phonon

!***********************************************************************
      end module phonon_variables
!***********************************************************************





!***********************************************************************
      module string_variables
!***********************************************************************

!     /*   centroid potential of mean force   */
      real(8), dimension(:), allocatable :: pot_string

!     /*   arc length grids per nbead   */
      integer :: ngrid_string

!     /*   automatic step control   */
      character(len=4) :: auto_string

!     //   maximum value of dt
      real(8) :: dtmax_string

!     //   minimum value of dt
      real(8) :: dtmin_string

!     //   mean of force integral
      real(8) :: pot_string_mean

!     //   maximum of force integral
      real(8) :: pot_string_max

!     //   minimum of force integral
      real(8) :: pot_string_min

!     /*   total number of grids of arc length   */
      integer :: narc

!     /*   reaction coordinate   */
      real(8), dimension(:), allocatable :: rc_arc

!     /*   potential along reaction coordinate   */
      real(8), dimension(:), allocatable :: pot_arc

!-----------------------------------------------------------------------

!     /*   reference points for coordinate spline   */
      real(8), dimension(:,:), allocatable :: s_ref, r1_ref, r2_ref

!     /*   reference points for force spline   */
      real(8), dimension(:,:), allocatable :: f1_ref, f2_ref

!     /*   reference points for potential spline   */
      real(8), dimension(:,:), allocatable :: pot1_ref, pot2_ref

!     /*   reference points for arc length spline   */
      real(8), dimension(:,:), allocatable :: arc_ref, s1_ref, s2_ref

!-----------------------------------------------------------------------

!     /*   data at maximum   */
      real(8), dimension(:), allocatable :: xmax_string
      real(8), dimension(:), allocatable :: ymax_string
      real(8), dimension(:), allocatable :: zmax_string
      real(8), dimension(:), allocatable :: dmax_string

      integer :: i_string_max

!***********************************************************************
      end module string_variables
!***********************************************************************





!***********************************************************************
      module cons_variables
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   basic parameters for constraints                           */
!-----------------------------------------------------------------------

!     /*   number of constraints   */
      integer:: ncons

!-----------------------------------------------------------------------
!     /*   type of constraints                                        */
!-----------------------------------------------------------------------

!     /*   number of types of constraints   */
      integer:: ntype_cons = 8

!     /*   type of constraints   */
      integer, dimension(:), allocatable:: itype_cons

!-----------------------------------------------------------------------
!     /*   constraints                                                */
!-----------------------------------------------------------------------

!     /*   position   */
      real(8), dimension(:,:), allocatable:: rcons

!     /*   actual position   */
      real(8), dimension(:,:), allocatable:: scons

!     /*   average actual position   */
      real(8), dimension(:,:), allocatable:: scons_avg

!-----------------------------------------------------------------------
!     /*   energy components                                          */
!-----------------------------------------------------------------------

!     /*   harmonic potential    */
      real(8):: potential_ref_cons

!     /*   energy of thermostat attached to the system   */
      real(8):: ebath_sys

!     /*   harmonic potential, each bead    */
      real(8), dimension(:),   allocatable:: pot_ref_cons

!-----------------------------------------------------------------------
!     /*   force components                                           */
!-----------------------------------------------------------------------

!     /*   constraint force: harmonic   */
      real(8), dimension(:,:), allocatable:: fref_cons

!     /*   average constraint force   */
      real(8), dimension(:,:), allocatable:: fref_cons_avg

!     /*   average constraint force squared   */
      real(8), dimension(:,:,:), allocatable:: f2ref_cons_avg

!     /*   average constraint hessian   */
      real(8), dimension(:,:), allocatable:: href_cons_avg

!     /*   force constant of harmonic potential   */
      real(8), dimension(:),   allocatable:: fc_cons

!     /*   free energy as force integral  */
      real(8), dimension(:), allocatable:: vref_cons

!     /*   free energy with jacobian  */
      real(8), dimension(:), allocatable:: aref_cons

!     /*   force constant of harmonic potential   */
      real(8), dimension(:),   allocatable:: gc_cons

!-----------------------------------------------------------------------
!     /*   options for constraints                                    */
!-----------------------------------------------------------------------

!     /*   period of harmonic term   */
      real(8) :: time_fc_cons

!-----------------------------------------------------------------------
!     /*   information for defining constraints                       */
!-----------------------------------------------------------------------

!     /*   atom i for geometric constraints   */
      integer, dimension(:), allocatable:: i_cons

!     /*   atom j for geometric constraints   */
      integer, dimension(:), allocatable:: j_cons

!     /*   atom k for geometric constraints   */
      integer, dimension(:), allocatable:: k_cons

!     /*   atom l for geometric constraints   */
      integer, dimension(:), allocatable:: l_cons

!     /*   rational function parameters for coordination number   */
      integer, dimension(:,:), allocatable:: nu_cons
      integer, dimension(:,:), allocatable:: mu_cons
      real(8), dimension(:,:), allocatable:: req_cons

!     /*   parameter for constraints   */
      real(8), dimension(:), allocatable :: params_cons

!     /*   periodic boundary condition   */
      integer, dimension(:), allocatable:: ipbc_cons

!-----------------------------------------------------------------------
!     /*   for PIMD                                                   */
!-----------------------------------------------------------------------

!     /*   reference forces for PIMD   */
      real(8), dimension(:,:), allocatable :: fx_ref_cons
      real(8), dimension(:,:), allocatable :: fy_ref_cons
      real(8), dimension(:,:), allocatable :: fz_ref_cons

!     /*   cv centroid for PIMD   */
      real(8), dimension(:), allocatable ::  scons_cent

!-----------------------------------------------------------------------
!     /*   for GEOOPT                                                 */
!-----------------------------------------------------------------------

      integer :: iter_cons
      integer :: niter_cons = 3
      real(8) :: diter_cons = 4.d0

!***********************************************************************
      end module cons_variables
!***********************************************************************





!***********************************************************************
      module om_variables
!***********************************************************************

!     /*   initialize variables   */
      implicit none

!     /*   om action   */
      real(8) :: action_all_om

!     /*   om action   */
      real(8) :: action_om

!     /*   reference om action   */
      real(8) :: action_ref_om

!     /*   force with respect to om action   */
      real(8), dimension(:,:), allocatable :: fx_om
      real(8), dimension(:,:), allocatable :: fy_om
      real(8), dimension(:,:), allocatable :: fz_om

!     /*   force with respect to om action   */
      real(8), dimension(:,:), allocatable :: fx_ref_om
      real(8), dimension(:,:), allocatable :: fy_ref_om
      real(8), dimension(:,:), allocatable :: fz_ref_om

!     /*   potential part of om action   */
      real(8), dimension(:), allocatable :: v_om

!     /*   friction: input in 1/fs, converted to au.  */
      real(8) :: gamma_om

!     /*   time step: input in fs, converted to au.   */
      real(8) :: dt_om

!     /*   total time   */
      real(8) :: t_om

!     /*   vanden-eijnden finite difference method   */
      real(8)                              :: potential_0
      real(8), dimension(:),   allocatable :: pot_0
      real(8), dimension(:,:), allocatable :: fx_p, fy_p, fz_p
      real(8), dimension(:,:), allocatable :: fx_m, fy_m, fz_m
      real(8), dimension(:,:), allocatable :: fx_0, fy_0, fz_0

!     /*   finite difference   */
      real(8), dimension(:), allocatable   :: fdscale_om

!     /*   normal mode force with respect to om action   */
      real(8), dimension(:,:), allocatable :: fux_om
      real(8), dimension(:,:), allocatable :: fuy_om
      real(8), dimension(:,:), allocatable :: fuz_om

!     /*   normal mode force with respect to om action   */
      real(8), dimension(:,:), allocatable :: fux_ref_om
      real(8), dimension(:,:), allocatable :: fuy_ref_om
      real(8), dimension(:,:), allocatable :: fuz_ref_om

!***********************************************************************
      end module om_variables
!***********************************************************************





!***********************************************************************
      module smash_variables
!***********************************************************************
!=======================================================================
!
!     This module is for PIMD-SMASH interface
!
!=======================================================================

!     /*   local variables   */
      implicit none

!     /*   extrapolation order of lcao coefficients   */
      integer :: ncmo_s

!     /*   energies   */
      real(8) :: ener_s

!     /*   gradients   */
      real(8), dimension(:,:), allocatable :: grad_s

!     /*   lcao coefficients for alpha orbitals   */
      real(8), dimension(:,:,:,:), allocatable :: cmo_s

!     /*   lcao coefficients for beta orbitals   */
      real(8), dimension(:,:,:,:), allocatable :: cmob_s

!     /*   number of beads per processor   */
      integer :: nbead_s

!     /*   extrapolation coefficients   */
      real(8), dimension(:), allocatable :: coeff_s

!     /*   number of atoms used in SMASH   */
      integer :: natom_smash

!     /*   dipole moment   */
      real(8) :: dipx_s, dipy_s, dipz_s

!     /*   point charges of qm image   */
      real(8), dimension(:), allocatable :: znuc2_s

!***********************************************************************
      end module smash_variables
!***********************************************************************





!***********************************************************************
      module afed_variables
!***********************************************************************
!=======================================================================
!
!     This module is for adiabatic free energy dynamics
!
!=======================================================================

!     /*   afed stage   */
      character(len=2) :: afed_status

!     /*   number of steps   */
      integer :: niter_afed

!     /*   step number   */
      integer :: iiter_afed

!     /*   first step   */
      integer :: iiter_start_afed

!     /*   last step   */
      integer :: iiter_end_afed

!     /*   number of molecular dynamics steps in preliminary run   */
      integer :: nstep_pre_afed

!     /*   number of molecular dynamics steps in productive run   */
      integer :: nstep_pro_afed

!     /*   cv   */
      real(8), dimension(:), allocatable :: rafed

!     /*   mean force   */
      real(8), dimension(:), allocatable :: f_afed

!     /*   old mean force   */
      real(8), dimension(:), allocatable :: f_old_afed

!     /*   mean hessian   */
      real(8), dimension(:,:), allocatable :: hess_afed

!     /*   old mean hessian   */
      real(8), dimension(:,:), allocatable :: hess_old_afed

!     /*   eigenvalues   */
      real(8), dimension(:), allocatable :: eigval_afed

!     /*   eigenvectors   */
      real(8), dimension(:,:), allocatable :: eigvec_afed

!     /*   free energy   */
      real(8) :: fenergy_afed

!     /*   unit vector   */
      real(8), dimension(:), allocatable :: e_afed

!     //   direction of current step
      real(8), dimension(:), allocatable :: d_afed

!     //   cv velocity of current step
      real(8), dimension(:), allocatable :: v_afed

!     /*   step parameter, finite difference parameter   */
      real(8), dimension(:,:), allocatable :: params_afed

!     /*   shift vector   */
      real(8), dimension(:), allocatable :: he_afed

!     /*   step size   */
      real(8) :: dt_afed

!     /*   step size for ascent search   */
      real(8) :: dt_ascent_afed

!     /*   step size for descent search   */
      real(8) :: dt_descent_afed

!     /*   step size at convergence   */
      real(8) :: dt_conv_afed

!     /*   gamma of ascent search   */
      real(8) :: gamma_ascent_afed

!     /*   fictitious mass   */
      real(8), dimension(:), allocatable :: fictmass_afed

!     /*   mean cv value   */
      real(8), dimension(:), allocatable :: scons_mean_afed

!     /*   angle between two last cv velocities   */
      real(8) :: xi_afed

!     /*   convergence flag   */
      integer :: iconv_afed

!     /*   convergence flag   */
      integer :: jconv_afed

!     /*   free energy of previous step   */
      real(8) :: fenergy_old_afed

!     //   position of the previous step
      real(8), dimension(:), allocatable :: rafed_old

!     //   direction of previous step
      real(8), dimension(:), allocatable :: d_old_afed

!     //   unit vector of previous step
      real(8), dimension(:), allocatable :: e_old_afed

!     //   cv velocity of previous step
      real(8), dimension(:), allocatable :: v_old_afed

!     //   cv velocity of current step
      real(8), dimension(:), allocatable :: v_save_afed

!     //   average position of the previous step
      real(8), dimension(:,:), allocatable :: scons_avg_old_afed

!     /*   norm of shift vector   */
      real(8) :: hehe_afed

!     /*   inner product of unit vector and shift vector   */
      real(8) :: ehe_afed

!     /*   maximum norm of shift vector   */
      real(8), parameter :: demax_ascent_afed = 0.25d0

!     /*   gad or evf   */
      character(len=4) :: algo_ascent_afed

!     /*   interval of explicit hessian computation in gad   */
      integer :: niter_refresh_afed

!     /*   iterations when switching gad to evf   */
      integer :: niter_gad2evf_afed

!     /*   for hessian update   */
      real(8), dimension(:),   allocatable :: r_old_afed
      real(8), dimension(:),   allocatable :: g_old_afed
      real(8), dimension(:,:), allocatable :: h_old_afed
      real(8), dimension(:),   allocatable :: r_new_afed
      real(8), dimension(:),   allocatable :: g_new_afed
      real(8), dimension(:,:), allocatable :: h_new_afed

!-----------------------------------------------------------------------
!     options
!-----------------------------------------------------------------------

!     /*   damping factor of step size   */
      real(8) :: dt_damp_afed

!     /*   maximum free energy criterion   */
      real(8) :: fenergy_max_afed

!     /*   finite difference parameter   */
      real(8) :: fdiff_sampling_afed

!     /*   bead sampling for ascent: ANALYTICAL or NUMERICAL   */
      character(len=11) :: ascent_sampling_afed

!     /*   print interval in convergence test   */
      integer :: iprint_test_afed

!     /*   print interval of trajectory   */
      integer :: iprint_xyz_afed

!-----------------------------------------------------------------------
!     variables for automated search
!-----------------------------------------------------------------------

!     /*   maximum number of consecutive misses of automated search   */
      integer :: nmiss_auto_afed

!     /*   maximum number of shots of automated search  */
      integer :: nshot_auto_afed

!     /*   radius criterion   */
      real(8) :: radius_auto_afed

!     //   root position of current step
      integer :: iroot_auto_afed

!     //   flag of auto
      logical :: finished_auto_afed

!     //   flag of auto
      logical :: restarted_auto_afed

!-----------------------------------------------------------------------
!     tamd and logmfd
!-----------------------------------------------------------------------

!     /*   step size for tamd   */
      real(8) :: dt_tamd

!     /*   step size for logmfd   */
      real(8) :: dt_logmfd

!     /*   temperature for tamd   */
      real(8) :: temperature_tamd

!     /*   hamiltonian for tamd and logmfd   */
      real(8) :: hamiltonian_afed

!     /*   kinetic energy for tamd and logmfd   */
      real(8) :: ekin_afed

!     /*   instantaneous temperature for tamd and logmfd   */
      real(8) :: temp_afed

!     /*   potential for tamd and logmfd   */
      real(8) :: potential_afed

!     /*   bath energy for tamd and logmfd   */
      real(8) :: ebath_afed

!     //   multiple time step for thermostat attached to cv
      integer :: nys_afed

!     /*   gkt of cv   */
      real(8) :: gkt_afed

!     /*   gnkt of cv   */
      real(8) :: gnkt_afed

!     //   thermostat position attached to cv
      real(8), dimension(:), allocatable :: rbath_afed

!     //   thermostat velocity attached to cv
      real(8), dimension(:), allocatable :: vbath_afed

!     //   thermostat force attached to cv
      real(8), dimension(:), allocatable :: fbath_afed

!     //   thermostat mass attached to cv
      real(8), dimension(:), allocatable :: qmass_afed

!     /*   characteristic time scale of cv   */
      real(8) :: time_bath_afed

!     /*   parameter associated to thermostat mass attached to cv   */
      integer :: niter_bath_afed

!     //   free energy surface defined by user: number of gaussians
      integer :: ng_user_afed

!     //   free energy surface defined by user: heights of gaussians
      real(8), dimension(:), allocatable :: gh_user_afed

!     //   free energy surface defined by user: widths of gaussians
      real(8), dimension(:,:), allocatable :: gw_user_afed

!     //   free energy surface defined by user: centers of gaussians
      real(8), dimension(:,:), allocatable :: gc_user_afed

!     //   initial five iterations
      integer, parameter :: iiter_skip_afed = 5

!     //   initial change of free energy value
      real(8), parameter :: dfenergy_init_afed = 0.00025d0

!     //   initial free energy value
      real(8) :: fenergy_init_afed

!     //   option for eigenvalue calculations
      integer :: ioption_eigen_afed

!     //   molecular dynamics cycle for productive runs: NVE or NVT
      character(len=4) :: mdcycle_pro_afed

!     /*   alpha parameter for logmfd  */
      real(8) :: alpha_logmfd

!     /*   gamma parameter for logmfd  */
      real(8) :: gamma_logmfd

!     /*   reference free energy for logmfd   */
      real(8) :: fenergy_ref_afed

!     /*   reference hamiltonian for logmfd   */
      real(8) :: hamiltonian_ref_afed

!     /*   replica work of cv   */
      real(8), dimension(:), allocatable :: work_afed

!     /*   replica weight of cv   */
      real(8), dimension(:), allocatable :: weight_afed

!     /*   replica weight sum of cv   */
      real(8) :: weight_sum_afed

!     /*   initial free energy for logmfd   */
      real(8) :: fenergy_ini_logmfd

!     /*   initial free energy for tamd   */
      real(8) :: fenergy_ini_tamd

!     /*   potential of velocity scaling for logmfd   */
      real(8) :: phi_logmfd

!     /*   potential of velocity scaling for tamd   */
      real(8) :: phi_tamd

!     /*   parameter of velocity scaling for logmfd   */
      real(8) :: scale_logmfd

!     /*   parameter of velocity scaling for tamd   */
      real(8) :: scale_tamd

!     //   weighting of mean force for afed   */
      integer :: ioption_afed

!     /*   intiial cv positions and velocities   */
      character(len=9) :: start_afed

!     /*   intiial thermostat positions and velocities for cv   */
      character(len=9) :: start_therm_afed

!     /*   intiial cv mass   */
      character(len=9) :: mass_afed

!***********************************************************************
      end module afed_variables
!***********************************************************************





!***********************************************************************
      module tass_variables
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   basic parameters for tassdynamics                          */
!-----------------------------------------------------------------------

!     /*   maximum number of gaussians   */
      integer:: mg_meta

!     /*   number of gaussians   */
      integer:: ng_meta

!     /*   number of collective variables   */
      integer:: ntass

!     /*   number of collective variables for afed   */
      integer:: ntass_afed

!     /*   number of afed collective variables for constrained md  */
      integer:: ntass_cons

!     /*   number of afed collective variables for metadynamics  */
      integer:: ntass_meta

!     /*   cv dimensions for reconstruction   */
      integer:: ntass_rec

!-----------------------------------------------------------------------
!     /*   reference step for tassdynamics                            */
!-----------------------------------------------------------------------

!     /*   reference step for harmonic potential:  number of steps   */
      integer :: nref_tass

!     /*   reference step for harmonic potential:  step size   */
      real(8) :: dt_ref_tass

!-----------------------------------------------------------------------
!     /*   type of collective variables                               */
!-----------------------------------------------------------------------

!     /*   number of types of collective variables   */
      integer:: ntype_tass = 8

!     /*   type of cv   */
      integer, dimension(:), allocatable:: itype_tass

!     /*   kind of cv   */
      integer, dimension(:), allocatable:: ikind_tass

!     /*   boundary condition of cv   */
      integer, dimension(:), allocatable:: ipbc_tass

!     /*   cvs for reconstruction  */
      integer, dimension(:), allocatable :: jtass_rec

!-----------------------------------------------------------------------
!     /*   starting options                                           */
!-----------------------------------------------------------------------

!     /*   starting option for collective variables   */
      integer:: icv_start

!     /*   starting option for hills   */
      integer:: ihills_start

!     /*   starting option for constraints   */
      integer:: icons_start

!-----------------------------------------------------------------------
!     /*   collective variables                                       */
!-----------------------------------------------------------------------

!     /*   cv position   */
      real(8), dimension(:,:), allocatable:: rtass

!     /*   cv velocity   */
      real(8), dimension(:,:), allocatable:: vtass

!     /*   actual cv position   */
      real(8), dimension(:,:), allocatable:: stass

!     /*   cv mass   */
      real(8), dimension(:),   allocatable:: fictmass_tass

!     /*   characteristic frequency of cv bath   */
      real(8) :: time_cv_bath

!     /*   characteristic angular frequency of cv   */
      real(8) :: omega_cv_bath

!-----------------------------------------------------------------------
!     /*   thermostats attached to collective variables               */
!-----------------------------------------------------------------------

!     /*   thermostat position attached to cv   */
      real(8), dimension(:,:,:,:), allocatable:: rbath_tass

!     /*   thermostat velocity attached to cv   */
      real(8), dimension(:,:,:,:), allocatable:: vbath_tass

!     /*   thermostat force attached to cv   */
      real(8), dimension(:,:,:,:), allocatable:: fbath_tass

!     /*   thermostat mass attached to cv   */
      real(8), dimension(:),       allocatable:: qmass_tass

!-----------------------------------------------------------------------
!     /*   energy components                                          */
!-----------------------------------------------------------------------

!     /*   kinetic energy of the system   */
      real(8):: ekin_sys

!     /*   kinetic energy of collective variables   */
      real(8):: ekin_tass

!     /*   gaussian potential   */
      real(8):: potential_tass

!     /*   harmonic potential    */
      real(8):: potential_ref_tass

!     /*   energy of thermostat attached to the system   */
      real(8):: ebath_sys

!     /*   energy of thermostat attached to cv   */
      real(8):: ebath_tass

!     /*   potential energy lifted by gaussians   */
      real(8):: pot_lifted

!     /*   harmonic potential, each bead    */
      real(8), dimension(:),   allocatable:: pot_ref_tass

!     /*   gaussian potential, each bead   */
      real(8), dimension(:),   allocatable:: pot_tass

!-----------------------------------------------------------------------
!     /*   force components                                           */
!-----------------------------------------------------------------------

!     /*   cv force: harmonic   */
      real(8), dimension(:,:), allocatable:: fref_tass

!     /*   force constant of harmonic potential   */
      real(8), dimension(:),   allocatable:: fc_tass

!     /*   cv force: gaussian   */
      real(8), dimension(:,:), allocatable:: ftass

!-----------------------------------------------------------------------
!     /*   hills                                                      */
!-----------------------------------------------------------------------

!     /*   gaussian center   */
      real(8), dimension(:,:), allocatable:: gc_meta

!     /*   latest gaussian center   */
      real(8), dimension(:,:), allocatable:: gc_meta_save

!     /*   gaussian width   */
      real(8), dimension(:,:), allocatable:: gw_meta

!     /*   current value of gaussian width   */
      real(8), dimension(:),   allocatable:: gw_meta_save

!     /*   gaussian width of cv type   */
      real(8), dimension(:),   allocatable:: gw_meta_type

!     /*   gaussian height   */
      real(8), dimension(:),   allocatable:: gh_meta

!     /*   current value of gaussian height   */
      real(8):: gh_meta_save

!-----------------------------------------------------------------------
!     /*   displacement of collective variables                       */
!-----------------------------------------------------------------------

!     /*   maximum displacement of cv   */
      real(8):: dmeta_max = 1.5d0

!-----------------------------------------------------------------------
!     /*   print inverval and option                                  */
!-----------------------------------------------------------------------

!     /*   print interval for reconstruction of pes   */
      integer:: iprint_rec_tass

!     /*   print interval for energies   */
      integer:: iprint_tass

!     /*   print interval for xyz trajectory   */
      integer:: iprint_xyz

!     /*   print interval for collective variables trajectory   */
      integer:: iprint_cv_tass

!     /*   parameters for hills reconstruction   */
      real(8), dimension(:,:), allocatable :: params_rec_meta

!c     /*   dimension of hills reconstruction   */
!      integer :: ndim_rec_tass

!     /*   axes of hills reconstruction   */
      integer, dimension(:), allocatable :: idim_rec_tass(:)

!-----------------------------------------------------------------------
!     /*   options for tassdynamics                                   */
!-----------------------------------------------------------------------

!     /*   time scale of cv: minimum time to fall off gaussian hill   */
      real(8) :: time_cv_meta

!     /*   period of harmonic term   */
      real(8) :: time_fc_meta

!     /*   time limit of adding gaussian   */
      real(8) :: time_limit_meta

!     /*   option   */
      integer :: ioption_meta

!     /*   option   */
      integer :: joption_meta

!-----------------------------------------------------------------------
!     /*   information for defining collective variables              */
!-----------------------------------------------------------------------

!     /*   atom i for geometric collective variables   */
      integer, dimension(:), allocatable:: i_tass

!     /*   atom j for geometric collective variables   */
      integer, dimension(:), allocatable:: j_tass

!     /*   atom k for geometric collective variables   */
      integer, dimension(:), allocatable:: k_tass

!     /*   atom l for geometric collective variables   */
      integer, dimension(:), allocatable:: l_tass

!     /*   rational function parameters for coordination number cv  */
      integer, dimension(:,:), allocatable:: nu_tass
      integer, dimension(:,:), allocatable:: mu_tass
      real(8), dimension(:,:), allocatable:: req_tass

!-----------------------------------------------------------------------
!     /*   other variables                                            */
!-----------------------------------------------------------------------

!     /*   time since last gaussian has been added   */
      real(8), dimension(:), allocatable :: time_tass_save

!     /*   angular frequency of harmonic term   */
      real(8) :: omega_tass

!-----------------------------------------------------------------------
!     /*   hills forces                                               */
!-----------------------------------------------------------------------

!     /*   hills forces in cartesian     */
      real(8), dimension(:,:), allocatable:: fux_tass
      real(8), dimension(:,:), allocatable:: fuy_tass
      real(8), dimension(:,:), allocatable:: fuz_tass

!     /*   hills forces in normal modes  */
      real(8), dimension(:,:), allocatable:: fx_tass
      real(8), dimension(:,:), allocatable:: fy_tass
      real(8), dimension(:,:), allocatable:: fz_tass

!-----------------------------------------------------------------------
!     /*   temperature parameter in well tempered tassdynamics        */
!-----------------------------------------------------------------------

      real(8) :: dtemp_meta

!-----------------------------------------------------------------------
!     /*   boundary parameters                                        */
!-----------------------------------------------------------------------

      real(8), dimension(:,:), allocatable :: params_meta

!-----------------------------------------------------------------------
!     /*   tass parameters for each cv type and cv kind               */
!-----------------------------------------------------------------------

      real(8), dimension(:,:), allocatable :: params_tass

!-----------------------------------------------------------------------
!     /*   number of molecular dynamics steps in productive run       */
!-----------------------------------------------------------------------

      integer :: nstep_pro_afed

!-----------------------------------------------------------------------
!     /*   averages                                                   */
!-----------------------------------------------------------------------

      real(8) :: dstep_tass

      real(8), dimension(:,:), allocatable :: stass_avg

      real(8), dimension(:,:), allocatable :: fref_tass_avg

!-----------------------------------------------------------------------
!     /*   tamd temperature for tass                                  */
!-----------------------------------------------------------------------

      real(8) :: temp_tamd_tass

!-----------------------------------------------------------------------
!     /*   gkt for tass                                               */
!-----------------------------------------------------------------------

      real(8) :: gkt_tass

!-----------------------------------------------------------------------
!     /*   step interval of gaussian deposition                       */
!-----------------------------------------------------------------------

      integer :: iprint_hills_tass

!-----------------------------------------------------------------------
!     /*   hills reconstruction                                       */
!-----------------------------------------------------------------------

!     //   gamma parameter in well tempered metadynamics
      real(8) :: gamma_meta

!     //   collective variables
      real(8), dimension(:,:), allocatable :: rcv

!     //   force on collective variables
      real(8), dimension(:,:), allocatable :: fref_cv

!     //   gaussian center
      real(8), dimension(:,:), allocatable :: gc_cv

!     //   gaussian height
      real(8), dimension(:), allocatable :: gh_cv

!     //   gaussian width
      real(8), dimension(:,:), allocatable :: gw_cv

!     //   bias potential
      real(8), dimension(:), allocatable :: vb_cv

!     //   c function
      real(8), dimension(:), allocatable :: ct_cv

!     //   hills potential at mesh (0d case)
      real(8) :: vhills_0d

!     //   hills potential at mesh (1d case)
      real(8), dimension(:), allocatable :: vhills_1d

!     //   hills potential at mesh (2d case)
      real(8), dimension(:,:), allocatable :: vhills_2d

!     //   hills potential at mesh (3d case)
      real(8), dimension(:,:,:), allocatable :: vhills_3d

!     //   probability distribution from metadynamics (1d case)
      real(8), dimension(:,:), allocatable :: pc_1d

!     //   probability distribution from metadynamics (2d case)
      real(8), dimension(:,:,:), allocatable :: pc_2d

!     //   probability distribution from metadynamics (3d case)
      real(8), dimension(:,:,:,:), allocatable :: pc_3d

!     //   probability distribution (1d case)
      real(8), dimension(:), allocatable :: prob_1d

!     //   probability distribution (2d case)
      real(8), dimension(:,:), allocatable :: prob_2d

!     //   probability distribution (3d case)
      real(8), dimension(:,:,:), allocatable :: prob_3d

!     //   free energy gradient from constrained md
      real(8), dimension(:), allocatable :: fegrad_cons

!     //   free energy component from constrained md
      real(8), dimension(:), allocatable :: fenergy_cons

!     //   free energy (1d case)
      real(8), dimension(:), allocatable :: fenergy_1d

!     //   free energy (2d case)
      real(8), dimension(:,:), allocatable :: fenergy_2d

!     //   free energy (3d case)
      real(8), dimension(:,:,:), allocatable :: fenergy_3d

!     //   number of cv steps
      integer :: nstep_cv

!     //   number of hills steps
      integer :: nstep_hills

!     /*   minimum value of mesh   */
      real(8), dimension(:), allocatable :: rmin_cv

!     /*   maximum value of mesh   */
      real(8), dimension(:), allocatable :: rmax_cv

!     /*   mesh size   */
      real(8), dimension(:), allocatable :: dr_cv

!     /*   number of meshes   */
      integer, dimension(:), allocatable :: nmesh_cv

!     /*   inverse of temperature   */
      real(8) :: beta_tamd

!     /*   inverse of temperature   */
      real(8) :: beta_a_tass

!     /*   inverse of temperature   */
      real(8) :: beta_b_tass

!     /*   label of metadynamics cv   */
      integer, dimension(:), allocatable :: label_meta_tass

!     /*   label of constrained cv   */
      integer, dimension(:), allocatable :: label_cons_tass

!     /*   label of tamd cv   */
      integer, dimension(:), allocatable :: label_afed_tass

!***********************************************************************
      end module tass_variables
!***********************************************************************





!***********************************************************************
      module qe_variables
!***********************************************************************

!-----------------------------------------------------------------------
!     general variables
!-----------------------------------------------------------------------

!     /*   current number of MD steps   */
      integer :: pimd_istep

!     /*   current bead   */
      integer :: pimd_ibead

!     /*   if 1, all processess generate output file   */
      integer :: qe_output_all_proc

!     /*   maximum number of MD steps sent to QE (= nstep in PIMD) */
      integer :: nstep_qe

!     /*   QE output results every n-step   */
      integer :: qe_output_every_nstep

!     /*   input file name  */
      character(len=80) :: qe_input_file_name

!     /*   flag_no_output  */
      logical :: flag_no_output = .false.

!-----------------------------------------------------------------------
!     energy, coordinate, and force
!-----------------------------------------------------------------------

!     /*    total energy (au)   */
      real(8) :: total_energy

!     /*    nuclear coordinate (au,cartesian)   */
      real(8), allocatable :: coord_x(:), coord_y(:), coord_z(:)

!     /*    force (au)   */
      real(8), allocatable :: force_x(:), force_y(:), force_z(:)

!     /*    stress tensor (au)   */
      real(8) :: stress_tensor(3,3)

!     /*    stress tensor (Ry/bohr**3) */
      real(8) :: stress_tensor_ry(3,3)

!     /*    cell parameter       */
      real(8) :: cell_param(3,3)

!-----------------------------------------------------------------------
!     wavefunction information
!-----------------------------------------------------------------------

!     /*   qe wavefunction type   */
      type qewave

!     /*   flag whether empty or not   */
      logical :: empty

!     /*   wavefunction CPTWFP(NRPLWV,NBANDS,NKPTS,ISPIN)   */
      complex(SELECTED_REAL_KIND(10)), pointer :: CPTWFP(:,:,:,:)

!     /*   qe wavefunction type   */
      end type qewave

!     /* wavefunctions for all beads   */
      type(qewave), allocatable, target :: vwaves(:)

!     /* wavefunctions for current bead   */
      type(qewave), pointer :: vwave

!***********************************************************************
      end module qe_variables
!***********************************************************************





!***********************************************************************
      module phase0_variables
!***********************************************************************

!-----------------------------------------------------------------------
!     general variables
!-----------------------------------------------------------------------

!     /*   current number of MD steps   */
      integer :: pimd_istep

!     /*   current bead   */
      integer :: pimd_ibead

!-----------------------------------------------------------------------
!     energy, coordinate, and force
!-----------------------------------------------------------------------

!     /*    total energy (au)   */
      real(8) :: total_energy

!     /*    nuclear coordinate (au,cartesian)   */
      real(8), allocatable :: coord_x(:), coord_y(:), coord_z(:)

!     /*    force (au)   */
      real(8), allocatable :: force_x(:), force_y(:), force_z(:)

!     /*    stress tensor (au)   */
      real(8) :: stress_tensor(3,3)

!     /*    cell parameter       */
      real(8) :: cell_param(3,3)

!     /*    parallelization option */
      integer :: ne=0, nk=0, ng=0

!***********************************************************************
      end module phase0_variables
!***********************************************************************





!***********************************************************************
      module polymers_variables
!***********************************************************************

!     /*   parallel number for polymers   */
      integer :: np_poly         = 1

!     /*   parallel number for pimd   */
      integer :: np_pimd         = 1

!     /*   cycles of pimd per step  */
      integer :: np_sweep        = 1

!     /*   mpi top group   */
      integer :: mpi_group_top   = 0

!     /*   mpi communicator of top group   */
      integer :: mpi_comm_top    = 0

!     /*   mpi rank of top group   */
      integer :: myrank_top      = 0

!     /*   mpi number of processors of top group   */
      integer :: nprocs_top      = 1

!-----------------------------------------------------------------------

!     /*   number of polymers   */
      integer :: npoly = 1

!     /*   number of cycles   */
      integer :: ncycle_poly

!     /*   cycle number   */
      integer :: icycle_poly

!     /*   starting cycle number   */
      integer :: icycle_poly_start

!     /*   ending cycle number   */
      integer :: icycle_poly_end

!     /*   centroid coordinates   */
      real(8), dimension(:,:), allocatable :: xc_poly
      real(8), dimension(:,:), allocatable :: yc_poly
      real(8), dimension(:,:), allocatable :: zc_poly

!     /*   centroid force   */
      real(8), dimension(:,:), allocatable :: fxc_poly
      real(8), dimension(:,:), allocatable :: fyc_poly
      real(8), dimension(:,:), allocatable :: fzc_poly

!     /*   tangent vector   */
      real(8), dimension(:,:), allocatable :: txc_poly
      real(8), dimension(:,:), allocatable :: tyc_poly
      real(8), dimension(:,:), allocatable :: tzc_poly

!     /*   centroid potential of mean force   */
      real(8), dimension(:),   allocatable :: pmfc_poly

!     /*   kinetic energy   */
      real(8), dimension(:), allocatable :: ekinvir_poly

!     /*   potential   */
      real(8), dimension(:), allocatable :: epot_poly

!     /*   entropy   */
      real(8), dimension(:), allocatable :: entropy_poly

!     /*   file unit for standard output   */
      integer :: iounit_poly      = 101

!     /*   file unit for convergence test   */
      integer :: iounit_rest_poly = 102

!     /*   print interval of average centroid force   */
      integer :: iprint_poly

!     /*   option   */
      character(len=10) :: ends_poly

!     /*   option   */
      character(len=10) :: projcmf_poly

!     /*   option   */
      character(len=10) :: guess_poly

!     /*   arc length grids per npoly   */
      integer :: ngrid_poly

!     /*   total number of grids of arc length   */
      integer :: narc

!     /*   step size in string method   */
      real(8) :: dt_poly

!     /*   polymer id   */
      integer :: jpoly

!     /*   step   */
      integer, dimension(:), allocatable :: istep_poly

!-----------------------------------------------------------------------

!     /*   reference points for coordinate spline   */
      real(8), dimension(:,:), allocatable :: s_ref, r1_ref, r2_ref

!     /*   reference points for force spline   */
      real(8), dimension(:,:), allocatable :: f1_ref, f2_ref

!     /*   reference points for arc length spline   */
      real(8), dimension(:,:), allocatable :: arc_ref, s1_ref, s2_ref

!-----------------------------------------------------------------------

!     /*   flag if cons.ini exists   */
      integer :: istart_cons_poly

!     /*   ideal cv positions   */
      real(8), dimension(:,:), allocatable :: rcons_poly

!     /*   actual cv positions   */
      real(8), dimension(:,:), allocatable :: scons_poly

!     /*   cv forces   */
      real(8), dimension(:,:), allocatable :: fcons_poly

!-----------------------------------------------------------------------

!     /*   reversible work   */
      real(8), dimension(:),   allocatable :: pmfc_arc

!     /*   reaction coordinate   */
      real(8), dimension(:),   allocatable :: rc_arc

!***********************************************************************
      end module polymers_variables
!***********************************************************************





!***********************************************************************
      module mtp_variables
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   Moment Tensor Potential                                    */
!-----------------------------------------------------------------------

      integer :: iset_mtp
      integer :: species_count
      integer :: radial_funcs_count
      integer :: alpha_moments_count
      integer :: alpha_index_basic_count
      integer :: alpha_index_times_count
      integer :: alpha_scalar_moments
      integer :: max_alpha_moment_mapping
      integer :: max_alpha_index_basic
      real(8) :: scaling
      real(8) ::min_dist
      real(8) ::max_dist
      integer, dimension(:,:), allocatable :: alpha_index_basic
      integer, dimension(:,:), allocatable :: alpha_index_times
      integer, dimension(:), allocatable :: alpha_moment_mapping
      real(8), dimension(:,:,:,:), allocatable :: radial_coeffs
      real(8), dimension(:), allocatable :: species_coeffs  
      real(8), dimension(:), allocatable :: moment_coeffs
      
      integer, dimension(3) :: nbox_mtp
      real(8), dimension(3,3) :: bigbox_mtp, bigboxinv_mtp

!***********************************************************************
      end module mtp_variables
!***********************************************************************
