!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     read atomic species
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine calc_read_species
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use calc_variables, only : input_style

!-----------------------------------------------------------------------
!     /*   option of initial geometry file                            */
!-----------------------------------------------------------------------

      if      ( input_style(1:4) .eq. 'OLD ' ) then

         call calc_read_species_old

      else if ( input_style(1:4) .eq. 'NEW ' ) then

         call calc_read_species_new

      end if

      return
      end





!***********************************************************************
      subroutine calc_read_species_old
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use calc_variables, only : &
     &   au_mass, amu_mass, physmass_spec, physmass, num_spec, &
     &   natom_spec, spec, species, ikind, iounit, natom, natom_spec, &
     &   int_spec, mspec, natom_kind

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: ierr, i, j, k, l

!     /*   number of atomic symbols   */
      integer :: nsymbol

!     /*   atomic symbols   */
      character(len=8), dimension(:), allocatable :: symbol

!     /*   atomic numbers   */
      integer, dimension(:), allocatable :: num_symbol

!-----------------------------------------------------------------------
!     /*   old version                                                */
!-----------------------------------------------------------------------

!     /*   reset error flag   */
      ierr = 0

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<nspec>', 7, iounit, ierr )

!     /*   read integer   */
      read ( iounit, *, iostat=ierr )

!     /*   read species   */
      do k = 1, mspec
         read ( iounit, *, iostat=ierr ) &
     &      spec(k), physmass_spec(k), natom_spec(k)
      end do

!     /*   file close   */
      close( iounit )

!-----------------------------------------------------------------------
!     /*   error termination if ierr is not zero                      */
!-----------------------------------------------------------------------

      call error_handling_calc &
     &   ( ierr, 'subroutine calc_read_species_old', 32 )

!-----------------------------------------------------------------------
!     /*   count number of atoms                                      */
!-----------------------------------------------------------------------

!     /*   reset counter   */
      j = 0

!     /*   loop of species   */
      do k = 1, mspec

!        /*  add number of atoms per species   */
         j = j + natom_spec(k)

!     /*   loop of species   */
      end do

!-----------------------------------------------------------------------
!     /*   check error                                                */
!-----------------------------------------------------------------------

!     /*   reset error flag   */
      ierr = 0

!     /*   if total number of atoms of all species is not natom   */
      if ( j .ne. natom ) then

!        /*   error message   */
         write( 6, '(a,i6,a,i6,a)' ) &
     &      'Error - number of atoms,', natom, &
     &      ', must match the sum over all species,', j, '.'

!        /*   error message   */
         write( 6, '(a)' ) &
     &      'Check keywords <natom> and <nspec>.'

!        /*   error message   */
         write( 6, '(a)' )

!        /*   error flag   */
         ierr = 1

!     /*   end of if statement   */
      end if

!     /*   error termination if ierr is not zero   */
      call error_handling_calc &
     &   ( ierr, 'subroutine calc_read_species_old', 32 )

!-----------------------------------------------------------------------
!     /*   make physical mass of atoms                                */
!-----------------------------------------------------------------------

!     /*   reset counter   */
      l = 0

!     /*   loop of species   */
      do k = 1, mspec

!        /*   [amu] -> [a.u.]   */
         physmass_spec(k) = physmass_spec(k) *amu_mass/au_mass

!        /*   loop of atoms per species   */
         do j = 1, natom_spec(k)

!           /*   update counter of atoms   */
            l = l + 1

!           /*   mass of species -> mass of atoms   */
            physmass(l) = physmass_spec(k)

!        /*   loop of atoms per species   */
         end do

!     /*   loop of species   */
      end do

!-----------------------------------------------------------------------
!     /*   atomic species                                             */
!-----------------------------------------------------------------------

!     /*   reset atom counter   */
      l = 0

!     /*   loop of species   */
      do k = 1, mspec

!        /*   loop of atoms per species   */
         do j = 1, natom_spec(k)

!           /*   update atom counter   */
            l = l + 1

!           /*   species of atoms   */
            species(l)  = spec(k)

!        /*   loop of atoms per species   */
         end do

!     /*   loop of species   */
      end do

!-----------------------------------------------------------------------
!     /*   read atomic symbols                                        */
!-----------------------------------------------------------------------

!     /*   number of atoms   */
      call read_int1_calc ( nsymbol, '<nsymbol>', 9, iounit )

!     /*   memory allocation: atomic symbols   */
      allocate( symbol(nsymbol) )

!     /*   memory allocation: atomic symbols   */
      allocate( num_symbol(nsymbol) )

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<nsymbol>', 9, iounit, ierr )

!     /*   number of atomic symbols   */
      read ( iounit, *, iostat=ierr )

!     /*   loop of symbols   */
      do i = 1, nsymbol

!        /*   read symbol, atomic number, atomic mass   */
         read ( iounit, *, iostat=ierr ) &
     &      symbol(i), num_symbol(i)

!     /*   loop of symbols   */
      end do

!     /*   file close   */
      close( iounit )

!     /*   on error, read default values   */
      if ( ierr .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'calc_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<nsymbol>', 9, iounit, ierr )

!        /*   number of atomic symbols   */
         read ( iounit, *, iostat=ierr )

!        /*   loop of symbols   */
         do i = 1, nsymbol

!           /*   read symbol, atomic number, atomic mass   */
            read ( iounit, *, iostat=ierr ) &
     &         symbol(i), num_symbol(i)

!        /*   loop of symbols   */
         end do

!        /*   file close   */
         close( iounit )

!     /*   on error, read default values   */
      end if

!-----------------------------------------------------------------------
!     /*   stop on error                                              */
!-----------------------------------------------------------------------

!     /*   check error   */
      if ( ierr .ne. 0 ) then

!        /*   error message   */
         write( 6, '(a)' ) 'Error - <nsymbol> read incorrectly.'
         write( 6, '(a)' )

!     /*   check error   */
      end if

!     /*   error termination   */
      call error_handling_calc &
     &   ( ierr, 'subroutine calc_read_species_old', 32 )

!-----------------------------------------------------------------------
!     /*   physmass: atomic mass of each atom                         */
!-----------------------------------------------------------------------

!     /*   flag   */
      ierr = 0

!     /*   loop of atoms   */
      do i = 1, natom

!        /*   flag   */
         k = 0

!        /*   loop of symbols   */
         do l = 1, nsymbol

!           /*   if symbol matched   */
            if ( species(i)(1:8) .eq. symbol(l)(1:8) ) then

!              /*   flag   */
               k = 1

!              /*   substitute atomic number   */
               int_spec(i) = num_symbol(l)

!              /*   go to next loop   */
               exit

!           /*   if symbol matched   */
            end if

!        /*   loop of symbols   */
         end do

!        /*   error flag   */
         if ( k .eq. 0 ) then

!           /*   error flag   */
            ierr = 1

!           /*   atom not found   */
            l = i

!           /*   exit from the loop  */
            exit

!        /*   error flag   */
         end if

!     /*   loop of atoms   */
      end do

!-----------------------------------------------------------------------
!     /*   stop on error                                              */
!-----------------------------------------------------------------------

!     /*   check error   */
      if ( ierr .ne. 0 ) then

!        /*   error message   */
         write( 6, '(a)' ) 'Error - <nsymbol> read incorrectly.'
         write( 6, '(a)' ) 'Atom type not designated: ' // species(l)
         write( 6, '(a)' ) 'Please add data in <nsymbol>.'
         write( 6, '(a)' )

!     /*   check error   */
      end if

!     /*   error termination   */
      call error_handling_calc &
     &   ( ierr, 'subroutine calc_read_species_old', 32 )

!-----------------------------------------------------------------------
!     /*   num_spec: atomic number, physmass_spec: atomic mass        */
!-----------------------------------------------------------------------

!     /*   loop of species   */
      do k = 1, mspec

!        /*   loop of symbols   */
         do l = 1, nsymbol

!           /*   if symbol matched   */
            if ( spec(k)(1:8) .eq. symbol(l)(1:8) ) then

!              /*   atomic number   */
               num_spec(k) = num_symbol(l)

!           /*   if symbol matched   */
            end if

!        /*   loop of symbols   */
         end do

!     /*   loop of species   */
      end do

!-----------------------------------------------------------------------
!     /*   ikind: species of a given atom                             */
!-----------------------------------------------------------------------

!     /*   counter   */
      l = 0

!     /*   loop of species   */
      do k = 1, mspec

!        /*   loop of atoms per species   */
         do j = 1, natom_spec(k)

!           /*   counter   */
            l = l + 1

!           /*   species number   */
            ikind(l) = k

!        /*   loop of atoms per species   */
         end do

!        /*   natom per kind   */
         natom_kind(k) = natom_spec(k)

!     /*   loop of species   */
      end do

      return
      end





!***********************************************************************
      subroutine calc_read_species_new
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use calc_variables, only : &
     &   au_mass, amu_mass, physmass_spec, physmass, num_spec, int_spec, &
     &   natom_spec, spec, species, ikind, iounit, natom, mspec, nkind, &
     &   natom_kind

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   number of atomic symbols   */
      integer :: nsymbol

!     /*   atomic symbols   */
      character(len=8), dimension(:), allocatable :: symbol

!     /*   atomic numbers   */
      integer, dimension(:), allocatable :: num_symbol

!     /*   atomic masses   */
      real(8), dimension(:), allocatable :: physmass_symbol

!     /*   real numbers   */
      integer :: ierr, i, j, k, l

!     /*   real numbers   */
      real(8) :: x, y, z

!     /*   characters   */
      character(len=120) :: charline

!-----------------------------------------------------------------------
!     /*   read species                                               */
!-----------------------------------------------------------------------

!     /*   file open   */
      open ( iounit, file = 'structure.dat' )

!     /*   skip a line   */
      read ( iounit, *, iostat=ierr )

!     /*   skip a line   */
      read ( iounit, *, iostat=ierr )

!     /*   read species   */
      do i = 1, natom
         read ( iounit, '(a)', iostat=ierr ) charline
         read ( charline, *, iostat=ierr ) species(i), x, y, z, ikind(i)
         if ( ierr .ne. 0 ) then
            read ( charline, *, iostat=ierr ) species(i), x, y, z
            if ( ierr .ne. 0 ) exit
            ikind(i) = 1
         end if
      end do

!     /*   file close   */
      close( iounit )

!-----------------------------------------------------------------------
!     /*   natom_kind: number of atoms per kind                       */
!-----------------------------------------------------------------------

!     /*   loop of kinds   */
      do j = 1, nkind

!        /*   counter   */
         l = 0

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   if matched update counter   */
            if ( j .eq. ikind(i) ) l = l + 1

!        /*   loop of atoms   */
         end do

!        /*   number of atoms per kind   */
         natom_kind(j) = l

!     /*   loop of kinds   */
      end do

!-----------------------------------------------------------------------
!     /*   spec: atomic symbol of each species                        */
!-----------------------------------------------------------------------

!     /*   species counter   */
      l = 1

!     /*   first atom   */
      spec(1) = species(1)

!     /*   loop of atoms   */
      do i = 2, natom

!        /*   flag   */
         k = 1

!        /*   loop of former atoms   */
         do j = 1, i-1

!           /*   if species match the ones of former atoms   */
            if ( species(j)(1:8) .eq. species(i)(1:8) ) then

!              /*   flag   */
               k = 0

!              /*   exit from loop   */
               exit

!           /*   if species match the ones of former atoms   */
            end if

!        /*   loop of former atoms   */
         end do

!        /*   update species counter   */
         l = l + k

!        /*   substitute species   */
         if ( k .eq. 1 ) spec(l) = species(i)

!     /*   loop of atoms   */
      end do

!-----------------------------------------------------------------------
!     /*   natom_spec: atoms per species                              */
!-----------------------------------------------------------------------

!     /*   initialize counter   */
      natom_spec(:) = 0

!     /*   loop of atoms   */
      do i = 1, natom

!        /*   loop of atomic species   */
         do l = 1, mspec

!           /*   if symbols matched   */
            if ( species(i)(1:8) .eq. spec(l)(1:8) ) then

!              /*   update counter   */
               natom_spec(l) = natom_spec(l) + 1

               /*   skip to next loop   */
               exit

!           /*   if symbols matched   */
            end if

!        /*   loop of atomic species   */
         end do

!     /*   loop of atoms   */
      end do

!-----------------------------------------------------------------------
!     /*   read atomic symbols                                        */
!-----------------------------------------------------------------------

!     /*   number of atoms   */
      call read_int1_calc ( nsymbol, '<nsymbol>', 9, iounit )

!     /*   memory allocation: atomic symbols   */
      allocate( symbol(nsymbol) )

!     /*   memory allocation: atomic symbols   */
      allocate( num_symbol(nsymbol) )

!     /*   memory allocation: atomic masses   */
      allocate( physmass_symbol(nsymbol) )

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<nsymbol>', 9, iounit, ierr )

!     /*   number of atomic symbols   */
      read ( iounit, *, iostat=ierr )

!     /*   loop of symbols   */
      do i = 1, nsymbol

!        /*   read symbol, atomic number, atomic mass   */
         read ( iounit, *, iostat=ierr ) &
     &      symbol(i), num_symbol(i), physmass_symbol(i)

!     /*   loop of symbols   */
      end do

!     /*   file close   */
      close( iounit )

!     /*   on error, read default values   */
      if ( ierr .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'calc_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<nsymbol>', 9, iounit, ierr )

!        /*   number of atomic symbols   */
         read ( iounit, *, iostat=ierr )

!        /*   loop of symbols   */
         do i = 1, nsymbol

!           /*   read symbol, atomic number, atomic mass   */
            read ( iounit, *, iostat=ierr ) &
     &         symbol(i), num_symbol(i), physmass_symbol(i)

!        /*   loop of symbols   */
         end do

!        /*   file close   */
         close( iounit )

!     /*   on error, read default values   */
      end if

!-----------------------------------------------------------------------
!     /*   stop on error                                              */
!-----------------------------------------------------------------------

!     /*   check error   */
      if ( ierr .ne. 0 ) then

!        /*   error message   */
         write( 6, '(a)' ) 'Error - <nsymbol> read incorrectly.'
         write( 6, '(a)' )

!     /*   check error   */
      end if

!     /*   error termination   */
      call error_handling_calc &
     &   ( ierr, 'subroutine calc_read_species_new', 32 )

!-----------------------------------------------------------------------
!     /*   physmass: atomic mass of each atom                         */
!-----------------------------------------------------------------------

!     /*   flag   */
      ierr = 0

!     /*   loop of atoms   */
      do i = 1, natom

!        /*   flag   */
         k = 0

!        /*   loop of symbols   */
         do l = 1, nsymbol

!           /*   if symbol matched   */
            if ( species(i)(1:8) .eq. symbol(l)(1:8) ) then

!              /*   flag   */
               k = 1

!              /*   substitute atomic number   */
               int_spec(i) = num_symbol(l)

!              /*   substitute mass   */
               physmass(i) = physmass_symbol(l)*amu_mass/au_mass

!              /*   go to next loop   */
               exit

!           /*   if symbol matched   */
            end if

!        /*   loop of symbols   */
         end do

!        /*   error flag   */
         if ( k .eq. 0 ) then

!           /*   error flag   */
            ierr = 1

!           /*   exit from the loop  */
            exit

!        /*   error flag   */
         end if

!     /*   loop of atoms   */
      end do

!-----------------------------------------------------------------------
!     /*   stop on error                                              */
!-----------------------------------------------------------------------

!     /*   check error   */
      if ( ierr .ne. 0 ) then

!        /*   error message   */
         write( 6, '(a)' ) 'Error - <nsymbol> read incorrectly.'
         write( 6, '(a)' )

!     /*   check error   */
      end if

!     /*   error termination   */
      call error_handling_calc &
     &   ( ierr, 'subroutine calc_read_species_new', 32 )

!-----------------------------------------------------------------------
!     /*   num_spec: atomic number, physmass_spec: atomic mass        */
!-----------------------------------------------------------------------

!     /*   loop of species   */
      do k = 1, mspec

!        /*   loop of symbols   */
         do l = 1, nsymbol

!           /*   if symbol matched   */
            if ( spec(k)(1:8) .eq. symbol(l)(1:8) ) then

!              /*   atomic number   */
               num_spec(k) = num_symbol(l)

!              /*   substitute mass   */
               physmass_spec(k) = physmass_symbol(l)*amu_mass/au_mass

!           /*   if symbol matched   */
            end if

!        /*   loop of symbols   */
         end do

!     /*   loop of species   */
      end do

      return
      end

