!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     periodic boundary condition for post process
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine calc_pbc
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use calc_variables, only : &
     &   box, boxinv, x, y, z, ux, uy, uz, natom, nbead, iboundary, &
     &   xo, yo, zo, jorigin_xyz, jxyz_atom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integer   */
      integer :: i, j

!     /*   real   */
      real(8) :: ac, bc, cc, da, db, dc, dx, dy, dz

!-----------------------------------------------------------------------
!     /*   centroid                                                   */
!-----------------------------------------------------------------------

      do i = 1, natom

         ux(i,1) = 0.d0
         uy(i,1) = 0.d0
         uz(i,1) = 0.d0

         do j = 1, nbead
            ux(i,1) = ux(i,1) + x(i,j)
            uy(i,1) = uy(i,1) + y(i,j)
            uz(i,1) = uz(i,1) + z(i,j)
         end do

         ux(i,1) = ux(i,1)/nbead
         uy(i,1) = uy(i,1)/nbead
         uz(i,1) = uz(i,1)/nbead

      end do

!-----------------------------------------------------------------------
!     /*   set origin                                                 */
!-----------------------------------------------------------------------

      if ( jorigin_xyz .eq. 3 ) then

         xo = ux(jxyz_atom,1)
         yo = uy(jxyz_atom,1)
         zo = uz(jxyz_atom,1)

      else

         continue

      end if

!-----------------------------------------------------------------------
!     /*   for free boundary                                          */
!-----------------------------------------------------------------------

      if      ( iboundary .eq. 0 ) then

         do i = 1, natom

            ux(i,1) = ux(i,1) - xo
            uy(i,1) = uy(i,1) - yo
            uz(i,1) = uz(i,1) - zo

            do j = 1, nbead

               x(i,j) = x(i,j) - xo
               y(i,j) = y(i,j) - yo
               z(i,j) = z(i,j) - zo

            end do

         end do

         xo = 0.d0
         yo = 0.d0
         zo = 0.d0

!-----------------------------------------------------------------------
!     /*   for periodic boundary                                      */
!-----------------------------------------------------------------------

      else if ( iboundary .eq. 1 ) then

         do i = 1, natom

            ac = boxinv(1,1)*(ux(i,1)-xo) &
     &         + boxinv(1,2)*(uy(i,1)-yo) &
     &         + boxinv(1,3)*(uz(i,1)-zo)
            bc = boxinv(2,1)*(ux(i,1)-xo) &
     &         + boxinv(2,2)*(uy(i,1)-yo) &
     &         + boxinv(2,3)*(uz(i,1)-zo)
            cc = boxinv(3,1)*(ux(i,1)-xo) &
     &         + boxinv(3,2)*(uy(i,1)-yo) &
     &         + boxinv(3,3)*(uz(i,1)-zo)

            da = - dble(nint(ac))
            db = - dble(nint(bc))
            dc = - dble(nint(cc))

            dx = box(1,1)*da + box(1,2)*db + box(1,3)*dc
            dy = box(2,1)*da + box(2,2)*db + box(2,3)*dc
            dz = box(3,1)*da + box(3,2)*db + box(3,3)*dc

            ux(i,1) = (ux(i,1)-xo) + dx
            uy(i,1) = (uy(i,1)-yo) + dy
            uz(i,1) = (uz(i,1)-zo) + dz

            do j = 1, nbead

               x(i,j) = (x(i,j)-xo) + dx
               y(i,j) = (y(i,j)-yo) + dy
               z(i,j) = (z(i,j)-zo) + dz

            end do

         end do

         xo = 0.d0
         yo = 0.d0
         zo = 0.d0

!-----------------------------------------------------------------------
!     /*   for periodic boundary                                      */
!-----------------------------------------------------------------------

      else if ( iboundary .eq. 2 ) then

         do i = 1, natom

            ac = boxinv(1,1)*(ux(i,1)-xo) &
     &         + boxinv(1,2)*(uy(i,1)-yo) &
     &         + boxinv(1,3)*(uz(i,1)-zo)
            bc = boxinv(2,1)*(ux(i,1)-xo) &
     &         + boxinv(2,2)*(uy(i,1)-yo) &
     &         + boxinv(2,3)*(uz(i,1)-zo)
            cc = boxinv(3,1)*(ux(i,1)-xo) &
     &         + boxinv(3,2)*(uy(i,1)-yo) &
     &         + boxinv(3,3)*(uz(i,1)-zo)

            da = - dble(nint(ac))
            db = - dble(nint(bc))
            dc = - dble(nint(cc))

            dx = box(1,1)*da + box(1,2)*db + box(1,3)*dc
            dy = box(2,1)*da + box(2,2)*db + box(2,3)*dc
            dz = box(3,1)*da + box(3,2)*db + box(3,3)*dc

            ux(i,1) = (ux(i,1)-xo) + dx
            uy(i,1) = (uy(i,1)-yo) + dy
            uz(i,1) = (uz(i,1)-zo) + dz

            do j = 1, nbead

               x(i,j) = (x(i,j)-xo) + dx
               y(i,j) = (y(i,j)-yo) + dy
               z(i,j) = (z(i,j)-zo) + dz

            end do

         end do

         xo = 0.d0
         yo = 0.d0
         zo = 0.d0

      end if

      return
      end
