!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     box optimization by LBFGS algorithm
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine boxoptcycle_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, potential, box, boxinv, fbox, tension, volume, &
     &   pressure, sigma_ref, boxinv_ref, volume_ref, natom, nstep, &
     &   iounit, iounit_std, istep, istep_start, istep_end, iexit, &
     &   iboundary

      use lbfgs_variables, only : &
     &   pos, pos0, grad, dm, ws, func, dmax, drms, fmax, frms, &
     &   dmax_tol, drms_tol, fmax_tol, frms_tol, postol, eps, &
     &   ndim, nup, iflag, iprint

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j, k, m

      real(8) :: r1, r2, r3, s1, s2, s3

      real(8), dimension(3,3) :: box0
      real(8), dimension(3,3) :: boxinv0

      real(8), dimension(natom,1) :: x0
      real(8), dimension(natom,1) :: y0
      real(8), dimension(natom,1) :: z0

      real(8) :: det3

!-----------------------------------------------------------------------
!     /*   error termination for free boundary                        */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) then

         write( 6, '(a)' ) &
     &     'Error - Boxopt is valid only for periodic boundary.'

         write( 6, '(a)' )

         call error_handling_MPI( 1, 'subroutine boxoptcycle_MPI', 26 )

      end if

!-----------------------------------------------------------------------
!     //   initialize step
!-----------------------------------------------------------------------

      istep = istep_start
      istep_end = istep

!-----------------------------------------------------------------------
!     //   new box size:  box  ->  pos
!-----------------------------------------------------------------------

      k = 0

      do m = 1, 3
      do j = 1, 3
         k = k + 1
         pos(k) = box(j,m)
      end do
      end do

!-----------------------------------------------------------------------
!     //   calculate potential and gradients
!-----------------------------------------------------------------------

      call getfbox_MPI

!     /*   add pressure and tension contributions   */
      call addfbox &
     &   ( fbox, volume, pressure, boxinv, box, sigma_ref )

!-----------------------------------------------------------------------
!     //   potential  ->  func
!-----------------------------------------------------------------------

      func = potential

!     /*   add pressure and tension contribution   */
      call addebox &
     &   ( func, pressure, volume, boxinv_ref, volume_ref, box, &
     &     tension )

!-----------------------------------------------------------------------
!     //   force  ->  grad
!-----------------------------------------------------------------------

      k = 0

      do m = 1, 3
      do j = 1, 3
         k = k + 1
         grad(k) = - fbox(j,m)
      end do
      end do

!-----------------------------------------------------------------------
!     //   old box size: pos0
!-----------------------------------------------------------------------

!     /*   store box matrix   */
      pos0(:) = pos(:)

!     /*   store coordinates   */
      x0(:,1) = x(:,1)
      y0(:,1) = y(:,1)
      z0(:,1) = z(:,1)

!     /*   store box matrix   */
      box0(:,:) = box(:,:)

!     /*   inverse box matrix   */
      call inv3 ( box0, boxinv0 )

!-----------------------------------------------------------------------
!     //   standard output
!-----------------------------------------------------------------------

      call standard_boxopt_MPI

!-----------------------------------------------------------------------
!     //   do some analysis
!-----------------------------------------------------------------------

      call analysis_MPI( 1 )

!-----------------------------------------------------------------------
!     //   terminate if converged
!-----------------------------------------------------------------------

      if ( ( fmax .lt. fmax_tol ) .and. ( frms .lt. frms_tol ) ) then
         return
      end if

!-----------------------------------------------------------------------
!     //   main loop start
!-----------------------------------------------------------------------

      do istep = istep_start+1, nstep

!-----------------------------------------------------------------------
!        //   current step
!-----------------------------------------------------------------------

         istep_end = istep

!-----------------------------------------------------------------------
!        //   old position: pos0
!-----------------------------------------------------------------------

!        /*   store box matrix   */
         pos0(:) = pos(:)

!        /*   store coordinates   */
         x0(:,1) = x(:,1)
         y0(:,1) = y(:,1)
         z0(:,1) = z(:,1)

!        /*   store box matrix   */
         box0(:,:) = box(:,:)

!        /*   inverse box matrix   */
         call inv3 ( box0, boxinv0 )

!-----------------------------------------------------------------------
!        //   call limited memory bfgs routine
!-----------------------------------------------------------------------

         call lbfgs ( ndim, nup, pos, func, grad, .false., dm, &
     &                iprint, eps, postol, ws, iflag )

!-----------------------------------------------------------------------
!        //   new box: pos  ->  box
!-----------------------------------------------------------------------

         k = 0

         do m = 1, 3
         do j = 1, 3
            k = k + 1
            box(j,m) = pos(k)
         end do
         end do

!        /*   inverse matrix of box   */
         call inv3 ( box, boxinv )

!        /*   shift box volume   */
         volume = det3( box )

!-----------------------------------------------------------------------
!        //   new position according to new box
!-----------------------------------------------------------------------

         do k = 1, natom

            r1 = x0(k,1)
            r2 = y0(k,1)
            r3 = z0(k,1)

            s1 = boxinv0(1,1)*r1 + boxinv0(1,2)*r2 + boxinv0(1,3)*r3
            s2 = boxinv0(2,1)*r1 + boxinv0(2,2)*r2 + boxinv0(2,3)*r3
            s3 = boxinv0(3,1)*r1 + boxinv0(3,2)*r2 + boxinv0(3,3)*r3

            x(k,1) = box(1,1)*s1 + box(1,2)*s2 + box(1,3)*s3
            y(k,1) = box(2,1)*s1 + box(2,2)*s2 + box(2,3)*s3
            z(k,1) = box(3,1)*s1 + box(3,2)*s2 + box(3,3)*s3

         end do

!-----------------------------------------------------------------------
!        //   calculate potential and gradients
!-----------------------------------------------------------------------

         call getfbox_MPI

!        /*   add pressure and tension contributions   */
         call addfbox &
     &      ( fbox, volume, pressure, boxinv, box, sigma_ref )

!-----------------------------------------------------------------------
!        //   potential  ->  func
!-----------------------------------------------------------------------

         func = potential

!        /*   add pressure and tension contribution   */
         call addebox &
     &      ( func, pressure, volume, boxinv_ref, volume_ref, box, &
     &        tension )

!-----------------------------------------------------------------------
!        //   force  ->  grad
!-----------------------------------------------------------------------

         k = 0

         do m = 1, 3
         do j = 1, 3
            k = k + 1
            grad(k) = - fbox(j,m)
         end do
         end do

!-----------------------------------------------------------------------
!        //   standard output
!-----------------------------------------------------------------------

         call standard_boxopt_MPI

!-----------------------------------------------------------------------
!        //   output restart
!-----------------------------------------------------------------------

         call backup_boxopt_MPI

!-----------------------------------------------------------------------
!        //   do some analysis
!-----------------------------------------------------------------------

         call analysis_MPI( 2 )

!-----------------------------------------------------------------------
!        //   terminate if converged
!-----------------------------------------------------------------------

         if ( ( dmax .lt. dmax_tol ) .and. ( drms .lt. drms_tol ) .and. &
     &        ( fmax .lt. fmax_tol ) .and. ( frms .lt. frms_tol ) ) then
            iflag = 0
         end if

         if ( iflag .le. 0 ) then
            iexit = 1
            exit
         end if

!-----------------------------------------------------------------------
!        /*   exit if `exit.dat' exists                               */
!-----------------------------------------------------------------------

         call softexit_MPI

         if ( iexit .eq. 1 ) exit

      end do

      return
      end

