!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     set boundary condition
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine boundary
!***********************************************************************

      use common_variables, only : &
     &   box, box_ref, boxinv, boxinv_ref, volume, volume_ref, &
     &   iboundary, iounit, ibox_start, au_length, char_boundary

      implicit none

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      integer :: ierr

      real(8) :: det3

      integer, save :: iset = 0

      real(8) :: bohr2ang = au_length * 1.d+10

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   read boundary condition                                    */
!-----------------------------------------------------------------------

      call read_char( char_boundary, 10, '<iboundary>', 11, iounit )

      if      ( char_boundary(1:2) .eq. '0 ' ) then
         iboundary = 0
      else if ( char_boundary(1:2) .eq. '1 ' ) then
         iboundary = 1
      else if ( char_boundary(1:2) .eq. '2 ' ) then
         iboundary = 2
      else if ( char_boundary(1:5) .eq. 'NONE ' ) then
         iboundary = 0
      else if ( char_boundary(1:5) .eq. 'BOHR ' ) then
         iboundary = 1
      else if ( char_boundary(1:9) .eq. 'ANGSTROM ' ) then
         iboundary = 1
      else
         iboundary = 0
      end if

!-----------------------------------------------------------------------
!     /*   free boundary: box not set                                 */
!-----------------------------------------------------------------------

      if      ( iboundary .eq. 0 ) then

         continue

!-----------------------------------------------------------------------
!     /*   periodic boundary                                          */
!-----------------------------------------------------------------------

      else if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then

!-----------------------------------------------------------------------
!        /*   read box size from input.dat and input.dat (ref)        */
!-----------------------------------------------------------------------

         if ( ibox_start .eq. 0 ) then

            open (iounit,file = 'input.dat')

!           /*   tag   */
            call search_tag ( '<iboundary>', 11, iounit, ierr )

!           /*   skip a line   */
            read(iounit,*,iostat=ierr)

!           /*   box size   */
            read(iounit,*,iostat=ierr) &
     &         box_ref(1,1), box_ref(1,2), box_ref(1,3)
            read(iounit,*,iostat=ierr) &
     &         box_ref(2,1), box_ref(2,2), box_ref(2,3)
            read(iounit,*,iostat=ierr) &
     &         box_ref(3,1), box_ref(3,2), box_ref(3,3)

            close(iounit)

!           /*   unit conversion   */
            if ( ierr .eq. 0 ) then
            if ( char_boundary(1:9) .eq. 'ANGSTROM ' ) then
               box_ref(:,:) = box_ref(:,:) / bohr2ang
            end if
            end if

            if ( ierr .ne. 0 ) then
               write(6,'(a)') 'Error - reading box from input.dat.'
               write(6,'(a)') 
            end if

!           /*   error termination   */
            call error_handling( ierr, 'subroutine boundary', 19 )

!           /*   reference cell matrix   */
            box(:,:) = box_ref(:,:)

!           /*   inverse matrix of cell matrix   */
            call inv3 ( box, boxinv )

!           /*   inverse matrix of cell matrix   */
            call inv3 ( box_ref, boxinv_ref )

!           /*   volume   */
            volume  = det3( box )

!           /*   reference volume   */
            volume_ref = det3( box_ref )

!-----------------------------------------------------------------------
!        /*   read box size from geometry.ini and input.dat (ref)     */
!-----------------------------------------------------------------------

         else if ( ibox_start .eq. 1 ) then

            open (iounit,file = 'input.dat')

!           /*   tag   */
            call search_tag ( '<iboundary>', 11, iounit, ierr )

!           /*   skip a line   */
            read(iounit,*,iostat=ierr)

!           /*   box size   */
            read(iounit,*,iostat=ierr) &
     &         box_ref(1,1), box_ref(1,2), box_ref(1,3)
            read(iounit,*,iostat=ierr) &
     &         box_ref(2,1), box_ref(2,2), box_ref(2,3)
            read(iounit,*,iostat=ierr) &
     &         box_ref(3,1), box_ref(3,2), box_ref(3,3)

            close(iounit)

!           /*   unit conversion   */
            if ( ierr .eq. 0 ) then
            if ( char_boundary(1:9) .eq. 'ANGSTROM ' ) then
               box_ref(:,:) = box_ref(:,:) / bohr2ang
            end if
            end if

            if ( ierr .ne. 0 ) then
               write(6,'(a)') 'Error - reading box from input.dat.'
               write(6,'(a)') 
            end if

!           /*   error termination   */
            call error_handling( ierr, 'subroutine boundary', 19 )

!           /*   inverse matrix of cell matrix   */
            call inv3 ( box_ref, boxinv_ref )

!           /*   reference volume   */
            volume_ref = det3( box_ref )

!           /*   read cell matrix   */
            call restart_box ( 1 )

!           /*   inverse matrix of cell matrix   */
            call inv3 ( box, boxinv )

!           /*   reference volume   */
            volume = det3( box )

         end if

      else

         write(6,'(a)') 'Error - iboundary should be 0, 1 or 2.'
         write(6,'(a)')

!        /*   error termination   */
         call error_handling( ierr, 'subroutine boundary', 19 )

      end if

!-----------------------------------------------------------------------
!     /*   set complete                                               */
!-----------------------------------------------------------------------

      iset = 1

      return
      end
