!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Dec 18, 2020 by M. Shiga
!      Description:     print xsf file
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine analysis_xsf_MPI( ioption )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, au_charge, au_energy, au_length, box, species, &
     &   pot, fx, fy, fz, istep, natom, nbead, iounit, iboundary, &
     &   istep, myrank

      use analysis_variables, only : &
     &   iprint_xsf

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: i, j, itest, ioption

!     /*   real numbers   */
      real(8) :: xa, ya, za, ax, ay, az, bx, by, bz, cx, cy, cz

!     /*   real numbers   */
      real(8) :: const_1, const_2, const_3

!     /*   potential and force   */
      real(8) :: pot_ev, fxj, fyj, fzj

!     /*   flag   */
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   print condition: every iprint_xsf steps                    */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 ) return
      if ((ioption.eq.0) .or. (ioption.eq.1) .or. (ioption.eq.3)) return
      if ( iprint_xsf .le. 0 ) return
      if ( mod(istep,iprint_xsf) .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   header                                                     */
!-----------------------------------------------------------------------

!     /*   initialization   */
      if ( iset .eq. 0 ) then

!        /*   test if file exists */
         call testfile ( 'trj.xsf', 7, itest )

!        /*   if file does not exist   */
         if ( itest .ne. 0 ) then

!            /*   file open   */
             open ( iounit, file = 'trj.xsf' )

!            /*   write header   */
             write( iounit, '(a)' ) 'ANIMSTEPS 99999999'

!            /*   file close   */
             close( iounit )

!        /*   if file does not exist   */
         end if

!        /*   set completed   */
         iset = 1

!     /*   initialization   */
      end if

!-----------------------------------------------------------------------
!     /*   print xsf file                                             */
!-----------------------------------------------------------------------

!     /*   conversion factor   */
      const_1 = au_length * 1.d+10

!     /*   conversion factor   */
      const_2 =  au_charge / au_energy

!     /*   conversion factor   */
      const_3 =  1.d0 / const_2 / const_1

!     /*   open sample file   */
      open( iounit, file = 'trj.xsf', access = 'append' )

!     /*   loop of beads   */
      do j = 1, nbead

!        /*   potential energy in eV   */
         pot_ev = pot(j) / const_2

!        /*   write one line   */
         write( iounit, '(a,f24.12,a)' ) &
     &      '# total energy = ', pot_ev, ' eV'

!     /*   loop of beads   */
      end do

!     /*   for periodic boundary condition   */
      if ( (iboundary .eq. 1) .or. (iboundary .eq. 2) ) then

!        /*   write one line   */
         write( iounit, '(a)' ) 'CRYSTAL'

!        /*   write one line   */
         write( iounit, '(a,i8)' ) 'PRIMVEC ', istep

!        /*   lattice vectors   */
         ax = box(1,1) * const_1
         ay = box(2,1) * const_1
         az = box(3,1) * const_1
         bx = box(1,2) * const_1
         by = box(2,2) * const_1
         bz = box(3,2) * const_1
         cx = box(1,3) * const_1
         cy = box(2,3) * const_1
         cz = box(3,3) * const_1

!        /*   write three lines   */
         write( iounit, '(3f16.8)' ) ax, ay, az
         write( iounit, '(3f16.8)' ) bx, by, bz
         write( iounit, '(3f16.8)' ) cx, cy, cz

!        /*   write one line   */
         write( iounit, '(a,i8)' ) 'PRIMCOORD ', istep

!        /*   write one line   */
         write( iounit, '(i8,i2)' ) natom*nbead, 1

!     /*   for free boundary condition   */
      else if ( iboundary .eq. 0 ) then

!        /*   write one line   */
         write( iounit, '(a,i8)' ) 'ATOMS ', istep

!     /*   for free or periodic boundary condition   */
      end if

!     /*   loop of beads   */
      do j = 1, nbead

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   geometry in angstroms   */
            xa = x(i,j) * const_1
            ya = y(i,j) * const_1
            za = z(i,j) * const_1

!           /*   geometry in eV per angstrom   */
            fxj = fx(i,j) * const_3 * dble(nbead)
            fyj = fy(i,j) * const_3 * dble(nbead)
            fzj = fz(i,j) * const_3 * dble(nbead)

!           /*   write one line   */
            write( iounit, '(a4,6f16.8)' ) &
     &         species(i)(1:4), xa, ya, za, fxj, fyj, fzj

!        /*   loop of atoms   */
         end do

!     /*   loop of beads   */
      end do

!     /*   close input file   */
      close( iounit )

      return
      end
