!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Mar 2, 2022 by M. Shiga
!      Description:     analysis in tass
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine analysis_tass_MPI ( ioption )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iounit

      use analysis_variables, only : iprint_dcd

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   option   */
      integer :: ioption

!     /*   flag   */
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   header                                                     */
!-----------------------------------------------------------------------

!     /*   initialization   */
      if ( iset .eq. 0 ) then

!        /*   read print interval   */
         call read_int1_MPI ( iprint_dcd,  '<iprint_dcd>',  12, iounit )

         /*   end initialization   */
         iset = 1

!     /*   initialization   */
      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  initialize/restart                           */
!-----------------------------------------------------------------------

      if      ( ioption .eq. 1 ) then

!        /*   energy components   */
         call analysis_ene_tass_MPI( 1 )

!        /*   xyz trajectory   */
         call analysis_xyz_tass_MPI( 1 )

!        /*   xsf trajectory   */
         call analysis_xsf_tass_MPI( 1 )

!        /*   dcd trajectory   */
         call analysis_dcd_beadwise_MPI( 1, 0 )

!        /*   cv trajectory   */
         call analysis_cv_tass_MPI ( 1 )

!        /*   reconstruction of hills   */
         call analysis_rec_tass_MPI( 1 )

      else if ( ioption .eq. 2 ) then

!        /*   energy components   */
         call analysis_ene_tass_MPI( 2 )

!        /*   xyz trajectory   */
         call analysis_xyz_tass_MPI( 2 )

!        /*   xsf trajectory   */
         call analysis_xsf_tass_MPI( 2 )

!        /*   dcd trajectory   */
         call analysis_dcd_beadwise_MPI( 2, 1 )

!        /*   cv trajectory   */
         call analysis_cv_tass_MPI ( 2 )

!        /*   reconstruction of hills   */
         call analysis_rec_tass_MPI( 2 )

      else if ( ioption .eq. 3 ) then

!        /*   energy components   */
         call analysis_ene_tass_MPI( 3 )

!        /*   xyz trajectory   */
         call analysis_xyz_tass_MPI( 3 )

!        /*   xsf trajectory   */
         call analysis_xsf_tass_MPI( 3 )

!        /*   dcd trajectory   */
         call analysis_dcd_beadwise_MPI( 3, 1 )

!        /*   cv trajectory   */
         call analysis_cv_tass_MPI ( 3 )

!        /*   reconstruction of hills   */
         call analysis_rec_tass_MPI( 3 )

      else

         continue

      end if

      return
      end





!***********************************************************************
      subroutine analysis_ene_tass_MPI( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, boltz, potential, hamiltonian, &
     &   natom, nbead, istep, myrank

      use tass_variables, only : &
     &   ekin_sys, ekin_tass, potential_tass, pot_lifted, ebath_tass, &
     &   potential_ref_tass, ebath_sys, iprint_tass, ntass

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: itest, ioption

      real(8) :: pot_combined, temp_sys, temp_tass

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   if set is not done   */
      if ( iset .eq. 0 ) then

!        /*   read print interval   */
         call read_int1_MPI( iprint_tass, '<iprint_tass>', 13, iounit )

!        /*   set done   */
         iset = 1

!        /*   return by option   */
         if ( iprint_tass .le. 0 ) return

!        /*   master process only   */
         if ( myrank .ne. 0 ) return

!        /*   check if file exists   */
         call testfile ( 'tass.out', 8, itest )

!        /*   if the file does not exist print header   */
         if ( itest .eq. 1 ) then

!           /*   open file   */
            open ( iounit, file = 'tass.out', access = 'append' )

!           /*   print header   */
            write( iounit, '(a)' ) &
     &      '========================================================'
            write( iounit, '(a)' ) &
     &      '    step       potential  potential_tass    pot_combined'
            write( iounit, '(a)' ) &
     &      '    step    pot_ref_tass    - pot_lifted        ekin_sys'
            write( iounit, '(a)' ) &
     &      '    step       ekin_tass       ebath_sys      ebath_tass'
            write( iounit, '(a)' ) &
     &      '    step     hamiltonian        temp_sys       temp_tass'
            write( iounit, '(a)' ) &
     &      '--------------------------------------------------------'

!           /*   close file   */
            close( iounit )

!        /*   if the file already exist do not print header   */
         else

            continue

         end if

!        /*   set done   */
         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 )  continue
      if ( ioption .eq. 2 )  continue
      if ( ioption .eq. 3 )  continue

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

!     /*   print interval   */
      if ( iprint_tass .le. 0 )  return

!     /*   master process only   */
      if ( myrank .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   print energies                                             */
!-----------------------------------------------------------------------

!     /*   print interval   */
      if ( mod(istep,iprint_tass) .eq. 0 ) then

!        /*   instant temperature of atoms   */
         temp_sys  = 2.d0*ekin_sys/dble(3*natom*nbead)/boltz

!        /*   instant temperature of cv   */
         temp_tass = 2.d0*ekin_tass/dble(ntass*nbead)/boltz

!        /*   real potential and hill potential combined   */
         pot_combined = potential + potential_tass

!        /*   open file   */
         open ( iounit, file = 'tass.out', access = 'append' )

!        /*   print energies   */

         write(iounit,'(i8,3e16.8)') &
     &      istep,          potential,  potential_tass,    pot_combined
         write(iounit,'(i8,3e16.8)') &
     &      istep, potential_ref_tass,    - pot_lifted,        ekin_sys
         write(iounit,'(i8,3e16.8)') &
     &      istep,          ekin_tass,       ebath_sys,      ebath_tass
         write(iounit,'(i8,3e16.8)') &
     &      istep,        hamiltonian,        temp_sys,       temp_tass

!        /*   close file   */
         close( iounit )

      end if

      return
      end





!***********************************************************************
      subroutine analysis_xyz_tass_MPI ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, au_length, x, y, z, species, &
     &   iounit, iounit_xyz, istep, natom, nbead, mbox, myrank

      use tass_variables, only : &
     &   iprint_xyz

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, k, ioption, m1, m2, m3

      real(8) :: xa, ya, za, xb, yb, zb

      character(len=11) :: filename

      character(len=3) :: char_num

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

!     /*   if set is not done   */
      if ( iset .eq. 0 ) then

!        /*   read print interval   */
         call read_int1_MPI ( iprint_xyz,  '<iprint_xyz>',  12, iounit )

!        /*   set done   */
         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

!     /*   print interval   */
      if ( iprint_xyz .le. 0 ) return

!     /*   master process only   */
      if ( myrank .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

!cc         open ( iounit_xyz,  file = 'trj.xyz' )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

!cc         open ( iounit_xyz,  file = 'trj.xyz', access = 'append' )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

!-----------------------------------------------------------------------
!     /*   print trajectory                                           */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_xyz) .eq. 0 ) then

            do k = 1, nbead

!              /*   bead number   */
               call int3_to_char( k, char_num )

!              /*   file name  */
               filename = 'trj.' // char_num(1:3) // '.xyz'

!              /*   open file   */
               open ( iounit_xyz, file = filename, access = 'append' )

!              /*   number of atoms, number of beads   */
               write(iounit_xyz,'(2i8)')      natom, nbead

!              /*   step number, potential   */
               write(iounit_xyz,'(i8,f16.8)') istep, pot(k)

               do i = 1, natom

!                 /*   [bohr]  -->  [angstrom]   */
                  xa = x(i,k) *au_length/1.d-10
                  ya = y(i,k) *au_length/1.d-10
                  za = z(i,k) *au_length/1.d-10

!                 /*   unfolded structure   */
                  xb = x(i,k)
                  yb = y(i,k)
                  zb = z(i,k)
                  m1 = mbox(1,i,k)
                  m2 = mbox(2,i,k)
                  m3 = mbox(3,i,k)
                  call pbc_unfold_MPI( xb, yb, zb, m1, m2, m3 )
                  xb = xb *au_length/1.d-10
                  yb = yb *au_length/1.d-10
                  zb = zb *au_length/1.d-10

!                 /*   print atomic species, cartesian coordinates   */
                  write( iounit_xyz, '(a4,6f16.8)' ) &
     &               species(i)(1:4), xa, ya, za, xb, yb, zb

               end do

!              /*   close file   */
               close( iounit_xyz )

            end do

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         continue

      end if

      return
      end





!***********************************************************************
      subroutine analysis_cv_tass_MPI ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, istep, nbead, myrank

      use tass_variables, only : &
     &   rtass, stass, fref_tass, iprint_cv_tass, ntass

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, ioption, itest

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

!     /*   if set is not done   */
      if ( iset .eq. 0 ) then

!        /*   read print interval   */
         call read_int1_MPI &
     &      ( iprint_cv_tass, '<iprint_cv_tass>',  16, iounit )

!        /*   master process only   */
         if ( myrank .eq. 0 ) then

!           /*   check if file exists   */
            call testfile ( 'cv.out', 6, itest )

!           /*   if the file does not exist print header   */
            if ( itest .eq. 1 ) then

!              /*   open file   */
               open ( iounit, file = 'cv.out', access = 'append' )

!              /*   print header   */
               write( iounit, '(a)' ) &
     &        '========================================================'
               write( iounit, '(a)' ) &
     &        '    step           rtass           stass       fref_tass'
               write( iounit, '(a)' ) &
     &        '--------------------------------------------------------'

!              /*   close file   */
               close( iounit )

!           /*   if the file already exists   */
            else

!              /*   do not print header   */
               continue

!           /*   end of if statement   */
            end if

!        /*   master process only   */
         end if

!        /*   set done   */
         iset = 1

!     /*   if set is not done   */
      end if

!-----------------------------------------------------------------------
!     /*   default print interval                                     */
!-----------------------------------------------------------------------

      if ( iprint_cv_tass .le. 0 ) then

         iprint_cv_tass = 10

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         return

      end if

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

!     /*   master process only   */
      if ( myrank .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

!        /*   print interval   */
         if ( mod(istep,iprint_cv_tass) .eq. 0 ) then

!           /*   open file   */
            open ( iounit,  file = 'cv.out', access = 'append' )

!           /*   step number, cv, actual cv   */
            do j = 1, nbead
            do i = 1, ntass
               write(iounit,'(i8,3f16.8)') &
     &            istep, rtass(i,j), stass(i,j), fref_tass(i,j)
            end do
            end do

!           /*   close file   */
            close( iounit )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         continue

      end if

      return
      end





!***********************************************************************
      subroutine analysis_xsf_tass_MPI( ioption )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, au_charge, au_energy, au_length, box, species, &
     &   pot, fx, fy, fz, istep, natom, nbead, iounit, iboundary, &
     &   istep, myrank

      use analysis_variables, only : &
     &   iprint_xsf

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: i, j, itest, ioption

!     /*   real numbers   */
      real(8) :: xa, ya, za, ax, ay, az, bx, by, bz, cx, cy, cz

!     /*   real numbers   */
      real(8) :: const_1, const_2, const_3

!     /*   potential and force   */
      real(8) :: pot_ev, fxj, fyj, fzj

!     /*   flag   */
      integer, save :: iset = 0

!     /*   file name   */
      character(len=11) :: char_file

!     /*   bead number   */
      character(len=3) :: char_num

!-----------------------------------------------------------------------
!     /*   header                                                     */
!-----------------------------------------------------------------------

!     /*   initialization   */
      if ( iset .eq. 0 ) then

!        /*   read print interval   */
         call read_int1_MPI ( iprint_xsf,  '<iprint_xsf>',  12, iounit )

!        /*   master rank only   */
         if ( ( myrank .eq. 0 ) .and. ( iprint_xsf .gt. 0 ) ) then

!           /*   loop of beads   */
            do j = 1, nbead

!              /*   bead number   */
               call int3_to_char( j, char_num )

!              /*   file name   */
               char_file = 'trj.' // char_num // '.xsf'

!              /*   test if file exists */
               call testfile ( char_file, 11, itest )

!              /*   if file does not exist   */
               if ( itest .ne. 0 ) then

!                 /*   file open   */
                  open ( iounit, file = char_file )

!                 /*   write header   */
                  write( iounit, '(a)' ) 'ANIMSTEPS 99999999'

!                 /*   file close   */
                  close( iounit )

!              /*   if file does not exist   */
               end if

!           /*   loop of beads   */
            end do

!        /*   master rank only   */
         end if

!        /*   set completed   */
         iset = 1

!     /*   initialization   */
      end if

!-----------------------------------------------------------------------
!     /*   print condition: every iprint_xsf steps                    */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 ) return
      if ( ( ioption .eq. 0 ) .or. ( ioption .eq. 1 ) ) return
      if ( iprint_xsf .lt. 0 ) return
      if ( mod(istep,iprint_xsf) .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   print xsf file                                             */
!-----------------------------------------------------------------------

!     /*   conversion factor   */
      const_1 = au_length * 1.d+10

!     /*   conversion factor   */
      const_2 =  au_charge / au_energy

!     /*   conversion factor   */
      const_3 =  1.d0 / const_2 / const_1

!     /*   loop of beads   */
      do j = 1, nbead

!        /*   bead number   */
         call int3_to_char( j, char_num )

!        /*   file name   */
         char_file = 'trj.' // char_num // '.xsf'

!        /*   open sample file   */
         open( iounit, file = char_file, access = 'append' )

!        /*   potential energy in eV   */
         pot_ev = pot(j) / const_2

!        /*   write one line   */
         write( iounit, '(a,f24.12,a)' ) &
     &      '# total energy = ', pot_ev, ' eV'

!        /*   for periodic boundary condition   */
         if ( (iboundary .eq. 1) .or. (iboundary .eq. 2) ) then

!           /*   write one line   */
            write( iounit, '(a)' ) 'CRYSTAL'

!           /*   write one line   */
            write( iounit, '(a,i8)' ) 'PRIMVEC ', istep

!           /*   lattice vectors   */
            ax = box(1,1) * const_1
            ay = box(2,1) * const_1
            az = box(3,1) * const_1
            bx = box(1,2) * const_1
            by = box(2,2) * const_1
            bz = box(3,2) * const_1
            cx = box(1,3) * const_1
            cy = box(2,3) * const_1
            cz = box(3,3) * const_1

!           /*   write three lines   */
            write( iounit, '(3f16.8)' ) ax, ay, az
            write( iounit, '(3f16.8)' ) bx, by, bz
            write( iounit, '(3f16.8)' ) cx, cy, cz

!           /*   write one line   */
            write( iounit, '(a,i8)' ) 'PRIMCOORD ', istep

!           /*   write one line   */
            write( iounit, '(i8,i2)' ) natom, 1

!        /*   for free boundary condition   */
         else if ( iboundary .eq. 0 ) then

!           /*   write one line   */
            write( iounit, '(a,i8)' ) 'ATOMS ', istep

!        /*   for free or periodic boundary condition   */
         end if

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   geometry in angstroms   */
            xa = x(i,j) * const_1
            ya = y(i,j) * const_1
            za = z(i,j) * const_1

!           /*   geometry in eV per angstrom   */
            fxj = fx(i,j) * const_3 * dble(nbead)
            fyj = fy(i,j) * const_3 * dble(nbead)
            fzj = fz(i,j) * const_3 * dble(nbead)

!           /*   write one line   */
            write( iounit, '(a4,6f16.8)' ) &
     &         species(i)(1:4), xa, ya, za, fxj, fyj, fzj

!        /*   loop of atoms   */
         end do

!        /*   close input file   */
         close( iounit )

!     /*   loop of beads   */
      end do

      return
      end

