!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     analysis in path integral hybrid Monte Carlo
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine analysis_second_hmc_MPI ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   nkind, iounit, natom_kind, iounit_avg, myrank

      use analysis_variables, only : &
     &   nkindpair, ikindpair, ikindpair_inv, npair_kindpair, &
     &   iprint_bond, iprint_eavg, iprint_trj, iprint_xyz, iprint_xsf, &
     &   iprint_box, iprint_dip, iprint_rdf, iprint_rgy, iprint_dcd

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: itest, ioption, l, j1, j2, k1, k2

!-----------------------------------------------------------------------
!     /*   ioption = 1:  initialize/restart                           */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

!        /*   number of atomic pairs   */

         nkindpair = nkind*(nkind+1)/2

!-----------------------------------------------------------------------
!        /*   ikindpair = label for a pair of kinds                   */
!-----------------------------------------------------------------------

         if ( .not. allocated( ikindpair ) ) &
     &      allocate ( ikindpair(nkind,nkind) )
         if ( .not. allocated( ikindpair_inv ) ) &
     &      allocate ( ikindpair_inv(nkindpair,2) )

         l = 0
         do k1 =  1, nkind
         do k2 = k1, nkind
            l = l + 1
            ikindpair(k1,k2)   = l
            ikindpair(k2,k1)   = l
            ikindpair_inv(l,1) = k1
            ikindpair_inv(l,2) = k2
         end do
         end do

!-----------------------------------------------------------------------
!        /*   npair_kindpair = number of atom pairs                   */
!-----------------------------------------------------------------------

         if ( .not. allocated( npair_kindpair ) ) &
     &      allocate ( npair_kindpair(nkindpair) )

         l = 0
         do k1 =  1, nkind
         do k2 = k1, nkind
            l = l + 1
            j1 = natom_kind(k1)
            j2 = natom_kind(k2)
            if ( k1 .ne. k2 ) npair_kindpair(l) = j1*j2
            if ( k1 .eq. k2 ) npair_kindpair(l) = j1*(j1-1)/2
         end do
         end do

!-----------------------------------------------------------------------
!        /*   step intervals of analysis                              */
!-----------------------------------------------------------------------

         call read_int1_MPI ( iprint_bond, '<iprint_bond>', 13, iounit )
         call read_int1_MPI ( iprint_eavg, '<iprint_eavg>', 13, iounit )
         call read_int1_MPI ( iprint_rgy,  '<iprint_rgy>',  12, iounit )
         call read_int1_MPI ( iprint_rdf,  '<iprint_rdf>',  12, iounit )
         call read_int1_MPI ( iprint_trj,  '<iprint_trj>',  12, iounit )
         call read_int1_MPI ( iprint_dip,  '<iprint_dip>',  12, iounit )
!         call read_int1_MPI ( iprint_mom,  '<iprint_mom>',  12, iounit )
         call read_int1_MPI ( iprint_xyz,  '<iprint_xyz>',  12, iounit )
         call read_int1_MPI ( iprint_xsf,  '<iprint_xsf>',  12, iounit )
         call read_int1_MPI ( iprint_dcd,  '<iprint_dcd>',  12, iounit )
         call read_int1_MPI ( iprint_box,  '<iprint_box>',  12, iounit )

!-----------------------------------------------------------------------
!        /*   check if file called `averages.ini' exists              */
!-----------------------------------------------------------------------

         if ( myrank .eq. 0 ) then
            call testfile ( 'averages.ini', 12, itest )
         end if

!        /*   MPI communication   */
         call my_mpi_bcast_int_0 ( itest )

!-----------------------------------------------------------------------
!        /*   if the file does not exist, initial start.              */
!-----------------------------------------------------------------------

         if ( itest .eq. 1 ) then

            call analysis_bond_MPI ( 0 )
            call analysis_rgy_MPI  ( 0 )
            call analysis_eavg_second_hmc_MPI ( 0 )
            call analysis_rdf_MPI  ( 0 )
            call analysis_trj_MPI  ( 0 )
            call analysis_dip_MPI  ( 0 )
!            call analysis_mom_MPI  ( 0 )
            call analysis_xyz_MPI  ( 0 )
            call analysis_xsf_MPI  ( 0 )
            call analysis_dcd_MPI  ( 0 )
            call analysis_box_MPI  ( 0 )
            call analysis_ratio_hmc_MPI( 0 )

#if defined(aenet2) || defined(aenet_pytorch)
            call analysis_aenet_MPI( 0 )
#endif

#ifdef n2p2
            call analysis_n2p2_MPI( 0 )
#endif


!-----------------------------------------------------------------------
!        /*   if the file exists, restart.                            */
!-----------------------------------------------------------------------

         else

            open ( iounit_avg, file = 'averages.ini' )

            call analysis_bond_MPI ( 1 )
            call analysis_rgy_MPI  ( 1 )
            call analysis_eavg_second_hmc_MPI ( 1 )
            call analysis_rdf_MPI  ( 1 )
            call analysis_trj_MPI  ( 1 )
            call analysis_dip_MPI  ( 1 )
!            call analysis_mom_MPI  ( 1 )
            call analysis_xyz_MPI  ( 1 )
            call analysis_xsf_MPI  ( 1 )
            call analysis_dcd_MPI  ( 1 )
            call analysis_box_MPI  ( 1 )
            call analysis_ratio_hmc_MPI( 1 )

#if defined(aenet2) || defined(aenet_pytorch)
            call analysis_aenet_MPI( 1 )
#endif

#ifdef n2p2
            call analysis_n2p2_MPI( 1 )
#endif

            close( iounit_avg )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  start analysis                               */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         call analysis_bond_MPI ( 2 )
         call analysis_rgy_MPI  ( 2 )
         call analysis_eavg_second_hmc_MPI ( 2 )
         call analysis_rdf_MPI  ( 2 )
         call analysis_trj_MPI  ( 2 )
         call analysis_dip_MPI  ( 2 )
!         call analysis_mom_MPI  ( 2 )
         call analysis_xyz_MPI  ( 2 )
         call analysis_xsf_MPI  ( 2 )
         call analysis_dcd_MPI  ( 2 )
         call analysis_box_MPI  ( 2 )
         call analysis_ratio_hmc_MPI( 2 )

#if defined(aenet2) || defined(aenet_pytorch)
            call analysis_aenet_MPI( 2 )
#endif

#ifdef n2p2
            call analysis_n2p2_MPI( 2 )
#endif

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         open ( iounit_avg, file = 'averages.ini' )

            call analysis_bond_MPI ( 3 )
            call analysis_rgy_MPI  ( 3 )
            call analysis_eavg_second_hmc_MPI ( 3 )
            call analysis_rdf_MPI  ( 3 )
            call analysis_trj_MPI  ( 3 )
            call analysis_dip_MPI  ( 3 )
!            call analysis_mom_MPI  ( 3 )
            call analysis_xyz_MPI  ( 3 )
            call analysis_xsf_MPI  ( 3 )
            call analysis_dcd_MPI  ( 3 )
            call analysis_box_MPI  ( 3 )
            call analysis_ratio_hmc_MPI( 3 )
#if defined(aenet2) || defined(aenet_pytorch)
            call analysis_aenet_MPI( 3 )
#endif

#ifdef n2p2
            call analysis_n2p2_MPI( 3 )
#endif

         close( iounit_avg )

      end if

      return
      end





!***********************************************************************
      subroutine analysis_eavg_second_hmc_MPI ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, temperature, physmass, x, y, z, ux, uy, uz, beta, &
     &   omega_p2, iounit_eavg, iounit_avg, nbead, natom, istep, &
     &   myrank, nprocs

      use analysis_variables, only : &
     &   epot, ekinpri, ekinvir, etot, epot_avg, ekinvir_avg, &
     &   ekinpri_avg, etot_avg, eprivir_avg, specific_heat, &
     &   iprint_eavg

      use hmc_variables, only : &
     &   fx_second, fy_second, fz_second, pot_second

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption, j

      real(8) :: ekinvir1, ekinvir2, ekinpri1, ekinpri2, eprivir

      real(8), save :: dstep = 0.d0

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_eavg .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         epot_avg      = 0.d0
         ekinvir_avg   = 0.d0
         ekinpri_avg   = 0.d0
         etot_avg      = 0.d0

         eprivir_avg = 0.d0

         if ( myrank .eq. 0 ) then

            open ( iounit_eavg, file = 'eavg.out' )

            write(iounit_eavg,'(a)') &
     &      '========' // &
     &      '================================' // &
     &      '================================' // &
     &      '================================' // &
     &      '================================' // &
     &      '================='
            write(iounit_eavg,'(a)') &
     &      '    step' // &
     &      '            epot         ekinvir' // &
     &      '         ekinpri            etot' // &
     &      '        epot_avg     ekinvir_avg' // &
     &      '     ekinpri_avg        etot_avg' // &
     &      '   specific_heat'
            write(iounit_eavg,'(a)') &
     &      '--------' // &
     &      '--------------------------------' // &
     &      '--------------------------------' // &
     &      '--------------------------------' // &
     &      '--------------------------------' // &
     &      '----------------'

            close( iounit_eavg )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         if ( myrank .eq. 0 ) then
            read ( iounit_avg, * ) dstep
            read ( iounit_avg, * ) epot_avg
            read ( iounit_avg, * ) ekinvir_avg
            read ( iounit_avg, * ) ekinpri_avg
            read ( iounit_avg, * ) etot_avg
            read ( iounit_avg, * ) eprivir_avg
         end if

         call my_mpi_bcast_real_0 ( dstep )
         call my_mpi_bcast_real_0 ( epot_avg )
         call my_mpi_bcast_real_0 ( ekinvir_avg )
         call my_mpi_bcast_real_0 ( ekinpri_avg )
         call my_mpi_bcast_real_0 ( etot_avg )
         call my_mpi_bcast_real_0 ( eprivir_avg )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         dstep = dstep + 1.d0

!-----------------------------------------------------------------------
!        /*   potential energy                                        */
!-----------------------------------------------------------------------

         epot = 0.d0

         do j = 1, nbead

            if ( mod( j-1, nprocs ) .ne. myrank ) cycle

            epot = epot + pot_second(j)

         end do

         call my_mpi_allreduce_real_0 ( epot )

         epot = epot/dble(nbead)

!-----------------------------------------------------------------------
!        /*   kinetic energy by virial estimator                      */
!-----------------------------------------------------------------------

!        /*   first term   */

         ekinvir1 = 1.5d0*dble(natom)*boltz*temperature

!        /*   second term   */

         ekinvir2 = 0.d0

         do j = 1, nbead

            if ( mod( j-1, nprocs ) .ne. myrank ) cycle

            do i = 1, natom
               ekinvir2 = ekinvir2 + (x(i,j)-ux(i,1))*fx_second(i,j) &
     &                             + (y(i,j)-uy(i,1))*fy_second(i,j) &
     &                             + (z(i,j)-uz(i,1))*fz_second(i,j)
            end do

         end do

         call my_mpi_allreduce_real_0 ( ekinvir2 )

         ekinvir2 = - 0.5d0*ekinvir2

!        /*   the sum of kinetic energy   */

         ekinvir = ekinvir1 + ekinvir2

!-----------------------------------------------------------------------
!        /*   kinetic energy by primitive estimator                   */
!-----------------------------------------------------------------------

!        /*   first term   */

         ekinpri1 = 1.5d0*dble(natom)*dble(nbead)*boltz*temperature

!        /*   second term   */

         ekinpri2 = 0.d0

         do j = 1, nbead

            if ( mod( j-1, nprocs ) .ne. myrank ) cycle

            do i = 1, natom

               if ( j .eq. nbead ) then

                  ekinpri2 = ekinpri2 &
     &            + physmass(i)*(x(i,nbead)-x(i,1))*(x(i,nbead)-x(i,1)) &
     &            + physmass(i)*(y(i,nbead)-y(i,1))*(y(i,nbead)-y(i,1)) &
     &            + physmass(i)*(z(i,nbead)-z(i,1))*(z(i,nbead)-z(i,1))

               else

                  ekinpri2 = ekinpri2 &
     &            + physmass(i)*(x(i,j)-x(i,j+1))*(x(i,j)-x(i,j+1)) &
     &            + physmass(i)*(y(i,j)-y(i,j+1))*(y(i,j)-y(i,j+1)) &
     &            + physmass(i)*(z(i,j)-z(i,j+1))*(z(i,j)-z(i,j+1))

               end if

            end do

         end do

         call my_mpi_allreduce_real_0 ( ekinpri2 )

         ekinpri2 = - 0.5d0*omega_p2*ekinpri2

!        /*   the sum of kinetic energy   */

         ekinpri = ekinpri1 + ekinpri2

!-----------------------------------------------------------------------
!        /*   total energy                                            */
!-----------------------------------------------------------------------

         etot = ekinvir + epot

!-----------------------------------------------------------------------
!        /*   accumulative averages                                   */
!-----------------------------------------------------------------------

         epot_avg    = epot   /dstep + epot_avg   *(dstep-1.d0)/dstep
         ekinvir_avg = ekinvir/dstep + ekinvir_avg*(dstep-1.d0)/dstep
         ekinpri_avg = ekinpri/dstep + ekinpri_avg*(dstep-1.d0)/dstep
         etot_avg    = etot   /dstep + etot_avg   *(dstep-1.d0)/dstep

!-----------------------------------------------------------------------
!        /*   specific heat                                           */
!-----------------------------------------------------------------------

         eprivir = ( ekinpri + epot ) * ( ekinvir + epot )

         eprivir_avg = eprivir / dstep &
     &                  + eprivir_avg * (dstep-1.d0)/dstep

         eprivir = ( ekinpri_avg + epot_avg ) &
     &           * ( ekinvir_avg + epot_avg )

         specific_heat &
     &       = beta * beta * ( eprivir_avg -  eprivir ) &
     &       + 1.5d0 * dble(natom)

!         specific_heat
!     &       = beta * beta * ( eprivir_avg - etot_avg*etot_avg  )
!     &       + 1.5d0 * dble(natom)

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_eavg) .eq. 0 ) then

            if ( myrank .eq. 0 ) then

               open ( iounit_eavg, file='eavg.out', access='append' )

               write(iounit_eavg,'(i8,8f16.8,f16.4)') &
     &            istep, epot, ekinvir, ekinpri, etot, &
     &            epot_avg, ekinvir_avg, ekinpri_avg, etot_avg, &
     &            specific_heat

               close( iounit_eavg )

            end if

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         if ( myrank .eq. 0 ) then

            write( iounit_avg, '(e24.16)' ) dstep
            write( iounit_avg, '(e24.16)' ) epot_avg
            write( iounit_avg, '(e24.16)' ) ekinvir_avg
            write( iounit_avg, '(e24.16)' ) ekinpri_avg
            write( iounit_avg, '(e24.16)' ) etot_avg
            write( iounit_avg, '(e24.16)' ) eprivir_avg

         end if

      end if

      return
      end

