!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 24, 2022 by M. Shiga
!      Description:     analysis in replica exchange hybrid Monte Carlo
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine analysis_rehmc_MPI ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   nkind, natom_kind, iounit_avg, iounit, &
     &   ipotential, myrank

      use analysis_variables, only : &
     &   nkindpair, ikindpair, npair_kindpair, iprint_xyz, iprint_xsf, &
     &   iprint_eavg, iprint_rdf, iprint_trj, ikindpair_inv, &
     &   iprint_rdf, iprint_bond, iprint_box, iprint_dcd

      use alchem_variables, only : &
     &   iprint_alc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: itest, ioption, l, j1, j2, k1, k2

!-----------------------------------------------------------------------
!     /*   ioption = 1:  initialize/restart                           */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

!        /*   number of atomic pairs   */

         nkindpair = nkind*(nkind+1)/2

!-----------------------------------------------------------------------
!        /*   ikindpair = label for a pair of kinds                   */
!-----------------------------------------------------------------------

         if ( .not. allocated( ikindpair ) ) &
     &      allocate ( ikindpair(nkind,nkind) )
         if ( .not. allocated( ikindpair_inv ) ) &
     &      allocate ( ikindpair_inv(nkindpair,2) )

         l = 0
         do k1 =  1, nkind
         do k2 = k1, nkind
            l = l + 1
            ikindpair(k1,k2)   = l
            ikindpair(k2,k1)   = l
            ikindpair_inv(l,1) = k1
            ikindpair_inv(l,2) = k2
         end do
         end do

!-----------------------------------------------------------------------
!        /*   npair_kindpair = number of atom pairs                   */
!-----------------------------------------------------------------------

         if ( .not. allocated( npair_kindpair ) ) &
     &      allocate ( npair_kindpair(nkindpair) )

         l = 0
         do k1 =  1, nkind
         do k2 = k1, nkind
            l = l + 1
            j1 = natom_kind(k1)
            j2 = natom_kind(k2)
            if ( k1 .ne. k2 ) npair_kindpair(l) = j1*j2
            if ( k1 .eq. k2 ) npair_kindpair(l) = j1*(j1-1)/2
         end do
         end do

!-----------------------------------------------------------------------
!        /*   step intervals of analysis                              */
!-----------------------------------------------------------------------

         call read_int1_MPI ( iprint_bond, '<iprint_bond>', 13, iounit )
         call read_int1_MPI ( iprint_eavg, '<iprint_eavg>', 13, iounit )
         call read_int1_MPI ( iprint_rdf,  '<iprint_rdf>',  12, iounit )
         call read_int1_MPI ( iprint_trj,  '<iprint_trj>',  12, iounit )
         call read_int1_MPI ( iprint_xyz,  '<iprint_xyz>',  12, iounit )
         call read_int1_MPI ( iprint_xsf,  '<iprint_xsf>',  12, iounit )
         call read_int1_MPI ( iprint_dcd,  '<iprint_dcd>',  12, iounit )
         call read_int1_MPI ( iprint_box,  '<iprint_box>',  12, iounit )

!         call read_int1_MPI ( iprint_rgy, '<iprint_rgy>',  12, iounit )
!         call read_int1_MPI ( iprint_dip, '<iprint_dip>',  12, iounit )
!         call read_int1_MPI ( iprint_mom, '<iprint_mom>',  12, iounit )

         if ( ipotential(1:7) .eq. 'ALCHEM ' ) then
            call read_int1_MPI &
     &         ( iprint_alc,  '<iprint_alc>',  12, iounit )
         end if

!-----------------------------------------------------------------------
!        /*   check if file called `averages.ini' exists              */
!-----------------------------------------------------------------------

         if ( myrank .eq. 0 ) then
            call testfile ( 'averages.ini', 12, itest )
         end if

         call my_mpi_bcast_int_0( itest )

!-----------------------------------------------------------------------
!        /*   if the file does not exist, initial start.              */
!-----------------------------------------------------------------------

         if ( itest .eq. 1 ) then

            call analysis_box_rehmc_MPI  ( 0 )
            call analysis_bond_rehmc_MPI ( 0 )
            call analysis_eavg_rehmc_MPI ( 0 )
            call analysis_rdf_rehmc_MPI  ( 0 )
            call analysis_trj_MPI  ( 0 )
            call analysis_xyz_rehmc_MPI  ( 0 )
            call analysis_xsf_rehmc_MPI  ( 0 )
            call analysis_dcd_beadwise_MPI  ( 0, 1 )
            call analysis_ratio_rehmc_MPI( 0 )

!            call analysis_rgy_MPI  ( 0 )
!            call analysis_dip_MPI  ( 0 )
!            call analysis_mom_MPI  ( 0 )

            if ( ipotential(1:7) .eq. 'ALCHEM ' ) then
               call analysis_alchem_MPI  ( 0 )
            end if

#if defined(aenet2) || defined(aenet_pytorch)
            call analysis_aenet_MPI( 0 )
#endif

!-----------------------------------------------------------------------
!        /*   if the file exists, restart.                            */
!-----------------------------------------------------------------------

         else

            if ( myrank .eq. 0 ) then
               open ( iounit_avg, file = 'averages.ini' )
            end if

            call analysis_box_rehmc_MPI  ( 1 )
            call analysis_bond_rehmc_MPI ( 1 )
            call analysis_eavg_rehmc_MPI ( 1 )
            call analysis_rdf_rehmc_MPI  ( 1 )
            call analysis_trj_MPI  ( 1 )
            call analysis_xyz_rehmc_MPI  ( 1 )
            call analysis_xsf_rehmc_MPI  ( 1 )
            call analysis_dcd_beadwise_MPI  ( 1, 1 )
            call analysis_ratio_rehmc_MPI( 1 )

!            call analysis_rgy_MPI  ( 1 )
!            call analysis_dip_MPI  ( 1 )
!            call analysis_mom_MPI  ( 1 )

            if ( ipotential(1:7) .eq. 'ALCHEM ' ) then
               call analysis_alchem_MPI  ( 1 )
            end if

#if defined(aenet2) || defined(aenet_pytorch)
            call analysis_aenet_MPI( 1 )
#endif

            if ( myrank .eq. 0 ) then
               close( iounit_avg )
            end if

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  start analysis                               */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

            call analysis_box_rehmc_MPI  ( 2 )
            call analysis_bond_rehmc_MPI ( 2 )
            call analysis_eavg_rehmc_MPI ( 2 )
            call analysis_rdf_rehmc_MPI  ( 2 )
            call analysis_trj_MPI  ( 2 )
            call analysis_xyz_rehmc_MPI  ( 2 )
            call analysis_xsf_rehmc_MPI  ( 2 )
            call analysis_dcd_beadwise_MPI  ( 2, 1 )
            call analysis_ratio_rehmc_MPI( 2 )

!            call analysis_rgy_MPI  ( 2 )
!            call analysis_dip_MPI  ( 2 )
!            call analysis_mom_MPI  ( 2 )

            if ( ipotential(1:7) .eq. 'ALCHEM ' ) then
               call analysis_alchem_MPI  ( 2 )
            end if

#if defined(aenet2) || defined(aenet_pytorch)
            call analysis_aenet_MPI( 2 )
#endif

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         if ( myrank .eq. 0 ) then
            open ( iounit_avg, file = 'averages.ini' )
         end if

            call analysis_box_rehmc_MPI  ( 3 )
            call analysis_bond_rehmc_MPI ( 3 )
            call analysis_eavg_rehmc_MPI ( 3 )
            call analysis_rdf_rehmc_MPI  ( 3 )
            call analysis_trj_MPI  ( 3 )
            call analysis_xyz_rehmc_MPI  ( 3 )
            call analysis_xsf_rehmc_MPI  ( 3 )
            call analysis_dcd_beadwise_MPI  ( 3, 1 )
            call analysis_ratio_rehmc_MPI( 3 )

!            call analysis_rgy_MPI  ( 3 )
!            call analysis_dip_MPI  ( 3 )
!            call analysis_mom_MPI  ( 3 )

         if ( ipotential(1:7) .eq. 'ALCHEM ' ) then
            call analysis_alchem_MPI  ( 3 )
         end if

#if defined(aenet2) || defined(aenet_pytorch)
            call analysis_aenet_MPI( 3 )
#endif

         if ( myrank .eq. 0 ) then
            close( iounit_avg )
         end if

      end if

      return
      end





!***********************************************************************
      subroutine analysis_bond_rehmc_MPI ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, box_bead, iounit_bond, iounit_avg, ikind, &
     &   natom, nbead, istep, nkind, myrank

      use analysis_variables, only : &
     &   iprint_bond, npair_kindpair, nkindpair, ikindpair

      use rehmc_variables, only : &
     &   rdist, rdist2, rdist_avg, rdist2_avg

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ioption, i, j, k, m

      real(8) :: r1, r2, rx, ry, rz

      real(8), dimension(3,3) :: box

      real(8), save :: dstep = 0.d0

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_bond .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         if ( .not. allocated( rdist ) ) &
     &      allocate ( rdist(nkindpair,nbead)  )
         if ( .not. allocated( rdist2 ) ) &
     &      allocate ( rdist2(nkindpair,nbead) )
         if ( .not. allocated( rdist_avg ) ) &
     &      allocate ( rdist_avg(nkindpair,nbead)  )
         if ( .not. allocated( rdist2_avg ) ) &
     &      allocate ( rdist2_avg(nkindpair,nbead) )

         rdist_avg(:,:)  = 0.d0
         rdist2_avg(:,:) = 0.d0

         if ( myrank .eq. 0 ) then

            open ( iounit_bond, file = 'bond.out' )

            write(iounit_bond,'(a)') &
     &      '==============================================' // &
     &      '========================'
            write(iounit_bond,'(a)') &
     &      '    step  bead  1   2        rdist      rdist2' // &
     &      '   rdist_avg  rdist_avg'
            write(iounit_bond,'(a)') &
     &      '----------------------------------------------' // &
     &      '------------------------'

            close( iounit_bond )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         if ( .not. allocated( rdist ) ) &
     &      allocate ( rdist(nkindpair,nbead)  )
         if ( .not. allocated( rdist2 ) ) &
     &      allocate ( rdist2(nkindpair,nbead) )
         if ( .not. allocated( rdist_avg ) ) &
     &      allocate ( rdist_avg(nkindpair,nbead)  )
         if ( .not. allocated( rdist2_avg ) ) &
     &      allocate ( rdist2_avg(nkindpair,nbead) )

         if ( myrank .eq. 0 ) then
            read ( iounit_avg, * ) dstep
            read ( iounit_avg, * ) rdist_avg(:,:)
            read ( iounit_avg, * ) rdist2_avg(:,:)
         end if

         call my_mpi_bcast_real_0 ( dstep )
         call my_mpi_bcast_real_2 ( rdist_avg, nkindpair, nbead )
         call my_mpi_bcast_real_2 ( rdist2_avg, nkindpair, nbead )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         dstep = dstep + 1.d0

         rdist(:,:)  = 0.d0
         rdist2(:,:) = 0.d0

         do k = 1, nbead

            box(:,:) = box_bead(:,:,k)

            do i = 1, natom-1
            do j = i+1, natom

               m = ikindpair ( ikind(i), ikind(j) )

               rx     = x(i,k) - x(j,k)
               ry     = y(i,k) - y(j,k)
               rz     = z(i,k) - z(j,k)

               call pbc_atom_MPI ( rx, ry, rz )

               r2     = rx*rx + ry*ry + rz*rz
               r1     = sqrt(r2)

               rdist(m,k)  = rdist(m,k)  + r1
               rdist2(m,k) = rdist2(m,k) + r2

            end do
            end do

         end do

         do k = 1, nbead
         do m = 1, nkindpair

            if ( npair_kindpair(m) .eq. 0 ) cycle

            rdist(m,k)  = rdist(m,k)  /npair_kindpair(m)
            rdist2(m,k) = rdist2(m,k) /npair_kindpair(m)

         end do
         end do

!-----------------------------------------------------------------------
!        /*   averages                                                */
!-----------------------------------------------------------------------

         rdist_avg(:,:)  = rdist(:,:) /dstep &
     &                   + rdist_avg(:,:)*(dstep-1.d0)/dstep

         rdist2_avg(:,:) = rdist2(:,:) /dstep &
     &                   + rdist2_avg(:,:)*(dstep-1.d0)/dstep

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         if ( myrank .eq. 0 ) then
         if ( mod(istep,iprint_bond) .eq. 0 ) then

            open ( iounit_bond, file = 'bond.out', access = 'append' )

            do k = 1, nbead

               m = 0

               do i = 1, nkind
               do j = i, nkind

                  m = m + 1

                  if ( npair_kindpair(m) .eq. 0 ) cycle

                  write(iounit_bond,'(i8,i6,1x,i3,1x,i3,4f12.5)') &
     &               istep, k, i, j, &
     &               rdist(m,k), rdist2(m,k), &
     &               rdist_avg(m,k), rdist2_avg(m,k)

               end do
               end do

            end do

            close( iounit_bond )

         end if
         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         if ( myrank .eq. 0 ) then
            write( iounit_avg, '(e24.16)' ) dstep
            write( iounit_avg, '(e24.16)' ) rdist_avg(:,:)
            write( iounit_avg, '(e24.16)' ) rdist2_avg(:,:)
         end if

      end if

      return
      end





!***********************************************************************
      subroutine analysis_eavg_rehmc_MPI( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, iounit_eavg, iounit_avg, nbead, istep, myrank

      use analysis_variables, only : &
     &   iprint_eavg

      use rehmc_variables, only : &
     &   pot_avg

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption, j

      real(8), save :: dstep = 0.d0

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_eavg .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         if ( .not. allocated( pot_avg ) ) &
     &      allocate ( pot_avg(nbead) )

         pot_avg(:)  = 0.d0

         if ( myrank .eq. 0 ) then

            open ( iounit_eavg, file = 'eavg.out' )

            write(iounit_eavg,'(a)') &
     &         '=============================================='
            write(iounit_eavg,'(a)') &
     &         '    step  bead       potential   potential_avg'
            write(iounit_eavg,'(a)') &
     &         '----------------------------------------------'

            close( iounit_eavg )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         if ( .not. allocated( pot_avg ) ) &
     &      allocate ( pot_avg(nbead) )

         if ( myrank .eq. 0 ) then

            read ( iounit_avg, * ) dstep
            read ( iounit_avg, * ) pot_avg(:)

         end if

         call my_mpi_bcast_real_0( dstep )
         call my_mpi_bcast_real_1( pot_avg, nbead )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         dstep = dstep + 1.d0

!-----------------------------------------------------------------------
!        /*   calculate potential averages                            */
!-----------------------------------------------------------------------

         do i = 1, nbead
            pot_avg(i) = pot(i) / dstep &
     &                 + pot_avg(i) * (dstep-1.d0)/dstep
         end do

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         if ( mod( istep, iprint_eavg ) .eq. 0 ) then

            if ( myrank .eq. 0 ) then

               open ( iounit_eavg, file='eavg.out', access='append' )

               do j = 1, nbead

                  write( iounit_eavg, '(i8,i6,2f16.8)' ) &
     &               istep, j, pot(j), pot_avg(j)

               end do

               close( iounit_eavg )

            end if

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         if ( myrank .eq. 0 ) then

            write( iounit_avg, '(e24.16)' ) dstep
            write( iounit_avg, '(e24.16)' ) pot_avg(:)

         end if

      end if

      return
      end





!***********************************************************************
      subroutine analysis_rdf_rehmc_MPI ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, volume_bead, x, y, z, box_bead, iounit, nkind, natom, &
     &   nbead, iboundary, iounit_avg, istep, ikind, natom_kind, &
     &   ensemble, myrank, nprocs

      use analysis_variables, only : &
     &   params_rdf, volume_bead_avg, iprint_rdf, nkindpair, &
     &   ikindpair, npair_kindpair

      use rehmc_variables, only : &
     &   rdf_bead_avg, rdf_bead, rdf_n_bead, rdf_g_bead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption, j, nmesh, k, l, m, n

      real(8) :: dr, dv, fact_n, rs, rl, rx, ry, rz, r, r1, r2

      real(8), dimension(nbead) :: volume_rdf

      real(8), dimension(3,3) :: box

      real(8) :: fact_g = 0.d0

      real(8), save :: dstep = 0.d0

      character(len=3) :: char_num

      character(len=80) :: char_file

      real(8) :: fact_i, fact_j, rdf_cn_i, rdf_cn_j

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_rdf .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   volume used in rdf                                         */
!-----------------------------------------------------------------------

      if      ( iboundary .eq. 0 ) then

         continue

      else if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then

         if ( ( ensemble(1:4) .eq. 'NVT ' ) .or. &
     &        ( ensemble(1:4) .eq. 'NPT ' ) ) then

            if ( istep .eq. 1 ) then
               volume_rdf(:) = volume_bead(:)
            else
               volume_rdf(:) = volume_bead_avg(:)
            end if

         else

            if ( istep .eq. 1 ) then
               volume_rdf(:) = volume_bead(:)
            else
               volume_rdf(:) = volume_bead(:)
            end if

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         call read_realn_MPI(params_rdf, 3, '<params_rdf>', 12, iounit)

!        /*   shortest interatomic distance   */
         rs = params_rdf(1)

!        /*   longest interatomic distance    */
         rl = params_rdf(2)

!        /*   mesh size    */
         dr = params_rdf(3)

!        /*   number of meshes    */
         nmesh = nint( ( rl - rs )/dr ) + 2

!        /*   memory allocation   */
         if ( .not. allocated( rdf_bead ) ) &
     &      allocate ( rdf_bead(nmesh,nkindpair,nbead) )
         if ( .not. allocated( rdf_n_bead ) ) &
     &      allocate ( rdf_n_bead(nmesh,nkindpair,nbead) )
         if ( .not. allocated( rdf_g_bead ) ) &
     &      allocate ( rdf_g_bead(nmesh,nkindpair,nbead) )
         if ( .not. allocated( rdf_bead_avg ) ) &
     &      allocate ( rdf_bead_avg(nmesh,nkindpair,nbead) )

         rdf_bead_avg(:,:,:)  = 0.d0

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         call read_realn_MPI(params_rdf, 3, '<params_rdf>', 12, iounit)

!        /*   shortest interatomic distance   */
         rs = params_rdf(1)

!        /*   longest interatomic distance    */
         rl = params_rdf(2)

!        /*   mesh size    */
         dr = params_rdf(3)

!        /*   number of meshes    */
         nmesh = nint( ( rl - rs )/dr ) + 2

!        /*   memory allocation   */
         if ( .not. allocated( rdf_bead ) ) &
     &      allocate ( rdf_bead(nmesh,nkindpair,nbead) )
         if ( .not. allocated( rdf_n_bead ) ) &
     &      allocate ( rdf_n_bead(nmesh,nkindpair,nbead) )
         if ( .not. allocated( rdf_g_bead ) ) &
     &      allocate ( rdf_g_bead(nmesh,nkindpair,nbead) )
         if ( .not. allocated( rdf_bead_avg ) ) &
     &      allocate ( rdf_bead_avg(nmesh,nkindpair,nbead) )

         if ( myrank .eq. 0 ) then
            read ( iounit_avg, * ) dstep
            read ( iounit_avg, * ) rdf_bead_avg(:,:,:)
         end if

         call my_mpi_bcast_real_0 ( dstep )
         call my_mpi_bcast_real_3 &
     &      ( rdf_bead_avg, nmesh, nkindpair, nbead )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         dstep = dstep + 1.d0

!-----------------------------------------------------------------------
!        /*   calculate interatomic rdf                               */
!-----------------------------------------------------------------------

!        /*   zero clear   */
         rdf_bead(:,:,:)  =  0.d0

!        /*   shortest interatomic distance   */
         rs = params_rdf(1)

!        /*   longest interatomic distance    */
         rl = params_rdf(2)

!        /*   mesh size    */
         dr = params_rdf(3)

!        /*   number of meshes    */
         nmesh = nint( ( rl - rs )/dr ) + 2

         do k = 1, nbead

            if ( mod( k-1, nprocs ) .ne. myrank ) cycle

            box(:,:) = box_bead(:,:,k)

            do i = 1, natom-1
            do j = i+1, natom

               m = ikindpair ( ikind(i), ikind(j) )

!              /*   calculate interatomic distance   */

               rx     = x(i,k) - x(j,k)
               ry     = y(i,k) - y(j,k)
               rz     = z(i,k) - z(j,k)

               call pbc_atom_MPI ( rx, ry, rz )

               r2     = rx*rx + ry*ry + rz*rz
               r1     = sqrt(r2)

!              /*   mesh point   */
               l = nint( (r1-rs)/dr ) + 1

!              /*   skip if out of the range   */
               if ( l .lt. 1     ) cycle
               if ( l .gt. nmesh ) cycle

!              /*   add one   */
               rdf_bead(l,m,k)  = rdf_bead(l,m,k)  + 1.d0

            end do
            end do

         end do

         call my_mpi_allreduce_real_3 &
     &      ( rdf_bead, nmesh, nkindpair, nbead )

         rdf_bead_avg(:,:,:) = rdf_bead(:,:,:)/dstep &
     &                       + rdf_bead_avg(:,:,:)*(dstep-1.d0)/dstep

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_rdf) .eq. 0 ) then

            do n = 1, nbead

               m = 0

               do i = 1, nkind
               do j = i, nkind

                  m = m + 1

                  if ( npair_kindpair(m) .eq. 0 ) cycle

!                 /*   normalization factor   */
                  fact_n = 1.d0/dble(npair_kindpair(m))/dr

                  do l = 1, nmesh

                     r = rs + (l-1)*dr

                     rdf_n_bead(l,m,n) = fact_n*rdf_bead_avg(l,m,n)

                     r1 = r + 0.5d0*dr
                     r2 = r - 0.5d0*dr

                     dv = 4.d0/3.d0*pi*( r1*r1*r1 - r2*r2*r2 )

                     if ( dv .eq. 0.d0 ) then
                        fact_g = 0.d0
                     else
                        if      ( iboundary .eq. 0 ) then
                           fact_g = fact_n*dr/(r1-r2)
                        else if ( iboundary .eq. 1 ) then
                           fact_g = fact_n*(dr*volume_rdf(n)/dv)
                        else if ( iboundary .eq. 2 ) then
                           fact_g = fact_n*(dr*volume_rdf(n)/dv)
                        end if
                     end if

                     rdf_g_bead(l,m,n) = fact_g*rdf_bead_avg(l,m,n)

                  end do

               end do
               end do

               if ( myrank .eq. 0 ) then

                  call int3_to_char( n, char_num )

                  char_file = ('rdf.' // char_num // '.out')

                  open ( iounit, file = char_file )

                  write(iounit,'(a)') &
     &               '================================' // &
     &               '===================================='
                  write(iounit,'(a)') &
     &               '   i   j      r [au]   pdf: d(r)' // &
     &               '   rdf: g(r)   cn: n1(r)   cn: n2(r)'
                  write(iounit,'(a)') &
     &               '--------------------------------' // &
     &               '------------------------------------'

                  m = 0

                  do i = 1, nkind
                  do j = i, nkind

                     m = m + 1

                     if ( npair_kindpair(m) .eq. 0 ) cycle

                     rdf_cn_i = 0.d0
                     rdf_cn_j = 0.d0

                     do l = 1, nmesh

                        r = rs + (l-1)*dr

                        r1 = r + 0.5d0*dr
                        r2 = max( r - 0.5d0*dr, 0.d0 )

                        dv = 4.d0/3.d0*pi*( r1*r1*r1 - r2*r2*r2 )

                        if      ( iboundary .eq. 0 ) then
                          fact_i = dble(natom_kind(i)) * (r1-r2)
                          fact_j = dble(natom_kind(j)) * (r1-r2)
                        else if ( iboundary .eq. 1 ) then
                          fact_i = dble(natom_kind(i)) *dv/volume_rdf(n)
                          fact_j = dble(natom_kind(j)) *dv/volume_rdf(n)
                        else if ( iboundary .eq. 2 ) then
                          fact_i = dble(natom_kind(i)) *dv/volume_rdf(n)
                          fact_j = dble(natom_kind(j)) *dv/volume_rdf(n)
                        else
                          fact_i = dble(natom_kind(i)) *dv/volume_rdf(n)
                          fact_j = dble(natom_kind(j)) *dv/volume_rdf(n)
                        end if

                        rdf_cn_i = rdf_cn_i + rdf_g_bead(l,m,n) * fact_i
                        rdf_cn_j = rdf_cn_j + rdf_g_bead(l,m,n) * fact_j

                        write(iounit,'(1x,i3,1x,i3,5f12.4)') &
     &                     i, j, r, &
     &                     rdf_n_bead(l,m,n), rdf_g_bead(l,m,n), &
     &                     rdf_cn_i, rdf_cn_j

                     end do

                  end do
                  end do

                  close( iounit )

               end if

            end do

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         if ( myrank .eq. 0 ) then

            write( iounit_avg, '(e24.16)' ) dstep
            write( iounit_avg, '(e24.16)' ) rdf_bead_avg(:,:,:)

         end if

      end if

      return
      end





!***********************************************************************
      subroutine analysis_alchem_MPI( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, temperature, nbead, iounit_avg, istep, iounit_alc, myrank

      use rehmc_variables, only : &
     &   pot_avg

      use alchem_variables, only : &
     &   pot_alchem, pot_alchem_avg, iprint_alc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption, j

      real(8) :: dfree, dpot, dentropy, dkin, &
     &           dfree_avg, dpot_avg, dentropy_avg, dkin_avg

      real(8), save :: dstep = 0.d0

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_alc .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         if ( .not. allocated( pot_alchem_avg ) ) &
     &      allocate ( pot_alchem_avg(nbead,nbead) )

         pot_alchem_avg(:,:)  = 0.d0

         if ( myrank .eq. 0 ) then

            open ( iounit_alc, file = 'alc.out' )

            write( iounit_alc, '(a)' ) &
     &      '========================================================' &
     &   // '================================================'
            write( iounit_alc, '(a)' ) &
     &      '    step              dA              dV              dS' &
     &   // '          dA_avg          dV_avg          dS_avg'
            write( iounit_alc, '(a)' ) &
     &      '--------------------------------------------------------' &
     &   // '------------------------------------------------'

            close( iounit_alc )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         if ( .not. allocated( pot_alchem_avg ) ) &
     &      allocate ( pot_alchem_avg(nbead,nbead) )

         if ( myrank .eq. 0 ) then
            read ( iounit_avg, * ) dstep
            read ( iounit_avg, * ) pot_alchem_avg(:,:)
         end if

         call my_mpi_bcast_real_0( dstep )
         call my_mpi_bcast_real_2( pot_alchem_avg, nbead, nbead )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         dstep = dstep + 1.d0

!-----------------------------------------------------------------------
!        /*   calculate potential averages                            */
!-----------------------------------------------------------------------

         do j = 1, nbead
         do i = 1, nbead
            pot_alchem_avg(i,j) &
     &                 = pot_alchem(i,j) / dstep &
     &                 + pot_alchem_avg(i,j) * (dstep-1.d0)/dstep
         end do
         end do

!-----------------------------------------------------------------------
!        /*   free energy difference: instantaneous                   */
!-----------------------------------------------------------------------

         dfree = 0.d0

         do i = 1, nbead-1
            dfree = dfree + pot_alchem(i+1,i) - pot(i)
            dfree = dfree - pot_alchem(i,i+1) + pot(i+1)
         end do

         dfree = dfree / 2.d0

         dkin = 0.d0

         dpot = pot(nbead) - pot(1)

         dentropy = ( dkin + dpot - dfree ) / temperature

!-----------------------------------------------------------------------
!        /*   free energy difference: average                         */
!-----------------------------------------------------------------------

         dfree_avg = 0.d0

         do i = 1, nbead-1
            dfree_avg = dfree_avg + pot_alchem_avg(i+1,i) - pot_avg(i)
            dfree_avg = dfree_avg - pot_alchem_avg(i,i+1) + pot_avg(i+1)
         end do

         dfree_avg = dfree_avg / 2.d0

         dkin_avg = 0.d0

         dpot_avg = pot_avg(nbead) - pot_avg(1)

         dentropy_avg = ( dkin_avg + dpot_avg - dfree_avg ) /temperature

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         if ( mod( istep, iprint_alc ) .eq. 0 ) then

         if ( myrank .eq. 0 ) then

            open( iounit_alc, file='alc.out', access='append' )

            write( iounit_alc, '(i8,6f16.8)' ) &
     &         istep, dfree, dpot, dentropy, &
     &         dfree_avg, dpot_avg, dentropy_avg

!            do l = 1, nbead
!            do j = 1, nbead, 8
!
!               k = min( j+8, nbead )
!
!               if ( j .eq. 1 ) then
!
!                  write( iounit_alc, '(i8,8f16.8)' )
!     &               istep, ( pot_alchem(i,l), i = j, k )
!
!                  write( iounit_alc, '(i8,8f16.8)' )
!     &               istep, ( pot_alchem_avg(i,l), i = j, k )
!
!               else
!
!                  write( iounit_alc, '(8x,8f16.8)' )
!     &                      ( pot_alchem(i,l), i = j, k )
!
!                  write( iounit_alc, '(8x,8f16.8)' )
!     &                      ( pot_alchem_avg(i,l), i = j, k )
!
!               end if
!
!            end do
!            end do

            close( iounit_alc )

         end if

      end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         if ( myrank .eq. 0 ) then

            write( iounit_avg, '(e24.16)' ) dstep
            write( iounit_avg, '(e24.16)' ) pot_alchem_avg(:,:)

         end if

      end if

      return
      end





!***********************************************************************
      subroutine analysis_ratio_rehmc_MPI ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit_avg, myrank

      use rehmc_variables, only : &
     &   ratio_rem, ratio_rex, ratio_hmc, naccept_rem, &
     &   naccept_rex, naccept_hmc, nreject_rem, &
     &   nreject_rex, nreject_hmc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ioption

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         continue

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         if ( myrank .eq. 0 ) then

            read ( iounit_avg, * ) naccept_hmc
            read ( iounit_avg, * ) nreject_hmc
            read ( iounit_avg, * ) ratio_hmc

            read ( iounit_avg, * ) naccept_rem
            read ( iounit_avg, * ) nreject_rem
            read ( iounit_avg, * ) ratio_rem

            read ( iounit_avg, * ) naccept_rex
            read ( iounit_avg, * ) nreject_rex
            read ( iounit_avg, * ) ratio_rex

         end if

         call my_mpi_bcast_int_0 ( naccept_hmc )
         call my_mpi_bcast_int_0 ( nreject_hmc )
         call my_mpi_bcast_real_0( ratio_hmc )

         call my_mpi_bcast_int_0 ( naccept_rem )
         call my_mpi_bcast_int_0 ( nreject_rem )
         call my_mpi_bcast_real_0( ratio_rem )

         call my_mpi_bcast_int_0 ( naccept_rex )
         call my_mpi_bcast_int_0 ( nreject_rex )
         call my_mpi_bcast_real_0( ratio_rex )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         continue

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         if ( myrank .eq. 0 ) then

            write( iounit_avg, '(i8)' )     naccept_hmc
            write( iounit_avg, '(i8)' )     nreject_hmc
            write( iounit_avg, '(e24.16)' ) ratio_hmc

            write( iounit_avg, '(i8)' )     naccept_rem
            write( iounit_avg, '(i8)' )     nreject_rem
            write( iounit_avg, '(e24.16)' ) ratio_rem

            write( iounit_avg, '(i8)' )     naccept_rex
            write( iounit_avg, '(i8)' )     nreject_rex
            write( iounit_avg, '(e24.16)' ) ratio_rex

         end if

      end if

      return
      end





!***********************************************************************
      subroutine analysis_xyz_rehmc_MPI ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, au_length, x, y, z, species, &
     &   iounit, iounit_xyz, istep, natom, nbead, myrank

      use meta_variables, only : &
     &   iprint_xyz

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, k, ioption

      real(8) :: xa, ya, za

      character(len=11) :: filename

      character(len=3) :: char_num

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

!     /*   if set is not done   */
      if ( iset .eq. 0 ) then

!        /*   read print interval   */
         call read_int1_MPI ( iprint_xyz,  '<iprint_xyz>',  12, iounit )

!        /*   set done   */
         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

!     /*   print interval   */
      if ( iprint_xyz .le. 0 ) return

!     /*   master process only   */
      if ( myrank .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

!cc         open ( iounit_xyz,  file = 'trj.xyz' )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

!cc         open ( iounit_xyz,  file = 'trj.xyz', access = 'append' )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

!-----------------------------------------------------------------------
!     /*   print trajectory                                           */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_xyz) .eq. 0 ) then

            do k = 1, nbead

!              /*   bead number   */
               call int3_to_char( k, char_num )

!              /*   file name  */
               filename = 'trj.' // char_num(1:3) // '.xyz'

!              /*   open file   */
               open ( iounit_xyz, file = filename, access = 'append' )

!              /*   number of atoms, number of beads   */
               write(iounit_xyz,'(2i8)')      natom, nbead

!              /*   step number, potential   */
               write(iounit_xyz,'(i8,f16.8)') istep, pot(k)

               do i = 1, natom

!                 /*   [bohr]  -->  [angstrom]   */
                  xa = x(i,k) *au_length/1.d-10
                  ya = y(i,k) *au_length/1.d-10
                  za = z(i,k) *au_length/1.d-10

!                 /*   print atomic species, cartesian coordinates   */
                  write( iounit_xyz, '(a4,3f16.8)' ) &
     &               species(i)(1:4), xa, ya, za

               end do

!              /*   close file   */
               close( iounit_xyz )

            end do

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         continue

      end if

      return
      end





!***********************************************************************
      subroutine analysis_box_rehmc_MPI ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   vx, vy, vz, volume_bead, pres_bead_iso, pres_bead, &
     &   vir_bead, physmass, iounit_box, box_bead, iboundary, nbead, &
     &   istep, iounit_box, iounit_avg, natom, ensemble, myrank

      use analysis_variables, only : &
     &   box_bead_avg, pres_bead_avg, volume_bead_avg, &
     &   pres_bead_iso_avg, iprint_box

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer, save :: iset = 0
      integer :: ioption, i, j

      real(8), save :: dstep = 0.d0

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         if ( .not. allocated( volume_bead_avg ) ) &
     &      allocate( volume_bead_avg(nbead) )
         if ( .not. allocated( box_bead_avg ) ) &
     &      allocate( pres_bead_iso_avg(nbead) )
         if ( .not. allocated( box_bead_avg ) ) &
     &      allocate( box_bead_avg(3,3,nbead) )
         if ( .not. allocated( pres_bead_avg ) ) &
     &      allocate( pres_bead_avg(3,3,nbead) )

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   only for periodic boundary                                 */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         if ( ( ensemble(1:4) .eq. 'NVT ' ) .or. &
     &        ( ensemble(1:4) .eq. 'NPT ' ) ) then

            volume_bead_avg(:)    = 0.d0
            box_bead_avg(:,:,:)   = 0.d0
            pres_bead_iso_avg(:)  = 0.d0
            pres_bead_avg(:,:,:)  = 0.d0

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         if ( ( ensemble(1:4) .eq. 'NVT ' ) .or. &
     &        ( ensemble(1:4) .eq. 'NPT ' ) ) then

            if ( myrank .eq. 0 ) then
               read ( iounit_avg, * ) dstep
               read ( iounit_avg, * ) volume_bead_avg(:)
               read ( iounit_avg, * ) box_bead_avg(:,:,:)
               read ( iounit_avg, * ) pres_bead_iso_avg(:)
               read ( iounit_avg, * ) pres_bead_avg(:,:,:)
            end if

            call my_mpi_bcast_real_0 ( dstep )
            call my_mpi_bcast_real_1 ( volume_bead_avg, nbead )
            call my_mpi_bcast_real_3 ( box_bead_avg, 3, 3, nbead )
            call my_mpi_bcast_real_1 ( pres_bead_iso_avg, nbead )
            call my_mpi_bcast_real_3 ( pres_bead_avg, 3, 3, nbead )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

!        /*   number of steps   */
         dstep = dstep + 1.d0

         if ( ( ensemble(1:4) .eq. 'NVT ' ) .or. &
     &        ( ensemble(1:4) .eq. 'NPT ' ) ) then

!           /*   calculate internal pressure   */

            do j = 1, nbead

               pres_bead(:,:,j) =  vir_bead(:,:,j)

               do i = 1, natom

                  pres_bead(1,1,j) = pres_bead(1,1,j) &
     &                             + physmass(i)*vx(i,j)*vx(i,j)
                  pres_bead(1,2,j) = pres_bead(1,2,j) &
     &                             + physmass(i)*vx(i,j)*vy(i,j)
                  pres_bead(1,3,j) = pres_bead(1,3,j) &
     &                             + physmass(i)*vx(i,j)*vz(i,j)
                  pres_bead(2,1,j) = pres_bead(2,1,j) &
     &                             + physmass(i)*vy(i,j)*vx(i,j)
                  pres_bead(2,2,j) = pres_bead(2,2,j) &
     &                             + physmass(i)*vy(i,j)*vy(i,j)
                  pres_bead(2,3,j) = pres_bead(2,3,j) &
     &                             + physmass(i)*vy(i,j)*vz(i,j)
                  pres_bead(3,1,j) = pres_bead(3,1,j) &
     &                             + physmass(i)*vz(i,j)*vx(i,j)
                  pres_bead(3,2,j) = pres_bead(3,2,j) &
     &                             + physmass(i)*vz(i,j)*vy(i,j)
                  pres_bead(3,3,j) = pres_bead(3,3,j) &
     &                             + physmass(i)*vz(i,j)*vz(i,j)

               end do

               pres_bead(:,:,j) = pres_bead(:,:,j) / volume_bead(j)

!              /*   isotropic pressure   */

               pres_bead_iso(j) = ( pres_bead(1,1,j) &
     &                            + pres_bead(2,2,j) &
     &                            + pres_bead(3,3,j) ) / 3.d0

!              /*   average volume   */

               volume_bead_avg(j)  = volume_bead(j) /dstep &
     &            + volume_bead_avg(j)*(dstep-1.d0)/dstep

!              /*   average box   */

               box_bead_avg(:,:,j)  = box_bead(:,:,j) /dstep &
     &            + box_bead_avg(:,:,j)*(dstep-1.d0)/dstep

!              /*   average volume   */

               pres_bead_iso_avg(j)  = pres_bead_iso(j) /dstep &
     &            + pres_bead_iso_avg(j)*(dstep-1.d0)/dstep

!              /*   average pressure   */

               pres_bead_avg(:,:,j)  = pres_bead(:,:,j) /dstep &
     &            + pres_bead_avg(:,:,j)*(dstep-1.d0)/dstep

            end do

            if ( iprint_box .gt. 0 ) then
            if ( mod(istep,iprint_box) .eq. 0 ) then
            if ( myrank .eq. 0 ) then

!              /*   output   */

               open ( iounit_box, file = 'box.out', access = 'append' )

               do j = 1, nbead

                  write( iounit_box, '(i8,2f16.2)' ) &
     &               istep, volume_bead(j), volume_bead_avg(j)
                  write( iounit_box, '(8x,6f16.6)' ) &
     &               box_bead(1,1,j), box_bead(1,2,j), box_bead(1,3,j), &
     &               box_bead_avg(1,1,j), box_bead_avg(1,2,j), &
     &               box_bead_avg(1,3,j)
                  write( iounit_box, '(8x,6f16.6)' ) &
     &               box_bead(2,1,j), box_bead(2,2,j), box_bead(2,3,j), &
     &               box_bead_avg(2,1,j), box_bead_avg(2,2,j), &
     &               box_bead_avg(2,3,j)
                  write( iounit_box, '(8x,6f16.6)' ) &
     &               box_bead(3,1,j), box_bead(3,2,j), box_bead(3,3,j), &
     &               box_bead_avg(3,1,j), box_bead_avg(3,2,j), &
     &               box_bead_avg(3,3,j)

                  write( iounit_box, '(i8,2f16.12)' ) &
     &               istep, pres_bead_iso(j), pres_bead_iso_avg(j)
                  write( iounit_box, '(8x,6f16.12)' ) &
     &               pres_bead(1,1,j), pres_bead(1,2,j), &
     &               pres_bead(1,3,j), &
     &               pres_bead_avg(1,1,j), pres_bead_avg(1,2,j), &
     &               pres_bead_avg(1,3,j)
                  write( iounit_box, '(8x,6f16.12)' ) &
     &               pres_bead(2,1,j), pres_bead(2,2,j), &
     &               pres_bead(2,3,j), &
     &               pres_bead_avg(2,1,j), pres_bead_avg(2,2,j), &
     &               pres_bead_avg(2,3,j)
                  write( iounit_box, '(8x,6f16.12)' ) &
     &               pres_bead(3,1,j), pres_bead(3,2,j), &
     &               pres_bead(3,3,j), &
     &               pres_bead_avg(3,1,j), pres_bead_avg(3,2,j), &
     &               pres_bead_avg(3,3,j)

               end do

               close( iounit_box )

            end if
            end if
            end if

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         if ( ( ensemble(1:4) .eq. 'NVT ' ) .or. &
     &        ( ensemble(1:4) .eq. 'NPT ' ) ) then

            if ( myrank .eq. 0 ) then
               write( iounit_avg, '(e24.16)' ) dstep
               write( iounit_avg, '(e24.16)' ) volume_bead_avg(:)
               write( iounit_avg, '(e24.16)' ) box_bead_avg(:,:,:)
               write( iounit_avg, '(e24.16)' ) pres_bead_iso_avg(:)
               write( iounit_avg, '(e24.16)' ) pres_bead_avg(:,:,:)
            end if

         end if

      end if

      return
      end





!***********************************************************************
      subroutine analysis_xsf_rehmc_MPI( ioption )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, au_charge, au_energy, au_length, box_bead, species, &
     &   pot, fx, fy, fz, istep, natom, nbead, iounit, iboundary, &
     &   istep, myrank

      use analysis_variables, only : &
     &   iprint_xsf

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: i, j, itest, ioption

!     /*   real numbers   */
      real(8) :: xa, ya, za, ax, ay, az, bx, by, bz, cx, cy, cz

!     /*   real numbers   */
      real(8) :: const_1, const_2, const_3

!     /*   potential and force   */
      real(8) :: pot_ev, fxj, fyj, fzj

!     /*   flag   */
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   print condition: every iprint_xsf steps                    */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 ) return
      if ( ( ioption .eq. 0 ) .or. ( ioption .eq. 1 ) ) return
      if ( iprint_xsf .le. 0 ) return
      if ( mod(istep,iprint_xsf) .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   header                                                     */
!-----------------------------------------------------------------------

!     /*   initialization   */
      if ( iset .eq. 0 ) then

!        /*   test if file exists */
         call testfile ( 'trj.xsf', 7, itest )

!        /*   if file does not exist   */
         if ( itest .ne. 0 ) then

!            /*   file open   */
             open ( iounit, file = 'trj.xsf' )

!            /*   write header   */
             write( iounit, '(a)' ) 'ANIMSTEPS 99999999'

!            /*   file close   */
             close( iounit )

!        /*   if file does not exist   */
         end if

!        /*   set completed   */
         iset = 1

!     /*   initialization   */
      end if

!-----------------------------------------------------------------------
!     /*   print xsf file                                             */
!-----------------------------------------------------------------------

!     /*   conversion factor   */
      const_1 = au_length * 1.d+10

!     /*   conversion factor   */
      const_2 =  au_charge / au_energy

!     /*   conversion factor   */
      const_3 =  1.d0 / const_2 / const_1

!     /*   open sample file   */
      open( iounit, file = 'trj.xsf', access = 'append' )

!     /*   loop of beads   */
      do j = 1, nbead

!        /*   potential energy in eV   */
         pot_ev = pot(j) / const_2

!        /*   write one line   */
         write( iounit, '(a,f24.12,a)' ) &
     &      '# total energy = ', pot_ev, ' eV'

!     /*   loop of beads   */
      end do

!     /*   for periodic boundary condition   */
      if ( (iboundary .eq. 1) .or. (iboundary .eq. 2) ) then

!        /*   write one line   */
         write( iounit, '(a)' ) 'CRYSTAL'

!        /*   write one line   */
         write( iounit, '(a,i8)' ) 'PRIMVEC ', istep

!        /*   lattice vectors   */
         ax = box_bead(1,1,j) * const_1
         ay = box_bead(2,1,j) * const_1
         az = box_bead(3,1,j) * const_1
         bx = box_bead(1,2,j) * const_1
         by = box_bead(2,2,j) * const_1
         bz = box_bead(3,2,j) * const_1
         cx = box_bead(1,3,j) * const_1
         cy = box_bead(2,3,j) * const_1
         cz = box_bead(3,3,j) * const_1

!        /*   write three lines   */
         write( iounit, '(3f16.8)' ) ax, ay, az
         write( iounit, '(3f16.8)' ) bx, by, bz
         write( iounit, '(3f16.8)' ) cx, cy, cz

!        /*   write one line   */
         write( iounit, '(a,i8)' ) 'PRIMCOORD ', istep

!        /*   write one line   */
         write( iounit, '(i8,i2)' ) natom*nbead, 1

!     /*   for free boundary condition   */
      else if ( iboundary .eq. 0 ) then

!        /*   write one line   */
         write( iounit, '(a,i8)' ) 'ATOMS ', istep

!     /*   for free or periodic boundary condition   */
      end if

!     /*   loop of beads   */
      do j = 1, nbead

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   geometry in angstroms   */
            xa = x(i,j) * const_1
            ya = y(i,j) * const_1
            za = z(i,j) * const_1

!           /*   geometry in eV per angstrom   */
            fxj = fx(i,j) * const_3 * dble(nbead)
            fyj = fy(i,j) * const_3 * dble(nbead)
            fzj = fz(i,j) * const_3 * dble(nbead)

!           /*   write one line   */
            write( iounit, '(a4,6f16.8)' ) &
     &         species(i)(1:4), xa, ya, za, fxj, fyj, fzj

!        /*   loop of atoms   */
         end do

!     /*   loop of beads   */
      end do

!     /*   close input file   */
      close( iounit )

      return
      end


!***********************************************************************
      subroutine analysis_rehmc_npt_MPI ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   nkind, natom_kind, iounit_avg, iounit, myrank

!      use common_variables, only :
!     &   ipotential

      use analysis_variables, only : &
     &   nkindpair, ikindpair, npair_kindpair, ikindpair_inv, &
     &   iprint_xyz, iprint_box, iprint_rdf, iprint_bond, iprint_dcd, &
     &   iprint_xsf, iprint_eavg, iprint_rdf, iprint_trj

!      use alchem_variables, only :
!     &   iprint_alc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: itest, ioption, l, j1, j2, k1, k2

!-----------------------------------------------------------------------
!     /*   ioption = 1:  initialize/restart                           */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

!        /*   number of atomic pairs   */

         nkindpair = nkind*(nkind+1)/2

!-----------------------------------------------------------------------
!        /*   ikindpair = label for a pair of kinds                   */
!-----------------------------------------------------------------------

         if ( .not. allocated( ikindpair ) ) &
     &      allocate ( ikindpair(nkind,nkind) )
         if ( .not. allocated( ikindpair_inv ) ) &
     &      allocate ( ikindpair_inv(nkindpair,2) )

         l = 0
         do k1 =  1, nkind
         do k2 = k1, nkind
            l = l + 1
            ikindpair(k1,k2)   = l
            ikindpair(k2,k1)   = l
            ikindpair_inv(l,1) = k1
            ikindpair_inv(l,2) = k2
         end do
         end do

!-----------------------------------------------------------------------
!        /*   npair_kindpair = number of atom pairs                   */
!-----------------------------------------------------------------------

         if ( .not. allocated( npair_kindpair ) ) &
     &      allocate ( npair_kindpair(nkindpair) )

         l = 0
         do k1 =  1, nkind
         do k2 = k1, nkind
            l = l + 1
            j1 = natom_kind(k1)
            j2 = natom_kind(k2)
            if ( k1 .ne. k2 ) npair_kindpair(l) = j1*j2
            if ( k1 .eq. k2 ) npair_kindpair(l) = j1*(j1-1)/2
         end do
         end do

!-----------------------------------------------------------------------
!        /*   step intervals of analysis                              */
!-----------------------------------------------------------------------

         call read_int1_MPI ( iprint_box,  '<iprint_box>',  12, iounit )
         call read_int1_MPI ( iprint_xyz,  '<iprint_xyz>',  12, iounit )
         call read_int1_MPI ( iprint_rdf,  '<iprint_rdf>',  12, iounit )
         call read_int1_MPI ( iprint_xsf,  '<iprint_xsf>',  12, iounit )
         call read_int1_MPI ( iprint_bond, '<iprint_bond>', 13, iounit )
         call read_int1_MPI ( iprint_eavg, '<iprint_eavg>', 13, iounit )
         call read_int1_MPI ( iprint_trj,  '<iprint_trj>',  12, iounit )
         call read_int1_MPI ( iprint_dcd,  '<iprint_dcd>',  12, iounit )

!        call read_int1_MPI ( iprint_rgy,  '<iprint_rgy>',  12, iounit )
!        call read_int1_MPI ( iprint_dip,  '<iprint_dip>',  12, iounit )
!        call read_int1_MPI ( iprint_mom,  '<iprint_mom>',  12, iounit )

!         if ( ipotential(1:7) .eq. 'ALCHEM ' ) then
!            call read_int1_MPI
!     &         ( iprint_alc,  '<iprint_alc>',  12, iounit )
!         end if

!-----------------------------------------------------------------------
!        /*   check if file called `averages.ini' exists              */
!-----------------------------------------------------------------------

         if ( myrank .eq. 0 ) then
            call testfile ( 'averages.ini', 12, itest )
         end if

         call my_mpi_bcast_int_0( itest )

!-----------------------------------------------------------------------
!        /*   if the file does not exist, initial start.              */
!-----------------------------------------------------------------------

         if ( itest .eq. 1 ) then

            call analysis_box_rehmc_MPI  ( 0 )
            call analysis_xyz_rehmc_MPI  ( 0 )
            call analysis_rdf_rehmc_MPI  ( 0 )
            call analysis_ratio_rehmc_MPI( 0 )
            call analysis_xsf_rehmc_MPI  ( 0 )
            call analysis_trj_MPI  ( 0 )
            call analysis_bond_rehmc_MPI ( 0 )
            call analysis_eavg_rehmc_MPI ( 0 )
            call analysis_dcd_beadwise_MPI  ( 0, 1 )

!            call analysis_rgy_MPI  ( 0 )
!            call analysis_dip_MPI  ( 0 )
!            call analysis_mom_MPI  ( 0 )
!
!            if ( ipotential(1:7) .eq. 'ALCHEM ' ) then
!               call analysis_alchem_MPI  ( 0 )
!            end if

#if defined(aenet2) || defined(aenet_pytorch)
            call analysis_aenet_MPI( 0 )
#endif

!-----------------------------------------------------------------------
!        /*   if the file exists, restart.                            */
!-----------------------------------------------------------------------

         else

            if ( myrank .eq. 0 ) then
               open ( iounit_avg, file = 'averages.ini')
            end if

            call analysis_box_rehmc_MPI  ( 1 )
            call analysis_xyz_rehmc_MPI  ( 1 )
            call analysis_rdf_rehmc_MPI  ( 1 )
            call analysis_ratio_rehmc_MPI( 1 )
            call analysis_xsf_rehmc_MPI  ( 1 )
            call analysis_trj_MPI  ( 1 )
            call analysis_bond_rehmc_MPI ( 1 )
            call analysis_eavg_rehmc_MPI ( 1 )
            call analysis_dcd_beadwise_MPI  ( 1, 1 )

!            call analysis_rgy_MPI  ( 1 )
!            call analysis_dip_MPI  ( 1 )
!            call analysis_mom_MPI  ( 1 )

!            if ( ipotential(1:7) .eq. 'ALCHEM ' ) then
!               call analysis_alchem_MPI  ( 1 )
!            end if

#if defined(aenet2) || defined(aenet_pytorch)
            call analysis_aenet_MPI( 1 )
#endif

            if ( myrank .eq. 0 ) then
               close( iounit_avg )
            end if

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  start analysis                               */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

            call analysis_box_rehmc_MPI  ( 2 )
            call analysis_xyz_rehmc_MPI  ( 2 )
            call analysis_rdf_rehmc_MPI  ( 2 )
            call analysis_ratio_rehmc_MPI( 2 )
            call analysis_xsf_rehmc_MPI  ( 2 )
            call analysis_trj_MPI  ( 2 )
            call analysis_bond_rehmc_MPI ( 2 )
            call analysis_eavg_rehmc_MPI ( 2 )
            call analysis_dcd_beadwise_MPI  ( 2, 1 )

!            call analysis_rgy_MPI  ( 2 )
!            call analysis_dip_MPI  ( 2 )
!            call analysis_mom_MPI  ( 2 )

!            if ( ipotential(1:7) .eq. 'ALCHEM ' ) then
!               call analysis_alchem_MPI  ( 2 )
!            end if

#if defined(aenet2) || defined(aenet_pytorch)
            call analysis_aenet_MPI( 2 )
#endif

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         if ( myrank .eq. 0 ) then
            open ( iounit_avg, file = 'averages.ini')
         end if

         call analysis_box_rehmc_MPI  ( 3 )
         call analysis_xyz_rehmc_MPI  ( 3 )
         call analysis_rdf_rehmc_MPI  ( 3 )
         call analysis_ratio_rehmc_MPI( 3 )
         call analysis_xsf_rehmc_MPI  ( 3 )
         call analysis_trj_MPI  ( 3 )
         call analysis_bond_rehmc_MPI ( 3 )
         call analysis_eavg_rehmc_MPI ( 3 )
         call analysis_dcd_beadwise_MPI  ( 3, 1 )

!         call analysis_rgy_MPI  ( 3 )
!         call analysis_dip_MPI  ( 3 )
!         call analysis_mom_MPI  ( 3 )

!         if ( ipotential(1:7) .eq. 'ALCHEM ' ) then
!            call analysis_alchem_MPI  ( 3 )
!         end if
         
#if defined(aenet2) || defined(aenet_pytorch)
            call analysis_aenet_MPI( 3 )
#endif

         if ( myrank .eq. 0 ) then
            close( iounit_avg )
         end if

      end if

      return
      end
