!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     tass reconstruction
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine analysis_rec_tass( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : istep

      use tass_variables, only : ntass_cons, iprint_rec_tass

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: ioption

!-----------------------------------------------------------------------
!     //   initial setup
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) return
      if ( ioption .eq. 1 ) return

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

!     /*   print interval   */
      if ( iprint_rec_tass .le. 0 )  return

!     /*   print interval   */
      if ( mod(istep,iprint_rec_tass) .ne. 0 ) return

!-----------------------------------------------------------------------
!     //   setup reconstruction
!-----------------------------------------------------------------------

      call analysis_rec_setup_tass

!-----------------------------------------------------------------------
!     //   main subroutines
!-----------------------------------------------------------------------

!     //   with constraint
      if      ( ntass_cons .eq. 1 ) then

         call analysis_rec_cons_tass

!     //   without constraint
      else if ( ntass_cons .eq. 0 ) then

         call analysis_rec_free_tass

      end if

      return
      end





!***********************************************************************
      subroutine analysis_rec_cons_tass
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : nbead

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize local variables
      implicit none

!     //   integers
      integer :: ibead

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

      call analysis_rec_cons_memory_tass( 0 )

!-----------------------------------------------------------------------
!     //   start loop of beads
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!        //   read cv.out and hills.ini
!-----------------------------------------------------------------------

         call analysis_rec_cons_read_tass( ibead )

!-----------------------------------------------------------------------
!        //   calculate c(t) potential in metadynamics
!-----------------------------------------------------------------------

         call analysis_rec_cons_ct_tass

!-----------------------------------------------------------------------
!        //   calculate v(s,t)
!-----------------------------------------------------------------------

         call analysis_rec_cons_vb_tass

!-----------------------------------------------------------------------
!        //   calculate free energy gradient
!-----------------------------------------------------------------------

         call analysis_rec_cons_fegrad_tass( ibead )

!-----------------------------------------------------------------------
!        //   calculate conditional probability
!-----------------------------------------------------------------------

         call analysis_rec_cons_pc_tass( ibead )

!-----------------------------------------------------------------------
!     //   end loop of beads
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     //   calculate free energy with respect to constraints
!-----------------------------------------------------------------------

      call analysis_rec_cons_add_tass

!-----------------------------------------------------------------------
!     //   calculate total free energy
!-----------------------------------------------------------------------

      call analysis_rec_cons_fenergy_tass

!-----------------------------------------------------------------------
!     //   print free energy
!-----------------------------------------------------------------------

      call analysis_rec_print_tass

!-----------------------------------------------------------------------
!     //   memory deallocation
!-----------------------------------------------------------------------

      call analysis_rec_cons_memory_tass( 1 )

      return
      end





!***********************************************************************
      subroutine analysis_rec_cons_add_tass
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   nbead, iounit

      use tass_variables, only : &
     &   fenergy_cons, dr_cv, fegrad_cons, label_cons_tass, ntass_cons

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize local variables
      implicit none

!     //   integers
      integer :: ibead, icvc

!-----------------------------------------------------------------------
!     //   constrained md only
!-----------------------------------------------------------------------

      if ( ntass_cons .eq. 0 ) return

!-----------------------------------------------------------------------
!     //   constrained cv
!-----------------------------------------------------------------------

      icvc = label_cons_tass(1)

!-----------------------------------------------------------------------
!     //   calculate projected free energy along constraint
!-----------------------------------------------------------------------

!     //   first bead zero
      fenergy_cons(1) = 0.d0

!     //   loop of beads
      do ibead = 2, nbead

!        //   add difference from previous bead
         fenergy_cons(ibead) = fenergy_cons(ibead-1) &
     &      + 0.5d0 * dr_cv(icvc) &
     &      * ( fegrad_cons(ibead) + fegrad_cons(ibead-1) )

!     //   loop of beads
      end do

!-----------------------------------------------------------------------
!     //   print projected free energy along constraint
!-----------------------------------------------------------------------

      open ( iounit, file = 'rec_1d.out' )

      do ibead = 1, nbead
         write( iounit, '(i8,f16.8)' ) ibead, fenergy_cons(ibead)
      end do

      close( iounit )

      return
      end





!***********************************************************************
      subroutine analysis_rec_cons_ct_tass
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use tass_variables, only : &
     &   vhills_0d, vhills_1d, vhills_2d, vhills_3d, ct_cv, &
     &   beta_a_tass, beta_b_tass, gw_cv, gh_cv, gc_cv, dr_cv, rmin_cv, &
     &   beta_tamd, dtemp_meta, nmesh_cv, ntass_meta, nstep_hills, &
     &   label_meta_tass, ipbc_tass

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize local variables
      implicit none

!     //   real numbers
      real(8) :: enum, eden, gwi, gwj, gwk, gci, gcj, gck, &
     &           gh, gwi_inv2, gwj_inv2, gwk_inv2, rmesh_icvm, &
     &           rmesh_jcvm, rmesh_kcvm, fcvi, fcvj, fcvk, &
     &           dcvi, dcvj, dcvk, vbias_mesh

!     //   integers
      integer :: mi, mj, mk, jstep_hills, icvm, jcvm, kcvm

!-----------------------------------------------------------------------
!     //   0D case: calculate c(t) potential in metadynamics
!-----------------------------------------------------------------------

!     //   0D case
      if ( ntass_meta .eq. 0 ) then

!        //   initialize metadynamics free energy
         vhills_0d = 0.d0

!        //   c function
         ct_cv(:) = 0.d0

!-----------------------------------------------------------------------
!     //   1D case: calculate c(t) potential in metadynamics
!-----------------------------------------------------------------------

!     //   1D case
      else if ( ntass_meta .eq. 1 ) then

!        //   metadynamics cv
         icvm = label_meta_tass(1)

!        //   initialize metadynamics free energy
         vhills_1d(:) = 0.d0

!        //   loop of hills steps
         do jstep_hills = 1, nstep_hills

!           //   numerator
            enum = 0.d0

!           //   denominator
            eden = 0.d0

!           //   loop of metadynamics meshes
            do mi = 1, nmesh_cv(icvm)

!              //   gaussian width
               gwi = gw_cv(icvm,jstep_hills)

!              //   gaussian center
               gci = gc_cv(icvm,jstep_hills)

!              //   gaussian height
               gh = gh_cv(jstep_hills)

!              //   gaussian width inverse squared
               gwi_inv2 = 1.d0 / ( gwi * gwi )

!              //   mesh point
               rmesh_icvm = rmin_cv(icvm) + dble(mi-1) * dr_cv(icvm)

!              //   difference between mesh point and gaussian center
               dcvi = rmesh_icvm - gci

!              /*   periodic boundary condition   */
               call pbc_tass( ipbc_tass(icvm), dcvi )

!              //   exponent value
               fcvi = 0.5d0 * dcvi * dcvi * gwi_inv2

!              //   hill height at hills mesh
               vhills_1d(mi) = vhills_1d(mi) &
     &            + gh * exp( - fcvi )

!              //   bias potential for conventional metadynamics
               vbias_mesh = vhills_1d(mi)

!              //   well tempered metadynamics
               if ( dtemp_meta .gt. 0.d0 ) then

!                 //   bias potential
                  vbias_mesh = log( 1.d0 + beta_b_tass * vbias_mesh ) &
     &                       / beta_b_tass

!              //   end of if statement
               end if

!              //   numerator
               enum = enum + exp( beta_a_tass * vbias_mesh )

!              //   denominator
               eden = eden + exp( beta_b_tass * vbias_mesh )

!           //   loop of metadynamics meshes
            end do

!           //   c function
            if ( eden .gt. 0.d0 ) then
               ct_cv(jstep_hills) = log( enum / eden ) / beta_tamd
            end if

!        //   loop of hills steps
         end do

!-----------------------------------------------------------------------
!     //   2D case: calculate c(t) potential in metadynamics
!-----------------------------------------------------------------------

!     //   2D case
      else if ( ntass_meta .eq. 2 ) then

!        //   metadynamics cv
         icvm = label_meta_tass(1)
         jcvm = label_meta_tass(2)

!        //   initialize metadynamics free energy
         vhills_2d(:,:) = 0.d0

!        //   loop of hills steps
         do jstep_hills = 1, nstep_hills

!           //   numerator
            enum = 0.d0

!           //   denominator
            eden = 0.d0

!           //   loop of metadynamics meshes
            do mi = 1, nmesh_cv(icvm)
            do mj = 1, nmesh_cv(jcvm)

!              //   gaussian width
               gwi = gw_cv(icvm,jstep_hills)
               gwj = gw_cv(jcvm,jstep_hills)

!              //   gaussian center
               gci = gc_cv(icvm,jstep_hills)
               gcj = gc_cv(jcvm,jstep_hills)

!              //   gaussian height
               gh = gh_cv(jstep_hills)

!              //   gaussian width inverse squared
               gwi_inv2 = 1.d0 / ( gwi * gwi )
               gwj_inv2 = 1.d0 / ( gwj * gwj )

!              //   mesh point
               rmesh_icvm = rmin_cv(icvm) + dble(mi-1) * dr_cv(icvm)
               rmesh_jcvm = rmin_cv(jcvm) + dble(mj-1) * dr_cv(jcvm)

!              //   difference between mesh point and gaussian center
               dcvi = rmesh_icvm - gci
               dcvj = rmesh_jcvm - gcj

!              /*   periodic boundary condition   */
               call pbc_tass( ipbc_tass(icvm), dcvi )
               call pbc_tass( ipbc_tass(jcvm), dcvj )

!              //   exponent value
               fcvi = 0.5d0 * dcvi * dcvi * gwi_inv2
               fcvj = 0.5d0 * dcvj * dcvj * gwj_inv2

!              //   hill height at hills mesh
               vhills_2d(mi,mj) = vhills_2d(mi,mj) &
     &            + gh * exp( - fcvi - fcvj )

!              //   bias potential for conventional metadynamics
               vbias_mesh = vhills_2d(mi,mj)

!              //   well tempered metadynamics
               if ( dtemp_meta .gt. 0.d0 ) then

!                 //   bias potential
                  vbias_mesh = log( 1.d0 + beta_b_tass * vbias_mesh ) &
     &                       / beta_b_tass

!              //   end of if statement
               end if

!              //   numerator
               enum = enum + exp( beta_a_tass * vbias_mesh )

!              //   denominator
               eden = eden + exp( beta_b_tass * vbias_mesh )

!           //   loop of metadynamics meshes
            end do
            end do

!           //   c function
            if ( eden .gt. 0.d0 ) then
               ct_cv(jstep_hills) = log( enum / eden ) / beta_tamd
            end if

!        //   loop of hills steps
         end do

!-----------------------------------------------------------------------
!     //   3D case:  calculate c(t) potential in metadynamics
!-----------------------------------------------------------------------

!     //   3D case
      else if ( ntass_meta .eq. 3 ) then

!        //   metadynamics cv
         icvm = label_meta_tass(1)
         jcvm = label_meta_tass(2)
         kcvm = label_meta_tass(3)

!        //   initialize metadynamics free energy
         vhills_3d(:,:,:) = 0.d0

!        //   loop of hills steps
         do jstep_hills = 1, nstep_hills

!           //   numerator
            enum = 0.d0

!           //   denominator
            eden = 0.d0

!           //   loop of metadynamics meshes
            do mi = 1, nmesh_cv(icvm)
            do mj = 1, nmesh_cv(jcvm)
            do mk = 1, nmesh_cv(kcvm)

!              //   gaussian width
               gwi = gw_cv(icvm,jstep_hills)
               gwj = gw_cv(jcvm,jstep_hills)
               gwk = gw_cv(kcvm,jstep_hills)

!              //   gaussian center
               gci = gc_cv(icvm,jstep_hills)
               gcj = gc_cv(jcvm,jstep_hills)
               gck = gc_cv(kcvm,jstep_hills)

!              //   gaussian height
               gh = gh_cv(jstep_hills)

!              //   gaussian width inverse squared
               gwi_inv2 = 1.d0 / ( gwi * gwi )
               gwj_inv2 = 1.d0 / ( gwj * gwj )
               gwk_inv2 = 1.d0 / ( gwk * gwk )

!              //   mesh point
               rmesh_icvm = rmin_cv(icvm) + dble(mi-1) * dr_cv(icvm)
               rmesh_jcvm = rmin_cv(jcvm) + dble(mj-1) * dr_cv(jcvm)
               rmesh_kcvm = rmin_cv(kcvm) + dble(mk-1) * dr_cv(kcvm)

!              //   difference between mesh point and gaussian center
               dcvi = rmesh_icvm - gci
               dcvj = rmesh_jcvm - gcj
               dcvk = rmesh_kcvm - gck

!              /*   periodic boundary condition   */
               call pbc_tass( ipbc_tass(icvm), dcvi )
               call pbc_tass( ipbc_tass(jcvm), dcvj )
               call pbc_tass( ipbc_tass(kcvm), dcvk )

!              //   exponent value
               fcvi = 0.5d0 * dcvi * dcvi * gwi_inv2
               fcvj = 0.5d0 * dcvj * dcvj * gwj_inv2
               fcvk = 0.5d0 * dcvk * dcvk * gwk_inv2

!              //   hill height at hills mesh
               vhills_3d(mi,mj,mk) = vhills_3d(mi,mj,mk) &
     &            + gh * exp( - fcvi - fcvj - fcvk )

!              //   bias potential for conventional metadynamics
               vbias_mesh = vhills_3d(mi,mj,mk)

!              //   well tempered metadynamics
               if ( dtemp_meta .gt. 0.d0 ) then

!                 //   bias potential
                  vbias_mesh = log( 1.d0 + beta_b_tass * vbias_mesh ) &
     &                       / beta_b_tass

!              //   end of if statement
               end if

!              //   numerator
               enum = enum + exp( beta_a_tass * vbias_mesh )

!              //   denominator
               eden = eden + exp( beta_b_tass * vbias_mesh )

!           //   loop of metadynamics meshes
            end do
            end do
            end do

!           //   c function
            if ( eden .gt. 0.d0 ) then
               ct_cv(jstep_hills) = log( enum / eden ) / beta_tamd
            end if

!        //   loop of hills steps
         end do

!     //   end of if statement
      end if

      return
      end





!***********************************************************************
      subroutine analysis_rec_cons_fegrad_tass( ibead )
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use tass_variables, only : &
     &   ct_cv, vb_cv, fref_cv, fegrad_cons, beta_tamd, &
     &   nstep_cv, label_cons_tass, iprint_cv_tass, iprint_hills_tass, &
     &   ntass_cons

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize local variables
      implicit none

!     //   real numbers
      real(8) :: enum, eden, dv, prob

!     //   integers
      integer :: jstep_hills, ibead, istep_cv, icvc

!-----------------------------------------------------------------------
!     //   constrained md only
!-----------------------------------------------------------------------

      if ( ntass_cons .eq. 0 ) return

!-----------------------------------------------------------------------
!     //   constrained cv
!-----------------------------------------------------------------------

      icvc = label_cons_tass(1)

!-----------------------------------------------------------------------
!     //   0D case: calculate free energy gradient
!-----------------------------------------------------------------------

!     //   0D case
      if ( ntass_cons .eq. 0 ) then

!        //   free energy gradient
         fegrad_cons(ibead) = 0.d0

!-----------------------------------------------------------------------
!     //   1D case: calculate free energy gradient
!-----------------------------------------------------------------------

!     //   1D case
      else if ( ntass_cons .eq. 1 ) then

!        //   initialize numerator
         enum = 0.d0

!        //   initialize denominator
         eden = 0.d0

!        //   loop of cv steps
         do istep_cv = 1, nstep_cv

!           //   hills step
            jstep_hills = ( istep_cv * iprint_cv_tass ) &
     &                  / iprint_hills_tass

!           //   dv = bias potential
            dv = vb_cv(istep_cv)

!           //   dv minus c potential
            if ( jstep_hills .ge. 1 ) dv = dv - ct_cv(jstep_hills)

!           //   probability
            prob = exp( beta_tamd * dv )

!           //   numerator
            enum = enum - fref_cv(icvc,istep_cv) * prob

!           //   denominator
            eden = eden + prob

!        //   loop of cv steps
         end do

!        //   free energy gradient
         if ( eden .gt. 0.d0 ) then
            fegrad_cons(ibead) = enum / eden
         else
            fegrad_cons(ibead) = 0.d0
         end if

!     //   end of if statement
      end if

      return
      end





!***********************************************************************
      subroutine analysis_rec_cons_fenergy_tass
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   nbead

      use tass_variables, only : &
     &   fenergy_cons, prob_1d, prob_2d, prob_3d, pc_2d, &
     &   fenergy_1d, fenergy_2d, fenergy_3d, beta_tamd, pc_1d, pc_3d, &
     &   ntass_rec, jtass_rec, nmesh_cv

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize local variables
      implicit none

!     //   real numbers
      real(8) :: prob, pp, eden, fmin

!     //   real numbers
      real(8) :: small_value = 1.d-38

!     //   real numbers
      real(8) :: huge_value  = 1.d+38

!     //   integers
      integer :: ibead, icv, jcv, kcv, mi, mj, mk

!-----------------------------------------------------------------------
!     //   denominator
!-----------------------------------------------------------------------

!     //   denominator
      eden = 0.d0

!     //   loop of beads
      do ibead = 1, nbead

!        //   denominator
         eden = eden + exp( - beta_tamd * fenergy_cons(ibead) )

!     //   loop of beads
      end do

!-----------------------------------------------------------------------
!     //   1D case: total free energy
!-----------------------------------------------------------------------

!     //   1D case
      if ( ntass_rec .eq. 1 ) then

!        //   cv labels
         icv  = jtass_rec(1)

!        //   numerator
         prob_1d(:) = 0.d0

!        //   loop of beads
         do ibead = 1, nbead

!           //   probability
            prob = exp( - beta_tamd * fenergy_cons(ibead) ) / eden

!           //   loop of meshes
            do mi = 1, nmesh_cv(icv)

!              //   probability
               pp = pc_1d(ibead,mi) * prob

!              //   numerator
               prob_1d(mi) = prob_1d(mi) + pp

!           //   loop of meshes
            end do

!        //   loop of beads
         end do

!        //   minimum
         fmin = huge_value

!        //   loop of meshes
         do mi = 1, nmesh_cv(icv)

!           //   probability
            prob = max( prob_1d(mi), small_value )

!           //   free energy
            fenergy_1d(mi) = - log ( prob ) / beta_tamd

!           //   minimum
            fmin = min( fenergy_1d(mi), fmin )

!        //   loop of meshes
         end do

!        //   shift free energy
         fenergy_1d(:) = fenergy_1d(:) - fmin

!-----------------------------------------------------------------------
!     //   2D case: total free energy
!-----------------------------------------------------------------------

!     //   2D case
      else if ( ntass_rec .eq. 2 ) then

!        //   cv labels
         icv  = jtass_rec(1)
         jcv  = jtass_rec(2)

!        //   numerator
         prob_2d(:,:) = 0.d0

!        //   loop of beads
         do ibead = 1, nbead

!           //   probability
            prob = exp( - beta_tamd * fenergy_cons(ibead) ) / eden

!           //   loop of meshes
            do mi = 1, nmesh_cv(icv)
            do mj = 1, nmesh_cv(jcv)

!              //   probability
               pp = pc_2d(ibead,mi,mj) * prob

!              //   numerator
               prob_2d(mi,mj) = prob_2d(mi,mj) + pp

!           //   loop of meshes
            end do
            end do

!        //   loop of beads
         end do

!        //   minimum
         fmin = huge_value

!        //   loop of meshes
         do mi = 1, nmesh_cv(icv)
         do mj = 1, nmesh_cv(jcv)

!           //   probability
            prob = max( prob_2d(mi,mj), small_value )

!           //   free energy
            fenergy_2d(mi,mj) = - log ( prob ) / beta_tamd

!           //   minimum
            fmin = min( fenergy_2d(mi,mj), fmin )

!        //   loop of meshes
         end do
         end do

!        //   shift free energy
         fenergy_2d(:,:) = fenergy_2d(:,:) - fmin

!-----------------------------------------------------------------------
!     //   3D case: total free energy
!-----------------------------------------------------------------------

!     //   3D case
      else if ( ntass_rec .eq. 3 ) then

!        //   cv labels
         icv  = jtass_rec(1)
         jcv  = jtass_rec(2)
         kcv  = jtass_rec(3)

!        //   numerator
         prob_3d(:,:,:) = 0.d0

!        //   loop of beads
         do ibead = 1, nbead

!           //   probability
            prob = exp( - beta_tamd * fenergy_cons(ibead) ) / eden

!           //   loop of meshes
            do mi = 1, nmesh_cv(icv)
            do mj = 1, nmesh_cv(jcv)
            do mk = 1, nmesh_cv(kcv)

!              //   probability
               pp = pc_3d(ibead,mi,mj,mk) * prob

!              //   numerator
               prob_3d(mi,mj,mk) = prob_3d(mi,mj,mk) + pp

!           //   loop of meshes
            end do
            end do
            end do

!        //   loop of beads
         end do

!        //   minimum
         fmin = huge_value

!        //   loop of meshes
         do mi = 1, nmesh_cv(icv)
         do mj = 1, nmesh_cv(jcv)
         do mk = 1, nmesh_cv(kcv)

!           //   probability
            prob = max( prob_3d(mi,mj,mk), small_value )

!           //   free energy
            fenergy_3d(mi,mj,mk) = - log ( prob ) / beta_tamd

!           //   minimum
            fmin = min( fenergy_3d(mi,mj,mk), fmin )

!        //   loop of meshes
         end do
         end do
         end do

!        //   shift free energy
         fenergy_3d(:,:,:) = fenergy_3d(:,:,:) - fmin

!     //   end of if statement
      end if

      return
      end





!***********************************************************************
      subroutine analysis_rec_cons_memory_tass( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   nbead, istep

      use tass_variables, only : &
     &   rcv, fref_cv, gh_cv, gw_cv, gc_cv, vb_cv, ct_cv, fegrad_cons, &
     &   fenergy_cons, fenergy_1d, fenergy_2d, fenergy_3d, vhills_1d, &
     &   vhills_2d, vhills_3d, pc_1d, pc_2d, pc_3d, prob_1d, prob_2d, &
     &   prob_3d, nmesh_cv, nstep_hills, ntass, nstep_cv, &
     &   label_meta_tass, label_cons_tass, jtass_rec, ntass_meta, &
     &   ntass_cons, ntass_rec, iprint_cv_tass, iprint_hills_tass

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize local variables
      implicit none

!     //   integers
      integer :: ioption

!     //   integers
      integer :: nci = 0, nmi = 0, nmj = 0, nmk = 0

!     //   integers
      integer :: mya = 0, myb = 0, myc = 0

!-----------------------------------------------------------------------
!     //   step parameters
!-----------------------------------------------------------------------

!     //   number of cv steps
      nstep_cv = istep / iprint_cv_tass

!     //   number of hills steps
      nstep_hills = istep / iprint_hills_tass

!-----------------------------------------------------------------------
!     //   number of meshes
!-----------------------------------------------------------------------

      if ( ntass_cons .ge. 1 ) nci = nmesh_cv(label_cons_tass(1))
      if ( ntass_meta .ge. 1 ) nmi = nmesh_cv(label_meta_tass(1))
      if ( ntass_meta .ge. 2 ) nmj = nmesh_cv(label_meta_tass(2))
      if ( ntass_meta .ge. 3 ) nmk = nmesh_cv(label_meta_tass(3))

!-----------------------------------------------------------------------
!     //   number of meshes
!-----------------------------------------------------------------------

      if ( ntass_rec .ge. 1 ) mya = nmesh_cv(jtass_rec(1))
      if ( ntass_rec .ge. 2 ) myb = nmesh_cv(jtass_rec(2))
      if ( ntass_rec .ge. 3 ) myc = nmesh_cv(jtass_rec(3))

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

!        //   collective variables
         if ( .not. allocated( rcv ) ) &
     &      allocate( rcv(ntass,max(1,nstep_cv)) )

!        //   force on collective variables
         if ( .not. allocated( fref_cv ) ) &
     &      allocate( fref_cv(ntass,max(1,nstep_cv)) )

!        //   gaussian height
         if ( .not. allocated( gh_cv ) ) &
     &      allocate( gh_cv(max(1,nstep_hills)) )

!        //   gaussian center
         if ( .not. allocated( gc_cv ) ) &
     &      allocate( gc_cv(ntass,max(1,nstep_hills)) )

!        //   gaussian width
         if ( .not. allocated( gw_cv ) ) &
     &      allocate( gw_cv(ntass,max(1,nstep_hills)) )

!        //   bias potential
         if ( .not. allocated( vb_cv ) ) &
     &      allocate( vb_cv(max(1,nstep_cv)) )

!        //   c function
         if ( .not. allocated( ct_cv ) ) &
     &      allocate( ct_cv(max(1,nstep_hills)) )

!        //   free energy gradient from constrained md
         if ( .not. allocated( fegrad_cons ) ) &
     &      allocate( fegrad_cons(nbead) )

!        //   free energy component from constrained md
         if ( .not. allocated( fenergy_cons ) ) &
     &      allocate( fenergy_cons(nbead) )

!        //   free energy component from metadynamics
         if ( ntass_meta .eq. 1 ) then
            if ( .not. allocated( vhills_1d ) ) &
     &         allocate( vhills_1d(nmi) )
         else if ( ntass_meta .eq. 2 ) then
            if ( .not. allocated( vhills_2d ) ) &
     &         allocate( vhills_2d(nmi,nmj) )
         else if ( ntass_meta .eq. 3 ) then
            if ( .not. allocated( vhills_3d ) ) &
     &         allocate( vhills_3d(nmi,nmj,nmk) )
         end if

!        //   unbiased probability distribution
         if ( ntass_rec .eq. 1 ) then
            if ( .not. allocated( pc_1d ) ) &
     &         allocate( pc_1d(nbead,mya) )
         else if ( ntass_rec .eq. 2 ) then
            if ( .not. allocated( pc_2d ) ) &
     &         allocate( pc_2d(nbead,mya,myb) )
         else if ( ntass_rec .eq. 3 ) then
            if ( .not. allocated( pc_3d ) ) &
     &         allocate( pc_3d(nbead,mya,myb,myc) )
         end if

!        //   tass free energy
         if ( ntass_rec .eq. 1 ) then
            if ( .not. allocated( fenergy_1d ) ) &
     &         allocate( fenergy_1d(mya) )
         else if ( ntass_rec .eq. 2 ) then
            if ( .not. allocated( fenergy_2d ) ) &
     &         allocate( fenergy_2d(mya,myb) )
         else if ( ntass_rec .eq. 3 ) then
            if ( .not. allocated( fenergy_3d ) ) &
     &         allocate( fenergy_3d(mya,myb,myc) )
         end if

!        //   probability distribution
         if ( ntass_rec .eq. 1 ) then
            if ( .not. allocated( prob_1d ) ) &
     &         allocate( prob_1d(mya) )
         else if ( ntass_rec .eq. 2 ) then
            if ( .not. allocated( prob_2d ) ) &
     &         allocate( prob_2d(mya,myb) )
         else if ( ntass_rec .eq. 3 ) then
            if ( .not. allocated( prob_3d ) ) &
     &         allocate( prob_3d(mya,myb,myc) )
         end if

!-----------------------------------------------------------------------
!     //   memory deallocation
!-----------------------------------------------------------------------

      else if ( ioption .eq. 1 ) then

!        //   collective variables
         if ( allocated( rcv ) ) &
     &      deallocate( rcv )

!        //   force on collective variables
         if ( allocated( fref_cv ) ) &
     &      deallocate( fref_cv )

!        //   gaussian height
         if ( allocated( gh_cv ) ) &
     &      deallocate( gh_cv )

!        //   gaussian center
         if ( allocated( gc_cv ) ) &
     &      deallocate( gc_cv )

!        //   gaussian width
         if ( allocated( gw_cv ) ) &
     &      deallocate( gw_cv )

!        //   bias potential
         if ( allocated( vb_cv ) ) &
     &      deallocate( vb_cv )

!        //   c function
         if ( allocated( ct_cv ) ) &
     &      deallocate( ct_cv )

!        //   free energy gradient from constrained md
         if ( allocated( fegrad_cons ) ) &
     &      deallocate( fegrad_cons )

!        //   free energy component from constrained md
         if ( allocated( fenergy_cons ) ) &
     &      deallocate( fenergy_cons )

!        //   free energy component from metadynamics
         if ( ntass_meta .eq. 1 ) then
            if ( allocated( vhills_1d ) ) &
     &         deallocate( vhills_1d )
         else if ( ntass_meta .eq. 2 ) then
            if ( allocated( vhills_2d ) ) &
     &         deallocate( vhills_2d )
         else if ( ntass_meta .eq. 3 ) then
            if ( allocated( vhills_3d ) ) &
     &         deallocate( vhills_3d )
         end if

!        //   unbiased probability distribution
         if ( ntass_rec .eq. 1 ) then
            if ( allocated( pc_1d ) ) &
     &         deallocate( pc_1d )
         else if ( ntass_rec .eq. 2 ) then
            if ( allocated( pc_2d ) ) &
     &         deallocate( pc_2d )
         else if ( ntass_rec .eq. 3 ) then
            if ( allocated( pc_3d ) ) &
     &         deallocate( pc_3d )
         end if

!        //   tass free energy
         if ( ntass_rec .eq. 1 ) then
            if ( allocated( fenergy_1d ) ) &
     &         deallocate( fenergy_1d )
         else if ( ntass_rec .eq. 2 ) then
            if ( allocated( fenergy_2d ) ) &
     &         deallocate( fenergy_2d )
         else if ( ntass_rec .eq. 3 ) then
            if ( allocated( fenergy_3d ) ) &
     &         deallocate( fenergy_3d )
         end if

!        //   probability distribution
         if ( ntass_rec .eq. 1 ) then
            if ( allocated( prob_1d ) ) &
     &         deallocate( prob_1d )
         else if ( ntass_rec .eq. 2 ) then
            if ( allocated( prob_2d ) ) &
     &         deallocate( prob_2d )
         else if ( ntass_rec .eq. 3 ) then
            if ( allocated( prob_3d ) ) &
     &         deallocate( prob_3d )
         end if

      end if

      return
      end





!***********************************************************************
      subroutine analysis_rec_cons_pc_tass( ibead )
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use tass_variables, only : &
     &   pc_1d, pc_2d, pc_3d, beta_tamd, vb_cv, dr_cv, ct_cv, rmin_cv, &
     &   rcv, jtass_rec, nmesh_cv, iprint_cv_tass, iprint_hills_tass, &
     &   nstep_cv, ntass_rec, jtass_rec

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize local variables
      implicit none

!     //   real numbers
      real(8) :: dv, eden, pc

!     //   integers
      integer :: ibead, jstep_hills, istep_cv, mi, mj, mk, icv, jcv, kcv

!-----------------------------------------------------------------------
!     //   1D case: calculate conditional probability
!-----------------------------------------------------------------------

!     //   1D case
      if ( ntass_rec .eq. 1 ) then

!        //   cv labels
         icv  = jtass_rec(1)

!        //   initialize numerator
         pc_1d(ibead,:) = 0.d0

!        //   initialize denominator
         eden = 0.d0

!        //   loop of cv steps
         do istep_cv = 1, nstep_cv

!           //   mesh index
            mi = nint((rcv(icv,istep_cv)-rmin_cv(icv))/dr_cv(icv)) + 1

!           //   check out of bounds
            if ( ( mi .lt. 1 ) .or. ( mi .gt. nmesh_cv(icv) ) ) cycle

!           //   hills step number
            jstep_hills = istep_cv * iprint_cv_tass / iprint_hills_tass

!           //   bias function
            dv = vb_cv(istep_cv)

!           //   minus c function
            if ( jstep_hills .ge. 1 ) dv = dv - ct_cv(jstep_hills)

!           //   probability
            pc = exp( beta_tamd * dv )

!           //   numerator
            pc_1d(ibead,mi) = pc_1d(ibead,mi) + pc

!           //   demoninator
            eden = eden + pc

!        //   loop of cv steps
         end do

!        //   probability
         if ( eden .gt. 0.d0 ) then
            pc_1d(ibead,:) = pc_1d(ibead,:) / eden
         end if

!-----------------------------------------------------------------------
!     //   2D case: calculate conditional probability
!-----------------------------------------------------------------------

!     //   2D case
      else if ( ntass_rec .eq. 2 ) then

!        //   cv labels
         icv  = jtass_rec(1)
         jcv  = jtass_rec(2)

!        //   initialize numerator
         pc_2d(ibead,:,:) = 0.d0

!        //   initialize denominator
         eden = 0.d0

!        //   loop of cv steps
         do istep_cv = 1, nstep_cv

!           //   mesh index
            mi = nint((rcv(icv,istep_cv)-rmin_cv(icv))/dr_cv(icv)) + 1
            mj = nint((rcv(jcv,istep_cv)-rmin_cv(jcv))/dr_cv(jcv)) + 1

!           //   check out of bounds
            if ( ( mi .lt. 1 ) .or. ( mi .gt. nmesh_cv(icv) ) ) cycle
            if ( ( mj .lt. 1 ) .or. ( mj .gt. nmesh_cv(jcv) ) ) cycle

!           //   hills step number
            jstep_hills = istep_cv * iprint_cv_tass / iprint_hills_tass

!           //   bias function
            dv = vb_cv(istep_cv)

!           //   minus c function
            if ( jstep_hills .ge. 1 ) dv = dv - ct_cv(jstep_hills)

!           //   probability
            pc = exp( beta_tamd * dv )

!           //   numerator
            pc_2d(ibead,mi,mj) = pc_2d(ibead,mi,mj) + pc

!           //   demoninator
            eden = eden + pc

!        //   loop of cv steps
         end do

!        //   probability
         if ( eden .gt. 0.d0 ) then
            pc_2d(ibead,:,:) = pc_2d(ibead,:,:) / eden
         end if

!-----------------------------------------------------------------------
!     //   3D case: calculate conditional probability
!-----------------------------------------------------------------------

!     //   3D case
      else if ( ntass_rec .eq. 3 ) then

!        //   cv labels
         icv  = jtass_rec(1)
         jcv  = jtass_rec(2)
         kcv  = jtass_rec(3)

!        //   initialize numerator
         pc_3d(ibead,:,:,:) = 0.d0

!        //   initialize denominator
         eden = 0.d0

!        //   loop of cv steps
         do istep_cv = 1, nstep_cv

!           //   mesh index
            mi = nint((rcv(icv,istep_cv)-rmin_cv(icv))/dr_cv(icv)) + 1
            mj = nint((rcv(jcv,istep_cv)-rmin_cv(jcv))/dr_cv(jcv)) + 1
            mk = nint((rcv(kcv,istep_cv)-rmin_cv(kcv))/dr_cv(kcv)) + 1

!           //   check out of bounds
            if ( ( mi .lt. 1 ) .or. ( mi .gt. nmesh_cv(icv) ) ) cycle
            if ( ( mj .lt. 1 ) .or. ( mj .gt. nmesh_cv(jcv) ) ) cycle
            if ( ( mk .lt. 1 ) .or. ( mj .gt. nmesh_cv(kcv) ) ) cycle

!           //   hills step number
            jstep_hills = istep_cv * iprint_cv_tass / iprint_hills_tass

!           //   bias function
            dv = vb_cv(istep_cv)

!           //   minus c function
            if ( jstep_hills .ge. 1 ) dv = dv - ct_cv(jstep_hills)

!           //   probability
            pc = exp( beta_tamd * dv )

!           //   numerator
            pc_3d(ibead,mi,mj,mk) = pc_3d(ibead,mi,mj,mk) + pc

!           //   demoninator
            eden = eden + pc

!        //   loop of cv steps
         end do

!        //   probability
         if ( eden .gt. 0.d0 ) then
            pc_3d(ibead,:,:,:) = pc_3d(ibead,:,:,:) / eden
         end if

!     //   end of if statement
      end if

      return
      end





!***********************************************************************
      subroutine analysis_rec_setup_tass
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, nbead, iounit

      use tass_variables, only : &
     &   rtass, params_rec_meta, rmin_cv, rmax_cv, dr_cv, beta_a_tass, &
     &   beta_b_tass, beta_tamd, dtemp_meta, gamma_meta, temp_tamd_tass, &
     &   nmesh_cv, ntass, ntass_cons, ntass_meta, itype_tass, &
     &   ntass_afed, label_meta_tass, label_cons_tass, label_afed_tass, &
     &   ikind_tass

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize local variables
      implicit none

!     //   integers
      integer :: i, icons, imeta, itamd

!     //   integers
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     //   initial setting
!-----------------------------------------------------------------------

!     //   initial visit only
      if ( iset .ne. 0 ) return

!     //   update flag
      iset = 1

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

!     /*   label of metadynamics cv   */
      if ( .not. allocated( label_meta_tass ) ) &
     &   allocate( label_meta_tass(ntass) )

!     /*   label of constrained cv   */
      if ( .not. allocated( label_cons_tass ) ) &
     &   allocate( label_cons_tass(ntass) )

!     /*   label of tamd cv   */
      if ( .not. allocated( label_afed_tass ) ) &
     &   allocate( label_afed_tass(ntass) )

!     /*   minimum value of mesh   */
      if ( .not. allocated( rmin_cv ) ) &
     &   allocate( rmin_cv(ntass) )

!     /*   maximum value of mesh   */
      if ( .not. allocated( rmax_cv ) ) &
     &   allocate( rmax_cv(ntass) )

!     /*   mesh size   */
      if ( .not. allocated( dr_cv ) ) &
     &   allocate( dr_cv(ntass) )

!     /*   number of meshes   */
      if ( .not. allocated( nmesh_cv ) ) &
     &   allocate( nmesh_cv(ntass) )

!-----------------------------------------------------------------------
!     //   temperature parameters
!-----------------------------------------------------------------------

!     //   beta parameter in tamd
      beta_tamd = 1.d0 / ( boltz * temp_tamd_tass )

!     //   gamma parameter in well tempered metadynamics
      gamma_meta = ( temp_tamd_tass + dtemp_meta ) / dtemp_meta

!     //   parameter in well tempered metadynamics
      beta_a_tass = beta_tamd * ( gamma_meta )

!     //   parameter in well tempered metadynamics
      beta_b_tass = beta_tamd * ( gamma_meta - 1.d0 )

!-----------------------------------------------------------------------
!     //   mesh parameters
!-----------------------------------------------------------------------

!     //   counter: tamd cv dimensions
      itamd = 0

!     //   counter: constraint cv dimensions
      icons = 0

!     //   counter: metadynamics cv dimensions
      imeta = 0

!     //   loop of cv
      do i = 1, ntass

!        //   cv type: tamd
         if      ( ikind_tass(i) .eq. 1 ) then

!           //   update counter
            itamd = itamd + 1

!           //   label
            label_afed_tass(itamd) = i

!           /*   mininum value of mesh   */
            rmin_cv(i)  = params_rec_meta(1,itype_tass(i))

!           /*   maximum value of mesh   */
            rmax_cv(i)  = params_rec_meta(2,itype_tass(i))

!           /*   mesh size   */
            dr_cv(i)    = params_rec_meta(3,itype_tass(i))

!           /*   number of meshes    */
            nmesh_cv(i) = nint( (rmax_cv(i)-rmin_cv(i)) / dr_cv(i) ) + 1

!        //   cv type: constraint
         else if ( ikind_tass(i) .eq. 2 ) then

!           //   update counter
            icons = icons + 1

!           //   label
            label_cons_tass(icons) = i

!           /*   mininum value of mesh   */
            rmin_cv(i)  = rtass(i,1)

!           /*   maximum value of mesh   */
            rmax_cv(i)  = rtass(i,nbead)

!           /*   mesh size   */
            dr_cv(i)    = ( rmax_cv(i) - rmin_cv(i) ) / dble(nbead-1)

!           /*   number of meshes    */
            nmesh_cv(i) = nbead

!        //   cv type: metadynamics
         else if ( ikind_tass(i) .eq. 3 ) then

!           //   update counter
            imeta = imeta + 1

!           //   label
            label_meta_tass(imeta) = i

!           /*   mininum value of mesh   */
            rmin_cv(i)  = params_rec_meta(1,itype_tass(i))

!           /*   maximum value of mesh   */
            rmax_cv(i)  = params_rec_meta(2,itype_tass(i))

!           /*   mesh size   */
            dr_cv(i)    = params_rec_meta(3,itype_tass(i))

!           /*   number of meshes    */
            nmesh_cv(i) = nint( (rmax_cv(i)-rmin_cv(i)) / dr_cv(i) ) + 1

!        //   cv type
         end if

!     //   loop of cv
      end do

!     //   tamd cv dimensions
      ntass_afed = itamd

!     //   constraint cv dimensions
      ntass_cons = icons

!     //   metadynamics cv dimensions
      ntass_meta = imeta

      return
      end





!***********************************************************************
      subroutine analysis_rec_print_tass
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit

      use tass_variables, only : &
     &   rmin_cv, dr_cv, fenergy_1d, fenergy_2d, fenergy_3d, &
     &   nmesh_cv, jtass_rec, ntass_rec

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize local variables
      implicit none

!     //   integers
      integer :: imesh1, imesh2, imesh3, mya, myb, myc

!     //   real numbers
      real(8) :: rmesh1, rmesh2, rmesh3

!-----------------------------------------------------------------------
!     //   1D case: print free energy
!-----------------------------------------------------------------------

      if ( ntass_rec .eq. 1 ) then

!        //   open file
         open( iounit, file = 'rec.out', status = 'unknown' )

!        //   cv
         mya = jtass_rec(1)

!        //   loop of meshes
         do imesh1 = 1, nmesh_cv(mya)

!           //   mesh point
            rmesh1 = rmin_cv(mya) + dble(imesh1-1) * dr_cv(mya)

!           //   print mesh and free energy
            write( iounit, '(2f16.8)' ) &
     &         rmesh1, fenergy_1d(imesh1)

!        //   loop of meshes
         end do

!        //   close file
         close( iounit )

!-----------------------------------------------------------------------
!     //   2D case: print free energy
!-----------------------------------------------------------------------

      else if ( ntass_rec .eq. 2 ) then

!        //   open file
         open( iounit, file = 'rec.out', status = 'unknown' )

!        //   cv
         mya = jtass_rec(1)
         myb = jtass_rec(2)

!        //   loop of meshes
         do imesh1 = 1, nmesh_cv(mya)

!           //   mesh point
            rmesh1 = rmin_cv(mya) + dble(imesh1-1) * dr_cv(mya)

!           //   loop of meshes
            do imesh2 = 1, nmesh_cv(myb)

!              //   mesh point
               rmesh2 = rmin_cv(myb) + dble(imesh2-1) * dr_cv(myb)

!              //   print mesh and free energy
               write( iounit, '(3f16.8)' ) &
     &            rmesh1, rmesh2, fenergy_2d(imesh1,imesh2)

!           //   loop of meshes
            end do

!           //   blank line
            write( iounit, * )

!        //   loop of meshes
         end do

!        //   close file
         close( iounit )

!-----------------------------------------------------------------------
!     //   3D case: print free energy
!-----------------------------------------------------------------------

      else if ( ntass_rec .eq. 3 ) then

!        //   open file
         open( iounit, file = 'rec.out', status = 'unknown' )

!        //   cv
         mya = jtass_rec(1)
         myb = jtass_rec(2)
         myc = jtass_rec(3)

!        //   loop of meshes
         do imesh1 = 1, nmesh_cv(mya)

!           //   mesh point
            rmesh1 = rmin_cv(mya) + dble(imesh1-1) * dr_cv(mya)

!           //   loop of meshes
            do imesh2 = 1, nmesh_cv(myb)

!              //   mesh point
               rmesh2 = rmin_cv(myb) + dble(imesh2-1) * dr_cv(myb)

!              //   loop of meshes
               do imesh3 = 1, nmesh_cv(myc)

!                 //   mesh point
                  rmesh3 = rmin_cv(myc) + dble(imesh3-1) * dr_cv(myc)

!                 //   print mesh and free energy
                  write( iounit, '(4f16.8)' ) rmesh1, rmesh2, rmesh3, &
     &               fenergy_3d(imesh1,imesh2,imesh3)

!              //   loop of meshes
               end do

!           //   loop of meshes
            end do

!        //   loop of meshes
         end do

!        //   close file
         close( iounit )

      end if

      return
      end





!***********************************************************************
      subroutine analysis_rec_cons_read_tass( ibead )
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, iounit, nbead

      use tass_variables, only : &
     &   rcv, fref_cv, gh_cv, gw_cv, gc_cv, nstep_hills, ntass, nstep_cv

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize local variables
      implicit none

!     //   real numbers
      real(8) :: scv

!     //   integers
      integer :: i, j, l, ibead

!-----------------------------------------------------------------------
!     //   read cv trajectory
!-----------------------------------------------------------------------

!     //   open file
      open( iounit, file = 'cv.out', status = 'unknown' )

!     //   skip three lines
      read( iounit, * )
      read( iounit, * )
      read( iounit, * )

!     //   loop of cv steps
      do j = 1, nstep_cv

!        //   skip lines
         do l = 1, ntass * ( ibead - 1 )
            read( iounit, * )
         end do

!        //   read cv and cv force
         do l = 1, ntass
            read( iounit, * ) i, rcv(l,j), scv, fref_cv(l,j)
         end do

!        //   skip lines
         do l = 1, ntass * ( nbead - ibead )
            read( iounit, * )
         end do

!     //   loop of cv steps
      end do

!     //   close file
      close( iounit )

!-----------------------------------------------------------------------
!     //   read hills
!-----------------------------------------------------------------------

!     //   open file
      open( iounit, file = 'hills.ini', status = 'unknown' )

!     //   loop of hills steps
      do j = 1, nstep_hills

!        //   skip lines
         do l = 1, ntass * ( ibead - 1 )
            read( iounit, * )
         end do

!        //   read gaussian heights, widths, centers
         do l = 1, ntass
            read( iounit, * ) i, gh_cv(j), gw_cv(l,j), gc_cv(l,j)
         end do

!        //   skip lines
         do l = 1, ntass * ( nbead - ibead )
            read( iounit, * )
         end do

!     //   loop of hills steps
      end do

!     //   close file
      close( iounit )

      return
      end





!***********************************************************************
      subroutine analysis_rec_cons_vb_tass
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz

      use tass_variables, only : &
     &   rcv, vb_cv, gw_cv, gh_cv, gc_cv, beta_b_tass, dtemp_meta, &
     &   nstep_cv, iprint_cv_tass, iprint_hills_tass, label_meta_tass, &
     &   ntass_meta, ipbc_tass

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize local variables
      implicit none

!     //   real numbers
      real(8) :: enum, gwi, gwj, gwk, gci, gcj, gck, gh, gwi_inv2, &
     &           gwj_inv2, gwk_inv2, fcvi, fcvj, fcvk, dcvi, dcvj, dcvk

!     //   integers
      integer :: jstep_hills, mstep_hills, istep_cv, icvm, jcvm, kcvm

!-----------------------------------------------------------------------
!     //   0D case: calculate bias potential in metadynamics
!-----------------------------------------------------------------------

!     //   0D case
      if ( ntass_meta .eq. 0 ) then

!        //   bias function at cv position
         vb_cv(:) = 0.d0

!-----------------------------------------------------------------------
!     //   1D case: calculate bias potential in metadynamics
!-----------------------------------------------------------------------

!     //   1D case
      else if ( ntass_meta .eq. 1 ) then

!        //   loop of cv steps
         do istep_cv = 1, nstep_cv

!           //   number of hills
            mstep_hills = ( istep_cv * iprint_cv_tass ) &
     &                  / iprint_hills_tass

!           //   sum of hills at cv position
            enum = 0.d0

!           //   loop of hills
            do jstep_hills = 1, mstep_hills

!              //   metadynamics cv
               icvm = label_meta_tass(1)

!              //   gaussian height
               gh  = gh_cv(jstep_hills)

!              //   gaussian width
               gwi = gw_cv(icvm,jstep_hills)

!              //   gaussian center
               gci = gc_cv(icvm,jstep_hills)

!              //   gaussian width inverse squared
               gwi_inv2 = 1.d0 / ( gwi * gwi )

!              //   distance between cv position and gaussian center
               dcvi = rcv(icvm,istep_cv) - gci

!              /*   periodic boundary condition   */
               call pbc_tass( ipbc_tass(icvm), dcvi )

!              //   exponent
               fcvi = 0.5d0 * dcvi * dcvi * gwi_inv2

!              //  sum of hills at cv position
               enum = enum + gh * exp( - fcvi )

!           //   loop of hills
            end do

!           //   well tempered metadynamics
            if ( dtemp_meta .gt. 0.d0 ) then

!              //   bias potential
               enum = log( 1.d0 + beta_b_tass*enum ) / beta_b_tass

!           //   end of if statement
            end if

!           //   bias function at cv position
            vb_cv(istep_cv) = enum

!        //   loop of cv steps
         end do

!-----------------------------------------------------------------------
!     //   2D case: calculate bias potential in metadynamics
!-----------------------------------------------------------------------

!     //   2D case
      else if ( ntass_meta .eq. 2 ) then

!        //   loop of cv steps
         do istep_cv = 1, nstep_cv

!           //   number of hills
            mstep_hills = ( istep_cv * iprint_cv_tass ) &
     &                  / iprint_hills_tass

!           //   sum of hills at cv position
            enum = 0.d0

!           //   loop of hills
            do jstep_hills = 1, mstep_hills

!              //   metadynamics cv
               icvm = label_meta_tass(1)
               jcvm = label_meta_tass(2)

!              //   gaussian height
               gh  = gh_cv(jstep_hills)

!              //   gaussian width
               gwi = gw_cv(icvm,jstep_hills)
               gwj = gw_cv(jcvm,jstep_hills)

!              //   gaussian center
               gci = gc_cv(icvm,jstep_hills)
               gcj = gc_cv(jcvm,jstep_hills)

!              //   gaussian width inverse squared
               gwi_inv2 = 1.d0 / ( gwi * gwi )
               gwj_inv2 = 1.d0 / ( gwj * gwj )

!              //   distance between cv position and gaussian center
               dcvi = rcv(icvm,istep_cv) - gci
               dcvj = rcv(jcvm,istep_cv) - gcj

!              /*   periodic boundary condition   */
               call pbc_tass( ipbc_tass(icvm), dcvi )
               call pbc_tass( ipbc_tass(jcvm), dcvj )

!              //   exponent
               fcvi = 0.5d0 * dcvi * dcvi * gwi_inv2
               fcvj = 0.5d0 * dcvj * dcvj * gwj_inv2

!              //  sum of hills at cv position
               enum = enum + gh * exp( - fcvi - fcvj )

!           //   loop of hills
            end do

!           //   well tempered metadynamics
            if ( dtemp_meta .gt. 0.d0 ) then

!              //   bias potential
               enum = log( 1.d0 + beta_b_tass*enum ) / beta_b_tass

!           //   end of if statement
            end if

!           //   bias function at cv position
            vb_cv(istep_cv) = enum

!        //   loop of cv steps
         end do

!-----------------------------------------------------------------------
!     //   3D case: calculate bias potential in metadynamics
!-----------------------------------------------------------------------

!     //   3D case
      else if ( ntass_meta .eq. 3 ) then

!        //   loop of cv steps
         do istep_cv = 1, nstep_cv

!           //   number of hills
            mstep_hills = ( istep_cv * iprint_cv_tass ) &
     &                  / iprint_hills_tass

!           //   sum of hills at cv position
            enum = 0.d0

!           //   loop of hills
            do jstep_hills = 1, mstep_hills

!              //   metadynamics cv
               icvm = label_meta_tass(1)
               jcvm = label_meta_tass(2)
               kcvm = label_meta_tass(3)

!              //   gaussian height
               gh  = gh_cv(jstep_hills)

!              //   gaussian width
               gwi = gw_cv(icvm,jstep_hills)
               gwj = gw_cv(jcvm,jstep_hills)
               gwk = gw_cv(kcvm,jstep_hills)

!              //   gaussian center
               gci = gc_cv(icvm,jstep_hills)
               gcj = gc_cv(jcvm,jstep_hills)
               gck = gc_cv(kcvm,jstep_hills)

!              //   gaussian width inverse squared
               gwi_inv2 = 1.d0 / ( gwi * gwi )
               gwj_inv2 = 1.d0 / ( gwj * gwj )
               gwk_inv2 = 1.d0 / ( gwk * gwk )

!              //   distance between cv position and gaussian center
               dcvi = rcv(icvm,istep_cv) - gci
               dcvj = rcv(jcvm,istep_cv) - gcj
               dcvk = rcv(kcvm,istep_cv) - gck

!              /*   periodic boundary condition   */
               call pbc_tass( ipbc_tass(icvm), dcvi )
               call pbc_tass( ipbc_tass(jcvm), dcvj )
               call pbc_tass( ipbc_tass(kcvm), dcvk )

!              //   exponent
               fcvi = 0.5d0 * dcvi * dcvi * gwi_inv2
               fcvj = 0.5d0 * dcvj * dcvj * gwj_inv2
               fcvk = 0.5d0 * dcvk * dcvk * gwk_inv2

!              //  sum of hills at cv position
               enum = enum + gh * exp( - fcvi - fcvj - fcvk )

!           //   loop of hills
            end do

!           //   well tempered metadynamics
            if ( dtemp_meta .gt. 0.d0 ) then

!              //   bias potential
               enum = log( 1.d0 + beta_b_tass*enum ) / beta_b_tass

!           //   end of if statement
            end if

!           //   bias function at cv position
            vb_cv(istep_cv) = enum

!        //   loop of cv steps
         end do

!     //   end of if statement
      end if

      return
      end





!***********************************************************************
      subroutine analysis_rec_free_tass
!***********************************************************************

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize local variables
      implicit none

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

      call analysis_rec_free_memory_tass( 0 )

!-----------------------------------------------------------------------
!     //   read cv.out and hills.ini
!-----------------------------------------------------------------------

      call analysis_rec_free_read_tass

!-----------------------------------------------------------------------
!     //   calculate c(t) potential in metadynamics
!-----------------------------------------------------------------------

      call analysis_rec_free_ct_tass

!-----------------------------------------------------------------------
!     //   calculate v(s,t)
!-----------------------------------------------------------------------

      call analysis_rec_free_vb_tass

!-----------------------------------------------------------------------
!     //   calculate probability
!-----------------------------------------------------------------------

      call analysis_rec_free_prob_tass

!-----------------------------------------------------------------------
!     //   calculate total free energy
!-----------------------------------------------------------------------

      call analysis_rec_free_fenergy_tass

!-----------------------------------------------------------------------
!     //   print free energy
!-----------------------------------------------------------------------

      call analysis_rec_print_tass

!-----------------------------------------------------------------------
!     //   memory deallocation
!-----------------------------------------------------------------------

      call analysis_rec_free_memory_tass( 1 )

      return
      end





!***********************************************************************
      subroutine analysis_rec_free_ct_tass
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   nbead

      use tass_variables, only : &
     &   vhills_0d, vhills_1d, vhills_2d, vhills_3d, ct_cv, &
     &   beta_a_tass, beta_b_tass, gw_cv, gh_cv, gc_cv, dr_cv, rmin_cv, &
     &   beta_tamd, dtemp_meta, nmesh_cv, ntass_meta, nstep_hills, &
     &   label_meta_tass, ipbc_tass

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize local variables
      implicit none

!     //   real numbers
      real(8) :: enum, eden, gwi, gwj, gwk, gci, gcj, gck, &
     &           gh, gwi_inv2, gwj_inv2, gwk_inv2, rmesh_icvm, &
     &           rmesh_jcvm, rmesh_kcvm, fcvi, fcvj, fcvk, &
     &           dcvi, dcvj, dcvk, vbias_mesh

!     //   integers
      integer :: mi, mj, mk, jstep_hills, icvm, jcvm, kcvm

!     //   integers
      integer :: lstep_hills, jbead

!-----------------------------------------------------------------------
!     //   0D case: calculate c(t) potential in metadynamics
!-----------------------------------------------------------------------

!     //   0D case
      if ( ntass_meta .eq. 0 ) then

!        //   initialize metadynamics free energy
         vhills_0d = 0.d0

!        //   c function
         ct_cv(:) = 0.d0

!-----------------------------------------------------------------------
!     //   1D case: calculate c(t) potential in metadynamics
!-----------------------------------------------------------------------

!     //   1D case
      else if ( ntass_meta .eq. 1 ) then

!        //   metadynamics cv
         icvm = label_meta_tass(1)

!        //   initialize metadynamics free energy
         vhills_1d(:) = 0.d0

!        //   loop of hills steps
         do jstep_hills = 1, nstep_hills

!        //   loop of beads
         do jbead = 1, nbead

!           //   hills bead step
            lstep_hills = ( jstep_hills - 1 ) * nbead + jbead

!           //   numerator
            enum = 0.d0

!           //   denominator
            eden = 0.d0

!           //   loop of metadynamics meshes
            do mi = 1, nmesh_cv(icvm)

!              //   gaussian width
               gwi = gw_cv(icvm,lstep_hills)

!              //   gaussian center
               gci = gc_cv(icvm,lstep_hills)

!              //   gaussian height
               gh = gh_cv(lstep_hills)

!              //   gaussian width inverse squared
               gwi_inv2 = 1.d0 / ( gwi * gwi )

!              //   mesh point
               rmesh_icvm = rmin_cv(icvm) + dble(mi-1) * dr_cv(icvm)

!              //   difference between mesh point and gaussian center
               dcvi = rmesh_icvm - gci

!              /*   periodic boundary condition   */
               call pbc_tass( ipbc_tass(icvm), dcvi )

!              //   exponent value
               fcvi = 0.5d0 * dcvi * dcvi * gwi_inv2

!              //   hill height at hills mesh
               vhills_1d(mi) = vhills_1d(mi) &
     &            + gh * exp( - fcvi )

!              //   bias potential for conventional metadynamics
               vbias_mesh = vhills_1d(mi)

!              //   well tempered metadynamics
               if ( dtemp_meta .gt. 0.d0 ) then

!                 //   bias potential
                  vbias_mesh = log( 1.d0 + beta_b_tass * vbias_mesh ) &
     &                       / beta_b_tass

!              //   end of if statement
               end if

!              //   numerator
               enum = enum + exp( beta_a_tass * vbias_mesh )

!              //   denominator
               eden = eden + exp( beta_b_tass * vbias_mesh )

!           //   loop of metadynamics meshes
            end do

!           //   c function
            if ( eden .gt. 0.d0 ) then
               ct_cv(lstep_hills) = log( enum / eden ) / beta_tamd
            end if

!        //   loop of beads
         end do

!        //   loop of hills steps
         end do

!-----------------------------------------------------------------------
!     //   2D case: calculate c(t) potential in metadynamics
!-----------------------------------------------------------------------

!     //   2D case
      else if ( ntass_meta .eq. 2 ) then

!        //   metadynamics cv
         icvm = label_meta_tass(1)
         jcvm = label_meta_tass(2)

!        //   initialize metadynamics free energy
         vhills_2d(:,:) = 0.d0

!        //   loop of hills steps
         do jstep_hills = 1, nstep_hills

!        //   loop of beads
         do jbead = 1, nbead

!           //   hills bead step
            lstep_hills = ( jstep_hills - 1 ) * nbead + jbead

!           //   numerator
            enum = 0.d0

!           //   denominator
            eden = 0.d0

!           //   loop of metadynamics meshes
            do mi = 1, nmesh_cv(icvm)
            do mj = 1, nmesh_cv(jcvm)

!              //   gaussian width
               gwi = gw_cv(icvm,lstep_hills)
               gwj = gw_cv(jcvm,lstep_hills)

!              //   gaussian center
               gci = gc_cv(icvm,lstep_hills)
               gcj = gc_cv(jcvm,lstep_hills)

!              //   gaussian height
               gh = gh_cv(lstep_hills)

!              //   gaussian width inverse squared
               gwi_inv2 = 1.d0 / ( gwi * gwi )
               gwj_inv2 = 1.d0 / ( gwj * gwj )

!              //   mesh point
               rmesh_icvm = rmin_cv(icvm) + dble(mi-1) * dr_cv(icvm)
               rmesh_jcvm = rmin_cv(jcvm) + dble(mj-1) * dr_cv(jcvm)

!              //   difference between mesh point and gaussian center
               dcvi = rmesh_icvm - gci
               dcvj = rmesh_jcvm - gcj

!              /*   periodic boundary condition   */
               call pbc_tass( ipbc_tass(icvm), dcvi )
               call pbc_tass( ipbc_tass(jcvm), dcvj )

!              //   exponent value
               fcvi = 0.5d0 * dcvi * dcvi * gwi_inv2
               fcvj = 0.5d0 * dcvj * dcvj * gwj_inv2

!              //   hill height at hills mesh
               vhills_2d(mi,mj) = vhills_2d(mi,mj) &
     &            + gh * exp( - fcvi - fcvj )

!              //   bias potential for conventional metadynamics
               vbias_mesh = vhills_2d(mi,mj)

!              //   well tempered metadynamics
               if ( dtemp_meta .gt. 0.d0 ) then

!                 //   bias potential
                  vbias_mesh = log( 1.d0 + beta_b_tass * vbias_mesh ) &
     &                       / beta_b_tass

!              //   end of if statement
               end if

!              //   numerator
               enum = enum + exp( beta_a_tass * vbias_mesh )

!              //   denominator
               eden = eden + exp( beta_b_tass * vbias_mesh )

!           //   loop of metadynamics meshes
            end do
            end do

!           //   c function
            if ( eden .gt. 0.d0 ) then
               ct_cv(lstep_hills) = log( enum / eden ) / beta_tamd
            end if

!        //   loop of beads
         end do

!        //   loop of hills steps
         end do

!-----------------------------------------------------------------------
!     //   3D case:  calculate c(t) potential in metadynamics
!-----------------------------------------------------------------------

!     //   3D case
      else if ( ntass_meta .eq. 3 ) then

!        //   metadynamics cv
         icvm = label_meta_tass(1)
         jcvm = label_meta_tass(2)
         kcvm = label_meta_tass(3)

!        //   initialize metadynamics free energy
         vhills_3d(:,:,:) = 0.d0

!        //   loop of hills steps
         do jstep_hills = 1, nstep_hills

!        //   loop of beads
         do jbead = 1, nbead

!           //   hills bead step
            lstep_hills = ( jstep_hills - 1 ) * nbead + jbead

!           //   numerator
            enum = 0.d0

!           //   denominator
            eden = 0.d0

!           //   loop of metadynamics meshes
            do mi = 1, nmesh_cv(icvm)
            do mj = 1, nmesh_cv(jcvm)
            do mk = 1, nmesh_cv(kcvm)

!              //   gaussian width
               gwi = gw_cv(icvm,lstep_hills)
               gwj = gw_cv(jcvm,lstep_hills)
               gwk = gw_cv(kcvm,lstep_hills)

!              //   gaussian center
               gci = gc_cv(icvm,lstep_hills)
               gcj = gc_cv(jcvm,lstep_hills)
               gck = gc_cv(kcvm,lstep_hills)

!              //   gaussian height
               gh = gh_cv(lstep_hills)

!              //   gaussian width inverse squared
               gwi_inv2 = 1.d0 / ( gwi * gwi )
               gwj_inv2 = 1.d0 / ( gwj * gwj )
               gwk_inv2 = 1.d0 / ( gwk * gwk )

!              //   mesh point
               rmesh_icvm = rmin_cv(icvm) + dble(mi-1) * dr_cv(icvm)
               rmesh_jcvm = rmin_cv(jcvm) + dble(mj-1) * dr_cv(jcvm)
               rmesh_kcvm = rmin_cv(kcvm) + dble(mk-1) * dr_cv(kcvm)

!              //   difference between mesh point and gaussian center
               dcvi = rmesh_icvm - gci
               dcvj = rmesh_jcvm - gcj
               dcvk = rmesh_kcvm - gck

!              /*   periodic boundary condition   */
               call pbc_tass( ipbc_tass(icvm), dcvi )
               call pbc_tass( ipbc_tass(jcvm), dcvj )
               call pbc_tass( ipbc_tass(kcvm), dcvk )

!              //   exponent value
               fcvi = 0.5d0 * dcvi * dcvi * gwi_inv2
               fcvj = 0.5d0 * dcvj * dcvj * gwj_inv2
               fcvk = 0.5d0 * dcvk * dcvk * gwk_inv2

!              //   hill height at hills mesh
               vhills_3d(mi,mj,mk) = vhills_3d(mi,mj,mk) &
     &            + gh * exp( - fcvi - fcvj - fcvk )

!              //   bias potential for conventional metadynamics
               vbias_mesh = vhills_3d(mi,mj,mk)

!              //   well tempered metadynamics
               if ( dtemp_meta .gt. 0.d0 ) then

!                 //   bias potential
                  vbias_mesh = log( 1.d0 + beta_b_tass * vbias_mesh ) &
     &                       / beta_b_tass

!              //   end of if statement
               end if

!              //   numerator
               enum = enum + exp( beta_a_tass * vbias_mesh )

!              //   denominator
               eden = eden + exp( beta_b_tass * vbias_mesh )

!           //   loop of metadynamics meshes
            end do
            end do
            end do

!           //   c function
            if ( eden .gt. 0.d0 ) then
               ct_cv(lstep_hills) = log( enum / eden ) / beta_tamd
            end if

!        //   loop of beads
         end do

!        //   loop of hills steps
         end do

!     //   end of if statement
      end if

      return
      end





!***********************************************************************
      subroutine analysis_rec_free_fenergy_tass
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use tass_variables, only : &
     &   prob_1d, prob_2d, prob_3d, fenergy_1d, fenergy_2d, fenergy_3d, &
     &   beta_tamd, ntass_rec, jtass_rec, nmesh_cv

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize local variables
      implicit none

!     //   real numbers
      real(8) :: prob, fmin

!     //   real numbers
      real(8) :: small_value = 1.d-38

!     //   real numbers
      real(8) :: huge_value  = 1.d+38

!     //   integers
      integer :: icv, jcv, kcv, mi, mj, mk

!-----------------------------------------------------------------------
!     //   1D case: total free energy
!-----------------------------------------------------------------------

!     //   1D case
      if ( ntass_rec .eq. 1 ) then

!        //   cv labels
         icv  = jtass_rec(1)

!        //   minimum
         fmin = huge_value

!        //   loop of meshes
         do mi = 1, nmesh_cv(icv)

!           //   probability
            prob = max( prob_1d(mi), small_value )

!           //   free energy
            fenergy_1d(mi) = - log ( prob ) / beta_tamd

!           //   minimum
            fmin = min( fenergy_1d(mi), fmin )

!        //   loop of meshes
         end do

!        //   shift free energy
         fenergy_1d(:) = fenergy_1d(:) - fmin

!-----------------------------------------------------------------------
!     //   2D case: total free energy
!-----------------------------------------------------------------------

!     //   2D case
      else if ( ntass_rec .eq. 2 ) then

!        //   cv labels
         icv  = jtass_rec(1)
         jcv  = jtass_rec(2)

!        //   minimum
         fmin = huge_value

!        //   loop of meshes
         do mi = 1, nmesh_cv(icv)
         do mj = 1, nmesh_cv(jcv)

!           //   probability
            prob = max( prob_2d(mi,mj), small_value )

!           //   free energy
            fenergy_2d(mi,mj) = - log ( prob ) / beta_tamd

!           //   minimum
            fmin = min( fenergy_2d(mi,mj), fmin )

!        //   loop of meshes
         end do
         end do

!        //   shift free energy
         fenergy_2d(:,:) = fenergy_2d(:,:) - fmin

!-----------------------------------------------------------------------
!     //   3D case: total free energy
!-----------------------------------------------------------------------

!     //   3D case
      else if ( ntass_rec .eq. 3 ) then

!        //   cv labels
         icv  = jtass_rec(1)
         jcv  = jtass_rec(2)
         kcv  = jtass_rec(3)

!        //   minimum
         fmin = huge_value

!        //   loop of meshes
         do mi = 1, nmesh_cv(icv)
         do mj = 1, nmesh_cv(jcv)
         do mk = 1, nmesh_cv(kcv)

!           //   probability
            prob = max( prob_3d(mi,mj,mk), small_value )

!           //   free energy
            fenergy_3d(mi,mj,mk) = - log ( prob ) / beta_tamd

!           //   minimum
            fmin = min( fenergy_3d(mi,mj,mk), fmin )

!        //   loop of meshes
         end do
         end do
         end do

!        //   shift free energy
         fenergy_3d(:,:,:) = fenergy_3d(:,:,:) - fmin

!     //   end of if statement
      end if

      return
      end





!***********************************************************************
      subroutine analysis_rec_free_memory_tass( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   nbead, istep

      use tass_variables, only : &
     &   rcv, fref_cv, gh_cv, gw_cv, gc_cv, vb_cv, ct_cv, fenergy_1d, &
     &   fenergy_2d, fenergy_3d, vhills_1d, vhills_2d, vhills_3d, &
     &   prob_1d, prob_2d, prob_3d, nmesh_cv, nstep_hills, ntass, &
     &   nstep_cv, label_meta_tass, label_cons_tass, jtass_rec, &
     &   ntass_meta, ntass_cons, ntass_rec, iprint_cv_tass, &
     &   iprint_hills_tass

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize local variables
      implicit none

!     //   integers
      integer :: ioption

!     //   integers
      integer :: nci = 0, nmi = 0, nmj = 0, nmk = 0

!     //   integers
      integer :: mya = 0, myb = 0, myc = 0

!-----------------------------------------------------------------------
!     //   step parameters
!-----------------------------------------------------------------------

!     //   number of cv steps
      nstep_cv = istep / iprint_cv_tass

!     //   number of hills steps
      nstep_hills = istep / iprint_hills_tass

!-----------------------------------------------------------------------
!     //   number of meshes
!-----------------------------------------------------------------------

      if ( ntass_cons .ge. 1 ) nci = nmesh_cv(label_cons_tass(1))
      if ( ntass_meta .ge. 1 ) nmi = nmesh_cv(label_meta_tass(1))
      if ( ntass_meta .ge. 2 ) nmj = nmesh_cv(label_meta_tass(2))
      if ( ntass_meta .ge. 3 ) nmk = nmesh_cv(label_meta_tass(3))

!-----------------------------------------------------------------------
!     //   number of meshes
!-----------------------------------------------------------------------

      if ( ntass_rec .ge. 1 ) mya = nmesh_cv(jtass_rec(1))
      if ( ntass_rec .ge. 2 ) myb = nmesh_cv(jtass_rec(2))
      if ( ntass_rec .ge. 3 ) myc = nmesh_cv(jtass_rec(3))

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

!        //   collective variables
         if ( .not. allocated( rcv ) ) &
     &      allocate( rcv(ntass,max(1,nstep_cv*nbead)) )

!        //   force on collective variables
         if ( .not. allocated( fref_cv ) ) &
     &      allocate( fref_cv(ntass,max(1,nstep_cv*nbead)) )

!        //   gaussian height
         if ( .not. allocated( gh_cv ) ) &
     &      allocate( gh_cv(max(1,nstep_hills*nbead)) )

!        //   gaussian center
         if ( .not. allocated( gc_cv ) ) &
     &      allocate( gc_cv(ntass,max(1,nstep_hills*nbead)) )

!        //   gaussian width
         if ( .not. allocated( gw_cv ) ) &
     &      allocate( gw_cv(ntass,max(1,nstep_hills*nbead)) )

!        //   bias potential
         if ( .not. allocated( vb_cv ) ) &
     &      allocate( vb_cv(max(1,nstep_cv*nbead)) )

!        //   c function
         if ( .not. allocated( ct_cv ) ) &
     &      allocate( ct_cv(max(1,nstep_hills*nbead)) )

!        //   free energy component from metadynamics
         if ( ntass_meta .eq. 1 ) then
            if ( .not. allocated( vhills_1d ) ) &
     &         allocate( vhills_1d(nmi) )
         else if ( ntass_meta .eq. 2 ) then
            if ( .not. allocated( vhills_2d ) ) &
     &         allocate( vhills_2d(nmi,nmj) )
         else if ( ntass_meta .eq. 3 ) then
            if ( .not. allocated( vhills_3d ) ) &
     &         allocate( vhills_3d(nmi,nmj,nmk) )
         end if

!        //   tass free energy
         if ( ntass_rec .eq. 1 ) then
            if ( .not. allocated( fenergy_1d ) ) &
     &         allocate( fenergy_1d(mya) )
         else if ( ntass_rec .eq. 2 ) then
            if ( .not. allocated( fenergy_2d ) ) &
     &         allocate( fenergy_2d(mya,myb) )
         else if ( ntass_rec .eq. 3 ) then
            if ( .not. allocated( fenergy_3d ) ) &
     &         allocate( fenergy_3d(mya,myb,myc) )
         end if

!        //   probability distribution
         if ( ntass_rec .eq. 1 ) then
            if ( .not. allocated( prob_1d ) ) &
     &         allocate( prob_1d(mya) )
         else if ( ntass_rec .eq. 2 ) then
            if ( .not. allocated( prob_2d ) ) &
     &         allocate( prob_2d(mya,myb) )
         else if ( ntass_rec .eq. 3 ) then
            if ( .not. allocated( prob_3d ) ) &
     &         allocate( prob_3d(mya,myb,myc) )
         end if

!-----------------------------------------------------------------------
!     //   memory deallocation
!-----------------------------------------------------------------------

      else if ( ioption .eq. 1 ) then

!        //   collective variables
         if ( allocated( rcv ) ) &
     &      deallocate( rcv )

!        //   force on collective variables
         if ( allocated( fref_cv ) ) &
     &      deallocate( fref_cv )

!        //   gaussian height
         if ( allocated( gh_cv ) ) &
     &      deallocate( gh_cv )

!        //   gaussian center
         if ( allocated( gc_cv ) ) &
     &      deallocate( gc_cv )

!        //   gaussian width
         if ( allocated( gw_cv ) ) &
     &      deallocate( gw_cv )

!        //   bias potential
         if ( allocated( vb_cv ) ) &
     &      deallocate( vb_cv )

!        //   c function
         if ( allocated( ct_cv ) ) &
     &      deallocate( ct_cv )

!        //   free energy component from metadynamics
         if ( ntass_meta .eq. 1 ) then
            if ( allocated( vhills_1d ) ) &
     &         deallocate( vhills_1d )
         else if ( ntass_meta .eq. 2 ) then
            if ( allocated( vhills_2d ) ) &
     &         deallocate( vhills_2d )
         else if ( ntass_meta .eq. 3 ) then
            if ( allocated( vhills_3d ) ) &
     &         deallocate( vhills_3d )
         end if

!        //   tass free energy
         if ( ntass_rec .eq. 1 ) then
            if ( allocated( fenergy_1d ) ) &
     &         deallocate( fenergy_1d )
         else if ( ntass_rec .eq. 2 ) then
            if ( allocated( fenergy_2d ) ) &
     &         deallocate( fenergy_2d )
         else if ( ntass_rec .eq. 3 ) then
            if ( allocated( fenergy_3d ) ) &
     &         deallocate( fenergy_3d )
         end if

!        //   probability distribution
         if ( ntass_rec .eq. 1 ) then
            if ( allocated( prob_1d ) ) &
     &         deallocate( prob_1d )
         else if ( ntass_rec .eq. 2 ) then
            if ( allocated( prob_2d ) ) &
     &         deallocate( prob_2d )
         else if ( ntass_rec .eq. 3 ) then
            if ( allocated( prob_3d ) ) &
     &         deallocate( prob_3d )
         end if

      end if

      return
      end





!***********************************************************************
      subroutine analysis_rec_free_prob_tass
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   nbead

      use tass_variables, only : &
     &   beta_tamd, vb_cv, dr_cv, ct_cv, rmin_cv, prob_1d, prob_2d, &
     &   prob_3d, rcv, jtass_rec, nmesh_cv, iprint_cv_tass, &
     &   iprint_hills_tass, nstep_cv, ntass_rec, jtass_rec

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize local variables
      implicit none

!     //   real numbers
      real(8) :: dv, eden, pc

!     //   integers
      integer :: jstep_hills, istep_cv, mi, mj, mk, icv, jcv, kcv

!     //   integers
      integer :: ibead, kstep_cv, lstep_hills

!-----------------------------------------------------------------------
!     //   1D case: calculate conditional probability
!-----------------------------------------------------------------------

!     //   1D case
      if ( ntass_rec .eq. 1 ) then

!        //   cv labels
         icv  = jtass_rec(1)

!        //   initialize numerator
         prob_1d(:) = 0.d0

!        //   initialize denominator
         eden = 0.d0

!        //   loop of cv steps
         do istep_cv = 1, nstep_cv

!        //   loop of beads
         do ibead = 1, nbead

!           //   cv bead step
            kstep_cv = ( istep_cv - 1 ) * nbead + ibead

!           //   mesh index
            mi = nint((rcv(icv,kstep_cv)-rmin_cv(icv))/dr_cv(icv)) + 1

!           //   check out of bounds
            if ( ( mi .lt. 1 ) .or. ( mi .gt. nmesh_cv(icv) ) ) cycle

!           //   hills step number
            jstep_hills = istep_cv * iprint_cv_tass / iprint_hills_tass

!           //   hill bead step
            lstep_hills = ( jstep_hills - 1 ) * nbead + ibead

!           //   bias function
            dv = vb_cv(kstep_cv)

!           //   minus c function
            if ( jstep_hills .ge. 1 ) dv = dv - ct_cv(lstep_hills)

!           //   probability
            pc = exp( beta_tamd * dv )

!           //   numerator
            prob_1d(mi) = prob_1d(mi) + pc

!           //   demoninator
            eden = eden + pc

!        //   loop of beads
         end do

!        //   loop of cv steps
         end do

!        //   probability
         if ( eden .gt. 0.d0 ) then
            prob_1d(:) = prob_1d(:) / eden
         end if

!-----------------------------------------------------------------------
!     //   2D case: calculate conditional probability
!-----------------------------------------------------------------------

!     //   2D case
      else if ( ntass_rec .eq. 2 ) then

!        //   cv labels
         icv  = jtass_rec(1)
         jcv  = jtass_rec(2)

!        //   initialize numerator
         prob_2d(:,:) = 0.d0

!        //   initialize denominator
         eden = 0.d0

!        //   loop of cv steps
         do istep_cv = 1, nstep_cv

!        //   loop of beads
         do ibead = 1, nbead

!           //   cv bead step
            kstep_cv = ( istep_cv - 1 ) * nbead + ibead

!           //   mesh index
            mi = nint((rcv(icv,kstep_cv)-rmin_cv(icv))/dr_cv(icv)) + 1
            mj = nint((rcv(jcv,kstep_cv)-rmin_cv(jcv))/dr_cv(jcv)) + 1

!           //   check out of bounds
            if ( ( mi .lt. 1 ) .or. ( mi .gt. nmesh_cv(icv) ) ) cycle
            if ( ( mj .lt. 1 ) .or. ( mj .gt. nmesh_cv(jcv) ) ) cycle

!           //   hills step number
            jstep_hills = istep_cv * iprint_cv_tass / iprint_hills_tass

!           //   hill bead step
            lstep_hills = ( jstep_hills - 1 ) * nbead + ibead

!           //   bias function
            dv = vb_cv(kstep_cv)

!           //   minus c function
            if ( jstep_hills .ge. 1 ) dv = dv - ct_cv(lstep_hills)

!           //   probability
            pc = exp( beta_tamd * dv )

!           //   numerator
            prob_2d(mi,mj) = prob_2d(mi,mj) + pc

!           //   demoninator
            eden = eden + pc

!        //   loop of beads
         end do

!        //   loop of cv steps
         end do

!        //   probability
         if ( eden .gt. 0.d0 ) then
            prob_2d(:,:) = prob_2d(:,:) / eden
         end if

!-----------------------------------------------------------------------
!     //   3D case: calculate conditional probability
!-----------------------------------------------------------------------

!     //   3D case
      else if ( ntass_rec .eq. 3 ) then

!        //   cv labels
         icv  = jtass_rec(1)
         jcv  = jtass_rec(2)
         kcv  = jtass_rec(3)

!        //   initialize numerator
         prob_3d(:,:,:) = 0.d0

!        //   initialize denominator
         eden = 0.d0

!        //   loop of cv steps
         do istep_cv = 1, nstep_cv

!        //   loop of beads
         do ibead = 1, nbead

!           //   cv bead step
            kstep_cv = ( istep_cv - 1 ) * nbead + ibead

!           //   mesh index
            mi = nint((rcv(icv,kstep_cv)-rmin_cv(icv))/dr_cv(icv)) + 1
            mj = nint((rcv(jcv,kstep_cv)-rmin_cv(jcv))/dr_cv(jcv)) + 1
            mk = nint((rcv(kcv,kstep_cv)-rmin_cv(kcv))/dr_cv(kcv)) + 1

!           //   check out of bounds
            if ( ( mi .lt. 1 ) .or. ( mi .gt. nmesh_cv(icv) ) ) cycle
            if ( ( mj .lt. 1 ) .or. ( mj .gt. nmesh_cv(jcv) ) ) cycle
            if ( ( mk .lt. 1 ) .or. ( mj .gt. nmesh_cv(kcv) ) ) cycle

!           //   hills step number
            jstep_hills = istep_cv * iprint_cv_tass / iprint_hills_tass

!           //   hill bead step
            lstep_hills = ( jstep_hills - 1 ) * nbead + ibead

!           //   bias function
            dv = vb_cv(kstep_cv)

!           //   minus c function
            if ( jstep_hills .ge. 1 ) dv = dv - ct_cv(lstep_hills)

!           //   probability
            pc = exp( beta_tamd * dv )

!           //   numerator
            prob_3d(mi,mj,mk) = prob_3d(mi,mj,mk) + pc

!           //   demoninator
            eden = eden + pc

!        //   loop of beads
         end do

!        //   loop of cv steps
         end do

!        //   probability
         if ( eden .gt. 0.d0 ) then
            prob_3d(:,:,:) = prob_3d(:,:,:) / eden
         end if

!     //   end of if statement
      end if

      return
      end





!***********************************************************************
      subroutine analysis_rec_free_read_tass
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, iounit, nbead

      use tass_variables, only : &
     &   rcv, fref_cv, gh_cv, gw_cv, gc_cv, nstep_hills, ntass, nstep_cv

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize local variables
      implicit none

!     //   real numbers
      real(8) :: scv

!     //   integers
      integer :: i, j, k, l, m

!-----------------------------------------------------------------------
!     //   read cv trajectory
!-----------------------------------------------------------------------

!     //   open file
      open( iounit, file = 'cv.out', status = 'old' )

!     //   skip three lines
      read( iounit, * )
      read( iounit, * )
      read( iounit, * )

!     //   loop of cv steps
      do k = 1, nstep_cv

!     //   loop of beads
      do m = 1, nbead

!        //   cv bead step
         j = ( k - 1 ) * nbead + m

!        //   read cv and cv force
         do l = 1, ntass
            read( iounit, * ) i, rcv(l,j), scv, fref_cv(l,j)
         end do

!     //   loop of beads
      end do

!     //   loop of cv steps
      end do

!     //   close file
      close( iounit )

!-----------------------------------------------------------------------
!     //   read hills
!-----------------------------------------------------------------------

!     //   open file
      open( iounit, file = 'hills.ini', status = 'old' )

!     //   loop of hills steps
      do k = 1, nstep_hills

!     //   loop of beads
      do m = 1, nbead

!        //   cv bead step
         j = ( k - 1 ) * nbead + m

!        //   read gaussian heights, widths, centers
         do l = 1, ntass
            read( iounit, * ) i, gh_cv(j), gw_cv(l,j), gc_cv(l,j)
         end do

!     //   loop of beads
      end do

!     //   loop of hills steps
      end do

!     //   close file
      close( iounit )

      return
      end





!***********************************************************************
      subroutine analysis_rec_free_vb_tass
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, nbead

      use tass_variables, only : &
     &   rcv, vb_cv, gw_cv, gh_cv, gc_cv, beta_b_tass, dtemp_meta, &
     &   nstep_cv, iprint_cv_tass, iprint_hills_tass, label_meta_tass, &
     &   ntass_meta, ipbc_tass

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize local variables
      implicit none

!     //   real numbers
      real(8) :: enum, gwi, gwj, gwk, gci, gcj, gck, gh, gwi_inv2, &
     &           gwj_inv2, gwk_inv2, fcvi, fcvj, fcvk, dcvi, dcvj, dcvk

!     //   integers
      integer :: jstep_hills, mstep_hills, istep_cv, icvm, jcvm, kcvm

!     //   integers
      integer :: kstep_cv, ibead, jbead, lstep_hills

!-----------------------------------------------------------------------
!     //   0D case: calculate bias potential in metadynamics
!-----------------------------------------------------------------------

!     //   0D case
      if ( ntass_meta .eq. 0 ) then

!        //   bias function at cv position
         vb_cv(:) = 0.d0

!-----------------------------------------------------------------------
!     //   1D case: calculate bias potential in metadynamics
!-----------------------------------------------------------------------

!     //   1D case
      else if ( ntass_meta .eq. 1 ) then

!        //   loop of cv steps
         do istep_cv = 1, nstep_cv

!        //   loop of beads
         do ibead = 1, nbead

!           //   cv bead step
            kstep_cv = ( istep_cv - 1 ) * nbead + ibead

!           //   number of hills
            mstep_hills = istep_cv * iprint_cv_tass / iprint_hills_tass

!           //   sum of hills at cv position
            enum = 0.d0

!           //   loop of hills
            do jstep_hills = 1, mstep_hills

!           //   loop of beads
            do jbead = 1, nbead

!              //   hills bead step
               lstep_hills = ( jstep_hills - 1 ) * nbead + jbead

!              //   metadynamics cv
               icvm = label_meta_tass(1)

!              //   gaussian height
               gh  = gh_cv(lstep_hills)

!              //   gaussian width
               gwi = gw_cv(icvm,lstep_hills)

!              //   gaussian center
               gci = gc_cv(icvm,lstep_hills)

!              //   gaussian width inverse squared
               gwi_inv2 = 1.d0 / ( gwi * gwi )

!              //   distance between cv position and gaussian center
               dcvi = rcv(icvm,kstep_cv) - gci

!              /*   periodic boundary condition   */
               call pbc_tass( ipbc_tass(icvm), dcvi )

!              //   exponent
               fcvi = 0.5d0 * dcvi * dcvi * gwi_inv2

!              //  sum of hills at cv position
               enum = enum + gh * exp( - fcvi )

!           //   loop of beads
            end do

!           //   loop of hills
            end do

!           //   well tempered metadynamics
            if ( dtemp_meta .gt. 0.d0 ) then

!              //   bias potential
               enum = log( 1.d0 + beta_b_tass*enum ) / beta_b_tass

!           //   end of if statement
            end if

!           //   bias function at cv position
            vb_cv(kstep_cv) = enum

!        //   loop of beads
         end do

!        //   loop of cv steps
         end do

!-----------------------------------------------------------------------
!     //   2D case: calculate bias potential in metadynamics
!-----------------------------------------------------------------------

!     //   2D case
      else if ( ntass_meta .eq. 2 ) then

!        //   loop of cv steps
         do istep_cv = 1, nstep_cv

!        //   loop of beads
         do ibead = 1, nbead

!           //   cv bead step
            kstep_cv = ( istep_cv - 1 ) * nbead + ibead

!           //   number of hills
            mstep_hills = istep_cv * iprint_cv_tass / iprint_hills_tass

!           //   sum of hills at cv position
            enum = 0.d0

!           //   loop of hills
            do jstep_hills = 1, mstep_hills

!           //   loop of beads
            do jbead = 1, nbead

!              //   hills bead step
               lstep_hills = ( jstep_hills - 1 ) * nbead + jbead

!              //   metadynamics cv
               icvm = label_meta_tass(1)
               jcvm = label_meta_tass(2)

!              //   gaussian height
               gh  = gh_cv(lstep_hills)

!              //   gaussian width
               gwi = gw_cv(icvm,lstep_hills)
               gwj = gw_cv(jcvm,lstep_hills)

!              //   gaussian center
               gci = gc_cv(icvm,lstep_hills)
               gcj = gc_cv(jcvm,lstep_hills)

!              //   gaussian width inverse squared
               gwi_inv2 = 1.d0 / ( gwi * gwi )
               gwj_inv2 = 1.d0 / ( gwj * gwj )

!              //   distance between cv position and gaussian center
               dcvi = rcv(icvm,kstep_cv) - gci
               dcvj = rcv(jcvm,kstep_cv) - gcj

!              /*   periodic boundary condition   */
               call pbc_tass( ipbc_tass(icvm), dcvi )
               call pbc_tass( ipbc_tass(jcvm), dcvj )

!              //   exponent
               fcvi = 0.5d0 * dcvi * dcvi * gwi_inv2
               fcvj = 0.5d0 * dcvj * dcvj * gwj_inv2

!              //  sum of hills at cv position
               enum = enum + gh * exp( - fcvi - fcvj )

!           //   loop of beads
            end do

!           //   loop of hills
            end do

!           //   well tempered metadynamics
            if ( dtemp_meta .gt. 0.d0 ) then

!              //   bias potential
               enum = log( 1.d0 + beta_b_tass*enum ) / beta_b_tass

!           //   end of if statement
            end if

!           //   bias function at cv position
            vb_cv(kstep_cv) = enum

!        //   loop of beads
         end do

!        //   loop of cv steps
         end do

!-----------------------------------------------------------------------
!     //   3D case: calculate bias potential in metadynamics
!-----------------------------------------------------------------------

!     //   3D case
      else if ( ntass_meta .eq. 3 ) then

!        //   loop of cv steps
         do istep_cv = 1, nstep_cv

!        //   loop of beads
         do ibead = 1, nbead

!           //   cv bead step
            kstep_cv = ( istep_cv - 1 ) * nbead + ibead

!           //   number of hills
            mstep_hills = istep_cv * iprint_cv_tass / iprint_hills_tass

!           //   sum of hills at cv position
            enum = 0.d0

!           //   loop of hills
            do jstep_hills = 1, mstep_hills

!           //   loop of beads
            do jbead = 1, nbead

!              //   hills bead step
               lstep_hills = ( jstep_hills - 1 ) * nbead + jbead

!              //   metadynamics cv
               icvm = label_meta_tass(1)
               jcvm = label_meta_tass(2)
               kcvm = label_meta_tass(3)

!              //   gaussian height
               gh  = gh_cv(lstep_hills)

!              //   gaussian width
               gwi = gw_cv(icvm,lstep_hills)
               gwj = gw_cv(jcvm,lstep_hills)
               gwk = gw_cv(kcvm,lstep_hills)

!              //   gaussian center
               gci = gc_cv(icvm,lstep_hills)
               gcj = gc_cv(jcvm,lstep_hills)
               gck = gc_cv(kcvm,lstep_hills)

!              //   gaussian width inverse squared
               gwi_inv2 = 1.d0 / ( gwi * gwi )
               gwj_inv2 = 1.d0 / ( gwj * gwj )
               gwk_inv2 = 1.d0 / ( gwk * gwk )

!              //   distance between cv position and gaussian center
               dcvi = rcv(icvm,kstep_cv) - gci
               dcvj = rcv(jcvm,kstep_cv) - gcj
               dcvk = rcv(kcvm,kstep_cv) - gck

!              /*   periodic boundary condition   */
               call pbc_tass( ipbc_tass(icvm), dcvi )
               call pbc_tass( ipbc_tass(jcvm), dcvj )
               call pbc_tass( ipbc_tass(kcvm), dcvk )

!              //   exponent
               fcvi = 0.5d0 * dcvi * dcvi * gwi_inv2
               fcvj = 0.5d0 * dcvj * dcvj * gwj_inv2
               fcvk = 0.5d0 * dcvk * dcvk * gwk_inv2

!              //  sum of hills at cv position
               enum = enum + gh * exp( - fcvi - fcvj - fcvk )

!           //   loop of beads
            end do

!           //   loop of hills
            end do

!           //   well tempered metadynamics
            if ( dtemp_meta .gt. 0.d0 ) then

!              //   bias potential
               enum = log( 1.d0 + beta_b_tass*enum ) / beta_b_tass

!           //   end of if statement
            end if

!           //   bias function at cv position
            vb_cv(kstep_cv) = enum

!        //   loop of beads
         end do

!        //   loop of cv steps
         end do

!     //   end of if statement
      end if

      return
      end

