!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, H. Kimizuka
!      Last updated:    Jan 7, 2019 by M. Shiga
!      Description:     polymers analysis
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine analysis_polymers_atom_MPI ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fux, fuy, fuz, pot, boltz, temperature, x, y, z, &
     &   ux, uy, uz, fx, fy, fz, natom, istep, iounit, nbead, myrank

      use polymers_variables, only : &
     &   iprint_poly, jpoly

      use analysis_variables, only : &
     &   fux_avg, fuy_avg, fuz_avg,epot, ekinvir, etot, epot_avg, &
     &   ekinvir_avg, etot_avg

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ioption, i, j

      real(8), save :: dstep = 0.d0

      real(8) :: ekinvir1, ekinvir2

      character(len=30) :: filename
      character(len=3)  :: char_num

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_poly .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 1:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         if ( istep .eq. 0 ) then

            if ( .not. allocated(fux_avg) ) allocate( fux_avg(natom) )
            if ( .not. allocated(fuy_avg) ) allocate( fuy_avg(natom) )
            if ( .not. allocated(fuz_avg) ) allocate( fuz_avg(natom) )

            fux_avg(:)  = 0.d0
            fuy_avg(:)  = 0.d0
            fuz_avg(:)  = 0.d0
            ekinvir_avg = 0.d0
            epot_avg    = 0.d0

            dstep       = 0.d0

            if ( myrank .eq. 0 ) then

               call int3_to_char( jpoly, char_num )

               filename = 'poly.' // char_num // '/favg.out'

               open ( iounit, file = filename, access ='append' )

               write( iounit, '(a)' ) &
     &            '====================' // &
     &            '====================================' // &
     &            '===================================='
               write( iounit, '(a)' ) &
     &            '    step' // &
     &            '           fux           fuy           fuz' // &
     &            '       fux_avg       fuy_avg       fuz_avg'
               write( iounit, '(a)' ) &
     &            '--------------------' // &
     &            '------------------------------------' // &
     &            '------------------------------------'

               close( iounit )

               filename = 'poly.' // char_num // '/eavg.out'

               open ( iounit, file = filename, access ='append' )

               write( iounit, '(a)' ) &
     &            '====================' // &
     &            '====================================' // &
     &            '===================================='
               write( iounit, '(a)' ) &
     &            '    step' // &
     &            '       ekinvir          epot          etot' // &
     &            '   ekinvir_avg      epot_avg      etot_avg'
               write( iounit, '(a)' ) &
     &            '--------------------' // &
     &            '------------------------------------' // &
     &            '------------------------------------'

               close( iounit )

            end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

         else

            if ( .not. allocated(fux_avg) ) allocate( fux_avg(natom) )
            if ( .not. allocated(fuy_avg) ) allocate( fuy_avg(natom) )
            if ( .not. allocated(fuz_avg) ) allocate( fuz_avg(natom) )

            if ( myrank .eq. 0 ) then

               call int3_to_char( jpoly, char_num )
               filename = 'poly.' // char_num // '/averages.ini'

               open ( iounit, file = filename )

               read ( iounit, * ) dstep
               read ( iounit, * ) fux_avg(:)
               read ( iounit, * ) fuy_avg(:)
               read ( iounit, * ) fuz_avg(:)
               read ( iounit, * ) ekinvir_avg
               read ( iounit, * ) epot_avg

               close( iounit )

            end if

            call my_mpi_bcast_real_0 ( dstep )
            call my_mpi_bcast_real_1 ( fux_avg, natom )
            call my_mpi_bcast_real_1 ( fuy_avg, natom )
            call my_mpi_bcast_real_1 ( fuz_avg, natom )
            call my_mpi_bcast_real_0 ( ekinvir_avg )
            call my_mpi_bcast_real_0 ( epot_avg )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         dstep = dstep + 1.d0

!-----------------------------------------------------------------------
!        /*   potential energy                                        */
!-----------------------------------------------------------------------

         epot = 0.d0

         do j = 1, nbead
            epot = epot + pot(j)
         end do

         epot = epot/dble(nbead)

!-----------------------------------------------------------------------
!        /*   kinetic energy by virial estimator                      */
!-----------------------------------------------------------------------

!        /*   first term   */

         ekinvir1 = 1.5d0*dble(natom)*boltz*temperature

!        /*   second term   */

         ekinvir2 = 0.d0

         do j = 1, nbead

            do i = 1, natom
               ekinvir2 = ekinvir2 + (x(i,j)-ux(i,1))*fx(i,j) &
     &                             + (y(i,j)-uy(i,1))*fy(i,j) &
     &                             + (z(i,j)-uz(i,1))*fz(i,j)
            end do

         end do

         ekinvir2 = - 0.5d0*ekinvir2

!        /*   the sum of kinetic energy   */

         ekinvir = ekinvir1 + ekinvir2

!-----------------------------------------------------------------------
!        /*   total energy                                            */
!-----------------------------------------------------------------------

         etot = ekinvir + epot

!-----------------------------------------------------------------------
!        /*   averages                                                */
!-----------------------------------------------------------------------

         fux_avg(:) = fux(:,1) /dstep &
     &              + fux_avg(:) * (dstep-1.d0)/dstep
         fuy_avg(:) = fuy(:,1) /dstep &
     &              + fuy_avg(:) * (dstep-1.d0)/dstep
         fuz_avg(:) = fuz(:,1) /dstep &
     &              + fuz_avg(:) * (dstep-1.d0)/dstep

         ekinvir_avg = ekinvir /dstep &
     &               + ekinvir_avg * (dstep-1.d0)/dstep

         epot_avg    = epot /dstep &
     &               + epot_avg * (dstep-1.d0)/dstep

         etot_avg    = ekinvir_avg + epot_avg

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_poly) .eq. 0 ) then

            if ( myrank .eq. 0 ) then

               call int3_to_char( jpoly, char_num )
               filename = 'poly.' // char_num // '/favg.out'

               open ( iounit, file = filename, access ='append' )

               do i = 1, natom

                  write( iounit, '(i8,6(1p,e14.6))' ) &
     &                  istep, fux(i,1), fuy(i,1), fuz(i,1), &
     &                      fux_avg(i), fuy_avg(i), fuz_avg(i)

               end do

               close( iounit )

               call int3_to_char( jpoly, char_num )
               filename = 'poly.' // char_num // '/eavg.out'

               open ( iounit, file = filename, access ='append' )

               write( iounit, '(i8,6(1p,e14.6))' ) &
     &            istep, ekinvir, epot, etot, &
     &            ekinvir_avg, epot_avg, etot_avg

               close( iounit )

            end if

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         if ( myrank .eq. 0 ) then

            call int3_to_char( jpoly, char_num )
            filename = 'poly.' // char_num // '/averages.ini'

            open ( iounit, file = filename )

            write( iounit, '(e24.16)' ) dstep
            write( iounit, '(e24.16)' ) fux_avg(:)
            write( iounit, '(e24.16)' ) fuy_avg(:)
            write( iounit, '(e24.16)' ) fuz_avg(:)
            write( iounit, '(e24.16)' ) ekinvir_avg
            write( iounit, '(e24.16)' ) epot_avg

            close( iounit )

         end if

      end if

      return
      end





!***********************************************************************
      subroutine analysis_polymers_cons_MPI ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, boltz, temperature, x, y, z, &
     &   ux, uy, uz, fx, fy, fz, natom, istep, iounit, nbead, myrank

      use cons_variables, only : &
     &   rcons, scons, fref_cons, scons_avg, fref_cons_avg, &
     &   ipbc_cons, ncons

      use polymers_variables, only : &
     &   iprint_poly, jpoly

      use analysis_variables, only : &
     &   epot, ekinvir, etot, epot_avg, ekinvir_avg, etot_avg, &
     &   iprint_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ioption, i, j, k

      real(8), save :: dstep = 0.d0

      real(8) :: ekinvir1, ekinvir2, dr, ds

      character(len=30) :: filename
      character(len=3)  :: char_num

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_poly .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 1:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         if ( istep .eq. 0 ) then

            ekinvir_avg = 0.d0
            epot_avg    = 0.d0

            scons_avg(:,:)        = 0.d0
            fref_cons_avg(:,:)    = 0.d0

            dstep       = 0.d0

            if ( myrank .eq. 0 ) then

               call int3_to_char( jpoly, char_num )

               filename = 'poly.' // char_num // '/eavg.out'

               open ( iounit, file = filename, access ='append' )

               write( iounit, '(a)' ) &
     &            '====================' // &
     &            '====================================' // &
     &            '===================================='
               write( iounit, '(a)' ) &
     &            '    step' // &
     &            '       ekinvir          epot          etot' // &
     &            '   ekinvir_avg      epot_avg      etot_avg'
               write( iounit, '(a)' ) &
     &            '--------------------' // &
     &            '------------------------------------' // &
     &            '------------------------------------'

               close( iounit )

            end if

            if ( myrank .eq. 0 ) then
            if ( iprint_cons .gt. 0 ) then

               call int3_to_char( jpoly, char_num )

               filename = 'poly.' // char_num // '/cons.out'

               open ( iounit, file = filename, access = 'append' )

               write( iounit, '(a)' ) &
     &            '===========================================' &
     &         // '========================='
               write( iounit, '(a)' ) &
     &            '    step       ideal      actual      force' &
     &         // '   actual_avg   force_avg'
               write( iounit, '(a)' ) &
     &            '-------------------------------------------' &
     &         // '-------------------------'

            close( iounit )

            end if
            end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

         else

            if ( myrank .eq. 0 ) then

               call int3_to_char( jpoly, char_num )
               filename = 'poly.' // char_num // '/averages.ini'

               open ( iounit, file = filename )

               read ( iounit, * ) dstep
               read ( iounit, * ) ekinvir_avg
               read ( iounit, * ) epot_avg

               do i = 1, ncons
                  write( iounit, * ) scons_avg(i,1)
               end do

               do i = 1, ncons
                  write( iounit, * ) fref_cons_avg(i,1)
               end do

               close( iounit )

            end if

            call my_mpi_bcast_real_0 ( dstep )
            call my_mpi_bcast_real_0 ( ekinvir_avg )
            call my_mpi_bcast_real_0 ( epot_avg )
            call my_mpi_bcast_real_2 ( scons_avg, ncons, nbead )
            call my_mpi_bcast_real_2 ( fref_cons_avg, ncons, nbead )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         dstep = dstep + 1.d0

!-----------------------------------------------------------------------
!        /*   potential energy                                        */
!-----------------------------------------------------------------------

         epot = 0.d0

         do j = 1, nbead
            epot = epot + pot(j)
         end do

         epot = epot/dble(nbead)

!-----------------------------------------------------------------------
!        /*   kinetic energy by virial estimator                      */
!-----------------------------------------------------------------------

!        /*   first term   */

         ekinvir1 = 1.5d0*dble(natom)*boltz*temperature

!        /*   second term   */

         ekinvir2 = 0.d0

         do j = 1, nbead

            do i = 1, natom
               ekinvir2 = ekinvir2 + (x(i,j)-ux(i,1))*fx(i,j) &
     &                             + (y(i,j)-uy(i,1))*fy(i,j) &
     &                             + (z(i,j)-uz(i,1))*fz(i,j)
            end do

         end do

         ekinvir2 = - 0.5d0*ekinvir2

!        /*   the sum of kinetic energy   */

         ekinvir = ekinvir1 + ekinvir2

!-----------------------------------------------------------------------
!        /*   total energy                                            */
!-----------------------------------------------------------------------

         etot = ekinvir + epot

!-----------------------------------------------------------------------
!        /*   averages                                                */
!-----------------------------------------------------------------------

         ekinvir_avg = ekinvir /dstep &
     &               + ekinvir_avg * (dstep-1.d0)/dstep

         epot_avg    = epot /dstep &
     &               + epot_avg * (dstep-1.d0)/dstep

         etot_avg    = ekinvir_avg + epot_avg

!-----------------------------------------------------------------------
!        /*   accumulative averages                                   */
!-----------------------------------------------------------------------

         do j = 1, ncons

            ds = 0.d0

            do i = 1, nbead
 
               dr = scons(j,i) - rcons(j,i)

               k = ipbc_cons(j)

               call pbc_cons( dr, k )

               dr = dr + rcons(j,i)

               ds = ds + dr

            end do

            ds = ds / dble(nbead)

            scons_avg(j,1) = ds / dstep &
     &         + scons_avg(j,1) * (dstep-1.d0)/dstep

         end do

!-----------------------------------------------------------------------
!        /*   accumulative averages                                   */
!-----------------------------------------------------------------------

         do j = 1, ncons

            dr = 0.d0

            do i = 1, nbead

               dr = dr + fref_cons(j,i)

            end do

            fref_cons_avg(j,1) = dr / dstep &
     &         + fref_cons_avg(j,1) * (dstep-1.d0)/dstep

         end do

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_poly) .eq. 0 ) then

            if ( myrank .eq. 0 ) then

               call int3_to_char( jpoly, char_num )
               filename = 'poly.' // char_num // '/eavg.out'

               open ( iounit, file = filename, access ='append' )

               write( iounit, '(i8,6(1p,e14.6))' ) &
     &            istep, ekinvir, epot, etot, &
     &            ekinvir_avg, epot_avg, etot_avg

               close( iounit )

            end if

         end if

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         if ( iprint_cons .gt. 0 ) then
         if ( mod(istep,iprint_cons) .eq. 0 ) then
         if ( myrank .eq. 0 ) then

            call int3_to_char( jpoly, char_num )

            filename = 'poly.' // char_num // '/cons.out'

            open ( iounit, file = filename, access = 'append' )

            do j = 1, nbead
            do i = 1, ncons
               write( iounit, '(i8,7f12.6)' ) &
     &            istep, rcons(i,j), scons(i,j), &
     &            fref_cons(i,j)*dble(nbead), &
     &            scons_avg(i,1), fref_cons_avg(i,1)
            end do
            end do

            close( iounit )

         end if
         end if
         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         if ( myrank .eq. 0 ) then

            call int3_to_char( jpoly, char_num )
            filename = 'poly.' // char_num // '/averages.ini'

            open ( iounit, file = filename )

            write( iounit, '(e24.16)' ) dstep
            write( iounit, '(e24.16)' ) ekinvir_avg
            write( iounit, '(e24.16)' ) epot_avg

            do i = 1, ncons
               write( iounit, '(e24.16)' ) scons_avg(i,1)
            end do

            do i = 1, ncons
               write( iounit, '(e24.16)' ) fref_cons_avg(i,1)
            end do

            close( iounit )

         end if

      end if

      return
      end
