!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Mar 2, 2022 by M. Shiga
!      Description:     analysis in metadynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine analysis_meta_MPI ( ioption )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iounit

      use analysis_variables, only : iprint_dcd

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   option   */
      integer :: ioption

!     /*   flag   */
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   header                                                     */
!-----------------------------------------------------------------------

!     /*   initialization   */
      if ( iset .eq. 0 ) then

!        /*   read print interval   */
         call read_int1_MPI ( iprint_dcd,  '<iprint_dcd>',  12, iounit )

         /*   end initialization   */
         iset = 1

!     /*   initialization   */
      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  initialize/restart                           */
!-----------------------------------------------------------------------

      if      ( ioption .eq. 0 ) then

#if defined(aenet2) || defined(aenet_pytorch)
         call analysis_aenet_MPI( 0 )
#endif

      else if ( ioption .eq. 1 ) then

!        /*   reconstruction of hills   */
         call analysis_rec_meta_MPI( 1 )

!        /*   energy components   */
         call analysis_ene_meta_MPI( 1 )

!        /*   xyz trajectory   */
         call analysis_xyz_meta_MPI( 1 )

!        /*   xyz trajectory   */
         call analysis_xsf_meta_MPI( 1 )

!        /*   dcd trajectory   */
         call analysis_dcd_beadwise_MPI( 1, 0 )

!        /*   cv trajectory   */
         call analysis_cv_meta_MPI ( 1 )

#if defined(aenet2) || defined(aenet_pytorch)
         call analysis_aenet_MPI( 1 )
#endif

      else if ( ioption .eq. 2 ) then

!        /*   reconstruction of hills   */
         call analysis_rec_meta_MPI( 2 )

!        /*   energy components   */
         call analysis_ene_meta_MPI( 2 )

!        /*   xyz trajectory   */
         call analysis_xyz_meta_MPI( 2 )

!        /*   xyz trajectory   */
         call analysis_xsf_meta_MPI( 2 )

!        /*   dcd trajectory   */
         call analysis_dcd_beadwise_MPI( 2, 0 )

!        /*   cv trajectory   */
         call analysis_cv_meta_MPI ( 2 )

#if defined(aenet2) || defined(aenet_pytorch)
         call analysis_aenet_MPI( 2 )
#endif

      else if ( ioption .eq. 3 ) then

!        /*   reconstruction of hills   */
         call analysis_rec_meta_MPI( 3 )

!        /*   energy components   */
         call analysis_ene_meta_MPI( 3 )

!        /*   xyz trajectory   */
         call analysis_xyz_meta_MPI( 3 )

!        /*   xyz trajectory   */
         call analysis_xsf_meta_MPI( 3 )

!        /*   dcd trajectory   */
         call analysis_dcd_beadwise_MPI( 3, 0 )

!        /*   cv trajectory   */
         call analysis_cv_meta_MPI ( 3 )

#if defined(aenet2) || defined(aenet_pytorch)
         call analysis_aenet_MPI( 3 )
#endif

      else

         continue

      end if

      return
      end





!***********************************************************************
      subroutine analysis_rec_meta_MPI( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use meta_variables, only : nmeta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ioption

!-----------------------------------------------------------------------
!     /*   hills reconstruction                                       */
!-----------------------------------------------------------------------

      if ( nmeta .eq. 1 )  call analysis_rec_1d_MPI( ioption )
      if ( nmeta .eq. 2 )  call analysis_rec_2d_MPI( ioption )
      if ( nmeta .eq. 3 )  call analysis_rec_3d_MPI( ioption )

      return
      end





!***********************************************************************
      subroutine analysis_ene_meta_MPI( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, boltz, potential, hamiltonian, &
     &   natom, nbead, istep, myrank

      use meta_variables, only : &
     &   ekin_sys, ekin_meta, potential_meta, pot_lifted, ebath_meta, &
     &   potential_ref_meta, ebath_sys, iprint_meta, nmeta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: itest, ioption

      real(8) :: pot_combined, temp_sys, temp_meta

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   if set is not done   */
      if ( iset .eq. 0 ) then

!        /*   read print interval   */
         call read_int1_MPI( iprint_meta, '<iprint_meta>', 13, iounit )

!        /*   set done   */
         iset = 1

!        /*   return by option   */
         if ( iprint_meta .le. 0 ) return

!        /*   master process only   */
         if ( myrank .ne. 0 ) return

!        /*   check if file exists   */
         call testfile ( 'meta.out', 8, itest )

!        /*   if the file does not exist print header   */
         if ( itest .eq. 1 ) then

!           /*   open file   */
            open ( iounit, file = 'meta.out', access = 'append' )

!           /*   print header   */
            write( iounit, '(a)' ) &
     &      '========================================================'
            write( iounit, '(a)' ) &
     &      '    step       potential  potential_meta    pot_combined'
            write( iounit, '(a)' ) &
     &      '    step    pot_ref_meta    - pot_lifted        ekin_sys'
            write( iounit, '(a)' ) &
     &      '    step       ekin_meta       ebath_sys      ebath_meta'
            write( iounit, '(a)' ) &
     &      '    step     hamiltonian        temp_sys       temp_meta'
            write( iounit, '(a)' ) &
     &      '--------------------------------------------------------'

!           /*   close file   */
            close( iounit )

!        /*   if the file already exist do not print header   */
         else

            continue

         end if

!        /*   set done   */
         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 )  continue
      if ( ioption .eq. 2 )  continue
      if ( ioption .eq. 3 )  continue

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

!     /*   print interval   */
      if ( iprint_meta .le. 0 )  return

!     /*   master process only   */
      if ( myrank .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   print energies                                             */
!-----------------------------------------------------------------------

!     /*   print interval   */
      if ( mod(istep,iprint_meta) .eq. 0 ) then

!        /*   instant temperature of atoms   */
         temp_sys  = 2.d0*ekin_sys/dble(3*natom*nbead)/boltz

!        /*   instant temperature of cv   */
         temp_meta = 0.d0
         if ( nmeta .gt. 0 ) then
            temp_meta = 2.d0*ekin_meta/dble(nmeta*nbead)/boltz
         end if

!        /*   real potential and hill potential combined   */
         pot_combined = potential + potential_meta

!        /*   open file   */
         open ( iounit, file = 'meta.out', access = 'append' )

!        /*   print energies   */

         write(iounit,'(i8,3d16.8)') &
     &      istep,          potential,  potential_meta,    pot_combined
         write(iounit,'(i8,3d16.8)') &
     &      istep, potential_ref_meta,    - pot_lifted,        ekin_sys
         write(iounit,'(i8,3d16.8)') &
     &      istep,          ekin_meta,       ebath_sys,      ebath_meta
         write(iounit,'(i8,3d16.8)') &
     &      istep,        hamiltonian,        temp_sys,       temp_meta

!        /*   close file   */
         close( iounit )

      end if

      return
      end





!***********************************************************************
      subroutine analysis_xyz_meta_MPI ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, au_length, x, y, z, species, &
     &   iounit, iounit_xyz, istep, natom, nbead, mbox, myrank

      use meta_variables, only : &
     &   iprint_xyz

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, k, ioption, m1, m2, m3

      real(8) :: xa, ya, za, xb, yb, zb

      character(len=11) :: filename

      character(len=3) :: char_num

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

!     /*   if set is not done   */
      if ( iset .eq. 0 ) then

!        /*   read print interval   */
         call read_int1_MPI ( iprint_xyz,  '<iprint_xyz>',  12, iounit )

!        /*   set done   */
         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

!     /*   print interval   */
      if ( iprint_xyz .le. 0 ) return

!     /*   master process only   */
      if ( myrank .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

!cc         open ( iounit_xyz,  file = 'trj.xyz' )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

!cc         open ( iounit_xyz,  file = 'trj.xyz', access = 'append' )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

!-----------------------------------------------------------------------
!     /*   print trajectory                                           */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_xyz) .eq. 0 ) then

            do k = 1, nbead

!              /*   bead number   */
               call int3_to_char( k, char_num )

!              /*   file name  */
               filename = 'trj.' // char_num(1:3) // '.xyz'

!              /*   open file   */
               open ( iounit_xyz, file = filename, access = 'append' )

!              /*   number of atoms, number of beads   */
               write(iounit_xyz,'(2i8)')      natom, nbead

!              /*   step number, potential   */
               write(iounit_xyz,'(i8,f16.8)') istep, pot(k)

               do i = 1, natom

!                 /*   [bohr]  -->  [angstrom]   */
                  xa = x(i,k) *au_length/1.d-10
                  ya = y(i,k) *au_length/1.d-10
                  za = z(i,k) *au_length/1.d-10

!                 /*   unfolded structure   */
                  xb = x(i,k)
                  yb = y(i,k)
                  zb = z(i,k)
                  m1 = mbox(1,i,k)
                  m2 = mbox(2,i,k)
                  m3 = mbox(3,i,k)
                  call pbc_unfold_MPI( xb, yb, zb, m1, m2, m3 )
                  xb = xb *au_length/1.d-10
                  yb = yb *au_length/1.d-10
                  zb = zb *au_length/1.d-10

!                 /*   print atomic species, cartesian coordinates   */
                  write( iounit_xyz, '(a4,6f16.8)' ) &
     &               species(i)(1:4), xa, ya, za, xb, yb, zb

               end do

!              /*   close file   */
               close( iounit_xyz )

            end do

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         continue

      end if

      return
      end





!***********************************************************************
      subroutine analysis_cv_meta_MPI ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, istep, nbead, myrank

      use meta_variables, only : &
     &   rmeta, smeta, iprint_cv_meta, nmeta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, ioption, itest

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

!     /*   if set is not done   */
      if ( iset .eq. 0 ) then

!        /*   read print interval   */
         call read_int1_MPI &
     &      ( iprint_cv_meta, '<iprint_cv_meta>',  16, iounit )

!        /*   set done   */
         iset = 1

!        /*   return by option   */
         if ( iprint_cv_meta .le. 0 ) return

!        /*   master process only   */
         if ( myrank .ne. 0 ) return

!        /*   check if file exists   */
         call testfile ( 'cv.out', 6, itest )

!        /*   if the file does not exist print header   */
         if ( itest .eq. 1 ) then

!           /*   open file   */
            open ( iounit, file = 'cv.out', access = 'append' )

!           /*   print header   */
            write( iounit, '(a)' ) &
     &      '========================================'
            write( iounit, '(a)' ) &
     &      '    step           rmeta           smeta'
            write( iounit, '(a)' ) &
     &      '----------------------------------------'

!           /*   close file   */
            close( iounit )

         else

            continue

         end if

!        /*   set done   */
         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

!     /*   print interval   */
      if ( iprint_cv_meta .le. 0 ) return

!     /*   master process only   */
      if ( myrank .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         return

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

!        /*   print interval   */
         if ( mod(istep,iprint_cv_meta) .eq. 0 ) then

!           /*   open file   */
            open ( iounit,  file = 'cv.out', access = 'append' )

!           /*   step number, cv, actual cv   */
            do j = 1, nbead
            do i = 1, nmeta
               write(iounit,'(i8,2f16.8)') istep, rmeta(i,j), smeta(i,j)
            end do
            end do

!           /*   close file   */
            close( iounit )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         continue

      end if

      return
      end





!***********************************************************************
      subroutine analysis_rec_1d_MPI( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, temperature, istep, iounit, myrank

      use meta_variables, only : &
     &   params_rec_meta, gh_meta, gc_meta, gw_meta, pot_1d, dtemp_meta, &
     &   ipbc_meta, iprint_rec_meta, ng_meta, itype_meta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, nmesh1, j1, ioption

      real(8) :: rmin1, rmax1, dr1, rmesh1

      real(8) :: f0, fa1, fb1, fc1, f10, f5, f11, f15

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   master process only                                        */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 )  return

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 )  continue
      if ( ioption .eq. 2 )  continue
      if ( ioption .eq. 3 )  continue

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_rec_meta .lt. 0 )  return

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( mod(istep,iprint_rec_meta) .ne. 0 )  return

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   constant   */
      f5 = boltz * dtemp_meta

!     /*   constant   */
      f15 = f5 + boltz*temperature

!     /*   constant   */
      f11 = 0.d0

!-----------------------------------------------------------------------
!     /*   prepare meshes                                             */
!-----------------------------------------------------------------------

!     /*   mininum value of mesh   */
      rmin1  = params_rec_meta(1,itype_meta(1))

!     /*   maximum value of mesh   */
      rmax1  = params_rec_meta(2,itype_meta(1))

!     /*   mesh size   */
      dr1    = params_rec_meta(3,itype_meta(1))

!     /*   number of meshes    */
      nmesh1 = nint( ( rmax1 - rmin1 )/dr1 ) + 1

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         if ( .not. allocated( pot_1d ) ) &
     &      allocate( pot_1d(nmesh1) )

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

!     /*   zero clear   */
      pot_1d(:) = 0.d0

!     /*   mesh point   */
      do j1 = 1, nmesh1

!        /*   sum   */
         f10 = 0.d0

!        /*   gaussians   */
         do i = 1, ng_meta

!           /*   height   */
            f0 = gh_meta(i)

!           /*   mesh position   */
            rmesh1 = rmin1 + (j1-1)*dr1

!           /*   displacement   */
            fa1 = rmesh1 - gc_meta(1,i)

!           /*   periodic boundary condition   */
            call pbc_meta( ipbc_meta(1), fa1 )

!           /*   width   */
            fb1 = gw_meta(1,i)

!           /*   exponent   */
            fc1 = 0.5d0*(fa1*fa1)/(fb1*fb1)

!           /*   skip small values   */
            if ( fc1 .gt. 75.d0 ) cycle

!           /*   potential   */
            f10 = f10 + f0*exp(-fc1)

!        /*   gaussians   */
         end do

!        /*   conventional metadynamics   */
         if ( dtemp_meta .le. 0.d0 ) then

!           /*   free energy   */
            pot_1d(j1) = - f10

!        /*   well tempered metadynamics   */
         else

!           /*   free energy   */
            pot_1d(j1) = - f15 * log( 1.d0 + f10/f5 )

!        /*   metadynamics   */
         end if

!        /*   minimum   */
         f11 = min( pot_1d(j1), f11 )

!     /*   mesh point   */
      end do

!-----------------------------------------------------------------------
!     /*   output mesh and hill potential                             */
!-----------------------------------------------------------------------

!     /*   open file   */
      open ( iounit, file = 'rec.out' )

!     /*   mesh point   */
      do j1 = 1, nmesh1

!        /*   mesh position   */
         rmesh1 = rmin1 + (j1-1)*dr1

!        /*   write   */
         write( iounit, '(2f16.8)' )  rmesh1, pot_1d(j1)-f11

!     /*   mesh point   */
      end do

!     /*   close file   */
      close( iounit )

!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine analysis_rec_2d_MPI( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, temperature, istep, iounit, myrank

      use meta_variables, only : &
     &   params_rec_meta, gh_meta, gc_meta, gw_meta, pot_2d, dtemp_meta, &
     &   ipbc_meta, iprint_rec_meta, ng_meta, itype_meta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, nmesh1, nmesh2, j1, j2, ioption

      real(8) :: rmin1, rmin2, rmax1, rmax2, dr1, dr2, rmesh1, rmesh2

      real(8) :: f0, fa1, fa2, fb1, fb2, fc1, fc2, f10, f5, f11, f15

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   master process only                                        */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 )  return

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 )  continue
      if ( ioption .eq. 2 )  continue
      if ( ioption .eq. 3 )  continue

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_rec_meta .lt. 0 )  return

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( mod(istep,iprint_rec_meta) .ne. 0 )  return

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   constant   */
      f5 = boltz * dtemp_meta

!     /*   constant   */
      f15 = f5 + boltz*temperature

!     /*   constant   */
      f11 = 0.d0

!-----------------------------------------------------------------------
!     /*   prepare meshes                                             */
!-----------------------------------------------------------------------

!     /*   mininum value of mesh   */
      rmin1  = params_rec_meta(1,itype_meta(1))
      rmin2  = params_rec_meta(1,itype_meta(2))

!     /*   maximum value of mesh   */
      rmax1  = params_rec_meta(2,itype_meta(1))
      rmax2  = params_rec_meta(2,itype_meta(2))

!     /*   mesh size   */
      dr1    = params_rec_meta(3,itype_meta(1))
      dr2    = params_rec_meta(3,itype_meta(2))

!     /*   number of meshes    */
      nmesh1 = nint( ( rmax1 - rmin1 )/dr1 ) + 1
      nmesh2 = nint( ( rmax2 - rmin2 )/dr2 ) + 1

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         if ( .not. allocated( pot_2d ) ) &
     &      allocate( pot_2d(nmesh1,nmesh2) )

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

!     /*   zero clear   */
      pot_2d(:,:) = 0.d0

!     /*   mesh point   */
      do j1 = 1, nmesh1
      do j2 = 1, nmesh2

!        /*   sum   */
         f10 = 0.d0

!        /*   gaussians   */
         do i = 1, ng_meta

!           /*   height   */
            f0 = gh_meta(i)

!           /*   mesh position   */
            rmesh1 = rmin1 + (j1-1)*dr1
            rmesh2 = rmin2 + (j2-1)*dr2

!           /*   displacement   */
            fa1 = rmesh1 - gc_meta(1,i)
            fa2 = rmesh2 - gc_meta(2,i)

!           /*   periodic boundary condition   */
            call pbc_meta( ipbc_meta(1), fa1 )
            call pbc_meta( ipbc_meta(2), fa2 )

!           /*   width   */
            fb1 = gw_meta(1,i)
            fb2 = gw_meta(2,i)

!           /*   exponent   */
            fc1 = 0.5d0*(fa1*fa1)/(fb1*fb1)
            fc2 = 0.5d0*(fa2*fa2)/(fb2*fb2)

!           /*   skip small values   */
            if ( (fc1+fc2) .gt. 75.d0 ) cycle

!           /*   potential   */
            f10 = f10 + f0*exp(-fc1-fc2)

!        /*   gaussians   */
         end do

!        /*   conventional metadynamics   */
         if ( dtemp_meta .le. 0.d0 ) then

!           /*   free energy   */
            pot_2d(j1,j2) = - f10

!        /*   well tempered metadynamics   */
         else

!           /*   free energy   */
            pot_2d(j1,j2) = - f15 * log( 1.d0 + f10/f5 )

!        /*   metadynamics   */
         end if

!        /*   minimum   */
         f11 = min( pot_2d(j1,j2), f11 )

!     /*   mesh point   */
      end do
      end do

!-----------------------------------------------------------------------
!     /*   output mesh and hill potential                             */
!-----------------------------------------------------------------------

!     /*   open file   */
      open ( iounit, file = 'rec.out' )

!     /*   mesh point   */
      do j1 = 1, nmesh1

!         /*   mesh point   */
         do j2 = 1, nmesh2

!           /*   mesh position   */
            rmesh1 = rmin1 + (j1-1)*dr1
            rmesh2 = rmin2 + (j2-1)*dr2

!           /*   write   */
            write( iounit, '(3f16.8)' ) &
     &         rmesh1, rmesh2, pot_2d(j1,j2)-f11

!         /*   mesh point   */
         end do

!        /*   write   */
         write( iounit, * )

!     /*   mesh point   */
      end do

!     /*   close file   */
      close( iounit )

!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine analysis_rec_3d_MPI( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iounit

      use meta_variables, only : cut_rec_3d

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ioption

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initial setting                                            */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_real1_MPI ( cut_rec_3d, '<cut_rec_3d>', 12, iounit )

      end if

!-----------------------------------------------------------------------
!     /*   start analysis                                             */
!-----------------------------------------------------------------------

      if ( cut_rec_3d .gt. 37.d0 ) then

         call analysis_rec_3d_full_MPI( ioption )

      else

         call analysis_rec_3d_cutoff_MPI( ioption )

      end if

      return
      end





!***********************************************************************
      subroutine analysis_rec_3d_full_MPI( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, temperature, istep, iounit, myrank

      use meta_variables, only : &
     &   params_rec_meta, gh_meta, gc_meta, gw_meta, pot_3d, dtemp_meta, &
     &   ipbc_meta, iprint_rec_meta, ng_meta, itype_meta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, nmesh1, nmesh2, nmesh3, j1, j2, j3, ioption

      real(8) :: rmin1, rmin2, rmin3, rmax1, rmax2, rmax3, &
     &           dr1, dr2, dr3, rmesh1, rmesh2, rmesh3

      real(8) :: f0, fa1, fa2, fa3, fb1, fb2, fb3, fc1, fc2, fc3

      real(8) :: f10, f5, f11, f15

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   master process only                                        */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 )  return

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 )  continue
      if ( ioption .eq. 2 )  continue
      if ( ioption .eq. 3 )  continue

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_rec_meta .lt. 0 )  return

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( mod(istep,iprint_rec_meta) .ne. 0 )  return

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   constant   */
      f5 = boltz * dtemp_meta

!     /*   constant   */
      f15 = f5 + boltz*temperature

!     /*   constant   */
      f11 = 0.d0

!-----------------------------------------------------------------------
!     /*   prepare meshes                                             */
!-----------------------------------------------------------------------

!     /*   mininum value of mesh   */
      rmin1  = params_rec_meta(1,itype_meta(1))
      rmin2  = params_rec_meta(1,itype_meta(2))
      rmin3  = params_rec_meta(1,itype_meta(3))

!     /*   maximum value of mesh   */
      rmax1  = params_rec_meta(2,itype_meta(1))
      rmax2  = params_rec_meta(2,itype_meta(2))
      rmax3  = params_rec_meta(2,itype_meta(3))

!     /*   mesh size   */
      dr1    = params_rec_meta(3,itype_meta(1))
      dr2    = params_rec_meta(3,itype_meta(2))
      dr3    = params_rec_meta(3,itype_meta(3))

!     /*   number of meshes    */
      nmesh1 = nint( ( rmax1 - rmin1 )/dr1 ) + 1
      nmesh2 = nint( ( rmax2 - rmin2 )/dr2 ) + 1
      nmesh3 = nint( ( rmax3 - rmin3 )/dr3 ) + 1

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         if ( .not. allocated( pot_3d ) ) &
     &      allocate( pot_3d(nmesh1,nmesh2,nmesh3) )

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

!     /*   zero clear   */
      pot_3d(:,:,:) = 0.d0

!     /*   mesh point   */
      do j1 = 1, nmesh1
      do j2 = 1, nmesh2
      do j3 = 1, nmesh3

!        /*   sum   */
         f10 = 0.d0

!        /*   gaussians   */
         do i = 1, ng_meta

!           /*   height   */
            f0 = gh_meta(i)

!           /*   mesh position   */
            rmesh1 = rmin1 + (j1-1)*dr1
            rmesh2 = rmin2 + (j2-1)*dr2
            rmesh3 = rmin3 + (j3-1)*dr3

!           /*   displacement   */
            fa1 = rmesh1 - gc_meta(1,i)
            fa2 = rmesh2 - gc_meta(2,i)
            fa3 = rmesh3 - gc_meta(3,i)

!           /*   periodic boundary condition   */
            call pbc_meta( ipbc_meta(1), fa1 )
            call pbc_meta( ipbc_meta(2), fa2 )
            call pbc_meta( ipbc_meta(3), fa3 )

!           /*   width   */
            fb1 = gw_meta(1,i)
            fb2 = gw_meta(2,i)
            fb3 = gw_meta(3,i)

!           /*   exponent   */
            fc1 = 0.5d0*(fa1*fa1)/(fb1*fb1)
            fc2 = 0.5d0*(fa2*fa2)/(fb2*fb2)
            fc3 = 0.5d0*(fa3*fa3)/(fb3*fb3)

!           /*   skip small values   */
            if ( (fc1+fc2+fc3) .gt. 75.d0 ) cycle

!           /*   potential   */
            f10 = f10 + f0*exp(-fc1-fc2-fc3)

!        /*   gaussians   */
         end do

!        /*   conventional metadynamics   */
         if ( dtemp_meta .le. 0.d0 ) then

!           /*   free energy   */
            pot_3d(j1,j2,j3) = - f10

!        /*   well tempered metadynamics   */
         else

!           /*   free energy   */
            pot_3d(j1,j2,j3) = - f15 * log( 1.d0 + f10/f5 )

!        /*   metadynamics   */
         end if

!        /*   minimum   */
         f11 = min( pot_3d(j1,j2,j3), f11 )

!     /*   mesh point   */
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   output mesh and hill potential                             */
!-----------------------------------------------------------------------

!     /*   open file   */
      open ( iounit, file = 'rec.out' )

!     /*   mesh point   */
      do j1 = 1, nmesh1
      do j2 = 1, nmesh2
      do j3 = 1, nmesh3

!        /*   mesh position   */
         rmesh1 = rmin1 + (j1-1)*dr1
         rmesh2 = rmin2 + (j2-1)*dr2
         rmesh3 = rmin3 + (j3-1)*dr3

!        /*   write   */
         write( iounit, '(4f16.8)' ) &
     &      rmesh1, rmesh2, rmesh3, pot_3d(j1,j2,j3)-f11

!     /*   mesh point   */
      end do
      end do
      end do

!     /*   close file   */
      close( iounit )

!-----------------------------------------------------------------------
!     /*   output mesh and hill potential in cube format              */
!-----------------------------------------------------------------------

!     /*   open file   */
      open ( iounit, file = 'rec.cube' )

      write( iounit, '(a)' ) " PIMD CUBE FILE."

      write( iounit, '(a)' ) &
     &   " OUTER LOOP: X, MIDDLE LOOP: Y, INNER LOOP: Z "

      write( iounit, '(i5,3e16.8)' ) &
     &   0, rmin1, rmin2, rmin3

!      write( iounit, '(i5,3e16.8)' )
!     &   nmesh1, rmax1-rmin1, 0.d0, 0.d0
!
!      write( iounit, '(i5,3e16.8)' )
!     &   nmesh2, 0.d0, rmax2-rmin2, 0.d0
!
!      write( iounit, '(i5,3e16.8)' )
!     &   nmesh3, 0.d0, 0.d0, rmax3-rmin3

      write( iounit, '(i5,3e16.8)' ) nmesh1, 1.d0, 0.d0, 0.d0
      write( iounit, '(i5,3e16.8)' ) nmesh2, 0.d0, 1.d0, 0.d0
      write( iounit, '(i5,3e16.8)' ) nmesh3, 0.d0, 0.d0, 1.d0

      do j1 = 1, nmesh1
      do j2 = 1, nmesh2
      do j3 = 1, nmesh3

!        /*   write   */
         write( iounit, '(f16.8)' ) pot_3d(j1,j2,j3)

      end do
      end do
      end do

!     /*   close file   */
      close( iounit )

      return
      end





!***********************************************************************
      subroutine analysis_rec_3d_cutoff_MPI( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, temperature, istep, iounit, myrank

      use meta_variables, only : &
     &   params_rec_meta, gh_meta, gc_meta, gw_meta, pot_3d, dtemp_meta, &
     &   ipbc_meta, iprint_rec_meta, ng_meta, itype_meta, cut_rec_3d, &
     &   j1_rec_3d, j2_rec_3d, j3_rec_3d, n1_rec_3d, n2_rec_3d, &
     &   n3_rec_3d

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   number of meshes   */
      integer :: nmesh1, nmesh2, nmesh3

!     /*   option   */
      integer :: ioption

!     /*   flag for initial setting   */
      integer, save :: iset = 0

!     /*   integers   */
      integer :: i, j1, j2, j3, k1, k2, k3

!     /*   real numbers   */
      real(8) :: f0, fa1, fa2, fa3, fb1, fb2, fb3, fc1, fc2, fc3

!     /*   real numbers   */
      real(8) :: f10, f5, f11, f15

!     /*   real numbers   */
      real(8) :: rmin1, rmin2, rmin3, rmax1, rmax2, rmax3

!     /*   real numbers   */
      real(8) :: dr1, dr2, dr3, rmesh1, rmesh2, rmesh3

!-----------------------------------------------------------------------
!     /*   master process only                                        */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 )  return

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 )  continue
      if ( ioption .eq. 2 )  continue
      if ( ioption .eq. 3 )  continue

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_rec_meta .lt. 0 )  return

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( mod(istep,iprint_rec_meta) .ne. 0 )  return

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   constant   */
      f5 = boltz * dtemp_meta

!     /*   constant   */
      f15 = f5 + boltz*temperature

!     /*   constant   */
      f11 = 0.d0

!-----------------------------------------------------------------------
!     /*   prepare meshes                                             */
!-----------------------------------------------------------------------

!     /*   mininum value of mesh   */
      rmin1  = params_rec_meta(1,itype_meta(1))
      rmin2  = params_rec_meta(1,itype_meta(2))
      rmin3  = params_rec_meta(1,itype_meta(3))

!     /*   maximum value of mesh   */
      rmax1  = params_rec_meta(2,itype_meta(1))
      rmax2  = params_rec_meta(2,itype_meta(2))
      rmax3  = params_rec_meta(2,itype_meta(3))

!     /*   mesh size   */
      dr1    = params_rec_meta(3,itype_meta(1))
      dr2    = params_rec_meta(3,itype_meta(2))
      dr3    = params_rec_meta(3,itype_meta(3))

!     /*   number of meshes    */
      nmesh1 = nint( ( rmax1 - rmin1 )/dr1 ) + 1
      nmesh2 = nint( ( rmax2 - rmin2 )/dr2 ) + 1
      nmesh3 = nint( ( rmax3 - rmin3 )/dr3 ) + 1

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         if ( .not. allocated( pot_3d ) ) &
     &      allocate( pot_3d(nmesh1,nmesh2,nmesh3) )

         if ( .not. allocated( j1_rec_3d ) ) &
     &      allocate( j1_rec_3d(nmesh1) )
         if ( .not. allocated( j2_rec_3d ) ) &
     &      allocate( j2_rec_3d(nmesh2) )
         if ( .not. allocated( j3_rec_3d ) ) &
     &      allocate( j3_rec_3d(nmesh3) )

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

!     /*   zero clear   */
      pot_3d(:,:,:) = 0.d0

!-----------------------------------------------------------------------
!     /*   loop of gaussians                                          */
!-----------------------------------------------------------------------

      do i = 1, ng_meta

!        /*   height   */
         f0 = gh_meta(i)

!        /*   width   */
         fb1 = gw_meta(1,i)
         fb2 = gw_meta(2,i)
         fb3 = gw_meta(3,i)

!-----------------------------------------------------------------------
!        /*   make list of meshes                                     */
!-----------------------------------------------------------------------

!        /*   counter   */
         k1 = 0

!        /*   loop of meshes   */
         do j1 = 1, nmesh1

!           /*   mesh position   */
            rmesh1 = rmin1 + (j1-1)*dr1

!           /*   displacement   */
            fa1 = rmesh1 - gc_meta(1,i)

!           /*   periodic boundary condition   */
            call pbc_meta( ipbc_meta(1), fa1 )

!           /*   exponent   */
            fc1 = 0.5d0*(fa1*fa1)/(fb1*fb1)

!           /*   apply cut off   */
            if ( fc1 .le. cut_rec_3d ) then

!              /*   counter   */
               k1 = k1 + 1

!              /*   list of meshes   */
               j1_rec_3d(k1) = j1

!           /*   apply cut off   */
            end if

!        /*   loop of meshes   */
         end do

!        /*   number of meshes   */
         n1_rec_3d = k1

!-----------------------------------------------------------------------
!        /*   make list of meshes                                     */
!-----------------------------------------------------------------------

!        /*   counter   */
         k2 = 0

!        /*   loop of meshes   */
         do j2 = 1, nmesh2

!           /*   mesh position   */
            rmesh2 = rmin2 + (j2-1)*dr2

!           /*   displacement   */
            fa2 = rmesh2 - gc_meta(2,i)

!           /*   periodic boundary condition   */
            call pbc_meta( ipbc_meta(2), fa2 )

!           /*   exponent   */
            fc2 = 0.5d0*(fa2*fa2)/(fb2*fb2)

!           /*   apply cut off   */
            if ( fc2 .le. cut_rec_3d ) then

!              /*   counter   */
               k2 = k2 + 1

!              /*   list of meshes   */
               j2_rec_3d(k2) = j2

!           /*   apply cut off   */
            end if

!        /*   loop of meshes   */
         end do

!        /*   number of meshes   */
         n2_rec_3d = k2

!-----------------------------------------------------------------------
!        /*   make list of meshes                                     */
!-----------------------------------------------------------------------

!        /*   counter   */
         k3 = 0

!        /*   loop of meshes   */
         do j3 = 1, nmesh3

!           /*   mesh position   */
            rmesh3 = rmin3 + (j3-1)*dr3

!           /*   displacement   */
            fa3 = rmesh3 - gc_meta(3,i)

!           /*   periodic boundary condition   */
            call pbc_meta( ipbc_meta(3), fa3 )

!           /*   exponent   */
            fc3 = 0.5d0*(fa3*fa3)/(fb3*fb3)

!           /*   apply cut off   */
            if ( fc3 .le. cut_rec_3d ) then

!              /*   counter   */
               k3 = k3 + 1

!              /*   list of meshes   */
               j3_rec_3d(k3) = j3

!           /*   apply cut off   */
            end if

!        /*   loop of meshes   */
         end do

!        /*   number of meshes   */
         n3_rec_3d = k3

!-----------------------------------------------------------------------
!        /*   calculate sum of hills                                  */
!-----------------------------------------------------------------------

!        /*   loop of mesh list   */
         do k1 = 1, n1_rec_3d
         do k2 = 1, n2_rec_3d
         do k3 = 1, n3_rec_3d

!           /*   meshes   */
            j1 = j1_rec_3d(k1)
            j2 = j2_rec_3d(k2)
            j3 = j3_rec_3d(k3)

!           /*   mesh position   */
            rmesh1 = rmin1 + (j1-1)*dr1
            rmesh2 = rmin2 + (j2-1)*dr2
            rmesh3 = rmin3 + (j3-1)*dr3

!           /*   displacement   */
            fa1 = rmesh1 - gc_meta(1,i)
            fa2 = rmesh2 - gc_meta(2,i)
            fa3 = rmesh3 - gc_meta(3,i)

!           /*   periodic boundary condition   */
            call pbc_meta( ipbc_meta(1), fa1 )
            call pbc_meta( ipbc_meta(2), fa2 )
            call pbc_meta( ipbc_meta(3), fa3 )

!           /*   exponent   */
            fc1 = 0.5d0*(fa1*fa1)/(fb1*fb1)
            fc2 = 0.5d0*(fa2*fa2)/(fb2*fb2)
            fc3 = 0.5d0*(fa3*fa3)/(fb3*fb3)

!           /*   skip small values   */
            if ( (fc1+fc2+fc3) .gt. 75.d0 ) cycle

!           /*   potential   */
            pot_3d(j1,j2,j3) = pot_3d(j1,j2,j3) + f0*exp(-fc1-fc2-fc3)

!        /*   loop of mesh list   */
         end do
         end do
         end do

!-----------------------------------------------------------------------
!     /*   loop of gaussians                                          */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   calculate minimum                                          */
!-----------------------------------------------------------------------

!     /*   conventional metadynamics   */
      if ( dtemp_meta .le. 0.d0 ) then

!        /*   mesh points   */
         do j1 = 1, nmesh1
         do j2 = 1, nmesh2
         do j3 = 1, nmesh3

!           /*   free energy   */
            pot_3d(j1,j2,j3) = - pot_3d(j1,j2,j3)

!           /*   minimum   */
            f11 = min( pot_3d(j1,j2,j3), f11 )

!        /*   mesh points   */
         end do
         end do
         end do

!     /*   well tempered metadynamics   */
      else

!        /*   mesh points   */
         do j1 = 1, nmesh1
         do j2 = 1, nmesh2
         do j3 = 1, nmesh3

!           /*   potential   */
            f10 = - pot_3d(j1,j2,j3)

!           /*   potential   */
            pot_3d(j1,j2,j3) = - f15 * log( 1.d0 + f10/f5 )

!           /*   minimum   */
            f11 = min( pot_3d(j1,j2,j3), f11 )

!        /*   mesh points   */
         end do
         end do
         end do

!     /*   metadynamics   */
      end if

!-----------------------------------------------------------------------
!     /*   output mesh and hill potential                             */
!-----------------------------------------------------------------------

!     /*   open file   */
      open ( iounit, file = 'rec.out' )

!     /*   mesh points   */
      do j1 = 1, nmesh1
      do j2 = 1, nmesh2
      do j3 = 1, nmesh3

!        /*   mesh position   */
         rmesh1 = rmin1 + (j1-1)*dr1
         rmesh2 = rmin2 + (j2-1)*dr2
         rmesh3 = rmin3 + (j3-1)*dr3

!        /*   write   */
         write( iounit, '(4f16.8)' ) &
     &      rmesh1, rmesh2, rmesh3, pot_3d(j1,j2,j3)-f11

!     /*   mesh points   */
      end do
      end do
      end do

!     /*   close file   */
      close( iounit )

!-----------------------------------------------------------------------
!     /*   output mesh and hill potential in cube format              */
!-----------------------------------------------------------------------

!     /*   open file   */
      open ( iounit, file = 'rec.cube' )

      write( iounit, '(a)' ) " PIMD CUBE FILE."

      write( iounit, '(a)' ) &
     &   " OUTER LOOP: X, MIDDLE LOOP: Y, INNER LOOP: Z "

      write( iounit, '(i5,3e16.8)' ) &
     &   0, rmin1, rmin2, rmin3

!      write( iounit, '(i5,3e16.8)' )
!     &   nmesh1, rmax1-rmin1, 0.d0, 0.d0
!
!      write( iounit, '(i5,3e16.8)' )
!     &   nmesh2, 0.d0, rmax2-rmin2, 0.d0
!
!      write( iounit, '(i5,3e16.8)' )
!     &   nmesh3, 0.d0, 0.d0, rmax3-rmin3

      write( iounit, '(i5,3e16.8)' ) nmesh1, 1.d0, 0.d0, 0.d0
      write( iounit, '(i5,3e16.8)' ) nmesh2, 0.d0, 1.d0, 0.d0
      write( iounit, '(i5,3e16.8)' ) nmesh3, 0.d0, 0.d0, 1.d0

      do j1 = 1, nmesh1
      do j2 = 1, nmesh2
      do j3 = 1, nmesh3

!        /*   write   */
         write( iounit, '(f16.8)' ) pot_3d(j1,j2,j3)-f11

      end do
      end do
      end do

!     /*   close file   */
      close( iounit )

      return
      end





!***********************************************************************
      subroutine analysis_xsf_meta_MPI( ioption )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, au_charge, au_energy, au_length, box, species, &
     &   pot, fx, fy, fz, istep, natom, nbead, iounit, iboundary, &
     &   istep, myrank

      use analysis_variables, only : &
     &   iprint_xsf

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: i, j, itest, ioption

!     /*   real numbers   */
      real(8) :: xa, ya, za, ax, ay, az, bx, by, bz, cx, cy, cz

!     /*   real numbers   */
      real(8) :: const_1, const_2, const_3

!     /*   potential and force   */
      real(8) :: pot_ev, fxj, fyj, fzj

!     /*   flag   */
      integer, save :: iset = 0

!     /*   file name   */
      character(len=11) :: char_file

!     /*   bead number   */
      character(len=3) :: char_num

!-----------------------------------------------------------------------
!     /*   header                                                     */
!-----------------------------------------------------------------------

!     /*   initialization   */
      if ( iset .eq. 0 ) then

!        /*   read print interval   */
         call read_int1_MPI ( iprint_xsf,  '<iprint_xsf>',  12, iounit )

!        /*   master rank only   */
         if ( ( myrank .eq. 0 ) .and. ( iprint_xsf .gt. 0 ) ) then

!           /*   loop of beads   */
            do j = 1, nbead

!              /*   bead number   */
               call int3_to_char( j, char_num )

!              /*   file name   */
               char_file = 'trj.' // char_num // '.xsf'

!              /*   test if file exists */
               call testfile ( char_file, 11, itest )

!              /*   if file does not exist   */
               if ( itest .ne. 0 ) then

!                 /*   file open   */
                  open ( iounit, file = char_file )

!                 /*   write header   */
                  write( iounit, '(a)' ) 'ANIMSTEPS 99999999'

!                 /*   file close   */
                  close( iounit )

!              /*   if file does not exist   */
               end if

!           /*   loop of beads   */
            end do

!        /*   master rank only   */
         end if

!        /*   set completed   */
         iset = 1

!     /*   initialization   */
      end if

!-----------------------------------------------------------------------
!     /*   print condition: every iprint_xsf steps                    */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 ) return
      if ( ( ioption .eq. 0 ) .or. ( ioption .eq. 1 ) ) return
      if ( iprint_xsf .lt. 0 ) return
      if ( mod(istep,iprint_xsf) .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   print xsf file                                             */
!-----------------------------------------------------------------------

!     /*   conversion factor   */
      const_1 = au_length * 1.d+10

!     /*   conversion factor   */
      const_2 =  au_charge / au_energy

!     /*   conversion factor   */
      const_3 =  1.d0 / const_2 / const_1

!     /*   loop of beads   */
      do j = 1, nbead

!        /*   bead number   */
         call int3_to_char( j, char_num )

!        /*   file name   */
         char_file = 'trj.' // char_num // '.xsf'

!        /*   open sample file   */
         open( iounit, file = char_file, access = 'append' )

!        /*   potential energy in eV   */
         pot_ev = pot(j) / const_2

!        /*   write one line   */
         write( iounit, '(a,f24.12,a)' ) &
     &      '# total energy = ', pot_ev, ' eV'

!        /*   for periodic boundary condition   */
         if ( (iboundary .eq. 1) .or. (iboundary .eq. 2) ) then

!           /*   write one line   */
            write( iounit, '(a)' ) 'CRYSTAL'

!           /*   write one line   */
            write( iounit, '(a,i8)' ) 'PRIMVEC ', istep

!           /*   lattice vectors   */
            ax = box(1,1) * const_1
            ay = box(2,1) * const_1
            az = box(3,1) * const_1
            bx = box(1,2) * const_1
            by = box(2,2) * const_1
            bz = box(3,2) * const_1
            cx = box(1,3) * const_1
            cy = box(2,3) * const_1
            cz = box(3,3) * const_1

!           /*   write three lines   */
            write( iounit, '(3f16.8)' ) ax, ay, az
            write( iounit, '(3f16.8)' ) bx, by, bz
            write( iounit, '(3f16.8)' ) cx, cy, cz

!           /*   write one line   */
            write( iounit, '(a,i8)' ) 'PRIMCOORD ', istep

!           /*   write one line   */
            write( iounit, '(i8,i2)' ) natom, 1

!        /*   for free boundary condition   */
         else if ( iboundary .eq. 0 ) then

!           /*   write one line   */
            write( iounit, '(a,i8)' ) 'ATOMS ', istep

!        /*   for free or periodic boundary condition   */
         end if

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   geometry in angstroms   */
            xa = x(i,j) * const_1
            ya = y(i,j) * const_1
            za = z(i,j) * const_1

!           /*   geometry in eV per angstrom   */
            fxj = fx(i,j) * const_3 * dble(nbead)
            fyj = fy(i,j) * const_3 * dble(nbead)
            fzj = fz(i,j) * const_3 * dble(nbead)

!           /*   write one line   */
            write( iounit, '(a4,6f16.8)' ) &
     &         species(i)(1:4), xa, ya, za, fxj, fyj, fzj

!        /*   loop of atoms   */
         end do

!        /*   close input file   */
         close( iounit )

!     /*   loop of beads   */
      end do

      return
      end
