!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     analysis in path integral hybrid Monte Carlo
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine analysis_fourth_hmc ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   nkind, iounit, natom_kind, iounit_avg

      use analysis_variables, only : &
     &   nkindpair, ikindpair, ikindpair_inv, npair_kindpair, &
     &   iprint_bond, iprint_eavg, iprint_trj, iprint_xyz, iprint_xsf, &
     &   iprint_box, iprint_dip, iprint_rdf, iprint_rgy, iprint_dcd

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: itest, ioption, l, j1, j2, k1, k2

!-----------------------------------------------------------------------
!     /*   ioption = 1:  initialize/restart                           */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

!        /*   number of atomic pairs   */

         nkindpair = nkind*(nkind+1)/2

!-----------------------------------------------------------------------
!        /*   ikindpair = label for a pair of kinds                   */
!-----------------------------------------------------------------------

         if ( .not. allocated( ikindpair ) ) &
     &      allocate ( ikindpair(nkind,nkind) )
         if ( .not. allocated( ikindpair_inv ) ) &
     &      allocate ( ikindpair_inv(nkindpair,2) )

         l = 0
         do k1 =  1, nkind
         do k2 = k1, nkind
            l = l + 1
            ikindpair(k1,k2)   = l
            ikindpair(k2,k1)   = l
            ikindpair_inv(l,1) = k1
            ikindpair_inv(l,2) = k2
         end do
         end do

!-----------------------------------------------------------------------
!        /*   npair_kindpair = number of atom pairs                   */
!-----------------------------------------------------------------------

         if ( .not. allocated( npair_kindpair ) ) &
     &      allocate ( npair_kindpair(nkindpair) )

         l = 0
         do k1 =  1, nkind
         do k2 = k1, nkind
            l = l + 1
            j1 = natom_kind(k1)
            j2 = natom_kind(k2)
            if ( k1 .ne. k2 ) npair_kindpair(l) = j1*j2
            if ( k1 .eq. k2 ) npair_kindpair(l) = j1*(j1-1)/2
         end do
         end do

!-----------------------------------------------------------------------
!        /*   step intervals of analysis                              */
!-----------------------------------------------------------------------

         call read_int1 ( iprint_bond, '<iprint_bond>', 13, iounit )
         call read_int1 ( iprint_eavg, '<iprint_eavg>', 13, iounit )
         call read_int1 ( iprint_rgy,  '<iprint_rgy>',  12, iounit )
         call read_int1 ( iprint_rdf,  '<iprint_rdf>',  12, iounit )
         call read_int1 ( iprint_trj,  '<iprint_trj>',  12, iounit )
         call read_int1 ( iprint_dip,  '<iprint_dip>',  12, iounit )
!         call read_int1 ( iprint_mom,  '<iprint_mom>',  12, iounit )
         call read_int1 ( iprint_xyz,  '<iprint_xyz>',  12, iounit )
         call read_int1 ( iprint_xsf,  '<iprint_xsf>',  12, iounit )
         call read_int1 ( iprint_dcd,  '<iprint_dcd>',  12, iounit )
         call read_int1 ( iprint_box,  '<iprint_box>',  12, iounit )

!-----------------------------------------------------------------------
!        /*   check if file called `averages.ini' exists              */
!-----------------------------------------------------------------------

         call testfile ( 'averages.ini', 12, itest )

!-----------------------------------------------------------------------
!        /*   if the file does not exist, initial start.              */
!-----------------------------------------------------------------------

         if ( itest .eq. 1 ) then

            call analysis_bond_fourth_hmc ( 0 )
            call analysis_rgy  ( 0 )
            call analysis_eavg_fourth_hmc ( 0 )
            call analysis_rdf  ( 0 )
            call analysis_trj  ( 0 )
            call analysis_dip  ( 0 )
!            call analysis_mom  ( 0 )
            call analysis_xyz  ( 0 )
            call analysis_xsf  ( 0 )
            call analysis_dcd  ( 0 )
            call analysis_box  ( 0 )
            call analysis_ratio_hmc( 0 )

!#ifdef aenet2
!            call analysis_aenet( 0 )
!#endif

!-----------------------------------------------------------------------
!        /*   if the file exists, restart.                            */
!-----------------------------------------------------------------------

         else

            open ( iounit_avg, file = 'averages.ini')

            call analysis_bond_fourth_hmc ( 1 )
            call analysis_rgy  ( 1 )
            call analysis_eavg_fourth_hmc ( 1 )
            call analysis_rdf  ( 1 )
            call analysis_trj  ( 1 )
            call analysis_dip  ( 1 )
!            call analysis_mom  ( 1 )
            call analysis_xyz  ( 1 )
            call analysis_xsf  ( 1 )
            call analysis_dcd  ( 1 )
            call analysis_box  ( 1 )
            call analysis_ratio_hmc( 1 )

!#ifdef aenet2
!            call analysis_aenet( 1 )
!#endif

            close( iounit_avg )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  start analysis                               */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

            call analysis_bond_fourth_hmc ( 2 )
            call analysis_rgy  ( 2 )
            call analysis_eavg_fourth_hmc ( 2 )
            call analysis_rdf  ( 2 )
            call analysis_trj  ( 2 )
            call analysis_dip  ( 2 )
!            call analysis_mom  ( 2 )
            call analysis_xyz  ( 2 )
            call analysis_xsf  ( 2 )
            call analysis_dcd  ( 2 )
            call analysis_box  ( 2 )
            call analysis_ratio_hmc( 2 )

!#ifdef aenet2
!            call analysis_aenet( 2 )
!#endif

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         open ( iounit_avg, file = 'averages.ini')

            call analysis_bond_fourth_hmc ( 3 )
            call analysis_rgy  ( 3 )
            call analysis_eavg_fourth_hmc ( 3 )
            call analysis_rdf  ( 3 )
            call analysis_trj  ( 3 )
            call analysis_dip  ( 3 )
!            call analysis_mom  ( 3 )
            call analysis_xyz  ( 3 )
            call analysis_xsf  ( 3 )
            call analysis_dcd  ( 3 )
            call analysis_box  ( 3 )
            call analysis_ratio_hmc( 3 )

!#ifdef aenet2
!            call analysis_aenet( 3 )
!#endif

         close( iounit_avg )

      end if

      return
      end





!***********************************************************************
      subroutine analysis_eavg_fourth_hmc ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, temperature, physmass, x, y, z, ux, uy, uz, beta, &
     &   omega_p2, fux_ref, fuy_ref, fuz_ref, iounit_eavg, &
     &   iounit_avg, nbead, natom, istep

      use analysis_variables, only : &
     &   epot, ekinpri, ekinvir, etot, epot_avg, ekinvir_avg, &
     &   ekinpri_avg, etot_avg, eprivir_avg, specific_heat, &
     &   iprint_eavg

      use hmc_variables, only : &
     &   fx_second, fy_second, fz_second, pot_second, pot_fourth, &
     &   potential_fourth

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption, j

      real(8) :: const, ekinpri1, ekinpri2, ekinpri3, gx, gy, gz, &
     &           ekinvir1, ekinvir2, ekinvir3, ekinvir4, ekinvir5, &
     &           eprivir, dummy

      real(8), save :: dstep = 0.d0

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_eavg .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         epot_avg      = 0.d0
         ekinvir_avg   = 0.d0
         ekinpri_avg   = 0.d0
         etot_avg      = 0.d0

         eprivir_avg = 0.d0

         open ( iounit_eavg, file = 'eavg.out' )

            write(iounit_eavg,'(a)') &
     &      '========' // &
     &      '================================' // &
     &      '================================' // &
     &      '================================' // &
     &      '================================' // &
     &      '================'
            write(iounit_eavg,'(a)') &
     &      '    step' // &
     &      '            epot         ekinvir' // &
     &      '         ekinpri            etot' // &
     &      '        epot_avg     ekinvir_avg' // &
     &      '     ekinpri_avg        etot_avg' // &
     &      '   specific_heat'
            write(iounit_eavg,'(a)') &
     &      '--------' // &
     &      '--------------------------------' // &
     &      '--------------------------------' // &
     &      '--------------------------------' // &
     &      '--------------------------------' // &
     &      '----------------'

            close( iounit_eavg )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         read ( iounit_avg, * ) dstep
         read ( iounit_avg, * ) epot_avg
         read ( iounit_avg, * ) ekinvir_avg
         read ( iounit_avg, * ) ekinpri_avg
         read ( iounit_avg, * ) etot_avg
         read ( iounit_avg, * ) eprivir_avg

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         dstep = dstep + 1.d0

!-----------------------------------------------------------------------
!        /*   potential energy                                        */
!-----------------------------------------------------------------------

         epot = 0.d0

         do j = 1, nbead
            epot = epot + pot_second(j) + 2.d0*pot_fourth(j)
         end do

         epot = epot/dble(nbead)

!-----------------------------------------------------------------------
!        /*   kinetic energy by primitive estimator                   */
!-----------------------------------------------------------------------

!        /*   first term   */

         ekinpri1 = 1.5d0*dble(natom)*dble(nbead)*boltz*temperature

!        /*   second term   */

         ekinpri2 = 0.d0

         do j = 1, nbead

            do i = 1, natom

               if ( j .eq. nbead ) then

                  ekinpri2 = ekinpri2 &
     &            + physmass(i)*(x(i,nbead)-x(i,1))*(x(i,nbead)-x(i,1)) &
     &            + physmass(i)*(y(i,nbead)-y(i,1))*(y(i,nbead)-y(i,1)) &
     &            + physmass(i)*(z(i,nbead)-z(i,1))*(z(i,nbead)-z(i,1))

               else

                  ekinpri2 = ekinpri2 &
     &            + physmass(i)*(x(i,j)-x(i,j+1))*(x(i,j)-x(i,j+1)) &
     &            + physmass(i)*(y(i,j)-y(i,j+1))*(y(i,j)-y(i,j+1)) &
     &            + physmass(i)*(z(i,j)-z(i,j+1))*(z(i,j)-z(i,j+1))

               end if

            end do

         end do

         ekinpri2 = - 0.5d0*omega_p2*ekinpri2

!        /*   third term   */

         ekinpri3 = potential_fourth

!        /*   the sum of kinetic energy   */

         ekinpri = ekinpri1 + ekinpri2 + ekinpri3

!-----------------------------------------------------------------------
!        /*   kinetic energy by virial estimator                      */
!-----------------------------------------------------------------------

!        /*   first term   */

         ekinvir1 = 1.5d0*dble(natom)*boltz*temperature

!        /*   second term   */

         ekinvir2 = 0.d0

         do j = 1, nbead

            do i = 1, natom
               ekinvir2 = ekinvir2 + (x(i,j)-ux(i,1))*fx_second(i,j) &
     &                             + (y(i,j)-uy(i,1))*fy_second(i,j) &
     &                             + (z(i,j)-uz(i,1))*fz_second(i,j)
            end do

         end do

         ekinvir2 = - 0.5d0*ekinvir2

         ekinvir3 = potential_fourth

         const = -1.5d0*dble(natom*(nbead-1))/dble(nbead)

         ekinvir4 = potential_fourth * const

         ekinvir5 = 0.d0

         call getforce_ref_xyz

         do j = 1, nbead

            const = - beta/dble(2*nbead)*pot_fourth(j)

            do i = 1, natom

               gx = fux_ref(i,j) + fx_second(i,j)
               gy = fuy_ref(i,j) + fy_second(i,j)
               gz = fuz_ref(i,j) + fz_second(i,j)

               ekinvir5 = ekinvir5 + (x(i,j)-ux(i,1))*gx*const &
     &                             + (y(i,j)-uy(i,1))*gy*const &
     &                             + (z(i,j)-uz(i,1))*gz*const

            end do

         end do

!-----------------------------------------------------------------------
!        /*   approximate estimation of kinetic energy                */
!-----------------------------------------------------------------------

!         /*  this does not work for large systems   */
!         ekinvir = ekinvir1 + ekinvir2 + ekinvir3 + ekinvir4 + ekinvir5
         dummy = ekinvir1 + ekinvir2 + ekinvir3 + ekinvir4 + ekinvir5

!        /*  this has only second order accuracy   */
         ekinvir = ekinvir1 + ekinvir2 + ekinvir3

!-----------------------------------------------------------------------
!        /*   total energy                                            */
!-----------------------------------------------------------------------

         etot = ekinpri + epot

!-----------------------------------------------------------------------
!        /*   accumulative averages                                   */
!-----------------------------------------------------------------------

         epot_avg    = epot   /dstep + epot_avg   *(dstep-1.d0)/dstep
         ekinvir_avg = ekinvir/dstep + ekinvir_avg*(dstep-1.d0)/dstep
         ekinpri_avg = ekinpri/dstep + ekinpri_avg*(dstep-1.d0)/dstep
         etot_avg    = etot   /dstep + etot_avg   *(dstep-1.d0)/dstep

!-----------------------------------------------------------------------
!        /*   specific heat                                           */
!-----------------------------------------------------------------------

         eprivir = ( ekinpri + epot ) * ( ekinvir + epot )

         eprivir_avg = eprivir / dstep &
     &                  + eprivir_avg * (dstep-1.d0)/dstep

         eprivir = ( ekinpri_avg + epot_avg ) &
     &           * ( ekinvir_avg + epot_avg )

         specific_heat &
     &       = beta * beta * ( eprivir_avg -  eprivir ) &
     &       + 1.5d0 * dble(natom)

!         specific_heat
!     &       = beta * beta * ( eprivir_avg - etot_avg*etot_avg  )
!     &       + 1.5d0 * dble(natom)

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_eavg) .eq. 0 ) then

            open ( iounit_eavg, file = 'eavg.out', access = 'append' )

            write( iounit_eavg, '(i8,8f16.8,f16.4)' ) &
     &         istep, &
     &         epot, ekinvir, ekinpri, etot, &
     &         epot_avg, ekinvir_avg, ekinpri_avg, etot_avg, &
     &         specific_heat

            close( iounit_eavg )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         write( iounit_avg, '(e24.16)' ) dstep
         write( iounit_avg, '(e24.16)' ) epot_avg
         write( iounit_avg, '(e24.16)' ) ekinvir_avg
         write( iounit_avg, '(e24.16)' ) ekinpri_avg
         write( iounit_avg, '(e24.16)' ) etot_avg
         write( iounit_avg, '(e24.16)' ) eprivir_avg

      end if

      return
      end





!***********************************************************************
      subroutine analysis_bond_fourth_hmc ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, beta, hbar, &
     &   iounit_bond, iounit_avg, physmass, &
     &   natom, nbead, istep, nkind, ikind

      use analysis_variables, only : &
     &   rbead, rcent, rkubo, rbead_avg, rcent_avg, rkubo_avg, &
     &   iprint_bond, npair_kindpair, nkindpair, ikindpair

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ioption, i, j, k, m

      real(8) :: r1, r2, rx, ry, rz, ax, ay, az, ar, s1, const

      real(8), save :: dstep = 0.d0

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_bond .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         if ( .not. allocated( rbead ) ) &
     &      allocate ( rbead(nkindpair)  )
         if ( .not. allocated( rcent ) ) &
     &      allocate ( rcent(nkindpair)  )
         if ( .not. allocated( rkubo ) ) &
     &      allocate ( rkubo(nkindpair)  )
         if ( .not. allocated( rbead_avg ) ) &
     &      allocate ( rbead_avg(nkindpair)  )
         if ( .not. allocated( rcent_avg ) ) &
     &      allocate ( rcent_avg(nkindpair)  )
         if ( .not. allocated( rkubo_avg ) ) &
     &      allocate ( rkubo_avg(nkindpair)  )

         rbead_avg(:)  = 0.d0
         rcent_avg(:)  = 0.d0
         rkubo_avg(:)  = 0.d0

         open ( iounit_bond, file = 'bond.out' )

         write(iounit_bond,'(a)') &
     &   '====================================================' // &
     &   '===================================='
         write(iounit_bond,'(a)') &
     &   '    step  1   2        rbead         2nd         4th' // &
     &   '   rbead_avg     2nd_avg     4th_avg'
         write(iounit_bond,'(a)') &
     &   '----------------------------------------------------' // &
     &   '------------------------------------'

         close( iounit_bond )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         if ( .not. allocated( rbead ) ) &
     &      allocate ( rbead(nkindpair)  )
         if ( .not. allocated( rcent ) ) &
     &      allocate ( rcent(nkindpair)  )
         if ( .not. allocated( rkubo ) ) &
     &      allocate ( rkubo(nkindpair)  )
         if ( .not. allocated( rbead_avg ) ) &
     &      allocate ( rbead_avg(nkindpair)  )
         if ( .not. allocated( rcent_avg ) ) &
     &      allocate ( rcent_avg(nkindpair)  )
         if ( .not. allocated( rkubo_avg ) ) &
     &      allocate ( rkubo_avg(nkindpair)  )

         read ( iounit_avg, * ) dstep
         read ( iounit_avg, * ) rbead_avg(:)
         read ( iounit_avg, * ) rcent_avg(:)
         read ( iounit_avg, * ) rkubo_avg(:)

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         dstep = dstep + 1.d0

!-----------------------------------------------------------------------
!        /*   calculate rbead:  bead average                          */
!-----------------------------------------------------------------------

         const = ( beta * hbar )**2 / dble( 12 * nbead )

         rbead(:)  = 0.d0
         rcent(:)  = 0.d0
         rkubo(:)  = 0.d0

         do k = 1, nbead

            do i = 1, natom-1
            do j = i+1, natom

               m = ikindpair ( ikind(i), ikind(j) )

               rx   = x(i,k) - x(j,k)
               ry   = y(i,k) - y(j,k)
               rz   = z(i,k) - z(j,k)

               call pbc_atom ( rx, ry, rz )

               r2   = rx*rx + ry*ry + rz*rz
               r1   = sqrt(r2)

               ax   = fx(i,k)/physmass(i) - fx(j,k)/physmass(j)
               ay   = fy(i,k)/physmass(i) - fy(j,k)/physmass(j)
               az   = fz(i,k)/physmass(i) - fz(j,k)/physmass(j)

               ar   = ax*rx + ay*ry + az*rz

               s1   = - const*ar / r1

               rbead(m)  = rbead(m)  + r1 + s1
               rcent(m)  = rcent(m)  + r1
               rkubo(m)  = rkubo(m)  + s1

            end do
            end do

         end do

         do m = 1, nkindpair

            if ( npair_kindpair(m) .eq. 0 ) cycle

            rcent(m)  = rcent(m)  /dble(nbead) /npair_kindpair(m)
            rkubo(m)  = rkubo(m)  /dble(nbead) /npair_kindpair(m)
            rbead(m)  = rbead(m)  /dble(nbead) /npair_kindpair(m)

         end do

!-----------------------------------------------------------------------
!        /*   averages                                                */
!-----------------------------------------------------------------------

         rcent_avg(:)  = rcent(:) /dstep &
     &                 + rcent_avg(:)*(dstep-1.d0)/dstep

         rkubo_avg(:)  = rkubo(:) /dstep &
     &                 + rkubo_avg(:)*(dstep-1.d0)/dstep

         rbead_avg(:)  = rbead(:) /dstep &
     &                 + rbead_avg(:)*(dstep-1.d0)/dstep

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_bond) .eq. 0 ) then

            open ( iounit_bond, file = 'bond.out', access = 'append' )

            m = 0

            do i = 1, nkind
            do j = i, nkind

               m = m + 1

               if ( npair_kindpair(m) .eq. 0 ) cycle

               write(iounit_bond,'(i8,1x,i3,1x,i3,6f12.5)') &
     &            istep, i, j, &
     &            rbead(m), rcent(m), rkubo(m), &
     &            rbead_avg(m), rcent_avg(m), rkubo_avg(m)

            end do
            end do

            close( iounit_bond )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         write( iounit_avg, '(e24.16)' ) dstep
         write( iounit_avg, '(e24.16)' ) rbead_avg(:)
         write( iounit_avg, '(e24.16)' ) rcent_avg(:)
         write( iounit_avg, '(e24.16)' ) rkubo_avg(:)

      end if

      return
      end

