!///////////////////////////////////////////////////////////////////////
!
!      Author:          B. Thomsen, M. Shiga
!      Last updated:    Mar 2, 2022 by M. Shiga
!      Description:     dcd output
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine analysis_dcd_MPI( ioption )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   au_length, box, istep, natom, nbead, &
     &   iboundary, istep, pi, ipotential, myrank, iprint_charge

      use analysis_variables, only : &
     &   iprint_dcd

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: ioption

!     /*   real numbers   */
      real(8) :: ax, ay, az, bx, by, bz, cx, cy, cz
      real(8) :: a, b, c, alpha, beta, gamma
      real(8) :: const_1

!     /*   flags   */
      integer, save :: iset = 0
      integer, save :: chargedat = 0

!-----------------------------------------------------------------------
!     /*   print condition: every iprint_dcd steps                    */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 ) return
      if ( iprint_dcd .le. 0 ) return
      if ( ioption .eq. 3 ) return
      if ( mod(istep,iprint_dcd) .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   header                                                     */
!-----------------------------------------------------------------------

!     /*   conversion factor   */
      const_1 = au_length * 1.d+10

!     /*   initialization   */
      if ( iset .eq. 0 ) then

!        /*   write dcd headers   */
         call output_dcd_header_MPI &
     &      ( INT(nbead*natom, 4), 'trj.dcd' )

         if ( (iboundary .eq. 1) .or. (iboundary .eq. 2) ) then

            call output_dcd_header_MPI &
     &         ( INT(nbead, 4), 'box.dcd' )

         end if

         if ( ipotential(1:5) .eq. 'DUAL ' ) then

            call output_dcd_header_MPI &
     &         ( INT(nbead*natom, 4), 'force_low.dcd' )

            call output_dcd_header_MPI &
     &         ( INT(nbead*natom, 4), 'force_high.dcd' )

            call output_dcd_header_MPI &
     &         ( INT(nbead*natom, 4), 'trial_trj.dcd' )

            call output_dcd_header_MPI &
     &         ( INT(nbead, 4), 'pot_high.dcd')

           call output_dcd_header_MPI &
     &         ( INT(nbead, 4), 'pot_low.dcd')
      
            if ( iprint_charge .eq. 1 ) then
                  call output_dcd_header_MPI &
     &                  ( INT(nbead, 4), 'charges.dcd' ) 
                  chargedat = 1
            end if

         else

            call output_dcd_header_MPI &
     &        ( INT(nbead*natom, 4), 'vel.dcd' )

            call output_dcd_header_MPI &
     &         ( INT(nbead*natom, 4), 'force.dcd' )

           if ( iprint_charge .eq. 1 ) then
                  call output_dcd_header_MPI &
     &                  ( INT(nbead, 4), 'charges.dcd' )
                  chargedat = 1
            end if 

         end if

         call output_dcd_header_MPI &
     &      ( INT(nbead, 4), 'pot.dcd' )

         call output_dcd_header_MPI &
     &      ( INT(nbead, 4), 'dip.dcd' )
     
!        /*   write template file for read in purposes   */
         call write_dcd_template_xyz_MPI( 1 )

!        /*   set completed   */
         iset = 1

!     /*   initialization   */
      end if

!     /*   return by option   */
      if( ioption .lt. 2 ) return

!-----------------------------------------------------------------------
!     /*   print dcd files                                            */
!-----------------------------------------------------------------------

!     /*   for periodic boundary condition   */
      if ( (iboundary .eq. 1) .or. (iboundary .eq. 2) ) then

!        /*   write box and virial info to file   */
         call write_dcd_box_MPI()

!        /*   lattice vectors   */
         ax = box(1,1) * const_1
         ay = box(2,1) * const_1
         az = box(3,1) * const_1
         bx = box(1,2) * const_1
         by = box(2,2) * const_1
         bz = box(3,2) * const_1
         cx = box(1,3) * const_1
         cy = box(2,3) * const_1
         cz = box(3,3) * const_1

         a = sqrt(ax*ax+ay*ay+az*az)
         b = sqrt(bx*bx+by*by+bz*bz)
         c = sqrt(cx*cx+cy*cy+cz*cz)

         alpha = acos((bx*cx+by*cy+bz*cz)/(b*c))
         beta = acos((ax*cx+ay*cy+az*cz)/(a*c))
         gamma  = acos((bx*ax+by*ay+bz*az)/(b*a))

         alpha = alpha*180.0/pi
         beta  = beta *180.0/pi
         gamma = gamma*180.0/pi

         call output_dcd_pbcinfo_MPI &
     &      ( 'trj.dcd', a, gamma, b, beta, alpha, c )

        if ( ipotential(1:5) .eq. 'DUAL ' ) then

           call output_dcd_pbcinfo_MPI &
     &        ( 'force_low.dcd', a, gamma, b, beta, alpha, c )

           call output_dcd_pbcinfo_MPI &
     &        ( 'force_high.dcd', a, gamma, b, beta, alpha, c )
     
           call output_dcd_pbcinfo_MPI &
     &         ( 'trial_trj.dcd', a, gamma, b, beta, alpha, c )

        else

           call output_dcd_pbcinfo_MPI &
     &        ( 'vel.dcd', a, gamma, b, beta, alpha, c )

           call output_dcd_pbcinfo_MPI &
     &        ( 'force.dcd', a, gamma, b, beta, alpha, c )

        end if

!     /*   for periodic boundary condition   */
      end if

!     /*   output the unwrapped trajectory for pbc   */
      if ( (iboundary .eq. 1) .or. (iboundary .eq. 2) ) then

         call output_dcd_wrapped_coord_MPI &
     &      ( 'trj.dcd', 0 )

      else

            call output_dcd_data_MPI &
     &            ( 'trj.dcd', 0, 1 )
      
      end if

!     /*    output the forces in dcd format   */
      if ( ipotential(1:5) .eq. 'DUAL ' ) then

         call output_dcd_data_MPI &
     &      ( 'force_low.dcd', 0, 4 )

         call output_dcd_data_MPI &
     &      ( 'force_high.dcd', 0, 5 )

         call output_dcd_data_MPI &
     &      ( 'trial_trj.dcd', 0, 6 )

         call write_dcd_energies_dual_MPI()
      else
!     /*    output the velocities in dcd format   */
         call output_dcd_data_MPI &
     &   ( 'vel.dcd', 0, 2 )

         call output_dcd_data_MPI &
     &      ( 'force.dcd', 0, 3 )

      end if

      call write_dcd_energies_MPI()
      call write_dcd_dipoles_MPI()

      if (chargedat .eq. 1 ) then
            call write_dcd_charges_MPI()
      end if 

      return
      end





!***********************************************************************
      subroutine analysis_dcd_beadwise_MPI ( ioption, beadwisebox )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   au_length, box, istep, natom, nbead, myrank, &
     &   iboundary, istep, pi, ipotential, box_bead

      use analysis_variables, only : &
     &   iprint_dcd

      use dual_variables, only : &
     &   idual_hi

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: j, ioption, beadwisebox

!     /*   real numbers   */
      real(8) :: ax, ay, az, bx, by, bz, cx, cy, cz
      real(8) :: a, b, c, alpha, beta, gamma, const_1

!     /*   flag   */
      integer, save :: iset = 0
      integer, save :: chargedat = 0

!     /*   bead number   */
      character(len=3) :: char_num

!     /*   filename   */
      character(len=11) :: char_file_dcd
      character(len=11) :: char_file_vel
      character(len=11) :: char_file_box
      character(len=13) :: char_file_force
      character(len=17) :: char_file_force_low
      character(len=18) :: char_file_force_high

!-----------------------------------------------------------------------
!     /*   print condition: every iprint_dcd steps                    */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 ) return
      if ( iprint_dcd .le. 0 ) return
      if ( ioption .eq. 3 ) return
      if ( mod(istep,iprint_dcd) .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   header                                                     */
!-----------------------------------------------------------------------

!     /*   conversion factor   */
      const_1 = au_length * 1.d+10

!     /*   initialization   */
      if ( ( iset .eq. 0 ) ) then

!        /* output of dipoles, energies, charges and box information */
         if ( (beadwisebox .eq. 0) .and.  &
     &        ( (iboundary .eq. 1) .or. (iboundary .eq. 2) ) ) then

            call output_dcd_header_MPI &
     &         ( INT(nbead, 4), 'box.dcd' )

         end if

         call output_dcd_header_MPI &
     &      ( INT(nbead, 4), 'dip.dcd' )

         call output_dcd_header_MPI &
     &      ( INT(nbead, 4), 'pot.dcd' )

!        /*   loop of beads   */
         do j = 1, nbead

!           /*   bead number   */
            call int3_to_char( j, char_num )

            if ( (beadwisebox .eq. 1) .and.  &
     &           ( (iboundary .eq. 1) .or. (iboundary .eq. 2) ) ) then

                  char_file_box = 'box.' // char_num // '.dcd'
                  call output_dcd_header_MPI &
     &                  ( INT(nbead, 4), char_file_box )

            end if

!           /*   file name   */
            char_file_dcd = 'trj.' // char_num // '.dcd'

!           /*   write dcd headers   */
            call output_dcd_header_MPI( INT(natom, 4), char_file_dcd )

            char_file_vel = 'vel.' // char_num // '.dcd'

            call output_dcd_header_MPI &
     &         ( INT(natom, 4), char_file_vel )

            if ( ipotential(1:5) .eq. 'DUAL ' ) then

               char_file_force_low &
     &            = 'force_low.' // char_num // '.dcd'

               char_file_force_high &
     &            = 'force_high.' // char_num // '.dcd'

               call output_dcd_header_MPI &
     &            ( INT(natom, 4), char_file_force_low )

               call output_dcd_header_MPI &
     &            ( INT(natom, 4), char_file_force_high )

            else

               char_file_force = 'force.' // char_num // '.dcd'

               call output_dcd_header_MPI &
     &            ( INT(natom, 4), char_file_force )

            end if

!        /*   loop of beads   */
         end do

         if ( ipotential(1:5) .eq. 'DUAL ' ) then

            if ( idual_hi(1:8) .eq. 'CP2KLIB ' ) then
                  call output_dcd_header_MPI &
     &                  ( INT(nbead, 4), 'charges.dcd' ) 
                  chargedat = 1
            end if

         else if ( ipotential(1:8) .eq. 'CP2KLIB ' ) then

            call output_dcd_header_MPI &
     &                  ( INT(nbead, 4), 'charges.dcd' )
            chargedat = 1

         end if 

!        /*   write template file for read in purposes   */
         call write_dcd_template_xyz_MPI( 0 )

!        /*   set completed   */
         iset = 1

!     /*   initialization   */
      end if

      if ( ioption .lt. 2 ) return
!-----------------------------------------------------------------------
!     /*   print dcd files                                            */
!-----------------------------------------------------------------------

!        /*   for periodic boundary condition   */
      if ( (iboundary .eq. 1) .or. (iboundary .eq. 2) ) then
         if ( beadwisebox .eq. 1 ) then
            call write_box_bead_dcd_MPI()
         else
            call write_dcd_box_MPI()
         end if
      end if
      
!     /*   write energies, charges and dipoles to dcd files */
      call write_dcd_energies_MPI()
      call write_dcd_dipoles_MPI()


!     /*   loop over each bead   */
      do j = 1, nbead

!        /*   bead number   */
         call int3_to_char( j, char_num )

!        /*   file name   */
         char_file_dcd = 'trj.' // char_num // '.dcd'

         char_file_vel = 'vel.' // char_num // '.dcd'

         char_file_force = 'force.' // char_num // '.dcd'
         char_file_force_low = 'force_low.' // char_num // '.dcd'
         char_file_force_high = 'force_high.' // char_num // '.dcd'

!        /*   for periodic boundary condition   */
         if ( (iboundary .eq. 1) .or. (iboundary .eq. 2) ) then

            if (beadwisebox .eq. 0) then
!                 /*   lattice vectors   */
                  ax = box(1,1) * const_1
                  ay = box(2,1) * const_1
                  az = box(3,1) * const_1
                  bx = box(1,2) * const_1
                  by = box(2,2) * const_1
                  bz = box(3,2) * const_1
                  cx = box(1,3) * const_1
                  cy = box(2,3) * const_1
                  cz = box(3,3) * const_1
            else
!                 /*   lattice vectors   */
                  ax = box_bead(1,1,j) * const_1
                  ay = box_bead(2,1,j) * const_1
                  az = box_bead(3,1,j) * const_1
                  bx = box_bead(1,2,j) * const_1
                  by = box_bead(2,2,j) * const_1
                  bz = box_bead(3,2,j) * const_1
                  cx = box_bead(1,3,j) * const_1
                  cy = box_bead(2,3,j) * const_1
                  cz = box_bead(3,3,j) * const_1
            end if

            a = sqrt(ax*ax+ay*ay+az*az)
            b = sqrt(bx*bx+by*by+bz*bz)
            c = sqrt(cx*cx+cy*cy+cz*cz)

            alpha = acos((bx*cx+by*cy+bz*cz)/(b*c))
            beta  = acos((bx*ax+by*ay+bz*az)/(b*a))
            gamma = acos((ax*cx+ay*cy+az*cz)/(a*c))

            alpha = alpha*180.0/pi
            beta  = beta *180.0/pi
            gamma = gamma*180.0/pi

            call output_dcd_pbcinfo_MPI &
     &         ( char_file_dcd, a, gamma, b, beta, alpha, c )

            call output_dcd_pbcinfo_MPI &
     &         ( char_file_vel, a, gamma, b, beta, alpha, c )

            if ( ipotential(1:5) .eq. 'DUAL ' ) then

               call output_dcd_pbcinfo_MPI &
     &            ( char_file_force_low, a, gamma, b, beta, alpha, &
     &              c )

               call output_dcd_pbcinfo_MPI &
     &            ( char_file_force_high, a, gamma, b, beta, alpha, &
     &              c )

           else

               call output_dcd_pbcinfo_MPI &
     &            ( char_file_force, a, gamma, b, beta, alpha, c )

           end if

!        /*   for periodic boundary condition   */
         end if

!        /*   output the coordinates in dcd format   */
         call output_dcd_data_MPI &
     &      ( char_file_dcd, j, 1 )

!        /*   output the unwrapped trajectory for pbc   */
         if ( (iboundary .eq. 1) .or. (iboundary .eq. 2) ) then

            call output_dcd_wrapped_coord_MPI &
     &         ( char_file_dcd, j )
         else
            call output_dcd_data_MPI( char_file_dcd, j, 1 )

         end if
!        /*   output the velocities in dcd format   */
         call output_dcd_data_MPI &
     &      ( char_file_vel, j, 2 )

!        /*   output the forces in dcd format   */
         if ( ipotential(1:5) .eq. 'DUAL ' ) then

            call output_dcd_data_MPI &
     &         ( char_file_force_low, j, 4 )

            call output_dcd_data_MPI &
     &         ( char_file_force_high, j, 5 )

            call write_dcd_energies_dual_MPI()

         else

            call output_dcd_data_MPI &
     &         ( char_file_force, j, 3 )

         end if

!     /*   loop of beads   */
      end do

      return
      end


!***********************************************************************
      subroutine write_box_bead_dcd_MPI( )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   box_bead, vir_bead, iounit, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: j, output

!     /*   real numbers   */
      real(8) :: box(3,3), vir(3,3)

!     /*   bead number   */
      character(len=3) :: char_num

!     /*   filenames   */
      character(len=11) :: char_file_box

!-----------------------------------------------------------------------
!     /*   Write Box and Virial information to dcd files              */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do j = 1, nbead

!        /*   bead number   */
         call int3_to_char( j, char_num )

         char_file_box = 'box.' // char_num // '.dcd'

         open ( iounit, file = char_file_box, &
     &          form = 'unformatted', access = 'stream', &
     &          status = 'old', position = 'append' )

         output = 8 * 9 * 2

         write( iounit ) output

         box(:,:) = box_bead(:,:,j)

         write( iounit ) box( 1, 1 )
         write( iounit ) box( 1, 2 )
         write( iounit ) box( 1, 3 )
         write( iounit ) box( 2, 1 )
         write( iounit ) box( 2, 2 )
         write( iounit ) box( 2, 3 )
         write( iounit ) box( 3, 1 )
         write( iounit ) box( 3, 2 )
         write( iounit ) box( 3, 3 )

         vir(:,:) = vir_bead(:,:,j)

         write( iounit ) vir( 1, 1 )
         write( iounit ) vir( 1, 2 )
         write( iounit ) vir( 1, 3 )
         write( iounit ) vir( 2, 1 )
         write( iounit ) vir( 2, 2 )
         write( iounit ) vir( 2, 3 )
         write( iounit ) vir( 3, 1 )
         write( iounit ) vir( 3, 2 )
         write( iounit ) vir( 3, 3 )

         close( iounit )

      end do

      return
      end


!***********************************************************************
      subroutine output_dcd_pbcinfo_MPI &
     &   ( filename, a, gamma, b, beta, alpha, c)
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iounit_dcd

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer(4) :: output

!     /*   real numbers   */
      real(8) :: a, b, c, alpha, beta, gamma

!     /*   character   */
      character(*) :: filename

!-----------------------------------------------------------------------
!     /*   start writing periodic box for dcd file                     */
!-----------------------------------------------------------------------

      open( iounit_dcd, file = filename, form = 'unformatted', &
     &      access = 'stream', status = 'old', position = 'append' )

!     /* size of the data in the pbc block */
      output = 48

      write( iounit_dcd ) output
      write( iounit_dcd ) a, gamma, b, beta, alpha, c
      write( iounit_dcd ) output

      close( iounit_dcd )

      return
      end





!***********************************************************************
      subroutine output_dcd_data_MPI( filename, beadidx, type )
!***********************************************************************

      use common_variables, only : &
     &   x, y, z, au_length, natom, nbead, iounit_dcd, &
     &   vx, vy, vz, au_time, fx, fy, fz

      use dual_variables, only : &
     & fx_high, fy_high, fz_high, fx_low, fy_low, fz_low, &
     & x_trial, y_trial, z_trial

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: beadidx, type, i, j
      integer(4) :: output

!     /*   real numbers   */
      real(8) :: const_1
      real(8), pointer, dimension (:,:) :: xdat, ydat, zdat

!     /*   character   */
      character(*) :: filename

!-----------------------------------------------------------------------
!     /*   Setup pointes and conversion factor for printing           */
!-----------------------------------------------------------------------

!     /*   pointers out of the if statement to avoid warning   */

      const_1 = au_length * 1.d+10
      xdat=>x
      ydat=>y
      zdat=>z

      if ( type == 1 ) then
         const_1 = au_length * 1.d+10
         xdat=>x
         ydat=>y
         zdat=>z
      else if ( type == 2 ) then
         const_1 = 1.d-2 * au_length / au_time
         xdat=>vx
         ydat=>vy
         zdat=>vz
      else if ( type == 3 ) then
         const_1 = dble(nbead)
         xdat=>fx
         ydat=>fy
         zdat=>fz
      else if ( type == 4 ) then
         const_1 = dble(nbead)
         xdat=>fx_low
         ydat=>fy_low
         zdat=>fz_low
      else if ( type == 5 ) then
         const_1 = dble(nbead)
         xdat=>fx_high
         ydat=>fy_high
         zdat=>fz_high
      else if ( type == 6 ) then
         const_1 = au_length * 1.d+10
         xdat=>x_trial
         ydat=>y_trial
         zdat=>z_trial
      else
         call error_handling_MPI( 1, 'subroutine output_dcd_data', 26 )
      end if

!-----------------------------------------------------------------------
!     /*   Write data to file                                         */
!-----------------------------------------------------------------------

      open( iounit_dcd, file = filename, form = 'unformatted', &
     &      access = 'stream', status = 'old', position = 'append' )

!     /*   If beadidx is set to 0, output all beads to one file   */
      if ( beadidx == 0 ) then

         output = INT(nbead*natom*4, 4)

         write( iounit_dcd ) output

!        /*   loop of beads   */
         do j = 1, nbead

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   write position   */
            write( iounit_dcd ) sngl( xdat(i,j) * const_1 )

!        /*   loop of atoms   */
         end do

!        /*   loop of beads   */
         end do

         write( iounit_dcd ) output
         write( iounit_dcd ) output

!        /*   loop of beads   */
         do j = 1, nbead

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   write position  */
            write( iounit_dcd ) sngl( ydat(i,j) * const_1 )

!        /*   loop of atoms   */
         end do

!        /*   loop of beads   */
         end do

         write( iounit_dcd ) output
         write( iounit_dcd ) output

!        /*   loop of beads   */
         do j = 1, nbead

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   write position   */
            write( iounit_dcd ) sngl( zdat(i,j) * const_1 )

!        /*   loop of atoms   */
         end do

!        /*   loop of beads   */
         end do

         write( iounit_dcd ) output

      else

         output = INT(natom*4, 4)

         write( iounit_dcd ) output

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   write position  */
            write( iounit_dcd ) sngl( xdat(i,beadidx) * const_1 )

!        /*   loop of atoms   */
         end do

         write( iounit_dcd ) output
         write( iounit_dcd ) output

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   write position  */
            write( iounit_dcd ) sngl( ydat(i,beadidx) * const_1 )

!        /*   loop of atoms   */
         end do

         write( iounit_dcd ) output
         write( iounit_dcd ) output

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   write position  */
            write( iounit_dcd ) sngl( zdat(i,beadidx) * const_1 )

!        /*   loop of atoms   */
         end do

         write( iounit_dcd ) output

      end if

      close ( iounit_dcd )

      return
      end





!***********************************************************************
      subroutine output_dcd_wrapped_coord_MPI( filename, beadidx )
!***********************************************************************

      use common_variables, only : &
     &   x, y, z, au_length, natom, nbead, iounit_dcd, mbox

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: beadidx, m1, m2, m3, i, j
      integer(4) :: output

!     /*   real numbers   */
      real(8) :: const_1, ax, ay, az

!     /*   character   */
      character(*) :: filename

      const_1 = au_length * 1.d+10

      open( iounit_dcd, file = filename, form = 'unformatted', &
     &      access = 'stream', status = 'old', position = 'append' )

      if ( beadidx == 0 ) then

         output = INT(nbead*natom*4, 4)

         write( iounit_dcd ) output

!        /*   loop of beads   */
         do j = 1, nbead

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   geometry in angstroms   */
            ax = x(i,j)

            m1 = mbox(1,i,j)
            m2 = mbox(2,i,j)
            m3 = mbox(3,i,j)

            call pbc_unfold_single_MPI( ax, 1, m1, m2, m3 )

!           /*   write position  */
            write( iounit_dcd ) sngl( ax * const_1 )

!        /*   loop of atoms   */
         end do

!        /*   loop of beads   */
         end do

         write( iounit_dcd ) output
         write( iounit_dcd ) output

!        /*   loop of beads   */
         do j = 1, nbead

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   geometry in angstroms   */
            ay = y(i,j)

            m1 = mbox(1,i,j)
            m2 = mbox(2,i,j)
            m3 = mbox(3,i,j)

            call pbc_unfold_single_MPI( ay, 2, m1, m2, m3 )

!           /*   write position  */
            write( iounit_dcd ) sngl( ay * const_1 )

!        /*   loop of atoms   */
         end do

!        /*   loop of beads   */
         end do

         write( iounit_dcd ) output
         write( iounit_dcd ) output

!        /*   loop of beads   */
         do j = 1, nbead

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   geometry in angstroms   */
            az = z(i,j)

            m1 = mbox(1,i,j)
            m2 = mbox(2,i,j)
            m3 = mbox(3,i,j)

            call pbc_unfold_single_MPI( az, 3, m1, m2, m3 )

!           /*   write position  */
            write( iounit_dcd ) sngl( az * const_1 )

!        /*   loop of atoms   */
         end do

!        /*   loop of beads   */
         end do

         write( iounit_dcd ) output

      else

         output = INT(natom*4, 4) 

         write( iounit_dcd ) output

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   geometry in angstroms   */
            ax = x(i,beadidx)

            m1 = mbox(1,i,beadidx)
            m2 = mbox(2,i,beadidx)
            m3 = mbox(3,i,beadidx)

            call pbc_unfold_single_MPI( ax, 1, m1, m2, m3 )

!           /*   write position  */
            write( iounit_dcd ) sngl( ax * const_1 )

!        /*   loop of atoms   */
         end do

         write( iounit_dcd ) output
         write( iounit_dcd ) output

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   geometry in angstroms   */
            ay = y(i,beadidx)

            m1 = mbox(1,i,beadidx)
            m2 = mbox(2,i,beadidx)
            m3 = mbox(3,i,beadidx)

            call pbc_unfold_single_MPI( ay, 2, m1, m2, m3 )

!           /*   write position  */
            write( iounit_dcd ) sngl( ay * const_1 )

!        /*   loop of atoms   */
         end do

         write( iounit_dcd ) output
         write( iounit_dcd ) output

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   geometry in angstroms   */
            az = z(i,beadidx)

            m1 = mbox(1,i,beadidx)
            m2 = mbox(2,i,beadidx)
            m3 = mbox(3,i,beadidx)

            call pbc_unfold_single_MPI( az, 3, m1, m2, m3 )

!           /*   write position   */
            write( iounit_dcd ) sngl( az * const_1 )

!        /*   loop of atoms   */
         end do

         write( iounit_dcd ) output

      end if

      close ( iounit_dcd )

      return
      end





!***********************************************************************
      subroutine output_dcd_header_MPI( numatoms, filename )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   istep, iounit, iboundary, dt, char_date

      use analysis_variables, only : &
     &   iprint_dcd

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: itest
      integer(4) :: output, zero, numatoms

!     /*   real number   */
      real(4) :: tmp

!     /*   characters   */
      character(len=80), save :: title = 'PIMD - TRAJECTORY\0'
      character(*) :: filename

!-----------------------------------------------------------------------
!     /*   start writing header for dcd file                          */
!-----------------------------------------------------------------------

!     /*   constant   */
      zero = 0

!     /*   test if file exists */
      call testfile ( filename, 7, itest )

!     /*   if file does not exist   */
      if ( itest .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = filename, form='UNFORMATTED', &
     &         access='Stream')

!        /*   write 84 - used to identify endianess   */
         output = 84
         write( iounit ) output

!        /*   four letters for description   */
         write( iounit ) 'CORD'

!        /*   NSET - number of frames   */
         write( iounit ) zero

!        /*   ISTRT - First step   */
         output = INT(istep, 4)
         write( iounit ) output

!        /*   NSAVC - step interval   */
         output = INT(iprint_dcd, 4)
         write( iounit ) output

!        /*   number of time steps in simulation   */
         write( iounit ) zero

!        /*   only used by NAMD   */
         write( iounit ) zero

!        /*   five zeroes   */
         write( iounit ) zero, zero, zero

!        /*   number of fixed atoms   */
         write( iounit ) zero

!        /*   timestep (AKMA units)   */
         tmp = sngl( dt/48.88821d0 )
         write( iounit ) tmp

!        /*   one if we are writing pbc information   */
         output = 0
         if( (iboundary .eq. 1) .or. (iboundary .eq. 2) ) output = 1
         write( iounit ) output

!        /*   8 zeroes   */
         write( iounit ) zero, zero, zero, zero, zero, zero, zero, zero

!        /*   pretend to be charmm 24   */
         output = 24
         write( iounit ) output

!        /*   84 (maybe length of title)   */
         output = 84
         write( iounit ) output

!        /*   164 (maybe total length of next read)   */
         output = 164
         write( iounit ) output

!        /*   2 (maybe number of strings to read)   */
         output = 2
         write( iounit ) output

!        /*   TITLE - 80 characters   */
         write( iounit ) title

!        /*   time and date   */
         title = char_date
         write ( iounit ) title

!        /*   164 (unknown)   */
         output = 164
         write( iounit ) output

!        /*   4 (maybe total length of next read)   */
         output = 4
         write( iounit ) output

!        /*   number of atoms   */
         write( iounit ) numatoms

!        /*   4 (maybe total length of next read)   */
         output = 4
         write( iounit ) output

!        /*   file close   */
         close( iounit )

!     /*   if file does not exist   */
      end if

      return
      end



!***********************************************************************
      subroutine write_dcd_energies_dual_MPI()
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only: &
     &  temp, hamiltonian, nbead, iounit

      use dual_variables, only: &
     &  pot_high, pot_low

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: i, output

!-----------------------------------------------------------------------
!     /*   write individual bead potentials, temp and hamiltonian     */
!-----------------------------------------------------------------------

      open ( iounit, file = 'pot_high.dcd', form = 'unformatted', &
     &      access = 'stream', status = 'old', position = 'append' )

      output = 8 * ( nbead + 2 )

      write( iounit ) output

      do i = 1, nbead
         write( iounit ) pot_high(i)
      end do

      write( iounit ) temp
      write( iounit ) hamiltonian

      close (iounit)

      open ( iounit, file = 'pot_low.dcd', form = 'unformatted', &
     &      access = 'stream', status = 'old', position = 'append' )

      output = 8 * ( nbead + 2 )

      write( iounit ) output

      do i = 1, nbead
         write( iounit ) pot_low(i)
      end do

      write( iounit ) temp
      write( iounit ) hamiltonian

      close (iounit)


      return
      end


!***********************************************************************
      subroutine write_dcd_energies_MPI()
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only: &
     &  pot, temp, hamiltonian, nbead, iounit

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: i, output

!-----------------------------------------------------------------------
!     /*   write individual bead potentials, temp and hamiltonian     */
!-----------------------------------------------------------------------

      open ( iounit, file = 'pot.dcd', form = 'unformatted', &
     &      access = 'stream', status = 'old', position = 'append' )

      output = 8 * ( nbead + 2 )

      write( iounit ) output

      do i = 1, nbead
         write( iounit ) pot(i)
      end do

      write( iounit ) temp
      write( iounit ) hamiltonian

      close (iounit)

      return
      end





!***********************************************************************
      subroutine write_dcd_dipoles_MPI()
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only: &
     &  dipx, dipy, dipz, nbead, iounit

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: i, output

!-----------------------------------------------------------------------
!     /*   write individual bead dipoles to file                      */
!-----------------------------------------------------------------------

      open ( iounit, file = 'dip.dcd', form = 'unformatted', &
     &      access = 'stream', status = 'old', position = 'append' )

      output = 8 * 3 * nbead
      write( iounit ) output
      do i = 1, nbead
         write( iounit ) dipx( i )
         write( iounit ) dipy( i )
         write( iounit ) dipz( i )
      end do

      close( iounit )

      return
      end





!***********************************************************************
      subroutine write_dcd_charges_MPI()
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only: &
     &  q_es, natom, nbead, iounit

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: i, j, output

!-----------------------------------------------------------------------
!     /*   write individual bead dipoles to file                      */
!-----------------------------------------------------------------------

      open ( iounit, file = 'charges.dcd', form = 'unformatted', &
     &      access = 'stream', status = 'old', position = 'append' )

      output = 4 * 2 * natom * nbead
      write( iounit ) output
      do i = 1, nbead
         do j = 1, natom
            write( iounit ) sngl( q_es( 1, j, i ) )
         end do
         do j = 1, natom
            write( iounit ) sngl( q_es( 2, j, i ) )
         end do
      end do

      close( iounit )

      return
      end





!***********************************************************************
      subroutine write_dcd_box_MPI()
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only: &
     & box, vir, iounit_dcd

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer(4):: output

!-----------------------------------------------------------------------
!     /*   Write Box and Virial information to dcd file               */
!-----------------------------------------------------------------------

      open ( iounit_dcd, file = 'box.dcd', form = 'unformatted', &
     &      access = 'stream', status = 'old', position = 'append' )

      output = 8 * 9 * 2

      write( iounit_dcd ) output

      write( iounit_dcd ) box( 1, 1 )
      write( iounit_dcd ) box( 1, 2 )
      write( iounit_dcd ) box( 1, 3 )
      write( iounit_dcd ) box( 2, 1 )
      write( iounit_dcd ) box( 2, 2 )
      write( iounit_dcd ) box( 2, 3 )
      write( iounit_dcd ) box( 3, 1 )
      write( iounit_dcd ) box( 3, 2 )
      write( iounit_dcd ) box( 3, 3 )

      write( iounit_dcd ) vir( 1, 1 )
      write( iounit_dcd ) vir( 1, 2 )
      write( iounit_dcd ) vir( 1, 3 )
      write( iounit_dcd ) vir( 2, 1 )
      write( iounit_dcd ) vir( 2, 2 )
      write( iounit_dcd ) vir( 2, 3 )
      write( iounit_dcd ) vir( 3, 1 )
      write( iounit_dcd ) vir( 3, 2 )
      write( iounit_dcd ) vir( 3, 3 )

      close( iounit_dcd )

      return
      end





!***********************************************************************
      subroutine write_dcd_template_xyz_MPI( ioption )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, au_length, natom, nbead, iounit, species

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: i, j, itest, ioption

!     /*   real numbers   */
      real(8) :: xa, ya, za
      real(8) :: const_1

!     /*   conversion factor   */
      const_1 = au_length * 1.d+10

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   test if file exists   */
      call testfile ( 'template.xyz', 7, itest )

!     /*   if file does not exist   */
      if ( itest .ne. 0 ) then

!        /*   open template xyz file   */
         open ( iounit, file = 'template.xyz')

!        /*   if ioption = 1 we write all beads as one   */
         if ( ioption .eq. 1 ) then

            write( iounit, '(i8)' ) natom*nbead
            write( iounit, '(a28)' ) 'Template file for dcd output'

            do j = 1, nbead
               do i = 1, natom
                  xa = x(i,j) * const_1
                  ya = y(i,j) * const_1
                  za = z(i,j) * const_1
                  write( iounit, '(a4,3f16.8)' ) &
     &               species(i)(1:4), xa, ya, za
               end do
            end do

!        /*   if ioption = 1 we write all beads as one   */
         end if

!        /*   if ioption = 0 we only write the first bead   */
         if ( ioption .eq. 0 ) then
            write( iounit, '(i8)' ) natom
            write( iounit, '(a28)' ) 'Template file for dcd output'

            do i = 1, natom
               xa = x(i,1) * const_1
               ya = y(i,1) * const_1
               za = z(i,1) * const_1
               write( iounit, '(a4,3f16.8)' ) &
     &            species(i)(1:4), xa, ya, za
            end do

!        /*   if ioption = 0 we only write the first bead   */
         end if

!        /*   close template xyz file   */
         close( iounit )

      /*   if file does not exist   */
      end if

      return
      end
