!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     analysis in molecular dynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine analysis_cart ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   nkind, natom_kind, iounit_avg, iounit

      use analysis_variables, only : &
     &   nkindpair, ikindpair, npair_kindpair, iprint_bond, &
     &   iprint_eavg, iprint_rdf, iprint_trj, iprint_dip, iprint_dcd, &
     &   iprint_xyz, iprint_xsf, iprint_box, iprint_cons, ikindpair_inv, &
     &   iprint_rdf_bead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: itest, ioption, l, j1, j2, k1, k2

!-----------------------------------------------------------------------
!     /*   ioption = 1:  initialize/restart                           */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

!        /*   number of atomic pairs   */

         nkindpair = nkind*(nkind+1)/2

!-----------------------------------------------------------------------
!        /*   ikindpair = label for a pair of kinds                   */
!-----------------------------------------------------------------------

         if ( .not. allocated(ikindpair) ) then
            allocate ( ikindpair(nkind,nkind) )
         end if
         if ( .not. allocated(ikindpair_inv) ) then
            allocate ( ikindpair_inv(nkindpair,2) )
         end if

         l = 0
         do k1 =  1, nkind
         do k2 = k1, nkind
            l = l + 1
            ikindpair(k1,k2)   = l
            ikindpair(k2,k1)   = l
            ikindpair_inv(l,1) = k1
            ikindpair_inv(l,2) = k2
         end do
         end do

!-----------------------------------------------------------------------
!        /*   npair_kindpair = number of atom pairs                   */
!-----------------------------------------------------------------------

         if ( .not. allocated(npair_kindpair) ) then
            allocate ( npair_kindpair(nkindpair) )
         end if

         l = 0
         do k1 =  1, nkind
         do k2 = k1, nkind
            l = l + 1
            j1 = natom_kind(k1)
            j2 = natom_kind(k2)
            if ( k1 .ne. k2 ) npair_kindpair(l) = j1*j2
            if ( k1 .eq. k2 ) npair_kindpair(l) = j1*(j1-1)/2
         end do
         end do

!-----------------------------------------------------------------------
!        /*   step intervals of analysis                              */
!-----------------------------------------------------------------------

         call read_int1 ( iprint_bond, '<iprint_bond>', 13, iounit )
         call read_int1 ( iprint_eavg, '<iprint_eavg>', 13, iounit )
         call read_int1 ( iprint_rdf,  '<iprint_rdf>',  12, iounit )
         call read_int1 ( iprint_trj,  '<iprint_trj>',  12, iounit )
         call read_int1 ( iprint_dip,  '<iprint_dip>',  12, iounit )
         call read_int1 ( iprint_xyz,  '<iprint_xyz>',  12, iounit )
         call read_int1 ( iprint_xsf,  '<iprint_xsf>',  12, iounit )
         call read_int1 ( iprint_dcd,  '<iprint_dcd>',  12, iounit )
         call read_int1 ( iprint_box,  '<iprint_box>',  12, iounit )
         call read_int1 ( iprint_cons, '<iprint_cons>', 13, iounit )
         call read_int1 ( iprint_rdf_bead, '<iprint_rdfbead>', &
     &                    16, iounit )

!-----------------------------------------------------------------------
!        /*   check if file called `averages.ini' exists              */
!-----------------------------------------------------------------------

         call testfile ( 'averages.ini', 12, itest )

!-----------------------------------------------------------------------
!        /*   if the file does not exist, initial start.              */
!-----------------------------------------------------------------------

         if ( itest .eq. 1 ) then

            call analysis_bond      ( 0 )
            call analysis_eavg_cart ( 0 )
            call analysis_rdf       ( 0 )
            call analysis_trj       ( 0 )
            call analysis_dip       ( 0 )
            call analysis_xyz       ( 0 )
            call analysis_xsf       ( 0 )
            call analysis_dcd       ( 0 )
            call analysis_box       ( 0 )
            call analysis_cons      ( 0 )
            call analysis_rdf_bead  ( 0 )

!#ifdef aenet2
!            call analysis_aenet( 0 )
!#endif

!-----------------------------------------------------------------------
!        /*   if the file exists, restart.                            */
!-----------------------------------------------------------------------

         else

            open ( iounit_avg, file = 'averages.ini')

            call analysis_bond      ( 1 )
            call analysis_eavg_cart ( 1 )
            call analysis_rdf       ( 1 )
            call analysis_trj       ( 1 )
            call analysis_dip       ( 1 )
            call analysis_xyz       ( 1 )
            call analysis_xsf       ( 1 )
            call analysis_dcd       ( 1 )
            call analysis_box       ( 1 )
            call analysis_cons      ( 1 )
            call analysis_rdf_bead  ( 1 )

!#ifdef aenet2
!            call analysis_aenet( 1 )
!#endif

            close( iounit_avg )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  start analysis                               */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         call analysis_bond      ( 2 )
         call analysis_eavg_cart ( 2 )
         call analysis_rdf       ( 2 )
         call analysis_trj       ( 2 )
         call analysis_dip       ( 2 )
         call analysis_xyz       ( 2 )
         call analysis_xsf       ( 2 )
         call analysis_dcd       ( 2 )
         call analysis_box       ( 2 )
         call analysis_cons      ( 2 )
         call analysis_rdf_bead  ( 2 )

!#ifdef aenet2
!         call analysis_aenet( 2 )
!#endif

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         open ( iounit_avg, file = 'averages.ini')

         call analysis_bond      ( 3 )
         call analysis_eavg_cart ( 3 )
         call analysis_rdf       ( 3 )
         call analysis_trj       ( 3 )
         call analysis_dip       ( 3 )
         call analysis_xyz       ( 3 )
         call analysis_xsf       ( 3 )
         call analysis_dcd       ( 3 )
         call analysis_box       ( 3 )
         call analysis_cons      ( 3 )
         call analysis_rdf_bead  ( 3 )

!#ifdef aenet2
!         call analysis_aenet( 3 )
!#endif

         close( iounit_avg )

      end if

      return
      end





!***********************************************************************
      subroutine analysis_eavg_cart ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, fictmass, vux, vuy, vuz, beta, temperature, &
     &   pot, iounit_eavg, iounit_avg, nbead, natom, istep

      use analysis_variables, only : &
     &   epot, ekinpri, etot, epot_avg, ekinpri_avg, etot_avg, &
     &   eprivir_avg, ekinvir, ekinvir_avg, specific_heat, iprint_eavg

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption, j

      real(8) :: eprivir

      real(8), save :: dstep = 0.d0

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_eavg .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         epot_avg      = 0.d0
         ekinpri_avg   = 0.d0
         etot_avg      = 0.d0

         eprivir_avg = 0.d0

         open ( iounit_eavg, file = 'eavg.out' )

         write(iounit_eavg,'(a)') &
     &      '========' // &
     &      '================================' // &
     &      '================' // &
     &      '================' // &
     &      '================================' // &
     &      '================'
         write(iounit_eavg,'(a)') &
     &      '    step' // &
     &      '            epot            ekin' // &
     &      '            etot' // &
     &      '        epot_avg' // &
     &      '        ekin_avg        etot_avg' // &
     &      '   specific_heat'
         write(iounit_eavg,'(a)') &
     &      '--------' // &
     &      '--------------------------------' // &
     &      '----------------' // &
     &      '----------------' // &
     &      '--------------------------------' // &
     &      '----------------'

         close( iounit_eavg )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         read ( iounit_avg, * ) dstep
         read ( iounit_avg, * ) epot_avg
         read ( iounit_avg, * ) ekinpri_avg
         read ( iounit_avg, * ) etot_avg
         read ( iounit_avg, * ) eprivir_avg

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         dstep = dstep + 1.d0

!-----------------------------------------------------------------------
!        /*   potential energy                                        */
!-----------------------------------------------------------------------

         epot = 0.d0

         do j = 1, nbead
            epot = epot + pot(j)
         end do

         epot = epot/dble(nbead)

!-----------------------------------------------------------------------
!        /*   kinetic energy                                          */
!-----------------------------------------------------------------------

         ekinpri = 0.d0

         do j = 1, nbead

            do i = 1, natom

               ekinpri = ekinpri &
     &            + fictmass(i,j)*vux(i,j)*vux(i,j) &
     &            + fictmass(i,j)*vuy(i,j)*vuy(i,j) &
     &            + fictmass(i,j)*vuz(i,j)*vuz(i,j)

            end do

         end do

         ekinpri = 0.5d0*ekinpri / dble(nbead)

!-----------------------------------------------------------------------
!        /*   total energy                                            */
!-----------------------------------------------------------------------

         etot = ekinpri + epot

!-----------------------------------------------------------------------
!        /*   accumulative averages                                   */
!-----------------------------------------------------------------------

         epot_avg    = epot   /dstep + epot_avg   *(dstep-1.d0)/dstep
         ekinpri_avg = ekinpri/dstep + ekinpri_avg*(dstep-1.d0)/dstep
         etot_avg    = etot   /dstep + etot_avg   *(dstep-1.d0)/dstep

!-----------------------------------------------------------------------
!        /*   specific heat                                           */
!-----------------------------------------------------------------------

         ekinvir = 1.5d0*dble(natom)*boltz*temperature
         ekinvir_avg = ekinvir

         eprivir = ( ekinvir + epot ) * ( ekinvir + epot )

         eprivir_avg = eprivir / dstep &
     &                  + eprivir_avg * (dstep-1.d0)/dstep

         eprivir = ( ekinvir_avg + epot_avg ) &
     &           * ( ekinvir_avg + epot_avg )

         specific_heat &
     &       = dble(nbead) * beta * beta * ( eprivir_avg -  eprivir ) &
     &       + 1.5d0 * dble(natom)

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_eavg) .eq. 0 ) then

            open ( iounit_eavg, file = 'eavg.out', access = 'append' )

            write(iounit_eavg,'(i8,6f16.8,f16.4)') &
     &         istep, &
     &         epot, ekinpri, etot, &
     &         epot_avg, ekinpri_avg, etot_avg, &
     &         specific_heat

            close( iounit_eavg )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         write( iounit_avg, '(e24.16)' ) dstep
         write( iounit_avg, '(e24.16)' ) epot_avg
         write( iounit_avg, '(e24.16)' ) ekinpri_avg
         write( iounit_avg, '(e24.16)' ) etot_avg
         write( iounit_avg, '(e24.16)' ) eprivir_avg

      end if

      return
      end





!***********************************************************************
      subroutine analysis_cons ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   beta, pi, iounit_cons, iounit_avg, iounit, nbead, istep, &
     &   method, afed_type

      use analysis_variables, only : &
     &   iprint_cons

      use cons_variables, only : &
     &   rcons, scons, fref_cons, scons_avg, fref_cons_avg, fc_cons, &
     &   f2ref_cons_avg, href_cons_avg, vref_cons, aref_cons, &
     &   ncons, itype_cons, ipbc_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption, j, k

      real(8) :: dr, dv, f1, f2, a1

      real(8), save :: dstep = 0.d0

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( ncons .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         scons_avg(:,:)            = 0.d0
         fref_cons_avg(:,:)        = 0.d0
         f2ref_cons_avg(:,:,:)     = 0.d0

         dstep = 0.d0

         if ( iprint_cons .gt. 0 ) then

         if ( iset .eq. 0 ) then

         open( iounit_cons, file = 'cons.out' )

         write(iounit_cons,'(a)') &
     &   '========================================================' // &
     &   '========================================================' // &
     &   '===='
         write(iounit_cons,'(a)') &
     &   '    step       ideal      actual      force   actual_avg' // &
     &   '   force_avg   potential free_energy    dist(CN)'         // &
     &   '    dist_avg'
         write(iounit_cons,'(a)') &
     &   '--------------------------------------------------------' // &
     &   '--------------------------------------------------------' // &
     &   '----'

         close( iounit_cons )

         iset = 1

         end if

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         if ( method(1:5) .eq. 'AFED ' ) then

            if ( afed_type(1:5) .eq. 'TEST ' ) then

               continue

            else

               dstep = 0.d0

               scons_avg(:,:)        = 0.d0

               fref_cons_avg(:,:)    = 0.d0

               f2ref_cons_avg(:,:,:) = 0.d0

            end if

         else

            read ( iounit_avg, * ) dstep

            do j = 1, nbead
            do i = 1, ncons
               read ( iounit_avg, * ) scons_avg(i,j)
            end do
            end do

            do j = 1, nbead
            do i = 1, ncons
               read ( iounit_avg, * ) fref_cons_avg(i,j)
            end do
            end do

            do k = 1, nbead
            do j = 1, ncons
            do i = 1, ncons
               read ( iounit_avg, * ) f2ref_cons_avg(i,j,k)
            end do
            end do
            end do

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         dstep = dstep + 1.d0

!-----------------------------------------------------------------------
!        /*   accumulative averages                                   */
!-----------------------------------------------------------------------

         do j = 1, ncons

            do i = 1, nbead

               dr = scons(j,i) - rcons(j,i)

               k = ipbc_cons(j)

               call pbc_cons( dr, k )

               dr = dr + rcons(j,i)

               scons_avg(j,i) = dr / dstep &
     &            + scons_avg(j,i) * (dstep-1.d0)/dstep

            end do

         end do

!-----------------------------------------------------------------------
!        /*   accumulative averages                                   */
!-----------------------------------------------------------------------

         fref_cons_avg(:,:) = fref_cons(:,:) / dstep &
     &                      + fref_cons_avg(:,:) * (dstep-1.d0)/dstep

         do k = 1, nbead
         do j = 1, ncons
         do i = 1, ncons
            f2ref_cons_avg(i,j,k) &
     &         = fref_cons(i,k) * fref_cons(j,k) / dstep &
     &         + f2ref_cons_avg(i,j,k) * (dstep-1.d0)/dstep
         end do
         end do
         end do

!-----------------------------------------------------------------------
!        /*   average hessian                                         */
!-----------------------------------------------------------------------

         do j = 1, ncons
         do i = 1, ncons

            href_cons_avg(i,j) = 0.d0

            if ( i .eq. j ) href_cons_avg(i,j) = fc_cons(i)

            do k = 1, nbead

               href_cons_avg(i,j) = href_cons_avg(i,j) &
     &            - beta * ( f2ref_cons_avg(i,j,k) &
     &            - fref_cons_avg(i,k) * fref_cons_avg(j,k) ) &
     &            / dble(nbead)

            end do

         end do
         end do

!-----------------------------------------------------------------------
!        /*   free energy as integral of mean force                   */
!-----------------------------------------------------------------------

         vref_cons(:) = 0.d0

         do i = 1, nbead-1

            vref_cons(i+1) = vref_cons(i)

            do j = 1, ncons

               dr = scons_avg(j,i+1) - scons_avg(j,i)

               k = ipbc_cons(j)

               call pbc_cons( dr, k )

               f1 = fref_cons_avg(j,i)
               f2 = fref_cons_avg(j,i+1)

               dv = - 0.5d0 * ( f1 + f2 ) * dr

               vref_cons(i+1) = vref_cons(i+1) + dv

            end do

         end do

!-----------------------------------------------------------------------
!        /*   add jacobian contribution                               */
!-----------------------------------------------------------------------

         aref_cons(:) = vref_cons(:)

         do j = 1, ncons
            if ( itype_cons(j) .eq. 1 ) then
              do i = 1, nbead
                  aref_cons(i) = aref_cons(i) &
     &                         + 2.d0/beta*log(scons_avg(j,i))
              end do
            end if
            if ( itype_cons(j) .eq. 2 ) then
              do i = 1, nbead
                  aref_cons(i) = aref_cons(i) &
     &               + 1.d0/beta*log(sin(pi/180.d0*scons_avg(j,i)))
              end do
            end if
         end do

!-----------------------------------------------------------------------
!        /*   shift zero to the minimum value                         */
!-----------------------------------------------------------------------

         a1 = vref_cons(1)

         do i = 2, nbead
            a1 = min( a1, vref_cons(i) )
         end do

         do i = 1, nbead
            vref_cons(i) = vref_cons(i) - a1
         end do

!-----------------------------------------------------------------------
!        /*   shift zero to the minimum value                         */
!-----------------------------------------------------------------------

         a1 = aref_cons(1)

         do i = 2, nbead
            a1 = min( a1, aref_cons(i) )
         end do

         do i = 1, nbead
            aref_cons(i) = aref_cons(i) - a1
         end do

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         if ( iprint_cons .gt. 0 ) then

         if ( mod(istep,iprint_cons) .eq. 0 ) then

            open ( iounit_cons, file = 'cons.out', access = 'append' )

            do j = 1, nbead
            do i = 1, ncons
                  write( iounit_cons, '(i8,9f12.6)' ) &
     &            istep, rcons(i,j), scons(i,j), fref_cons(i,j), &
     &            scons_avg(i,j), fref_cons_avg(i,j), &
     &            vref_cons(j), aref_cons(j)
            end do
            end do

            close( iounit_cons )

!            open( iounit, file = 'cons_avg.out', access = 'append' )
!
!            do j = 1, ncons
!            do i = 1, ncons
!
!               write( iounit, '(i8,f16.8)' )
!     &            istep, href_cons_avg(i,j)
!
!            end do
!            end do
!
!            close( iounit )

         end if

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         write( iounit_avg, '(d24.16)' ) dstep

         do j = 1, nbead
         do i = 1, ncons
            write( iounit_avg, '(d24.16)' ) scons_avg(i,j)
         end do
         end do

         do j = 1, nbead
         do i = 1, ncons
            write( iounit_avg, '(d24.16)' ) fref_cons_avg(i,j)
         end do
         end do

         do k = 1, nbead
         do j = 1, ncons
         do i = 1, ncons
            write( iounit_avg, '(d24.16)' ) f2ref_cons_avg(i,j,k)
         end do
         end do
         end do

      end if

      return
      end
