!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     analysis in path integral molecular dynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine analysis ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   nkind, natom_kind, iounit_avg, iounit

      use analysis_variables, only : &
     &   nkindpair, ikindpair, npair_kindpair, iprint_bond, iprint_cons, &
     &   iprint_eavg, iprint_rgy, iprint_rdf, iprint_trj, iprint_dip, &
     &   iprint_mom, iprint_xyz, iprint_xsf, iprint_box, ikindpair_inv, &
     &   iprint_rdf_bead, iprint_rdf_cent, iprint_dcd, iprint_akin

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: itest, ioption, l, j1, j2, k1, k2

!-----------------------------------------------------------------------
!     /*   ioption = 1:  initialize/restart                           */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

!        /*   number of atomic pairs   */

         nkindpair = nkind*(nkind+1)/2

!-----------------------------------------------------------------------
!        /*   ikindpair = label for a pair of kinds                   */
!-----------------------------------------------------------------------

         if ( .not. allocated( ikindpair ) ) &
     &      allocate ( ikindpair(nkind,nkind) )
         if ( .not. allocated( ikindpair_inv ) ) &
     &      allocate ( ikindpair_inv(nkindpair,2) )

         l = 0
         do k1 =  1, nkind
         do k2 = k1, nkind
            l = l + 1
            ikindpair(k1,k2)   = l
            ikindpair(k2,k1)   = l
            ikindpair_inv(l,1) = k1
            ikindpair_inv(l,2) = k2
         end do
         end do

!-----------------------------------------------------------------------
!        /*   npair_kindpair = number of atom pairs                   */
!-----------------------------------------------------------------------

         if ( .not. allocated( npair_kindpair ) ) &
     &      allocate ( npair_kindpair(nkindpair) )

         l = 0
         do k1 =  1, nkind
         do k2 = k1, nkind
            l = l + 1
            j1 = natom_kind(k1)
            j2 = natom_kind(k2)
            if ( k1 .ne. k2 ) npair_kindpair(l) = j1*j2
            if ( k1 .eq. k2 ) npair_kindpair(l) = j1*(j1-1)/2
         end do
         end do

!-----------------------------------------------------------------------
!        /*   step intervals of analysis                              */
!-----------------------------------------------------------------------

         call read_int1 ( iprint_bond, '<iprint_bond>', 13, iounit )
         call read_int1 ( iprint_eavg, '<iprint_eavg>', 13, iounit )
         call read_int1 ( iprint_rgy,  '<iprint_rgy>',  12, iounit )
         call read_int1 ( iprint_rdf,  '<iprint_rdf>',  12, iounit )
         call read_int1 ( iprint_trj,  '<iprint_trj>',  12, iounit )
         call read_int1 ( iprint_dip,  '<iprint_dip>',  12, iounit )
         call read_int1 ( iprint_mom,  '<iprint_mom>',  12, iounit )
         call read_int1 ( iprint_xyz,  '<iprint_xyz>',  12, iounit )
         call read_int1 ( iprint_xsf,  '<iprint_xsf>',  12, iounit )
         call read_int1 ( iprint_dcd,  '<iprint_dcd>',  12, iounit )
         call read_int1 ( iprint_box,  '<iprint_box>',  12, iounit )
         call read_int1 ( iprint_cons, '<iprint_cons>', 13, iounit )
         call read_int1 ( iprint_rdf_bead, '<iprint_rdfbead>', &
     &                    16, iounit )
         call read_int1 ( iprint_rdf_cent, '<iprint_rdfcent>', &
     &                    16, iounit )

         call read_int1 ( iprint_akin, '<iprint_akin>', 13, iounit )

!-----------------------------------------------------------------------
!        /*   check if file called `averages.ini' exists              */
!-----------------------------------------------------------------------

         call testfile ( 'averages.ini', 12, itest )

!-----------------------------------------------------------------------
!        /*   if the file does not exist, initial start.              */
!-----------------------------------------------------------------------

         if ( itest .eq. 1 ) then

            call analysis_bond      ( 0 )
            call analysis_rgy       ( 0 )
            call analysis_eavg      ( 0 )
            call analysis_rdf       ( 0 )
            call analysis_trj       ( 0 )
            call analysis_dip       ( 0 )
            call analysis_mom       ( 0 )
            call analysis_xyz       ( 0 )
            call analysis_xsf       ( 0 )
            call analysis_dcd       ( 0 )
            call analysis_box       ( 0 )
            call analysis_cons_cent ( 0 )
            call analysis_rdf_bead  ( 0 )
            call analysis_rdf_cent  ( 0 )

            call analysis_akin      ( 0 )

!#ifdef aenet2
!            call analysis_aenet( 0 )
!#endif

!-----------------------------------------------------------------------
!        /*   if the file exists, restart.                            */
!-----------------------------------------------------------------------

         else

            open ( iounit_avg, file = 'averages.ini' )

            call analysis_bond      ( 1 )
            call analysis_rgy       ( 1 )
            call analysis_eavg      ( 1 )
            call analysis_rdf       ( 1 )
            call analysis_trj       ( 1 )
            call analysis_dip       ( 1 )
            call analysis_mom       ( 1 )
            call analysis_xyz       ( 1 )
            call analysis_xsf       ( 1 )
            call analysis_dcd       ( 1 )
            call analysis_box       ( 1 )
            call analysis_cons_cent ( 1 )
            call analysis_rdf_bead  ( 1 )
            call analysis_rdf_cent  ( 1 )

            call analysis_akin      ( 1 )

!#ifdef aenet2
!            call analysis_aenet( 1 )
!#endif

            close( iounit_avg )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  start analysis                               */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         call analysis_bond      ( 2 )
         call analysis_rgy       ( 2 )
         call analysis_eavg      ( 2 )
         call analysis_rdf       ( 2 )
         call analysis_trj       ( 2 )
         call analysis_dip       ( 2 )
         call analysis_mom       ( 2 )
         call analysis_xyz       ( 2 )
         call analysis_xsf       ( 2 )
         call analysis_dcd       ( 2 )
         call analysis_box       ( 2 )
         call analysis_cons_cent ( 2 )
         call analysis_rdf_bead  ( 2 )
         call analysis_rdf_cent  ( 2 )

         call analysis_akin      ( 2 )

!#ifdef aenet2
!         call analysis_aenet( 2 )
!#endif

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         open ( iounit_avg, file = 'averages.ini' )

         call analysis_bond      ( 3 )
         call analysis_rgy       ( 3 )
         call analysis_eavg      ( 3 )
         call analysis_rdf       ( 3 )
         call analysis_trj       ( 3 )
         call analysis_dip       ( 3 )
         call analysis_mom       ( 3 )
         call analysis_xyz       ( 3 )
         call analysis_xsf       ( 3 )
         call analysis_dcd       ( 3 )
         call analysis_box       ( 3 )
         call analysis_cons_cent ( 3 )
         call analysis_rdf_bead  ( 3 )
         call analysis_rdf_cent  ( 3 )

         call analysis_akin      ( 3 )

!#ifdef aenet2
!         call analysis_aenet( 3 )
!#endif

         close( iounit_avg )

      end if

      return
      end





!***********************************************************************
      subroutine analysis_bond ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, ux, uy, uz, iounit_bond, iounit_avg, &
     &   natom, nbead, istep, nkind, ikind

      use analysis_variables, only : &
     &   rbead, rbead2, rcent, rcent2, rkubo, rkubo2, rbead_avg, &
     &   rbead2_avg, rcent_avg, rcent2_avg, rkubo_avg, rkubo2_avg, &
     &   iprint_bond, npair_kindpair, nkindpair, ikindpair

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ioption, i, j, k, m

      real(8) :: r1, r2, rc, rc2, rcx, rcy, rcz, rx, ry, rz

      real(8), save :: dstep = 0.d0

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_bond .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         if ( .not. allocated( rbead ) ) &
     &      allocate ( rbead(nkindpair)  )
         if ( .not. allocated( rbead2 ) ) &
     &      allocate ( rbead2(nkindpair) )
         if ( .not. allocated( rcent ) ) &
     &      allocate ( rcent(nkindpair)  )
         if ( .not. allocated( rcent2 ) ) &
     &      allocate ( rcent2(nkindpair) )
         if ( .not. allocated( rkubo ) ) &
     &      allocate ( rkubo(nkindpair)  )
         if ( .not. allocated( rkubo2 ) ) &
     &      allocate ( rkubo2(nkindpair) )
         if ( .not. allocated( rbead_avg ) ) &
     &      allocate ( rbead_avg(nkindpair)  )
         if ( .not. allocated( rbead2_avg ) ) &
     &      allocate ( rbead2_avg(nkindpair) )
         if ( .not. allocated( rcent_avg ) ) &
     &      allocate ( rcent_avg(nkindpair)  )
         if ( .not. allocated( rcent2_avg ) ) &
     &      allocate ( rcent2_avg(nkindpair) )
         if ( .not. allocated( rkubo_avg ) ) &
     &      allocate ( rkubo_avg(nkindpair)  )
         if ( .not. allocated( rkubo2_avg ) ) &
     &      allocate ( rkubo2_avg(nkindpair) )

         rbead_avg(:)  = 0.d0
         rbead2_avg(:) = 0.d0
         rcent_avg(:)  = 0.d0
         rcent2_avg(:) = 0.d0
         rkubo_avg(:)  = 0.d0
         rkubo2_avg(:) = 0.d0

         open ( iounit_bond, file = 'bond.out' )

         write(iounit_bond,'(a)') &
     &   '================' // &
     &   '====================================' // &
     &   '====================================' // &
     &   '====================================' // &
     &   '===================================='
         write(iounit_bond,'(a)') &
     &   '    step  1   2 ' // &
     &   '       rkubo       rcent       rbead' // &
     &   '      rkubo2      rcent2      rbead2' // &
     &   '   rkubo_avg   rcent_avg   rbead_avg' // &
     &   '  rkubo2_avg  rcent2_avg  rbead2_avg'
         write(iounit_bond,'(a)') &
     &   '----------------' // &
     &   '------------------------------------' // &
     &   '------------------------------------' // &
     &   '------------------------------------' // &
     &   '------------------------------------'

         close( iounit_bond )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         if ( .not. allocated( rbead ) ) &
     &      allocate ( rbead(nkindpair)  )
         if ( .not. allocated( rbead2 ) ) &
     &      allocate ( rbead2(nkindpair) )
         if ( .not. allocated( rcent ) ) &
     &      allocate ( rcent(nkindpair)  )
         if ( .not. allocated( rcent2 ) ) &
     &      allocate ( rcent2(nkindpair) )
         if ( .not. allocated( rkubo ) ) &
     &      allocate ( rkubo(nkindpair)  )
         if ( .not. allocated( rkubo2 ) ) &
     &      allocate ( rkubo2(nkindpair) )
         if ( .not. allocated( rbead_avg ) ) &
     &      allocate ( rbead_avg(nkindpair)  )
         if ( .not. allocated( rbead2_avg ) ) &
     &      allocate ( rbead2_avg(nkindpair) )
         if ( .not. allocated( rcent_avg ) ) &
     &      allocate ( rcent_avg(nkindpair)  )
         if ( .not. allocated( rcent2_avg ) ) &
     &      allocate ( rcent2_avg(nkindpair) )
         if ( .not. allocated( rkubo_avg ) ) &
     &      allocate ( rkubo_avg(nkindpair)  )
         if ( .not. allocated( rkubo2_avg ) ) &
     &      allocate ( rkubo2_avg(nkindpair) )

         read ( iounit_avg, * ) dstep
         read ( iounit_avg, * ) rbead_avg(:)
         read ( iounit_avg, * ) rbead2_avg(:)
         read ( iounit_avg, * ) rcent_avg(:)
         read ( iounit_avg, * ) rcent2_avg(:)
         read ( iounit_avg, * ) rkubo_avg(:)
         read ( iounit_avg, * ) rkubo2_avg(:)

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         dstep = dstep + 1.d0

!-----------------------------------------------------------------------
!        /*   calculate rbead:  bead average                          */
!        /*   calculate rcent:  centroid average                      */
!        /*   calculate rkubo:  kubo canonical average                */
!-----------------------------------------------------------------------

         rbead(:)  = 0.d0
         rbead2(:) = 0.d0

         rkubo(:) = 0.d0
         rkubo2(:) = 0.d0

         do k = 1, nbead

            do i = 1, natom-1
            do j = i+1, natom

               m = ikindpair ( ikind(i), ikind(j) )

               rx     = x(i,k) - x(j,k)
               ry     = y(i,k) - y(j,k)
               rz     = z(i,k) - z(j,k)

               call pbc_atom ( rx, ry, rz )

               r2     = rx*rx + ry*ry + rz*rz
               r1     = sqrt(r2)

               rbead(m)  = rbead(m)  + r1
               rbead2(m) = rbead2(m) + r2

               rkubo(m)  = rkubo(m)  + r1

            end do
            end do

         end do

         do m = 1, nkindpair

            if ( npair_kindpair(m) .eq. 0 ) cycle

            rbead(m)  = rbead(m)  /dble(nbead) /npair_kindpair(m)
            rbead2(m) = rbead2(m) /dble(nbead) /npair_kindpair(m)

            rkubo(m)  = rkubo(m)  /dble(nbead) /npair_kindpair(m)
            rkubo2(m) = rkubo(m)*rkubo(m)

         end do

         rcent(:)  = 0.d0
         rcent2(:) = 0.d0

         do i = 1, natom-1
         do j = i+1, natom

            m = ikindpair ( ikind(i), ikind(j) )

            rcx    = ux(i,1) - ux(j,1)
            rcy    = uy(i,1) - uy(j,1)
            rcz    = uz(i,1) - uz(j,1)

            call pbc_atom ( rcx, rcy, rcz )

            rc2    = rcx*rcx + rcy*rcy + rcz*rcz
            rc     = sqrt(rc2)
            rcent(m)  = rcent(m)  + rc
            rcent2(m) = rcent2(m) + rc2

         end do
         end do

         do m = 1, nkindpair

            if ( npair_kindpair(m) .eq. 0 ) cycle

            rcent(m)  = rcent(m)  /npair_kindpair(m)
            rcent2(m) = rcent2(m) /npair_kindpair(m)

         end do

!-----------------------------------------------------------------------
!        /*   averages                                                */
!-----------------------------------------------------------------------

         rbead_avg(:)  = rbead(:) /dstep &
     &                 + rbead_avg(:)*(dstep-1.d0)/dstep
         rbead2_avg(:) = rbead2(:)/dstep &
     &                 + rbead2_avg(:)*(dstep-1.d0)/dstep

         rcent_avg(:)  = rcent(:) /dstep &
     &                 + rcent_avg(:)*(dstep-1.d0)/dstep
         rcent2_avg(:) = rcent2(:)/dstep &
     &                 + rcent2_avg(:)*(dstep-1.d0)/dstep

         rkubo_avg(:)  = rkubo(:) /dstep &
     &                 + rkubo_avg(:)*(dstep-1.d0)/dstep
         rkubo2_avg(:) = rkubo2(:)/dstep &
     &                 + rkubo2_avg(:)*(dstep-1.d0)/dstep

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_bond) .eq. 0 ) then

            open ( iounit_bond, file = 'bond.out', access = 'append' )

            m = 0

            do i = 1, nkind
            do j = i, nkind

               m = m + 1

               if ( npair_kindpair(m) .eq. 0 ) cycle

               write(iounit_bond,'(i8,1x,i3,1x,i3,12f12.6)') &
     &            istep, i, j, &
     &            rkubo(m),  rcent(m), rbead(m), &
     &            rkubo2(m), rcent2(m), rbead2(m), &
     &            rkubo_avg(m),  rcent_avg(m),  rbead_avg(m), &
     &            rkubo2_avg(m), rcent2_avg(m), rbead2_avg(m)

            end do
            end do

            close( iounit_bond )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         write( iounit_avg, '(e24.16)' ) dstep
         write( iounit_avg, '(e24.16)' ) rbead_avg(:)
         write( iounit_avg, '(e24.16)' ) rbead2_avg(:)
         write( iounit_avg, '(e24.16)' ) rcent_avg(:)
         write( iounit_avg, '(e24.16)' ) rcent2_avg(:)
         write( iounit_avg, '(e24.16)' ) rkubo_avg(:)
         write( iounit_avg, '(e24.16)' ) rkubo2_avg(:)

      end if

      return
      end





!***********************************************************************
      subroutine analysis_rgy ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, ux, uy, uz, iounit_rgy, nkind, natom, nbead, &
     &   natom_kind, istep, iounit_avg, ikind

      use analysis_variables, only : &
     &   rgy, rgy2, rgy_avg, rgy2_avg, iprint_rgy

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption, k, m

      real(8) :: r2, rx, ry, rz

      real(8), save :: dstep = 0.d0

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_rgy .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         if ( .not. allocated( rgy ) ) &
     &      allocate ( rgy(nkind)  )
         if ( .not. allocated( rgy2 ) ) &
     &      allocate ( rgy2(nkind)  )
         if ( .not. allocated( rgy_avg ) ) &
     &      allocate ( rgy_avg(nkind)  )
         if ( .not. allocated( rgy2_avg ) ) &
     &      allocate ( rgy2_avg(nkind)  )

         rgy2_avg(:)  = 0.d0

         open ( iounit_rgy, file = 'rgy.out' )

         write(iounit_rgy,'(a)') &
     &   '===================================='
         write(iounit_rgy,'(a)') &
     &   '    step  1          rgy     rgy_avg'
         write(iounit_rgy,'(a)') &
     &   '------------------------------------'

         close( iounit_rgy )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         if ( .not. allocated( rgy ) ) &
     &      allocate ( rgy(nkind)  )
         if ( .not. allocated( rgy2 ) ) &
     &      allocate ( rgy2(nkind)  )
         if ( .not. allocated( rgy_avg ) ) &
     &      allocate ( rgy_avg(nkind)  )
         if ( .not. allocated( rgy2_avg ) ) &
     &      allocate ( rgy2_avg(nkind)  )

         read ( iounit_avg, * ) dstep
         read ( iounit_avg, * ) rgy2_avg(:)

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         dstep = dstep + 1.d0

!-----------------------------------------------------------------------
!        /*   calculate rgy2:  radius of gyration **2                 */
!-----------------------------------------------------------------------

         rgy2(:)  = 0.d0

         do k = 1, nbead

            do i = 1, natom

               m = ikind(i)

               rx     = x(i,k) - ux(i,1)
               ry     = y(i,k) - uy(i,1)
               rz     = z(i,k) - uz(i,1)

               call pbc_atom ( rx, ry, rz )

               r2     = rx*rx + ry*ry + rz*rz

               rgy2(m) = rgy2(m) + r2

            end do

         end do

         do m = 1, nkind

            if ( natom_kind(m) .eq. 0 ) cycle

            rgy2(m)  = rgy2(m) /dble(nbead) /natom_kind(m)
            rgy(m)   = sqrt(rgy2(m))

         end do

!-----------------------------------------------------------------------
!        /*   averages                                                */
!-----------------------------------------------------------------------

         rgy2_avg(:)   = rgy2(:)/dstep &
     &                 + rgy2_avg(:)*(dstep-1.d0)/dstep

         rgy_avg(:)    = sqrt(rgy2_avg(:))

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_rgy) .eq. 0 ) then

            open ( iounit_rgy, file = 'rgy.out', access = 'append' )

            do i = 1, nkind

               write(iounit_rgy,'(i8,1x,i3,2f24.16)') &
     &            istep, i, rgy(i), rgy_avg(i)

            end do

            close( iounit_rgy )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         write( iounit_avg, '(e24.16)' ) dstep
         write( iounit_avg, '(e24.16)' ) rgy2_avg(:)

      end if

      return
      end





!***********************************************************************
      subroutine analysis_eavg ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, temperature, physmass, x, y, z, ux, uy, uz, fx, fy, fz, &
     &   pot, omega_p2, beta, iounit_eavg, iounit_avg, nbead, natom, &
     &   istep

      use analysis_variables, only : &
     &   epot, ekinpri, ekinvir, etot, epot_avg, ekinvir_avg, &
     &   ekinpri_avg, etot_avg, eprivir_avg, specific_heat, &
     &   iprint_eavg

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption, j

      real(8) :: ekinvir1, ekinvir2, ekinpri1, ekinpri2, eprivir

      real(8), save :: dstep = 0.d0

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_eavg .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         epot_avg      = 0.d0
         ekinvir_avg   = 0.d0
         ekinpri_avg   = 0.d0
         etot_avg      = 0.d0

         eprivir_avg = 0.d0

         open ( iounit_eavg, file = 'eavg.out' )

            write(iounit_eavg,'(a)') &
     &      '========' // &
     &      '================================' // &
     &      '================================' // &
     &      '================================' // &
     &      '================================' // &
     &      '================'
            write(iounit_eavg,'(a)') &
     &      '    step' // &
     &      '            epot         ekinvir' // &
     &      '         ekinpri            etot' // &
     &      '        epot_avg     ekinvir_avg' // &
     &      '     ekinpri_avg        etot_avg' // &
     &      '   specific_heat'
            write(iounit_eavg,'(a)') &
     &      '--------' // &
     &      '--------------------------------' // &
     &      '--------------------------------' // &
     &      '--------------------------------' // &
     &      '--------------------------------' // &
     &      '----------------'

         close( iounit_eavg )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         read ( iounit_avg, * ) dstep
         read ( iounit_avg, * ) epot_avg
         read ( iounit_avg, * ) ekinvir_avg
         read ( iounit_avg, * ) ekinpri_avg
         read ( iounit_avg, * ) etot_avg
         read ( iounit_avg, * ) eprivir_avg

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         dstep = dstep + 1.d0

!-----------------------------------------------------------------------
!        /*   potential energy                                        */
!-----------------------------------------------------------------------

         epot = 0.d0

         do j = 1, nbead
            epot = epot + pot(j)
         end do

         epot = epot/dble(nbead)

!-----------------------------------------------------------------------
!        /*   kinetic energy by virial estimator                      */
!-----------------------------------------------------------------------

!        /*   first term   */

         ekinvir1 = 1.5d0*dble(natom)*boltz*temperature

!        /*   second term   */

         ekinvir2 = 0.d0

         do j = 1, nbead

            do i = 1, natom
               ekinvir2 = ekinvir2 + (x(i,j)-ux(i,1))*fx(i,j) &
     &                             + (y(i,j)-uy(i,1))*fy(i,j) &
     &                             + (z(i,j)-uz(i,1))*fz(i,j)
            end do

         end do

         ekinvir2 = - 0.5d0*ekinvir2

!        /*   the sum of kinetic energy   */

         ekinvir = ekinvir1 + ekinvir2

!-----------------------------------------------------------------------
!        /*   kinetic energy by primitive estimator                   */
!-----------------------------------------------------------------------

!        /*   first term   */

         ekinpri1 = 1.5d0*dble(natom)*dble(nbead)*boltz*temperature

!        /*   second term   */

         ekinpri2 = 0.d0

         do j = 1, nbead

            do i = 1, natom

               if ( j .eq. nbead ) then

                  ekinpri2 = ekinpri2 &
     &            + physmass(i)*(x(i,nbead)-x(i,1))*(x(i,nbead)-x(i,1)) &
     &            + physmass(i)*(y(i,nbead)-y(i,1))*(y(i,nbead)-y(i,1)) &
     &            + physmass(i)*(z(i,nbead)-z(i,1))*(z(i,nbead)-z(i,1))

               else

                  ekinpri2 = ekinpri2 &
     &            + physmass(i)*(x(i,j)-x(i,j+1))*(x(i,j)-x(i,j+1)) &
     &            + physmass(i)*(y(i,j)-y(i,j+1))*(y(i,j)-y(i,j+1)) &
     &            + physmass(i)*(z(i,j)-z(i,j+1))*(z(i,j)-z(i,j+1))

               end if

            end do

         end do

         ekinpri2 = - 0.5d0*omega_p2*ekinpri2

!        /*   the sum of kinetic energy   */

         ekinpri = ekinpri1 + ekinpri2

!-----------------------------------------------------------------------
!        /*   total energy                                            */
!-----------------------------------------------------------------------

         etot = ekinvir + epot

!-----------------------------------------------------------------------
!        /*   accumulative averages                                   */
!-----------------------------------------------------------------------

         epot_avg    = epot   /dstep + epot_avg   *(dstep-1.d0)/dstep
         ekinvir_avg = ekinvir/dstep + ekinvir_avg*(dstep-1.d0)/dstep
         ekinpri_avg = ekinpri/dstep + ekinpri_avg*(dstep-1.d0)/dstep
         etot_avg    = etot   /dstep + etot_avg   *(dstep-1.d0)/dstep

!-----------------------------------------------------------------------
!        /*   specific heat                                           */
!-----------------------------------------------------------------------

         eprivir = ( ekinpri + epot ) * ( ekinvir + epot )

         eprivir_avg = eprivir / dstep &
     &                  + eprivir_avg * (dstep-1.d0)/dstep

         eprivir = ( ekinpri_avg + epot_avg ) &
     &           * ( ekinvir_avg + epot_avg )

         specific_heat &
     &       = beta * beta * ( eprivir_avg -  eprivir ) &
     &       + 1.5d0 * dble(natom)

!         specific_heat
!     &       = beta * beta * ( eprivir_avg - etot_avg*etot_avg  )
!     &       + 1.5d0 * dble(natom)

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_eavg) .eq. 0 ) then

            open ( iounit_eavg, file = 'eavg.out', access = 'append' )

            write( iounit_eavg,'(i8,8f16.8,f16.4)' ) &
     &         istep, &
     &         epot, ekinvir, ekinpri, etot, &
     &         epot_avg, ekinvir_avg, ekinpri_avg, etot_avg, &
     &         specific_heat

            close( iounit_eavg )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         write( iounit_avg, '(e24.16)' ) dstep
         write( iounit_avg, '(e24.16)' ) epot_avg
         write( iounit_avg, '(e24.16)' ) ekinvir_avg
         write( iounit_avg, '(e24.16)' ) ekinpri_avg
         write( iounit_avg, '(e24.16)' ) etot_avg
         write( iounit_avg, '(e24.16)' ) eprivir_avg

      end if

      return
      end





!***********************************************************************
      subroutine analysis_rdf ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, volume, x, y, z, iounit, nkind, natom, nbead, ensemble, &
     &   iboundary, iounit_avg, istep, ikind, natom_kind, box, boxinv

      use analysis_variables, only : &
     &   rdf, rdf_n, rdf_g, rdf_avg, params_rdf, volume_avg, &
     &   iprint_rdf, nkindpair, ikindpair, npair_kindpair

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption, j, nmesh, k, l, m

      real(8) :: dr, dv, fact_n, rs, rl, rx, ry, rz, r, r1, r2

      real(8) :: volume_rdf = 0.d0

      real(8), save :: dstep = 0.d0

      real(8) :: fact_g = 0.d0

      real(8) :: fact_i, fact_j, rdf_cn_i, rdf_cn_j

      real(8) :: absa, absb, absc, a, b, c, bigbox(3,3), bigboxinv(3,3)

      integer :: nbox_rdf(3), jx, jy, jz, j2

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_rdf .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   volume used in rdf                                         */
!-----------------------------------------------------------------------

      if      ( iboundary .eq. 0 ) then

         continue

      else if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then

         if ( ( ensemble(1:4) .eq. 'NPT ' ) .or. &
     &        ( ensemble(1:4) .eq. 'NTT ' ) .or. &
     &        ( ensemble(1:4) .eq. 'NPH ' ) .or. &
     &        ( ensemble(1:4) .eq. 'NTH ' ) ) then

            if ( istep .eq. 1 ) then
               volume_rdf = volume
            else
               volume_rdf = volume_avg
            end if

         else

            if ( istep .eq. 1 ) then
               volume_rdf = volume
            else
               volume_rdf = volume
            end if

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         call read_realn( params_rdf, 3, '<params_rdf>', 12, iounit )

!        /*   shortest interatomic distance   */
         rs = params_rdf(1)

!        /*   longest interatomic distance    */
         rl = params_rdf(2)

!        /*   mesh size    */
         dr = params_rdf(3)

!        /*   number of meshes    */
         nmesh = nint( ( rl - rs )/dr ) + 2

!        /*   memory allocation   */
         if ( .not. allocated( rdf ) ) &
     &      allocate ( rdf(nmesh,nkindpair) )
         if ( .not. allocated( rdf_n ) ) &
     &      allocate ( rdf_n(nmesh,nkindpair) )
         if ( .not. allocated( rdf_g ) ) &
     &      allocate ( rdf_g(nmesh,nkindpair) )
         if ( .not. allocated( rdf_avg ) ) &
     &      allocate ( rdf_avg(nmesh,nkindpair) )

         rdf_avg(:,:)  = 0.d0

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         call read_realn( params_rdf, 3, '<params_rdf>', 12, iounit )

!        /*   shortest interatomic distance   */
         rs = params_rdf(1)

!        /*   longest interatomic distance    */
         rl = params_rdf(2)

!        /*   mesh size    */
         dr = params_rdf(3)

!        /*   number of meshes    */
         nmesh = nint( ( rl - rs )/dr ) + 2

!        /*   memory allocation   */
         if ( .not. allocated( rdf ) ) &
     &      allocate ( rdf(nmesh,nkindpair) )
         if ( .not. allocated( rdf_n ) ) &
     &      allocate ( rdf_n(nmesh,nkindpair) )
         if ( .not. allocated( rdf_g ) ) &
     &      allocate ( rdf_g(nmesh,nkindpair) )
         if ( .not. allocated( rdf_avg ) ) &
     &      allocate ( rdf_avg(nmesh,nkindpair) )

         read ( iounit_avg, * ) dstep
         read ( iounit_avg, * ) rdf_avg(:,:)

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         dstep = dstep + 1.d0

!-----------------------------------------------------------------------
!        /*   prepare meshes                                          */
!-----------------------------------------------------------------------

!        /*   zero clear   */
         rdf(:,:)  =  0.d0

!        /*   shortest interatomic distance   */
         rs = params_rdf(1)

!        /*   longest interatomic distance    */
         rl = params_rdf(2)

!        /*   mesh size    */
         dr = params_rdf(3)

!        /*   number of meshes    */
         nmesh = nint( ( rl - rs )/dr ) + 2

!-----------------------------------------------------------------------
!        /*   calculate interatomic rdf                               */
!-----------------------------------------------------------------------

         absa = sqrt ( boxinv(1,1)*boxinv(1,1) &
     &               + boxinv(1,2)*boxinv(1,2) &
     &               + boxinv(1,3)*boxinv(1,3) )
         absb = sqrt ( boxinv(2,1)*boxinv(2,1) &
     &               + boxinv(2,2)*boxinv(2,2) &
     &               + boxinv(2,3)*boxinv(2,3) )
         absc = sqrt ( boxinv(3,1)*boxinv(3,1) &
     &               + boxinv(3,2)*boxinv(3,2) &
     &               + boxinv(3,3)*boxinv(3,3) )

         nbox_rdf(1) = int(2.d0*rl*absa) + 1
         nbox_rdf(2) = int(2.d0*rl*absb) + 1
         nbox_rdf(3) = int(2.d0*rl*absc) + 1

         if ( nbox_rdf(1)*nbox_rdf(2)*nbox_rdf(3) .eq. 1 ) then

            do k = 1, nbead

               do i = 1, natom-1
               do j = i+1, natom

                  m = ikindpair ( ikind(i), ikind(j) )

!                 /*   calculate interatomic distance   */

                  rx     = x(i,k) - x(j,k)
                  ry     = y(i,k) - y(j,k)
                  rz     = z(i,k) - z(j,k)

                  call pbc_atom ( rx, ry, rz )

                  r2     = rx*rx + ry*ry + rz*rz
                  r1     = sqrt(r2)

!                 /*   mesh point   */
                  l = nint( (r1-rs)/dr ) + 1

!                 /*   skip if out of the range   */
                  if ( l .lt. 1     ) cycle
                  if ( l .gt. nmesh ) cycle

!                 /*   add one   */
                  rdf(l,m)  = rdf(l,m)  + 1.d0

               end do
               end do

            end do

         else

            bigbox(:,1) = dble(nbox_rdf(1))*box(:,1)
            bigbox(:,2) = dble(nbox_rdf(2))*box(:,2)
            bigbox(:,3) = dble(nbox_rdf(3))*box(:,3)

            call inv3 ( bigbox, bigboxinv )

            do k = 1, nbead

            do i = 1, natom
            do j = i, natom

               m = ikindpair ( ikind(i), ikind(j) )

               do jx = 0, nbox_rdf(1)-1
               do jy = 0, nbox_rdf(2)-1
               do jz = 0, nbox_rdf(3)-1

                  j2 = jx*jx + jy*jy + jz*jz

                  if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

!                 /*   calculate interatomic distance   */

                  rx  = x(i,k) - x(j,k)
                  ry  = y(i,k) - y(j,k)
                  rz  = z(i,k) - z(j,k)

                  rx = rx - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
                  ry = ry - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
                  rz = rz - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

                  a = bigboxinv(1,1)*rx + bigboxinv(1,2)*ry &
     &              + bigboxinv(1,3)*rz
                  b = bigboxinv(2,1)*rx + bigboxinv(2,2)*ry &
     &              + bigboxinv(2,3)*rz
                  c = bigboxinv(3,1)*rx + bigboxinv(3,2)*ry &
     &              + bigboxinv(3,3)*rz

                  a = a - nint(a)
                  b = b - nint(b)
                  c = c - nint(c)

                  rx = bigbox(1,1)*a + bigbox(1,2)*b + bigbox(1,3)*c
                  ry = bigbox(2,1)*a + bigbox(2,2)*b + bigbox(2,3)*c
                  rz = bigbox(3,1)*a + bigbox(3,2)*b + bigbox(3,3)*c

                  r2 = rx*rx + ry*ry + rz*rz
                  r1 = sqrt(r2)

!                 /*   mesh point   */
                  l = nint( (r1-rs)/dr ) + 1

!                 /*   skip if out of the range   */
                  if ( l .lt. 1     ) cycle
                  if ( l .gt. nmesh ) cycle

!                 /*   add one   */
                  rdf(l,m)  = rdf(l,m)  + 0.5d0

               end do
               end do
               end do

               if ( i .eq. j ) cycle

               do jx = 0, nbox_rdf(1)-1
               do jy = 0, nbox_rdf(2)-1
               do jz = 0, nbox_rdf(3)-1

                  j2 = jx*jx + jy*jy + jz*jz

                  if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

                  rx = x(j,k) - x(i,k)
                  ry = y(j,k) - y(i,k)
                  rz = z(j,k) - z(i,k)

                  rx = rx - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
                  ry = ry - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
                  rz = rz - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

                  a = bigboxinv(1,1)*rx + bigboxinv(1,2)*ry &
     &              + bigboxinv(1,3)*rz
                  b = bigboxinv(2,1)*rx + bigboxinv(2,2)*ry &
     &              + bigboxinv(2,3)*rz
                  c = bigboxinv(3,1)*rx + bigboxinv(3,2)*ry &
     &              + bigboxinv(3,3)*rz

                  a = a - nint(a)
                  b = b - nint(b)
                  c = c - nint(c)

                  rx = bigbox(1,1)*a + bigbox(1,2)*b + bigbox(1,3)*c
                  ry = bigbox(2,1)*a + bigbox(2,2)*b + bigbox(2,3)*c
                  rz = bigbox(3,1)*a + bigbox(3,2)*b + bigbox(3,3)*c

                  r2 = rx*rx + ry*ry + rz*rz
                  r1 = sqrt(r2)

!                 /*   mesh point   */
                  l = nint( (r1-rs)/dr ) + 1

!                 /*   skip if out of the range   */
                  if ( l .lt. 1     ) cycle
                  if ( l .gt. nmesh ) cycle

!                 /*   add one   */
                  rdf(l,m)  = rdf(l,m)  + 0.5d0

               end do
               end do
               end do

            end do
            end do

            end do

         end if

         rdf_avg(:,:) = rdf(:,:)/dstep + rdf_avg(:,:)*(dstep-1.d0)/dstep

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_rdf) .eq. 0 ) then

            m = 0

            do i = 1, nkind
            do j = i, nkind

               m = m + 1

               if ( npair_kindpair(m) .eq. 0 ) cycle

!              /*   normalization factor   */
               fact_n = 1.d0/dble(nbead*npair_kindpair(m))/dr

               do l = 1, nmesh

                  r = rs + (l-1)*dr

                  rdf_n(l,m) = fact_n*rdf_avg(l,m)

                  r1 = r + 0.5d0*dr
                  r2 = max( r - 0.5d0*dr, 0.d0 )

                  dv = 4.d0/3.d0*pi*( r1*r1*r1 - r2*r2*r2 )

                  if ( dv .eq. 0.d0 ) then
                     fact_g = 0.d0
                  else
                     if      ( iboundary .eq. 0 ) then
                        fact_g = fact_n*dr/(r1-r2)
                     else if ( iboundary .eq. 1 ) then
                        fact_g = fact_n*(dr*volume_rdf/dv)
                     else if ( iboundary .eq. 2 ) then
                        fact_g = fact_n*(dr*volume_rdf/dv)
                     end if
                  end if

                  rdf_g(l,m) = fact_g*rdf_avg(l,m)

               end do

            end do
            end do

            open ( iounit, file = 'rdf.out' )

            write(iounit,'(a)') &
     &         '================================' // &
     &         '===================================='
            write(iounit,'(a)') &
     &         '   i   j      r [au]   pdf: d(r)' // &
     &         '   rdf: g(r)   cn: n1(r)   cn: n2(r)'
            write(iounit,'(a)') &
     &         '--------------------------------' // &
     &         '------------------------------------'

            m = 0

            do i = 1, nkind
            do j = i, nkind

               m = m + 1

               if ( npair_kindpair(m) .eq. 0 ) cycle

               rdf_cn_i = 0.d0
               rdf_cn_j = 0.d0

               do l = 1, nmesh

                  r = rs + (l-1)*dr

                  r1 = r + 0.5d0*dr
                  r2 = max( r - 0.5d0*dr, 0.d0 )

                  dv = 4.d0/3.d0*pi*( r1*r1*r1 - r2*r2*r2 )

                  if      ( iboundary .eq. 0 ) then
                     fact_i = dble(natom_kind(i)) * (r1-r2)
                     fact_j = dble(natom_kind(j)) * (r1-r2)
                  else if ( iboundary .eq. 1 ) then
                     fact_i = dble(natom_kind(i)) * dv / volume_rdf
                     fact_j = dble(natom_kind(j)) * dv / volume_rdf
                  else if ( iboundary .eq. 2 ) then
                     fact_i = dble(natom_kind(i)) * dv / volume_rdf
                     fact_j = dble(natom_kind(j)) * dv / volume_rdf
                  else
                     fact_i = dble(natom_kind(i)) * dv / volume_rdf
                     fact_j = dble(natom_kind(j)) * dv / volume_rdf
                  end if

                  rdf_cn_i = rdf_cn_i + rdf_g(l,m) * fact_i
                  rdf_cn_j = rdf_cn_j + rdf_g(l,m) * fact_j

                  write(iounit,'(1x,i3,1x,i3,5f12.4)') &
     &               i, j, r, rdf_n(l,m), rdf_g(l,m), &
     &               rdf_cn_i, rdf_cn_j

               end do

            end do
            end do

            close( iounit )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         write( iounit_avg, '(e24.16)' ) dstep
         write( iounit_avg, '(e24.16)' ) rdf_avg(:,:)

      end if

      return
      end





!***********************************************************************
      subroutine analysis_rdf_bead ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, volume, x, y, z, iounit, nkind, natom, nbead, ensemble, &
     &   iboundary, iounit_avg, istep, ikind, natom_kind, box, boxinv

      use analysis_variables, only : &
     &   rdf_bead, rdf_n_bead, rdf_g_bead, rdf_avg_bead,  &
     &   params_rdf, volume_avg, iprint_rdf_bead, nkindpair,  &
     &   ikindpair, npair_kindpair

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption, j, nmesh, k, l, m

      real(8) :: dr, dv, fact_n, rs, rl, rx, ry, rz, r, r1, r2

      real(8) :: volume_rdf = 0.d0

      real(8), save :: dstep = 0.d0

      real(8) :: fact_g = 0.d0

      real(8) :: fact_i, fact_j, rdf_cn_i, rdf_cn_j

      real(8) :: absa, absb, absc, a, b, c, bigbox(3,3), bigboxinv(3,3)

      integer :: nbox_rdf(3), jx, jy, jz, j2

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_rdf_bead .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   volume used in rdf                                         */
!-----------------------------------------------------------------------

      if      ( iboundary .eq. 0 ) then

         continue

      else if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then

         if ( ( ensemble(1:4) .eq. 'NPT ' ) .or. &
     &        ( ensemble(1:4) .eq. 'NTT ' ) .or. &
     &        ( ensemble(1:4) .eq. 'NPH ' ) .or. &
     &        ( ensemble(1:4) .eq. 'NTH ' ) ) then

            if ( istep .eq. 1 ) then
               volume_rdf = volume
            else
               volume_rdf = volume_avg
            end if

         else

            if ( istep .eq. 1 ) then
               volume_rdf = volume
            else
               volume_rdf = volume
            end if

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         call read_realn( params_rdf, 3, '<params_rdf>' &
     &   , 12, iounit )

!        /*   shortest interatomic distance   */
         rs = params_rdf(1)

!        /*   longest interatomic distance    */
         rl = params_rdf(2)

!        /*   mesh size    */
         dr = params_rdf(3)

!        /*   number of meshes    */
         nmesh = nint( ( rl - rs )/dr ) + 2

!        /*   memory allocation   */
         if ( .not. allocated( rdf_bead ) ) &
     &      allocate ( rdf_bead(nmesh,nkindpair*nbead) )
         if ( .not. allocated( rdf_n_bead ) ) &
     &      allocate ( rdf_n_bead(nmesh,nkindpair*nbead) )
         if ( .not. allocated( rdf_g_bead ) ) &
     &      allocate ( rdf_g_bead(nmesh,nkindpair*nbead) )
         if ( .not. allocated( rdf_avg_bead ) ) &
     &      allocate ( rdf_avg_bead(nmesh,nkindpair*nbead) )

         rdf_avg_bead(:,:)  = 0.d0

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         call read_realn( params_rdf, 3, '<params_rdf>', 12, iounit )

!        /*   shortest interatomic distance   */
         rs = params_rdf(1)

!        /*   longest interatomic distance    */
         rl = params_rdf(2)

!        /*   mesh size    */
         dr = params_rdf(3)

!        /*   number of meshes    */
         nmesh = nint( ( rl - rs )/dr ) + 2

!        /*   memory allocation   */
         if ( .not. allocated( rdf_bead ) ) &
     &      allocate ( rdf_bead(nmesh,nkindpair*nbead) )
         if ( .not. allocated( rdf_n_bead ) ) &
     &      allocate ( rdf_n_bead(nmesh,nkindpair*nbead) )
         if ( .not. allocated( rdf_g_bead ) ) &
     &      allocate ( rdf_g_bead(nmesh,nkindpair*nbead) )
         if ( .not. allocated( rdf_avg_bead ) ) &
     &      allocate ( rdf_avg_bead(nmesh,nkindpair*nbead) )

         read ( iounit_avg, * ) dstep
         read ( iounit_avg, * ) rdf_avg_bead(:,:)

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         dstep = dstep + 1.d0

!-----------------------------------------------------------------------
!        /*   prepare meshes                                          */
!-----------------------------------------------------------------------

!        /*   zero clear   */
         rdf_bead(:,:)  =  0.d0

!        /*   shortest interatomic distance   */
         rs = params_rdf(1)

!        /*   longest interatomic distance    */
         rl = params_rdf(2)

!        /*   mesh size    */
         dr = params_rdf(3)

!        /*   number of meshes    */
         nmesh = nint( ( rl - rs )/dr ) + 2

!-----------------------------------------------------------------------
!        /*   calculate interatomic rdf                               */
!-----------------------------------------------------------------------

         absa = sqrt ( boxinv(1,1)*boxinv(1,1) &
     &               + boxinv(1,2)*boxinv(1,2) &
     &               + boxinv(1,3)*boxinv(1,3) )
         absb = sqrt ( boxinv(2,1)*boxinv(2,1) &
     &               + boxinv(2,2)*boxinv(2,2) &
     &               + boxinv(2,3)*boxinv(2,3) )
         absc = sqrt ( boxinv(3,1)*boxinv(3,1) &
     &               + boxinv(3,2)*boxinv(3,2) &
     &               + boxinv(3,3)*boxinv(3,3) )

         nbox_rdf(1) = int(2.d0*rl*absa) + 1
         nbox_rdf(2) = int(2.d0*rl*absb) + 1
         nbox_rdf(3) = int(2.d0*rl*absc) + 1

         if ( nbox_rdf(1)*nbox_rdf(2)*nbox_rdf(3) .eq. 1 ) then

            do k = 1, nbead

               do i = 1, natom-1
               do j = i+1, natom

                  m = ikindpair ( ikind(i), ikind(j) )

!                 /*   calculate interatomic distance   */

                  rx     = x(i,k) - x(j,k)
                  ry     = y(i,k) - y(j,k)
                  rz     = z(i,k) - z(j,k)

                  call pbc_atom ( rx, ry, rz )

                  r2     = rx*rx + ry*ry + rz*rz
                  r1     = sqrt(r2)

!                 /*   mesh point   */
                  l = nint( (r1-rs)/dr ) + 1

!                 /*   skip if out of the range   */
                  if ( l .lt. 1     ) cycle
                  if ( l .gt. nmesh ) cycle

!                 /*   add one   */
                  rdf_bead(l,m+nkindpair*(k-1))  =  &
     &                    rdf_bead(l,m+nkindpair*(k-1))  + 1.d0

               end do
               end do

            end do

         else

            bigbox(:,1) = dble(nbox_rdf(1))*box(:,1)
            bigbox(:,2) = dble(nbox_rdf(2))*box(:,2)
            bigbox(:,3) = dble(nbox_rdf(3))*box(:,3)

            call inv3 ( bigbox, bigboxinv )

            do k = 1, nbead

            do i = 1, natom
            do j = i, natom

               m = ikindpair ( ikind(i), ikind(j) )

               do jx = 0, nbox_rdf(1)-1
               do jy = 0, nbox_rdf(2)-1
               do jz = 0, nbox_rdf(3)-1

                  j2 = jx*jx + jy*jy + jz*jz

                  if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

!                 /*   calculate interatomic distance   */

                  rx  = x(i,k) - x(j,k)
                  ry  = y(i,k) - y(j,k)
                  rz  = z(i,k) - z(j,k)

                  rx = rx - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
                  ry = ry - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
                  rz = rz - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

                  a = bigboxinv(1,1)*rx + bigboxinv(1,2)*ry &
     &              + bigboxinv(1,3)*rz
                  b = bigboxinv(2,1)*rx + bigboxinv(2,2)*ry &
     &              + bigboxinv(2,3)*rz
                  c = bigboxinv(3,1)*rx + bigboxinv(3,2)*ry &
     &              + bigboxinv(3,3)*rz

                  a = a - nint(a)
                  b = b - nint(b)
                  c = c - nint(c)

                  rx = bigbox(1,1)*a + bigbox(1,2)*b + bigbox(1,3)*c
                  ry = bigbox(2,1)*a + bigbox(2,2)*b + bigbox(2,3)*c
                  rz = bigbox(3,1)*a + bigbox(3,2)*b + bigbox(3,3)*c

                  r2 = rx*rx + ry*ry + rz*rz
                  r1 = sqrt(r2)

!                 /*   mesh point   */
                  l = nint( (r1-rs)/dr ) + 1

!                 /*   skip if out of the range   */
                  if ( l .lt. 1     ) cycle
                  if ( l .gt. nmesh ) cycle

!                 /*   add one   */
                  rdf_bead(l,m+nkindpair*(k-1)) =  &
     &                   rdf_bead(l,m+nkindpair*(k-1)) + 0.5d0

               end do
               end do
               end do

               if ( i .eq. j ) cycle

               do jx = 0, nbox_rdf(1)-1
               do jy = 0, nbox_rdf(2)-1
               do jz = 0, nbox_rdf(3)-1

                  j2 = jx*jx + jy*jy + jz*jz

                  if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

                  rx = x(j,k) - x(i,k)
                  ry = y(j,k) - y(i,k)
                  rz = z(j,k) - z(i,k)

                  rx = rx - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
                  ry = ry - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
                  rz = rz - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

                  a = bigboxinv(1,1)*rx + bigboxinv(1,2)*ry &
     &              + bigboxinv(1,3)*rz
                  b = bigboxinv(2,1)*rx + bigboxinv(2,2)*ry &
     &              + bigboxinv(2,3)*rz
                  c = bigboxinv(3,1)*rx + bigboxinv(3,2)*ry &
     &              + bigboxinv(3,3)*rz

                  a = a - nint(a)
                  b = b - nint(b)
                  c = c - nint(c)

                  rx = bigbox(1,1)*a + bigbox(1,2)*b + bigbox(1,3)*c
                  ry = bigbox(2,1)*a + bigbox(2,2)*b + bigbox(2,3)*c
                  rz = bigbox(3,1)*a + bigbox(3,2)*b + bigbox(3,3)*c

                  r2 = rx*rx + ry*ry + rz*rz
                  r1 = sqrt(r2)

!                 /*   mesh point   */
                  l = nint( (r1-rs)/dr ) + 1

!                 /*   skip if out of the range   */
                  if ( l .lt. 1     ) cycle
                  if ( l .gt. nmesh ) cycle

!                 /*   add one   */
                  rdf_bead(l,m+nkindpair*(k-1)) =  &
     &                  rdf_bead(l,m+nkindpair*(k-1)) + 0.5d0

               end do
               end do
               end do

            end do
            end do

            end do

         end if

         rdf_avg_bead(:,:) = rdf_bead(:,:)/dstep +  &
     &        rdf_avg_bead(:,:)*(dstep-1.d0)/dstep

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_rdf_bead) .eq. 0 ) then


            do k = 1, nbead
               m = 0
            do i = 1, nkind
            do j = i, nkind

               m = m + 1

               if ( npair_kindpair(m) .eq. 0 ) cycle

!              /*   normalization factor   */
               fact_n = 1.d0/dble(npair_kindpair(m))/dr

               do l = 1, nmesh

                  r = rs + (l-1)*dr

                  rdf_n_bead(l,m+(k-1)*nkindpair) =  &
     &                  fact_n*rdf_avg_bead(l,m+(k-1)*nkindpair)

                  r1 = r + 0.5d0*dr
                  r2 = max( r - 0.5d0*dr, 0.d0 )

                  dv = 4.d0/3.d0*pi*( r1*r1*r1 - r2*r2*r2 )

                  if ( dv .eq. 0.d0 ) then
                     fact_g = 0.d0
                  else
                     if      ( iboundary .eq. 0 ) then
                        fact_g = fact_n*dr/(r1-r2)
                     else if ( iboundary .eq. 1 ) then
                        fact_g = fact_n*(dr*volume_rdf/dv)
                     else if ( iboundary .eq. 2 ) then
                        fact_g = fact_n*(dr*volume_rdf/dv)
                     end if
                  end if

                  rdf_g_bead(l,m+(k-1)*nkindpair) =  &
     &                  fact_g*rdf_avg_bead(l,m+(k-1)*nkindpair)

               end do

            end do
            end do
            end do

            open ( iounit, file = 'rdf_bead.out' )

            write(iounit,'(a)') &
     &         '================================' // &
     &         '=========================================='
            write(iounit,'(a)') &
     &         '   bead  i   j      r [au]   pdf: d(r)' // &
     &         '   rdf: g(r)   cn: n1(r)   cn: n2(r)'
            write(iounit,'(a)') &
     &         '--------------------------------' // &
     &         '------------------------------------------'

            do k = 1, nbead
               m = 0
            do i = 1, nkind
            do j = i, nkind

               m = m + 1

               if ( npair_kindpair(m) .eq. 0 ) cycle

               rdf_cn_i = 0.d0
               rdf_cn_j = 0.d0

               do l = 1, nmesh

                  r = rs + (l-1)*dr

                  r1 = r + 0.5d0*dr
                  r2 = max( r - 0.5d0*dr, 0.d0 )

                  dv = 4.d0/3.d0*pi*( r1*r1*r1 - r2*r2*r2 )

                  if      ( iboundary .eq. 0 ) then
                     fact_i = dble(natom_kind(i)) * (r1-r2)
                     fact_j = dble(natom_kind(j)) * (r1-r2)
                  else if ( iboundary .eq. 1 ) then
                     fact_i = dble(natom_kind(i)) * dv / volume_rdf
                     fact_j = dble(natom_kind(j)) * dv / volume_rdf
                  else if ( iboundary .eq. 2 ) then
                     fact_i = dble(natom_kind(i)) * dv / volume_rdf
                     fact_j = dble(natom_kind(j)) * dv / volume_rdf
                  else
                     fact_i = dble(natom_kind(i)) * dv / volume_rdf
                     fact_j = dble(natom_kind(j)) * dv / volume_rdf
                  end if

                  rdf_cn_i = rdf_cn_i +  &
     &                       rdf_g_bead(l,m+(k-1)*nkindpair) * fact_i
                  rdf_cn_j = rdf_cn_j +  &
     &                       rdf_g_bead(l,m+(k-1)*nkindpair) * fact_j

                  write( iounit, '(1x,i3,1x,i3,1x,i3,5f12.4)' ) &
     &               k, i, j, r, rdf_n_bead(l,m+(k-1)*nkindpair),  &
     &               rdf_g_bead(l,m+(k-1)*nkindpair), &
     &               rdf_cn_i, rdf_cn_j

               end do

            end do
            end do
            end do

            close( iounit )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         write( iounit_avg, '(e24.16)' ) dstep
         write( iounit_avg, '(e24.16)' ) rdf_avg_bead(:,:)

      end if

      return
      end





!***********************************************************************
      subroutine analysis_rdf_cent ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, volume, x, y, z, iounit, nkind, natom, nbead, ensemble, &
     &   iboundary, iounit_avg, istep, ikind, natom_kind, box, boxinv

      use analysis_variables, only : &
     &   rdf_cent, rdf_n_cent, rdf_g_cent, rdf_avg_cent, params_rdf, &
     &   volume_avg, iprint_rdf_cent, nkindpair,  &
     &   ikindpair, npair_kindpair

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption, j, nmesh, k, l, m

      real(8) :: dr, dv, fact_n, rs, rl, rx, ry, rz, r, r1, r2, rgather

      real(8) :: volume_rdf = 0.d0

      real(8), save :: dstep = 0.d0

      real(8) :: fact_g = 0.d0

      real(8) :: fact_i, fact_j, rdf_cn_i, rdf_cn_j

      real(8) :: absa, absb, absc, a, b, c, bigbox(3,3), bigboxinv(3,3)

      integer :: nbox_rdf(3), jx, jy, jz, j2

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_rdf_cent .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   volume used in rdf                                         */
!-----------------------------------------------------------------------

      if      ( iboundary .eq. 0 ) then

         continue

      else if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then

         if ( ( ensemble(1:4) .eq. 'NPT ' ) .or. &
     &        ( ensemble(1:4) .eq. 'NTT ' ) .or. &
     &        ( ensemble(1:4) .eq. 'NPH ' ) .or. &
     &        ( ensemble(1:4) .eq. 'NTH ' ) ) then

            if ( istep .eq. 1 ) then
               volume_rdf = volume
            else
               volume_rdf = volume_avg
            end if

         else

            if ( istep .eq. 1 ) then
               volume_rdf = volume
            else
               volume_rdf = volume
            end if

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         call read_realn(params_rdf, 3, '<params_rdf>', 12, iounit)

!        /*   shortest interatomic distance   */
         rs = params_rdf(1)

!        /*   longest interatomic distance    */
         rl = params_rdf(2)

!        /*   mesh size    */
         dr = params_rdf(3)

!        /*   number of meshes    */
         nmesh = nint( ( rl - rs )/dr ) + 2

!        /*   memory allocation   */
         if ( .not. allocated( rdf_cent ) ) &
     &      allocate ( rdf_cent(nmesh,nkindpair) )
         if ( .not. allocated( rdf_n_cent ) ) &
     &      allocate ( rdf_n_cent(nmesh,nkindpair) )
         if ( .not. allocated( rdf_g_cent ) ) &
     &      allocate ( rdf_g_cent(nmesh,nkindpair) )
         if ( .not. allocated( rdf_avg_cent ) ) &
     &      allocate ( rdf_avg_cent(nmesh,nkindpair) )

         rdf_avg_cent(:,:)  = 0.d0

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         call read_realn( params_rdf, 3, '<params_rdf>', 12, iounit )

!        /*   shortest interatomic distance   */
         rs = params_rdf(1)

!        /*   longest interatomic distance    */
         rl = params_rdf(2)

!        /*   mesh size    */
         dr = params_rdf(3)

!        /*   number of meshes    */
         nmesh = nint( ( rl - rs )/dr ) + 2

!        /*   memory allocation   */
         if ( .not. allocated( rdf_cent ) ) &
     &      allocate ( rdf_cent(nmesh,nkindpair) )
         if ( .not. allocated( rdf_n_cent ) ) &
     &      allocate ( rdf_n_cent(nmesh,nkindpair) )
         if ( .not. allocated( rdf_g_cent ) ) &
     &      allocate ( rdf_g_cent(nmesh,nkindpair) )
         if ( .not. allocated( rdf_avg_cent ) ) &
     &      allocate ( rdf_avg_cent(nmesh,nkindpair) )

         read ( iounit_avg, * ) dstep
         read ( iounit_avg, * ) rdf_avg_cent(:,:)

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         dstep = dstep + 1.d0

!-----------------------------------------------------------------------
!        /*   prepare meshes                                          */
!-----------------------------------------------------------------------

!        /*   zero clear   */
         rdf_cent(:,:)  =  0.d0

!        /*   shortest interatomic distance   */
         rs = params_rdf(1)

!        /*   longest interatomic distance    */
         rl = params_rdf(2)

!        /*   mesh size    */
         dr = params_rdf(3)

!        /*   number of meshes    */
         nmesh = nint( ( rl - rs )/dr ) + 2

!-----------------------------------------------------------------------
!        /*   calculate interatomic rdf                               */
!-----------------------------------------------------------------------

         absa = sqrt ( boxinv(1,1)*boxinv(1,1) &
     &               + boxinv(1,2)*boxinv(1,2) &
     &               + boxinv(1,3)*boxinv(1,3) )
         absb = sqrt ( boxinv(2,1)*boxinv(2,1) &
     &               + boxinv(2,2)*boxinv(2,2) &
     &               + boxinv(2,3)*boxinv(2,3) )
         absc = sqrt ( boxinv(3,1)*boxinv(3,1) &
     &               + boxinv(3,2)*boxinv(3,2) &
     &               + boxinv(3,3)*boxinv(3,3) )

         nbox_rdf(1) = int(2.d0*rl*absa) + 1
         nbox_rdf(2) = int(2.d0*rl*absb) + 1
         nbox_rdf(3) = int(2.d0*rl*absc) + 1

         if ( nbox_rdf(1)*nbox_rdf(2)*nbox_rdf(3) .eq. 1 ) then


            do i = 1, natom-1
            do j = i+1, natom
               rgather = 0.0d0
               m = ikindpair ( ikind(i), ikind(j) )
               do k = 1, nbead


!                 /*   calculate interatomic distance   */

                  rx     = x(i,k) - x(j,k)
                  ry     = y(i,k) - y(j,k)
                  rz     = z(i,k) - z(j,k)

                  call pbc_atom ( rx, ry, rz )

                  r2     = rx*rx + ry*ry + rz*rz
                  r1     = sqrt(r2)
                  
                  rgather = rgather + r1 / dble(nbead)

               end do
!              /*   mesh point   */
               l = nint( (rgather-rs)/dr ) + 1

!              /*   skip if out of the range   */
               if ( l .lt. 1     ) cycle
               if ( l .gt. nmesh ) cycle

!              /*   add one   */
               rdf_cent(l,m)  = rdf_cent(l,m)  + 1.d0
            end do
            end do

         else

            bigbox(:,1) = dble(nbox_rdf(1))*box(:,1)
            bigbox(:,2) = dble(nbox_rdf(2))*box(:,2)
            bigbox(:,3) = dble(nbox_rdf(3))*box(:,3)

            call inv3 ( bigbox, bigboxinv )


            do i = 1, natom
            do j = i, natom
            
            do k = 1, nbead

               m = ikindpair ( ikind(i), ikind(j) )
               rgather = 0.0d0
               
               do jx = 0, nbox_rdf(1)-1
               do jy = 0, nbox_rdf(2)-1
               do jz = 0, nbox_rdf(3)-1

                  j2 = jx*jx + jy*jy + jz*jz

                  if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

!                 /*   calculate interatomic distance   */

                  rx  = x(i,k) - x(j,k)
                  ry  = y(i,k) - y(j,k)
                  rz  = z(i,k) - z(j,k)

                  rx = rx - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
                  ry = ry - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
                  rz = rz - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

                  a = bigboxinv(1,1)*rx + bigboxinv(1,2)*ry &
     &              + bigboxinv(1,3)*rz
                  b = bigboxinv(2,1)*rx + bigboxinv(2,2)*ry &
     &              + bigboxinv(2,3)*rz
                  c = bigboxinv(3,1)*rx + bigboxinv(3,2)*ry &
     &              + bigboxinv(3,3)*rz

                  a = a - nint(a)
                  b = b - nint(b)
                  c = c - nint(c)

                  rx = bigbox(1,1)*a + bigbox(1,2)*b + bigbox(1,3)*c
                  ry = bigbox(2,1)*a + bigbox(2,2)*b + bigbox(2,3)*c
                  rz = bigbox(3,1)*a + bigbox(3,2)*b + bigbox(3,3)*c

                  r2 = rx*rx + ry*ry + rz*rz
                  r1 = sqrt(r2)

                  rgather = rgather + r1/dble(nbead)

               end do
               end do
               end do
!                 /*   mesh point   */
                  l = nint( (rgather-rs)/dr ) + 1

!                 /*   skip if out of the range   */
                  if ( l .lt. 1     ) cycle
                  if ( l .gt. nmesh ) cycle

!                 /*   add one   */
                  rdf_cent(l,m)  = rdf_cent(l,m)  + 0.5d0
               
            end do
             
               if ( i .eq. j ) cycle

            do k = 1, nbead
               rgather = 0.0d0
               m = ikindpair ( ikind(i), ikind(j) )
               do jx = 0, nbox_rdf(1)-1
               do jy = 0, nbox_rdf(2)-1
               do jz = 0, nbox_rdf(3)-1

                  j2 = jx*jx + jy*jy + jz*jz

                  if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

                  rx = x(j,k) - x(i,k)
                  ry = y(j,k) - y(i,k)
                  rz = z(j,k) - z(i,k)

                  rx = rx - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
                  ry = ry - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
                  rz = rz - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

                  a = bigboxinv(1,1)*rx + bigboxinv(1,2)*ry &
     &              + bigboxinv(1,3)*rz
                  b = bigboxinv(2,1)*rx + bigboxinv(2,2)*ry &
     &              + bigboxinv(2,3)*rz
                  c = bigboxinv(3,1)*rx + bigboxinv(3,2)*ry &
     &              + bigboxinv(3,3)*rz

                  a = a - nint(a)
                  b = b - nint(b)
                  c = c - nint(c)

                  rx = bigbox(1,1)*a + bigbox(1,2)*b + bigbox(1,3)*c
                  ry = bigbox(2,1)*a + bigbox(2,2)*b + bigbox(2,3)*c
                  rz = bigbox(3,1)*a + bigbox(3,2)*b + bigbox(3,3)*c

                  r2 = rx*rx + ry*ry + rz*rz
                  r1 = sqrt(r2)

                  rgather = rgather + r1 / dble(nbead)

               end do
               end do
               end do
!                 /*   mesh point   */
                  l = nint( (rgather-rs)/dr ) + 1

!                 /*   skip if out of the range   */
                  if ( l .lt. 1     ) cycle
                  if ( l .gt. nmesh ) cycle

!                 /*   add one   */
                  rdf_cent(l,m)  = rdf_cent(l,m)  + 0.5d0
            end do
            
            end do
            end do


         end if

         rdf_avg_cent(:,:) = rdf_cent(:,:)/dstep  &
     &           + rdf_avg_cent(:,:)*(dstep-1.d0)/dstep

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_rdf_cent) .eq. 0 ) then

            m = 0

            do i = 1, nkind
            do j = i, nkind

               m = m + 1

               if ( npair_kindpair(m) .eq. 0 ) cycle

!              /*   normalization factor   */
               fact_n = 1.d0/dble(npair_kindpair(m))/dr

               do l = 1, nmesh

                  r = rs + (l-1)*dr

                  rdf_n_cent(l,m) = fact_n*rdf_avg_cent(l,m)

                  r1 = r + 0.5d0*dr
                  r2 = max( r - 0.5d0*dr, 0.d0 )

                  dv = 4.d0/3.d0*pi*( r1*r1*r1 - r2*r2*r2 )

                  if ( dv .eq. 0.d0 ) then
                     fact_g = 0.d0
                  else
                     if      ( iboundary .eq. 0 ) then
                        fact_g = fact_n*dr/(r1-r2)
                     else if ( iboundary .eq. 1 ) then
                        fact_g = fact_n*(dr*volume_rdf/dv)
                     else if ( iboundary .eq. 2 ) then
                        fact_g = fact_n*(dr*volume_rdf/dv)
                     end if
                  end if

                  rdf_g_cent(l,m) = fact_g*rdf_avg_cent(l,m)

               end do

            end do
            end do

            open ( iounit, file = 'rdf_cent.out' )

            write(iounit,'(a)') &
     &         '================================' // &
     &         '===================================='
            write(iounit,'(a)') &
     &         '   i   j      r [au]   pdf: d(r)' // &
     &         '   rdf: g(r)   cn: n1(r)   cn: n2(r)'
            write(iounit,'(a)') &
     &         '--------------------------------' // &
     &         '------------------------------------'

            m = 0

            do i = 1, nkind
            do j = i, nkind

               m = m + 1

               if ( npair_kindpair(m) .eq. 0 ) cycle

               rdf_cn_i = 0.d0
               rdf_cn_j = 0.d0

               do l = 1, nmesh

                  r = rs + (l-1)*dr

                  r1 = r + 0.5d0*dr
                  r2 = max( r - 0.5d0*dr, 0.d0 )

                  dv = 4.d0/3.d0*pi*( r1*r1*r1 - r2*r2*r2 )

                  if      ( iboundary .eq. 0 ) then
                     fact_i = dble(natom_kind(i)) * (r1-r2)
                     fact_j = dble(natom_kind(j)) * (r1-r2)
                  else if ( iboundary .eq. 1 ) then
                     fact_i = dble(natom_kind(i)) * dv / volume_rdf
                     fact_j = dble(natom_kind(j)) * dv / volume_rdf
                  else if ( iboundary .eq. 2 ) then
                     fact_i = dble(natom_kind(i)) * dv / volume_rdf
                     fact_j = dble(natom_kind(j)) * dv / volume_rdf
                  else
                     fact_i = dble(natom_kind(i)) * dv / volume_rdf
                     fact_j = dble(natom_kind(j)) * dv / volume_rdf
                  end if

                  rdf_cn_i = rdf_cn_i + rdf_g_cent(l,m) * fact_i
                  rdf_cn_j = rdf_cn_j + rdf_g_cent(l,m) * fact_j

                  write( iounit, '(1x,i3,1x,i3,5f12.4)' ) &
     &               i, j, r, rdf_n_cent(l,m), rdf_g_cent(l,m), &
     &               rdf_cn_i, rdf_cn_j

               end do

            end do
            end do

            close( iounit )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         write( iounit_avg, '(e24.16)' ) dstep
         write( iounit_avg, '(e24.16)' ) rdf_avg_cent(:,:)

      end if

      return
      end





!***********************************************************************
      subroutine analysis_trj ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, vx, vy, vz, fx, fy, fz, pot, natom, nbead, &
     &   iounit, iounit_trj, istep

      use analysis_variables, only : &
     &   iformat_trj, iprint_trj

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption, k

      real(8) :: fxn, fyn, fzn

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_trj .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         call read_int1( iformat_trj, '<iformat_trj>', 13, iounit )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         call read_int1( iformat_trj, '<iformat_trj>', 13, iounit )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

!-----------------------------------------------------------------------
!     /*   print trajectory                                           */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_trj) .eq. 0 ) then

            open ( iounit_trj,  file = 'trj.out', access = 'append' )

            do k = 1, nbead
            do i = 1, natom

               fxn = fx(i,k)*nbead
               fyn = fy(i,k)*nbead
               fzn = fz(i,k)*nbead

               if ( iformat_trj .eq. 1 ) then

                  write(iounit_trj,'(i8,10e24.16)') &
     &               istep,  x(i,k),  y(i,k),  z(i,k), &
     &                      vx(i,k), vy(i,k), vz(i,k), &
     &                      fxn, fyn, fzn, pot(k)

               else if ( iformat_trj .eq. 2 ) then

                  write(iounit_trj,'(i8,10e16.8)') &
     &               istep,  x(i,k),  y(i,k),  z(i,k), &
     &                      vx(i,k), vy(i,k), vz(i,k), &
     &                      fxn, fyn, fzn, pot(k)

               else

                  write(iounit_trj,'(i8,10e24.16)') &
     &               istep,  x(i,k),  y(i,k),  z(i,k), &
     &                      vx(i,k), vy(i,k), vz(i,k), &
     &                      fxn, fyn, fzn, pot(k)

               end if

            end do
            end do

            close( iounit_trj )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         continue

      end if

      return
      end





!***********************************************************************
      subroutine analysis_dip ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   dipx, dipy, dipz, iounit_dip, istep, nbead, iounit_dip

      use analysis_variables, only : &
     &   iprint_dip

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_dip .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         continue

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         continue

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

!-----------------------------------------------------------------------
!     /*   print dipole moment                                        */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_dip) .eq. 0 ) then

            open ( iounit_dip, file = 'dipole.out', access = 'append' )

            do i = 1, nbead

               write(iounit_dip,'(i8,3f24.16)') &
     &            istep, dipx(i), dipy(i), dipz(i)

            end do

            close( iounit_dip )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         continue

      end if

      return
      end





!***********************************************************************
      subroutine analysis_mom ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, vux, vuy, vuz, vx, vy, vz, fux, fuy, fuz, boltz, &
     &   xg, yg, zg, ux, uy, uz, x, y, z, physmass, fx, fy, fz, &
     &   natom, nbead, iounit_mom, istep

      use analysis_variables, only : &
     &   temp_tot, temp_cent, temp_cart, p_tot, p_cent, p_cart, &
     &   a_tot, a_cent, a_cart, t_tot, t_cent, t_cart, iprint_mom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption, k

      real(8) :: fm, pm, sumx, sumy, sumz

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_mom .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         open ( iounit_mom,  file = 'momentum.out' )

         write(iounit_mom,'(a)') &
     &      '========' // &
     &      '================================================' // &
     &      '================================================' // &
     &      '================================================' // &
     &      '================================================'
         write(iounit_mom,'(a)') &
     &      '    step' // &
     &      '        temp_tot       temp_cent       temp_cart' // &
     &      '           p_tot          p_cent          p_cart' // &
     &      '           a_tot          a_cent          a_cart' // &
     &      '           t_tot          t_cent          t_cart'
         write(iounit_mom,'(a)') &
     &      '--------' // &
     &      '------------------------------------------------' // &
     &      '------------------------------------------------' // &
     &      '------------------------------------------------' // &
     &      '------------------------------------------------'

         close( iounit_mom )

         return

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         continue

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

!-----------------------------------------------------------------------
!        /*   center of mass:  xg, yg, zg                             */
!-----------------------------------------------------------------------

         call center_of_mass

!-----------------------------------------------------------------------
!        /*   temperature:  total                                     */
!-----------------------------------------------------------------------

         sumx = 0.d0
         sumy = 0.d0
         sumz = 0.d0

         do k = 1, nbead
         do i = 1, natom
            sumx = sumx + fictmass(i,k)*vux(i,k)*vux(i,k)
            sumy = sumy + fictmass(i,k)*vuy(i,k)*vuy(i,k)
            sumz = sumz + fictmass(i,k)*vuz(i,k)*vuz(i,k)
         end do
         end do

         sumx = sumx/(nbead*natom*boltz)
         sumy = sumy/(nbead*natom*boltz)
         sumz = sumz/(nbead*natom*boltz)

         temp_tot = ( sumx + sumy + sumz ) / 3.d0

!-----------------------------------------------------------------------
!     /*   temperature:  centroid                                     */
!-----------------------------------------------------------------------

         sumx = 0.d0
         sumy = 0.d0
         sumz = 0.d0

         do i = 1, natom
            sumx = sumx + fictmass(i,1)*vux(i,1)*vux(i,1)
            sumy = sumy + fictmass(i,1)*vuy(i,1)*vuy(i,1)
            sumz = sumz + fictmass(i,1)*vuz(i,1)*vuz(i,1)
         end do

         sumx = sumx/(natom*boltz)
         sumy = sumy/(natom*boltz)
         sumz = sumz/(natom*boltz)

         temp_cent = ( sumx + sumy + sumz ) / 3.d0

!-----------------------------------------------------------------------
!        /*   temperature:  cartesian                                 */
!-----------------------------------------------------------------------

         sumx = 0.d0
         sumy = 0.d0
         sumz = 0.d0

         do k = 1, nbead
         do i = 1, natom
            sumx = sumx + physmass(i)*vx(i,k)*vx(i,k)
            sumy = sumy + physmass(i)*vy(i,k)*vy(i,k)
            sumz = sumz + physmass(i)*vz(i,k)*vz(i,k)
         end do
         end do

         sumx = sumx/(nbead*natom*boltz)
         sumy = sumy/(nbead*natom*boltz)
         sumz = sumz/(nbead*natom*boltz)

         temp_cart = ( sumx + sumy + sumz ) / 3.d0

!-----------------------------------------------------------------------
!        /*   linear momentum:  total                                 */
!-----------------------------------------------------------------------

         sumx = 0.d0
         sumy = 0.d0
         sumz = 0.d0

         do k = 1, nbead
         do i = 1, natom
            sumx = sumx + fictmass(i,k)*vux(i,k)
            sumy = sumy + fictmass(i,k)*vuy(i,k)
            sumz = sumz + fictmass(i,k)*vuz(i,k)
         end do
         end do

         p_tot   = sqrt( sumx*sumx + sumy*sumy + sumz*sumz )

!-----------------------------------------------------------------------
!        /*   linear momentum:  centroid                              */
!-----------------------------------------------------------------------

         sumx = 0.d0
         sumy = 0.d0
         sumz = 0.d0

         do i = 1, natom
            sumx = sumx + fictmass(i,1)*vux(i,1)
            sumy = sumy + fictmass(i,1)*vuy(i,1)
            sumz = sumz + fictmass(i,1)*vuz(i,1)
         end do

         p_cent  = sqrt( sumx*sumx + sumy*sumy + sumz*sumz )

!-----------------------------------------------------------------------
!        /*   linear momentum:  cartesian                             */
!-----------------------------------------------------------------------

         sumx = 0.d0
         sumy = 0.d0
         sumz = 0.d0

         do k = 1, nbead
         do i = 1, natom
            sumx = sumx + physmass(i)*vx(i,k)
            sumy = sumy + physmass(i)*vy(i,k)
            sumz = sumz + physmass(i)*vz(i,k)
         end do
         end do

         p_cart  = sqrt( sumx*sumx + sumy*sumy + sumz*sumz )

!-----------------------------------------------------------------------
!        /*   angular momentum:  total                                */
!-----------------------------------------------------------------------

         sumx = 0.d0
         sumy = 0.d0
         sumz = 0.d0

         do k = 1, nbead
         do i = 1, natom
            fm   = fictmass(i,k)
            sumx = sumx + fm*((uy(i,k)-yg(k))*vuz(i,k) &
     &                       -(uz(i,k)-zg(k))*vuy(i,k))
            sumy = sumy + fm*((uz(i,k)-zg(k))*vux(i,k) &
     &                       -(ux(i,k)-xg(k))*vuz(i,k))
            sumz = sumz + fm*((ux(i,k)-xg(k))*vuy(i,k) &
     &                       -(uy(i,k)-yg(k))*vux(i,k))
         end do
         end do

         a_tot   = sqrt( sumx*sumx + sumy*sumy + sumz*sumz )

!-----------------------------------------------------------------------
!        /*   angular momentum:  centroid                             */
!-----------------------------------------------------------------------

         sumx = 0.d0
         sumy = 0.d0
         sumz = 0.d0

         do i = 1, natom
            fm   = fictmass(i,1)
            sumx = sumx + fm*((uy(i,1)-yg(1))*vuz(i,1) &
     &                       -(uz(i,1)-zg(1))*vuy(i,1))
            sumy = sumy + fm*((uz(i,1)-zg(1))*vux(i,1) &
     &                       -(ux(i,1)-xg(1))*vuz(i,1))
            sumz = sumz + fm*((ux(i,1)-xg(1))*vuy(i,1) &
     &                       -(uy(i,1)-yg(1))*vux(i,1))
         end do

         a_cent  = sqrt( sumx*sumx + sumy*sumy + sumz*sumz )

!-----------------------------------------------------------------------
!        /*   angular momentum:  cartesian                            */
!-----------------------------------------------------------------------

         sumx = 0.d0
         sumy = 0.d0
         sumz = 0.d0

         do k = 1, nbead
         do i = 1, natom
            pm   = physmass(i)
            sumx = sumx + pm*((y(i,k)-yg(1))*vz(i,k) &
     &                       -(z(i,k)-zg(1))*vy(i,k))
            sumy = sumy + pm*((z(i,k)-zg(1))*vx(i,k) &
     &                       -(x(i,k)-xg(1))*vz(i,k))
            sumz = sumz + pm*((x(i,k)-xg(1))*vy(i,k) &
     &                       -(y(i,k)-yg(1))*vx(i,k))
         end do
         end do

         a_cart  = sqrt( sumx*sumx + sumy*sumy + sumz*sumz )

!-----------------------------------------------------------------------
!        /*   torque:  total                                          */
!-----------------------------------------------------------------------

         sumx = 0.d0
         sumy = 0.d0
         sumz = 0.d0

         do k = 1, nbead
         do i = 1, natom
            sumx = sumx + (uy(i,k)-yg(k))*fuz(i,k) &
     &                   -(uz(i,k)-zg(k))*fuy(i,k)
            sumy = sumy + (uz(i,k)-zg(k))*fux(i,k) &
     &                   -(ux(i,k)-xg(k))*fuz(i,k)
            sumz = sumz + (ux(i,k)-xg(k))*fuy(i,k) &
     &                   -(uy(i,k)-yg(k))*fux(i,k)
         end do
         end do

         t_tot   = sqrt( sumx*sumx + sumy*sumy + sumz*sumz )

!-----------------------------------------------------------------------
!     /*   torque:  centroid                                          */
!-----------------------------------------------------------------------

         sumx = 0.d0
         sumy = 0.d0
         sumz = 0.d0

         do i = 1, natom
            sumx = sumx + (uy(i,1)-yg(1))*fuz(i,1) &
     &                   -(uz(i,1)-zg(1))*fuy(i,1)
            sumy = sumy + (uz(i,1)-zg(1))*fux(i,1) &
     &                   -(ux(i,1)-xg(1))*fuz(i,1)
            sumz = sumz + (ux(i,1)-xg(1))*fuy(i,1) &
     &                   -(uy(i,1)-yg(1))*fux(i,1)
         end do

         t_cent  = sqrt( sumx*sumx + sumy*sumy + sumz*sumz )

!-----------------------------------------------------------------------
!        /*   torque:  cartesian                                      */
!-----------------------------------------------------------------------

         sumx = 0.d0
         sumy = 0.d0
         sumz = 0.d0

         do k = 1, nbead
         do i = 1, natom
            sumx = sumx + fy(i,k)*(z(i,k)-zg(1)) &
     &                  - fz(i,k)*(y(i,k)-yg(1))
            sumy = sumy + fz(i,k)*(x(i,k)-xg(1)) &
     &                  - fx(i,k)*(z(i,k)-zg(1))
            sumz = sumz + fx(i,k)*(y(i,k)-yg(1)) &
     &                  - fy(i,k)*(x(i,k)-xg(1))
         end do
         end do

         t_cart  = sqrt( sumx*sumx + sumy*sumy + sumz*sumz )

!-----------------------------------------------------------------------
!        /*   print                                                   */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_mom) .eq. 0 ) then

            open ( iounit_mom, file='momentum.out', access='append' )

            write( iounit_mom, '(i8,12d16.8)' ) &
     &         istep, temp_tot, temp_cent, temp_cart, &
     &                   p_tot,    p_cent,    p_cart, &
     &                   a_tot,    a_cent,    a_cart, &
     &                   t_tot,    t_cent,    t_cart

            close( iounit_mom )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         continue

      end if

      return
      end





!***********************************************************************
      subroutine analysis_xyz ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, ux, uy, uz, pot, au_length, iounit_xyz, &
     &   iounit, istep, species, natom, nbead, ikind, nkind, mbox

      use analysis_variables, only : &
     &   iprint_xyz, iformat_xyz, ikind_xyz

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, l, ioption, m1, m2, m3, ierr

!     /*   real   */
      real(8) :: xa, ya, za, xb, yb, zb

!     /*   real   */
      real(8), parameter :: bohr2ang = au_length/1.d-10

!     /*   integer   */
      integer, save :: natom_xyz

!     /*   logical   */
      logical :: file_opened

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_xyz .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         ierr = 0

         inquire( unit = iounit_xyz, opened = file_opened )

         if ( file_opened ) then
            close( iounit_xyz )
            open ( iounit_xyz,  file = 'trj.xyz' )
         else
            open ( iounit_xyz,  file = 'trj.xyz' )
         end if

         call error_handling( ierr, 'subroutine analysis_xyz', 23 )

         call read_int1( iformat_xyz, '<iformat_xyz>', 13, iounit )

         call read_intn( ikind_xyz, 2, '<ikind_xyz>', 11, iounit )

         if ( ikind_xyz(2) .le. 0 ) ikind_xyz(2) = nkind

         l = 0
         do i = 1, natom
            if ( ikind(i) .lt. ikind_xyz(1) ) cycle
            if ( ikind(i) .gt. ikind_xyz(2) ) cycle
            l = l + 1
         end do

         natom_xyz = l

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         ierr = 0

         inquire( unit = iounit_xyz, opened = file_opened )

         if ( file_opened ) then
            close( iounit_xyz )
            open ( iounit_xyz,  file = 'trj.xyz', access = 'append' )
         else
            open ( iounit_xyz,  file = 'trj.xyz', access = 'append' )
         end if

         call error_handling( ierr, 'subroutine analysis_xyz', 23 )

         call read_int1( iformat_xyz, '<iformat_xyz>', 13, iounit )

         call read_intn( ikind_xyz, 2, '<ikind_xyz>', 11, iounit )

         if ( ikind_xyz(2) .le. 0 ) ikind_xyz(2) = nkind

         l = 0
         do i = 1, natom
            if ( ikind(i) .lt. ikind_xyz(1) ) cycle
            if ( ikind(i) .gt. ikind_xyz(2) ) cycle
            l = l + 1
         end do

         natom_xyz = l

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

!-----------------------------------------------------------------------
!     /*   print trajectory                                           */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_xyz) .eq. 0 ) then

!           ===  each bead independently  ===

            if      ( iformat_xyz .eq. 1 ) then

               do j = 1, nbead
                  write( iounit_xyz, '(i8)' ) natom_xyz
                  write( iounit_xyz, '(2i8,f16.8)' ) istep, j, pot(j)
                  do i = 1, natom
                     if ( ikind(i) .lt. ikind_xyz(1) ) cycle
                     if ( ikind(i) .gt. ikind_xyz(2) ) cycle
                     xa = x(i,j) * bohr2ang
                     ya = y(i,j) * bohr2ang
                     za = z(i,j) * bohr2ang
                     xb = x(i,j)
                     yb = y(i,j)
                     zb = z(i,j)
                     m1 = mbox(1,i,j)
                     m2 = mbox(2,i,j)
                     m3 = mbox(3,i,j)
                     call pbc_unfold( xb, yb, zb, m1, m2, m3 )
                     xb = xb * bohr2ang
                     yb = yb * bohr2ang
                     zb = zb * bohr2ang
                     write( iounit_xyz, '(a4,6f16.8)' ) &
     &                  species(i)(1:4), xa, ya, za, xb, yb, zb
                  end do
               end do

               flush( iounit_xyz )

!           ===  all beads into one  ===

            else if ( iformat_xyz .eq. 2 ) then

               write( iounit_xyz, '(i8)' ) natom_xyz*nbead
               write( iounit_xyz, '(i8)' ) istep

               do j = 1, nbead
                  do i = 1, natom
                     if ( ikind(i) .lt. ikind_xyz(1) ) cycle
                     if ( ikind(i) .gt. ikind_xyz(2) ) cycle
                     xa = x(i,j) * bohr2ang
                     ya = y(i,j) * bohr2ang
                     za = z(i,j) * bohr2ang
                     xb = x(i,j)
                     yb = y(i,j)
                     zb = z(i,j)
                     m1 = mbox(1,i,j)
                     m2 = mbox(2,i,j)
                     m3 = mbox(3,i,j)
                     call pbc_unfold( xb, yb, zb, m1, m2, m3 )
                     xb = xb * bohr2ang
                     yb = yb * bohr2ang
                     zb = zb * bohr2ang
                     write( iounit_xyz, '(a4,6f16.8)' ) &
     &                  species(i)(1:4), xa, ya, za, xb, yb, zb
                  end do
               end do

               flush( iounit_xyz )

!           ===  the centroid  ===

            else if ( iformat_xyz .eq. 3 ) then

               write( iounit_xyz, '(i8)' ) natom_xyz
               write( iounit_xyz, '(i8)' ) istep
               do i = 1, natom
                  if ( ikind(i) .lt. ikind_xyz(1) ) cycle
                  if ( ikind(i) .gt. ikind_xyz(2) ) cycle
                  xa = ux(i,1) * bohr2ang
                  ya = uy(i,1) * bohr2ang
                  za = uz(i,1) * bohr2ang
                  xb = ux(i,1)
                  yb = uy(i,1)
                  zb = uz(i,1)
                  m1 = 0
                  m2 = 0
                  m3 = 0
                  do j = 1, nbead
                     m1 = m1 + mbox(1,i,j)
                     m2 = m2 + mbox(2,i,j)
                     m3 = m3 + mbox(3,i,j)
                  end do
                  m1 = m1 / nbead
                  m2 = m2 / nbead
                  m3 = m3 / nbead
                  call pbc_unfold( xb, yb, zb, m1, m2, m3 )
                  xb = xb * bohr2ang
                  yb = yb * bohr2ang
                  zb = zb * bohr2ang
                  write( iounit_xyz, '(a4,6f16.8)' ) &
     &               species(i)(1:4), xa, ya, za, xb, yb, zb
               end do

               flush( iounit_xyz )

            end if

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         continue

      end if

      return
      end





!***********************************************************************
      subroutine analysis_box ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   vux, vuy, vuz, box, pres, volume, pres_iso, stress, strain, &
     &   stress_iso, strain_iso, vir, fictmass, iounit_box, iboundary, &
     &   istep, iounit_box, iounit_avg, natom, ensemble

      use analysis_variables, only : &
     &   box_avg, pres_avg, volume_avg, pres_iso_avg, stress_avg, &
     &   strain_avg, stress_iso_avg, strain_iso_avg, iprint_box

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ioption, i

      real(8) :: dstep = 0.d0

!-----------------------------------------------------------------------
!     /*   only for periodic boundary                                 */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         if      ( ( ensemble(1:4) .eq. 'NPT ' ) .or. &
     &             ( ensemble(1:4) .eq. 'NPH ' ) ) then

            volume_avg      = 0.d0
            box_avg(:,:)    = 0.d0
            pres_iso_avg    = 0.d0
            pres_avg(:,:)   = 0.d0

         else if ( ( ensemble(1:4) .eq. 'NTT ' ) .or. &
     &             ( ensemble(1:4) .eq. 'NTH ' ) ) then

            volume_avg      = 0.d0
            box_avg(:,:)    = 0.d0
            pres_iso_avg    = 0.d0
            pres_avg(:,:)   = 0.d0
            stress_avg(:,:) = 0.d0
            strain_avg(:,:) = 0.d0

         else

            volume_avg      = 0.d0
            box_avg(:,:)    = 0.d0
            pres_iso_avg    = 0.d0
            pres_avg(:,:)   = 0.d0

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         if      ( ( ensemble(1:4) .eq. 'NPT ' ) .or. &
     &             ( ensemble(1:4) .eq. 'NPH ' ) ) then

            read ( iounit_avg, * ) dstep
            read ( iounit_avg, * ) volume_avg
            read ( iounit_avg, * ) box_avg(:,:)
            read ( iounit_avg, * ) pres_iso_avg
            read ( iounit_avg, * ) pres_avg(:,:)

         else if ( ( ensemble(1:4) .eq. 'NTT ' ) .or. &
     &             ( ensemble(1:4) .eq. 'NTH ' ) ) then

            read ( iounit_avg, * ) dstep
            read ( iounit_avg, * ) volume_avg
            read ( iounit_avg, * ) box_avg(:,:)
            read ( iounit_avg, * ) pres_iso_avg
            read ( iounit_avg, * ) pres_avg(:,:)
            read ( iounit_avg, * ) stress_avg(:,:)
            read ( iounit_avg, * ) strain_avg(:,:)

         else

            read ( iounit_avg, * ) dstep
            read ( iounit_avg, * ) volume_avg
            read ( iounit_avg, * ) box_avg(:,:)
            read ( iounit_avg, * ) pres_iso_avg
            read ( iounit_avg, * ) pres_avg(:,:)

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

!        /*   number of steps   */

         dstep = dstep + 1.d0

         if      ( ( ensemble(1:4) .eq. 'NPT ' ) .or. &
     &             ( ensemble(1:4) .eq. 'NPH ' ) ) then

!           /*   average volume   */

            volume_avg  = volume /dstep &
     &                  + volume_avg*(dstep-1.d0)/dstep

!           /*   average box   */

            box_avg(:,:)  = box(:,:) /dstep &
     &                    + box_avg(:,:)*(dstep-1.d0)/dstep

!           /*   average volume   */

            pres_iso_avg  = pres_iso /dstep &
     &                    + pres_iso_avg*(dstep-1.d0)/dstep

!           /*   average pressure   */

            pres_avg(:,:)  = pres(:,:) /dstep &
     &                     + pres_avg(:,:)*(dstep-1.d0)/dstep

         else if ( ( ensemble(1:4) .eq. 'NTT ' ) .or. &
     &             ( ensemble(1:4) .eq. 'NTH ' ) ) then

!           /*   average volume   */

            volume_avg  = volume /dstep &
     &                  + volume_avg*(dstep-1.d0)/dstep

!           /*   average box   */

            box_avg(:,:)  = box(:,:) /dstep &
     &                    + box_avg(:,:)*(dstep-1.d0)/dstep

!           /*   average volume   */

            pres_iso_avg  = pres_iso /dstep &
     &                    + pres_iso_avg*(dstep-1.d0)/dstep

!           /*   average pressure   */

            pres_avg(:,:)  = pres(:,:) /dstep &
     &                     + pres_avg(:,:)*(dstep-1.d0)/dstep

!           /*   average stress   */

            stress_avg(:,:)  = stress(:,:) /dstep &
     &                       + stress_avg(:,:)*(dstep-1.d0)/dstep

!           /*   average strain   */

            strain_avg(:,:)  = strain(:,:) /dstep &
     &                       + strain_avg(:,:)*(dstep-1.d0)/dstep

!           /*   isotropic strain   */

            strain_iso &
     &         = ( strain(1,1) &
     &           + strain(2,2) &
     &           + strain(3,3) ) / 3.d0

!           /*   isotropic stress   */

            stress_iso &
     &         = ( stress(1,1) &
     &           + stress(2,2) &
     &           + stress(3,3) ) / 3.d0

!           /*   average isotropic strain   */

            strain_iso_avg &
     &         = ( strain_avg(1,1) &
     &           + strain_avg(2,2) &
     &           + strain_avg(3,3) ) / 3.d0

!           /*   average isotropic stress   */

            stress_iso_avg &
     &         = ( stress_avg(1,1) &
     &           + stress_avg(2,2) &
     &           + stress_avg(3,3) ) / 3.d0

         else

!           /*   calculate internal pressure   */

            pres(:,:) =  vir(:,:)

            do i = 1, natom

               pres(1,1) = pres(1,1) + fictmass(i,1)*vux(i,1)*vux(i,1)
               pres(1,2) = pres(1,2) + fictmass(i,1)*vux(i,1)*vuy(i,1)
               pres(1,3) = pres(1,3) + fictmass(i,1)*vux(i,1)*vuz(i,1)
               pres(2,1) = pres(2,1) + fictmass(i,1)*vuy(i,1)*vux(i,1)
               pres(2,2) = pres(2,2) + fictmass(i,1)*vuy(i,1)*vuy(i,1)
               pres(2,3) = pres(2,3) + fictmass(i,1)*vuy(i,1)*vuz(i,1)
               pres(3,1) = pres(3,1) + fictmass(i,1)*vuz(i,1)*vux(i,1)
               pres(3,2) = pres(3,2) + fictmass(i,1)*vuz(i,1)*vuy(i,1)
               pres(3,3) = pres(3,3) + fictmass(i,1)*vuz(i,1)*vuz(i,1)

            end do

            pres(:,:) = pres(:,:) / volume

!           /*   isotropic pressure   */

            pres_iso = ( pres(1,1) + pres(2,2) + pres(3,3) ) / 3.d0

!           /*   average volume   */

            volume_avg  = volume /dstep &
     &                  + volume_avg*(dstep-1.d0)/dstep

!           /*   average box   */

            box_avg(:,:)  = box(:,:) /dstep &
     &                    + box_avg(:,:)*(dstep-1.d0)/dstep

!           /*   average volume   */

            pres_iso_avg  = pres_iso /dstep &
     &                    + pres_iso_avg*(dstep-1.d0)/dstep

!           /*   average pressure   */

            pres_avg(:,:)  = pres(:,:) /dstep &
     &                     + pres_avg(:,:)*(dstep-1.d0)/dstep

         end if

         if ( iprint_box .gt. 0 ) then
         if ( mod(istep,iprint_box) .eq. 0 ) then

            open ( iounit_box, file = 'box.out', access = 'append' )

!           /*   output   */

            if     ( ( ensemble(1:4) .eq. 'NPH ' ) .or. &
     &               ( ensemble(1:4) .eq. 'NPT ' ) ) then

               write( iounit_box, '(i8,2f16.2)' ) &
     &            istep, volume, volume_avg
               write( iounit_box, '(8x,6f16.6)' ) &
     &                   box(1,1), box(1,2), box(1,3), &
     &                   box_avg(1,1), box_avg(1,2), box_avg(1,3)
               write( iounit_box, '(8x,6f16.6)' ) &
     &                   box(2,1), box(2,2), box(2,3), &
     &                   box_avg(2,1), box_avg(2,2), box_avg(2,3)
               write( iounit_box, '(8x,6f16.6)' ) &
     &                   box(3,1), box(3,2), box(3,3), &
     &                   box_avg(3,1), box_avg(3,2), box_avg(3,3)

               write( iounit_box, '(i8,2f16.12)' ) &
     &            istep, pres_iso, pres_iso_avg
               write( iounit_box, '(8x,6f16.12)' ) &
     &                   pres(1,1), pres(1,2), pres(1,3), &
     &                   pres_avg(1,1), pres_avg(1,2), pres_avg(1,3)
               write( iounit_box, '(8x,6f16.12)' ) &
     &                   pres(2,1), pres(2,2), pres(2,3), &
     &                   pres_avg(2,1), pres_avg(2,2), pres_avg(2,3)
               write( iounit_box, '(8x,6f16.12)' ) &
     &                   pres(3,1), pres(3,2), pres(3,3), &
     &                   pres_avg(3,1), pres_avg(3,2), pres_avg(3,3)

            else if ( ( ensemble(1:4) .eq. 'NTH ' ) .or. &
     &                ( ensemble(1:4) .eq. 'NTT ' ) ) then

               write( iounit_box, '(i8,2f16.2)' ) &
     &            istep, volume, volume_avg
               write( iounit_box, '(8x,6f16.6)' ) &
     &                   box(1,1), box(1,2), box(1,3), &
     &                   box_avg(1,1), box_avg(1,2), box_avg(1,3)
               write( iounit_box, '(8x,6f16.6)' ) &
     &                   box(2,1), box(2,2), box(2,3), &
     &                   box_avg(2,1), box_avg(2,2), box_avg(2,3)
               write( iounit_box, '(8x,6f16.6)' ) &
     &                   box(3,1), box(3,2), box(3,3), &
     &                   box_avg(3,1), box_avg(3,2), box_avg(3,3)

               write( iounit_box, '(i8,2f16.12)' ) &
     &            istep, pres_iso, pres_iso_avg
               write( iounit_box, '(8x,6f16.12)' ) &
     &                   pres(1,1), pres(1,2), pres(1,3), &
     &                   pres_avg(1,1), pres_avg(1,2), pres_avg(1,3)
               write( iounit_box, '(8x,6f16.12)' ) &
     &                   pres(2,1), pres(2,2), pres(2,3), &
     &                   pres_avg(2,1), pres_avg(2,2), pres_avg(2,3)
               write( iounit_box, '(8x,6f16.12)' ) &
     &                   pres(3,1), pres(3,2), pres(3,3), &
     &                   pres_avg(3,1), pres_avg(3,2), pres_avg(3,3)

               write( iounit_box, '(i8,2f16.12)' ) &
     &            istep, stress_iso, stress_iso_avg
               write( iounit_box, '(8x,6f16.12)' ) &
     &               stress(1,1), stress(1,2), stress(1,3), &
     &               stress_avg(1,1), stress_avg(1,2), stress_avg(1,3)
               write( iounit_box, '(8x,6f16.12)' ) &
     &               stress(2,1), stress(2,2), stress(2,3), &
     &               stress_avg(2,1), stress_avg(2,2), stress_avg(2,3)
               write( iounit_box, '(8x,6f16.12)' ) &
     &               stress(3,1), stress(3,2), stress(3,3), &
     &               stress_avg(3,1), stress_avg(3,2), stress_avg(3,3)

               write( iounit_box, '(i8,2f16.10)' ) &
     &            istep, strain_iso, strain_iso_avg
               write( iounit_box, '(8x,6f16.10)' ) &
     &               strain(1,1), strain(1,2), strain(1,3), &
     &               strain_avg(1,1), strain_avg(1,2), strain_avg(1,3)
               write( iounit_box, '(8x,6f16.10)' ) &
     &               strain(2,1), strain(2,2), strain(2,3), &
     &               strain_avg(2,1), strain_avg(2,2), strain_avg(2,3)
               write( iounit_box, '(8x,6f16.10)' ) &
     &               strain(3,1), strain(3,2), strain(3,3), &
     &               strain_avg(3,1), strain_avg(3,2), strain_avg(3,3)

            else

               write( iounit_box, '(i8,2f16.2)' ) &
     &            istep, volume, volume_avg
               write( iounit_box, '(8x,6f16.6)' ) &
     &                   box(1,1), box(1,2), box(1,3), &
     &                   box_avg(1,1), box_avg(1,2), box_avg(1,3)
               write( iounit_box, '(8x,6f16.6)' ) &
     &                   box(2,1), box(2,2), box(2,3), &
     &                   box_avg(2,1), box_avg(2,2), box_avg(2,3)
               write( iounit_box, '(8x,6f16.6)' ) &
     &                   box(3,1), box(3,2), box(3,3), &
     &                   box_avg(3,1), box_avg(3,2), box_avg(3,3)

               write( iounit_box, '(i8,2f16.12)' ) &
     &            istep, pres_iso, pres_iso_avg
               write( iounit_box, '(8x,6f16.12)' ) &
     &                   pres(1,1), pres(1,2), pres(1,3), &
     &                   pres_avg(1,1), pres_avg(1,2), pres_avg(1,3)
               write( iounit_box, '(8x,6f16.12)' ) &
     &                   pres(2,1), pres(2,2), pres(2,3), &
     &                   pres_avg(2,1), pres_avg(2,2), pres_avg(2,3)
               write( iounit_box, '(8x,6f16.12)' ) &
     &                   pres(3,1), pres(3,2), pres(3,3), &
     &                   pres_avg(3,1), pres_avg(3,2), pres_avg(3,3)

            end if


            close( iounit_box )

         end if
         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         if      ( ( ensemble(1:4) .eq. 'NPT ' ) .or. &
     &             ( ensemble(1:4) .eq. 'NPH ' ) ) then

            write( iounit_avg, * ) dstep
            write( iounit_avg, * ) volume_avg
            write( iounit_avg, * ) box_avg(:,:)
            write( iounit_avg, * ) pres_iso_avg
            write( iounit_avg, * ) pres_avg(:,:)

         else if ( ( ensemble(1:4) .eq. 'NTT ' ) .or. &
     &             ( ensemble(1:4) .eq. 'NTH ' ) ) then

            write( iounit_avg, * ) dstep
            write( iounit_avg, * ) volume_avg
            write( iounit_avg, * ) box_avg(:,:)
            write( iounit_avg, * ) pres_iso_avg
            write( iounit_avg, * ) pres_avg(:,:)
            write( iounit_avg, * ) stress_avg(:,:)
            write( iounit_avg, * ) strain_avg(:,:)

         else

            write( iounit_avg, * ) dstep
            write( iounit_avg, * ) volume_avg
            write( iounit_avg, * ) box_avg(:,:)
            write( iounit_avg, * ) pres_iso_avg
            write( iounit_avg, * ) pres_avg(:,:)

         end if

      end if

      return
      end





!***********************************************************************
      subroutine analysis_cons_cent ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, iounit_cons, iounit_avg, iounit, nbead, istep, method

      use analysis_variables, only : &
     &   iprint_cons

      use cons_variables, only : &
     &   rcons, scons, fref_cons, scons_avg, fref_cons_avg, &
     &   ncons, ipbc_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption, j, k

      real(8) :: dr, ds

      real(8), save :: dstep = 0.d0

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( ncons .le. 0 ) return
      if ( method(1:6) .eq. 'GEOOPT' ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         scons_avg(:,:)            = 0.d0
         fref_cons_avg(:,:)        = 0.d0

         dstep = 0.d0

         if ( iprint_cons .gt. 0 ) then
         if ( iset .eq. 0 ) then

         open( iounit_cons, file = 'cons.out' )

         write(iounit_cons,'(a)') &
     &   '========================================================' // &
     &   '===================================='
         write(iounit_cons,'(a)') &
     &   '    step       ideal      actual      force    actual_avg' // &
     &   '   force_avg   dist(CN)    dist_avg'
         write(iounit_cons,'(a)') &
     &   '--------------------------------------------------------' // &
     &   '------------------------------------'

         close( iounit_cons )

         iset = 1

         end if
         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         read ( iounit_avg, * ) dstep

         do i = 1, ncons
            read ( iounit_avg, * ) scons_avg(i,1)
         end do

         do i = 1, ncons
            read ( iounit_avg, * ) fref_cons_avg(i,1)
         end do

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         dstep = dstep + 1.d0

!-----------------------------------------------------------------------
!        /*   accumulative averages                                   */
!-----------------------------------------------------------------------

         do j = 1, ncons

            ds = 0.d0

            do i = 1, nbead

               dr = scons(j,i) - rcons(j,i)

               k = ipbc_cons(j)

               call pbc_cons( dr, k )

               dr = dr + rcons(j,i)

               ds = ds + dr

            end do

            ds = ds / dble(nbead)

            scons_avg(j,1) = ds / dstep &
     &         + scons_avg(j,1) * (dstep-1.d0)/dstep

         end do

!-----------------------------------------------------------------------
!        /*   accumulative averages                                   */
!-----------------------------------------------------------------------

         do j = 1, ncons

            dr = 0.d0

            do i = 1, nbead

               dr = dr + fref_cons(j,i)

            end do

            fref_cons_avg(j,1) = dr / dstep &
     &         + fref_cons_avg(j,1) * (dstep-1.d0)/dstep

         end do

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         if ( iprint_cons .gt. 0 ) then
         if ( mod(istep,iprint_cons) .eq. 0 ) then

            open ( iounit_cons, file = 'cons.out', access = 'append' )

            do j = 1, nbead
            do i = 1, ncons
               write( iounit_cons, '(i8,7f12.6)' ) &
     &            istep, rcons(i,j), scons(i,j), &
     &            fref_cons(i,j)*dble(nbead), &
     &            scons_avg(i,1), fref_cons_avg(i,1)
            end do
            end do

            close( iounit_cons )

         end if
         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         write( iounit_avg, '(d24.16)' ) dstep

         do i = 1, ncons
            write( iounit_avg, '(d24.16)' ) scons_avg(i,1)
         end do

         do i = 1, ncons
            write( iounit_avg, '(d24.16)' ) fref_cons_avg(i,1)
         end do

      end if

      return
      end





!***********************************************************************
      subroutine analysis_akin ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, temperature, x, y, z, ux, uy, uz, fx, fy, fz, &
     &   nbead, natom, istep, iounit_avg, iounit

      use analysis_variables, only : &
     &   akin, akin_avg, iprint_akin

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ioption, i, j

      real(8), save :: dstep = 0.d0

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_akin .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         allocate( akin(natom) )
         allocate( akin_avg(natom) )

         akin_avg(:) = 0.d0

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         allocate( akin(natom) )
         allocate( akin_avg(natom) )

         read ( iounit_avg, * ) dstep
         do i = 1, natom
            read ( iounit_avg, * ) akin_avg(i)
         end do

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         dstep = dstep + 1.d0

!-----------------------------------------------------------------------
!        /*   kinetic energy by virial estimator                      */
!-----------------------------------------------------------------------

         do i = 1, natom
            akin(i) = 1.5d0*boltz*temperature
         end do

         do j = 1, nbead
         do i = 1, natom
            akin(i) = akin(i) - 0.5d0*(x(i,j)-ux(i,1))*fx(i,j) &
     &                        - 0.5d0*(y(i,j)-uy(i,1))*fy(i,j) &
     &                        - 0.5d0*(z(i,j)-uz(i,1))*fz(i,j)
         end do
         end do

!-----------------------------------------------------------------------
!        /*   accumulative averages                                   */
!-----------------------------------------------------------------------

         akin_avg(:) = akin(:)/dstep + akin_avg(:)*(dstep-1.d0)/dstep

!-----------------------------------------------------------------------
!        /*   output                                                  */
!-----------------------------------------------------------------------

         if ( mod(istep,iprint_akin) .eq. 0 ) then

            open ( iounit, file='akin.out', access='append' )

            do i = 1, natom
               write( iounit, '(i8,2f16.8)' ) &
     &            istep, akin(i), akin_avg(i)
            end do

            close( iounit )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         write( iounit_avg, '(e24.16)' ) dstep
         do i = 1, natom
            write( iounit_avg, '(e24.16)' ) akin_avg(i)
         end do

      end if

      return
      end
