!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, H. Kimizuka
!      Last updated:    Apr 28, 2019 by M. Shiga
!      Description:     polymers alignment by string method
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine align_polymers_atom_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : physmass, natom

      use polymers_variables, only : xc_poly, yc_poly, zc_poly, &
     &    arc_ref, narc, npoly

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   arc length   */
      real(8) :: arc

!     /*   paramter s from reactant (0) to product (1)   */
      real(8) :: s

!     /*   real numbers   */
      real(8) :: dx, dy, dz, factor, sum

!     /*   integers   */
      integer :: i, j, k

!     /*   mass weighted coordinates   */
      real(8), dimension(3*natom) :: r1, r2

!-----------------------------------------------------------------------
!     /*   initialize spline - s:r                                    */
!-----------------------------------------------------------------------

      call spline_init_crd_poly_atom

!-----------------------------------------------------------------------
!     /*   calculate arc length                                       */
!-----------------------------------------------------------------------

!     /*   arc length at the reactant is zero   */
      arc_ref(1,1) = 0.d0

!     /*   parameter s at the reactant is zero  */
      s = 0.d0

!     /*   get old mass weighted coordinates   */
      call spline_crd_poly_atom( r1, s )

!     /*   scan from the reactant to the product   */
      do j = 2, narc

!        /*   parameter s   */
         s = dble(j-1) / dble(narc-1)

!        /*   get new mass weighted coordinates   */
         call spline_crd_poly_atom( r2, s )

!        /*   calculate arc length by linear approximation   */

         k = 0

         sum = 0.d0

         do i = 1, natom

            k = k + 1

            dx = r2(k) - r1(k)

            k = k + 1

            dy = r2(k) - r1(k)

            k = k + 1

            dz = r2(k) - r1(k)

            sum = sum + dx*dx + dy*dy + dz*dz

         end do

!        /*   accumulate arc length from the reactant   */
         arc_ref(j,1) = arc_ref(j-1,1) + sqrt(sum)

!        /*   save old mass weighted coordinates */
         r1(:) = r2(:)

      end do

!-----------------------------------------------------------------------
!     /*   initialize spline - arc:s                                  */
!-----------------------------------------------------------------------

      call spline_init_arc_poly

!-----------------------------------------------------------------------
!     /*   redistribute polymer                                       */
!-----------------------------------------------------------------------

      do j = 2, npoly-1

!        /*  uniform distribution of arc length from the reactant   */
         arc = arc_ref(narc,1) * dble(j-1) / dble(npoly-1)

!        /*   get parameter s  */
         call spline_arc_poly( s, arc )

!        /*   get mass weighted coordinates  */
         call spline_crd_poly_atom( r1, s )

!        /*   cartesian coordinates  */

         k = 0

         do i = 1, natom

            factor = 1.d0 / sqrt( physmass(i) )

            k = k + 1

            xc_poly(i,j) = r1(k) * factor

            k = k + 1

            yc_poly(i,j) = r1(k) * factor

            k = k + 1

            zc_poly(i,j) = r1(k) * factor

         end do

      end do

      return
      end





!***********************************************************************
      subroutine spline_init_crd_poly_atom
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : physmass, natom

      use polymers_variables, only : xc_poly, yc_poly, zc_poly, &
     &    s_ref, r1_ref, r2_ref, npoly

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j, k

!     /*   real number   */
      real(8) :: factor

!-----------------------------------------------------------------------
!     /*   reference data                                             */
!-----------------------------------------------------------------------

      do j = 1, npoly

         k = 0

         do i = 1, natom

            factor = sqrt(physmass(i))

            k = k + 1

            s_ref(j,k)  = dble(j-1) / dble(npoly-1)
            r1_ref(j,k) = factor * xc_poly(i,j)

            k = k + 1

            s_ref(j,k)  = dble(j-1) / dble(npoly-1)
            r1_ref(j,k) = factor * yc_poly(i,j)

            k = k + 1

            s_ref(j,k)  = dble(j-1) / dble(npoly-1)
            r1_ref(j,k) = factor * zc_poly(i,j)

         end do

      end do

!-----------------------------------------------------------------------
!     /*   initialize spline                                          */
!-----------------------------------------------------------------------

      call spline_init_poly( s_ref, r1_ref, r2_ref, npoly, 3*natom )

      return
      end





!***********************************************************************
      subroutine spline_init_frc_poly_atom
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : natom

      use polymers_variables, only : fxc_poly, fyc_poly, fzc_poly, &
     &    s_ref, f1_ref, f2_ref, npoly

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j, k

!-----------------------------------------------------------------------
!     /*   reference data                                             */
!-----------------------------------------------------------------------

      do j = 1, npoly

         k = 0

         do i = 1, natom

            k = k + 1

            f1_ref(j,k) = fxc_poly(i,j)

            k = k + 1

            f1_ref(j,k) = fyc_poly(i,j)

            k = k + 1

            f1_ref(j,k) = fzc_poly(i,j)

         end do

      end do

!-----------------------------------------------------------------------
!     /*   initialize spline                                          */
!-----------------------------------------------------------------------

      call spline_init_poly( s_ref, f1_ref, f2_ref, npoly, 3*natom )

      return
      end





!***********************************************************************
      subroutine spline_crd_poly_atom( r, s )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : natom

      use polymers_variables, only : s_ref, r1_ref, r2_ref, npoly

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, k

!     /*   real values   */
      real(8) :: s

!     /*   mass weighted coordinates   */
      real(8), dimension(3*natom) :: r

!-----------------------------------------------------------------------
!     /*   interpolation                                              */
!-----------------------------------------------------------------------

      do i = 1, 3*natom

         k = spline_locate_func( s_ref(:,i), npoly, s )

         r(i) = spline_string_func &
     &             ( s_ref(:,i), r1_ref(:,i), r2_ref(:,i), s, k )

      end do

      return
      contains
      include 'spline_locate_func.F90'
      include 'spline_string_func.F90'
      end





!***********************************************************************
      subroutine spline_frc_poly_atom( f, s )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : natom

      use polymers_variables, only : s_ref, f1_ref, f2_ref, npoly

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, k

!     /*   real values   */
      real(8) :: s

!     /*   mass weighted coordinates   */
      real(8), dimension(3*natom) :: f

!-----------------------------------------------------------------------
!     /*   interpolation                                              */
!-----------------------------------------------------------------------

      do i = 1, 3*natom

         k = spline_locate_func( s_ref(:,i), npoly, s )

         f(i) = spline_string_func &
     &             ( s_ref(:,i), f1_ref(:,i), f2_ref(:,i), s, k )

      end do

      return
      contains
      include 'spline_locate_func.F90'
      include 'spline_string_func.F90'
      end





!***********************************************************************
      subroutine spline_crd_frc_poly_atom( r, f, s )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : natom

      use polymers_variables, only : s_ref, r1_ref, r2_ref, &
     &                                      f1_ref, f2_ref, npoly

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, k

!     /*   real values   */
      real(8) :: s

!     /*   mass weighted coordinates and forces  */
      real(8), dimension(3*natom) :: r, f

!-----------------------------------------------------------------------
!     /*   interpolation                                              */
!-----------------------------------------------------------------------

      do i = 1, 3*natom

         k = spline_locate_func( s_ref(:,i), npoly, s )

         r(i) = spline_string_func &
     &             ( s_ref(:,i), r1_ref(:,i), r2_ref(:,i), s, k )

         f(i) = spline_string_func &
     &             ( s_ref(:,i), f1_ref(:,i), f2_ref(:,i), s, k )

      end do

      return
      contains
      include 'spline_locate_func.F90'
      include 'spline_string_func.F90'
      end





!***********************************************************************
      subroutine align_polymers_cons_MPI( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : ncons
      use afed_variables, only : fictmass_afed

      use polymers_variables, only : &
     &   rcons_poly, scons_poly, arc_ref, narc, npoly

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   arc length   */
      real(8) :: arc

!     /*   paramter s from reactant (0) to product (1)   */
      real(8) :: s

!     /*   real numbers   */
      real(8) :: sum

!     /*   integers   */
      integer :: i, j

!     /*   mass weighted coordinates   */
      real(8), dimension(ncons) :: r1, r2

!     /*   integers   */
      integer :: ioption

!-----------------------------------------------------------------------
!     /*   initialize spline - s:r                                    */
!-----------------------------------------------------------------------

      call spline_init_crd_poly_cons( 0 )

!-----------------------------------------------------------------------
!     /*   calculate arc length                                       */
!-----------------------------------------------------------------------

!     /*   arc length at the reactant is zero   */
      arc_ref(1,1) = 0.d0

!     /*   parameter s at the reactant is zero  */
      s = 0.d0

!     /*   get old mass weighted coordinates   */
      call spline_crd_poly_cons( r1, s )

!     /*   scan from the reactant to the product   */
      do j = 2, narc

!        /*   parameter s   */
         s = dble(j-1) / dble(narc-1)

!        /*   get new mass weighted coordinates   */
         call spline_crd_poly_cons( r2, s )

!        /*   calculate arc length by linear approximation   */
         sum = 0.d0
         do i = 1, ncons
            sum = sum + ( r2(i) - r1(i) ) * ( r2(i) - r1(i) )
         end do

!        /*   accumulate arc length from the reactant   */
         arc_ref(j,1) = arc_ref(j-1,1) + sqrt(sum)

!        /*   save old mass weighted coordinates */
         r1(:) = r2(:)

      end do

!-----------------------------------------------------------------------
!     /*   initialize spline - arc:s                                  */
!-----------------------------------------------------------------------

      call spline_init_arc_poly

!-----------------------------------------------------------------------
!     /*   redistribute polymer                                       */
!-----------------------------------------------------------------------

      do j = 2, npoly-1

!        /*  uniform distribution of arc length from the reactant   */
         arc = arc_ref(narc,1) * dble(j-1) / dble(npoly-1)

!        /*   get parameter s  */
         call spline_arc_poly( s, arc )

!        /*   get mass weighted coordinates  */
         call spline_crd_poly_cons( r1, s )

!        /*   cartesian coordinates  */
         if ( ioption .eq. 0 ) then
            do i = 1, ncons
               rcons_poly(i,j) = r1(i) / sqrt( fictmass_afed(i) )
            end do
         else
            do i = 1, ncons
               scons_poly(i,j) = r1(i) / sqrt( fictmass_afed(i) )
            end do
         end if

      end do

      return
      end





!***********************************************************************
      subroutine spline_init_crd_poly_cons( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : ncons
      use afed_variables, only : fictmass_afed

      use polymers_variables, only : &
     &   rcons_poly, scons_poly, s_ref, r1_ref, r2_ref, npoly

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j

!     /*   integers   */
      integer :: ioption

!-----------------------------------------------------------------------
!     /*   reference data                                             */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         do j = 1, npoly
         do i = 1, ncons
            s_ref(j,i)  = dble(j-1) / dble(npoly-1)
            r1_ref(j,i) = sqrt(fictmass_afed(i)) * rcons_poly(i,j)
         end do
         end do

      else

         do j = 1, npoly
         do i = 1, ncons
            s_ref(j,i)  = dble(j-1) / dble(npoly-1)
            r1_ref(j,i) = sqrt(fictmass_afed(i)) * scons_poly(i,j)
         end do
         end do

      end if

!-----------------------------------------------------------------------
!     /*   initialize spline                                          */
!-----------------------------------------------------------------------

      call spline_init_poly( s_ref, r1_ref, r2_ref, npoly, ncons )

      return
      end





!***********************************************************************
      subroutine spline_init_frc_poly_cons
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : ncons

      use polymers_variables, only : &
     &   fcons_poly, s_ref, f1_ref, f2_ref, npoly

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j

!-----------------------------------------------------------------------
!     /*   reference data                                             */
!-----------------------------------------------------------------------

      do j = 1, npoly
      do i = 1, ncons
         f1_ref(j,i) = fcons_poly(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   initialize spline                                          */
!-----------------------------------------------------------------------

      call spline_init_poly( s_ref, f1_ref, f2_ref, npoly, ncons )

      return
      end





!***********************************************************************
      subroutine spline_crd_poly_cons( r, s )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : ncons

      use polymers_variables, only : s_ref, r1_ref, r2_ref, npoly

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, k

!     /*   real values   */
      real(8) :: s

!     /*   mass weighted coordinates   */
      real(8), dimension(ncons) :: r

!-----------------------------------------------------------------------
!     /*   interpolation                                              */
!-----------------------------------------------------------------------

      do i = 1, ncons

         k = spline_locate_func( s_ref(:,i), npoly, s )

         r(i) = spline_string_func &
     &             ( s_ref(:,i), r1_ref(:,i), r2_ref(:,i), s, k )

      end do

      return
      contains
      include 'spline_locate_func.F90'
      include 'spline_string_func.F90'
      end





!***********************************************************************
      subroutine spline_frc_poly_cons( f, s )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : ncons

      use polymers_variables, only : s_ref, f1_ref, f2_ref, npoly

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, k

!     /*   real values   */
      real(8) :: s

!     /*   mass weighted coordinates   */
      real(8), dimension(ncons) :: f

!-----------------------------------------------------------------------
!     /*   interpolation                                              */
!-----------------------------------------------------------------------

      do i = 1, ncons

         k = spline_locate_func( s_ref(:,i), npoly, s )

         f(i) = spline_string_func &
     &             ( s_ref(:,i), f1_ref(:,i), f2_ref(:,i), s, k )

      end do

      return
      contains
      include 'spline_locate_func.F90'
      include 'spline_string_func.F90'
      end





!***********************************************************************
      subroutine spline_crd_frc_poly_cons( r, f, s )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : ncons

      use polymers_variables, only : s_ref, r1_ref, r2_ref, &
     &                                      f1_ref, f2_ref, npoly

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, k

!     /*   real values   */
      real(8) :: s

!     /*   mass weighted coordinates and forces  */
      real(8), dimension(ncons) :: r, f

!-----------------------------------------------------------------------
!     /*   interpolation                                              */
!-----------------------------------------------------------------------

      do i = 1, ncons

         k = spline_locate_func( s_ref(:,i), npoly, s )

         r(i) = spline_string_func &
     &             ( s_ref(:,i), r1_ref(:,i), r2_ref(:,i), s, k )

         f(i) = spline_string_func &
     &             ( s_ref(:,i), f1_ref(:,i), f2_ref(:,i), s, k )

      end do

      return
      contains
      include 'spline_locate_func.F90'
      include 'spline_string_func.F90'
      end





!***********************************************************************
      subroutine spline_init_arc_poly
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use polymers_variables, only : arc_ref, s1_ref, s2_ref, narc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: j

!-----------------------------------------------------------------------
!     /*   reference data                                             */
!-----------------------------------------------------------------------

      do j = 1, narc

         s1_ref(j,1) = dble(j-1) / dble(narc-1)

      end do

!-----------------------------------------------------------------------
!     /*   initialize spline                                          */
!-----------------------------------------------------------------------

      call spline_init_poly( arc_ref, s1_ref, s2_ref, narc, 1 )

      return
      end





!***********************************************************************
      subroutine spline_arc_poly( s, arc )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use polymers_variables, only : arc_ref, s1_ref, s2_ref, narc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: k

!     /*   real values   */
      real(8) :: s, arc

!-----------------------------------------------------------------------
!     /*   interpolation                                              */
!-----------------------------------------------------------------------

      k = spline_locate_func( arc_ref(:,1), narc, arc )

      s = spline_string_func &
     &       ( arc_ref(:,1), s1_ref(:,1), s2_ref(:,1), arc, k )

      return
      contains
      include 'spline_locate_func.F90'
      include 'spline_string_func.F90'
      end





!***********************************************************************
      subroutine spline_init_poly ( x, y, y2, n, m )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, n, m, k

      real(8) :: yp1, ypn, sig, p, qn, un

      real(8) :: x(n,m), y(n,m), y2(n,m), u(n)

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do j = 1, m

         yp1 = (y(2,j)-y(1,j))/(x(2,j)-x(1,j))
         ypn = (y(n,j)-y(n-1,j))/(x(n,j)-x(n-1,j))

         y2(1,j) = -0.5d0
         u(1) = (3.d0/(x(2,j)-x(1,j)))*((y(2,j)-y(1,j)) &
     &           /(x(2,j)-x(1,j))-yp1)

         do i = 2, n-1
            sig = (x(i,j)-x(i-1,j))/(x(i+1,j)-x(i-1,j))
            p = sig*y2(i-1,j)+2.d0
            y2(i,j) = (sig-1.d0)/p
            u(i) = (6.d0*((y(i+1,j)-y(i,j)) &
     &              /(x(i+1,j)-x(i,j))-(y(i,j)-y(i-1,j)) &
     &              /(x(i,j)-x(i-1,j)))/(x(i+1,j)-x(i-1,j)) &
     &              -sig*u(i-1))/p
         end do

         qn = 0.5d0
         un = (3.d0/(x(n,j)-x(n-1,j)))*(ypn-(y(n,j)-y(n-1,j)) &
     &       /(x(n,j)-x(n-1,j)))

         y2(n,j) = (un-qn*u(n-1))/(qn*y2(n-1,j)+1.d0)

         do k = n-1, 1, -1
            y2(k,j) = y2(k,j)*y2(k+1,j) + u(k)
         end do

      end do

      return
      end





!***********************************************************************
      subroutine spline_poly( xa, ya, y2a, n, x, y, i, m )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: klo, khi, k, n, m, i

      real(8) :: x, h, a, b, y

      real(8) :: xa(n,m), ya(n,m), y2a(n,m)

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      if ( x .lt. xa(1,i) ) then

         klo = 1
         khi = 2

      else if ( x .gt. xa(n,i) ) then

         klo = n-1
         khi = n

      else

         klo = 1
         khi = n

         do

            if ( khi-klo .gt. 1 ) then
               k = (khi+klo)/2
               if ( xa(k,i) .gt. x ) then
                  khi = k
               else
                  klo = k
               end if
            else
               exit
            end if

         end do

      end if

      h = xa(khi,i)-xa(klo,i)

      if ( h .eq. 0.d0 ) then
         write(6,'(a)')
         write(6,'(a)') 'Error termination: bad input in spline.'
         write(6,'(a)')
         call my_mpi_abort
      end if

      a = (xa(khi,i)-x)/h
      b = (x-xa(klo,i))/h

      y = a*ya(klo,i)+b*ya(khi,i) &
     &  + ((a*a*a-a)*y2a(klo,i)+(b*b*b-b)*y2a(khi,i))*(h*h)/6.d0

      return
      end
