!///////////////////////////////////////////////////////////////////////
!
!      Author:          T. Morishita, M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     Logarithmic mean force dynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setup_logmfd_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, pi, iounit, beta, nbead, iounit_logmfd, &
     &   logmfd_type

      use cons_variables, only : &
     &   ncons, itype_cons, ntype_cons

      use afed_variables, only : &
     &   f_afed, e_afed,  f_old_afed, fictmass_afed, scons_mean_afed, &
     &   v_afed, d_old_afed, dt_logmfd, rafed_old, time_bath_afed, &
     &   alpha_logmfd, gamma_logmfd, fenergy_max_afed, v_old_afed, &
     &   rafed, params_afed, d_afed, gnkt_afed, work_afed, weight_afed, &
     &   rbath_afed, vbath_afed, fbath_afed, qmass_afed, gkt_afed, &
     &   fenergy_ini_logmfd, e_old_afed, scons_avg_old_afed, hess_afed, &
     &   he_afed, iconv_afed, nstep_pre_afed, v_save_afed, &
     &   niter_afed, nstep_pro_afed, niter_bath_afed, ioption_afed, &
     &   start_afed, mass_afed, hess_old_afed, start_therm_afed

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: itest, i, k, ierr

      real(8) :: factor_mass

      character(len=8) :: params_char

!-----------------------------------------------------------------------
!     //   convergence
!-----------------------------------------------------------------------

      iconv_afed = 0

!-----------------------------------------------------------------------
!     //   setup constrained molecular dynamics
!-----------------------------------------------------------------------

      call setup_md_nvt

!-----------------------------------------------------------------------
!     //   get collective variables
!-----------------------------------------------------------------------

      call get_cv_cons

!-----------------------------------------------------------------------
!     //   read parameters
!-----------------------------------------------------------------------

!c     //   method
!      call read_char ( afed_type, 8, '<afed_type>', 11, iounit )

!c     //   ensemble
!      call read_char ( logmfd_type, 4, '<logmfd_type>', 13, iounit )

!     //   read number of steps
      call read_int1 ( niter_afed, '<niter_afed>', 12, iounit )

!     //   read number of steps for production run
      call read_int1 &
     &   ( nstep_pro_afed, '<nstep_pro_afed>', 16, iounit )

!     //   read number of steps for preliminary run
      call read_int1 &
     &   ( nstep_pre_afed, '<nstep_pre_afed>', 16, iounit )

!     //   read maximum free energy criterion
      call read_real1 &
     &   ( fenergy_max_afed, '<fenergy_max_afed>', 18, iounit )

!     //   step size for logarithmic mean force dynamics
      call read_real1 &
     &   ( dt_logmfd, '<dt_logmfd>', 11, iounit )

!     //   temperature for logarithmic mean force dynamics
      call read_real1 &
     &   ( alpha_logmfd, '<alpha_logmfd>', 14, iounit)

!     //   temperature for logarithmic mean force dynamics
      call read_real1 &
     &   ( gamma_logmfd, '<gamma_logmfd>', 14, iounit)

!     //   initial conserved quantity for LogMFD
      call read_real1 &
     &   ( fenergy_ini_logmfd, '<fenergy_ini_logmfd>', 20, iounit )

!     //   parameter associated to thermostat masses
      call read_int1 &
     &   ( niter_bath_afed, '<niter_bath_afed>', 17, iounit )

!     //   read mean force option
      call read_int1 ( ioption_afed, '<ioption_afed>', 14, iounit )

!     //   correction of fictmass
      call read_char ( mass_afed, 9, '<mass_afed>', 11, iounit )

!     //   initial values of cv positions and velocities
      call read_char ( start_afed, 9, '<start_afed>', 12, iounit )

!     //   initial values of thermostat positions and velocities for cv
      call read_char  &
     &     ( start_therm_afed, 9, '<start_therm_afed>', 18, iounit )

!-----------------------------------------------------------------------
!     //   default settings of alpha, gamma
!-----------------------------------------------------------------------

      ierr = 0

      if ( ( alpha_logmfd .gt. 0.d0 ) .and. &
     &     ( gamma_logmfd .gt. 0.d0 ) ) then

         alpha_logmfd = alpha_logmfd * beta
         gamma_logmfd = gamma_logmfd / beta

         continue

      else if ( ( alpha_logmfd .gt. 0.d0 ) .and. &
     &          ( gamma_logmfd .lt. 0.d0 ) ) then

         alpha_logmfd = alpha_logmfd * beta
         gamma_logmfd  = 1.d0 / alpha_logmfd

      else

         ierr = 1

         write( 6, '(a)' ) &
     &      'Error - keyword <alpha_logmfd> ' // &
     &      'or <gamma_logmfd> is incorrect.'

      end if

!     /*   check error   */
      call error_handling &
     &     ( ierr, 'subroutine setup_logmfd_afed', 28 )

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

!     //   memory allocation: mean force
      if ( .not. allocated( f_afed ) ) &
     &   allocate( f_afed(ncons) )

!     //   memory allocation: old mean force
      if ( .not. allocated( f_old_afed ) ) &
     &   allocate( f_old_afed(ncons) )

!     //   memory allocation: mean hessian
      if ( .not. allocated( hess_afed ) ) &
     &   allocate( hess_afed(ncons,ncons) )

!     //   memory allocation: mean hessian
      if ( .not. allocated( hess_old_afed ) ) &
     &   allocate( hess_old_afed(ncons,ncons) )

!     //   memory allocation: unit vector
      if ( .not. allocated( e_afed ) ) &
     &   allocate( e_afed(ncons) )

!     //   memory allocation: fictitious mass
      if ( .not. allocated( fictmass_afed ) ) &
     &   allocate( fictmass_afed(ncons) )

!     //   memory allocation: mean cv value
      if ( .not. allocated( scons_mean_afed ) ) &
     &   allocate( scons_mean_afed(ncons) )

!     //   memory allocation: cv
      if ( .not. allocated( rafed ) ) &
     &   allocate( rafed(ncons) )

!     //   memory allocation: parameters
      if ( .not. allocated( params_afed ) ) &
     &   allocate( params_afed(ntype_cons,3) )

!     //   memory allocation: shift vector
      if ( .not. allocated( he_afed ) ) &
     &   allocate( he_afed(ncons) )

!     //   memory allocation: direction of current step
      if ( .not. allocated( d_afed ) ) &
     &   allocate( d_afed(ncons) )

!     //   memory allocation: direction of previous step
      if ( .not. allocated( d_old_afed ) ) &
     &   allocate( d_old_afed(ncons) )

!     //   memory allocation: unit vector of previous step
      if ( .not. allocated( e_old_afed ) ) &
     &   allocate( e_old_afed(ncons) )

!     //   memory allocation: position of previous step
      if ( .not. allocated( rafed_old ) ) &
     &   allocate( rafed_old(ncons) )

!     //   memory allocation: average position of previous step
      if ( .not. allocated( scons_avg_old_afed ) ) &
     &   allocate( scons_avg_old_afed(ncons,nbead) )

!     //   memory allocation: weight parameter of replica
      if ( .not. allocated( weight_afed ) ) &
     &   allocate( weight_afed(nbead) )

!     //   memory allocation: work parameter of replica
      if ( .not. allocated( work_afed ) ) &
     &   allocate( work_afed(nbead) )

!     //   memory allocation: cv velocity of current step
      if ( .not. allocated( v_afed ) ) &
     &   allocate( v_afed(ncons) )

!     //   memory allocation: cv velocity of previous step
      if ( .not. allocated( v_old_afed ) ) &
     &   allocate( v_old_afed(ncons) )

!     //   memory allocation: thermostat position attached to cv
      if ( .not. allocated( rbath_afed ) ) &
     &   allocate( rbath_afed(1) )

!     //   memory allocation: thermostat velocity attached to cv
      if ( .not. allocated( vbath_afed ) ) &
     &   allocate( vbath_afed(1) )

!     //   memory allocation: thermostat force attached to cv
      if ( .not. allocated( fbath_afed ) ) &
     &   allocate( fbath_afed(1) )

!     //   memory allocation: thermostat mass attached to cv
      if ( .not. allocated( qmass_afed ) ) &
     &   allocate( qmass_afed(1) )

!     //   memory allocation: cv velocity of current step
      if ( .not. allocated( v_save_afed ) ) &
     &   allocate( v_save_afed(ncons) )

!-----------------------------------------------------------------------
!     /*   read parameters for constraint                             */
!-----------------------------------------------------------------------

      ierr = 0

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<params_afed>', 13, iounit, ierr )

!     /*   parameters of constraint type   */

      if ( ierr .eq. 0 ) then

         do i = 1, ntype_cons

            read ( iounit, *, iostat=ierr ) params_char

            backspace( iounit )

            if      ( ( params_char(1:6) .eq. '1     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIST  ' ) ) then
               k = 1
            else if ( ( params_char(1:6) .eq. '2     ' ) .or. &
     &                ( params_char(1:6) .eq. 'ANGL  ' ) ) then
               k = 2
            else if ( ( params_char(1:6) .eq. '3     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIH   ' ) ) then
               k = 3
            else if ( ( params_char(1:6) .eq. '4     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIFF  ' ) ) then
               k = 4
            else if ( ( params_char(1:6) .eq. '5     ' ) .or. &
     &                ( params_char(1:6) .eq. 'CN    ' ) ) then
               k = 5
            else if ( ( params_char(1:6) .eq. '6     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DCN   ' ) ) then
               k = 6
            else if ( ( params_char(1:6) .eq. '7     ' ) .or. &
     &                ( params_char(1:6) .eq. 'XYZ   ' ) ) then
               k = 7
            else if ( ( params_char(1:6) .eq. '8     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DXYZ  ' ) ) then
               k = 8
            else
               ierr = 1
            end if

            if ( ierr .ne. 0 ) exit

            read ( iounit, *, iostat=ierr ) &
     &         params_char, params_afed(k,1), params_afed(k,2), &
     &                      params_afed(k,3)

            if ( ierr .ne. 0 ) exit

         end do

      end if

!     /*   file close   */
      close( iounit )

!     /*   if error is found, read default values   */

      if ( ierr .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<params_afed>', 13, iounit, ierr )

!        /*   parameters of constraint type   */

         do i = 1, ntype_cons

            read ( iounit, *, iostat=ierr ) params_char

            backspace( iounit )

            if      ( ( params_char(1:6) .eq. '1     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIST  ' ) ) then
               k = 1
            else if ( ( params_char(1:6) .eq. '2     ' ) .or. &
     &                ( params_char(1:6) .eq. 'ANGL  ' ) ) then
               k = 2
            else if ( ( params_char(1:6) .eq. '3     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIH   ' ) ) then
               k = 3
            else if ( ( params_char(1:6) .eq. '4     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIFF  ' ) ) then
               k = 4
            else if ( ( params_char(1:6) .eq. '5     ' ) .or. &
     &                ( params_char(1:6) .eq. 'CN    ' ) ) then
               k = 5
            else if ( ( params_char(1:6) .eq. '6     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DCN   ' ) ) then
               k = 6
            else if ( ( params_char(1:6) .eq. '7     ' ) .or. &
     &                ( params_char(1:6) .eq. 'XYZ   ' ) ) then
               k = 7
            else if ( ( params_char(1:6) .eq. '8     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DXYZ  ' ) ) then
               k = 8
            else
               ierr = 1
            end if

            if ( ierr .ne. 0 ) exit

            read ( iounit, *, iostat=ierr ) &
     &         params_char, params_afed(k,1), params_afed(k,2), &
     &                      params_afed(k,3)

            if ( ierr .ne. 0 ) exit

         end do

!        /*   file close   */
         close( iounit )

!     /*   end   */
      end if

      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - keyword <params_afed> is incorrect.'
      end if

!     /*   check error   */
      call error_handling  &
     &     ( ierr, 'subroutine setup_logmfd_afed', 28 )

!-----------------------------------------------------------------------
!     /*   fictitious mass                                            */
!-----------------------------------------------------------------------

      do i = 1, ncons

         k = itype_cons(i)

         fictmass_afed(i) = dt_logmfd**2 / beta / params_afed(k,1)**2

      end do

!-----------------------------------------------------------------------
!     /*   fictitious mass with correction                            */
!-----------------------------------------------------------------------

!     /*   manual only   */
      if ( mass_afed(1:7) .eq. 'MANUAL ' ) then

!        /*    file open   */
         open ( iounit, file = 'input.dat' )

!        /*    search for tag    */
         call search_tag ( '<mass_afed>', 11, iounit, ierr )

!        /*    read line   */
         read ( iounit, *, iostat=ierr )

!        /*    read scaling factor of cv fictitious masses   */
         do i = 1, ncons
            read ( iounit, *, iostat=ierr ) factor_mass
            fictmass_afed(i) = fictmass_afed(i) * factor_mass
         end do

!        /*   file close   */
         close( iounit )

!     /*   manual only   */
      end if

!     /*   check error   */
      call error_handling &
     &   ( ierr, 'subroutine setup_logmfd_afed', 28 )

!-----------------------------------------------------------------------
!     /*   gkt, gnkt for AFED                                         */
!-----------------------------------------------------------------------

      gkt_afed = 1.d0 / beta

      gnkt_afed = dble(ncons) / beta

!-----------------------------------------------------------------------
!     /*   thermostat mass                                            */
!-----------------------------------------------------------------------

      qmass_afed(1) = 0.d0

      if ( logmfd_type(1:4) .eq. 'NVT ' ) then
         time_bath_afed = dble(niter_bath_afed) * dt_logmfd
         qmass_afed(1) = gnkt_afed * time_bath_afed**2
      end if

!-----------------------------------------------------------------------
!     //   initialize or restart
!-----------------------------------------------------------------------

!     //   check existence of restart file
      call testfile( 'afed.ini', 8, itest )

!     //   restart file: not found
      if ( itest .eq. 1 ) then

!        //   initialize
         call init_logmfd_afed

!     //   restart file: found
      else

!        //   restart
         call restart_logmfd_afed

!     //   restart file
      end if

      return
      end





!***********************************************************************
      subroutine init_logmfd_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   nbead, logmfd_type, iounit

      use cons_variables, only : &
     &   scons, ncons, ipbc_cons

      use afed_variables, only : &
     &   rafed, dt_afed, fenergy_afed, fenergy_ref_afed, dt_logmfd, &
     &   afed_status, iiter_start_afed, fenergy_ini_logmfd, &
     &   start_afed, v_afed, rbath_afed, vbath_afed, start_therm_afed

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      real(8) :: s, savg

      integer :: i, j, ierr

!-----------------------------------------------------------------------
!     //   set afed_status: status
!-----------------------------------------------------------------------

      afed_status(1:2) = 'LO'

!-----------------------------------------------------------------------
!     //    set iiter_start_afed: step number
!-----------------------------------------------------------------------

      iiter_start_afed = 0

!-----------------------------------------------------------------------
!     //   set free energy
!-----------------------------------------------------------------------

      fenergy_afed = fenergy_ini_logmfd

!-----------------------------------------------------------------------
!     //   set reference free energy
!-----------------------------------------------------------------------

      fenergy_ref_afed = 0.d0

      if ( logmfd_type(1:4) .eq. 'VS  ' ) then

         fenergy_ref_afed = fenergy_ini_logmfd

      end if

!-----------------------------------------------------------------------
!     //   set cv position and velocity
!-----------------------------------------------------------------------

!     /*   manual setting: read from input file   */
      if ( start_afed(1:7) .eq. 'MANUAL ' ) then

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag   */
         call search_tag ( '<start_afed>', 12, iounit, ierr )

!        /*   read initial cv positions   */
         read ( iounit, *, iostat=ierr )

         do i = 1, ncons
            read ( iounit, *, iostat=ierr ) rafed(i), v_afed(i)
         end do

!        /*   default   */
         if ( ierr .ne. 0 ) then

!           /*    file close   */
            close( iounit )

!           /*    file open   */
            open ( iounit, file = 'input_default.dat' )

!           /*   search for tag   */
            call search_tag ( '<start_afed>', 12, iounit, ierr )

!           /*   read initial cv positions   */
            read ( iounit, *, iostat=ierr )

            do i = 1, ncons
               read ( iounit, *, iostat=ierr ) rafed(i), v_afed(i)
            end do

!        /*   default   */
         end if

!        /*   file close   */
         close( iounit )

!        /* check error   */
         call error_handling &
     &        ( ierr, 'subroutine init_logmfd_afed', 27 )

!     /*   default setting: set from geometry   */
      else

!        /*   cv position set to be the center of mass of scons   */
         do i = 1, ncons

            rafed(i) = 0.d0

            savg = 0.d0

            do j = 1, nbead

               s = scons(i,j) - scons(i,1)

               call pbc_cons( s, ipbc_cons(i) )

               s = s + scons(i,1)

               savg = savg + s

            end do

            savg = savg / dble(nbead)

            rafed(i) = rafed(i) + savg

         end do

!        /*   apply periodic boundary condition   */
         call pbc_cv_afed

!        /*   random cv velocity   */
         call init_velocity_tamd_afed

      end if

!-----------------------------------------------------------------------
!     //   set bath
!-----------------------------------------------------------------------

!      call init_bath_tamd_afed

!-----------------------------------------------------------------------
!     //   set dt_afed: afed step size
!-----------------------------------------------------------------------

      dt_afed = dt_logmfd

!-----------------------------------------------------------------------
!     //   set bath and get reference hamiltonian
!-----------------------------------------------------------------------

      if      ( logmfd_type(1:4) .eq. 'NVE ' ) then

         vbath_afed(:) = 0.d0
         rbath_afed(:) = 0.d0

         call getref_logmfd_nve_afed( 0 )

      else if ( logmfd_type(1:4) .eq. 'NVT ' ) then

!        /*    thermostat position and velocity are set manually   */
         if ( start_therm_afed(1:7) .eq. 'MANUAL ' ) then

!           /*   initialize   */
            ierr = 0

!           /*    file open   */
            open ( iounit, file = 'input.dat' )

!           /*    search for tag    */
            call search_tag &
     &         ( '<start_therm_afed>', 18, iounit, ierr )

!           /*    read line   */
            read ( iounit, *, iostat=ierr )

!           /*    set initial cv positions   */
            read ( iounit, *, iostat=ierr ) &
     &        rbath_afed(1), vbath_afed(1)

!           /*   default   */
            if ( ierr .ne. 0 ) then

!              /*    file close   */
               close( iounit )

!              /*    file open   */
               open ( iounit, file = 'input_default.dat' )

!              /*    search for tag    */
               call search_tag &
     &            ( '<start_therm_afed>', 18, iounit, ierr )

!              /*    read line   */
               read ( iounit, *, iostat=ierr )

!              /*    set initial cv positions   */
               read ( iounit, *, iostat=ierr ) &
     &            rbath_afed(1), vbath_afed(1)

!           /*   default   */
            end if

!           /*    file close   */
            close( iounit )

!           /*   check error   */
            call error_handling &
     &           ( ierr, 'subroutine init_logmfd_afed', 27 )

         else

            call init_bath_tamd_afed

         end if

         call getref_logmfd_nvt_afed( 0 )

      else if ( logmfd_type(1:4) .eq. 'VS  ' ) then

         vbath_afed(:) = 0.d0
         rbath_afed(:) = 0.d0

         call getref_logmfd_vs_afed( 0 )

      end if

      return
      end





!***********************************************************************
      subroutine restart_logmfd_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, nbead, logmfd_type

      use afed_variables, only : &
     &   fenergy_afed, rafed, dt_afed, v_afed, fenergy_afed, vbath_afed, &
     &   fenergy_ref_afed, dt_logmfd, fictmass_afed, rbath_afed, &
     &   qmass_afed, iiter_start_afed, afed_status, &
     &   work_afed, hamiltonian_ref_afed, phi_logmfd

      use cons_variables, only : &
     &   ncons

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ierr, ierr2

      real(8) :: p_afed, pbath_afed

!-----------------------------------------------------------------------
!     //   check status
!-----------------------------------------------------------------------

      open ( iounit, file = 'afed.ini' )
      read ( iounit, * ) afed_status
      close( iounit )

!-----------------------------------------------------------------------
!    //   read restart file: LOGMFD
!-----------------------------------------------------------------------

!     //   LOGMFD
      if ( afed_status(1:2) .eq. 'LO' ) then

!        //   error flag
         ierr = 0

!        //   open file
         open ( iounit, file = 'afed.ini' )

!        //   read afed_status: status
         read ( iounit, * ) afed_status

!        //   read iiter_start_afed: afed step number
         read ( iounit, * ) iiter_start_afed

!        //   read free energy
         read ( iounit, * ) fenergy_afed

!        //   read collective variables
         do i = 1, ncons
            read ( iounit, * ) rafed(i), p_afed
            v_afed(i) = p_afed / sqrt( fictmass_afed(i) )
         end do

!        //   read reference hamiltonian
         if ( ( logmfd_type(1:4) .eq. 'NVE ' ) .or.  &
     &        ( logmfd_type(1:4) .eq. 'NVT ' ) ) then
            read ( iounit, * ) hamiltonian_ref_afed
         else if ( logmfd_type(1:4) .eq. 'VS  ' ) then
            read ( iounit, * ) phi_logmfd
         end if

!        //   read reference free energy
         read ( iounit, * ) fenergy_ref_afed

!        //   read bath variables
         if ( logmfd_type(1:4) .eq. 'NVT ' ) then
            read ( iounit, *, iostat=ierr ) rbath_afed(1), pbath_afed
            vbath_afed(1) = pbath_afed / sqrt( qmass_afed(1) )
         end if

!        //   read work parameters of replica
         do i = 1, nbead
            read ( iounit, *, iostat=ierr2 ) work_afed(i)
         end do

         if ( ierr2 .ne. 0 ) work_afed(:) = 0.d0

!        //   close file
         close( iounit )

!        //   initialize bath
         if ( ierr .ne. 0 ) call init_bath_tamd_afed

!     //   not LOGMFD
      else

!        //   open file
         open ( iounit, file = 'afed.ini' )

!        //   read afed_status: status
         read ( iounit, * ) afed_status

!        //   read iiter_start_afed: afed step number
         read ( iounit, * ) iiter_start_afed

!        //   read free energy
         read ( iounit, * ) fenergy_afed

!        //   read collective variables
         do i = 1, ncons
            read ( iounit, * ) rafed(i)
         end do

!        //   read work parameters of replica
         do i = 1, nbead
            read ( iounit, * ) work_afed(i)
         end do

!        //   close file
         close( iounit )

!        //   reset afed_status: status
         afed_status(1:2) = 'LO'

!        //   reset afed step number
         iiter_start_afed = 0

!        //   initialize velocity
         call init_velocity_tamd_afed

!        //   initialize bath
         call init_bath_tamd_afed

!        //   initialize reference free energy
         fenergy_ref_afed = 0.d0

!     //   LOGMFD or not LOGMFD
      end if

!-----------------------------------------------------------------------
!     //   set dt_afed: afed step size
!-----------------------------------------------------------------------

      dt_afed = dt_logmfd

      return
      end





!***********************************************************************
      subroutine afedcycle_logmfd_nve
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iexit

      use afed_variables, only : &
     &   iiter_afed, iiter_start_afed, iiter_end_afed, niter_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

!     //   starting step
      iiter_afed = iiter_start_afed

!     //   current step
      iiter_end_afed = iiter_afed

!c     //   get reference hamiltonian
!      call getref_logmfd_nve_afed( 0 )

!     //   run constrained molecular dynamics and get meanforce
      call meanforce_afed

!     //   stop on soft exit
      if ( iexit .eq. 1 ) return

!     //   gradient vector
      call eupdate_descent_afed

!     //   calculate free energy
      call getref_logmfd_nve_afed( 1 )

!     //   standard output
      call standard_logmfd_afed_nve

!     //   analysis
      call analysis_afed

!     //   save data
      call backup_logmfd_nve_afed

!     //   start iteration loop
      do iiter_afed = iiter_start_afed+1, niter_afed

!        //   current step
         iiter_end_afed = iiter_afed

!        //   save last step
         call save_cv_afed

!        //   move cv in forward direction
         call vupdate_logmfd_afed

!        //   move cv in forward direction
         call rupdate_logmfd_afed

!        //   run constrained molecular dynamics and get meanforce
         call meanforce_afed

!        //   stop on soft exit
         if ( iexit .eq. 1 ) return

!        //   gradient vector
         call eupdate_descent_afed

!        //   calculate free energy
         call getref_logmfd_nve_afed( 1 )

!        //   standard output
         call standard_logmfd_afed_nve

!        //   analysis
         call analysis_afed

!        //   save data
         call backup_logmfd_nve_afed

!     //   end iteration loop
      end do

!     /*   last step   */
      iiter_afed = iiter_end_afed

      return
      end





!***********************************************************************
      subroutine standard_logmfd_afed_nve
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, iounit_afed, iounit_logmfd, nbead

      use cons_variables, only : &
     &   scons_avg, ncons, ipbc_cons, itype_cons

      use afed_variables, only : &
     &   fenergy_afed, f_afed, scons_mean_afed, v_afed, params_afed, &
     &   hamiltonian_afed, ekin_afed, temp_afed, potential_afed, &
     &   rafed, fenergy_max_afed, fictmass_afed, ebath_afed, &
     &   alpha_logmfd, gamma_logmfd, fenergy_ref_afed, &
     &   afed_status, iiter_afed, iconv_afed, iiter_start_afed, &
     &   qmass_afed, vbath_afed, rbath_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8) :: ds, sgnf, absf, pot_tmp

      integer :: i, j

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   check error: initialize flag                               */
!-----------------------------------------------------------------------

      iconv_afed = 0

!-----------------------------------------------------------------------
!     /*   check error: out of bounds                                 */
!-----------------------------------------------------------------------

      do i = 1, ncons

         if ( ( rafed(i) .lt. params_afed(itype_cons(i),2) ) .or. &
     &        ( rafed(i) .gt. params_afed(itype_cons(i),3) ) ) then

!           //   flag
            iconv_afed = -1

!           //   velocity reflected
            v_afed(i) = - v_afed(i)

         end if

      end do

!-----------------------------------------------------------------------
!     /*   check error: free energy too high                          */
!-----------------------------------------------------------------------

      if ( fenergy_afed .gt. fenergy_max_afed ) then

!        //   flag
         iconv_afed = -2

!        //   velocity reversed
         v_afed(:) = - v_afed(:)

      end if

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      write( 6, '(a)' )

      if ( iconv_afed .eq. 0 ) then

         write( 6, '(a)' ) &
     &        'AFED continues normally.'
         write( 6, '(a)' )

      else if ( iconv_afed .eq. -1 ) then

         write( 6, '(a)' ) &
     &         'AFED out of bounds, velocity is reflected.'
         write( 6, '(a)' )

      else if ( iconv_afed .eq. -2 ) then

         write( 6, '(a)' ) &
     &         'AFED free energy too high, velocity is reversed.'
         write( 6, '(a)' )

      end if

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( ( iset .eq. 0 ) .and. ( iiter_afed .eq. 0 ) ) then

         open ( iounit_afed, file = 'afed.out', access = 'append' )

         write( iounit_afed, '(a)' ) &
     &      '===================================================' // &
     &      '==========================='
         write( iounit_afed, '(a)' ) &
     &      ' step st cv  r-ideal   r-mean   f-energy     -df/dr' // &
     &      '    dr/dt   h-energy   temp'
         write( iounit_afed, '(a)' ) &
     &      '---------------------------------------------------' // &
     &      '---------------------------'

         close( iounit_afed )

         open( iounit_logmfd, file = 'logmfd.out', access = 'append' )

         write(iounit_logmfd, '(a)') '# Mass for CV particles :'
         write(iounit_logmfd, '(100f16.8)') fictmass_afed(:)

         write(iounit_logmfd, '(a)') '# Mass for thermostat   :' 
         write(iounit_logmfd, '(f16.8)') qmass_afed(1)

         write(iounit_logmfd, '(a)') &
     &   '# iter_md, Flog, 2*Ekin/gkb[K], eta, Veta, (Xq,Xt,Vt,Ft), ...'

         close( iounit_logmfd)

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   mean cv value                                              */
!-----------------------------------------------------------------------

      do i = 1, ncons

         scons_mean_afed(i) = 0.d0

         do j = 1, nbead

            ds = scons_avg(i,j) - rafed(i)

            call pbc_cons( ds, ipbc_cons(i) )

            ds = ds + rafed(i)

            scons_mean_afed(i) = scons_mean_afed(i) + ds

         end do

         scons_mean_afed(i) = scons_mean_afed(i) / dble(nbead)

      end do

!-----------------------------------------------------------------------
!     //   kinetic energy
!-----------------------------------------------------------------------

      ekin_afed = 0.d0

      do i = 1, ncons
         ekin_afed = ekin_afed &
     &      + 0.5d0 * fictmass_afed(i) * v_afed(i) * v_afed(i)
      end do

!-----------------------------------------------------------------------
!     //   potential energy
!-----------------------------------------------------------------------

!     //   sign of free energy
      sgnf = dsign( 1.d0, fenergy_afed - fenergy_ref_afed )

!     //   absolute value of free energy
      absf = abs( fenergy_afed - fenergy_ref_afed )

!     //   potential energy
      potential_afed = sgnf * gamma_logmfd &
     &               * log ( alpha_logmfd * absf + 1.d0 )

      pot_tmp = sgnf * log ( 4.d0 * absf + 1.d0) / 4.d0

!-----------------------------------------------------------------------
!     //   bath energy
!-----------------------------------------------------------------------

      ebath_afed = 0.d0

!-----------------------------------------------------------------------
!     //   total hamiltonian and temperature
!-----------------------------------------------------------------------

!     //   hamiltonian
      hamiltonian_afed = ekin_afed + potential_afed + ebath_afed

!     //   instataneous temperature
      temp_afed = 2.d0 * ekin_afed / dble(ncons) / boltz

!-----------------------------------------------------------------------
!     /*   standard output                                            */
!-----------------------------------------------------------------------

!     /*   wall clock time   */
      call getdate

      write( 6, '(a)' ) &
     &   '===================================================' // &
     &   '==========================='
      write( 6, '(a)' ) &
     &   ' step st cv  r-ideal   r-mean   f-energy     -df/dr' // &
     &   '    dr/dt   h-energy   temp'
      write( 6, '(a)' ) &
     &   '---------------------------------------------------' // &
     &   '---------------------------'

!-----------------------------------------------------------------------
!     /*   standard output                                            */
!-----------------------------------------------------------------------

!     /*   output   */
      do i = 1, ncons

         write( 6, &
     &      '(i5,a3,i3,2f9.3,f11.6,f11.6,f9.4,f11.6,i7)') &
     &      iiter_afed, afed_status(1:2), i, rafed(i), &
     &      scons_mean_afed(i), fenergy_afed, &
     &      f_afed(i), v_afed(i), hamiltonian_afed, nint(temp_afed)

      end do

      write( 6, * )

!-----------------------------------------------------------------------
!     /*   output to file: afed.out                                   */
!-----------------------------------------------------------------------

!     /*   no duplication   */
      if ( ( iiter_afed .eq. 0 ) .or. &
     &     ( iiter_afed .gt. iiter_start_afed ) ) then

!     /*   open file   */
      open( iounit_afed, file = 'afed.out', access = 'append' )

!     /*   output   */
      do i = 1, ncons

         write( iounit_afed, &
     &      '(i5,a3,i3,2f9.3,f11.6,f11.6,f9.4,f11.6,i7)') &
     &      iiter_afed, afed_status(1:2), i, rafed(i), &
     &      scons_mean_afed(i), fenergy_afed, &
     &      f_afed(i), v_afed(i), hamiltonian_afed, nint(temp_afed)

      end do

!     /*   close file   */
      close( iounit_afed )

!     /*   no duplication   */
      end if

!-----------------------------------------------------------------------
!     /*   output to file: logmfd.out                                 */
!-----------------------------------------------------------------------

!     /*   no duplication   */
      if ( ( iiter_afed .eq. 0 ) .or. &
     &     ( iiter_afed .gt. iiter_start_afed ) ) then

      open( iounit_logmfd, file = 'logmfd.out', access = 'append')

      write(iounit_logmfd, "(i8)") iiter_afed
      write(iounit_logmfd, "(e18.9)") fenergy_afed
      write(iounit_logmfd, "(1e18.9)") temp_afed
      write(iounit_logmfd, "(2e18.9)") rbath_afed(1), vbath_afed(1)

      do i = 1, ncons
         write(iounit_logmfd, "(4e18.9)") &
     &        scons_mean_afed(i), rafed(i), v_afed(i), f_afed(i)
      end do

      write( iounit_logmfd, * )

      close( iounit_logmfd )

!     /*   no duplication   */
      end if

      return
      end





!***********************************************************************
      subroutine rupdate_logmfd_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use afed_variables, only : &
     &   rafed, v_afed, dt_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   update cv position                                         */
!-----------------------------------------------------------------------

      rafed(:) = rafed(:) + v_afed(:) * dt_afed

!-----------------------------------------------------------------------
!     /*   apply periodic boundary condition                          */
!-----------------------------------------------------------------------

      call pbc_cv_afed

      return
      end





!***********************************************************************
      subroutine vupdate_logmfd_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   v_afed, f_afed, fictmass_afed, dt_afed, alpha_logmfd, &
     &   gamma_logmfd, fenergy_afed, fenergy_ref_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   real numbers
      real(8) :: frc, absf

!     //   integers
      integer :: i

!-----------------------------------------------------------------------
!     /*   update cv velocity                                         */
!-----------------------------------------------------------------------

!     //   absolute value of free energy
      absf = abs( fenergy_afed - fenergy_ref_afed )

!     //   loop of cv
      do i = 1, ncons

!        //   force applied to cv
         frc =   alpha_logmfd * gamma_logmfd &
     &       / ( alpha_logmfd * absf + 1.d0 ) &
     &       * f_afed(i)

!        //   update cv velocity
         v_afed(i) = v_afed(i) + frc / fictmass_afed(i) * dt_afed

!     //   loop of cv
      end do

      return
      end





!***********************************************************************
      subroutine backup_logmfd_nve_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, nbead

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   fenergy_afed, rafed, v_afed, fictmass_afed, fenergy_ref_afed, &
     &   iiter_afed, afed_status, work_afed, hamiltonian_ref_afed

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: i

      real(8) :: p_afed

!-----------------------------------------------------------------------
!     //   constrained molecular dynamics
!-----------------------------------------------------------------------

      call backup_md_nvt

!-----------------------------------------------------------------------
!        //   open file
!-----------------------------------------------------------------------

      open ( iounit, file = 'afed.ini' )

!-----------------------------------------------------------------------
!        //   write status
!-----------------------------------------------------------------------
      
      write( iounit, '(a2)' ) afed_status

!-----------------------------------------------------------------------
!        //   write step number
!-----------------------------------------------------------------------

      write( iounit, '(i8)' ) iiter_afed

!-----------------------------------------------------------------------
!        //   write free energy
!-----------------------------------------------------------------------

      write( iounit, '(e24.16)' ) fenergy_afed

!-----------------------------------------------------------------------
!        //   write position and velocity of collective variables
!-----------------------------------------------------------------------

      do i = 1, ncons
         p_afed = sqrt( fictmass_afed(i) ) * v_afed(i)
         write( iounit, '(2e24.16)' ) rafed(i), p_afed
      end do

!-----------------------------------------------------------------------
!        //   write reference hamiltonian
!-----------------------------------------------------------------------

      write( iounit, '(e24.16)' ) hamiltonian_ref_afed

!-----------------------------------------------------------------------
!        //   write reference free energy
!-----------------------------------------------------------------------

      write( iounit, '(e24.16)' ) fenergy_ref_afed

!-----------------------------------------------------------------------
!        //   write work parameter of replica
!-----------------------------------------------------------------------

      do i = 1, nbead
         write( iounit, '(e24.16)' ) work_afed(i)
      end do

!-----------------------------------------------------------------------
!        //   close file
!-----------------------------------------------------------------------

      close( iounit )

!-----------------------------------------------------------------------
!     //   wait for other processors
!-----------------------------------------------------------------------

!      call my_mpi_barrier

      return
      end





!***********************************************************************
      subroutine backup_logmfd_nvt_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, nbead

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   fenergy_afed, rafed, v_afed, fictmass_afed, rbath_afed, &
     &   vbath_afed, qmass_afed, fenergy_ref_afed, iiter_afed, &
     &   afed_status, work_afed, hamiltonian_ref_afed

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: i

      real(8) :: p_afed, pbath_afed

!-----------------------------------------------------------------------
!     //   constrained molecular dynamics
!-----------------------------------------------------------------------

      call backup_md_nvt

!-----------------------------------------------------------------------
!     //   open file
!-----------------------------------------------------------------------

      open ( iounit, file = 'afed.ini' )

!-----------------------------------------------------------------------
!     //   write status
!-----------------------------------------------------------------------

      write( iounit, '(a2)' ) afed_status

!-----------------------------------------------------------------------
!     //   write step number
!-----------------------------------------------------------------------

      write( iounit, '(i8)' ) iiter_afed

!-----------------------------------------------------------------------
!     //   write free energy
!-----------------------------------------------------------------------

      write( iounit, '(e24.16)' ) fenergy_afed

!-----------------------------------------------------------------------
!     //   write position and velocity of collective variables
!-----------------------------------------------------------------------

      do i = 1, ncons
         p_afed = sqrt( fictmass_afed(i) ) * v_afed(i)
         write( iounit, '(2e24.16)' ) rafed(i), p_afed
      end do

!-----------------------------------------------------------------------
!     //   write reference hamiltonian
!-----------------------------------------------------------------------

      write( iounit, '(e24.16)' ) hamiltonian_ref_afed

!-----------------------------------------------------------------------
!     //   write reference free energy
!-----------------------------------------------------------------------

      write( iounit, '(e24.16)' ) fenergy_ref_afed

!-----------------------------------------------------------------------
!     //   write position and velocity of collective variables
!-----------------------------------------------------------------------

      pbath_afed = sqrt( qmass_afed(1) ) * vbath_afed(1)
      write( iounit, '(2e24.16)' ) rbath_afed(1), pbath_afed

!-----------------------------------------------------------------------
!     //   write work parameter of replica
!-----------------------------------------------------------------------

      do i = 1, nbead
         write( iounit, '(e24.16)' ) work_afed(i)
      end do

!-----------------------------------------------------------------------
!        //   close file
!-----------------------------------------------------------------------
      
      close( iounit )

      return
      end





!***********************************************************************
      subroutine afedcycle_logmfd_nvt
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iexit

      use afed_variables, only : &
     &   iiter_afed, iiter_start_afed, iiter_end_afed, niter_afed, &
     &   v_afed, v_save_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: iviter
      integer :: nviter = 5

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

!     //   starting step
      iiter_afed = iiter_start_afed

!     //   current step
      iiter_end_afed = iiter_afed

!c     //   get reference hamiltonian
!      call getref_logmfd_nvt_afed( 0 )

!     //   run constrained molecular dynamics and get meanforce
      call meanforce_afed

!     //   stop on soft exit
      if ( iexit .eq. 1 ) return

!     //   gradient vector
      call eupdate_descent_afed

!     //   calculate free energy
      call getref_logmfd_nvt_afed( 1 )

!     //   standard output ! orig
      call standard_logmfd_nvt_afed

!     //   analysis
      call analysis_afed

!     //   save data
      call backup_logmfd_nvt_afed

!     //   start iteration loop
      do iiter_afed = iiter_start_afed+1, niter_afed

!        //   current step
         iiter_end_afed = iiter_afed

!        //   save last step
         call save_cv_afed

!        //   save current velocity : v_afed
         v_save_afed(:) = v_afed(:)

!        //   update velocity by dt/2 : v_afed
         do iviter = 1, nviter

!           //   calculate latest free energy : fenergy_afed
            call getref_logmfd_nvt_afed( 1 )

!           //   restore v_afed
            v_afed(:) = v_save_afed(:)

!           //   evolve v_afed by dt/2
            call vupdate_logmfd_nvt_afed

         end do

!        //   calculate latest free energy : fenergy_afed
         call getref_logmfd_nvt_afed( 1 )

!        //   update velocity by dt/2 : v_afed
         call vupdate_logmfd_nvt_afed

!        //   update position by dt : r_afed
         call rupdate_logmfd_afed

!        //   update bath by dt : rbath_afed, vbath_afed
         call bupdate_logmfd_nvt_afed

!        //   run constrained molecular dynamics and get meanforce
         call meanforce_afed

!        //   stop on soft exit
         if ( iexit .eq. 1 ) return

!        //   gradient vector
         call eupdate_descent_afed

!        //   calculate latest free energy : fenergy_afed
         call getref_logmfd_nvt_afed( 1 )

!        //   standard output ! orig
         call standard_logmfd_nvt_afed

!        //   analysis
         call analysis_afed

!        //   save data
         call backup_logmfd_nvt_afed

!     //   end iteration loop
      end do

!     /*   last step   */
      iiter_afed = iiter_end_afed

      return
      end





!***********************************************************************
      subroutine vupdate_logmfd_nvt_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   v_afed, f_afed, fictmass_afed, dt_afed, alpha_logmfd, &
     &   gamma_logmfd, fenergy_afed, fenergy_ref_afed, vbath_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   real numbers
      real(8) :: dt_half, dt_quarter, frc, absf

!     //   integers
      integer :: i

!-----------------------------------------------------------------------
!     /*   half of AFED step size                                     */
!-----------------------------------------------------------------------

      dt_half = 0.5d0 * dt_afed

!-----------------------------------------------------------------------
!     /*   quarter of AFED step size                                  */
!-----------------------------------------------------------------------

      dt_quarter = 0.25d0 * dt_afed

!-----------------------------------------------------------------------
!     /*   update cv velocity                                         */
!-----------------------------------------------------------------------

!     //   absolute value of free energy
      absf = abs( fenergy_afed - fenergy_ref_afed )

!     //   loop of cv
      do i = 1, ncons

!        //   update cv velocity with cv bath velocity by dt/4
         v_afed(i) = v_afed(i) * exp( - dt_quarter * vbath_afed(1))

!        //   force applied to cv
         frc =   alpha_logmfd * gamma_logmfd &
     &       / ( alpha_logmfd * absf + 1.d0 ) &
     &       * f_afed(i)

!        //   update cv velocity
         v_afed(i) = v_afed(i) + frc / fictmass_afed(i) * dt_half

!        //   update cv velocity with cv bath velocity by dt/4
         v_afed(i) = v_afed(i) * exp( - dt_quarter * vbath_afed(1))

!     //   loop of cv
      end do

      return
      end





!***********************************************************************
      subroutine bupdate_logmfd_nvt_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   v_afed, fictmass_afed, dt_afed, &
     &   rbath_afed, vbath_afed, fbath_afed, qmass_afed, gnkt_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   real numbers
      real(8) :: dt_half, dkin

!     //   integers
      integer :: i

!-----------------------------------------------------------------------
!     /*   half of AFED step size                                     */
!-----------------------------------------------------------------------

      dt_half = 0.5d0 * dt_afed

!-----------------------------------------------------------------------
!     /*   update bath parameters                                     */
!-----------------------------------------------------------------------

!     //   update cv bath position by dt/2
      rbath_afed(1) = rbath_afed(1) + dt_half * vbath_afed(1)

!     //   calculate total kinetic energy of the system
      dkin = 0.d0

      do i = 1, ncons
         dkin = dkin + fictmass_afed(i) * v_afed(i) * v_afed(i)
      end do

      fbath_afed(1) = (dkin - gnkt_afed) / qmass_afed(1)

!     //   update cv bath velocity by dt
      vbath_afed(1) = vbath_afed(1) + dt_afed * fbath_afed(1)

!     //   update cv bath position by dt/2
      rbath_afed(1) = rbath_afed(1) + dt_half * vbath_afed(1)

      return
      end





!***********************************************************************
      subroutine standard_logmfd_nvt_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, iounit_afed, iounit_logmfd, nbead

      use cons_variables, only : &
     &   scons_avg, ncons, ipbc_cons, itype_cons

      use afed_variables, only : &
     &   fenergy_afed, f_afed, scons_mean_afed, v_afed, params_afed, &
     &   hamiltonian_afed, ekin_afed, temp_afed, potential_afed, &
     &   rafed, fenergy_max_afed, fictmass_afed, ebath_afed, qmass_afed, &
     &   qmass_afed, vbath_afed, rbath_afed, gnkt_afed, &
     &   alpha_logmfd, gamma_logmfd, fenergy_ref_afed, afed_status, &
     &   iiter_afed, iconv_afed, iiter_start_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8) :: ds, sgnf, absf

      integer :: i, j

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   check error: initialize flag                               */
!-----------------------------------------------------------------------

      iconv_afed = 0

!-----------------------------------------------------------------------
!     /*   check error: out of bounds                                 */
!-----------------------------------------------------------------------

      do i = 1, ncons

         if ( ( rafed(i) .lt. params_afed(itype_cons(i),2) ) .or. &
     &        ( rafed(i) .gt. params_afed(itype_cons(i),3) ) ) then

!           //   flag
            iconv_afed = -1

!           //   velocity reflected
            v_afed(i) = - v_afed(i)

         end if

      end do

!-----------------------------------------------------------------------
!     /*   check error: free energy too high                          */
!-----------------------------------------------------------------------

      if ( fenergy_afed .gt. fenergy_max_afed ) then

!        //   flag
         iconv_afed = -2

!        //   velocity reversed
         v_afed(:) = - v_afed(:)

      end if

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      write( 6, '(a)' )

      if ( iconv_afed .eq. 0 ) then

         write( 6, '(a)' ) &
     &         'AFED continues normally.'
         write( 6, '(a)' )

      else if ( iconv_afed .eq. -1 ) then

         write( 6, '(a)' ) &
     &         'AFED out of bounds, velocity is reflected.'
         write( 6, '(a)' )

      else if ( iconv_afed .eq. -2 ) then

         write( 6, '(a)' ) &
     &         'AFED free energy too high, velocity is reversed.'
         write( 6, '(a)' )

      end if

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( ( iset .eq. 0 ) .and. ( iiter_afed .eq. 0 ) ) then

         open ( iounit_afed, file = 'afed.out', access = 'append' )

         write( iounit_afed, '(a)' ) &
     &      '===================================================' // &
     &      '==========================='
         write( iounit_afed, '(a)' ) &
     &      ' step st cv  r-ideal   r-mean   f-energy     -df/dr' // &
     &      '    dr/dt   h-energy   temp'
         write( iounit_afed, '(a)' ) &
     &      '---------------------------------------------------' // &
     &      '---------------------------'

         close( iounit_afed )

         open( iounit_logmfd, file = 'logmfd.out', access = 'append')

         write(iounit_logmfd, '(a)') '# Mass for CV particles :'
         write(iounit_logmfd, '(100f16.8)') fictmass_afed(:)

         write(iounit_logmfd, '(a)') '# Mass for thermostat   :'
         write(iounit_logmfd, '(f16.8)') qmass_afed(1)

         write(iounit_logmfd, '(a)') &
     &   '# iter_md, Flog, 2*Ekin/gkb[K], eta, Veta, (Xq,Xt,Vt,Ft), ...'

         close( iounit_logmfd)

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   mean cv value                                              */
!-----------------------------------------------------------------------

      do i = 1, ncons

         scons_mean_afed(i) = 0.d0

         do j = 1, nbead

            ds = scons_avg(i,j) - rafed(i)

            call pbc_cons( ds, ipbc_cons(i) )

            ds = ds + rafed(i)

            scons_mean_afed(i) = scons_mean_afed(i) + ds

         end do

         scons_mean_afed(i) = scons_mean_afed(i) / dble(nbead)

      end do

!-----------------------------------------------------------------------
!     //   kinetic energy
!-----------------------------------------------------------------------

      ekin_afed = 0.d0

      do i = 1, ncons
         ekin_afed = ekin_afed &
     &      + 0.5d0 * fictmass_afed(i) * v_afed(i) * v_afed(i)
      end do

!-----------------------------------------------------------------------
!     //   potential energy
!-----------------------------------------------------------------------

!     //   sign of free energy
      sgnf = dsign( 1.d0, fenergy_afed - fenergy_ref_afed )

!     //   absolute value of free energy
      absf = abs( fenergy_afed - fenergy_ref_afed )

!     //   potential energy
      potential_afed = sgnf * gamma_logmfd &
     &               * log ( alpha_logmfd * absf + 1.d0 )

!-----------------------------------------------------------------------
!     //   bath energy
!-----------------------------------------------------------------------

      ebath_afed = &
     &   + 0.5d0 * qmass_afed(1) * vbath_afed(1) * vbath_afed(1) &
     &   + gnkt_afed * rbath_afed(1)

!-----------------------------------------------------------------------
!     //   total hamiltonian and temperature
!-----------------------------------------------------------------------

!     //   hamiltonian
      hamiltonian_afed = ekin_afed + potential_afed + ebath_afed

!     //   instataneous temperature
      temp_afed = 2.d0 * ekin_afed / dble(ncons) / boltz


!-----------------------------------------------------------------------
!     /*   standard output                                            */
!-----------------------------------------------------------------------

!     /*   wall clock time   */
      call getdate

      write( 6, '(a)' ) &
     &   '===================================================' // &
     &   '==========================='
      write( 6, '(a)' ) &
     &   ' step st cv  r-ideal   r-mean   f-energy     -df/dr' // &
     &   '    dr/dt   h-energy   temp'
      write( 6, '(a)' ) &
     &   '---------------------------------------------------' // &
     &   '---------------------------'

!-----------------------------------------------------------------------
!     /*   standard output                                            */
!-----------------------------------------------------------------------

!     /*   output   */
      do i = 1, ncons

         write( 6, &
     &      '(i5,a3,i3,2f9.3,f11.6,f11.6,f9.4,f11.6,i7)') &
     &      iiter_afed, afed_status(1:2), i, rafed(i), &
     &      scons_mean_afed(i), fenergy_afed, &
     &      f_afed(i), v_afed(i), hamiltonian_afed, nint(temp_afed)

      end do

      write( 6, * )

!-----------------------------------------------------------------------
!     /*   output to file: afed.out                                   */
!-----------------------------------------------------------------------

!     /*   no duplication   */
      if ( ( iiter_afed .eq. 0 ) .or. &
     &     ( iiter_afed .gt. iiter_start_afed ) ) then

!     /*   open file   */
      open( iounit_afed, file = 'afed.out', access = 'append' )

!     /*   output   */
      do i = 1, ncons

         write( iounit_afed, &
     &      '(i5,a3,i3,2f9.3,f11.6,f11.6,f9.4,f11.6,i7)') &
     &      iiter_afed, afed_status(1:2), i, rafed(i), &
     &      scons_mean_afed(i), fenergy_afed, &
     &      f_afed(i), v_afed(i), hamiltonian_afed, nint(temp_afed)

      end do

!     /*   close file   */
      close( iounit_afed )

!     /*   no duplication   */
      end if

!-----------------------------------------------------------------------
!     /*   output to file: logmfd.out                                 */
!-----------------------------------------------------------------------

!     /*   no duplication   */
      if ( ( iiter_afed .eq. 0 ) .or. &
     &     ( iiter_afed .gt. iiter_start_afed ) ) then

      open( iounit_logmfd, file = 'logmfd.out', access = 'append' )

      write(iounit_logmfd, "(i8)") iiter_afed
      write(iounit_logmfd, "(e18.9)") fenergy_afed
      write(iounit_logmfd, "(1e18.9)") temp_afed
      write(iounit_logmfd, "(2e18.9)") rbath_afed(1), vbath_afed(1)

      do i = 1, ncons
         write(iounit_logmfd, "(4e18.9)") &
     &        scons_mean_afed(i), rafed(i), v_afed(i), f_afed(i)
      end do

      write( iounit_logmfd, * )

      close( iounit_logmfd )

!     /*   no duplication   */
      end if

      return
      end





!***********************************************************************
      subroutine getref_logmfd_nve_afed( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   fenergy_afed, ekin_afed, potential_afed, v_afed, &
     &   fictmass_afed, ebath_afed, alpha_logmfd, &
     &   gamma_logmfd, fenergy_ref_afed, hamiltonian_ref_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption

      real(8) :: sgnf, absf

!-----------------------------------------------------------------------
!     //   option 0:  get reference hamiltonian
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

!        //   kinetic energy

         ekin_afed = 0.d0

         do i = 1, ncons
            ekin_afed = ekin_afed &
     &         + 0.5d0 * fictmass_afed(i) * v_afed(i) * v_afed(i)
         end do

!     //   potential energy

!        //   sign of free energy
         sgnf = dsign( 1.d0, fenergy_afed - fenergy_ref_afed )

!        //   absolute value of free energy
         absf = abs( fenergy_afed - fenergy_ref_afed )

!        //   potential energy
         potential_afed = sgnf * gamma_logmfd &
     &                  * log ( alpha_logmfd * absf + 1.d0 )

!        //   bath energy

         ebath_afed = 0.d0

!        //   reference hamiltonian

         hamiltonian_ref_afed = ekin_afed + potential_afed + ebath_afed

!-----------------------------------------------------------------------
!     //   option 0:  get reference hamiltonian
!-----------------------------------------------------------------------

      end if

!-----------------------------------------------------------------------
!     //   option 1:  get reference free energy
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

!        //   kinetic energy

         ekin_afed = 0.d0

         do i = 1, ncons
            ekin_afed = ekin_afed &
     &         + 0.5d0 * fictmass_afed(i) * v_afed(i) * v_afed(i)
         end do

!        //   bath energy

         ebath_afed = 0.d0

!        //   potential

         potential_afed = hamiltonian_ref_afed - ekin_afed - ebath_afed

!        //   sign of potential energy
         sgnf = dsign( 1.d0, potential_afed )

!        //   absolute value of potential energy
         absf = abs( potential_afed )

!        //   reference free energy
         fenergy_afed = fenergy_ref_afed &
     &     + sgnf * ( exp( absf / gamma_logmfd ) - 1.d0 ) / alpha_logmfd

!-----------------------------------------------------------------------
!     //   option 1:  get reference free energy
!-----------------------------------------------------------------------

      end if

      return
      end





!***********************************************************************
      subroutine getref_logmfd_nvt_afed( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   fenergy_afed, ekin_afed, potential_afed, gamma_logmfd, &
     &   fictmass_afed, ebath_afed, alpha_logmfd, qmass_afed, &
     &   rbath_afed, vbath_afed, gnkt_afed, v_afed, &
     &   fenergy_ref_afed, hamiltonian_ref_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption

      real(8) :: sgnf, absf

!-----------------------------------------------------------------------
!     //   option 0:  get reference hamiltonian
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

!        //   kinetic energy

         ekin_afed = 0.d0

         do i = 1, ncons
            ekin_afed = ekin_afed &
     &         + 0.5d0 * fictmass_afed(i) * v_afed(i) * v_afed(i)
         end do

!        //   sign of free energy
         sgnf = dsign( 1.d0, fenergy_afed - fenergy_ref_afed )

!        //   absolute value of free energy
         absf = abs( fenergy_afed - fenergy_ref_afed )

!        //   potential energy

         potential_afed = sgnf * gamma_logmfd &
     &                  * log ( alpha_logmfd * absf + 1.d0 )

!        //   bath energy

         ebath_afed = &
     &      + 0.5d0 * qmass_afed(1) * vbath_afed(1) * vbath_afed(1) &
     &      + gnkt_afed * rbath_afed(1)

!        //   reference hamiltonian
         hamiltonian_ref_afed = ekin_afed + potential_afed + ebath_afed

!-----------------------------------------------------------------------
!     //   option 0:  get reference hamiltonian
!-----------------------------------------------------------------------

      end if

!-----------------------------------------------------------------------
!     //   option 1:  get reference free energy
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

!        //   kinetic energy

         ekin_afed = 0.d0

         do i = 1, ncons
            ekin_afed = ekin_afed &
     &         + 0.5d0 * fictmass_afed(i) * v_afed(i) * v_afed(i)
         end do

!        //   bath energy

         ebath_afed = &
     &      + 0.5d0 * qmass_afed(1) * vbath_afed(1) * vbath_afed(1) &
     &      + gnkt_afed * rbath_afed(1)

!        //   potential
         potential_afed = hamiltonian_ref_afed - ekin_afed - ebath_afed

!        //   sign of potential energy
         sgnf = dsign( 1.d0, potential_afed )

!        //   absolute value of potential energy
         absf = abs(potential_afed)

!        //   reference free energy

         fenergy_afed = fenergy_ref_afed &
     &      + sgnf * ( exp( absf / gamma_logmfd ) - 1.d0 ) &
     &      / alpha_logmfd


!-----------------------------------------------------------------------
!     //   option 1:  get reference free energy
!-----------------------------------------------------------------------

      end if

      return
      end





!***********************************************************************
      subroutine afedcycle_logmfd_vs
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iexit

      use afed_variables, only : &
     &   iiter_afed, iiter_start_afed, iiter_end_afed, niter_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

!     //   starting step
      iiter_afed = iiter_start_afed

!     //   current step
      iiter_end_afed = iiter_afed

!c     //   get reference hamiltonian
!      call getref_logmfd_vs_afed( 0 )

!     //   run constrained molecular dynamics and get meanforce
      call meanforce_afed

!     //   stop on soft exit
      if ( iexit .eq. 1 ) return

!     //   gradient vector
      call eupdate_descent_afed

!     //   calculate free energy
      call getref_logmfd_vs_afed( 1 )

!     //   standard output
      call standard_logmfd_vs_afed

!     //   analysis
      call analysis_afed

!     //   save data
      call backup_logmfd_vs_afed

!     //   start iteration loop
      do iiter_afed = iiter_start_afed+1, niter_afed

!        //   current step
         iiter_end_afed = iiter_afed

!        //   save last step
         call save_cv_afed

!        //   calculate latest free energy : fenergy_afed
         call getref_logmfd_vs_afed( 1 )

!        //   update velocity by dt : v_afed
         call vupdate_logmfd_vs_afed

!        //   velocity scaling
         call vscaling_logmfd

!        //   update position by dt : r_afed
         call rupdate_logmfd_afed

!        //   update potential by dt : rbath_afed, vbath_afed
         call pupdate_logmfd_vs_afed

!        //   run constrained molecular dynamics and get meanforce
         call meanforce_afed

!        //   stop on soft exit
         if ( iexit .eq. 1 ) return

!        //   gradient vector
         call eupdate_descent_afed

!        //   calculate latest free energy : fenergy_afed
         call getref_logmfd_vs_afed( 1 )

!        //   standard output
         call standard_logmfd_vs_afed

!        //   analysis
         call analysis_afed

!        //   save data
         call backup_logmfd_nvt_afed

!     //   end iteration loop
      end do

!     /*   last step   */
      iiter_afed = iiter_end_afed

      return
      end





!***********************************************************************
      subroutine getref_logmfd_vs_afed( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use afed_variables, only : &
     &   fenergy_afed, gamma_logmfd, alpha_logmfd, phi_logmfd

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ioption
      real(8) :: sgnf, absf, sgnp, absp

!-----------------------------------------------------------------------
!     //   option 0:  get reference hamiltonian
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

!        //   pontential of logmfd by velocity scalling

         sgnf = dsign( 1.d0, fenergy_afed)
         absf = abs(fenergy_afed)

         phi_logmfd = sgnf * gamma_logmfd  &
     &              * log(alpha_logmfd * absf + 1.d0)

!-----------------------------------------------------------------------
!     //   option 0:  get reference hamiltonian
!-----------------------------------------------------------------------

      end if

!-----------------------------------------------------------------------
!     //   option 1:  get reference free energy
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

!        //   reference free energy
         
         sgnp = dsign( 1.d0, phi_logmfd)
         absp = abs(phi_logmfd)

         fenergy_afed = sgnp * ( exp( absp / gamma_logmfd ) - 1.d0 ) &
     &                  / alpha_logmfd

!-----------------------------------------------------------------------
!     //   option 1:  get reference free energy
!-----------------------------------------------------------------------

      end if

      return
      end





!***********************************************************************
      subroutine vupdate_logmfd_vs_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   v_afed, f_afed, fictmass_afed, dt_afed, alpha_logmfd, &
     &   gamma_logmfd, fenergy_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   real numbers
      real(8) :: frc

!     //   integers
      integer :: i

!-----------------------------------------------------------------------
!     /*   update cv velocity                                         */
!-----------------------------------------------------------------------

!     //   loop of cv
      do i = 1, ncons

!        //   force applied to cv
         frc =   alpha_logmfd * gamma_logmfd &
     &       / ( alpha_logmfd * abs(fenergy_afed) + 1.d0 ) &
     &       * f_afed(i)

!        //   update cv velocity
         v_afed(i) = v_afed(i) + frc / fictmass_afed(i) * dt_afed

!     //   loop of cv
      end do

      return
      end





!***********************************************************************
      subroutine vscaling_logmfd
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   ekin_afed, fictmass_afed, gnkt_afed, v_afed, &
     &   scale_logmfd

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

!-----------------------------------------------------------------------

!     //   kinetic energy

      ekin_afed = 0.d0

      do i = 1, ncons
         ekin_afed = ekin_afed &
     &        + 0.5d0 * fictmass_afed(i) * v_afed(i) * v_afed(i)
      end do

      scale_logmfd = sqrt(gnkt_afed / (2.d0 * ekin_afed))

      do j = 1, ncons
         v_afed(j) = scale_logmfd * v_afed(j)
      end do

      return
      end





!***********************************************************************
      subroutine pupdate_logmfd_vs_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use afed_variables, only : &
     &   gnkt_afed, phi_logmfd, scale_logmfd

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------

!     //   pontential of logmfd by velocity scalling

      phi_logmfd = phi_logmfd + gnkt_afed * log(scale_logmfd)

      return
      end





!***********************************************************************
      subroutine standard_logmfd_vs_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, iounit_afed, iounit_logmfd, nbead

      use cons_variables, only : &
     &   scons_avg, ncons, ipbc_cons, itype_cons

      use afed_variables, only : &
     &   fenergy_afed, f_afed, scons_mean_afed, v_afed, params_afed, &
     &   hamiltonian_afed, ekin_afed, temp_afed, rafed, &
     &   fenergy_max_afed, fictmass_afed, &
     &   iiter_afed, iconv_afed, iiter_start_afed, &
     &   afed_status, rbath_afed, vbath_afed, qmass_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8) :: ds

      integer :: i, j

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   check error: initialize flag                               */
!-----------------------------------------------------------------------

      iconv_afed = 0

!-----------------------------------------------------------------------
!     /*   check error: out of bounds                                 */
!-----------------------------------------------------------------------

      do i = 1, ncons

         if ( ( rafed(i) .lt. params_afed(itype_cons(i),2) ) .or. &
     &        ( rafed(i) .gt. params_afed(itype_cons(i),3) ) ) then

!           //   flag
            iconv_afed = -1

!           //   velocity reflected
            v_afed(i) = - v_afed(i)

         end if

      end do

!-----------------------------------------------------------------------
!     /*   check error: free energy too high                          */
!-----------------------------------------------------------------------

      if ( fenergy_afed .gt. fenergy_max_afed ) then

!        //   flag
         iconv_afed = -2

!        //   velocity reversed
         v_afed(:) = - v_afed(:)

      end if

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      write( 6, '(a)' )

      if ( iconv_afed .eq. 0 ) then

         write( 6, '(a)' ) &
     &         'AFED continues normally.'
         write( 6, '(a)' )

      else if ( iconv_afed .eq. -1 ) then

         write( 6, '(a)' ) &
     &         'AFED out of bounds, velocity is reflected.'
         write( 6, '(a)' )

      else if ( iconv_afed .eq. -2 ) then

         write( 6, '(a)' ) &
     &         'AFED free energy too high, velocity is reversed.'
         write( 6, '(a)' )

      end if

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( ( iset .eq. 0 ) .and. ( iiter_afed .eq. 0 ) ) then

            open ( iounit_afed, file = 'afed.out', access = 'append' )

            write( iounit_afed, '(a)' ) &
     &      '===================================================' // &
     &      '==========================='
            write( iounit_afed, '(a)' ) &
     &      ' step st cv  r-ideal   r-mean   f-energy     -df/dr' // &
     &      '    dr/dt   h-energy   temp'
            write( iounit_afed, '(a)' ) &
     &      '---------------------------------------------------' // &
     &      '---------------------------'

            close( iounit_afed )

            open( iounit_logmfd, file = 'logmfd.out', access = 'append')

            write(iounit_logmfd, '(a)') '# Mass for CV particles :'
            write(iounit_logmfd, '(100f16.8)') fictmass_afed(:)

            write(iounit_logmfd, '(a)') '# Mass for thermostat   :'
            write(iounit_logmfd, '(f16.8)') qmass_afed(1)

            write(iounit_logmfd, '(a)') &
     &   '# iter_md, Flog, 2*Ekin/gkb[K], eta, Veta, (Xq,Xt,Vt,Ft), ...'

            close( iounit_logmfd)

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   mean cv value                                              */
!-----------------------------------------------------------------------

      do i = 1, ncons

         scons_mean_afed(i) = 0.d0

         do j = 1, nbead

            ds = scons_avg(i,j) - rafed(i)

            call pbc_cons( ds, ipbc_cons(i) )

            ds = ds + rafed(i)

            scons_mean_afed(i) = scons_mean_afed(i) + ds

         end do

         scons_mean_afed(i) = scons_mean_afed(i) / dble(nbead)

      end do

!-----------------------------------------------------------------------
!     //   kinetic energy
!-----------------------------------------------------------------------

      ekin_afed = 0.d0

      do i = 1, ncons
         ekin_afed = ekin_afed &
     &      + 0.5d0 * fictmass_afed(i) * v_afed(i) * v_afed(i)
      end do

!-----------------------------------------------------------------------
!     //   total hamiltonian and temperature
!-----------------------------------------------------------------------

!     //   hamiltonian
      hamiltonian_afed = 0.d0

!     //   instataneous temperature
      temp_afed = 2.d0 * ekin_afed / dble(ncons) / boltz

!-----------------------------------------------------------------------
!     /*   standard output                                            */
!-----------------------------------------------------------------------

!     /*   wall clock time   */
      call getdate

      write( 6, '(a)' ) &
     &   '===================================================' // &
     &   '==========================='
      write( 6, '(a)' ) &
     &   ' step st cv  r-ideal   r-mean   f-energy     -df/dr' // &
     &   '    dr/dt   h-energy   temp'
      write( 6, '(a)' ) &
     &   '---------------------------------------------------' // &
     &   '---------------------------'

!-----------------------------------------------------------------------
!     /*   standard output                                            */
!-----------------------------------------------------------------------

!     /*   output   */
      do i = 1, ncons

         write( 6, &
     &      '(i5,a3,i3,2f9.3,f11.6,f11.6,f9.4,f11.6,i7)') &
     &      iiter_afed, afed_status(1:2), i, rafed(i), &
     &      scons_mean_afed(i), fenergy_afed, &
     &      f_afed(i), v_afed(i), hamiltonian_afed, nint(temp_afed)

      end do

      write( 6, * )

!-----------------------------------------------------------------------
!     /*   output to file: afed.out                                   */
!-----------------------------------------------------------------------

!     /*   no duplication   */
      if ( ( iiter_afed .eq. 0 ) .or. &
     &     ( iiter_afed .gt. iiter_start_afed ) ) then

!     /*   open file   */
      open( iounit_afed, file = 'afed.out', access = 'append' )

!     /*   output   */
      do i = 1, ncons

         write( iounit_afed, &
     &      '(i5,a3,i3,2f9.3,f11.6,f11.6,f9.4,f11.6,i7)') &
     &      iiter_afed, afed_status(1:2), i, rafed(i), &
     &      scons_mean_afed(i), fenergy_afed, &
     &      f_afed(i), v_afed(i), hamiltonian_afed, nint(temp_afed)

      end do

!     /*   close file   */
      close( iounit_afed )

!     /*   no duplication   */
      end if

!-----------------------------------------------------------------------
!     /*   output to file: afed.out                                   */
!-----------------------------------------------------------------------

!     /*   no duplication   */
      if ( ( iiter_afed .eq. 0 ) .or. &
     &     ( iiter_afed .gt. iiter_start_afed ) ) then

      open( iounit_logmfd, file = 'logmfd.out', access = 'append')

      write(iounit_logmfd, "(i8)") iiter_afed
      write(iounit_logmfd, "(e18.9)") fenergy_afed
      write(iounit_logmfd, "(1e18.9)") temp_afed
      write(iounit_logmfd, "(2e18.9)") rbath_afed(1), vbath_afed(1)

      do i = 1, ncons
         write(iounit_logmfd, "(4e18.9)") &
     &      scons_mean_afed(i), rafed(i), v_afed(i), f_afed(i)
      end do

      write( iounit_logmfd, * )

      close( iounit_logmfd )

!     /*   no duplication   */
      end if

      return
      end





!***********************************************************************
      subroutine backup_logmfd_vs_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, nbead

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   fenergy_afed, rafed, v_afed, fictmass_afed, phi_logmfd, &
     &   fenergy_ref_afed, iiter_afed, afed_status, work_afed

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: i

      real(8) :: p_afed

!-----------------------------------------------------------------------
!     //   constrained molecular dynamics
!-----------------------------------------------------------------------

      call backup_md_nvt

!-----------------------------------------------------------------------
!     //   open file
!-----------------------------------------------------------------------

      open ( iounit, file = 'afed.ini' )

!-----------------------------------------------------------------------
!     //   write status
!-----------------------------------------------------------------------

      write( iounit, '(a2)' ) afed_status

!-----------------------------------------------------------------------
!     //   write step number
!-----------------------------------------------------------------------

      write( iounit, '(i8)' ) iiter_afed

!-----------------------------------------------------------------------
!     //   write free energy
!-----------------------------------------------------------------------

      write( iounit, '(e24.16)' ) fenergy_afed

!-----------------------------------------------------------------------
!     //   write position and velocity of collective variables
!-----------------------------------------------------------------------

      do i = 1, ncons
         p_afed = sqrt( fictmass_afed(i) ) * v_afed(i)
         write( iounit, '(2e24.16)' ) rafed(i), p_afed
      end do

!-----------------------------------------------------------------------
!     //   write reference hamiltonian
!-----------------------------------------------------------------------

      write( iounit, '(e24.16)' ) phi_logmfd

!-----------------------------------------------------------------------
!     //   write reference free energy
!-----------------------------------------------------------------------

      write( iounit, '(e24.16)' ) fenergy_ref_afed

!-----------------------------------------------------------------------
!     //   write work parameter of replica
!-----------------------------------------------------------------------

      do i = 1, nbead
         write( iounit, '(e24.16)' ) work_afed(i)
      end do

!-----------------------------------------------------------------------
!     //   close file
!-----------------------------------------------------------------------

      close( iounit )

      return
      end

