!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jan 29, 2019 by M. Shiga
!      Description:     Adiabatic free energy dynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine afedcycle_grad_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iexit

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

!     //   run constrained molecular dynamics and get meanforce
      call meanforce_afed_MPI

!     //   stop on soft exit
      if ( iexit .eq. 1 ) return

!     //   gradient vector
      call eupdate_descent_afed

!     //   eigenvector
      call geteigen_afed_MPI

!     //   standard output
      call standard_adescent_afed_MPI

!     //   analysis
      call analysis_afed_MPI

!     //   save data
      call backup_adescent_afed_MPI

      return
      end





!***********************************************************************
      subroutine backup_adescent_afed_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, ipos_start, ivel_start, ibath_start, myrank

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   e_afed, fenergy_afed, rafed, dt_afed, iiter_afed, afed_status, &
     &   iroot_auto_afed

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: i

!-----------------------------------------------------------------------
!     //   constrained molecular dynamics
!-----------------------------------------------------------------------

      call backup_md_nvt_MPI

!-----------------------------------------------------------------------
!     //   for next time
!-----------------------------------------------------------------------

      ipos_start = 1
      ivel_start = 1
      ibath_start = 1

!-----------------------------------------------------------------------
!     //   master rank only
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

!-----------------------------------------------------------------------
!        //   open file
!-----------------------------------------------------------------------

         open ( iounit, file = 'afed.ini' )

!-----------------------------------------------------------------------
!        //   write status
!-----------------------------------------------------------------------

         write( iounit, '(a2)' ) afed_status

!-----------------------------------------------------------------------
!        //   write step number
!-----------------------------------------------------------------------

         write( iounit, '(i8)' ) iiter_afed

!-----------------------------------------------------------------------
!        //   write free energy
!-----------------------------------------------------------------------

         write( iounit, '(e24.16)' ) fenergy_afed

!-----------------------------------------------------------------------
!        //   write collective variable, unit vector
!-----------------------------------------------------------------------

         do i = 1, ncons
            write( iounit, '(2e24.16)' ) rafed(i), e_afed(i)
         end do

!-----------------------------------------------------------------------
!        //   write root point of trajectory
!-----------------------------------------------------------------------

         write( iounit, '(i8)' ) iroot_auto_afed

!-----------------------------------------------------------------------
!        //   write step size
!-----------------------------------------------------------------------

         write( iounit, '(e24.16)' ) dt_afed

!-----------------------------------------------------------------------
!        //   close file
!-----------------------------------------------------------------------

         close( iounit )

!-----------------------------------------------------------------------
!     //   master rank only
!-----------------------------------------------------------------------

      end if

!-----------------------------------------------------------------------
!     //   wait for other processors
!-----------------------------------------------------------------------

      call my_mpi_barrier

      return
      end





!***********************************************************************
      subroutine meanforce_analytical_afed_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iexit, istep_start, nstep

      use afed_variables, only : &
     &   nstep_pre_afed, nstep_pro_afed, mdcycle_pro_afed, ioption_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   constrained molecular dynamics: preliminary run            */
!-----------------------------------------------------------------------

!     /*   initialize bath   */
      call init_bath_cart_revised_MPI

!     /*   set constraints   */
      call averages_analytical_afed( 0 )

!     /*   reset step number   */
      istep_start = 0

!     /*   reset number of steps   */
      nstep = nstep_pre_afed

!     /*   reset average forces   */
      call analysis_cons_MPI ( 0 )

!     /*   preliminary run    */
      call mdcycle_nvt_cons_MPI

!     /*   stop on soft exit   */
      if ( iexit .eq. 1 ) return

!-----------------------------------------------------------------------
!     /*   constrained molecular dynamics: productive run             */
!-----------------------------------------------------------------------

!     /*   reset step number   */
      istep_start = 0

!     /*   reset number of steps   */
      nstep = nstep_pro_afed

!     /*   reset average forces   */
      call analysis_cons_MPI ( 0 )

!     /*   productive run    */

      if ( mdcycle_pro_afed(1:4) .eq. 'NVE ' ) then
         call mdcycle_nve_cons_MPI
      else
         call mdcycle_nvt_cons_MPI
      end if

!     /*   stop on soft exit   */
      if ( iexit .eq. 1 ) return

!     /*   calculate averages   */
      call averages_analytical_afed( ioption_afed )

      return
      end





!***********************************************************************
      subroutine geteigen_afed_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use afed_variables, only : &
     &   hess_afed, fictmass_afed, eigval_afed, eigvec_afed, e_afed

      use cons_variables, only : &
     &   ncons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     //   real numbers
      real(8) :: a(ncons,ncons)

!     //   integers
      integer :: i, j

!-----------------------------------------------------------------------
!     /*   mass weighted hessian in cv space                          */
!-----------------------------------------------------------------------

      do i = 1, ncons
      do j = 1, ncons
         a(i,j) = 0.5d0 * ( hess_afed(i,j) + hess_afed(j,i) ) &
     &          / sqrt(fictmass_afed(i)*fictmass_afed(j))
      end do
      end do

!-----------------------------------------------------------------------
!     /*   diagonalize hessian                                        */
!-----------------------------------------------------------------------

      call ddiag_MPI( a, eigval_afed, eigvec_afed, ncons )

!-----------------------------------------------------------------------
!     /*   overwrite unit vector by lowest mode eigenvector           */
!-----------------------------------------------------------------------

      e_afed(:) = eigvec_afed(:,1)

      return
      end





!***********************************************************************
      subroutine analysis_afed_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, au_length, natom, iounit, nbead, species, myrank

      use afed_variables, only : &
     &   fenergy_afed, iiter_afed, afed_status, iprint_xyz_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j

!     /*   real numbers   */
      real(8), parameter :: bohr2ang = au_length/1.d-10

!     /*   real numbers   */
      real(8) :: xa, ya, za

!     /*   integer   */
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_int1_MPI &
     &      ( iprint_xyz_afed, '<iprint_xyz_afed>', 17, iounit )

         iset = 1

      end if

      if ( iprint_xyz_afed .le. 0 ) return

      if ( mod(iiter_afed,iprint_xyz_afed) .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   print xyz file                                             */
!-----------------------------------------------------------------------

!     /*   master only   */
      if ( myrank .ne. 0 ) return

!     /*   xyz format   */
      open ( iounit, file = 'afed.xyz', access = 'append' )

      do j = 1, nbead

         write( iounit, '(i5)' ) natom

         write( iounit, '(a2,2x,i5,a5,i5,a5,f10.5)' ) &
     &      afed_status(1:2), iiter_afed, ' ITER', j, ' BEAD', &
     &      fenergy_afed

         do i = 1, natom
            xa = x(i,j) * bohr2ang
            ya = y(i,j) * bohr2ang
            za = z(i,j) * bohr2ang
            write( iounit, '(a4,3f10.5)' ) species(i)(1:4), xa, ya, za
         end do

      end do

      close( iounit )

      return
      end





!***********************************************************************
      subroutine pbc_cv_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : ncons, ipbc_cons

      use afed_variables, only : rafed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i

!-----------------------------------------------------------------------
!     /*   periodic boundary condition                                */
!-----------------------------------------------------------------------

      do i = 1, ncons

         if ( ipbc_cons(i) .eq. 1 ) then

            rafed(i) = rafed(i) - 360.d0*nint(rafed(i)/360.d0)

         else

            continue

         end if

      end do

      return
      end





!***********************************************************************
      subroutine meanforce_numerical_afed_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iexit, istep_start, nstep

      use afed_variables, only : &
     &   nstep_pre_afed, nstep_pro_afed, mdcycle_pro_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   constrained molecular dynamics: preliminary run            */
!-----------------------------------------------------------------------

!     /*   initialize bath   */
      call init_bath_cart_revised_MPI

!     /*   set constraints   */
      call averages_numerical_afed( 0 )

!     /*   reset step number   */
      istep_start = 0

!     /*   reset number of steps   */
      nstep = nstep_pre_afed

!     /*   reset average forces   */
      call analysis_cons_MPI ( 0 )

!     /*   preliminary run    */
      call mdcycle_nvt_cons_MPI

!     /*   stop on soft exit   */
      if ( iexit .eq. 1 ) return

!-----------------------------------------------------------------------
!     /*   constrained molecular dynamics: productive run             */
!-----------------------------------------------------------------------

!     /*   reset step number   */
      istep_start = 0

!     /*   reset number of steps   */
      nstep = nstep_pro_afed

!     /*   reset average forces   */
      call analysis_cons_MPI ( 0 )

!     /*   productive run    */

      if ( mdcycle_pro_afed(1:4) .eq. 'NVE ' ) then
         call mdcycle_nve_cons_MPI
      else
         call mdcycle_nvt_cons_MPI
      end if

!     /*   stop on soft exit   */
      if ( iexit .eq. 1 ) return

!     /*   calculate averages   */
      call averages_numerical_afed( 1 )

      return
      end





!***********************************************************************
      subroutine lsfit_lin( xref, yref, yfit, nref, c0, c1 )
!***********************************************************************

      implicit none

      integer :: nref

      real(8) :: xref(nref), yref(nref), yfit(nref)

      integer :: i

      real(8) :: c0, c1, dn, sumx, sumy, sumxx, sumxy

      sumx  = 0.d0
      sumy  = 0.d0
      sumxx = 0.d0
      sumxy = 0.d0

      do i = 1, nref
         sumx  = sumx  + xref(i)
         sumy  = sumy  + yref(i)
         sumxx = sumxx + xref(i)*xref(i)
         sumxy = sumxy + xref(i)*yref(i)
      end do

      dn = dble(nref)

      c1 = ( dn*sumxy-sumx*sumy ) / ( dn*sumxx - sumx*sumx )
      c0 = ( sumy - c1*sumx ) / dn

      do i = 1, nref
         yfit(i) = c1*xref(i) + c0
      end do

      return
      end





!***********************************************************************
      subroutine lsfit_quad( xref, yref, yfit, nref, c0, c1, c2 )
!***********************************************************************

      implicit none

      integer :: nref

      real(8) ::  xref(nref), yref(nref), yfit(nref)

      real(8) :: a(3,3), ainv(3,3), b(3)

      integer :: i

      real(8) :: c0, c1, c2, x1, y1, x2, x3, x4

!----------------------------------------------------------------------

      a(:,:) = 0.d0
      b(:)   = 0.d0

      do i = 1, nref

         x1 = xref(i)
         y1 = yref(i)

         x2 = x1 * x1
         x3 = x1 * x2
         x4 = x2 * x2

         a(1,1) = a(1,1) + x4
         a(1,2) = a(1,2) + x3
         a(1,3) = a(1,3) + x2
         a(2,1) = a(2,1) + x3
         a(2,2) = a(2,2) + x2
         a(2,3) = a(2,3) + x1
         a(3,1) = a(3,1) + x2
         a(3,2) = a(3,2) + x1
         a(3,3) = a(3,3) + 1.d0

         b(1) = b(1) + x2 * y1
         b(2) = b(2) + x1 * y1
         b(3) = b(3) + y1

      end do

      call inv3 ( a, ainv )

      c2 = 0.d0
      c1 = 0.d0
      c0 = 0.d0

      do i = 1, 3
         c2 = c2 + ainv(1,i) * b(i)
         c1 = c1 + ainv(2,i) * b(i)
         c0 = c0 + ainv(3,i) * b(i)
      end do

      do i = 1, nref

         x1 = xref(i)
         x2 = x1 * x1

         yfit(i) = c2 * x2 + c1 * x1 + c0

      end do

      return
      end





!***********************************************************************
      subroutine averages_afed( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : nbead

      use afed_variables, only : ascent_sampling_afed, afed_status

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ioption

!-----------------------------------------------------------------------
!     /*   options                                                    */
!-----------------------------------------------------------------------

      if      ( afed_status(1:2) .ne. 'AS' ) then

         call averages_analytical_afed( ioption )

      else if ( nbead .eq. 1 ) then

         call averages_analytical_afed( ioption )

      else if ( ascent_sampling_afed(1:11) .eq. 'ANALYTICAL ' ) then

         call averages_analytical_afed( ioption )

      else

         call averages_numerical_afed( ioption )

      end if

      return
      end





!***********************************************************************
      subroutine averages_analytical_afed( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   beta, nbead, myrank, iounit, iounit_weight

      use cons_variables, only : &
     &   fc_cons, fref_cons_avg, f2ref_cons_avg, rcons, ncons

      use afed_variables, only : &
     &   f_afed, hess_afed, he_afed, e_afed, fictmass_afed, rafed, &
     &   work_afed, weight_afed, weight_sum_afed, v_afed, dt_afed, &
     &   iiter_afed, iiter_start_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8) :: f2_afed, f_tmp

      integer :: i, j, k, ioption, itest

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   set constraints                                            */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         do i = 1, nbead
         do j = 1, ncons
            rcons(j,i) = rafed(j)
         end do
         end do

      end if

!-----------------------------------------------------------------------
!     /*   calculate averages                                         */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         do i = 1, ncons

            f_afed(i) = 0.d0

            do k = 1, nbead
               f_afed(i) = f_afed(i) + fref_cons_avg(i,k)
            end do

            f_afed(i) = f_afed(i) / dble(nbead)

         end do

!-----------------------------------------------------------------------
!        /*   mean hessian                                            */
!-----------------------------------------------------------------------

         hess_afed(:,:) = 0.d0

         do i = 1, ncons
            hess_afed(i,i) = fc_cons(i)
         end do

         do j = 1, ncons
         do i = 1, ncons

            f2_afed = 0.d0

            do k = 1, nbead
               f2_afed = f2_afed + f2ref_cons_avg(i,j,k)
            end do

            f2_afed = f2_afed / dble(nbead)

            hess_afed(i,j) = hess_afed(i,j) &
     &                  - beta * ( f2_afed - f_afed(i) * f_afed(j) )

         end do
         end do

!-----------------------------------------------------------------------
!        /*   symmetrize hessian                                      */
!-----------------------------------------------------------------------

         do i = 1, ncons-1
         do j = i+1, ncons

            hess_afed(i,j) = 0.5d0 * ( hess_afed(i,j) + hess_afed(j,i) )

            hess_afed(j,i) = hess_afed(i,j)

         end do
         end do

!-----------------------------------------------------------------------
!        /*   gentlest ascent: (hessian * unit vector)                */
!-----------------------------------------------------------------------

         do i = 1, ncons

            he_afed(i) = 0.d0

            do j = 1, ncons

               he_afed(i) = he_afed(i) + hess_afed(i,j) * e_afed(j) &
     &                 / sqrt(fictmass_afed(i)*fictmass_afed(j))

            end do

         end do

      end if

!-----------------------------------------------------------------------
!     /*   calculate weights                                          */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         weight_afed = 0.d0
         weight_sum_afed = 0.d0

         do i = 1, nbead

            f_tmp = 0.d0

            do j = 1, ncons

               f_tmp = f_tmp + fref_cons_avg(j,i) * v_afed(j)

            end do

            if ( iiter_afed .eq. 0 ) then

               work_afed(i) = 0.d0

            else if (iiter_afed .gt. iiter_start_afed) then

!fixed          work_afed(i) = work_afed(i) + f_tmp * dt_afed
               work_afed(i) = work_afed(i) - f_tmp * dt_afed

            end if

            weight_afed(i) = exp(- beta * work_afed(i))
            weight_sum_afed = weight_sum_afed + weight_afed(i)

         end do

         weight_afed(:) = weight_afed(:) / weight_sum_afed

         do j = 1, ncons

            f_afed(j) = 0.d0

            do i = 1, nbead
               f_afed(j) = f_afed(j) &
     &                   + fref_cons_avg(j,i) * weight_afed(i)
            end do

         end do

!-----------------------------------------------------------------------
!        /*   print weights                                           */
!-----------------------------------------------------------------------

         if ( iset .eq. 0 ) then

            if ( myrank .ne. 0 ) then

            call testfile( 'afed_weight.out', 15, itest )

            if ( itest .eq. 1 ) then

               open ( iounit_weight, file = 'afed_weight.out' )

               write( iounit_weight, '(a)' ) &
     &         '====================================================' // &
     &         '=========================='

               write( iounit_weight, '(a)' ) &
     &         '    step    replica weight'

               write( iounit_weight, '(a)' ) &
     &         '----------------------------------------------------' // &
     &         '--------------------------'

               close( iounit_weight )

            end if

            end if

            iset = 1

         end if

!        /*   no duplication   */
         if ( ( iiter_afed .eq. 0 ) .or. &
     &        ( iiter_afed .gt. iiter_start_afed ) ) then

            if ( myrank .eq. 0 ) then

               open ( iounit_weight, file = 'afed_weight.out', &
     &                access = 'append' )

               write( iounit_weight, '(i8,100f16.8)' ) &
     &            iiter_afed, weight_afed(:)

               close( iounit_weight )

            end if

         end if

!-----------------------------------------------------------------------
!        /*   mean hessian                                            */
!-----------------------------------------------------------------------

         hess_afed(:,:) = 0.d0

         do i = 1, ncons
            hess_afed(i,i) = fc_cons(i)
         end do

         do j = 1, ncons
         do i = 1, ncons

            f2_afed = 0.d0

            do k = 1, nbead
               f2_afed = f2_afed + f2ref_cons_avg(i,j,k)
            end do

            f2_afed = f2_afed / dble(nbead)

            hess_afed(i,j) = hess_afed(i,j) &
     &                  - beta * ( f2_afed - f_afed(i) * f_afed(j) )

         end do
         end do

!-----------------------------------------------------------------------
!        /*   symmetrize hessian                                      */
!-----------------------------------------------------------------------

         do i = 1, ncons-1
         do j = i+1, ncons

            hess_afed(i,j) = 0.5d0 * ( hess_afed(i,j) + hess_afed(j,i) )

            hess_afed(j,i) = hess_afed(i,j)

         end do
         end do

!-----------------------------------------------------------------------
!        /*   gentlest ascent: (hessian * unit vector)                */
!-----------------------------------------------------------------------

         do i = 1, ncons

            he_afed(i) = 0.d0

            do j = 1, ncons

               he_afed(i) = he_afed(i) + hess_afed(i,j) * e_afed(j) &
     &                 / sqrt(fictmass_afed(i)*fictmass_afed(j))

            end do

         end do

      end if

      return
      end





!***********************************************************************
      subroutine averages_numerical_afed( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   beta, nbead

      use cons_variables, only : &
     &   fc_cons, fref_cons_avg, f2ref_cons_avg, rcons, ncons, ipbc_cons

      use afed_variables, only : &
     &   f_afed, hess_afed, fictmass_afed, he_afed, e_afed, rafed, &
     &   fdiff_sampling_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8) :: f2_afed, eps, c0, c1, c2, ds

      real(8) :: xref(nbead), yref(nbead), yfit(nbead)

      integer :: i, j, k, ioption

!-----------------------------------------------------------------------
!     /*   set constraints                                            */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         do k = 1, nbead
         do i = 1, ncons

            eps = dble(2*k-nbead-1)/dble(nbead-1)*fdiff_sampling_afed

            rcons(i,k) = rafed(i) &
     &          + e_afed(i) / sqrt(fictmass_afed(i)) * eps

         end do
         end do

         do k = 1, nbead
         do i = 1, ncons

            ds = rcons(i,k)

            call pbc_cons( ds, ipbc_cons(i) )

            rcons(i,k) = ds

         end do
         end do

      end if

!-----------------------------------------------------------------------
!     /*   calculate averages                                         */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         do i = 1, ncons

            f_afed(i) = 0.d0

            do k = 1, nbead

               eps = dble(2*k-nbead-1)/dble(nbead-1)*fdiff_sampling_afed

               xref(k) = eps
               yref(k) = fref_cons_avg(i,k)

            end do

            if ( nbead .eq. 2 ) then
               call lsfit_lin( xref, yref, yfit, nbead, c0, c1 )
            else
               call lsfit_quad( xref, yref, yfit, nbead, c0, c1, c2 )
            end if

            f_afed(i)  =   c0

            he_afed(i) = - c1 / sqrt(fictmass_afed(i))

         end do

!-----------------------------------------------------------------------
!        /*   mean hessian                                            */
!-----------------------------------------------------------------------

         hess_afed(:,:) = 0.d0

         do i = 1, ncons
            hess_afed(i,i) = fc_cons(i)
         end do

         do j = 1, ncons
         do i = 1, ncons

            do k = 1, nbead

               eps = dble(2*k-nbead-1)/dble(nbead-1)*fdiff_sampling_afed

               xref(k) = eps
               yref(k) = f2ref_cons_avg(i,j,k)

            end do

            if ( nbead .eq. 2 ) then
               call lsfit_lin( xref, yref, yfit, nbead, c0, c1 )
            else
               call lsfit_quad( xref, yref, yfit, nbead, c0, c1, c2 )
            end if

            f2_afed = c0

            hess_afed(i,j) = hess_afed(i,j) &
     &                  - beta * ( f2_afed - f_afed(i) * f_afed(j) )

         end do
         end do

         do j = 1, ncons-1
         do i = j+1, ncons
            hess_afed(i,j) = 0.5d0 * ( hess_afed(i,j) + hess_afed(j,i) )
            hess_afed(j,i) = hess_afed(i,j)
         end do
         end do

      end if

      return
      end





!***********************************************************************
      subroutine setup_adescent_afed_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, nbead, myrank

      use cons_variables, only : &
     &   ncons, itype_cons, ntype_cons

      use afed_variables, only : &
     &   f_afed, hess_afed, gamma_ascent_afed, e_afed, he_afed, &
     &   f_old_afed, fictmass_afed, scons_mean_afed, rafed_old, &
     &   eigval_afed, eigvec_afed, v_afed, d_old_afed, hess_old_afed, &
     &   radius_auto_afed, dt_conv_afed, dt_damp_afed, fenergy_max_afed, &
     &   rafed, dt_descent_afed, dt_ascent_afed, ascent_sampling_afed, &
     &   fdiff_sampling_afed, params_afed, v_old_afed, algo_ascent_afed, &
     &   e_old_afed, scons_avg_old_afed, weight_afed, work_afed, &
     &   nmiss_auto_afed, nshot_auto_afed, ioption_eigen_afed, d_afed, &
     &   niter_afed, nstep_pre_afed, nstep_pro_afed, iprint_test_afed, &
     &   mdcycle_pro_afed, iconv_afed, ioption_afed, niter_refresh_afed, &
     &   niter_gad2evf_afed

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: itest, i, k, ierr

      character(len=8) :: params_char

!-----------------------------------------------------------------------
!     //   convergence
!-----------------------------------------------------------------------

      iconv_afed = 0

!-----------------------------------------------------------------------
!     //   setup constrained molecular dynamics
!-----------------------------------------------------------------------

      call setup_md_nvt_MPI

!-----------------------------------------------------------------------
!     //   get collective variables
!-----------------------------------------------------------------------

      call get_cv_cons_MPI

!-----------------------------------------------------------------------
!     //   read parameters
!-----------------------------------------------------------------------

!c     //   method
!      call read_char_MPI ( afed_type, 8, '<afed_type>', 11, iounit )

!     //   read number of steps
      call read_int1_MPI ( niter_afed, '<niter_afed>', 12, iounit )

!     //   read number of steps for production run
      call read_int1_MPI &
     &   ( nstep_pro_afed, '<nstep_pro_afed>', 16, iounit )

!     //   read number of steps for preliminary run
      call read_int1_MPI &
     &   ( nstep_pre_afed, '<nstep_pre_afed>', 16, iounit )

!     //   read step size
      call read_real1_MPI &
     &   ( gamma_ascent_afed, '<gamma_ascent_afed>', 19, iounit )

!     //   read number of shots
      call read_int1_MPI &
     &   ( nshot_auto_afed, '<nshot_auto_afed>', 17, iounit )

!     //   read radius criterion
      call read_real1_MPI &
     &   ( radius_auto_afed, '<radius_auto_afed>', 18, iounit )

!     //   step size at convergence
      call read_real1_MPI &
     &   ( dt_conv_afed, '<dt_conv_afed>', 14, iounit )

!     //   damping factor of step size
      call read_real1_MPI &
     &   ( dt_damp_afed, '<dt_damp_afed>', 14, iounit )

!     //   read maximum free energy criterion
      call read_real1_MPI &
     &   ( fenergy_max_afed, '<fenergy_max_afed>', 18, iounit )

!     //   read maximum number of consecutive fails
      call read_int1_MPI &
     &   ( nmiss_auto_afed, '<nmiss_auto_afed>', 17, iounit )

!     //   bead sampling for ascent: gradient and hessian
      call read_char_MPI &
     &   ( ascent_sampling_afed, 11, '<ascent_sampling_afed>', 22, &
     &      iounit )

!     //   finite difference parameter
      call read_real1_MPI &
     &   ( fdiff_sampling_afed, '<fdiff_sampling_afed>', 21, iounit )

!     //   step size for ascent dynamics
      call read_real1_MPI &
     &   ( dt_ascent_afed, '<dt_ascent_afed>', 16, iounit )

!     //   step size for descent dynamics
      call read_real1_MPI &
     &   ( dt_descent_afed, '<dt_descent_afed>', 17, iounit )

!     //   read print interval of convergence test
      call read_int1_MPI &
     &   ( iprint_test_afed, '<iprint_test_afed>', 18, iounit )

!     //   read option for eigenvalue calculations
      call read_int1_MPI &
     &   ( ioption_eigen_afed, '<ioption_eigen_afed>', 20, iounit )

!     //   molecular dynamics cycle in productive runs
      call read_char_MPI &
     &   ( mdcycle_pro_afed, 4, '<mdcycle_pro_afed>', 18, iounit )

!     //   read mean force option
      call read_int1_MPI ( ioption_afed, '<ioption_afed>', 14, iounit )

!     //   gad or evf
      call read_char_MPI &
     &   ( algo_ascent_afed, 4, '<algo_ascent_afed>', 18, iounit )

!     //   maximum iterations of gad without evf refreshment
      call read_int1_MPI &
     &   ( niter_refresh_afed, '<niter_refresh_afed>', 20, iounit )

!     //   iterations when switching gad to evf
      call read_int1_MPI &
     &   ( niter_gad2evf_afed, '<niter_gad2evf_afed>', 20, iounit )

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

!     //   memory allocation: mean force
      if ( .not. allocated( f_afed ) ) &
     &   allocate( f_afed(ncons) )

!     //   memory allocation: old mean force
      if ( .not. allocated( f_old_afed ) ) &
     &   allocate( f_old_afed(ncons) )

!     //   memory allocation: mean hessian
      if ( .not. allocated( hess_afed ) ) &
     &   allocate( hess_afed(ncons,ncons) )

!     //   memory allocation: old mean hessian
      if ( .not. allocated( hess_old_afed ) ) &
     &   allocate( hess_old_afed(ncons,ncons) )

!     //   memory allocation: unit vector
      if ( .not. allocated( e_afed ) ) &
     &   allocate( e_afed(ncons) )

!     //   memory allocation: fictitious mass
      if ( .not. allocated( fictmass_afed ) ) &
     &   allocate( fictmass_afed(ncons) )

!     //   memory allocation: mean cv value
      if ( .not. allocated( scons_mean_afed ) ) &
     &   allocate( scons_mean_afed(ncons) )

!     //   memory allocation: eigenvalues
      if ( .not. allocated( eigval_afed ) ) &
     &   allocate( eigval_afed(ncons) )

!     //   memory allocation: eigenvectors
      if ( .not. allocated( eigvec_afed ) ) &
     &   allocate( eigvec_afed(ncons,ncons) )

!     //   memory allocation: cv
      if ( .not. allocated( rafed ) ) &
     &   allocate( rafed(ncons) )

!     //   memory allocation: parameters
      if ( .not. allocated( params_afed ) ) &
     &   allocate( params_afed(ntype_cons,3) )

!     //   memory allocation: shift vector
      if ( .not. allocated( he_afed ) ) &
     &   allocate( he_afed(ncons) )

!     //   memory allocation: direction of current step
      if ( .not. allocated( d_afed ) ) &
     &   allocate( d_afed(ncons) )

!     //   memory allocation: direction of previous step
      if ( .not. allocated( d_old_afed ) ) &
     &   allocate( d_old_afed(ncons) )

!     //   memory allocation: unit vector of previous step
      if ( .not. allocated( e_old_afed ) ) &
     &   allocate( e_old_afed(ncons) )

!     //   memory allocation: position of previous step
      if ( .not. allocated( rafed_old ) ) &
     &   allocate( rafed_old(ncons) )

!     //   memory allocation: average position of previous step
      if ( .not. allocated( scons_avg_old_afed ) ) &
     &   allocate( scons_avg_old_afed(ncons,nbead) )

!     //   memory allocation: cv velocity of current step
      if ( .not. allocated( v_afed ) ) &
     &   allocate( v_afed(ncons) )

!     //   memory allocation: cv velocity of previous step
      if ( .not. allocated( v_old_afed ) ) &
     &   allocate( v_old_afed(ncons) )

!     //   memory allocation: weight parameter of replica
      if ( .not. allocated( weight_afed ) ) &
     &   allocate( weight_afed(nbead) )

!     //   memory allocation: work parameter of replica
      if ( .not. allocated( work_afed ) ) &
     &   allocate( work_afed(nbead) )

!-----------------------------------------------------------------------
!     /*   read parameters for constraint                             */
!-----------------------------------------------------------------------

!     /*   master process only   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<params_afed>', 13, iounit, ierr )

!     /*   parameters of constraint type   */

      if ( ierr .eq. 0 ) then

         do i = 1, ntype_cons

            read ( iounit, *, iostat=ierr ) params_char

            backspace( iounit )

            if      ( ( params_char(1:6) .eq. '1     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIST  ' ) ) then
               k = 1
            else if ( ( params_char(1:6) .eq. '2     ' ) .or. &
     &                ( params_char(1:6) .eq. 'ANGL  ' ) ) then
               k = 2
            else if ( ( params_char(1:6) .eq. '3     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIH   ' ) ) then
               k = 3
            else if ( ( params_char(1:6) .eq. '4     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIFF  ' ) ) then
               k = 4
            else if ( ( params_char(1:6) .eq. '5     ' ) .or. &
     &                ( params_char(1:6) .eq. 'CN    ' ) ) then
               k = 5
            else if ( ( params_char(1:6) .eq. '6     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DCN   ' ) ) then
               k = 6
            else if ( ( params_char(1:6) .eq. '7     ' ) .or. &
     &                ( params_char(1:6) .eq. 'XYZ   ' ) ) then
               k = 7
            else if ( ( params_char(1:6) .eq. '8     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DXYZ  ' ) ) then
               k = 8
            else
               ierr = 1
            end if

            if ( ierr .ne. 0 ) exit

            read ( iounit, *, iostat=ierr ) &
     &         params_char, params_afed(k,1), params_afed(k,2), &
     &                      params_afed(k,3)

            if ( ierr .ne. 0 ) exit

         end do

      end if

!     /*   file close   */
      close( iounit )

!     /*   if error is found, read default values   */

      if ( ierr .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<params_afed>', 13, iounit, ierr )

!        /*   parameters of constraint type   */

         do i = 1, ntype_cons

            read ( iounit, *, iostat=ierr ) params_char

            backspace( iounit )

            if      ( ( params_char(1:6) .eq. '1     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIST  ' ) ) then
               k = 1
            else if ( ( params_char(1:6) .eq. '2     ' ) .or. &
     &                ( params_char(1:6) .eq. 'ANGL  ' ) ) then
               k = 2
            else if ( ( params_char(1:6) .eq. '3     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIH   ' ) ) then
               k = 3
            else if ( ( params_char(1:6) .eq. '4     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIFF  ' ) ) then
               k = 4
            else if ( ( params_char(1:6) .eq. '5     ' ) .or. &
     &                ( params_char(1:6) .eq. 'CN    ' ) ) then
               k = 5
            else if ( ( params_char(1:6) .eq. '6     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DCN   ' ) ) then
               k = 6
            else if ( ( params_char(1:6) .eq. '7     ' ) .or. &
     &                ( params_char(1:6) .eq. 'XYZ   ' ) ) then
               k = 7
            else if ( ( params_char(1:6) .eq. '8     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DXYZ  ' ) ) then
               k = 8
            else
               ierr = 1
            end if

            if ( ierr .ne. 0 ) exit

            read ( iounit, *, iostat=ierr ) &
     &         params_char, params_afed(k,1), params_afed(k,2), &
     &                      params_afed(k,3)

            if ( ierr .ne. 0 ) exit

         end do

!        /*   file close   */
         close( iounit )

!     /*   end   */
      end if

      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - keyword <params_afed> is incorrect.'
      end if

!     /*   master process only   */
      end if

!     /*   broadcast   */
      call my_mpi_bcast_int_0 ( ierr )

!     /*   check error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine setup_adescent_afed_MPI', 34 )

!     /*   broadcast   */
      call my_mpi_bcast_real_2 ( params_afed, ntype_cons, 3 )

!-----------------------------------------------------------------------
!     /*   fictitious mass                                            */
!-----------------------------------------------------------------------

      do i = 1, ncons

         k = itype_cons(i)

         fictmass_afed(i) = 1.d0 / params_afed(k,1)**2

      end do

!-----------------------------------------------------------------------
!     //   initialize or restart
!-----------------------------------------------------------------------

!     //   check existence of restart file
      if ( myrank .eq. 0 ) call testfile( 'afed.ini', 8, itest )

!     //   broadcast
      call my_mpi_bcast_int_0( itest )

!     //   restart file: not found
      if ( itest .eq. 1 ) then

!        //   initialize
         call init_adescent_afed_MPI

!     //   restart file: found
      else

!        //   restart
         call restart_adescent_afed_MPI

!     //   restart file
      end if

      return
      end





!***********************************************************************
      subroutine init_adescent_afed_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   nbead, afed_type

      use cons_variables, only : &
     &   scons, ncons, ipbc_cons

      use afed_variables, only : &
     &   e_afed, rafed, dt_ascent_afed, dt_afed, dt_descent_afed, &
     &   dt_conv_afed, v_afed, xi_afed, fenergy_afed, afed_status, &
     &   iconv_afed, jconv_afed, iroot_auto_afed, iiter_start_afed

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      real(8) :: s, savg

      integer :: i, j

!-----------------------------------------------------------------------
!     //   set afed_status: status
!-----------------------------------------------------------------------

      if ( afed_type(1:5) .eq. 'TEST '    ) afed_status(1:2) = 'TE'
      if ( afed_type(1:5) .eq. 'GRAD '    ) afed_status(1:2) = 'GR'
      if ( afed_type(1:8) .eq. 'HESSIAN ' ) afed_status(1:2) = 'HE'
      if ( afed_type(1:8) .eq. 'DESCENT ' ) afed_status(1:2) = 'DE'
      if ( afed_type(1:7) .eq. 'ASCENT '  ) afed_status(1:2) = 'AS'
      if ( afed_type(1:5) .eq. 'AUTO '    ) afed_status(1:2) = 'DE'

!-----------------------------------------------------------------------
!     //    set iiter_start_afed: step number
!-----------------------------------------------------------------------

      iiter_start_afed = 0

!-----------------------------------------------------------------------
!     //   set free energy
!-----------------------------------------------------------------------

      fenergy_afed = 0.d0

!-----------------------------------------------------------------------
!     //   set rafed: cv position to the center of mass of scons
!-----------------------------------------------------------------------

      do i = 1, ncons

         rafed(i) = 0.d0

         savg = 0.d0

         do j = 1, nbead

            s = scons(i,j) - scons(i,1)

            call pbc_cons( s, ipbc_cons(i) )

            s = s + scons(i,1)

            savg = savg + s

         end do

         savg = savg / dble(nbead)

         rafed(i) = rafed(i) + savg

      end do

!     //   apply periodic boundary condition
      call pbc_cv_afed

!-----------------------------------------------------------------------
!     //   set dt_afed: afed step size
!-----------------------------------------------------------------------

      if ( afed_type(1:5) .eq. 'TEST '    ) dt_afed = 0.d0
      if ( afed_type(1:5) .eq. 'GRAD '    ) dt_afed = 0.d0
      if ( afed_type(1:8) .eq. 'HESSIAN ' ) dt_afed = 0.d0
      if ( afed_type(1:8) .eq. 'DESCENT ' ) dt_afed = dt_descent_afed
      if ( afed_type(1:7) .eq. 'ASCENT '  ) dt_afed = dt_ascent_afed
      if ( afed_type(1:5) .eq. 'AUTO '    ) dt_afed = dt_descent_afed

!-----------------------------------------------------------------------
!     //   reset iconv_afed: full convergence
!-----------------------------------------------------------------------

      if ( ( afed_type(1:5) .eq. 'TEST '    ) .or. &
     &     ( afed_type(1:5) .eq. 'GRAD '    ) .or. &
     &     ( afed_type(1:8) .eq. 'HESSIAN ' ) ) then

!        //  not converged
         iconv_afed = 0

      end if

      if ( ( afed_type(1:8) .eq. 'DESCENT ' ) .or. &
     &     ( afed_type(1:7) .eq. 'ASCENT '  ) .or. &
     &     ( afed_type(1:5) .eq. 'AUTO '    ) ) then

         if ( dt_afed .gt. dt_conv_afed ) then

!           //  not converged
            iconv_afed = 0

         else

!           //  already converged
            iconv_afed = 1

         end if

      end if

!-----------------------------------------------------------------------
!     //   set velocity
!-----------------------------------------------------------------------

      v_afed(:) = 0.d0

!-----------------------------------------------------------------------
!     //   set unit vector
!-----------------------------------------------------------------------

      e_afed(:) = 0.d0
      e_afed(1) = 1.d0

!-----------------------------------------------------------------------
!     //   root point of trajectory
!-----------------------------------------------------------------------

      iroot_auto_afed = 0

!-----------------------------------------------------------------------
!     //   set jconv_afed: close to convergence
!-----------------------------------------------------------------------

!     //   initialize convergence: not converging
      jconv_afed = 0

!-----------------------------------------------------------------------
!     //   set angle
!-----------------------------------------------------------------------

      xi_afed = 1.d0

      return
      end





!***********************************************************************
      subroutine restart_adescent_afed_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, afed_type, myrank

      use afed_variables, only : &
     &   e_afed, fenergy_afed, rafed, dt_afed, v_afed, fenergy_afed, &
     &   dt_conv_afed, xi_afed, dt_descent_afed, dt_ascent_afed, &
     &   iroot_auto_afed, iiter_start_afed, iconv_afed, jconv_afed, &
     &   afed_status

      use cons_variables, only : &
     &   ncons

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: i

!-----------------------------------------------------------------------
!     //   check status
!-----------------------------------------------------------------------

!     //   master rank
      if ( myrank .eq. 0 ) then

         open ( iounit, file = 'afed.ini' )
            read ( iounit, * ) afed_status
         close( iounit )

!-----------------------------------------------------------------------
!        //   read restart file: TAMD
!-----------------------------------------------------------------------

!        //   TAMD or LOGMFD
         if ( ( afed_status(1:2) .eq. 'TA' ) .or. &
     &        ( afed_status(1:2) .eq. 'LO' ) ) then

!           //   open file
            open ( iounit, file = 'afed.ini' )

!           //   read afed_status: status
            read ( iounit, * ) afed_status

!           //   read iiter_start_afed: afed step number
            read ( iounit, * ) iiter_start_afed

!           //   read free energy
            read ( iounit, * ) fenergy_afed

!           //   read collective variables
            do i = 1, ncons
               read ( iounit, * ) rafed(i)
            end do

!           //   close file
            close( iounit )

!           //   reset afed_status: status
            if ( afed_type(1:5) .eq. 'TEST '    ) &
     &         afed_status(1:2) = 'TE'
            if ( afed_type(1:5) .eq. 'GRAD '    ) &
     &         afed_status(1:2) = 'GR'
            if ( afed_type(1:8) .eq. 'HESSIAN ' ) &
     &         afed_status(1:2) = 'HE'
            if ( afed_type(1:8) .eq. 'DESCENT ' ) &
     &         afed_status(1:2) = 'DE'
            if ( afed_type(1:7) .eq. 'ASCENT '  ) &
     &         afed_status(1:2) = 'AS'
            if ( afed_type(1:5) .eq. 'AUTO '    ) &
     &         afed_status(1:2) = 'DE'

!           //   reset unit vector
            e_afed(:) = 0.d0
            e_afed(1) = 1.d0

!           //   reset root point of trajectory
            iroot_auto_afed = 0

!           //   reset dt_afed: afed step size
            if ( afed_type(1:5) .eq. 'TEST '    ) &
     &         dt_afed = 0.d0
            if ( afed_type(1:5) .eq. 'GRAD '    ) &
     &         dt_afed = 0.d0
            if ( afed_type(1:8) .eq. 'HESSIAN ' ) &
     &         dt_afed = 0.d0
            if ( afed_type(1:8) .eq. 'DESCENT ' ) &
     &         dt_afed = dt_descent_afed
            if ( afed_type(1:7) .eq. 'ASCENT '  ) &
     &         dt_afed = dt_ascent_afed
            if ( afed_type(1:5) .eq. 'AUTO '    ) &
     &         dt_afed = dt_descent_afed

!        //   not TAMD or LOGMFD
         else

!           //   open file
            open ( iounit, file = 'afed.ini' )

!           //   read afed_status: status
            read ( iounit, * ) afed_status

!           //   read iiter_start_afed: afed step number
            read ( iounit, * ) iiter_start_afed

!           //   read free energy
            read ( iounit, * ) fenergy_afed

!           //   read collective variables
            do i = 1, ncons
               read ( iounit, * ) rafed(i), e_afed(i)
            end do

!           //   root point of trajectory
            read ( iounit, * ) iroot_auto_afed

!           //   read dt_afed: afed step size
            read ( iounit, * ) dt_afed

!           //   close file
            close( iounit )

!        //   TAMD, LOGMFD or others
         end if

!     //   master rank
      end if

!-----------------------------------------------------------------------
!     //   broadcast data
!-----------------------------------------------------------------------

      call my_mpi_bcast_char_0( afed_status, 2 )
      call my_mpi_bcast_int_0 ( iiter_start_afed )
      call my_mpi_bcast_real_0( fenergy_afed )
      call my_mpi_bcast_real_1( rafed, ncons )
      call my_mpi_bcast_real_1( e_afed, ncons )
      call my_mpi_bcast_int_0 ( iroot_auto_afed )
      call my_mpi_bcast_real_0( dt_afed )

!-----------------------------------------------------------------------
!     //   reset iconv_afed: full convergence
!-----------------------------------------------------------------------

      if ( ( afed_type(1:5) .eq. 'TEST '    ) .or. &
     &     ( afed_type(1:5) .eq. 'GRAD '    ) .or. &
     &     ( afed_type(1:8) .eq. 'HESSIAN ' ) ) then

!        //  not converged
         iconv_afed = 0

      end if

      if ( ( afed_type(1:8) .eq. 'DESCENT ' ) .or. &
     &     ( afed_type(1:7) .eq. 'ASCENT '  ) .or. &
     &     ( afed_type(1:5) .eq. 'AUTO '    ) ) then

         if ( dt_afed .gt. dt_conv_afed ) then

!           //  not converged
            iconv_afed = 0

         else

!           //  already converged
            iconv_afed = 1

         end if

      end if

!-----------------------------------------------------------------------
!     //   reset jconv_afed: close to convergence
!-----------------------------------------------------------------------

!     //   initialize convergence: not converging
      jconv_afed = 0

!-----------------------------------------------------------------------
!     //   reset velocity
!-----------------------------------------------------------------------

      v_afed(:) = 0.d0

!-----------------------------------------------------------------------
!     //   reset angle
!-----------------------------------------------------------------------

      xi_afed = 1.d0

      return
      end





!***********************************************************************
      subroutine afedcycle_test_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iexit, istep_start, nstep, istep

      use afed_variables, only : &
     &   nstep_pre_afed, nstep_pro_afed, iprint_test_afed, &
     &   mdcycle_pro_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   set constraints                                            */
!-----------------------------------------------------------------------

      call averages_afed( 0 )

!-----------------------------------------------------------------------
!     /*   constrained molecular dynamics: preliminary run            */
!-----------------------------------------------------------------------

!     /*   reset step number   */
      istep_start = 0

!     /*   reset number of steps   */
      nstep = nstep_pre_afed

!     /*   reset average forces   */
      call analysis_cons_MPI ( 0 )

!     /*   preliminary run    */
      call mdcycle_nvt_cons_MPI

!     /*   stop on soft exit   */
      if ( iexit .eq. 1 ) return

!-----------------------------------------------------------------------
!     /*   constrained molecular dynamics: productive run             */
!-----------------------------------------------------------------------

!     /*   reset step number   */
      istep_start = 0

!     /*   reset average forces   */
      call analysis_cons_MPI ( 0 )

      do

!        /*   reset number of steps   */
         nstep = min( istep_start+iprint_test_afed, nstep_pro_afed )

!        /*   productive run    */

         if ( mdcycle_pro_afed(1:4) .eq. 'NVE ' ) then
            call mdcycle_nve_cons_MPI
         else
            call mdcycle_nvt_cons_MPI
         end if

!        //   stop on soft exit
         if ( iexit .eq. 1 ) return

!        /*   output restart   */
         call backup_md_nvt_MPI

!        /*   calculate averages   */
         call averages_afed( 1 )

!        //   standard output
         call standard_adescent_afed_MPI

!        //   exit
         if ( istep .ge. nstep_pro_afed ) return

!        /*   reset step number   */
         istep_start = istep

      end do

      return
      end





!***********************************************************************
      subroutine afedcycle_descent_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iexit

      use afed_variables, only : &
     &   fenergy_afed, fenergy_init_afed, iiter_start_afed, &
     &   iiter_end_afed, niter_afed, iconv_afed, iiter_afed, &
     &   jconv_afed, ioption_eigen_afed, afed_status

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

!     //   starting step
      iiter_afed = iiter_start_afed

!     //   current step
      iiter_end_afed = iiter_afed

!     //   run constrained molecular dynamics and get meanforce
      call meanforce_afed_MPI

!     //   initial free energy
      fenergy_init_afed = fenergy_afed

!     //   stop on soft exit
      if ( iexit .eq. 1 ) return

!     //   update d, e, f
      call updates_descent_afed

!     //   descent status started
      if ( ( afed_status(1:2) .ne. 'D1' ) .and. &
     &     ( afed_status(1:2) .ne. 'D2' ) ) afed_status(1:2) = 'DE'

!     //   check convergence and accept or reject cv move
      call save_cv_afed

!     /*   judge if converged   */
      call judge_converged_afed

!     //   standard output
      call standard_adescent_afed_MPI

!     //   analysis
      call analysis_afed_MPI

!     //   save data
      call backup_adescent_afed_MPI

!     //   stop on convergence or error
      call softexit_afed_MPI

!     //   start iteration loop
      do iiter_afed = iiter_start_afed+1, niter_afed

!        //   stop on convergence or error
         if ( iconv_afed .ne. 0 ) exit

!        //   current step
         iiter_end_afed = iiter_afed

!        //   move cv in forward direction
         call rupdate_forward_afed

!        //   run constrained molecular dynamics and get meanforce
         call meanforce_afed_MPI

!        //   prepare for soft exit
         if ( iexit .eq. 1 ) call restore_cv_afed

!        //   stop on soft exit
         if ( iexit .eq. 1 ) return

!        //   update d, e, f
         call updates_descent_afed

!        //   update if last step is accepted
         call fupdate_afed

!        /*   judge if converging   */
         call judge_converging_afed

!        //   accept and save this step
         if ( jconv_afed .eq. 0 ) call save_cv_afed

!        //   reject and restore last step
         if ( jconv_afed .eq. 1 ) call restore_cv_afed

!        /*   update dt   */
         call update_dt_afed

!        /*   judge if converged   */
         call judge_converged_afed

!        //   standard output
         call standard_adescent_afed_MPI

!        //   analysis
         call analysis_afed_MPI

!        //   save data
         call backup_adescent_afed_MPI

!     //   end iteration loop
      end do

!     //   confirm minimum
      if ( ( iconv_afed .eq. 1 ) .and. &
     &     ( ioption_eigen_afed .eq. 1 ) ) then

!        //   calculate hessian
         call meanhessian_afed_MPI

!        //   eigenvalues and eigenvectors (overwrite e_afed)
         call geteigen_afed_MPI

!        //   print eigenvalues and eigenvectors
         call printeigen_afed_MPI

!     //   confirm minimum
      end if

!     /*   last step   */
      iiter_afed = iiter_end_afed

      return
      end





!***********************************************************************
      subroutine afedcycle_ascent_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iexit

      use afed_variables, only : &
     &   iiter_afed, iiter_start_afed, iiter_end_afed, niter_afed, &
     &   iconv_afed, jconv_afed, ioption_eigen_afed, afed_status

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

!     //   starting step
      iiter_afed = iiter_start_afed

!     //   current step
      iiter_end_afed = iiter_afed

!     //   run constrained molecular dynamics and get meanforce
      call meanforce_afed_MPI

!     //   stop on soft exit
      if ( iexit .eq. 1 ) return

!     //   update d, e, f
      call updates_ascent_afed_MPI

!     //   ascent status started
      afed_status(1:2) = 'AS'

!     //   check convergence and accept or reject cv move
      call save_cv_afed

!     /*   judge if converged   */
      call judge_converged_afed

!     //   standard output
      call standard_adescent_afed_MPI

!     //   analysis
      call analysis_afed_MPI

!     //   save data
      call backup_adescent_afed_MPI

!     //   stop on convergence or error
      call softexit_afed_MPI

!     //   start iteration loop
      do iiter_afed = iiter_start_afed+1, niter_afed

!        //   stop on convergence or error
         if ( iconv_afed .ne. 0 ) exit

!        //   current step
         iiter_end_afed = iiter_afed

!        //   move cv in forward direction
         call rupdate_forward_afed

!        //   run constrained molecular dynamics and get meanforce
         call meanforce_afed_MPI

!        //   prepare for soft exit
         if ( iexit .eq. 1 ) call restore_cv_afed

!        //   stop on soft exit
         if ( iexit .eq. 1 ) return

!        //   update d, e, f
         call updates_ascent_afed_MPI

!        //   update if last step is accepted
         call fupdate_afed

!        /*   judge if converging   */
         call judge_converging_afed

!        //   accept and save this step
         if ( jconv_afed .eq. 0 ) call save_cv_afed

!        //   reject and restore last step
         if ( jconv_afed .eq. 1 ) call restore_cv_afed

!        //   reset unit vector
         if ( jconv_afed .eq. 1 ) call ereset_ascent_afed

!        /*   update dt   */
         call update_dt_afed

!        /*   judge if converged   */
         call judge_converged_afed

!        //   standard output
         call standard_adescent_afed_MPI

!        //   analysis
         call analysis_afed_MPI

!        //   save data
         call backup_adescent_afed_MPI

!     //   end iteration loop
      end do

!     //   confirm saddle
      if ( ( iconv_afed .eq. 1 ) .and. &
     &     ( ioption_eigen_afed .eq. 1 ) ) then

!        //   calculate hessian
         call meanhessian_afed_MPI

!        //   eigenvalues and eigenvectors (overwrite e_afed)
         call geteigen_afed_MPI

!        //   print eigenvalues and eigenvectors
         call printeigen_afed_MPI

!     //   confirm saddle
      end if

!     /*   last step   */
      iiter_afed = iiter_end_afed

      return
      end





!***********************************************************************
      subroutine softexit_afed_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : myrank

      use afed_variables, only : iconv_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      if ( iconv_afed .le. -1 ) then

         if ( myrank .eq. 0 ) then

            write( 6, '(a)' ) &
     &         'Error termination: CV is out of bounds at the outset.'

         end if

         call my_mpi_finalize_2

         stop

      end if

      return
      end





!***********************************************************************
      subroutine updates_descent_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use afed_variables, only : &
     &   d_afed, e_afed, fenergy_afed, fenergy_init_afed, &
     &   dfenergy_init_afed, iiter_afed, iiter_start_afed, &
     &   iiter_skip_afed, afed_status

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     //   free energy change
      real(8) :: dfenergy_afed

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   restarted step   */
      if ( iiter_afed .eq. iiter_start_afed ) then

!        /*   zeroth step   */
         if ( iiter_afed .eq. 0 ) then

!           /*   update direction   */
            if ( afed_status(1:2) .eq. 'DE' ) call dupdate_descent_afed
            if ( afed_status(1:2) .eq. 'D1' ) d_afed(:) = + e_afed(:)
            if ( afed_status(1:2) .eq. 'D2' ) d_afed(:) = - e_afed(:)

!           //   get cv velocity
            if ( afed_status(1:2) .eq. 'DE' ) call vupdate_descent_afed
            if ( afed_status(1:2) .eq. 'D1' ) call vupdate_shift_afed
            if ( afed_status(1:2) .eq. 'D2' ) call vupdate_shift_afed

!           /*   unit vector in force direction   */
            if ( afed_status(1:2) .eq. 'DE' ) call eupdate_descent_afed

!        /*   other steps   */
         else

!           /*   update direction   */
            call dupdate_descent_afed

!           //   get cv velocity
            call vupdate_descent_afed

!           /*   unit vector in force direction   */
            call eupdate_descent_afed

!        /*   other steps   */
         end if

!     /*   continued step   */
      else

!        //   for DE
         if ( afed_status(1:2) .eq. 'DE' ) then

!           /*   update direction   */
            call dupdate_descent_afed

!           //   get cv velocity
            call vupdate_descent_afed

!           /*   unit vector in force direction   */
            call eupdate_descent_afed

!        //    for initial five iterations in D1 and D2
         else if ( iiter_afed .le. iiter_skip_afed ) then

!           //   free energy change
            dfenergy_afed = fenergy_init_afed - fenergy_afed

!           //   if free energy change is sufficiently large
            if ( dfenergy_afed .gt. dfenergy_init_afed ) then

!              /*   update direction   */
               call dupdate_descent_afed

!              //   get cv velocity
               call vupdate_descent_afed

!              /*   unit vector in force direction   */
               call eupdate_descent_afed

!           //   if free energy change is sufficiently large
            end if

!        //    for latter iterations in D1 and D2
         else

!           /*   update direction   */
            call dupdate_descent_afed

!           //   get cv velocity
            call vupdate_descent_afed

!           /*   unit vector in force direction   */
            call eupdate_descent_afed

!        //   for DE, D1, D2
         end if

!     /*   restarted or continued step   */
      end if

      return
      end





!***********************************************************************
      subroutine updates_ascent_afed_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   myrank

      use afed_variables, only : &
     &   iiter_afed, iiter_start_afed, niter_refresh_afed, &
     &   niter_gad2evf_afed, algo_ascent_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      logical :: initstep, irestart, igad2evf, irefresh

!-----------------------------------------------------------------------
!     /*   niter_refresh_afed should not exceed niter_gad2evf_afed    */
!-----------------------------------------------------------------------

      if ( niter_refresh_afed .gt. niter_gad2evf_afed ) &
     &   niter_refresh_afed = niter_gad2evf_afed

!-----------------------------------------------------------------------
!     /*   condition                                                  */
!-----------------------------------------------------------------------

      initstep = .false.
      irestart = .false.
      igad2evf = .false.
      irefresh = .false.

      if ( iiter_afed .eq. 0 )                  initstep = .true.

      if ( iiter_afed .eq. iiter_start_afed )   irestart = .true.

      if ( ( algo_ascent_afed(1:4) .eq. 'EVF' ) .or. &
     &     ( ( niter_gad2evf_afed .gt. 0 ) .and. &
     &       ( iiter_afed .ge. niter_gad2evf_afed ) ) ) &
     &                                          igad2evf = .true.

      if ( ( niter_refresh_afed .gt. 0 ) .and. &
     &     ( mod(iiter_afed,niter_refresh_afed) .eq. 0 ) ) &
     &                                          irefresh = .true.

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      if ( initstep ) then

!        //   print a message
         if ( myrank .eq. 0 ) then
            write( 6, '(a)' ) 'Random displacement applied.'
            write( 6, '(a)' ) 
         end if

!        /*   update direction   */
         call dupdate_random_afed_MPI

!        //   get cv velocity
         call vupdate_shift_afed

!        /*   unit vector in velocity direction   */
         call eupdate_velocity_afed

      else if ( igad2evf .and. ( irestart .or. irefresh ) ) then

!        //   print a message
         if ( myrank .eq. 0 ) then
            write( 6, '(a)' ) 'Calculating Hessian, EVF update.'
            write( 6, '(a)' ) 
         end if

!        //   calculate hessian
         call meanhessian_afed_MPI
         call update_hessian_afed( 0 )

!        //   eigenvalues and eigenvectors (overwrite e_afed)
         call geteigen_afed_MPI

!        //   print eigenvalues and eigenvectors
         call printeigen_afed_MPI

!        /*   update direction   */
         call dupdate_ascent_afed

!        /*   update velocity vector   */
         call vupdate_ascent_afed

      else if ( igad2evf ) then

!        //   print a message
         if ( myrank .eq. 0 ) then
            write( 6, '(a)' ) 'Quasi-Hessian, EVF update.'
            write( 6, '(a)' ) 
         end if

!        //   calculate hessian
         call update_hessian_afed( 0 )

!        //   eigenvalues and eigenvectors (overwrite e_afed)
         call geteigen_afed_MPI

!        //   print eigenvalues and eigenvectors
         call printeigen_afed_MPI

!        /*   update direction   */
         call dupdate_ascent_afed

!        /*   update velocity vector   */
         call vupdate_ascent_afed

      else

!        //   print a message
         if ( myrank .eq. 0 ) then
            write( 6, '(a)' ) 'GAD update.'
            write( 6, '(a)' ) 
         end if

!        /*   update direction   */
         call dupdate_ascent_afed

!        /*   update velocity vector   */
         call vupdate_ascent_afed

      end if

      return
      end





!***********************************************************************
      subroutine standard_adescent_afed_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit_afed, nbead, myrank

      use cons_variables, only : &
     &   scons_avg, ncons, ipbc_cons, itype_cons

      use afed_variables, only : &
     &   fenergy_afed, f_afed, scons_mean_afed, v_afed, e_afed, dt_afed, &
     &   afed_status, rafed, params_afed, dt_afed, fenergy_max_afed, &
     &   iiter_afed, iconv_afed, xi_afed, jconv_afed, iiter_start_afed, &
     &   niter_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8) :: ds

      integer :: i, j

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   check error: out of bounds                                 */
!-----------------------------------------------------------------------

      do i = 1, ncons

         if ( ( rafed(i) .lt. params_afed(itype_cons(i),2) ) .or. &
     &        ( rafed(i) .gt. params_afed(itype_cons(i),3) ) ) then

            iconv_afed = -1

            exit

         end if

      end do

!-----------------------------------------------------------------------
!     /*   check error: free energy too high                          */
!-----------------------------------------------------------------------

      if ( fenergy_afed .gt. fenergy_max_afed ) iconv_afed = -2

!-----------------------------------------------------------------------
!     /*   check error: exceeding maximum number of iterations        */
!-----------------------------------------------------------------------

      if ( iiter_afed .ge. niter_afed ) iconv_afed = -3

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

         write( 6, '(a)' )

         if      ( iconv_afed .eq. 1 ) then

         write( 6, '(a,a)' ) &
     &      'AFED step reached stationary point, ', &
     &      'run terminates normally.'

         else if ( iconv_afed .eq. 0 ) then

            if ( jconv_afed .eq. 1 ) then

            write( 6, '(a)' ) &
     &      'AFED step rejected, run continues with adjusted step.'

         else

            write( 6, '(a)' ) &
     &      'AFED step accepted, run continues normally.'

         end if

         else if ( iconv_afed .eq. -1 ) then

            write( 6, '(a)' ) &
     &      'AFED did not converge: Out of bounds.'
            write( 6, '(a)' )

         else if ( iconv_afed .eq. -2 ) then

            write( 6, '(a)' ) &
     &      'AFED did not converge: Free energy too high.'
            write( 6, '(a)' )

         else if ( iconv_afed .eq. -3 ) then

            write( 6, '(a)' ) &
     &      'AFED did not converge: Exceeding maximum iterations.'
            write( 6, '(a)' )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   change status                                              */
!-----------------------------------------------------------------------

      if      ( iconv_afed .eq. 1 ) then

         if ( afed_status(1:2) .eq. 'DE' ) afed_status(1:2) = 'EQ'
         if ( afed_status(1:2) .eq. 'D1' ) afed_status(1:2) = 'EQ'
         if ( afed_status(1:2) .eq. 'D2' ) afed_status(1:2) = 'EQ'
         if ( afed_status(1:2) .eq. 'AS' ) afed_status(1:2) = 'TS'

      else if ( iconv_afed .eq. -1 ) then

         afed_status(1:2) = 'OB'

      else if ( iconv_afed .eq. -2 ) then

         afed_status(1:2) = 'HI'

      else if ( iconv_afed .eq. -3 ) then

         afed_status(1:2) = 'EX'

      end if

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( ( iset .eq. 0 ) .and. ( iiter_afed .eq. 0 ) ) then

         if ( myrank .eq. 0 ) then

            open ( iounit_afed, file = 'afed.out', access = 'append' )

            write( iounit_afed, '(a)' )
            write( iounit_afed, '(a)' ) &
     &      '====================================================' // &
     &      '=========================='
            write( iounit_afed, '(a)' ) &
     &      ' step st cv  r-ideal   r-mean   f-energy     -df/dr' // &
     &      '     dr/dt     dt     n  xi'
            write( iounit_afed, '(a)' ) &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            close( iounit_afed )

         end if

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   mean cv value                                              */
!-----------------------------------------------------------------------

      do i = 1, ncons

         scons_mean_afed(i) = 0.d0

         do j = 1, nbead

            ds = scons_avg(i,j) - rafed(i)

            call pbc_cons( ds, ipbc_cons(i) )

            ds = ds + rafed(i)

            scons_mean_afed(i) = scons_mean_afed(i) + ds

         end do

         scons_mean_afed(i) = scons_mean_afed(i) / dble(nbead)

      end do

!-----------------------------------------------------------------------
!     /*   standard output                                            */
!-----------------------------------------------------------------------

!     /*   master rank only   */
      if ( myrank .ne. 0 ) return

!     /*   wall clock time   */
      call getdate

      write( 6, '(a)' )
      write( 6, '(a)' ) &
     &   '====================================================' // &
     &   '=========================='
      write( 6, '(a)' ) &
     &   ' step st cv  r-ideal   r-mean   f-energy     -df/dr' // &
     &   '     dr/dt     dt     n  xi'
      write( 6, '(a)' ) &
     &   '----------------------------------------------------' // &
     &   '--------------------------'

!-----------------------------------------------------------------------
!     /*   standard output                                            */
!-----------------------------------------------------------------------

!     /*   output   */
      do i = 1, ncons

         write( 6, &
     &      '(i5,a3,i3,2f9.3,f11.6,f11.6,f10.3,f7.3,f6.2,i4)') &
     &      iiter_afed, afed_status(1:2), i, rafed(i), &
     &      scons_mean_afed(i), fenergy_afed, &
     &      f_afed(i), v_afed(i), &
     &      dt_afed, e_afed(i), nint(xi_afed)

      end do

      write( 6, * )

!-----------------------------------------------------------------------
!     /*   output to file                                             */
!-----------------------------------------------------------------------

!     /*   no duplication   */
      if ( ( iiter_afed .eq. 0 ) .or. &
     &     ( iiter_afed .gt. iiter_start_afed ) ) then

!     /*   open file   */
      open( iounit_afed, file = 'afed.out', access = 'append' )

!     /*   output   */
      do i = 1, ncons

         write( iounit_afed, &
     &      '(i5,a3,i3,2f9.3,f11.6,f11.6,f10.3,f7.3,f6.2,i4)') &
     &      iiter_afed, afed_status(1:2), i, rafed(i), &
     &      scons_mean_afed(i), fenergy_afed, &
     &      f_afed(i), v_afed(i), &
     &      dt_afed, e_afed(i), nint(xi_afed)

      end do

!     /*   close file   */
      close( iounit_afed )

!     /*   no duplication   */
      end if

      return
      end





!***********************************************************************
      subroutine vupdate_ascent_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   v_afed, fictmass_afed, d_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i

!-----------------------------------------------------------------------
!     /*   update velocity                                            */
!-----------------------------------------------------------------------

      do i = 1, ncons
         v_afed(i) = d_afed(i) / sqrt(fictmass_afed(i))
      end do

      return
      end





!***********************************************************************
      subroutine dupdate_ascent_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   f_afed, d_afed, fictmass_afed, e_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i

      real(8) :: fe, absd

!-----------------------------------------------------------------------
!     /*   gentlest ascent: (force * unit vector)                     */
!-----------------------------------------------------------------------

      fe = 0.d0

      do i = 1, ncons
         fe = fe + f_afed(i) * e_afed(i) / sqrt(fictmass_afed(i))
      end do

!-----------------------------------------------------------------------
!     /*   gentlest ascent: (ascent force vector)                     */
!-----------------------------------------------------------------------

      do i = 1, ncons
         d_afed(i) = f_afed(i) / sqrt(fictmass_afed(i)) &
     &             - 2.d0 * fe * e_afed(i)
      end do

!-----------------------------------------------------------------------
!     /*   gentlest ascent: (normalize ascent force)                  */
!-----------------------------------------------------------------------

      absd = 0.d0

      do i = 1, ncons
         absd = absd + d_afed(i)*d_afed(i)
      end do

      absd = sqrt( absd )

      do i = 1, ncons
         d_afed(i) = d_afed(i) / absd
      end do

      return
      end



!***********************************************************************
      subroutine eupdate_ascent_afed_MPI
!***********************************************************************

      use afed_variables, only : algo_ascent_afed

      if      ( algo_ascent_afed(1:4) .eq. 'GAD ' ) then

         call eupdate_ascent_gad_afed_MPI

      else if ( algo_ascent_afed(1:4) .eq. 'EVF ' ) then

         call eupdate_ascent_evf_afed_MPI

      else

         call eupdate_ascent_gad_afed_MPI

      end if

      return
      end





!***********************************************************************
      subroutine eupdate_ascent_evf_afed_MPI
!***********************************************************************

!     //   calculate hessian
      call meanhessian_afed_MPI

!     //   eigenvalues and eigenvectors (overwrite e_afed)
      call geteigen_afed_MPI

!     //   print eigenvalues and eigenvectors
      call printeigen_afed_MPI

      return
      end





!***********************************************************************
      subroutine eupdate_ascent_gad_afed_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, myrank

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   dt_afed, gamma_ascent_afed, e_afed, he_afed, ehe_afed, &
     &   hehe_afed, demax_ascent_afed, f_afed, fictmass_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i

      real(8) :: const, c, theta, fabs, fe, phi

!-----------------------------------------------------------------------
!     /*   inner product of e and he vectors                          */
!-----------------------------------------------------------------------

      ehe_afed = 0.d0

      do i = 1, ncons
         ehe_afed = ehe_afed + e_afed(i) * he_afed(i)
      end do

!-----------------------------------------------------------------------
!     /*   inner product of he and he vectors                         */
!-----------------------------------------------------------------------

      hehe_afed = 0.d0

      do i = 1, ncons
         hehe_afed = hehe_afed + he_afed(i) * he_afed(i)
      end do

!-----------------------------------------------------------------------
!     /*   gentlest ascent: (update unit vector, revised)             */
!-----------------------------------------------------------------------

      const = dt_afed/gamma_ascent_afed

      const = min( demax_ascent_afed/sqrt(hehe_afed), const )

!-----------------------------------------------------------------------
!     /*   print message                                              */
!-----------------------------------------------------------------------

      theta = 180.d0 - acos( ehe_afed / sqrt(hehe_afed ) ) * 180.d0 / pi

      fabs = 0.d0

      do i = 1, ncons
         fabs = fabs + f_afed(i) * f_afed(i) / fictmass_afed(i)
      end do

      fabs = sqrt(fabs)

      fe = 0.d0

      do i = 1, ncons
         fe = fe + f_afed(i) * e_afed(i) / sqrt(fictmass_afed(i))
      end do

      phi = acos( fe / fabs ) * 180.d0 / pi

      if ( myrank .eq. 0 ) then

         write( 6, '(a,f10.5)' ) 'GAD ehe     value:', ehe_afed
         write( 6, '(a,f10.5)' ) 'GAD abs(he) value:', sqrt(hehe_afed)
         write( 6, '(a,f10.2)' ) 'GAD theta   value:', theta
         write( 6, '(a,f10.2)' ) 'GAD c       value:', const
         write( 6, '(a,f10.5)' ) 'GAD fabs    value:', fabs
         write( 6, '(a,f10.2)' ) 'GAD phi     value:', phi

         write( 6, '(a)' ) 

         if ( ehe_afed .gt. 0.d0 ) then
            write( 6, '(a)' ) 'GAD guiding vector is not updated.'
         else
            write( 6, '(a)' ) 'GAD guiding vector is updated.'
         end if

         write( 6, '(a)' ) 

      end if

!-----------------------------------------------------------------------
!     /*   gentlest ascent: (update unit vector, revised)             */
!-----------------------------------------------------------------------

      if ( ehe_afed .gt. 0.d0 ) return

!-----------------------------------------------------------------------
!     /*   gentlest ascent: (update unit vector, revised)             */
!-----------------------------------------------------------------------

      c = 0.d0

      do i = 1, ncons
         c = c + ( e_afed(i) - const * he_afed(i) )**2
      end do

      c = sqrt( c )

      do i = 1, ncons
         e_afed(i) = ( e_afed(i) - const * he_afed(i) ) / c
      end do

      return
      end





!***********************************************************************
      subroutine ereset_ascent_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   e_afed, he_afed, hehe_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i

      integer :: ioption = 0

!-----------------------------------------------------------------------
!     /*   ioption = 0: do nothing                                    */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         return

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1: e reset by he                                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         hehe_afed = 0.d0

         do i = 1, ncons
            hehe_afed = hehe_afed + he_afed(i) * he_afed(i)
         end do

         do i = 1, ncons
            e_afed(i) = - he_afed(i) / sqrt(hehe_afed)
         end do

      end if

      return
      end





!***********************************************************************
      subroutine rupdate_forward_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use afed_variables, only : rafed, v_afed, dt_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      rafed(:) = rafed(:) + v_afed(:) * dt_afed

!-----------------------------------------------------------------------
!     /*   apply periodic boundary condition                          */
!-----------------------------------------------------------------------

      call pbc_cv_afed

      return
      end





!***********************************************************************
      subroutine judge_converging_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : pi

      use afed_variables, only : &
     &   xi_afed, d_old_afed, d_afed, jconv_afed, iiter_afed

      use cons_variables, only : &
     &   ncons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8) :: ax, ay, xy

      integer :: i

      integer, parameter :: iiter_skip_afed = 5

!-----------------------------------------------------------------------
!     /*   calculate inner products                                   */
!-----------------------------------------------------------------------

      ax = 0.d0
      ay = 0.d0
      xy = 0.d0

      do i = 1, ncons
         ax = ax + d_afed(i)     * d_afed(i)
         ay = ay + d_old_afed(i) * d_old_afed(i)
         xy = xy + d_afed(i)     * d_old_afed(i)
      end do

!-----------------------------------------------------------------------
!     /*   calculate angle                                            */
!-----------------------------------------------------------------------

      xi_afed = xy / sqrt( ax * ay )

      xi_afed = min( max( xi_afed, -1.d0 ), 1.d0 )

      xi_afed = acos( xi_afed ) * 180.d0 / pi

!-----------------------------------------------------------------------
!     /*   judge if converging                                        */
!-----------------------------------------------------------------------

!     //   first five iterations
      if ( iiter_afed .le. iiter_skip_afed ) then

!        //   not converging
         jconv_afed = 0

!        //   return
         return

!     //   otherwise
      end if

!-----------------------------------------------------------------------
!     /*   judge if converging                                        */
!-----------------------------------------------------------------------

!     //   if xi angle is larger than 90 degrees
      if ( xi_afed .gt. 90.d0 ) then

!       //   last iteration: not converging
        if ( jconv_afed .eq. 0 ) then

!           //   converging
            jconv_afed = 1

!        //   last iteration: converging
         else

!           //   not converging (this avoids converging consecutively)
            jconv_afed = 0

!        //   last iteration
         end if

!     //   if xi angle is smaller than 90 degrees
      else

!        //   not converging
         jconv_afed = 0

!     //   otherwise
      end if

      return
      end





!***********************************************************************
      subroutine judge_converged_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use afed_variables, only : &
     &   dt_afed, dt_conv_afed, iconv_afed

!-----------------------------------------------------------------------
!     /*   judge if converged                                         */
!-----------------------------------------------------------------------

!     //   check convergence
      if ( dt_afed .lt. dt_conv_afed ) then

!        //   converged
         iconv_afed = 1

!     //   check convergence
      else

!        //   unconverged
         iconv_afed = 0

!     //   check convergence
      end if

      return
      end





!***********************************************************************
      subroutine vupdate_descent_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use afed_variables, only : &
     &   fictmass_afed, v_afed, d_afed

      use cons_variables, only : &
     &   ncons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: i

!-----------------------------------------------------------------------
!     /*   velocity vector                                            */
!-----------------------------------------------------------------------

      do i = 1, ncons
         v_afed(i) = d_afed(i) / sqrt(fictmass_afed(i))
      end do

      return
      end





!***********************************************************************
      subroutine update_dt_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only: &
     &   iounit

      use afed_variables, only : &
     &   dt_afed, dt_damp_afed, dt_ascent_afed, &
     &   dt_descent_afed, iiter_afed, afed_status, jconv_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer, save :: iset = 0
      integer, save :: kiter_afed = 0

      integer, parameter :: iiter_skip_afed = 5

!-----------------------------------------------------------------------
!     /*   skip the rest for initial steps                            */
!-----------------------------------------------------------------------

      if ( iiter_afed .le. iiter_skip_afed ) return

!-----------------------------------------------------------------------
!     /*   step size is damped if jconv_afed = 1                      */
!-----------------------------------------------------------------------

      if ( jconv_afed .eq. 1 ) dt_afed = dt_afed * dt_damp_afed

!-----------------------------------------------------------------------
!     /*   step size is rescaled if jconv_afed = 1 for 5 times        */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         kiter_afed = 0

         iset = 1

      else

         if ( jconv_afed .eq. 0 ) then

            kiter_afed = kiter_afed + 1

            if ( kiter_afed .eq. 5 ) then

               dt_afed = dt_afed/dt_damp_afed

               if ( afed_status(1:2) .eq. 'DE' ) &
     &            dt_afed = min( dt_afed, dt_descent_afed )

               if ( afed_status(1:2) .eq. 'D1' ) &
     &            dt_afed = min( dt_afed, dt_descent_afed )

               if ( afed_status(1:2) .eq. 'D2' ) &
     &            dt_afed = min( dt_afed, dt_descent_afed )

               if ( afed_status(1:2) .eq. 'AS' ) &
     &            dt_afed = min( dt_afed, dt_ascent_afed )

               kiter_afed = 0

            end if

         else

            kiter_afed = 0

         end if

      end if

      return
      end





!***********************************************************************
      subroutine eupdate_descent_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   f_afed, e_afed, fictmass_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i

      real(8) :: abse

!-----------------------------------------------------------------------
!     /*   unit vector along force                                    */
!-----------------------------------------------------------------------

      do i = 1, ncons
         e_afed(i) = f_afed(i) / sqrt(fictmass_afed(i))
      end do

      abse = 0.d0

      do i = 1, ncons
         abse = abse + e_afed(i)*e_afed(i)
      end do

      abse = sqrt( abse )

      do i = 1, ncons
         e_afed(i) = e_afed(i) / abse
      end do

      return
      end





!***********************************************************************
      subroutine eupdate_velocity_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   v_afed, e_afed, fictmass_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i

      real(8) :: abse

!-----------------------------------------------------------------------
!     /*   unit vector along velocity                                 */
!-----------------------------------------------------------------------

      do i = 1, ncons
         e_afed(i) = sqrt(fictmass_afed(i)) * v_afed(i)
      end do

      abse = 0.d0

      do i = 1, ncons
         abse = abse + e_afed(i)*e_afed(i)
      end do

      abse = sqrt( abse )

      do i = 1, ncons
         e_afed(i) = e_afed(i) / abse
      end do

      return
      end





!***********************************************************************
      subroutine dupdate_descent_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   f_afed, d_afed, fictmass_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i

      real(8) :: absd

!-----------------------------------------------------------------------
!     /*   unit vector along force                                    */
!-----------------------------------------------------------------------

      do i = 1, ncons
         d_afed(i) = f_afed(i) / sqrt(fictmass_afed(i))
      end do

      absd = 0.d0

      do i = 1, ncons
         absd = absd + d_afed(i)*d_afed(i)
      end do

      absd = sqrt( absd )

      do i = 1, ncons
         d_afed(i) = d_afed(i) / absd
      end do

      return
      end





!***********************************************************************
      subroutine fupdate_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   fenergy_afed, dt_afed, f_afed, f_old_afed, v_afed, v_old_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   integers
      integer :: i

!-----------------------------------------------------------------------
!     /*   update free energy                                         */
!-----------------------------------------------------------------------

      do i = 1, ncons
         fenergy_afed = fenergy_afed &
     &                - 0.5d0 * f_afed(i) * v_afed(i) * dt_afed &
     &                - 0.5d0 * f_old_afed(i) * v_old_afed(i) * dt_afed
      end do

      return
      end





!***********************************************************************
      subroutine save_cv_afed
!***********************************************************************

!     //   shared variables
      use afed_variables, only : &
     &   d_afed, d_old_afed, f_afed, f_old_afed, rafed, rafed_old, &
     &   v_afed, v_old_afed, e_afed, e_old_afed, scons_avg_old_afed, &
     &   fenergy_old_afed, fenergy_afed, hess_afed, hess_old_afed

!     //   shared variables
      use cons_variables, only : scons_avg

!     //   save free energy
      fenergy_old_afed = fenergy_afed

!     //   save old direction
      d_old_afed(:) = d_afed(:)

!     //   save old unit vector
      e_old_afed(:) = e_afed(:)

!     //   save old force
      f_old_afed(:) = f_afed(:)

!     //   save old position
      rafed_old(:)  = rafed(:)

!     //   save old average position
      scons_avg_old_afed(:,:) = scons_avg(:,:)

!     //   save old velocity
      v_old_afed(:) = v_afed(:)

!     //   save old hessian mizuho
      hess_old_afed(:,:) = hess_afed(:,:)

      return
      end





!***********************************************************************
      subroutine restore_cv_afed
!***********************************************************************

!     //   shared variables
      use afed_variables, only : &
     &   d_afed, d_old_afed, f_afed, f_old_afed, rafed, rafed_old, &
     &   v_afed, v_old_afed, e_afed, e_old_afed, scons_avg_old_afed, &
     &   fenergy_old_afed, fenergy_afed, hess_afed, hess_old_afed

!     //   shared variables
      use cons_variables, only : scons_avg

!     //   save free energy
      fenergy_afed = fenergy_old_afed

!     //   restore old direction
      d_afed(:) = d_old_afed(:)

!     //   restore old unit vector
      e_afed(:) = e_old_afed(:)

!     //   restore old force
      f_afed(:) = f_old_afed(:)

!     //   restore old position
      rafed(:)  = rafed_old(:)

!     //   restore old average position
      scons_avg(:,:) = scons_avg_old_afed(:,:)

!     //   restore old velocity
      v_afed(:) = v_old_afed(:)

!     //   restore old hessian
      hess_afed(:,:) = hess_old_afed(:,:)

      return
      end





!***********************************************************************
      subroutine dupdate_random_afed_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : myrank

      use cons_variables, only : ncons

      use afed_variables, only : d_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   integers
      integer :: i

!     //   real numbers
      real(8) :: absd, gasdev

!-----------------------------------------------------------------------
!     /*   random direction                                           */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

      absd = 0.d0

      do i = 1, ncons
         d_afed(i) = gasdev()
         absd = absd + d_afed(i)*d_afed(i)
      end do

      absd = sqrt(absd)

      do i = 1, ncons
         d_afed(i) = d_afed(i) / absd
      end do

      end if

      call my_mpi_bcast_real_1( d_afed, ncons )

      return
      end





!***********************************************************************
      subroutine vupdate_shift_afed
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use afed_variables, only : &
     &   fictmass_afed, d_afed, v_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   shift from minimum
      real(8) :: const_shift_afed = 1.5d0

!-----------------------------------------------------------------------
!     /*   update velocity in random direction                        */
!-----------------------------------------------------------------------

      v_afed(:) = d_afed(:) / sqrt(fictmass_afed(:)) * const_shift_afed

      return
      end





!***********************************************************************
      subroutine afedcycle_auto_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iexit, iounit

      use afed_variables, only : &
     &   afed_status, finished_auto_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   start main loop                                            */
!-----------------------------------------------------------------------

!     //   loop of shots
      do

!        //   navigate
         call afed_auto_navigate_MPI

!        //   stop if auto converged
         if ( finished_auto_afed ) exit

!        //   initial minimum search
         if ( afed_status(1:2) .eq. 'DE' ) then

!           //   run descent trajectory
            call afedcycle_descent_MPI

!           //   stop on error
            if ( iexit .eq. 1 ) exit

!           //   record landmark
            call afed_auto_record_MPI

!           //   go back
            cycle

!        //   initial minimum search
         end if

!        //  saddle point search
         if ( afed_status(1:2) .eq. 'AS' ) then

!           //   saddle point search
            call afedcycle_ascent_MPI

!           //   stop on error
            if ( iexit .eq. 1 ) exit

!           //   record landmark
            call afed_auto_record_MPI

!           //   go back
            cycle

!        //  saddle point search
         end if

!        //   minimum search in plus direction
         if ( afed_status(1:2) .eq. 'D1' ) then

!           //   minimum point search
            call afedcycle_descent_MPI

!           //   stop on error
            if ( iexit .eq. 1 ) exit

!           //   record landmark
            call afed_auto_record_MPI

!           //   go back
            cycle

!        //  minimum search
         end if

!        //   minimum search in plus direction
         if ( afed_status(1:2) .eq. 'D2' ) then

!           //   minimum point search
            call afedcycle_descent_MPI

!           //   stop on error
            if ( iexit .eq. 1 ) exit

!           //   record landmark
            call afed_auto_record_MPI

!           //   go back
            cycle

!        //   minimum search
         end if

!     //   loop of shots
      end do

      return
      end





!***********************************************************************
      subroutine afed_auto_record_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, myrank

      use cons_variables, only : &
     &   ncons, ipbc_cons

      use afed_variables, only : &
     &   fenergy_afed, rafed,  e_afed, fictmass_afed, &
     &   radius_auto_afed, afed_status, iroot_auto_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   cv position of reference landmark
      real(8), dimension(ncons) :: rafed_ref

!     //   radius between current reference landmarks
      real(8) :: r

!     //   distance between current and reference landmarks
      real(8) :: dr

!     //   free energy of reference landmark
      real(8) :: dble_fenergy

!     //   current landmark number: equal to last line
      integer :: icur

!     //   reference landmark number: equal to line number
      integer :: iref

!     //   line number: equal to landmark number
      integer :: iline

!     //   landmark id of reference landmark
      integer :: jref

!     //   landmark id of current landmark
      integer :: jcur

!     //   root landmark id of reference landmark
      integer :: kref

!     //   root landmark id of current landmark
      integer :: kcur

!     //   number of lines
      integer :: nline

!     //   error flag
      integer :: ierr

!     //   status of current landmark
      character(len=2) :: char_status

!     //   label of current landmark: NEW or OLD
      character(len=3) :: char_label_cur

!     //   label of reference landmark: NEW or OLD
      character(len=3) :: char_label_ref

!     //   file name
      character(len=10) :: char_file

!     //   integer
      integer :: j

!     //   character
      character(len=3) :: char_num

!-----------------------------------------------------------------------
!     //   count number of lines
!-----------------------------------------------------------------------

!     //   master rank only
      if ( myrank .eq. 0 ) then

!        //   open file
         open ( iounit, file = 'auto.ini' )

!        //   counter
         nline = 0

!        //   line by line
         do

!           //   read line
            read( iounit, *, iostat=ierr ) iref

!           //   find last line
            if ( ierr .ne. 0 ) exit

!           //   counter
            nline = nline + 1

!        //   line by line
         end do

!        //   close file
         close( iounit )

!     //  master rank only
      end if

!-----------------------------------------------------------------------
!     //   new or old landmark
!-----------------------------------------------------------------------

!     //   master rank only
      if ( myrank .eq. 0 ) then

!        //   initialize current landmark label
         char_label_cur = 'NEW'

!        //   initialize current landmark id
         jcur = nline + 1

!        //   open file
         open ( iounit, file = 'auto.ini' )

!        //   line by line
         do iline = 1, nline

!           //   read line of reference landmark

            read( iounit, *, iostat=ierr ) &
     &         iref, char_status, char_label_ref, dble_fenergy, &
     &         jref, char_file, kref, rafed_ref(1:ncons)

!           //   initialize
            r = 0.d0

!           //   loop of cv dimensions
            do j = 1, ncons

!              //   distance from reference landmark
               dr = rafed_ref(j) - rafed(j)

!              //   apply periodic boundary condition of cv
               call pbc_cons( dr, ipbc_cons(j) )

!              //   distance in mass weighted cv
               r = r + fictmass_afed(j)*dr*dr

!           //   loop of cv dimensions
            end do

!           //   mass weighted distance from reference landmark
            r = sqrt(r)

!           //   if it is close to reference landmark
            if ( r .lt. radius_auto_afed ) then

!              //   identified as old landmark
               char_label_cur = 'OLD'

!              //   number id of current landmark
               if ( char_label_ref(1:3) .eq. 'NEW' ) jcur = jref

!           //   otherwise
            end if

!        //   line by line
         end do

!        //   close file
         close( iounit )

!     //  master rank only
      end if

!-----------------------------------------------------------------------
!     //   add landmark data to the list
!-----------------------------------------------------------------------

!     //   master rank only
      if ( myrank .eq. 0 ) then

!        //   landmark number of current landmark
         icur = nline + 1

!        //   root of current landmark
         kcur = iroot_auto_afed

!        //   numbering
         call int3_to_char( icur, char_num )

!        //   file name
         char_file = afed_status(1:2) // '.' // char_num // '.tar'

!        //   open file
         open ( iounit, file = 'auto.ini', access = 'append' )

!        //   write data: 1/3

         write( iounit, &
     &      '(i5,1x,a2,1x,a3,f9.5,i5,1x,a10,i5)', advance = 'no' ) &
     &      icur, afed_status(1:2), char_label_cur(1:3), fenergy_afed, &
     &      jcur, char_file, kcur

!        //   write data: 2/3

         if ( afed_status(1:2) .eq. 'TS' ) then

            do j = 1, ncons
               write( iounit, '(f9.3)', advance = 'no' ) rafed(j)
            end do

         else

            do j = 1, ncons-1
               write( iounit, '(f9.3)', advance = 'no' ) rafed(j)
            end do

            write( iounit, '(f9.3)' ) rafed(ncons)

         end if

!        //   write data: 3/3

         if ( afed_status(1:2) .eq. 'TS' ) then

            do j = 1, ncons-1
               write( iounit, '(f7.3)', advance = 'no' ) e_afed(j)
            end do

            write( iounit, '(f7.3)' ) e_afed(ncons)

         end if

!        //   close file
         close( iounit )

!     //  master rank only
      end if

!-----------------------------------------------------------------------
!     //   print *.ini
!-----------------------------------------------------------------------

!     //   save data
      call backup_adescent_afed_MPI

!     //   synchronize
      call my_mpi_barrier

!-----------------------------------------------------------------------
!     //   save restart files
!-----------------------------------------------------------------------

!     //   master rank only
      if ( myrank .eq. 0 ) then

!        //   compress into tar file
         call compress_tar_afed( char_file )

!     //   master rank only
      end if

!     //   synchronize
      call my_mpi_barrier

      return
      end





!***********************************************************************
      subroutine afed_auto_navigate_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, myrank

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   dt_descent_afed, dt_ascent_afed, dt_afed, e_afed, rafed, &
     &   fenergy_afed, iconv_afed, jconv_afed, afed_status, &
     &   nmiss_auto_afed, nshot_auto_afed, iiter_start_afed, &
     &   iroot_auto_afed, finished_auto_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   cv position of reference landmark
      real(8), dimension(:,:), allocatable :: rafed_ref

!     //   unit vector of reference landmark
      real(8), dimension(:,:), allocatable :: e_afed_ref

!     //   number of lines
      integer :: nline

!     //   line number: equal to landmark number
      integer :: iline

!     //   landmark id of EQ
      integer, dimension(:), allocatable :: id_eq

!     //   landmark id of TS
      integer, dimension(:), allocatable :: id_ts

!     //   number of trajectories shot from EQ
      integer, dimension(:), allocatable :: ntraj_eq

!     //   number of trajectories shot from TS
      integer, dimension(:), allocatable :: ntraj_ts

!     //   number of accepted trajectories shot from EQ
      integer, dimension(:), allocatable :: ntraj_accept_eq

!     //   number of rejected trajectories shot from EQ
      integer, dimension(:), allocatable :: ntraj_reject_eq

!     //   number of consecutive misses shot from EQ
      integer, dimension(:), allocatable :: ntraj_missed_eq

!     //   number of trajectories
      integer :: ntraj

!     //   label of reference landmark: NEW or OLD
      character(len=3) :: char_label_ref

!     //   reference landmark number: equal to line number
      integer :: iref

!     //   landmark id of reference landmark
      integer :: jref

!     //   root landmark id of reference landmark
      integer :: kref

!     //   error flag
      integer :: ierr

!     //   status of reference landmark
      character(len=2) :: char_status

!     //   label of reference EQ landmark: NEW or OLD
      character(len=3), dimension(:), allocatable :: label_eq

!     //   label of reference TS landmark: NEW or OLD
      character(len=3), dimension(:), allocatable :: label_ts

!     //   free energy of reference landmark: NEW or OLD
      real(8), dimension(:), allocatable :: fenergy_ref

!     //   file name
      character(len=10) :: char_file

!     //   number of EQ reference landmarks
      integer :: neq

!     //   number of TS reference landmarks
      integer :: nts

!     //   EQ reference landmark
      integer :: jeq

!     //   TS reference landmark
      integer :: jts

!     //   free energy of reference landmark
      real(8) :: dble_fenergy

!     //   character
      character(len=3) :: char_num

!     //   integer
      integer :: j

!-----------------------------------------------------------------------
!     //   initialize flag of auto
!-----------------------------------------------------------------------

      finished_auto_afed = .false.

!-----------------------------------------------------------------------
!     //   return for continued runs
!-----------------------------------------------------------------------

      if ( afed_status(1:2) .eq. 'DE' ) return
      if ( afed_status(1:2) .eq. 'D1' ) return
      if ( afed_status(1:2) .eq. 'D2' ) return
      if ( afed_status(1:2) .eq. 'AS' ) return

!-----------------------------------------------------------------------
!     //   count number of reference EQ and reference TS landmarks
!-----------------------------------------------------------------------

!     //   master rank only
      if ( myrank .eq. 0 ) then

!        //   line number
         iline = 0

!        //   number of EQ landmarks
         neq = 0

!        //   number of TS landmarks
         nts = 0

!        //   open file
         open ( iounit, file = 'auto.ini' )

!        //   line by line
         do

!           //   read line of reference landmark

            read( iounit, *, iostat=ierr ) &
     &         iref, char_status, char_label_ref, dble_fenergy, &
     &         jref, char_file, kref

!           //   exit from loop if last line found
            if ( ierr .ne. 0 ) exit

!           //   update line number
            iline = iline + 1

!           //   if EQ found
            if ( char_status(1:2) .eq. 'EQ' ) then

!              //   update EQ counter
               neq = neq + 1

!           //   if EQ is found
            end if

!           //   if TS found
            if ( char_status(1:2) .eq. 'TS' ) then

!              //   update TS counter
               nts = nts + 1

!           //   if EQ is found
            end if

!        //   line by line
         end do

!        //   close file
         close( iounit )

!        //   number of lines
         nline = iline

!     //  master rank only
      end if

!     //   broadcast
      call my_mpi_bcast_int_0( neq )
      call my_mpi_bcast_int_0( nts )
      call my_mpi_bcast_int_0( nline )

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

!     //   id of EQ reference landmark
      if ( .not. allocated( id_eq ) ) &
     &   allocate( id_eq(max(neq,1)) )

!     //   id of TS reference landmark
      if ( .not. allocated( id_ts ) ) &
     &   allocate( id_ts(max(nts,1)) )

!     //   label of EQ reference landmark
      if ( .not. allocated( label_eq ) ) &
     &   allocate( label_eq(max(neq,1)) )

!     //   label of TS reference landmark
      if ( .not. allocated( label_ts ) ) &
     &   allocate( label_ts(max(nts,1)) )

!     //   number of shots from EQ
      if ( .not. allocated( ntraj_eq ) ) &
     &   allocate( ntraj_eq(max(neq,1)) )

!     //   number of shots from TS
      if ( .not. allocated( ntraj_ts ) ) &
     &   allocate( ntraj_ts(max(nts,1)) )

!     //   number of accepted shots from EQ
      if ( .not. allocated( ntraj_accept_eq ) ) &
     &   allocate( ntraj_accept_eq(max(neq,1)) )

!     //   number of rejected shots from EQ
      if ( .not. allocated( ntraj_reject_eq ) ) &
     &   allocate( ntraj_reject_eq(max(neq,1)) )

!     //   number of consecutive missed shots from EQ
      if ( .not. allocated( ntraj_missed_eq ) ) &
     &   allocate( ntraj_missed_eq(max(neq,1)) )

!     //   cv position of reference landmark
      if ( .not. allocated( rafed_ref ) ) &
     &   allocate( rafed_ref(ncons,max(nline,1)) )

!     //   unit vector of reference landmark
      if ( .not. allocated( e_afed_ref ) ) &
     &   allocate( e_afed_ref(ncons,max(nline,1)) )

!     //   unit vector of reference landmark
      if ( .not. allocated( fenergy_ref ) ) &
     &   allocate( fenergy_ref(max(nline,1)) )

!-----------------------------------------------------------------------
!     //   initialize
!-----------------------------------------------------------------------

!     //   id of EQ reference landmark
      id_eq(:) = 0

!     //   id of TS reference landmark
      id_ts(:) = 0

!     //   label of EQ reference landmark
      label_eq(:) = 'OLD'

!     //   label of TS reference landmark
      label_ts(:) = 'OLD'

!     //   number of shots from EQ
      ntraj_eq(:) = 0

!     //   number of shots from TS
      ntraj_ts(:) = 0

!     //   number of accepted shots from EQ
      ntraj_accept_eq(:) = 0

!     //   number of rejected shots from TS
      ntraj_reject_eq(:) = 0

!     //   number of consecutive missed shots from EQ
      ntraj_missed_eq(:) = 0

!     //   integer
      j = 0

!     //   cv position of reference landmark
      rafed_ref(:,:) = 0.d0

!     //   unit vector of reference landmark
      e_afed_ref(:,:) = 0.d0

!     //   free energy of reference landmark
      fenergy_ref(:) = 0.d0

!-----------------------------------------------------------------------
!     //   start DE if no minimum found
!-----------------------------------------------------------------------

!     //  no minimum
      if ( neq .eq. 0 ) then

!        //   reset status
         afed_status = 'DE'

!        //   root point of trajectory
         iroot_auto_afed = 0

!        //   jump to line 100
         go to 100

!     //  no minimum
      end if

!-----------------------------------------------------------------------
!     //   identify shots from TS
!-----------------------------------------------------------------------

!     //   master rank only
      if ( myrank .eq. 0 ) then

!        //   TS counter
         jts = 0

!        //   open file
         open ( iounit, file = 'auto.ini' )

!        //   line by line
         do iline = 1, nline

!           //   read line of reference landmark
            read( iounit, *, iostat=ierr ) iref, char_status

!           //   TS found
            if ( char_status(1:2) .eq. 'TS' ) then

!              //   go back one line
               backspace( iounit )

!              //   read line of reference landmark

               read( iounit, *, iostat=ierr ) &
     &            iref, char_status, char_label_ref, fenergy_ref(iref), &
     &            jref, char_file, kref, rafed_ref(1:ncons,iref), &
     &            e_afed_ref(1:ncons,iref)

!              //   update TS counter
               jts = jts + 1

!              //   id of TS
               id_ts(jts) = jref

!              //   label of TS
               label_ts(jts) = char_label_ref

!           //   EQ found
            else if ( char_status(1:2) .eq. 'EQ' ) then

!              //   go back one line
               backspace( iounit )

!              //   read line of reference landmark

               read( iounit, *, iostat=ierr ) &
     &            iref, char_status, char_label_ref, fenergy_ref(iref), &
     &            jref, char_file, kref, rafed_ref(1:ncons,iref)

!           //   otherwise
            end if

!        //   line by line
         end do

!        //   close file
         close( iounit )

!     //  master rank only
      end if

!     //   broadcast
      call my_mpi_bcast_int_1( id_ts, max(nts,1) )
      call my_mpi_bcast_char_1( label_ts, 3, max(nts,1) )
      call my_mpi_bcast_real_2( rafed_ref, ncons, max(nline,1) )
      call my_mpi_bcast_real_2( e_afed_ref, ncons, max(nline,1) )
      call my_mpi_bcast_real_1( fenergy_ref, max(nline,1) )

!-----------------------------------------------------------------------
!     //   number of shots from TS leading to EQ
!-----------------------------------------------------------------------

!     //   master rank only
      if ( myrank .eq. 0 ) then

!        //   loop of TS
         do jts = 1, nts

!           //   counter: number of trajectories from TS
            ntraj_ts(jts) = 0

!           //   open file
            open ( iounit, file = 'auto.ini' )

!           //   line by line
            do iline = 1, nline

!              //   read line of reference landmark

               read( iounit, *, iostat=ierr ) &
     &            iref, char_status, char_label_ref, dble_fenergy, &
     &            jref, char_file, kref

!              //   found trajectory leading to EQ
               if ( ( char_status(1:2) .eq. 'EQ' ) .and. &
     &              ( kref .eq. id_ts(jts) ) ) then

!                 //   update counter
                  ntraj_ts(jts) = ntraj_ts(jts) + 1

!              //   found trajectory leading to EQ
               end if

!              //   found trajectory leading to OB
               if ( ( char_status(1:2) .eq. 'OB' ) .and. &
     &              ( kref .eq. id_ts(jts) ) ) then

!                 //   update counter
                  ntraj_ts(jts) = ntraj_ts(jts) + 1

!              //   found trajectory leading to OB
               end if

!              //   found trajectory leading to HI
               if ( ( char_status(1:2) .eq. 'HI' ) .and. &
     &              ( kref .eq. id_ts(jts) ) ) then

!                 //   update counter
                  ntraj_ts(jts) = ntraj_ts(jts) + 1

!              //   found trajectory leading to HI
               end if

!              //   found trajectory leading to EX
               if ( ( char_status(1:2) .eq. 'EX' ) .and. &
     &              ( kref .eq. id_ts(jts) ) ) then

!                 //   update counter
                  ntraj_ts(jts) = ntraj_ts(jts) + 1

!              //   found trajectory leading to EX
               end if

!           //   line by line
            end do

!           //   close file
            close( iounit )

!        //   loop of TS
         end do

!     //  master rank only
      end if

!     //   broadcast
      call my_mpi_bcast_int_1( ntraj_ts, max(nts,1) )

!-----------------------------------------------------------------------
!     //    identify shots from EQ
!-----------------------------------------------------------------------

!     //   master rank only
      if ( myrank .eq. 0 ) then

!        //   EQ counter
         jeq = 0

!        //   open file
         open ( iounit, file = 'auto.ini' )

!        //   line by line
         do iline = 1, nline

!           //   read line of reference landmark

            read( iounit, *, iostat=ierr ) &
     &         iref, char_status, char_label_ref, dble_fenergy, &
     &         jref, char_file, kref

!           //   exit loop if last line found
            if ( ierr .ne. 0 ) exit

!           //   EQ found
            if ( char_status(1:2) .eq. 'EQ' ) then

!              //   update EQ counter
               jeq = jeq + 1

!              //   id of EQ
               id_eq(jeq) = jref

!              //   label of TS
               label_eq(jeq) = char_label_ref

!           //   EQ found
            end if

!        //   line by line
         end do

!        //   close file
         close( iounit )

!     //  master rank only
      end if

!     //   broadcast
      call my_mpi_bcast_int_1( id_eq, max(neq,1) )
      call my_mpi_bcast_char_1( label_eq, 3, max(neq,1) )

!-----------------------------------------------------------------------
!     //   number of shots from EQ
!-----------------------------------------------------------------------

!     //   master rank only
      if ( myrank .eq. 0 ) then

!        //   loop of EQ
         do jeq = 1, neq

!           //   open file
            open ( iounit, file = 'auto.ini' )

!           //   line by line
            do iline = 1, nline

!              //   read line of reference landmark

               read( iounit, *, iostat=ierr ) &
     &            iref, char_status, char_label_ref, dble_fenergy, &
     &            jref, char_file, kref, rafed_ref(1:ncons,iref)

!              //   shots from EQ
               if ( kref .eq. id_eq(jeq) ) then

!                 //   update shot counter
                  ntraj_eq(jeq) = ntraj_eq(jeq) + 1

!                 //   shots from EQ leading to new TS
                  if ( ( char_status(1:2) .eq. 'TS' ) .and. &
     &                 ( char_label_ref(1:3) .eq. 'NEW' ) ) then

!                    //   count number of accepted trajectories
                     ntraj_accept_eq(jeq) = ntraj_accept_eq(jeq) + 1

!                    //   count number of consecutive rejections
                     ntraj_missed_eq(jeq) = 0

!                 //   shots from EQ failed to find new TS
                  else

!                    //   count number of rejected trajectories
                     ntraj_reject_eq(jeq) = ntraj_reject_eq(jeq) + 1

!                    //   count number of consecutive rejections
                     ntraj_missed_eq(jeq) = ntraj_missed_eq(jeq) + 1

!                 //   shots from EQ
                  end if

!              //   shots from EQ
               end if

!           //   line by line
            end do

!           //   close file
            close( iounit )

!        //   loop of EQ
         end do

!     //  master rank only
      end if

!     //   broadcast
      call my_mpi_bcast_int_1( ntraj_eq, max(neq,1) )
      call my_mpi_bcast_int_1( ntraj_accept_eq, max(neq,1) )
      call my_mpi_bcast_int_1( ntraj_reject_eq, max(neq,1) )
      call my_mpi_bcast_int_1( ntraj_missed_eq, max(neq,1) )

!-----------------------------------------------------------------------
!     //   restart D1 from new TS with no shot
!-----------------------------------------------------------------------

!     //   master rank only
      if ( myrank .eq. 0 ) then

!        //   loop of TS
         do jts = 1, nts

!           //   only for new TS
            if ( label_ts(jts)(1:3) .eq. 'OLD' ) cycle

!           //   if no shot from TS
            if ( ntraj_ts(jts) .eq. 0 ) then

!              //   id of TS origin
               iroot_auto_afed = id_ts(jts)

!              //   restore restart files: geometry.ini, afed.ini

               call int3_to_char( iroot_auto_afed, char_num )
               char_file = 'TS.' // char_num // '.tar'
               call decompress_tar_afed( char_file )

!              //   exit from cycle
               exit

!           //   if no shot from TS
            end if

!        //   loop of TS
         end do

!     //  master rank only
      end if

!     //   synchronize
      call my_mpi_barrier

!     //   setup molecular dynamics
      call setup_md_nvt_MPI

!     //   loop of TS
      do jts = 1, nts

!        //   only for new TS
         if ( label_ts(jts)(1:3) .eq. 'OLD' ) cycle

!        //   if no shot from TS
         if ( ntraj_ts(jts) .eq. 0 ) then

!           //   id of TS origin
            iroot_auto_afed = id_ts(jts)

!           //   reset status
            afed_status = 'D1'

!           //   initialize step
            iiter_start_afed = 0

!           //   reset step size
            dt_afed = dt_descent_afed

!           //   reset convergence flag
            iconv_afed = 0

!           //   reset convergence flag
            jconv_afed = 0

!           //   number of trajectories from current TS
            ntraj = ntraj_ts(jts) + 1

!           //   cv position
            rafed(:) = rafed_ref(:,iroot_auto_afed)

!           //   unit vector
            e_afed(:) = e_afed_ref(:,iroot_auto_afed)

!           //   free energy
            fenergy_afed = fenergy_ref(iroot_auto_afed)

!           //   TS
            j = jts

!           //   jump to line 100
            go to 100

!        //   if no shot from TS
         end if

!     //   loop of TS
      end do

!-----------------------------------------------------------------------
!     //   restart D2 from new TS with one shot
!-----------------------------------------------------------------------

!     //   master rank only
      if ( myrank .eq. 0 ) then

!        //   loop of TS
         do jts = 1, nts

!           //   only for new TS
            if ( label_ts(jts)(1:3) .eq. 'OLD' ) cycle

!           //   if one shot from TS
            if ( ntraj_ts(jts) .eq. 1 ) then

!              //   id of TS origin
               iroot_auto_afed = id_ts(jts)

!              //   restore restart files: geometry.ini, afed.ini

               call int3_to_char( iroot_auto_afed, char_num )
               char_file = 'TS.' // char_num // '.tar'
               call decompress_tar_afed( char_file )

!              //   exit from cycle
               exit

!           //   if no shot from TS
            end if

!        //   loop of TS
         end do

!     //  master rank only
      end if

!     //   synchronize
      call my_mpi_barrier

!     //   setup molecular dynamics
      call setup_md_nvt_MPI

!     //   loop of TS
      do jts = 1, nts

!        //   only for new TS
         if ( label_ts(jts)(1:3) .eq. 'OLD' ) cycle

!        //   if no shot from TS
         if ( ntraj_ts(jts) .eq. 1 ) then

!           //   id of TS origin
            iroot_auto_afed = id_ts(jts)

!           //   reset status
            afed_status = 'D2'

!           //   initialize step
            iiter_start_afed = 0

!           //   reset step size
            dt_afed = dt_descent_afed

!           //   reset convergence flag
            iconv_afed = 0

!           //   reset convergence flag
            jconv_afed = 0

!           //   number of trajectories from current TS
            ntraj = ntraj_ts(jts) + 1

!           //   cv position
            rafed(:) = rafed_ref(:,iroot_auto_afed)

!           //   unit vector
            e_afed(:) = e_afed_ref(:,iroot_auto_afed)

!           //   free energy
            fenergy_afed = fenergy_ref(iroot_auto_afed)

!           //   TS
            j = jts

!           //   jump to line 100
            go to 100

!        //   if one shot from TS
         end if

!     //   loop of TS
      end do

!-----------------------------------------------------------------------
!     //   check all new TS with two shots
!-----------------------------------------------------------------------

!     //   error flag
      ierr = 0

!     //   loop of TS
      do jts = 1, nts

!        //   only for new TS
         if ( label_ts(jts)(1:3) .eq. 'OLD' ) cycle

!        //   if two shots from TS
         if ( ntraj_ts(jts) .eq. 2 ) cycle

!        //   error flag
         ierr = 1

!        //   TS
         j = jts

!        //   jump to line 100
         go to 200

!     //   loop of TS
      end do

!-----------------------------------------------------------------------
!     //   restart AS from new EQ
!-----------------------------------------------------------------------

!     //   master rank only
      if ( myrank .eq. 0 ) then

!        //   loop of EQ
         do jeq = 1, neq

!           //   only for new EQ
            if ( label_eq(jeq)(1:3) .eq. 'OLD' ) cycle

!           //   if number of shots from EQ is less than threshold
            if ( ( ntraj_eq(jeq)        .lt. nshot_auto_afed ) .and. &
     &           ( ntraj_missed_eq(jeq) .lt. nmiss_auto_afed ) ) then

!              //   id of EQ origin
               iroot_auto_afed = id_eq(jeq)

!              //   restore restart files: geometry.ini, afed.ini

               call int3_to_char( iroot_auto_afed, char_num )
               char_file = 'EQ.' // char_num // '.tar'
               call decompress_tar_afed( char_file )

!              //   exit from cycle
               exit

!           //   if number of shots from EQ is less than threshold
            end if

!        //   loop of EQ
         end do

!     //  master rank only
      end if

!     //   synchronize
      call my_mpi_barrier

!     //   setup molecular dynamics
      call setup_md_nvt_MPI

!     //   loop of EQ
      do jeq = 1, neq

!        //   only for new EQ
         if ( label_eq(jeq)(1:3) .eq. 'OLD' ) cycle

!        //   if number of shots from EQ is less than threshold
         if ( ( ntraj_eq(jeq)        .lt. nshot_auto_afed ) .and. &
     &        ( ntraj_missed_eq(jeq) .lt. nmiss_auto_afed ) ) then

!           //   id of EQ origin
            iroot_auto_afed = id_eq(jeq)

!           //   reset status
            afed_status = 'AS'

!           //   initialize step
            iiter_start_afed = 0

!           //   reset step size
            dt_afed = dt_ascent_afed

!           //   reset convergence flag
            iconv_afed = 0

!           //   reset convergence flag
            jconv_afed = 0

!           //   number of trajectories from current EQ
            ntraj = ntraj_eq(jeq) + 1

!           //   cv position
            rafed(:) = rafed_ref(:,iroot_auto_afed)

!           //   free energy
            fenergy_afed = fenergy_ref(iroot_auto_afed)

!           //   EQ
            j = jeq

!        //   jump to line 100
         go to 100

!        //   if number of shots from EQ is less than threshold
         end if

!     //   loop of EQ
      end do

!-----------------------------------------------------------------------
!     //   initialize flag of auto
!-----------------------------------------------------------------------

!     //   finished
      finished_auto_afed = .true.

!     //   jump to line 200
      go to 200

!-----------------------------------------------------------------------
!     //   line 100
!-----------------------------------------------------------------------

  100 continue

!-----------------------------------------------------------------------
!     //   print
!-----------------------------------------------------------------------

!     //   master rank only
      if ( myrank .eq. 0 ) then

         if ( ( afed_status(1:2) .eq. 'D1' ) .or. &
     &        ( afed_status(1:2) .eq. 'D2' ) ) then

            write( 6, '(a)' ) 'AFED Restarted from following point.'
            write( 6, '(a)' ) '====================================='
            write( 6, '(a)' ) 'st     id  shots'
            write( 6, '(a)' ) '-------------------------------------'

            write( 6, '(a2,2i7)' ) 'TS', id_ts(j), ntraj_ts(j)

            write( 6, '(a)' ) '-------------------------------------'
            write( 6, '(a)' ) 

         end if

         if ( afed_status(1:2) .eq. 'AS' ) then

            write( 6, '(a)' ) 'AFED Restarted from following point.'
            write( 6, '(a)' ) '====================================='
            write( 6, '(a)' ) 'st     id  shots accept reject misses'
            write( 6, '(a)' ) '-------------------------------------'

            write( 6, '(a2,6i7)' ) &
     &         'EQ', id_eq(j), ntraj_eq(j), ntraj_accept_eq(j), &
     &         ntraj_reject_eq(j), ntraj_missed_eq(j)

            write( 6, '(a)' ) '-------------------------------------'
            write( 6, '(a)' ) 

         end if

!     //  master rank only
      end if

!-----------------------------------------------------------------------
!     //   line 200
!-----------------------------------------------------------------------

  200 continue

!-----------------------------------------------------------------------
!     //   memory deallocation
!-----------------------------------------------------------------------

!     //   id of EQ reference landmark
      if ( allocated( id_eq ) ) &
     &   deallocate( id_eq )

!     //   id of TS reference landmark
      if ( allocated( id_ts ) ) &
     &   deallocate( id_ts )

!     //   label of EQ reference landmark
      if ( allocated( label_eq ) ) &
     &   deallocate( label_eq )

!     //   label of TS reference landmark
      if ( allocated( label_ts ) ) &
     &   deallocate( label_ts )

!     //   number of shots from EQ
      if ( allocated( ntraj_eq ) ) &
     &   deallocate( ntraj_eq )

!     //   number of shots from TS
      if ( allocated( ntraj_ts ) ) &
     &   deallocate( ntraj_ts )

!     //   number of accepted shots from EQ
      if ( allocated( ntraj_accept_eq ) ) &
     &   deallocate( ntraj_accept_eq )

!     //   number of rejected shots from TS
      if ( allocated( ntraj_reject_eq ) ) &
     &   deallocate( ntraj_reject_eq )

!     //   number of consecutive missed shots from EQ
      if ( allocated( ntraj_missed_eq ) ) &
     &   deallocate( ntraj_missed_eq )

!     //   cv position of reference landmark
      if ( allocated( rafed_ref ) ) &
     &   deallocate( rafed_ref )

!     //   unit vector of reference landmark
      if ( allocated( e_afed_ref ) ) &
     &   deallocate( e_afed_ref )

!     //   free energy of reference landmark
      if ( allocated( fenergy_ref ) ) &
     &   deallocate( fenergy_ref )

      return
      end





!***********************************************************************
      subroutine compress_tar_afed( char_file )
!***********************************************************************

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      character(len=10) :: char_file

!-----------------------------------------------------------------------
!     //   compress *.ini file
!-----------------------------------------------------------------------

      call system &
     &   ( 'tar cvfz ' // char_file // ' geometry.ini afed.ini' )

      return
      end





!***********************************************************************
      subroutine decompress_tar_afed( char_file )
!***********************************************************************

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      character(len=10) :: char_file

!-----------------------------------------------------------------------
!     //   compress *.ini file
!-----------------------------------------------------------------------

      call system( 'tar xvfz ' // char_file )

      return
      end





!***********************************************************************
      subroutine meanforce_afed_MPI
!***********************************************************************

      use common_variables, only : &
     &   ipotential, nbead

      use afed_variables, only : &
     &   ascent_sampling_afed, afed_status

      implicit none

      if ( ipotential(1:6) .eq. 'HILLS ' ) then

         call meanforce_hills_afed_MPI

      else if ( nbead .eq. 1 ) then

         call meanforce_analytical_afed_MPI

      else if ( afed_status(1:2) .eq. 'AS' ) then

         if ( ascent_sampling_afed(1:11) .eq. 'ANALYTICAL ' ) then

            call meanforce_analytical_afed_MPI

         else

            call meanforce_numerical_afed_MPI

         end if

      else if ( ( afed_status(1:2) .eq. 'EQ' ) .or. &
     &          ( afed_status(1:2) .eq. 'TS' ) .or. &
     &          ( afed_status(1:2) .eq. 'DE' ) .or. &
     &          ( afed_status(1:2) .eq. 'D1' ) .or. &
     &          ( afed_status(1:2) .eq. 'D2' ) .or. &
     &          ( afed_status(1:2) .eq. 'HE' ) ) then

         call meanforce_analytical_afed_MPI

      else

         call meanforce_analytical_afed_MPI

      end if

      return
      end





!***********************************************************************
      subroutine meanforce_hills_afed_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fdiff, nbead, iounit

      use cons_variables, only : &
     &   rcons, scons, scons_avg, ncons

      use afed_variables, only : &
     &   f_afed, hess_afed, he_afed, e_afed, fictmass_afed, rafed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: fp(ncons), fm(ncons), dr

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   set finite difference parameter                            */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_real1_MPI ( fdiff, '<fdiff>', 7, iounit )

         call analysis_cart_MPI( 1 )

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   set constraints                                            */
!-----------------------------------------------------------------------

      do i = 1, nbead
         rcons(:,i)     = rafed(:)
         scons(:,i)     = rafed(:)
         scons_avg(:,i) = rafed(:)
      end do

!-----------------------------------------------------------------------
!     /*   mean force                                                 */
!-----------------------------------------------------------------------

      do i = 1, ncons

         dr = fdiff / sqrt(fictmass_afed(i))

         rcons(i,1) = rcons(i,1) + dr

         call getforce_hills_afed_MPI

         fp(:) = f_afed(:)

         rcons(i,1) = rcons(i,1) - 2.d0 * dr

         call getforce_hills_afed_MPI

         fm(:) = f_afed(:)

         rcons(i,1) = rcons(i,1) + dr

         hess_afed(i,:) = - ( fp(:) - fm(:) ) / ( 2.d0 * dr )

      end do

      call getforce_hills_afed_MPI

!-----------------------------------------------------------------------
!     /*   symmetrize hessian                                         */
!-----------------------------------------------------------------------

      do i = 1, ncons-1
      do j = i+1, ncons

         hess_afed(i,j) = 0.5d0 * ( hess_afed(i,j) + hess_afed(j,i) )

         hess_afed(j,i) = hess_afed(i,j)

      end do
      end do

!-----------------------------------------------------------------------
!     /*   gentlest ascent: (hessian * unit vector)                   */
!-----------------------------------------------------------------------

      do i = 1, ncons

         he_afed(i) = 0.d0

         do j = 1, ncons

            he_afed(i) = he_afed(i) + hess_afed(i,j) * e_afed(j) &
     &                 / sqrt(fictmass_afed(i)*fictmass_afed(j))

         end do

      end do

      return
      end





!***********************************************************************
      subroutine getforce_hills_afed_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, myrank

      use cons_variables, only : &
     &   rcons, ipbc_cons, ncons

      use afed_variables, only : &
     &   fenergy_afed, f_afed, gc_user_afed, gh_user_afed, &
     &   gw_user_afed, ng_user_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, ierr

      real(8) :: f0, f1, f2, f3, f4

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initial setting                                            */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'hills.dat' )

            k = 0

            do

               do i = 1, ncons
                  read ( iounit, *, iostat=ierr ) j, f1, f2, f3
               end do

               if ( ierr .ne. 0 ) exit

               k = k + 1

            end do

            ng_user_afed = k

            close( iounit )

         end if

         call my_mpi_bcast_int_0 ( ng_user_afed )

         if ( .not. allocated( gh_user_afed ) ) &
     &      allocate( gh_user_afed(ng_user_afed) )
         if ( .not. allocated( gc_user_afed ) ) &
     &      allocate( gc_user_afed(ncons,ng_user_afed) )
         if ( .not. allocated( gw_user_afed ) ) &
     &      allocate( gw_user_afed(ncons,ng_user_afed) )

         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'hills.dat' )

            do k = 1, ng_user_afed

               do i = 1, ncons
                  read ( iounit, *, iostat=ierr ) &
     &               j, gh_user_afed(k), gw_user_afed(i,k), &
     &               gc_user_afed(i,k)
               end do

            end do

            close( iounit )

         end if

         call my_mpi_bcast_real_1( gh_user_afed, ncons )
         call my_mpi_bcast_real_2( gw_user_afed, ncons, ng_user_afed )
         call my_mpi_bcast_real_2( gc_user_afed, ncons, ng_user_afed )

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      fenergy_afed = 0.d0

      f_afed(:)  = 0.d0

!-----------------------------------------------------------------------
!     /*   sum of gaussian hills                                      */
!-----------------------------------------------------------------------

      do i = 1, ng_user_afed

!        /*   height   */
         f0 = gh_user_afed(i)

         do k = 1, ncons

!           /*   displacement   */
            f1 = rcons(k,1) - gc_user_afed(k,i)

!           /*   periodic boundary condition   */
            call pbc_cons( f1, ipbc_cons(k) )

!           /*   width   */
            f2 = gw_user_afed(k,i)

!           /*   exponent   */
            f3 = 0.5d0*(f1*f1)/(f2*f2)

!           /*   potential   */
            f0 = f0*exp(-f3)

         end do

         fenergy_afed = fenergy_afed + f0

         do k = 1, ncons

!           /*   displacement   */
            f1 = rcons(k,1) - gc_user_afed(k,i)

!           /*   periodic boundary condition   */
            call pbc_cons( f1, ipbc_cons(k) )

!           /*   width   */
            f2 = gw_user_afed(k,i)

!           /*   prefactor   */
            f4 = f1/(f2*f2)

!           /*   force   */
            f_afed(k) = f_afed(k) + f0*f4

         end do

      end do

      return
      end





!***********************************************************************
      subroutine meanhessian_afed_MPI
!***********************************************************************

      use afed_variables, only : ascent_sampling_afed

      if   ( ascent_sampling_afed(1:11) .eq. 'ANALYTICAL ' ) then

         call meanhessian_analytical_afed_MPI

      else

         call meanhessian_numerical_afed_MPI

      end if

      return
      end





!***********************************************************************
      subroutine meanhessian_analytical_afed_MPI
!***********************************************************************

      call meanforce_afed_MPI

      return
      end





!***********************************************************************
      subroutine meanhessian_numerical_afed_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   hess_afed, he_afed, e_afed, e_old_afed, fictmass_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8), dimension(ncons,ncons) :: a

!-----------------------------------------------------------------------
!     /*   calculate mean hessian by finite differencing              */
!-----------------------------------------------------------------------

!     //   save original unit vector
      e_old_afed(:) = e_afed(:)

!     //   loop of finite differencing
      do i = 1, ncons

!        //   reset unit vector in direction i
         e_afed(:) = 0.d0
         e_afed(i) = 1.d0

!        //   constrained molecular dynamics
         call meanforce_afed_MPI

!        //   hessian (without mass weight)
         do j = 1, ncons
            a(j,i) = he_afed(j) &
     &         * sqrt(fictmass_afed(i)*fictmass_afed(j))
         end do

!     //   loop of finite differencing
      end do

!     //   restore original unit vector
      e_afed(:) = e_old_afed(:)

!     //   hessian (without mass weight)
      hess_afed(:,:) = a(:,:)

!-----------------------------------------------------------------------
!     /*   symmetrize hessian                                         */
!-----------------------------------------------------------------------

      do i = 1, ncons-1
      do j = i+1, ncons

         hess_afed(i,j) = 0.5d0 * ( hess_afed(i,j) + hess_afed(j,i) )

         hess_afed(j,i) = hess_afed(i,j)

      end do
      end do

      return
      end





!***********************************************************************
      subroutine printeigen_afed_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, iounit, myrank

      use cons_variables, only : &
     &   ncons

      use afed_variables, only : &
     &   rafed, fenergy_afed, e_old_afed, eigval_afed, e_afed, f_afed, &
     &   eigvec_afed, hess_afed, he_afed, fictmass_afed, afed_status

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8) :: phi_afed, cos_phi_afed, fmax_afed, frms_afed

      real(8), parameter :: fmax_conv_afed = 1.d-3
      real(8), parameter :: frms_conv_afed = 1.d-4

      integer :: icheck_afed

      integer :: i, j

      real(8) :: fi

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     //   only master process
      if ( myrank .ne. 0 ) return

!     //   open file
      open ( iounit, file = 'afed_eigen.out', access = 'append' )

!-----------------------------------------------------------------------
!     /*   maximum value of mass-weighted forces                      */
!-----------------------------------------------------------------------

      fmax_afed = f_afed(1) / sqrt(fictmass_afed(1))

      do i = 2, ncons
         fi = f_afed(i) / sqrt(fictmass_afed(i))
         fmax_afed = max( fmax_afed, fi )
      end do

!-----------------------------------------------------------------------
!     /*   root-mean-square of mass-weighted forces                   */
!-----------------------------------------------------------------------

      frms_afed = 0.d0

      do i = 1, ncons
         fi = f_afed(i) / sqrt(fictmass_afed(i))
         frms_afed = frms_afed + fi*fi
      end do

      frms_afed = frms_afed / dble(ncons)

      frms_afed = sqrt( frms_afed )

!-----------------------------------------------------------------------
!     /*   convergence of stationary points                           */
!-----------------------------------------------------------------------

      icheck_afed = 0

      if ( afed_status(1:2) .eq. 'EQ' ) then

         if ( eigval_afed(1) .lt. 0.d0 ) &
     &      icheck_afed = icheck_afed + 1

         if ( frms_afed .gt. frms_conv_afed ) &
     &      icheck_afed = icheck_afed + 100

         if ( fmax_afed .gt. fmax_conv_afed ) &
     &      icheck_afed = icheck_afed + 1000

      else if ( afed_status(1:2) .eq. 'TS' ) then

         if ( ncons .eq. 1 ) then

            if ( eigval_afed(1) .gt. 0.d0 ) &
     &         icheck_afed = icheck_afed + 1

         else

            if ( eigval_afed(1) .gt. 0.d0 ) &
     &         icheck_afed = icheck_afed + 1

            if ( eigval_afed(2) .lt. 0.d0 ) &
     &         icheck_afed = icheck_afed + 10

         end if

         if ( frms_afed .gt. frms_conv_afed ) &
     &      icheck_afed = icheck_afed + 100

         if ( fmax_afed .gt. fmax_conv_afed ) &
     &      icheck_afed = icheck_afed + 1000

      end if

!-----------------------------------------------------------------------
!     /*   print status                                               */
!-----------------------------------------------------------------------

      write( iounit, '(a,3x,a)' ) &
     &   'STATUS:    ', afed_status(1:2)

!-----------------------------------------------------------------------
!     /*   print check                                                */
!-----------------------------------------------------------------------

      if ( ( afed_status(1:2) .eq. 'EQ' ) .or. &
     &     ( afed_status(1:2) .eq. 'TS' ) ) then

         if ( icheck_afed .eq. 0 ) then

            write( iounit, '(a,3x,a)' ) &
     &         'CHECK:     ', 'OK'

         else

            write( iounit, '(a,3x,a,i5)' ) &
     &         'CHECK:     ', ' X', icheck_afed

         end if

      end if

!-----------------------------------------------------------------------
!     /*   print free energy                                          */
!-----------------------------------------------------------------------

      write( iounit, '(a,1x,f11.6)' ) &
     &   'F-ENERGY:  ', fenergy_afed

!-----------------------------------------------------------------------
!     /*   print mean force                                           */
!-----------------------------------------------------------------------

      write( iounit, '(a,1x)', advance = 'no'  ) &
     &   'MEANFORCE: '

      do i = 1, ncons-1
         write( iounit, '(f11.6)', advance = 'no' ) f_afed(i)
      end do

      write( iounit, '(f11.6)' ) f_afed(ncons)

!-----------------------------------------------------------------------
!     /*   print mean force (scaled)                                  */
!-----------------------------------------------------------------------

      write( iounit, '(a,1x)', advance = 'no'  ) &
     &   '(SCALED):  '

      do i = 1, ncons-1
         write( iounit, '(f11.6)', advance = 'no' ) &
     &      f_afed(i)/sqrt(fictmass_afed(i))
      end do

      write( iounit, '(f11.6)' ) &
     &   f_afed(ncons)/sqrt(fictmass_afed(ncons))

!-----------------------------------------------------------------------
!     /*   print cv position                                          */
!-----------------------------------------------------------------------

      write( iounit, '(a)', advance = 'no' ) &
     &   'POSITION:  '

      do i = 1, ncons-1
         write( iounit, '(f9.3,2x)', advance = 'no' ) rafed(i)
      end do

      write( iounit, '(f9.3)' ) rafed(ncons)

!-----------------------------------------------------------------------
!     /*   print cv position (scaled)                                 */
!-----------------------------------------------------------------------

      write( iounit, '(a)', advance = 'no'  ) &
     &   '(SCALED):  '

      do i = 1, ncons-1
         write( iounit, '(f9.3,2x)', advance = 'no' ) &
     &      sqrt(fictmass_afed(i))*rafed(i)
      end do

      write( iounit, '(f9.3,2x)' ) &
     &   sqrt(fictmass_afed(ncons))*rafed(ncons)

!-----------------------------------------------------------------------
!     /*   print eigenvalues                                          */
!-----------------------------------------------------------------------

      if ( ( afed_status(1:2) .eq. 'EQ' ) .or. &
     &     ( afed_status(1:2) .eq. 'TS' ) .or. &
     &     ( afed_status(1:2) .eq. 'HE' ) ) then

         write( iounit, '(a,1x)', advance = 'no' ) &
     &      'EIGVALUES: '

         do i = 1, ncons-1
            write( iounit, '(f11.6)', advance = 'no' ) eigval_afed(i)
         end do

         write( iounit, '(f11.6)' ) eigval_afed(ncons)

      end if

!-----------------------------------------------------------------------
!     /*   print eigenvector of lowest mode                           */
!-----------------------------------------------------------------------

      if ( ( afed_status(1:2) .eq. 'EQ' ) .or. &
     &     ( afed_status(1:2) .eq. 'TS' ) .or. &
     &     ( afed_status(1:2) .eq. 'HE' ) ) then

         write( iounit, '(a,2x)', advance = 'no' ) &
     &      'EIGVECTOR: '

         do i = 1, ncons-1
            write( iounit, '(f7.3,4x)', advance = 'no' ) &
     &         eigvec_afed(i,1)
         end do

         write( iounit, '(f7.3)' ) eigvec_afed(ncons,1)

      end if

!-----------------------------------------------------------------------
!     /*   print hessian                                              */
!-----------------------------------------------------------------------

      if ( afed_status(1:2) .eq. 'HE' ) then

         do j = 1, ncons

            if ( j .eq. 1 ) then
               write( iounit, '(a,3x)', advance = 'no' ) &
     &            'HESSIAN: '
            else
               write( iounit, '(a,3x)', advance = 'no' ) &
     &            '         '
            end if

            do i = 1, ncons-1
               write( iounit, '(f11.6)', advance = 'no' ) hess_afed(i,j)
            end do

            write( iounit, '(f11.6)' ) hess_afed(ncons,j)

         end do

      end if

!-----------------------------------------------------------------------
!     /*   print mass-weighted hessian                                */
!-----------------------------------------------------------------------

      if ( afed_status(1:2) .eq. 'HE' ) then

         do j = 1, ncons

            if ( j .eq. 1 ) then
               write( iounit, '(a,2x)', advance = 'no' ) &
     &            'HESS/MASS:'
            else
               write( iounit, '(a,3x)', advance = 'no' ) &
     &            '         '
            end if

            do i = 1, ncons-1
               write( iounit, '(f11.6)', advance = 'no' ) &
     &            hess_afed(i,j) &
     &            / sqrt(fictmass_afed(i)*fictmass_afed(j))
            end do

            write( iounit, '(f11.6)' ) &
     &         hess_afed(ncons,j) &
     &         / sqrt(fictmass_afed(ncons)*fictmass_afed(j))

         end do

      end if

!-----------------------------------------------------------------------
!     /*   print hessian times unit vector                            */
!-----------------------------------------------------------------------

      if ( afed_status(1:2) .eq. 'TS' ) then

         write( iounit, '(a,3x)', advance = 'no' ) &
     &      'HESS*VEC:'

         do i = 1, ncons-1
            write( iounit, '(f11.6)', advance = 'no' ) he_afed(i)
         end do

         write( iounit, '(f11.6)' ) he_afed(ncons)

      end if

!-----------------------------------------------------------------------
!     /*   print angle between unit vector and eigenvector            */
!-----------------------------------------------------------------------

      if ( afed_status(1:2) .eq. 'TS' ) then

         cos_phi_afed = 0.d0

         do i = 1, ncons
            cos_phi_afed = cos_phi_afed + e_afed(i) * e_old_afed(i)
         end do

         cos_phi_afed = min( max( cos_phi_afed, -1.d0 ), 1.d0 )

         phi_afed = acos(cos_phi_afed) * 180.d0 / pi

         phi_afed = phi_afed - nint(phi_afed/180.d0) * 180.d0

         write( iounit, '(a,i5)' ) &
     &      'DEVIATION: ', nint(phi_afed)

      end if

!-----------------------------------------------------------------------
!     /*   end                                                        */
!-----------------------------------------------------------------------

!     //   add a blank line
      write( iounit, '(a)' )

!     //   close file
      close( iounit )

      return
      end





!***********************************************************************
      subroutine afedcycle_hessian_MPI
!***********************************************************************

!     //   local variables
      implicit none

!     //   calculate hessian
      call meanhessian_afed_MPI

!     //   eigenvalues and eigenvectors (overwrite e_afed)
      call geteigen_afed_MPI

!     //   print eigenvalues and eigenvectors
      call printeigen_afed_MPI

!     //   standard output
      call standard_adescent_afed_MPI

!     //   analysis
      call analysis_afed_MPI

!     //   save data
      call backup_adescent_afed_MPI

      return
      end





!***********************************************************************
      subroutine update_hessian_afed( ioption )
!***********************************************************************
!-----------------------------------------------------------------------
!
!     Murtagh−Sargent−Powell update formula
!
!-----------------------------------------------------------------------

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use afed_variables, only : &
     &   fictmass_afed, rafed, f_afed, hess_afed, r_old_afed, &
     &   g_old_afed, h_old_afed, r_new_afed, g_new_afed, h_new_afed

      use cons_variables, only : ncons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, ioption

      real(8) :: dx(ncons), j0(ncons), u0(ncons), w(ncons,ncons)
      real(8) :: c1, c2, c3, e, phi
      real(8) :: eps = 1.d-4

!-----------------------------------------------------------------------
!     /*   old cv, gradient, hessian                                  */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         if ( .not. allocated(r_old_afed) ) &
     &      allocate( r_old_afed(ncons) )

         if ( .not. allocated(g_old_afed) ) &
     &      allocate( g_old_afed(ncons) )

         if ( .not. allocated(h_old_afed) ) &
     &      allocate( h_old_afed(ncons,ncons) )

         do i = 1, ncons
            r_old_afed(i) = sqrt(fictmass_afed(i)) * rafed(i)
         end do

         do i = 1, ncons
            g_old_afed(i) = - f_afed(i) / sqrt(fictmass_afed(i))
         end do

         do i = 1, ncons
         do j = 1, ncons
            h_old_afed(i,j) = hess_afed(i,j) &
     &                 / sqrt(fictmass_afed(i)*fictmass_afed(j))
         end do
         end do

         return

!-----------------------------------------------------------------------
!     /*   new cv, gradient, hessian                                  */
!-----------------------------------------------------------------------

      else

         if ( .not. allocated(r_new_afed) ) &
     &      allocate( r_new_afed(ncons) )

         if ( .not. allocated(g_new_afed) ) &
     &      allocate( g_new_afed(ncons) )

         if ( .not. allocated(h_new_afed) ) &
     &      allocate( h_new_afed(ncons,ncons) )

         do i = 1, ncons
            r_new_afed(i) = sqrt(fictmass_afed(i)) * rafed(i)
         end do

         do i = 1, ncons
            g_new_afed(i) = - f_afed(i) / sqrt(fictmass_afed(i))
         end do

      end if

!-----------------------------------------------------------------------
!     /*   dx = delta x0 vector                                       */
!-----------------------------------------------------------------------

      do i = 1, ncons
         dx(i) = r_new_afed(i) - r_old_afed(i)
      end do

!-----------------------------------------------------------------------
!     /*   j0 vector                                                  */
!-----------------------------------------------------------------------

      do i = 1, ncons
         j0(i) = g_new_afed(i) - g_old_afed(i)
         do j = 1, ncons
            j0(i) = j0(i) + h_old_afed(i,j) * dx(j)
         end do
      end do

!-----------------------------------------------------------------------
!     /*   phi value                                                  */
!-----------------------------------------------------------------------

      c1 = 0.d0
      c2 = 0.d0
      c3 = 0.d0
      do i = 1, ncons
         c1 = c1 + dx(i) * j0(i)
         c2 = c2 + dx(i) * dx(i)
         c3 = c3 + j0(i) * j0(i)
      end do

      if ( c2 .lt. eps ) return
      if ( c3 .lt. eps ) return

      phi = c1 * c1 / ( c2 * c3 )

!-----------------------------------------------------------------------
!     /*   w matrix                                                   */
!-----------------------------------------------------------------------

      do i = 1, ncons
      do j = 1, ncons
         w(i,j) = phi * dx(i) * dx(j) + ( 1.d0 - phi ) * j0(i) * j0(j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   u0 vector                                                  */
!-----------------------------------------------------------------------

      do i = 1, ncons
         u0(i) = 0.d0
         do j = 1, ncons
            u0(i) = u0(i) + w(i,j) * dx(j)
         end do
      end do

      e = 0.d0
      do i = 1, ncons
         e = e + u0(i) * dx(i)
      end do

      do i = 1, ncons
         u0(i) = u0(i) / e
      end do

!-----------------------------------------------------------------------
!     /*   new hessian                                                */
!-----------------------------------------------------------------------

      do i = 1, ncons
      do j = 1, ncons
         h_new_afed(i,j) = h_old_afed(i,j) &
     &      + j0(i)*u0(j) + j0(j)*u0(i) - c1*u0(i)*u0(j)
      end do
      end do

      h_old_afed(:,:) = h_new_afed(:,:)

!-----------------------------------------------------------------------
!     /*   new hessian                                                */
!-----------------------------------------------------------------------

      do i = 1, ncons
      do j = 1, ncons
         hess_afed(i,j) &
     &      = h_new_afed(i,j) * sqrt(fictmass_afed(i)*fictmass_afed(j))
      end do
      end do

      return
      end

