!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, Y. Nagai
!      Last updated:    Dec 30, 2020 by M. Shiga
!      Description:     update step in hybrid Monte Carlo
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine adjust_step_rehmc
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

!     /*   variables in common module   */
      use common_variables, only: istep_hmc, iounit, istep

!     /*   variables in hmc module   */
      use rehmc_variables, only : ratio_min_hmc, ratio_max_hmc, &
     &   istep_min_hmc, istep_max_hmc, istep_mul_hmc, istep_adjust_hmc, &
     &   naccept_hmc, nreject_hmc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize local variables   */
      implicit none

!     /*   number of metropolis trials   */
      integer :: ntrials_hmc  = 0

!     /*   integers   */
      integer :: itest

!     /*   flag   */
      integer, save :: iset = 0

!     /*   number of acceptance in last adjustment   */
      integer, save :: naccept_last  = 0

!     /*   number of rejection in last adjustment   */
      integer, save :: nreject_last  = 0

!     /*   number of trials in last adjustment   */
      integer, save :: ntrials_last  = 0

!     /*   saved total number of acceptance   */
      integer, save :: naccept_save  = 0

!     /*   saved total number of rejection   */
      integer, save :: nreject_save  = 0

!     /*   saved total number of trials   */
      integer, save :: ntrials_save  = 0

!     /*   acceptance ratio in last adjustment   */
      real(8), save :: ratio_last = 0.d0

!     /*   total acceptance ratio   */
      real(8), save :: ratio_hmc  = 0.d0

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

!     /*   do not adjust istep_hmc  */
      if ( ( istep_adjust_hmc .lt. 0 ) .or. ( istep_mul_hmc .eq. 1 ) ) &
     &   return

!-----------------------------------------------------------------------
!     /*   total number of trials and acceptance ratio                */
!-----------------------------------------------------------------------

!     /*   total number of trials   */
      ntrials_hmc = naccept_hmc + nreject_hmc

!     /*   total acceptance ratio   */
      if (  ntrials_hmc .gt. 0 ) then
         ratio_hmc = dble(naccept_hmc) / dble(ntrials_hmc)
      end if

!-----------------------------------------------------------------------
!     /*   reset when first visit to this routine                     */
!-----------------------------------------------------------------------

!     /*   first visit  */
      if ( iset .eq. 0 ) then

!        /*   number of acceptance and rejection in last adjustment   */
         naccept_last = 0
         nreject_last = 0

!        /*   number of trials in last adjustment   */
         ntrials_last = naccept_last + nreject_last

!        /*   total number of acceptance and rejection   */
         naccept_save = naccept_hmc
         nreject_save = nreject_hmc

!        /*   total number of trials   */
         ntrials_save = naccept_save + nreject_save

!        /*   check existence of file   */
         call testfile ( 'hmc.out', 7, itest )

!        /*   if file does not exist   */
         if ( itest .ne. 0 ) then

!           /*   open file   */
            open ( iounit, file ='hmc.out', access='append' )

!           /*   print header   */
            write( iounit, '(a)' ) '================================'
            write( iounit, '(a)' ) '    step   moves   last%  total%'
            write( iounit, '(a)' ) '--------------------------------'

!           /*   close file   */
            close( iounit )

!        /*   if file does not exist   */
         end if

!        /*   flag   */
         iset = 1

!        /*   finish   */
         return

!     /*   first visit  */
      end if

!-----------------------------------------------------------------------
!     /*   update acceptance/rejection counter                        */
!-----------------------------------------------------------------------

!     /*   condition   */
      if ( ( mod(ntrials_hmc,istep_adjust_hmc) .ne. 0 ) .or. &
     &     ( ntrials_save .lt. istep_adjust_hmc-1 ) ) then

!        /*   number of acceptance and rejection in last rung   */
         naccept_last = naccept_last + ( naccept_hmc - naccept_save )
         nreject_last = nreject_last + ( nreject_hmc - nreject_save )

!        /*   number of trials in last adjustment   */
         ntrials_last = naccept_last + nreject_last

!        /*   total number of acceptance and rejection   */
         naccept_save = naccept_hmc
         nreject_save = nreject_hmc

!        /*   total number of trials   */
         ntrials_save = naccept_save + nreject_save

!        /*   finish   */
         return

!     /*   condition   */
      end if

!-----------------------------------------------------------------------
!     /*   calculate acceptance ratio                                 */
!-----------------------------------------------------------------------

!     /*   number of acceptance and rejection in last rung   */
      naccept_last = naccept_last + ( naccept_hmc - naccept_save )
      nreject_last = nreject_last + ( nreject_hmc - nreject_save )

!     /*   number of trials in last adjustment   */
      ntrials_last = naccept_last + nreject_last

!     /*   acceptance ratio in last adjustment   */
      ratio_last = dble(naccept_last) / dble(ntrials_last)

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

!     /*   open file   */
      open ( iounit, file = 'hmc.out', access = 'append' )

!     /*   print data   */
      write( iounit, '(2i8,2f8.5)' ) &
     &   istep, istep_hmc, ratio_last, ratio_hmc

!     /*   close file   */
      close( iounit )

!-----------------------------------------------------------------------
!     /*   modify istep_hmc                                           */
!-----------------------------------------------------------------------

!     /*   if acceptance ratio is large  */
      if ( ratio_last .gt. ratio_max_hmc ) then

!        /*   increase istep_hmc   */
         istep_hmc = istep_hmc * istep_mul_hmc

!        /*   ensure istep_hmc is larger than minimum value   */
         istep_hmc = min( istep_hmc, istep_max_hmc )

!     /*   end of if statement   */
      end if

!     /*   if acceptance ratio is small  */
      if ( ratio_last .lt. ratio_min_hmc ) then

!        /*   decrease istep_hmc   */
         istep_hmc = istep_hmc / istep_mul_hmc

!        /*   ensure istep_hmc is larger than maximum value   */
         istep_hmc = max( istep_hmc, istep_min_hmc )

!     /*   end of if statement   */
      end if

!-----------------------------------------------------------------------
!     /*   reset                                                      */
!-----------------------------------------------------------------------

!     /*   number of acceptance in last adjustment   */
      naccept_last = 0

!     /*   number of rejection in last adjustment   */
      nreject_last = 0

!     /*   number of trials in last adjustment   */
      ntrials_last = naccept_last + nreject_last

!     /*   save total number of acceptance   */
      naccept_save = naccept_hmc

!     /*   save total number of rejection   */
      nreject_save = nreject_hmc

!     /*   save total number of trials   */
      ntrials_save = naccept_save + nreject_save

      return
      end
