! Copyright 2014-2017  Kazuya Ishimura
!
! Licensed under the Apache License, Version 2.0 (the "License");
! you may not use this file except in compliance with the License.
! You may obtain a copy of the License at
!
!     http://www.apache.org/licenses/LICENSE-2.0
!
! Unless required by applicable law or agreed to in writing, software
! distributed under the License is distributed on an "AS IS" BASIS,
! WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
! See the License for the specific language governing permissions and
! limitations under the License.
!
!------------------------------------------------------------------------
  subroutine guessmo(cmoa,cmob,overinv,h1mtrx,ortho, &
&                    nproc1,nproc2,myrank1,myrank2,mpi_comm1,mpi_comm2)
!------------------------------------------------------------------------
!
! In   : h1mtrx  (one-electron Hamiltonian matrx)
!        ortho   (Orthogonalization matrix)
! Out  : cmoa    (Alpha initial guess orbitals)
!        cmob    (Beta initial guess orbitals)
! Inout: overinv (Overlap integral inverse matrix, and then work space)
!
      use modparallel, only : master
      use modguess, only : guess
      use modbasis, only : nao
      use modjob, only : scftype
      implicit none
      integer,intent(in) :: nproc1, nproc2, myrank1, myrank2, mpi_comm1, mpi_comm2
      real(8),intent(in) :: h1mtrx(nao*(nao+1)/2), ortho(nao*nao)
      real(8),intent(inout):: cmoa(nao*nao), cmob(nao*nao), overinv(nao*nao)
!
      if(master) then
        write(*,'(" ---------------------")')
        write(*,'("   Guess calculation")')
        write(*,'(" ---------------------")')
      endif
!
      if(guess == 'HUCKEL') then
        if(master) write(*,'("   Guess MOs are generated using extended Huckel orbitals.")')
        call huckelguess(cmoa,overinv,nproc2,myrank2,mpi_comm2)
        if(master) write(*,*)
        if(scftype == 'UHF') call dcopy(nao*nao,cmoa,1,cmob,1)
!
      elseif(guess == 'DFTB') then
        if(master) write(*,'("   Guess MOs are generated using DFTB orbitals.")')
        call dftbguess(cmoa,overinv,nproc2,myrank2,mpi_comm2)
        if(master) write(*,*)
        if(scftype == 'UHF') call dcopy(nao*nao,cmoa,1,cmob,1)
!
      elseif(guess == 'UPDATE') then
        call updatemo(cmoa,overinv,nproc2,myrank2,mpi_comm2)
        if(scftype == 'UHF') call dcopy(nao*nao,cmoa,1,cmob,1)
!
      elseif(guess == 'CHECK') then
        if(master) write(*,'("   Guess MOs are read from checkpoint file and projected.",/)')
        call checkguess(cmoa,cmob,overinv,nproc2,myrank2,mpi_comm1,mpi_comm2)
      elseif(guess == 'HCORE') then
        if(master) &
&         write(*,'("   Guess MOs are generated by diagonalizing core Hamiltonian matrix.",/)')
        call hcoreguess(cmoa,h1mtrx,ortho,overinv,nproc2,myrank2,mpi_comm2)
        if(scftype == 'UHF') call dcopy(nao*nao,cmoa,1,cmob,1)
      else
        if(master) write(*,'(" Error! This program does not support guess= ",a10,"!")') guess
        call iabort
      endif
!
      if(master) then
        write(*,'(" If the calculation stops at the beginning of SCF, " &
&                 "increase the stack size.")')
        write(*,'(" bash: Write the following two lines in the ~/.bashrc file.")')
        write(*,'("       ulimit -s unlimited")')
        write(*,'("       export OMP_STACKSIZE=1G")')
        write(*,'(" tcsh: Write the following two lines in the ~/.tcshrc file.")')
        write(*,'("       unlimit")')
        write(*,'("       setenv OMP_STACKSIZE 1G",/)')
      endif
      return
end


!------------------------------------------------------------
  subroutine huckelguess(cmo,overinv,nproc,myrank,mpi_comm)
!------------------------------------------------------------
!
! Calculate initial guess MOs by extended Huckel
!
! In  : overinv (overlap integral inverse matrix)
! Out : cmo     (initial guess orbitals)
!
      use modguess, only : nao_g, spher_g, coord_g
      use modbasis, only : nao
      use modmolecule, only : coord, natom
      implicit none
      integer,intent(in) :: nproc, myrank, mpi_comm
      integer :: i, j, nao_g2
      real(8),intent(in):: overinv(nao*nao)
      real(8),intent(out):: cmo(nao*nao)
      real(8),allocatable :: hmo(:), overlap(:), work1(:), work2(:), eigen(:)
!
! Set basis functions
!
      spher_g=.true.
      call setbasis_g(1)
!
! Set coordinate
!
      do i= 1,natom
        do j= 1,3
          coord_g(j,i)=coord(j,i)
        enddo
      enddo
!
! Set required arrays
!
      nao_g2= nao_g*nao_g
      call memset(2*nao_g2+2*nao*nao_g+nao_g)
      allocate(hmo(nao_g2),overlap(nao*nao_g),work1(nao*nao_g),work2(nao_g2),eigen(nao_g))
!
! Calculate Extended Huckel method
!
      call calchuckelg(hmo,eigen,nproc,myrank,mpi_comm)
!
! Calculate overlap integrals between input basis and Huckel basis
!
      call calcover2(overlap,work1,nproc,myrank,mpi_comm)
!
! Project orbitals from Huckel to SCF
!
      call projectmo(cmo,overinv,overlap,hmo,work1,work2,eigen,nproc,myrank,mpi_comm)
      deallocate(hmo,overlap,work1,work2,eigen)
      call memunset(2*nao_g2+2*nao*nao_g+nao_g)
      return
end


!----------------------------------------------------------
  subroutine calchuckelg(hmo,eigen,nproc,myrank,mpi_comm)
!----------------------------------------------------------
!
! Driver of extended Huckel calculation for guess generation
!
! Out : hmo (extended Huckel orbitals)
!
      use modparallel, only : master
      use modguess, only : nao_g, nmo_g
      use modprint, only : iprint
      implicit none
      integer,intent(in) :: nproc, myrank, mpi_comm
      real(8),parameter :: zero=0.0D+00, one=1.0D+00
      real(8),intent(out) :: hmo(nao_g*nao_g), eigen(nao_g)
!
! Calculate overlap integrals
! (guess basis)x(guess basis)
!
      call calcover1(hmo)
!
! Set ionization potentials
!
      call huckelip(eigen,1)
!
! Form extended Huckel matrix
!
      call formhuckel(hmo,eigen)
!
! Diagonalize extended Huckel matrix
!
      call diag('V','U',nao_g,hmo,nao_g,eigen,nproc,myrank,mpi_comm)
!
      if(master.and.(iprint >= 3)) then
        write(*,'("   Eigenvalues of Huckel guess")')
        write(*,'(4x,8f9.3)')eigen
      endif
!
      nmo_g= nao_g
      return
end


!--------------------------------------------------------------
  subroutine calchuckelgcore(hmo,eigen,nproc,myrank,mpi_comm)
!--------------------------------------------------------------
!
! Driver of extended Huckel calculation for only core orbitals
!
! Out : hmo (extended Huckel orbitals)
!
      use modguess, only : nao_gcore
      implicit none
      integer,intent(in) :: nproc, myrank, mpi_comm
      real(8),parameter :: zero=0.0D+00, one=1.0D+00
      real(8),intent(out) :: hmo(nao_gcore*nao_gcore), eigen(nao_gcore)
!
! Calculate overlap integrals
! (guess basis)x(guess basis)
!
      call calcover1core(hmo)
!
! Set ionization potentials
!
      call huckelip(eigen,2)
!
! Form extended Huckel matrix
!
      call formhuckelcore(hmo,eigen)
!
! Diagonalize extended Huckel matrix
!
      call diag('V','U',nao_gcore,hmo,nao_gcore,eigen,nproc,myrank,mpi_comm)
!
      return
end


!----------------------------------------------------------------------------------------
  subroutine projectmo(cmo,overinv,overlap,hmo,work1,work2,eigen,nproc,myrank,mpi_comm)
!----------------------------------------------------------------------------------------
!
! Project orbitals from Huckel to SCF
!    C1= S11^-1 * S12 * C2 [C2t * S12t * S11^-1 * S12 * C2]^-1/2
!
! In  :  overinv (overlap integral inverse matrix of SCF basis set)
! Inout: overlap (overlap integral of guess and SCF basis sets)
!        hmo (extended Huckel orbitals)
! Out :  cmo (initial guess orbitals)
!
      use modguess, only : nao_g, nmo_g
      use modbasis, only : nao
      implicit none
      integer,intent(in) :: nproc, myrank, mpi_comm
      integer :: i, j, nskip
      real(8),parameter :: zero=0.0D+00, one=1.0D+00, small=1.0D-5
      real(8),intent(in) :: overinv(nao,nao)
      real(8),intent(inout) :: overlap(nao,nao_g), hmo(nao_g,nao_g)
      real(8),intent(out) :: cmo(nao,nao), work1(nao_g,nao_g), work2(nao_g,nao_g)
      real(8),intent(out) :: eigen(nao_g)
      real(8) :: eigeninv
!
! Calculate S12*C2
!
      call dgemm('N','N',nao,nmo_g,nao_g,one,overlap,nao,hmo,nao_g,zero,cmo,nao)
!
! Calculate S11^-1*S12*C2
!
      call dsymm('L','U',nao,nmo_g,one,overinv,nao,cmo,nao,zero,overlap,nao)
!
! Calculate C2t*S12t*S11^-1*S12*C2
!
      call dgemm('T','N',nmo_g,nmo_g,nao,one,cmo,nao,overlap,nao,zero,work1,nao_g)
!
! Calculate (C2t*S12t*S11^-1*S12*C2)^-1/2
!
      call diag('V','U',nmo_g,work1,nao_g,eigen,nproc,myrank,mpi_comm)
!
      nskip= 0
      do i= 1,nmo_g
        if(eigen(i) <= small) then
          nskip= nskip+1
        else
          cycle
        endif
      enddo
!
!$OMP parallel do private(eigeninv)
      do i= nskip+1,nmo_g
        eigeninv= one/sqrt(eigen(i))
        do j= 1,nmo_g
          work2(j,i)= work1(j,i)*eigeninv
        enddo
      enddo
!$OMP end parallel do
      call dgemm('N','T',nmo_g,nmo_g,nmo_g-nskip,one,work1(1,nskip+1),nao_g,work2(1,nskip+1), &
&                nao_g,zero,hmo,nao_g)
!
! Calculate C1
!
      call dgemm('N','N',nao,nmo_g,nmo_g,one,overlap,nao,hmo,nao_g,zero,cmo,nao)
!
      return
end


!------------------------------------
  subroutine huckelip(energy,itype)
!------------------------------------
!
! Set ionization potentials
!
! In  : itype  (1: normal Huckel calculation)
!              (2: Huckel calculation for core orbitals)
! Out : energy (ionization potential)
!
      use modparallel, only : master
      use modmolecule, only : natom, numatomic
      use modecp, only : flagecp, izcore
      implicit none
      integer,intent(in) :: itype
      integer :: iao, iatom, i
      real(8),parameter :: one=1.0D+00
      real(8),intent(out) :: energy(*)
      real(8) :: row1(2)=(/-5.000D-01,-9.180D-01/)
      real(8) :: row2(3,3:10)
      real(8) :: row3(5,11:18)
      real(8) :: row4(8,19:36)
      real(8) :: row5(11,37:54)
      real(8) :: row6(15,55:86)
! The order is 1S,2S,2P
      data row2/-2.48D+00,-1.960D-01, 0.000D+00, -4.73D+00,-3.090D-01, 0.000D+00,&
&               -7.70D+00,-4.950D-01,-3.100D-01, -1.13D+01,-7.060D-01,-4.330D-01,&
&               -1.56D+01,-9.450D-01,-5.680D-01, -2.07D+01,-1.244D+00,-6.320D-01,&
&               -2.64D+01,-1.573D+00,-7.300D-01, -3.28D+01,-1.930D+00,-8.500D-01/
! The order is 1S,2S,2P,3S,3P
      data row3/-4.05D+01,-2.80D+00,-1.52D+00,-1.820D-01, 0.000D+00,&
&               -4.90D+01,-3.77D+00,-2.28D+00,-2.530D-01, 0.000D+00,&
&               -5.85D+01,-4.91D+00,-3.22D+00,-3.930D-01,-2.100D-01,&
&               -6.88D+01,-6.16D+00,-4.26D+00,-5.400D-01,-2.970D-01,&
&               -8.00D+01,-7.51D+00,-5.40D+00,-6.960D-01,-3.920D-01,&
&               -9.20D+01,-9.00D+00,-6.68D+00,-8.800D-01,-4.370D-01,&
&               -1.04D+02,-1.06D+01,-8.07D+00,-1.073D+00,-5.060D-01,&
&               -1.186D+02,-1.23D+01,-9.57D+00,-1.278D+00,-5.910D-01/
! The order is 1S,2S,2P,3S,3P,3D,4S,4P
      data row4/-1.335D+02,-1.450D+01,-1.150D+01,-1.750D+00,-9.500D-01,&
&                0.000D+00,-1.470D-01, 0.000D+00,&
&               -1.494D+02,-1.680D+01,-1.360D+01,-2.240D+00,-1.340D+00,&
&                0.000D+00,-1.960D-01, 0.000D+00,&
&               -1.659D+02,-1.908D+01,-1.567D+01,-2.570D+00,-1.575D+00,&
&               -1.193D+00,-4.240D-01,-2.080D-01,&
&               -1.833D+02,-2.142D+01,-1.779D+01,-2.874D+00,-1.795D+00,&
&               -3.430D-01,-2.100D-01,-1.000D-01,&
&               -2.013D+02,-2.370D+01,-1.980D+01,-2.990D+00,-1.840D+00,&
&               -4.410D-01,-2.200D-01,-1.000D-01,&
&               -2.204D+02,-2.620D+01,-2.210D+01,-3.290D+00,-2.050D+00,&
&               -3.210D-01,-2.140D-01,-1.000D-01,&
&               -2.404D+02,-2.890D+01,-2.460D+01,-3.620D+00,-2.300D+00,&
&               -3.730D-01,-2.220D-01,-1.000D-01,&
&               -2.612D+02,-3.170D+01,-2.720D+01,-3.960D+00,-2.550D+00,&
&               -3.830D-01,-2.270D-01,-1.000D-01,&
&               -2.829D+02,-3.460D+01,-2.990D+01,-4.300D+00,-2.800D+00,&
&               -4.060D-01,-2.300D-01,-1.000D-01,&
&               -3.054D+02,-3.770D+01,-3.270D+01,-4.650D+00,-3.060D+00,&
&               -4.340D-01,-2.330D-01,-1.000D-01,&
&               -3.288D+02,-4.080D+01,-3.560D+01,-5.010D+00,-3.320D+00,&
&               -4.570D-01,-2.360D-01,-1.000D-01,&
&               -3.533D+02,-4.440D+01,-3.890D+01,-5.630D+00,-3.840D+00,&
&               -4.910D-01,-2.380D-01,-1.000D-01,&
&               -3.788D+02,-4.820D+01,-4.250D+01,-6.400D+00,-4.480D+00,&
&               -7.830D-01,-2.930D-01,-1.000D-01,&
&               -4.052D+02,-5.210D+01,-4.620D+01,-7.190D+00,-5.170D+00,&
&               -1.635D+00,-5.530D-01,-2.870D-01,&
&               -4.326D+02,-5.630D+01,-5.020D+01,-8.030D+00,-5.880D+00,&
&               -2.113D+00,-6.860D-01,-3.690D-01,&
&               -4.609D+02,-6.070D+01,-5.430D+01,-8.930D+00,-6.660D+00,&
&               -2.650D+00,-8.380D-01,-4.030D-01,&
&               -4.901D+02,-6.520D+01,-5.860D+01,-9.870D+00,-7.480D+00,&
&               -3.220D+00,-9.930D-01,-4.570D-01,&
&               -5.202D+02,-6.990D+01,-6.300D+01,-1.080D+01,-8.330D+00,&
&               -3.825D+00,-1.153D+00,-5.240D-01/
! The order is 1S,2S,2P,3S,3P,3D,4S,4P,4D,5S,5P
      data row5/-5.510D+02,-7.500D+01,-6.790D+01,-1.210D+01,-9.500D+00,-4.700D+00,&
&               -1.520D+00,-8.100D-01, 0.000D+00,-1.380D-01, 0.000D+00,&
&               -5.837D+02,-8.040D+01,-7.300D+01,-1.350D+01,-1.070D+01,-5.700D+00,&
&               -1.900D+00,-1.100D+00, 0.000D+00,-1.780D-01, 0.000D+00,&
&               -6.168D+02,-8.581D+01,-7.816D+01,-1.476D+01,-1.185D+01,-6.599D+00,&
&               -2.168D+00,-1.300D+00,-2.499D-01,-1.958D-01,-1.000D-01,&
&               -6.507D+02,-9.138D+01,-8.348D+01,-1.606D+01,-1.302D+01,-7.515D+00,&
&               -2.418D+00,-1.487D+00,-3.365D-01,-2.070D-01,-1.000D-01,&
&               -6.854D+02,-9.700D+01,-8.880D+01,-1.720D+01,-1.400D+01,-8.300D+00,&
&               -2.530D+00,-1.550D+00,-2.990D-01,-2.140D-01,-1.000D-01,&
&               -7.212D+02,-1.029D+02,-9.450D+01,-1.860D+01,-1.530D+01,-9.300D+00,&
&               -2.760D+00,-1.720D+00,-3.570D-01,-2.220D-01,-1.000D-01,&
&               -7.579D+02,-1.089D+02,-1.002D+02,-2.000D+01,-1.660D+01,-1.030D+01,&
&               -3.000D+00,-1.910D+00,-3.770D-01,-2.220D-01,-1.000D-01,&
&               -7.955D+02,-1.152D+02,-1.062D+02,-2.140D+01,-1.780D+01,-1.130D+01,&
&               -3.260D+00,-2.100D+00,-4.120D-01,-2.220D-01,-1.000D-01,&
&               -8.340D+02,-1.216D+02,-1.124D+02,-2.290D+01,-1.920D+01,-1.240D+01,&
&               -3.500D+00,-2.290D+00,-4.510D-01,-2.200D-01,-1.000D-01,&
&               -8.735D+02,-1.281D+02,-1.187D+02,-2.440D+01,-2.050D+01,-1.350D+01,&
&               -3.750D+00,-2.480D+00,-4.880D-01,-2.200D-01,-1.000D-01,&
&               -9.138D+02,-1.349D+02,-1.252D+02,-2.590D+01,-2.190D+01,-1.470D+01,&
&               -4.000D+00,-2.680D+00,-5.370D-01,-2.200D-01,-1.000D-01,&
&               -9.554D+02,-1.421D+02,-1.321D+02,-2.770D+01,-2.360D+01,-1.610D+01,&
&               -4.450D+00,-3.050D+00,-7.630D-01,-2.650D-01,-1.000D-01,&
&               -9.978D+02,-1.494D+02,-1.392D+02,-2.960D+01,-2.540D+01,-1.760D+01,&
&               -4.980D+00,-3.510D+00,-1.063D+00,-3.720D-01,-1.970D-01,&
&               -1.041D+03,-1.570D+02,-1.465D+02,-3.160D+01,-2.720D+01,-1.920D+01,&
&               -5.510D+00,-3.970D+00,-1.369D+00,-4.760D-01,-2.650D-01,&
&               -1.086D+03,-1.648D+02,-1.540D+02,-3.360D+01,-1.920D+01,-2.080D+01,&
&               -6.060D+00,-4.450D+00,-1.688D+00,-5.820D-01,-3.350D-01,&
&               -1.131D+03,-1.728D+02,-1.617D+02,-3.580D+01,-3.110D+01,-2.250D+01,&
&               -6.650D+00,-4.950D+00,-2.038D+00,-7.010D-01,-3.600D-01,&
&               -1.177D+03,-1.809D+02,-1.697D+02,-3.790D+01,-3.310D+01,-2.430D+01,&
&               -7.240D+00,-5.470D+00,-2.401D+00,-8.210D-01,-4.030D-01,&
&               -1.224D+03,-1.893D+02,-1.778D+02,-4.020D+01,-3.520D+01,-2.610D+01,&
&               -7.860D+00,-6.010D+00,-2.778D+00,-9.440D-01,-4.570D-01/
! The order is 1S,2S,2P,3S,3P,3D,4S,4P,4D,4F,5S,5P,5D,6S,6P
      data row6/-2.5455D+03,-3.9630D+02,-3.7260D+02,-8.5390D+01,-7.5190D+01,-5.6450D+01,&
&               -1.7390D+01,-1.3540D+01,-6.7590D+00, 0.0000D+00,-2.4630D+00,-1.3670D+00,&
&                0.0000D+00,-2.4730D-01, 0.0000D+00,&
&               -2.6442D+03,-4.1430D+02,-3.9010D+02,-9.0560D+01,-8.0080D+01,-6.0800D+01,&
&               -1.9110D+01,-1.5100D+01,-8.0030D+00, 0.0000D+00,-3.0250D+00,-1.8080D+00,&
&                0.0000D+00,-3.1500D-01, 0.0000D+00,&
&               -2.7445D+03,-4.3260D+02,-4.0780D+02,-9.5690D+01,-8.4920D+01,-6.5110D+01,&
&               -2.0690D+01,-1.6520D+01,-9.1070D+00,-7.1800D-01,-3.4090D+00,-2.0990D+00,&
&                0.0000D+00,-3.4080D-01, 0.0000D+00,&
&               -2.8462D+03,-4.5050D+02,-4.2520D+02,-1.0020D+02,-8.9100D+01,-6.8770D+01,&
&               -2.1620D+01,-1.7310D+01,-9.6360D+00,-8.9300D-01,-3.5080D+00,-2.1560D+00,&
&                0.0000D+00,-3.4525D-01, 0.0000D+00,&
&               -2.9491D+03,-4.6820D+02,-4.4241D+02,-1.0400D+02,-9.2671D+01,-7.1830D+01,&
&               -2.1930D+01,-1.7492D+01,-9.6043D+00,-9.5104D-01,-3.3236D+00,-1.9759D+00,&
&                0.0000D+00,-3.2825D-01, 0.0000D+00,&
&               -3.0545D+03,-4.8680D+02,-4.6051D+02,-1.0859D+02,-9.6973D+01,-7.5608D+01,&
&               -2.2842D+01,-1.8266D+01,-1.0117D+01,-1.0277D+00,-3.4109D+00,-2.0226D+00,&
&                0.0000D+00,-3.3206D-01, 0.0000D+00,&
&               -3.1616D+03,-5.0580D+02,-4.7894D+02,-1.1324D+02,-1.0134D+02,-7.9449D+01,&
&               -2.3755D+01,-1.9038D+01,-1.0627D+01,-1.0952D+00,-3.4952D+00,-2.0667D+00,&
&                0.0000D+00,-3.3573D-01, 0.0000D+00,&
&               -3.2706D+03,-5.2510D+02,-4.9772D+02,-1.1796D+02,-1.0577D+02,-8.3355D+01,&
&               -2.4669D+01,-1.9811D+01,-1.1136D+01,-1.1553D+00,-3.5773D+00,-2.1091D+00,&
&                0.0000D+00,-3.3927D-01, 0.0000D+00,&
&               -3.3814D+03,-5.4470D+02,-5.1684D+02,-1.2275D+02,-1.1027D+02,-8.7329D+01,&
&               -2.5587D+01,-2.0586D+01,-1.1647D+01,-1.2092D+00,-3.6578D+00,-2.1500D+00,&
&                0.0000D+00,-3.4272D-01, 0.0000D+00,&
&               -3.4947D+03,-5.6537D+02,-5.3692D+02,-1.2831D+02,-1.1555D+02,-9.2059D+01,&
&               -2.7151D+01,-2.1995D+01,-1.2750D+01,-1.2170D+00,-4.0451D+00,-2.4489D+00,&
&                0.0000D+00,-3.6994D-01, 0.0000D+00,&
&               -3.6093D+03,-5.8572D+02,-5.5674D+02,-1.3326D+02,-1.2020D+02,-9.6182D+01,&
&               -2.8088D+01,-2.2787D+01,-1.9000D+00,-1.2010D+00,-4.1303D+00,-2.4936D+00,&
&                0.0000D+00,-3.7392D-01, 0.0000D+00,&
&               -3.7251D+03,-6.0580D+02,-5.7627D+02,-1.3756D+02,-1.2422D+02,-9.9678D+01,&
&               -2.8371D+01,-2.2939D+01,-1.3196D+01,-1.3411D+00,-3.8927D+00,-2.2663D+00,&
&                0.0000D+00,-3.5258D-01, 0.0000D+00,&
&               -3.8433D+03,-6.2681D+02,-5.9678D+02,-1.4265D+02,-1.2901D+02,-1.0394D+02,&
&               -2.9313D+01,-2.3736D+01,-1.3720D+01,-1.3769D+00,-3.9695D+00,-2.3035D+00,&
&                0.0000D+00,-3.5574D-01, 0.0000D+00,&
&               -3.9634D+03,-6.4820D+02,-6.1764D+02,-1.4781D+02,-1.3389D+02,-1.0830D+02,&
&               -3.0300D+01,-2.4539D+01,-1.4249D+01,-1.4093D+00,-4.0459D+00,-2.3401D+00,&
&                0.0000D+00,-3.5885D-01, 0.0000D+00,&
&               -4.0854D+03,-6.6990D+02,-6.3884D+02,-1.5306D+02,-1.3884D+02,-1.1260D+02,&
&               -3.1220D+01,-2.5349D+01,-1.4783D+01,-1.4385D+00,-4.1219D+00,-2.3761D+00,&
&                0.0000D+00,-3.6191D-01, 0.0000D+00,&
&               -4.2092D+03,-6.9200D+02,-6.6039D+02,-1.5838D+02,-1.4386D+02,-1.1720D+02,&
&               -3.2190D+01,-2.6167D+01,-1.5323D+01,-1.4648D+00,-4.1976D+00,-2.4117D+00,&
&                0.0000D+00,-3.6492D-01, 0.0000D+00,&
&               -4.3354D+03,-7.1510D+02,-6.8298D+02,-1.6454D+02,-1.4972D+02,-1.2250D+02,&
&               -3.3880D+01,-2.7690D+01,-1.6530D+01,-2.1540D+00,-4.6340D+00,-2.7520D+00,&
&               -4.8670D-01,-3.9770D-01, 0.0000D+00,&
&               -4.4636D+03,-7.3870D+02,-7.0600D+02,-1.7080D+02,-1.5570D+02,-1.2790D+02,&
&               -3.5610D+01,-2.9260D+01,-1.7770D+01,-2.8720D+00,-5.0500D+00,-3.0740D+00,&
&               -5.9830D-01,-4.2080D-01, 0.0000D+00,&
&               -4.5937D+03,-7.6260D+02,-7.2940D+02,-1.7730D+02,-1.6190D+02,-1.3350D+02,&
&               -3.7390D+01,-3.0870D+01,-1.9060D+01,-3.6300D+00,-5.4590D+00,-3.3930D+00,&
&               -7.0330D-01,-4.3950D-01, 0.0000D+00,&
&               -4.7256D+03,-7.8690D+02,-7.5310D+02,-1.8390D+02,-1.6810D+02,-1.3920D+02,&
&               -3.9220D+01,-3.2530D+01,-2.0390D+01,-4.4290D+00,-5.8680D+00,-3.7130D+00,&
&               -8.0580D-01,-4.5540D-01, 0.0000D+00,&
&               -4.8595D+03,-8.1170D+02,-7.7730D+02,-1.9060D+02,-1.7460D+02,-1.4510D+02,&
&               -4.1100D+01,-3.4240D+01,-2.1770D+01,-5.2670D+00,-6.2770D+00,-4.0340D+00,&
&               -9.0760D-01,-4.6930D-01, 0.0000D+00,&
&               -4.9953D+03,-8.3680D+02,-8.0190D+02,-1.9740D+02,-1.8110D+02,-1.5100D+02,&
&               -4.3030D+01,-3.5990D+01,-2.3180D+01,-6.1430D+00,-6.6880D+00,-4.3590D+00,&
&               -1.0096D+00,-4.8183D-01, 0.0000D+00,&
&               -5.1329D+03,-8.6240D+02,-8.2690D+02,-2.0450D+02,-1.8780D+02,-1.5720D+02,&
&               -4.4990D+01,-3.7780D+01,-2.4630D+01,-7.0580D+00,-7.1020D+00,-4.6880D+00,&
&               -1.1124D+00,-4.9317D-01, 0.0000D+00,&
&               -5.2721D+03,-8.8800D+02,-8.5200D+02,-2.1130D+02,-1.9430D+02,-1.6310D+02,&
&               -4.6680D+01,-3.9290D+01,-2.5800D+01,-7.6870D+00,-7.2130D+00,-4.7470D+00,&
&               -9.5290D-01,-4.3590D-01, 0.0000D+00,&
&               -5.4136D+03,-9.1440D+02,-8.7780D+02,-2.1850D+02,-2.0120D+02,-1.6940D+02,&
&               -4.8710D+01,-4.1140D+01,-2.7310D+01,-8.6570D+00,-7.6180D+00,-5.0690D+00,&
&               -1.0420D+00,-4.4155D-01, 0.0000D+00,&
&               -5.5573D+03,-9.4150D+02,-9.0440D+02,-2.2630D+02,-2.0870D+02,-1.7630D+02,&
&               -5.1150D+01,-4.3400D+01,-2.9220D+01,-1.0020D+01,-8.3640D+00,-5.7020D+00,&
&               -1.4284D+00,-5.2209D-01, 0.0000D+00,&
&               -5.7030D+03,-9.6910D+02,-9.3150D+02,-2.3430D+02,-2.1640D+02,-1.8340D+02,&
&               -5.3770D+01,-4.5840D+01,-3.1310D+01,-1.1570D+01,-9.2370D+00,-6.4630D+00,&
&               -1.9370D+00,-7.2220D-01,-3.8480D-01,&
&               -5.8507D+03,-9.9720D+02,-9.5900D+02,-2.4250D+02,-2.2430D+02,-1.9070D+02,&
&               -5.6450D+01,-4.8330D+01,-3.3450D+01,-1.3170D+01,-1.0120D+01,-7.2290D+00,&
&               -2.4490D+00,-9.1770D-01,-4.7970D-01,&
&               -6.0002D+03,-1.0257D+03,-9.8690D+02,-2.5080D+02,-2.3230D+02,-1.9810D+02,&
&               -5.9200D+01,-5.0900D+01,-3.5660D+01,-1.4840D+01,-1.1020D+01,-8.0100D+00,&
&               -2.9750D+00,-1.1160D+00,-5.7240D-01,&
&               -6.1517D+03,-1.0546D+03,-1.0153D+03,-2.5930D+02,-2.4050D+02,-2.0570D+02,&
&               -6.2010D+01,-5.3530D+01,-3.7930D+01,-1.6570D+01,-1.1930D+01,-8.8060D+00,&
&               -3.5170D+00,-1.3200D+00,-6.6540D-01,&
&               -6.3052D+03,-1.0840D+03,-1.0441D+03,-2.6800D+02,-2.4880D+02,-2.1350D+02,&
&               -6.4890D+01,-5.6220D+01,-4.0260D+01,-1.8360D+01,-1.2860D+01,-9.6200D+00,&
&               -4.0760D+00,-1.5310D+00,-7.5970D-01,&
&               -6.4605D+03,-1.1138D+03,-1.0734D+03,-2.7680D+02,-2.5730D+02,-2.2140D+02,&
&               -6.7840D+01,-5.8980D+01,-4.2660D+01,-2.0220D+01,-1.3810D+01,-1.0450D+01,&
&               -4.6530D+00,-1.7480D+00,-8.5600D-01/
! The order is 1S,2S,2P,3S,3P,3D,4S,4P,4D,4F,5S,5P,5D,5F,6S,6P,6D,7S
!
      iao= 0
      if(itype == 1) then
!
! Set valence ionization potentials
!
        do iatom= 1,natom
          select case(numatomic(iatom))
! H - He
            case(1:2)
              iao= iao+1
              energy(iao)= row1(numatomic(iatom))
! Li - Ne
            case(3:10)
              iao= iao+1
              energy(iao)= row2(2,numatomic(iatom))
              do i= 1,3
                iao= iao+1
                energy(iao)= row2(3,numatomic(iatom))
              enddo
! Na - Ar
            case(11:18)
              iao= iao+1
              energy(iao)= row3(4,numatomic(iatom))
              do i= 1,3
                iao= iao+1
                energy(iao)= row3(5,numatomic(iatom))
              enddo
! K  - Ca, Ga - Kr
            case(19:20,31:36)
              iao= iao+1
              energy(iao)= row4(7,numatomic(iatom))
              do i= 1,3
                iao= iao+1
                energy(iao)= row4(8,numatomic(iatom))
              enddo
! Sc - Zn
            case(21:30)
              iao= iao+1
              energy(iao)= row4(7,numatomic(iatom))
              do i= 1,3
                iao= iao+1
                energy(iao)= row4(8,numatomic(iatom))
              enddo
              do i= 1,5
                iao= iao+1
                energy(iao)= row4(6,numatomic(iatom))
              enddo
! Rb - Sr, In - Xe
            case(37:38,49:54)
              iao= iao+1
              energy(iao)= row5(10,numatomic(iatom))
              do i= 1,3
                iao= iao+1
                energy(iao)= row5(11,numatomic(iatom))
              enddo
! Y  - Cd
            case(39:48)
              iao= iao+1
              energy(iao)= row5(10,numatomic(iatom))
              do i= 1,3
                iao= iao+1
                energy(iao)= row5(11,numatomic(iatom))
              enddo
              do i= 1,5
                iao= iao+1
                energy(iao)= row5(9,numatomic(iatom))
              enddo
! Cs - Ba
            case(55:56)
              iao= iao+1
              energy(iao)= row6(14,numatomic(iatom))
! La - Yb
            case(57:70)
              iao= iao+1
              energy(iao)= row6(14,numatomic(iatom))
              do i= 1,7
                iao= iao+1
                energy(iao)= row6(9,numatomic(iatom))
              enddo
!ishi
              if(numatomic(iatom)==58) then
                do i= 1,3
                  iao= iao+1
                  energy(iao)=-0.20D0
                enddo
                do i= 1,5
                  iao= iao+1
                  energy(iao)=-0.35D0
                enddo
              endif
! Lu - Hg
            case(71:80)
              iao= iao+1
              energy(iao)= row6(14,numatomic(iatom))
              do i= 1,5
                iao= iao+1
                energy(iao)= row6(13,numatomic(iatom))
              enddo
! Tl - Rn
            case(81:86)
              iao= iao+1
              energy(iao)= row6(14,numatomic(iatom))
              do i= 1,3
                iao= iao+1
                energy(iao)= row6(15,numatomic(iatom))
              enddo
! Bq9 - X
            case(-9:0)
!
            case default
              if(master) write(*,'(" Error! This program supports up to Rn in huckelip.")')
              call iabort
          end select
        enddo
!
! Set core ionization potentials
!
        do iatom= 1,natom
          select case(numatomic(iatom))
! H  - He
            case(1:2)
! Li - Ne
            case(3:10)
!  1s
              if(.not.flagecp.or.(izcore(iatom) < 2)) then
                iao= iao+1
                energy(iao)= row2(1,numatomic(iatom))
              endif
! Na - Ar
            case(11:18)
!  1s
              if(.not.flagecp.or.(izcore(iatom) < 2)) then
                iao= iao+1
                energy(iao)=row3(1,numatomic(iatom))
              endif
!  2s+2p
              if(.not.flagecp.or.(izcore(iatom) < 10)) then
                iao= iao+1
                energy(iao)=row3(2,numatomic(iatom))
                do i= 1,3
                  iao= iao+1
                  energy(iao)=row3(3,numatomic(iatom))
                enddo
              endif
! K  - Kr
            case(19:36)
!  1s
              if(.not.flagecp.or.(izcore(iatom) < 2)) then
                iao= iao+1
                energy(iao)= row4(1,numatomic(iatom))
              endif
!  2s+2p
              if(.not.flagecp.or.(izcore(iatom) < 10)) then
                iao= iao+1
                energy(iao)= row4(2,numatomic(iatom))
                do i= 1,3
                  iao= iao+1
                  energy(iao)= row4(3,numatomic(iatom))
                enddo
              endif
!  3s+3p
              if(.not.flagecp.or.(izcore(iatom) < 18)) then
                iao= iao+1
                energy(iao)= row4(4,numatomic(iatom))
                do i= 1,3
                  iao= iao+1
                  energy(iao)= row4(5,numatomic(iatom))
                enddo
              endif
!  3d
              if(numatomic(iatom) >= 31) then
                if(.not.flagecp.or.(izcore(iatom) < 28)) then
                  do i= 1,5
                    iao= iao+1
                    energy(iao)= row4(6,numatomic(iatom))
                  enddo
                endif
              endif
! Rb - Xe
            case(37:54)
!  1s
              if(.not.flagecp.or.(izcore(iatom) < 2)) then
                iao= iao+1
                energy(iao)= row5(1,numatomic(iatom))
              endif
!  2s+2p
              if(.not.flagecp.or.(izcore(iatom) < 10)) then
                iao= iao+1
                energy(iao)= row5(2,numatomic(iatom))
                do i= 1,3
                  iao= iao+1
                  energy(iao)= row5(3,numatomic(iatom))
                enddo
              endif
!  3s+3p
              if(.not.flagecp.or.(izcore(iatom) < 18)) then
                iao= iao+1
                energy(iao)= row5(4,numatomic(iatom))
                do i= 1,3
                  iao= iao+1
                  energy(iao)= row5(5,numatomic(iatom))
                enddo
              endif
!  3d
              if(.not.flagecp.or.(izcore(iatom) < 28)) then
                do i= 1,5
                  iao= iao+1
                  energy(iao)= row5(6,numatomic(iatom))
                enddo
              endif
!  4s+4p  
              if(.not.flagecp.or.(izcore(iatom) < 36)) then
                iao= iao+1
                energy(iao)= row5(7,numatomic(iatom))
                do i= 1,3
                  iao= iao+1
                  energy(iao)= row5(8,numatomic(iatom))
                enddo
              endif
!  4d
              if(numatomic(iatom) >= 49) then
                if(.not.flagecp.or.(izcore(iatom) < 46)) then
                  do i= 1,5
                    iao= iao+1
                    energy(iao)= row5(9,numatomic(iatom))
                  enddo
                endif
              endif
! Cs - Rn
            case(55:86)
!  1s
              if(.not.flagecp.or.(izcore(iatom) < 2)) then
                iao= iao+1
                energy(iao)= row6(1,numatomic(iatom))
              endif
!  2s+2p  
              if(.not.flagecp.or.(izcore(iatom) < 10)) then
                iao= iao+1
                energy(iao)= row6(2,numatomic(iatom))
                do i= 1,3
                  iao= iao+1
                  energy(iao)= row6(3,numatomic(iatom))
                enddo
              endif
!  3s+3p  
              if(.not.flagecp.or.(izcore(iatom) < 18)) then
                iao= iao+1
                energy(iao)= row6(4,numatomic(iatom))
                do i= 1,3
                  iao= iao+1
                  energy(iao)= row6(5,numatomic(iatom))
                enddo
              endif
!  3d
              if(.not.flagecp.or.(izcore(iatom) < 28)) then
                do i= 1,5
                  iao= iao+1
                  energy(iao)= row6(6,numatomic(iatom))
                enddo
              endif
!  4s+4p  
              if(.not.flagecp.or.(izcore(iatom) < 36)) then
                iao= iao+1
                energy(iao)= row6(7,numatomic(iatom))
                do i= 1,3
                  iao= iao+1
                  energy(iao)= row6(8,numatomic(iatom))
                enddo
              endif
!  4d
              if(.not.flagecp.or.(izcore(iatom) < 46)) then
                do i= 1,5
                  iao= iao+1
                  energy(iao)= row6(9,numatomic(iatom))
                enddo
              endif
!  4f
              if(numatomic(iatom) >= 71) then
                if(.not.flagecp.or.(izcore(iatom) < 60)) then
                  do i= 1,7
                    iao= iao+1
                    energy(iao)= row6(10,numatomic(iatom))
                  enddo
                endif
              endif
!  5s+5p  
              if(.not.flagecp.or.((izcore(iatom) /= 54).and.(izcore(iatom) < 68))) then
                iao= iao+1
                energy(iao)= row6(11,numatomic(iatom))
                do i= 1,3
                  iao= iao+1
                  energy(iao)= row6(12,numatomic(iatom))
                enddo
              endif
!  5d
              if(numatomic(iatom) >= 81) then
                if(.not.flagecp.or.(izcore(iatom) < 78)) then
                  do i= 1,5
                    iao= iao+1
                    energy(iao)= row6(13,numatomic(iatom))
                  enddo
                endif
              endif
! Bq9 - X
            case(-9:0)
            case default
              if(master) write(*,'(" Error! This program supports up to Rn in huckelip.")')
              call iabort
          end select
        enddo
!
      elseif(itype == 2) then
        do iatom= 1,natom
          select case(numatomic(iatom))
! H  - He
            case(1:2)
! Li - Ne
            case(3:10)
!  1s
              if(izcore(iatom) >= 2) then
                iao= iao+1
                energy(iao)= row2(1,numatomic(iatom))
              endif
! Na - Ar
            case(11:18)
!  1s
              if(izcore(iatom) >= 2) then
                iao= iao+1
                energy(iao)=row3(1,numatomic(iatom))
              endif
!  2s+2p
              if(izcore(iatom) >= 10) then
                iao= iao+1
                energy(iao)=row3(2,numatomic(iatom))
                do i= 1,3
                  iao= iao+1
                  energy(iao)=row3(3,numatomic(iatom))
                enddo
              endif
! K  - Kr
            case(19:36)
!  1s
              if(izcore(iatom) >= 2) then
                iao= iao+1
                energy(iao)= row4(1,numatomic(iatom))
              endif
!  2s+2p
              if(izcore(iatom) >= 10) then
                iao= iao+1
                energy(iao)= row4(2,numatomic(iatom))
                do i= 1,3
                  iao= iao+1
                  energy(iao)= row4(3,numatomic(iatom))
                enddo
              endif
!  3s+3p
              if(izcore(iatom) >= 18) then
                iao= iao+1
                energy(iao)= row4(4,numatomic(iatom))
                do i= 1,3
                  iao= iao+1
                  energy(iao)= row4(5,numatomic(iatom))
                enddo
              endif
!  3d
              if(numatomic(iatom) >= 31) then
                if(izcore(iatom) >= 28) then
                  do i= 1,5
                    iao= iao+1
                    energy(iao)= row4(6,numatomic(iatom))
                  enddo
                endif
              endif
! Rb - Xe
            case(37:54)
!  1s
              if(izcore(iatom) >= 2) then
                iao= iao+1
                energy(iao)= row5(1,numatomic(iatom))
              endif
!  2s+2p
              if(izcore(iatom) >= 10) then
                iao= iao+1
                energy(iao)= row5(2,numatomic(iatom))
                do i= 1,3
                  iao= iao+1
                  energy(iao)= row5(3,numatomic(iatom))
                enddo
              endif
!  3s+3p
              if(izcore(iatom) >= 18) then
                iao= iao+1
                energy(iao)= row5(4,numatomic(iatom))
                do i= 1,3
                  iao= iao+1
                  energy(iao)= row5(5,numatomic(iatom))
                enddo
              endif
!  3d
              if(izcore(iatom) >= 28) then
                do i= 1,5
                  iao= iao+1
                  energy(iao)= row5(6,numatomic(iatom))
                enddo
              endif
!  4s+4p  
              if(izcore(iatom) >= 36) then
                iao= iao+1
                energy(iao)= row5(7,numatomic(iatom))
                do i= 1,3
                  iao= iao+1
                  energy(iao)= row5(8,numatomic(iatom))
                enddo
              endif
!  4d
              if(numatomic(iatom) >= 49) then
                if(izcore(iatom) >= 46) then
                  do i= 1,5
                    iao= iao+1
                    energy(iao)= row5(9,numatomic(iatom))
                  enddo
                endif
              endif
! Cs - Rn
            case(55:86)
!  1s
              if(izcore(iatom) >= 2) then
                iao= iao+1
                energy(iao)= row6(1,numatomic(iatom))
              endif
!  2s+2p
              if(izcore(iatom) >= 10) then
                iao= iao+1
                energy(iao)= row6(2,numatomic(iatom))
                do i= 1,3
                  iao= iao+1
                  energy(iao)= row6(3,numatomic(iatom))
                enddo
              endif
!  3s+3p  
              if(izcore(iatom) >= 18) then
                iao= iao+1
                energy(iao)= row6(4,numatomic(iatom))
                do i= 1,3
                  iao= iao+1
                  energy(iao)= row6(5,numatomic(iatom))
                enddo
              endif
!  3d
              if(izcore(iatom) >= 28) then
                do i= 1,5
                  iao= iao+1
                  energy(iao)= row6(6,numatomic(iatom))
                enddo
              endif
!  4s+4p  
              if(izcore(iatom) >= 36) then
                iao= iao+1
                energy(iao)= row6(7,numatomic(iatom))
                do i= 1,3
                  iao= iao+1
                  energy(iao)= row6(8,numatomic(iatom))
                enddo
              endif
!  4d
              if(izcore(iatom) >= 46) then
                do i= 1,5
                  iao= iao+1
                  energy(iao)= row6(9,numatomic(iatom))
                enddo
              endif
!  4f
              if(numatomic(iatom) >= 71) then
                if(izcore(iatom) >= 60) then
                  do i= 1,7
                    iao= iao+1
                    energy(iao)= row6(10,numatomic(iatom))
                  enddo
                endif
              endif
!  5s+5p  
              if((izcore(iatom) == 54).or.(izcore(iatom) >= 68)) then
                iao= iao+1
                energy(iao)= row6(11,numatomic(iatom))
                do i= 1,3
                  iao= iao+1
                  energy(iao)= row6(12,numatomic(iatom))
                enddo
              endif
!  5d
              if(numatomic(iatom) >= 81) then
                if(izcore(iatom) >= 78) then
                  do i= 1,5
                    iao= iao+1
                    energy(iao)= row6(13,numatomic(iatom))
                  enddo
                endif
              endif
! Bq9 - X
            case(-9:0)
            case default
              if(master) write(*,'(" Error! This program supports up to Rn in huckelip.")')
              call iabort
          end select
        enddo
      endif
!
      return
end


!---------------------------------------
  subroutine formhuckel(huckel,energy)
!---------------------------------------
!
! Form extended Huckel matrix
!
! In   : energy (ionization potential)
! Inout: huckel[in] (overlap integral of guess basis set)
!              [out](extended Huckel Hamiltonian)
!
      use modguess, only : nao_g, nao_v
      implicit none
      integer :: i, j
      real(8),parameter :: factor=0.875D+00  !(=1.75/2.0)
      real(8),parameter :: fdown=0.05D+00
      real(8),intent(in) :: energy(nao_g)
      real(8),intent(inout) :: huckel(nao_g,nao_g)
!
! Generate Huckel matrix from overlap integrals and ionization potentials
!
!$OMP parallel do schedule(static,1) private(j)
      do i= 1,nao_g
        do j= 1,i-1
          huckel(j,i)=factor*huckel(j,i)*(energy(i)+energy(j))
        enddo
        huckel(i,i)= energy(i)
      enddo
!$OMP end parallel do
!
! Scale down core-core and core-valence elements
!
!$OMP parallel do schedule(static,1) private(j)
      do i= nao_v+1,nao_g
        do j= 1,i-1
          huckel(j,i)= fdown*huckel(j,i)
        enddo
      enddo
!$OMP end parallel do
      return
end


!-------------------------------------------
  subroutine formhuckelcore(huckel,energy)
!-------------------------------------------
!
! Form extended Huckel matrix for only core orbitals
!
! In   : energy (ionization potential)
! Inout: huckel[in] (overlap integral of guess basis set)
!              [out](extended Huckel Hamiltonian)
!
      use modguess, only : nao_gcore
      implicit none
      integer :: i, j
      real(8),parameter :: factor=0.875D+00  !(=1.75/2.0)
      real(8),parameter :: fdown=0.05D+00
      real(8),intent(in) :: energy(nao_gcore)
      real(8),intent(inout) :: huckel(nao_gcore,nao_gcore)
!
! Generate Huckel matrix from overlap integrals and ionization potentials
! Scale down core-core and core-valence elements
!
!$OMP parallel do schedule(static,1) private(j)
      do i= 1,nao_gcore
        do j= 1,i-1
          huckel(j,i)= fdown*factor*huckel(j,i)*(energy(i)+energy(j))
        enddo
        huckel(i,i)= energy(i)
      enddo
!$OMP end parallel do
      return
end


!--------------------------------
  subroutine calcover1(overlap)
!--------------------------------
!
! Driver of overlap integral calculation
! (guess basis)x(guess basis)
!
! Out : overlap (overlap integral of guess basis set)
!
      use modguess, only : nshell_g, nao_g
      implicit none
      integer :: ish, jsh
      real(8),intent(out) :: overlap(nao_g*nao_g)
!
!$OMP parallel do private(jsh)
      do ish= nshell_g,1,-1
        do jsh= 1,ish
          call intover1(overlap,ish,jsh)
        enddo
      enddo
!$OMP end parallel do
      return
end


!------------------------------------
  subroutine calcover1core(overlap)
!------------------------------------
!
! Driver of overlap integral calculation for only core orbitals
! (guess basis)x(guess basis)
!
! Out : overlap (overlap integral of guess basis set)
!
      use modguess, only : nshell_gcore, nao_gcore
      implicit none
      integer :: ish, jsh
      real(8),intent(out) :: overlap(nao_gcore*nao_gcore)
!
!$OMP parallel do private(jsh)
      do ish= nshell_gcore,1,-1
        do jsh= 1,ish
          call intover1core(overlap,ish,jsh)
        enddo
      enddo
!$OMP end parallel do
      return
end


!-----------------------------------------------------------
  subroutine calcover2(overlap,work,nproc,myrank,mpi_comm)
!-----------------------------------------------------------
!
! Driver of overlap integral calculation
! (input basis)x(guess basis)
!
! Out : overlap (overlap integral of guess and SCF basis sets)
!       work    (work array)
!
      use modbasis, only : nshell, nao
      use modguess, only : nshell_g, nao_g
      implicit none
      integer,intent(in) :: nproc, myrank, mpi_comm
      integer :: ish, jsh
      real(8),parameter :: zero=0.0D+00
      real(8),intent(out) :: overlap(nao*nao_g), work(nao*nao_g)
!
      work(:)= zero
!$OMP parallel
      do ish= nshell_g-myrank,1,-nproc
!$OMP do
        do jsh= 1,nshell
          call intover2(work,ish,jsh)
        enddo
!$OMP enddo
      enddo
!$OMP end parallel
!
      call para_allreducer(work,overlap,nao*nao_g,mpi_comm)
      return
end


!---------------------------------------------------------------
  subroutine calcover2core(overlap,work,nproc,myrank,mpi_comm)
!---------------------------------------------------------------
!
! Driver of overlap integral calculation for only core orbitals
! (input basis)x(guess basis)
!
! Out : overlap (overlap integral of guess and SCF basis sets)
!       work    (work array)
!
      use modguess, only : nshell_g, nao_g, nshell_gcore, nao_gcore
      implicit none
      integer,intent(in) :: nproc, myrank, mpi_comm
      integer :: ish, jsh
      real(8),parameter :: zero=0.0D+00
      real(8),intent(out) :: overlap(nao_gcore*nao_g), work(nao_gcore*nao_g)
!
      work(:)= zero
!$OMP parallel
      do ish= nshell_g-myrank,1,-nproc
!$OMP do
        do jsh= 1,nshell_gcore
          call intover2core(work,ish,jsh)
        enddo
!$OMP enddo
      enddo
!$OMP end parallel
!
      call para_allreducer(work,overlap,nao_gcore*nao_g,mpi_comm)
      return
end


!---------------------------------------
  subroutine intover1(overlap,ish,jsh)
!---------------------------------------
!
! Overlap integral calculation
! (guess basis)x(guess basis)
!
! In  : ish, jsh (shell index)
! Out : overlap (overlap integral of guess basis set)
!
      use modparam, only : mxprsh
      use modthresh, only : threshex
      use modguess, only : locatom_g, locprim_g, locbf_g, mprim_g, mbf_g, mtype_g, &
&                          ex_g, coeff_g, nao_g, coord_g
      implicit none
      integer,intent(in) :: ish, jsh
      integer :: iatom, jatom, iloc, jloc, ilocbf, jlocbf, iprim, jprim
      integer :: nbfij(2), nprimij(2), nangij(2), ii, jj, maxj
      real(8),intent(out) :: overlap(nao_g,nao_g)
      real(8) :: sint(28,28), exij(mxprsh,2), coij(mxprsh,2), coordij(3,2)
      logical :: iandj
!
! Set parameters
!
      iandj =(ish == jsh)
      nangij(1)= mtype_g(ish)
      nangij(2)= mtype_g(jsh)
      nprimij(1)= mprim_g(ish)
      nprimij(2)= mprim_g(jsh)
      nbfij(1)  = mbf_g(ish)
      nbfij(2)  = mbf_g(jsh)
      iatom = locatom_g(ish)
      iloc  = locprim_g(ish)
      ilocbf= locbf_g(ish)
      jatom = locatom_g(jsh)
      jloc  = locprim_g(jsh)
      jlocbf= locbf_g(jsh)
      do ii= 1,3
        coordij(ii,1)= coord_g(ii,iatom)
        coordij(ii,2)= coord_g(ii,jatom)
      enddo
      do iprim= 1,nprimij(1)
        exij(iprim,1)= ex_g(iloc+iprim)
        coij(iprim,1)= coeff_g(iloc+iprim)
      enddo
      do jprim= 1,nprimij(2)
        exij(jprim,2)= ex_g(jloc+jprim)
        coij(jprim,2)= coeff_g(jloc+jprim)
      enddo
!
! Calculate overlap integrals
!
      call ints(sint,exij,coij,coordij,nprimij,nangij,nbfij,mxprsh,threshex)
!
      maxj= nbfij(2)
      do ii= 1,nbfij(1)
        if(iandj) maxj= ii
        do jj= 1,maxj
          overlap(jlocbf+jj,ilocbf+ii)= sint(jj,ii)
        enddo
      enddo
      return
end


!---------------------------------------
  subroutine intover2(overlap,ish,jsh)
!---------------------------------------
!
! Overlap integral calculation
! (input basis)x(guess basis)
!
! In  : ish, jsh (shell index)
! Out : overlap (overlap integral of guess and SCF basis sets)
!
      use modparam, only : mxprsh
      use modthresh, only : threshex
      use modmolecule, only : coord
      use modbasis, only : locatom, locprim, locbf, mprim, mbf, mtype, ex, coeff, nao
      use modguess, only : locatom_g, locprim_g, locbf_g, mprim_g, mbf_g, mtype_g, &
&                          ex_g, coeff_g, nao_g, coord_g
      implicit none
      integer,intent(in) :: ish, jsh
      integer :: iatom, jatom, iloc, jloc, ilocbf, jlocbf, iprim, jprim
      integer :: nbfij(2), nprimij(2), nangij(2), ii, jj
      real(8),parameter :: zero=0.0D+0, one=1.0D+0
      real(8),intent(out) :: overlap(nao,nao_g)
      real(8) :: sint(28,28), exij(mxprsh,2), coij(mxprsh,2), coordij(3,2)
!
! Set parameters
!
      nangij(1)= mtype_g(ish)
      nangij(2)= mtype(jsh)
      nprimij(1)= mprim_g(ish)
      nprimij(2)= mprim(jsh)
      nbfij(1)  = mbf_g(ish)
      nbfij(2)  = mbf(jsh)
      iatom = locatom_g(ish)
      iloc  = locprim_g(ish)
      ilocbf= locbf_g(ish)
      jatom = locatom(jsh)
      jloc  = locprim(jsh)
      jlocbf= locbf(jsh)
      do ii= 1,3
        coordij(ii,1)= coord_g(ii,iatom)
        coordij(ii,2)= coord(ii,jatom)
      enddo
      do iprim= 1,nprimij(1)
        exij(iprim,1)= ex_g(iloc+iprim)
        coij(iprim,1)= coeff_g(iloc+iprim)
      enddo
      do jprim= 1,nprimij(2)
        exij(jprim,2)= ex(jloc+jprim)
        coij(jprim,2)= coeff(jloc+jprim)
      enddo
!
! Calculate overlap integrals
!
      call ints(sint,exij,coij,coordij,nprimij,nangij,nbfij,mxprsh,threshex)
!
      do ii= 1,nbfij(1)
        do jj= 1,nbfij(2)
          overlap(jlocbf+jj,ilocbf+ii)= sint(jj,ii)
        enddo
      enddo
      return
end


!-------------------------------------------
  subroutine intover1core(overlap,ish,jsh)
!-------------------------------------------
!
! Overlap integral calculation
! (guess basis)x(guess basis)
!
! In  : ish, jsh (shell index)
! Out : overlap (overlap integral of guess basis set)
!
      use modparam, only : mxprsh
      use modthresh, only : threshex
      use modguess, only : locatom_gcore, locprim_gcore, locbf_gcore, mprim_gcore, mbf_gcore, &
&                          mtype_gcore, ex_gcore, coeff_gcore, nao_gcore, coord_g
      implicit none
      integer,intent(in) :: ish, jsh
      integer :: iatom, jatom, iloc, jloc, ilocbf, jlocbf, iprim, jprim
      integer :: nbfij(2), nprimij(2), nangij(2), ii, jj, maxj
      real(8),intent(out) :: overlap(nao_gcore,nao_gcore)
      real(8) :: sint(28,28), exij(mxprsh,2), coij(mxprsh,2), coordij(3,2)
      logical :: iandj
!
! Set parameters
!
      iandj =(ish == jsh)
      nangij(1)= mtype_gcore(ish)
      nangij(2)= mtype_gcore(jsh)
      nprimij(1)= mprim_gcore(ish)
      nprimij(2)= mprim_gcore(jsh)
      nbfij(1)  = mbf_gcore(ish)
      nbfij(2)  = mbf_gcore(jsh)
      iatom = locatom_gcore(ish)
      iloc  = locprim_gcore(ish)
      ilocbf= locbf_gcore(ish)
      jatom = locatom_gcore(jsh)
      jloc  = locprim_gcore(jsh)
      jlocbf= locbf_gcore(jsh)
      do ii= 1,3
        coordij(ii,1)= coord_g(ii,iatom)
        coordij(ii,2)= coord_g(ii,jatom)
      enddo
      do iprim= 1,nprimij(1)
        exij(iprim,1)= ex_gcore(iloc+iprim)
        coij(iprim,1)= coeff_gcore(iloc+iprim)
      enddo
      do jprim= 1,nprimij(2)
        exij(jprim,2)= ex_gcore(jloc+jprim)
        coij(jprim,2)= coeff_gcore(jloc+jprim)
      enddo
!
! Calculate overlap integrals
!
      call ints(sint,exij,coij,coordij,nprimij,nangij,nbfij,mxprsh,threshex)
!
      maxj= nbfij(2)
      do ii= 1,nbfij(1)
        if(iandj) maxj= ii
        do jj= 1,maxj
          overlap(jlocbf+jj,ilocbf+ii)= sint(jj,ii)
        enddo
      enddo
      return
end


!-------------------------------------------
  subroutine intover2core(overlap,ish,jsh)
!-------------------------------------------
!
! Overlap integral calculation for only core orbitals
! (core-guess basis)x(guess basis)
!
! In  : ish, jsh (shell index)
! Out : overlap (overlap integral of core-guess and guess basis sets)
!
      use modparam, only : mxprsh
      use modthresh, only : threshex
      use modguess, only : locatom_g, locprim_g, locbf_g, mprim_g, mbf_g, mtype_g, &
&                          ex_g, coeff_g, nao_g, coord_g, &
&                          locatom_gcore, locprim_gcore, locbf_gcore, mprim_gcore, &
&                          mbf_gcore, mtype_gcore, ex_gcore, coeff_gcore, nao_gcore
      implicit none
      integer,intent(in) :: ish, jsh
      integer :: iatom, jatom, iloc, jloc, ilocbf, jlocbf, iprim, jprim
      integer :: nbfij(2), nprimij(2), nangij(2), ii, jj
      real(8),parameter :: zero=0.0D+0, one=1.0D+0
      real(8),intent(out) :: overlap(nao_gcore,nao_g)
      real(8) :: sint(28,28), exij(mxprsh,2), coij(mxprsh,2), coordij(3,2)
!
! Set parameters
!
      nangij(1)= mtype_g(ish)
      nangij(2)= mtype_gcore(jsh)
      nprimij(1)= mprim_g(ish)
      nprimij(2)= mprim_gcore(jsh)
      nbfij(1)  = mbf_g(ish)
      nbfij(2)  = mbf_gcore(jsh)
      iatom = locatom_g(ish)
      iloc  = locprim_g(ish)
      ilocbf= locbf_g(ish)
      jatom = locatom_gcore(jsh)
      jloc  = locprim_gcore(jsh)
      jlocbf= locbf_gcore(jsh)
      do ii= 1,3
        coordij(ii,1)= coord_g(ii,iatom)
        coordij(ii,2)= coord_g(ii,jatom)
      enddo
      do iprim= 1,nprimij(1)
        exij(iprim,1)= ex_g(iloc+iprim)
        coij(iprim,1)= coeff_g(iloc+iprim)
      enddo
      do jprim= 1,nprimij(2)
        exij(jprim,2)= ex_gcore(jloc+jprim)
        coij(jprim,2)= coeff_gcore(jloc+jprim)
      enddo
!
! Calculate overlap integrals
!
      call ints(sint,exij,coij,coordij,nprimij,nangij,nbfij,mxprsh,threshex)
!
      do ii= 1,nbfij(1)
        do jj= 1,nbfij(2)
          overlap(jlocbf+jj,ilocbf+ii)= sint(jj,ii)
        enddo
      enddo
      return
end


!---------------------------------------------------------
  subroutine updatemo(cmo,overinv,nproc,myrank,mpi_comm)
!---------------------------------------------------------
!
! Read and project MOs
!
! Inout : overinv (overlap integral inverse matrix)
!         cmo     (initial guess orbitals)
!
      use modguess, only : nao_g, nmo_g
      use modbasis, only : nao
      implicit none
      integer,intent(in) :: nproc, myrank, mpi_comm
      integer :: ndim
      real(8),intent(inout) :: cmo(nao*nao), overinv(nao*nao)
      real(8),allocatable :: overlap(:,:), work1(:), work2(:), eigen(:)
!
!     ndim= min(nmo,neleca+5)
      ndim= nmo_g
!
! Set arrays
!
      call memset(2*nao*nao_g+ndim*ndim+nao*ndim+ndim)
      allocate(overlap(nao*nao_g,2),work1(ndim*ndim),work2(nao*ndim),eigen(ndim))
!
! Calculate overlap integrals between previous and present bases
!
      call calcover2(overlap(1,1),overlap(1,2),nproc,myrank,mpi_comm)
!
! Project orbitals from previous basis to current basis
!
      call projectmo2(cmo,overinv,overlap,work1,work2,eigen,ndim,nproc,myrank,mpi_comm)
!
! Unset arrays
!
      deallocate(overlap,work1,work2,eigen)
      call memunset(2*nao*nao_g+ndim*ndim+nao*ndim+ndim)
!
      return
end


!------------------------------------------------------------------------------------------
  subroutine projectmo2(cmo,overinv,overlap,work1,work2,eigen,ndim,nproc,myrank,mpi_comm)
!------------------------------------------------------------------------------------------
!
! Project orbitals from previous basis to current basis
!    C1= S11^-1 * S12 * C2 [C2t * S12t * S11^-1 * S12 * C2]^-1/2
!
! In this routine, nao >= ndim is assumed.
!
! Inout : cmo     (previous and updated orbitals)
!         overinv (overlap integral inverse matrix of SCF basis set)
!         overlap (overlap integral of guess and SCF basis sets)
!
      use modguess, only : nao_g
      use modbasis, only : nao
      implicit none
      integer,intent(in) :: ndim, nproc, myrank, mpi_comm
      integer :: i, j
      real(8),parameter :: zero=0.0D+00, one=1.0D+00
      real(8),intent(inout) :: cmo(nao,nao), overinv(nao,nao), overlap(nao,nao_g)
      real(8),intent(out) :: work1(ndim,ndim), work2(ndim,nao), eigen(ndim)
      real(8) :: eigeninv
!
      if(nao < ndim) then
        write(*,'(" Error! Nao is less than ndim in projectmo2.")')
        call iabort
      endif
!
! Calculate S12*C2
!
      call dgemm('N','N',nao,ndim,nao_g,one,overlap,nao,cmo,nao_g,zero,work2,nao)
!
! Calculate S11^-1*S12*C2
!
      call dsymm('L','U',nao,ndim,one,overinv,nao,work2,nao,zero,overlap,nao)
!
! Calculate C2t*S12t*S11^-1*S12*C2
!
      call dgemm('T','N',ndim,ndim,nao,one,work2,nao,overlap,nao,zero,work1,ndim)
!
! Calculate (C2t*S12t*S11^-1*S12*C2)^-1/2
!
      call diag('V','U',ndim,work1,ndim,eigen,nproc,myrank,mpi_comm)
!$OMP parallel do private(eigeninv)
      do i= 1,ndim
        eigeninv= one/sqrt(eigen(i))
        do j= 1,ndim
          work2(j,i)= work1(j,i)*eigeninv
        enddo
      enddo
!$OMP end parallel do
      call dgemm('N','T',ndim,ndim,ndim,one,work1,ndim,work2,ndim,zero,overinv,ndim)
!
! Calculate C1
!
      call dgemm('N','N',nao,ndim,ndim,one,overlap,nao,overinv,ndim,zero,cmo,nao)
      return
end


!------------------------------------------------------------------------------
  subroutine checkguess(cmoa,cmob,overinv,nproc2,myrank2,mpi_comm1,mpi_comm2)
!------------------------------------------------------------------------------
!
! Read and project MOs
!
! In    : overinv (Overlap integral inverse matrix)
! Out   : cmoa    (Alpha initial guess orbitals)
!         cmob    (Beta initial guess orbitals)
!
      use modparallel, only : master
      use modjob, only : scftype
      use modguess, only : nao_g, nmo_g, nao_gcore
      use modbasis, only : nao
      use modmolecule, only : neleca, nelecb, charge
      implicit none
      integer,intent(in) :: nproc2, myrank2, mpi_comm1, mpi_comm2
      integer :: neleca_g, nelecb_g, nelect, nelect_g
      integer :: ncore, ndim, max
      real(8),intent(in) :: overinv(nao*nao)
      real(8),intent(inout) :: cmoa(nao*nao), cmob(nao*nao)
      real(8),allocatable :: cmoa_g(:,:), cmob_g(:,:)
      real(8),allocatable :: overlap(:), work1(:), work2(:), work3(:), eigen(:)
      real(8),allocatable :: coremo(:,:)
      real(8) :: charge_g
      character(len=16) :: scftype_g
      logical :: flagecp_g
!
! Read checkpoint information
!
      call readcheckinfo(scftype_g,charge_g,flagecp_g,neleca_g,nelecb_g,mpi_comm1)
!
      if(scftype == 'RHF') then
        call memset(nao_g*nao_g)
        allocate(cmoa_g(nao_g,nao_g))
      elseif(scftype == 'UHF') then
        call memset(nao_g*nao_g*2)
        allocate(cmoa_g(nao_g,nao_g),cmob_g(nao_g,nao_g))
      endif
!
! Read guess basis functions and MOs from checkpoint file
!
      call readcheckguess(cmoa_g,cmob_g,scftype_g,mpi_comm1)
!
! Orthonormalize guess basis functions
!
      call bsnrmlz_g
!
! Adjust the number of projected MOs
!
      nelect  =neleca  +nelecb  -nint(charge)
      nelect_g=neleca_g+nelecb_g-nint(charge_g)
      ncore=(nelect_g-nelect)/2
!
      if(ncore < 0) then
        if(master) write(*,'(" Error! The number of electrons in checkpoint file is too small.")')
        call iabort
      elseif((ncore /= 0).and.flagecp_g) then
        if(master) then
          write(*,'(" Error! The ECP functions in checkpoint and input files are different.")')
        endif
        call iabort
      endif
!
      if(ncore > 0) then
        call setbasis_g(2)
        call memset(nao_gcore*(nao_gcore+nao_g*3+1))
        allocate(coremo(nao_gcore,nao_gcore),work1(nao_g*nao_gcore),work2(nao_g*nao_gcore), &
&                work3(nao_g*nao_gcore),eigen(nao_gcore))
        call calccoremo(cmoa_g,cmob_g,coremo,work1,work2,work3,eigen,scftype_g, &
&                       nproc2,myrank2,mpi_comm2)
        call memunset(nao_gcore*(nao_gcore+nao_g*3+1))
        deallocate(coremo,work1,work2,work3,eigen)
        nmo_g= nmo_g-ncore
      endif
!
! Set arrays
!
      ndim= max(nao_g,nao)
      call memset(nao_g*(nao*2+ndim+nao_g+1))
      allocate(overlap(nao*nao_g),work1(ndim*nao_g),work2(nao*nao_g),work3(nao_g*nao_g), &
&              eigen(nao_g))
!
! Calculate overlap integrals between previous and present bases
!
      call calcover2(overlap,work2,nproc2,myrank2,mpi_comm2)
!
! Project orbitals from previous basis to current basis
!
      call projectmo3(cmoa,cmoa_g,overinv,overlap,work1,work2,work3,eigen, &
&                     ndim,nproc2,myrank2,mpi_comm2)
      if(scftype == 'UHF') then
        if(scftype_g == 'RHF') then
          call dcopy(nao*nao,cmoa,1,cmob,1)
        elseif(scftype_g == 'UHF') then
          call projectmo3(cmob,cmob_g,overinv,overlap,work1,work2,work3,eigen, &
&                         ndim,nproc2,myrank2,mpi_comm2)
        endif
      endif
!
! Unset arrays
!
      call memunset(nao_g*(nao*2+ndim+nao_g+1))
      deallocate(overlap,work1,work2,work3, &
&                eigen)
!
      if(scftype == 'RHF') then
        call memunset(nao_g*nao_g)
        deallocate(cmoa_g)
      elseif(scftype == 'UHF') then
        call memunset(nao_g*nao_g*2)
        deallocate(cmoa_g,cmob_g)
      endif
!
      return
end


!-----------------------------------------------------------------------------
  subroutine projectmo3(cmo,cmo_g,overinv,overlap,work1,work2,work3,eigen, &
&                       ndim,nproc,myrank,mpi_comm)
!-----------------------------------------------------------------------------
!
! Project orbitals from previous basis to current basis
!    C1= S11^-1 * S12 * C2 [C2t * S12t * S11^-1 * S12 * C2]^-1/2
!
! In    : cmo_g   (Previous orbitals)
!         overinv (Overlap integral inverse matrix of SCF basis set)
!         overlap (Overlap integral of guess and SCF basis sets)
! Out   : cmo     (Projected orbitals)
!
      use modguess, only : nao_g, nmo_g
      use modbasis, only : nao
      implicit none
      integer,intent(in) :: ndim, nproc, myrank, mpi_comm
      integer :: ii, jj, maxmo
      real(8),parameter :: zero=0.0D+00, one=1.0D+00, pm5=1.0D-05
      real(8),intent(in) :: overinv(nao,nao), overlap(nao,nao_g)
      real(8),intent(out) :: cmo(nao,nao), work1(ndim,nao_g), work2(nao,nao_g)
      real(8),intent(out) :: work3(nao_g,nao_g), eigen(nao_g)
      real(8),intent(inout) :: cmo_g(nao_g,nao_g)
      real(8) :: eigeninv
!
! Calculate S12*C2
!
      call dgemm('N','N',nao,nmo_g,nao_g,one,overlap,nao,cmo_g,nao_g,zero,work1,ndim)
!
! Calculate S11^-1*S12*C2
!
      call dsymm('L','U',nao,nmo_g,one,overinv,nao,work1,ndim,zero,work2,nao)
!
! Calculate C2t*S12t*S11^-1*S12*C2
!
      call dgemm('T','N',nmo_g,nmo_g,nao,one,work1,ndim,work2,nao,zero,cmo_g,nao_g)
!
! Calculate (C2t*S12t*S11^-1*S12*C2)^-1/2
!
      call diag('V','U',nmo_g,cmo_g,nao_g,eigen,nproc,myrank,mpi_comm)
!
!$OMP parallel do private(eigeninv)
      do ii= 1,nmo_g
        if(eigen(ii) > pm5) then
          eigeninv= one/sqrt(eigen(ii))
        else
          eigeninv= zero
        endif
        do jj= 1,nmo_g
          work1(jj,ii)= cmo_g(jj,ii)*eigeninv
        enddo
      enddo
!$OMP end parallel do
!
      maxmo= nmo_g
      if(nmo_g > nao) maxmo= nao
      call dgemm('N','T',nmo_g,maxmo,nmo_g,one,cmo_g,nao_g,work1,ndim,zero,work3,nao_g)
!
! Calculate C1
!
      call dgemm('N','N',nao,maxmo,nmo_g,one,work2,nao,work3,nao_g,zero,cmo,nao)
!
      return
end

!------------------------------------------------------------------------------------
  subroutine calccoremo(cmoa_g,cmob_g,coremo,overlap,work2,work3,eigen,scftype_g, &
&                       nproc,myrank,mpi_comm)
!------------------------------------------------------------------------------------
!
! Calculate and remove core orbitals corresponding ECP
!
      use modguess, only : nao_g, nmo_g, nao_gcore
      use modjob, only : scftype
      implicit none
      integer,intent(in) :: nproc, myrank, mpi_comm
      integer :: ii, jj, idamax, icount
      real(8),parameter :: zero=0.0D+00, one=1.0D+00
      real(8),intent(inout) :: cmoa_g(nao_g,nao_g), cmob_g(nao_g,nao_g)
      real(8),intent(out) :: coremo(nao_gcore,nao_gcore), overlap(nao_gcore,nao_g)
      real(8),intent(out) :: work2(nao_gcore,nao_g), work3(nao_g,nao_gcore), eigen(nao_gcore)
      character(len=16),intent(in) :: scftype_g
      logical :: skipmo(nao_g)
!
!
! Calculate Extended Huckel method
!
      call calchuckelgcore(coremo,eigen,nproc,myrank,mpi_comm)
!
! Calculate overlap integrals between core-guess basis and guess bases
!
      call calcover2core(overlap,work2,nproc,myrank,mpi_comm)
!
! Calculate Cguess-t*Overlap*Ccore
!
      call dgemm('N','N',nao_gcore,nmo_g,nao_g,one,overlap,nao_gcore,cmoa_g,nao_g,zero, &
&                work2,nao_gcore)
      call dgemm('T','N',nmo_g,nao_gcore,nao_gcore,one,work2,nao_gcore,coremo,nao_gcore,zero, &
&                work3,nao_g)
!
! Check correspoing orbitals
!
      skipmo(:)=.false.
      do ii= 1,nao_gcore
        skipmo(idamax(nmo_g,work3(1,ii),1))=.true.
      enddo
!
! Skip core orbitals
!
      icount= 0
      do ii= 1,nmo_g
        if(skipmo(ii)) cycle
        icount= icount+1
        do jj= 1,nao_g
          cmoa_g(jj,icount)= cmoa_g(jj,ii)
        enddo 
      enddo 
!
      if((scftype == 'UHF').and.(scftype_g == 'UHF')) then
!
! Calculate Cguess-t*Overlap*Ccore
!
        call dgemm('N','N',nao_gcore,nmo_g,nao_g,one,overlap,nao_gcore,cmob_g,nao_g,zero, &
&                  work2,nao_gcore)
        call dgemm('T','N',nmo_g,nao_gcore,nao_gcore,one,work2,nao_gcore,coremo,nao_gcore,zero, &
&                  work3,nao_g)
!
! Check correspoing orbitals
!
        skipmo(:)=.false.
        do ii= 1,nao_gcore
          skipmo(idamax(nmo_g,work3(1,ii),1))=.true.
        enddo
!
! Skip core orbitals
!
        icount= 0
        do ii= 1,nmo_g
          if(skipmo(ii)) cycle
          icount= icount+1
          do jj= 1,nao_g
            cmob_g(jj,icount)= cmob_g(jj,ii)
          enddo
        enddo
      endif
!
      return
end

!---------------------------------------------------------------------
  subroutine hcoreguess(cmo,h1mtrx,ortho,work,nproc,myrank,mpi_comm)
!---------------------------------------------------------------------
!
! Calculate initial guess MOs by diagonalizing core Hamiltonian matrix
!
! In   : h1mtrx (One-electron core Hamiltonian matrx)
!        ortho  (Orthogonalization matrix)
! Out  : cmo    (Initial guess orbitals)
!        work   (Work space)
!
      use modbasis, only : nao
      use modmolecule, only : nmo
      implicit none
      integer,intent(in) :: nproc, myrank, mpi_comm
      integer :: nao3, idis(nproc,14), isize
      real(8),intent(in) :: h1mtrx(nao*(nao+1)/2), ortho(nao*nao)
      real(8),intent(out) :: cmo(nao*nao), work(nao*nao)
      real(8),allocatable :: work2(:), eigen(:)
!
      nao3= nao*(nao+1)/2
      call distarray(idis,nmo,nao,nao3,nao,nao,nao,nao,nproc)
      isize= idis(myrank+1,3)
!
      call memset(isize+nao)
      allocate(work2(isize),eigen(nao))
!
      call diagfock(h1mtrx,work,ortho,cmo,work2,eigen,idis,nproc,myrank,mpi_comm)
!
      deallocate(work2,eigen)
      call memunset(isize+nao)
!
      return
end


!--------------------------------
  subroutine gcheckreorder(cmo)
!--------------------------------
!
! Reorder guess molecular orbitals
!
      use modguess, only : locbf_g, mbf_g, mtype_g, nao_g, nmo_g, nshell_g
      implicit none
      integer :: ish, iloc, imo, iao
      real(8),intent(inout) :: cmo(nao_g,nao_g)
      real(8) :: work(28)
!
!ishimura
      do ish= 1,nshell_g
        if(mtype_g(ish) >= 2) then
          iloc= locbf_g(ish)
          select case (mbf_g(ish))
            case(5)
              do imo= 1,nmo_g
                do iao= 1,5
                  work(iao)= cmo(iloc+iao,imo)
                enddo
                cmo(iloc+1,imo)= work(5)
                cmo(iloc+2,imo)= work(3)
                cmo(iloc+3,imo)= work(1)
                cmo(iloc+4,imo)= work(2)
                cmo(iloc+5,imo)= work(4)
              enddo
            case(6)
              do imo= 1,nmo_g
                do iao= 1,6
                  work(iao)= cmo(iloc+iao,imo)
                enddo
                cmo(iloc+1,imo)= work(1)
                cmo(iloc+2,imo)= work(4)
                cmo(iloc+3,imo)= work(5)
                cmo(iloc+4,imo)= work(2)
                cmo(iloc+5,imo)= work(6)
                cmo(iloc+6,imo)= work(3)
              enddo
            case(7)
              do imo= 1,nmo_g
                do iao= 1,7
                  work(iao)= cmo(iloc+iao,imo)
                enddo
                cmo(iloc+1,imo)= work(7)
                cmo(iloc+2,imo)= work(5)
                cmo(iloc+3,imo)= work(3)
                cmo(iloc+4,imo)= work(1)
                cmo(iloc+5,imo)= work(2)
                cmo(iloc+6,imo)= work(4)
                cmo(iloc+7,imo)= work(6)
              enddo
            case(10)
              do imo= 1,nmo_g
                do iao= 1,10
                  work(iao)= cmo(iloc+iao,imo)
                enddo
                cmo(iloc+ 1,imo)= work( 1)
                cmo(iloc+ 2,imo)= work( 4)
                cmo(iloc+ 3,imo)= work( 5)
                cmo(iloc+ 4,imo)= work( 6)
                cmo(iloc+ 5,imo)= work(10)
                cmo(iloc+ 6,imo)= work( 8)
                cmo(iloc+ 7,imo)= work( 2)
                cmo(iloc+ 8,imo)= work( 7)
                cmo(iloc+ 9,imo)= work( 9)
                cmo(iloc+10,imo)= work( 3)
              enddo
            case(9)
              do imo= 1,nmo_g
                do iao= 1,9
                  work(iao)= cmo(iloc+iao,imo)
                enddo
                cmo(iloc+1,imo)= work(9)
                cmo(iloc+2,imo)= work(7)
                cmo(iloc+3,imo)= work(5)
                cmo(iloc+4,imo)= work(3)
                cmo(iloc+5,imo)= work(1)
                cmo(iloc+6,imo)= work(2)
                cmo(iloc+7,imo)= work(4)
                cmo(iloc+8,imo)= work(6)
                cmo(iloc+9,imo)= work(8)
              enddo
            case(15)
              do imo= 1,nmo_g
                do iao= 1,15
                  work(iao)= cmo(iloc+iao,imo)
                enddo
                cmo(iloc+ 1,imo)= work( 1)
                cmo(iloc+ 2,imo)= work( 4)
                cmo(iloc+ 3,imo)= work( 5)
                cmo(iloc+ 4,imo)= work(10)
                cmo(iloc+ 5,imo)= work(13)
                cmo(iloc+ 6,imo)= work(11)
                cmo(iloc+ 7,imo)= work( 6)
                cmo(iloc+ 8,imo)= work(14)
                cmo(iloc+ 9,imo)= work(15)
                cmo(iloc+10,imo)= work( 8)
                cmo(iloc+11,imo)= work( 2)
                cmo(iloc+12,imo)= work( 7)
                cmo(iloc+13,imo)= work(12)
                cmo(iloc+14,imo)= work( 9)
                cmo(iloc+15,imo)= work( 3)
              enddo
            case default
              write(*,'(" Error! Subroutine gcheckreorder supports up to g functions.")')
              call iabort
          end select
        endif
      enddo
!
      return
end


!----------------------------------------------------------
  subroutine dftbguess(cmo,overinv,nproc,myrank,mpi_comm)
!----------------------------------------------------------
!
! Calculate initial guess MOs by extended Huckel
!
! In  : overinv (overlap integral inverse matrix)
! Out : cmo     (initial guess orbitals)
!
      use modguess, only : nao_g, spher_g, coord_g
      use modbasis, only : nao
      use modmolecule, only : coord, natom
      implicit none
      integer,intent(in) :: nproc, myrank, mpi_comm
      integer :: i, j, nao_g2, nao_g3
      real(8),intent(in):: overinv(nao*nao)
      real(8),intent(out):: cmo(nao*nao)
      real(8),allocatable :: hmo(:), overlap(:), work1(:), work2(:), eigen(:)
!
! Set basis functions
!
      spher_g=.true.
      call setbasis_g(1)
!
! Set coordinate
!
      do i= 1,natom
        do j= 1,3
          coord_g(j,i)=coord(j,i)
        enddo
      enddo
!
! Set required arrays
!
      nao_g2= nao_g*nao_g
      nao_g3= max(nao,nao_g)*nao_g
      call memset(2*nao_g2+2*nao_g3+nao_g)
      allocate(hmo(nao_g2),overlap(nao_g3),work1(nao_g3),work2(nao_g2),eigen(nao_g))
!
! Calculate Extended Huckel method
!
      call calcdftb(hmo,overlap,work1,work2,eigen,nproc,myrank,mpi_comm)
!
! Calculate overlap integrals between input basis and Huckel basis
!
      call calcover2(overlap,work1,nproc,myrank,mpi_comm)
!
! Project orbitals from Huckel to SCF
!
      call projectmo(cmo,overinv,overlap,hmo,work1,work2,eigen,nproc,myrank,mpi_comm)
      deallocate(hmo,overlap,work1,work2,eigen)
      call memunset(2*nao_g2+2*nao*nao_g+nao_g)
      return
end


!-----------------------------------------------------------------------------
  subroutine calcdftb(dftbmo,overlap,ortho,work,eigen,nproc,myrank,mpi_comm)
!-----------------------------------------------------------------------------
!
! Driver of DFTB calculation
!
!
! Out : dftbmo (DFTB MOs)
!       overlap,ortho,work,eigen (work space)
!
      use modparallel, only : master
      use modguess, only : nao_g, nmo_g, spher_g, coord_g, nshell_v
      use modthresh, only : threshover
      use modmolecule, only : coord, natom, neleca, nelecb
      implicit none
      integer,parameter :: maxiter=1000
      integer,intent(in) :: nproc, myrank, mpi_comm
      integer :: ii, jj, nelecdftb(2), iter
      real(8),parameter :: zero=0.0D+00, one=1.0D+00
      real(8),intent(out) :: dftbmo(nao_g,nao_g), overlap(nao_g,nao_g)
      real(8),intent(out) :: ortho(nao_g,nao_g), work(nao_g,nao_g), eigen(nao_g)
      real(8),allocatable :: dftb0(:), dftb1(:), gamma12(:), uhub(:), qmulliken(:,:)
      real(8) :: qmax
!
! Set basis functions
!
      spher_g=.true.
      call setbasis_g(1)
!
! Set coordinate
!
      do ii= 1,natom
        do jj= 1,3
          coord_g(jj,ii)=coord(jj,ii)
        enddo
      enddo
!
      call memset(2*nao_g*nao_g+nshell_v*nshell_v+3*nshell_v)
      allocate(dftb0(nao_g*nao_g),dftb1(nao_g*nao_g),gamma12(nshell_v*nshell_v), &
&              uhub(nshell_v),qmulliken(nshell_v,2))
!
! Calculate overlap integrals
! (guess basis)x(guess basis)
!
      call calcover1(overlap)
      do ii= 1,nao_g
        do jj= 1,ii
          work(jj,ii)= overlap(jj,ii)
        enddo
      enddo
!
! Calculate orthogonalization matrix
!
      call mtrxcanon(ortho,work,eigen,nao_g,nmo_g,threshover,nproc,myrank,mpi_comm)
!
! Set parameters
!
      call huckelip(eigen,1)
      call dftbip(eigen,uhub)
      nelecdftb(1)= neleca
      nelecdftb(2)= nelecb
!
! Calculate core Hamiltonian matrix for DFTB
!
      dftb0=zero
      call formdftb0(dftb0,overlap,eigen)
!
! Calculate gamma12 calculation for DFTB
!
      call calcgamma12(gamma12,uhub)
!
! Set initial Mulliken charge
!
      qmulliken(1:nshell_v,1)= zero
!
      do iter= 1,maxiter
        dftb1(:)= dftb0(:)
!
! Calculate Hamiltonian matrix for DFTB
!
        call formdftb1(dftb1,overlap,qmulliken,gamma12)
!
! Diagonalize Hamiltonian matrix
!
        call canonicalize(dftb1,ortho,work,nao_g,nmo_g)
        call diag('V','U',nmo_g,dftb1,nao_g,eigen,nproc,myrank,mpi_comm)
        call dgemm('N','N',nao_g,nmo_g,nmo_g,one,ortho,nao_g,dftb1,nao_g,zero,dftbmo,nao_g)
!
! Calculate new Mulliken charge
!
        qmulliken(1:nshell_v,2)= qmulliken(1:nshell_v,1)
        call calcdftbmulliken(overlap,dftbmo,work,qmulliken,nelecdftb)
        call diffqmulliken(qmulliken,qmax)
        if(qmax < 1.0D-4) exit
        if(iter == maxiter) then
          if(master) write(*,'(" Error! DFTB calculation did not converge.")')
          call iabort
        endif
      enddo
!
      return
end


!---------------------------------------------
  subroutine formdftb0(dftb0,overlap,energy)
!---------------------------------------------
!
! Form charge independent DFTB matrix elements
!
      use modparam, only : mxprsh
      use modguess, only : nao_v, nshell_v, nao_g, locprim_g, locbf_g, locatom_g, mprim_g, &
&                          mbf_g, mtype_g, ex_g, coeff_g, coord_g
      use modthresh, only : threshex
      use modmolecule, only : numatomic
      implicit none
      integer,parameter :: len1=5
      integer :: ii, jj, ish, jsh, iprim, jprim, nangij(2), nprimij(2), nbfij(2)
      integer :: iatom, jatom, iloc, jloc, ilocbf, jlocbf
      real(8),parameter :: zero=0.0D+00, factor=0.875D+00, fdown=0.05D+00
      real(8),intent(in) :: overlap(nao_g,nao_g), energy(nao_g)
      real(8),intent(out) :: dftb0(nao_g,nao_g)
      real(8) :: coordij(3,2), exij(mxprsh,2), coij(mxprsh,2), znucdftb(2)
      real(8) :: sint(len1,len1), tint(len1,len1), cint(len1,len1)
      real(8) :: znucvalence(86)= &
&     (/ 1.0D+00, 2.0D+00, 1.0D+00, 2.0D+00, 3.0D+00, 4.0D+00, 5.0D+00, 6.0D+00, 7.0D+00,&
&        8.0D+00, 1.0D+00, 2.0D+00, 3.0D+00, 4.0D+00, 5.0D+00, 6.0D+00, 7.0D+00, 8.0D+00,&
&        1.0D+00, 2.0D+00, 3.0D+00, 4.0D+00, 5.0D+00, 6.0D+00, 7.0D+00, 8.0D+00, 9.0D+00,&
&        1.0D+01, 1.1D+01, 1.2D+01, 3.0D+00, 4.0D+00, 5.0D+00, 6.0D+00, 7.0D+00, 8.0D+00,&
&        1.0D+00, 2.0D+00, 3.0D+00, 4.0D+00, 5.0D+00, 6.0D+00, 7.0D+00, 8.0D+00, 9.0D+00,&
&        1.0D+01, 1.1D+01, 1.2D+01, 3.0D+00, 4.0D+00, 5.0D+00, 6.0D+00, 7.0D+00, 8.0D+00,&
&        1.0D+00, 2.0D+00, 3.0D+00, 4.0D+00, 5.0D+00, 6.0D+00, 7.0D+00, 8.0D+00, 9.0D+00,&
&        1.0D+01, 1.1D+01, 1.2D+01, 1.3D+01, 1.4D+01, 1.5D+01, 1.6D+01, 3.0D+00, 4.0D+00,&
&        5.0D+00, 6.0D+00, 7.0D+00, 8.0D+00, 9.0D+00, 1.0D+01, 1.1D+01, 1.2D+01, 3.0D+00,&
&        4.0D+00, 5.0D+00, 6.0D+00, 7.0D+00, 8.0D+00/)
!
! Diagonal elements
!
      do ii= 1,nao_g
        dftb0(ii,ii)= energy(ii)
      enddo
!
! Off-diagonal elements of valence functions
!
!$OMP parallel do schedule(static,1) private(nangij,nprimij,nbfij,iatom,iloc,ilocbf,coordij, &
!$OMP exij,coij,znucdftb,jatom,jloc,jlocbf,sint,tint,cint)
      do ish= 1,nshell_v
        nangij(1)= mtype_g(ish)
        nprimij(1)= mprim_g(ish)
        nbfij(1)  = mbf_g(ish)
        iatom = locatom_g(ish)
        iloc  = locprim_g(ish)
        ilocbf= locbf_g(ish)
        do ii= 1,3
          coordij(ii,1)= coord_g(ii,iatom)
        enddo
        do iprim= 1,nprimij(1)
          exij(iprim,1)= ex_g(iloc+iprim)
          coij(iprim,1)= coeff_g(iloc+iprim)
        enddo
        znucdftb(1)= znucvalence(numatomic(iatom))
!
        do jsh= 1,ish-1
          jatom = locatom_g(jsh)
          if(iatom /= jatom) then
            nangij(2)= mtype_g(jsh)
            nprimij(2)= mprim_g(jsh)
            nbfij(2)  = mbf_g(jsh)
            jloc  = locprim_g(jsh)
            jlocbf= locbf_g(jsh)
            do ii= 1,3
              coordij(ii,2)= coord_g(ii,jatom)
            enddo
            do jprim= 1,nprimij(2)
              exij(jprim,2)= ex_g(jloc+jprim)
              coij(jprim,2)= coeff_g(jloc+jprim)
            enddo
!
! Overlap and kinetic integrals
!
            call intst(sint,tint,exij,coij,coordij, &
&                      nprimij,nangij,nbfij,len1,mxprsh,threshex)
!
! 1-electron Coulomb integrals
!
            znucdftb(2)= znucvalence(numatomic(jatom))
            if((nangij(1) <= 2).and.(nangij(2) <= 2)) then
              call int1cmd(cint,exij,coij,coordij,coordij,znucdftb,2, &
&                          nprimij,nangij,nbfij,len1,mxprsh,threshex)
            else
              call int1rys(cint,exij,coij,coordij,coordij,znucdftb,2, &
&                          nprimij,nangij,nbfij,len1,mxprsh,threshex)
            endif
!
! Add kinetic and Coulomb integrals
!
            if((nangij(1) <= 1).and.(nangij(2) <= 1))then
              do ii= 1,nbfij(1)
                do jj= 1,nbfij(2)
                  dftb0(jlocbf+jj,ilocbf+ii)= tint(jj,ii)+cint(jj,ii)
                enddo
              enddo
            else
              do ii= 1,nbfij(1)
                do jj= 1,nbfij(2)
!ishimura
! Scale-down the nuclear attraction term
!
                  dftb0(jlocbf+jj,ilocbf+ii)= tint(jj,ii)+cint(jj,ii)*0.1D0
                enddo
              enddo
            endif
          endif
        enddo
      enddo
!$OMP end parallel do
!
! Off-diagonal elements of core functions
!
!$OMP parallel do schedule(static,1)
      do ii= nao_v+1,nao_g
        do jj= 1,ii-1
          dftb0(jj,ii)= factor*fdown*overlap(jj,ii)*(energy(ii)+energy(jj))
        enddo
      enddo
!$OMP end parallel do
!
      return
end


!--------------------------------------------------------
  subroutine formdftb1(dftb1,overlap,qmulliken,gamma12)
!--------------------------------------------------------
!
! Form charge dependent DFTB matrix elements
!
      use modguess, only : nao_g, locbf_g, mbf_g, nshell_v
      implicit none
      integer :: ish, jsh, ksh, nbfi, nbfj, ilocbf, jlocbf, ii, jj
      real(8),parameter :: zero=0.0D+00, half=0.5D+00
      real(8),intent(in) :: overlap(nao_g,nao_g), qmulliken(nshell_v), gamma12(nshell_v,nshell_v)
      real(8),intent(inout) :: dftb1(nao_g,*)
      real(8) :: cterm
!
!$OMP parallel do schedule(static,1) private(nbfi,ilocbf,nbfj,jlocbf,cterm)
      do ish= 1,nshell_v
        nbfi  = mbf_g(ish)
        ilocbf= locbf_g(ish)
        do jsh= 1,ish
          nbfj  = mbf_g(jsh)
          jlocbf= locbf_g(jsh)
          cterm= zero
          do ksh= 1,nshell_v
            cterm= cterm+(gamma12(ksh,ish)+gamma12(ksh,jsh))*qmulliken(ksh)
          enddo
          do ii= 1,nbfi
            do jj= 1,nbfj
              dftb1(jlocbf+jj,ilocbf+ii)= dftb1(jlocbf+jj,ilocbf+ii) &
&                                        +half*cterm*overlap(jlocbf+jj,ilocbf+ii)
            enddo
          enddo
        enddo
      enddo
!
      return
end


!---------------------------------------
  subroutine calcgamma12(gamma12,uhub)
!---------------------------------------
!
! Driver of gamma12 calculation for DFTB
!
      use modguess, only : coord_g, locatom_g, nshell_v
      use modmolecule, only : numatomic
      implicit none
      integer :: iatom, jatom, ish, jsh
      real(8),intent(in) :: uhub(nshell_v)
      real(8),intent(out) :: gamma12(nshell_v,nshell_v)
      real(8) :: rr
      logical :: hbondi, hbondij
!
!$OMP parallel do schedule(static,1) private(iatom,hbondi,jatom,rr,hbondij)
      do ish= 1,nshell_v
        iatom= locatom_g(ish)
        hbondi=(numatomic(iatom) == 1)
        do jsh= 1,ish
          jatom= locatom_g(jsh)
          rr= sqrt((coord_g(1,iatom)-coord_g(1,jatom))**2+(coord_g(2,iatom)-coord_g(2,jatom))**2 &
&                 +(coord_g(3,iatom)-coord_g(3,jatom))**2)
          hbondij= hbondi.or.(numatomic(jatom) == 1)
          call dftbgamma(gamma12(jsh,ish),uhub(ish),uhub(jsh),rr,hbondij)
          gamma12(ish,jsh)= gamma12(jsh,ish)
        enddo
      enddo
!
      return
end


!--------------------------------------------------------------------
  subroutine calcdftbmulliken(overlap,cmo,work,qmulliken,nelecdftb)
!--------------------------------------------------------------------
!
! Calculate Mulliken population for DFTB
!
      use modguess, only : nao_g, nshell_v, locbf_g, mbf_g, mtype_g, locatom_g
      use modmolecule, only : numatomic
      implicit none
      integer,intent(in) :: nelecdftb(2)
      integer :: ii, jj, ish, locbfi
      real(8),parameter :: zero=0.0D+00, one=1.0D+00, two=2.0D+00
      real(8),intent(in) :: overlap(nao_g,nao_g), cmo(nao_g,nao_g)
      real(8),intent(out) :: work(nao_g,nao_g), qmulliken(nshell_v)
      real(8) :: grossorb(nao_g)
      real(8) :: znucshell(0:2,84)
      data znucshell/ &
&          1.0D+0, 0.0D+0, 0.0D+0,  2.0D+0, 0.0D+0, 0.0D+0,  1.0D+0, 0.0D+0, 0.0D+0, &
&          2.0D+0, 0.0D+0, 0.0D+0,  2.0D+0, 1.0D+0, 0.0D+0,  2.0D+0, 2.0D+0, 0.0D+0, &
&          2.0D+0, 3.0D+0, 0.0D+0,  2.0D+0, 4.0D+0, 0.0D+0,  2.0D+0, 5.0D+0, 0.0D+0, &
&          2.0D+0, 6.0D+0, 0.0D+0,  1.0D+0, 0.0D+0, 0.0D+0,  2.0D+0, 0.0D+0, 0.0D+0, &
&          2.0D+0, 1.0D+0, 0.0D+0,  2.0D+0, 2.0D+0, 0.0D+0,  2.0D+0, 3.0D+0, 0.0D+0, &
&          2.0D+0, 4.0D+0, 0.0D+0,  2.0D+0, 5.0D+0, 0.0D+0,  2.0D+0, 6.0D+0, 0.0D+0, &
&          1.0D+0, 0.0D+0, 0.0D+0,  2.0D+0, 0.0D+0, 0.0D+0,  2.0D+0, 0.0D+0, 1.0D+0, &
&          2.0D+0, 0.0D+0, 2.0D+0,  2.0D+0, 0.0D+0, 3.0D+0,  1.0D+0, 0.0D+0, 5.0D+0, &
&          2.0D+0, 0.0D+0, 5.0D+0,  2.0D+0, 0.0D+0, 6.0D+0,  2.0D+0, 0.0D+0, 7.0D+0, &
&          1.0D+0, 0.0D+0, 9.0D+0,  1.0D+0, 0.0D+0, 1.0D+1,  2.0D+0, 0.0D+0, 1.0D+1, &
&          2.0D+0, 1.0D+0, 0.0D+0,  2.0D+0, 2.0D+0, 0.0D+0,  2.0D+0, 3.0D+0, 0.0D+0, &
&          2.0D+0, 4.0D+0, 0.0D+0,  2.0D+0, 5.0D+0, 0.0D+0,  2.0D+0, 6.0D+0, 0.0D+0, &
&          1.0D+0, 0.0D+0, 0.0D+0,  2.0D+0, 0.0D+0, 0.0D+0,  2.0D+0, 0.0D+0, 1.0D+0, &
&          2.0D+0, 0.0D+0, 2.0D+0,  2.0D+0, 0.0D+0, 3.0D+0,  2.0D+0, 0.0D+0, 4.0D+0, &
&          2.0D+0, 0.0D+0, 5.0D+0,  1.0D+0, 0.0D+0, 7.0D+0,  1.0D+0, 0.0D+0, 8.0D+0, &
&          1.0D+0, 0.0D+0, 9.0D+0,  1.0D+0, 0.0D+0, 1.0D+1,  2.0D+0, 0.0D+0, 1.0D+1, &
&          2.0D+0, 1.0D+0, 0.0D+0,  2.0D+0, 2.0D+0, 0.0D+0,  2.0D+0, 3.0D+0, 0.0D+0, &
&          2.0D+0, 4.0D+0, 0.0D+0,  2.0D+0, 5.0D+0, 0.0D+0,  2.0D+0, 6.0D+0, 0.0D+0, &
&          1.0D+0, 0.0D+0, 0.0D+0,  2.0D+0, 0.0D+0, 0.0D+0,  0.0D+0, 0.0D+0, 0.0D+0, &
&          0.0D+0, 0.0D+0, 0.0D+0,  0.0D+0, 0.0D+0, 0.0D+0,  0.0D+0, 0.0D+0, 0.0D+0, &
&          0.0D+0, 0.0D+0, 0.0D+0,  0.0D+0, 0.0D+0, 0.0D+0,  0.0D+0, 0.0D+0, 0.0D+0, &
&          0.0D+0, 0.0D+0, 0.0D+0,  0.0D+0, 0.0D+0, 0.0D+0,  0.0D+0, 0.0D+0, 0.0D+0, &
&          0.0D+0, 0.0D+0, 0.0D+0,  0.0D+0, 0.0D+0, 0.0D+0,  0.0D+0, 0.0D+0, 0.0D+0, &
&          0.0D+0, 0.0D+0, 0.0D+0,  2.0D+0, 0.0D+0, 1.0D+0,  2.0D+0, 0.0D+0, 2.0D+0, &
&          2.0D+0, 0.0D+0, 3.0D+0,  2.0D+0, 0.0D+0, 4.0D+0,  2.0D+0, 0.0D+0, 5.0D+0, &
&          2.0D+0, 0.0D+0, 6.0D+0,  2.0D+0, 0.0D+0, 7.0D+0,  2.0D+0, 0.0D+0, 8.0D+0, &
&          1.0D+0, 0.0D+0, 1.0D+1,  2.0D+0, 0.0D+0, 1.0D+1,  2.0D+0, 1.0D+0, 0.0D+0, &
&          2.0D+0, 2.0D+0, 0.0D+0,  2.0D+0, 3.0D+0, 0.0D+0,  2.0D+0, 4.0D+0, 0.0D+0/
!
      grossorb(:)= zero
      qmulliken(:)= zero
!
      if(nelecdftb(1) == nelecdftb(2)) then
!
! Calculate Gross orbital population
!
        call dgemm('N','T',nao_g,nao_g,nelecdftb(1),two,cmo,nao_g,cmo,nao_g,zero,work,nao_g)
!$OMP parallel do schedule(static,1) reduction(+:grossorb)
        do ii= 1,nao_g
          do jj= 1,ii-1
            grossorb(ii)= grossorb(ii)+work(jj,ii)*overlap(jj,ii)
            grossorb(jj)= grossorb(jj)+work(jj,ii)*overlap(jj,ii)
          enddo
          grossorb(ii)= grossorb(ii)+work(ii,ii)*overlap(ii,ii)
        enddo
!
! Calculate Gross atom population
!
        do ish= 1,nshell_v
          locbfi= locbf_g(ish)
          do ii= 1,mbf_g(ish)
            qmulliken(ish)= qmulliken(ish)+grossorb(locbfi+ii)
          enddo
        enddo
        do ish= 1,nshell_v
          qmulliken(ish)= qmulliken(ish)-znucshell(mtype_g(ish),numatomic(locatom_g(ish)))
        enddo
      else
        write(*,'(" Error! DFTB for open-shell has not been implemented yet!")')
        call iabort
      endif
!
      return
end


!-------------------------------------------
  subroutine diffqmulliken(qmulliken,qmax)
!-------------------------------------------
!
! Calculate maximum absolute change of Mulliken charge
!
      use modguess, only : nshell_v
      implicit none
      integer :: ishell
      real(8),parameter :: zero=0.0D+00
      real(8),intent(in) :: qmulliken(nshell_v,2)
      real(8),intent(out) :: qmax
      real(8) :: qtmp
!
      qmax= zero
      do ishell= 1,nshell_v
        qtmp= qmulliken(ishell,2)-qmulliken(ishell,1)
        if(abs(qtmp) > qmax) qmax= abs(qtmp)
      enddo
!
      return
end


!-----------------------------------------------------
  subroutine dftbgamma(gammaab,uhuba,uhubb,rr,hbond)
!-----------------------------------------------------
!
! Calculate gamma value for DFTB
!
      implicit none
      real(8),parameter :: one=1.0D+00, half=0.5D+00, three=3.0D+00, nine=9.0D+00
      real(8),parameter :: p16=1.6D+00, p33=3.3D+01, p48=4.8D+01
      real(8),parameter :: factau=3.2D+00, facg=0.3125D+00, gthresh=1.0D-05
      real(8),intent(in) :: uhuba, uhubb, rr
      real(8),intent(out) :: gammaab
      real(8) :: taua, taub, denom, tautmp, tauavg, coeffhbond, rrinv, tmp, saa
      real(8) :: taua2, taub2, taua4, taub4, denom2, sab, sba
      logical,intent(in) :: hbond
!
      taua= factau*uhuba
      taub= factau*uhubb
      denom= one/(taua+taub)
      tautmp= taua*taub*denom
      tauavg= p16*tautmp*(one+tautmp*denom)
      if(hbond) then
        coeffhbond= exp(-((uhuba+uhubb)*half)**4*rr*rr)
      else
        coeffhbond= one
      endif
!
      if(rr < gthresh) then
        gammaab = facg*tauavg
      else
        rrinv= one/rr
        if(abs(taua-taub) < gthresh) then
          tmp= tauavg*rr
          saa= exp(-tmp)*rrinv*(one+(p33+(nine+tmp)*tmp)*tmp/p48)
          gammaab= rrinv-coeffhbond*saa
        else
          taua2= taua*taua
          taub2= taub*taub
          taua4= taua2*taua2
          taub4= taub2*taub2
          denom= one/(taua2-taub2)
          denom2= denom*denom
          sab= exp(-taua*rr)*taub4*denom2*(half*taua-(taub2-three*taua2)*denom*rrinv)
          sba= exp(-taub*rr)*taua4*denom2*(half*taub+(taua2-three*taub2)*denom*rrinv)
          gammaab= rrinv-coeffhbond*(sab+sba)
        endif
      endif
!
      return
end


!---------------------------------
  subroutine dftbip(energy,uhub)
!---------------------------------
!
! Set valence ionization potentials for DFTB
!
      use modparallel, only : master
      use modmolecule, only : natom, numatomic
      use modguess, only : nao_g, nshell_v
      implicit none
      integer :: iao, iatom, i, ishell
      real(8),intent(inout) :: energy(nao_g), uhub(nshell_v)
      real(8) :: dftbips(84)= &
&     (/-0.238603D+0, -0.579318D+0, -0.105624D+0, -0.206152D+0, -0.347026D+0, -0.505337D+0,&
&       -0.682915D+0, -0.880592D+0, -1.098828D+0, -1.337930D+0, -0.100836D+0, -0.172918D+0,&
&       -0.284903D+0, -0.397349D+0, -0.513346D+0, -0.634144D+0, -0.760399D+0, -0.892514D+0,&
&       -0.085219D+0, -0.138404D+0, -0.153708D+0, -0.164133D+0, -0.172774D+0, -0.147221D+0,&
&       -0.187649D+0, -0.194440D+0, -0.200975D+0, -0.165046D+0, -0.169347D+0, -0.219658D+0,&
&       -0.328789D+0, -0.431044D+0, -0.532564D+0, -0.635202D+0, -0.739820D+0, -0.846921D+0,&
&       -0.081999D+0, -0.129570D+0, -0.150723D+0, -0.163093D+0, -0.172061D+0, -0.179215D+0,&
&       -0.185260D+0, -0.155713D+0, -0.157939D+0, -0.159936D+0, -0.161777D+0, -0.207892D+0,&
&       -0.301650D+0, -0.387547D+0, -0.471377D+0, -0.555062D+0, -0.639523D+0, -0.725297D+0,&
&       -0.076658D+0, -0.118676D+0, -0.135171D+0,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,&
&        0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,&
&        0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     , -0.171078D+0, -0.187557D+0,&
&       -0.199813D+0, -0.209733D+0, -0.218183D+0, -0.225640D+0, -0.232400D+0, -0.238659D+0,&
&       -0.211421D+0, -0.250189D+0, -0.350442D+0, -0.442037D+0, -0.531518D+0, -0.620946D+0/)
      real(8) :: uhubs(84)= &
&     (/ 0.419731D+0,  0.742961D+0,  0.174131D+0,  0.270796D+0,  0.333879D+0,  0.399218D+0,&
&        0.464356D+0,  0.528922D+0,  0.592918D+0,  0.656414D+0,  0.165505D+0,  0.224983D+0,&
&        0.261285D+0,  0.300005D+0,  0.338175D+0,  0.375610D+0,  0.412418D+0,  0.448703D+0,&
&        0.136368D+0,  0.177196D+0,  0.189558D+0,  0.201341D+0,  0.211913D+0,  0.200284D+0,&
&        0.230740D+0,  0.239398D+0,  0.247710D+0,  0.235429D+0,  0.243169D+0,  0.271212D+0,&
&        0.279898D+0,  0.304342D+0,  0.330013D+0,  0.355433D+0,  0.380376D+0,  0.404852D+0,&
&        0.130512D+0,  0.164724D+0,  0.176814D+0,  0.189428D+0,  0.200280D+0,  0.209759D+0,&
&        0.218221D+0,  0.212289D+0,  0.219321D+0,  0.225725D+0,  0.231628D+0,  0.251776D+0,&
&        0.257192D+0,  0.275163D+0,  0.294185D+0,  0.313028D+0,  0.331460D+0,  0.349484D+0,&
&        0.120590D+0,  0.149382D+0,  0.160718D+0,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,&
&        0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,&
&        0.0D+0     ,  0.0D+0     ,  0.0D+00    ,  0.0D+0     ,  0.187365D+0,  0.200526D+0,&
&        0.212539D+0,  0.223288D+0,  0.233028D+0,  0.241981D+0,  0.250317D+0,  0.258165D+0,&
&        0.255962D+0,  0.272767D+0,  0.267448D+0,  0.280804D+0,  0.296301D+0,  0.311976D+0/)
      real(8) :: dftbipp(3:84)= &
      (/                            -0.040054D+0, -0.074172D+0, -0.132547D+0, -0.194236D+0,&
&       -0.260544D+0, -0.331865D+0, -0.408337D+0, -0.490009D+0, -0.027320D+0, -0.048877D+0,&
&       -0.099666D+0, -0.149976D+0, -0.202363D+0, -0.257553D+0, -0.315848D+0, -0.377389D+0,&
&       -0.029573D+0, -0.051543D+0, -0.053913D+0, -0.053877D+0, -0.053055D+0, -0.036319D+0,&
&       -0.050354D+0, -0.048699D+0, -0.046909D+0, -0.027659D+0, -0.025621D+0, -0.040997D+0,&
&       -0.094773D+0, -0.143136D+0, -0.190887D+0, -0.239256D+0, -0.288792D+0, -0.339778D+0,&
&       -0.027523D+0, -0.047197D+0, -0.052925D+0, -0.053976D+0, -0.053629D+0, -0.052675D+0,&
&       -0.051408D+0, -0.033507D+0, -0.031248D+0, -0.029100D+0, -0.027061D+0, -0.043481D+0,&
&       -0.092539D+0, -0.135732D+0, -0.177383D+0, -0.218721D+0, -0.260330D+0, -0.302522D+0,&
&       -0.027142D+0, -0.045680D+0, -0.049659D+0,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,&
&        0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,&
&        0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     , -0.049388D+0, -0.051266D+0,&
&       -0.051078D+0, -0.049978D+0, -0.048416D+0, -0.046602D+0, -0.044644D+0, -0.042604D+0,&
&       -0.028258D+0, -0.038408D+0, -0.087069D+0, -0.128479D+0, -0.167900D+0, -0.206503D+0/) 
      real(8) :: uhubp(3:84)= &
      (/                             0.131681D+0,  0.224651D+0,  0.296157D+0,  0.364696D+0,&
&        0.430903D+0,  0.495405D+0,  0.558631D+0,  0.620878D+0,  0.087777D+0,  0.150727D+0,&
&        0.203216D+0,  0.247841D+0,  0.289262D+0,  0.328724D+0,  0.366885D+0,  0.404106D+0,&
&        0.081938D+0,  0.128252D+0,  0.137969D+0,  0.144515D+0,  0.149029D+0,  0.123012D+0,&
&        0.155087D+0,  0.156593D+0,  0.157219D+0,  0.106180D+0,  0.097312D+0,  0.153852D+0,&
&        0.205025D+0,  0.240251D+0,  0.271613D+0,  0.300507D+0,  0.327745D+0,  0.353804D+0,&
&        0.073660D+0,  0.115222D+0,  0.127903D+0,  0.136205D+0,  0.141661D+0,  0.145599D+0,&
&        0.148561D+0,  0.117901D+0,  0.113146D+0,  0.107666D+0,  0.099994D+0,  0.150506D+0,&
&        0.189913D+0,  0.217398D+0,  0.241589D+0,  0.263623D+0,  0.284168D+0,  0.303641D+0,&
&        0.069450D+0,  0.105176D+0,  0.115479D+0,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,&
&        0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,&
&        0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.126480D+0,  0.135605D+0,&
&        0.141193D+0,  0.144425D+0,  0.146247D+0,  0.146335D+0,  0.145121D+0,  0.143184D+0,&
&        0.090767D+0,  0.134398D+0,  0.185496D+0,  0.209811D+0,  0.231243D+0,  0.250546D+0/)
      real(8) :: dftbipd(21:84)= &
      (/                            -0.118911D+0, -0.156603D+0, -0.189894D+0, -0.107113D+0,&
&       -0.248949D+0, -0.275927D+0, -0.301635D+0, -0.170792D+0, -0.185263D+0, -0.372826D+0,&
&        0.043096D+0,  0.062123D+0,  0.078654D+0,  0.104896D+0,  0.126121D+0,  0.140945D+0,&
&        0.030672D+0, -0.041363D+0, -0.092562D+0, -0.132380D+0, -0.170468D+0, -0.207857D+0,&
&       -0.244973D+0, -0.191289D+0, -0.218418D+0, -0.245882D+0, -0.273681D+0, -0.431379D+0,&
&        0.135383D+0,  0.125834D+0,  0.118556D+0,  0.114419D+0,  0.112860D+0,  0.111715D+0,&
&       -0.007997D+0, -0.074037D+0, -0.113716D+0,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,&
&        0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,&
&        0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     , -0.064434D+0, -0.098991D+0,&
&       -0.132163D+0, -0.164874D+0, -0.197477D+0, -0.230140D+0, -0.262953D+0, -0.295967D+0,&
&       -0.252966D+0, -0.362705D+0,  0.081292D+0,  0.072602D+0,  0.073863D+0,  0.081795D+0/)
      real(8) :: uhubd(21:84)= &
&     (/                             0.322610D+0,  0.351019D+0,  0.376535D+0,  0.312190D+0,&
&        0.422038D+0,  0.442914D+0,  0.462884D+0,  0.401436D+0,  0.420670D+0,  0.518772D+0,&
&        0.051561D+0,  0.101337D+0,  0.127856D+0,  0.165858D+0,  0.189059D+0,  0.200972D+0,&
&        0.180808D+0,  0.234583D+0,  0.239393D+0,  0.269067D+0,  0.294607D+0,  0.317562D+0,&
&        0.338742D+0,  0.329726D+0,  0.350167D+0,  0.369605D+0,  0.388238D+0,  0.430023D+0,&
&        0.156519D+0,  0.171708D+0,  0.184848D+0,  0.195946D+0,  0.206534D+0,  0.211949D+0,&
&        0.159261D+0,  0.199559D+0,  0.220941D+0,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,&
&        0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,&
&        0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.0D+0     ,  0.220882D+0,  0.249246D+0,&
&        0.273105D+0,  0.294154D+0,  0.313288D+0,  0.331031D+0,  0.347715D+0,  0.363569D+0,&
&        0.361156D+0,  0.393392D+0,  0.119520D+0,  0.128603D+0,  0.142210D+0,  0.158136D+0/)
!
      iao= 0
      ishell= 0
      do iatom= 1,natom
        select case(numatomic(iatom))
! H - He
            case(1:2)
              iao= iao+1
              energy(iao)= dftbips(numatomic(iatom))
              ishell= ishell+1
              uhub(ishell)= uhubs(numatomic(iatom))
! Li - Ar
            case(3:18)
              iao= iao+1
              energy(iao)= dftbips(numatomic(iatom))
              do i= 1,3
                iao= iao+1
                energy(iao)= dftbipp(numatomic(iatom))
              enddo
              ishell= ishell+1
              uhub(ishell)= uhubs(numatomic(iatom))
              ishell= ishell+1
              uhub(ishell)= uhubp(numatomic(iatom))
! K  - Ca, Ga - Kr
            case(19:20,31:36)
              iao= iao+1
              energy(iao)= dftbips(numatomic(iatom))
              do i= 1,3
                iao= iao+1
                energy(iao)= dftbipp(numatomic(iatom))
              enddo
              ishell= ishell+1
              uhub(ishell)= uhubs(numatomic(iatom))
              ishell= ishell+1
              uhub(ishell)= uhubp(numatomic(iatom))
! Sc - Zn
            case(21:30)
              iao= iao+1
              energy(iao)= dftbips(numatomic(iatom))
              do i= 1,3
                iao= iao+1
                energy(iao)= dftbipp(numatomic(iatom))
              enddo
              do i= 1,5
                iao= iao+1
                energy(iao)= dftbipd(numatomic(iatom))
              enddo
              ishell= ishell+1
              uhub(ishell)= uhubs(numatomic(iatom))
              ishell= ishell+1
              uhub(ishell)= uhubp(numatomic(iatom))
              ishell= ishell+1
              uhub(ishell)= uhubd(numatomic(iatom))
! Rb - Sr, In - Xe
            case(37:38,49:54)
              iao= iao+1
              energy(iao)= dftbips(numatomic(iatom))
              do i= 1,3
                iao= iao+1
                energy(iao)= dftbipp(numatomic(iatom))
              enddo
              ishell= ishell+1
              uhub(ishell)= uhubs(numatomic(iatom))
              ishell= ishell+1
              uhub(ishell)= uhubp(numatomic(iatom))
! Y  - Cd
            case(39:48)
              iao= iao+1
              energy(iao)= dftbips(numatomic(iatom))
              do i= 1,3
                iao= iao+1
                energy(iao)= dftbipp(numatomic(iatom))
              enddo
              do i= 1,5
                iao= iao+1
                energy(iao)= dftbipd(numatomic(iatom))
              enddo
              ishell= ishell+1
              uhub(ishell)= uhubs(numatomic(iatom))
              ishell= ishell+1
              uhub(ishell)= uhubp(numatomic(iatom))
              ishell= ishell+1
              uhub(ishell)= uhubd(numatomic(iatom))
! Cs - Ba
            case(55:56)
              iao= iao+1
              energy(iao)= dftbips(numatomic(iatom))
              ishell= ishell+1
              uhub(ishell)= uhubs(numatomic(iatom))
! La
! no d function in bshuzmini6_g
!           case(57)
!             iao= iao+1
!             energy(iao)= dftbips(numatomic(iatom))
!             do i= 1,5
!               iao= iao+1
!               energy(iao)= dftbipd(numatomic(iatom))
!             enddo
!             ishell= ishell+1
!             uhub(ishell)= uhubs(numatomic(iatom))
!             ishell= ishell+1
!             uhub(ishell)= uhubd(numatomic(iatom))
! Lu - Hg
            case(71:80)
              iao= iao+1
              energy(iao)= dftbips(numatomic(iatom))
              do i= 1,5
                iao= iao+1
                energy(iao)= dftbipd(numatomic(iatom))
              enddo
              ishell= ishell+1
              uhub(ishell)= uhubs(numatomic(iatom))
              ishell= ishell+1
              uhub(ishell)= uhubd(numatomic(iatom))
! Tl - Po
            case(81:84)
              iao= iao+1
              energy(iao)= dftbips(numatomic(iatom))
              do i= 1,3
                iao= iao+1
                energy(iao)= dftbipp(numatomic(iatom))
              enddo
              ishell= ishell+1
              uhub(ishell)= uhubs(numatomic(iatom))
              ishell= ishell+1
              uhub(ishell)= uhubp(numatomic(iatom))
!
            case default
              if(master) write(*,'(" Error! This program supports H-La and Lu-Po in dftbip.")')
              call iabort
        end select
      enddo
!
      return
end

