! Copyright 2014-2017  Kazuya Ishimura
!
! Licensed under the Apache License, Version 2.0 (the "License");
! you may not use this file except in compliance with the License.
! You may obtain a copy of the License at
!
!     http://www.apache.org/licenses/LICENSE-2.0
!
! Unless required by applicable law or agreed to in writing, software
! distributed under the License is distributed on an "AS IS" BASIS,
! WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
! See the License for the specific language governing permissions and
! limitations under the License.
!
!------------------------------
  subroutine setecp(mpi_comm)
!------------------------------
!
! Driver of setting ecp functions
!
      use modparallel, only : master
      use modecp, only : ecp, maxangecp, izcore, locecp, mprimecp, execp, coeffecp, mtypeecp
      use modmolecule, only : natom, znuc
      implicit none
      integer,intent(in) :: mpi_comm
      integer :: iatom, iprim
!
      maxangecp(1:natom)= -1
      izcore(1:natom)= 0
      locecp(:,1:natom)= 0
      mprimecp(:,1:natom)= 0
      iprim= 0
!
      if(master) then
        if((ecp == 'LANL2DZ').or.(ecp == 'LANL2MB')) then
          do iatom= 1,natom
            call ecplanl2(iatom,iprim)
          enddo
        elseif(ecp == 'GEN') then
          call readecp
          call setgenecp(iprim)
        else
          write(*,'(" Error! This program does not support ECP function,",a10,".")')ecp
          call iabort
        endif
      endif
!
      call para_bcasti(iprim,1,0,mpi_comm)
      call para_bcasti(maxangecp,natom,0,mpi_comm)
      call para_bcasti(izcore,natom,0,mpi_comm)
      call para_bcasti(locecp(0,1),natom*6,0,mpi_comm)
      call para_bcasti(mprimecp(0,1),natom*6,0,mpi_comm)
      call para_bcastr(execp,iprim,0,mpi_comm)
      call para_bcastr(coeffecp,iprim,0,mpi_comm)
      call para_bcasti(mtypeecp,iprim,0,mpi_comm)
! 
      do iatom= 1,natom
        znuc(iatom)= znuc(iatom)-izcore(iatom)
      enddo
!
      return
end


!------------------------------
  subroutine setgenecp(iprim)
!------------------------------
!
! Driver of setting ECP functions from input file
! This routine should be called only from master node.
!
      use modmolecule, only : natom, numatomic
      use modecp, only : maxangecp, izcore, mtypeecp, locecp, mprimecp, execp, coeffecp, &
&                        maxgenangecp, izgencore, mgentypeecp, locgenecp, mgenprimecp, &
&                        exgenecp, coeffgenecp, atomecp
      implicit none
      integer,intent(inout) :: iprim
      integer :: iatom, nn, lmax, iang, numprim, locgen, ii
!
      iprim= 0
!
      do iatom= 1,natom
        nn= numatomic(iatom)
        select case(atomecp(nn))
          case('LANL2DZ','LANL2MB')
            call ecplanl2(iatom,iprim)
          case('')
          case default
            write(*,'(" Error! This program does not support ECP function ",a10,".")')atomecp(nn)
            call iabort
        end select
!
        lmax= maxgenangecp(nn)
        if(lmax /= -1) then
          maxangecp(iatom)= lmax
          izcore(iatom)= izgencore(nn)
          do iang= 0,lmax
            numprim= mgenprimecp(iang,nn)
            locgen= locgenecp(iang,nn)
            locecp(iang,iatom)= iprim
            mprimecp(iang,iatom)= numprim
            do ii= 1,numprim
              iprim= iprim+1
              execp(iprim)= exgenecp(locgen+ii)
              coeffecp(iprim)= coeffgenecp(locgen+ii)
              mtypeecp(iprim)= mgentypeecp(locgen+ii)
            enddo
          enddo
        endif
      enddo
!
      return
end


!-----------------------------------
  subroutine ecplanl2(iatom,iprim)
!-----------------------------------
!
! Set Hay-Wadt ECP functions
!
      use modmolecule, only : numatomic
      use modparam, only : mxprim
      use modecp, only : maxangecp, izcore, mtypeecp, locecp, mprimecp, execp, coeffecp
      implicit none
      integer,intent(in) :: iatom
      integer,intent(inout) :: iprim
      integer :: nangecp3(16)=(/1,2,2,2,2, 0,1,2,2,2, 0,1,2,2,2,2/)
      integer :: iecp4(19:36)=(/  0, 13, 26, 40, 53, 66, 79, 92,105, &
&                               116,127,138,156,176,196,216,236,254/)
      integer :: iecp5(37:54)=(/  0, 18, 37, 55, 72, 89,106,123,142, &
&                               161,181,201,223,243,263,284,305,325/)
      integer :: iecp6(55:83)
      data iecp6(55:57) /  0, 17, 37/
      data iecp6(72:83) / 56, 84,112,140,168,194,220,245,270,299,328,358/
      integer :: j, numecp(0:4), loctmp, nangecp4(20), nangecp5(22), nangecp6(30)
      real(8) :: ecpex3(16,11:18)
      data ecpex3 &
&      /0.17555025900D+03,0.35051679100D+02,0.79060270000D+01,0.23365719000D+01,0.77998670000D+00,&
&       0.24336058460D+03,0.41576475900D+02,0.13264916700D+02,0.36797165000D+01,0.97642090000D+00,&
&       0.12572650682D+04,0.18962488100D+03,0.54524775900D+02,0.13744995500D+02,0.36813579000D+01,&
&       0.94611060000D+00,0.23754848040D+03,0.47752036700D+02,0.10783785200D+02,0.31992124000D+01,&
&       0.10636953000D+01,0.34830086310D+03,0.59468013300D+02,0.19076758200D+02,0.52965613000D+01,&
&       0.13867373000D+01,0.12568739085D+04,0.18986088390D+03,0.54694963100D+02,0.13899013700D+02,&
&       0.39597181000D+01,0.12552787000D+01,0.30472919260D+03,0.61529976800D+02,0.13925900600D+02,&
&       0.41463626000D+01,0.13715443000D+01,0.46784377560D+03,0.79399221600D+02,0.25403596700D+02,&
&       0.69954696000D+01,0.17860129000D+01,0.77627171900D+03,0.11849922540D+03,0.34410727600D+02,&
&       0.87859563000D+01,0.23406228000D+01,0.73983860000D+00,0.50531376930D+03,0.10322210260D+03,&
&       0.23456924800D+02,0.67505693000D+01,0.21603140000D+01,0.68949107190D+03,0.11417285100D+03,&
&       0.35742433600D+02,0.94529586000D+01,0.22543590000D+01,0.88937935500D+02,0.76777353800D+02,&
&       0.56148098700D+02,0.21187401400D+02,0.68277276000D+01,0.21001192000D+01,0.46212114230D+03,&
&       0.93686370100D+02,0.21234909400D+02,0.63388415000D+01,0.20620684000D+01,0.78083182300D+02,&
&       0.58957681000D+02,0.36057125500D+02,0.11246445300D+02,0.26757561000D+01,0.75161788000D+02,&
&       0.57454404100D+02,0.47948174800D+02,0.18458836000D+02,0.59414190000D+01,0.18487507000D+01,&
&       0.53266852220D+03,0.10813422480D+03,0.24569766400D+02,0.73702438000D+01,0.23712569000D+01,&
&       0.10631767810D+03,0.10082458330D+03,0.53585847200D+02,0.15370633200D+02,0.31778402000D+01,&
&       0.10197091850D+03,0.93280897300D+02,0.65143177200D+02,0.24634744000D+02,0.78120535000D+01,&
&       0.23112730000D+01,0.94813000000D+02,0.16564400000D+03,0.30831700000D+02,0.10584100000D+02,&
&       0.37704000000D+01,0.12883910000D+03,0.12037860000D+03,0.63562200000D+02,0.18069500000D+02,&
&       0.38142000000D+01,0.21652630000D+03,0.46572300000D+02,0.14746850000D+03,0.48986900000D+02,&
&       0.13209600000D+02,0.31831000000D+01,0.71152421750D+03,0.14467086890D+03,0.32924699200D+02,&
&       0.99103877000D+01,0.31328926000D+01,0.15285530330D+03,0.82142479200D+02,0.83715480000D+02,&
&       0.23855716100D+02,0.43128823000D+01,0.21255737470D+03,0.26648847120D+03,0.13992224770D+03,&
&       0.50109765900D+02,0.14553727600D+02,0.34828623000D+01/
      real(8) :: ecpcoeff3(16,11:18)
      data ecpcoeff3 &
&      /-.10000000000D+02,-.47490202400D+02,-.17228300700D+02,-.60637782000D+01,-.72993930000D+00,&
&       0.30000000000D+01,0.36284762600D+02,0.72930488000D+02,0.23840115100D+02,0.60123861000D+01,&
&       0.50000000000D+01,0.11744956830D+03,0.42339867040D+03,0.10932472970D+03,0.31370165600D+02,&
&       0.71241813000D+01,-.10000000000D+02,-.55899396800D+02,-.20139195700D+02,-.70679107000D+01,&
&       -.81331090000D+00,0.30000000000D+01,0.44007566000D+02,0.10738613440D+03,0.35828908800D+02,&
&       0.10114343500D+02,0.50000000000D+01,0.11710536720D+03,0.42059720730D+03,0.10761229590D+03,&
&       0.29100257600D+02,0.70875570000D+01,-.10000000000D+02,-.63807983700D+02,-.22897217400D+02,&
&       -.80063232000D+01,-.88293450000D+00,0.30000000000D+01,0.50970568200D+02,0.14387164600D+03,&
&       0.48005575300D+02,0.14011418000D+02,0.50000000000D+01,0.93059542400D+02,0.26676863290D+03,&
&       0.70072580500D+02,0.17229083900D+02,0.71053310000D+00,-.10000000000D+02,-.84923608700D+02,&
&       -.30329941000D+02,-.12104904600D+02,-.18945408000D+01,0.30000000000D+01,0.60520683300D+02,&
&       0.20130861370D+03,0.65939974500D+02,0.19030078900D+02,0.50000000000D+01,0.66413817000D+01,&
&       0.24759724160D+03,0.12937153800D+03,0.47461710700D+02,0.11737657400D+02,-.10000000000D+02,&
&       -.79486465800D+02,-.28366825100D+02,-.98577589000D+01,-.10163783000D+01,0.30000000000D+01,&
&       0.12910415400D+02,0.15002502980D+03,0.71708314600D+02,0.23039701200D+02,0.50000000000D+01,&
&       0.63446507000D+01,0.19855851040D+03,0.11114708200D+03,0.40394414400D+02,0.64483233000D+01,&
&       -.10000000000D+02,-.85359384600D+02,-.30451329000D+02,-.10374588600D+02,-.98992950000D+00,&
&       0.30000000000D+01,0.10628403600D+02,0.22363604690D+03,0.93646084500D+02,0.28760906500D+02,&
&       0.50000000000D+01,0.60969842000D+01,0.28544255000D+03,0.14714484130D+03,0.53656977800D+02,&
&       0.89249559000D+01,-.10000000000D+02,0.66272917000D+02,-.28968595000D+02,-.12866337000D+02,&
&       -.17102170000D+01,0.30000000000D+01,0.12852851000D+02,0.27567239800D+03,0.11567771200D+03,&
&       0.35060609000D+02,0.50000000000D+01,0.74794860000D+01,0.61303200000D+03,0.28080068500D+03,&
&       0.10787882400D+03,0.15343956000D+02,-.10000000000D+02,-.99060666900D+02,-.35271176700D+02,&
&       -.11815194700D+02,-.10453382000D+01,0.30000000000D+01,0.11246562100D+02,0.25074128120D+03,&
&       0.13916065430D+03,0.41289798100D+02,0.50000000000D+01,0.52787358000D+01,0.63171351660D+03,&
&       0.30516498090D+03,0.10648076150D+03,0.17021576500D+02/
      real(8) :: ecpex4(272)= &
&     (/0.10671081575D+04,0.21841851420D+03,0.50043350200D+02,0.14937450000D+02,0.50996937000D+01,&
&       0.16550398800D+02,0.19164159100D+02,0.48910684000D+01,0.95364585500D+02,0.11912090620D+03,&
&       0.57079581500D+02,0.17445091000D+02,0.44636131000D+01,0.12704543808D+04,0.26024950570D+03,&
&       0.59706850100D+02,0.17681815000D+02,0.59668207000D+01,0.18602991200D+02,0.21413897100D+02,&
&       0.56851879000D+01,0.13476872330D+03,0.18184580040D+03,0.80924013200D+02,0.23380699300D+02,&
&       0.51868564000D+01,0.23731665280D+03,0.42648900400D+02,0.10621122900D+02,0.69255232400D+02,&
&       0.58895341600D+02,0.27263745900D+02,0.54737304000D+01,0.51523607000D+01,0.77251297000D+02,&
&       0.44452715500D+02,0.53685475200D+02,0.18942117000D+02,0.47970573000D+01,0.44142925000D+01,&
&       0.26532639090D+03,0.47768781500D+02,0.11890333400D+02,0.81473069600D+02,0.72649672400D+02,&
&       0.31812821300D+02,0.61664468000D+01,0.58268347000D+01,0.50296694300D+02,0.63508975400D+02,&
&       0.26099608400D+02,0.56022573000D+01,0.52171069000D+01,0.29644875820D+03,0.53501505200D+02,&
&       0.13320451000D+02,0.93219365400D+02,0.71102956700D+02,0.33876699300D+02,0.68691200000D+01,&
&       0.64804722000D+01,0.52095397700D+02,0.65677841100D+02,0.27470376400D+02,0.62352020000D+01,&
&       0.58139909000D+01,0.32927205390D+03,0.59548759700D+02,0.14826596600D+02,0.10002606530D+03,&
&       0.67846812400D+02,0.34677100600D+02,0.76757005000D+01,0.72077427000D+01,0.47972332100D+02,&
&       0.59622487300D+02,0.26043260000D+02,0.65582692000D+01,0.61268236000D+01,0.35739144690D+03,&
&       0.64647738900D+02,0.16096083300D+02,0.10741272150D+03,0.11149589730D+03,0.46556834600D+02,&
&       0.83688135000D+01,0.77237489000D+01,0.80041510300D+02,0.10560436460D+03,0.40830046600D+02,&
&       0.80098457000D+01,0.73390928000D+01,0.39261497870D+03,0.71175697900D+02,0.17732028100D+02,&
&       0.12605718950D+03,0.13812642510D+03,0.54209885800D+02,0.92837966000D+01,0.86289082000D+01,&
&       0.83175949000D+02,0.10605599380D+03,0.42828493700D+02,0.87701805000D+01,0.80397818000D+01,&
&       0.43410761040D+03,0.78840275800D+02,0.19603774900D+02,0.15123075820D+03,0.16967900260D+03,&
&       0.64990942300D+02,0.12507628700D+02,0.92731992400D+02,0.12119101830D+03,0.48626590900D+02,&
&       0.11996697100D+02,0.46993243310D+03,0.85423641100D+02,0.21267498400D+02,0.16216860970D+03,&
&       0.17653332320D+03,0.68956201000D+02,0.13579283800D+02,0.69018150600D+02,0.27559555960D+03,&
&       0.47131545300D+02,0.12987407500D+02,0.51199517630D+03,0.93280107400D+02,0.23220666900D+02,&
&       0.17311808540D+03,0.18524198860D+03,0.73151784700D+02,0.14688415700D+02,0.10071913690D+03,&
&       0.13083456650D+03,0.53868372000D+02,0.14098946900D+02,0.38673796600D+03,0.72858735900D+02,&
&       0.15906617000D+02,0.43502340000D+01,0.12842199000D+01,0.19086785800D+02,0.50231080000D+01,&
&       0.12701744000D+01,0.10671287000D+01,0.92641900000D+00,0.43492775000D+02,0.20869266900D+02,&
&       0.21711837800D+02,0.63616915000D+01,0.12291195000D+01,0.13585180000D+02,0.98373050000D+01,&
&       0.83731130000D+00,0.26090511840D+03,0.50424456400D+02,0.10719552000D+02,0.31563995000D+01,&
&       0.95023850000D+00,0.33905759950D+03,0.95962522700D+02,0.30098936000D+02,0.78907039000D+01,&
&       0.21154234000D+01,0.30120270800D+02,0.14614382500D+02,0.12463943300D+02,0.51039219000D+01,&
&       0.16566033000D+01,0.32270430600D+02,0.10944400400D+02,0.45111791000D+01,0.15561710000D+01,&
&       0.55509260000D+00,0.31821675830D+03,0.61537096700D+02,0.13298689900D+02,0.38985215000D+01,&
&       0.12137666000D+01,0.20518869320D+03,0.68979027800D+02,0.27919487900D+02,0.85481650000D+01,&
&       0.23173734000D+01,0.33248800200D+02,0.15777724700D+02,0.14926072200D+02,0.58416394000D+01,&
&       0.18349575000D+01,0.42020634300D+02,0.19209636300D+02,0.94133917000D+01,0.33282907000D+01,&
&       0.85223310000D+00,0.37547488030D+03,0.72655476900D+02,0.15916267700D+02,0.46420251000D+01,&
&       0.14789182000D+01,0.10588538700D+03,0.62595184600D+02,0.34501743100D+02,0.10375859400D+02,&
&       0.25379347000D+01,0.37647144000D+02,0.17203466500D+02,0.18907570900D+02,0.68153452000D+01,&
&       0.20611853000D+01,0.52073133500D+02,0.28003782400D+02,0.14705120800D+02,0.47973194000D+01,&
&       0.11062763000D+01,0.43319313360D+03,0.83895215700D+02,0.18583913900D+02,0.53955286000D+01,&
&       0.17474326000D+01,0.20289861930D+03,0.78382048700D+02,0.35075303700D+02,0.10876954300D+02,&
&       0.28005941000D+01,0.44301187500D+02,0.20387420600D+02,0.23188994800D+02,0.79777664000D+01,&
&       0.22988146000D+01,0.73362826300D+02,0.48383561800D+02,0.25629721100D+02,0.71705822000D+01,&
&       0.13639538000D+01,0.21361439690D+03,0.41058538000D+02,0.87086530000D+01,0.26074661000D+01,&
&       0.54198068200D+02,0.32905355800D+02,0.13674489000D+02,0.30341152000D+01,0.54256334000D+02,&
&       0.26009559300D+02,0.28201299500D+02,0.94341061000D+01,0.25321764000D+01,0.87632872100D+02,&
&       0.61737337700D+02,0.32438510400D+02,0.87537199000D+01,0.16633189000D+01,0.23754708530D+03,&
&       0.45740675400D+02,0.97073237000D+01,0.29320738000D+01,0.70760691800D+02,0.33680974500D+02,&
&       0.13807528100D+02,0.33188254000D+01,0.57767680400D+02,0.27677945300D+02,0.30475744900D+02,&
&       0.10364263900D+02,0.28470827000D+01,0.85390227700D+02,0.36078199300D+02,0.38038766800D+02,&
&       0.10606988200D+02,0.19390148000D+01/)
      real(8) :: ecpcoeff4(272)= &
&     (/-.10000000000D+02,-.12434693060D+03,-.43993746800D+02,-.16594615800D+02,-.18728686000D+01,&
&       0.30000000000D+01,0.10482860900D+02,0.45456156900D+02,0.50000000000D+01,0.55748307000D+01,&
&       0.30964931890D+03,0.12559518090D+03,0.23197764900D+02,-.10000000000D+02,-.13636841900D+03,&
&       -.48340360000D+02,-.18514190200D+02,-.22114740000D+01,0.30000000000D+01,0.11722802700D+02,&
&       0.54211756100D+02,0.50000000000D+01,0.51639340000D+01,0.44286860220D+03,0.17502559890D+03,&
&       0.28821743400D+02,-.10000000000D+02,-.48969555300D+02,-.87066044000D+01,0.30000000000D+01,&
&       0.17058455900D+02,0.16795954120D+03,0.17983821130D+03,-.12115231440D+03,0.50000000000D+01,&
&       0.61088195000D+01,0.19293849140D+03,0.12722747550D+03,0.68094597400D+02,-.41646910800D+02,&
&       -.10000000000D+02,-.51842781600D+02,-.91429145000D+01,0.30000000000D+01,0.19482557900D+02,&
&       0.20733492790D+03,0.23567445010D+03,-.16687843870D+03,0.50000000000D+01,0.55348822000D+01,&
&       0.17784193840D+03,0.10742071530D+03,-.71906590200D+02,-.10000000000D+02,-.54890980800D+02,&
&       -.97010586000D+01,0.30000000000D+01,0.22534568900D+02,0.21656340090D+03,0.25268225380D+03,&
&       -.17626716540D+03,0.50000000000D+01,0.57003555000D+01,0.18290542740D+03,0.11727562260D+03,&
&       -.78582913400D+02,-.10000000000D+02,-.57940939800D+02,-.10262328500D+02,0.30000000000D+01,&
&       0.25253459900D+02,0.21064723660D+03,0.23528195870D+03,-.15253434610D+03,0.50000000000D+01,&
&       0.56390266000D+01,0.16483621980D+03,0.10601680180D+03,-.69858546800D+02,-.10000000000D+02,&
&       -.60299028700D+02,-.10421783400D+02,0.30000000000D+01,0.16259181900D+02,0.27693739280D+03,&
&       0.24131743420D+03,-.14646353290D+03,0.50000000000D+01,0.57589756000D+01,0.28529186540D+03,&
&       0.14342226470D+03,-.88703185100D+02,-.10000000000D+02,-.63266751800D+02,-.10961333800D+02,&
&       0.30000000000D+01,0.18172913700D+02,0.33912311640D+03,0.31710680120D+03,-.20734216490D+03,&
&       0.50000000000D+01,0.59535930000D+01,0.29426655270D+03,0.15442446350D+03,-.95316424900D+02,&
&       -.10000000000D+02,-.66724027600D+02,-.11719224300D+02,0.30000000000D+01,0.20155311000D+02,&
&       0.41337622440D+03,0.13387277050D+03,0.50000000000D+01,0.61892353000D+01,0.32632159190D+03,&
&       0.73878897000D+02,-.10000000000D+02,-.69408480500D+02,-.12095102000D+02,0.30000000000D+01,&
&       0.22025361800D+02,0.44301810880D+03,0.14556964110D+03,0.50000000000D+01,0.49882824000D+01,&
&       0.25669458530D+03,0.78475445000D+02,-.10000000000D+02,-.72554828200D+02,-.12745023100D+02,&
&       0.30000000000D+01,0.23835182500D+02,0.47389304880D+03,0.15763458230D+03,0.50000000000D+01,&
&       0.64990936000D+01,0.35146053950D+03,0.85501603600D+02,-.18000000000D+02,-.12435274030D+03,&
&       -.30660182200D+02,-.10635898900D+02,-.76836230000D+00,0.30000000000D+01,0.22523422500D+02,&
&       0.48446594200D+02,-.44556011900D+02,0.12998395800D+02,0.50000000000D+01,0.20743558900D+02,&
&       0.90302715800D+02,0.74661031600D+02,0.98894424000D+01,-.48490359000D+01,0.36913379000D+01,&
&       -.50373190000D+00,-.28000000000D+02,-.16162486340D+03,-.49265975100D+02,-.16930594200D+02,&
&       -.15149893000D+01,0.30000000000D+01,0.93859744900D+02,0.30864954850D+03,0.94729060700D+02,&
&       0.27093106600D+02,0.50000000000D+01,0.22849531500D+02,0.43397490700D+02,0.48620392300D+02,&
&       0.14736548300D+02,0.30000000000D+01,0.24299483400D+02,0.35441581300D+02,0.91055575000D+01,&
&       0.96011210000D+00,-.28000000000D+02,-.18098916760D+03,-.55004390900D+02,-.19790652600D+02,&
&       -.18533572000D+01,0.30000000000D+01,0.65226255800D+02,0.22523545220D+03,0.94012547200D+02,&
&       0.29941500500D+02,0.50000000000D+01,0.23477815700D+02,0.45098041400D+02,0.56332695700D+02,&
&       0.16605864000D+02,0.30000000000D+01,0.23737151800D+02,0.56479224900D+02,0.25890183500D+02,&
&       0.30229836000D+01,-.28000000000D+02,-.19836873570D+03,-.60456379800D+02,-.22318827400D+02,&
&       -.21638290000D+01,0.30000000000D+01,0.31848677700D+02,0.20099582150D+03,0.10263876040D+03,&
&       0.34184535900D+02,0.50000000000D+01,0.24250574700D+02,0.48917496300D+02,0.66533225900D+02,&
&       0.19657766600D+02,0.30000000000D+01,0.23346071200D+02,0.80785491900D+02,0.40268134700D+02,&
&       0.48332790000D+01,-.28000000000D+02,-.21438417620D+03,-.65691878200D+02,-.24615393200D+02,&
&       -.24481497000D+01,0.30000000000D+01,0.62029539000D+02,0.25885559840D+03,0.11878001530D+03,&
&       0.38235527900D+02,0.50000000000D+01,0.24797345800D+02,0.63757564000D+02,0.79051283100D+02,&
&       0.22952018300D+02,0.30000000000D+01,0.22470590700D+02,0.14054928870D+03,0.63578183500D+02,&
&       0.70753614000D+01,-.28000000000D+02,-.13492688520D+03,-.41927191300D+02,-.59336420000D+01,&
&       0.30000000000D+01,0.27343064200D+02,0.11880288470D+03,0.43435487600D+02,0.50000000000D+01,&
&       0.25050425200D+02,0.92615746300D+02,0.95824901600D+02,0.26268498300D+02,0.30000000000D+01,&
&       0.22553355700D+02,0.17812419880D+03,0.76992416200D+02,0.94818270000D+01,-.28000000000D+02,&
&       -.14342599550D+03,-.44372383800D+02,-.62911252000D+01,0.30000000000D+01,0.37189364700D+02,&
&       0.13229739580D+03,0.47934698500D+02,0.50000000000D+01,0.26156400800D+02,0.96317650100D+02,&
&       0.10366327940D+03,0.31077931500D+02,0.30000000000D+01,0.24435653700D+02,0.85722940200D+02,&
&       0.91697057800D+02,0.11830070900D+02/)
      real(8) :: ecpex5(346)= &
&     (/0.87959836640D+03,0.14201886950D+03,0.40934688200D+02,0.99084802000D+01,0.31556041000D+01,&
&       0.70980721500D+02,0.38694980700D+02,0.15908203500D+02,0.37152659000D+01,0.93305868400D+02,&
&       0.47122137000D+02,0.48805064000D+02,0.14378494900D+02,0.30547807000D+01,0.33047743000D+02,&
&       0.99601880000D+01,0.37403577000D+01,0.81242080000D+00,0.78238046310D+03,0.12465423380D+03,&
&       0.36987496600D+02,0.98828819000D+01,0.32829588000D+01,0.59324063100D+02,0.55203847200D+02,&
&       0.20469209200D+02,0.39588141000D+01,0.92120199100D+02,0.46813255900D+02,0.48656643200D+02,&
&       0.14950323800D+02,0.34268785000D+01,0.65829130100D+02,0.32728262100D+02,0.21114603000D+02,&
&       0.91071292000D+01,0.28110754000D+01,0.57843103490D+03,0.15277920040D+03,0.44930152400D+02,&
&       0.11458791800D+02,0.37523267000D+01,0.59471511400D+02,0.17217355300D+02,0.18479709300D+02,&
&       0.43276192000D+01,0.45727100500D+02,0.49459588600D+02,0.18995237300D+02,0.36603193000D+01,&
&       0.62826843500D+02,0.31889790400D+02,0.18364657200D+02,0.73062400000D+01,0.24051635000D+01,&
&       0.64593218730D+03,0.13475474010D+03,0.42307461900D+02,0.12000322700D+02,0.41260454000D+01,&
&       0.11762518620D+03,0.22964608900D+02,0.45225298000D+01,0.47195314500D+02,0.48035603300D+02,&
&       0.19454145600D+02,0.40512875000D+01,0.79907398300D+02,0.45826379800D+02,0.26990352200D+02,&
&       0.96835718000D+01,0.27995666000D+01,0.34296384050D+03,0.13993080140D+03,0.43352340500D+02,&
&       0.12444853300D+02,0.43204837000D+01,0.11286306170D+03,0.22967867600D+02,0.49340673000D+01,&
&       0.63680196300D+02,0.23091242900D+02,0.24428364700D+02,0.42310090000D+01,0.99335963600D+02,&
&       0.64058647200D+02,0.37089101100D+02,0.12070857400D+02,0.31575323000D+01,0.53796678070D+03,&
&       0.14789829380D+03,0.45735889800D+02,0.13291146700D+02,0.47059961000D+01,0.11029917600D+03,&
&       0.23201464500D+02,0.53530131000D+01,0.63290139700D+02,0.23331530200D+02,0.24675942300D+02,&
&       0.46493040000D+01,0.10448399770D+03,0.66230724500D+02,0.39128317600D+02,0.13116443700D+02,&
&       0.36280263000D+01,0.50111994090D+03,0.15024688440D+03,0.46709791400D+02,0.13933901400D+02,&
&       0.49854069000D+01,0.10817759950D+03,0.23659435100D+02,0.57814977000D+01,0.63306201400D+02,&
&       0.23693553200D+02,0.25052585400D+02,0.50717084000D+01,0.10277213970D+03,0.45671375600D+02,&
&       0.48261199700D+02,0.15708911500D+02,0.39484576000D+01,0.55437963030D+03,0.15510668710D+03,&
&       0.48497626300D+02,0.14770159400D+02,0.52077363000D+01,0.66711806000D+02,0.77350363200D+02,&
&       0.18357144500D+02,0.11840472700D+02,0.81179479000D+01,0.54993791500D+02,0.13939921200D+02,&
&       0.15211824600D+02,0.10546069100D+02,0.75539486000D+01,0.60344459500D+02,0.45210030500D+02,&
&       0.19119007400D+02,0.42712090000D+01,0.60032430320D+03,0.15769101760D+03,0.49884199500D+02,&
&       0.15596689500D+02,0.55099296000D+01,0.59344252600D+02,0.83742606100D+02,0.18453024800D+02,&
&       0.12419460600D+02,0.88172913000D+01,0.53430906800D+02,0.65667184300D+02,0.16836986200D+02,&
&       0.11304213600D+02,0.80312444000D+01,0.64399365300D+02,0.43462505300D+02,0.19402030100D+02,&
&       0.46879328000D+01,0.59833364440D+03,0.16242982900D+03,0.51571477100D+02,0.16488826000D+02,&
&       0.58287656000D+01,0.73380630400D+02,0.14755043800D+02,0.17835020400D+02,0.12711147700D+02,&
&       0.93292063000D+01,0.55668987700D+02,0.64233777100D+02,0.17625495200D+02,0.11905815500D+02,&
&       0.85100832000D+01,0.49999472800D+02,0.39747754700D+02,0.11432136600D+02,0.91790080000D+01,&
&       0.75624429000D+01,0.56870062370D+03,0.16235790660D+03,0.51102575500D+02,0.16920582200D+02,&
&       0.61669596000D+01,0.76097465800D+02,0.15332735900D+02,0.18771534500D+02,0.13366329400D+02,&
&       0.98236948000D+01,0.56331804300D+02,0.69060909800D+02,0.19271799800D+02,0.12577065400D+02,&
&       0.87956670000D+01,0.53464107800D+02,0.40197545700D+02,0.11908607300D+02,0.97528183000D+01,&
&       0.81788997000D+01,0.35007624900D+03,0.81175777800D+02,0.24774781800D+02,0.12576033000D+02,&
&       0.49889712000D+01,0.18449910000D+01,0.53094940000D+00,0.54578619300D+02,0.16008231600D+02,&
&       0.52767462000D+01,0.97310700000D+00,0.93056460000D+00,0.41358176600D+02,0.12121498200D+02,&
&       0.34478806000D+01,0.93621750000D+00,0.85371049900D+02,0.73050078100D+02,0.28426288700D+02,&
&       0.39450158000D+01,0.35349379000D+01,0.64821940000D+00,0.18020710000D+00,0.33017652800D+02,&
&       0.97219749000D+01,0.23481578000D+01,0.72268530000D+00,0.17314032280D+03,0.58275243100D+02,&
&       0.21129430400D+02,0.59581061000D+01,0.14830272000D+01,0.35943803300D+02,0.11243108300D+02,&
&       0.35340830000D+01,0.14120937000D+01,0.12828947000D+01,0.42627681700D+02,0.14564933000D+02,&
&       0.55505378000D+01,0.18351485000D+01,0.52881220000D+00,0.21690210000D+00,0.33309718200D+02,&
&       0.98846832000D+01,0.25019593000D+01,0.78333230000D+00,0.15560622930D+03,0.56514546400D+02,&
&       0.22658364800D+02,0.65837080000D+01,0.15947702000D+01,0.40466670500D+02,0.13621844000D+02,&
&       0.41092688000D+01,0.15046137000D+01,0.13646762000D+01,0.60191505200D+02,0.20496114200D+02,&
&       0.84438328000D+01,0.26103790000D+01,0.65013170000D+00,0.22727235580D+03,0.77142988200D+02,&
&       0.26011650100D+02,0.91421159000D+01,0.29281082000D+01,0.92311360000D+00,0.11600108880D+03,&
&       0.59675506500D+02,0.27613584900D+02,0.78045081000D+01,0.16934418000D+01,0.27805079600D+02,&
&       0.10392968500D+02,0.36381067000D+01,0.19839681000D+01,0.15546562000D+01,0.83218303000D+02,&
&       0.30307050100D+02,0.13201412000D+02,0.36961611000D+01,0.77111440000D+00,0.28503232920D+03,&
&       0.75910579800D+02,0.26182785900D+02,0.90635141000D+01,0.29543257000D+01,0.91699420000D+00,&
&       0.18833275040D+03,0.68630692100D+02,0.27440219500D+02,0.78835282000D+01,0.18443362000D+01,&
&       0.54082279500D+02,0.28543471400D+02,0.16866085700D+02,0.55244176000D+01,0.14625225000D+01,&
&       0.34531264800D+02,0.11269936600D+02,0.22704361000D+01,0.17766992000D+01,0.14251666000D+01,&
&       0.10715702000D+01,0.44193602800D+02,0.12936760900D+02,0.31956412000D+01,0.85898060000D+00,&
&       0.12792026700D+03,0.78621146500D+02,0.36514623700D+02,0.99065681000D+01,0.19420086000D+01,&
&       0.13003530400D+02,0.76033140400D+02,0.24196168400D+02,0.64053433000D+01,0.15851786000D+01,&
&       0.40427810800D+02,0.28908437500D+02,0.15626893600D+02,0.41442856000D+01,0.93772350000D+00,&
&       0.44251383130D+03,0.10600055340D+03,0.32612374300D+02,0.94739667000D+01,0.31147195000D+01,&
&       0.91944720000D+00,0.14874007240D+03,0.74768743600D+02,0.34154841300D+02,0.96899675000D+01,&
&       0.21065947000D+01,0.81495157000D+01,0.11392356400D+03,0.35126643100D+02,0.81449579000D+01,&
&       0.17187138000D+01,0.49451838400D+02,0.38948091800D+02,0.20016159800D+02,0.50614904000D+01,&
&       0.10672963000D+01/)
      real(8) :: ecpcoeff5(346)= &
&     (/-.36436200000D-01,-.20803305200D+02,-.11546619100D+03,-.42506129300D+02,-.55247889000D+01,&
&       0.29747910000D+01,0.36117043500D+02,0.15227410680D+03,0.53715190600D+02,0.49551026000D+01,&
&       0.24272623600D+02,0.21523414110D+03,0.15166876660D+03,0.33875028000D+02,0.30465224000D+01,&
&       0.29424362400D+02,0.33443180800D+02,0.11102946000D+02,-.38432300000D-01,-.20617427100D+02,&
&       -.10117377440D+03,-.38774360300D+02,-.46479243000D+01,0.29989022000D+01,0.25655266900D+02,&
&       0.18318185330D+03,0.58438473900D+02,0.49551189000D+01,0.25447236700D+02,0.20380027800D+03,&
&       0.15505187400D+03,0.39360519200D+02,0.30056451000D+01,0.26706411900D+02,0.74575690100D+02,&
&       0.63174212100D+02,0.20296116200D+02,-.40481700000D-01,-.20619434400D+02,-.11675222790D+03,&
&       -.43797580600D+02,-.54247609000D+01,0.29801339000D+01,0.34783467600D+02,0.28845324600D+02,&
&       0.64764208800D+02,0.49885783000D+01,0.19650656400D+02,0.19409431810D+03,0.43134976900D+02,&
&       0.30066647000D+01,0.25987925000D+02,0.85717289700D+02,0.48779256800D+02,0.11453510400D+02,&
&       -.42584300000D-01,-.20222240900D+02,-.10186951720D+03,-.41619578400D+02,-.46986160000D+01,&
&       0.27910559000D+01,0.41948945900D+02,0.67727186600D+02,0.49911144000D+01,0.20719317200D+02,&
&       0.19558677580D+03,0.48287717600D+02,0.30049226000D+01,0.25937798900D+02,0.12512449340D+03,&
&       0.70763402200D+02,0.15049282200D+02,-.44740100000D-01,-.20053510000D+02,-.10372440210D+03,&
&       -.41045916500D+02,-.42046219000D+01,0.27964669000D+01,0.42884578600D+02,0.75187696600D+02,&
&       0.49461846000D+01,0.24831637600D+02,0.13791832920D+03,0.50977847100D+02,0.30064059000D+01,&
&       0.25534745800D+02,0.17895974520D+03,0.92957749000D+02,0.18474425600D+02,-.46949200000D-01,&
&       -.20208008400D+02,-.10621163020D+03,-.41810736800D+02,-.42054103000D+01,0.28063717000D+01,&
&       0.44516201200D+02,0.82778522700D+02,0.49420876000D+01,0.25860497600D+02,0.13247087420D+03,&
&       0.57314979400D+02,0.30054591000D+01,0.26363785100D+02,0.18338491990D+03,0.98445306800D+02,&
&       0.22490137700D+02,-.49211500000D-01,-.20146730800D+02,-.10557928800D+03,-.41590119700D+02,&
&       -.39415111000D+01,0.28222374000D+01,0.46733070800D+02,0.90531865700D+02,0.49390782000D+01,&
&       0.26922963100D+02,0.12850797960D+03,0.63749025000D+02,0.29921902000D+01,0.28182137900D+02,&
&       0.89054846500D+02,0.12118943250D+03,0.25586697300D+02,-.51527000000D-01,-.20181653600D+02,&
&       -.10599669150D+03,-.42216678800D+02,-.37675024000D+01,0.29578344000D+01,0.25374870700D+02,&
&       0.53612623720D+03,-.65120572210D+03,0.38138169430D+03,0.49651557000D+01,0.23886150100D+02,&
&       0.46446313440D+03,-.71444517880D+03,0.37755035940D+03,0.30352988000D+01,0.23290172300D+02,&
&       0.14609266200D+03,0.28912977000D+02,-.53895800000D-01,-.20131628200D+02,-.10536541210D+03,&
&       -.42327437000D+02,-.36654043000D+01,0.29753728000D+01,0.25123030600D+02,0.62609261450D+03,&
&       -.81225493850D+03,0.46737293400D+03,0.49537213000D+01,0.20487111600D+02,0.59801201390D+03,&
&       -.71840590280D+03,0.38281731510D+03,0.30279532000D+01,0.24752651600D+02,0.14268442890D+03,&
&       0.32140685700D+02,-.56317700000D-01,-.20128803600D+02,-.10581979230D+03,-.42573334500D+02,&
&       -.36165086000D+01,0.30003651000D+01,0.32435009300D+02,0.45908303830D+03,-.86806290290D+03,&
&       0.51447260980D+03,0.49593099000D+01,0.21171102900D+02,0.60505600920D+03,-.72696418460D+03,&
&       0.39632748830D+03,0.30508745000D+01,0.22250658000D+02,0.67483576980D+03,-.10408554048D+04,&
&       0.50593751470D+03,-.58793000000D-01,-.20114514600D+02,-.10427331140D+03,-.40453978700D+02,&
&       -.34420009000D+01,0.29861527000D+01,0.35157646000D+02,0.45018099060D+03,-.86602483080D+03,&
&       0.52311101760D+03,0.49640671000D+01,0.21502821900D+02,0.54602754530D+03,-.60038225560D+03,&
&       0.34829492890D+03,0.30467486000D+01,0.23365670500D+02,0.77725401170D+03,-.12388602423D+04,&
&       0.60806771210D+03,-.61321400000D-01,-.24935559300D+02,-.87772876500D+02,-.19066503000D+02,&
&       -.16190125400D+02,-.39441254000D+01,-.30974760000D+00,0.29050102000D+01,0.39628919800D+02,&
&       0.57443579000D+02,0.90235116600D+02,-.71713923600D+02,0.19585463000D+01,0.37566984400D+02,&
&       0.44087510400D+02,0.12567859900D+02,0.29753499000D+01,0.24705437300D+02,0.21821631430D+03,&
&       0.11899453700D+03,-.82499967100D+02,-.24137580000D+00,-.63903100000D-01,-.29908290100D+02,&
&       -.63344682400D+02,-.19005842800D+02,-.25161921000D+01,0.32301607000D+01,0.76780914000D+02,&
&       0.24547707600D+03,0.79289272100D+02,0.26408946000D+02,0.44690290000D+01,0.43647339900D+02,&
&       0.50756540600D+02,-.48673001700D+02,0.58769265300D+02,0.31020770000D+01,0.36544509000D+02,&
&       0.48804180800D+02,0.19436236100D+02,0.39855647000D+01,-.66538000000D-01,-.29746941600D+02,&
&       -.61830727800D+02,-.18941063400D+02,-.22384982000D+01,0.29482777000D+01,0.63761471800D+02,&
&       0.23052767880D+03,0.85071331400D+02,0.28821564400D+02,0.26941946000D+01,0.44827107900D+02,&
&       0.57622337000D+02,-.44685331200D+02,0.57135181500D+02,0.30259297000D+01,0.39534107100D+02,&
&       0.68468650400D+02,0.29112376400D+02,0.58428577000D+01,-.69226100000D-01,-.31543471200D+02,&
&       -.11584073060D+03,-.42022421200D+02,-.20046393800D+02,-.28796028000D+01,0.29331303000D+01,&
&       0.44110481900D+02,0.22801980500D+03,0.92561016200D+02,0.31253724400D+02,0.49947369000D+01,&
&       0.38022386000D+02,0.62052065400D+02,-.49548938900D+02,0.51366600000D+02,0.29738470000D+01,&
&       0.41649582500D+02,0.10459195620D+03,0.42318785700D+02,0.79413503000D+01,-.71967500000D-01,&
&       -.31374075400D+02,-.11297119960D+03,-.41959918500D+02,-.19127400600D+02,-.22917778000D+01,&
&       0.29286639000D+01,0.68962677300D+02,0.27696370630D+03,0.10447017650D+03,0.34838480500D+02,&
&       0.49732188000D+01,0.38898363400D+02,0.13731392460D+03,0.69737032300D+02,0.22748717800D+02,&
&       0.31548778000D+01,0.33115364900D+02,0.21030563180D+03,-.32119637460D+03,0.15248530010D+03,&
&       -.74762100000D-01,-.30081122400D+02,-.75372272100D+02,-.22056375800D+02,-.16979585000D+01,&
&       0.29380036000D+01,0.41247126700D+02,0.28786800950D+03,0.11437585060D+03,0.37654771400D+02,&
&       0.22222630000D+01,0.39409083100D+02,0.17740750020D+03,0.77988946200D+02,0.25754764100D+02,&
&       0.70524360000D+01,0.33304163500D+02,0.18694538750D+03,0.71968836100D+02,0.93630657000D+01,&
&       -.77609900000D-01,-.32433295300D+02,-.15108533510D+03,-.46497762600D+02,-.17808099000D+02,&
&       -.16481163000D+01,0.29170308000D+01,0.52062064400D+02,0.29328656720D+03,0.12286103730D+03,&
&       0.41686727700D+02,0.20968515000D+01,0.30022042400D+02,0.17287576360D+03,0.67760984100D+02,&
&       0.29897091300D+02,0.70414935000D+01,0.33339174400D+02,0.24928930770D+03,0.89357616600D+02,&
&       0.11620632400D+02/)
      real(8) :: ecpex6(388)= &
&     (/0.63678401300D+03,0.14789124460D+03,0.42572491100D+02,0.11242788300D+02,0.35689707000D+01,&
&       0.13107462000D+01,0.40864807000D+02,0.10932932900D+02,0.22294903000D+01,0.11885981710D+03,&
&       0.30684795800D+02,0.10176299900D+02,0.18272850000D+01,0.16008170400D+02,0.26488291000D+01,&
&       0.14555567200D+02,0.67036040000D+00,0.62096904880D+03,0.14636488260D+03,0.42320711400D+02,&
&       0.11213515100D+02,0.36963891000D+01,0.13169502000D+01,0.14036692000D+03,0.39143654700D+02,&
&       0.12955349300D+02,0.24308751000D+01,0.99492226100D+02,0.68271130900D+02,0.36151818100D+02,&
&       0.10105153700D+02,0.20232648000D+01,0.95275253600D+02,0.34660460800D+02,0.15489104000D+02,&
&       0.50015895000D+01,0.13236266000D+01,0.74091660340D+03,0.14665714990D+03,0.42358541300D+02,&
&       0.11164750000D+02,0.37556343000D+01,0.12871827000D+01,0.12568626650D+03,0.37576895500D+02,&
&       0.12695486900D+02,0.26276082000D+01,0.17917822470D+03,0.42619566100D+02,0.13413595700D+02,&
&       0.20950392000D+01,0.10476820640D+03,0.40193034000D+02,0.18505079800D+02,0.58051185000D+01,&
&       0.14583220000D+01,0.68555029310D+03,0.15180804850D+03,0.38052209900D+02,0.10205754400D+02,&
&       0.34376287000D+01,0.89975390000D+00,0.38010967250D+03,0.72462345050D+03,0.28147646030D+03,&
&       0.89359098800D+02,0.22976577300D+02,0.36700035000D+01,0.34703000000D+01,0.22831357710D+03,&
&       0.64087844700D+02,0.20513928900D+02,0.31643736000D+01,0.29880924000D+01,0.14371023990D+03,&
&       0.63185917000D+02,0.32022744900D+02,0.10767492200D+02,0.31003260000D+01,0.85039585300D+02,&
&       0.41819882000D+02,0.19652496700D+02,0.60777120000D+01,0.14776693000D+01,0.76822675330D+03,&
&       0.17359402180D+03,0.43704226800D+02,0.11618131900D+02,0.40314618000D+01,0.10905378000D+01,&
&       0.36541115590D+03,0.69713009150D+03,0.27178103790D+03,0.87307426300D+02,0.22749233900D+02,&
&       0.38680082000D+01,0.36718891000D+01,0.22821073670D+03,0.62618546200D+02,0.19646567000D+02,&
&       0.33651559000D+01,0.31497236000D+01,0.15127555540D+03,0.67979491100D+02,0.34747470600D+02,&
&       0.11565642500D+02,0.32930148000D+01,0.82836802900D+02,0.38071987200D+02,0.17454703400D+02,&
&       0.56785931000D+01,0.15814041000D+01,0.83944891200D+03,0.19285324820D+03,0.48665197400D+02,&
&       0.12922172700D+02,0.45748890000D+01,0.12681796000D+01,0.31342675180D+03,0.69931554620D+03,&
&       0.25989237410D+03,0.85499998000D+02,0.22763592500D+02,0.40764317000D+01,0.38827162000D+01,&
&       0.22439264240D+03,0.61673693100D+02,0.19146904300D+02,0.35565710000D+01,0.33263210000D+01,&
&       0.16152789580D+03,0.75581460700D+02,0.38911585200D+02,0.12542627100D+02,0.34615187000D+01,&
&       0.91210272700D+02,0.45415275600D+02,0.22045296700D+02,0.69810413000D+01,0.18380480000D+01,&
&       0.18338143836D+04,0.41405894560D+03,0.59034954000D+02,0.11917709400D+02,0.36531764000D+01,&
&       0.12764184000D+01,0.24700230850D+03,0.63171957670D+03,0.24928004410D+03,0.84294731500D+02,&
&       0.23040791200D+02,0.42997111000D+01,0.40974390000D+01,0.19368977260D+03,0.56930839400D+02,&
&       0.18722760200D+02,0.37340020000D+01,0.35127483000D+01,0.12623408240D+03,0.80709082600D+02,&
&       0.42552971400D+02,0.13407449400D+02,0.36904228000D+01,0.11277864590D+03,0.56781596800D+02,&
&       0.28583196600D+02,0.85734158000D+01,0.20892721000D+01,0.84887432920D+03,0.40046400040D+03,&
&       0.58035946600D+02,0.12045468200D+02,0.34442512000D+01,0.21626722490D+03,0.34385820980D+03,&
&       0.13605236060D+03,0.35038945700D+02,0.44724452000D+01,0.37675119000D+01,0.32606001340D+03,&
&       0.95633020600D+02,0.32526104300D+02,0.37699730000D+01,0.33895859000D+01,0.13103199290D+03,&
&       0.89246490300D+02,0.46397698000D+02,0.14186605300D+02,0.37668843000D+01,0.13473681480D+03,&
&       0.73877131800D+02,0.38103042500D+02,0.10533769500D+02,0.23338246000D+01,0.82358801470D+03,&
&       0.36466133360D+03,0.55708280100D+02,0.12046454400D+02,0.35120610000D+01,0.18804907700D+03,&
&       0.34041947120D+03,0.12823736730D+03,0.33864496100D+02,0.47560005000D+01,0.39649974000D+01,&
&       0.28972911390D+03,0.87463378900D+02,0.30436376600D+02,0.40553412000D+01,0.35525341000D+01,&
&       0.13640171060D+03,0.95077692500D+02,0.49225841000D+02,0.15087414500D+02,0.40405764000D+01,&
&       0.12735079080D+03,0.66236437400D+02,0.34429922900D+02,0.10199572100D+02,0.25409702000D+01,&
&       0.72893940560D+03,0.32065678000D+03,0.52868017400D+02,0.12028012800D+02,0.35238913000D+01,&
&       0.40944373580D+03,0.27454192310D+03,0.12756585700D+03,0.32903663100D+02,0.50593880000D+01,&
&       0.41506556000D+01,0.46617288920D+03,0.12078882590D+03,0.36411879100D+02,0.56985408000D+01,&
&       0.24956507630D+03,0.12666785850D+03,0.63143058600D+02,0.17905947000D+02,0.42239373000D+01,&
&       0.12181587990D+03,0.60875703000D+02,0.31476714700D+02,0.98811751000D+01,0.27319874000D+01,&
&       0.62262879560D+03,0.13628436070D+03,0.33154978100D+02,0.99894895000D+01,0.30481312000D+01,&
&       0.19473743040D+03,0.35153274470D+03,0.12232704020D+03,0.32091461700D+02,0.52451812000D+01,&
&       0.44916223000D+01,0.42061588010D+03,0.10944178150D+03,0.34171428000D+02,0.59879750000D+01,&
&       0.21926661580D+03,0.12272977860D+03,0.63106336900D+02,0.18368452000D+02,0.44972844000D+01,&
&       0.10855060370D+03,0.56479552700D+02,0.29206915900D+02,0.95440543000D+01,0.28965118000D+01,&
&       0.31834603850D+03,0.95618719100D+02,0.17565263100D+02,0.86336796000D+01,0.30025515000D+01,&
&       0.11004827000D+01,0.18000820180D+03,0.60978206300D+02,0.22944466800D+02,0.68681457000D+01,&
&       0.13345231000D+01,0.26054813650D+03,0.89572895900D+02,0.37146144100D+02,0.12496035900D+02,&
&       0.37833475000D+01,0.90419930000D+00,0.72058623200D+02,0.53876873800D+02,0.19655698900D+02,&
&       0.34386734000D+01,0.27552072000D+01,0.49232620000D+00,0.86900848100D+02,0.31384480900D+02,&
&       0.33568331900D+02,0.10604695700D+02,0.22121230000D+01,0.76705000000D+00,0.37379107830D+03,&
&       0.99022204400D+02,0.17684464000D+02,0.86281221000D+01,0.30353877000D+01,0.11191950000D+01,&
&       0.16988123650D+03,0.58243495800D+02,0.21728148400D+02,0.66359251000D+01,0.14615973000D+01,&
&       0.24894388090D+03,0.84660141400D+02,0.33967410900D+02,0.11523910400D+02,0.35354358000D+01,&
&       0.95129150000D+00,0.86203682900D+02,0.76028055800D+02,0.26284676400D+02,0.33191097000D+01,&
&       0.27009368000D+01,0.54921050000D+00,0.11594616750D+03,0.43744398800D+02,0.46493723100D+02,&
&       0.13284899900D+02,0.25347097000D+01,0.76705000000D+00,0.37658037860D+03,0.86484001400D+02,&
&       0.26678427600D+02,0.94261986000D+01,0.27101719000D+01,0.87920310000D+00,0.13242487960D+03,&
&       0.47237604400D+02,0.17631272700D+02,0.54744712000D+01,0.12634856000D+01,0.76514470000D+00,&
&       0.67896645400D+02,0.24989822500D+02,0.10705293900D+02,0.32792568000D+01,0.84525220000D+00,&
&       0.64162450000D+00,0.68833600500D+02,0.24281587400D+02,0.94532762000D+01,0.24788185000D+01,&
&       0.24789161000D+01,0.55517380000D+00,0.12810213220D+03,0.54802915400D+02,0.24552930800D+02,&
&       0.81144792000D+01,0.16931290000D+01,0.76705000000D+00,0.22330919710D+03,0.86469921600D+02,&
&       0.26849220300D+02,0.93823622000D+01,0.27614708000D+01,0.94239230000D+00,0.15597839140D+03,&
&       0.54459113900D+02,0.20179455800D+02,0.61273104000D+01,0.11805983000D+01,0.99324640000D+00,&
&       0.74509138600D+02,0.28048764700D+02,0.12016850800D+02,0.36914327000D+01,0.90808240000D+00,&
&       0.73946170000D+00,0.68085667900D+02,0.23725035500D+02,0.92437053000D+01,0.18452292000D+01,&
&       0.12209859000D+01,0.87408610000D+00,0.13435389830D+03,0.51581279700D+02,0.54930728400D+02,&
&       0.17047534600D+02,0.37702824000D+01,0.76705000000D+00/)
      real(8) :: ecpcoeff6(388)= &
&     (/-.80510900000D-01,-.33431144800D+02,-.19392708870D+03,-.56830143300D+02,-.18650497500D+02,&
&       -.25411399000D+01,0.30499934000D+01,0.37967751400D+02,0.45234067300D+02,0.46406951000D+01,&
&       0.62031353300D+02,0.12959094810D+03,0.31518682500D+02,0.73054473000D+01,0.33278576000D+02,&
&       0.53129694800D+02,0.16211887800D+02,-.83465200000D-01,-.33325767100D+02,-.19086072320D+03,&
&       -.55198417200D+02,-.18023634000D+02,-.21978281000D+01,0.28131160000D+01,0.55305062600D+02,&
&       0.14955134020D+03,0.50507855300D+02,0.49191812000D+01,0.39387207500D+02,0.33507535840D+03,&
&       0.13125351530D+03,0.36317502500D+02,0.29764140000D+01,0.46057114100D+02,0.11716585880D+03,&
&       0.54013081500D+02,0.15578490600D+02,-.86472700000D-01,-.33516880300D+02,-.18907098290D+03,&
&       -.54246436700D+02,-.17135134200D+02,-.18092185000D+01,0.28894943000D+01,0.61952159200D+02,&
&       0.16480744900D+03,0.55695460900D+02,0.45978412000D+01,0.71671973600D+02,0.17589658960D+03,&
&       0.37913640400D+02,0.29705496000D+01,0.46523994600D+02,0.13740781760D+03,0.63494518300D+02,&
&       0.18099865400D+02,-.60000000000D+02,-.58849806640D+03,-.19977488200D+03,-.77236512100D+02,&
&       -.14653259600D+02,-.67622040000D+00,0.30000000000D+01,0.31339553200D+02,0.12516866138D+04,&
&       0.70089001060D+03,0.26604458450D+03,0.46501961550D+03,-.36954045680D+03,0.20000000000D+01,&
&       0.60277697200D+02,0.19221749830D+03,0.33964600890D+03,-.26610665430D+03,0.30000000000D+01,&
&       0.54657676000D+02,0.21758114270D+03,0.12513630840D+03,0.37620319500D+02,0.40000000000D+01,&
&       0.49635839300D+02,0.17846572220D+03,0.82007717300D+02,0.63063943000D+01,-.60000000000D+02,&
&       -.62971293130D+03,-.22117505030D+03,-.83476968300D+02,-.17837027500D+02,-.81606090000D+00,&
&       0.30000000000D+01,0.33979898500D+02,0.12425776244D+04,0.70616638840D+03,0.27624645630D+03,&
&       0.51322831320D+03,-.41296019160D+03,0.20000000000D+01,0.61817416300D+02,0.19449754790D+03,&
&       0.30783466710D+03,-.23063463810D+03,0.30000000000D+01,0.55267097800D+02,0.23629575450D+03,&
&       0.13447550720D+03,0.41140118300D+02,0.40000000000D+01,0.50753376000D+02,0.16203128770D+03,&
&       0.72061929800D+02,0.60009895000D+01,-.60000000000D+02,-.66419879200D+03,-.23861436510D+03,&
&       -.88419240700D+02,-.20606232600D+02,-.92837920000D+00,0.30000000000D+01,0.39119296700D+02,&
&       0.11809692974D+04,0.72895642100D+03,0.29355911400D+03,0.56267314930D+03,-.45738071850D+03,&
&       0.20000000000D+01,0.63894839300D+02,0.20589018370D+03,0.31214271530D+03,-.23139612810D+03,&
&       0.30000000000D+01,0.55331525600D+02,0.26719766530D+03,0.14684855780D+03,0.44105524300D+02,&
&       0.40000000000D+01,0.50306552300D+02,0.19073630980D+03,0.91760555200D+02,0.84247312000D+01,&
&       -.14971040000D+00,-.16692555781D+04,-.34666611290D+03,-.96668489200D+02,-.11073856700D+02,&
&       -.57985520000D+00,0.31497104000D+01,0.45296996900D+02,0.11164279585D+04,0.79364190650D+03,&
&       0.31820991930D+03,0.58631050250D+03,-.47569041640D+03,0.21497104000D+01,0.64229573600D+02,&
&       0.21885046070D+03,0.34441891680D+03,-.26047272670D+03,0.31497104000D+01,0.46571486800D+02,&
&       0.30438345950D+03,0.15715896520D+03,0.48570121100D+02,0.39569610000D+01,0.52358678100D+02,&
&       0.23603408240D+03,0.11657872660D+03,0.11105053000D+02,-.15372930000D+00,-.16429695542D+04,&
&       -.33679545670D+03,-.94983902700D+02,-.10351621800D+02,0.31537293000D+01,0.25994823300D+02,&
&       0.85948657560D+03,0.38013765040D+03,0.24270106660D+03,-.12517531070D+03,0.21537293000D+01,&
&       0.62868427500D+02,0.29313229870D+03,0.26446483880D+03,-.17264203140D+03,0.31537293000D+01,&
&       0.45339920000D+02,0.33585750320D+03,0.16630374570D+03,0.49023782500D+02,0.39557982000D+01,&
&       0.51952195700D+02,0.31425558660D+03,0.14718398940D+03,0.13962294700D+02,-.15780140000D+00,&
&       -.15175270446D+04,-.31653065290D+03,-.91888094100D+02,-.92241773000D+01,0.31578014000D+01,&
&       0.26832257700D+02,0.80042500070D+03,0.36940506830D+03,0.24241718990D+03,-.11821732820D+03,&
&       0.21578014000D+01,0.61967861000D+02,0.26905819860D+03,0.23116547930D+03,-.13369526670D+03,&
&       0.31578014000D+01,0.45934980300D+02,0.35903446680D+03,0.17647401190D+03,0.54515528600D+02,&
&       0.39546197000D+01,0.52977365500D+02,0.27486433830D+03,0.13720473380D+03,0.14863330500D+02,&
&       -.16192680000D+00,-.13202873852D+04,-.29831781350D+03,-.87583706500D+02,-.81493274000D+01,&
&       0.27334218000D+01,0.59702432900D+02,0.89145895500D+03,0.36844676560D+03,0.23802630900D+03,&
&       -.10705564540D+03,0.18878568000D+01,0.76013862900D+02,0.34355111160D+03,0.11949117860D+03,&
&       0.29343678000D+01,0.59330657100D+02,0.45244451940D+03,0.21047694790D+03,0.58625411200D+02,&
&       0.39534253000D+01,0.53855518200D+02,0.24743051330D+03,0.12781879760D+03,0.15377204600D+02,&
&       -.60000000000D+02,-.55552923120D+03,-.16800197850D+03,-.63039987500D+02,-.42516681000D+01,&
&       0.30000000000D+01,0.38602088000D+02,0.86483707270D+03,0.37499355200D+03,0.28979101000D+03,&
&       -.15245327730D+03,0.20000000000D+01,0.73888562500D+02,0.32667298720D+03,0.12658145910D+03,&
&       0.30000000000D+01,0.55679314900D+02,0.44919873350D+03,0.21502690910D+03,0.64084099500D+02,&
&       0.40000000000D+01,0.51806533500D+02,0.23121831130D+03,0.11900473860D+03,0.15342418800D+02,&
&       -.17033720000D+00,-.24518967600D+02,-.28230532800D+02,-.50216021700D+02,-.13832553400D+02,&
&       -.13977261000D+01,0.27440421000D+01,0.68867134300D+02,0.24680060730D+03,0.97786420600D+02,&
&       0.26546589400D+02,0.19446525000D+01,0.64931582200D+02,0.27235217900D+03,0.13857944260D+03,&
&       0.41215819900D+02,0.15171275600D+02,0.31153493000D+01,0.44558021700D+02,0.23025883070D+03,&
&       0.98527115600D+02,-.43371893700D+02,-.32608800000D+00,0.40027249000D+01,0.53960022000D+02,&
&       0.25275602600D+02,0.14591842280D+03,0.11763763600D+02,-.67597360000D+00,-.17462220000D+00,&
&       -.24585805700D+02,-.28521471900D+02,-.47997734800D+02,-.13173333700D+02,-.11972529000D+01,&
&       0.27960747000D+01,0.74906273100D+02,0.26433238230D+03,0.10234276830D+03,0.30305458200D+02,&
&       0.19719244000D+01,0.70036844800D+02,0.28676457660D+03,0.14140036470D+03,0.45049013000D+02,&
&       0.16465531700D+02,0.30725115000D+01,0.43180960300D+02,0.30276964840D+03,0.11630419710D+03,&
&       -.59306644900D+02,-.24625940000D+00,0.39888161000D+01,0.54448118600D+02,0.89875601600D+02,&
&       0.18737447880D+03,0.15355292700D+02,-.64393030000D+00,-.17896050000D+00,-.54397233700D+02,&
&       -.19970617590D+03,-.79122394100D+02,-.24986902000D+02,-.44397939000D+01,0.28115386000D+01,&
&       0.65036720500D+02,0.21278685450D+03,0.72105317500D+02,0.33014094000D+02,-.57708461000D+01,&
&       0.48754911000D+01,0.63914810200D+02,0.14810643580D+03,0.47310630100D+02,0.21030670200D+02,&
&       -.70930772000D+01,0.32161388000D+01,0.55738608600D+02,0.12141683510D+03,0.19345606400D+02,&
&       0.15367516800D+02,0.61298724000D+01,0.41353682000D+01,0.67512844600D+02,0.25873731070D+03,&
&       0.11324782640D+03,0.34168020100D+02,-.65531956000D+01,-.18335200000D+00,-.53338949200D+02,&
&       -.20191178410D+03,-.76311529500D+02,-.24257846800D+02,-.40319700000D+01,0.27916510000D+01,&
&       0.71855381700D+02,0.25243274440D+03,0.89604510900D+02,0.61174413600D+02,-.33047190600D+02,&
&       0.48462914000D+01,0.65688339600D+02,0.16626196730D+03,0.54139332900D+02,0.28620488000D+02,&
&       -.12345220400D+02,0.32183832000D+01,0.54895268100D+02,0.11655407080D+03,0.69076066200D+02,&
&       -.72641087000D+02,0.38940005500D+02,0.39864075000D+01,0.63026207800D+02,0.92525995900D+02,&
&       0.24099736720D+03,0.62184457700D+02,-.17251134000D+01/)
!
      select case(numatomic(iatom))
!
! Set Ne - Ar ECP functions
!
        case(11:18)
          maxangecp(iatom)= 2
          izcore(iatom)= 10
!  
          locecp(0,iatom)= iprim
          mprimecp(0,iatom)= 5
          do j= 1,5
            execp(iprim+j)= ecpex3(j,numatomic(iatom))
            coeffecp(iprim+j)= ecpcoeff3(j,numatomic(iatom))
            mtypeecp(iprim+j)= nangecp3(j)
          enddo
!  
          locecp(1,iatom)= iprim+5
          mprimecp(1,iatom)= 5
          do j= 6,10
            execp(iprim+j)= ecpex3(j,numatomic(iatom))
            coeffecp(iprim+j)= ecpcoeff3(j,numatomic(iatom))
            mtypeecp(iprim+j)= nangecp3(j)
          enddo
!  
          locecp(2,iatom)= iprim+10
          mprimecp(2,iatom)= 6
          do j= 11,16
            execp(iprim+j)= ecpex3(j,numatomic(iatom))
            coeffecp(iprim+j)= ecpcoeff3(j,numatomic(iatom))
            mtypeecp(iprim+j)= nangecp3(j)
          enddo
!  
          iprim= iprim+16
!
! Set K - Kr ECP functions
!
        case (19:36)
          if(numatomic(iatom) <= 29) then
            maxangecp(iatom)= 2
            izcore(iatom)= 10
          elseif(numatomic(iatom) == 30) then
            maxangecp(iatom)= 3
            izcore(iatom)= 18
          else
            maxangecp(iatom)= 3
            izcore(iatom)= 28
          endif
          if(numatomic(iatom) <= 20) then
            numecp(0:2)=(/5,3,5/)
            nangecp4(1:13)=(/1,2,2,2,2,0,1,2,0,1,2,2,2/)
          elseif(numatomic(iatom) == 21) then
            numecp(0:2)=(/3,5,6/)
            nangecp4(1:14)=(/1,2,2,0,1,2,2,2,0,1,2,2,2,2/)
          elseif(numatomic(iatom) <= 26) then
            numecp(0:2)=(/3,5,5/)
            nangecp4(1:13)=(/1,2,2,0,1,2,2,2,0,1,2,2,2/)
          elseif(numatomic(iatom) <= 29) then
            numecp(0:2)=(/3,4,4/)
            nangecp4(1:11)=(/1,2,2,0,1,2,2,0,1,2,2/)
          elseif(numatomic(iatom) == 30) then
            numecp(0:3)=(/5,5,5,3/)
            nangecp4(1:18)=(/1,2,2,2,2,0,1,2,2,2,0,1,2,2,2,2,2,2/)
          elseif(numatomic(iatom) <= 34) then
            numecp(0:3)=(/5,5,5,5/)
            nangecp4(1:20)=(/1,2,2,2,2,0,1,2,2,2,0,1,2,2,2,0,1,2,2,2/)
          else
            numecp(0:3)=(/4,4,5,5/)
            nangecp4(1:18)=(/1,2,2,2,0,1,2,2,0,1,2,2,2,0,1,2,2,2/)
          endif
!
          locecp(0,iatom)= iprim
          mprimecp(0,iatom)= numecp(0)
          do j= 1,numecp(0)
            execp(iprim+j)= ecpex4(j+iecp4(numatomic(iatom)))
            coeffecp(iprim+j)= ecpcoeff4(j+iecp4(numatomic(iatom)))
            mtypeecp(iprim+j)= nangecp4(j)
          enddo
          iprim= iprim+numecp(0)
          loctmp= numecp(0)
!
          locecp(1,iatom)= iprim
          mprimecp(1,iatom)= numecp(1)
          do j= 1,numecp(1)
            execp(iprim+j)= ecpex4(j+loctmp+iecp4(numatomic(iatom)))
            coeffecp(iprim+j)= ecpcoeff4(j+loctmp+iecp4(numatomic(iatom)))
            mtypeecp(iprim+j)= nangecp4(j+loctmp)
          enddo
          iprim= iprim+numecp(1)
          loctmp= loctmp+numecp(1)
!
          locecp(2,iatom)= iprim
          mprimecp(2,iatom)= numecp(2)
          do j= 1,numecp(2)
            execp(iprim+j)= ecpex4(j+loctmp+iecp4(numatomic(iatom)))
            coeffecp(iprim+j)= ecpcoeff4(j+loctmp+iecp4(numatomic(iatom)))
            mtypeecp(iprim+j)= nangecp4(j+loctmp)
          enddo
          iprim= iprim+numecp(2)
          loctmp= loctmp+numecp(2)
!
          if(numatomic(iatom) >= 30) then
            locecp(3,iatom)= iprim
            mprimecp(3,iatom)= numecp(3)
            do j= 1,numecp(3)
              execp(iprim+j)= ecpex4(j+loctmp+iecp4(numatomic(iatom)))
              coeffecp(iprim+j)= ecpcoeff4(j+loctmp+iecp4(numatomic(iatom)))
              mtypeecp(iprim+j)= nangecp4(j+loctmp)
            enddo
            iprim= iprim+numecp(3)
          endif
!
! Set Rb - Xe ECP functions
!
        case (37:54)
          maxangecp(iatom)= 3
          if(numatomic(iatom) <= 47) then
            izcore(iatom)= 28
          elseif(numatomic(iatom) == 48) then
            izcore(iatom)= 36
          else
            izcore(iatom)= 46
          endif
          select case(numatomic(iatom))
            case(37)
              numecp(0:3)=(/5,4,5,4/)
              nangecp5(1:18)=(/0,1,2,2,2,0,1,2,2,0,1,2,2,2,0,1,2,2/)
            case(38)
              numecp(0:3)=(/5,4,5,5/)
              nangecp5(1:19)=(/0,1,2,2,2,0,1,2,2,0,1,2,2,2,0,1,2,2,2/)
            case(39)
              numecp(0:3)=(/5,4,4,5/)
              nangecp5(1:18)=(/0,1,2,2,2,0,1,2,2,0,1,2,2,0,1,2,2,2/)
            case(40,43)
              numecp(0:3)=(/5,3,4,5/)
              nangecp5(1:17)=(/0,1,2,2,2,0,1,2,0,1,2,2,0,1,2,2,2/)
            case(44,45)
              numecp(0:3)=(/5,5,5,4/)
              nangecp5(1:19)=(/0,1,2,2,2,0,1,2,2,2,0,1,2,2,2,0,1,2,2/)
            case(46,47,49,50,53)
              numecp(0:3)=(/5,5,5,5/)
              nangecp5(1:20)=(/0,1,2,2,2,0,1,2,2,2,0,1,2,2,2,0,1,2,2,2/)
            case(48)
              numecp(0:3)=(/7,5,4,6/)
              nangecp5(1:22)=(/0,1,2,2,2,2,2,0,1,2,2,2,0,1,2,2,0,1,2,2,2,2/)
            case(51,52,54)
              numecp(0:3)=(/6,5,5,5/)
              nangecp5(1:21)=(/0,1,2,2,2,2,0,1,2,2,2,0,1,2,2,2,0,1,2,2,2/)
          end select
!
          locecp(0,iatom)= iprim
          mprimecp(0,iatom)= numecp(0)
          do j= 1,numecp(0)
            execp(iprim+j)= ecpex5(j+iecp5(numatomic(iatom)))
            coeffecp(iprim+j)= ecpcoeff5(j+iecp5(numatomic(iatom)))
            mtypeecp(iprim+j)= nangecp5(j)
          enddo
          iprim= iprim+numecp(0)
          loctmp= numecp(0)
!
          locecp(1,iatom)= iprim
          mprimecp(1,iatom)= numecp(1)
          do j= 1,numecp(1)
            execp(iprim+j)= ecpex5(j+loctmp+iecp5(numatomic(iatom)))
            coeffecp(iprim+j)= ecpcoeff5(j+loctmp+iecp5(numatomic(iatom)))
            mtypeecp(iprim+j)= nangecp5(j+loctmp)
          enddo
          iprim= iprim+numecp(1)
          loctmp= loctmp+numecp(1)
!
          locecp(2,iatom)= iprim
          mprimecp(2,iatom)= numecp(2)
          do j= 1,numecp(2)
            execp(iprim+j)= ecpex5(j+loctmp+iecp5(numatomic(iatom)))
            coeffecp(iprim+j)= ecpcoeff5(j+loctmp+iecp5(numatomic(iatom)))
            mtypeecp(iprim+j)= nangecp5(j+loctmp)
          enddo
          iprim= iprim+numecp(2)
          loctmp= loctmp+numecp(2)
!
          locecp(3,iatom)= iprim
          mprimecp(3,iatom)= numecp(3)
          do j= 1,numecp(3)
            execp(iprim+j)= ecpex5(j+loctmp+iecp5(numatomic(iatom)))
            coeffecp(iprim+j)= ecpcoeff5(j+loctmp+iecp5(numatomic(iatom)))
            mtypeecp(iprim+j)= nangecp5(j+loctmp)
          enddo
          iprim= iprim+numecp(3)
!
! Set Cs - La, Hf - Bi ECP functions
!
        case(55:57,72:83)
          if(numatomic(iatom) <= 55) then
            maxangecp(iatom)= 3
          else
            maxangecp(iatom)= 4
          endif
          if(numatomic(iatom) <= 55) then
            izcore(iatom)= 46
          elseif(numatomic(iatom) <= 79) then
            izcore(iatom)= 60
          elseif(numatomic(iatom) <= 81) then
            izcore(iatom)= 68
          else
            izcore(iatom)= 78
          endif
          select case(numatomic(iatom))
            case(55)
              numecp(0:3)=(/6,3,4,4/)
              nangecp6(1:17)=(/0,1,2,2,2,2,0,1,2,0,1,2,2,0,1,2,2/)
            case(56)
              numecp(0:3)=(/6,4,5,5/)
              nangecp6(1:20)=(/0,1,2,2,2,2,0,1,2,2,0,1,2,2,2,0,1,2,2,2/)
            case(57)
              numecp(0:3)=(/6,4,4,5/)
              nangecp6(1:19)=(/0,1,2,2,2,2,0,1,2,2,0,1,2,2,0,1,2,2,2/)
            case(72:75)
              numecp(0:4)=(/6,7,5,5,5/)
              nangecp6(1:28)=(/1,2,2,2,2,2,0,1,2,2,2,2,2,0,1,2,2,2,0,1,2,2,2,0,1,2,2,2/)
            case(76:77)
              numecp(0:4)=(/5,6,5,5,5/)
              nangecp6(1:26)=(/1,2,2,2,2,0,1,2,2,2,2,0,1,2,2,2,0,1,2,2,2,0,1,2,2,2/)
            case(78:79)
              numecp(0:4)=(/5,6,4,5,5/)
              nangecp6(1:25)=(/1,2,2,2,2,0,1,2,2,2,2,0,1,2,2,0,1,2,2,2,0,1,2,2,2/)
            case(80:81)
              numecp(0:4)=(/6,5,6,6,6/)
              nangecp6(1:29)=(/0,1,1,2,2,2,0,1,2,2,2,0,1,2,2,2,2,0,1,2,2,2,2,0,1,2,2,2,2/)
            case(82:83)
              numecp(0:4)=(/6,6,6,6,6/)
              nangecp6(1:30)=(/0,1,2,2,2,2,0,1,2,2,2,2,0,1,2,2,2,2,0,1,2,2,2,2,0,1,2,2,2,2/)
          end select
!
          locecp(0,iatom)= iprim
          mprimecp(0,iatom)= numecp(0)
          do j= 1,numecp(0)
            execp(iprim+j)= ecpex6(j+iecp6(numatomic(iatom)))
            coeffecp(iprim+j)= ecpcoeff6(j+iecp6(numatomic(iatom)))
            mtypeecp(iprim+j)= nangecp6(j)
          enddo
          iprim= iprim+numecp(0)
          loctmp= numecp(0)
!
          locecp(1,iatom)= iprim
          mprimecp(1,iatom)= numecp(1)
          do j= 1,numecp(1)
            execp(iprim+j)= ecpex6(j+loctmp+iecp6(numatomic(iatom)))
            coeffecp(iprim+j)= ecpcoeff6(j+loctmp+iecp6(numatomic(iatom)))
            mtypeecp(iprim+j)= nangecp6(j+loctmp)
          enddo
          iprim= iprim+numecp(1)
          loctmp= loctmp+numecp(1)
!
          locecp(2,iatom)= iprim
          mprimecp(2,iatom)= numecp(2)
          do j= 1,numecp(2)
            execp(iprim+j)= ecpex6(j+loctmp+iecp6(numatomic(iatom)))
            coeffecp(iprim+j)= ecpcoeff6(j+loctmp+iecp6(numatomic(iatom)))
            mtypeecp(iprim+j)= nangecp6(j+loctmp)
          enddo
          iprim= iprim+numecp(2)
          loctmp= loctmp+numecp(2)
!
          locecp(3,iatom)= iprim
          mprimecp(3,iatom)= numecp(3)
          do j= 1,numecp(3)
            execp(iprim+j)= ecpex6(j+loctmp+iecp6(numatomic(iatom)))
            coeffecp(iprim+j)= ecpcoeff6(j+loctmp+iecp6(numatomic(iatom)))
            mtypeecp(iprim+j)= nangecp6(j+loctmp)
          enddo
          iprim= iprim+numecp(3)
          loctmp= loctmp+numecp(3)
!
          if(numatomic(iatom) >= 72) then
            locecp(4,iatom)= iprim
            mprimecp(4,iatom)= numecp(4)
            do j= 1,numecp(4)
              execp(iprim+j)= ecpex6(j+loctmp+iecp6(numatomic(iatom)))
              coeffecp(iprim+j)= ecpcoeff6(j+loctmp+iecp6(numatomic(iatom)))
              mtypeecp(iprim+j)= nangecp6(j+loctmp)
            enddo
            iprim= iprim+numecp(4)
          endif
        case (:0)
        case (84:)
          write(*,'(" Error! This program supports Na - La, Hf - Bi Lanl2dz ECP.")')
          call iabort
      endselect
!
      if(iprim > mxprim ) then
        write(*,'(" Error! The number of ECP functions exceeds mxprim",i6,".")')mxprim
        call iabort
      endif
      return
end









