! Copyright 2014-2017  Kazuya Ishimura
!
! Licensed under the Apache License, Version 2.0 (the "License");
! you may not use this file except in compliance with the License.
! You may obtain a copy of the License at
!
!     http://www.apache.org/licenses/LICENSE-2.0
!
! Unless required by applicable law or agreed to in writing, software
! distributed under the License is distributed on an "AS IS" BASIS,
! WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
! See the License for the specific language governing permissions and
! limitations under the License.
!
!--------------------------------
  subroutine setbasis(mpi_comm)
!--------------------------------
!
! Driver of setting basis functions
!
      use modparallel, only : master
      use modmolecule, only : natom, numatomic
      use modbasis, only : basis, locprim, locbf, locatom, mprim, mbf, mtype, &
&                          nshell, nao, nprim, ex, coeff, coeffinp
      use modecp, only : flagecp
      implicit none
      integer,intent(in) :: mpi_comm
      integer :: ishell, iatom, i
!
      if(master) then
        locprim(1)=0
        locbf(1)=0
        ishell= 0
!
        select case(basis)
          case('STO-3G')
            do iatom= 1, natom
              call bssto3g(iatom,ishell)
            enddo
          case('3-21G')
            do iatom= 1, natom
              call bs321g(iatom,ishell)
            enddo
          case('6-31G')
            do iatom= 1, natom
              call bs631g(iatom,ishell)
            enddo
          case('6-31G(D)','6-31G*')
            do iatom= 1, natom
              call bs631gd(iatom,ishell)
            enddo
          case('6-31G(D,P)','6-31G**')
            do iatom= 1, natom
              call bs631gdp(iatom,ishell)
            enddo
          case('6-311G')
            do iatom= 1, natom
              call bs6311g(iatom,ishell)
            enddo
          case('6-311G(D)','6-311G*')
            do iatom= 1, natom
              call bs6311gd(iatom,ishell)
            enddo
          case('6-311G(D,P)','6-311G**')
            do iatom= 1, natom
              call bs6311gdp(iatom,ishell)
            enddo
          case('CC-PVDZ')
            do iatom= 1, natom
              call bsccpvdz(iatom,ishell)
            enddo
          case('CC-PVTZ')
            do iatom= 1, natom
              call bsccpvtz(iatom,ishell)
            enddo
          case('CC-PVQZ')
            do iatom= 1, natom
              call bsccpvqz(iatom,ishell)
            enddo
          case('D95V')
            do iatom= 1, natom
              call bsd95v(iatom,ishell)
            enddo
          case('LANL2DZ')
            if(.not.flagecp) then
              if(master) write(*,'(" Error! ECP is not set!.")')
              call iabort
            endif
            do iatom= 1, natom
              if(numatomic(iatom) < 11) then
                call bsd95v(iatom,ishell)
              else
                call bslanl2dz(iatom,ishell)
              endif
            enddo
          case('GEN')
            call readbasis
            call setgenbasis(ishell)
          case('CHECK')
            call setcheckbasis
            ishell= nshell
          case default
            write(*,'(" Error! Basis set ",a16,"is not supported.")') basis
            call iabort
        end select
!
        nshell= ishell
      endif
!
      call para_bcasti(nshell,1,0,mpi_comm)
      call para_bcasti(locprim,nshell+1,0,mpi_comm)
      call para_bcasti(locbf  ,nshell+1,0,mpi_comm)
      call para_bcasti(locatom,nshell  ,0,mpi_comm)
!
      nao= locbf(nshell+1)
      nprim= locprim(nshell+1)
!
      call para_bcastr(ex   ,nprim,0,mpi_comm)
      call para_bcastr(coeff,nprim,0,mpi_comm)
      call para_bcasti(mprim,nshell,0,mpi_comm)
      call para_bcasti(mbf  ,nshell,0,mpi_comm)
      call para_bcasti(mtype,nshell,0,mpi_comm)
!
      do i= 1,nprim
        coeffinp(i)= coeff(i)
      enddo
!
      call bsnrmlz
      return
end


!---------------------
  subroutine bsnrmlz
!---------------------
!
! Normalize basis functions
!
      use modbasis, only : nshell, mtype, ex, coeff, locprim, mprim
      implicit none
      integer :: ishell
!
!$OMP parallel do
      do ishell= 1,nshell
        call bsnor(ishell,ex,coeff,locprim,mprim,mtype)
      enddo
!$OMP end parallel do
      return
end


!--------------------------------------------------------
  subroutine bsnor(ishell,ex,coeff,locprim,mprim,mtype)
!--------------------------------------------------------
!
! Normalize basis functions
!
      use modparam, only : mxprim, mxshell
      implicit none
      integer,intent(in) :: ishell, locprim(mxshell+1), mprim(mxshell), mtype(mxshell)
      integer :: iloc, iprim, jprim, nprimi, nangi
      real(8),parameter :: zero=0.0D+00, one=1.0D+00, two=2.0D+00
      real(8),parameter :: pi32=5.568327996831708D+00   ! pi32=pi**(3/2)
      real(8),parameter :: pii32=0.1795871221251666D+00 ! pii32=pi**(-3/2)
      real(8),intent(in) :: ex(mxprim)
      real(8),intent(inout) :: coeff(mxprim)
      real(8) :: exij, fac1, fac2
      real(8) :: factor(0:6)=(/1.0D+0,1.0D+0,3.0D+0,15.0D+0,105.0D+0,945.0D+0,10395.0D+0/)
!
! Multiply normalization factor to each primitive function
!
      iloc= locprim(ishell)
      nprimi= mprim(ishell)
      nangi= mtype(ishell)
      do iprim= iloc+1,iloc+nprimi
        exij= ex(iprim)*two
        fac1= pii32*exij*sqrt(exij)*(two*exij)**nangi
        coeff(iprim)= coeff(iprim)*sqrt(fac1/factor(nangi))
      enddo
!
! Normalize contracted functions
!
      fac2= zero
      do iprim= iloc+1,iloc+nprimi
        do jprim= iloc+1,iprim-1
          exij= ex(iprim)+ex(jprim)
          fac1= exij*sqrt(exij)*(two*exij)**nangi
          fac2= fac2+coeff(iprim)*coeff(jprim)*two*factor(nangi)/fac1
        enddo
        exij= ex(iprim)*two
        fac1= exij*sqrt(exij)*(two*exij)**nangi
        fac2= fac2+coeff(iprim)*coeff(iprim)*factor(nangi)/fac1
      enddo
      fac2= one/sqrt(fac2*pi32)
      do iprim= iloc+1,iloc+nprimi
        coeff(iprim)= coeff(iprim)*fac2
      enddo
      return
end


!-----------------------
  subroutine bsnrmlz_g
!-----------------------
!
! Normalize basis functions for guess calculation
!
      use modguess, only : nshell_g, mtype_g, ex_g, coeff_g, locprim_g, mprim_g
      implicit none
      integer :: ishell
!
!$OMP parallel do
      do ishell= 1,nshell_g
        call bsnor(ishell,ex_g,coeff_g,locprim_g,mprim_g,mtype_g)
      enddo
!$OMP end parallel do
      return
end


!---------------------------
  subroutine bsnrmlz_gcore
!---------------------------
!
! Normalize basis functions for guess calculation of core orbitals
!
      use modguess, only : nshell_gcore, mtype_gcore, ex_gcore, coeff_gcore, &
&                          locprim_gcore, mprim_gcore
      implicit none
      integer :: ishell
!
!$OMP parallel do
      do ishell= 1,nshell_gcore
        call bsnor(ishell,ex_gcore,coeff_gcore,locprim_gcore,mprim_gcore,mtype_gcore)
      enddo
!$OMP end parallel do
      return
end


!---------------------------------
  subroutine setgenbasis(ishell)
!---------------------------------
!
! Driver of setting basis functions from input file
! This routine must be called only from master node.
!
      use modmolecule, only : natom, numatomic
      use modbasis, only : locprim, locbf, locatom, mprim, mbf, mtype, &
&                          ex, coeff, locgenprim, mgenprim, mgentype, &
&                          locgenshell, ngenshell, exgen, coeffgen, atombasis, spher
      use modecp, only : flagecp
      implicit none
      integer,intent(out) :: ishell
      integer :: iatom, nn, ii, jj, ll, lprim
      character(len=3) :: table(-9:112)= &
&     (/'Bq9','Bq8','Bq7','Bq6','Bq5','Bq4','Bq3','Bq2','Bq ','X  ',&
&       'H  ','He ','Li ','Be ','B  ','C  ','N  ','O  ','F  ','Ne ','Na ','Mg ','Al ','Si ','P  ',&
&       'S  ','Cl ','Ar ','K  ','Ca ','Sc ','Ti ','V  ','Cr ','Mn ','Fe ','Co ','Ni ','Cu ','Zn ',&
&       'Ga ','Ge ','As ','Se ','Br ','Kr ','Rb ','Sr ','Y  ','Zr ','Nb ','Mo ','Tc ','Ru ','Rh ',&
&       'Pd ','Ag ','Cd ','In ','Sn ','Sb ','Te ','I  ','Xe ','Cs ','Ba ','La ','Ce ','Pr ','Nd ',&
&       'Pm ','Sm ','Eu ','Gd ','Tb ','Dy ','Ho ','Er ','Tm ','Yb ','Lu ','Hf ','Ta ','W  ','Re ',&
&       'Os ','Ir ','Pt ','Au ','Hg ','Tl ','Pb ','Bi ','Po ','At ','Rn ','Fr ','Ra ','Ac ','Th ',&
&       'Pa ','U  ','Np ','Pu ','Am ','Cm ','Bk ','Cf ','Es ','Fm ','Md ','No ','Lr ','Rf ','Db ',&
&       'Sg ','Bh ','Hs ','Mt ','Uun','Uuu','Uub'/)
!
      locprim(1)=0
      locbf(1)=0
      ishell= 0
!
      do iatom= 1,natom
        nn= numatomic(iatom)
        select case(atombasis(nn))
          case('STO-3G')
            call bssto3g(iatom,ishell)
          case('3-21G')
            call bs321g(iatom,ishell)
          case('6-31G')
            call bs631g(iatom,ishell)
          case('6-31G(D)','6-31G*')
            call bs631gd(iatom,ishell)
          case('6-31G(D,P)','6-31G**')
            call bs631gdp(iatom,ishell)
          case('6-311G')
            call bs6311g(iatom,ishell)
          case('6-311G(D)','6-311G*')
            call bs6311gd(iatom,ishell)
          case('6-311G(D,P)','6-311G**')
            call bs6311gdp(iatom,ishell)
          case('CC-PVDZ')
            call bsccpvdz(iatom,ishell)
          case('CC-PVTZ')
            call bsccpvtz(iatom,ishell)
          case('CC-PVQZ')
            call bsccpvqz(iatom,ishell)
          case('D95V')
            call bsd95v(iatom,ishell)
          case('LANL2DZ')
            if(.not.flagecp) then
              write(*,'(" Error! ECP is not set!.")')
              call iabort
            endif
            if(nn > 0) then
              if(nn < 11) then
                call bsd95v(iatom,ishell)
              else
                call bslanl2dz(iatom,ishell)
              endif
            endif
          case('')
            if((nn > 0).and.(ngenshell(nn) == 0)) then
              write(*,'(" Error! Basis set for ",a3,"is not set.")')table(nn)
              call iabort
            endif
          case default
            write(*,'(" Error! This program is not support the basis set, ",a12,".")')atombasis(nn)
            call iabort
        end select
!
        ll= locgenshell(nn)
        do ii= 1,ngenshell(nn)
          ishell= ishell+1 
          lprim= locgenprim(ll+ii)
          do jj= 1,mgenprim(ll+ii)
            ex(locprim(ishell)+jj)= exgen(lprim+jj)
            coeff(locprim(ishell)+jj)= coeffgen(lprim+jj)
          enddo
          mprim(ishell)= mgenprim(ll+ii)
          mtype(ishell)= mgentype(ll+ii)
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+mgenprim(ll+ii)
          select case(mgentype(ll+ii))
            case(0)
              mbf(ishell)= 1
              locbf(ishell+1)= locbf(ishell)+1
            case(1)
              mbf(ishell)= 3
              locbf(ishell+1)= locbf(ishell)+3
            case(2)
              if(spher) then
                mbf(ishell)= 5
                locbf(ishell+1)= locbf(ishell)+5
              else
                mbf(ishell)= 6
                locbf(ishell+1)= locbf(ishell)+6
              endif
            case(3)
              if(spher) then
                mbf(ishell)= 7
                locbf(ishell+1)= locbf(ishell)+7
              else
                mbf(ishell)= 10
                locbf(ishell+1)= locbf(ishell)+10
              endif
            case(4)
              if(spher) then
                mbf(ishell)= 9
                locbf(ishell+1)= locbf(ishell)+9
              else
                mbf(ishell)= 15
                locbf(ishell+1)= locbf(ishell)+15
              endif
            case(5)
              if(spher) then
                mbf(ishell)= 11
                locbf(ishell+1)= locbf(ishell)+11
              else
                mbf(ishell)= 21
                locbf(ishell+1)= locbf(ishell)+21
              endif
            case(6)
              if(spher) then
                mbf(ishell)= 13
                locbf(ishell+1)= locbf(ishell)+13
              else
                mbf(ishell)= 28
                locbf(ishell+1)= locbf(ishell)+28
              endif
            case default
              write(*,'(" Error! This program supports up to i function.")')
              call iabort
          end select
        enddo
      enddo
!
      return
end


!-----------------------------------
  subroutine bssto3g(iatom,ishell)
!-----------------------------------
!
! Set basis functions of STO-3G
!
      use modmolecule, only : numatomic
      use modbasis, only : ex, coeff, locprim, locbf, locatom, mprim, mbf, mtype, spher
      use modparam, only : mxao, mxshell, mxprim
      implicit none
      integer,intent(in) :: iatom
      integer,intent(inout) :: ishell
      integer :: j
      integer :: is(54)=(/0,3, 6,12,18,24,30,36,42,48, 54,63,72,81,90,99,108,117, &
&                       126,138,150,162,174,186,198,210,222,234,246,258,270,282,294,306,318,330, &
                        342,357,372,387,402,417,432,447,462,477,492,507,522,537,552,567,582,597/)
      integer :: ip(54)=(/0,0, 0, 3, 6, 9,12,15,18,21, 24,30,36,42,48,54, 60, 66, &
&                        72, 81, 90, 99,108,117,126,135,144,153,162,171,180,189,198,207,216,225, &
&                       234,246,258,270,282,294,306,318,330,342,354,366,378,390,402,414,426,438/)
      integer :: id(21:54)=(/ 0, 3, 6, 9,12,15,18,21,24,27,30,33,36,39,42,45, &
&                        48, 51, 54, 60, 66, 72, 78, 84, 90, 96,102,108,114,120,126,132,138,144/)
      real(8) :: exsto(612)= &
&     (/ 0.3425250914D+01, 0.6239137298D+00, 0.1688554040D+00, 0.6362421394D+01, 0.1158922999D+01,&
&        0.3136497915D+00, 0.1611957475D+02, 0.2936200663D+01, 0.7946504870D+00, 0.6362897469D+00,&
&        0.1478600533D+00, 0.4808867840D-01, 0.3016787069D+02, 0.5495115306D+01, 0.1487192653D+01,&
&        0.1314833110D+01, 0.3055389383D+00, 0.9937074560D-01, 0.4879111318D+02, 0.8887362172D+01,&
&        0.2405267040D+01, 0.2236956142D+01, 0.5198204999D+00, 0.1690617600D+00, 0.7161683735D+02,&
&        0.1304509632D+02, 0.3530512160D+01, 0.2941249355D+01, 0.6834830964D+00, 0.2222899159D+00,&
&        0.9910616896D+02, 0.1805231239D+02, 0.4885660238D+01, 0.3780455879D+01, 0.8784966449D+00,&
&        0.2857143744D+00, 0.1307093214D+03, 0.2380886605D+02, 0.6443608313D+01, 0.5033151319D+01,&
&        0.1169596125D+01, 0.3803889600D+00, 0.1666791340D+03, 0.3036081233D+02, 0.8216820672D+01,&
&        0.6464803249D+01, 0.1502281245D+01, 0.4885884864D+00, 0.2070156070D+03, 0.3770815124D+02,&
&        0.1020529731D+02, 0.8246315120D+01, 0.1916266291D+01, 0.6232292721D+00, 0.2507724300D+03,&
&        0.4567851117D+02, 0.1236238776D+02, 0.1204019274D+02, 0.2797881859D+01, 0.9099580170D+00,&
&        0.1478740622D+01, 0.4125648801D+00, 0.1614750979D+00, 0.2992374137D+03, 0.5450646845D+02,&
&        0.1475157752D+02, 0.1512182352D+02, 0.3513986579D+01, 0.1142857498D+01, 0.1395448293D+01,&
&        0.3893265318D+00, 0.1523797659D+00, 0.3514214767D+03, 0.6401186067D+02, 0.1732410761D+02,&
&        0.1889939621D+02, 0.4391813233D+01, 0.1428353970D+01, 0.1395448293D+01, 0.3893265318D+00,&
&        0.1523797659D+00, 0.4077975514D+03, 0.7428083305D+02, 0.2010329229D+02, 0.2319365606D+02,&
&        0.5389706871D+01, 0.1752899952D+01, 0.1478740622D+01, 0.4125648801D+00, 0.1614750979D+00,&
&        0.4683656378D+03, 0.8531338559D+02, 0.2308913156D+02, 0.2803263958D+02, 0.6514182577D+01,&
&        0.2118614352D+01, 0.1743103231D+01, 0.4863213771D+00, 0.1903428909D+00, 0.5331257359D+03,&
&        0.9710951830D+02, 0.2628162542D+02, 0.3332975173D+02, 0.7745117521D+01, 0.2518952599D+01,&
&        0.2029194274D+01, 0.5661400518D+00, 0.2215833792D+00, 0.6013456136D+03, 0.1095358542D+03,&
&        0.2964467686D+02, 0.3896041889D+02, 0.9053563477D+01, 0.2944499834D+01, 0.2129386495D+01,&
&        0.5940934274D+00, 0.2325241410D+00, 0.6744465184D+03, 0.1228512753D+03, 0.3324834945D+02,&
&        0.4516424392D+02, 0.1049519900D+02, 0.3413364448D+01, 0.2621366518D+01, 0.7313546050D+00,&
&        0.2862472356D+00, 0.7715103681D+03, 0.1405315766D+03, 0.3803332899D+02, 0.5240203979D+02,&
&        0.1217710710D+02, 0.3960373165D+01, 0.3651583985D+01, 0.1018782663D+01, 0.3987446295D+00,&
&        0.5039822505D+00, 0.1860011465D+00, 0.8214006743D-01, 0.8540324951D+03, 0.1555630851D+03,&
&        0.4210144179D+02, 0.5956029944D+02, 0.1384053270D+02, 0.4501370797D+01, 0.4374706256D+01,&
&        0.1220531941D+01, 0.4777079300D+00, 0.4558489757D+00, 0.1682369410D+00, 0.7429520700D-01,&
&        0.9416624250D+03, 0.1715249862D+03, 0.4642135516D+02, 0.6717668771D+02, 0.1561041754D+02,&
&        0.5076992278D+01, 0.4698159231D+01, 0.1433088313D+01, 0.5529300240D+00, 0.6309328384D+00,&
&        0.2328538976D+00, 0.1028307363D+00, 0.1033571245D+04, 0.1882662926D+03, 0.5095220601D+02,&
&        0.7525120460D+02, 0.1748676162D+02, 0.5687237606D+01, 0.5395535474D+01, 0.1645810296D+01,&
&        0.6350047770D+00, 0.7122640246D+00, 0.2628702203D+00, 0.1160862609D+00, 0.1130762517D+04,&
&        0.2059698041D+03, 0.5574346711D+02, 0.8378385011D+02, 0.1946956493D+02, 0.6332106784D+01,&
&        0.6141151276D+01, 0.1873246881D+01, 0.7227568825D+00, 0.7122640246D+00, 0.2628702203D+00,&
&        0.1160862609D+00, 0.1232320450D+04, 0.2244687082D+03, 0.6074999251D+02, 0.9277462423D+02,&
&        0.2155882749D+02, 0.7011599810D+01, 0.6899488096D+01, 0.2104563782D+01, 0.8120061343D+00,&
&        0.7547780537D+00, 0.2785605708D+00, 0.1230152851D+00, 0.1337153266D+04, 0.2435641365D+03,&
&        0.6591796062D+02, 0.1020220021D+03, 0.2370771923D+02, 0.7710486098D+01, 0.7701960922D+01,&
&        0.2349343572D+01, 0.9064497870D+00, 0.6709822860D+00, 0.2476346630D+00, 0.1093580780D+00,&
&        0.1447400411D+04, 0.2636457916D+03, 0.7135284019D+02, 0.1119194891D+03, 0.2600768236D+02,&
&        0.8458505490D+01, 0.8548569754D+01, 0.2607586250D+01, 0.1006087840D+01, 0.5921156814D+00,&
&        0.2185279254D+00, 0.9650423590D-01, 0.1557298704D+04, 0.2836639029D+03, 0.7677052234D+02,&
&        0.1218344741D+03, 0.2831171164D+02, 0.9207847321D+01, 0.9480851678D+01, 0.2891961952D+01,&
&        0.1115808827D+01, 0.5921156814D+00, 0.2185279254D+00, 0.9650423590D-01, 0.1679771028D+04,&
&        0.3059723896D+03, 0.8280806943D+02, 0.1328588899D+03, 0.3087354878D+02, 0.1004103627D+02,&
&        0.1033074335D+02, 0.3151206003D+01, 0.1215833241D+01, 0.6309328384D+00, 0.2328538976D+00,&
&        0.1028307363D+00, 0.1801806730D+04, 0.3282013450D+03, 0.8882409228D+02, 0.1441212184D+03,&
&        0.3349067173D+02, 0.1089220588D+02, 0.1130775402D+02, 0.3449225397D+01, 0.1330818388D+01,&
&        0.6309328384D+00, 0.2328538976D+00, 0.1028307363D+00, 0.1929432301D+04, 0.3514485021D+03,&
&        0.9511568021D+02, 0.1558416755D+03, 0.3621425391D+02, 0.1177799934D+02, 0.1228152744D+02,&
&        0.3746257327D+01, 0.1445422541D+01, 0.8897138854D+00, 0.3283603790D+00, 0.1450074055D+00,&
&        0.2061424532D+04, 0.3754910517D+03, 0.1016225324D+03, 0.1677618680D+03, 0.3898425028D+02,&
&        0.1267888813D+02, 0.1261505520D+02, 0.3847993927D+01, 0.1484675684D+01, 0.7985243736D+00,&
&        0.2947057141D+00, 0.1301451506D+00, 0.2196384229D+04, 0.4000741292D+03, 0.1082756726D+03,&
&        0.1803890380D+03, 0.4191853304D+02, 0.1363320795D+02, 0.1419665619D+02, 0.4330432640D+01,&
&        0.1670815538D+01, 0.9858325600D+00, 0.3638342150D+00, 0.1606730254D+00, 0.2337065673D+04,&
&        0.4256994298D+03, 0.1152108790D+03, 0.1931970535D+03, 0.4489484040D+02, 0.1460119548D+02,&
&        0.1587163584D+02, 0.4841354819D+01, 0.1867945198D+01, 0.1107681464D+01, 0.4088041239D+00,&
&        0.1805322114D+00, 0.2480626814D+04, 0.4518492708D+03, 0.1222880464D+03, 0.2061578780D+03,&
&        0.4790665727D+02, 0.1558073180D+02, 0.1763999414D+02, 0.5380760465D+01, 0.2076064666D+01,&
&        0.1214644297D+01, 0.4482801363D+00, 0.1979652346D+00, 0.2629997471D+04, 0.4790573224D+03,&
&        0.1296516070D+03, 0.2198350255D+03, 0.5108493222D+02, 0.1661440546D+02, 0.1950173109D+02,&
&        0.5948649577D+01, 0.2295173940D+01, 0.1396037488D+01, 0.5152256318D+00, 0.2275290713D+00,&
&        0.2782160055D+04, 0.5067739270D+03, 0.1371528019D+03, 0.2339514118D+03, 0.5436527681D+02,&
&        0.1768127533D+02, 0.2145684671D+02, 0.6545022156D+01, 0.2525273021D+01, 0.1590049336D+01,&
&        0.5868282053D+00, 0.2591495227D+00, 0.2938601529D+04, 0.5352699370D+03, 0.1448649340D+03,&
&        0.2485070370D+03, 0.5774769100D+02, 0.1878134100D+02, 0.2350534097D+02, 0.7169878201D+01,&
&        0.2766361909D+01, 0.2247796820D+01, 0.8295783930D+00, 0.3663505650D+00, 0.4869939919D+00,&
&        0.2622161565D+00, 0.1158254875D+00, 0.3100983951D+04, 0.5648480978D+03, 0.1528699389D+03,&
&        0.2635019007D+03, 0.6123217493D+02, 0.1991460372D+02, 0.2557886692D+02, 0.7802369707D+01,&
&        0.3010396794D+01, 0.2461032403D+01, 0.9082757340D+00, 0.4011041400D+00, 0.4370804803D+00,&
&        0.2353408164D+00, 0.1039541771D+00, 0.3266026869D+04, 0.5949108710D+03, 0.1610060990D+03,&
&        0.2779377240D+03, 0.6458675000D+02, 0.2100561600D+02, 0.3067132600D+02, 0.8557222000D+01,&
&        0.3349239000D+01, 0.2667688000D+01, 0.9845440000D+00, 0.4347850000D+00, 0.2074240000D+00,&
&        0.1116850000D+00, 0.4933300000D-01, 0.3435348677D+04, 0.6257530498D+03, 0.1693531958D+03,&
&        0.2937830292D+03, 0.6826885797D+02, 0.2220315144D+02, 0.3073293103D+02, 0.9374523538D+01,&
&        0.3616982618D+01, 0.2827607815D+01, 0.1101055827D+01, 0.4846874856D+00, 0.8878301887D+00,&
&        0.3457164736D+00, 0.1521852428D+00, 0.3610742864D+04, 0.6577013201D+03, 0.1779996445D+03,&
&        0.3100675728D+03, 0.7205303569D+02, 0.2343388348D+02, 0.3301997858D+02, 0.1007214594D+02,&
&        0.3886147028D+01, 0.3144798430D+01, 0.1224568208D+01, 0.5390579399D+00, 0.4869939919D+00,&
&        0.2622161565D+00, 0.1158254875D+00, 0.3788666115D+04, 0.6901102623D+03, 0.1867707691D+03,&
&        0.3264309567D+03, 0.7585553420D+02, 0.2467057401D+02, 0.3546948129D+02, 0.1081932234D+02,&
&        0.4174430912D+01, 0.3496895188D+01, 0.1361672861D+01, 0.5994117456D+00, 0.5129625081D+00,&
&        0.2761985970D+00, 0.1220017773D+00, 0.3970868257D+04, 0.7232986098D+03, 0.1957528311D+03,&
&        0.3435846323D+03, 0.7984167952D+02, 0.2596699219D+02, 0.3808991983D+02, 0.1161863962D+02,&
&        0.4482832367D+01, 0.3829752708D+01, 0.1491285854D+01, 0.6564677040D+00, 0.4616999826D+00,&
&        0.2485968963D+00, 0.1098096207D+00, 0.4159274210D+04, 0.7576169894D+03, 0.2050407239D+03,&
&        0.3607986561D+03, 0.8384184843D+02, 0.2726797127D+02, 0.4071751678D+02, 0.1242014044D+02,&
&        0.4792076302D+01, 0.4197516371D+01, 0.1634491118D+01, 0.7195070139D+00, 0.4131354848D+00,&
&        0.2224479167D+00, 0.9825915660D-01, 0.4350077794D+04, 0.7923721005D+03, 0.2144468133D+03,&
&        0.3784334264D+03, 0.8793978981D+02, 0.2860074899D+02, 0.4352179455D+02, 0.1327553454D+02,&
&        0.5122113939D+01, 0.4540857408D+01, 0.1768186338D+01, 0.7783599789D+00, 0.4131354848D+00,&
&        0.2224479167D+00, 0.9825915660D-01, 0.4545160269D+04, 0.8279066168D+03, 0.2240638402D+03,&
&        0.3964889433D+03, 0.9213550365D+02, 0.2996532535D+02, 0.4641945097D+02, 0.1415941211D+02,&
&        0.5463141383D+01, 0.4919104589D+01, 0.1915473830D+01, 0.8431962954D+00, 0.4370804803D+00,&
&        0.2353408164D+00, 0.1039541771D+00, 0.4744521634D+04, 0.8642205383D+03, 0.2338918045D+03,&
&        0.4149652069D+03, 0.9642898995D+02, 0.3136170035D+02, 0.4941048605D+02, 0.1507177314D+02,&
&        0.5815158634D+01, 0.5290230450D+01, 0.2059988316D+01, 0.9068119281D+00, 0.4370804803D+00,&
&        0.2353408164D+00, 0.1039541771D+00, 0.4950261905D+04, 0.9016963856D+03, 0.2440342313D+03,&
&        0.4334469385D+03, 0.1007237469D+03, 0.3275848861D+02, 0.5259279235D+02, 0.1604247800D+02,&
&        0.6189686744D+01, 0.5674851796D+01, 0.2209757875D+01, 0.9727408566D+00, 0.5949150981D+00,&
&        0.3203250000D+00, 0.1414931855D+00, 0.5158224714D+04, 0.9395770707D+03, 0.2542862231D+03,&
&        0.4523313223D+03, 0.1051120716D+03, 0.3418570799D+02, 0.5597539769D+02, 0.1707428044D+02,&
&        0.6587788204D+01, 0.5048549180D+01, 0.1965878882D+01, 0.8653847237D+00, 0.5669230612D+00,&
&        0.3052530187D+00, 0.1348356264D+00, 0.5370466413D+04, 0.9782371611D+03, 0.2647491522D+03,&
&        0.4720515322D+03, 0.1096946243D+03, 0.3567609636D+02, 0.5915141188D+02, 0.1804306600D+02,&
&        0.6961575790D+01, 0.5583138529D+01, 0.2174045204D+01, 0.9570200509D+00, 0.6235816420D+00,&
&        0.3357601616D+00, 0.1483111678D+00, 0.5586987002D+04, 0.1017676657D+04, 0.2754230189D+03,&
&        0.4921924888D+03, 0.1143749494D+03, 0.3719828336D+02, 0.6252179775D+02, 0.1907114112D+02,&
&        0.7358239131D+01, 0.6120693149D+01, 0.2383366187D+01, 0.1049163663D+01, 0.6529226928D+00,&
&        0.3515585034D+00, 0.1552895732D+00, 0.5810061591D+04, 0.1058309972D+04, 0.2864199797D+03,&
&        0.5127541920D+03, 0.1191530471D+03, 0.3875226900D+02, 0.6598556227D+02, 0.2012769970D+02,&
&        0.7765892279D+01, 0.6707956921D+01, 0.2612043655D+01, 0.1149828048D+01, 0.7012713483D+00,&
&        0.3775912653D+00, 0.1667887020D+00, 0.6035183623D+04, 0.1099316231D+04, 0.2975178737D+03,&
&        0.5337366418D+03, 0.1240289171D+03, 0.4033805328D+02, 0.6954270545D+02, 0.2121274175D+02,&
&        0.8184535234D+01, 0.7295991196D+01, 0.2841021154D+01, 0.1250624506D+01, 0.7900364582D+00,&
&        0.4253857892D+00, 0.1879003836D+00, 0.6264584546D+04, 0.1141101895D+04, 0.3088267052D+03,&
&        0.5551398381D+03, 0.1290025597D+03, 0.4195563620D+02, 0.7307773504D+02, 0.2229103845D+02,&
&        0.8600575622D+01, 0.7908728280D+01, 0.3079617799D+01, 0.1355655337D+01, 0.8910101433D+00,&
&        0.4797538759D+00, 0.2119157236D+00/)
      real(8) :: cssto(612)= &
&     (/ 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00, 0.1543289673D+00, 0.5353281423D+00,&
&        0.4446345422D+00, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722919D-01,&
&        0.3995128261D+00, 0.7001154689D+00, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00, 0.1543289673D+00, 0.5353281423D+00,&
&        0.4446345422D+00,-0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00, 0.1543289673D+00,&
&        0.5353281423D+00, 0.4446345422D+00,-0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00,&
&        0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722919D-01, 0.3995128261D+00,&
&        0.7001154689D+00, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722919D-01,&
&        0.3995128261D+00, 0.7001154689D+00, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00, 0.1543289673D+00, 0.5353281423D+00,&
&        0.4446345422D+00,-0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00, 0.1543289673D+00,&
&        0.5353281423D+00, 0.4446345422D+00,-0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00,&
&       -0.2196203690D+00, 0.2255954336D+00, 0.9003984260D+00, 0.1543289673D+00, 0.5353281423D+00,&
&        0.4446345422D+00,-0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2196203690D+00,&
&        0.2255954336D+00, 0.9003984260D+00, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2196203690D+00, 0.2255954336D+00,&
&        0.9003984260D+00, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722919D-01,&
&        0.3995128261D+00, 0.7001154689D+00,-0.2196203690D+00, 0.2255954336D+00, 0.9003984260D+00,&
&        0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722919D-01, 0.3995128261D+00,&
&        0.7001154689D+00,-0.2196203690D+00, 0.2255954336D+00, 0.9003984260D+00, 0.1543289673D+00,&
&        0.5353281423D+00, 0.4446345422D+00,-0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00,&
&       -0.2196203690D+00, 0.2255954336D+00, 0.9003984260D+00, 0.1543289673D+00, 0.5353281423D+00,&
&        0.4446345422D+00,-0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2196203690D+00,&
&        0.2255954336D+00, 0.9003984260D+00, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2196203690D+00, 0.2255954336D+00,&
&        0.9003984260D+00, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01,&
&        0.3995128261D+00, 0.7001154689D+00,-0.2196203690D+00, 0.2255954336D+00, 0.9003984260D+00,&
&       -0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00,&
&        0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2196203690D+00,&
&        0.2255954336D+00, 0.9003984260D+00,-0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01,&
&        0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00,&
&        0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00, 0.9166769611D+00,-0.3088441215D+00,&
&        0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00,&
&        0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,&
&       -0.2277635023D+00, 0.2175436044D+00, 0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01,&
&        0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01,&
&        0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00, 0.9166769611D+00,&
&       -0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00,&
&        0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00,&
&        0.2175436044D+00, 0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01,&
&        0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00,&
&        0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00, 0.9166769611D+00,-0.3088441215D+00,&
&        0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00,&
&        0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,&
&       -0.2277635023D+00, 0.2175436044D+00, 0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01,&
&        0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01,&
&        0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00, 0.9166769611D+00,&
&       -0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00,&
&        0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00,&
&        0.2175436044D+00, 0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01,&
&        0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00,&
&        0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00, 0.9166769611D+00,-0.3088441215D+00,&
&        0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00,&
&        0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,&
&       -0.2277635023D+00, 0.2175436044D+00, 0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01,&
&        0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01,&
&        0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00, 0.9166769611D+00,&
&       -0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00,&
&        0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00,&
&        0.2175436044D+00, 0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01,&
&        0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00,&
&        0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00, 0.9166769611D+00,-0.3088441215D+00,&
&        0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095330D-01, 0.1155787450D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095330D-01, 0.1155787450D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095330D-01, 0.1155787450D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095330D-01, 0.1155787450D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095330D-01, 0.1155787450D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095330D-01, 0.1155787450D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095330D-01, 0.1155787450D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095330D-01, 0.1155787450D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095330D-01, 0.1155787450D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095330D-01, 0.1155787450D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095340D-01, 0.1115578745D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095340D-01, 0.1115578745D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095340D-01, 0.1115578745D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095340D-01, 0.1115578745D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095340D-01, 0.1115578745D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095338D-01, 0.1115578745D+01,-0.3842642608D+00,&
&       -0.1972567438D+00, 0.1375495512D+01/)
      real(8) :: cpsto(450)= &
&     (/ 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.1559162750D+00, 0.6076837186D+00,&
&        0.3919573931D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.1559162750D+00,&
&        0.6076837186D+00, 0.3919573931D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00,&
&        0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.1559162750D+00, 0.6076837186D+00,&
&        0.3919573931D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.1559162750D+00,&
&        0.6076837186D+00, 0.3919573931D+00, 0.1058760429D-01, 0.5951670053D+00, 0.4620010120D+00,&
&        0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.1058760429D-01, 0.5951670053D+00,&
&        0.4620010120D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.1058760429D-01,&
&        0.5951670053D+00, 0.4620010120D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00,&
&        0.1058760429D-01, 0.5951670053D+00, 0.4620010120D+00, 0.1559162750D+00, 0.6076837186D+00,&
&        0.3919573931D+00, 0.1058760429D-01, 0.5951670053D+00, 0.4620010120D+00, 0.1559162750D+00,&
&        0.6076837186D+00, 0.3919573931D+00, 0.1058760429D-01, 0.5951670053D+00, 0.4620010120D+00,&
&        0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.1058760429D-01, 0.5951670053D+00,&
&        0.4620010120D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.1058760429D-01,&
&        0.5951670053D+00, 0.4620010120D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00,&
&        0.1058760430D-01, 0.5951670053D+00, 0.4620010120D+00,-0.1215468600D+00, 0.5715227604D+00,&
&        0.5498949471D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.1058760430D-01,&
&        0.5951670053D+00, 0.4620010120D+00,-0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00,&
&        0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00,&
&        0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00, 0.1559162750D+00,&
&        0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,&
&       -0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00, 0.1559162750D+00, 0.6076837186D+00,&
&        0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1215468600D+00,&
&        0.5715227604D+00, 0.5498949471D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00,&
&        0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00,&
&        0.5498949471D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02,&
&        0.5777664691D+00, 0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00,&
&        0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00,&
&        0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00, 0.1559162750D+00,&
&        0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,&
&       -0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00, 0.1559162750D+00, 0.6076837186D+00,&
&        0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1215468600D+00,&
&        0.5715227604D+00, 0.5498949471D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00,&
&        0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00,&
&        0.5498949471D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02,&
&        0.5777664691D+00, 0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00,&
&        0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00,&
&        0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00, 0.1559162750D+00,&
&        0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,&
&       -0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00, 0.1559162750D+00, 0.6076837186D+00,&
&        0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1215468600D+00,&
&        0.5715227604D+00, 0.5498949471D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00,&
&        0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00,&
&        0.5498949471D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02,&
&        0.5777664691D+00, 0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00,&
&        0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00,&
&        0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00, 0.1559162750D+00,&
&        0.6076837186D+00, 0.3919573931D+00, 0.4951511100D-02, 0.5777664691D+00, 0.4846460366D+00,&
&       -0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00,-0.3481691526D+00, 0.6290323690D+00,&
&        0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511100D-02,&
&        0.5777664691D+00, 0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00,&
&       -0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00,&
&        0.3919573931D+00, 0.4951511100D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1283927634D+00,&
&        0.5852047641D+00, 0.5439442040D+00,-0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00,&
&        0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511100D-02, 0.5777664691D+00,&
&        0.4846460366D+00,-0.1283927634D+00, 0.5852047641D+00, 0.5439442040D+00,-0.3481691526D+00,&
&        0.6290323690D+00, 0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00,&
&        0.4951511100D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1283927634D+00, 0.5852047641D+00,&
&        0.5439442040D+00,-0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00, 0.1559162750D+00,&
&        0.6076837186D+00, 0.3919573931D+00, 0.4951511100D-02, 0.5777664691D+00, 0.4846460366D+00,&
&       -0.1283927634D+00, 0.5852047641D+00, 0.5439442040D+00,-0.3481691526D+00, 0.6290323690D+00,&
&        0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511100D-02,&
&        0.5777664691D+00, 0.4846460366D+00,-0.1283927634D+00, 0.5852047641D+00, 0.5439442040D+00,&
&       -0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00,&
&        0.3919573931D+00, 0.4951511100D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1283927634D+00,&
&        0.5852047641D+00, 0.5439442040D+00,-0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00,&
&        0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511100D-02, 0.5777664691D+00,&
&        0.4846460366D+00,-0.1283927634D+00, 0.5852047641D+00, 0.5439442040D+00,-0.3481691526D+00,&
&        0.6290323690D+00, 0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00,&
&        0.4951511100D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1283927634D+00, 0.5852047641D+00,&
&        0.5439442040D+00,-0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00, 0.1559162750D+00,&
&        0.6076837186D+00, 0.3919573931D+00, 0.4951511100D-02, 0.5777664691D+00, 0.4846460366D+00,&
&       -0.1283927634D+00, 0.5852047641D+00, 0.5439442040D+00,-0.3481691526D+00, 0.6290323690D+00,&
&        0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511100D-02,&
&        0.5777664691D+00, 0.4846460366D+00,-0.1283927634D+00, 0.5852047641D+00, 0.5439442040D+00,&
&       -0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00,&
&        0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1283927634D+00,&
&        0.5852047641D+00, 0.5439442040D+00,-0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00,&
&        0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00,&
&        0.4846460366D+00,-0.1283927634D+00, 0.5852047641D+00, 0.5439442040D+00,-0.3481691526D+00,&
&        0.6290323690D+00, 0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00,&
&        0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1283927634D+00, 0.5852047641D+00,&
&        0.5439442040D+00,-0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00, 0.1559162750D+00,&
&        0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,&
&       -0.1283927634D+00, 0.5852047641D+00, 0.5439442040D+00,-0.3481691526D+00, 0.6290323690D+00,&
&        0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02,&
&        0.5777664691D+00, 0.4846460366D+00,-0.1283927634D+00, 0.5852047641D+00, 0.5439442040D+00,&
&       -0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00,&
&        0.3919573931D+00, 0.4951511155D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1283927634D+00,&
&        0.5852047641D+00, 0.5439442040D+00,-0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00/)
      real(8) :: exdsto(150)= &
&     (/ 0.5517000679D+00, 0.1682861055D+00, 0.6493001120D-01, 0.1645981194D+01, 0.5020767280D+00,&
&        0.1937168100D+00, 0.2964817927D+01, 0.9043639676D+00, 0.3489317337D+00, 0.4241479241D+01,&
&        0.1293786360D+01, 0.4991829993D+00, 0.5426950461D+01, 0.1655392868D+01, 0.6387020320D+00,&
&        0.6411803475D+01, 0.1955804428D+01, 0.7546101510D+00, 0.7664527389D+01, 0.2337925151D+01,&
&        0.9020442050D+00, 0.8627722755D+01, 0.2631730438D+01, 0.1015403419D+01, 0.9647911930D+01,&
&        0.2942920654D+01, 0.1135470278D+01, 0.1094737077D+02, 0.3339297018D+01, 0.1288404602D+01,&
&        0.1261505520D+02, 0.3847993927D+01, 0.1484675684D+01, 0.1419665619D+02, 0.4330432640D+01,&
&        0.1670815538D+01, 0.1587163584D+02, 0.4841354819D+01, 0.1867945198D+01, 0.1763999414D+02,&
&        0.5380760465D+01, 0.2076064666D+01, 0.1950173109D+02, 0.5948649577D+01, 0.2295173940D+01,&
&        0.2145684671D+02, 0.6545022156D+01, 0.2525273021D+01, 0.2350534097D+02, 0.7169878201D+01,&
&        0.2766361909D+01, 0.2557886692D+02, 0.7802369707D+01, 0.3010396794D+01, 0.5660043000D+01,&
&        0.1774715000D+01, 0.6912950000D+00, 0.2128212000D+01, 0.9625940000D+00, 0.4728610000D+00,&
&        0.3073293103D+02, 0.9374523538D+01, 0.3616982618D+01, 0.4869939919D+00, 0.2622161565D+00,&
&        0.1158254875D+00, 0.3301997858D+02, 0.1007214594D+02, 0.3886147028D+01, 0.1344878866D+01,&
&        0.5236888594D+00, 0.2305291251D+00, 0.3546948129D+02, 0.1081932234D+02, 0.4174430912D+01,&
&        0.1702112315D+01, 0.6627937127D+00, 0.2917634240D+00, 0.3808991983D+02, 0.1161863962D+02,&
&        0.4482832367D+01, 0.2101373228D+01, 0.8182638428D+00, 0.3602017580D+00, 0.4071751678D+02,&
&        0.1242014044D+02, 0.4792076302D+01, 0.2390895761D+01, 0.9310024167D+00, 0.4098295558D+00,&
&        0.4352179455D+02, 0.1327553454D+02, 0.5122113939D+01, 0.2779066094D+01, 0.1082153932D+01,&
&        0.4763668250D+00, 0.4641945097D+02, 0.1415941211D+02, 0.5463141383D+01, 0.3025977448D+01,&
&        0.1178299934D+01, 0.5186905316D+00, 0.4941048605D+02, 0.1507177314D+02, 0.5815158634D+01,&
&        0.3283395668D+01, 0.1278537254D+01, 0.5628152469D+00, 0.5259279235D+02, 0.1604247800D+02,&
&        0.6189686744D+01, 0.3642963976D+01, 0.1418551290D+01, 0.6244497700D+00, 0.5597539769D+02,&
&        0.1707428044D+02, 0.6587788204D+01, 0.5048549180D+01, 0.1965878882D+01, 0.8653847237D+00,&
&        0.5915141188D+02, 0.1804306600D+02, 0.6961575790D+01, 0.5583138529D+01, 0.2174045204D+01,&
&        0.9570200509D+00, 0.6252179775D+02, 0.1907114112D+02, 0.7358239131D+01, 0.6120693149D+01,&
&        0.2383366187D+01, 0.1049163663D+01, 0.6598556227D+02, 0.2012769970D+02, 0.7765892279D+01,&
&        0.6707956921D+01, 0.2612043655D+01, 0.1149828048D+01, 0.6954270545D+02, 0.2121274175D+02,&
&        0.8184535234D+01, 0.7295991196D+01, 0.2841021154D+01, 0.1250624506D+01, 0.7307773504D+02,&
&        0.2229103845D+02, 0.8600575622D+01, 0.7908728280D+01, 0.3079617799D+01, 0.1355655337D+01/)
      real(8) :: cdsto(150)= &
&     (/ 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00,&
&        0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00,&
&        0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00,&
&        0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00,&
&        0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00,&
&        0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00,&
&        0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00,&
&        0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00,&
&        0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00,&
&        0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00,&
&        0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00,&
&        0.6555473627D+00, 0.2865732590D+00, 0.1250662138D+00, 0.6686785577D+00, 0.3052468245D+00,&
&        0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.1250662138D+00, 0.6686785577D+00,&
&        0.3052468245D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.1250662138D+00,&
&        0.6686785577D+00, 0.3052468245D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00,&
&        0.1250662138D+00, 0.6686785577D+00, 0.3052468245D+00, 0.2197679508D+00, 0.6555473627D+00,&
&        0.2865732590D+00, 0.1250662138D+00, 0.6686785577D+00, 0.3052468245D+00, 0.2197679508D+00,&
&        0.6555473627D+00, 0.2865732590D+00, 0.1250662138D+00, 0.6686785577D+00, 0.3052468245D+00,&
&        0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.1250662138D+00, 0.6686785577D+00,&
&        0.3052468245D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.1250662138D+00,&
&        0.6686785577D+00, 0.3052468245D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00,&
&        0.1250662138D+00, 0.6686785577D+00, 0.3052468245D+00, 0.2197679508D+00, 0.6555473627D+00,&
&        0.2865732590D+00, 0.1250662138D+00, 0.6686785577D+00, 0.3052468245D+00, 0.2197679508D+00,&
&        0.6555473627D+00, 0.2865732590D+00, 0.1250662138D+00, 0.6686785577D+00, 0.3052468245D+00,&
&        0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.1250662138D+00, 0.6686785577D+00,&
&        0.3052468245D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.1250662138D+00,&
&        0.6686785577D+00, 0.3052468245D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00,&
&        0.1250662138D+00, 0.6686785577D+00, 0.3052468245D+00, 0.2197679508D+00, 0.6555473627D+00,&
&        0.2865732590D+00, 0.1250662138D+00, 0.6686785577D+00, 0.3052468245D+00, 0.2197679508D+00,&
&        0.6555473627D+00, 0.2865732590D+00, 0.1250662138D+00, 0.6686785577D+00, 0.3052468245D+00/)
!
      if(numatomic(iatom) > 54) then
        write(*,'(" Error! This program supports H - Xe STO-3G basis set.")')
        call iabort
      elseif(numatomic(iatom) <= 0) then
        return
      endif
!
! Set 1S functions
!
      ishell= ishell+1
      do j= 1,3
        ex(locprim(ishell)+j)= exsto(is(numatomic(iatom))+j)
        coeff(locprim(ishell)+j)= cssto(is(numatomic(iatom))+j)
      enddo
      mprim(ishell)= 3
      mbf(ishell)= 1
      mtype(ishell)= 0
      locatom(ishell)= iatom
      locprim(ishell+1)= locprim(ishell)+3
      locbf(ishell+1) = locbf(ishell)+1
!
! Set 2SP functions
!
      if(numatomic(iatom) >= 3) then
! S function
        ishell= ishell+1
        do j= 1,3
          ex(locprim(ishell)+j)= exsto(is(numatomic(iatom))+j+3)
          coeff(locprim(ishell)+j)= cssto(is(numatomic(iatom))+j+3)
        enddo
        mprim(ishell)= 3
        mbf(ishell)= 1
        mtype(ishell)= 0
        locatom(ishell)= iatom
        locprim(ishell+1)= locprim(ishell)+3
        locbf(ishell+1) = locbf(ishell)+1
! P function
        ishell= ishell+1
        do j= 1,3
          ex(locprim(ishell)+j)= exsto(is(numatomic(iatom))+j+3)
          coeff(locprim(ishell)+j)= cpsto(ip(numatomic(iatom))+j)
        enddo
        mprim(ishell)= 3
        mbf(ishell)= 3
        mtype(ishell)= 1
        locatom(ishell)= iatom
        locprim(ishell+1)= locprim(ishell)+3
        locbf(ishell+1) = locbf(ishell)+3
      endif
!
! Set 3SP functions
!
      if(numatomic(iatom) >= 11) then
! S function
        ishell= ishell+1
        do j= 1,3
          ex(locprim(ishell)+j)= exsto(is(numatomic(iatom))+j+6)
          coeff(locprim(ishell)+j)= cssto(is(numatomic(iatom))+j+6)
        enddo
        mprim(ishell)= 3
        mbf(ishell)= 1
        mtype(ishell)= 0
        locatom(ishell)= iatom
        locprim(ishell+1)= locprim(ishell)+3
        locbf(ishell+1) = locbf(ishell)+1
! P function
        ishell= ishell+1
        do j= 1,3
          ex(locprim(ishell)+j)= exsto(is(numatomic(iatom))+j+6)
          coeff(locprim(ishell)+j)= cpsto(ip(numatomic(iatom))+j+3)
        enddo
        mprim(ishell)= 3
        mbf(ishell)= 3
        mtype(ishell)= 1
        locatom(ishell)= iatom
        locprim(ishell+1)= locprim(ishell)+3
        locbf(ishell+1) = locbf(ishell)+3
      endif
!
! Set 4SP functions
!
      if(numatomic(iatom) >= 19) then
! S function
        ishell= ishell+1
        do j= 1,3
          ex(locprim(ishell)+j)= exsto(is(numatomic(iatom))+j+9)
          coeff(locprim(ishell)+j)= cssto(is(numatomic(iatom))+j+9)
        enddo
        mprim(ishell)= 3
        mbf(ishell)= 1
        mtype(ishell)= 0
        locatom(ishell)= iatom
        locprim(ishell+1)= locprim(ishell)+3
        locbf(ishell+1) = locbf(ishell)+1
! P function
        ishell= ishell+1
        do j= 1,3
          ex(locprim(ishell)+j)= exsto(is(numatomic(iatom))+j+9)
          coeff(locprim(ishell)+j)= cpsto(ip(numatomic(iatom))+j+6)
        enddo
        mprim(ishell)= 3
        mbf(ishell)= 3
        mtype(ishell)= 1
        locatom(ishell)= iatom
        locprim(ishell+1)= locprim(ishell)+3
        locbf(ishell+1) = locbf(ishell)+3
      endif
!
! Set 3D functions
!
      if(numatomic(iatom) >= 21) then
! D function
        ishell= ishell+1
        do j= 1,3
          ex(locprim(ishell)+j)= exdsto(id(numatomic(iatom))+j)
          coeff(locprim(ishell)+j)= cdsto(id(numatomic(iatom))+j)
        enddo
        mprim(ishell)= 3
        mtype(ishell)= 2
        locatom(ishell)= iatom
        locprim(ishell+1)= locprim(ishell)+3
        if(spher) then
          mbf(ishell)= 5
          locbf(ishell+1) = locbf(ishell)+5
        else
          mbf(ishell)= 6
          locbf(ishell+1) = locbf(ishell)+6
        endif
      endif
!
! Set 5SP functions
!
      if(numatomic(iatom) >= 37) then
! S function
        ishell= ishell+1
        do j= 1,3
          ex(locprim(ishell)+j)= exsto(is(numatomic(iatom))+j+12)
          coeff(locprim(ishell)+j)= cssto(is(numatomic(iatom))+j+12)
        enddo
        mprim(ishell)= 3
        mbf(ishell)= 1
        mtype(ishell)= 0
        locatom(ishell)= iatom
        locprim(ishell+1)= locprim(ishell)+3
        locbf(ishell+1) = locbf(ishell)+1
! P function
        ishell= ishell+1
        do j= 1,3
          ex(locprim(ishell)+j)= exsto(is(numatomic(iatom))+j+12)
          coeff(locprim(ishell)+j)= cpsto(ip(numatomic(iatom))+j+9)
        enddo
        mprim(ishell)= 3
        mbf(ishell)= 3
        mtype(ishell)= 1
        locatom(ishell)= iatom
        locprim(ishell+1)= locprim(ishell)+3
        locbf(ishell+1) = locbf(ishell)+3
      endif
!
! Set 3D functions
!
      if(numatomic(iatom) >= 39) then
! D function
        ishell= ishell+1
        do j= 1,3
          ex(locprim(ishell)+j)= exdsto(id(numatomic(iatom))+j+3)
          coeff(locprim(ishell)+j)= cdsto(id(numatomic(iatom))+j+3)
        enddo
        mprim(ishell)= 3
        mtype(ishell)= 2
        locatom(ishell)= iatom
        locprim(ishell+1)= locprim(ishell)+3
        if(spher) then
          mbf(ishell)= 5
          locbf(ishell+1) = locbf(ishell)+5
        else
          mbf(ishell)= 6
          locbf(ishell+1) = locbf(ishell)+6
        endif
      endif
!
      if(ishell > mxshell) then
        write(*,'(" Error! The number of basis shells exceeds mxshell",i6,".")')mxshell
        call iabort
      endif
      if(locprim(ishell+1) > mxprim ) then
        write(*,'(" Error! The number of primitive basis functions exceeds mxprim",i6,".")')mxprim
        call iabort
      endif
      if(locbf(ishell+1) > mxao ) then
        write(*,'(" Error! The number of basis functions exceeds mxao",i6,".")')mxao
        call iabort
      endif
      return
end


!-------------------------------
  subroutine setbasis_g(itype)
!-------------------------------
!
! Driver of setting basis functions for guess calculations
!
! In : itype =1 (For normal extended Huckel calculation)
!            =2 (For extended Huckel calculation of only core orbitals)
!
      use modmolecule, only : natom, numatomic
      use modguess, only : locprim_g, locbf_g, nshell_v, nao_v, nprim_v, nshell_g, &
&                          nao_g, nprim_g, nao_c, &
&                          locprim_gcore, locbf_gcore, nao_gcore, nprim_gcore, nshell_gcore
      implicit none
      integer,intent(in) :: itype
      integer :: ishell, iatom
!
      select case(itype)
!
! For normal extended Huckel calculation
!
        case(1)
          ishell= 0
          locprim_g(1)=0
          locbf_g(1)=0
!
! Set valence basis functions
!
          do iatom= 1,natom
            if((numatomic(iatom) >= 1).and.(numatomic(iatom) <= 54)) then
              call bssto3g_g(iatom,ishell,1)
            elseif(numatomic(iatom) >= 55) then
              call bshuzmini6_g(iatom,ishell,1)
            endif
          enddo
          nshell_v= ishell
          nao_v   = locbf_g(ishell+1)
          nprim_v = locprim_g(ishell+1)
!
! Set core basis functions
!
          do iatom= 1,natom
            if((numatomic(iatom) >= 1).and.(numatomic(iatom) <= 54)) then
              call bssto3g_g(iatom,ishell,2)
            elseif(numatomic(iatom) >= 55) then
              call bshuzmini6_g(iatom,ishell,2)
            endif
          enddo
          nshell_g= ishell
          nao_g   = locbf_g(ishell+1)
          nao_c   = nao_g-nao_v
          nprim_g = locprim_g(ishell+1)
          call bsnrmlz_g
!
! For extended Huckel calculation of only core orbitals
!
        case(2)
          ishell= 0
          locprim_gcore(1)=0
          locbf_gcore(1)=0
          do iatom= 1,natom
            if((numatomic(iatom) >= 1).and.(numatomic(iatom) <= 54)) then
              call bssto3g_g(iatom,ishell,3)
            elseif(numatomic(iatom) >= 55) then
              call bshuzmini6_g(iatom,ishell,3)
            endif
          enddo
          nshell_gcore= ishell
          nao_gcore   = locbf_gcore(ishell+1)
          nprim_gcore = locprim_gcore(ishell+1)
          call bsnrmlz_gcore
      end select
!
      return
end


!-------------------------------------------
  subroutine bssto3g_g(iatom,ishell,itype)
!-------------------------------------------
!
! Set basis functions of STO-3G for guess calculation
! itype = 1 : valence functions
!       = 2 : core functions
!       = 3 : core functions for core calculation
!
      use modmolecule, only : numatomic
      use modguess, only : ex_g, coeff_g, locprim_g, locbf_g, locatom_g, mprim_g, mbf_g, &
&                          mtype_g, spher_g, ex_gcore, coeff_gcore, locprim_gcore, &
&                          locbf_gcore, locatom_gcore, mprim_gcore, mbf_gcore, mtype_gcore
      use modparam, only : mxao, mxshell, mxprim
      use modecp, only : flagecp, izcore
      implicit none
      integer,intent(in) :: iatom, itype
      integer,intent(inout):: ishell
      integer :: j
      integer :: is(54)=(/0,3, 6,12,18,24,30,36,42,48, 54,63,72,81,90,99,108,117, &
&                       126,138,150,162,174,186,198,210,222,234,246,258,270,282,294,306,318,330, &
                        342,357,372,387,402,417,432,447,462,477,492,507,522,537,552,567,582,597/)
      integer :: ip(54)=(/0,0, 0, 3, 6, 9,12,15,18,21, 24,30,36,42,48,54, 60, 66, &
&                        72, 81, 90, 99,108,117,126,135,144,153,162,171,180,189,198,207,216,225, &
&                       234,246,258,270,282,294,306,318,330,342,354,366,378,390,402,414,426,438/)
      integer :: id(21:54)=(/ 0, 3, 6, 9,12,15,18,21,24,27,30,33,36,39,42,45, &
&                        48, 51, 54, 60, 66, 72, 78, 84, 90, 96,102,108,114,120,126,132,138,144/)
      real(8) :: exsto(612)= &
&     (/ 0.3425250914D+01, 0.6239137298D+00, 0.1688554040D+00, 0.6362421394D+01, 0.1158922999D+01,&
&        0.3136497915D+00, 0.1611957475D+02, 0.2936200663D+01, 0.7946504870D+00, 0.6362897469D+00,&
&        0.1478600533D+00, 0.4808867840D-01, 0.3016787069D+02, 0.5495115306D+01, 0.1487192653D+01,&
&        0.1314833110D+01, 0.3055389383D+00, 0.9937074560D-01, 0.4879111318D+02, 0.8887362172D+01,&
&        0.2405267040D+01, 0.2236956142D+01, 0.5198204999D+00, 0.1690617600D+00, 0.7161683735D+02,&
&        0.1304509632D+02, 0.3530512160D+01, 0.2941249355D+01, 0.6834830964D+00, 0.2222899159D+00,&
&        0.9910616896D+02, 0.1805231239D+02, 0.4885660238D+01, 0.3780455879D+01, 0.8784966449D+00,&
&        0.2857143744D+00, 0.1307093214D+03, 0.2380886605D+02, 0.6443608313D+01, 0.5033151319D+01,&
&        0.1169596125D+01, 0.3803889600D+00, 0.1666791340D+03, 0.3036081233D+02, 0.8216820672D+01,&
&        0.6464803249D+01, 0.1502281245D+01, 0.4885884864D+00, 0.2070156070D+03, 0.3770815124D+02,&
&        0.1020529731D+02, 0.8246315120D+01, 0.1916266291D+01, 0.6232292721D+00, 0.2507724300D+03,&
&        0.4567851117D+02, 0.1236238776D+02, 0.1204019274D+02, 0.2797881859D+01, 0.9099580170D+00,&
&        0.1478740622D+01, 0.4125648801D+00, 0.1614750979D+00, 0.2992374137D+03, 0.5450646845D+02,&
&        0.1475157752D+02, 0.1512182352D+02, 0.3513986579D+01, 0.1142857498D+01, 0.1395448293D+01,&
&        0.3893265318D+00, 0.1523797659D+00, 0.3514214767D+03, 0.6401186067D+02, 0.1732410761D+02,&
&        0.1889939621D+02, 0.4391813233D+01, 0.1428353970D+01, 0.1395448293D+01, 0.3893265318D+00,&
&        0.1523797659D+00, 0.4077975514D+03, 0.7428083305D+02, 0.2010329229D+02, 0.2319365606D+02,&
&        0.5389706871D+01, 0.1752899952D+01, 0.1478740622D+01, 0.4125648801D+00, 0.1614750979D+00,&
&        0.4683656378D+03, 0.8531338559D+02, 0.2308913156D+02, 0.2803263958D+02, 0.6514182577D+01,&
&        0.2118614352D+01, 0.1743103231D+01, 0.4863213771D+00, 0.1903428909D+00, 0.5331257359D+03,&
&        0.9710951830D+02, 0.2628162542D+02, 0.3332975173D+02, 0.7745117521D+01, 0.2518952599D+01,&
&        0.2029194274D+01, 0.5661400518D+00, 0.2215833792D+00, 0.6013456136D+03, 0.1095358542D+03,&
&        0.2964467686D+02, 0.3896041889D+02, 0.9053563477D+01, 0.2944499834D+01, 0.2129386495D+01,&
&        0.5940934274D+00, 0.2325241410D+00, 0.6744465184D+03, 0.1228512753D+03, 0.3324834945D+02,&
&        0.4516424392D+02, 0.1049519900D+02, 0.3413364448D+01, 0.2621366518D+01, 0.7313546050D+00,&
&        0.2862472356D+00, 0.7715103681D+03, 0.1405315766D+03, 0.3803332899D+02, 0.5240203979D+02,&
&        0.1217710710D+02, 0.3960373165D+01, 0.3651583985D+01, 0.1018782663D+01, 0.3987446295D+00,&
&        0.5039822505D+00, 0.1860011465D+00, 0.8214006743D-01, 0.8540324951D+03, 0.1555630851D+03,&
&        0.4210144179D+02, 0.5956029944D+02, 0.1384053270D+02, 0.4501370797D+01, 0.4374706256D+01,&
&        0.1220531941D+01, 0.4777079300D+00, 0.4558489757D+00, 0.1682369410D+00, 0.7429520700D-01,&
&        0.9416624250D+03, 0.1715249862D+03, 0.4642135516D+02, 0.6717668771D+02, 0.1561041754D+02,&
&        0.5076992278D+01, 0.4698159231D+01, 0.1433088313D+01, 0.5529300240D+00, 0.6309328384D+00,&
&        0.2328538976D+00, 0.1028307363D+00, 0.1033571245D+04, 0.1882662926D+03, 0.5095220601D+02,&
&        0.7525120460D+02, 0.1748676162D+02, 0.5687237606D+01, 0.5395535474D+01, 0.1645810296D+01,&
&        0.6350047770D+00, 0.7122640246D+00, 0.2628702203D+00, 0.1160862609D+00, 0.1130762517D+04,&
&        0.2059698041D+03, 0.5574346711D+02, 0.8378385011D+02, 0.1946956493D+02, 0.6332106784D+01,&
&        0.6141151276D+01, 0.1873246881D+01, 0.7227568825D+00, 0.7122640246D+00, 0.2628702203D+00,&
&        0.1160862609D+00, 0.1232320450D+04, 0.2244687082D+03, 0.6074999251D+02, 0.9277462423D+02,&
&        0.2155882749D+02, 0.7011599810D+01, 0.6899488096D+01, 0.2104563782D+01, 0.8120061343D+00,&
&        0.7547780537D+00, 0.2785605708D+00, 0.1230152851D+00, 0.1337153266D+04, 0.2435641365D+03,&
&        0.6591796062D+02, 0.1020220021D+03, 0.2370771923D+02, 0.7710486098D+01, 0.7701960922D+01,&
&        0.2349343572D+01, 0.9064497870D+00, 0.6709822860D+00, 0.2476346630D+00, 0.1093580780D+00,&
&        0.1447400411D+04, 0.2636457916D+03, 0.7135284019D+02, 0.1119194891D+03, 0.2600768236D+02,&
&        0.8458505490D+01, 0.8548569754D+01, 0.2607586250D+01, 0.1006087840D+01, 0.5921156814D+00,&
&        0.2185279254D+00, 0.9650423590D-01, 0.1557298704D+04, 0.2836639029D+03, 0.7677052234D+02,&
&        0.1218344741D+03, 0.2831171164D+02, 0.9207847321D+01, 0.9480851678D+01, 0.2891961952D+01,&
&        0.1115808827D+01, 0.5921156814D+00, 0.2185279254D+00, 0.9650423590D-01, 0.1679771028D+04,&
&        0.3059723896D+03, 0.8280806943D+02, 0.1328588899D+03, 0.3087354878D+02, 0.1004103627D+02,&
&        0.1033074335D+02, 0.3151206003D+01, 0.1215833241D+01, 0.6309328384D+00, 0.2328538976D+00,&
&        0.1028307363D+00, 0.1801806730D+04, 0.3282013450D+03, 0.8882409228D+02, 0.1441212184D+03,&
&        0.3349067173D+02, 0.1089220588D+02, 0.1130775402D+02, 0.3449225397D+01, 0.1330818388D+01,&
&        0.6309328384D+00, 0.2328538976D+00, 0.1028307363D+00, 0.1929432301D+04, 0.3514485021D+03,&
&        0.9511568021D+02, 0.1558416755D+03, 0.3621425391D+02, 0.1177799934D+02, 0.1228152744D+02,&
&        0.3746257327D+01, 0.1445422541D+01, 0.8897138854D+00, 0.3283603790D+00, 0.1450074055D+00,&
&        0.2061424532D+04, 0.3754910517D+03, 0.1016225324D+03, 0.1677618680D+03, 0.3898425028D+02,&
&        0.1267888813D+02, 0.1261505520D+02, 0.3847993927D+01, 0.1484675684D+01, 0.7985243736D+00,&
&        0.2947057141D+00, 0.1301451506D+00, 0.2196384229D+04, 0.4000741292D+03, 0.1082756726D+03,&
&        0.1803890380D+03, 0.4191853304D+02, 0.1363320795D+02, 0.1419665619D+02, 0.4330432640D+01,&
&        0.1670815538D+01, 0.9858325600D+00, 0.3638342150D+00, 0.1606730254D+00, 0.2337065673D+04,&
&        0.4256994298D+03, 0.1152108790D+03, 0.1931970535D+03, 0.4489484040D+02, 0.1460119548D+02,&
&        0.1587163584D+02, 0.4841354819D+01, 0.1867945198D+01, 0.1107681464D+01, 0.4088041239D+00,&
&        0.1805322114D+00, 0.2480626814D+04, 0.4518492708D+03, 0.1222880464D+03, 0.2061578780D+03,&
&        0.4790665727D+02, 0.1558073180D+02, 0.1763999414D+02, 0.5380760465D+01, 0.2076064666D+01,&
&        0.1214644297D+01, 0.4482801363D+00, 0.1979652346D+00, 0.2629997471D+04, 0.4790573224D+03,&
&        0.1296516070D+03, 0.2198350255D+03, 0.5108493222D+02, 0.1661440546D+02, 0.1950173109D+02,&
&        0.5948649577D+01, 0.2295173940D+01, 0.1396037488D+01, 0.5152256318D+00, 0.2275290713D+00,&
&        0.2782160055D+04, 0.5067739270D+03, 0.1371528019D+03, 0.2339514118D+03, 0.5436527681D+02,&
&        0.1768127533D+02, 0.2145684671D+02, 0.6545022156D+01, 0.2525273021D+01, 0.1590049336D+01,&
&        0.5868282053D+00, 0.2591495227D+00, 0.2938601529D+04, 0.5352699370D+03, 0.1448649340D+03,&
&        0.2485070370D+03, 0.5774769100D+02, 0.1878134100D+02, 0.2350534097D+02, 0.7169878201D+01,&
&        0.2766361909D+01, 0.2247796820D+01, 0.8295783930D+00, 0.3663505650D+00, 0.4869939919D+00,&
&        0.2622161565D+00, 0.1158254875D+00, 0.3100983951D+04, 0.5648480978D+03, 0.1528699389D+03,&
&        0.2635019007D+03, 0.6123217493D+02, 0.1991460372D+02, 0.2557886692D+02, 0.7802369707D+01,&
&        0.3010396794D+01, 0.2461032403D+01, 0.9082757340D+00, 0.4011041400D+00, 0.4370804803D+00,&
&        0.2353408164D+00, 0.1039541771D+00, 0.3266026869D+04, 0.5949108710D+03, 0.1610060990D+03,&
&        0.2779377240D+03, 0.6458675000D+02, 0.2100561600D+02, 0.3067132600D+02, 0.8557222000D+01,&
&        0.3349239000D+01, 0.2667688000D+01, 0.9845440000D+00, 0.4347850000D+00, 0.2074240000D+00,&
&        0.1116850000D+00, 0.4933300000D-01, 0.3435348677D+04, 0.6257530498D+03, 0.1693531958D+03,&
&        0.2937830292D+03, 0.6826885797D+02, 0.2220315144D+02, 0.3073293103D+02, 0.9374523538D+01,&
&        0.3616982618D+01, 0.2827607815D+01, 0.1101055827D+01, 0.4846874856D+00, 0.8878301887D+00,&
&        0.3457164736D+00, 0.1521852428D+00, 0.3610742864D+04, 0.6577013201D+03, 0.1779996445D+03,&
&        0.3100675728D+03, 0.7205303569D+02, 0.2343388348D+02, 0.3301997858D+02, 0.1007214594D+02,&
&        0.3886147028D+01, 0.3144798430D+01, 0.1224568208D+01, 0.5390579399D+00, 0.4869939919D+00,&
&        0.2622161565D+00, 0.1158254875D+00, 0.3788666115D+04, 0.6901102623D+03, 0.1867707691D+03,&
&        0.3264309567D+03, 0.7585553420D+02, 0.2467057401D+02, 0.3546948129D+02, 0.1081932234D+02,&
&        0.4174430912D+01, 0.3496895188D+01, 0.1361672861D+01, 0.5994117456D+00, 0.5129625081D+00,&
&        0.2761985970D+00, 0.1220017773D+00, 0.3970868257D+04, 0.7232986098D+03, 0.1957528311D+03,&
&        0.3435846323D+03, 0.7984167952D+02, 0.2596699219D+02, 0.3808991983D+02, 0.1161863962D+02,&
&        0.4482832367D+01, 0.3829752708D+01, 0.1491285854D+01, 0.6564677040D+00, 0.4616999826D+00,&
&        0.2485968963D+00, 0.1098096207D+00, 0.4159274210D+04, 0.7576169894D+03, 0.2050407239D+03,&
&        0.3607986561D+03, 0.8384184843D+02, 0.2726797127D+02, 0.4071751678D+02, 0.1242014044D+02,&
&        0.4792076302D+01, 0.4197516371D+01, 0.1634491118D+01, 0.7195070139D+00, 0.4131354848D+00,&
&        0.2224479167D+00, 0.9825915660D-01, 0.4350077794D+04, 0.7923721005D+03, 0.2144468133D+03,&
&        0.3784334264D+03, 0.8793978981D+02, 0.2860074899D+02, 0.4352179455D+02, 0.1327553454D+02,&
&        0.5122113939D+01, 0.4540857408D+01, 0.1768186338D+01, 0.7783599789D+00, 0.4131354848D+00,&
&        0.2224479167D+00, 0.9825915660D-01, 0.4545160269D+04, 0.8279066168D+03, 0.2240638402D+03,&
&        0.3964889433D+03, 0.9213550365D+02, 0.2996532535D+02, 0.4641945097D+02, 0.1415941211D+02,&
&        0.5463141383D+01, 0.4919104589D+01, 0.1915473830D+01, 0.8431962954D+00, 0.4370804803D+00,&
&        0.2353408164D+00, 0.1039541771D+00, 0.4744521634D+04, 0.8642205383D+03, 0.2338918045D+03,&
&        0.4149652069D+03, 0.9642898995D+02, 0.3136170035D+02, 0.4941048605D+02, 0.1507177314D+02,&
&        0.5815158634D+01, 0.5290230450D+01, 0.2059988316D+01, 0.9068119281D+00, 0.4370804803D+00,&
&        0.2353408164D+00, 0.1039541771D+00, 0.4950261905D+04, 0.9016963856D+03, 0.2440342313D+03,&
&        0.4334469385D+03, 0.1007237469D+03, 0.3275848861D+02, 0.5259279235D+02, 0.1604247800D+02,&
&        0.6189686744D+01, 0.5674851796D+01, 0.2209757875D+01, 0.9727408566D+00, 0.5949150981D+00,&
&        0.3203250000D+00, 0.1414931855D+00, 0.5158224714D+04, 0.9395770707D+03, 0.2542862231D+03,&
&        0.4523313223D+03, 0.1051120716D+03, 0.3418570799D+02, 0.5597539769D+02, 0.1707428044D+02,&
&        0.6587788204D+01, 0.5048549180D+01, 0.1965878882D+01, 0.8653847237D+00, 0.5669230612D+00,&
&        0.3052530187D+00, 0.1348356264D+00, 0.5370466413D+04, 0.9782371611D+03, 0.2647491522D+03,&
&        0.4720515322D+03, 0.1096946243D+03, 0.3567609636D+02, 0.5915141188D+02, 0.1804306600D+02,&
&        0.6961575790D+01, 0.5583138529D+01, 0.2174045204D+01, 0.9570200509D+00, 0.6235816420D+00,&
&        0.3357601616D+00, 0.1483111678D+00, 0.5586987002D+04, 0.1017676657D+04, 0.2754230189D+03,&
&        0.4921924888D+03, 0.1143749494D+03, 0.3719828336D+02, 0.6252179775D+02, 0.1907114112D+02,&
&        0.7358239131D+01, 0.6120693149D+01, 0.2383366187D+01, 0.1049163663D+01, 0.6529226928D+00,&
&        0.3515585034D+00, 0.1552895732D+00, 0.5810061591D+04, 0.1058309972D+04, 0.2864199797D+03,&
&        0.5127541920D+03, 0.1191530471D+03, 0.3875226900D+02, 0.6598556227D+02, 0.2012769970D+02,&
&        0.7765892279D+01, 0.6707956921D+01, 0.2612043655D+01, 0.1149828048D+01, 0.7012713483D+00,&
&        0.3775912653D+00, 0.1667887020D+00, 0.6035183623D+04, 0.1099316231D+04, 0.2975178737D+03,&
&        0.5337366418D+03, 0.1240289171D+03, 0.4033805328D+02, 0.6954270545D+02, 0.2121274175D+02,&
&        0.8184535234D+01, 0.7295991196D+01, 0.2841021154D+01, 0.1250624506D+01, 0.7900364582D+00,&
&        0.4253857892D+00, 0.1879003836D+00, 0.6264584546D+04, 0.1141101895D+04, 0.3088267052D+03,&
&        0.5551398381D+03, 0.1290025597D+03, 0.4195563620D+02, 0.7307773504D+02, 0.2229103845D+02,&
&        0.8600575622D+01, 0.7908728280D+01, 0.3079617799D+01, 0.1355655337D+01, 0.8910101433D+00,&
&        0.4797538759D+00, 0.2119157236D+00/)
      real(8) :: cssto(612)= &
&     (/ 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00, 0.1543289673D+00, 0.5353281423D+00,&
&        0.4446345422D+00, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722919D-01,&
&        0.3995128261D+00, 0.7001154689D+00, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00, 0.1543289673D+00, 0.5353281423D+00,&
&        0.4446345422D+00,-0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00, 0.1543289673D+00,&
&        0.5353281423D+00, 0.4446345422D+00,-0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00,&
&        0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722919D-01, 0.3995128261D+00,&
&        0.7001154689D+00, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722919D-01,&
&        0.3995128261D+00, 0.7001154689D+00, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00, 0.1543289673D+00, 0.5353281423D+00,&
&        0.4446345422D+00,-0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00, 0.1543289673D+00,&
&        0.5353281423D+00, 0.4446345422D+00,-0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00,&
&       -0.2196203690D+00, 0.2255954336D+00, 0.9003984260D+00, 0.1543289673D+00, 0.5353281423D+00,&
&        0.4446345422D+00,-0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2196203690D+00,&
&        0.2255954336D+00, 0.9003984260D+00, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2196203690D+00, 0.2255954336D+00,&
&        0.9003984260D+00, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722919D-01,&
&        0.3995128261D+00, 0.7001154689D+00,-0.2196203690D+00, 0.2255954336D+00, 0.9003984260D+00,&
&        0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722919D-01, 0.3995128261D+00,&
&        0.7001154689D+00,-0.2196203690D+00, 0.2255954336D+00, 0.9003984260D+00, 0.1543289673D+00,&
&        0.5353281423D+00, 0.4446345422D+00,-0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00,&
&       -0.2196203690D+00, 0.2255954336D+00, 0.9003984260D+00, 0.1543289673D+00, 0.5353281423D+00,&
&        0.4446345422D+00,-0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2196203690D+00,&
&        0.2255954336D+00, 0.9003984260D+00, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2196203690D+00, 0.2255954336D+00,&
&        0.9003984260D+00, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01,&
&        0.3995128261D+00, 0.7001154689D+00,-0.2196203690D+00, 0.2255954336D+00, 0.9003984260D+00,&
&       -0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00,&
&        0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2196203690D+00,&
&        0.2255954336D+00, 0.9003984260D+00,-0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01,&
&        0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00,&
&        0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00, 0.9166769611D+00,-0.3088441215D+00,&
&        0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00,&
&        0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,&
&       -0.2277635023D+00, 0.2175436044D+00, 0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01,&
&        0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01,&
&        0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00, 0.9166769611D+00,&
&       -0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00,&
&        0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00,&
&        0.2175436044D+00, 0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01,&
&        0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00,&
&        0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00, 0.9166769611D+00,-0.3088441215D+00,&
&        0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00,&
&        0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,&
&       -0.2277635023D+00, 0.2175436044D+00, 0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01,&
&        0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01,&
&        0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00, 0.9166769611D+00,&
&       -0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00,&
&        0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00,&
&        0.2175436044D+00, 0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01,&
&        0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00,&
&        0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00, 0.9166769611D+00,-0.3088441215D+00,&
&        0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00,&
&        0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,&
&       -0.2277635023D+00, 0.2175436044D+00, 0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01,&
&        0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01,&
&        0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00, 0.9166769611D+00,&
&       -0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00,&
&        0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00,&
&        0.2175436044D+00, 0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01,&
&        0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,-0.9996722920D-01, 0.3995128261D+00,&
&        0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00, 0.9166769611D+00,-0.3088441215D+00,&
&        0.1960641170D-01, 0.1131034442D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3088441215D+00, 0.1960641170D-01, 0.1131034442D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095330D-01, 0.1155787450D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095330D-01, 0.1155787450D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095330D-01, 0.1155787450D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095330D-01, 0.1155787450D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095330D-01, 0.1155787450D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095330D-01, 0.1155787450D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095330D-01, 0.1155787450D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095330D-01, 0.1155787450D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095330D-01, 0.1155787450D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095330D-01, 0.1155787450D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095340D-01, 0.1115578745D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095340D-01, 0.1115578745D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095340D-01, 0.1115578745D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095340D-01, 0.1115578745D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722920D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095340D-01, 0.1115578745D+01,-0.3842642607D+00,&
&       -0.1972567438D+00, 0.1375495512D+01, 0.1543289673D+00, 0.5353281423D+00, 0.4446345422D+00,&
&       -0.9996722919D-01, 0.3995128261D+00, 0.7001154689D+00,-0.2277635023D+00, 0.2175436044D+00,&
&        0.9166769611D+00,-0.3306100626D+00, 0.5761095338D-01, 0.1115578745D+01,-0.3842642608D+00,&
&       -0.1972567438D+00, 0.1375495512D+01/)
      real(8) :: cpsto(450)= &
&     (/ 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.1559162750D+00, 0.6076837186D+00,&
&        0.3919573931D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.1559162750D+00,&
&        0.6076837186D+00, 0.3919573931D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00,&
&        0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.1559162750D+00, 0.6076837186D+00,&
&        0.3919573931D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.1559162750D+00,&
&        0.6076837186D+00, 0.3919573931D+00, 0.1058760429D-01, 0.5951670053D+00, 0.4620010120D+00,&
&        0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.1058760429D-01, 0.5951670053D+00,&
&        0.4620010120D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.1058760429D-01,&
&        0.5951670053D+00, 0.4620010120D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00,&
&        0.1058760429D-01, 0.5951670053D+00, 0.4620010120D+00, 0.1559162750D+00, 0.6076837186D+00,&
&        0.3919573931D+00, 0.1058760429D-01, 0.5951670053D+00, 0.4620010120D+00, 0.1559162750D+00,&
&        0.6076837186D+00, 0.3919573931D+00, 0.1058760429D-01, 0.5951670053D+00, 0.4620010120D+00,&
&        0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.1058760429D-01, 0.5951670053D+00,&
&        0.4620010120D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.1058760429D-01,&
&        0.5951670053D+00, 0.4620010120D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00,&
&        0.1058760430D-01, 0.5951670053D+00, 0.4620010120D+00,-0.1215468600D+00, 0.5715227604D+00,&
&        0.5498949471D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.1058760430D-01,&
&        0.5951670053D+00, 0.4620010120D+00,-0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00,&
&        0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00,&
&        0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00, 0.1559162750D+00,&
&        0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,&
&       -0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00, 0.1559162750D+00, 0.6076837186D+00,&
&        0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1215468600D+00,&
&        0.5715227604D+00, 0.5498949471D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00,&
&        0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00,&
&        0.5498949471D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02,&
&        0.5777664691D+00, 0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00,&
&        0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00,&
&        0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00, 0.1559162750D+00,&
&        0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,&
&       -0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00, 0.1559162750D+00, 0.6076837186D+00,&
&        0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1215468600D+00,&
&        0.5715227604D+00, 0.5498949471D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00,&
&        0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00,&
&        0.5498949471D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02,&
&        0.5777664691D+00, 0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00,&
&        0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00,&
&        0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00, 0.1559162750D+00,&
&        0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,&
&       -0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00, 0.1559162750D+00, 0.6076837186D+00,&
&        0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1215468600D+00,&
&        0.5715227604D+00, 0.5498949471D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00,&
&        0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00,&
&        0.5498949471D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02,&
&        0.5777664691D+00, 0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00,&
&        0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00,&
&        0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00, 0.1559162750D+00,&
&        0.6076837186D+00, 0.3919573931D+00, 0.4951511100D-02, 0.5777664691D+00, 0.4846460366D+00,&
&       -0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00,-0.3481691526D+00, 0.6290323690D+00,&
&        0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511100D-02,&
&        0.5777664691D+00, 0.4846460366D+00,-0.1215468600D+00, 0.5715227604D+00, 0.5498949471D+00,&
&       -0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00,&
&        0.3919573931D+00, 0.4951511100D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1283927634D+00,&
&        0.5852047641D+00, 0.5439442040D+00,-0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00,&
&        0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511100D-02, 0.5777664691D+00,&
&        0.4846460366D+00,-0.1283927634D+00, 0.5852047641D+00, 0.5439442040D+00,-0.3481691526D+00,&
&        0.6290323690D+00, 0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00,&
&        0.4951511100D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1283927634D+00, 0.5852047641D+00,&
&        0.5439442040D+00,-0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00, 0.1559162750D+00,&
&        0.6076837186D+00, 0.3919573931D+00, 0.4951511100D-02, 0.5777664691D+00, 0.4846460366D+00,&
&       -0.1283927634D+00, 0.5852047641D+00, 0.5439442040D+00,-0.3481691526D+00, 0.6290323690D+00,&
&        0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511100D-02,&
&        0.5777664691D+00, 0.4846460366D+00,-0.1283927634D+00, 0.5852047641D+00, 0.5439442040D+00,&
&       -0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00,&
&        0.3919573931D+00, 0.4951511100D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1283927634D+00,&
&        0.5852047641D+00, 0.5439442040D+00,-0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00,&
&        0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511100D-02, 0.5777664691D+00,&
&        0.4846460366D+00,-0.1283927634D+00, 0.5852047641D+00, 0.5439442040D+00,-0.3481691526D+00,&
&        0.6290323690D+00, 0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00,&
&        0.4951511100D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1283927634D+00, 0.5852047641D+00,&
&        0.5439442040D+00,-0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00, 0.1559162750D+00,&
&        0.6076837186D+00, 0.3919573931D+00, 0.4951511100D-02, 0.5777664691D+00, 0.4846460366D+00,&
&       -0.1283927634D+00, 0.5852047641D+00, 0.5439442040D+00,-0.3481691526D+00, 0.6290323690D+00,&
&        0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511100D-02,&
&        0.5777664691D+00, 0.4846460366D+00,-0.1283927634D+00, 0.5852047641D+00, 0.5439442040D+00,&
&       -0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00,&
&        0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1283927634D+00,&
&        0.5852047641D+00, 0.5439442040D+00,-0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00,&
&        0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00,&
&        0.4846460366D+00,-0.1283927634D+00, 0.5852047641D+00, 0.5439442040D+00,-0.3481691526D+00,&
&        0.6290323690D+00, 0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00,&
&        0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1283927634D+00, 0.5852047641D+00,&
&        0.5439442040D+00,-0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00, 0.1559162750D+00,&
&        0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02, 0.5777664691D+00, 0.4846460366D+00,&
&       -0.1283927634D+00, 0.5852047641D+00, 0.5439442040D+00,-0.3481691526D+00, 0.6290323690D+00,&
&        0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00, 0.3919573931D+00, 0.4951511200D-02,&
&        0.5777664691D+00, 0.4846460366D+00,-0.1283927634D+00, 0.5852047641D+00, 0.5439442040D+00,&
&       -0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00, 0.1559162750D+00, 0.6076837186D+00,&
&        0.3919573931D+00, 0.4951511155D-02, 0.5777664691D+00, 0.4846460366D+00,-0.1283927634D+00,&
&        0.5852047641D+00, 0.5439442040D+00,-0.3481691526D+00, 0.6290323690D+00, 0.6662832743D+00/)
      real(8) :: exdsto(150)= &
&     (/ 0.5517000679D+00, 0.1682861055D+00, 0.6493001120D-01, 0.1645981194D+01, 0.5020767280D+00,&
&        0.1937168100D+00, 0.2964817927D+01, 0.9043639676D+00, 0.3489317337D+00, 0.4241479241D+01,&
&        0.1293786360D+01, 0.4991829993D+00, 0.5426950461D+01, 0.1655392868D+01, 0.6387020320D+00,&
&        0.6411803475D+01, 0.1955804428D+01, 0.7546101510D+00, 0.7664527389D+01, 0.2337925151D+01,&
&        0.9020442050D+00, 0.8627722755D+01, 0.2631730438D+01, 0.1015403419D+01, 0.9647911930D+01,&
&        0.2942920654D+01, 0.1135470278D+01, 0.1094737077D+02, 0.3339297018D+01, 0.1288404602D+01,&
&        0.1261505520D+02, 0.3847993927D+01, 0.1484675684D+01, 0.1419665619D+02, 0.4330432640D+01,&
&        0.1670815538D+01, 0.1587163584D+02, 0.4841354819D+01, 0.1867945198D+01, 0.1763999414D+02,&
&        0.5380760465D+01, 0.2076064666D+01, 0.1950173109D+02, 0.5948649577D+01, 0.2295173940D+01,&
&        0.2145684671D+02, 0.6545022156D+01, 0.2525273021D+01, 0.2350534097D+02, 0.7169878201D+01,&
&        0.2766361909D+01, 0.2557886692D+02, 0.7802369707D+01, 0.3010396794D+01, 0.5660043000D+01,&
&        0.1774715000D+01, 0.6912950000D+00, 0.2128212000D+01, 0.9625940000D+00, 0.4728610000D+00,&
&        0.3073293103D+02, 0.9374523538D+01, 0.3616982618D+01, 0.4869939919D+00, 0.2622161565D+00,&
&        0.1158254875D+00, 0.3301997858D+02, 0.1007214594D+02, 0.3886147028D+01, 0.1344878866D+01,&
&        0.5236888594D+00, 0.2305291251D+00, 0.3546948129D+02, 0.1081932234D+02, 0.4174430912D+01,&
&        0.1702112315D+01, 0.6627937127D+00, 0.2917634240D+00, 0.3808991983D+02, 0.1161863962D+02,&
&        0.4482832367D+01, 0.2101373228D+01, 0.8182638428D+00, 0.3602017580D+00, 0.4071751678D+02,&
&        0.1242014044D+02, 0.4792076302D+01, 0.2390895761D+01, 0.9310024167D+00, 0.4098295558D+00,&
&        0.4352179455D+02, 0.1327553454D+02, 0.5122113939D+01, 0.2779066094D+01, 0.1082153932D+01,&
&        0.4763668250D+00, 0.4641945097D+02, 0.1415941211D+02, 0.5463141383D+01, 0.3025977448D+01,&
&        0.1178299934D+01, 0.5186905316D+00, 0.4941048605D+02, 0.1507177314D+02, 0.5815158634D+01,&
&        0.3283395668D+01, 0.1278537254D+01, 0.5628152469D+00, 0.5259279235D+02, 0.1604247800D+02,&
&        0.6189686744D+01, 0.3642963976D+01, 0.1418551290D+01, 0.6244497700D+00, 0.5597539769D+02,&
&        0.1707428044D+02, 0.6587788204D+01, 0.5048549180D+01, 0.1965878882D+01, 0.8653847237D+00,&
&        0.5915141188D+02, 0.1804306600D+02, 0.6961575790D+01, 0.5583138529D+01, 0.2174045204D+01,&
&        0.9570200509D+00, 0.6252179775D+02, 0.1907114112D+02, 0.7358239131D+01, 0.6120693149D+01,&
&        0.2383366187D+01, 0.1049163663D+01, 0.6598556227D+02, 0.2012769970D+02, 0.7765892279D+01,&
&        0.6707956921D+01, 0.2612043655D+01, 0.1149828048D+01, 0.6954270545D+02, 0.2121274175D+02,&
&        0.8184535234D+01, 0.7295991196D+01, 0.2841021154D+01, 0.1250624506D+01, 0.7307773504D+02,&
&        0.2229103845D+02, 0.8600575622D+01, 0.7908728280D+01, 0.3079617799D+01, 0.1355655337D+01/)
      real(8) :: cdsto(150)= &
&     (/ 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00,&
&        0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00,&
&        0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00,&
&        0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00,&
&        0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00,&
&        0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00,&
&        0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00,&
&        0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00,&
&        0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00,&
&        0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00,&
&        0.2865732590D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.2197679508D+00,&
&        0.6555473627D+00, 0.2865732590D+00, 0.1250662138D+00, 0.6686785577D+00, 0.3052468245D+00,&
&        0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.1250662138D+00, 0.6686785577D+00,&
&        0.3052468245D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.1250662138D+00,&
&        0.6686785577D+00, 0.3052468245D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00,&
&        0.1250662138D+00, 0.6686785577D+00, 0.3052468245D+00, 0.2197679508D+00, 0.6555473627D+00,&
&        0.2865732590D+00, 0.1250662138D+00, 0.6686785577D+00, 0.3052468245D+00, 0.2197679508D+00,&
&        0.6555473627D+00, 0.2865732590D+00, 0.1250662138D+00, 0.6686785577D+00, 0.3052468245D+00,&
&        0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.1250662138D+00, 0.6686785577D+00,&
&        0.3052468245D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.1250662138D+00,&
&        0.6686785577D+00, 0.3052468245D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00,&
&        0.1250662138D+00, 0.6686785577D+00, 0.3052468245D+00, 0.2197679508D+00, 0.6555473627D+00,&
&        0.2865732590D+00, 0.1250662138D+00, 0.6686785577D+00, 0.3052468245D+00, 0.2197679508D+00,&
&        0.6555473627D+00, 0.2865732590D+00, 0.1250662138D+00, 0.6686785577D+00, 0.3052468245D+00,&
&        0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.1250662138D+00, 0.6686785577D+00,&
&        0.3052468245D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00, 0.1250662138D+00,&
&        0.6686785577D+00, 0.3052468245D+00, 0.2197679508D+00, 0.6555473627D+00, 0.2865732590D+00,&
&        0.1250662138D+00, 0.6686785577D+00, 0.3052468245D+00, 0.2197679508D+00, 0.6555473627D+00,&
&        0.2865732590D+00, 0.1250662138D+00, 0.6686785577D+00, 0.3052468245D+00, 0.2197679508D+00,&
&        0.6555473627D+00, 0.2865732590D+00, 0.1250662138D+00, 0.6686785577D+00, 0.3052468245D+00/)
!
! Set valence basis functions
!
      if(itype == 1) then
        select case(numatomic(iatom))
! H-He
          case(1:2)
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= exsto(is(numatomic(iatom))+j)
              coeff_g(locprim_g(ishell)+j)= cssto(is(numatomic(iatom))+j)
            enddo
            mprim_g(ishell)= 3
            mbf_g(ishell)= 1
            mtype_g(ishell)= 0
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            locbf_g(ishell+1) = locbf_g(ishell)+1
! Li-Ne
          case(3:10)
!   S function
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= exsto(is(numatomic(iatom))+j+3)
              coeff_g(locprim_g(ishell)+j)= cssto(is(numatomic(iatom))+j+3)
            enddo
            mprim_g(ishell)= 3
            mbf_g(ishell)= 1
            mtype_g(ishell)= 0
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            locbf_g(ishell+1) = locbf_g(ishell)+1
!   P function
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= exsto(is(numatomic(iatom))+j+3)
              coeff_g(locprim_g(ishell)+j)= cpsto(ip(numatomic(iatom))+j)
            enddo
            mprim_g(ishell)= 3
            mbf_g(ishell)= 3
            mtype_g(ishell)= 1
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            locbf_g(ishell+1) = locbf_g(ishell)+3
! Na-Ar
          case(11:18)
!   S function
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= exsto(is(numatomic(iatom))+j+6)
              coeff_g(locprim_g(ishell)+j)= cssto(is(numatomic(iatom))+j+6)
            enddo
            mprim_g(ishell)= 3
            mbf_g(ishell)= 1
            mtype_g(ishell)= 0
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            locbf_g(ishell+1) = locbf_g(ishell)+1
!   P function
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= exsto(is(numatomic(iatom))+j+6)
              coeff_g(locprim_g(ishell)+j)= cpsto(ip(numatomic(iatom))+j+3)
            enddo
            mprim_g(ishell)= 3
            mbf_g(ishell)= 3
            mtype_g(ishell)= 1
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            locbf_g(ishell+1) = locbf_g(ishell)+3
! K -Kr
          case(19:36)
!   S function
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= exsto(is(numatomic(iatom))+j+9)
              coeff_g(locprim_g(ishell)+j)= cssto(is(numatomic(iatom))+j+9)
            enddo
            mprim_g(ishell)= 3
            mbf_g(ishell)= 1
            mtype_g(ishell)= 0
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            locbf_g(ishell+1) = locbf_g(ishell)+1
!   P function
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= exsto(is(numatomic(iatom))+j+9)
              coeff_g(locprim_g(ishell)+j)= cpsto(ip(numatomic(iatom))+j+6)
            enddo
            mprim_g(ishell)= 3
            mbf_g(ishell)= 3
            mtype_g(ishell)= 1
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            locbf_g(ishell+1) = locbf_g(ishell)+3
!   D function
            if((numatomic(iatom) >= 21).and.(numatomic(iatom) <= 30)) then
              ishell= ishell+1
              do j= 1,3
                ex_g(locprim_g(ishell)+j)= exdsto(id(numatomic(iatom))+j)
                coeff_g(locprim_g(ishell)+j)= cdsto(id(numatomic(iatom))+j)
              enddo
              mprim_g(ishell)= 3
              mtype_g(ishell)= 2
              locatom_g(ishell)= iatom
              locprim_g(ishell+1)= locprim_g(ishell)+3
              if(spher_g) then
                mbf_g(ishell)= 5
                locbf_g(ishell+1) = locbf_g(ishell)+5
              else
                mbf_g(ishell)= 6
                locbf_g(ishell+1) = locbf_g(ishell)+6
              endif
            endif
! Rb-Xe
          case(37:54)
!   S function
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= exsto(is(numatomic(iatom))+j+12)
              coeff_g(locprim_g(ishell)+j)= cssto(is(numatomic(iatom))+j+12)
            enddo
            mprim_g(ishell)= 3
            mbf_g(ishell)= 1
            mtype_g(ishell)= 0
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            locbf_g(ishell+1) = locbf_g(ishell)+1
!   P function
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= exsto(is(numatomic(iatom))+j+12)
              coeff_g(locprim_g(ishell)+j)= cpsto(ip(numatomic(iatom))+j+9)
            enddo
            mprim_g(ishell)= 3
            mbf_g(ishell)= 3
            mtype_g(ishell)= 1
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            locbf_g(ishell+1) = locbf_g(ishell)+3
!   D function
            if((numatomic(iatom) >= 39).and.(numatomic(iatom) <= 48)) then
              ishell= ishell+1
              do j= 1,3
                ex_g(locprim_g(ishell)+j)= exdsto(id(numatomic(iatom))+j+3)
                coeff_g(locprim_g(ishell)+j)= cdsto(id(numatomic(iatom))+j+3)
              enddo
              mprim_g(ishell)= 3
              mtype_g(ishell)= 2
              locatom_g(ishell)= iatom
              locprim_g(ishell+1)= locprim_g(ishell)+3
              if(spher_g) then
                mbf_g(ishell)= 5
                locbf_g(ishell+1) = locbf_g(ishell)+5
              else
                mbf_g(ishell)= 6
                locbf_g(ishell+1) = locbf_g(ishell)+6
              endif
            endif
          case default
            write(*,'(" Error! This program supports H - Xe basis set in subroutine bssto3g_g.")')
            call iabort
        endselect
!
! Set core basis functions
!
      elseif(itype == 2) then
! Set 1S functions
        if(numatomic(iatom) >= 3) then
          if(.not.flagecp.or.(izcore(iatom) < 2))then
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= exsto(is(numatomic(iatom))+j)
              coeff_g(locprim_g(ishell)+j)= cssto(is(numatomic(iatom))+j)
            enddo
            mprim_g(ishell)= 3
            mbf_g(ishell)= 1
            mtype_g(ishell)= 0
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            locbf_g(ishell+1) = locbf_g(ishell)+1
          endif
        endif
! Set 2SP functions
        if(numatomic(iatom) >= 11) then
          if(.not.flagecp.or.(izcore(iatom) < 10))then
!   S function
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= exsto(is(numatomic(iatom))+j+3)
              coeff_g(locprim_g(ishell)+j)= cssto(is(numatomic(iatom))+j+3)
            enddo
            mprim_g(ishell)= 3
            mbf_g(ishell)= 1
            mtype_g(ishell)= 0
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            locbf_g(ishell+1) = locbf_g(ishell)+1
!   P function
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= exsto(is(numatomic(iatom))+j+3)
              coeff_g(locprim_g(ishell)+j)= cpsto(ip(numatomic(iatom))+j)
            enddo
            mprim_g(ishell)= 3
            mbf_g(ishell)= 3
            mtype_g(ishell)= 1
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            locbf_g(ishell+1) = locbf_g(ishell)+3
          endif
        endif
! Set 3SPD functions
        if(numatomic(iatom) >= 19) then
          if(.not.flagecp.or.(izcore(iatom) < 18))then
!   S function
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= exsto(is(numatomic(iatom))+j+6)
              coeff_g(locprim_g(ishell)+j)= cssto(is(numatomic(iatom))+j+6)
            enddo
            mprim_g(ishell)= 3
            mbf_g(ishell)= 1
            mtype_g(ishell)= 0
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            locbf_g(ishell+1) = locbf_g(ishell)+1
!   P function
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= exsto(is(numatomic(iatom))+j+6)
              coeff_g(locprim_g(ishell)+j)= cpsto(ip(numatomic(iatom))+j+3)
            enddo
            mprim_g(ishell)= 3
            mbf_g(ishell)= 3
            mtype_g(ishell)= 1
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            locbf_g(ishell+1) = locbf_g(ishell)+3
          endif
        endif
!   D function
        if(numatomic(iatom) >= 31) then
          if(.not.flagecp.or.(izcore(iatom) < 28))then
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= exdsto(id(numatomic(iatom))+j)
              coeff_g(locprim_g(ishell)+j)= cdsto(id(numatomic(iatom))+j)
            enddo
            mprim_g(ishell)= 3
            mtype_g(ishell)= 2
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            if(spher_g) then
              mbf_g(ishell)= 5
              locbf_g(ishell+1) = locbf_g(ishell)+5
            else
              mbf_g(ishell)= 6
              locbf_g(ishell+1) = locbf_g(ishell)+6
            endif
          endif
        endif
! Set 4SPD functions
        if(numatomic(iatom) >= 37) then
          if(.not.flagecp.or.(izcore(iatom) < 36))then
!   S function
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= exsto(is(numatomic(iatom))+j+9)
              coeff_g(locprim_g(ishell)+j)= cssto(is(numatomic(iatom))+j+9)
            enddo
            mprim_g(ishell)= 3
            mbf_g(ishell)= 1
            mtype_g(ishell)= 0
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            locbf_g(ishell+1) = locbf_g(ishell)+1
!   P function
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= exsto(is(numatomic(iatom))+j+9)
              coeff_g(locprim_g(ishell)+j)= cpsto(ip(numatomic(iatom))+j+6)
            enddo
            mprim_g(ishell)= 3
            mbf_g(ishell)= 3
            mtype_g(ishell)= 1
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            locbf_g(ishell+1) = locbf_g(ishell)+3
          endif
        endif
!   D function
        if(numatomic(iatom) >= 49) then
          if(.not.flagecp.or.(izcore(iatom) < 46))then
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= exdsto(id(numatomic(iatom))+j+3)
              coeff_g(locprim_g(ishell)+j)= cdsto(id(numatomic(iatom))+j+3)
            enddo
            mprim_g(ishell)= 3
            mtype_g(ishell)= 2
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            if(spher_g) then
              mbf_g(ishell)= 5
              locbf_g(ishell+1) = locbf_g(ishell)+5
            else
              mbf_g(ishell)= 6
              locbf_g(ishell+1) = locbf_g(ishell)+6
            endif
          endif
        endif
!
! Set core basis functions for core orbital calculation
!
      elseif(itype == 3) then
! Set 1S functions
        if(numatomic(iatom) >= 3) then
          if(izcore(iatom) >= 2)then
            ishell= ishell+1
            do j= 1,3
              ex_gcore(locprim_gcore(ishell)+j)= exsto(is(numatomic(iatom))+j)
              coeff_gcore(locprim_gcore(ishell)+j)= cssto(is(numatomic(iatom))+j)
            enddo
            mprim_gcore(ishell)= 3
            mbf_gcore(ishell)= 1
            mtype_gcore(ishell)= 0
            locatom_gcore(ishell)= iatom
            locprim_gcore(ishell+1)= locprim_gcore(ishell)+3
            locbf_gcore(ishell+1) = locbf_gcore(ishell)+1
          endif
        endif
! Set 2SP functions
        if(numatomic(iatom) >= 11) then
          if(izcore(iatom) >= 10)then
!   S function
            ishell= ishell+1
            do j= 1,3
              ex_gcore(locprim_gcore(ishell)+j)= exsto(is(numatomic(iatom))+j+3)
              coeff_gcore(locprim_gcore(ishell)+j)= cssto(is(numatomic(iatom))+j+3)
            enddo
            mprim_gcore(ishell)= 3
            mbf_gcore(ishell)= 1
            mtype_gcore(ishell)= 0
            locatom_gcore(ishell)= iatom
            locprim_gcore(ishell+1)= locprim_gcore(ishell)+3
            locbf_gcore(ishell+1) = locbf_gcore(ishell)+1
!   P function
            ishell= ishell+1
            do j= 1,3
              ex_gcore(locprim_gcore(ishell)+j)= exsto(is(numatomic(iatom))+j+3)
              coeff_gcore(locprim_gcore(ishell)+j)= cpsto(ip(numatomic(iatom))+j)
            enddo
            mprim_gcore(ishell)= 3
            mbf_gcore(ishell)= 3
            mtype_gcore(ishell)= 1
            locatom_gcore(ishell)= iatom
            locprim_gcore(ishell+1)= locprim_gcore(ishell)+3
            locbf_gcore(ishell+1) = locbf_gcore(ishell)+3
          endif
        endif
! Set 3SPD functions
        if(numatomic(iatom) >= 19) then
          if(izcore(iatom) >= 18)then
!   S function
            ishell= ishell+1
            do j= 1,3
              ex_gcore(locprim_gcore(ishell)+j)= exsto(is(numatomic(iatom))+j+6)
              coeff_gcore(locprim_gcore(ishell)+j)= cssto(is(numatomic(iatom))+j+6)
            enddo
            mprim_gcore(ishell)= 3
            mbf_gcore(ishell)= 1
            mtype_gcore(ishell)= 0
            locatom_gcore(ishell)= iatom
            locprim_gcore(ishell+1)= locprim_gcore(ishell)+3
            locbf_gcore(ishell+1) = locbf_gcore(ishell)+1
!   P function
            ishell= ishell+1
            do j= 1,3
              ex_gcore(locprim_gcore(ishell)+j)= exsto(is(numatomic(iatom))+j+6)
              coeff_gcore(locprim_gcore(ishell)+j)= cpsto(ip(numatomic(iatom))+j+3)
            enddo
            mprim_gcore(ishell)= 3
            mbf_gcore(ishell)= 3
            mtype_gcore(ishell)= 1
            locatom_gcore(ishell)= iatom
            locprim_gcore(ishell+1)= locprim_gcore(ishell)+3
            locbf_gcore(ishell+1) = locbf_gcore(ishell)+3
          endif
        endif
!   D function
        if(numatomic(iatom) >= 31) then
          if(izcore(iatom) >= 28)then
            ishell= ishell+1
            do j= 1,3
              ex_gcore(locprim_gcore(ishell)+j)= exdsto(id(numatomic(iatom))+j)
              coeff_gcore(locprim_gcore(ishell)+j)= cdsto(id(numatomic(iatom))+j)
            enddo
            mprim_gcore(ishell)= 3
            mtype_gcore(ishell)= 2
            locatom_gcore(ishell)= iatom
            locprim_gcore(ishell+1)= locprim_gcore(ishell)+3
            mbf_gcore(ishell)= 5
            locbf_gcore(ishell+1) = locbf_gcore(ishell)+5
          endif
        endif
! Set 4SPD functions
        if(numatomic(iatom) >= 37) then
          if(izcore(iatom) >= 36)then
!   S function
            ishell= ishell+1
            do j= 1,3
              ex_gcore(locprim_gcore(ishell)+j)= exsto(is(numatomic(iatom))+j+9)
              coeff_gcore(locprim_gcore(ishell)+j)= cssto(is(numatomic(iatom))+j+9)
            enddo
            mprim_gcore(ishell)= 3
            mbf_gcore(ishell)= 1
            mtype_gcore(ishell)= 0
            locatom_gcore(ishell)= iatom
            locprim_gcore(ishell+1)= locprim_gcore(ishell)+3
            locbf_gcore(ishell+1) = locbf_gcore(ishell)+1
!   P function
            ishell= ishell+1
            do j= 1,3
              ex_gcore(locprim_gcore(ishell)+j)= exsto(is(numatomic(iatom))+j+9)
              coeff_gcore(locprim_gcore(ishell)+j)= cpsto(ip(numatomic(iatom))+j+6)
            enddo
            mprim_gcore(ishell)= 3
            mbf_gcore(ishell)= 3
            mtype_gcore(ishell)= 1
            locatom_gcore(ishell)= iatom
            locprim_gcore(ishell+1)= locprim_gcore(ishell)+3
            locbf_gcore(ishell+1) = locbf_gcore(ishell)+3
          endif
        endif
!   D function
        if(numatomic(iatom) >= 49) then
          if(izcore(iatom) >= 46)then
            ishell= ishell+1
            do j= 1,3
              ex_gcore(locprim_gcore(ishell)+j)= exdsto(id(numatomic(iatom))+j+3)
              coeff_gcore(locprim_gcore(ishell)+j)= cdsto(id(numatomic(iatom))+j+3)
            enddo
            mprim_gcore(ishell)= 3
            mtype_gcore(ishell)= 2
            locatom_gcore(ishell)= iatom
            locprim_gcore(ishell+1)= locprim_gcore(ishell)+3
            mbf_gcore(ishell)= 5
            locbf_gcore(ishell+1) = locbf_gcore(ishell)+5
          endif
        endif
      endif
!
      if(ishell > mxshell) then
        write(*,'(" Error! The number of basis shells exceeds mxshell",i6,".")')mxshell
        call iabort
      endif
      if(itype <= 2) then
        if(locprim_g(ishell+1) > mxprim ) then
          write(*,'(" Error! The number of primitive basis functions exceeds mxprim",&
&               i6,".")')mxprim
          call iabort
        endif
        if(locbf_g(ishell+1) > mxao ) then
          write(*,'(" Error! The number of basis functions exceeds mxao",i6,".")')mxao
          call iabort
        endif
      elseif(itype == 3) then
        if(locprim_gcore(ishell+1) > mxprim ) then
          write(*,'(" Error! The number of primitive basis functions exceeds mxprim",&
&               i6,".")')mxprim
          call iabort
        endif
        if(locbf_gcore(ishell+1) > mxao ) then
          write(*,'(" Error! The number of basis functions exceeds mxao",i6,".")')mxao
          call iabort
        endif
      endif
      return
end


!-----------------------------------
  subroutine bs321g(iatom,ishell)
!-----------------------------------
!
! Set basis functions of 3-21G
!
      use modmolecule, only : numatomic
      use modbasis, only : ex, coeff, locprim, locbf, locatom, mprim, mbf, mtype, spher
      use modparam, only : mxao, mxshell, mxprim
      implicit none
      integer,intent(in) :: iatom
      integer,intent(inout) :: ishell
      integer :: j
      integer :: is1(2)= (/0,6/)
      integer :: is2(3:10)= (/0,6,12,18,24,30,36,42/)
      integer :: ip2(3:10)= (/0,3,6,9,12,15,18,21/)
      integer :: is3(11:18)=(/0,9,18,27,36,45,54,63/)
      integer :: ip3(11:18)=(/0,6,12,18,24,30,36,42/)
      integer :: is4(19:36)=(/0,12,24,36,48,60,72,84,96,108,120,132,144,156,168,180,192,204/)
      integer :: ip4(19:36)=(/0,9,18,27,36,45,54,63,72,81,90,99,108,117,126,135,144,153/)
      integer :: id4(21:36)=(/0,3,6,9,12,15,18,21,24,27,30,33,36,39,42,45/)
      integer :: is5(37:54)=(/0,15,30,45,60,75,90,105,120,135,150,165,180,195,210,225,240,255/)
      integer :: ip5(37:54)=(/0,12,24,36,48,60,72,84,96,108,120,132,144,156,168,180,192,204/)
      integer :: id5(37:54)=(/0,3,6,12,18,24,30,36,42,48,54,60,66,72,78,84,90,96/)
      real(8) :: exs1(6)= &
&     (/ 5.4471780D+00, 8.2454700D-01, 1.8319200D-01, 1.3626700D+01, 1.9993500D+00, &
&        3.8299300D-01/)
      real(8) :: coeffs1(6)= &
&     (/ 1.5628500D-01, 9.0469100D-01, 1.0000000D+00, 1.7523000D-01, 8.9348300D-01, &
&        1.0000000D+00/)
      real(8) :: exs2(48)= &
&     (/ 3.6838200D+01, 5.4817200D+00, 1.1132700D+00, 5.4020500D-01, 1.0225500D-01, &
&        2.8565000D-02, 7.1887600D+01, 1.0728900D+01, 2.2220500D+00, 1.2954800D+00, &
&        2.6888100D-01, 7.7350000D-02, 1.1643400D+02, 1.7431400D+01, 3.6801600D+00, &
&        2.2818700D+00, 4.6524800D-01, 1.2432800D-01, 1.7225600D+02, 2.5910900D+01, &
&        5.5333500D+00, 3.6649800D+00, 7.7054500D-01, 1.9585700D-01, 2.4276600D+02, &
&        3.6485100D+01, 7.8144900D+00, 5.4252200D+00, 1.1491500D+00, 2.8320500D-01, &
&        3.2203700D+02, 4.8430800D+01, 1.0420600D+01, 7.4029400D+00, 1.5762000D+00, &
&        3.7368400D-01, 4.1380100D+02, 6.2244600D+01, 1.3434000D+01, 9.7775900D+00, &
&        2.0861700D+00, 4.8238300D-01, 5.1572400D+02, 7.7653800D+01, 1.6813600D+01, &
&        1.2483000D+01, 2.6645100D+00, 6.0625000D-01/)
      real(8) :: coeffs2(48)= &
&     (/ 6.9668600D-02, 3.8134600D-01, 6.8170200D-01,-2.6312700D-01, 1.1433900D+00, &
&        1.0000000D+00, 6.4426300D-02, 3.6609600D-01, 6.9593400D-01,-4.2106400D-01, &
&        1.2240700D+00, 1.0000000D+00, 6.2960500D-02, 3.6330400D-01, 6.9725500D-01, &
&       -3.6866200D-01, 1.1994400D+00, 1.0000000D+00, 6.1766900D-02, 3.5879400D-01, &
&        7.0071300D-01,-3.9589700D-01, 1.2158400D+00, 1.0000000D+00, 5.9865700D-02, &
&        3.5295500D-01, 7.0651300D-01,-4.1330100D-01, 1.2244200D+00, 1.0000000D+00, &
&        5.9239400D-02, 3.5150000D-01, 7.0765800D-01,-4.0445300D-01, 1.2215600D+00, &
&        1.0000000D+00, 5.8548300D-02, 3.4930800D-01, 7.0963200D-01,-4.0732700D-01, &
&        1.2231400D+00, 1.0000000D+00, 5.8143000D-02, 3.4795100D-01, 7.1071400D-01, &
&       -4.0992200D-01, 1.2243100D+00, 1.0000000D+00/)
      real(8) :: coeffp2(24)= &
&     (/ 1.6154600D-01, 9.1566300D-01, 1.0000000D+00, 2.0513200D-01, 8.8252800D-01, &
&        1.0000000D+00, 2.3115200D-01, 8.6676400D-01, 1.0000000D+00, 2.3646000D-01, &
&        8.6061900D-01, 1.0000000D+00, 2.3797200D-01, 8.5895300D-01, 1.0000000D+00, &
&        2.4458600D-01, 8.5395500D-01, 1.0000000D+00, 2.4668000D-01, 8.5232100D-01, &
&        1.0000000D+00, 2.4746000D-01, 8.5174300D-01, 1.0000000D+00/)
      real(8) :: exs3(72)= &
&     (/ 5.4761300D+02, 8.2067800D+01, 1.7691700D+01, 1.7540700D+01, 3.7939800D+00, &
&        9.0644100D-01, 5.0182400D-01, 6.0945800D-02, 2.4434900D-02, 6.5284100D+02, &
&        9.8380500D+01, 2.1299600D+01, 2.3372700D+01, 5.1995300D+00, 1.3150800D+00, &
&        6.1134900D-01, 1.4184100D-01, 4.6401100D-02, 7.7573700D+02, 1.1695200D+02, &
&        2.5332600D+01, 2.9479600D+01, 6.6331400D+00, 1.7267500D+00, 9.4616000D-01, &
&        2.0250600D-01, 6.3908800D-02, 9.1065500D+02, 1.3733600D+02, 2.9760100D+01, &
&        3.6671600D+01, 8.3172900D+00, 2.2164500D+00, 1.0791300D+00, 3.0242200D-01, &
&        9.3339200D-02, 1.0549000D+03, 1.5919500D+02, 3.4530400D+01, 4.4286600D+01, &
&        1.0101900D+01, 2.7399700D+00, 1.2186500D+00, 3.9554600D-01, 1.2281100D-01, &
&        1.2106200D+03, 1.8274700D+02, 3.9667300D+01, 5.2223600D+01, 1.1962900D+01, &
&        3.2891100D+00, 1.2238400D+00, 4.5730300D-01, 1.4226900D-01, 1.3764000D+03, &
&        2.0785700D+02, 4.5155400D+01, 6.0801400D+01, 1.3976500D+01, 3.8871000D+00, &
&        1.3529900D+00, 5.2695500D-01, 1.6671400D-01, 1.5537100D+03, 2.3467800D+02, &
&        5.1012100D+01, 7.0045300D+01, 1.6147300D+01, 4.5349200D+00, 1.5420900D+00, &
&        6.0726700D-01, 1.9537300D-01/)
      real(8) :: coeffs3(72)= &
&     (/ 6.7491100D-02, 3.9350500D-01, 6.6560500D-01,-1.1193700D-01, 2.5465400D-01, &
&        8.4441700D-01,-2.1966000D-01, 1.0891200D+00, 1.0000000D+00, 6.7598200D-02, &
&        3.9177800D-01, 6.6666100D-01,-1.1024600D-01, 1.8411900D-01, 8.9639900D-01, &
&       -3.6110100D-01, 1.2150500D+00, 1.0000000D+00, 6.6834700D-02, 3.8906100D-01, &
&        6.6946800D-01,-1.0790200D-01, 1.4624500D-01, 9.2373000D-01,-3.2032700D-01, &
&        1.1841200D+00, 1.0000000D+00, 6.6082300D-02, 3.8622900D-01, 6.7238000D-01, &
&       -1.0451100D-01, 1.0741000D-01, 9.5144600D-01,-3.7610800D-01, 1.2516500D+00, &
&        1.0000000D+00, 6.5541000D-02, 3.8403600D-01, 6.7454100D-01,-1.0213000D-01, &
&        8.1592000D-02, 9.6978800D-01,-3.7149500D-01, 1.2709900D+00, 1.0000000D+00, &
&        6.5007000D-02, 3.8204000D-01, 6.7654500D-01,-1.0031000D-01, 6.5088000D-02, &
&        9.8145500D-01,-2.8608900D-01, 1.2280600D+00, 1.0000000D+00, 6.4582700D-02, &
&        3.8036300D-01, 6.7819000D-01,-9.8763900D-02, 5.1133800D-02, 9.9133700D-01, &
&       -2.2240100D-01, 1.1825200D+00, 1.0000000D+00, 6.4170700D-02, 3.7879700D-01, &
&        6.7975200D-01,-9.7466100D-02, 3.9056900D-02, 9.9991600D-01,-1.7686600D-01, &
&        1.1469000D+00, 1.0000000D+00/)
      real(8) :: coeffp3(48)= &
&     (/ 1.2823300D-01, 4.7153300D-01, 6.0427300D-01, 9.0665000D-03, 9.9720200D-01, &
&        1.0000000D+00, 1.2101400D-01, 4.6281000D-01, 6.0690700D-01, 2.4263300D-02, &
&        9.8667300D-01, 1.0000000D+00, 1.1757400D-01, 4.6117400D-01, 6.0553500D-01, &
&        5.1938300D-02, 9.7266000D-01, 1.0000000D+00, 1.1335500D-01, 4.5757800D-01, &
&        6.0742700D-01, 6.7103000D-02, 9.5688300D-01, 1.0000000D+00, 1.1085100D-01, &
&        4.5649500D-01, 6.0693600D-01, 9.1582000D-02, 9.3492400D-01, 1.0000000D+00, &
&        1.0964600D-01, 4.5764900D-01, 6.0426100D-01, 1.6477700D-01, 8.7085500D-01, &
&        1.0000000D+00, 1.0859800D-01, 4.5868200D-01, 6.0196200D-01, 2.1921600D-01, &
&        8.2232100D-01, 1.0000000D+00, 1.0761900D-01, 4.5957600D-01, 6.0004100D-01, &
&        2.5568700D-01, 7.8984200D-01, 1.0000000D+00/)
      real(8) :: exs4(216)= &
&     (/ 1.7211755D+03, 2.6001633D+02, 5.6624554D+01, 7.1557200D+01, 1.5438940D+01, &
&        4.4745510D+00, 4.1212750D+00, 1.1886210D+00, 3.7567400D-01, 2.4457700D-01, &
&        3.8972000D-02, 1.6063000D-02, 1.9154348D+03, 2.8953324D+02, 6.3106352D+01, &
&        8.0397440D+01, 1.7330750D+01, 5.0836240D+00, 4.7822290D+00, 1.4625580D+00, &
&        4.7922300D-01, 4.3968200D-01, 5.9130000D-02, 2.3897000D-02, 2.1198870D+03, &
&        3.2042990D+02, 6.9898930D+01, 8.9764500D+01, 1.9385100D+01, 5.7314230D+00, &
&        5.4919380D+00, 1.7437420D+00, 5.6622700D-01, 5.1680200D-01, 6.7214000D-02, &
&        2.5985000D-02, 2.3350200D+03, 3.5304410D+02, 7.7058450D+01, 9.9573870D+01, &
&        2.1546710D+01, 6.4139650D+00, 6.2382790D+00, 1.9961080D+00, 6.4649000D-01, &
&        5.7328500D-01, 7.3119000D-02, 2.6538000D-02, 2.5638770D+03, 3.8753400D+02, &
&        8.4598230D+01, 1.0979380D+02, 2.3769210D+01, 7.1229610D+00, 6.9812040D+00, &
&        2.2198390D+00, 7.1980300D-01, 6.3126200D-01, 8.0062000D-02, 2.8865000D-02, &
&        2.7982940D+03, 4.2313700D+02, 9.2438860D+01, 1.2028060D+02, 2.6037270D+01, &
&        7.8441720D+00, 7.7932760D+00, 2.4971960D+00, 8.0514200D-01, 7.0392100D-01, &
&        8.6162000D-02, 3.2199000D-02, 3.0416860D+03, 4.6009010D+02, 1.0059580D+02, &
&        1.3176730D+02, 2.8569150D+01, 8.6605010D+00, 8.5690810D+00, 2.7681780D+00, &
&        8.8728800D-01, 7.6744300D-01, 9.2025000D-02, 3.3265000D-02, 3.2991840D+03, &
&        4.9908860D+02, 1.0916140D+02, 1.4346520D+02, 3.1168580D+01, 9.4836120D+00, &
&        9.4645650D+00, 3.1003730D+00, 9.8649300D-01, 8.5341200D-01, 9.8812000D-02, &
&        3.6442000D-02, 3.5647620D+03, 5.3939080D+02, 1.1804490D+02, 1.5543820D+02, &
&        3.3815610D+01, 1.0333230D+01, 1.0381520D+01, 3.3827140D+00, 1.0769540D+00, &
&        9.0901500D-01, 1.0504100D-01, 3.7257000D-02, 3.8480050D+03, 5.8203070D+02, &
&        1.2736740D+02, 1.6828960D+02, 3.6656330D+01, 1.1232120D+01, 1.1358770D+01, &
&        3.7388460D+00, 1.1824630D+00, 9.8890400D-01, 1.1102500D-01, 3.9258000D-02, &
&        4.1343020D+03, 6.2549120D+02, 1.3695560D+02, 1.8149600D+02, 3.9574310D+01, &
&        1.2162460D+01, 1.2351110D+01, 4.0496510D+00, 1.2792250D+00, 1.0498040D+00, &
&        1.1693300D-01, 7.5100000D-04, 4.4322880D+03, 6.7066010D+02, 1.4690240D+02, &
&        1.9500420D+02, 4.2568890D+01, 1.3121430D+01, 1.3402310D+01, 4.3999060D+00, &
&        1.3851480D+00, 1.1215580D+00, 1.2294400D-01, 4.2193000D-02, 4.7518979D+03, &
&        7.1892054D+02, 1.5744592D+02, 2.0958340D+02, 4.5691710D+01, 1.4132970D+01, &
&        1.4599540D+01, 4.8608420D+00, 1.5491110D+00, 1.2679430D+00, 1.8839900D-01, &
&        5.7237000D-02, 5.0737499D+03, 7.6772417D+02, 1.6818881D+02, 2.2443600D+02, &
&        4.8955430D+01, 1.5183700D+01, 1.5912570D+01, 5.4414370D+00, 1.7426030D+00, &
&        1.4665380D+00, 2.6309300D-01, 8.4821000D-02, 5.4076138D+03, 8.1817436D+02, &
&        1.7926569D+02, 2.3777830D+02, 5.4256620D+01, 1.6328030D+01, 1.7101850D+01, &
&        5.8051440D+00, 1.9020840D+00, 1.6754040D+00, 3.4165600D-01, 1.1363000D-01, &
&        5.7513215D+03, 8.7025721D+02, 1.9072949D+02, 2.5501640D+02, 5.5576540D+01, &
&        1.7356610D+01, 1.8445680D+01, 6.3287590D+00, 2.0967580D+00, 1.8726330D+00, &
&        4.1747400D-01, 1.3709100D-01, 6.1032899D+03, 9.2369743D+02, 2.0252031D+02, &
&        2.7060150D+02, 5.8253570D+01, 1.8469330D+01, 1.9761420D+01, 6.8217520D+00, &
&        2.2916290D+00, 2.1312060D+00, 4.9935400D-01, 1.6476400D-01, 6.4466307D+03, &
&        9.7687570D+02, 2.1447955D+02, 2.8764460D+02, 6.2620090D+01, 1.9691740D+01, &
&        2.1123210D+01, 7.3032860D+00, 2.4888500D+00, 2.3613740D+00, 5.8601600D-01, &
&        1.9444700D-01/)
      real(8) :: coeffs4(216)= &
&     (/ 6.4874700D-02, 3.8085930D-01, 6.7736810D-01,-1.0934290D-01, 1.1306400D-01, &
&        9.4625750D-01,-2.6997300D-01, 3.6463230D-01, 8.1075330D-01,-2.6882500D-01, &
&        1.1289830D+00, 1.0000000D+00, 6.4624000D-02, 3.7983800D-01, 6.7832900D-01, &
&       -1.0930300D-01, 1.0890000D-01, 9.4927700D-01,-2.8160700D-01, 3.4105100D-01, &
&        8.3810400D-01,-2.6970500D-01, 1.1132930D+00, 1.0000000D+00, 6.4421000D-02, &
&        3.7916000D-01, 6.7896300D-01,-1.0938400D-01, 1.0507000D-01, 9.5220500D-01, &
&       -2.8521100D-01, 3.2415500D-01, 8.5659200D-01,-2.6267800D-01, 1.1080790D+00, &
&        1.0000000D+00, 6.4217000D-02, 3.7841200D-01, 6.7968100D-01,-1.0947200D-01, &
&        1.0194300D-01, 9.5462200D-01,-2.8613700D-01, 3.2182800D-01, 8.5955100D-01, &
&       -2.4245000D-01, 1.1000750D+00, 1.0000000D+00, 6.3948000D-02, 3.7759400D-01, &
&        6.8054200D-01,-1.0983600D-01, 1.0070700D-01, 9.5563300D-01,-2.8845900D-01, &
&        3.3643600D-01, 8.4819000D-01,-2.3649000D-01, 1.0977210D+00, 1.0000000D+00, &
&        6.3824000D-02, 3.7708400D-01, 6.8098900D-01,-1.1777900D-01, 1.0143100D-01, &
&        9.5719800D-01,-2.8885700D-01, 3.3511500D-01, 8.5024800D-01,-2.3225100D-01, &
&        1.0936710D+00, 1.0000000D+00, 6.3745000D-02, 3.7674900D-01, 6.8124700D-01, &
&       -1.1029600D-01, 9.8190000D-02, 9.5765900D-01,-2.9171400D-01, 3.4396300D-01, &
&        8.4519800D-01,-2.3000400D-01, 1.0914500D+00, 1.0000000D+00, 6.3586000D-02, &
&        3.7620200D-01, 6.8178500D-01,-1.1055200D-01, 9.6847000D-02, 9.5879700D-01, &
&       -2.9205600D-01, 3.3752400D-01, 8.5194200D-01,-2.2794400D-01, 1.0882870D+00, &
&        1.0000000D+00, 6.3487000D-02, 3.7581800D-01, 6.8212200D-01,-1.1098700D-01, &
&        9.6767000D-02, 9.5899200D-01,-2.9226200D-01, 3.4325100D-01, 8.4696300D-01, &
&       -2.1746000D-01, 1.0849980D+00, 1.0000000D+00, 6.3266000D-02, 3.7517100D-01, &
&        6.8282400D-01,-1.1111500D-01, 9.5324000D-02, 9.6016100D-01,-2.9206000D-01, &
&        3.3754100D-01, 8.5253300D-01,-2.1368700D-01, 1.0819330D+00, 1.0000000D+00, &
&        6.3188000D-02, 3.7484500D-01, 6.8310000D-01,-1.1132000D-01, 9.4487000D-02, &
&        9.6087900D-01,-2.9222300D-01, 3.4299100D-01, 8.4794600D-01,-2.0650800D-01, &
&        1.0792730D+00, 1.0000000D+00, 6.3093000D-02, 3.7450400D-01, 6.8341600D-01, &
&       -1.1162800D-01, 9.4336000D-02, 9.6110000D-01,-2.9178100D-01, 3.4261400D-01, &
&        8.4828400D-01,-2.0237100D-01, 1.0770350D+00, 1.0000000D+00, 6.2840000D-02, &
&        3.7361100D-01, 6.8436300D-01,-1.1151600D-01, 9.2696000D-02, 9.6228700D-01, &
&        2.9102900D-01,-3.2318800D-01,-8.6439100D-01,-2.8513100D-01, 1.1280220D+00, &
&        1.0000000D+00, 6.2725000D-02, 3.7316700D-01, 6.8478700D-01,-1.1151500D-01, &
&        9.1200000D-02, 9.6344900D-01,-2.8956500D-01, 2.9388300D-01, 8.8919900D-01, &
&       -3.9673400D-01, 1.1906700D+00, 1.0000000D+00, 6.2601000D-02, 3.7277900D-01, &
&        6.8518400D-01,-1.1283800D-01, 8.7227000D-02, 9.6818800D-01,-2.9145400D-01, &
&        2.9696200D-01, 8.8657900D-01,-5.0576100D-01, 1.2517640D+00, 1.0000000D+00, &
&        6.2493000D-02, 3.7236800D-01, 6.8558000D-01,-1.1190800D-01, 9.0999000D-02, &
&        9.6366800D-01,-2.9179300D-01, 2.8462100D-01, 8.9730500D-01,-5.6776400D-01, &
&        1.2941270D+00, 1.0000000D+00, 6.2418000D-02, 3.7204100D-01, 6.8587300D-01, &
&       -1.1214900D-01, 9.3145000D-02, 9.6167900D-01,-2.9387000D-01, 2.8026600D-01, &
&        9.0203600D-01,-6.5180300D-01, 1.3360120D+00, 1.0000000D+00, 6.2540000D-02, &
&        3.7210700D-01, 6.8561100D-01,-1.1206100D-01, 9.0139000D-02, 9.6433000D-01, &
&       -2.9581700D-01, 2.7921700D-01, 9.0373000D-01,-7.2024500D-01, 1.3768460D+00, &
&        1.0000000D+00/)
      real(8) :: coeffp4(162)= &
&     (/ 1.3396540D-01, 5.3026730D-01, 5.1179920D-01, 1.9949220D-02, 4.3402130D-01, &
&        6.4532260D-01, 3.0810350D-04, 9.9987870D-01, 1.0000000D+00, 1.3543300D-01, &
&        5.3722200D-01, 5.0180400D-01, 1.9009000D-02, 4.3603800D-01, 6.3867100D-01, &
&        3.0800000D-04, 9.9989600D-01, 1.0000000D+00, 1.3632800D-01, 5.4186000D-01, &
&        4.9505500D-01, 1.7614000D-02, 4.3364500D-01, 6.4255100D-01, 3.2700000D-04, &
&        9.9989300D-01, 1.0000000D+00, 1.3729700D-01, 5.4587500D-01, 4.8906800D-01, &
&        1.9237000D-02, 4.4044200D-01, 6.3562000D-01, 2.9200000D-04, 9.9990700D-01, &
&        1.0000000D+00, 1.3842100D-01, 5.5048900D-01, 4.8241700D-01, 2.1821000D-02, &
&        4.5676200D-01, 6.1867500D-01, 1.9000000D-04, 9.9994000D-01, 1.0000000D+00, &
&        1.3987800D-01, 5.5598300D-01, 4.7481800D-01, 2.2185000D-02, 4.6162500D-01, &
&        6.1453900D-01, 1.8000000D-04, 9.9994500D-01, 1.0000000D+00, 1.4045400D-01, &
&        5.5780200D-01, 4.7150100D-01, 2.4224000D-02, 4.6866000D-01, 6.0742100D-01, &
&        3.0800000D-04, 9.9990700D-01, 1.0000000D+00, 1.4110100D-01, 5.6038700D-01, &
&        4.6764400D-01, 2.3762000D-02, 4.6891100D-01, 6.0831100D-01,-4.2600000D-04, &
&        1.0001240D+00, 1.0000000D+00, 1.4206400D-01, 5.6344400D-01, 4.6302400D-01, &
&        2.6313000D-02, 4.7691700D-01, 5.9915400D-01, 2.2800000D-04, 9.9993400D-01, &
&        1.0000000D+00, 1.4249000D-01, 5.6554700D-01, 4.5999300D-01, 2.6138000D-02, &
&        4.7659800D-01, 6.0038000D-01, 2.9400000D-04, 9.9991700D-01, 1.0000000D+00, &
&        1.4308400D-01, 5.6775600D-01, 4.5671400D-01, 2.7727000D-02, 4.8352400D-01, &
&        5.9297800D-01, 1.3900000D-04, 9.9996100D-01, 1.0000000D+00, 1.4380600D-01, &
&        5.7000200D-01, 4.5331200D-01, 2.8705000D-02, 4.8625200D-01, 5.9023500D-01, &
&        3.4400000D-04, 9.9990500D-01, 1.0000000D+00, 1.4426600D-01, 5.7317800D-01, &
&        4.4908600D-01, 2.6562000D-02, 4.8331400D-01, 5.9243000D-01, 3.0183000D-02, &
&        9.8846600D-01, 1.0000000D+00, 1.4464000D-01, 5.7538000D-01, 4.4599500D-01, &
&        2.2973000D-02, 4.7324500D-01, 6.0327800D-01, 2.7893000D-02, 9.8749000D-01, &
&        1.0000000D+00, 1.4968000D-01, 5.6232200D-01, 4.5932300D-01, 2.5686000D-02, &
&        4.8339700D-01, 5.8878500D-01, 2.5282000D-02, 9.8743300D-01, 1.0000000D+00, &
&        1.4614900D-01, 5.8137100D-01, 4.3746000D-01, 2.4421000D-02, 4.8336500D-01, &
&        5.8790400D-01, 2.8255000D-02, 9.8490600D-01, 1.0000000D+00, 1.4775100D-01, &
&        6.0105600D-01, 4.1287000D-01, 2.5007000D-02, 4.8661000D-01, 5.8242300D-01, &
&        2.8708000D-02, 9.8407000D-01, 1.0000000D+00, 1.4752800D-01, 5.8689200D-01, &
&        4.2950700D-01, 2.6070000D-02, 4.9225000D-01, 5.7427400D-01, 2.8775000D-02, &
&        9.8333900D-01, 1.0000000D+00/)
      real(8) :: exd4(48)= &
&     (/ 5.7222150D+00, 1.3608490D+00, 3.2265200D-01, 7.0836660D+00, 1.7096340D+00, &
&        4.1412300D-01, 8.3429170D+00, 2.0329440D+00, 4.9571200D-01, 9.6253390D+00, &
&        2.3622640D+00, 5.7709400D-01, 1.1068840D+01, 2.7307070D+00, 6.6850900D-01, &
&        1.2354490D+01, 3.0556050D+00, 7.3859100D-01, 1.3740700D+01, 3.4089830D+00, &
&        8.1864100D-01, 1.5220690D+01, 3.7860200D+00, 9.0455700D-01, 1.6759380D+01, &
&        4.1789770D+00, 9.9432700D-01, 1.8368200D+01, 4.5913040D+00, 1.0902030D+00, &
&        2.1292530D+01, 5.3931660D+00, 1.3338830D+00, 2.4321421D+01, 6.2238140D+00, &
&        1.5887370D+00, 2.7437209D+01, 7.0840440D+00, 1.8558230D+00, 3.0627464D+01, &
&        7.9712760D+00, 2.1348100D+00, 3.3965097D+01, 8.9008310D+00, 2.4284360D+00, &
&        3.7368103D+01, 9.8543130D+00, 2.7327950D+00/)
      real(8) :: coeffd4(48)= &
&     (/ 2.6523600D-01, 8.5586100D-01, 1.0000000D+00, 2.6292100D-01, 8.5577200D-01, &
&        1.0000000D+00, 2.6406200D-01, 8.5396600D-01, 1.0000000D+00, 2.6559600D-01, &
&        8.5215600D-01, 1.0000000D+00, 2.6527200D-01, 8.5179500D-01, 1.0000000D+00, &
&        2.6861100D-01, 8.4927200D-01, 1.0000000D+00, 2.7095500D-01, 8.4734200D-01, &
&        1.0000000D+00, 2.7260600D-01, 8.4592800D-01, 1.0000000D+00, 2.7411200D-01, &
&        8.4462500D-01, 1.0000000D+00, 2.7538600D-01, 8.4347700D-01, 1.0000000D+00, &
&        1.6199000D-01, 5.1167400D-01, 5.8987300D-01, 1.5779900D-01, 5.1149200D-01, &
&        5.8577000D-01, 1.5449500D-01, 5.1143200D-01, 5.8219400D-01, 1.5198600D-01, &
&        5.1164000D-01, 5.7869400D-01, 1.4966700D-01, 5.1174800D-01, 5.7591500D-01, &
&        1.4794700D-01, 5.1217200D-01, 5.7295000D-01/)
      real(8) :: exs5(270)= &
&     (/ 6.8167225D+03, 1.0330007D+03, 2.2690861D+02, 3.0412830D+02, 6.6260580D+01, &
&        2.0919450D+01, 2.2465330D+01, 7.8774680D+00, 2.7052710D+00, 2.6921160D+00, &
&        7.2305600D-01, 2.5983800D-01, 1.8971400D-01, 3.3997000D-02, 1.4712000D-02, &
&        7.2154735D+03, 1.0928519D+03, 2.3998182D+02, 3.2212460D+02, 7.0090460D+01, &
&        2.2176410D+01, 2.3927630D+01, 8.4751140D+00, 2.9429340D+00, 2.9409660D+00, &
&        8.5235600D-01, 3.2153700D-01, 3.4804200D-01, 4.8177000D-02, 2.1803000D-02, &
&        7.6464210D+03, 1.1568630D+03, 2.5371520D+02, 3.4185400D+02, 7.4209860D+01, &
&        2.3513520D+01, 1.8862600D+01, 1.6454050D+01, 3.4845000D+00, 3.2217330D+00, &
&        1.0507050D+00, 3.9259200D-01, 4.3276400D-01, 5.7012000D-02, 2.3754000D-02, &
&        8.0845920D+03, 1.2216680D+03, 2.6769170D+02, 3.6102120D+02, 7.8304950D+01, &
&        2.4845230D+01, 2.0006280D+01, 1.7574150D+01, 3.7429850D+00, 3.5547880D+00, &
&        1.1789920D+00, 4.4469700D-01, 5.0504900D-01, 6.2116000D-02, 2.5580000D-02, &
&        8.4665170D+03, 1.2812610D+03, 2.8123110D+02, 3.7947290D+02, 8.2335890D+01, &
&        2.6222480D+01, 2.1162940D+01, 1.8589780D+01, 4.0099810D+00, 3.8363750D+00, &
&        1.3033250D+00, 4.9343100D-01, 5.7237300D-01, 6.8203000D-02, 2.7157000D-02, &
&        8.8994910D+03, 1.3467640D+03, 2.9563520D+02, 3.9931390D+02, 8.6593560D+01, &
&        2.7639040D+01, 2.2502920D+01, 1.9491710D+01, 4.2781800D+00, 4.1630210D+00, &
&        1.4353050D+00, 5.4378200D-01, 6.3180100D-01, 7.3258000D-02, 2.8025000D-02, &
&        9.3294820D+03, 1.4125060D+03, 3.1026430D+02, 4.1881750D+02, 9.1250780D+01, &
&        2.9112120D+01, 2.5910640D+01, 2.3267700D+01, 4.7070830D+00, 4.4411380D+00, &
&        1.5956390D+00, 5.9556000D-01, 6.7388100D-01, 7.7241000D-02, 2.8696000D-02, &
&        9.7861610D+03, 1.4814770D+03, 3.2541220D+02, 4.3986650D+02, 9.5762730D+01, &
&        3.0605660D+01, 2.7277370D+01, 2.4510820D+01, 5.0089460D+00, 4.7658120D+00, &
&        1.7345310D+00, 6.4663600D-01, 7.4066200D-01, 8.2171000D-02, 3.0097000D-02, &
&        1.0217710D+04, 1.5484120D+03, 3.4049900D+02, 4.6075930D+02, 1.0032890D+02, &
&        3.2139710D+01, 2.8793290D+01, 2.5917680D+01, 5.3206400D+00, 5.1097480D+00, &
&        1.8754140D+00, 6.9955800D-01, 8.0057100D-01, 8.7321000D-02, 3.1407000D-02, &
&        1.0728740D+04, 1.6240740D+03, 3.5679370D+02, 4.8247830D+02, 1.0505900D+02, &
&        3.3681450D+01, 3.0186540D+01, 2.7166420D+01, 5.6359340D+00, 5.4753740D+00, &
&        1.9976040D+00, 7.4393000D-01, 8.9016300D-01, 9.2821000D-02, 3.3774000D-02, &
&        1.1190780D+04, 1.6950770D+03, 3.7267520D+02, 5.0461620D+02, 1.0987180D+02, &
&        3.5295130D+01, 3.1568770D+01, 2.8343970D+01, 5.9451270D+00, 5.8002560D+00, &
&        2.1272560D+00, 7.9355100D-01, 9.2854400D-01, 9.7255000D-02, 3.4933000D-02, &
&        1.1686090D+04, 1.7701110D+03, 3.8920900D+02, 5.2760040D+02, 1.1483290D+02, &
&        3.6958290D+01, 3.3015480D+01, 2.9545430D+01, 6.2785080D+00, 6.1505960D+00, &
&        2.2597460D+00, 8.4142600D-01, 9.4906900D-01, 1.0148800D-01, 3.5987000D-02, &
&        1.2214547D+04, 1.8489136D+03, 4.0636833D+02, 5.5044230D+02, 1.1977440D+02, &
&        3.8669270D+01, 4.7029310D+01, 2.2496420D+01, 6.6971170D+00, 6.5723600D+00, &
&        2.5021580D+00, 9.4202500D-01, 1.0012210D+00, 1.6597000D-01, 5.4340000D-02, &
&        1.2741674D+04, 1.9284692D+03, 4.2380797D+02, 5.7428750D+02, 1.2495370D+02, &
&        4.0395760D+01, 4.8806620D+01, 2.3835880D+01, 7.0482960D+00, 6.9733780D+00, &
&        2.6930400D+00, 1.0259580D+00, 1.1314630D+00, 2.0340900D-01, 7.0564000D-02, &
&        1.3289383D+04, 2.0105218D+03, 4.4169815D+02, 5.9888900D+02, 1.3003850D+02, &
&        4.2132860D+01, 5.1513330D+01, 2.4435950D+01, 7.4209310D+00, 7.3142350D+00, &
&        2.8440530D+00, 1.1058550D+00, 1.2786370D+00, 2.4123200D-01, 8.6630000D-02, &
&        1.3796560D+04, 2.0888798D+03, 4.5939319D+02, 6.2326310D+02, 1.3536000D+02, &
&        4.4000480D+01, 5.4190780D+01, 2.5820390D+01, 7.8095830D+00, 7.7642170D+00, &
&        3.0439320D+00, 1.1992530D+00, 1.3403640D+00, 2.7808800D-01, 9.6726000D-02, &
&        1.4351186D+04, 2.1730741D+03, 4.7787205D+02, 6.4818870D+02, 1.4030640D+02, &
&        4.5698800D+01, 5.6694690D+01, 2.7488750D+01, 8.2090960D+00, 8.1916790D+00, &
&        3.2445960D+00, 1.3004890D+00, 1.4513800D+00, 3.2810300D-01, 1.1507600D-01, &
&        1.4902236D+04, 2.2565383D+03, 4.9637317D+02, 6.7366110D+02, 1.4584910D+02, &
&        4.7577080D+01, 5.9167520D+01, 2.8611590D+01, 8.5965960D+00, 8.6386760D+00, &
&        3.4628180D+00, 1.4010400D+00, 1.5784740D+00, 3.7508100D-01, 1.3317900D-01/)
      real(8) :: coeffs5(270)= &
&     (/ 6.2496000D-02, 3.7195000D-01, 6.8572900D-01,-1.1233000D-01, 9.0751000D-02, &
&        9.6394100D-01,-3.0048500D-01, 2.7835700D-01, 9.0761000D-01,-3.3116200D-01, &
&        5.0969900D-01, 6.9824600D-01,-2.7119300D-01, 1.1415500D+00, 1.0000000D+00, &
&        6.2282000D-02, 3.7131000D-01, 6.8644400D-01,-1.1223500D-01, 8.9544000D-02, &
&        9.6481300D-01,-3.0247200D-01, 2.7008400D-01, 9.1592000D-01,-3.5198500D-01, &
&        4.9725500D-01, 7.2386000D-01,-2.8514700D-01, 1.1209390D+00, 1.0000000D+00, &
&        6.1891000D-02, 3.7020700D-01, 6.8775600D-01,-1.1190000D-01, 8.6805000D-02, &
&        9.6678500D-01,-1.4778730D+00, 1.3472590D+00, 1.0062310D+00,-3.6995800D-01, &
&        4.3086400D-01, 8.0208700D-01,-3.4645800D-01, 1.1327770D+00, 1.0000000D+00, &
&        6.1578000D-02, 3.6939900D-01, 6.8872800D-01,-1.1190700D-01, 8.5840000D-02, &
&        9.6751300D-01,-1.5443490D+00, 1.4095960D+00, 1.0093490D+00,-3.7938700D-01, &
&        4.2328500D-01, 8.1406800D-01,-3.2951200D-01, 1.1207090D+00, 1.0000000D+00, &
&        6.1804000D-02, 3.6980500D-01, 6.8807900D-01,-1.1210600D-01, 8.6503000D-02, &
&        9.6705700D-01,-1.5551310D+00, 1.4179390D+00, 1.0106200D+00,-3.8910400D-01, &
&        4.3497000D-01, 8.1159000D-01,-3.1560900D-01, 1.1140470D+00, 1.0000000D+00, &
&        6.1706000D-02, 3.6945400D-01, 6.8843400D-01,-1.1214400D-01, 8.6011000D-02, &
&        9.6743300D-01,-1.4223060D+00, 1.2841850D+00, 1.0108660D+00,-3.9642300D-01, &
&        4.3707900D-01, 8.1484600D-01,-3.0336200D-01, 1.1084130D+00, 1.0000000D+00, &
&        6.1712000D-02, 3.6933700D-01, 6.8847200D-01,-1.1240300D-01, 8.5318000D-02, &
&        9.6817700D-01,-1.3804460D+00, 1.1979000D+00, 1.0526490D+00,-4.0411400D-01, &
&        4.3983800D-01, 8.2193600D-01,-2.7000300D-01, 1.0991500D+00, 1.0000000D+00, &
&        6.1605000D-02, 3.6898200D-01, 6.8884500D-01,-1.1239100D-01, 8.4694000D-02, &
&        9.6863800D-01,-1.3955530D+00, 1.2108520D+00, 1.0540450D+00,-4.1036300D-01, &
&        4.4800300D-01, 8.1980800D-01,-2.6396600D-01, 1.0948570D+00, 1.0000000D+00, &
&        6.1732000D-02, 3.6915300D-01, 6.8851400D-01,-1.1244600D-01, 8.4381000D-02, &
&        9.6890200D-01,-1.4040910D+00, 1.2161690D+00, 1.0565550D+00,-4.1264700D-01, &
&        4.5188500D-01, 8.1889900D-01,-2.5534800D-01, 1.0913080D+00, 1.0000000D+00, &
&        6.1429000D-02, 3.6832800D-01, 6.8950300D-01,-1.1267900D-01, 8.4612000D-02, &
&        9.6878700D-01,-1.4185470D+00, 1.2294440D+00, 1.0570830D+00,-4.1726100D-01, &
&        4.7058800D-01, 8.0463600D-01,-2.7843200D-01, 1.0930280D+00, 1.0000000D+00, &
&        6.1495000D-02, 3.6840500D-01, 6.8932500D-01,-1.1265800D-01, 8.4028000D-02, &
&        9.6923400D-01,-1.4220280D+00, 1.2340980D+00, 1.0556830D+00,-4.1961700D-01, &
&        4.8435000D-01, 7.9520300D-01,-2.5230000D-01, 1.0873920D+00, 1.0000000D+00, &
&        6.1427000D-02, 3.6815700D-01, 6.8957200D-01,-1.1259200D-01, 8.3270000D-02, &
&        9.6979800D-01,-1.4064710D+00, 1.2181560D+00, 1.0555200D+00,-4.2292100D-01, &
&        4.9877100D-01, 7.8507600D-01,-2.2155500D-01, 1.0809440D+00, 1.0000000D+00, &
&        6.1248000D-02, 3.6767500D-01, 6.9013600D-01,-1.1270900D-01, 8.3444000D-02, &
&        9.6968800D-01,-2.7589500D-01, 5.9773000D-02, 1.0821480D+00,-4.2848300D-01, &
&        4.6336400D-01, 8.2196800D-01,-4.3641700D-01, 1.1898930D+00, 1.0000000D+00, &
&        6.1135000D-02, 3.6729300D-01, 6.9054500D-01,-1.1274600D-01, 8.2863000D-02, &
&        9.7015100D-01,-2.8245300D-01, 6.6056000D-02, 1.0819870D+00,-4.3403600D-01, &
&        4.6102900D-01, 8.2855800D-01,-5.2520800D-01, 1.2292260D+00, 1.0000000D+00, &
&        6.0984000D-02, 3.6684900D-01, 6.9105000D-01,-1.1272000D-01, 8.2644000D-02, &
&        9.7025800D-01,-2.7704300D-01, 5.7503000D-02, 1.0847030D+00,-4.4038100D-01, &
&        4.7373400D-01, 8.2213500D-01,-6.0169500D-01, 1.2586920D+00, 1.0000000D+00, &
&        6.1086000D-02, 3.6696300D-01, 6.9079400D-01,-1.1282000D-01, 8.2252000D-02, &
&        9.7060100D-01,-2.7441200D-01, 5.1830000D-02, 1.0876220D+00,-4.4673100D-01, &
&        4.6947000D-01, 8.2981100D-01,-5.9047000D-01, 1.2819680D+00, 1.0000000D+00, &
&        6.1003000D-02, 3.6664000D-01, 6.9113100D-01,-1.1285100D-01, 8.3228000D-02, &
&        9.6975200D-01,-2.7369700D-01, 4.6500000D-02, 1.0915760D+00,-4.5082800D-01, &
&        4.6320900D-01, 8.3863600D-01,-6.6585200D-01, 1.3285840D+00, 1.0000000D+00, &
&        6.0997000D-02, 3.6662900D-01, 6.9111500D-01,-1.1291300D-01, 8.2905000D-02, &
&        9.7002900D-01,-2.7397700D-01, 4.5530000D-02, 1.0925280D+00,-4.5584100D-01, &
&        4.6173600D-01, 8.4428800D-01,-7.2777200D-01, 1.3627970D+00, 1.0000000D+00/)
      real(8) :: coeffp5(216)= &
&     (/ 1.4844100D-01, 5.8912500D-01, 4.2582500D-01, 2.4454000D-02, 4.9445400D-01, &
&        5.7185700D-01, 1.1901000D-02, 4.9517300D-01, 5.7372400D-01, 3.0800000D-04, &
&        9.9986500D-01, 1.0000000D+00, 1.4883700D-01, 5.9194700D-01, 4.2217100D-01, &
&        2.4837000D-02, 4.9347800D-01, 5.7098300D-01, 9.7230000D-03, 4.9832200D-01, &
&        5.6505600D-01, 3.0800000D-04, 9.9989300D-01, 1.0000000D+00, 1.4857200D-01, &
&        5.9430700D-01, 4.1960400D-01,-7.0414100D-01, 1.0578620D+00, 7.3938200D-01, &
&        2.4940000D-03, 4.5376200D-01, 6.1306800D-01,-1.3370000D-03, 1.0004400D+00, &
&        1.0000000D+00, 1.4876200D-01, 5.9656900D-01, 4.1678500D-01,-7.5684800D-01, &
&        1.1120900D+00, 7.3680200D-01, 2.5990000D-03, 4.5997600D-01, 6.0584900D-01, &
&       -1.2490000D-03, 1.0003840D+00, 1.0000000D+00, 1.4966700D-01, 5.9871800D-01, &
&        4.1323800D-01,-7.5546200D-01, 1.1139660D+00, 7.3276700D-01, 3.2900000D-03, &
&        4.7164500D-01, 5.9369900D-01,-1.1330000D-03, 1.0003380D+00, 1.0000000D+00, &
&        1.5006600D-01, 6.0076900D-01, 4.1038600D-01,-6.6806600D-01, 1.0303460D+00, &
&        7.2834800D-01, 2.9630000D-03, 4.7914700D-01, 5.8648700D-01,-1.0790000D-03, &
&        1.0003130D+00, 1.0000000D+00, 1.5007200D-01, 6.0005700D-01, 4.1098600D-01, &
&       -1.6552710D+00, 1.9860200D+00, 7.2903400D-01, 1.2291000D-02, 4.6320700D-01, &
&        5.9838300D-01,-9.2000000D-04, 1.0002640D+00, 1.0000000D+00, 1.5037900D-01, &
&        6.0192900D-01, 4.0846400D-01,-1.6686180D+00, 2.0027900D+00, 7.2514400D-01, &
&        1.1274000D-02, 4.7270300D-01, 5.8984300D-01,-7.6200000D-04, 1.0002120D+00, &
&        1.0000000D+00, 1.5085800D-01, 6.0351400D-01, 4.0602500D-01,-1.7122180D+00, &
&        2.0476030D+00, 7.2298300D-01, 9.3740000D-03, 4.8155400D-01, 5.8223100D-01, &
&       -7.7600000D-04, 1.0002120D+00, 1.0000000D+00, 1.5107900D-01, 6.0509200D-01, &
&        4.0398000D-01,-1.7098170D+00, 2.0493080D+00, 7.1863000D-01, 1.1584000D-02, &
&        4.9745500D-01, 5.6552700D-01,-1.2720000D-03, 1.0003320D+00, 1.0000000D+00, &
&        1.5148000D-01, 6.0651400D-01, 4.0183000D-01,-1.6733660D+00, 2.0189760D+00, &
&        7.1268900D-01, 1.4304000D-02, 5.0719400D-01, 5.5397400D-01,-1.4810000D-03, &
&        1.0003880D+00, 1.0000000D+00, 1.5180500D-01, 6.0776000D-01, 3.9996300D-01, &
&       -1.6090240D+00, 1.9595680D+00, 7.0802700D-01, 1.4482000D-02, 5.1866100D-01, &
&        5.4266600D-01,-1.5400000D-03, 1.0004120D+00, 1.0000000D+00, 1.5237000D-01, &
&        6.0965100D-01, 3.9702500D-01,-1.4084800D-01, 5.2908700D-01, 6.6206800D-01, &
&        1.0913000D-02, 5.0367600D-01, 5.5818100D-01, 2.3163000D-02, 9.9033100D-01, &
&        1.0000000D+00, 1.5258000D-01, 6.1101100D-01, 3.9515500D-01,-1.5096300D-01, &
&        5.3996800D-01, 6.6048200D-01, 1.1951000D-02, 5.0671900D-01, 5.5291100D-01, &
&        2.1071000D-02, 9.9059100D-01, 1.0000000D+00, 1.5306700D-01, 6.1359700D-01, &
&        3.9169900D-01,-1.3787000D-01, 5.3635500D-01, 6.5086800D-01, 1.5305000D-02, &
&        5.1608300D-01, 5.3875700D-01, 2.2253000D-02, 9.8964300D-01, 1.0000000D+00, &
&        1.5341900D-01, 6.1490000D-01, 3.8951600D-01,-1.4333100D-01, 5.3918800D-01, &
&        6.5228500D-01, 1.2746000D-02, 5.2212300D-01, 5.3266100D-01, 2.5588000D-02, &
&        9.8710200D-01, 1.0000000D+00, 1.5411400D-01, 6.1946200D-01, 3.8375800D-01, &
&       -1.5232200D-01, 5.4376900D-01, 6.5616800D-01, 1.1870000D-02, 5.2652500D-01, &
&        5.2660800D-01, 2.7541000D-02, 9.8513700D-01, 1.0000000D+00, 1.5442700D-01, &
&        6.2061700D-01, 3.8200400D-01,-1.5185700D-01, 5.4715100D-01, 6.5195400D-01, &
&        9.5850000D-03, 5.2981900D-01, 5.2359800D-01, 2.8072000D-02, 9.8426400D-01, &
&        1.0000000D+00/)
      real(8) :: exd5(102)= &
&     (/ 4.0866031D+01, 1.0840885D+01, 3.0508340D+00, 4.4566115D+01, 1.1881489D+01, &
&        3.3875580D+00, 5.0353750D+01, 1.3530780D+01, 3.9449960D+00, 1.5301370D+00, &
&        6.3006700D-01, 2.1658800D-01, 5.4723230D+01, 1.4774160D+01, 4.3589610D+00, &
&        1.8628420D+00, 6.4331300D-01, 1.9939500D-01, 5.9012190D+01, 1.6012790D+01, &
&        4.7771850D+00, 1.9704430D+00, 6.6193500D-01, 2.0599700D-01, 6.3780450D+01, &
&        1.7373580D+01, 5.2307840D+00, 2.2709370D+00, 7.5465300D-01, 2.3514200D-01, &
&        6.8783750D+01, 1.8803890D+01, 5.7052280D+00, 2.5991640D+00, 8.6227600D-01, &
&        2.7060700D-01, 7.3983300D+01, 2.0281490D+01, 6.1942980D+00, 2.8891080D+00, &
&        9.5396100D-01, 2.9588100D-01, 7.9255970D+01, 2.1789450D+01, 6.6975180D+00, &
&        3.1909080D+00, 1.0545750D+00, 3.2607900D-01, 8.4236910D+01, 2.3249190D+01, &
&        7.1967600D+00, 3.4730770D+00, 1.1480500D+00, 3.5481100D-01, 8.9933350D+01, &
&        2.4874960D+01, 7.7381910D+00, 3.7965570D+00, 1.2566440D+00, 3.8813300D-01, &
&        9.5472740D+01, 2.6481960D+01, 8.2828860D+00, 4.0821410D+00, 1.3572790D+00, &
&        4.2083100D-01, 1.0217356D+02, 2.8394632D+01, 8.9248050D+00, 4.5353640D+00, &
&        1.5371480D+00, 4.9949200D-01, 1.0805630D+02, 3.0131576D+01, 9.5300360D+00, &
&        4.9626100D+00, 1.7120830D+00, 5.7719500D-01, 1.1580955D+02, 3.2305835D+01, &
&        1.0250328D+01, 5.4862100D+00, 1.9216200D+00, 6.6606300D-01, 1.2140830D+02, &
&        3.4015217D+01, 1.0869138D+01, 5.8031110D+00, 2.0580660D+00, 7.3283000D-01, &
&        1.2809026D+02, 3.5982378D+01, 1.1551116D+01, 6.1461520D+00, 2.2209370D+00, &
&        8.0991200D-01, 1.3491331D+02, 3.7956387D+01, 1.2227475D+01, 6.6004930D+00, &
&        2.3980510D+00, 8.8648200D-01/)
      real(8) :: coeffd5(102)= &
&     (/ 1.4660400D-01, 5.1272500D-01, 5.6998000D-01, 1.4512700D-01, 5.1306800D-01, &
&        5.6766400D-01, 1.3679600D-01, 5.0190600D-01, 5.7886000D-01, 3.3840300D-01, &
&        7.2932900D-01, 1.0000000D+00, 1.3482400D-01, 5.0055400D-01, 5.7878200D-01, &
&        2.8503200D-01, 7.9720700D-01, 1.0000000D+00, 1.3371000D-01, 5.0003900D-01, &
&        5.7753800D-01, 3.1068100D-01, 7.8006900D-01, 1.0000000D+00, 1.3173900D-01, &
&        4.9853200D-01, 5.7817800D-01, 3.1126400D-01, 7.8103400D-01, 1.0000000D+00, &
&        1.2969300D-01, 4.9661900D-01, 5.7954700D-01, 3.0921900D-01, 7.8290600D-01, &
&        1.0000000D+00, 1.2776000D-01, 4.9514700D-01, 5.8065500D-01, 3.1599400D-01, &
&        7.7806600D-01, 1.0000000D+00, 1.2619000D-01, 4.9395400D-01, 5.8133000D-01, &
&        3.2104000D-01, 7.7385200D-01, 1.0000000D+00, 1.2564300D-01, 4.9372000D-01, &
&        5.8034300D-01, 3.2815400D-01, 7.6802700D-01, 1.0000000D+00, 1.2401600D-01, &
&        4.9238300D-01, 5.8149700D-01, 3.3142600D-01, 7.6516300D-01, 1.0000000D+00, &
&        1.2308300D-01, 4.9167700D-01, 5.8154100D-01, 3.3794100D-01, 7.5916800D-01, &
&        1.0000000D+00, 1.2055600D-01, 4.8849800D-01, 5.8501900D-01, 2.5085700D-01, &
&        5.6931100D-01, 3.8406400D-01, 1.1982400D-01, 4.8759100D-01, 5.8498700D-01, &
&        2.5294900D-01, 5.7276100D-01, 3.6903900D-01, 1.1662800D-01, 4.8343600D-01, &
&        5.9014000D-01, 2.4836600D-01, 5.7431500D-01, 3.6430400D-01, 1.1691400D-01, &
&        4.8355600D-01, 5.8838600D-01, 2.6019400D-01, 5.7977600D-01, 3.4055800D-01, &
&        1.1586400D-01, 4.8204900D-01, 5.8944500D-01, 2.6818200D-01, 5.8006100D-01, &
&        3.2622600D-01, 1.1501100D-01, 4.8159500D-01, 5.8961300D-01, 2.7188400D-01, &
&        5.8555700D-01, 3.1274600D-01/)
!
! Set functions for H - He
!
      select case(numatomic(iatom))
        case(1:2)
! 1S
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= exs1(is1(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs1(is1(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 2
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs1(is1(numatomic(iatom))+3)
          coeff(locprim(ishell)+1)= coeffs1(is1(numatomic(iatom))+3)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
! Set functions for Li - Ne
!
        case(3:10)
! 1S
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exs2(is2(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs2(is2(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
! 2SP
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= exs2(is2(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffs2(is2(numatomic(iatom))+j+3)
          enddo
          mprim(ishell)= 2
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= exs2(is2(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffp2(ip2(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 2
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs2(is2(numatomic(iatom))+6)
          coeff(locprim(ishell)+1)= coeffs2(is2(numatomic(iatom))+6)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs2(is2(numatomic(iatom))+6)
          coeff(locprim(ishell)+1)= coeffp2(ip2(numatomic(iatom))+3)
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
! Set functions for Na - Ar
!
        case(11:18)
! 1S
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exs3(is3(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs3(is3(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
! 2SP
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exs3(is3(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffs3(is3(numatomic(iatom))+j+3)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exs3(is3(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffp3(ip3(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+3
! 3SP
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= exs3(is3(numatomic(iatom))+j+6)
            coeff(locprim(ishell)+j)= coeffs3(is3(numatomic(iatom))+j+6)
          enddo
          mprim(ishell)= 2
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= exs3(is3(numatomic(iatom))+j+6)
            coeff(locprim(ishell)+j)= coeffp3(ip3(numatomic(iatom))+j+3)
          enddo
          mprim(ishell)= 2
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs3(is3(numatomic(iatom))+9)
          coeff(locprim(ishell)+1)= coeffs3(is3(numatomic(iatom))+9)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs3(is3(numatomic(iatom))+9)
          coeff(locprim(ishell)+1)= coeffp3(ip3(numatomic(iatom))+6)
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
! Set functions for K - Kr
!
        case(19:36)
! 1S
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exs4(is4(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs4(is4(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
! 2SP
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exs4(is4(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffs4(is4(numatomic(iatom))+j+3)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exs4(is4(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffp4(ip4(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+3
! 3SP
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exs4(is4(numatomic(iatom))+j+6)
            coeff(locprim(ishell)+j)= coeffs4(is4(numatomic(iatom))+j+6)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exs4(is4(numatomic(iatom))+j+6)
            coeff(locprim(ishell)+j)= coeffp4(ip4(numatomic(iatom))+j+3)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+3
! 4SP
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= exs4(is4(numatomic(iatom))+j+9)
            coeff(locprim(ishell)+j)= coeffs4(is4(numatomic(iatom))+j+9)
          enddo
          mprim(ishell)= 2
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= exs4(is4(numatomic(iatom))+j+9)
            coeff(locprim(ishell)+j)= coeffp4(ip4(numatomic(iatom))+j+6)
          enddo
          mprim(ishell)= 2
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs4(is4(numatomic(iatom))+12)
          coeff(locprim(ishell)+1)= coeffs4(is4(numatomic(iatom))+12)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs4(is4(numatomic(iatom))+12)
          coeff(locprim(ishell)+1)= coeffp4(ip4(numatomic(iatom))+9)
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
! 3D
          select case(numatomic(iatom))
            case(21:30)
              ishell= ishell+1
              do j= 1,2
                ex(locprim(ishell)+j)= exd4(id4(numatomic(iatom))+j)
                coeff(locprim(ishell)+j)= coeffd4(id4(numatomic(iatom))+j)
              enddo
              mprim(ishell)= 2
              mtype(ishell)= 2
              locatom(ishell)= iatom
              locprim(ishell+1)= locprim(ishell)+2
              if(spher) then
                mbf(ishell)= 5
                locbf(ishell+1)= locbf(ishell)+5
              else
                mbf(ishell)= 6
                locbf(ishell+1)= locbf(ishell)+6
              endif
!
              ishell= ishell+1
              ex(locprim(ishell)+1)= exd4(id4(numatomic(iatom))+3)
              coeff(locprim(ishell)+1)= coeffd4(id4(numatomic(iatom))+3)
              mprim(ishell)= 1
              mtype(ishell)= 2
              locatom(ishell)= iatom
              locprim(ishell+1)= locprim(ishell)+1
              if(spher) then
                mbf(ishell)= 5
                locbf(ishell+1)= locbf(ishell)+5
              else
                mbf(ishell)= 6
                locbf(ishell+1)= locbf(ishell)+6
              endif
!
            case(31:36)
              ishell= ishell+1
              do j= 1,3
                ex(locprim(ishell)+j)= exd4(id4(numatomic(iatom))+j)
                coeff(locprim(ishell)+j)= coeffd4(id4(numatomic(iatom))+j)
              enddo
              mprim(ishell)= 3
              mtype(ishell)= 2
              locatom(ishell)= iatom
              locprim(ishell+1)= locprim(ishell)+3
              if(spher) then
                mbf(ishell)= 5
                locbf(ishell+1)= locbf(ishell)+5
              else
                mbf(ishell)= 6
                locbf(ishell+1)= locbf(ishell)+6
              endif
          end select
!
! Set functions for Rb - Xe
!
        case(37:54)
! 1S
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exs5(is5(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs5(is5(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
! 2SP
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exs5(is5(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffs5(is5(numatomic(iatom))+j+3)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exs5(is5(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffp5(ip5(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+3
! 3SP
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exs5(is5(numatomic(iatom))+j+6)
            coeff(locprim(ishell)+j)= coeffs5(is5(numatomic(iatom))+j+6)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exs5(is5(numatomic(iatom))+j+6)
            coeff(locprim(ishell)+j)= coeffp5(ip5(numatomic(iatom))+j+3)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+3
! 4SP
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exs5(is5(numatomic(iatom))+j+9)
            coeff(locprim(ishell)+j)= coeffs5(is5(numatomic(iatom))+j+9)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exs5(is5(numatomic(iatom))+j+9)
            coeff(locprim(ishell)+j)= coeffp5(ip5(numatomic(iatom))+j+6)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+3
! 5SP
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= exs5(is5(numatomic(iatom))+j+12)
            coeff(locprim(ishell)+j)= coeffs5(is5(numatomic(iatom))+j+12)
          enddo
          mprim(ishell)= 2
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= exs5(is5(numatomic(iatom))+j+12)
            coeff(locprim(ishell)+j)= coeffp5(ip5(numatomic(iatom))+j+9)
          enddo
          mprim(ishell)= 2
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs5(is5(numatomic(iatom))+15)
          coeff(locprim(ishell)+1)= coeffs5(is5(numatomic(iatom))+15)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs5(is5(numatomic(iatom))+15)
          coeff(locprim(ishell)+1)= coeffp5(ip5(numatomic(iatom))+12)
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
! 3D
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exd5(id5(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffd5(id5(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1)= locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1)= locbf(ishell)+6
          endif
! 4D
          select case(numatomic(iatom))
            case(39:48)
              ishell= ishell+1
              do j= 1,2
                ex(locprim(ishell)+j)= exd5(id5(numatomic(iatom))+j+3)
                coeff(locprim(ishell)+j)= coeffd5(id5(numatomic(iatom))+j+3)
              enddo
              mprim(ishell)= 2
              mtype(ishell)= 2
              locatom(ishell)= iatom
              locprim(ishell+1)= locprim(ishell)+2
              if(spher) then
                mbf(ishell)= 5
                locbf(ishell+1)= locbf(ishell)+5
              else
                mbf(ishell)= 6
                locbf(ishell+1)= locbf(ishell)+6
              endif
!
              ishell= ishell+1
              ex(locprim(ishell)+1)= exd5(id5(numatomic(iatom))+6)
              coeff(locprim(ishell)+1)= coeffd5(id5(numatomic(iatom))+6)
              mprim(ishell)= 1
              mtype(ishell)= 2
              locatom(ishell)= iatom
              locprim(ishell+1)= locprim(ishell)+1
              if(spher) then
                mbf(ishell)= 5
                locbf(ishell+1)= locbf(ishell)+5
              else
                mbf(ishell)= 6
                locbf(ishell+1)= locbf(ishell)+6
              endif
!
            case(49:54)
              ishell= ishell+1
              do j= 1,3
                ex(locprim(ishell)+j)= exd5(id5(numatomic(iatom))+j+3)
                coeff(locprim(ishell)+j)= coeffd5(id5(numatomic(iatom))+j+3)
              enddo
              mprim(ishell)= 3
              mtype(ishell)= 2
              locatom(ishell)= iatom
              locprim(ishell+1)= locprim(ishell)+3
              if(spher) then
                mbf(ishell)= 5
                locbf(ishell+1)= locbf(ishell)+5
              else
                mbf(ishell)= 6
                locbf(ishell+1)= locbf(ishell)+6
              endif
          end select
        case (:0)
        case default
          write(*,'(" Error! This program supports H - Xe 3-21G basis set.")')
          call iabort
      end select
!
      if(ishell > mxshell) then
        write(*,'(" Error! The number of basis shells exceeds mxshell",i6,".")')mxshell
        call iabort
      endif
      if(locprim(ishell+1) > mxprim ) then
        write(*,'(" Error! The number of primitive basis functions exceeds mxprim", &
&             i6,".")')mxprim
        call iabort
      endif
      if(locbf(ishell+1) > mxao ) then
        write(*,'(" Error! The number of basis functions exceeds mxao",i6,".")')mxao
        call iabort
      endif
!
      return
end


!-----------------------------------
  subroutine bs631g(iatom,ishell)
!-----------------------------------
!
! Set basis functions of 6-31G
!
      use modmolecule, only : numatomic
      use modbasis, only : ex, coeff, locprim, locbf, locatom, mprim, mbf, mtype, spher
      use modparam, only : mxao, mxshell, mxprim
      implicit none
      integer,intent(in) :: iatom
      integer,intent(inout) :: ishell
      integer :: j
      integer :: is(30)= (/0,4,8,18,28,38,48,58,68,78,88,104,120,136,152,168,184,200, &
&                          216,238,260,282,304,326,348,370,392,414,436,458/)
      integer :: ip(30)= (/0,0,0,4,8,12,16,20,24,28,32,42,52,62,72,82,92,102, &
&                          112,128,144,160,176,192,208,224,240,256,272,288/)
      integer :: id(21:30)= (/0,4,8,12,16,20,24,28,32,36/)
      real(8) :: ex631g(480)= &
&     (/1.873113696D+01,2.825394365D+00,6.401216923D-01,1.612777588D-01,3.842163400D+01, &
&       5.778030000D+00,1.241774000D+00,2.979640000D-01,6.424189150D+02,9.679851530D+01, &
&       2.209112120D+01,6.201070250D+00,1.935117680D+00,6.367357890D-01,2.324918408D+00, &
&       6.324303556D-01,7.905343475D-02,3.596197175D-02,1.264585690D+03,1.899368060D+02, &
&       4.315908900D+01,1.209866270D+01,3.806323220D+00,1.272890300D+00,3.196463098D+00, &
&       7.478133038D-01,2.199663302D-01,8.230990070D-02,2.068882250D+03,3.106495700D+02, &
&       7.068303300D+01,1.986108030D+01,6.299304840D+00,2.127026970D+00,4.727971071D+00, &
&       1.190337736D+00,3.594116829D-01,1.267512469D-01,3.047524880D+03,4.573695180D+02, &
&       1.039486850D+02,2.921015530D+01,9.286662960D+00,3.163926960D+00,7.868272350D+00, &
&       1.881288540D+00,5.442492580D-01,1.687144782D-01,4.173511460D+03,6.274579110D+02, &
&       1.429020930D+02,4.023432930D+01,1.282021290D+01,4.390437010D+00,1.162636186D+01, &
&       2.716279807D+00,7.722183966D-01,2.120314975D-01,5.484671660D+03,8.252349460D+02, &
&       1.880469580D+02,5.296450000D+01,1.689757040D+01,5.799635340D+00,1.553961625D+01, &
&       3.599933586D+00,1.013761750D+00,2.700058226D-01,7.001713090D+03,1.051366090D+03, &
&       2.392856900D+02,6.739744530D+01,2.151995730D+01,7.403101300D+00,2.084795280D+01, &
&       4.808308340D+00,1.344069860D+00,3.581513930D-01,8.425851530D+03,1.268519400D+03, &
&       2.896214140D+02,8.185900400D+01,2.625150790D+01,9.094720510D+00,2.653213100D+01, &
&       6.101755010D+00,1.696271530D+00,4.458187000D-01,9.993200000D+03,1.499890000D+03, &
&       3.419510000D+02,9.467960000D+01,2.973450000D+01,1.000630000D+01,1.509630000D+02, &
&       3.558780000D+01,1.116830000D+01,3.902010000D+00,1.381770000D+00,4.663820000D-01, &
&       4.979660000D-01,8.435290000D-02,6.663500000D-02,2.595440000D-02,1.172280000D+04, &
&       1.759930000D+03,4.008460000D+02,1.128070000D+02,3.599970000D+01,1.218280000D+01, &
&       1.891800000D+02,4.521190000D+01,1.435630000D+01,5.138860000D+00,1.906520000D+00, &
&       7.058870000D-01,9.293400000D-01,2.690350000D-01,1.173790000D-01,4.210610000D-02, &
&       1.398310000D+04,2.098750000D+03,4.777050000D+02,1.343600000D+02,4.287090000D+01, &
&       1.451890000D+01,2.396680000D+02,5.744190000D+01,1.828590000D+01,6.599140000D+00, &
&       2.490490000D+00,9.445450000D-01,1.277900000D+00,3.975900000D-01,1.600950000D-01, &
&       5.565770000D-02,1.611590000D+04,2.425580000D+03,5.538670000D+02,1.563400000D+02, &
&       5.006830000D+01,1.701780000D+01,2.927180000D+02,6.987310000D+01,2.233630000D+01, &
&       8.150390000D+00,3.134580000D+00,1.225430000D+00,1.727380000D+00,5.729220000D-01, &
&       2.221920000D-01,7.783690000D-02,1.941330000D+04,2.909420000D+03,6.613640000D+02, &
&       1.857590000D+02,5.919430000D+01,2.003100000D+01,3.394780000D+02,8.101010000D+01, &
&       2.587800000D+01,9.452210000D+00,3.665660000D+00,1.467460000D+00,2.156230000D+00, &
&       7.489970000D-01,2.831450000D-01,9.983170000D-02,2.191710000D+04,3.301490000D+03, &
&       7.541460000D+02,2.127110000D+02,6.798960000D+01,2.305150000D+01,4.237350000D+02, &
&       1.007100000D+02,3.215990000D+01,1.180790000D+01,4.631100000D+00,1.870250000D+00, &
&       2.615840000D+00,9.221670000D-01,3.412870000D-01,1.171670000D-01,2.518010000D+04, &
&       3.780350000D+03,8.604740000D+02,2.421450000D+02,7.733490000D+01,2.624700000D+01, &
&       4.917650000D+02,1.169840000D+02,3.741530000D+01,1.378340000D+01,5.452150000D+00, &
&       2.225880000D+00,3.186490000D+00,1.144270000D+00,4.203770000D-01,1.426570000D-01, &
&       2.834830000D+04,4.257620000D+03,9.698570000D+02,2.732630000D+02,8.736950000D+01, &
&       2.968670000D+01,5.758910000D+02,1.368160000D+02,4.380980000D+01,1.620940000D+01, &
&       6.460840000D+00,2.651140000D+00,3.860280000D+00,1.413730000D+00,5.166460000D-01, &
&       1.738880000D-01,3.159442000D+04,4.744330000D+03,1.080419000D+03,3.042338000D+02, &
&       9.724586000D+01,3.302495000D+01,6.227625000D+02,1.478839000D+02,4.732735000D+01, &
&       1.751495000D+01,6.922722000D+00,2.768277000D+00,1.184802000D+01,4.079211000D+00, &
&       1.763481000D+00,7.889270000D-01,3.503870000D-01,1.463440000D-01,7.168010000D-01, &
&       2.337410000D-01,3.867500000D-02,1.652100000D-02,3.526486000D+04,5.295503000D+03, &
&       1.206020000D+03,3.396839000D+02,1.086264000D+02,3.692103000D+01,7.063096000D+02, &
&       1.678187000D+02,5.382558000D+01,2.001638000D+01,7.970279000D+00,3.212059000D+00, &
&       1.419518000D+01,4.880828000D+00,2.160390000D+00,9.878990000D-01,4.495170000D-01, &
&       1.873870000D-01,1.032271000D+00,3.811710000D-01,6.513100000D-02,2.601000000D-02, &
&       3.908898000D+04,5.869792000D+03,1.336910000D+03,3.766031000D+02,1.204679000D+02, &
&       4.098032000D+01,7.862852000D+02,1.868870000D+02,6.000935000D+01,2.225883000D+01, &
&       8.885149000D+00,3.609211000D+00,2.984355000D+01,9.542383000D+00,4.056790000D+00, &
&       1.704703000D+00,7.062340000D-01,2.795360000D-01,1.065609000D+00,4.259330000D-01, &
&       7.632000000D-02,2.959400000D-02,4.315295000D+04,6.479571000D+03,1.475675000D+03, &
&       4.156991000D+02,1.330006000D+02,4.527222000D+01,8.746826000D+02,2.079785000D+02, &
&       6.687918000D+01,2.487347000D+01,9.968441000D+00,4.063826000D+00,3.364363000D+01, &
&       1.087565000D+01,4.628225000D+00,1.950126000D+00,8.094520000D-01,3.204740000D-01, &
&       1.224148000D+00,4.842630000D-01,8.409600000D-02,3.203600000D-02,4.735433000D+04, &
&       7.110787000D+03,1.619591000D+03,4.563379000D+02,1.460606000D+02,4.975791000D+01, &
&       9.681484000D+02,2.302821000D+02,7.414591000D+01,2.764107000D+01,1.111475000D+01, &
&       4.543113000D+00,3.764050000D+01,1.228238000D+01,5.233366000D+00,2.208950000D+00, &
&       9.178800000D-01,3.634120000D-01,1.392781000D+00,5.439130000D-01,9.147600000D-02, &
&       3.431200000D-02,5.178981000D+04,7.776849000D+03,1.771385000D+03,4.991588000D+02, &
&       1.597982000D+02,5.447021000D+01,1.064328000D+03,2.532138000D+02,8.160924000D+01, &
&       3.048193000D+01,1.229439000D+01,5.037722000D+00,4.156291000D+01,1.367627000D+01, &
&       5.844390000D+00,2.471609000D+00,1.028308000D+00,4.072500000D-01,1.571464000D+00, &
&       6.055800000D-01,9.856100000D-02,3.645900000D-02,5.634714000D+04,8.460943000D+03, &
&       1.927325000D+03,5.432343000D+02,1.739905000D+02,5.936005000D+01,1.165412000D+03, &
&       2.773276000D+02,8.947278000D+01,3.348256000D+01,1.354037000D+01,5.557972000D+00, &
&       4.583532000D+01,1.518777000D+01,6.500710000D+00,2.751583000D+00,1.145404000D+00, &
&       4.536870000D-01,1.757999000D+00,6.670220000D-01,1.051290000D-01,3.841800000D-02, &
&       6.113262000D+04,9.179342000D+03,2.090857000D+03,5.892479000D+02,1.887543000D+02, &
&       6.444629000D+01,1.259980000D+03,2.998761000D+02,9.684917000D+01,3.631020000D+01, &
&       1.472996000D+01,6.066075000D+00,5.043485000D+01,1.683929000D+01,7.192086000D+00, &
&       3.053420000D+00,1.273643000D+00,5.040910000D-01,1.950316000D+00,7.367210000D-01, &
&       1.141770000D-01,4.114800000D-02,6.614899000D+04,9.933077000D+03,2.262816000D+03, &
&       6.379154000D+02,2.044122000D+02,6.982538000D+01,1.378841000D+03,3.282694000D+02, &
&       1.060946000D+02,3.983275000D+01,1.618622000D+01,6.667788000D+00,5.452355000D+01, &
&       1.829783000D+01,7.867348000D+00,3.340534000D+00,1.393756000D+00,5.513260000D-01, &
&       2.151947000D+00,8.110630000D-01,1.210170000D-01,4.303700000D-02,7.139635000D+04, &
&       1.072084000D+04,2.442129000D+03,6.884265000D+02,2.206153000D+02,7.539373000D+01, &
&       1.492532000D+03,3.554013000D+02,1.149534000D+02,4.322043000D+01,1.759710000D+01, &
&       7.257765000D+00,5.935261000D+01,2.002181000D+01,8.614561000D+00,3.660531000D+00, &
&       1.528111000D+00,6.040570000D-01,2.379276000D+00,8.858390000D-01,1.285290000D-01, &
&       4.519500000D-02,7.679438000D+04,1.153070000D+04,2.626575000D+03,7.404903000D+02, &
&       2.373528000D+02,8.115818000D+01,1.610814000D+03,3.836367000D+02,1.241733000D+02, &
&       4.674678000D+01,1.906569000D+01,7.871567000D+00,6.445732000D+01,2.185212000D+01, &
&       9.405343000D+00,3.999168000D+00,1.670297000D+00,6.596270000D-01,2.600088000D+00, &
&       9.630940000D-01,1.361610000D-01,4.733200000D-02,8.240094000D+04,1.237255000D+04, &
&       2.818351000D+03,7.945717000D+02,2.547232000D+02,8.713880000D+01,1.732569000D+03, &
&       4.127149000D+02,1.336780000D+02,5.038585000D+01,2.058358000D+01,8.505940000D+00, &
&       6.936492000D+01,2.362082000D+01,1.018471000D+01,4.334082000D+00,1.810918000D+00, &
&       7.148410000D-01,2.823842000D+00,1.039543000D+00,1.432640000D-01,4.929600000D-02/)
      real(8) :: cs631g(480)= &
&     (/ 3.349460434D-02, 2.347269535D-01, 8.137573262D-01, 1.000000000D+00, 2.376600000D-02, &
&        1.546790000D-01, 4.696300000D-01, 1.000000000D+00, 2.142607810D-03, 1.620887150D-02, &
&        7.731557250D-02, 2.457860520D-01, 4.701890040D-01, 3.454708450D-01,-3.509174590D-02, &
&       -1.912328440D-01, 1.083987800D+00, 1.000000000D+00, 1.944757590D-03, 1.483505200D-02, &
&        7.209054630D-02, 2.371541500D-01, 4.691986520D-01, 3.565202280D-01,-1.126487290D-01, &
&       -2.295064090D-01, 1.186916770D+00, 1.000000000D+00, 1.866274590D-03, 1.425148170D-02, &
&        6.955161850D-02, 2.325729330D-01, 4.670787120D-01, 3.634314400D-01,-1.303937970D-01, &
&       -1.307889510D-01, 1.130944480D+00, 1.000000000D+00, 1.834737130D-03, 1.403732280D-02, &
&        6.884262220D-02, 2.321844430D-01, 4.679413480D-01, 3.623119850D-01,-1.193324200D-01, &
&       -1.608541520D-01, 1.143456440D+00, 1.000000000D+00, 1.834772160D-03, 1.399462700D-02, &
&        6.858655180D-02, 2.322408730D-01, 4.690699480D-01, 3.604551990D-01,-1.149611820D-01, &
&       -1.691174790D-01, 1.145851950D+00, 1.000000000D+00, 1.831074430D-03, 1.395017220D-02, &
&        6.844507810D-02, 2.327143360D-01, 4.701928980D-01, 3.585208530D-01,-1.107775490D-01, &
&       -1.480262620D-01, 1.130767010D+00, 1.000000000D+00, 1.819616900D-03, 1.391607960D-02, &
&        6.840532450D-02, 2.331857600D-01, 4.712674390D-01, 3.566185460D-01,-1.085069750D-01, &
&       -1.464516580D-01, 1.128688580D+00, 1.000000000D+00, 1.884348050D-03, 1.433689940D-02, &
&        7.010962330D-02, 2.373732660D-01, 4.730071260D-01, 3.484012410D-01,-1.071182870D-01, &
&       -1.461638210D-01, 1.127773500D+00, 1.000000000D+00, 1.937660000D-03, 1.480700000D-02, &
&        7.270550000D-02, 2.526290000D-01, 4.932420000D-01, 3.131690000D-01,-3.542080000D-03, &
&       -4.395880000D-02,-1.097520000D-01, 1.873980000D-01, 6.466990000D-01, 3.060580000D-01, &
&       -2.485030000D-01,-1.317040000D-01, 1.233520000D+00, 1.000000000D+00, 1.977830000D-03, &
&        1.511400000D-02, 7.391080000D-02, 2.491910000D-01, 4.879280000D-01, 3.196620000D-01, &
&       -3.237170000D-03,-4.100790000D-02,-1.126000000D-01, 1.486330000D-01, 6.164970000D-01, &
&        3.648290000D-01,-2.122900000D-01,-1.079850000D-01, 1.175840000D+00, 1.000000000D+00, &
&        1.942670000D-03, 1.485990000D-02, 7.284940000D-02, 2.468300000D-01, 4.872580000D-01, &
&        3.234960000D-01,-2.926190000D-03,-3.740830000D-02,-1.144870000D-01, 1.156350000D-01, &
&        6.125950000D-01, 3.937990000D-01,-2.276060000D-01, 1.445830000D-03, 1.092790000D+00, &
&        1.000000000D+00, 1.959480000D-03, 1.492880000D-02, 7.284780000D-02, 2.461300000D-01, &
&        4.859140000D-01, 3.250020000D-01,-2.780940000D-03,-3.571460000D-02,-1.149850000D-01, &
&        9.356340000D-02, 6.030170000D-01, 4.189590000D-01,-2.446300000D-01, 4.315720000D-03, &
&        1.098180000D+00, 1.000000000D+00, 1.851600000D-03, 1.420620000D-02, 6.999950000D-02, &
&        2.400790000D-01, 4.847620000D-01, 3.352000000D-01,-2.782170000D-03,-3.604990000D-02, &
&       -1.166310000D-01, 9.683280000D-02, 6.144180000D-01, 4.037980000D-01,-2.529230000D-01, &
&        3.285170000D-02, 1.081250000D+00, 1.000000000D+00, 1.869240000D-03, 1.423030000D-02, &
&        6.969620000D-02, 2.384870000D-01, 4.833070000D-01, 3.380740000D-01,-2.376770000D-03, &
&       -3.169300000D-02,-1.133170000D-01, 5.609000000D-02, 5.922550000D-01, 4.550060000D-01, &
&       -2.503740000D-01, 6.695700000D-02, 1.054510000D+00, 1.000000000D+00, 1.832960000D-03, &
&        1.403420000D-02, 6.909740000D-02, 2.374520000D-01, 4.830340000D-01, 3.398560000D-01, &
&       -2.297390000D-03,-3.071370000D-02,-1.125280000D-01, 4.501630000D-02, 5.893530000D-01, &
&        4.652060000D-01,-2.518270000D-01, 6.158900000D-02, 1.060180000D+00, 1.000000000D+00, &
&        1.825260000D-03, 1.396860000D-02, 6.870730000D-02, 2.362040000D-01, 4.822140000D-01, &
&        3.420430000D-01,-2.159720000D-03,-2.907750000D-02,-1.108270000D-01, 2.769990000D-02, &
&        5.776130000D-01, 4.886880000D-01,-2.555920000D-01, 3.780660000D-02, 1.080560000D+00, &
&        1.000000000D+00, 1.828010000D-03, 1.399403000D-02, 6.887129000D-02, 2.369760000D-01, &
&        4.829040000D-01, 3.404795000D-01,-2.502976000D-03,-3.315550000D-02,-1.226387000D-01, &
&        5.353643000D-02, 6.193860000D-01, 4.345878000D-01, 1.277689000D-02, 2.098767000D-01, &
&       -3.095274000D-03,-5.593884000D-01,-5.134760000D-01,-6.598035000D-02,-5.237772000D-02, &
&       -2.798503000D-01, 1.141547000D+00, 1.000000000D+00, 1.813501000D-03, 1.388493000D-02, &
&        6.836162000D-02, 2.356188000D-01, 4.820639000D-01, 3.429819000D-01, 2.448225000D-03, &
&        3.241504000D-02, 1.226219000D-01,-4.316965000D-02,-6.126995000D-01,-4.487540000D-01, &
&        1.084500000D-02, 2.088333000D-01, 3.150338000D-02,-5.526518000D-01,-5.437997000D-01, &
&       -6.669342000D-02,-4.439720000D-02,-3.284563000D-01, 1.163010000D+00, 1.000000000D+00, &
&        1.803263000D-03, 1.380769000D-02, 6.800396000D-02, 2.347099000D-01, 4.815690000D-01, &
&        3.445652000D-01, 2.451863000D-03, 3.259579000D-02, 1.238242000D-01,-4.359890000D-02, &
&       -6.177181000D-01,-4.432823000D-01,-2.586302000D-03, 7.188424000D-02, 2.503260000D-01, &
&       -2.991003000D-01,-7.446818000D-01,-1.799776000D-01, 6.482978000D-02, 3.253756000D-01, &
&       -1.170806000D+00, 1.000000000D+00, 1.791872000D-03, 1.372392000D-02, 6.762830000D-02, &
&        2.337642000D-01, 4.810696000D-01, 3.462280000D-01, 2.431008000D-03, 3.233027000D-02, &
&        1.242520000D-01,-3.903905000D-02,-6.171789000D-01,-4.473097000D-01,-2.940358000D-03, &
&        7.163103000D-02, 2.528915000D-01,-2.966401000D-01,-7.432215000D-01,-1.853520000D-01, &
&        6.351465000D-02, 3.151404000D-01,-1.162595000D+00, 1.000000000D+00, 1.784513000D-03, &
&        1.366754000D-02, 6.736122000D-02, 2.330552000D-01, 4.806316000D-01, 3.474802000D-01, &
&        2.410599000D-03, 3.207243000D-02, 1.245942000D-01,-3.482177000D-02,-6.167374000D-01, &
&       -4.509844000D-01,-3.233199000D-03, 7.130744000D-02, 2.543820000D-01,-2.933887000D-01, &
&       -7.415695000D-01,-1.909410000D-01, 6.139703000D-02, 3.061130000D-01,-1.154890000D+00, &
&        1.000000000D+00, 1.776182000D-03, 1.360476000D-02, 6.706925000D-02, 2.323104000D-01, &
&        4.802410000D-01, 3.487653000D-01, 2.399669000D-03, 3.194886000D-02, 1.250868000D-01, &
&       -3.221866000D-02,-6.172284000D-01,-4.525936000D-01,-3.454216000D-03, 7.218428000D-02, &
&        2.544820000D-01,-2.934534000D-01,-7.385455000D-01,-1.947157000D-01, 5.892219000D-02, &
&        2.976055000D-01,-1.147506000D+00, 1.000000000D+00, 1.771580000D-03, 1.357081000D-02, &
&        6.690605000D-02, 2.318541000D-01, 4.799046000D-01, 3.495737000D-01, 2.388751000D-03, &
&        3.181708000D-02, 1.254670000D-01,-2.955431000D-02,-6.175160000D-01,-4.544458000D-01, &
&       -3.665856000D-03, 7.231971000D-02, 2.544486000D-01,-2.910380000D-01,-7.359860000D-01, &
&       -1.997617000D-01, 5.628572000D-02, 2.897491000D-01,-1.140653000D+00, 1.000000000D+00, &
&        1.766111000D-03, 1.353038000D-02, 6.673128000D-02, 2.314823000D-01, 4.797058000D-01, &
&        3.501976000D-01, 2.438014000D-03, 3.224048000D-02, 1.265724000D-01,-3.139902000D-02, &
&       -6.207593000D-01,-4.502914000D-01,-3.873256000D-03, 7.196598000D-02, 2.556591000D-01, &
&       -2.882837000D-01,-7.342822000D-01,-2.049353000D-01, 5.694869000D-02, 2.882915000D-01, &
&       -1.138159000D+00, 1.000000000D+00, 1.759787000D-03, 1.348162000D-02, 6.649342000D-02, &
&        2.307939000D-01, 4.792919000D-01, 3.514097000D-01, 2.376276000D-03, 3.167450000D-02, &
&        1.262888000D-01,-2.584552000D-02,-6.183491000D-01,-4.567008000D-01,-3.993004000D-03, &
&        7.409663000D-02, 2.542000000D-01,-2.921657000D-01,-7.318703000D-01,-2.040784000D-01, &
&        5.379843000D-02, 2.759971000D-01,-1.129692000D+00, 1.000000000D+00, 1.753003000D-03, &
&        1.343122000D-02, 6.627041000D-02, 2.302508000D-01, 4.790186000D-01, 3.523444000D-01, &
&        2.370714000D-03, 3.160566000D-02, 1.266335000D-01,-2.417037000D-02,-6.187775000D-01, &
&       -4.576770000D-01,-4.162002000D-03, 7.425111000D-02, 2.541360000D-01,-2.903477000D-01, &
&       -7.302121000D-01,-2.076057000D-01, 5.157888000D-02, 2.707611000D-01,-1.124770000D+00, &
&        1.000000000D+00, 1.748161000D-03, 1.339602000D-02, 6.610885000D-02, 2.298265000D-01, &
&        4.787675000D-01, 3.530739000D-01, 2.364055000D-03, 3.153635000D-02, 1.269452000D-01, &
&       -2.262840000D-02,-6.192080000D-01,-4.585393000D-01,-4.331075000D-03, 7.412307000D-02, &
&        2.542108000D-01,-2.874843000D-01,-7.291436000D-01,-2.113951000D-01, 5.027577000D-02, &
&        2.650040000D-01,-1.120155000D+00, 1.000000000D+00, 1.743329000D-03, 1.335966000D-02, &
&        6.594365000D-02, 2.294151000D-01, 4.785453000D-01, 3.537753000D-01, 2.361459000D-03, &
&        3.150177000D-02, 1.272774000D-01,-2.145928000D-02,-6.197652000D-01,-4.590180000D-01, &
&       -4.440098000D-03, 7.505253000D-02, 2.533111000D-01,-2.881897000D-01,-7.267052000D-01, &
&       -2.133439000D-01, 4.898543000D-02, 2.592793000D-01,-1.115711000D+00, 1.000000000D+00/)
      real(8) :: cp631g(304)= &
&     (/ 8.941508040D-03, 1.410094640D-01, 9.453636950D-01, 1.000000000D+00, 5.598019980D-02, &
&        2.615506110D-01, 7.939723390D-01, 1.000000000D+00, 7.459757990D-02, 3.078466770D-01, &
&        7.434568340D-01, 1.000000000D+00, 6.899906660D-02, 3.164239610D-01, 7.443082910D-01, &
&        1.000000000D+00, 6.757974390D-02, 3.239072960D-01, 7.408951400D-01, 1.000000000D+00, &
&        7.087426820D-02, 3.397528390D-01, 7.271585770D-01, 1.000000000D+00, 7.162872430D-02, &
&        3.459121030D-01, 7.224699570D-01, 1.000000000D+00, 7.190958850D-02, 3.495133720D-01, &
&        7.199405120D-01, 1.000000000D+00, 5.001660000D-03, 3.551090000D-02, 1.428250000D-01, &
&        3.386200000D-01, 4.515790000D-01, 2.732710000D-01,-2.302250000D-02, 9.503590000D-01, &
&        5.985790000D-02, 1.000000000D+00, 4.928130000D-03, 3.498880000D-02, 1.407250000D-01, &
&        3.336420000D-01, 4.449400000D-01, 2.692540000D-01,-2.241920000D-02, 1.922710000D-01, &
&        8.461810000D-01, 1.000000000D+00, 4.602850000D-03, 3.319900000D-02, 1.362820000D-01, &
&        3.304760000D-01, 4.491460000D-01, 2.657040000D-01,-1.751260000D-02, 2.445330000D-01, &
&        8.049340000D-01, 1.000000000D+00, 4.438260000D-03, 3.266790000D-02, 1.347210000D-01, &
&        3.286780000D-01, 4.496400000D-01, 2.613720000D-01,-1.779510000D-02, 2.535390000D-01, &
&        8.006690000D-01, 1.000000000D+00, 4.564620000D-03, 3.369360000D-02, 1.397550000D-01, &
&        3.393620000D-01, 4.509210000D-01, 2.385860000D-01,-1.776530000D-02, 2.740580000D-01, &
&        7.854210000D-01, 1.000000000D+00, 4.061010000D-03, 3.068130000D-02, 1.304520000D-01, &
&        3.272050000D-01, 4.528510000D-01, 2.560420000D-01,-1.451050000D-02, 3.102630000D-01, &
&        7.544830000D-01, 1.000000000D+00, 3.989400000D-03, 3.031770000D-02, 1.298800000D-01, &
&        3.279510000D-01, 4.535270000D-01, 2.521540000D-01,-1.429930000D-02, 3.235720000D-01, &
&        7.435070000D-01, 1.000000000D+00, 3.806650000D-03, 2.923050000D-02, 1.264670000D-01, &
&        3.235100000D-01, 4.548960000D-01, 2.566300000D-01,-1.591970000D-02, 3.246460000D-01, &
&        7.439900000D-01, 1.000000000D+00, 4.094637000D-03, 3.145199000D-02, 1.351558000D-01, &
&        3.390500000D-01, 4.629455000D-01, 2.242638000D-01,-1.221377000D-02,-6.900537000D-03, &
&        2.007466000D-01, 4.281332000D-01, 3.970156000D-01, 1.104718000D-01, 3.164300000D-02, &
&       -4.046160000D-02, 1.012029000D+00, 1.000000000D+00, 4.020371000D-03, 3.100601000D-02, &
&        1.337279000D-01, 3.367983000D-01, 4.631281000D-01, 2.257532000D-01,-1.289621000D-02, &
&       -1.025198000D-02, 1.959781000D-01, 4.357933000D-01, 3.996452000D-01, 9.713636000D-02, &
&       -4.298621000D-01, 6.935829000D-03, 9.705933000D-01, 1.000000000D+00, 4.039530000D-03, &
&        3.122570000D-02, 1.349833000D-01, 3.424793000D-01, 4.623113000D-01, 2.177524000D-01, &
&       -6.096652000D-03,-2.628884000D-02, 5.091001000D-02, 3.798097000D-01, 5.170883000D-01, &
&        1.829772000D-01,-2.938440000D-01, 9.235323000D-02, 9.847930000D-01, 1.000000000D+00, &
&        4.017679000D-03, 3.113966000D-02, 1.349077000D-01, 3.431672000D-01, 4.625760000D-01, &
&        2.154603000D-01,-6.311620000D-03,-2.697638000D-02, 5.316847000D-02, 3.845549000D-01, &
&        5.127662000D-01, 1.811135000D-01,-2.112070000D-01, 7.771998000D-02, 9.898214000D-01, &
&        1.000000000D+00, 3.995005000D-03, 3.104061000D-02, 1.347747000D-01, 3.437279000D-01, &
&        4.628759000D-01, 2.135547000D-01,-6.494056000D-03,-2.753453000D-02, 5.516284000D-02, &
&        3.879672000D-01, 5.090258000D-01, 1.803840000D-01,-1.891265000D-01, 8.005453000D-02, &
&        9.877399000D-01, 1.000000000D+00, 3.986997000D-03, 3.104662000D-02, 1.350518000D-01, &
&        3.448865000D-01, 4.628571000D-01, 2.110426000D-01,-6.722497000D-03,-2.806471000D-02, &
&        5.820028000D-02, 3.916988000D-01, 5.047823000D-01, 1.790290000D-01,-1.930100000D-01, &
&        9.605620000D-02, 9.817609000D-01, 1.000000000D+00, 3.977318000D-03, 3.103112000D-02, &
&        1.351894000D-01, 3.457387000D-01, 4.629205000D-01, 2.090592000D-01,-6.887578000D-03, &
&       -2.846816000D-02, 6.031832000D-02, 3.938961000D-01, 5.013769000D-01, 1.792264000D-01, &
&       -5.035024000D-01, 2.345011000D-01, 9.141257000D-01, 1.000000000D+00, 4.028019000D-03, &
&        3.144647000D-02, 1.368317000D-01, 3.487236000D-01, 4.617931000D-01, 2.043058000D-01, &
&       -7.017128000D-03,-2.877660000D-02, 6.181383000D-02, 3.954946000D-01, 4.989059000D-01, &
&        1.791251000D-01,-4.593796000D-01, 2.852139000D-01, 9.076485000D-01, 1.000000000D+00, &
&        3.971488000D-03, 3.108174000D-02, 1.357439000D-01, 3.476827000D-01, 4.626340000D-01, &
&        2.051632000D-01,-7.290772000D-03,-2.926027000D-02, 6.564150000D-02, 4.000652000D-01, &
&        4.950236000D-01, 1.758240000D-01,-2.165496000D-01, 1.240488000D-01, 9.724064000D-01, &
&        1.000000000D+00, 3.967554000D-03, 3.109479000D-02, 1.359517000D-01, 3.485136000D-01, &
&        4.625498000D-01, 2.035186000D-01,-7.421452000D-03,-2.953410000D-02, 6.731852000D-02, &
&        4.016660000D-01, 4.926623000D-01, 1.756893000D-01,-1.887663000D-01, 1.015199000D-01, &
&        9.790906000D-01, 1.000000000D+00, 3.963307000D-03, 3.110223000D-02, 1.361350000D-01, &
&        3.492914000D-01, 4.624780000D-01, 2.020102000D-01,-7.523725000D-03,-2.975687000D-02, &
&        6.849654000D-02, 4.027141000D-01, 4.908490000D-01, 1.759268000D-01,-1.702911000D-01, &
&        9.310133000D-02, 9.814336000D-01, 1.000000000D+00, 3.963125000D-03, 3.113411000D-02, &
&        1.363931000D-01, 3.501266000D-01, 4.623179000D-01, 2.004995000D-01,-7.689262000D-03, &
&       -2.997982000D-02, 7.082411000D-02, 4.046141000D-01, 4.882325000D-01, 1.751970000D-01, &
&       -1.586763000D-01, 8.379327000D-02, 9.840547000D-01, 1.000000000D+00/)
      real(8) :: exd631g(40)= &
&     (/1.114701000D+01,2.821043000D+00,8.196200000D-01,2.214680000D-01,1.369085000D+01, &
&       3.513154000D+00,1.040434000D+00,2.869620000D-01,1.605025000D+01,4.160063000D+00, &
&       1.243265000D+00,3.442770000D-01,1.841930000D+01,4.812661000D+00,1.446447000D+00, &
&       4.004130000D-01,2.094355000D+01,5.510486000D+00,1.665038000D+00,4.617330000D-01, &
&       2.314994000D+01,6.122368000D+00,1.846601000D+00,5.043610000D-01,2.559306000D+01, &
&       6.800990000D+00,2.051647000D+00,5.556710000D-01,2.819147000D+01,7.523584000D+00, &
&       2.271228000D+00,6.116030000D-01,3.085341000D+01,8.264985000D+00,2.495332000D+00, &
&       6.676580000D-01,3.370764000D+01,9.061106000D+00,2.738383000D+00,7.302940000D-01/)
      real(8) :: cd631g(40)= &
&     (/ 8.747672000D-02, 3.795635000D-01, 7.180393000D-01, 1.000000000D+00, 8.589418000D-02, &
&        3.784671000D-01, 7.161239000D-01, 1.000000000D+00, 8.599899000D-02, 3.802996000D-01, &
&        7.127659000D-01, 1.000000000D+00, 8.650816000D-02, 3.826699000D-01, 7.093772000D-01, &
&        1.000000000D+00, 8.672702000D-02, 3.841883000D-01, 7.069071000D-01, 1.000000000D+00, &
&        8.876935000D-02, 3.896319000D-01, 7.014816000D-01, 1.000000000D+00, 9.004748000D-02, &
&        3.931703000D-01, 6.976844000D-01, 1.000000000D+00, 9.098881000D-02, 3.958208000D-01, &
&        6.947154000D-01, 1.000000000D+00, 9.199905000D-02, 3.985021000D-01, 6.917897000D-01, &
&        1.000000000D+00, 9.262648000D-02, 4.002980000D-01, 6.896608000D-01, 1.000000000D+00/)
!
! Set  functions for H - He
!
      select case(numatomic(iatom))
        case(1:2)
! 1S
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= ex631g(is(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= cs631g(is(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= ex631g(is(numatomic(iatom))+4)
          coeff(locprim(ishell)+1)= cs631g(is(numatomic(iatom))+4)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
! Set  functions for Li - Ne
!
        case(3:10)
! 1S
          ishell= ishell+1
          do j= 1,6
            ex(locprim(ishell)+j)= ex631g(is(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= cs631g(is(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 6
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+6
          locbf(ishell+1) = locbf(ishell)+1
! 2SP
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= ex631g(is(numatomic(iatom))+j+6)
            coeff(locprim(ishell)+j)= cs631g(is(numatomic(iatom))+j+6)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= ex631g(is(numatomic(iatom))+j+6)
            coeff(locprim(ishell)+j)= cp631g(ip(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= ex631g(is(numatomic(iatom))+10)
          coeff(locprim(ishell)+1)= cs631g(is(numatomic(iatom))+10)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1

          ishell= ishell+1
          ex(locprim(ishell)+1)= ex631g(is(numatomic(iatom))+10)
          coeff(locprim(ishell)+1)= cp631g(ip(numatomic(iatom))+4)
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
! Set  functions for Na - Ar
!
        case(11:18)
! 1S
          ishell= ishell+1
          do j= 1,6
            ex(locprim(ishell)+j)= ex631g(is(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= cs631g(is(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 6
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+6
          locbf(ishell+1) = locbf(ishell)+1
! 2SP
          ishell= ishell+1
          do j= 1,6
            ex(locprim(ishell)+j)= ex631g(is(numatomic(iatom))+j+6)
            coeff(locprim(ishell)+j)= cs631g(is(numatomic(iatom))+j+6)
          enddo
          mprim(ishell)= 6
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+6
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,6
            ex(locprim(ishell)+j)= ex631g(is(numatomic(iatom))+j+6)
            coeff(locprim(ishell)+j)= cp631g(ip(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 6
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+6
          locbf(ishell+1) = locbf(ishell)+3
! 3SP
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= ex631g(is(numatomic(iatom))+j+12)
            coeff(locprim(ishell)+j)= cs631g(is(numatomic(iatom))+j+12)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= ex631g(is(numatomic(iatom))+j+12)
            coeff(locprim(ishell)+j)= cp631g(ip(numatomic(iatom))+j+6)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= ex631g(is(numatomic(iatom))+16)
          coeff(locprim(ishell)+1)= cs631g(is(numatomic(iatom))+16)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= ex631g(is(numatomic(iatom))+16)
          coeff(locprim(ishell)+1)= cp631g(ip(numatomic(iatom))+10)
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
! Set  functions for Na - Ar
!
        case(19:30)
! 1S
          ishell= ishell+1
          do j= 1,6
            ex(locprim(ishell)+j)= ex631g(is(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= cs631g(is(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 6
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+6
          locbf(ishell+1) = locbf(ishell)+1
! 2SP  
          ishell= ishell+1
          do j= 1,6
            ex(locprim(ishell)+j)= ex631g(is(numatomic(iatom))+j+6)
            coeff(locprim(ishell)+j)= cs631g(is(numatomic(iatom))+j+6)
          enddo
          mprim(ishell)= 6
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+6
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,6
            ex(locprim(ishell)+j)= ex631g(is(numatomic(iatom))+j+6)
            coeff(locprim(ishell)+j)= cp631g(ip(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 6
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+6
          locbf(ishell+1) = locbf(ishell)+3
! 3SP
          ishell= ishell+1
          do j= 1,6
            ex(locprim(ishell)+j)= ex631g(is(numatomic(iatom))+j+12)
            coeff(locprim(ishell)+j)= cs631g(is(numatomic(iatom))+j+12)
          enddo
          mprim(ishell)= 6
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+6
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,6
            ex(locprim(ishell)+j)= ex631g(is(numatomic(iatom))+j+12)
            coeff(locprim(ishell)+j)= cp631g(ip(numatomic(iatom))+j+6)
          enddo
          mprim(ishell)= 6
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+6
          locbf(ishell+1) = locbf(ishell)+3
! 4SP
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= ex631g(is(numatomic(iatom))+j+18)
            coeff(locprim(ishell)+j)= cs631g(is(numatomic(iatom))+j+18)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= ex631g(is(numatomic(iatom))+j+18)
            coeff(locprim(ishell)+j)= cp631g(ip(numatomic(iatom))+j+12)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= ex631g(is(numatomic(iatom))+22)
          coeff(locprim(ishell)+1)= cs631g(is(numatomic(iatom))+22)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= ex631g(is(numatomic(iatom))+22)
          coeff(locprim(ishell)+1)= cp631g(ip(numatomic(iatom))+16)
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
! 3D
          if(numatomic(iatom) >= 21) then
            ishell= ishell+1
            do j= 1,3
              ex(locprim(ishell)+j)= exd631g(id(numatomic(iatom))+j)
              coeff(locprim(ishell)+j)= cd631g(id(numatomic(iatom))+j)
            enddo
            mprim(ishell)= 3
            mtype(ishell)= 2
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+3
            if(spher) then
              mbf(ishell)= 5
              locbf(ishell+1) = locbf(ishell)+5
            else
              mbf(ishell)= 6
              locbf(ishell+1) = locbf(ishell)+6
            endif
!
            ishell= ishell+1
            ex(locprim(ishell)+1)= exd631g(id(numatomic(iatom))+4)
            coeff(locprim(ishell)+1)= cd631g(id(numatomic(iatom))+4)
            mprim(ishell)= 1
            mtype(ishell)= 2
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+1
            if(spher) then
              mbf(ishell)= 5
              locbf(ishell+1) = locbf(ishell)+5
            else
              mbf(ishell)= 6
              locbf(ishell+1) = locbf(ishell)+6
            endif
          endif
        case (:0)
        case default
          write(*,'(" Error! This program supports H - Zn 6-31G basis set.")')
          call iabort
      end select
!
      if(ishell > mxshell) then
        write(*,'(" Error! The number of basis shells exceeds mxshell",i6,".")')mxshell
        call iabort
      endif
      if(locprim(ishell+1) > mxprim ) then
        write(*,'(" Error! The number of primitive basis functions exceeds mxprim", &
&             i6,".")')mxprim
        call iabort
      endif
      if(locbf(ishell+1) > mxao ) then
        write(*,'(" Error! The number of basis functions exceeds mxao",i6,".")')mxao
        call iabort
      endif
      return
end


!-----------------------------------
  subroutine bs631gd(iatom,ishell)
!-----------------------------------
!
! Set basis functions of 6-31G(d)
!
      implicit none
      integer,intent(in) :: iatom
      integer,intent(inout) :: ishell
!
      call bs631g(iatom,ishell)
!
      call bs631gs(iatom,ishell)
!
      return
end


!------------------------------------
  subroutine bs631gdp(iatom,ishell)
!------------------------------------
!
! Set basis functions of 6-31G(d,p)
!
      use modmolecule, only : numatomic
      implicit none
      integer,intent(in) :: iatom
      integer,intent(inout) :: ishell
!
      call bs631g(iatom,ishell)
!
      select case(numatomic(iatom))
        case(1:2)
          call bs631gss(iatom,ishell)
        case(3:)
          call bs631gs(iatom,ishell)
      end select
!
      return
end


!-----------------------------------
  subroutine bs631gs(iatom,ishell)
!-----------------------------------
!
! Set basis polarization function of 6-31G(d)
!
      use modmolecule, only : numatomic
      use modbasis, only : ex, coeff, locprim, locbf, locatom, mprim, mtype, mbf, spher
      use modparam, only : mxao, mxshell, mxprim
      implicit none
      integer,intent(in) :: iatom
      integer,intent(inout) :: ishell
      real(8),parameter :: one=1.0D+00
      real(8) :: ex631gs(3:30)= &
&     (/2.00D-01,4.00D-01,6.00D-01,8.00D-01,8.00D-01,8.00D-01,8.00D-01,8.00D-01, &
&       1.75D-01,1.75D-01,3.25D-01,4.50D-01,5.50D-01,6.50D-01,7.50D-01,8.50D-01, &
&       2.00D-01,2.00D-01,8.00D-01,8.00D-01,8.00D-01,8.00D-01,8.00D-01,8.00D-01, &
&       8.00D-01,8.00D-01,8.00D-01,8.00D-01/)
!
      select case (numatomic(iatom))
        case(:2)
        case(3:20)
          ishell= ishell+1
          ex(locprim(ishell)+1)= ex631gs(numatomic(iatom))
          coeff(locprim(ishell)+1)= one
          mprim(ishell)= 1
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1)= locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1)= locbf(ishell)+6
          endif
        case(21:30)
          ishell= ishell+1
          ex(locprim(ishell)+1)= ex631gs(numatomic(iatom))
          coeff(locprim(ishell)+1)= one
          mprim(ishell)= 1
          mtype(ishell)= 3
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 7
            locbf(ishell+1)= locbf(ishell)+7
          else
            mbf(ishell)= 10
            locbf(ishell+1)= locbf(ishell)+10
          endif
        case default
          write(*,'(" Error! This program supports Li - Zn 6-31G* basis set.")')
          call iabort
      end select
!
      if(ishell > mxshell) then
        write(*,'(" Error! The number of basis shells exceeds mxshell",i6,".")')mxshell
        call iabort
      endif
      if(locprim(ishell+1) > mxprim ) then
        write(*,'(" Error! The number of primitive basis functions exceeds mxprim", &
&             i6,".")')mxprim
        call iabort
      endif
      if(locbf(ishell+1) > mxao ) then
        write(*,'(" Error! The number of basis functions exceeds mxao",i6,".")')mxao
        call iabort
      endif
      return
end


!------------------------------------
  subroutine bs631gss(iatom,ishell)
!------------------------------------
!
! Set basis polarization function of Hydrogen 6-31G(d,p)
!
      use modmolecule, only : numatomic
      use modbasis, only : ex, coeff, locprim, locbf, locatom, mprim, mtype, mbf
      use modparam, only : mxao, mxshell, mxprim
      implicit none
      integer,intent(in) :: iatom
      integer,intent(inout) :: ishell
      real(8),parameter :: one=1.0D+00, ex631gss=1.1D+00 
!
      select case (numatomic(iatom))
        case(1:2)
          ishell= ishell+1
          ex(locprim(ishell)+1)= ex631gss
          coeff(locprim(ishell)+1)= one
          mprim(ishell)= 1
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          mbf(ishell)= 3
          locbf(ishell+1) = locbf(ishell)+3
      end select
!
      if(ishell > mxshell) then
        write(*,'(" Error! The number of basis shells exceeds mxshell",i6,".")')mxshell
        call iabort
      endif
      if(locprim(ishell+1) > mxprim ) then
        write(*,'(" Error! The number of primitive basis functions exceeds mxprim", &
&             i6,".")')mxprim
        call iabort
      endif
      if(locbf(ishell+1) > mxao ) then
        write(*,'(" Error! The number of basis functions exceeds mxao",i6,".")')mxao
        call iabort
      endif
      return
end


!-----------------------------------
  subroutine bs6311g(iatom,ishell)
!-----------------------------------
!
! Set basis functions of 6-311G
!
      use modmolecule, only : numatomic
      use modbasis, only : ex, coeff, locprim, locbf, locatom, mprim, mbf, mtype, spher
      use modparam, only : mxao, mxshell, mxprim
      implicit none
      integer,intent(in) :: iatom
      integer,intent(inout) :: ishell
      integer :: j, num
      integer :: is1(2), is2(3:10), ip2(3:10), is3(11:18), ip3(11:18)
      integer :: is4(19:36), ip4(19:36),id4(19:36)
      data is1( 1: 2) /0,5/
      data is2( 3:10) /0,11,22,33,44,55,66,77/
      data ip2( 3:10) /0, 5,10,15,20,25,30,35/
      data is3(11:18) /0,13,26,39,52,65,78,91/
      data ip3(11:18) /0, 9,18,27,36,45,54,64/
      data is4(19:20) /0,14/
      data ip4(19:20) /0,11/
      data id4(19:20) /0, 3/
      data is4(31:36) /28,43,58,73,88,103/
      data ip4(31:36) /22,35,48,61,74,87/
      data id4(31:36) / 6,11,16,21,26,31/
      real(8) :: exs1(10)= (/ &
&       3.386500000D+01,5.094790000D+00,1.158790000D+00,3.258400000D-01,1.027410000D-01, &
&       9.812430000D+01,1.476890000D+01,3.318830000D+00,8.740470000D-01,2.445640000D-01/)
      real(8) :: coeffs1(10)= (/ &
&        2.549380000D-02, 1.903730000D-01, 8.521610000D-01, 1.000000000D+00, 1.000000000D+00, &
&        2.874520000D-02, 2.080610000D-01, 8.376350000D-01, 1.000000000D+00, 1.000000000D+00/)
      real(8) :: exs2(88)= (/ &
&       9.004600000D+02,1.344330000D+02,3.043650000D+01,8.626390000D+00,2.483320000D+00, &
&       3.031790000D-01,4.868900000D+00,8.569240000D-01,2.432270000D-01,6.350700000D-02, &
&       2.436830000D-02,1.682800000D+03,2.517150000D+02,5.741160000D+01,1.651710000D+01, &
&       4.853640000D+00,6.268630000D-01,8.309380000D+00,1.740750000D+00,4.858160000D-01, &
&       1.636130000D-01,5.672850000D-02,2.858890000D+03,4.281400000D+02,9.752820000D+01, &
&       2.796930000D+01,8.215770000D+00,1.112780000D+00,1.324150000D+01,3.001660000D+00, &
&       9.128560000D-01,3.154540000D-01,9.885630000D-02,4.563240000D+03,6.820240000D+02, &
&       1.549730000D+02,4.445530000D+01,1.302900000D+01,1.827730000D+00,2.096420000D+01, &
&       4.803310000D+00,1.459330000D+00,4.834560000D-01,1.455850000D-01,6.293480000D+03, &
&       9.490440000D+02,2.187760000D+02,6.369160000D+01,1.882820000D+01,2.720230000D+00, &
&       3.063310000D+01,7.026140000D+00,2.112050000D+00,6.840090000D-01,2.008780000D-01, &
&       8.588500000D+03,1.297230000D+03,2.992960000D+02,8.737710000D+01,2.567890000D+01, &
&       3.740040000D+00,4.211750000D+01,9.628370000D+00,2.853320000D+00,9.056610000D-01, &
&       2.556110000D-01,1.142710000D+04,1.722350000D+03,3.957460000D+02,1.151390000D+02, &
&       3.360260000D+01,4.919010000D+00,5.544410000D+01,1.263230000D+01,3.717560000D+00, &
&       1.165450000D+00,3.218920000D-01,1.399570000D+04,2.117100000D+03,4.904250000D+02, &
&       1.438330000D+02,4.192650000D+01,6.156840000D+00,6.912110000D+01,1.583500000D+01, &
&       4.673260000D+00,1.457560000D+00,3.970570000D-01/)
      real(8) :: coeffs2(88)= (/ &
&        2.287040000D-03, 1.763500000D-02, 8.734340000D-02, 2.809770000D-01, 6.587410000D-01, &
&        1.187120000D-01, 9.332930000D-02, 9.430450000D-01,-2.798270000D-03, 1.000000000D+00, &
&        1.000000000D+00, 2.285740000D-03, 1.759380000D-02, 8.633150000D-02, 2.818350000D-01, &
&        6.405940000D-01, 1.444670000D-01, 1.086210000D-01, 9.273010000D-01,-2.971690000D-03, &
&        1.000000000D+00, 1.000000000D+00, 2.153750000D-03, 1.658230000D-02, 8.218700000D-02, &
&        2.766180000D-01, 6.293160000D-01, 1.737700000D-01, 1.174430000D-01, 9.180020000D-01, &
&       -2.651050000D-03, 1.000000000D+00, 1.000000000D+00, 1.966650000D-03, 1.523060000D-02, &
&        7.612690000D-02, 2.608010000D-01, 6.164620000D-01, 2.210060000D-01, 1.146600000D-01, &
&        9.199990000D-01,-3.030680000D-03, 1.000000000D+00, 1.000000000D+00, 1.969790000D-03, &
&        1.496130000D-02, 7.350060000D-02, 2.489370000D-01, 6.024600000D-01, 2.562020000D-01, &
&        1.119060000D-01, 9.216660000D-01,-2.569190000D-03, 1.000000000D+00, 1.000000000D+00, &
&        1.895150000D-03, 1.438590000D-02, 7.073200000D-02, 2.400010000D-01, 5.947970000D-01, &
&        2.808020000D-01, 1.138890000D-01, 9.208110000D-01,-3.274470000D-03, 1.000000000D+00, &
&        1.000000000D+00, 1.800930000D-03, 1.374190000D-02, 6.813340000D-02, 2.333250000D-01, &
&        5.890860000D-01, 2.995050000D-01, 1.145360000D-01, 9.205120000D-01,-3.378040000D-03, &
&        1.000000000D+00, 1.000000000D+00, 1.832760000D-03, 1.388270000D-02, 6.806870000D-02, &
&        2.313280000D-01, 5.858900000D-01, 3.058830000D-01, 1.191490000D-01, 9.173750000D-01, &
&       -4.058390000D-03, 1.000000000D+00, 1.000000000D+00/)
      real(8) :: coeffp2(40)= (/ &
&        3.276610000D-02, 1.597920000D-01, 8.856670000D-01, 1.000000000D+00, 1.000000000D+00, &
&        3.613440000D-02, 2.169580000D-01, 8.418390000D-01, 1.000000000D+00, 1.000000000D+00, &
&        4.181000000D-02, 2.365750000D-01, 8.162140000D-01, 1.000000000D+00, 1.000000000D+00, &
&        4.024870000D-02, 2.375940000D-01, 8.158540000D-01, 1.000000000D+00, 1.000000000D+00, &
&        3.831190000D-02, 2.374030000D-01, 8.175920000D-01, 1.000000000D+00, 1.000000000D+00, &
&        3.651140000D-02, 2.371530000D-01, 8.197020000D-01, 1.000000000D+00, 1.000000000D+00, &
&        3.546090000D-02, 2.374510000D-01, 8.204580000D-01, 1.000000000D+00, 1.000000000D+00, &
&        3.565740000D-02, 2.394770000D-01, 8.184610000D-01, 1.000000000D+00, 1.000000000D+00/)
      real(8) :: exs3(104)= (/ &
&       3.616640000D+04,5.372580000D+03,1.213210000D+03,3.396230000D+02,1.095530000D+02, &
&       3.877730000D+01,3.877730000D+01,1.457590000D+01,5.269930000D+00,1.827770000D+00, &
&       6.199480000D-01,5.724000000D-02,2.404800000D-02,4.386650000D+04,6.605370000D+03, &
&       1.513260000D+03,4.323170000D+02,1.421490000D+02,5.139830000D+01,5.139830000D+01, &
&       1.991960000D+01,8.024740000D+00,2.508170000D+00,8.715310000D-01,1.081880000D-01, &
&       4.013000000D-02,5.486648900D+04,8.211766500D+03,1.866176100D+03,5.311293400D+02, &
&       1.751179700D+02,6.400550000D+01,6.400550000D+01,2.529250700D+01,1.053491000D+01, &
&       3.206711000D+00,1.152555000D+00,1.766780000D-01,6.523700000D-02,6.937923000D+04, &
&       1.035494000D+04,2.333879600D+03,6.571429500D+02,2.143011300D+02,7.762916800D+01, &
&       7.762916800D+01,3.063080700D+01,1.280129500D+01,3.926866000D+00,1.452343000D+00, &
&       2.562340000D-01,9.427900000D-02,7.749240000D+04,1.160580000D+04,2.645960000D+03, &
&       7.549760000D+02,2.487550000D+02,9.115650000D+01,9.115650000D+01,3.622570000D+01, &
&       1.521130000D+01,4.794170000D+00,1.807930000D+00,3.568160000D-01,1.147830000D-01, &
&       9.341340000D+04,1.396170000D+04,3.169910000D+03,9.024560000D+02,2.971580000D+02, &
&       1.087020000D+02,1.087020000D+02,4.315530000D+01,1.810790000D+01,5.560090000D+00, &
&       2.131830000D+00,4.204030000D-01,1.360450000D-01,1.058190000D+05,1.587200000D+04, &
&       3.619650000D+03,1.030800000D+03,3.399080000D+02,1.245380000D+02,1.245380000D+02, &
&       4.951350000D+01,2.080560000D+01,6.583460000D+00,2.564680000D+00,5.597630000D-01, &
&       1.832730000D-01,1.180223800D+05,1.768354100D+04,4.027765700D+03,1.145397700D+03, &
&       3.771637500D+02,1.381596900D+02,1.381596900D+02,5.498911700D+01,2.317066700D+01, &
&       7.377860000D+00,2.923688000D+00,6.504050000D-01,2.328250000D-01/)
      real(8) :: coeffs3(104)= (/ &
&        1.032000000D-03, 8.071000000D-03, 4.212900000D-02, 1.697890000D-01, 5.146210000D-01, &
&        3.798170000D-01, 3.747620000D-01, 5.757690000D-01, 1.129330000D-01, 1.000000000D+00, &
&        1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 9.180000000D-04, 7.047000000D-03, &
&        3.594100000D-02, 1.414610000D-01, 4.267640000D-01, 4.979750000D-01, 2.513550000D-01, &
&        6.186710000D-01, 1.884170000D-01, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, &
&        1.000000000D+00, 8.390000000D-04, 6.527000000D-03, 3.366600000D-02, 1.329020000D-01, &
&        4.012660000D-01, 5.313380000D-01, 2.023050000D-01, 6.247900000D-01, 2.274390000D-01, &
&        1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 7.570000000D-04, &
&        5.932000000D-03, 3.108800000D-02, 1.249670000D-01, 3.868970000D-01, 5.548880000D-01, &
&        1.778810000D-01, 6.277650000D-01, 2.476230000D-01, 1.000000000D+00, 1.000000000D+00, &
&        1.000000000D+00, 1.000000000D+00, 7.810000000D-04, 6.068000000D-03, 3.116000000D-02, &
&        1.234310000D-01, 3.782090000D-01, 5.632620000D-01, 1.602550000D-01, 6.276470000D-01, &
&        2.638490000D-01, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, &
&        7.430000000D-04, 5.793000000D-03, 2.995400000D-02, 1.190280000D-01, 3.684320000D-01, &
&        5.772990000D-01, 1.431860000D-01, 6.244650000D-01, 2.833660000D-01, 1.000000000D+00, &
&        1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 7.380000000D-04, 5.718000000D-03, &
&        2.949500000D-02, 1.172860000D-01, 3.629490000D-01, 5.841490000D-01, 1.341770000D-01, &
&        6.242500000D-01, 2.917560000D-01, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, &
&        1.000000000D+00, 7.470000000D-04, 5.790000000D-03, 2.991900000D-02, 1.192060000D-01, &
&        3.690280000D-01, 5.764590000D-01, 1.439270000D-01, 6.229380000D-01, 2.839640000D-01, &
&        1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00/)
      real(8) :: exp3(74)= (/ &
&       1.446450000D+02,3.390740000D+01,1.062850000D+01,3.823890000D+00,1.444290000D+00, &
&       5.526210000D-01,1.887200000D-01,4.650100000D-02,1.628500000D-02,1.938540000D+02, &
&       4.544200000D+01,1.418640000D+01,5.057510000D+00,1.888610000D+00,7.226520000D-01, &
&       2.364170000D-01,9.335800000D-02,3.480900000D-02,2.592836200D+02,6.107687000D+01, &
&       1.930323700D+01,7.010882000D+00,2.673865000D+00,1.036596000D+00,3.168190000D-01, &
&       1.142570000D-01,4.139700000D-02,3.354831900D+02,7.890036600D+01,2.498815000D+01, &
&       9.219711000D+00,3.621140000D+00,1.451310000D+00,5.049770000D-01,1.863170000D-01, &
&       6.543200000D-02,3.848430000D+02,9.055210000D+01,2.913390000D+01,1.088620000D+01, &
&       4.352590000D+00,1.777060000D+00,6.970050000D-01,2.535320000D-01,6.849300000D-02, &
&       4.950400000D+02,1.172210000D+02,3.777490000D+01,1.405840000D+01,5.565740000D+00, &
&       2.262970000D+00,8.079940000D-01,2.774600000D-01,7.714100000D-02,5.897760000D+02, &
&       1.398490000D+02,4.514130000D+01,1.687330000D+01,6.741100000D+00,6.741100000D+00, &
&       2.771520000D+00,1.023870000D+00,3.813680000D-01,1.094370000D-01,6.630620100D+02, &
&       1.570928100D+02,5.023110000D+01,1.863534500D+01,7.446537000D+00,7.446537000D+00, &
&       3.095698000D+00,1.106463000D+00,4.156010000D-01,1.454490000D-01/)
      real(8) :: coeffp3(74)= (/ &
&        1.148500000D-02, 8.238300000D-02, 3.196580000D-01, 7.012950000D-01, 6.385060000D-01, &
&        4.253650000D-01, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 1.018800000D-02, &
&        7.536000000D-02, 3.074190000D-01, 7.175750000D-01, 6.673390000D-01, 3.946490000D-01, &
&        1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 9.448000000D-03, 7.097400000D-02, &
&        2.956360000D-01, 7.282190000D-01, 6.444670000D-01, 4.174130000D-01, 1.000000000D+00, &
&        1.000000000D+00, 1.000000000D+00, 8.866000000D-03, 6.829900000D-02, 2.909580000D-01, &
&        7.321170000D-01, 6.198790000D-01, 4.391480000D-01, 1.000000000D+00, 1.000000000D+00, &
&        1.000000000D+00, 9.206000000D-03, 6.987400000D-02, 2.924700000D-01, 7.281030000D-01, &
&        6.283490000D-01, 4.280440000D-01, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, &
&        8.309000000D-03, 6.402400000D-02, 2.776140000D-01, 7.450760000D-01, 6.137120000D-01, &
&        4.438180000D-01, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 2.391000000D-03, &
&        1.850400000D-02, 8.137700000D-02, 2.215520000D-01, 7.725690000D-01,-1.572244000D+00, &
&        9.923890000D-01, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 3.082000000D-03, &
&        2.416500000D-02, 1.082230000D-01, 2.941920000D-01, 6.878620000D-01,-1.214482000D-01, &
&        1.632370000D-01, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00/)
      real(8) :: exs4(118)= (/ &
&       1.825940000D+05,2.736900000D+04,6.229170000D+03,1.764580000D+03,5.770510000D+02, &
&       2.102490000D+02,8.261780000D+01,3.323320000D+01,8.106490000D+00,3.334030000D+00, &
&       8.455440000D-01,3.282160000D-01,3.640350000D-02,1.764630000D-02,2.026990000D+05, &
&       3.038250000D+04,6.915080000D+03,1.959020000D+03,6.409360000D+02,2.339770000D+02, &
&       9.228920000D+01,3.725450000D+01,9.131980000D+00,3.817790000D+00,1.049350000D+00, &
&       4.286600000D-01,6.282260000D-02,2.601620000D-02,3.338000000D+05,5.010000000D+04, &
&       1.151000000D+04,3.292000000D+03,1.089000000D+03,4.010000000D+02,4.010000000D+02, &
&       1.593000000D+02,6.759000000D+01,2.489000000D+01,1.068000000D+01,3.386000000D+00, &
&       1.331000000D+00,1.853000000D-01,6.621000000D-02,3.575000000D+05,5.367000000D+04, &
&       1.230000000D+04,3.512000000D+03,1.161000000D+03,4.280000000D+02,4.280000000D+02, &
&       1.700000000D+02,7.206000000D+01,2.669000000D+01,1.150000000D+01,3.742000000D+00, &
&       1.499000000D+00,2.292000000D-01,8.675000000D-02,3.812000000D+05,5.724000000D+04, &
&       1.311000000D+04,3.743000000D+03,1.238000000D+03,4.563000000D+02,4.563000000D+02, &
&       1.814000000D+02,7.707000000D+01,2.857000000D+01,1.236000000D+01,4.117000000D+00, &
&       1.678000000D+00,2.761000000D-01,1.105000000D-01,4.054000000D+05,6.085000000D+04, &
&       1.391000000D+04,3.989000000D+03,1.324000000D+03,4.870000000D+02,4.870000000D+02, &
&       1.932000000D+02,8.219000000D+01,3.052000000D+01,1.325000000D+01,4.510000000D+00, &
&       1.867000000D+00,3.190000000D-01,1.120000000D-01,4.397000000D+05,6.603000000D+04, &
&       1.514000000D+04,4.317000000D+03,1.414000000D+03,5.239000000D+02,5.239000000D+02, &
&       2.077000000D+02,8.654000000D+01,3.052000000D+01,1.298000000D+01,4.412000000D+00, &
&       1.862000000D+00,3.932000000D-01,1.400000000D-01,4.562000000D+05,6.846000000D+04, &
&       1.565000000D+04,4.500000000D+03,1.497000000D+03,5.505000000D+02,5.505000000D+02, &
&       2.182000000D+02,9.297000000D+01,3.459000000D+01,1.512000000D+01,5.346000000D+00, &
&       2.273000000D+00,4.841000000D-01,1.849000000D-01/)
      real(8) :: coeffs4(118)= (/ &
&        2.277470000D-04, 1.766400000D-03, 9.194970000D-03, 3.745510000D-02, 1.220450000D-01, &
&        2.989900000D-01, 4.051470000D-01, 2.925320000D-01, 1.000000000D+00, 1.000000000D+00, &
&        1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 2.229640000D-04, &
&        1.729320000D-03, 9.002260000D-03, 3.666990000D-02, 1.194100000D-01, 2.918250000D-01, &
&        4.044150000D-01, 2.963130000D-01, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, &
&        1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 8.380500000D-04, 6.246600000D-03, &
&        3.202800000D-02, 1.270900000D-01, 3.909200000D-01, 5.464300000D-01, 1.807800000D-01, &
&        6.223900000D-01, 2.495300000D-01, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, &
&        1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 8.389800000D-04, 6.263500000D-03, &
&        3.203600000D-02, 1.275100000D-01, 3.916500000D-01, 5.452800000D-01, 1.816000000D-01, &
&        6.224800000D-01, 2.487200000D-01, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, &
&        1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 8.332700000D-04, 6.219800000D-03, &
&        3.203000000D-02, 1.272900000D-01, 3.913200000D-01, 5.458100000D-01, 1.809700000D-01, &
&        6.218000000D-01, 2.498800000D-01, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, &
&        1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 8.310000000D-04, 6.233100000D-03, &
&        3.207100000D-02, 1.263600000D-01, 3.889900000D-01, 5.488800000D-01, 1.796300000D-01, &
&        6.222700000D-01, 2.506700000D-01, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, &
&        1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 8.130000000D-04, 6.285000000D-03, &
&        3.192000000D-02, 1.288000000D-01, 3.946000000D-01, 5.413000000D-01, 1.831000000D-01, &
&        6.176000000D-01, 2.538000000D-01, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, &
&        1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 8.111000000D-04, 6.255200000D-03, &
&        3.202200000D-02, 1.254600000D-01, 3.866600000D-01, 5.518800000D-01, 1.771100000D-01, &
&        6.224800000D-01, 2.527900000D-01, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, &
&        1.000000000D+00, 1.000000000D+00, 1.000000000D+00/)
      real(8) :: exp4(100)= (/ &
&       8.910540000D+02,2.110160000D+02,6.767140000D+01,2.527150000D+01,1.013900000D+01, &
&       4.201860000D+00,1.625070000D+00,6.437700000D-01,2.461300000D-01,4.544000000D-02, &
&       1.616000000D-02,1.019760000D+03,2.415960000D+02,7.763700000D+01,2.911540000D+01, &
&       1.176260000D+01,4.922890000D+00,1.906450000D+00,7.369000000D-01,2.764200000D-01, &
&       6.027000000D-02,1.791000000D-02,2.181000000D+03,5.154000000D+02,1.648000000D+02, &
&       6.140000000D+01,2.498000000D+01,1.044000000D+01,1.044000000D+01,5.589000000D+00, &
&       2.517000000D+00,1.053000000D+00,2.915000000D-01,1.210000000D-01,3.990000000D-02, &
&       2.345000000D+03,5.542000000D+02,1.773000000D+02,6.613000000D+01,2.690000000D+01, &
&       1.126000000D+01,1.126000000D+01,6.116000000D+00,2.819000000D+00,1.211000000D+00, &
&       3.568000000D-01,1.621000000D-01,6.084000000D-02,2.528000000D+03,5.973000000D+02, &
&       1.907000000D+02,7.096000000D+01,2.889000000D+01,1.213000000D+01,1.213000000D+01, &
&       6.533000000D+00,3.028000000D+00,1.344000000D+00,5.823000000D-01,2.558000000D-01, &
&       8.651000000D-02,2.706000000D+03,6.386000000D+02,2.038000000D+02,7.596000000D+01, &
&       3.102000000D+01,1.305000000D+01,1.305000000D+01,6.986000000D+00,3.234000000D+00, &
&       1.475000000D+00,7.275000000D-01,2.869000000D-01,9.679000000D-02,2.957000000D+03, &
&       7.003000000D+02,2.246000000D+02,8.259000000D+01,3.319000000D+01,1.420000000D+01, &
&       1.420000000D+01,7.438000000D+00,3.526000000D+00,1.595000000D+00,8.462000000D-01, &
&       3.186000000D-01,1.096000000D-01,3.074000000D+03,7.257000000D+02,2.315000000D+02, &
&       8.645000000D+01,3.555000000D+01,1.514000000D+01,1.514000000D+01,8.031000000D+00, &
&       3.799000000D+00,1.849000000D+00,8.676000000D-01,3.329000000D-01,1.199000000D-01/)
      real(8) :: coeffp4(100)= (/ &
&        2.184290000D-03, 1.758910000D-02, 8.177750000D-02, 2.456560000D-01, 4.339840000D-01, &
&        3.623770000D-01, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, &
&        1.000000000D+00, 2.059860000D-03, 1.665010000D-02, 7.776460000D-02, 2.418060000D-01, &
&        4.325780000D-01, 3.673250000D-01, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, &
&        1.000000000D+00, 1.000000000D+00, 2.256500000D-02, 1.836300000D-01, 8.598400000D-01, &
&        3.434500000D-01, 5.056100000D-01, 2.622400000D-01, 6.413000000D-02, 3.788400000D-01, &
&        6.166900000D-01, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, &
&        2.251400000D-02, 1.833500000D-01, 8.600300000D-01, 3.430600000D-01, 5.065200000D-01, &
&        2.614100000D-01, 6.724600000D-02, 3.723800000D-01, 6.176300000D-01, 1.000000000D+00, &
&        1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 2.227100000D-02, 1.819900000D-01, &
&        8.613300000D-01, 3.433800000D-01, 5.059800000D-01, 2.612800000D-01, 6.853600000D-02, &
&        3.828200000D-01, 6.059900000D-01, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, &
&        1.000000000D+00, 2.214600000D-02, 1.818000000D-01, 8.615500000D-01, 3.424300000D-01, &
&        5.054100000D-01, 2.623700000D-01, 7.016300000D-02, 3.841900000D-01, 6.036800000D-01, &
&        1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 2.226000000D-02, &
&        1.802000000D-01, 8.624000000D-01, 3.440000000D-01, 5.071000000D-01, 2.590000000D-01, &
&        7.965000000D-02, 3.734000000D-01, 6.049000000D-01, 1.000000000D+00, 1.000000000D+00, &
&        1.000000000D+00, 1.000000000D+00, 2.199800000D-02, 1.810200000D-01, 8.622300000D-01, &
&        3.415500000D-01, 5.025500000D-01, 2.645300000D-01, 7.167000000D-02, 3.970800000D-01, &
&        5.876200000D-01, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00, 1.000000000D+00/)
      real(8) :: exd4(36)= (/ &
&       1.337000000D+01,3.421000000D+00,1.063000000D+00,1.508000000D+01,3.926000000D+00, &
&       1.233000000D+00,6.714000000D+01,1.894000000D+01,6.426000000D+00,2.190000000D+00, &
&       6.725000000D-01,7.484000000D+01,2.123000000D+01,7.297000000D+00,2.549000000D+00, &
&       8.165000000D-01,8.605000000D+01,2.449000000D+01,8.442000000D+00,2.981000000D+00, &
&       9.790000000D-01,9.901000000D+01,2.841000000D+01,9.863000000D+00,3.514000000D+00, &
&       1.171000000D+00,1.348000000D+02,3.639000000D+01,1.216000000D+01,4.341000000D+00, &
&       1.535000000D+00,1.225000000D+02,3.537000000D+01,1.239000000D+01,4.496000000D+00, &
&       1.546000000D+00/)
      real(8) :: coeffd4(36)= (/ &
&        3.160160000D-02, 1.568790000D-01, 3.905820000D-01, 3.689470000D-02, 1.778200000D-01, &
&        4.255130000D-01, 3.095700000D-02, 1.748000000D-01, 4.434600000D-01, 5.670200000D-01, &
&        1.000000000D+00, 3.039000000D-02, 1.731900000D-01, 4.409000000D-01, 5.653200000D-01, &
&        1.000000000D+00, 2.798900000D-02, 1.650200000D-01, 4.373600000D-01, 5.725500000D-01, &
&        1.000000000D+00, 2.559600000D-02, 1.545900000D-01, 4.287800000D-01, 5.862000000D-01, &
&        1.000000000D+00, 1.831000000D-02, 1.350000000D-01, 4.261000000D-01, 6.043000000D-01, &
&        1.000000000D+00, 2.337700000D-02, 1.461000000D-01, 4.223000000D-01, 5.945200000D-01, &
&        1.000000000D+00/)
!
! Set functions for H - He
!
      select case(numatomic(iatom))
        case(1:2)
! 1S
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exs1(is1(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs1(is1(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs1(is1(numatomic(iatom))+4)
          coeff(locprim(ishell)+1)= coeffs1(is1(numatomic(iatom))+4)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs1(is1(numatomic(iatom))+5)
          coeff(locprim(ishell)+1)= coeffs1(is1(numatomic(iatom))+5)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
! Set functions for Li - Ne
!
        case(3:10)
! 1S
          ishell= ishell+1
          do j= 1,6
            ex(locprim(ishell)+j)= exs2(is2(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs2(is2(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 6
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+6
          locbf(ishell+1) = locbf(ishell)+1
! 2SP
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exs2(is2(numatomic(iatom))+j+6)
            coeff(locprim(ishell)+j)= coeffs2(is2(numatomic(iatom))+j+6)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exs2(is2(numatomic(iatom))+j+6)
            coeff(locprim(ishell)+j)= coeffp2(ip2(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs2(is2(numatomic(iatom))+10)
          coeff(locprim(ishell)+1)= coeffs2(is2(numatomic(iatom))+10)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs2(is2(numatomic(iatom))+10)
          coeff(locprim(ishell)+1)= coeffp2(ip2(numatomic(iatom))+4)
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs2(is2(numatomic(iatom))+11)
          coeff(locprim(ishell)+1)= coeffs2(is2(numatomic(iatom))+11)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs2(is2(numatomic(iatom))+11)
          coeff(locprim(ishell)+1)= coeffp2(ip2(numatomic(iatom))+5)
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
! Set  functions for Na - Ar
!
        case(11:18)
! S functions
          ishell= ishell+1
          do j= 1,6
            ex(locprim(ishell)+j)= exs3(is3(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs3(is3(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 6
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+6
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exs3(is3(numatomic(iatom))+j+6)
            coeff(locprim(ishell)+j)= coeffs3(is3(numatomic(iatom))+j+6)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs3(is3(numatomic(iatom))+10)
          coeff(locprim(ishell)+1)= coeffs3(is3(numatomic(iatom))+10)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs3(is3(numatomic(iatom))+11)
          coeff(locprim(ishell)+1)= coeffs3(is3(numatomic(iatom))+11)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs3(is3(numatomic(iatom))+12)
          coeff(locprim(ishell)+1)= coeffs3(is3(numatomic(iatom))+12)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs3(is3(numatomic(iatom))+13)
          coeff(locprim(ishell)+1)= coeffs3(is3(numatomic(iatom))+13)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
! P functions
          ishell= ishell+1
          num= 4
          if(numatomic(iatom) >= 17) num= 5
          do j= 1,num
            ex(locprim(ishell)+j)= exp3(ip3(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp3(ip3(numatomic(iatom))+j)
          enddo
          mprim(ishell)= num
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+num
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= exp3(ip3(numatomic(iatom))+j+num)
            coeff(locprim(ishell)+j)= coeffp3(ip3(numatomic(iatom))+j+num)
          enddo
          mprim(ishell)= 2
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exp3(ip3(numatomic(iatom))+num+3)
          coeff(locprim(ishell)+1)= coeffp3(ip3(numatomic(iatom))+num+3)
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exp3(ip3(numatomic(iatom))+num+4)
          coeff(locprim(ishell)+1)= coeffp3(ip3(numatomic(iatom))+num+4)
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exp3(ip3(numatomic(iatom))+num+5)
          coeff(locprim(ishell)+1)= coeffp3(ip3(numatomic(iatom))+num+5)
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
! Set  functions for K, Ca, Ga - Kr 
!
        case(19:20,31:36)
! S functions
          ishell= ishell+1
          do j= 1,6
            ex(locprim(ishell)+j)= exs4(is4(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs4(is4(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 6
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+6
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          num= 2
          if(numatomic(iatom) >= 31) num= 3
          do j= 1,num
            ex(locprim(ishell)+j)= exs4(is4(numatomic(iatom))+j+6)
            coeff(locprim(ishell)+j)= coeffs4(is4(numatomic(iatom))+j+6)
          enddo
          mprim(ishell)= num
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+num
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs4(is4(numatomic(iatom))+num+7)
          coeff(locprim(ishell)+1)= coeffs4(is4(numatomic(iatom))+num+7)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs4(is4(numatomic(iatom))+num+8)
          coeff(locprim(ishell)+1)= coeffs4(is4(numatomic(iatom))+num+8)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs4(is4(numatomic(iatom))+num+9)
          coeff(locprim(ishell)+1)= coeffs4(is4(numatomic(iatom))+num+9)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs4(is4(numatomic(iatom))+num+10)
          coeff(locprim(ishell)+1)= coeffs4(is4(numatomic(iatom))+num+10)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs4(is4(numatomic(iatom))+num+11)
          coeff(locprim(ishell)+1)= coeffs4(is4(numatomic(iatom))+num+11)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exs4(is4(numatomic(iatom))+num+12)
          coeff(locprim(ishell)+1)= coeffs4(is4(numatomic(iatom))+num+12)
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
! P functions
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exp4(ip4(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp4(ip4(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exp4(ip4(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffp4(ip4(numatomic(iatom))+j+3)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          num= 1
          if(numatomic(iatom) >= 31) num= 3
          do j= 1,num
            ex(locprim(ishell)+j)= exp4(ip4(numatomic(iatom))+j+6)
            coeff(locprim(ishell)+j)= coeffp4(ip4(numatomic(iatom))+j+6)
          enddo
          mprim(ishell)= num
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+num
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exp4(ip4(numatomic(iatom))+num+7)
          coeff(locprim(ishell)+1)= coeffp4(ip4(numatomic(iatom))+num+7)
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exp4(ip4(numatomic(iatom))+num+8)
          coeff(locprim(ishell)+1)= coeffp4(ip4(numatomic(iatom))+num+8)
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exp4(ip4(numatomic(iatom))+num+9)
          coeff(locprim(ishell)+1)= coeffp4(ip4(numatomic(iatom))+num+9)
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          ex(locprim(ishell)+1)= exp4(ip4(numatomic(iatom))+num+10)
          coeff(locprim(ishell)+1)= coeffp4(ip4(numatomic(iatom))+num+10)
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
! D functions
          ishell= ishell+1
          num= 3
          if(numatomic(iatom) >= 31) num= 4
          do j= 1,num
            ex(locprim(ishell)+j)= exd4(id4(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffd4(id4(numatomic(iatom))+j)
          enddo
          mprim(ishell)= num
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+num
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1)= locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1)= locbf(ishell)+6
          endif
          if(numatomic(iatom) >= 31) then
            ishell= ishell+1
            ex(locprim(ishell)+1)= exd4(id4(numatomic(iatom))+num+1)
            coeff(locprim(ishell)+1)= coeffd4(id4(numatomic(iatom))+num+1)
            mprim(ishell)= 1
            mtype(ishell)= 2
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+1
            if(spher) then
              mbf(ishell)= 5
              locbf(ishell+1)= locbf(ishell)+5
            else
              mbf(ishell)= 6
              locbf(ishell+1)= locbf(ishell)+6
            endif
          endif
        case(:0)
        case default
          write(*,'(" Error! This program supports H - Ca, Ga - Kr 6-311G basis set.")')
          call iabort
      end select
!
      if(ishell > mxshell) then
        write(*,'(" Error! The number of basis shells exceeds mxshell",i6,".")')mxshell
        call iabort
      endif
      if(locprim(ishell+1) > mxprim ) then
        write(*,'(" Error! The number of primitive basis functions exceeds mxprim", &
&             i6,".")')mxprim
        call iabort
      endif
      if(locbf(ishell+1) > mxao ) then
        write(*,'(" Error! The number of basis functions exceeds mxao",i6,".")')mxao
        call iabort
      endif
!
      return
end


!-----------------------------------
  subroutine bs6311gd(iatom,ishell)
!-----------------------------------
!
! Set basis functions of 6-311G(d)
!
      implicit none
      integer,intent(in) :: iatom
      integer,intent(inout) :: ishell
!
      call bs6311g(iatom,ishell)
!
      call bs6311gs(iatom,ishell)
!
      return
end


!------------------------------------
  subroutine bs6311gdp(iatom,ishell)
!------------------------------------
!
! Set basis functions of 6-311G(d,p)
!
      use modmolecule, only : numatomic
      implicit none
      integer,intent(in) :: iatom
      integer,intent(inout) :: ishell
!
      call bs6311g(iatom,ishell)
!
      select case(numatomic(iatom))
        case(1:2)
          call bs6311gss(iatom,ishell)
        case(3:)
        call bs6311gs(iatom,ishell)
      end select
!
      return
end


!-----------------------------------
  subroutine bs6311gs(iatom,ishell)
!-----------------------------------
!
! Set basis polarization function of 6-311G(d)
!
      use modmolecule, only : numatomic
      use modbasis, only : ex, coeff, locprim, locbf, locatom, mprim, mtype, mbf, spher
      use modparam, only : mxao, mxshell, mxprim
      implicit none
      integer,intent(in) :: iatom
      integer,intent(inout) :: ishell
      real(8),parameter :: one=1.0D+00
      real(8) :: ex6311gs(3:36)
      data ex6311gs(3:20) &
       /2.000D-01,2.550D-01,4.010D-01,6.260D-01,9.130D-01,1.292D+00,1.750D+00,2.304D+00, &
&       1.750D-01,1.750D-01,3.250D-01,4.500D-01,5.500D-01,6.500D-01,7.500D-01,8.500D-01, &
&       2.290D-01,2.600D-01/
      data ex6311gs(31:36) &
&      /1.690D-01,2.280D-01,2.640D-01,3.050D-01,4.510D-01,3.950D-01/
!
      select case (numatomic(iatom))
        case(:2)
        case(3:20,31:36)
          ishell= ishell+1
          ex(locprim(ishell)+1)= ex6311gs(numatomic(iatom))
          coeff(locprim(ishell)+1)= one
          mprim(ishell)= 1
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1)= locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1)= locbf(ishell)+6
          endif
        case default
          write(*,'(" Error! This program supports Li - Ca, Ga - Kr 6-311G* basis set.")')
          call iabort
      end select
!
      if(ishell > mxshell) then
        write(*,'(" Error! The number of basis shells exceeds mxshell",i6,".")')mxshell
        call iabort
      endif
      if(locprim(ishell+1) > mxprim ) then
        write(*,'(" Error! The number of primitive basis functions exceeds mxprim", &
&             i6,".")')mxprim
        call iabort
      endif
      if(locbf(ishell+1) > mxao ) then
        write(*,'(" Error! The number of basis functions exceeds mxao",i6,".")')mxao
        call iabort
      endif
      return
end


!------------------------------------
  subroutine bs6311gss(iatom,ishell)
!------------------------------------
!
! Set basis polarization function of Hydrogen 6-311G(d,p)
!
      use modmolecule, only : numatomic
      use modbasis, only : ex, coeff, locprim, locbf, locatom, mprim, mtype, mbf
      use modparam, only : mxao, mxshell, mxprim
      implicit none
      integer,intent(in) :: iatom
      integer,intent(inout) :: ishell
      real(8),parameter :: one=1.0D+00, ex6311gss=0.75D+00
!
      select case (numatomic(iatom))
        case(1:2)
          ishell= ishell+1
          ex(locprim(ishell)+1)= ex6311gss
          coeff(locprim(ishell)+1)= one
          mprim(ishell)= 1
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          mbf(ishell)= 3
          locbf(ishell+1) = locbf(ishell)+3
      end select
!
      if(ishell > mxshell) then
        write(*,'(" Error! The number of basis shells exceeds mxshell",i6,".")')mxshell
        call iabort
      endif
      if(locprim(ishell+1) > mxprim ) then
        write(*,'(" Error! The number of primitive basis functions exceeds mxprim", &
&             i6,".")')mxprim
        call iabort
      endif
      if(locbf(ishell+1) > mxao ) then
        write(*,'(" Error! The number of basis functions exceeds mxao",i6,".")')mxao
        call iabort
      endif
      return
end


!-----------------------------------
  subroutine bsccpvdz(iatom,ishell)
!-----------------------------------
!
! Set basis functions of cc-pVDZ
!
      use modmolecule, only : numatomic
      use modbasis, only : ex, coeff, locprim, locbf, locatom, mprim, mbf, mtype, spher
      use modparam, only : mxao, mxshell, mxprim
      implicit none
      integer,intent(in) :: iatom
      integer,intent(inout) :: ishell
      integer :: j, num
      integer :: ise(36)=(/0,4,8,17,26,35,44,53,62,71,80,92,104,116,128,140,152,164, 176,176, &
&                          190,210,230,250,270,290,310,330,350,370,390,404,418,432,446,460/)
      integer :: isc(36)=(/0,4,8,25,42,59,76,93,110,127,144,178,212,246,280,314,348,382, &
&                          416,416,469,565,661,757,853,949,1045,1141,1237,1333,1429,1482, &
&                          1535,1588,1641,1694/)
      integer :: ipe(36)=(/0,1,2,6,10,14,18,22,26,30,34,42,50,58,66,74,82,90, 98,98, &
&                          109,125,141,157,173,189,205,221,237,253,269,280,291,302,313,324/)
      integer :: ipc(36)=(/0,1,2,6,10,14,18,22,26,30,34,49,64,79,94,109,124,139, 154,154, &
&                          185,246,307,368,429,490,551,612,673,734,795,826,857,888,919,950/)
      integer :: ide(36)= (/0,0,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15, 16,16, &
&                           21,29,37,45,53,61,69,77,85,93,101,107,113,119,125,131/)
      integer :: idc(20:36)= (/0,5,20,35,50,65,80,95,110,125,140,155,161,167,173,179,185/)
      integer :: ife(21:30)= (/0,2,4,6,8,10,12,14,16,18/)
      integer :: ifc(21:30)= (/0,2,4,6,8,10,12,14,16,18/)
      real(8),parameter :: one=1.0D+00
      real(8) :: exps(474)= &
&     (/ 0.1301000D+02,0.1962000D+01,0.4446000D+00,0.1220000D+00,0.3836000D+02,&
&        0.5770000D+01,0.1240000D+01,0.2976000D+00,0.1469000D+04,0.2205000D+03,&
&        0.5026000D+02,0.1424000D+02,0.4581000D+01,0.1580000D+01,0.5640000D+00,&
&        0.7345000D-01,0.2805000D-01,0.2940000D+04,0.4412000D+03,0.1005000D+03,&
&        0.2843000D+02,0.9169000D+01,0.3196000D+01,0.1159000D+01,0.1811000D+00,&
&        0.5890000D-01,0.4570000D+04,0.6859000D+03,0.1565000D+03,0.4447000D+02,&
&        0.1448000D+02,0.5131000D+01,0.1898000D+01,0.3329000D+00,0.1043000D+00,&
&        0.6665000D+04,0.1000000D+04,0.2280000D+03,0.6471000D+02,0.2106000D+02,&
&        0.7495000D+01,0.2797000D+01,0.5215000D+00,0.1596000D+00,0.9046000D+04,&
&        0.1357000D+04,0.3093000D+03,0.8773000D+02,0.2856000D+02,0.1021000D+02,&
&        0.3838000D+01,0.7466000D+00,0.2248000D+00,0.1172000D+05,0.1759000D+04,&
&        0.4008000D+03,0.1137000D+03,0.3703000D+02,0.1327000D+02,0.5025000D+01,&
&        0.1013000D+01,0.3023000D+00,0.1471000D+05,0.2207000D+04,0.5028000D+03,&
&        0.1426000D+03,0.4647000D+02,0.1670000D+02,0.6356000D+01,0.1316000D+01,&
&        0.3897000D+00,0.1788000D+05,0.2683000D+04,0.6115000D+03,0.1735000D+03,&
&        0.5664000D+02,0.2042000D+02,0.7810000D+01,0.1653000D+01,0.4869000D+00,&
&        0.3170000D+05,0.4755000D+04,0.1082000D+04,0.3064000D+03,0.9953000D+02,&
&        0.3542000D+02,0.1330000D+02,0.4392000D+01,0.1676000D+01,0.5889000D+00,&
&        0.5640000D-01,0.2307000D-01,0.4739000D+05,0.7108000D+04,0.1618000D+04,&
&        0.4584000D+03,0.1493000D+03,0.5359000D+02,0.2070000D+02,0.8384000D+01,&
&        0.2542000D+01,0.8787000D+00,0.1077000D+00,0.3999000D-01,0.6415000D+05,&
&        0.9617000D+04,0.2189000D+04,0.6205000D+03,0.2027000D+03,0.7315000D+02,&
&        0.2855000D+02,0.1177000D+02,0.3300000D+01,0.1173000D+01,0.1752000D+00,&
&        0.6473000D-01,0.7886000D+05,0.1182000D+05,0.2692000D+04,0.7634000D+03,&
&        0.2496000D+03,0.9028000D+02,0.3529000D+02,0.1451000D+02,0.4053000D+01,&
&        0.1482000D+01,0.2517000D+00,0.9243000D-01,0.9484000D+05,0.1422000D+05,&
&        0.3236000D+04,0.9171000D+03,0.2995000D+03,0.1081000D+03,0.4218000D+02,&
&        0.1728000D+02,0.4858000D+01,0.1818000D+01,0.3372000D+00,0.1232000D+00,&
&        0.1108000D+06,0.1661000D+05,0.3781000D+04,0.1071000D+04,0.3498000D+03,&
&        0.1263000D+03,0.4926000D+02,0.2016000D+02,0.5720000D+01,0.2182000D+01,&
&        0.4327000D+00,0.1570000D+00,0.1279000D+06,0.1917000D+05,0.4363000D+04,&
&        0.1236000D+04,0.4036000D+03,0.1457000D+03,0.5681000D+02,0.2323000D+02,&
&        0.6644000D+01,0.2575000D+01,0.5371000D+00,0.1938000D+00,0.1457000D+06,&
&        0.2184000D+05,0.4972000D+04,0.1408000D+04,0.4597000D+03,0.1659000D+03,&
&        0.6469000D+02,0.2644000D+02,0.7628000D+01,0.2996000D+01,0.6504000D+00,&
&        0.2337000D+00,0.1900007D+06,0.2848146D+05,0.6482701D+04,0.1835891D+04,&
&        0.5987243D+03,0.2158841D+03,0.8401242D+02,0.3422488D+02,0.1002497D+02,&
&        0.4055920D+01,0.1020261D+01,0.4268650D+00,0.6334700D-01,0.2630100D-01,&
&        0.2715278D+07,0.4065984D+06,0.9253004D+05,0.2620792D+05,0.8549429D+04,&
&        0.3085975D+04,0.1203172D+04,0.4984869D+03,0.2167360D+03,0.9787476D+02,&
&        0.4520433D+02,0.2021187D+02,0.9574751D+01,0.4540346D+01,0.1995687D+01,&
&        0.9422150D+00,0.4178450D+00,0.9576100D-01,0.5135100D-01,0.2387800D-01,&
&        0.3014643D+07,0.4514329D+06,0.1027338D+06,0.2909817D+05,0.9492330D+04,&
&        0.3426346D+04,0.1335896D+04,0.5535026D+03,0.2406925D+03,0.1087293D+03,&
&        0.5026457D+02,0.2258004D+02,0.1071432D+02,0.5093546D+01,0.2244183D+01,&
&        0.1059570D+01,0.4688490D+00,0.1061430D+00,0.5526200D-01,0.2546500D-01,&
&        0.3321857D+07,0.4974356D+06,0.1132027D+06,0.3206333D+05,0.1045962D+05,&
&        0.3775506D+04,0.1472040D+04,0.6099331D+03,0.2652634D+03,0.1198607D+03,&
&        0.5544891D+02,0.2498372D+02,0.1188056D+02,0.5660311D+01,0.2495703D+01,&
&        0.1177866D+01,0.5200440D+00,0.1159650D+00,0.5893800D-01,0.2694600D-01,&
&        0.6177194D+07,0.9249295D+06,0.2104865D+06,0.5962005D+05,0.1945076D+05,&
&        0.7022056D+04,0.2738763D+04,0.1135814D+04,0.4950923D+03,0.2247487D+03,&
&        0.1053836D+03,0.5019359D+02,0.2224957D+02,0.1098265D+02,0.5383665D+01,&
&        0.2343685D+01,0.1105202D+01,0.4878480D+00,0.8959900D-01,0.3342300D-01,&
&        0.3960805D+07,0.5931155D+06,0.1349768D+06,0.3823067D+05,0.1247154D+05,&
&        0.4501743D+04,0.1755212D+04,0.7273039D+03,0.3163678D+03,0.1430098D+03,&
&        0.6621805D+02,0.2991896D+02,0.1430318D+02,0.6839451D+01,0.3012374D+01,&
&        0.1418808D+01,0.6236240D+00,0.1340980D+00,0.6554800D-01,0.2958400D-01,&
&        0.4316265D+07,0.6463424D+06,0.1470897D+06,0.4166152D+05,0.1359077D+05,&
&        0.4905750D+04,0.1912746D+04,0.7926043D+03,0.3448065D+03,0.1558999D+03,&
&        0.7223091D+02,0.3272506D+02,0.1566762D+02,0.7503483D+01,0.3312223D+01,&
&        0.1558471D+01,0.6839140D+00,0.1467570D+00,0.7058300D-01,0.3144900D-01,&
&        0.4675675D+07,0.7001615D+06,0.1593373D+06,0.4513046D+05,0.1472238D+05,&
&        0.5314222D+04,0.2072018D+04,0.8586188D+03,0.3735497D+03,0.1689229D+03,&
&        0.7829639D+02,0.3552123D+02,0.1704144D+02,0.8173000D+01,0.3610318D+01,&
&        0.1697047D+01,0.7435320D+00,0.1583440D+00,0.7503600D-01,0.3309100D-01,&
&        0.5045991D+07,0.7556142D+06,0.1719568D+06,0.4870479D+05,0.1588841D+05,&
&        0.5735123D+04,0.2236137D+04,0.9266468D+03,0.4031743D+03,0.1823476D+03,&
&        0.8454885D+02,0.3839634D+02,0.1845859D+02,0.8863548D+01,0.3916227D+01,&
&        0.1838870D+01,0.8043620D+00,0.1697970D+00,0.7930600D-01,0.3467700D-01,&
&        0.5430321D+07,0.8131665D+06,0.1850544D+06,0.5241466D+05,0.1709868D+05,&
&        0.6171994D+04,0.2406481D+04,0.9972584D+03,0.4339289D+03,0.1962869D+03,&
&        0.9104280D+02,0.4138425D+02,0.1993278D+02,0.9581891D+01,0.4234516D+01,&
&        0.1985814D+01,0.8670830D+00,0.1813390D+00,0.8365700D-01,0.3626700D-01,&
&        0.5820021D+07,0.8715234D+06,0.1983350D+06,0.5617631D+05,0.1832582D+05,&
&        0.6614955D+04,0.2579199D+04,0.1068849D+04,0.4651045D+03,0.2104130D+03,&
&        0.9761629D+02,0.4438020D+02,0.2142308D+02,0.1030891D+02,0.4553645D+01,&
&        0.2132821D+01,0.9296970D+00,0.1921470D+00,0.8759500D-01,0.3770200D-01,&
&        0.4851300D+06,0.7271900D+05,0.1655200D+05,0.4687800D+04,0.1529100D+04,&
&        0.5518100D+03,0.2151800D+03,0.8817400D+02,0.2715400D+02,0.1150300D+02,&
&        0.3301800D+01,0.1331400D+01,0.1931600D+00,0.7089500D-01,0.5218000D+06,&
&        0.7821400D+05,0.1780300D+05,0.5041900D+04,0.1644500D+04,0.5934300D+03,&
&        0.2313600D+03,0.9476200D+02,0.2927400D+02,0.1245000D+02,0.3646300D+01,&
&        0.1502500D+01,0.2450300D+00,0.9159400D-01,0.5595838D+06,0.8387933D+05,&
&        0.1909267D+05,0.5407392D+04,0.1763756D+04,0.6364567D+03,0.2480884D+03,&
&        0.1015785D+03,0.3147551D+02,0.1343728D+02,0.4008690D+01,0.1684929D+01,&
&        0.3000190D+00,0.1135870D+00,0.5989900D+06,0.8978300D+05,0.2043500D+05,&
&        0.5786900D+04,0.1887300D+04,0.6809700D+03,0.2653900D+03,0.1086300D+03,&
&        0.3376000D+02,0.1446500D+02,0.4389000D+01,0.1878300D+01,0.3585900D+00,&
&        0.1364900D+00,0.6401000D+06,0.9593800D+05,0.2183300D+05,0.6181900D+04,&
&        0.2015700D+04,0.7271000D+03,0.2832800D+03,0.1159100D+03,0.3612400D+02,&
&        0.1553200D+02,0.4785700D+01,0.2081700D+01,0.4202800D+00,0.1606900D+00,&
&        0.6813588D+06,0.1021265D+06,0.2324371D+05,0.6582007D+04,0.2146429D+04,&
&        0.7743378D+03,0.3016702D+03,0.1234118D+03,0.3856755D+02,0.1663738D+02,&
&        0.5198795D+01,0.2294814D+01,0.4852110D+00,0.1862700D+00/)
      real(8) :: coeffs(1747)= &
&     (/ 0.1968500D-01,0.1379770D+00,0.4781480D+00,0.1000000D+01,0.2380900D-01,&
&        0.1548910D+00,0.4699870D+00,0.1000000D+01,0.7660000D-03,0.5892000D-02,&
&        0.2967100D-01,0.1091800D+00,0.2827890D+00,0.4531230D+00,0.2747740D+00,&
&        0.9751000D-02,-.1200000D-03,-.9230000D-03,-.4689000D-02,-.1768200D-01,&
&        -.4890200D-01,-.9600900D-01,-.1363800D+00,0.5751020D+00,0.1000000D+01,&
&        0.6800000D-03,0.5236000D-02,0.2660600D-01,0.9999300D-01,0.2697020D+00,&
&        0.4514690D+00,0.2950740D+00,0.1258700D-01,-.1230000D-03,-.9660000D-03,&
&        -.4831000D-02,-.1931400D-01,-.5328000D-01,-.1207230D+00,-.1334350D+00,&
&        0.5307670D+00,0.1000000D+01,0.6960000D-03,0.5353000D-02,0.2713400D-01,&
&        0.1013800D+00,0.2720550D+00,0.4484030D+00,0.2901230D+00,0.1432200D-01,&
&        -.1390000D-03,-.1097000D-02,-.5444000D-02,-.2191600D-01,-.5975100D-01,&
&        -.1387320D+00,-.1314820D+00,0.5395260D+00,0.1000000D+01,0.6920000D-03,&
&        0.5329000D-02,0.2707700D-01,0.1017180D+00,0.2747400D+00,0.4485640D+00,&
&        0.2850740D+00,0.1520400D-01,-.1460000D-03,-.1154000D-02,-.5725000D-02,&
&        -.2331200D-01,-.6395500D-01,-.1499810D+00,-.1272620D+00,0.5445290D+00,&
&        0.1000000D+01,0.7000000D-03,0.5389000D-02,0.2740600D-01,0.1032070D+00,&
&        0.2787230D+00,0.4485400D+00,0.2782380D+00,0.1544000D-01,-.1530000D-03,&
&        -.1208000D-02,-.5992000D-02,-.2454400D-01,-.6745900D-01,-.1580780D+00,&
&        -.1218310D+00,0.5490030D+00,0.1000000D+01,0.7100000D-03,0.5470000D-02,&
&        0.2783700D-01,0.1048000D+00,0.2830620D+00,0.4487190D+00,0.2709520D+00,&
&        0.1545800D-01,-.1600000D-03,-.1263000D-02,-.6267000D-02,-.2571600D-01,&
&        -.7092400D-01,-.1654110D+00,-.1169550D+00,0.5573680D+00,0.1000000D+01,&
&        0.7210000D-03,0.5553000D-02,0.2826700D-01,0.1064440D+00,0.2868140D+00,&
&        0.4486410D+00,0.2647610D+00,0.1533300D-01,-.1650000D-03,-.1308000D-02,&
&        -.6495000D-02,-.2669100D-01,-.7369000D-01,-.1707760D+00,-.1123270D+00,&
&        0.5628140D+00,0.1000000D+01,0.7380000D-03,0.5677000D-02,0.2888300D-01,&
&        0.1085400D+00,0.2909070D+00,0.4483240D+00,0.2580260D+00,0.1506300D-01,&
&        -.1720000D-03,-.1357000D-02,-.6737000D-02,-.2766300D-01,-.7620800D-01,&
&        -.1752270D+00,-.1070380D+00,0.5670500D+00,0.1000000D+01,0.4588780D-03,&
&        0.3550700D-02,0.1826180D-01,0.7166500D-01,0.2123460D+00,0.4162030D+00,&
&        0.3730200D+00,0.6250540D-01,-.6245320D-02,0.2433740D-02,-.4423810D-03,&
&        -.1121620D-03,-.8685120D-03,-.4513300D-02,-.1814360D-01,-.5807990D-01,&
&        -.1376530D+00,-.1939080D+00,0.8580090D-01,0.6044190D+00,0.4417190D+00,&
&        0.1305470D-01,0.1701600D-04,0.1306930D-03,0.6877840D-03,0.2723590D-02,&
&        0.8955290D-02,0.2078320D-01,0.3193800D-01,-.1913680D-01,-.1025950D+00,&
&        -.1989450D+00,0.6559520D+00,0.1000000D+01,0.3460230D-03,0.2680770D-02,&
&        0.1383670D-01,0.5517670D-01,0.1696600D+00,0.3647030D+00,0.4068560D+00,&
&        0.1350890D+00,0.4908840D-02,0.2864600D-03,0.2645900D-04,-.8778390D-04,&
&        -.6747250D-03,-.3556030D-02,-.1421540D-01,-.4767480D-01,-.1148920D+00,&
&        -.2006760D+00,-.3412240D-01,0.5704540D+00,0.5423090D+00,0.2181280D-01,&
&        0.1696280D-04,0.1298650D-03,0.6888310D-03,0.2735330D-02,0.9312240D-02,&
&        0.2232650D-01,0.4111950D-01,0.5456420D-02,-.1340120D+00,-.2561760D+00,&
&        0.6058560D+00,0.1000000D+01,0.2902500D-03,0.2250640D-02,0.1164590D-01,&
&        0.4673770D-01,0.1462990D+00,0.3302830D+00,0.4158610D+00,0.1892530D+00,&
&        0.1158890D-01,-.1283850D-02,0.4258830D-03,-.7580480D-04,-.5817910D-03,&
&        -.3081130D-02,-.1231120D-01,-.4197810D-01,-.1033710D+00,-.1963080D+00,&
&        -.8300020D-01,0.5410400D+00,0.5787960D+00,0.2881470D-01,0.1750780D-04,&
&        0.1342080D-03,0.7124420D-03,0.2843300D-02,0.9768420D-02,0.2418500D-01,&
&        0.4749930D-01,0.2036210D-01,-.1587880D+00,-.3116940D+00,0.6201470D+00,&
&        0.1000000D+01,0.2704430D-03,0.2097170D-02,0.1085060D-01,0.4367540D-01,&
&        0.1376530D+00,0.3166440D+00,0.4185810D+00,0.2102120D+00,0.1449520D-01,&
&        -.2035900D-02,0.6241860D-03,-.7231770D-04,-.5551160D-03,-.2938050D-02,&
&        -.1176870D-01,-.4029070D-01,-.1006090D+00,-.1965280D+00,-.1023820D+00,&
&        0.5271900D+00,0.5932510D+00,0.3326520D-01,0.1851130D-04,0.1422360D-03,&
&        0.7521850D-03,0.3022790D-02,0.1036770D-01,0.2625630D-01,0.5239890D-01,&
&        0.2909590D-01,-.1780030D+00,-.3468740D+00,0.6230200D+00,0.1000000D+01,&
&        0.2555090D-03,0.1981930D-02,0.1027600D-01,0.4148230D-01,0.1319840D+00,&
&        0.3086620D+00,0.4206470D+00,0.2228780D+00,0.1640350D-01,-.2542550D-02,&
&        0.7480500D-03,-.6969390D-04,-.5352660D-03,-.2837090D-02,-.1139830D-01,&
&        -.3929290D-01,-.9963640D-01,-.1979830D+00,-.1148600D+00,0.5185950D+00,&
&        0.6018470D+00,0.3686120D-01,0.1911990D-04,0.1472230D-03,0.7779120D-03,&
&        0.3145460D-02,0.1082000D-01,0.2799570D-01,0.5639780D-01,0.3581900D-01,&
&        -.1933870D+00,-.3720970D+00,0.6242460D+00,0.1000000D+01,0.2476350D-03,&
&        0.1920260D-02,0.9961920D-02,0.4029750D-01,0.1286040D+00,0.3034800D+00,&
&        0.4214320D+00,0.2307810D+00,0.1789710D-01,-.2975160D-02,0.8495220D-03,&
&        -.6870390D-04,-.5276810D-03,-.2796710D-02,-.1126510D-01,-.3888340D-01,&
&        -.9950250D-01,-.1997400D+00,-.1233600D+00,0.5131940D+00,0.6071200D+00,&
&        0.3967530D-01,0.1990770D-04,0.1534830D-03,0.8095030D-03,0.3289740D-02,&
&        0.1129670D-01,0.2963850D-01,0.5998510D-01,0.4132480D-01,-.2074740D+00,&
&        -.3928890D+00,0.6328400D+00,0.1000000D+01,0.2411530D-03,0.1870950D-02,&
&        0.9708270D-02,0.3931530D-01,0.1259320D+00,0.2993410D+00,0.4218860D+00,&
&        0.2372010D+00,0.1915310D-01,-.3347920D-02,0.9298830D-03,-.6789220D-04,&
&        -.5218360D-03,-.2765130D-02,-.1115370D-01,-.3859190D-01,-.9948480D-01,&
&        -.2013920D+00,-.1303130D+00,0.5094430D+00,0.6107250D+00,0.4215490D-01,&
&        0.2049860D-04,0.1582980D-03,0.8336390D-03,0.3398800D-02,0.1167380D-01,&
&        0.3096220D-01,0.6295330D-01,0.4602570D-01,-.2193120D+00,-.4087730D+00,&
&        0.6384650D+00,0.1000000D+01,0.2367000D-03,0.1835230D-02,0.9528600D-02,&
&        0.3862830D-01,0.1240810D+00,0.2964710D+00,0.4220680D+00,0.2417110D+00,&
&        0.2005090D-01,-.3610000D-02,0.9756070D-03,-.6749100D-04,-.5185220D-03,&
&        -.2748250D-02,-.1110070D-01,-.3848200D-01,-.9975990D-01,-.2030880D+00,&
&        -.1356080D+00,0.5071950D+00,0.6128980D+00,0.4429680D-01,0.2104570D-04,&
&        0.1625650D-03,0.8554630D-03,0.3497450D-02,0.1201560D-01,0.3213680D-01,&
&        0.6552790D-01,0.4993700D-01,-.2297690D+00,-.4210060D+00,0.6423310D+00,&
&        0.1000000D+01, 0.22145000D-03,0.17183000D-02,0.89234800D-02,0.36301830D-01,&
&        0.11762223D+00,0.28604352D+00,0.42260708D+00,0.25774366D+00,0.23918930D-01,&
&        -.49521800D-02,0.17177900D-02,-.89209000D-03,0.24510000D-03,-.64530000D-04,&
&        -.49662000D-03,-.26282600D-02,-.10668450D-01,-.37135090D-01,-.98042840D-01,&
&        -.20342692D+00,-.15244655D+00,0.48279406D+00,0.62923839D+00,0.61648420D-01,&
&        -.14799710D-01,0.36108900D-02,0.22230000D-04,0.17170000D-03,0.90452000D-03,&
&        0.37034300D-02,0.12837500D-01,0.34754590D-01,0.73034910D-01,0.61000830D-01,&
&        -.24292928D+00,-.48708500D+00,0.56502804D+00,0.65574386D+00,0.26728940D-01,&
&        0.53100000D-05,0.41110000D-04,0.21568000D-03,0.88827000D-03,0.30581300D-02,&
&        0.83760800D-02,0.17410560D-01,0.15154530D-01,-.62079190D-01,-.12611803D+00,&
&        0.17360694D+00,0.37822943D+00,-.65964698D+00,0.10000000D+01,0.81472210D-05,&
&        0.63347880D-04,0.33303840D-03,0.14040550D-02,0.50817250D-02,0.16269260D-01,&
&        0.46245770D-01,0.11372230D+00,0.22576360D+00,0.31067000D+00,0.21919060D+00,&
&        0.72158790D-01,0.11870300D+00,0.12205320D+00,0.21367950D-01,-.53572460D-03,&
&        0.24357740D-03,-.87966170D-04,0.78782460D-04,-.47221090D-05,-.36718290D-04,&
&        -.19308830D-03,-.81468700D-03,-.29555260D-02,-.95200350D-02,-.27468580D-01,&
&        -.69915280D-01,-.14992510D+00,-.24591530D+00,-.24012930D+00,0.35679870D-01,&
&        0.49150230D+00,0.49113810D+00,0.91206330D-01,-.53567230D-02,0.88128360D-03,&
&        -.76055360D-03,0.63401160D-03,0.91399050D-06,0.71085130D-05,0.37381260D-04,&
&        0.15788280D-03,0.57376860D-03,0.18592440D-02,0.54331820D-02,0.14253870D-01,&
&        0.32461440D-01,0.60034540D-01,0.69161050D-01,-.21130840D-01,-.26668320D+00,&
&        -.43675910D+00,0.64982430D-01,0.70095990D+00,0.45155620D+00,0.30119100D-01,&
&        -.13294800D-01,-.22019510D-06,-.17114190D-05,-.90084690D-05,-.37999970D-04,&
&        -.13832270D-03,-.44736920D-03,-.13106910D-02,-.34298600D-02,-.78475790D-02,&
&        -.14471890D-01,-.16906690D-01,0.53961150D-02,0.66710620D-01,0.11783560D+00,&
&        -.27381340D-01,-.22601490D+00,-.30735390D+00,0.25440540D+00,0.59815900D+00,&
&        -.37572380D-06,-.29819070D-05,-.15225860D-04,-.66846860D-04,-.23131290D-03,&
&        -.79597290D-03,-.21619610D-02,-.62064590D-02,-.12619050D-01,-.27394590D-01,&
&        -.23365160D-01,-.57346270D-02,0.15360250D+00,0.14471000D+00,0.93596990D-01,&
&        -.86877300D+00,0.21145970D-01,0.22754980D+01,-.11907700D+01,0.10000000D+01,&
&        0.80607820D-05,0.62675180D-04,0.32950060D-03,0.13892030D-02,0.50284690D-02,&
&        0.16104190D-01,0.45812320D-01,0.11286130D+00,0.22481930D+00,0.31145710D+00,&
&        0.22249950D+00,0.72931280D-01,0.11606830D+00,0.11947740D+00,0.20978680D-01,&
&        -.50917150D-03,0.22178590D-03,-.76368960D-04,0.77195390D-04,-.46304860D-05,&
&        -.36004510D-04,-.18934200D-03,-.79887810D-03,-.28986980D-02,-.93397010D-02,&
&        -.26974640D-01,-.68789130D-01,-.14810370D+00,-.24452530D+00,-.24199160D+00,&
&        0.31837900D-01,0.49326860D+00,0.49396550D+00,0.91963130D-01,-.53169920D-02,&
&        0.80856240D-03,-.69184590D-03,0.60865120D-03,0.92305590D-06,0.71789740D-05,&
&        0.37751340D-04,0.15945320D-03,0.57951500D-03,0.18784140D-02,0.54927470D-02,&
&        0.14432970D-01,0.32964080D-01,0.61254930D-01,0.71341130D-01,-.19731500D-01,&
&        -.27418690D+00,-.44409770D+00,0.77760840D-01,0.70684440D+00,0.44138920D+00,&
&        0.27997690D-01,-.12107900D-01,-.21803230D-06,-.16948600D-05,-.89192080D-05,&
&        -.37636330D-04,-.13695750D-03,-.44328940D-03,-.12988680D-02,-.34067520D-02,&
&        -.78108290D-02,-.14492450D-01,-.17081360D-01,0.48976660D-02,0.67531080D-01,&
&        0.11733180D+00,-.29850250D-01,-.22776340D+00,-.29281150D+00,0.26653000D+00,&
&        0.59124060D+00,-.39751260D-06,-.31610800D-05,-.16093750D-04,-.70929470D-04,&
&        -.24427100D-03,-.84578920D-03,-.22822080D-02,-.66198730D-02,-.13350240D-01,&
&        -.29558300D-01,-.24770390D-01,-.84146240D-02,0.16938550D+00,0.15007870D+00,&
&        0.97877770D-01,-.96536080D+00,0.14897210D+00,0.21911790D+01,-.12433250D+01,&
&        0.10000000D+01,0.80399990D-05,0.62514020D-04,0.32865530D-03,0.13856970D-02,&
&        0.50162170D-02,0.16069310D-01,0.45742420D-01,0.11285440D+00,0.22543440D+00,&
&        0.31404610D+00,0.22678190D+00,0.73340690D-01,0.11024740D+00,0.11313580D+00,&
&        0.19712950D-01,-.47190880D-03,0.18616060D-03,-.62085980D-04,0.72953140D-04,&
&        -.45030030D-05,-.35012950D-04,-.18413390D-03,-.77692160D-03,-.28195050D-02,&
&        -.90874860D-02,-.26271340D-01,-.67127260D-01,-.14511300D+00,-.24124830D+00,&
&        -.24163140D+00,0.30673620D-01,0.49704150D+00,0.49588750D+00,0.91818680D-01,&
&        -.53925140D-02,0.71023800D-03,-.63631280D-03,0.59799320D-03,0.93206480D-06,&
&        0.72493060D-05,0.38119670D-04,0.16102380D-03,0.58522100D-03,0.18975020D-02,&
&        0.55509090D-02,0.14605840D-01,0.33429740D-01,0.62357220D-01,0.73124350D-01,&
&        -.19114720D-01,-.28172490D+00,-.44881510D+00,0.92026960D-01,0.71101170D+00,&
&        0.43092740D+00,0.26045890D-01,-.11010490D-01,-.21589440D-06,-.16785190D-05,&
&        -.88312130D-05,-.37277690D-04,-.13560990D-03,-.43923510D-03,-.12869480D-02,&
&        -.33821490D-02,-.77656460D-02,-.14479850D-01,-.17155020D-01,0.46101010D-02,&
&        0.68278310D-01,0.11613680D+00,-.32770490D-01,-.22800000D+00,-.27939910D+00,&
&        0.27711650D+00,0.58529990D+00,-.40934160D-06,-.32589560D-05,-.16563900D-04,&
&        -.73166890D-04,-.25127840D-03,-.87326570D-03,-.23476540D-02,-.68531500D-02,&
&        -.13764200D-01,-.30846790D-01,-.25622080D-01,-.10051230D-01,0.17953300D+00,&
&        0.15224000D+00,0.94838870D-01,-.10148760D+01,0.23088100D+00,0.21133210D+01,&
&        -.12530480D+01,0.10000000D+01,0.41286670D-05,0.32107670D-04,0.16884160D-03,&
&        0.71285200D-03,0.25893250D-02,0.83773500D-02,0.24417250D-01,0.63651350D-01,&
&        0.14276180D+00,0.25412750D+00,0.30095120D+00,0.17665130D+00,0.69367090D-01,&
&        0.11795790D+00,0.89161870D-01,0.11036300D-01,-.35460480D-03,0.10573110D-03,&
&        0.11146400D-04,-.23017720D-05,-.17895360D-04,-.94161740D-04,-.39750740D-03,&
&        -.14470250D-02,-.46946220D-02,-.13823870D-01,-.36746430D-01,-.86471850D-01,&
&        -.16967350D+00,-.25070890D+00,-.19611560D+00,0.14572440D+00,0.54667060D+00,&
&        0.39794340D+00,0.52770070D-01,-.43745370D-02,0.32040350D-03,-.51420770D-04,&
&        0.48629570D-06,0.37766450D-05,0.19906640D-04,0.83891640D-04,0.30657060D-03,&
&        0.99441070D-03,0.29619590D-02,0.79694730D-02,0.19550170D-01,0.40850350D-01,&
&        0.69290030D-01,0.61469840D-01,-.69813020D-01,-.35175970D+00,-.38286290D+00,&
&        0.26764010D+00,0.71759500D+00,0.30208140D+00,0.77495140D-02,-.11024510D-06,&
&        -.85302330D-06,-.45203580D-05,-.18916120D-04,-.69743440D-04,-.22378670D-03,&
&        -.67545030D-03,-.17893460D-02,-.44778580D-02,-.91401440D-02,-.16105620D-01,&
&        -.13348700D-01,0.14260270D-01,0.89316900D-01,0.88852790D-01,-.63687760D-01,&
&        -.27832620D+00,-.18300710D+00,0.67909370D+00,0.21798930D-06,0.16129400D-05,&
&        0.91118420D-05,0.35006450D-04,0.14353150D-03,0.40358960D-03,0.14251770D-02,&
&        0.31140090D-02,0.98144490D-02,0.14746980D-01,0.39115120D-01,0.91708880D-02,&
&        0.15598780D-01,-.28168440D+00,-.68952610D-02,-.17697810D+00,0.14430610D+01,&
&        -.10293180D+01,-.13076670D+01,0.10000000D+01,0.82421270D-05,0.64085870D-04,&
&        0.33692530D-03,0.14206480D-02,0.51436830D-02,0.16485690D-01,0.46985600D-01,&
&        0.11624370D+00,0.23352770D+00,0.32928370D+00,0.24403040D+00,0.72198060D-01,&
&        0.76878060D-01,0.78522350D-01,0.12941090D-01,-.37848730D-03,-.25032030D-04,&
&        -.24215170D-04,0.34620710D-04,-.39360950D-05,-.30604810D-04,-.16096260D-03,&
&        -.67923480D-03,-.24661820D-02,-.79576290D-02,-.23072480D-01,-.59329560D-01,&
&        -.12994510D+00,-.22123520D+00,-.22925500D+00,0.35807330D-01,0.51076020D+00,&
&        0.50083070D+00,0.90118300D-01,-.69099090D-02,-.19129250D-03,-.60323120D-03,&
&        0.56216080D-03,0.94627090D-06,0.73605840D-05,0.38699350D-04,0.16351100D-03,&
&        0.59417750D-03,0.19277370D-02,0.56417310D-02,0.14878480D-01,0.34147830D-01,&
&        0.64057940D-01,0.75576590D-01,-.19460700D-01,-.29578740D+00,-.45211700D+00,&
&        0.12245310D+00,0.71697560D+00,0.40927120D+00,0.22219690D-01,-.90112020D-02,&
&        -.20953910D-06,-.16294390D-05,-.85705920D-05,-.36192720D-04,-.13161460D-03,&
&        -.42668100D-03,-.12502700D-02,-.32946650D-02,-.75818600D-02,-.14228640D-01,&
&        -.16937960D-01,0.44542980D-02,0.68670420D-01,0.11133350D+00,-.39008200D-01,&
&        -.22157550D+00,-.25443590D+00,0.28658660D+00,0.57557410D+00,-.41212310D-06,&
&        -.32820990D-05,-.16674330D-04,-.73699990D-04,-.25294950D-03,-.88014250D-03,&
&        -.23654820D-02,-.69263540D-02,-.13938510D-01,-.31438400D-01,-.26257490D-01,&
&        -.10483130D-01,0.18564720D+00,0.15248390D+00,0.74113680D-01,-.10180970D+01,&
&        0.29803720D+00,0.19719890D+01,-.11792530D+01,0.10000000D+01,0.80488030D-05,&
&        0.62583060D-04,0.32902390D-03,0.13873550D-02,0.50232560D-02,0.16101400D-01,&
&        0.45900340D-01,0.11361540D+00,0.22838690D+00,0.32211590D+00,0.23836610D+00,&
&        0.74046670D-01,0.92141970D-01,0.93397900D-01,0.15739650D-01,-.41866820D-03,&
&        0.53763180D-04,-.38166540D-04,0.43196030D-04,-.41559540D-05,-.32314010D-04,&
&        -.16995250D-03,-.71713690D-03,-.26036250D-02,-.83991090D-02,-.24341090D-01,&
&        -.62519480D-01,-.13659290D+00,-.23127070D+00,-.23837340D+00,0.31238370D-01,&
&        0.50868180D+00,0.49876950D+00,0.90335520D-01,-.60053370D-02,0.23124540D-03,&
&        -.56436800D-03,0.49922600D-03,0.95321780D-06,0.74146050D-05,0.38983930D-04,&
&        0.16471520D-03,0.59859800D-03,0.19423900D-02,0.56872370D-02,0.15013290D-01,&
&        0.34524550D-01,0.64958200D-01,0.77161940D-01,-.18734110D-01,-.30091850D+00,&
&        -.45546610D+00,0.12864630D+00,0.71833160D+00,0.40517430D+00,0.21682270D-01,&
&        -.83435660D-02,-.20630080D-06,-.16041690D-05,-.84384370D-05,-.35631510D-04,&
&        -.12959980D-03,-.42015340D-03,-.12319540D-02,-.32489220D-02,-.74937170D-02,&
&        -.14101020D-01,-.16916000D-01,0.42189960D-02,0.68338100D-01,0.10982010D+00,&
&        -.40090050D-01,-.21747390D+00,-.24651350D+00,0.27314350D+00,0.57483210D+00,&
&        -.40093670D-06,-.31892550D-05,-.16230790D-04,-.71579200D-04,-.24639580D-03,&
&        -.85449070D-03,-.23075930D-02,-.67282920D-02,-.13661650D-01,-.30622400D-01,&
&        -.26311370D-01,-.97601830D-02,0.18019060D+00,0.15296340D+00,0.55054130D-01,&
&        -.95513640D+00,0.25868130D+00,0.18340490D+01,-.93332400D+00,0.10000000D+01,&
&        0.79790260D-05,0.62040710D-04,0.32617350D-03,0.13753600D-02,0.49799970D-02,&
&        0.15964340D-01,0.45520860D-01,0.11273850D+00,0.22682620D+00,0.32030740D+00,&
&        0.23740210D+00,0.74776860D-01,0.95818720D-01,0.96499110D-01,0.16233620D-01,&
&        -.45354970D-03,0.51135190D-04,-.41745080D-04,0.40275770D-04,-.42002400D-05,&
&        -.32658310D-04,-.17176440D-03,-.72478530D-03,-.26314620D-02,-.84892720D-02,&
&        -.24606190D-01,-.63220590D-01,-.13819570D+00,-.23406800D+00,-.24150020D+00,&
&        0.30353120D-01,0.51013410D+00,0.49749390D+00,0.89707460D-01,-.59410340D-02,&
&        0.21753620D-03,-.54801550D-03,0.45258040D-03,0.95926920D-06,0.74618510D-05,&
&        0.39231370D-04,0.16577060D-03,0.60243350D-03,0.19552170D-02,0.57263260D-02,&
&        0.15129840D-01,0.34839730D-01,0.65703510D-01,0.78315030D-01,-.18770370D-01,&
&        -.30626630D+00,-.45664290D+00,0.13781690D+00,0.71936760D+00,0.39925790D+00,&
&        0.20799330D-01,-.78206630D-02,-.20288400D-06,-.15775800D-05,-.82988130D-05,&
&        -.35041540D-04,-.12746550D-03,-.41326950D-03,-.12122610D-02,-.31993180D-02,&
&        -.73909720D-02,-.13936490D-01,-.16785750D-01,0.41498560D-02,0.67976460D-01,&
&        0.10758070D+00,-.41660220D-01,-.21280440D+00,-.23813600D+00,0.26507880D+00,&
&        0.57227740D+00,-.38630530D-06,-.30687880D-05,-.15648260D-04,-.68835880D-04,&
&        -.23773670D-03,-.82131730D-03,-.22296300D-02,-.64678410D-02,-.13254630D-01,&
&        -.29466860D-01,-.25990660D-01,-.84998070D-02,0.17273160D+00,0.15121890D+00,&
&        0.35545090D-01,-.88293530D+00,0.21435300D+00,0.17118650D+01,-.71400370D+00,&
&        0.10000000D+01,0.82089960D-05,0.63828840D-04,0.33558000D-03,0.14150750D-02,&
&        0.51244440D-02,0.16432560D-01,0.46893980D-01,0.11635340D+00,0.23505110D+00,&
&        0.33501840D+00,0.25347790D+00,0.73009010D-01,0.61842440D-01,0.63029560D-01,&
&        0.10080630D-01,-.22445280D-03,-.59327670D-04,-.11585620D-04,0.81151090D-05,&
&        -.36578490D-05,-.28440940D-04,-.14959280D-03,-.63130090D-03,-.22930520D-02,&
&        -.74051230D-02,-.21520320D-01,-.55609740D-01,-.12301760D+00,-.21301040D+00,&
&        -.22658370D+00,0.35467960D-01,0.51816970D+00,0.50256300D+00,0.89556740D-01,&
&        -.70313110D-02,-.43391670D-03,-.58317110D-03,0.42287880D-03,0.95941490D-06,&
&        0.74626140D-05,0.39238430D-04,0.16578680D-03,0.60259050D-03,0.19556620D-02,&
&        0.57303910D-02,0.15147560D-01,0.34934990D-01,0.65980720D-01,0.78930830D-01,&
&        -.19062490D-01,-.30959210D+00,-.45586100D+00,0.14829310D+00,0.71340390D+00,&
&        0.39760630D+00,0.22955230D-01,-.91517580D-02,-.20137530D-06,-.15658320D-05,&
&        -.82371820D-05,-.34781050D-04,-.12652650D-03,-.41025890D-03,-.12038340D-02,&
&        -.31790620D-02,-.73538280D-02,-.13890220D-01,-.16778750D-01,0.41633780D-02,&
&        0.68147030D-01,0.10610290D+00,-.43399800D-01,-.20949500D+00,-.23102710D+00,&
&        0.25905320D+00,0.56914260D+00,-.39242450D-06,-.31139090D-05,-.15904470D-04,&
&        -.69813940D-04,-.24178480D-03,-.83261950D-03,-.22702940D-02,-.65574270D-02,&
&        -.13542880D-01,-.29897680D-01,-.26931060D-01,-.78276930D-02,0.17416670D+00,&
&        0.15954680D+00,0.19955500D-01,-.88970000D+00,0.24868920D+00,0.16130120D+01,&
&        -.59902770D+00,0.10000000D+01,0.78010260D-05,0.60656660D-04,0.31889640D-03,&
&        0.13446870D-02,0.48690500D-02,0.15610130D-01,0.44520770D-01,0.11031110D+00,&
&        0.22203420D+00,0.31337390D+00,0.23151210D+00,0.76409200D-01,0.11038180D+00,&
&        0.10943720D+00,0.18363110D-01,-.60430840D-03,0.50922450D-04,-.55407300D-04,&
&        0.39694820D-04,-.44047060D-05,-.34248010D-04,-.18012380D-03,-.76004550D-03,&
&        -.27593480D-02,-.89009700D-02,-.25793780D-01,-.66238610D-01,-.14459270D+00,&
&        -.24401100D+00,-.25048370D+00,0.28525770D-01,0.51158740D+00,0.49280610D+00,&
&        0.87884370D-01,-.58202810D-02,0.20135080D-03,-.51825530D-03,0.37315030D-03,&
&        0.97046820D-06,0.75492450D-05,0.39688920D-04,0.16772000D-03,0.60951010D-03,&
&        0.19788460D-02,0.57980490D-02,0.15341580D-01,0.35404840D-01,0.67020980D-01,&
&        0.80269450D-01,-.19272310D-01,-.31601290D+00,-.45731620D+00,0.15508410D+00,&
&        0.72028720D+00,0.38851220D+00,0.19243260D-01,-.71038070D-02,-.19593540D-06,&
&        -.15234720D-05,-.80148080D-05,-.33839920D-04,-.12311910D-03,-.39920850D-03,&
&        -.11719000D-02,-.30961410D-02,-.71719930D-02,-.13566210D-01,-.16439890D-01,&
&        0.41076280D-02,0.66939640D-01,0.10282210D+00,-.44229450D-01,-.20311910D+00,&
&        -.22300220D+00,0.25179750D+00,0.56500910D+00,-.35322290D-06,-.27988120D-05,&
&        -.14325170D-04,-.62709460D-04,-.21794900D-03,-.74743160D-03,-.20492710D-02,&
&        -.58852030D-02,-.12268850D-01,-.26831470D-01,-.24792610D-01,-.59847460D-02,&
&        0.15571240D+00,0.14366830D+00,0.83741030D-02,-.74607110D+00,0.12443670D+00,&
&        0.15101100D+01,-.34771220D+00,0.10000000D+01,0.85492410D-05,0.66474100D-04,&
&        0.34949620D-03,0.14738320D-02,0.53383300D-02,0.17127080D-01,0.48940850D-01,&
&        0.12179340D+00,0.24765890D+00,0.35824310D+00,0.27981740D+00,0.68574910D-01,&
&        -.13110920D-02,0.19140010D-02,-.87592200D-03,0.37400960D-03,-.14013990D-03,&
&        0.47571320D-04,-.36427110D-04,-.26400690D-05,-.20527200D-04,-.10798590D-03,&
&        -.45585770D-03,-.16577580D-02,-.53684920D-02,-.15712490D-01,-.41225580D-01,&
&        -.94064590D-01,-.17199540D+00,-.19585230D+00,0.45329070D-01,0.52444420D+00,&
&        0.50061420D+00,0.89455270D-01,-.21462620D-02,0.21121130D-02,-.41339800D-03,&
&        0.32097520D-03,0.99671030D-06,0.77541630D-05,0.40760190D-04,0.17228110D-03,&
&        0.62593700D-03,0.20328550D-02,0.59546460D-02,0.15766400D-01,0.36376380D-01,&
&        0.68923430D-01,0.82380930D-01,-.20113600D-01,-.32525260D+00,-.46028990D+00,&
&        0.16355460D+00,0.72971180D+00,0.37697510D+00,0.14332240D-01,-.66712100D-02,&
&        0.19958180D-06,0.15529730D-05,0.81612590D-05,0.34507470D-04,0.12532750D-03,&
&        0.40729900D-03,0.11927340D-02,0.31631400D-02,0.73039420D-02,0.13912790D-01,&
&        0.16706200D-01,-.40355860D-02,-.69688610D-01,-.10301050D+00,0.44714420D-01,&
&        0.21500270D+00,0.22201630D+00,-.31147760D+00,-.56934290D+00,-.54359100D-06,&
&        -.43368940D-05,-.21975720D-04,-.97473920D-04,-.33316150D-03,-.11661920D-02,&
&        -.31193080D-02,-.92395040D-02,-.18554710D-01,-.42811890D-01,-.35710950D-01,&
&        -.16383500D-01,0.26446640D+00,0.20865880D+00,-.17743820D-01,-.13538730D+01,&
&        0.81829260D+00,0.16950360D+01,-.13886560D+01,0.10000000D+01,0.20680000D-03,&
&        0.16047000D-02,0.83402000D-02,0.34024800D-01,0.11116990D+00,0.27539300D+00,&
&        0.42126280D+00,0.27389060D+00,0.28372000D-01,-.62931000D-02,0.20606000D-02,&
&        -.92690000D-03,0.22730000D-03,-.64300000D-04,-.49540000D-03,-.26208000D-02,&
&        -.10683900D-01,-.37412300D-01,-.10096360D+00,-.21451410D+00,-.17522970D+00,&
&        0.48315990D+00,0.63236770D+00,0.68494200D-01,-.11871200D-01,0.26652000D-02,&
&        0.24500000D-04,0.18950000D-03,0.99640000D-03,0.41082000D-02,0.14293800D-01,&
&        0.39803400D-01,0.85594000D-01,0.79630500D-01,-.29391070D+00,-.52639140D+00,&
&        0.58642490D+00,0.67263470D+00,0.27612300D-01,-.57000000D-05,-.44000000D-04,&
&        -.23050000D-03,-.95440000D-03,-.33055000D-02,-.92888000D-02,-.19864400D-01,&
&        -.19088800D-01,0.73235600D-01,0.13415260D+00,-.18319290D+00,-.35713080D+00,&
&        0.62460130D+00,0.10000000D+01,0.20450000D-03,0.15868000D-02,0.82480000D-02,&
&        0.33664900D-01,0.11012490D+00,0.27356070D+00,0.42106700D+00,0.27667910D+00,&
&        0.29218000D-01,-.65903000D-02,0.22430000D-02,-.10382000D-02,0.26950000D-03,&
&        -.63800000D-04,-.49160000D-03,-.26002000D-02,-.10608000D-01,-.37160200D-01,&
&        -.10057900D+00,-.21439770D+00,-.17826170D+00,0.47774040D+00,0.63559830D+00,&
&        0.72217400D-01,-.12726500D-01,0.29608000D-02,0.24600000D-04,0.19000000D-03,&
&        0.99930000D-03,0.41200000D-02,0.14355700D-01,0.40037500D-01,0.86579400D-01,&
&        0.81586100D-01,-.29347700D+00,-.53679830D+00,0.56379850D+00,0.69471820D+00,&
&        0.31573000D-01,-.63000000D-05,-.48600000D-04,-.25530000D-03,-.10560000D-02,&
&        -.36674000D-02,-.10305300D-01,-.22220000D-01,-.21527500D-01,0.80675200D-01,&
&        0.15249580D+00,-.19805280D+00,-.40739540D+00,0.64772880D+00,0.10000000D+01,&
&        0.20240000D-03,0.15709000D-02,0.81662000D-02,0.33339900D-01,0.10917260D+00,&
&        0.27188530D+00,0.42085090D+00,0.27922570D+00,0.30030100D-01,-.68804000D-02,&
&        0.24240000D-02,-.11491000D-02,0.30950000D-03,-.63400000D-04,-.48830000D-03,&
&        -.25821000D-02,-.10540200D-01,-.36932500D-01,-.10023550D+00,-.21429480D+00,&
&        -.18105260D+00,0.47254100D+00,0.63861940D+00,0.75810700D-01,-.13527800D-01,&
&        0.31970000D-02,0.24600000D-04,0.19070000D-03,0.10031000D-02,0.41353000D-02,&
&        0.14425900D-01,0.40296200D-01,0.87567000D-01,0.83517800D-01,-.29329350D+00,&
&        -.54705200D+00,0.54387380D+00,0.71435910D+00,0.35344300D-01,-.68000000D-05,&
&        -.52500000D-04,-.27560000D-03,-.11389000D-02,-.39646000D-02,-.11142300D-01,&
&        -.24199100D-01,-.23633900D-01,0.86631700D-01,0.16858390D+00,-.20914250D+00,&
&        -.45009180D+00,0.66039780D+00,0.10000000D+01,0.20040000D-03,0.15554000D-02,&
&        0.80872000D-02,0.33034400D-01,0.10829240D+00,0.27033610D+00,0.42062360D+00,&
&        0.28159220D+00,0.30811000D-01,-.71617000D-02,0.26022000D-02,-.12583000D-02,&
&        0.34650000D-03,-.62900000D-04,-.48500000D-03,-.25644000D-02,-.10476100D-01,&
&        -.36722300D-01,-.99922500D-01,-.21419730D+00,-.18365930D+00,0.46754540D+00,&
&        0.64147400D+00,0.79256900D-01,-.14269700D-01,0.33792000D-02,0.24700000D-04,&
&        0.19130000D-03,0.10068000D-02,0.41514000D-02,0.14499100D-01,0.40565800D-01,&
&        0.88536400D-01,0.85421200D-01,-.29325810D+00,-.55707270D+00,0.52614360D+00,&
&        0.73203710D+00,0.38824600D-01,-.72000000D-05,-.55900000D-04,-.29380000D-03,&
&        -.12136000D-02,-.42340000D-02,-.11903500D-01,-.26020600D-01,-.25614800D-01,&
&        0.91942700D-01,0.18387000D+00,-.21884610D+00,-.48965240D+00,0.67758180D+00,&
&        0.10000000D+01,0.19840000D-03,0.15400000D-02,0.80096000D-02,0.32734100D-01,&
&        0.10744800D+00,0.26889460D+00,0.42044110D+00,0.28380410D+00,0.31545500D-01,&
&        -.74268000D-02,0.27728000D-02,-.13635000D-02,0.38120000D-03,-.62500000D-04,&
&        -.48160000D-03,-.25466000D-02,-.10411200D-01,-.36517900D-01,-.99629500D-01,&
&        -.21413100D+00,-.18609110D+00,0.46282610D+00,0.64411410D+00,0.82550200D-01,&
&        -.14969400D-01,0.35288000D-02,0.24800000D-04,0.19190000D-03,0.10100000D-02,&
&        0.41659000D-02,0.14568300D-01,0.40834500D-01,0.89485900D-01,0.87278600D-01,&
&        -.29336440D+00,-.56671090D+00,0.51056580D+00,0.74772140D+00,0.42151200D-01,&
&        -.76000000D-05,-.58800000D-04,-.30920000D-03,-.12766000D-02,-.44634000D-02,&
&        -.12557500D-01,-.27614500D-01,-.27394500D-01,0.96409400D-01,0.19768710D+00,&
&        -.22666930D+00,-.52411650D+00,0.68898650D+00,0.10000000D+01,0.19690000D-03,&
&        0.15286000D-02,0.79500000D-02,0.32493800D-01,0.10672400D+00,0.26757010D+00,&
&        0.42018510D+00,0.28580150D+00,0.32246100D-01,-.76828000D-02,0.29393000D-02,&
&        -.14662000D-02,0.41440000D-03,-.62200000D-04,-.47940000D-03,-.25341000D-02,&
&        -.10363600D-01,-.36351600D-01,-.99373700D-01,-.21406100D+00,-.18831920D+00,&
&        0.45838160D+00,0.64656640D+00,0.85657900D-01,-.15612300D-01,0.36490000D-02,&
&        0.24900000D-04,0.19280000D-03,0.10149000D-02,0.41857000D-02,0.14645900D-01,&
&        0.41107000D-01,0.90395500D-01,0.89062300D-01,-.29357180D+00,-.57596980D+00,&
&        0.49685780D+00,0.76168950D+00,0.45326700D-01,-.79000000D-05,-.61400000D-04,&
&        -.32300000D-03,-.13330000D-02,-.46672000D-02,-.13135200D-01,-.29034200D-01,&
&        -.29017300D-01,0.10026640D+00,0.21038180D+00,-.23324710D+00,-.55464970D+00,&
&        0.69695220D+00,0.10000000D+01/)
      real(8) :: expp(335)= &
&     (/ 0.7270000D+00,0.1275000D+01,0.1534000D+01,0.2749000D+00,0.7362000D-01,&
&        0.2403000D-01,0.3619000D+01,0.7110000D+00,0.1951000D+00,0.6018000D-01,&
&        0.6001000D+01,0.1241000D+01,0.3364000D+00,0.9538000D-01,0.9439000D+01,&
&        0.2002000D+01,0.5456000D+00,0.1517000D+00,0.1355000D+02,0.2917000D+01,&
&        0.7973000D+00,0.2185000D+00,0.1770000D+02,0.3854000D+01,0.1046000D+01,&
&        0.2753000D+00,0.2267000D+02,0.4977000D+01,0.1347000D+01,0.3471000D+00,&
&        0.2839000D+02,0.6270000D+01,0.1695000D+01,0.4317000D+00,0.1381000D+03,&
&        0.3224000D+02,0.9985000D+01,0.3484000D+01,0.1231000D+01,0.4177000D+00,&
&        0.6513000D-01,0.2053000D-01,0.1799000D+03,0.4214000D+02,0.1313000D+02,&
&        0.4628000D+01,0.1670000D+01,0.5857000D+00,0.1311000D+00,0.4112000D-01,&
&        0.2588000D+03,0.6089000D+02,0.1914000D+02,0.6881000D+01,0.2574000D+01,&
&        0.9572000D+00,0.2099000D+00,0.5986000D-01,0.3159000D+03,0.7442000D+02,&
&        0.2348000D+02,0.8488000D+01,0.3217000D+01,0.1229000D+01,0.2964000D+00,&
&        0.8768000D-01,0.3705000D+03,0.8733000D+02,0.2759000D+02,0.1000000D+02,&
&        0.3825000D+01,0.1494000D+01,0.3921000D+00,0.1186000D+00,0.3997000D+03,&
&        0.9419000D+02,0.2975000D+02,0.1077000D+02,0.4119000D+01,0.1625000D+01,&
&        0.4726000D+00,0.1407000D+00,0.4176000D+03,0.9833000D+02,0.3104000D+02,&
&        0.1119000D+02,0.4249000D+01,0.1624000D+01,0.5322000D+00,0.1620000D+00,&
&        0.4537000D+03,0.1068000D+03,0.3373000D+02,0.1213000D+02,0.4594000D+01,&
&        0.1678000D+01,0.5909000D+00,0.1852000D+00,0.1072043D+04,0.2538439D+03,&
&        0.8131626D+02,0.3024183D+02,0.1210110D+02,0.5022554D+01,0.1909220D+01,&
&        0.7713040D+00,0.3005700D+00,0.7664900D-01,0.2777200D-01,0.1059219D+05,&
&        0.2507533D+04,0.8144571D+03,0.3115195D+03,0.1319617D+03,0.5998718D+02,&
&        0.2866250D+02,0.1410851D+02,0.7103706D+01,0.3609200D+01,0.1776070D+01,&
&        0.8547600D+00,0.4022390D+00,0.1546650D+00,0.6494500D-01,0.2635900D-01,&
&        0.1191203D+05,0.2819947D+04,0.9159479D+03,0.3503842D+03,0.1484825D+03,&
&        0.6753944D+02,0.3230332D+02,0.1592786D+02,0.8038035D+01,0.4093916D+01,&
&        0.2022390D+01,0.9761020D+00,0.4595950D+00,0.1771520D+00,0.7351700D-01,&
&        0.2940100D-01,0.1327320D+05,0.3142126D+04,0.1020588D+04,0.3904407D+03,&
&        0.1655043D+03,0.7532006D+02,0.3605503D+02,0.1780436D+02,0.9002929D+01,&
&        0.4594544D+01,0.2276760D+01,0.1101178D+01,0.5186380D+00,0.2005650D+00,&
&        0.8129100D-01,0.3179500D-01,0.1445420D+05,0.3421676D+04,0.1111387D+04,&
&        0.4251918D+03,0.1802623D+03,0.8206117D+02,0.3929726D+02,0.1941959D+02,&
&        0.9828899D+01,0.5016810D+01,0.2487091D+01,0.1198780D+01,0.5586950D+00,&
&        0.2089240D+00,0.8460800D-01,0.3325800D-01,0.1620586D+05,0.3836274D+04,&
&        0.1246048D+04,0.4767535D+03,0.2021895D+03,0.9209487D+02,0.4414720D+02,&
&        0.2185468D+02,0.1108596D+02,0.5674108D+01,0.2823170D+01,0.1368621D+01,&
&        0.6444310D+00,0.2483820D+00,0.9725500D-01,0.3663300D-01,0.1774569D+05,&
&        0.4200721D+04,0.1364429D+04,0.5220806D+03,0.2214595D+03,0.1009096D+03,&
&        0.4840115D+02,0.2398536D+02,0.1218250D+02,0.6242298D+01,0.3110944D+01,&
&        0.1509958D+01,0.7108450D+00,0.2731900D+00,0.1042330D+00,0.3829100D-01,&
&        0.1926778D+05,0.4560986D+04,0.1481436D+04,0.5668671D+03,0.2404910D+03,&
&        0.1096105D+03,0.5259491D+02,0.2608361D+02,0.1326143D+02,0.6799778D+01,&
&        0.3393414D+01,0.1648766D+01,0.7762820D+00,0.2980030D+00,0.1136180D+00,&
&        0.4162400D-01,0.2102792D+05,0.4977560D+04,0.1616740D+04,0.6186718D+03,&
&        0.2625183D+03,0.1196907D+03,0.5746585D+02,0.2852829D+02,0.1452148D+02,&
&        0.7453850D+01,0.3723553D+01,0.1809813D+01,0.8513360D+00,0.3248140D+00,&
&        0.1195220D+00,0.4236600D-01,0.2276057D+05,0.5387679D+04,0.1749945D+04,&
&        0.6696653D+03,0.2841948D+03,0.1296077D+03,0.6225415D+02,0.3092964D+02,&
&        0.1575827D+02,0.8094211D+01,0.4046921D+01,0.1967869D+01,0.9252950D+00,&
&        0.3529920D+00,0.1273070D+00,0.4435600D-01,0.2441198D+05,0.5778518D+04,&
&        0.1876862D+04,0.7182361D+03,0.3048327D+03,0.1390453D+03,0.6680417D+02,&
&        0.3320699D+02,0.1692816D+02,0.8696229D+01,0.4350510D+01,0.2116523D+01,&
&        0.9953870D+00,0.3781120D+00,0.1345790D+00,0.4628200D-01,0.3248600D+04,&
&        0.7699700D+03,0.2482000D+03,0.9336400D+02,0.3825100D+02,0.1642200D+02,&
&        0.6791800D+01,0.2833600D+01,0.1106200D+01,0.2225000D+00,0.6177200D-01,&
&        0.3568100D+04,0.8457200D+03,0.2727400D+03,0.1026800D+03,0.4214800D+02,&
&        0.1814900D+02,0.7593400D+01,0.3196400D+01,0.1274300D+01,0.2825800D+00,&
&        0.8409000D-01,0.3886356D+04,0.9212020D+03,0.2971932D+03,0.1119751D+03,&
&        0.4603462D+02,0.1987419D+02,0.8386088D+01,0.3558728D+01,0.1447282D+01,&
&        0.3477790D+00,0.1076990D+00,0.4135600D+04,0.9803400D+03,0.3163500D+03,&
&        0.1192500D+03,0.4906800D+02,0.2121200D+02,0.8946200D+01,0.3823600D+01,&
&        0.1588300D+01,0.4096900D+00,0.1245900D+00,0.4340800D+04,0.1028900D+04,&
&        0.3320200D+03,0.1251600D+03,0.5151100D+02,0.2228100D+02,0.9341700D+01,&
&        0.4013200D+01,0.1700200D+01,0.4719400D+00,0.1442100D+00,0.4474270D+04,&
&        0.1060579D+04,0.3422081D+03,0.1289984D+03,0.5308722D+02,0.2295942D+02,&
&        0.9507300D+01,0.4083055D+01,0.1750446D+01,0.5291900D+00,0.1643690D+00/)
      real(8) :: coeffp(981)= &
&     (/ 0.1000000D+01,0.1000000D+01,0.2278400D-01,0.1391070D+00,0.5003750D+00,&
&        0.1000000D+01,0.2911100D-01,0.1693650D+00,0.5134580D+00,0.1000000D+01,&
&        0.3548100D-01,0.1980720D+00,0.5052300D+00,0.1000000D+01,0.3810900D-01,&
&        0.2094800D+00,0.5085570D+00,0.1000000D+01,0.3991900D-01,0.2171690D+00,&
&        0.5103190D+00,0.1000000D+01,0.4301800D-01,0.2289130D+00,0.5087280D+00,&
&        0.1000000D+01,0.4487800D-01,0.2357180D+00,0.5085210D+00,0.1000000D+01,&
&        0.4608700D-01,0.2401810D+00,0.5087440D+00,0.1000000D+01,0.5796410D-02,&
&        0.4157560D-01,0.1628730D+00,0.3594010D+00,0.4499880D+00,0.2275070D+00,&
&        0.8082470D-02,-.5815310D-03,-.4073060D-02,-.1679370D-01,-.3532680D-01,&
&        -.5219710D-01,-.1683590D-01,0.4346130D+00,0.1000000D+01,0.5381610D-02,&
&        0.3924180D-01,0.1574450D+00,0.3585350D+00,0.4572260D+00,0.2159180D+00,&
&        0.6649480D-02,-.8659480D-03,-.6159780D-02,-.2615190D-01,-.5706470D-01,&
&        -.8739060D-01,-.1229900D-01,0.5020850D+00,0.1000000D+01,0.4068470D-02,&
&        0.3068150D-01,0.1291490D+00,0.3208310D+00,0.4538150D+00,0.2750660D+00,&
&        0.1908070D-01,-.7480530D-03,-.5457960D-02,-.2453710D-01,-.5821380D-01,&
&        -.9837560D-01,-.2600640D-01,0.4640200D+00,0.1000000D+01,0.3926560D-02,&
&        0.2988110D-01,0.1272120D+00,0.3209430D+00,0.4554290D+00,0.2685630D+00,&
&        0.1883360D-01,-.8583020D-03,-.6303280D-02,-.2882550D-01,-.6945600D-01,&
&        -.1194930D+00,-.1995810D-01,0.5102680D+00,0.1000000D+01,0.3950050D-02,&
&        0.3024920D-01,0.1295540D+00,0.3275940D+00,0.4569920D+00,0.2530860D+00,&
&        0.1687980D-01,-.9598320D-03,-.7111770D-02,-.3271220D-01,-.7957840D-01,&
&        -.1350160D+00,-.9105850D-02,0.5378020D+00,0.1000000D+01,0.4475410D-02,&
&        0.3417080D-01,0.1442500D+00,0.3539280D+00,0.4590850D+00,0.2063830D+00,&
&        0.1021410D-01,-.1162510D-02,-.8656640D-02,-.3908860D-01,-.9346250D-01,&
&        -.1479940D+00,0.3019040D-01,0.5615730D+00,0.1000000D+01,0.5259820D-02,&
&        0.3983320D-01,0.1646550D+00,0.3873220D+00,0.4570720D+00,0.1516360D+00,&
&        0.1816150D-02,-.1435700D-02,-.1077960D-01,-.4700750D-01,-.1110300D+00,&
&        -.1532750D+00,0.8946090D-01,0.5794440D+00,0.1000000D+01,0.5705550D-02,&
&        0.4304600D-01,0.1765910D+00,0.4068630D+00,0.4525490D+00,0.1228010D+00,&
&        -.4459960D-02,-.1606550D-02,-.1217140D-01,-.5207890D-01,-.1237370D+00,&
&        -.1516190D+00,0.1424250D+00,0.5845010D+00,0.1000000D+01,0.19816600D-02,&
&        0.16129440D-01,0.76578510D-01,0.23269594D+00,0.42445210D+00,0.37326402D+00,&
&        0.78685300D-01,-.59992700D-02,0.26425700D-02,-.85694000D-03,-.64891000D-03,&
&        -.52790700D-02,-.25811310D-01,-.80628920D-01,-.15846552D+00,-.12816816D+00,&
&        0.25610103D+00,0.58724068D+00,0.30372561D+00,0.14164510D-01,0.13595000D-03,&
&        0.10942000D-02,0.54268000D-02,0.16747180D-01,0.33898630D-01,0.25311830D-01,&
&        -.58957130D-01,-.15876120D+00,-.85545230D-01,0.54464665D+00,0.10000000D+01,&
&        0.45000000D-04,0.40100000D-03,0.23020000D-02,0.10037000D-01,0.34954000D-01,&
&        0.97909000D-01,0.21068000D+00,0.33009300D+00,0.33102700D+00,0.15796000D+00,&
&        0.22099000D-01,-.16050000D-02,-.13260000D-02,-.28000000D-03,0.34000000D-04,&
&        -.15000000D-04,-.13100000D-03,-.75700000D-03,-.33180000D-02,-.11706000D-01,&
&        -.33604000D-01,-.74879000D-01,-.12254800D+00,-.13027600D+00,0.14596000D-01,&
&        0.30918400D+00,0.46299800D+00,0.30495700D+00,0.50878000D-01,-.44930000D-02,&
&        -.40000000D-05,-.32000000D-04,-.18500000D-03,-.80800000D-03,-.28700000D-02,&
&        -.82070000D-02,-.18473000D-01,-.30101000D-01,-.32943000D-01,0.79580000D-02,&
&        0.87993000D-01,0.15237700D+00,0.97170000D-01,-.25693800D+00,-.58781500D+00,&
&        0.40000000D-05,0.39000000D-04,0.22100000D-03,0.98400000D-03,0.34230000D-02,&
&        0.99930000D-02,0.21916000D-01,0.37008000D-01,0.37794000D-01,-.43790000D-02,&
&        -.11016400D+00,-.16101700D+00,-.18248200D+00,0.38861100D+00,0.69110000D+00,&
&        0.10000000D+01,0.44000000D-04,0.39100000D-03,0.22480000D-02,0.98230000D-02,&
&        0.34338000D-01,0.96666000D-01,0.20941700D+00,0.33018900D+00,0.33193600D+00,&
&        0.15848800D+00,0.22310000D-01,-.15660000D-02,-.13240000D-02,-.27100000D-03,&
&        0.32000000D-04,-.15000000D-04,-.13100000D-03,-.75500000D-03,-.33190000D-02,&
&        -.11750000D-01,-.33922000D-01,-.76164000D-01,-.12570200D+00,-.13309800D+00,&
&        0.17406000D-01,0.31516500D+00,0.46181400D+00,0.29985600D+00,0.50000000D-01,&
&        -.42300000D-02,0.40000000D-05,0.31000000D-04,0.18200000D-03,0.79500000D-03,&
&        0.28330000D-02,0.81540000D-02,0.18472000D-01,0.30400000D-01,0.33047000D-01,&
&        -.82510000D-02,-.88554000D-01,-.14961200D+00,-.94227000D-01,0.25084600D+00,&
&        0.58664300D+00,0.40000000D-05,0.39000000D-04,0.22300000D-03,0.99200000D-03,&
&        0.34760000D-02,0.10172000D-01,0.22576000D-01,0.38238000D-01,0.39337000D-01,&
&        -.61060000D-02,-.11296200D+00,-.16811400D+00,-.16593200D+00,0.39140300D+00,&
&        0.68184000D+00,0.10000000D+01,0.43000000D-04,0.38400000D-03,0.22100000D-02,&
&        0.96780000D-02,0.33936000D-01,0.95917000D-01,0.20885300D+00,0.33066000D+00,&
&        0.33231200D+00,0.15818800D+00,0.22252000D-01,-.15650000D-02,-.13530000D-02,&
&        -.26500000D-03,0.29000000D-04,-.15000000D-04,-.13100000D-03,-.75500000D-03,&
&        -.33250000D-02,-.11811000D-01,-.34256000D-01,-.77363000D-01,-.12845600D+00,&
&        -.13507800D+00,0.20838000D-01,0.32049900D+00,0.46026000D+00,0.29534600D+00,&
&        0.49046000D-01,-.38240000D-02,0.40000000D-05,0.32000000D-04,0.18300000D-03,&
&        0.80200000D-03,0.28620000D-02,0.82870000D-02,0.18870000D-01,0.31307000D-01,&
&        0.33660000D-01,-.94790000D-02,-.92313000D-01,-.14898900D+00,-.83644000D-01,&
&        0.24933900D+00,0.58051500D+00,0.40000000D-05,0.39000000D-04,0.22300000D-03,&
&        0.99600000D-03,0.34980000D-02,0.10296000D-01,0.22962000D-01,0.39208000D-01,&
&        0.39943000D-01,-.71210000D-02,-.11622500D+00,-.16949600D+00,-.15537400D+00,&
&        0.39502200D+00,0.67890800D+00,0.10000000D+01,0.44000000D-04,0.38900000D-03,&
&        0.22410000D-02,0.98210000D-02,0.34471000D-01,0.97460000D-01,0.21198500D+00,&
&        0.33399000D+00,0.33013700D+00,0.15222700D+00,0.20425000D-01,-.13600000D-02,&
&        -.11950000D-02,-.19700000D-03,0.23000000D-04,-.15000000D-04,-.13500000D-03,&
&        -.77700000D-03,-.34270000D-02,-.12189000D-01,-.35388000D-01,-.79915000D-01,&
&        -.13233500D+00,-.13540100D+00,0.32008000D-01,0.33384900D+00,0.46177300D+00,&
&        0.28129000D+00,0.41843000D-01,-.40020000D-02,0.40000000D-05,0.32000000D-04,&
&        0.18500000D-03,0.81000000D-03,0.29060000D-02,0.83910000D-02,0.19193000D-01,&
&        0.31564000D-01,0.33417000D-01,-.12907000D-01,-.93659000D-01,-.14997700D+00,&
&        -.67234000D-01,0.27075900D+00,0.57580700D+00,0.40000000D-05,0.40000000D-04,&
&        0.22900000D-03,0.10190000D-02,0.36020000D-02,0.10550000D-01,0.23702000D-01,&
&        0.39988000D-01,0.40437000D-01,-.12074000D-01,-.11893900D+00,-.17810000D+00,&
&        -.12386500D+00,0.42972200D+00,0.65078600D+00,0.10000000D+01,0.42000000D-04,&
&        0.37300000D-03,0.21490000D-02,0.94450000D-02,0.33297000D-01,0.94759000D-01,&
&        0.20814400D+00,0.33180500D+00,0.33317500D+00,0.15760100D+00,0.21445000D-01,&
&        -.25580000D-02,-.20270000D-02,-.36000000D-03,0.34000000D-04,-.15000000D-04,&
&        -.12900000D-03,-.74800000D-03,-.33080000D-02,-.11811000D-01,-.34533000D-01,&
&        -.78785000D-01,-.13218300D+00,-.13719500D+00,0.27075000D-01,0.32889100D+00,&
&        0.45728000D+00,0.28890800D+00,0.47433000D-01,-.35220000D-02,0.30000000D-05,&
&        0.30000000D-04,0.17200000D-03,0.76200000D-03,0.27260000D-02,0.79760000D-02,&
&        0.18287000D-01,0.30776000D-01,0.32373000D-01,-.99780000D-02,-.90529000D-01,&
&        -.13800400D+00,-.77965000D-01,0.22956000D+00,0.57612200D+00,0.40000000D-05,&
&        0.40000000D-04,0.22600000D-03,0.10130000D-02,0.35750000D-02,0.10612000D-01,&
&        0.23902000D-01,0.41279000D-01,0.41475000D-01,-.94580000D-02,-.12369500D+00,&
&        -.17439200D+00,-.12917000D+00,0.40034800D+00,0.66964600D+00,0.10000000D+01,&
&        0.41000000D-04,0.36900000D-03,0.21290000D-02,0.93690000D-02,0.33097000D-01,&
&        0.94431000D-01,0.20807700D+00,0.33233300D+00,0.33298700D+00,0.15684300D+00,&
&        0.21549000D-01,-.20950000D-02,-.17390000D-02,-.30000000D-03,0.29000000D-04,&
&        -.15000000D-04,-.13000000D-03,-.75100000D-03,-.33290000D-02,-.11912000D-01,&
&        -.34933000D-01,-.79989000D-01,-.13463600D+00,-.13859800D+00,0.30278000D-01,&
&        0.33321600D+00,0.45615300D+00,0.28505100D+00,0.46144000D-01,-.32490000D-02,&
&        0.30000000D-05,0.29000000D-04,0.16500000D-03,0.73400000D-03,0.26260000D-02,&
&        0.77250000D-02,0.17733000D-01,0.30055000D-01,0.31094000D-01,-.10048000D-01,&
&        -.88306000D-01,-.12982400D+00,-.76937000D-01,0.21266100D+00,0.57306100D+00,&
&        0.50000000D-05,0.42000000D-04,0.24100000D-03,0.10850000D-02,0.38310000D-02,&
&        0.11423000D-01,0.25792000D-01,0.44818000D-01,0.44598000D-01,-.11177000D-01,&
&        -.13813400D+00,-.18828500D+00,-.10739900D+00,0.44486300D+00,0.64023900D+00,&
&        0.10000000D+01,0.41000000D-04,0.36900000D-03,0.21280000D-02,0.93720000D-02,&
&        0.33155000D-01,0.94752000D-01,0.20909300D+00,0.33372200D+00,0.33220800D+00,&
&        0.15461300D+00,0.20902000D-01,-.20240000D-02,-.16970000D-02,-.28000000D-03,&
&        0.26000000D-04,-.15000000D-04,-.13100000D-03,-.75800000D-03,-.33630000D-02,&
&        -.12054000D-01,-.35424000D-01,-.81287000D-01,-.13690800D+00,-.13901900D+00,&
&        0.35468000D-01,0.33849800D+00,0.45443300D+00,0.27979300D+00,0.44776000D-01,&
&        -.31510000D-02,-.30000000D-05,-.29000000D-04,-.16700000D-03,-.74200000D-03,&
&        -.26620000D-02,-.78410000D-02,-.18051000D-01,-.30580000D-01,-.31312000D-01,&
&        0.11311000D-01,0.89990000D-01,0.13073300D+00,0.71808000D-01,-.22165800D+00,&
&        -.57102500D+00,0.50000000D-05,0.45000000D-04,0.25500000D-03,0.11440000D-02,&
&        0.40610000D-02,0.12095000D-01,0.27476000D-01,0.47557000D-01,0.47302000D-01,&
&        -.14418000D-01,-.15006200D+00,-.19909200D+00,-.79783000D-01,0.45903500D+00,&
&        0.61749500D+00,0.10000000D+01,0.41000000D-04,0.36300000D-03,0.20970000D-02,&
&        0.92500000D-02,0.32796000D-01,0.94004000D-01,0.20828000D+00,0.33365400D+00,&
&        0.33290400D+00,0.15537200D+00,0.20859000D-01,-.24400000D-02,-.19980000D-02,&
&        -.33800000D-03,0.35000000D-04,-.15000000D-04,-.12900000D-03,-.74900000D-03,&
&        -.33280000D-02,-.11947000D-01,-.35242000D-01,-.81204000D-01,-.13749300D+00,&
&        -.13922600D+00,0.36016000D-01,0.33912800D+00,0.45047200D+00,0.28178300D+00,&
&        0.47898000D-01,-.29870000D-02,0.30000000D-05,0.26000000D-04,0.15200000D-03,&
&        0.67800000D-03,0.24270000D-02,0.72010000D-02,0.16578000D-01,0.28392000D-01,&
&        0.28599000D-01,-.10132000D-01,-.82912000D-01,-.11599800D+00,-.72795000D-01,&
&        0.19564000D+00,0.56709900D+00,0.60000000D-05,0.53000000D-04,0.30500000D-03,&
&        0.13640000D-02,0.48760000D-02,0.14503000D-01,0.33296000D-01,0.57482000D-01,&
&        0.58702000D-01,-.19904000D-01,-.19469500D+00,-.23961300D+00,-.22320000D-02,&
&        0.52143500D+00,0.54554000D+00,0.10000000D+01,0.40000000D-04,0.36100000D-03,&
&        0.20830000D-02,0.91970000D-02,0.32660000D-01,0.93795000D-01,0.20827400D+00,&
&        0.33399300D+00,0.33249300D+00,0.15472800D+00,0.21271000D-01,-.16900000D-02,&
&        -.15160000D-02,-.24200000D-03,0.23000000D-04,-.15000000D-04,-.13100000D-03,&
&        -.75500000D-03,-.33590000D-02,-.12081000D-01,-.35703000D-01,-.82502000D-01,&
&        -.13989000D+00,-.14072900D+00,0.38766000D-01,0.34269500D+00,0.45231000D+00,&
&        0.27705400D+00,0.43885000D-01,-.28020000D-02,0.30000000D-05,0.25000000D-04,&
&        0.14700000D-03,0.65600000D-03,0.23510000D-02,0.70040000D-02,0.16131000D-01,&
&        0.27770000D-01,0.27567000D-01,-.10115000D-01,-.81009000D-01,-.11040900D+00,&
&        -.71732000D-01,0.18793000D+00,0.56462900D+00,0.50000000D-05,0.49000000D-04,&
&        0.27800000D-03,0.12530000D-02,0.44470000D-02,0.13370000D-01,0.30469000D-01,&
&        0.53447000D-01,0.52639000D-01,-.16881000D-01,-.17944800D+00,-.20958800D+00,&
&        -.39633000D-01,0.50213000D+00,0.58111100D+00,0.10000000D+01,0.41000000D-04,&
&        0.36100000D-03,0.20880000D-02,0.92210000D-02,0.32773000D-01,0.94179000D-01,&
&        0.20913200D+00,0.33456900D+00,0.33035900D+00,0.15234700D+00,0.22984000D-01,&
&        0.16070000D-02,0.46800000D-03,0.66000000D-04,-.20000000D-05,-.15000000D-04,&
&        -.13500000D-03,-.78200000D-03,-.34780000D-02,-.12520000D-01,-.37016000D-01,&
&        -.85559000D-01,-.14471800D+00,-.14344200D+00,0.43595000D-01,0.34888800D+00,&
&        0.45386500D+00,0.26859400D+00,0.38868000D-01,-.24920000D-02,0.30000000D-05,&
&        0.25000000D-04,0.14400000D-03,0.64500000D-03,0.23110000D-02,0.68980000D-02,&
&        0.15882000D-01,0.27350000D-01,0.26621000D-01,-.10858000D-01,-.79853000D-01,&
&        -.10612700D+00,-.68883000D-01,0.18438500D+00,0.56178800D+00,0.50000000D-05,&
&        0.42000000D-04,0.23800000D-03,0.10880000D-02,0.38210000D-02,0.11644000D-01,&
&        0.26167000D-01,0.46750000D-01,0.43309000D-01,-.13429000D-01,-.15389700D+00,&
&        -.16741300D+00,-.84995000D-01,0.45081300D+00,0.64086900D+00,0.10000000D+01,&
&        0.15260000D-02,0.12748600D-01,0.63374200D-01,0.20657750D+00,0.40929630D+00,&
&        0.39191830D+00,0.10294410D+00,-.72030000D-03,0.20950000D-02,-.32900000D-03,&
&        -.58030000D-03,-.48647000D-02,-.24839400D-01,-.84175900D-01,-.18008850D+00,&
&        -.15855550D+00,0.23553760D+00,0.58205870D+00,0.33666190D+00,0.17191200D-01,&
&        0.95000000D-04,0.78320000D-03,0.40855000D-02,0.13598700D-01,0.30269500D-01,&
&        0.24179000D-01,-.42377700D-01,-.12656610D+00,-.49944400D-01,0.44941990D+00,&
&        0.10000000D+01,0.14591000D-02,0.12217600D-01,0.61049000D-01,0.20080390D+00,&
&        0.40389420D+00,0.39700270D+00,0.11054810D+00,0.76800000D-04,0.21263000D-02,&
&        -.37440000D-03,-.56300000D-03,-.47354000D-02,-.24264300D-01,-.83090000D-01,&
&        -.18002470D+00,-.16632950D+00,0.21937170D+00,0.58202390D+00,0.34777200D+00,&
&        0.19245500D-01,0.11150000D-03,0.92120000D-03,0.48273000D-02,0.16227200D-01,&
&        0.36635400D-01,0.30786700D-01,-.48064300D-01,-.15598040D+00,-.63237000D-01,&
&        0.50408190D+00,0.10000000D+01,0.14097000D-02,0.11827700D-01,0.59328000D-01,&
&        0.19651150D+00,0.39978910D+00,0.40046530D+00,0.11641960D+00,0.69180000D-03,&
&        0.21633000D-02,-.41500000D-03,-.55190000D-03,-.46550000D-02,-.23917600D-01,&
&        -.82562700D-01,-.18067910D+00,-.17248480D+00,0.20867000D+00,0.58236220D+00,&
&        0.35374650D+00,0.20643900D-01,0.12360000D-03,0.10240000D-02,0.53805000D-02,&
&        0.18244300D-01,0.41597900D-01,0.36299800D-01,-.52356900D-01,-.17916670D+00,&
&        -.74047700D-01,0.53580940D+00,0.10000000D+01,0.14127000D-02,0.11858800D-01,&
&        0.59515300D-01,0.19722010D+00,0.40074390D+00,0.39947400D+00,0.11533640D+00,&
&        0.22190000D-03,0.22838000D-02,-.47560000D-03,-.56100000D-03,-.47340000D-02,&
&        -.24350400D-01,-.84107100D-01,-.18413840D+00,-.17350040D+00,0.21672630D+00,&
&        0.58500990D+00,0.34168160D+00,0.19912500D-01,0.13660000D-03,0.11308000D-02,&
&        0.59581000D-02,0.20186600D-01,0.46193900D-01,0.39405000D-01,-.59284600D-01,&
&        -.20146630D+00,-.68782100D-01,0.55959440D+00,0.10000000D+01,0.14448000D-02,&
&        0.12128800D-01,0.60807700D-01,0.20093580D+00,0.40474190D+00,0.39571510D+00,&
&        0.11022130D+00,-.90900000D-03,0.24832000D-02,-.57440000D-03,-.58190000D-03,&
&        -.49065000D-02,-.25251400D-01,-.86944500D-01,-.18934220D+00,-.17108820D+00,&
&        0.23687550D+00,0.58984000D+00,0.31719440D+00,0.17983300D-01,0.15180000D-03,&
&        0.12563000D-02,0.66224000D-02,0.22381600D-01,0.50971700D-01,0.41400900D-01,&
&        -.70397000D-01,-.22325400D+00,-.56417900D-01,0.58080790D+00,0.10000000D+01,&
&        0.15195000D-02,0.12742400D-01,0.63646500D-01,0.20856350D+00,0.41224230D+00,&
&        0.38781030D+00,0.10038200D+00,-.25078000D-02,0.27139000D-02,-.69770000D-03,&
&        -.62080000D-03,-.52212000D-02,-.26846300D-01,-.91582300D-01,-.19681640D+00,&
&        -.16347500D+00,0.27382040D+00,0.59815920D+00,0.27504530D+00,0.12770600D-01,&
&        0.17010000D-03,0.14064000D-02,0.73963000D-02,0.24825400D-01,0.55715500D-01,&
&        0.41213200D-01,-.87605700D-01,-.24405860D+00,-.29500700D-01,0.60122950D+00,&
&        0.10000000D+01/)
      real(8) :: expd(137)= &
&     (/ 0.1239000D+00,0.2380000D+00,0.3430000D+00,0.5500000D+00,0.8170000D+00,&
&        0.1185000D+01,0.1640000D+01,0.2202000D+01,0.9730000D-01,0.1870000D+00,&
&        0.1890000D+00,0.2750000D+00,0.3730000D+00,0.4790000D+00,0.6000000D+00,&
&        0.7380000D+00,0.1031820D+02,0.2592420D+01,0.7617000D+00,0.2083800D+00,&
&        0.5370000D-01,0.5051380D+02,0.1474050D+02,0.5195000D+01,0.2028460D+01,&
&        0.8040860D+00,0.3076890D+00,0.1113920D+00,0.3735200D-01,0.6401300D+02,&
&        0.1881790D+02,0.6728700D+01,0.2664130D+01,0.1078680D+01,0.4232090D+00,&
&        0.1559990D+00,0.5188400D-01,0.7761150D+02,0.2291590D+02,0.8279540D+01,&
&        0.3309930D+01,0.1358630D+01,0.5413500D+00,0.2023560D+00,0.6756800D-01,&
&        0.8857680D+02,0.2620450D+02,0.9517470D+01,0.3822480D+01,0.1575120D+01,&
&        0.6289280D+00,0.2344240D+00,0.7681500D-01,0.1006630D+03,0.2983360D+02,&
&        0.1088940D+02,0.4393580D+01,0.1817820D+01,0.7278270D+00,0.2712950D+00,&
&        0.8830900D-01,0.1133440D+03,0.3364140D+02,0.1233100D+02,0.4994780D+01,&
&        0.2072800D+01,0.8307530D+00,0.3091780D+00,0.1001300D+00,0.1262640D+03,&
&        0.3752260D+02,0.1380210D+02,0.5609270D+01,0.2333690D+01,0.9364150D+00,&
&        0.3482370D+00,0.1123530D+00,0.1402527D+03,0.4172610D+02,0.1539810D+02,&
&        0.6277100D+01,0.2618500D+01,0.1052600D+01,0.3916000D+00,0.1262000D+00,&
&        0.1738970D+03,0.5188690D+02,0.1934190D+02,0.7975720D+01,0.3398230D+01,&
&        0.1409320D+01,0.5488580D+00,0.1901990D+00,0.2056177D+03,0.6144981D+02,&
&        0.2305689D+02,0.9577739D+01,0.4133734D+01,0.1747518D+01,0.6995600D+00,&
&        0.2516080D+00,0.6533700D+02,0.1849700D+02,0.6315000D+01,0.2163500D+01,&
&        0.6667500D+00,0.1884000D+00,0.7476200D+02,0.2130200D+02,0.7343600D+01,&
&        0.2565100D+01,0.8197000D+00,0.2470000D+00,0.8442423D+02,0.2418159D+02,&
&        0.8401777D+01,0.2980502D+01,0.9790030D+00,0.3098000D+00,0.9447200D+02,&
&        0.2718000D+02,0.9506800D+01,0.3416800D+01,0.1147900D+01,0.3682000D+00,&
&        0.1048300D+03,0.3027200D+02,0.1064900D+02,0.3869600D+01,0.1323900D+01,&
&        0.4098000D+00,0.1155253D+03,0.3346525D+02,0.1183046D+02,0.4339771D+01,&
&        0.1507524D+01,0.5030000D+00/)
      real(8) :: coeffd(191)= &
&     (/ 0.32849000D-01,0.14819200D+00,0.31092100D+00,0.45219500D+00,0.10000000D+01,&
&        0.42660000D-02,0.27708000D-01,0.10000100D+00,0.23158100D+00,0.34603300D+00,&
&        0.37337400D+00,0.26428800D+00,-.43890000D-02,-.28363000D-01,-.10513700D+00,&
&        -.23485400D+00,-.32460900D+00,-.64289000D-01,0.60174900D+00,0.10000000D+01,&
&        0.38870000D-02,0.26399000D-01,0.97511000D-01,0.23284800D+00,0.35315200D+00,&
&        0.37218600D+00,0.24767200D+00,-.39700000D-02,-.26873000D-01,-.10227500D+00,&
&        -.23772800D+00,-.31211400D+00,-.42378000D-01,0.58865800D+00,0.10000000D+01,&
&        0.35950000D-02,0.25210000D-01,0.94786000D-01,0.23036300D+00,0.35289400D+00,&
&        0.37041400D+00,0.24571800D+00,-.38180000D-02,-.26717000D-01,-.10369000D+00,&
&        -.24768900D+00,-.31152300D+00,-.22827000D-01,0.56972600D+00,0.10000000D+01,&
&        0.36210000D-02,0.25766000D-01,0.97556000D-01,0.23631200D+00,0.35828600D+00,&
&        0.36854300D+00,0.23549400D+00,-.41220000D-02,-.29307000D-01,-.11506200D+00,&
&        -.27306800D+00,-.31442300D+00,0.42097000D-01,0.59140300D+00,0.10000000D+01,&
&        0.35790000D-02,0.25827000D-01,0.98559000D-01,0.23832700D+00,0.35870700D+00,&
&        0.36509200D+00,0.23373800D+00,-.34540000D-02,-.24925000D-01,-.97635000D-01,&
&        -.23669200D+00,-.29235000D+00,-.49730000D-02,0.50658800D+00,0.10000000D+01,&
&        0.35300000D-02,0.25784000D-01,0.99119000D-01,0.23907300D+00,0.35719900D+00,&
&        0.36218800D+00,0.23646100D+00,-.38900000D-02,-.28442000D-01,-.11242900D+00,&
&        -.27425700D+00,-.31554600D+00,0.57109000D-01,0.56360400D+00,0.10000000D+01,&
&        0.35100000D-02,0.25884000D-01,0.10005800D+00,0.24054700D+00,0.35684300D+00,&
&        0.35957900D+00,0.23662900D+00,-.40670000D-02,-.30053000D-01,-.11962000D+00,&
&        -.29151300D+00,-.31804800D+00,0.91698000D-01,0.56082300D+00,0.10000000D+01,&
&        0.33760000D-02,0.25141000D-01,0.97746000D-01,0.23470900D+00,0.34694500D+00,&
&        0.35106800D+00,0.25025500D+00,-.34950000D-02,-.26015000D-01,-.10387600D+00,&
&        -.25207000D+00,-.29458000D+00,0.11520000D-02,0.43858900D+00,0.10000000D+01,&
&        0.27000000D-02,0.20909000D-01,0.84408000D-01,0.21399900D+00,0.33598000D+00,&
&        0.35730100D+00,0.26457800D+00,-.33630000D-02,-.26079000D-01,-.10823100D+00,&
&        -.28221700D+00,-.34719000D+00,0.26711000D-01,0.49204700D+00,0.10000000D+01,&
&        0.23420000D-02,0.18606000D-01,0.77102000D-01,0.20202600D+00,0.32945400D+00,&
&        0.36097600D+00,0.27165700D+00,0.32790000D-02,0.26176000D-01,0.11136700D+00,&
&        0.30458100D+00,0.38629900D+00,-.58375000D-01,-.53887600D+00,0.10000000D+01,&
&        0.27382500D-01,0.15108050D+00,0.37492170D+00,0.47507990D+00,0.29827500D+00,&
&        0.10000000D+01,0.25768400D-01,0.14544210D+00,0.37137210D+00,0.48000020D+00,&
&        0.28968000D+00,0.10000000D+01,0.24528800D-01,0.14113400D+00,0.36875790D+00,&
&        0.48406260D+00,0.28244340D+00,0.10000000D+01,0.23498200D-01,0.13751830D+00,&
&        0.36648240D+00,0.48747170D+00,0.27657690D+00,0.10000000D+01,0.22658300D-01,&
&        0.13458950D+00,0.36471810D+00,0.49041960D+00,0.27138850D+00,0.10000000D+01,&
&        0.21955700D-01,0.13216200D+00,0.36334840D+00,0.49295820D+00,0.26675600D+00,&
&        0.10000000D+01/)
      real(8) :: expf(20)= &
&     (/ 0.7126000D+00,0.1636000D+00,0.1227400D+01,0.2788000D+00,0.1748800D+01,&
&        0.4057000D+00,0.2221100D+01,0.5231000D+00,0.2703200D+01,0.6438000D+00,&
&        0.3224300D+01,0.7758000D+00,0.3772400D+01,0.9170000D+00,0.4345500D+01,&
&        0.1068000D+01,0.5028600D+01,0.1259400D+01,0.5734400D+01,0.1461500D+01/)
      real(8) :: coefff(20)= &
&     (/ 0.3617450D+00,0.8218680D+00,0.3581580D+00,0.8257940D+00,0.3900680D+00,&
&        0.8008410D+00,0.4235450D+00,0.7741140D+00,0.4267760D+00,0.7697990D+00,&
&        0.4222490D+00,0.7714680D+00,0.4239660D+00,0.7684290D+00,0.4174290D+00,&
&        0.7714830D+00,0.4242800D+00,0.7630250D+00,0.4311320D+00,0.7546420D+00/)
!
      select case(numatomic(iatom))
!
! Set H - He functions
!
        case(1:2)
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+3)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
! Set Li - Ne functions
!
        case(3:10)
          ishell= ishell+1
          do j= 1,8
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 8
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+8
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,8
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+8)
          enddo
          mprim(ishell)= 8
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+8
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j+8)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+16)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+3)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expd(ide(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= one
          enddo
          mprim(ishell)= 1
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1) = locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1) = locbf(ishell)+6
          endif
!
! Set Na - Ar functions
!
        case(11:18)
! S functions
          ishell= ishell+1
          do j= 1,11
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 11
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+11
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,11
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+11)
          enddo
          mprim(ishell)= 11
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+11
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,11
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+22)
          enddo
          mprim(ishell)= 11
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+11
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j+11)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+33)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
! P functions
          ishell= ishell+1
          do j= 1,7
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 7
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+7
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,7
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+7)
          enddo
          mprim(ishell)= 7
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+7
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j+7)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+14)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
! D functions
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expd(ide(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= one
          enddo
          mprim(ishell)= 1
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1) = locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1) = locbf(ishell)+6
          endif
!
! Set Ca, Ga - Kr functions
!
        case(20,31:36)
! S functions
          ishell= ishell+1
          do j= 1,13
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 13
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+13
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,13
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+13)
          enddo
          mprim(ishell)= 13
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+13
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,13
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+26)
          enddo
          mprim(ishell)= 13
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+13
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,13
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+39)
          enddo
          mprim(ishell)= 13
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+13
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j+13)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+52)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
! P functions
          ishell= ishell+1
          do j= 1,10
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 10
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+10
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,10
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+10)
          enddo
          mprim(ishell)= 10
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+10
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,10
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+20)
          enddo
          mprim(ishell)= 10
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+10
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j+10)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+30)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
! D functions
          ishell= ishell+1
          num= 4
          if(numatomic(iatom) >= 31) num= 5
          do j= 1,num
            ex(locprim(ishell)+j)= expd(ide(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffd(idc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= num
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+num
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1) = locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1) = locbf(ishell)+6
          endif
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expd(ide(numatomic(iatom))+j+num)
            coeff(locprim(ishell)+j)= coeffd(idc(numatomic(iatom))+j+num)
          enddo
          mprim(ishell)= 1
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1) = locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1) = locbf(ishell)+6
          endif
!
! Set Sc - Zn functions
!
        case(21:30)
! S functions
          ishell= ishell+1
          do j= 1,19
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 19
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+19
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,19
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+19)
          enddo
          mprim(ishell)= 19
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+19
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,19
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+38)
          enddo
          mprim(ishell)= 19
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+19
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,19
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+57)
          enddo
          mprim(ishell)= 19
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+19
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,19
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+76)
          enddo
          mprim(ishell)= 19
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+19
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j+19)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+95)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
! P functions
          ishell= ishell+1
          do j= 1,15
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 15
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+15
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,15
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+15)
          enddo
          mprim(ishell)= 15
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+15
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,15
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+30)
          enddo
          mprim(ishell)= 15
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+15
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,15
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+45)
          enddo
          mprim(ishell)= 15
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+15
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j+15)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+60)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
! D functions
          ishell= ishell+1
          do j= 1,7
            ex(locprim(ishell)+j)= expd(ide(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffd(idc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 7
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+7
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1)= locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1)= locbf(ishell)+6
          endif
!
          ishell= ishell+1
          do j= 1,7
            ex(locprim(ishell)+j)= expd(ide(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffd(idc(numatomic(iatom))+j+7)
          enddo
          mprim(ishell)= 7
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+7
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1)= locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1)= locbf(ishell)+6
          endif
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expd(ide(numatomic(iatom))+j+7)
            coeff(locprim(ishell)+j)= coeffd(idc(numatomic(iatom))+j+14)
          enddo
          mprim(ishell)= 1
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1)= locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1)= locbf(ishell)+6
          endif
! F functions
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= expf(ife(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coefff(ifc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 2
          mtype(ishell)= 3
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          if(spher) then
            mbf(ishell)= 7
            locbf(ishell+1)= locbf(ishell)+7
          else
            mbf(ishell)= 10
            locbf(ishell+1)= locbf(ishell)+10
          endif
        case(:0)
        case default
          write(*,'(" Error! This program supports H - Ar, Ca - Kr cc-PVDZ basis set.")')
          call iabort
      endselect
!
      if(ishell > mxshell) then
        write(*,'(" Error! The number of basis shells exceeds mxshell",i6,".")')mxshell
        call iabort
      endif
      if(locprim(ishell+1) > mxprim ) then
        write(*,'(" Error! The number of primitive basis functions exceeds mxprim", &
&             i6,".")')mxprim
        call iabort
      endif
      if(locbf(ishell+1) > mxao ) then
        write(*,'(" Error! The number of basis functions exceeds mxao",i6,".")')mxao
        call iabort
      endif
      return
end


!------------------------------------
  subroutine bsccpvtz(iatom,ishell)
!------------------------------------
!
! Set basis functions of cc-pVTZ
!
      use modmolecule, only : numatomic
      use modbasis, only : ex, coeff, locprim, locbf, locatom, mprim, mbf, mtype, spher
      use modparam, only : mxao, mxshell, mxprim
      implicit none
      integer,intent(in) :: iatom
      integer,intent(inout) :: ishell
      integer :: j, k, num
      integer :: ise(36)=(/0,5, 11,22,33,43,53,63,73,83, 93,109,124,139,154,169,184,199, 214,214,&
&                          234,254,274,294,314,334,354,374,394,414,434,454,474,494,514,534/)
      integer :: isc(36)=(/0,5, 11,31,51,69,87,105,123,141, 159,203,244,285,326,367,408,449, &
&                          490,490,564,679,794,909,1024,1139,1254,1369,1484,1599, &
&                          1714,1788,1862,1936,2010,2084/)
      integer :: ipe(36)=(/0,2, 4,9,14,19,24,29,34,39, 44,54,64,73,82,91,100,109, 118,118, &
&                          132,148,164,180,196,212,228,244,260,276,292,305,318,331,344,357/)
      integer :: ipc(36)=(/0,2, 4,9,14,19,24,29,34,39, 44,62,80,96,112,128,144,160, &
&                          176,176,214,290,366,442,518,594,670,746,822,898, &
&                          974,1009,1044,1079,1114,1149/)
      integer :: ide(36)=(/0,1, 2,4,6,8,10,12,14,16, 18,20,22,24,26,28,30,32, &
&                          34,34,40,48,56,64,72,80,88,96,104,112,120,129,138,147,156,165/)
      integer :: idc(20:36)=(/0,6,28,50,72,94,116,138,160,182,204,226,235,244,253,262,271/)
      integer :: ife(36)=(/0,0, 0,1,2,3,4,5,6,7, 8,9,10,11,12,13,14,15, &
&                          16,16,17,19,21,23,25,27,29,31,33,35,37,38,39,40,41,42/)
      integer :: ige(21:30)=(/0,1,2,3,4,5,6,7,8,9/)
      real(8),parameter :: one=1.0D+00
      real(8) :: exps(554)= &
&     (/ 0.3387000D+02,0.5095000D+01,0.1159000D+01,0.3258000D+00,0.1027000D+00,&
&        0.2340000D+03,0.3516000D+02,0.7989000D+01,0.2212000D+01,0.6669000D+00,&
&        0.2089000D+00,0.5988000D+04,0.8989000D+03,0.2059000D+03,0.5924000D+02,&
&        0.1987000D+02,0.7406000D+01,0.2930000D+01,0.1189000D+01,0.4798000D+00,&
&        0.7509000D-01,0.2832000D-01,0.6863000D+04,0.1030000D+04,0.2347000D+03,&
&        0.6656000D+02,0.2169000D+02,0.7734000D+01,0.2916000D+01,0.1130000D+01,&
&        0.1101000D+00,0.2577000D+00,0.4409000D-01,0.5473000D+04,0.8209000D+03,&
&        0.1868000D+03,0.5283000D+02,0.1708000D+02,0.5999000D+01,0.2208000D+01,&
&        0.2415000D+00,0.5879000D+00,0.8610000D-01,0.8236000D+04,0.1235000D+04,&
&        0.2808000D+03,0.7927000D+02,0.2559000D+02,0.8997000D+01,0.3319000D+01,&
&        0.3643000D+00,0.9059000D+00,0.1285000D+00,0.1142000D+05,0.1712000D+04,&
&        0.3893000D+03,0.1100000D+03,0.3557000D+02,0.1254000D+02,0.4644000D+01,&
&        0.5118000D+00,0.1293000D+01,0.1787000D+00,0.1533000D+05,0.2299000D+04,&
&        0.5224000D+03,0.1473000D+03,0.4755000D+02,0.1676000D+02,0.6207000D+01,&
&        0.6882000D+00,0.1752000D+01,0.2384000D+00,0.1950000D+05,0.2923000D+04,&
&        0.6645000D+03,0.1875000D+03,0.6062000D+02,0.2142000D+02,0.7950000D+01,&
&        0.8815000D+00,0.2257000D+01,0.3041000D+00,0.2435000D+05,0.3650000D+04,&
&        0.8296000D+03,0.2340000D+03,0.7561000D+02,0.2673000D+02,0.9927000D+01,&
&        0.1102000D+01,0.2836000D+01,0.3782000D+00,0.4230000D+06,0.6334000D+05,&
&        0.1441000D+05,0.4077000D+04,0.1328000D+04,0.4786000D+03,0.1862000D+03,&
&        0.7692000D+02,0.3332000D+02,0.1500000D+02,0.6869000D+01,0.2683000D+01,&
&        0.1109000D+01,0.6015000D-01,0.4540000D+00,0.2382000D-01,0.1649000D+06,&
&        0.2471000D+05,0.5628000D+04,0.1596000D+04,0.5210000D+03,0.1880000D+03,&
&        0.7301000D+02,0.2990000D+02,0.1254000D+02,0.4306000D+01,0.1826000D+01,&
&        0.7417000D+00,0.7612000D-01,0.1457000D+00,0.3310000D-01,0.2055000D+06,&
&        0.3078000D+05,0.7006000D+04,0.1985000D+04,0.6491000D+03,0.2350000D+03,&
&        0.9162000D+02,0.3767000D+02,0.1591000D+02,0.5850000D+01,0.2542000D+01,&
&        0.1057000D+01,0.1455000D+00,0.2931000D+00,0.5650000D-01,0.2549000D+06,&
&        0.3819000D+05,0.8690000D+04,0.2462000D+04,0.8048000D+03,0.2913000D+03,&
&        0.1136000D+03,0.4675000D+02,0.1982000D+02,0.7708000D+01,0.3340000D+01,&
&        0.1402000D+01,0.2070000D+00,0.4387000D+00,0.7944000D-01,0.3124000D+06,&
&        0.4680000D+05,0.1065000D+05,0.3018000D+04,0.9868000D+03,0.3574000D+03,&
&        0.1396000D+03,0.5763000D+02,0.2460000D+02,0.1012000D+02,0.4283000D+01,&
&        0.1805000D+01,0.2782000D+00,0.6158000D+00,0.1055000D+00,0.3741000D+06,&
&        0.5605000D+05,0.1276000D+05,0.3615000D+04,0.1183000D+04,0.4288000D+03,&
&        0.1678000D+03,0.6947000D+02,0.2984000D+02,0.1272000D+02,0.5244000D+01,&
&        0.2219000D+01,0.3490000D+00,0.7767000D+00,0.1322000D+00,0.4561000D+06,&
&        0.6833000D+05,0.1555000D+05,0.4405000D+04,0.1439000D+04,0.5204000D+03,&
&        0.2031000D+03,0.8396000D+02,0.3620000D+02,0.1583000D+02,0.6334000D+01,&
&        0.2694000D+01,0.4313000D+00,0.9768000D+00,0.1625000D+00,0.5450000D+06,&
&        0.8164000D+05,0.1858000D+05,0.5261000D+04,0.1717000D+04,0.6199000D+03,&
&        0.2416000D+03,0.9979000D+02,0.4315000D+02,0.1914000D+02,0.7488000D+01,&
&        0.3205000D+01,0.5204000D+00,0.1196000D+01,0.1954000D+00,0.2402654D+07,&
&        0.3597898D+06,0.8187809D+05,0.2319089D+05,0.7565212D+04,0.2730702D+04,&
&        0.1064640D+04,0.4410605D+03,0.1917269D+03,0.8653774D+02,0.3989924D+02,&
&        0.1764065D+02,0.8359990D+01,0.3951330D+01,0.1713400D+01,0.8108600D+00,&
&        0.3602500D+00,0.4484000D-01,0.8108000D-01,0.2143000D-01,0.2715278D+07,&
&        0.4065984D+06,0.9253004D+05,0.2620792D+05,0.8549429D+04,0.3085975D+04,&
&        0.1203172D+04,0.4984869D+03,0.2167360D+03,0.9787476D+02,0.4520433D+02,&
&        0.2021187D+02,0.9574751D+01,0.4540346D+01,0.1995687D+01,0.9422150D+00,&
&        0.4178450D+00,0.9576100D-01,0.5135100D-01,0.2387800D-01,0.3014643D+07,&
&        0.4514329D+06,0.1027338D+06,0.2909817D+05,0.9492330D+04,0.3426346D+04,&
&        0.1335896D+04,0.5535026D+03,0.2406925D+03,0.1087293D+03,0.5026457D+02,&
&        0.2258004D+02,0.1071432D+02,0.5093546D+01,0.2244183D+01,0.1059570D+01,&
&        0.4688490D+00,0.1061430D+00,0.5526200D-01,0.2546500D-01,0.3321857D+07,&
&        0.4974356D+06,0.1132027D+06,0.3206333D+05,0.1045962D+05,0.3775506D+04,&
&        0.1472040D+04,0.6099331D+03,0.2652634D+03,0.1198607D+03,0.5544891D+02,&
&        0.2498372D+02,0.1188056D+02,0.5660311D+01,0.2495703D+01,0.1177866D+01,&
&        0.5200440D+00,0.1159650D+00,0.5893800D-01,0.2694600D-01,0.6177194D+07,&
&        0.9249295D+06,0.2104865D+06,0.5962005D+05,0.1945076D+05,0.7022056D+04,&
&        0.2738763D+04,0.1135814D+04,0.4950923D+03,0.2247487D+03,0.1053836D+03,&
&        0.5019359D+02,0.2224957D+02,0.1098265D+02,0.5383665D+01,0.2343685D+01,&
&        0.1105202D+01,0.4878480D+00,0.8959900D-01,0.3342300D-01,0.3960805D+07,&
&        0.5931155D+06,0.1349768D+06,0.3823067D+05,0.1247154D+05,0.4501743D+04,&
&        0.1755212D+04,0.7273039D+03,0.3163678D+03,0.1430098D+03,0.6621805D+02,&
&        0.2991896D+02,0.1430318D+02,0.6839451D+01,0.3012374D+01,0.1418808D+01,&
&        0.6236240D+00,0.1340980D+00,0.6554800D-01,0.2958400D-01,0.4316265D+07,&
&        0.6463424D+06,0.1470897D+06,0.4166152D+05,0.1359077D+05,0.4905750D+04,&
&        0.1912746D+04,0.7926043D+03,0.3448065D+03,0.1558999D+03,0.7223091D+02,&
&        0.3272506D+02,0.1566762D+02,0.7503483D+01,0.3312223D+01,0.1558471D+01,&
&        0.6839140D+00,0.1467570D+00,0.7058300D-01,0.3144900D-01,0.4675675D+07,&
&        0.7001615D+06,0.1593373D+06,0.4513046D+05,0.1472238D+05,0.5314222D+04,&
&        0.2072018D+04,0.8586188D+03,0.3735497D+03,0.1689229D+03,0.7829639D+02,&
&        0.3552123D+02,0.1704144D+02,0.8173000D+01,0.3610318D+01,0.1697047D+01,&
&        0.7435320D+00,0.1583440D+00,0.7503600D-01,0.3309100D-01,0.5045991D+07,&
&        0.7556142D+06,0.1719568D+06,0.4870479D+05,0.1588841D+05,0.5735123D+04,&
&        0.2236137D+04,0.9266468D+03,0.4031743D+03,0.1823476D+03,0.8454885D+02,&
&        0.3839634D+02,0.1845859D+02,0.8863548D+01,0.3916227D+01,0.1838870D+01,&
&        0.8043620D+00,0.1697970D+00,0.7930600D-01,0.3467700D-01,0.5430321D+07,&
&        0.8131665D+06,0.1850544D+06,0.5241466D+05,0.1709868D+05,0.6171994D+04,&
&        0.2406481D+04,0.9972584D+03,0.4339289D+03,0.1962869D+03,0.9104280D+02,&
&        0.4138425D+02,0.1993278D+02,0.9581891D+01,0.4234516D+01,0.1985814D+01,&
&        0.8670830D+00,0.1813390D+00,0.8365700D-01,0.3626700D-01,0.5820021D+07,&
&        0.8715234D+06,0.1983350D+06,0.5617631D+05,0.1832582D+05,0.6614955D+04,&
&        0.2579199D+04,0.1068849D+04,0.4651045D+03,0.2104130D+03,0.9761629D+02,&
&        0.4438020D+02,0.2142308D+02,0.1030891D+02,0.4553645D+01,0.2132821D+01,&
&        0.9296970D+00,0.1921470D+00,0.8759500D-01,0.3770200D-01,0.6558157D+07,&
&        0.9820253D+06,0.2234677D+06,0.6328829D+05,0.2064294D+05,0.7450522D+04,&
&        0.2905074D+04,0.1204210D+04,0.5243045D+03,0.2374656D+03,0.1105787D+03,&
&        0.5137462D+02,0.2444085D+02,0.1176859D+02,0.5342119D+01,0.2495036D+01,&
&        0.1098773D+01,0.2601800D+00,0.1270790D+00,0.5440800D-01,0.7447967D+07,&
&        0.1115318D+07,0.2538426D+06,0.7191529D+05,0.2347018D+05,0.8477492D+04,&
&        0.3308391D+04,0.1372605D+04,0.5982201D+03,0.2713860D+03,0.1269780D+03,&
&        0.6022207D+02,0.2801858D+02,0.1351752D+02,0.6309406D+01,0.2904534D+01,&
&        0.1287556D+01,0.1677320D+00,0.3365520D+00,0.7106900D-01,0.8482340D+07,&
&        0.1270151D+07,0.2890570D+06,0.8187985D+05,0.2671656D+05,0.9647584D+04,&
&        0.3764119D+04,0.1561566D+04,0.6808147D+03,0.3092412D+03,0.1452574D+03,&
&        0.6973905D+02,0.3177032D+02,0.1539176D+02,0.7341526D+01,0.3323716D+01,&
&        0.1485867D+01,0.2115000D+00,0.4210860D+00,0.8897400D-01,0.9563600D+07,&
&        0.1432100D+07,0.3259100D+06,0.9231200D+05,0.3011600D+05,0.1087200D+05,&
&        0.4240100D+04,0.1758400D+04,0.7665900D+03,0.3484300D+03,0.1640300D+03,&
&        0.7914200D+02,0.3552400D+02,0.1730500D+02,0.8378400D+01,0.3740500D+01,&
&        0.1689000D+01,0.2552000D+00,0.5092700D+00,0.1065100D+00,0.1063900D+08,&
&        0.1593400D+07,0.3626100D+06,0.1027000D+06,0.3350100D+05,0.1209300D+05,&
&        0.4715900D+04,0.1955600D+04,0.8526100D+03,0.3876700D+03,0.1826800D+03,&
&        0.8824500D+02,0.3926300D+02,0.1923400D+02,0.9405700D+01,0.4160100D+01,&
&        0.1899500D+01,0.3011400D+00,0.6047200D+00,0.1251500D+00,0.1171811D+08,&
&        0.1754604D+07,0.3992813D+06,0.1130846D+06,0.3688593D+05,0.1331221D+05,&
&        0.5189988D+04,0.2151660D+04,0.9380325D+03,0.4265573D+03,0.2010666D+03,&
&        0.9709761D+02,0.4299872D+02,0.2117707D+02,0.1042675D+02,0.4585008D+01,&
&        0.2117603D+01,0.3492250D+00,0.7070570D+00,0.1448210D+00/)
      real(8) :: coeffs(2158)= &
&     (/ 0.6068000D-02,0.4530800D-01,0.2028220D+00,0.1000000D+01,0.1000000D+01,&
&        0.2587000D-02,0.1953300D-01,0.9099800D-01,0.2720500D+00,0.1000000D+01,&
&        0.1000000D+01,0.1330000D-03,0.1025000D-02,0.5272000D-02,0.2092900D-01,&
&        0.6634000D-01,0.1657750D+00,0.3150380D+00,0.3935230D+00,0.1908700D+00,&
&        -.2100000D-04,-.1610000D-03,-.8200000D-03,-.3326000D-02,-.1051900D-01,&
&        -.2809700D-01,-.5593600D-01,-.9923700D-01,-.1121890D+00,0.1000000D+01,&
&        0.1000000D+01,0.2360000D-03,0.1826000D-02,0.9452000D-02,0.3795700D-01,&
&        0.1199650D+00,0.2821620D+00,0.4274040D+00,0.2662780D+00,-.7275000D-02,&
&        -.4300000D-04,-.3330000D-03,-.1736000D-02,-.7012000D-02,-.2312600D-01,&
&        -.5813800D-01,-.1145560D+00,-.1359080D+00,0.5774410D+00,0.1000000D+01,&
&        0.1000000D+01,0.5550000D-03,0.4291000D-02,0.2194900D-01,0.8444100D-01,&
&        0.2385570D+00,0.4350720D+00,0.3419550D+00,-.9545000D-02,-.1120000D-03,&
&        -.8680000D-03,-.4484000D-02,-.1768300D-01,-.5363900D-01,-.1190050D+00,&
&        -.1658240D+00,0.5959810D+00,0.1000000D+01,0.1000000D+01,0.5310000D-03,&
&        0.4108000D-02,0.2108700D-01,0.8185300D-01,0.2348170D+00,0.4344010D+00,&
&        0.3461290D+00,-.8983000D-02,-.1130000D-03,-.8780000D-03,-.4540000D-02,&
&        -.1813300D-01,-.5576000D-01,-.1268950D+00,-.1703520D+00,0.5986840D+00,&
&        0.1000000D+01,0.1000000D+01,0.5230000D-03,0.4045000D-02,0.2077500D-01,&
&        0.8072700D-01,0.2330740D+00,0.4335010D+00,0.3474720D+00,-.8508000D-02,&
&        -.1150000D-03,-.8950000D-03,-.4624000D-02,-.1852800D-01,-.5733900D-01,&
&        -.1320760D+00,-.1725100D+00,0.5999440D+00,0.1000000D+01,0.1000000D+01,&
&        0.5080000D-03,0.3929000D-02,0.2024300D-01,0.7918100D-01,0.2306870D+00,&
&        0.4331180D+00,0.3502600D+00,-.8154000D-02,-.1150000D-03,-.8950000D-03,&
&        -.4636000D-02,-.1872400D-01,-.5846300D-01,-.1364630D+00,-.1757400D+00,&
&        0.6034180D+00,0.1000000D+01,0.1000000D+01,0.5070000D-03,0.3923000D-02,&
&        0.2020000D-01,0.7901000D-01,0.2304390D+00,0.4328720D+00,0.3499640D+00,&
&        -.7892000D-02,-.1170000D-03,-.9120000D-03,-.4717000D-02,-.1908600D-01,&
&        -.5965500D-01,-.1400100D+00,-.1767820D+00,0.6050430D+00,0.1000000D+01,&
&        0.1000000D+01,0.5020000D-03,0.3881000D-02,0.1999700D-01,0.7841800D-01,&
&        0.2296760D+00,0.4327220D+00,0.3506420D+00,-.7645000D-02,-.1180000D-03,&
&        -.9150000D-03,-.4737000D-02,-.1923300D-01,-.6036900D-01,-.1425080D+00,&
&        -.1777100D+00,0.6058360D+00,0.1000000D+01,0.1000000D+01,0.1806180D-04,&
&        0.1404300D-03,0.7384380D-03,0.3111820D-02,0.1120810D-01,0.3528280D-01,&
&        0.9598970D-01,0.2137350D+00,0.3486880D+00,0.3245660D+00,0.1126330D+00,&
&        0.7067970D-02,0.5980100D-03,-.5308700D-05,-.4406530D-05,-.3434430D-04,&
&        -.1801140D-03,-.7639000D-03,-.2752480D-02,-.8860160D-02,-.2479390D-01,&
&        -.6059950D-01,-.1164460D+00,-.1624370D+00,-.4388910D-01,0.3379170D+00,&
&        0.5613470D+00,0.4067540D-02,0.6630190D-06,0.5157690D-05,0.2712500D-04,&
&        0.1146350D-03,0.4151180D-03,0.1329780D-02,0.3755950D-02,0.9140250D-02,&
&        0.1798590D-01,0.2514770D-01,0.7635220D-02,-.6145890D-01,-.1157210D+00,&
&        0.6264060D+00,0.1000000D+01,0.1000000D+01,0.7299290D-04,0.5666520D-03,&
&        0.2962690D-02,0.1229620D-01,0.4273240D-01,0.1230130D+00,0.2748320D+00,&
&        0.4018180D+00,0.2646970D+00,0.3326120D-01,-.4413350D-02,0.2060240D-02,&
&        0.7081950D-03,-.1842480D-04,-.1435000D-03,-.7487100D-03,-.3144070D-02,&
&        -.1104810D-01,-.3360580D-01,-.8259460D-01,-.1593140D+00,-.1528880D+00,&
&        0.1908490D+00,0.5799640D+00,0.3720290D+00,-.1193440D-01,0.3551760D-05,&
&        0.2764200D-04,0.1444040D-03,0.6057440D-03,0.2135270D-02,0.6499340D-02,&
&        0.1614460D-01,0.3157660D-01,0.3163740D-01,-.4391400D-01,-.1510930D+00,&
&        -.2176680D+00,0.5472450D+00,0.1000000D+01,0.1000000D+01,0.6788360D-04,&
&        0.5271490D-03,0.2762030D-02,0.1147280D-01,0.3981880D-01,0.1150400D+00,&
&        0.2608870D+00,0.3963860D+00,0.2845970D+00,0.4445830D-01,-.4898380D-02,&
&        0.2612530D-02,0.7220680D-03,-.1763770D-04,-.1371950D-03,-.7189100D-03,&
&        -.3011460D-02,-.1060140D-01,-.3213450D-01,-.8031560D-01,-.1567940D+00,&
&        -.1683760D+00,0.1268790D+00,0.5614940D+00,0.4366130D+00,-.1145630D-01,&
&        0.4073150D-05,0.3165660D-04,0.1661160D-03,0.6949920D-03,0.2455110D-02,&
&        0.7445980D-02,0.1882530D-01,0.3727720D-01,0.4194960D-01,-.3543750D-01,&
&        -.1751320D+00,-.2762030D+00,0.6528090D+00,0.1000000D+01,0.1000000D+01,&
&        0.6251010D-04,0.4855530D-03,0.2545160D-02,0.1058660D-01,0.3687870D-01,&
&        0.1074790D+00,0.2479360D+00,0.3909270D+00,0.3020260D+00,0.5592360D-01,&
&        -.4024060D-02,0.2580300D-02,0.6079300D-03,-.1663700D-04,-.1293100D-03,&
&        -.6788280D-03,-.2841170D-02,-.1005510D-01,-.3057740D-01,-.7772560D-01,&
&        -.1542360D+00,-.1803680D+00,0.7982180D-01,0.5474410D+00,0.4801190D+00,&
&        -.1069960D-01,0.4262570D-05,0.3310620D-04,0.1740150D-03,0.7275740D-03,&
&        0.2583330D-02,0.7863540D-02,0.2021550D-01,0.4073200D-01,0.4993580D-01,&
&        -.2493960D-01,-.1903500D+00,-.3183500D+00,0.6811800D+00,0.1000000D+01,&
&        0.1000000D+01,0.5769600D-04,0.4482960D-03,0.2349390D-02,0.9782650D-02,&
&        0.3414670D-01,0.1002040D+00,0.2343720D+00,0.3824340D+00,0.3180880D+00,&
&        0.7077880D-01,-.1817990D-02,0.2161800D-02,0.4322970D-03,-.1567090D-04,&
&        -.1217240D-03,-.6396720D-03,-.2674260D-02,-.9498310D-02,-.2893490D-01,&
&        -.7451210D-01,-.1499380D+00,-.1894670D+00,0.3632700D-01,0.5288160D+00,&
&        0.5191150D+00,-.9256950D-02,0.4306310D-05,0.3341940D-04,0.1758850D-03,&
&        0.7343400D-03,0.2617750D-02,0.7978520D-02,0.2079400D-01,0.4244460D-01,&
&        0.5634360D-01,-.1273580D-01,-.1964950D+00,-.3535550D+00,0.7009120D+00,&
&        0.1000000D+01,0.1000000D+01,0.5421400D-04,0.4208550D-03,0.2206980D-02,&
&        0.9192580D-02,0.3211230D-01,0.9466830D-01,0.2236300D+00,0.3743930D+00,&
&        0.3291080D+00,0.8470380D-01,0.4408510D-03,0.1648270D-02,0.3013060D-03,&
&        -.1498370D-04,-.1161980D-03,-.6115830D-03,-.2553700D-02,-.9087080D-02,&
&        -.2770450D-01,-.7200200D-01,-.1464390D+00,-.1951500D+00,0.8191930D-02,&
&        0.5166010D+00,0.5421780D+00,-.9180720D-02,0.4350660D-05,0.3371400D-04,&
&        0.1776740D-03,0.7411160D-03,0.2645910D-02,0.8074870D-02,0.2122760D-01,&
&        0.4383230D-01,0.6127160D-01,-.3615100D-02,-.2045100D+00,-.3818710D+00,&
&        0.7141470D+00,0.1000000D+01,0.1000000D+01,0.4929700D-04,0.3830290D-03,&
&        0.2008540D-02,0.8385580D-02,0.2947030D-01,0.8783250D-01,0.2114730D+00,&
&        0.3653640D+00,0.3408840D+00,0.1021330D+00,0.3116750D-02,0.1057510D-02,&
&        0.1561360D-03,-.1383040D-04,-.1072790D-03,-.5650830D-03,-.2361350D-02,&
&        -.8458860D-02,-.2596380D-01,-.6863620D-01,-.1418740D+00,-.1993190D+00,&
&        -.1956620D-01,0.4997410D+00,0.5637360D+00,-.8350910D-02,0.4185460D-05,&
&        0.3243950D-04,0.1711050D-03,0.7141760D-03,0.2567050D-02,0.7885520D-02,&
&        0.2108670D-01,0.4422640D-01,0.6516700D-01,0.6030120D-02,-.2064950D+00,&
&        -.4058710D+00,0.7256610D+00,0.1000000D+01,0.1000000D+01,0.4558280D-04,&
&        0.3541080D-03,0.1857970D-02,0.7768510D-02,0.2742320D-01,0.8238360D-01,&
&        0.2012300D+00,0.3567810D+00,0.3495630D+00,0.1182660D+00,0.5601900D-02,&
&        0.4834730D-03,0.2920250D-04,-.1295510D-04,-.1004280D-03,-.5295830D-03,&
&        -.2213960D-02,-.7968450D-02,-.2458030D-01,-.6577980D-01,-.1379420D+00,&
&        -.2016300D+00,-.4128340D-01,0.4846800D+00,0.5792240D+00,-.7275530D-02,&
&        0.4049900D-05,0.3136910D-04,0.1656460D-03,0.6916620D-03,0.2497900D-02,&
&        0.7710740D-02,0.2087140D-01,0.4439650D-01,0.6802240D-01,0.1413500D-01,&
&        -.2074890D+00,-.4250450D+00,0.7336270D+00,0.1000000D+01,0.1000000D+01,&
&        0.93100000D-05,0.72390000D-04,0.38059000D-03,0.16045300D-02,0.58078000D-02,&
&        0.18595660D-01,0.52877760D-01,0.13015149D+00,0.25931471D+00,0.36149610D+00,&
&        0.26411160D+00,0.57093980D-01,-.18220000D-02,0.21116400D-02,-.97709000D-03,&
&        0.45581000D-03,-.19146000D-03,-.78740000D-04,-.27000000D-05,-.21020000D-04,&
&        -.11052000D-03,-.46666000D-03,-.16951300D-02,-.54834600D-02,-.15965970D-01,&
&        -.41513840D-01,-.92863870D-01,-.16531657D+00,-.17664074D+00,0.64444220D-01,&
&        0.51087962D+00,0.49463804D+00,0.87500940D-01,-.35910100D-02,0.24922000D-02,&
&        0.64587000D-03,0.93000000D-06,0.72500000D-05,0.38110000D-04,0.16101000D-03,&
&        0.58466000D-03,0.18950400D-02,0.55252500D-02,0.14470100D-01,0.32715810D-01,&
&        0.60031900D-01,0.67016830D-01,-.25936600D-01,-.26747467D+00,-.42697256D+00,&
&        0.67964050D-01,0.71020539D+00,0.44180057D+00,-.11869240D-01,-.22000000D-06,&
&        -.17300000D-05,-.91000000D-05,-.38440000D-04,-.13965000D-03,-.45251000D-03,&
&        -.13203800D-02,-.34584100D-02,-.78368600D-02,-.14415170D-01,-.16216480D-01,&
&        0.63442000D-02,0.67406260D-01,0.11447396D+00,-.26344770D-01,-.23369885D+00,&
&        -.31607538D+00,0.56111035D+00,0.10000000D+01,0.10000000D+01,0.81472210D-05,&
&        0.63347880D-04,0.33303840D-03,0.14040550D-02,0.50817250D-02,0.16269260D-01,&
&        0.46245770D-01,0.11372230D+00,0.22576360D+00,0.31067000D+00,0.21919060D+00,&
&        0.72158790D-01,0.11870300D+00,0.12205320D+00,0.21367950D-01,-.53572460D-03,&
&        0.24357740D-03,-.87966170D-04,0.78782460D-04,-.47221090D-05,-.36718290D-04,&
&        -.19308830D-03,-.81468700D-03,-.29555260D-02,-.95200350D-02,-.27468580D-01,&
&        -.69915280D-01,-.14992510D+00,-.24591530D+00,-.24012930D+00,0.35679870D-01,&
&        0.49150230D+00,0.49113810D+00,0.91206330D-01,-.53567230D-02,0.88128360D-03,&
&        -.76055360D-03,0.63401160D-03,0.91399050D-06,0.71085130D-05,0.37381260D-04,&
&        0.15788280D-03,0.57376860D-03,0.18592440D-02,0.54331820D-02,0.14253870D-01,&
&        0.32461440D-01,0.60034540D-01,0.69161050D-01,-.21130840D-01,-.26668320D+00,&
&        -.43675910D+00,0.64982430D-01,0.70095990D+00,0.45155620D+00,0.30119100D-01,&
&        -.13294800D-01,-.22019510D-06,-.17114190D-05,-.90084690D-05,-.37999970D-04,&
&        -.13832270D-03,-.44736920D-03,-.13106910D-02,-.34298600D-02,-.78475790D-02,&
&        -.14471890D-01,-.16906690D-01,0.53961150D-02,0.66710620D-01,0.11783560D+00,&
&        -.27381340D-01,-.22601490D+00,-.30735390D+00,0.25440540D+00,0.59815900D+00,&
&        -.37572380D-06,-.29819070D-05,-.15225860D-04,-.66846860D-04,-.23131290D-03,&
&        -.79597290D-03,-.21619610D-02,-.62064590D-02,-.12619050D-01,-.27394590D-01,&
&        -.23365160D-01,-.57346270D-02,0.15360250D+00,0.14471000D+00,0.93596990D-01,&
&        -.86877300D+00,0.21145970D-01,0.22754980D+01,-.11907700D+01,-.59627680D-06,&
&        -.48228530D-05,-.23950050D-04,-.10903350D-03,-.36019440D-03,-.13112330D-02,&
&        -.33224350D-02,-.10375200D-01,-.18921740D-01,-.47552870D-01,-.29316920D-01,&
&        -.30900880D-01,0.30745970D+00,0.18031000D+00,0.23581780D+00,-.23885440D+01,&
&        0.23294070D+01,0.99182520D+00,-.35072400D+01,0.10000000D+01,0.80607820D-05,&
&        0.62675180D-04,0.32950060D-03,0.13892030D-02,0.50284690D-02,0.16104190D-01,&
&        0.45812320D-01,0.11286130D+00,0.22481930D+00,0.31145710D+00,0.22249950D+00,&
&        0.72931280D-01,0.11606830D+00,0.11947740D+00,0.20978680D-01,-.50917150D-03,&
&        0.22178590D-03,-.76368960D-04,0.77195390D-04,-.46304860D-05,-.36004510D-04,&
&        -.18934200D-03,-.79887810D-03,-.28986980D-02,-.93397010D-02,-.26974640D-01,&
&        -.68789130D-01,-.14810370D+00,-.24452530D+00,-.24199160D+00,0.31837900D-01,&
&        0.49326860D+00,0.49396550D+00,0.91963130D-01,-.53169920D-02,0.80856240D-03,&
&        -.69184590D-03,0.60865120D-03,0.92305590D-06,0.71789740D-05,0.37751340D-04,&
&        0.15945320D-03,0.57951500D-03,0.18784140D-02,0.54927470D-02,0.14432970D-01,&
&        0.32964080D-01,0.61254930D-01,0.71341130D-01,-.19731500D-01,-.27418690D+00,&
&        -.44409770D+00,0.77760840D-01,0.70684440D+00,0.44138920D+00,0.27997690D-01,&
&        -.12107900D-01,-.21803230D-06,-.16948600D-05,-.89192080D-05,-.37636330D-04,&
&        -.13695750D-03,-.44328940D-03,-.12988680D-02,-.34067520D-02,-.78108290D-02,&
&        -.14492450D-01,-.17081360D-01,0.48976660D-02,0.67531080D-01,0.11733180D+00,&
&        -.29850250D-01,-.22776340D+00,-.29281150D+00,0.26653000D+00,0.59124060D+00,&
&        -.39751260D-06,-.31610800D-05,-.16093750D-04,-.70929470D-04,-.24427100D-03,&
&        -.84578920D-03,-.22822080D-02,-.66198730D-02,-.13350240D-01,-.29558300D-01,&
&        -.24770390D-01,-.84146240D-02,0.16938550D+00,0.15007870D+00,0.97877770D-01,&
&        -.96536080D+00,0.14897210D+00,0.21911790D+01,-.12433250D+01,-.63024830D-06,&
&        -.50845850D-05,-.25344820D-04,-.11482600D-03,-.38174970D-03,-.13796450D-02,&
&        -.35316660D-02,-.10922250D-01,-.20295540D-01,-.50244040D-01,-.33163090D-01,&
&        -.30980630D-01,0.32446730D+00,0.21556890D+00,0.14996160D+00,-.24207410D+01,&
&        0.25308740D+01,0.34236000D+00,-.27337220D+01,0.10000000D+01,0.80399990D-05,&
&        0.62514020D-04,0.32865530D-03,0.13856970D-02,0.50162170D-02,0.16069310D-01,&
&        0.45742420D-01,0.11285440D+00,0.22543440D+00,0.31404610D+00,0.22678190D+00,&
&        0.73340690D-01,0.11024740D+00,0.11313580D+00,0.19712950D-01,-.47190880D-03,&
&        0.18616060D-03,-.62085980D-04,0.72953140D-04,-.45030030D-05,-.35012950D-04,&
&        -.18413390D-03,-.77692160D-03,-.28195050D-02,-.90874860D-02,-.26271340D-01,&
&        -.67127260D-01,-.14511300D+00,-.24124830D+00,-.24163140D+00,0.30673620D-01,&
&        0.49704150D+00,0.49588750D+00,0.91818680D-01,-.53925140D-02,0.71023800D-03,&
&        -.63631280D-03,0.59799320D-03,0.93206480D-06,0.72493060D-05,0.38119670D-04,&
&        0.16102380D-03,0.58522100D-03,0.18975020D-02,0.55509090D-02,0.14605840D-01,&
&        0.33429740D-01,0.62357220D-01,0.73124350D-01,-.19114720D-01,-.28172490D+00,&
&        -.44881510D+00,0.92026960D-01,0.71101170D+00,0.43092740D+00,0.26045890D-01,&
&        -.11010490D-01,-.21589440D-06,-.16785190D-05,-.88312130D-05,-.37277690D-04,&
&        -.13560990D-03,-.43923510D-03,-.12869480D-02,-.33821490D-02,-.77656460D-02,&
&        -.14479850D-01,-.17155020D-01,0.46101010D-02,0.68278310D-01,0.11613680D+00,&
&        -.32770490D-01,-.22800000D+00,-.27939910D+00,0.27711650D+00,0.58529990D+00,&
&        -.40934160D-06,-.32589560D-05,-.16563900D-04,-.73166890D-04,-.25127840D-03,&
&        -.87326570D-03,-.23476540D-02,-.68531500D-02,-.13764200D-01,-.30846790D-01,&
&        -.25622080D-01,-.10051230D-01,0.17953300D+00,0.15224000D+00,0.94838870D-01,&
&        -.10148760D+01,0.23088100D+00,0.21133210D+01,-.12530480D+01,-.65399630D-06,&
&        -.52673990D-05,-.26320920D-04,-.11887190D-03,-.39685520D-03,-.14274600D-02,&
&        -.36791810D-02,-.11307740D-01,-.21282210D-01,-.52182710D-01,-.35990140D-01,&
&        -.30887590D-01,0.33806570D+00,0.23895590D+00,0.76090680D-01,-.24081130D+01,&
&        0.26335920D+01,-.53585100D-01,-.22301350D+01,0.10000000D+01,0.41286670D-05,&
&        0.32107670D-04,0.16884160D-03,0.71285200D-03,0.25893250D-02,0.83773500D-02,&
&        0.24417250D-01,0.63651350D-01,0.14276180D+00,0.25412750D+00,0.30095120D+00,&
&        0.17665130D+00,0.69367090D-01,0.11795790D+00,0.89161870D-01,0.11036300D-01,&
&        -.35460480D-03,0.10573110D-03,0.11146400D-04,-.23017720D-05,-.17895360D-04,&
&        -.94161740D-04,-.39750740D-03,-.14470250D-02,-.46946220D-02,-.13823870D-01,&
&        -.36746430D-01,-.86471850D-01,-.16967350D+00,-.25070890D+00,-.19611560D+00,&
&        0.14572440D+00,0.54667060D+00,0.39794340D+00,0.52770070D-01,-.43745370D-02,&
&        0.32040350D-03,-.51420770D-04,0.48629570D-06,0.37766450D-05,0.19906640D-04,&
&        0.83891640D-04,0.30657060D-03,0.99441070D-03,0.29619590D-02,0.79694730D-02,&
&        0.19550170D-01,0.40850350D-01,0.69290030D-01,0.61469840D-01,-.69813020D-01,&
&        -.35175970D+00,-.38286290D+00,0.26764010D+00,0.71759500D+00,0.30208140D+00,&
&        0.77495140D-02,-.11024510D-06,-.85302330D-06,-.45203580D-05,-.18916120D-04,&
&        -.69743440D-04,-.22378670D-03,-.67545030D-03,-.17893460D-02,-.44778580D-02,&
&        -.91401440D-02,-.16105620D-01,-.13348700D-01,0.14260270D-01,0.89316900D-01,&
&        0.88852790D-01,-.63687760D-01,-.27832620D+00,-.18300710D+00,0.67909370D+00,&
&        0.21798930D-06,0.16129400D-05,0.91118420D-05,0.35006450D-04,0.14353150D-03,&
&        0.40358960D-03,0.14251770D-02,0.31140090D-02,0.98144490D-02,0.14746980D-01,&
&        0.39115120D-01,0.91708880D-02,0.15598780D-01,-.28168440D+00,-.68952610D-02,&
&        -.17697810D+00,0.14430610D+01,-.10293180D+01,-.13076670D+01,-.36690100D-06,&
&        -.27683530D-05,-.15210140D-04,-.60662880D-04,-.23749590D-03,-.70770710D-03,&
&        -.23343890D-02,-.55550620D-02,-.15851420D-01,-.27378880D-01,-.61069750D-01,&
&        -.28872270D-01,0.68137140D-02,0.43276700D+00,0.19684100D+00,-.37646570D+00,&
&        -.20524730D+01,0.29757410D+01,-.22117050D+01,0.10000000D+01,0.82421270D-05,&
&        0.64085870D-04,0.33692530D-03,0.14206480D-02,0.51436830D-02,0.16485690D-01,&
&        0.46985600D-01,0.11624370D+00,0.23352770D+00,0.32928370D+00,0.24403040D+00,&
&        0.72198060D-01,0.76878060D-01,0.78522350D-01,0.12941090D-01,-.37848730D-03,&
&        -.25032030D-04,-.24215170D-04,0.34620710D-04,-.39360950D-05,-.30604810D-04,&
&        -.16096260D-03,-.67923480D-03,-.24661820D-02,-.79576290D-02,-.23072480D-01,&
&        -.59329560D-01,-.12994510D+00,-.22123520D+00,-.22925500D+00,0.35807330D-01,&
&        0.51076020D+00,0.50083070D+00,0.90118300D-01,-.69099090D-02,-.19129250D-03,&
&        -.60323120D-03,0.56216080D-03,0.94627090D-06,0.73605840D-05,0.38699350D-04,&
&        0.16351100D-03,0.59417750D-03,0.19277370D-02,0.56417310D-02,0.14878480D-01,&
&        0.34147830D-01,0.64057940D-01,0.75576590D-01,-.19460700D-01,-.29578740D+00,&
&        -.45211700D+00,0.12245310D+00,0.71697560D+00,0.40927120D+00,0.22219690D-01,&
&        -.90112020D-02,-.20953910D-06,-.16294390D-05,-.85705920D-05,-.36192720D-04,&
&        -.13161460D-03,-.42668100D-03,-.12502700D-02,-.32946650D-02,-.75818600D-02,&
&        -.14228640D-01,-.16937960D-01,0.44542980D-02,0.68670420D-01,0.11133350D+00,&
&        -.39008200D-01,-.22157550D+00,-.25443590D+00,0.28658660D+00,0.57557410D+00,&
&        -.41212310D-06,-.32820990D-05,-.16674330D-04,-.73699990D-04,-.25294950D-03,&
&        -.88014250D-03,-.23654820D-02,-.69263540D-02,-.13938510D-01,-.31438400D-01,&
&        -.26257490D-01,-.10483130D-01,0.18564720D+00,0.15248390D+00,0.74113680D-01,&
&        -.10180970D+01,0.29803720D+00,0.19719890D+01,-.11792530D+01,-.68054930D-06,&
&        -.54605160D-05,-.27438930D-04,-.12302910D-03,-.41462690D-03,-.14751510D-02,&
&        -.38591400D-02,-.11684050D-01,-.22571850D-01,-.53997590D-01,-.40323510D-01,&
&        -.27425260D-01,0.34855930D+00,0.27578180D+00,-.54998120D-01,-.22925450D+01,&
&        0.26356440D+01,-.33791130D+00,-.17896070D+01,0.10000000D+01,0.80488030D-05,&
&        0.62583060D-04,0.32902390D-03,0.13873550D-02,0.50232560D-02,0.16101400D-01,&
&        0.45900340D-01,0.11361540D+00,0.22838690D+00,0.32211590D+00,0.23836610D+00,&
&        0.74046670D-01,0.92141970D-01,0.93397900D-01,0.15739650D-01,-.41866820D-03,&
&        0.53763180D-04,-.38166540D-04,0.43196030D-04,-.41559540D-05,-.32314010D-04,&
&        -.16995250D-03,-.71713690D-03,-.26036250D-02,-.83991090D-02,-.24341090D-01,&
&        -.62519480D-01,-.13659290D+00,-.23127070D+00,-.23837340D+00,0.31238370D-01,&
&        0.50868180D+00,0.49876950D+00,0.90335520D-01,-.60053370D-02,0.23124540D-03,&
&        -.56436800D-03,0.49922600D-03,0.95321780D-06,0.74146050D-05,0.38983930D-04,&
&        0.16471520D-03,0.59859800D-03,0.19423900D-02,0.56872370D-02,0.15013290D-01,&
&        0.34524550D-01,0.64958200D-01,0.77161940D-01,-.18734110D-01,-.30091850D+00,&
&        -.45546610D+00,0.12864630D+00,0.71833160D+00,0.40517430D+00,0.21682270D-01,&
&        -.83435660D-02,-.20630080D-06,-.16041690D-05,-.84384370D-05,-.35631510D-04,&
&        -.12959980D-03,-.42015340D-03,-.12319540D-02,-.32489220D-02,-.74937170D-02,&
&        -.14101020D-01,-.16916000D-01,0.42189960D-02,0.68338100D-01,0.10982010D+00,&
&        -.40090050D-01,-.21747390D+00,-.24651350D+00,0.27314350D+00,0.57483210D+00,&
&        -.40093670D-06,-.31892550D-05,-.16230790D-04,-.71579200D-04,-.24639580D-03,&
&        -.85449070D-03,-.23075930D-02,-.67282920D-02,-.13661650D-01,-.30622400D-01,&
&        -.26311370D-01,-.97601830D-02,0.18019060D+00,0.15296340D+00,0.55054130D-01,&
&        -.95513640D+00,0.25868130D+00,0.18340490D+01,-.93332400D+00,-.69660420D-06,&
&        -.55680360D-05,-.28136840D-04,-.12524180D-03,-.42607870D-03,-.14990600D-02,&
&        -.39791030D-02,-.11856860D-01,-.23467340D-01,-.54677360D-01,-.43938200D-01,&
&        -.23761030D-01,0.34359280D+00,0.31929600D+00,-.13432070D+00,-.22210200D+01,&
&        0.25711420D+01,-.22924040D+00,-.18324520D+01,0.10000000D+01,0.79790260D-05,&
&        0.62040710D-04,0.32617350D-03,0.13753600D-02,0.49799970D-02,0.15964340D-01,&
&        0.45520860D-01,0.11273850D+00,0.22682620D+00,0.32030740D+00,0.23740210D+00,&
&        0.74776860D-01,0.95818720D-01,0.96499110D-01,0.16233620D-01,-.45354970D-03,&
&        0.51135190D-04,-.41745080D-04,0.40275770D-04,-.42002400D-05,-.32658310D-04,&
&        -.17176440D-03,-.72478530D-03,-.26314620D-02,-.84892720D-02,-.24606190D-01,&
&        -.63220590D-01,-.13819570D+00,-.23406800D+00,-.24150020D+00,0.30353120D-01,&
&        0.51013410D+00,0.49749390D+00,0.89707460D-01,-.59410340D-02,0.21753620D-03,&
&        -.54801550D-03,0.45258040D-03,0.95926920D-06,0.74618510D-05,0.39231370D-04,&
&        0.16577060D-03,0.60243350D-03,0.19552170D-02,0.57263260D-02,0.15129840D-01,&
&        0.34839730D-01,0.65703510D-01,0.78315030D-01,-.18770370D-01,-.30626630D+00,&
&        -.45664290D+00,0.13781690D+00,0.71936760D+00,0.39925790D+00,0.20799330D-01,&
&        -.78206630D-02,-.20288400D-06,-.15775800D-05,-.82988130D-05,-.35041540D-04,&
&        -.12746550D-03,-.41326950D-03,-.12122610D-02,-.31993180D-02,-.73909720D-02,&
&        -.13936490D-01,-.16785750D-01,0.41498560D-02,0.67976460D-01,0.10758070D+00,&
&        -.41660220D-01,-.21280440D+00,-.23813600D+00,0.26507880D+00,0.57227740D+00,&
&        -.38630530D-06,-.30687880D-05,-.15648260D-04,-.68835880D-04,-.23773670D-03,&
&        -.82131730D-03,-.22296300D-02,-.64678410D-02,-.13254630D-01,-.29466860D-01,&
&        -.25990660D-01,-.84998070D-02,0.17273160D+00,0.15121890D+00,0.35545090D-01,&
&        -.88293530D+00,0.21435300D+00,0.17118650D+01,-.71400370D+00,-.71746870D-06,&
&        -.57221570D-05,-.29009700D-04,-.12858450D-03,-.43984340D-03,-.15375860D-02,&
&        -.41160220D-02,-.12155140D-01,-.24404410D-01,-.56019760D-01,-.46894490D-01,&
&        -.21688280D-01,0.34777890D+00,0.35005970D+00,-.20485510D+00,-.21632110D+01,&
&        0.25240910D+01,-.15900110D+00,-.18252670D+01,0.10000000D+01,0.82089960D-05,&
&        0.63828840D-04,0.33558000D-03,0.14150750D-02,0.51244440D-02,0.16432560D-01,&
&        0.46893980D-01,0.11635340D+00,0.23505110D+00,0.33501840D+00,0.25347790D+00,&
&        0.73009010D-01,0.61842440D-01,0.63029560D-01,0.10080630D-01,-.22445280D-03,&
&        -.59327670D-04,-.11585620D-04,0.81151090D-05,-.36578490D-05,-.28440940D-04,&
&        -.14959280D-03,-.63130090D-03,-.22930520D-02,-.74051230D-02,-.21520320D-01,&
&        -.55609740D-01,-.12301760D+00,-.21301040D+00,-.22658370D+00,0.35467960D-01,&
&        0.51816970D+00,0.50256300D+00,0.89556740D-01,-.70313110D-02,-.43391670D-03,&
&        -.58317110D-03,0.42287880D-03,0.95941490D-06,0.74626140D-05,0.39238430D-04,&
&        0.16578680D-03,0.60259050D-03,0.19556620D-02,0.57303910D-02,0.15147560D-01,&
&        0.34934990D-01,0.65980720D-01,0.78930830D-01,-.19062490D-01,-.30959210D+00,&
&        -.45586100D+00,0.14829310D+00,0.71340390D+00,0.39760630D+00,0.22955230D-01,&
&        -.91517580D-02,-.20137530D-06,-.15658320D-05,-.82371820D-05,-.34781050D-04,&
&        -.12652650D-03,-.41025890D-03,-.12038340D-02,-.31790620D-02,-.73538280D-02,&
&        -.13890220D-01,-.16778750D-01,0.41633780D-02,0.68147030D-01,0.10610290D+00,&
&        -.43399800D-01,-.20949500D+00,-.23102710D+00,0.25905320D+00,0.56914260D+00,&
&        -.39242450D-06,-.31139090D-05,-.15904470D-04,-.69813940D-04,-.24178480D-03,&
&        -.83261950D-03,-.22702940D-02,-.65574270D-02,-.13542880D-01,-.29897680D-01,&
&        -.26931060D-01,-.78276930D-02,0.17416670D+00,0.15954680D+00,0.19955500D-01,&
&        -.88970000D+00,0.24868920D+00,0.16130120D+01,-.59902770D+00,-.73036330D-06,&
&        -.58020130D-05,-.29585470D-04,-.13014990D-03,-.44953320D-03,-.15533140D-02,&
&        -.42198840D-02,-.12254190D-01,-.25197070D-01,-.56217170D-01,-.50222290D-01,&
&        -.16774120D-01,0.33880210D+00,0.39849750D+00,-.30320530D+00,-.20796190D+01,&
&        0.25005420D+01,-.21690020D+00,-.17091780D+01,0.10000000D+01,0.78010260D-05,&
&        0.60656660D-04,0.31889640D-03,0.13446870D-02,0.48690500D-02,0.15610130D-01,&
&        0.44520770D-01,0.11031110D+00,0.22203420D+00,0.31337390D+00,0.23151210D+00,&
&        0.76409200D-01,0.11038180D+00,0.10943720D+00,0.18363110D-01,-.60430840D-03,&
&        0.50922450D-04,-.55407300D-04,0.39694820D-04,-.44047060D-05,-.34248010D-04,&
&        -.18012380D-03,-.76004550D-03,-.27593480D-02,-.89009700D-02,-.25793780D-01,&
&        -.66238610D-01,-.14459270D+00,-.24401100D+00,-.25048370D+00,0.28525770D-01,&
&        0.51158740D+00,0.49280610D+00,0.87884370D-01,-.58202810D-02,0.20135080D-03,&
&        -.51825530D-03,0.37315030D-03,0.97046820D-06,0.75492450D-05,0.39688920D-04,&
&        0.16772000D-03,0.60951010D-03,0.19788460D-02,0.57980490D-02,0.15341580D-01,&
&        0.35404840D-01,0.67020980D-01,0.80269450D-01,-.19272310D-01,-.31601290D+00,&
&        -.45731620D+00,0.15508410D+00,0.72028720D+00,0.38851220D+00,0.19243260D-01,&
&        -.71038070D-02,-.19593540D-06,-.15234720D-05,-.80148080D-05,-.33839920D-04,&
&        -.12311910D-03,-.39920850D-03,-.11719000D-02,-.30961410D-02,-.71719930D-02,&
&        -.13566210D-01,-.16439890D-01,0.41076280D-02,0.66939640D-01,0.10282210D+00,&
&        -.44229450D-01,-.20311910D+00,-.22300220D+00,0.25179750D+00,0.56500910D+00,&
&        -.35322290D-06,-.27988120D-05,-.14325170D-04,-.62709460D-04,-.21794900D-03,&
&        -.74743160D-03,-.20492710D-02,-.58852030D-02,-.12268850D-01,-.26831470D-01,&
&        -.24792610D-01,-.59847460D-02,0.15571240D+00,0.14366830D+00,0.83741030D-02,&
&        -.74607110D+00,0.12443670D+00,0.15101100D+01,-.34771220D+00,-.75082670D-06,&
&        -.59720180D-05,-.30396820D-04,-.13404050D-03,-.46157780D-03,-.16010640D-02,&
&        -.43309420D-02,-.12654340D-01,-.25868640D-01,-.58354280D-01,-.51323220D-01,&
&        -.19089530D-01,0.35861160D+00,0.38858180D+00,-.30571060D+00,-.20698960D+01,&
&        0.24317740D+01,-.21219740D-01,-.18202510D+01,0.10000000D+01,0.85492410D-05,&
&        0.66474100D-04,0.34949620D-03,0.14738320D-02,0.53383300D-02,0.17127080D-01,&
&        0.48940850D-01,0.12179340D+00,0.24765890D+00,0.35824310D+00,0.27981740D+00,&
&        0.68574910D-01,-.13110920D-02,0.19140010D-02,-.87592200D-03,0.37400960D-03,&
&        -.14013990D-03,0.47571320D-04,-.36427110D-04,-.26400690D-05,-.20527200D-04,&
&        -.10798590D-03,-.45585770D-03,-.16577580D-02,-.53684920D-02,-.15712490D-01,&
&        -.41225580D-01,-.94064590D-01,-.17199540D+00,-.19585230D+00,0.45329070D-01,&
&        0.52444420D+00,0.50061420D+00,0.89455270D-01,-.21462620D-02,0.21121130D-02,&
&        -.41339800D-03,0.32097520D-03,0.99671030D-06,0.77541630D-05,0.40760190D-04,&
&        0.17228110D-03,0.62593700D-03,0.20328550D-02,0.59546460D-02,0.15766400D-01,&
&        0.36376380D-01,0.68923430D-01,0.82380930D-01,-.20113600D-01,-.32525260D+00,&
&        -.46028990D+00,0.16355460D+00,0.72971180D+00,0.37697510D+00,0.14332240D-01,&
&        -.66712100D-02,0.19958180D-06,0.15529730D-05,0.81612590D-05,0.34507470D-04,&
&        0.12532750D-03,0.40729900D-03,0.11927340D-02,0.31631400D-02,0.73039420D-02,&
&        0.13912790D-01,0.16706200D-01,-.40355860D-02,-.69688610D-01,-.10301050D+00,&
&        0.44714420D-01,0.21500270D+00,0.22201630D+00,-.31147760D+00,-.56934290D+00,&
&        -.54359100D-06,-.43368940D-05,-.21975720D-04,-.97473920D-04,-.33316150D-03,&
&        -.11661920D-02,-.31193080D-02,-.92395040D-02,-.18554710D-01,-.42811890D-01,&
&        -.35710950D-01,-.16383500D-01,0.26446640D+00,0.20865880D+00,-.17743820D-01,&
&        -.13538730D+01,0.81829260D+00,0.16950360D+01,-.13886560D+01,0.80407910D-06,&
&        0.62829360D-05,0.32818680D-04,0.13988580D-03,0.50298360D-03,0.16554190D-02,&
&        0.47786770D-02,0.12924790D-01,0.29258310D-01,0.57918160D-01,0.66406810D-01,&
&        -.73889660D-02,-.33299890D+00,-.59178650D+00,0.90114060D+00,0.15859510D+01,&
&        -.27880080D+01,0.20718840D+01,-.60120250D+00,0.10000000D+01,0.80000000D-05,&
&        0.62200000D-04,0.32700000D-03,0.13794000D-02,0.49993000D-02,0.16060500D-01,&
&        0.46012400D-01,0.11522240D+00,0.23739210D+00,0.35319890D+00,0.29155000D+00,&
&        0.81212900D-01,0.76550000D-03,0.16124000D-02,-.75300000D-03,0.31340000D-03,&
&        -.13060000D-03,0.51300000D-04,-.25000000D-05,-.19300000D-04,-.10140000D-03,&
&        -.42810000D-03,-.15582000D-02,-.50469000D-02,-.14805600D-01,-.38948200D-01,&
&        -.89683200D-01,-.16640760D+00,-.20040100D+00,0.11494300D-01,0.49581340D+00,&
&        0.52955500D+00,0.11101850D+00,-.70000000D-03,0.22283000D-02,-.50140000D-03,&
&        0.90000000D-06,0.74000000D-05,0.38700000D-04,0.16330000D-03,0.59440000D-03,&
&        0.19292000D-02,0.56689000D-02,0.15028200D-01,0.35022200D-01,0.67113500D-01,&
&        0.85015600D-01,-.47212000D-02,-.30167370D+00,-.48254890D+00,0.89169500D-01,&
&        0.72878300D+00,0.42885420D+00,0.20724900D-01,0.20000000D-06,0.17000000D-05,&
&        0.90000000D-05,0.38000000D-04,0.13820000D-03,0.44890000D-03,0.13188000D-02,&
&        0.35016000D-02,0.81673000D-02,0.15733800D-01,0.20028400D-01,-.10136000D-02,&
&        -.75016200D-01,-.12579800D+00,0.30085700D-01,0.24881690D+00,0.28437060D+00,&
&        -.31105940D+00,0.10000000D+01,0.10000000D+01,0.74000000D-05,0.57400000D-04,&
&        0.30190000D-03,0.12733000D-02,0.46123000D-02,0.14821400D-01,0.42553600D-01,&
&        0.10730550D+00,0.22451780D+00,0.34531310D+00,0.30452610D+00,0.99067000D-01,&
&        0.41317000D-02,0.10347000D-02,-.48560000D-03,0.18050000D-03,-.91200000D-04,&
&        -.26600000D-04,-.23000000D-05,-.17900000D-04,-.94000000D-04,-.39640000D-03,&
&        -.14425000D-02,-.46675000D-02,-.13715300D-01,-.36179700D-01,-.84167900D-01,&
&        -.15887670D+00,-.20338070D+00,-.25141000D-01,0.45751520D+00,0.55719390D+00,&
&        0.13970550D+00,0.22645000D-02,0.20927000D-02,0.32810000D-03,0.90000000D-06,&
&        0.69000000D-05,0.36200000D-04,0.15280000D-03,0.55630000D-03,0.18018000D-02,&
&        0.53085000D-02,0.14087700D-01,0.33201300D-01,0.64462100D-01,0.86954000D-01,&
&        0.11874500D-01,-.27245340D+00,-.50014520D+00,0.10855400D-01,0.72164690D+00,&
&        0.48052130D+00,-.10229700D-01,-.20000000D-06,-.18000000D-05,-.93000000D-05,&
&        -.39200000D-04,-.14260000D-03,-.46210000D-03,-.13614000D-02,-.36175000D-02,&
&        -.85359000D-02,-.16650600D-01,-.22591100D-01,-.32147000D-02,0.74499800D-01,&
&        0.14403330D+00,-.80815000D-02,-.27041630D+00,-.34016070D+00,0.61906570D+00,&
&        0.10000000D+01,0.10000000D+01,0.68000000D-05,0.52800000D-04,0.27740000D-03,&
&        0.11702000D-02,0.42421000D-02,0.13655700D-01,0.39339900D-01,0.99929200D-01,&
&        0.21215550D+00,0.33638660D+00,0.31551250D+00,0.11813120D+00,0.80076000D-02,&
&        0.32930000D-03,-.15230000D-03,0.24700000D-04,-.36600000D-04,-.94000000D-05,&
&        -.21000000D-05,-.16500000D-04,-.86600000D-04,-.36530000D-03,-.13309000D-02,&
&        -.43093000D-02,-.12697300D-01,-.33616000D-01,-.78947000D-01,-.15144580D+00,&
&        -.20420140D+00,-.55736700D-01,0.41876070D+00,0.57587620D+00,0.16968420D+00,&
&        0.60662000D-02,0.17605000D-02,0.23160000D-03,0.80000000D-06,0.64000000D-05,&
&        0.33700000D-04,0.14230000D-03,0.51880000D-03,0.16802000D-02,0.49677000D-02,&
&        0.13211500D-01,0.31456600D-01,0.61844600D-01,0.87956600D-01,0.25754800D-01,&
&        -.24554590D+00,-.50905720D+00,-.55574900D-01,0.70837960D+00,0.52310270D+00,&
&        -.11117600D-01,-.20000000D-06,-.18000000D-05,-.93000000D-05,-.39200000D-04,&
&        -.14290000D-03,-.46290000D-03,-.13687000D-02,-.36440000D-02,-.86884000D-02,&
&        -.17155600D-01,-.24551400D-01,-.73524000D-02,0.72008700D-01,0.15762540D+00,&
&        0.14207400D-01,-.28515930D+00,-.38853470D+00,0.64953370D+00,0.10000000D+01,&
&        0.10000000D+01,0.63000000D-05,0.48900000D-04,0.25740000D-03,0.10861000D-02,&
&        0.39399000D-02,0.12704100D-01,0.36715600D-01,0.93867200D-01,0.20176770D+00,&
&        0.32805400D+00,0.32383340D+00,0.13523370D+00,0.11707500D-01,-.34360000D-03,&
&        0.16650000D-03,-.11880000D-03,0.20400000D-04,0.83000000D-05,-.20000000D-05,&
&        -.15300000D-04,-.80600000D-04,-.34000000D-03,-.12397000D-02,-.40177000D-02,&
&        -.11867200D-01,-.31534000D-01,-.74643900D-01,-.14521790D+00,-.20384410D+00,&
&        -.78871100D-01,0.38458250D+00,0.58652700D+00,0.19735910D+00,0.10010200D-01,&
&        0.13160000D-02,0.11090000D-03,0.80000000D-06,0.60000000D-05,0.31700000D-04,&
&        0.13370000D-03,0.48830000D-03,0.15821000D-02,0.46919000D-02,0.12509800D-01,&
&        0.30038100D-01,0.59727100D-01,0.88469600D-01,0.36392000D-01,-.22353290D+00,&
&        -.51224620D+00,-.10842240D+00,0.69363720D+00,0.55587110D+00,-.11383200D-01,&
&        -.20000000D-06,-.18000000D-05,-.93000000D-05,-.39100000D-04,-.14280000D-03,&
&        -.46270000D-03,-.13722000D-02,-.36628000D-02,-.88061000D-02,-.17586700D-01,&
&        -.26207400D-01,-.10996400D-01,0.69569700D-01,0.16839470D+00,0.34616000D-01,&
&        -.29787020D+00,-.43225690D+00,0.67572170D+00,0.10000000D+01,0.10000000D+01,&
&        0.59000000D-05,0.46100000D-04,0.24220000D-03,0.10226000D-02,0.37113000D-02,&
&        0.11978500D-01,0.34692700D-01,0.89123900D-01,0.19345570D+00,0.32090190D+00,&
&        0.32992330D+00,0.14941210D+00,0.14993800D-01,-.91650000D-03,0.43800000D-03,&
&        -.23980000D-03,0.73600000D-04,0.23900000D-04,-.19000000D-05,-.14500000D-04,&
&        -.76100000D-04,-.32100000D-03,-.11709000D-02,-.37968000D-02,-.11230700D-01,&
&        -.29927700D-01,-.71270600D-01,-.14031360D+00,-.20307630D+00,-.96098500D-01,&
&        0.35580860D+00,0.59217920D+00,0.22159770D+00,0.13764800D-01,0.83950000D-03,&
&        -.85000000D-05,0.70000000D-06,0.57000000D-05,0.30300000D-04,0.12750000D-03,&
&        0.46590000D-03,0.15096000D-02,0.44852000D-02,0.11983500D-01,0.28957100D-01,&
&        0.58156600D-01,0.88813300D-01,0.44524400D-01,-.20603870D+00,-.51270170D+00,&
&        -.15093490D+00,0.67892030D+00,0.58176970D+00,-.11182500D-01,-.20000000D-06,&
&        -.18000000D-05,-.93000000D-05,-.39100000D-04,-.14280000D-03,-.46280000D-03,&
&        -.13750000D-02,-.36784000D-02,-.88981000D-02,-.17952900D-01,-.27573200D-01,&
&        -.14095300D-01,0.67256100D-01,0.17669280D+00,0.52886100D-01,-.30759550D+00,&
&        -.47006580D+00,0.69803410D+00,0.10000000D+01,0.10000000D+01,0.56000000D-05,&
&        0.43800000D-04,0.23050000D-03,0.97330000D-03,0.35337000D-02,0.11416700D-01,&
&        0.33132500D-01,0.85446400D-01,0.18691240D+00,0.31497610D+00,0.33433340D+00,&
&        0.16088100D+00,0.17843500D-01,-.13793000D-02,0.65720000D-03,-.33880000D-03,&
&        0.12110000D-03,0.36600000D-04,-.18000000D-05,-.13800000D-04,-.72600000D-04,&
&        -.30630000D-03,-.11177000D-02,-.36270000D-02,-.10743200D-01,-.28699200D-01,&
&        -.68667900D-01,-.13651550D+00,-.20224580D+00,-.10905690D+00,0.33187680D+00,&
&        0.59482500D+00,0.24248250D+00,0.17224100D-01,0.36850000D-03,-.11530000D-03,&
&        0.70000000D-06,0.55000000D-05,0.29200000D-04,0.12280000D-03,0.44910000D-03,&
&        0.14557000D-02,0.43319000D-02,0.11596500D-01,0.28158500D-01,0.57033900D-01,&
&        0.89135600D-01,0.50842100D-01,-.19210300D+00,-.51210400D+00,-.18570070D+00,&
&        0.66541190D+00,0.60239250D+00,-.10645300D-01,-.20000000D-06,-.18000000D-05,&
&        -.93000000D-05,-.39100000D-04,-.14300000D-03,-.46390000D-03,-.13801000D-02,&
&        -.37001000D-02,-.89921000D-02,-.18302100D-01,-.28755900D-01,-.16732400D-01,&
&        0.65241000D-01,0.18344220D+00,0.69218300D-01,-.31560340D+00,-.50315010D+00,&
&        0.71715710D+00,0.10000000D+01,0.10000000D+01/)
      real(8) :: expp(370)= &
&     (/ 0.1407000D+01,0.3880000D+00,0.3044000D+01,0.7580000D+00,0.3266000D+01,&
&        0.6511000D+00,0.1696000D+00,0.5578000D-01,0.2050000D-01,0.7436000D+01,&
&        0.1577000D+01,0.4352000D+00,0.1438000D+00,0.4994000D-01,0.1205000D+02,&
&        0.2613000D+01,0.7475000D+00,0.2385000D+00,0.7698000D-01,0.1871000D+02,&
&        0.4133000D+01,0.1200000D+01,0.3827000D+00,0.1209000D+00,0.2663000D+02,&
&        0.5948000D+01,0.1742000D+01,0.5550000D+00,0.1725000D+00,0.3446000D+02,&
&        0.7749000D+01,0.2280000D+01,0.7156000D+00,0.2140000D+00,0.4388000D+02,&
&        0.9926000D+01,0.2930000D+01,0.9132000D+00,0.2672000D+00,0.5470000D+02,&
&        0.1243000D+02,0.3679000D+01,0.1143000D+01,0.3300000D+00,0.2433000D+03,&
&        0.5739000D+02,0.1810000D+02,0.6575000D+01,0.2521000D+01,0.9607000D+00,&
&        0.3512000D+00,0.9827000D-01,0.3734000D-01,0.1500000D-01,0.3169000D+03,&
&        0.7486000D+02,0.2372000D+02,0.8669000D+01,0.3363000D+01,0.1310000D+01,&
&        0.4911000D+00,0.2364000D+00,0.8733000D-01,0.3237000D-01,0.4444000D+03,&
&        0.1051000D+03,0.3347000D+02,0.1233000D+02,0.4869000D+01,0.1961000D+01,&
&        0.1888000D+00,0.7834000D+00,0.5557000D-01,0.4815000D+03,0.1139000D+03,&
&        0.3623000D+02,0.1334000D+02,0.5252000D+01,0.2120000D+01,0.2528000D+00,&
&        0.8561000D+00,0.7889000D-01,0.5049000D+03,0.1194000D+03,0.3796000D+02,&
&        0.1395000D+02,0.5457000D+01,0.2177000D+01,0.2877000D+00,0.8010000D+00,&
&        0.9714000D-01,0.5744000D+03,0.1358000D+03,0.4319000D+02,0.1587000D+02,&
&        0.6208000D+01,0.2483000D+01,0.3229000D+00,0.8688000D+00,0.1098000D+00,&
&        0.6633000D+03,0.1568000D+03,0.4998000D+02,0.1842000D+02,0.7240000D+01,&
&        0.2922000D+01,0.3818000D+00,0.1022000D+01,0.1301000D+00,0.7618000D+03,&
&        0.1802000D+03,0.5750000D+02,0.2124000D+02,0.8388000D+01,0.3416000D+01,&
&        0.4523000D+00,0.1206000D+01,0.1545000D+00,0.4061289D+04,0.9622465D+03,&
&        0.3121686D+03,0.1187144D+03,0.4980670D+02,0.2225998D+02,0.1028764D+02,&
&        0.4861154D+01,0.2248773D+01,0.1033662D+01,0.4641320D+00,0.6739000D-01,&
&        0.1987500D+00,0.2542000D-01,0.1059219D+05,0.2507533D+04,0.8144571D+03,&
&        0.3115195D+03,0.1319617D+03,0.5998718D+02,0.2866250D+02,0.1410851D+02,&
&        0.7103706D+01,0.3609200D+01,0.1776070D+01,0.8547600D+00,0.4022390D+00,&
&        0.1546650D+00,0.6494500D-01,0.2635900D-01,0.1191203D+05,0.2819947D+04,&
&        0.9159479D+03,0.3503842D+03,0.1484825D+03,0.6753944D+02,0.3230332D+02,&
&        0.1592786D+02,0.8038035D+01,0.4093916D+01,0.2022390D+01,0.9761020D+00,&
&        0.4595950D+00,0.1771520D+00,0.7351700D-01,0.2940100D-01,0.1327320D+05,&
&        0.3142126D+04,0.1020588D+04,0.3904407D+03,0.1655043D+03,0.7532006D+02,&
&        0.3605503D+02,0.1780436D+02,0.9002929D+01,0.4594544D+01,0.2276760D+01,&
&        0.1101178D+01,0.5186380D+00,0.2005650D+00,0.8129100D-01,0.3179500D-01,&
&        0.1445420D+05,0.3421676D+04,0.1111387D+04,0.4251918D+03,0.1802623D+03,&
&        0.8206117D+02,0.3929726D+02,0.1941959D+02,0.9828899D+01,0.5016810D+01,&
&        0.2487091D+01,0.1198780D+01,0.5586950D+00,0.2089240D+00,0.8460800D-01,&
&        0.3325800D-01,0.1620586D+05,0.3836274D+04,0.1246048D+04,0.4767535D+03,&
&        0.2021895D+03,0.9209487D+02,0.4414720D+02,0.2185468D+02,0.1108596D+02,&
&        0.5674108D+01,0.2823170D+01,0.1368621D+01,0.6444310D+00,0.2483820D+00,&
&        0.9725500D-01,0.3663300D-01,0.1774569D+05,0.4200721D+04,0.1364429D+04,&
&        0.5220806D+03,0.2214595D+03,0.1009096D+03,0.4840115D+02,0.2398536D+02,&
&        0.1218250D+02,0.6242298D+01,0.3110944D+01,0.1509958D+01,0.7108450D+00,&
&        0.2731900D+00,0.1042330D+00,0.3829100D-01,0.1926778D+05,0.4560986D+04,&
&        0.1481436D+04,0.5668671D+03,0.2404910D+03,0.1096105D+03,0.5259491D+02,&
&        0.2608361D+02,0.1326143D+02,0.6799778D+01,0.3393414D+01,0.1648766D+01,&
&        0.7762820D+00,0.2980030D+00,0.1136180D+00,0.4162400D-01,0.2102792D+05,&
&        0.4977560D+04,0.1616740D+04,0.6186718D+03,0.2625183D+03,0.1196907D+03,&
&        0.5746585D+02,0.2852829D+02,0.1452148D+02,0.7453850D+01,0.3723553D+01,&
&        0.1809813D+01,0.8513360D+00,0.3248140D+00,0.1195220D+00,0.4236600D-01,&
&        0.2276057D+05,0.5387679D+04,0.1749945D+04,0.6696653D+03,0.2841948D+03,&
&        0.1296077D+03,0.6225415D+02,0.3092964D+02,0.1575827D+02,0.8094211D+01,&
&        0.4046921D+01,0.1967869D+01,0.9252950D+00,0.3529920D+00,0.1273070D+00,&
&        0.4435600D-01,0.2441198D+05,0.5778518D+04,0.1876862D+04,0.7182361D+03,&
&        0.3048327D+03,0.1390453D+03,0.6680417D+02,0.3320699D+02,0.1692816D+02,&
&        0.8696229D+01,0.4350510D+01,0.2116523D+01,0.9953870D+00,0.3781120D+00,&
&        0.1345790D+00,0.4628200D-01,0.8050167D+04,0.1907536D+04,0.6186275D+03,&
&        0.2353242D+03,0.9889965D+02,0.4424822D+02,0.2061743D+02,0.9780516D+01,&
&        0.4441238D+01,0.1964045D+01,0.8335780D+00,0.1934450D+00,0.5611700D-01,&
&        0.6979598D+04,0.1654165D+04,0.5360286D+03,0.2035371D+03,0.8523753D+02,&
&        0.3784196D+02,0.1740651D+02,0.7881492D+01,0.3533213D+01,0.1521473D+01,&
&        0.1990930D+00,0.5627040D+00,0.6703100D-01,0.7423861D+04,0.1759517D+04,&
&        0.5702292D+03,0.2165800D+03,0.9073425D+02,0.4030879D+02,0.1855550D+02,&
&        0.8396543D+01,0.3767367D+01,0.1629701D+01,0.2225030D+00,0.5682630D+00,&
&        0.8040500D-01,0.8004300D+04,0.1896900D+04,0.6147100D+03,0.2335000D+03,&
&        0.9785600D+02,0.4351400D+02,0.2006300D+02,0.9112700D+01,0.4106300D+01,&
&        0.1794900D+01,0.2461500D+00,0.6243200D+00,0.8891700D-01,0.8676500D+04,&
&        0.2055900D+04,0.6662300D+03,0.2531000D+03,0.1061200D+03,0.4724200D+02,&
&        0.2182500D+02,0.9968400D+01,0.4517100D+01,0.1998200D+01,0.2814500D+00,&
&        0.7098800D+00,0.1020400D+00,0.9366309D+04,0.2219554D+04,0.7194529D+03,&
&        0.2734645D+03,0.1147523D+03,0.5115557D+02,0.2368268D+02,0.1087548D+02,&
&        0.4955131D+01,0.2217267D+01,0.3221540D+00,0.8064100D+00,0.1176190D+00/)
      real(8) :: coeffp(1184)= &
&     (/ 0.1000000D+01,0.1000000D+01,0.1000000D+01,0.1000000D+01,0.8630000D-02,&
&        0.4753800D-01,0.2097720D+00,0.1000000D+01,0.1000000D+01,0.1073600D-01,&
&        0.6285400D-01,0.2481800D+00,0.1000000D+01,0.1000000D+01,0.1311800D-01,&
&        0.7989600D-01,0.2772750D+00,0.1000000D+01,0.1000000D+01,0.1403100D-01,&
&        0.8686600D-01,0.2902160D+00,0.1000000D+01,0.1000000D+01,0.1467000D-01,&
&        0.9176400D-01,0.2986830D+00,0.1000000D+01,0.1000000D+01,0.1592800D-01,&
&        0.9974000D-01,0.3104920D+00,0.1000000D+01,0.1000000D+01,0.1666500D-01,&
&        0.1044720D+00,0.3172600D+00,0.1000000D+01,0.1000000D+01,0.1715100D-01,&
&        0.1076560D+00,0.3216810D+00,0.1000000D+01,0.1000000D+01,0.2243920D-02,&
&        0.1739970D-01,0.7741250D-01,0.2191020D+00,0.3785220D+00,0.3949020D+00,&
&        0.1604240D+00,0.2333110D-02,-.2224010D-03,-.1742770D-02,-.7754560D-02,&
&        -.2251870D-01,-.3843300D-01,-.4501770D-01,-.1921320D-01,0.1826970D+00,&
&        0.1000000D+01,0.1000000D+01,0.2075320D-02,0.1628690D-01,0.7386970D-01,&
&        0.2142970D+00,0.3821540D+00,0.3981780D+00,0.1528780D+00,-.4375400D-02,&
&        -.3297270D-03,-.2587540D-02,-.1191200D-01,-.3502270D-01,-.6399680D-01,&
&        -.7044360D-01,-.3758360D-01,0.1770430D+00,0.1000000D+01,0.1000000D+01,&
&        0.1627860D-02,0.1306870D-01,0.6123410D-01,0.1878700D+00,0.3604520D+00,&
&        0.4084540D+00,0.9765140D-02,-.2863410D-03,-.2423080D-02,-.1086580D-01,&
&        -.3643070D-01,-.6410740D-01,-.9722390D-01,0.5034480D+00,0.1000000D+01,&
&        0.1000000D+01,0.1920450D-02,0.1535520D-01,0.7139910D-01,0.2130520D+00,&
&        0.3903540D+00,0.3937210D+00,0.3956300D-02,-.4052200D-03,-.3358960D-02,&
&        -.1528600D-01,-.4892180D-01,-.8550080D-01,-.1121370D+00,0.5519190D+00,&
&        0.1000000D+01,0.1000000D+01,0.2337280D-02,0.1854100D-01,0.8496930D-01,&
&        0.2446150D+00,0.4227660D+00,0.3684390D+00,-.3790050D-02,-.5552360D-03,&
&        -.4459130D-02,-.2063500D-01,-.6176940D-01,-.1089240D+00,-.1055990D+00,&
&        0.5769810D+00,0.1000000D+01,0.1000000D+01,0.2422640D-02,0.1927960D-01,&
&        0.8854010D-01,0.2546540D+00,0.4339840D+00,0.3549530D+00,-.5029770D-02,&
&        -.6201020D-03,-.4938820D-02,-.2326470D-01,-.6851950D-01,-.1238960D+00,&
&        -.9694990D-01,0.5693940D+00,0.1000000D+01,0.1000000D+01,0.2404480D-02,&
&        0.1921480D-01,0.8850970D-01,0.2560200D+00,0.4369270D+00,0.3503340D+00,&
&        -.4584230D-02,-.6521450D-03,-.5194450D-02,-.2469380D-01,-.7281670D-01,&
&        -.1340300D+00,-.9477420D-01,0.5646670D+00,0.1000000D+01,0.1000000D+01,&
&        0.2369760D-02,0.1901990D-01,0.8808070D-01,0.2563770D+00,0.4387110D+00,&
&        0.3475690D+00,-.5238820D-02,-.6672110D-03,-.5327170D-02,-.2554940D-01,&
&        -.7571970D-01,-.1411330D+00,-.9327680D-01,0.5624500D+00,0.1000000D+01,&
&        0.1000000D+01,0.19799000D-03,0.17320800D-02,0.95337900D-02,0.38390120D-01,&
&        0.11675881D+00,0.25626874D+00,0.37978080D+00,0.30829326D+00,0.85920900D-01,&
&        0.21206700D-02,0.12888000D-02,0.14728000D-03,-.64550000D-04,-.56458000D-03,&
&        -.31312500D-02,-.12740860D-01,-.39914030D-01,-.90504480D-01,-.14261898D+00,&
&        -.10980904D+00,0.15162490D+00,0.46176411D+00,0.43260031D+00,0.25287400D-02,&
&        0.13360000D-04,0.11852000D-03,0.64872000D-03,0.26799300D-02,0.82851300D-02,&
&        0.19212350D-01,0.29549840D-01,0.24312320D-01,-.41112300D-01,-.10419758D+00,&
&        -.15036537D+00,0.59861164D+00,0.10000000D+01,0.10000000D+01,0.45000000D-04,&
&        0.40100000D-03,0.23020000D-02,0.10037000D-01,0.34954000D-01,0.97909000D-01,&
&        0.21068000D+00,0.33009300D+00,0.33102700D+00,0.15796000D+00,0.22099000D-01,&
&        -.16050000D-02,-.13260000D-02,-.28000000D-03,0.34000000D-04,-.15000000D-04,&
&        -.13100000D-03,-.75700000D-03,-.33180000D-02,-.11706000D-01,-.33604000D-01,&
&        -.74879000D-01,-.12254800D+00,-.13027600D+00,0.14596000D-01,0.30918400D+00,&
&        0.46299800D+00,0.30495700D+00,0.50878000D-01,-.44930000D-02,-.40000000D-05,&
&        -.32000000D-04,-.18500000D-03,-.80800000D-03,-.28700000D-02,-.82070000D-02,&
&        -.18473000D-01,-.30101000D-01,-.32943000D-01,0.79580000D-02,0.87993000D-01,&
&        0.15237700D+00,0.97170000D-01,-.25693800D+00,-.58781500D+00,0.40000000D-05,&
&        0.39000000D-04,0.22100000D-03,0.98400000D-03,0.34230000D-02,0.99930000D-02,&
&        0.21916000D-01,0.37008000D-01,0.37794000D-01,-.43790000D-02,-.11016400D+00,&
&        -.16101700D+00,-.18248200D+00,0.38861100D+00,0.69110000D+00,0.90000000D-05,&
&        0.74000000D-04,0.47900000D-03,0.18690000D-02,0.74240000D-02,0.18763000D-01,&
&        0.48717000D-01,0.65440000D-01,0.10794800D+00,-.58713000D-01,-.17030600D+00,&
&        -.75497700D+00,0.62011700D+00,0.11675480D+01,-.10095310D+01,0.10000000D+01,&
&        0.44000000D-04,0.39100000D-03,0.22480000D-02,0.98230000D-02,0.34338000D-01,&
&        0.96666000D-01,0.20941700D+00,0.33018900D+00,0.33193600D+00,0.15848800D+00,&
&        0.22310000D-01,-.15660000D-02,-.13240000D-02,-.27100000D-03,0.32000000D-04,&
&        -.15000000D-04,-.13100000D-03,-.75500000D-03,-.33190000D-02,-.11750000D-01,&
&        -.33922000D-01,-.76164000D-01,-.12570200D+00,-.13309800D+00,0.17406000D-01,&
&        0.31516500D+00,0.46181400D+00,0.29985600D+00,0.50000000D-01,-.42300000D-02,&
&        0.40000000D-05,0.31000000D-04,0.18200000D-03,0.79500000D-03,0.28330000D-02,&
&        0.81540000D-02,0.18472000D-01,0.30400000D-01,0.33047000D-01,-.82510000D-02,&
&        -.88554000D-01,-.14961200D+00,-.94227000D-01,0.25084600D+00,0.58664300D+00,&
&        0.40000000D-05,0.39000000D-04,0.22300000D-03,0.99200000D-03,0.34760000D-02,&
&        0.10172000D-01,0.22576000D-01,0.38238000D-01,0.39337000D-01,-.61060000D-02,&
&        -.11296200D+00,-.16811400D+00,-.16593200D+00,0.39140300D+00,0.68184000D+00,&
&        0.90000000D-05,0.73000000D-04,0.46800000D-03,0.18560000D-02,0.72930000D-02,&
&        0.18856000D-01,0.48405000D-01,0.67378000D-01,0.10454100D+00,-.53760000D-01,&
&        -.19174900D+00,-.69472000D+00,0.56016200D+00,0.11813360D+01,-.99327200D+00,&
&        0.10000000D+01,0.43000000D-04,0.38400000D-03,0.22100000D-02,0.96780000D-02,&
&        0.33936000D-01,0.95917000D-01,0.20885300D+00,0.33066000D+00,0.33231200D+00,&
&        0.15818800D+00,0.22252000D-01,-.15650000D-02,-.13530000D-02,-.26500000D-03,&
&        0.29000000D-04,-.15000000D-04,-.13100000D-03,-.75500000D-03,-.33250000D-02,&
&        -.11811000D-01,-.34256000D-01,-.77363000D-01,-.12845600D+00,-.13507800D+00,&
&        0.20838000D-01,0.32049900D+00,0.46026000D+00,0.29534600D+00,0.49046000D-01,&
&        -.38240000D-02,0.40000000D-05,0.32000000D-04,0.18300000D-03,0.80200000D-03,&
&        0.28620000D-02,0.82870000D-02,0.18870000D-01,0.31307000D-01,0.33660000D-01,&
&        -.94790000D-02,-.92313000D-01,-.14898900D+00,-.83644000D-01,0.24933900D+00,&
&        0.58051500D+00,0.40000000D-05,0.39000000D-04,0.22300000D-03,0.99600000D-03,&
&        0.34980000D-02,0.10296000D-01,0.22962000D-01,0.39208000D-01,0.39943000D-01,&
&        -.71210000D-02,-.11622500D+00,-.16949600D+00,-.15537400D+00,0.39502200D+00,&
&        0.67890800D+00,0.90000000D-05,0.75000000D-04,0.47900000D-03,0.19060000D-02,&
&        0.75120000D-02,0.19530000D-01,0.50402000D-01,0.70685000D-01,0.10877500D+00,&
&        -.58536000D-01,-.21544800D+00,-.67734200D+00,0.58597900D+00,0.11291080D+01,&
&        -.97405100D+00,0.10000000D+01,0.44000000D-04,0.38900000D-03,0.22410000D-02,&
&        0.98210000D-02,0.34471000D-01,0.97460000D-01,0.21198500D+00,0.33399000D+00,&
&        0.33013700D+00,0.15222700D+00,0.20425000D-01,-.13600000D-02,-.11950000D-02,&
&        -.19700000D-03,0.23000000D-04,-.15000000D-04,-.13500000D-03,-.77700000D-03,&
&        -.34270000D-02,-.12189000D-01,-.35388000D-01,-.79915000D-01,-.13233500D+00,&
&        -.13540100D+00,0.32008000D-01,0.33384900D+00,0.46177300D+00,0.28129000D+00,&
&        0.41843000D-01,-.40020000D-02,0.40000000D-05,0.32000000D-04,0.18500000D-03,&
&        0.81000000D-03,0.29060000D-02,0.83910000D-02,0.19193000D-01,0.31564000D-01,&
&        0.33417000D-01,-.12907000D-01,-.93659000D-01,-.14997700D+00,-.67234000D-01,&
&        0.27075900D+00,0.57580700D+00,0.40000000D-05,0.40000000D-04,0.22900000D-03,&
&        0.10190000D-02,0.36020000D-02,0.10550000D-01,0.23702000D-01,0.39988000D-01,&
&        0.40437000D-01,-.12074000D-01,-.11893900D+00,-.17810000D+00,-.12386500D+00,&
&        0.42972200D+00,0.65078600D+00,0.10000000D-04,0.79000000D-04,0.51200000D-03,&
&        0.20230000D-02,0.80550000D-02,0.20772000D-01,0.54224000D-01,0.74685000D-01,&
&        0.11598900D+00,-.76584000D-01,-.24393100D+00,-.68018100D+00,0.73366400D+00,&
&        0.99912000D+00,-.10170810D+01,0.10000000D+01,0.42000000D-04,0.37300000D-03,&
&        0.21490000D-02,0.94450000D-02,0.33297000D-01,0.94759000D-01,0.20814400D+00,&
&        0.33180500D+00,0.33317500D+00,0.15760100D+00,0.21445000D-01,-.25580000D-02,&
&        -.20270000D-02,-.36000000D-03,0.34000000D-04,-.15000000D-04,-.12900000D-03,&
&        -.74800000D-03,-.33080000D-02,-.11811000D-01,-.34533000D-01,-.78785000D-01,&
&        -.13218300D+00,-.13719500D+00,0.27075000D-01,0.32889100D+00,0.45728000D+00,&
&        0.28890800D+00,0.47433000D-01,-.35220000D-02,0.30000000D-05,0.30000000D-04,&
&        0.17200000D-03,0.76200000D-03,0.27260000D-02,0.79760000D-02,0.18287000D-01,&
&        0.30776000D-01,0.32373000D-01,-.99780000D-02,-.90529000D-01,-.13800400D+00,&
&        -.77965000D-01,0.22956000D+00,0.57612200D+00,0.40000000D-05,0.40000000D-04,&
&        0.22600000D-03,0.10130000D-02,0.35750000D-02,0.10612000D-01,0.23902000D-01,&
&        0.41279000D-01,0.41475000D-01,-.94580000D-02,-.12369500D+00,-.17439200D+00,&
&        -.12917000D+00,0.40034800D+00,0.66964600D+00,0.10000000D-04,0.81000000D-04,&
&        0.51200000D-03,0.20710000D-02,0.81110000D-02,0.21557000D-01,0.55316000D-01,&
&        0.80263000D-01,0.11773000D+00,-.65069000D-01,-.28291400D+00,-.64916600D+00,&
&        0.69256700D+00,0.97899000D+00,-.92135700D+00,0.10000000D+01,0.41000000D-04,&
&        0.36900000D-03,0.21290000D-02,0.93690000D-02,0.33097000D-01,0.94431000D-01,&
&        0.20807700D+00,0.33233300D+00,0.33298700D+00,0.15684300D+00,0.21549000D-01,&
&        -.20950000D-02,-.17390000D-02,-.30000000D-03,0.29000000D-04,-.15000000D-04,&
&        -.13000000D-03,-.75100000D-03,-.33290000D-02,-.11912000D-01,-.34933000D-01,&
&        -.79989000D-01,-.13463600D+00,-.13859800D+00,0.30278000D-01,0.33321600D+00,&
&        0.45615300D+00,0.28505100D+00,0.46144000D-01,-.32490000D-02,0.30000000D-05,&
&        0.29000000D-04,0.16500000D-03,0.73400000D-03,0.26260000D-02,0.77250000D-02,&
&        0.17733000D-01,0.30055000D-01,0.31094000D-01,-.10048000D-01,-.88306000D-01,&
&        -.12982400D+00,-.76937000D-01,0.21266100D+00,0.57306100D+00,0.50000000D-05,&
&        0.42000000D-04,0.24100000D-03,0.10850000D-02,0.38310000D-02,0.11423000D-01,&
&        0.25792000D-01,0.44818000D-01,0.44598000D-01,-.11177000D-01,-.13813400D+00,&
&        -.18828500D+00,-.10739900D+00,0.44486300D+00,0.64023900D+00,0.11000000D-04,&
&        0.87000000D-04,0.54100000D-03,0.22260000D-02,0.85930000D-02,0.23339000D-01,&
&        0.58844000D-01,0.88289000D-01,0.12319200D+00,-.63186000D-01,-.35490200D+00,&
&        -.61970800D+00,0.81298600D+00,0.81911800D+00,-.90170500D+00,0.10000000D+01,&
&        0.41000000D-04,0.36900000D-03,0.21280000D-02,0.93720000D-02,0.33155000D-01,&
&        0.94752000D-01,0.20909300D+00,0.33372200D+00,0.33220800D+00,0.15461300D+00,&
&        0.20902000D-01,-.20240000D-02,-.16970000D-02,-.28000000D-03,0.26000000D-04,&
&        -.15000000D-04,-.13100000D-03,-.75800000D-03,-.33630000D-02,-.12054000D-01,&
&        -.35424000D-01,-.81287000D-01,-.13690800D+00,-.13901900D+00,0.35468000D-01,&
&        0.33849800D+00,0.45443300D+00,0.27979300D+00,0.44776000D-01,-.31510000D-02,&
&        -.30000000D-05,-.29000000D-04,-.16700000D-03,-.74200000D-03,-.26620000D-02,&
&        -.78410000D-02,-.18051000D-01,-.30580000D-01,-.31312000D-01,0.11311000D-01,&
&        0.89990000D-01,0.13073300D+00,0.71808000D-01,-.22165800D+00,-.57102500D+00,&
&        0.50000000D-05,0.45000000D-04,0.25500000D-03,0.11440000D-02,0.40610000D-02,&
&        0.12095000D-01,0.27476000D-01,0.47557000D-01,0.47302000D-01,-.14418000D-01,&
&        -.15006200D+00,-.19909200D+00,-.79783000D-01,0.45903500D+00,0.61749500D+00,&
&        0.11000000D-04,0.92000000D-04,0.56300000D-03,0.23540000D-02,0.89760000D-02,&
&        0.24810000D-01,0.61580000D-01,0.94777000D-01,0.12669300D+00,-.63754000D-01,&
&        -.41856600D+00,-.56786500D+00,0.87540600D+00,0.71826300D+00,-.87472200D+00,&
&        0.10000000D+01,0.41000000D-04,0.36300000D-03,0.20970000D-02,0.92500000D-02,&
&        0.32796000D-01,0.94004000D-01,0.20828000D+00,0.33365400D+00,0.33290400D+00,&
&        0.15537200D+00,0.20859000D-01,-.24400000D-02,-.19980000D-02,-.33800000D-03,&
&        0.35000000D-04,-.15000000D-04,-.12900000D-03,-.74900000D-03,-.33280000D-02,&
&        -.11947000D-01,-.35242000D-01,-.81204000D-01,-.13749300D+00,-.13922600D+00,&
&        0.36016000D-01,0.33912800D+00,0.45047200D+00,0.28178300D+00,0.47898000D-01,&
&        -.29870000D-02,0.30000000D-05,0.26000000D-04,0.15200000D-03,0.67800000D-03,&
&        0.24270000D-02,0.72010000D-02,0.16578000D-01,0.28392000D-01,0.28599000D-01,&
&        -.10132000D-01,-.82912000D-01,-.11599800D+00,-.72795000D-01,0.19564000D+00,&
&        0.56709900D+00,0.60000000D-05,0.53000000D-04,0.30500000D-03,0.13640000D-02,&
&        0.48760000D-02,0.14503000D-01,0.33296000D-01,0.57482000D-01,0.58702000D-01,&
&        -.19904000D-01,-.19469500D+00,-.23961300D+00,-.22320000D-02,0.52143500D+00,&
&        0.54554000D+00,0.11000000D-04,0.95000000D-04,0.58000000D-03,0.24510000D-02,&
&        0.92820000D-02,0.26009000D-01,0.64096000D-01,0.10071000D+00,0.13253900D+00,&
&        -.65089000D-01,-.48975600D+00,-.49845500D+00,0.96635700D+00,0.52837900D+00,&
&        -.86767600D+00,0.10000000D+01,0.40000000D-04,0.36100000D-03,0.20830000D-02,&
&        0.91970000D-02,0.32660000D-01,0.93795000D-01,0.20827400D+00,0.33399300D+00,&
&        0.33249300D+00,0.15472800D+00,0.21271000D-01,-.16900000D-02,-.15160000D-02,&
&        -.24200000D-03,0.23000000D-04,-.15000000D-04,-.13100000D-03,-.75500000D-03,&
&        -.33590000D-02,-.12081000D-01,-.35703000D-01,-.82502000D-01,-.13989000D+00,&
&        -.14072900D+00,0.38766000D-01,0.34269500D+00,0.45231000D+00,0.27705400D+00,&
&        0.43885000D-01,-.28020000D-02,0.30000000D-05,0.25000000D-04,0.14700000D-03,&
&        0.65600000D-03,0.23510000D-02,0.70040000D-02,0.16131000D-01,0.27770000D-01,&
&        0.27567000D-01,-.10115000D-01,-.81009000D-01,-.11040900D+00,-.71732000D-01,&
&        0.18793000D+00,0.56462900D+00,0.50000000D-05,0.49000000D-04,0.27800000D-03,&
&        0.12530000D-02,0.44470000D-02,0.13370000D-01,0.30469000D-01,0.53447000D-01,&
&        0.52639000D-01,-.16881000D-01,-.17944800D+00,-.20958800D+00,-.39633000D-01,&
&        0.50213000D+00,0.58111100D+00,0.11000000D-04,0.96000000D-04,0.59000000D-03,&
&        0.24840000D-02,0.94630000D-02,0.26453000D-01,0.65689000D-01,0.10273200D+00,&
&        0.13704100D+00,-.70961000D-01,-.50470800D+00,-.47805600D+00,0.94289200D+00,&
&        0.54469900D+00,-.83276600D+00,0.10000000D+01,0.41000000D-04,0.36100000D-03,&
&        0.20880000D-02,0.92210000D-02,0.32773000D-01,0.94179000D-01,0.20913200D+00,&
&        0.33456900D+00,0.33035900D+00,0.15234700D+00,0.22984000D-01,0.16070000D-02,&
&        0.46800000D-03,0.66000000D-04,-.20000000D-05,-.15000000D-04,-.13500000D-03,&
&        -.78200000D-03,-.34780000D-02,-.12520000D-01,-.37016000D-01,-.85559000D-01,&
&        -.14471800D+00,-.14344200D+00,0.43595000D-01,0.34888800D+00,0.45386500D+00,&
&        0.26859400D+00,0.38868000D-01,-.24920000D-02,0.30000000D-05,0.25000000D-04,&
&        0.14400000D-03,0.64500000D-03,0.23110000D-02,0.68980000D-02,0.15882000D-01,&
&        0.27350000D-01,0.26621000D-01,-.10858000D-01,-.79853000D-01,-.10612700D+00,&
&        -.68883000D-01,0.18438500D+00,0.56178800D+00,0.50000000D-05,0.42000000D-04,&
&        0.23800000D-03,0.10880000D-02,0.38210000D-02,0.11644000D-01,0.26167000D-01,&
&        0.46750000D-01,0.43309000D-01,-.13429000D-01,-.15389700D+00,-.16741300D+00,&
&        -.84995000D-01,0.45081300D+00,0.64086900D+00,0.12000000D-04,0.96000000D-04,&
&        0.59400000D-03,0.24840000D-02,0.95370000D-02,0.26479000D-01,0.66366000D-01,&
&        0.10245800D+00,0.13868300D+00,-.80140000D-01,-.49606900D+00,-.46351000D+00,&
&        0.87453100D+00,0.62979000D+00,-.81168600D+00,0.10000000D+01,0.31690000D-03,&
&        0.27648000D-02,0.15120400D-01,0.59958300D-01,0.17331200D+00,0.34108200D+00,&
&        0.38969670D+00,0.18398170D+00,0.21889600D-01,0.11608000D-02,0.27350000D-03,&
&        -.12030000D-03,-.10492000D-02,-.58102000D-02,-.23434500D-01,-.70827000D-01,&
&        -.14655110D+00,-.17696600D+00,0.36382100D-01,0.42328480D+00,0.49525860D+00,&
&        0.17974280D+00,0.20000000D-04,0.16890000D-03,0.96680000D-03,0.37797000D-02,&
&        0.11908200D-01,0.23569300D-01,0.31423300D-01,-.13618800D-01,-.73400300D-01,&
&        -.12647850D+00,0.15857900D-01,0.10000000D+01,0.10000000D+01,0.45690000D-03,&
&        0.39562000D-02,0.21314300D-01,0.81871500D-01,0.22237320D+00,0.39056590D+00,&
&        0.35604150D+00,0.10703120D+00,0.36941000D-02,0.19219000D-02,0.19170000D-03,&
&        -.17560000D-03,-.15305000D-02,-.83145000D-02,-.32871800D-01,-.93166100D-01,&
&        -.17555420D+00,-.14679120D+00,0.18629340D+00,0.52648620D+00,0.39708590D+00,&
&        -.33478000D-02,0.34800000D-04,0.30140000D-03,0.16487000D-02,0.64982000D-02,&
&        0.18638300D-01,0.35061300D-01,0.29589200D-01,-.47245800D-01,-.12498470D+00,&
&        -.12108010D+00,0.57547300D+00,0.10000000D+01,0.10000000D+01,0.45990000D-03,&
&        0.39823000D-02,0.21463800D-01,0.82461700D-01,0.22389020D+00,0.39207040D+00,&
&        0.35422380D+00,0.10486410D+00,0.33664000D-02,0.18495000D-02,0.22970000D-03,&
&        -.17940000D-03,-.15641000D-02,-.84999000D-02,-.33632700D-01,-.95322800D-01,&
&        -.17936260D+00,-.14666820D+00,0.19660160D+00,0.53720880D+00,0.38573610D+00,&
&        -.54006000D-02,0.39900000D-04,0.34880000D-03,0.18953000D-02,0.75325000D-02,&
&        0.21431500D-01,0.40780700D-01,0.32524900D-01,-.54883200D-01,-.15119220D+00,&
&        -.12490110D+00,0.58552930D+00,0.10000000D+01,0.10000000D+01,0.45050000D-03,&
&        0.39049000D-02,0.21090100D-01,0.81292000D-01,0.22178410D+00,0.39072700D+00,&
&        0.35597140D+00,0.10732720D+00,0.36985000D-02,0.18032000D-02,0.22080000D-03,&
&        -.17830000D-03,-.15554000D-02,-.84727000D-02,-.33624500D-01,-.95826700D-01,&
&        -.18139070D+00,-.15031520D+00,0.19482630D+00,0.54155540D+00,0.38372990D+00,&
&        -.50132000D-02,0.43000000D-04,0.37700000D-03,0.20465000D-02,0.81899000D-02,&
&        0.23335600D-01,0.44981300D-01,0.35747500D-01,-.58686600D-01,-.17095730D+00,&
&        -.12935830D+00,0.57780690D+00,0.10000000D+01,0.10000000D+01,0.43570000D-03,&
&        0.37815000D-02,0.20478200D-01,0.79283400D-01,0.21784730D+00,0.38785850D+00,&
&        0.35943500D+00,0.11219950D+00,0.43874000D-02,0.17809000D-02,0.21220000D-03,&
&        -.17480000D-03,-.15263000D-02,-.83399000D-02,-.33220300D-01,-.95418000D-01,&
&        -.18240260D+00,-.15583080D+00,0.18678990D+00,0.54277330D+00,0.38733090D+00,&
&        -.43784000D-02,0.45100000D-04,0.39640000D-03,0.21555000D-02,0.86720000D-02,&
&        0.24868000D-01,0.48547200D-01,0.39615600D-01,-.60574900D-01,-.18716990D+00,&
&        -.13777570D+00,0.57608960D+00,0.10000000D+01,0.10000000D+01,0.42310000D-03,&
&        0.36743000D-02,0.19931200D-01,0.77422200D-01,0.21403860D+00,0.38485560D+00,&
&        0.36263400D+00,0.11708180D+00,0.51210000D-02,0.17539000D-02,0.21990000D-03,&
&        -.17200000D-03,-.15025000D-02,-.82269000D-02,-.32856600D-01,-.95013500D-01,&
&        -.18331060D+00,-.16121610D+00,0.17876440D+00,0.54378850D+00,0.39133870D+00,&
&        -.47800000D-02,0.46600000D-04,0.41000000D-03,0.22328000D-02,0.90144000D-02,&
&        0.26011500D-01,0.51334000D-01,0.43092900D-01,-.61504000D-01,-.20034240D+00,&
&        -.14573640D+00,0.57645810D+00,0.10000000D+01,0.10000000D+01/)
      real(8) :: expd(174)= &
&     (/ 0.1057000D+01,0.1965000D+01,0.1874000D+00,0.8010000D-01,0.3480000D+00,&
&        0.1803000D+00,0.6610000D+00,0.1990000D+00,0.1097000D+01,0.3180000D+00,&
&        0.1654000D+01,0.4690000D+00,0.2314000D+01,0.6450000D+00,0.3107000D+01,&
&        0.8550000D+00,0.4014000D+01,0.1096000D+01,0.1367000D+00,0.6360000D-01,&
&        0.1260000D+00,0.2940000D+00,0.1090000D+00,0.3330000D+00,0.1590000D+00,&
&        0.4810000D+00,0.2160000D+00,0.6520000D+00,0.2690000D+00,0.8190000D+00,&
&        0.1046000D+01,0.3440000D+00,0.4100000D+00,0.1254000D+01,0.1694623D+02,&
&        0.4472120D+01,0.1438090D+01,0.4669900D+00,0.1415100D+00,0.4164000D-01,&
&        0.5051380D+02,0.1474050D+02,0.5195000D+01,0.2028460D+01,0.8040860D+00,&
&        0.3076890D+00,0.1113920D+00,0.3735200D-01,0.6401300D+02,0.1881790D+02,&
&        0.6728700D+01,0.2664130D+01,0.1078680D+01,0.4232090D+00,0.1559990D+00,&
&        0.5188400D-01,0.7761150D+02,0.2291590D+02,0.8279540D+01,0.3309930D+01,&
&        0.1358630D+01,0.5413500D+00,0.2023560D+00,0.6756800D-01,0.8857680D+02,&
&        0.2620450D+02,0.9517470D+01,0.3822480D+01,0.1575120D+01,0.6289280D+00,&
&        0.2344240D+00,0.7681500D-01,0.1006630D+03,0.2983360D+02,0.1088940D+02,&
&        0.4393580D+01,0.1817820D+01,0.7278270D+00,0.2712950D+00,0.8830900D-01,&
&        0.1133440D+03,0.3364140D+02,0.1233100D+02,0.4994780D+01,0.2072800D+01,&
&        0.8307530D+00,0.3091780D+00,0.1001300D+00,0.1262640D+03,0.3752260D+02,&
&        0.1380210D+02,0.5609270D+01,0.2333690D+01,0.9364150D+00,0.3482370D+00,&
&        0.1123530D+00,0.1402527D+03,0.4172610D+02,0.1539810D+02,0.6277100D+01,&
&        0.2618500D+01,0.1052600D+01,0.3916000D+00,0.1262000D+00,0.1738970D+03,&
&        0.5188690D+02,0.1934190D+02,0.7975720D+01,0.3398230D+01,0.1409320D+01,&
&        0.5488580D+00,0.1901990D+00,0.2056177D+03,0.6144981D+02,0.2305689D+02,&
&        0.9577739D+01,0.4133734D+01,0.1747518D+01,0.6995600D+00,0.2516080D+00,&
&        0.2441474D+03,0.7306759D+02,0.2759208D+02,0.1154652D+02,0.5048628D+01,&
&        0.2178465D+01,0.9002530D+00,0.3373270D+00,0.1169000D+00,0.2822391D+03,&
&        0.8454996D+02,0.3207366D+02,0.1349750D+02,0.5958550D+01,0.2610788D+01,&
&        0.1103987D+01,0.4240490D+00,0.1520000D+00,0.3210196D+03,0.9624931D+02,&
&        0.3664496D+02,0.1549396D+02,0.6891138D+01,0.3054831D+01,0.1314241D+01,&
&        0.5134300D+00,0.1877000D+00,0.3618500D+03,0.1085500D+03,0.4143300D+02,&
&        0.1757900D+02,0.7862700D+01,0.3518000D+01,0.1534800D+01,0.6081300D+00,&
&        0.2220000D+00,0.4038300D+03,0.1211700D+03,0.4634500D+02,0.1972100D+02,&
&        0.8862400D+01,0.3996200D+01,0.1763600D+01,0.7061900D+00,0.2639000D+00,&
&        0.4461613D+03,0.1339648D+03,0.5134591D+02,0.2191691D+02,0.9893725D+01,&
&        0.4492527D+01,0.2002293D+01,0.8084090D+00,0.3006000D+00/)
      real(8) :: coeffd(280)= &
&     (/ 0.15473000D-01,0.78874000D-01,0.20878000D+00,0.33021300D+00,0.10000000D+01,&
&        0.10000000D+01,0.42660000D-02,0.27708000D-01,0.10000100D+00,0.23158100D+00,&
&        0.34603300D+00,0.37337400D+00,0.26428800D+00,-.43890000D-02,-.28363000D-01,&
&        -.10513700D+00,-.23485400D+00,-.32460900D+00,-.64289000D-01,0.60174900D+00,&
&        0.58590000D-02,0.37323000D-01,0.14192400D+00,0.30689600D+00,0.30818900D+00,&
&        -.76071700D+00,-.20477500D+00,0.10000000D+01,0.38870000D-02,0.26399000D-01,&
&        0.97511000D-01,0.23284800D+00,0.35315200D+00,0.37218600D+00,0.24767200D+00,&
&        -.39700000D-02,-.26873000D-01,-.10227500D+00,-.23772800D+00,-.31211400D+00,&
&        -.42378000D-01,0.58865800D+00,0.64180000D-02,0.43809000D-01,0.17232500D+00,&
&        0.37533200D+00,0.22494700D+00,-.79190700D+00,-.10442800D+00,0.10000000D+01,&
&        0.35950000D-02,0.25210000D-01,0.94786000D-01,0.23036300D+00,0.35289400D+00,&
&        0.37041400D+00,0.24571800D+00,-.38180000D-02,-.26717000D-01,-.10369000D+00,&
&        -.24768900D+00,-.31152300D+00,-.22827000D-01,0.56972600D+00,0.60010000D-02,&
&        0.42206000D-01,0.17075100D+00,0.38551800D+00,0.20620400D+00,-.77866900D+00,&
&        -.11472100D+00,0.10000000D+01,0.36210000D-02,0.25766000D-01,0.97556000D-01,&
&        0.23631200D+00,0.35828600D+00,0.36854300D+00,0.23549400D+00,-.41220000D-02,&
&        -.29307000D-01,-.11506200D+00,-.27306800D+00,-.31442300D+00,0.42097000D-01,&
&        0.59140300D+00,0.59540000D-02,0.42532000D-01,0.17451100D+00,0.39390000D+00,&
&        0.14927900D+00,-.81024500D+00,0.10911000D-01,0.10000000D+01,0.35790000D-02,&
&        0.25827000D-01,0.98559000D-01,0.23832700D+00,0.35870700D+00,0.36509200D+00,&
&        0.23373800D+00,-.34540000D-02,-.24925000D-01,-.97635000D-01,-.23669200D+00,&
&        -.29235000D+00,-.49730000D-02,0.50658800D+00,0.56850000D-02,0.41171000D-01,&
&        0.16932200D+00,0.38591200D+00,0.18690900D+00,-.67811300D+00,-.25826000D+00,&
&        0.10000000D+01,0.35300000D-02,0.25784000D-01,0.99119000D-01,0.23907300D+00,&
&        0.35719900D+00,0.36218800D+00,0.23646100D+00,-.38900000D-02,-.28442000D-01,&
&        -.11242900D+00,-.27425700D+00,-.31554600D+00,0.57109000D-01,0.56360400D+00,&
&        0.56950000D-02,0.42001000D-01,0.17354000D+00,0.41015700D+00,0.11325200D+00,&
&        -.76968000D+00,-.31643000D-01,0.10000000D+01,0.35100000D-02,0.25884000D-01,&
&        0.10005800D+00,0.24054700D+00,0.35684300D+00,0.35957900D+00,0.23662900D+00,&
&        -.40670000D-02,-.30053000D-01,-.11962000D+00,-.29151300D+00,-.31804800D+00,&
&        0.91698000D-01,0.56082300D+00,0.54700000D-02,0.40813000D-01,0.16897800D+00,&
&        0.40935900D+00,0.86316000D-01,-.76900800D+00,0.33550000D-02,0.10000000D+01,&
&        0.33760000D-02,0.25141000D-01,0.97746000D-01,0.23470900D+00,0.34694500D+00,&
&        0.35106800D+00,0.25025500D+00,-.34950000D-02,-.26015000D-01,-.10387600D+00,&
&        -.25207000D+00,-.29458000D+00,0.11520000D-02,0.43858900D+00,0.50520000D-02,&
&        0.38087000D-01,0.15611300D+00,0.38615600D+00,0.17560500D+00,-.62680700D+00,&
&        -.34277500D+00,0.10000000D+01,0.27000000D-02,0.20909000D-01,0.84408000D-01,&
&        0.21399900D+00,0.33598000D+00,0.35730100D+00,0.26457800D+00,-.33630000D-02,&
&        -.26079000D-01,-.10823100D+00,-.28221700D+00,-.34719000D+00,0.26711000D-01,&
&        0.49204700D+00,0.41330000D-02,0.33085000D-01,0.13833600D+00,0.39016600D+00,&
&        0.16984200D+00,-.68301800D+00,-.26579700D+00,0.10000000D+01,0.23420000D-02,&
&        0.18606000D-01,0.77102000D-01,0.20202600D+00,0.32945400D+00,0.36097600D+00,&
&        0.27165700D+00,0.32790000D-02,0.26176000D-01,0.11136700D+00,0.30458100D+00,&
&        0.38629900D+00,-.58375000D-01,-.53887600D+00,0.37400000D-02,0.31825000D-01,&
&        0.13222900D+00,0.42455000D+00,0.12037000D+00,-.76266100D+00,-.11282300D+00,&
&        0.10000000D+01,0.20270000D-02,0.16508800D-01,0.70382300D-01,0.19114300D+00,&
&        0.32540920D+00,0.36781990D+00,0.27446850D+00,0.10000000D+01,0.10000000D+01,&
&        0.18275000D-02,0.15154500D-01,0.66046000D-01,0.18394700D+00,0.32278720D+00,&
&        0.37294590D+00,0.27517300D+00,0.10000000D+01,0.10000000D+01,0.16840000D-02,&
&        0.14158600D-01,0.62825900D-01,0.17849930D+00,0.32094520D+00,0.37735150D+00,&
&        0.27502310D+00,0.10000000D+01,0.10000000D+01,0.15655000D-02,0.13326200D-01,&
&        0.60152700D-01,0.17402930D+00,0.31956900D+00,0.38120290D+00,0.27460860D+00,&
&        0.10000000D+01,0.10000000D+01,0.14732000D-02,0.12672500D-01,0.58045100D-01,&
&        0.17051030D+00,0.31859580D+00,0.38450230D+00,0.27377370D+00,0.10000000D+01,&
&        0.10000000D+01,0.14044000D-02,0.12171500D-01,0.56391900D-01,0.16764300D+00,&
&        0.31773680D+00,0.38726470D+00,0.27280060D+00,0.10000000D+01,0.10000000D+01/)
      real(8) :: expf(43)= &
&     (/ 0.1829000D+00,0.3250000D+00,0.4900000D+00,0.7610000D+00,0.1093000D+01,&
&        0.1428000D+01,0.1917000D+01,0.2544000D+01,0.1397000D+00,0.2520000D+00,&
&        0.2440000D+00,0.3360000D+00,0.4520000D+00,0.5570000D+00,0.7060000D+00,&
&        0.8900000D+00,0.1509000D+00,0.7267000D+00,0.1665000D+00,0.1248300D+01,&
&        0.2836000D+00,0.1774900D+01,0.4125000D+00,0.2251000D+01,0.5315000D+00,&
&        0.2738200D+01,0.6543000D+00,0.3275800D+01,0.7920000D+00,0.3813700D+01,&
&        0.9309000D+00,0.4504600D+01,0.1128600D+01,0.5082100D+01,0.1279700D+01,&
&        0.5792200D+01,0.1485100D+01,0.2881000D+00,0.3458000D+00,0.4158000D+00,&
&        0.4620000D+00,0.5515000D+00,0.6622000D+00/)
      real(8) :: expg(10)= &
&     (/ 0.4187000D+00,0.7251000D+00,0.1136800D+01,0.1445900D+01,0.1733500D+01,&
&        0.2089700D+01,0.2470900D+01,0.2977100D+01,0.3483500D+01,0.4114400D+01/)
!
      select case(numatomic(iatom))
!
! Set H - He functions
!
        case(1:2)
!  S function
          ishell= ishell+1
          num=4
          if(numatomic(iatom) == 1) num=3
          do j= 1,num
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= num
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+num
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j+num)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+num)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j+num+1)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+num+1)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!  P function
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j+1)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+1)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!  D function
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expd(ide(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= one
          enddo
          mprim(ishell)= 1
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1) = locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1) = locbf(ishell)+6
          endif
!
! Set Li - Ne functions
!
        case(3:10)
!  S function
          ishell= ishell+1
          num= 8
          if(numatomic(iatom) < 5) num= 9
          do j= 1,num
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= num
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+num
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,num
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+num)
          enddo
          mprim(ishell)= num
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+num
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j+num)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+num*2)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j+num+1)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+num*2+1)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!  P function
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+3)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j+4)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+4)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!  D function
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expd(ide(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= one
          enddo
          mprim(ishell)= 1
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1) = locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1) = locbf(ishell)+6
          endif
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expd(ide(numatomic(iatom))+j+1)
            coeff(locprim(ishell)+j)= one
          enddo
          mprim(ishell)= 1
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1) = locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1) = locbf(ishell)+6
          endif
!  F function
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expf(ife(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= one
          enddo
          mprim(ishell)= 1
          mtype(ishell)= 3
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 7
            locbf(ishell+1) = locbf(ishell)+7
          else
            mbf(ishell)= 10
            locbf(ishell+1) = locbf(ishell)+10
          endif
!
! Set Na - Ar functions
!
        case(11:18)
!  S function
          ishell= ishell+1
          num= 13
          if(numatomic(iatom) == 11) num= 14
          do j= 1,num
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= num
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+num
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,num
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+num)
          enddo
          mprim(ishell)= num
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+num
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,num
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+num*2)
          enddo
          mprim(ishell)= num
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+num
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j+num)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+num*3)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j+num+1)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+num*3+1)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!  P function
          ishell= ishell+1
          num= 7
          if(numatomic(iatom) < 13) num= 8
          do j= 1,num
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= num
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+num
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,num
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+num)
          enddo
          mprim(ishell)= num
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+num
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j+num)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+num*2)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j+num+1)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+num*2+1)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!  D function
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expd(ide(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= one
          enddo
          mprim(ishell)= 1
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1) = locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1) = locbf(ishell)+6
          endif
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expd(ide(numatomic(iatom))+j+1)
            coeff(locprim(ishell)+j)= one
          enddo
          mprim(ishell)= 1
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1) = locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1) = locbf(ishell)+6
          endif
!  F function
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expf(ife(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= one
          enddo
          mprim(ishell)= 1
          mtype(ishell)= 3
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 7
            locbf(ishell+1) = locbf(ishell)+7
          else
            mbf(ishell)= 10
            locbf(ishell+1) = locbf(ishell)+10
          endif
!
! Set Ca, Ga - Kr functions
!
        case(20,31:36)
!  S function
          do k= 0,3
            ishell= ishell+1
            do j= 1,18
              ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
              coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+18*k)
            enddo
            mprim(ishell)= 18
            mbf(ishell)= 1
            mtype(ishell)= 0
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+18
            locbf(ishell+1) = locbf(ishell)+1
          enddo
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j+18)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+72)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j+19)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+73)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!  P function
          num= 12
          if(numatomic(iatom) >= 31) num= 11
          do k=0,2
            ishell= ishell+1
            do j= 1,num
              ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j)
              coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+num*k)
            enddo
            mprim(ishell)= num
            mbf(ishell)= 3
            mtype(ishell)= 1
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+num
            locbf(ishell+1) = locbf(ishell)+3
          enddo
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j+num)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+num*3)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j+num+1)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+num*3+1)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!  D function
          ishell= ishell+1
          num= 4
          if(numatomic(iatom) >= 31) num= 7
          do j= 1,num
            ex(locprim(ishell)+j)= expd(ide(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffd(idc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= num
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+num
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1)= locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1)= locbf(ishell)+6
          endif
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expd(ide(numatomic(iatom))+j+num)
            coeff(locprim(ishell)+j)= coeffd(idc(numatomic(iatom))+j+num)
          enddo
          mprim(ishell)= 1
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1)= locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1)= locbf(ishell)+6
          endif
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expd(ide(numatomic(iatom))+j+num+1)
            coeff(locprim(ishell)+j)= coeffd(idc(numatomic(iatom))+j+num+1)
          enddo
          mprim(ishell)= 1
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1)= locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1)= locbf(ishell)+6
          endif
!  F function
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expf(ife(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= one
          enddo
          mprim(ishell)= 1
          mtype(ishell)= 3
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 7
            locbf(ishell+1)= locbf(ishell)+7
          else
            mbf(ishell)= 10
            locbf(ishell+1)= locbf(ishell)+10
          endif
!
! Set Sc - Zn functions
!
        case(21:30)
!  S function
          do k= 0,5
            ishell= ishell+1
            do j= 1,19
              ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
              coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+19*k)
            enddo
            mprim(ishell)= 19
            mbf(ishell)= 1
            mtype(ishell)= 0
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+19
            locbf(ishell+1) = locbf(ishell)+1
          enddo
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j+19)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+114)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!  P function
          do k=0,4
            ishell= ishell+1
            do j= 1,15
              ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j)
              coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+15*k)
            enddo
            mprim(ishell)= 15
            mbf(ishell)= 3
            mtype(ishell)= 1
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+15
            locbf(ishell+1) = locbf(ishell)+3
          enddo
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j+15)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+75)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!  D function
          do k=0,2
            ishell= ishell+1
            do j= 1,7
              ex(locprim(ishell)+j)= expd(ide(numatomic(iatom))+j)
              coeff(locprim(ishell)+j)= coeffd(idc(numatomic(iatom))+j+7*k)
            enddo
            mprim(ishell)= 7
            mtype(ishell)= 2
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+7
            if(spher) then
              mbf(ishell)= 5
              locbf(ishell+1)= locbf(ishell)+5
            else
              mbf(ishell)= 6
              locbf(ishell+1)= locbf(ishell)+6
            endif
          enddo
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expd(ide(numatomic(iatom))+j+7)
            coeff(locprim(ishell)+j)= coeffd(idc(numatomic(iatom))+j+21)
          enddo
          mprim(ishell)= 1
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1)= locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1)= locbf(ishell)+6
          endif
!  F function
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expf(ife(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= one
          enddo
          mprim(ishell)= 1
          mtype(ishell)= 3
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 7
            locbf(ishell+1)= locbf(ishell)+7
          else
            mbf(ishell)= 10
            locbf(ishell+1)= locbf(ishell)+10
          endif
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expf(ife(numatomic(iatom))+j+1)
            coeff(locprim(ishell)+j)= one
          enddo
          mprim(ishell)= 1
          mtype(ishell)= 3
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 7
            locbf(ishell+1)= locbf(ishell)+7
          else
            mbf(ishell)= 10
            locbf(ishell+1)= locbf(ishell)+10
          endif
! G function
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expg(ige(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= one
          enddo
          mprim(ishell)= 1
          mtype(ishell)= 4
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 9
            locbf(ishell+1)= locbf(ishell)+9
          else
            mbf(ishell)= 15
            locbf(ishell+1)= locbf(ishell)+15
          endif
        case(:0)
        case default
          write(*,'(" Error! This program supports H - Ar, Ca - Kr cc-PVTZ basis set.")')
          call iabort
      endselect
!
      if(ishell > mxshell) then
        write(*,'(" Error! The number of basis shells exceeds mxshell",i6,".")')mxshell
        call iabort
      endif
      if(locprim(ishell+1) > mxprim ) then
        write(*,'(" Error! The number of primitive basis functions exceeds mxprim", &
&             i6,".")')mxprim
        call iabort
      endif
      if(locbf(ishell+1) > mxao ) then
        write(*,'(" Error! The number of basis functions exceeds mxao",i6,".")')mxao
        call iabort
      endif
      return
end


!------------------------------------
  subroutine bsccpvqz(iatom,ishell)
!------------------------------------
!
! Set basis functions of cc-pVQZ
!
      use modmolecule, only : numatomic
      use modbasis, only : ex, coeff, locprim, locbf, locatom, mprim, mbf, mtype, spher
      use modparam, only : mxao, mxshell, mxprim
      implicit none
      integer,intent(in) :: iatom
      integer,intent(inout) :: ishell
      integer :: j, k, nprim
      integer :: ise(18)=(/0,6, 13,25,37,49,61,73,85,97, 109,128,144,160,176,192,208,224/)
      integer :: isc(18)=(/0,6, 13,34,55,76,97,118,139,160, 181,232,274,316,358,400,442,484/)
      integer :: ipe(18)=(/0,3, 6,12,18,24,30,36,42,48, 54,66,78,89,100,111,122,133/)
      integer :: ipc(18)=(/0,3, 6,12,18,24,30,36,42,48, 54,75,96,115,134,153,172,191/)
      integer :: ide(18)=(/0,2, 4,7,10,13,16,19,22,25, 28,31,34,37,40,43,46,49/)
      integer :: ife(18)=(/0,1, 2,4,6,8,10,12,14,16, 18,20,22,24,26,28,30,32/)
      integer :: ige(18)=(/0,0, 0,1,2,3,4,5,6,7, 8,9,10,11,12,13,14,15/)
      real(8),parameter :: one=1.0D+00
      real(8) :: exps(240)= &
&     (/ 0.8264000D+02,0.1241000D+02,0.2824000D+01,0.7977000D+00,0.2581000D+00,&
&        0.8989000D-01,0.5285000D+03,0.7931000D+02,0.1805000D+02,0.5085000D+01,&
&        0.1609000D+01,0.5363000D+00,0.1833000D+00,0.6601000D+04,0.9897000D+03,&
&        0.2257000D+03,0.6429000D+02,0.2118000D+02,0.7724000D+01,0.3003000D+01,&
&        0.1212000D+01,0.4930000D+00,0.9515000D-01,0.4791000D-01,0.2220000D-01,&
&        0.1463000D+05,0.2191000D+04,0.4982000D+03,0.1409000D+03,0.4586000D+02,&
&        0.1647000D+02,0.6319000D+01,0.2535000D+01,0.1035000D+01,0.2528000D+00,&
&        0.1052000D+00,0.4261000D-01,0.2387000D+05,0.3575000D+04,0.8128000D+03,&
&        0.2297000D+03,0.7469000D+02,0.2681000D+02,0.1032000D+02,0.4178000D+01,&
&        0.1727000D+01,0.4704000D+00,0.1896000D+00,0.7394000D-01,0.3398000D+05,&
&        0.5089000D+04,0.1157000D+04,0.3266000D+03,0.1061000D+03,0.3811000D+02,&
&        0.1475000D+02,0.6035000D+01,0.2530000D+01,0.7355000D+00,0.2905000D+00,&
&        0.1111000D+00,0.4584000D+05,0.6868000D+04,0.1563000D+04,0.4424000D+03,&
&        0.1443000D+03,0.5218000D+02,0.2034000D+02,0.8381000D+01,0.3529000D+01,&
&        0.1054000D+01,0.4118000D+00,0.1552000D+00,0.6142000D+05,0.9199000D+04,&
&        0.2091000D+04,0.5909000D+03,0.1923000D+03,0.6932000D+02,0.2697000D+02,&
&        0.1110000D+02,0.4682000D+01,0.1428000D+01,0.5547000D+00,0.2067000D+00,&
&        0.7453000D+05,0.1117000D+05,0.2543000D+04,0.7210000D+03,0.2359000D+03,&
&        0.8560000D+02,0.3355000D+02,0.1393000D+02,0.5915000D+01,0.1843000D+01,&
&        0.7124000D+00,0.2637000D+00,0.9992000D+05,0.1496000D+05,0.3399000D+04,&
&        0.9589000D+03,0.3112000D+03,0.1117000D+03,0.4332000D+02,0.1780000D+02,&
&        0.7503000D+01,0.2337000D+01,0.9001000D+00,0.3301000D+00,0.1224000D+07,&
&        0.1832000D+06,0.4170000D+05,0.1181000D+05,0.3853000D+04,0.1391000D+04,&
&        0.5425000D+03,0.2249000D+03,0.9793000D+02,0.4431000D+02,0.2065000D+02,&
&        0.9729000D+01,0.4228000D+01,0.1969000D+01,0.8890000D+00,0.3964000D+00,&
&        0.6993000D-01,0.3289000D-01,0.1612000D-01,0.3276000D+06,0.4905000D+05,&
&        0.1115000D+05,0.3152000D+04,0.1025000D+04,0.3688000D+03,0.1432000D+03,&
&        0.5896000D+02,0.2540000D+02,0.1115000D+02,0.4004000D+01,0.1701000D+01,&
&        0.7060000D+00,0.1410000D+00,0.6808000D-01,0.3063000D-01,0.4196000D+06,&
&        0.6283000D+05,0.1429000D+05,0.4038000D+04,0.1312000D+04,0.4705000D+03,&
&        0.1818000D+03,0.7446000D+02,0.3190000D+02,0.1396000D+02,0.5180000D+01,&
&        0.2265000D+01,0.9664000D+00,0.2447000D+00,0.1184000D+00,0.5021000D-01,&
&        0.5130000D+06,0.7682000D+05,0.1747000D+05,0.4935000D+04,0.1602000D+04,&
&        0.5741000D+03,0.2215000D+03,0.9054000D+02,0.3874000D+02,0.1695000D+02,&
&        0.6452000D+01,0.2874000D+01,0.1250000D+01,0.3599000D+00,0.1699000D+00,&
&        0.7066000D-01,0.6152000D+06,0.9212000D+05,0.2095000D+05,0.5920000D+04,&
&        0.1922000D+04,0.6880000D+03,0.2650000D+03,0.1082000D+03,0.4622000D+02,&
&        0.2023000D+02,0.7859000D+01,0.3547000D+01,0.1564000D+01,0.4888000D+00,&
&        0.2266000D+00,0.9331000D-01,0.7278000D+06,0.1090000D+06,0.2480000D+05,&
&        0.7014000D+04,0.2278000D+04,0.8147000D+03,0.3134000D+03,0.1277000D+03,&
&        0.5448000D+02,0.2385000D+02,0.9428000D+01,0.4290000D+01,0.1909000D+01,&
&        0.6270000D+00,0.2873000D+00,0.1172000D+00,0.8349000D+06,0.1250000D+06,&
&        0.2843000D+05,0.8033000D+04,0.2608000D+04,0.9339000D+03,0.3600000D+03,&
&        0.1470000D+03,0.6288000D+02,0.2760000D+02,0.1108000D+02,0.5075000D+01,&
&        0.2278000D+01,0.7775000D+00,0.3527000D+00,0.1431000D+00,0.9506000D+06,&
&        0.1423000D+06,0.3236000D+05,0.9145000D+04,0.2970000D+04,0.1064000D+04,&
&        0.4108000D+03,0.1680000D+03,0.7199000D+02,0.3167000D+02,0.1289000D+02,&
&        0.5929000D+01,0.2678000D+01,0.9416000D+00,0.4239000D+00,0.1714000D+00/)
      real(8) :: coeffs(526)= &
&     (/ 0.2006000D-02,0.1534300D-01,0.7557900D-01,0.1000000D+01,0.1000000D+01,&
&        0.1000000D+01,0.9400000D-03,0.7214000D-02,0.3597500D-01,0.1277820D+00,&
&        0.1000000D+01,0.1000000D+01,0.1000000D+01,0.1170000D-03,0.9110000D-03,&
&        0.4728000D-02,0.1919700D-01,0.6304700D-01,0.1632080D+00,0.3148270D+00,&
&        0.3939360D+00,0.1969180D+00,-.1800000D-04,-.1420000D-03,-.7410000D-03,&
&        -.3020000D-02,-.1012300D-01,-.2709400D-01,-.5735900D-01,-.9389500D-01,&
&        -.1210910D+00,0.1000000D+01,0.1000000D+01,0.1000000D+01,0.9200000D-04,&
&        0.7130000D-03,0.3735000D-02,0.1546800D-01,0.5287400D-01,0.1456940D+00,&
&        0.3026810D+00,0.4049360D+00,0.2223870D+00,-.1700000D-04,-.1300000D-03,&
&        -.6790000D-03,-.2857000D-02,-.9813000D-02,-.2860900D-01,-.6376000D-01,&
&        -.1172310D+00,-.1212020D+00,0.1000000D+01,0.1000000D+01,0.1000000D+01,&
&        0.8800000D-04,0.6870000D-03,0.3600000D-02,0.1494900D-01,0.5143500D-01,&
&        0.1433020D+00,0.3009350D+00,0.4035260D+00,0.2253400D+00,-.1800000D-04,&
&        -.1390000D-03,-.7250000D-03,-.3063000D-02,-.1058100D-01,-.3136500D-01,&
&        -.7101200D-01,-.1321030D+00,-.1230720D+00,0.1000000D+01,0.1000000D+01,&
&        0.1000000D+01,0.9100000D-04,0.7040000D-03,0.3693000D-02,0.1536000D-01,&
&        0.5292900D-01,0.1470430D+00,0.3056310D+00,0.3993450D+00,0.2170510D+00,&
&        -.1900000D-04,-.1510000D-03,-.7850000D-03,-.3324000D-02,-.1151200D-01,&
&        -.3416000D-01,-.7717300D-01,-.1414930D+00,-.1180190D+00,0.1000000D+01,&
&        0.1000000D+01,0.1000000D+01,0.9200000D-04,0.7170000D-03,0.3749000D-02,&
&        0.1553200D-01,0.5314600D-01,0.1467870D+00,0.3046630D+00,0.3976840D+00,&
&        0.2176410D+00,-.2000000D-04,-.1590000D-03,-.8240000D-03,-.3478000D-02,&
&        -.1196600D-01,-.3538800D-01,-.8007700D-01,-.1467220D+00,-.1163600D+00,&
&        0.1000000D+01,0.1000000D+01,0.1000000D+01,0.9000000D-04,0.6980000D-03,&
&        0.3664000D-02,0.1521800D-01,0.5242300D-01,0.1459210D+00,0.3052580D+00,&
&        0.3985080D+00,0.2169800D+00,-.2000000D-04,-.1590000D-03,-.8290000D-03,&
&        -.3508000D-02,-.1215600D-01,-.3626100D-01,-.8299200D-01,-.1520900D+00,&
&        -.1153310D+00,0.1000000D+01,0.1000000D+01,0.1000000D+01,0.9500000D-04,&
&        0.7380000D-03,0.3858000D-02,0.1592600D-01,0.5428900D-01,0.1495130D+00,&
&        0.3082520D+00,0.3948530D+00,0.2110310D+00,-.2200000D-04,-.1720000D-03,&
&        -.8910000D-03,-.3748000D-02,-.1286200D-01,-.3806100D-01,-.8623900D-01,&
&        -.1558650D+00,-.1109140D+00,0.1000000D+01,0.1000000D+01,0.1000000D+01,&
&        0.8600000D-04,0.6690000D-03,0.3518000D-02,0.1466700D-01,0.5096200D-01,&
&        0.1437440D+00,0.3045620D+00,0.4001050D+00,0.2186440D+00,-.2000000D-04,&
&        -.1580000D-03,-.8240000D-03,-.3500000D-02,-.1223300D-01,-.3701700D-01,&
&        -.8611300D-01,-.1583810D+00,-.1142880D+00,0.1000000D+01,0.1000000D+01,&
&        0.1000000D+01,0.4788940D-05,0.3723950D-04,0.1958310D-03,0.8266980D-03,&
&        0.3002510D-02,0.9703100D-02,0.2823370D-01,0.7320580D-01,0.1628970D+00,&
&        0.2887080D+00,0.3468290D+00,0.2068650D+00,0.3280090D-01,-.6477360D-03,&
&        0.1458780D-02,-.1783460D-03,-.1169580D-05,-.9091100D-05,-.4784990D-04,&
&        -.2019620D-03,-.7358370D-03,-.2387460D-02,-.7049690D-02,-.1878560D-01,&
&        -.4461530D-01,-.8977410D-01,-.1429400D+00,-.1243150D+00,0.9996480D-01,&
&        0.4170800D+00,0.4751230D+00,0.1632680D+00,0.1758710D-06,0.1365940D-05,&
&        0.7197950D-05,0.3033490D-04,0.1107520D-03,0.3585960D-03,0.1062720D-02,&
&        0.2826870D-02,0.6767420D-02,0.1364800D-01,0.2228140D-01,0.1960110D-01,&
&        -.1677080D-01,-.7737340D-01,-.1135010D+00,-.1391300D+00,0.1000000D+01,&
&        0.1000000D+01,0.1000000D+01,0.3096080D-04,0.2409540D-03,0.1266600D-02,&
&        0.5333590D-02,0.1907700D-01,0.5880580D-01,0.1514540D+00,0.3007160D+00,&
&        0.3811490D+00,0.2135840D+00,0.2312100D-01,-.2307570D-02,0.1289000D-02,&
&        -.7831730D-05,-.6079350D-04,-.3211970D-03,-.1349550D-02,-.4905700D-02,&
&        -.1535610D-01,-.4234090D-01,-.9406030D-01,-.1634250D+00,-.1247540D+00,&
&        0.2356230D+00,0.5775630D+00,0.3352320D+00,0.1509080D-05,0.1171340D-04,&
&        0.6189800D-04,0.2600880D-03,0.9462180D-03,0.2965950D-02,0.8212450D-02,&
&        0.1839770D-01,0.3266570D-01,0.2573150D-01,-.5353510D-01,-.1568950D+00,&
&        -.2066590D+00,0.1000000D+01,0.1000000D+01,0.1000000D+01,0.2782190D-04,&
&        0.2163300D-03,0.1137540D-02,0.4796350D-02,0.1723890D-01,0.5380660D-01,&
&        0.1413260D+00,0.2892680D+00,0.3848250D+00,0.2328520D+00,0.2933300D-01,&
&        -.3005740D-02,0.1666730D-02,-.7237540D-05,-.5617330D-04,-.2965280D-03,&
&        -.1249130D-02,-.4551010D-02,-.1443930D-01,-.4034640D-01,-.9226180D-01,&
&        -.1645100D+00,-.1412960D+00,0.1953650D+00,0.5724750D+00,0.3740410D+00,&
&        0.1671500D-05,0.1296410D-04,0.6851010D-04,0.2882740D-03,0.1052760D-02,&
&        0.3338780D-02,0.9392170D-02,0.2160470D-01,0.3958730D-01,0.3491800D-01,&
&        -.5284150D-01,-.1918780D+00,-.2541150D+00,0.1000000D+01,0.1000000D+01,&
&        0.1000000D+01,0.2609200D-04,0.2029050D-03,0.1067150D-02,0.4505970D-02,&
&        0.1623590D-01,0.5089130D-01,0.1351550D+00,0.2812920D+00,0.3853360D+00,&
&        0.2456510D+00,0.3431450D-01,-.3348840D-02,0.1876250D-02,-.6948800D-05,&
&        -.5396410D-04,-.2847160D-03,-.1202030D-02,-.4383970D-02,-.1397760D-01,&
&        -.3935160D-01,-.9142830D-01,-.1656090D+00,-.1525050D+00,0.1685240D+00,&
&        0.5692840D+00,0.3980560D+00,0.1780680D-05,0.1381480D-04,0.7300050D-04,&
&        0.3076660D-03,0.1125630D-02,0.3584350D-02,0.1017280D-01,0.2375200D-01,&
&        0.4434830D-01,0.4190410D-01,-.5025040D-01,-.2165780D+00,-.2864480D+00,&
&        0.1000000D+01,0.1000000D+01,0.1000000D+01,0.2474500D-04,0.1924650D-03,&
&        0.1012020D-02,0.4272610D-02,0.1541610D-01,0.4859760D-01,0.1300600D+00,&
&        0.2745140D+00,0.3854020D+00,0.2559340D+00,0.3912370D-01,-.3680100D-02,&
&        0.2082110D-02,-.6722050D-05,-.5223110D-04,-.2753610D-03,-.1163070D-02,&
&        -.4242810D-02,-.1361140D-01,-.3851140D-01,-.9066430D-01,-.1665840D+00,&
&        -.1614470D+00,0.1467810D+00,0.5666820D+00,0.4164330D+00,0.1847400D-05,&
&        0.1433800D-04,0.7572280D-04,0.3192050D-03,0.1168510D-02,0.3742670D-02,&
&        0.1068170D-01,0.2526570D-01,0.4792830D-01,0.4770960D-01,-.4665250D-01,&
&        -.2349680D+00,-.3113370D+00,0.1000000D+01,0.1000000D+01,0.1000000D+01,&
&        0.2360250D-04,0.1834820D-03,0.9642780D-03,0.4065370D-02,0.1469730D-01,&
&        0.4650810D-01,0.1255080D+00,0.2684330D+00,0.3848090D+00,0.2653720D+00,&
&        0.4373260D-01,-.3788070D-02,0.2180830D-02,-.6521790D-05,-.5066310D-04,&
&        -.2668330D-03,-.1126010D-02,-.4111860D-02,-.1324540D-01,-.3770040D-01,&
&        -.8985540D-01,-.1670980D+00,-.1693540D+00,0.1278240D+00,0.5648620D+00,&
&        0.4317670D+00,0.1894060D-05,0.1469480D-04,0.7754600D-04,0.3265090D-03,&
&        0.1196860D-02,0.3847990D-02,0.1105390D-01,0.2646450D-01,0.5087710D-01,&
&        0.5300300D-01,-.4255180D-01,-.2508530D+00,-.3331520D+00,0.1000000D+01,&
&        0.1000000D+01,0.1000000D+01,0.2316880D-04,0.1801540D-03,0.9477820D-03,&
&        0.4001390D-02,0.1446290D-01,0.4565860D-01,0.1232480D+00,0.2643690D+00,&
&        0.3829890D+00,0.2709340D+00,0.4714040D-01,-.3717660D-02,0.2191580D-02,&
&        -.6496490D-05,-.5048950D-04,-.2661130D-03,-.1124990D-02,-.4104970D-02,&
&        -.1319870D-01,-.3753420D-01,-.8972330D-01,-.1676710D+00,-.1747630D+00,&
&        0.1149090D+00,0.5636180D+00,0.4416060D+00,0.1966450D-05,0.1526200D-04,&
&        0.8060860D-04,0.3399600D-03,0.1245510D-02,0.3996120D-02,0.1147510D-01,&
&        0.2755040D-01,0.5329170D-01,0.5712460D-01,-.3952010D-01,-.2643430D+00,&
&        -.3492910D+00,0.1000000D+01,0.1000000D+01,0.1000000D+01,0.2275450D-04,&
&        0.1769450D-03,0.9312820D-03,0.3928600D-02,0.1420640D-01,0.4481140D-01,&
&        0.1210010D+00,0.2605790D+00,0.3813640D+00,0.2760580D+00,0.5051790D-01,&
&        -.3598660D-02,0.2187980D-02,-.6462010D-05,-.5023460D-04,-.2648040D-03,&
&        -.1118950D-02,-.4082760D-02,-.1312160D-01,-.3728550D-01,-.8947090D-01,&
&        -.1680540D+00,-.1795940D+00,0.1029530D+00,0.5626300D+00,0.4503550D+00,&
&        0.2020560D-05,0.1568510D-04,0.8286170D-04,0.3492640D-03,0.1279760D-02,&
&        0.4103650D-02,0.1177890D-01,0.2838680D-01,0.5524060D-01,0.6074920D-01,&
&        -.3620120D-01,-.2753980D+00,-.3628450D+00,0.1000000D+01,0.1000000D+01,&
&        0.1000000D+01/)
      real(8) :: expp(144)= &
&     (/ 0.2292000D+01,0.8380000D+00,0.2920000D+00,0.5994000D+01,0.1745000D+01,&
&        0.5600000D+00,0.6250000D+01,0.1370000D+01,0.3672000D+00,0.1192000D+00,&
&        0.4474000D-01,0.1795000D-01,0.1403000D+02,0.3168000D+01,0.9024000D+00,&
&        0.3036000D+00,0.1130000D+00,0.4286000D-01,0.2226000D+02,0.5058000D+01,&
&        0.1487000D+01,0.5071000D+00,0.1812000D+00,0.6463000D-01,0.3451000D+02,&
&        0.7915000D+01,0.2368000D+01,0.8132000D+00,0.2890000D+00,0.1007000D+00,&
&        0.4933000D+02,0.1137000D+02,0.3435000D+01,0.1182000D+01,0.4173000D+00,&
&        0.1428000D+00,0.6342000D+02,0.1466000D+02,0.4459000D+01,0.1531000D+01,&
&        0.5302000D+00,0.1750000D+00,0.8039000D+02,0.1863000D+02,0.5694000D+01,&
&        0.1953000D+01,0.6702000D+00,0.2166000D+00,0.9968000D+02,0.2315000D+02,&
&        0.7108000D+01,0.2441000D+01,0.8339000D+00,0.2662000D+00,0.4134000D+03,&
&        0.9798000D+02,0.3137000D+02,0.1162000D+02,0.4671000D+01,0.1918000D+01,&
&        0.7775000D+00,0.3013000D+00,0.2275000D+00,0.7527000D-01,0.3126000D-01,&
&        0.1342000D-01,0.5396000D+03,0.1279000D+03,0.4102000D+02,0.1525000D+02,&
&        0.6166000D+01,0.2561000D+01,0.1060000D+01,0.4176000D+00,0.2690000D+00,&
&        0.1223000D+00,0.5476000D-01,0.2388000D-01,0.8913000D+03,0.2113000D+03,&
&        0.6828000D+02,0.2570000D+02,0.1063000D+02,0.4602000D+01,0.2015000D+01,&
&        0.8706000D+00,0.2972000D+00,0.1100000D+00,0.3989000D-01,0.1122000D+04,&
&        0.2660000D+03,0.8592000D+02,0.3233000D+02,0.1337000D+02,0.5800000D+01,&
&        0.2559000D+01,0.1124000D+01,0.3988000D+00,0.1533000D+00,0.5728000D-01,&
&        0.1367000D+04,0.3240000D+03,0.1046000D+03,0.3937000D+02,0.1626000D+02,&
&        0.7056000D+01,0.3130000D+01,0.1394000D+01,0.5179000D+00,0.2032000D+00,&
&        0.7698000D-01,0.1546000D+04,0.3664000D+03,0.1184000D+03,0.4453000D+02,&
&        0.1838000D+02,0.7965000D+01,0.3541000D+01,0.1591000D+01,0.6205000D+00,&
&        0.2420000D+00,0.9014000D-01,0.1703000D+04,0.4036000D+03,0.1303000D+03,&
&        0.4905000D+02,0.2026000D+02,0.8787000D+01,0.3919000D+01,0.1765000D+01,&
&        0.7207000D+00,0.2839000D+00,0.1060000D+00,0.1890000D+04,0.4478000D+03,&
&        0.1446000D+03,0.5446000D+02,0.2251000D+02,0.9774000D+01,0.4368000D+01,&
&        0.1959000D+01,0.8260000D+00,0.3297000D+00,0.1242000D+00/)
      real(8) :: coeffp(210)= &
&     (/ 0.1000000D+01,0.1000000D+01,0.1000000D+01,0.1000000D+01,0.1000000D+01,&
&        0.1000000D+01,0.3388000D-02,0.1931600D-01,0.7910400D-01,0.1000000D+01,&
&        0.1000000D+01,0.1000000D+01,0.4099000D-02,0.2562600D-01,0.1037680D+00,&
&        0.1000000D+01,0.1000000D+01,0.1000000D+01,0.5095000D-02,0.3320600D-01,&
&        0.1323140D+00,0.1000000D+01,0.1000000D+01,0.1000000D+01,0.5378000D-02,&
&        0.3613200D-01,0.1424930D+00,0.1000000D+01,0.1000000D+01,0.1000000D+01,&
&        0.5533000D-02,0.3796200D-01,0.1490280D+00,0.1000000D+01,0.1000000D+01,&
&        0.1000000D+01,0.6044000D-02,0.4179900D-01,0.1611430D+00,0.1000000D+01,&
&        0.1000000D+01,0.1000000D+01,0.6347000D-02,0.4420400D-01,0.1685140D+00,&
&        0.1000000D+01,0.1000000D+01,0.1000000D+01,0.6566000D-02,0.4597900D-01,&
&        0.1734190D+00,0.1000000D+01,0.1000000D+01,0.1000000D+01,0.9081960D-03,&
&        0.7417730D-02,0.3574640D-01,0.1185200D+00,0.2614030D+00,0.3783950D+00,&
&        0.3346320D+00,0.1268440D+00,-.1471170D-01,-.9017410D-04,-.7393420D-03,&
&        -.3573090D-02,-.1201420D-01,-.2671780D-01,-.3927530D-01,-.3760830D-01,&
&        -.4332280D-01,0.5180030D-01,0.1000000D+01,0.1000000D+01,0.1000000D+01,&
&        0.8339690D-03,0.6892150D-02,0.3378740D-01,0.1144010D+00,0.2595140D+00,&
&        0.3850950D+00,0.3353730D+00,0.1106410D+00,-.1213150D-01,-.1320760D-03,&
&        -.1095380D-02,-.5394950D-02,-.1855720D-01,-.4273750D-01,-.6476840D-01,&
&        -.6278180D-01,-.2449120D-01,0.1047610D+00,0.1000000D+01,0.1000000D+01,&
&        0.1000000D+01,0.4917550D-03,0.4158430D-02,0.2125380D-01,0.7640580D-01,&
&        0.1942770D+00,0.3344280D+00,0.3750260D+00,0.2040410D+00,-.8886950D-04,&
&        -.7458230D-03,-.3870250D-02,-.1393500D-01,-.3668600D-01,-.6277970D-01,&
&        -.7896020D-01,-.2885890D-01,0.1000000D+01,0.1000000D+01,0.1000000D+01,&
&        0.4481430D-03,0.3816390D-02,0.1981050D-01,0.7270170D-01,0.1898390D+00,&
&        0.3356720D+00,0.3793650D+00,0.2011930D+00,-.9648830D-04,-.8119710D-03,&
&        -.4300870D-02,-.1575020D-01,-.4295410D-01,-.7525740D-01,-.9714460D-01,&
&        -.2275070D-01,0.1000000D+01,0.1000000D+01,0.1000000D+01,0.4210150D-03,&
&        0.3609850D-02,0.1892170D-01,0.7055600D-01,0.1881570D+00,0.3387090D+00,&
&        0.3819430D+00,0.1952610D+00,-.1008270D-03,-.8544990D-03,-.4571160D-02,&
&        -.1703270D-01,-.4752040D-01,-.8527860D-01,-.1096760D+00,-.1611810D-01,&
&        0.1000000D+01,0.1000000D+01,0.1000000D+01,0.4411830D-03,0.3775710D-02,&
&        0.1983600D-01,0.7420630D-01,0.1973270D+00,0.3518510D+00,0.3786870D+00,&
&        0.1709310D+00,-.1131100D-03,-.9585810D-03,-.5134710D-02,-.1926410D-01,&
&        -.5359800D-01,-.9603330D-01,-.1181830D+00,0.9231940D-02,0.1000000D+01,&
&        0.1000000D+01,0.1000000D+01,0.4740390D-03,0.4064120D-02,0.2133550D-01,&
&        0.7946110D-01,0.2089270D+00,0.3649450D+00,0.3717250D+00,0.1462920D+00,&
&        -.1282660D-03,-.1093560D-02,-.5834290D-02,-.2192580D-01,-.6013850D-01,&
&        -.1069290D+00,-.1224540D+00,0.3836190D-01,0.1000000D+01,0.1000000D+01,&
&        0.1000000D+01,0.4957520D-03,0.4251720D-02,0.2232770D-01,0.8308780D-01,&
&        0.2171100D+00,0.3745070D+00,0.3664450D+00,0.1292450D+00,-.1388630D-03,&
&        -.1188700D-02,-.6325530D-02,-.2388130D-01,-.6492380D-01,-.1154440D+00,&
&        -.1236510D+00,0.6490550D-01,0.1000000D+01,0.1000000D+01,0.1000000D+01/)
      real(8) :: expd(52)= &
&     (/ 0.2062000D+01,0.6620000D+00,0.4299000D+01,0.1223000D+01,0.3440000D+00,&
&        0.1530000D+00,0.6800000D-01,0.1072000D+01,0.4410000D+00,0.1811000D+00,&
&        0.1110000D+01,0.4020000D+00,0.1450000D+00,0.1848000D+01,0.6490000D+00,&
&        0.2280000D+00,0.2837000D+01,0.9680000D+00,0.3350000D+00,0.3775000D+01,&
&        0.1300000D+01,0.4440000D+00,0.5014000D+01,0.1725000D+01,0.5860000D+00,&
&        0.6471000D+01,0.2213000D+01,0.7470000D+00,0.1538000D+00,0.8650000D-01,&
&        0.4870000D-01,0.1060000D+00,0.1944000D+00,0.3570000D+00,0.8040000D-01,&
&        0.1990000D+00,0.4940000D+00,0.1200000D+00,0.3020000D+00,0.7600000D+00,&
&        0.1650000D+00,0.4130000D+00,0.1036000D+01,0.2030000D+00,0.5040000D+00,&
&        0.1250000D+01,0.2540000D+00,0.6280000D+00,0.1551000D+01,0.3110000D+00,&
&        0.7630000D+00,0.1873000D+01/)
      real(8) :: expf(34)= &
&     (/ 0.1397000D+01,0.2680000D+01,0.2460000D+00,0.1292000D+00,0.4810000D+00,&
&        0.2550000D+00,0.8820000D+00,0.3110000D+00,0.1419000D+01,0.4850000D+00,&
&        0.2027000D+01,0.6850000D+00,0.2666000D+01,0.8590000D+00,0.3562000D+01,&
&        0.1148000D+01,0.4657000D+01,0.1524000D+01,0.1912000D+00,0.1036000D+00,&
&        0.1810000D+00,0.3590000D+00,0.1540000D+00,0.4010000D+00,0.2120000D+00,&
&        0.5410000D+00,0.2800000D+00,0.7030000D+00,0.3350000D+00,0.8690000D+00,&
&        0.4230000D+00,0.1089000D+01,0.5430000D+00,0.1325000D+01/)
      real(8) :: expg(16)= &
&     (/ 0.2380000D+00,0.4150000D+00,0.6730000D+00,0.1011000D+01,0.1427000D+01,&
&        0.1846000D+01,0.2376000D+01,0.2983000D+01,0.1722000D+00,0.3070000D+00,&
&        0.3570000D+00,0.4610000D+00,0.5970000D+00,0.6830000D+00,0.8270000D+00,&
&        0.1007000D+01/)
!
      if(numatomic(iatom) > 18) then
        write(*,'(" Error! This program supports H - Ar cc-PVQZ basis set.")')
        call iabort
      endif
!
      select case(numatomic(iatom))
!
! Set H - He functions
!
        case(1:2)
!  S function
          ishell= ishell+1
          nprim=4
          if(numatomic(iatom) == 1) nprim=3
          do j= 1,nprim
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= nprim
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+nprim
          locbf(ishell+1) = locbf(ishell)+1
!
          do k= nprim,nprim+2
            ishell= ishell+1
            ex(locprim(ishell)+1)= exps(ise(numatomic(iatom))+1+k)
            coeff(locprim(ishell)+1)= coeffs(isc(numatomic(iatom))+1+k)
            mprim(ishell)= 1
            mbf(ishell)= 1
            mtype(ishell)= 0
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+1
            locbf(ishell+1) = locbf(ishell)+1
          enddo
!  P function
          do k= 0,2
            ishell= ishell+1
            ex(locprim(ishell)+1)= expp(ipe(numatomic(iatom))+1+k)
            coeff(locprim(ishell)+1)= coeffp(ipc(numatomic(iatom))+1+k)
            mprim(ishell)= 1
            mbf(ishell)= 3
            mtype(ishell)= 1
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+1
            locbf(ishell+1) = locbf(ishell)+3
          enddo
!
!  D function
          do k= 0,1
            ishell= ishell+1
            ex(locprim(ishell)+1)= expd(ide(numatomic(iatom))+1+k)
            coeff(locprim(ishell)+1)= one
            mprim(ishell)= 1
            mtype(ishell)= 2
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+1
            if(spher) then
              mbf(ishell)= 5
              locbf(ishell+1) = locbf(ishell)+5
            else
              mbf(ishell)= 6
              locbf(ishell+1) = locbf(ishell)+6
            endif
          enddo
!  F function
          ishell= ishell+1
          ex(locprim(ishell)+1)= expf(ife(numatomic(iatom))+1)
          coeff(locprim(ishell)+1)= one
          mprim(ishell)= 1
          mtype(ishell)= 3
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 7
            locbf(ishell+1) = locbf(ishell)+7
          else
            mbf(ishell)= 10
            locbf(ishell+1) = locbf(ishell)+10
          endif
!
! Set Li - Ne functions
!
        case(3:10)
!  S function
          do k= 0,1
            ishell= ishell+1
            do j= 1,9
              ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
              coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+k*9)
            enddo
            mprim(ishell)= 9
            mbf(ishell)= 1
            mtype(ishell)= 0
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+9
            locbf(ishell+1) = locbf(ishell)+1
          enddo
!
          do k= 0,2
            ishell= ishell+1
            ex(locprim(ishell)+1)= exps(ise(numatomic(iatom))+10+k)
            coeff(locprim(ishell)+1)= coeffs(isc(numatomic(iatom))+19+k)
            mprim(ishell)= 1
            mbf(ishell)= 1
            mtype(ishell)= 0
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+1
            locbf(ishell+1) = locbf(ishell)+1
          enddo
!  P function
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+3
!
          do k= 0,2
            ishell= ishell+1
            ex(locprim(ishell)+1)= expp(ipe(numatomic(iatom))+4+k)
            coeff(locprim(ishell)+1)= coeffp(ipc(numatomic(iatom))+4+k)
            mprim(ishell)= 1
            mbf(ishell)= 3
            mtype(ishell)= 1
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+1
            locbf(ishell+1) = locbf(ishell)+3
          enddo
!  D function
          do k= 0,2
            ishell= ishell+1
            ex(locprim(ishell)+1)= expd(ide(numatomic(iatom))+1+k)
            coeff(locprim(ishell)+1)= one
            mprim(ishell)= 1
            mtype(ishell)= 2
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+1
            if(spher) then
              mbf(ishell)= 5
              locbf(ishell+1) = locbf(ishell)+5
            else
              mbf(ishell)= 6
              locbf(ishell+1) = locbf(ishell)+6
            endif
          enddo
!  F function
          do k= 0,1
            ishell= ishell+1
            ex(locprim(ishell)+1)= expf(ife(numatomic(iatom))+1+k)
            coeff(locprim(ishell)+1)= one
            mprim(ishell)= 1
            mtype(ishell)= 3
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+1
            if(spher) then
              mbf(ishell)= 7
              locbf(ishell+1) = locbf(ishell)+7
            else
              mbf(ishell)= 10
              locbf(ishell+1) = locbf(ishell)+10
            endif
          enddo
!  G function
          ishell= ishell+1
          ex(locprim(ishell)+1)= expg(ige(numatomic(iatom))+1)
          coeff(locprim(ishell)+1)= one
          mprim(ishell)= 1
          mtype(ishell)= 4
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 9
            locbf(ishell+1) = locbf(ishell)+9
          else
            mbf(ishell)= 15
            locbf(ishell+1) = locbf(ishell)+15
          endif
!
! Set Na - Ar functions
!
        case(11:18)
!  S function
          do k= 0,2
            ishell= ishell+1
            nprim= 13
            if(numatomic(iatom) == 11) nprim= 16
            do j= 1,nprim
              ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
              coeff(locprim(ishell)+j)= coeffs(isc(numatomic(iatom))+j+k*nprim)
            enddo
            mprim(ishell)= nprim
            mbf(ishell)= 1
            mtype(ishell)= 0
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+nprim
            locbf(ishell+1) = locbf(ishell)+1
          enddo
!
          do k= 0,2
            ishell= ishell+1
            ex(locprim(ishell)+1)= exps(ise(numatomic(iatom))+1+nprim+k)
            coeff(locprim(ishell)+1)= coeffs(isc(numatomic(iatom))+1+nprim*3+k)
            mprim(ishell)= 1
            mbf(ishell)= 1
            mtype(ishell)= 0
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+1
            locbf(ishell+1) = locbf(ishell)+1
          enddo
!  P function
          do k= 0,1
            ishell= ishell+1
            nprim= 8
            if(numatomic(iatom) < 13) nprim= 9
            do j= 1,nprim
              ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j)
              coeff(locprim(ishell)+j)= coeffp(ipc(numatomic(iatom))+j+k*nprim)
            enddo
            mprim(ishell)= nprim
            mbf(ishell)= 3
            mtype(ishell)= 1
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+nprim
            locbf(ishell+1) = locbf(ishell)+3
          enddo
!
          do k= 0,2
            ishell= ishell+1
            ex(locprim(ishell)+1)= expp(ipe(numatomic(iatom))+1+nprim+k)
            coeff(locprim(ishell)+1)= coeffp(ipc(numatomic(iatom))+1+nprim*2+k)
            mprim(ishell)= 1
            mbf(ishell)= 3
            mtype(ishell)= 1
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+1
            locbf(ishell+1) = locbf(ishell)+3
          enddo
!  D function
          do k= 0,2
            ishell= ishell+1
            ex(locprim(ishell)+1)= expd(ide(numatomic(iatom))+1+k)
            coeff(locprim(ishell)+1)= one
            mprim(ishell)= 1
            mtype(ishell)= 2
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+1
            if(spher) then
              mbf(ishell)= 5
              locbf(ishell+1) = locbf(ishell)+5
            else
              mbf(ishell)= 6
              locbf(ishell+1) = locbf(ishell)+6
            endif
          enddo
!  F function
          do k= 0,1
            ishell= ishell+1
            ex(locprim(ishell)+1)= expf(ife(numatomic(iatom))+1+k)
            coeff(locprim(ishell)+1)= one
            mprim(ishell)= 1
            mtype(ishell)= 3
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+1
            if(spher) then
              mbf(ishell)= 7
              locbf(ishell+1) = locbf(ishell)+7
            else
              mbf(ishell)= 10
              locbf(ishell+1) = locbf(ishell)+10
            endif
          enddo
!  G function
          ishell= ishell+1
          ex(locprim(ishell)+1)= expg(ige(numatomic(iatom))+1)
          coeff(locprim(ishell)+1)= one
          mprim(ishell)= 1
          mtype(ishell)= 4
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 9
            locbf(ishell+1) = locbf(ishell)+9
          else
            mbf(ishell)= 15
            locbf(ishell+1) = locbf(ishell)+15
          endif
      endselect
!
      if(ishell > mxshell) then
        write(*,'(" Error! The number of basis shells exceeds mxshell",i6,".")')mxshell
        call iabort
      endif
      if(locprim(ishell+1) > mxprim ) then
        write(*,'(" Error! The number of primitive basis functions exceeds mxprim", &
&             i6,".")')mxprim
        call iabort
      endif
      if(locbf(ishell+1) > mxao ) then
        write(*,'(" Error! The number of basis functions exceeds mxao",i6,".")')mxao
        call iabort
      endif
      return
end


!----------------------------------
  subroutine bsd95v(iatom,ishell)
!----------------------------------
!
! Set basis functions of D95V
!
      use modmolecule, only : numatomic
      use modbasis, only : ex, coeff, locprim, locbf, locatom, mprim, mbf, mtype
      use modparam, only : mxao, mxshell, mxprim
      implicit none
      integer,intent(in) :: iatom
      integer,intent(inout) :: ishell
      integer :: j, nump
      integer :: ise(10)=(/0,0,4,14,24,34,44,54,64,74/)
      integer :: ipe(10)=(/0,0,0,4,9,14,19,24,29,34/)
      real(8) :: exps(84)= &
      (/ 0.1923840D+02, 0.2898700D+01, 0.6535000D+00, 0.1776000D+00, 0.9213000D+03, &
&        0.1387000D+03, 0.3194000D+02, 0.9353000D+01, 0.3158000D+01, 0.1157000D+01, &
&        0.4446000D+00, 0.4446000D+00, 0.7666000D-01, 0.2864000D-01, 0.1741000D+04, &
&        0.2621000D+03, 0.6033000D+02, 0.1762000D+02, 0.5933000D+01, 0.2185000D+01, &
&        0.8590000D+00, 0.2185000D+01, 0.1806000D+00, 0.5835000D-01, 0.2788000D+04, &
&        0.4190000D+03, 0.9647000D+02, 0.2807000D+02, 0.9376000D+01, 0.3406000D+01, &
&        0.1306000D+01, 0.3406000D+01, 0.3245000D+00, 0.1022000D+00, 0.4233000D+04, &
&        0.6349000D+03, 0.1461000D+03, 0.4250000D+02, 0.1419000D+02, 0.5148000D+01, &
&        0.1967000D+01, 0.5148000D+01, 0.4962000D+00, 0.1533000D+00, 0.5909000D+04, &
&        0.8875000D+03, 0.2047000D+03, 0.5984000D+02, 0.2000000D+02, 0.7193000D+01, &
&        0.2686000D+01, 0.7193000D+01, 0.7000000D+00, 0.2133000D+00, 0.7817000D+04, &
&        0.1176000D+04, 0.2732000D+03, 0.8117000D+02, 0.2718000D+02, 0.9532000D+01, &
&        0.3414000D+01, 0.9532000D+01, 0.9398000D+00, 0.2846000D+00, 0.9995000D+04, &
&        0.1506000D+04, 0.3503000D+03, 0.1041000D+03, 0.3484000D+02, 0.1222000D+02, &
&        0.4369000D+01, 0.1222000D+02, 0.1208000D+01, 0.3634000D+00, 0.1210000D+05, &
&        0.1821000D+04, 0.4328000D+03, 0.1325000D+03, 0.4377000D+02, 0.1491000D+02, &
&        0.5127000D+01, 0.1491000D+02, 0.1491000D+01, 0.4468000D+00/)
      real(8) :: coeffs(84)= &
      (/ 0.3282800D-01, 0.2312040D+00, 0.8172260D+00, 0.1000000D+01, 0.1367000D-02, &
&        0.1042500D-01, 0.4985900D-01, 0.1607010D+00, 0.3446040D+00, 0.4251970D+00, &
&        0.1694680D+00, -.2223110D+00, 0.1116477D+01, 0.1000000D+01, 0.1305000D-02, &
&        0.9955000D-02, 0.4803100D-01, 0.1585770D+00, 0.3513250D+00, 0.4270060D+00, &
&        0.1604900D+00, -.1852940D+00, 0.1057014D+01, 0.1000000D+01, 0.1288000D-02, &
&        0.9835000D-02, 0.4764800D-01, 0.1600690D+00, 0.3628940D+00, 0.4335820D+00, &
&        0.1400820D+00, -.1793300D+00, 0.1062594D+01, 0.1000000D+01, 0.1220000D-02, &
&        0.9342000D-02, 0.4545200D-01, 0.1546570D+00, 0.3588660D+00, 0.4386320D+00, &
&        0.1459180D+00, -.1683670D+00, 0.1060091D+01, 0.1000000D+01, 0.1190000D-02, &
&        0.9099000D-02, 0.4414500D-01, 0.1504640D+00, 0.3567410D+00, 0.4465330D+00, &
&        0.1456030D+00, -.1604050D+00, 0.1058215D+01, 0.1000000D+01, 0.1176000D-02, &
&        0.8968000D-02, 0.4286800D-01, 0.1439300D+00, 0.3556300D+00, 0.4612480D+00, &
&        0.1402060D+00, -.1541530D+00, 0.1056914D+01, 0.1000000D+01, 0.1160000D-02, &
&        0.8870000D-02, 0.4238000D-01, 0.1429290D+00, 0.3553720D+00, 0.4620850D+00, &
&        0.1408480D+00, -.1484520D+00, 0.1055270D+01, 0.1000000D+01, 0.1200000D-02, &
&        0.9092000D-02, 0.4130500D-01, 0.1378670D+00, 0.3624330D+00, 0.4722470D+00, &
&        0.1300350D+00, -.1408100D+00, 0.1053327D+01, 0.1000000D+01/)
      real(8) :: expp(39)= &
      (/ 0.1488000D+01, 0.2667000D+00, 0.7201000D-01, 0.2370000D-01, 0.6710000D+01, &
&        0.1442000D+01, 0.4103000D+00, 0.1397000D+00, 0.4922000D-01, 0.1134000D+02, &
&        0.2436000D+01, 0.6836000D+00, 0.2134000D+00, 0.7011000D-01, 0.1816000D+02, &
&        0.3986000D+01, 0.1143000D+01, 0.3594000D+00, 0.1146000D+00, 0.2679000D+02, &
&        0.5956000D+01, 0.1707000D+01, 0.5314000D+00, 0.1654000D+00, 0.3518000D+02, &
&        0.7904000D+01, 0.2305000D+01, 0.7171000D+00, 0.2137000D+00, 0.4436000D+02, &
&        0.1008000D+02, 0.2996000D+01, 0.9383000D+00, 0.2733000D+00, 0.5645000D+02, &
&        0.1292000D+02, 0.3865000D+01, 0.1203000D+01, 0.3444000D+00/)
      real(8) :: coeffp(39)= &
      (/ 0.3877000D-01, 0.2362570D+00, 0.8304480D+00, 0.1000000D+01, 0.1637800D-01, &
&        0.9155300D-01, 0.3414690D+00, 0.6854280D+00, 0.1000000D+01, 0.1798800D-01, &
&        0.1103430D+00, 0.3830720D+00, 0.6478950D+00, 0.1000000D+01, 0.1853900D-01, &
&        0.1154360D+00, 0.3861880D+00, 0.6401140D+00, 0.1000000D+01, 0.1825400D-01, &
&        0.1164610D+00, 0.3901780D+00, 0.6371020D+00, 0.1000000D+01, 0.1958000D-01, &
&        0.1242000D+00, 0.3947140D+00, 0.6273760D+00, 0.1000000D+01, 0.2087600D-01, &
&        0.1301070D+00, 0.3961660D+00, 0.6204040D+00, 0.1000000D+01, 0.2087500D-01, &
&        0.1300320D+00, 0.3956790D+00, 0.6214500D+00, 0.1000000D+01/)
!
      if((numatomic(iatom) >= 11).or.(numatomic(iatom) == 2)) then
        write(*,'(" Error! This program supports H - Ne D95V basis set.")')
        call iabort
      endif
!
      select case(numatomic(iatom))
!
! Set H - He functions
!
        case(1)
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(ise(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffs(ise(numatomic(iatom))+j+3)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
! Set Li - Ne functions
!
        case(3:10)
          ishell= ishell+1
          do j= 1,7
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs(ise(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 7
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+7
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j+7)
            coeff(locprim(ishell)+j)= coeffs(ise(numatomic(iatom))+j+7)
          enddo
          mprim(ishell)= 2
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps(ise(numatomic(iatom))+j+9)
            coeff(locprim(ishell)+j)= coeffs(ise(numatomic(iatom))+j+9)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
!
          nump= 4
          if(numatomic(iatom) == 4) nump= 3
          ishell= ishell+1
          do j= 1,nump
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp(ipe(numatomic(iatom))+j)
          enddo
          mprim(ishell)= nump
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+nump
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp(ipe(numatomic(iatom))+j+nump)
            coeff(locprim(ishell)+j)= coeffp(ipe(numatomic(iatom))+j+nump)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
      endselect
!
      if(ishell > mxshell) then
        write(*,'(" Error! The number of basis shells exceeds mxshell",i6,".")')mxshell
        call iabort
      endif
      if(locprim(ishell+1) > mxprim ) then
        write(*,'(" Error! The number of primitive basis functions exceeds mxprim", &
&             i6,".")')mxprim
        call iabort
      endif
      if(locbf(ishell+1) > mxao ) then
        write(*,'(" Error! The number of basis functions exceeds mxao",i6,".")')mxao
        call iabort
      endif
      return
end


!-------------------------------------
  subroutine bslanl2dz(iatom,ishell)
!-------------------------------------
!
! Set basis functions of LanL2DZ
!
      use modmolecule, only : numatomic
      use modbasis, only : ex, coeff, locprim, locbf, locatom, mprim, mbf, mtype, spher
      use modparam, only : mxao, mxshell, mxprim
      implicit none
      integer,intent(in) :: iatom
      integer,intent(inout) :: ishell
      integer :: j, nump
      integer :: ise3(11:18)=(/0,3,6,9,12,15,18,21/)
      integer :: ise4(19:36)=(/0,8,16,24,32,40,48,56,64,72,80,88,91,94,97,100,103,106/)
      integer :: ipe4(19:36)=(/0,5,10,15,20,25,30,35,40,45,50,55,57,60,63, 66, 69, 72/)
      integer :: ide4(21:30)=(/0,5,10,15,20,25,30,35,40,45/)
      integer :: ise5(37:54)=(/0,8,16,24,32,40,48,56,64,72,80,88,91,94,97,100,103,106/)
      integer :: ipe5(37:54)=(/0,6,12,18,24,30,36,42,48,54,60,66,69,72,75, 78, 81, 84/)
      integer :: ide5(39:48)=(/0,4, 8,12,16,20,24,28,32,36/)
      integer :: ise6(55:83), ipe6(55:83), ide6(55:83)
      data ise6(55:57) /0,8,16/
      data ipe6(55:57) /0,6,12/
      data ide6(55:57) /0,0,0/
      data ise6(72:83) /24,32,40,48,56,64,72,80,88,91,94,97/
      data ipe6(72:83) /18,24,30,36,42,48,54,60,66,69,72,75/
      data ide6(72:81) / 3, 6, 9,12,15,18,21,24,27,30/
      real(8) :: exps3(24)= &
&     (/ 0.49720000D+00, 0.56000000D-01, 0.22100000D-01, 0.72500000D+00, 0.11120000D+00, &
&        0.40400000D-01, 0.96150000D+00, 0.18190000D+00, 0.65700000D-01, 0.12220000D+01, &
&        0.25950000D+00, 0.93100000D-01, 0.15160000D+01, 0.33690000D+00, 0.12110000D+00, &
&        0.18500000D+01, 0.40350000D+00, 0.14380000D+00, 0.22310000D+01, 0.47200000D+00, &
&        0.16310000D+00, 0.26130000D+01, 0.57360000D+00, 0.20140000D+00/)
      real(8) :: coeffs3(24)= &
&     (/ -.27535740D+00, 0.10989969D+01, 0.10000000D+01, -.40584540D+00, 0.11688704D+01, &
&        0.10000000D+01, -.50215460D+00, 0.12342547D+01, 0.10000000D+01, -.57073390D+00, &
&        0.12823826D+01, 0.10000000D+01, -.58620890D+00, 0.12994376D+01, 0.10000000D+01, &
&        -.53243350D+00, 0.12763801D+01, 0.10000000D+01, -.49005890D+00, 0.12542684D+01, &
&        0.10000000D+01, -.51104630D+00, 0.12701236D+01, 0.10000000D+01/)
      real(8) :: expp3(24)= &
&     (/ 0.66970000D+00, 0.63600000D-01, 0.20400000D-01, 0.12400000D+01, 0.13460000D+00, &
&        0.42200000D-01, 0.19280000D+01, 0.20130000D+00, 0.58000000D-01, 0.25800000D+01, &
&        0.29840000D+00, 0.88500000D-01, 0.37050000D+01, 0.39340000D+00, 0.11900000D+00, &
&        0.49450000D+01, 0.48700000D+00, 0.13790000D+00, 0.62960000D+01, 0.63330000D+00, &
&        0.18190000D+00, 0.78600000D+01, 0.73870000D+00, 0.20810000D+00/)
      real(8) :: coeffp3(24)= &
&     (/ -.68384500D-01, 0.10140550D+01, 0.10000000D+01, -.74975300D-01, 0.10178183D+01, &
&        0.10000000D+01, -.71258400D-01, 0.10162966D+01, 0.10000000D+01, -.77725000D-01, &
&        0.10197870D+01, 0.10000000D+01, -.69147200D-01, 0.10161988D+01, 0.10000000D+01, &
&        -.60811600D-01, 0.10132686D+01, 0.10000000D+01, -.63564100D-01, 0.10141355D+01, &
&        0.10000000D+01, -.55516700D-01, 0.10115982D+01, 0.10000000D+01/)
      real(8) :: exps4(109)= &
&     (/ 0.30720000D+01, 0.67520000D+00, 0.25450000D+00, 0.30720000D+01, 0.67520000D+00, &
&        0.25450000D+00, 0.52900000D-01, 0.20900000D-01, 0.34840000D+01, 0.85510000D+00, &
&        0.31920000D+00, 0.34840000D+01, 0.85510000D+00, 0.31920000D+00, 0.14470000D+00, &
&        0.35000000D-01, 0.37170000D+01, 0.10970000D+01, 0.41640000D+00, 0.37170000D+01, &
&        0.10970000D+01, 0.41640000D+00, 0.76100000D-01, 0.28400000D-01, 0.43720000D+01, &
&        0.10980000D+01, 0.41780000D+00, 0.43720000D+01, 0.10980000D+01, 0.41780000D+00, &
&        0.87200000D-01, 0.31400000D-01, 0.45900000D+01, 0.14930000D+01, 0.55700000D+00, &
&        0.45900000D+01, 0.14930000D+01, 0.55700000D+00, 0.97500000D-01, 0.34200000D-01, &
&        0.53610000D+01, 0.14490000D+01, 0.54960000D+00, 0.53610000D+01, 0.14490000D+01, &
&        0.54960000D+00, 0.10520000D+00, 0.36400000D-01, 0.59140000D+01, 0.16050000D+01, &
&        0.62600000D+00, 0.59140000D+01, 0.16050000D+01, 0.62600000D+00, 0.11150000D+00, &
&        0.38000000D-01, 0.64220000D+01, 0.18260000D+01, 0.71350000D+00, 0.64220000D+01, &
&        0.18260000D+01, 0.71350000D+00, 0.10210000D+00, 0.36300000D-01, 0.71760000D+01, &
&        0.20090000D+01, 0.80550000D+00, 0.71760000D+01, 0.20090000D+01, 0.80550000D+00, &
&        0.10700000D+00, 0.37500000D-01, 0.76200000D+01, 0.22940000D+01, 0.87600000D+00, &
&        0.76200000D+01, 0.22940000D+01, 0.87600000D+00, 0.11530000D+00, 0.39600000D-01, &
&        0.81760000D+01, 0.25680000D+01, 0.95870000D+00, 0.81760000D+01, 0.25680000D+01, &
&        0.95870000D+00, 0.11530000D+00, 0.39600000D-01, 0.79970000D+00, 0.17520000D+00, &
&        0.55600000D-01, 0.83060000D+00, 0.33920000D+00, 0.91800000D-01, 0.89350000D+00, &
&        0.44240000D+00, 0.11620000D+00, 0.96350000D+00, 0.54270000D+00, 0.14070000D+00, &
&        0.10330000D+01, 0.65210000D+00, 0.16600000D+00, 0.11590000D+01, 0.71070000D+00, &
&        0.19050000D+00, 0.12270000D+01, 0.84570000D+00, 0.21670000D+00/)
      real(8) :: coeffs4(109)= &
&     (/ -.30830670D+00, 0.78207110D+00, 0.41428830D+00, 0.18940870D+00, -.54723320D+00, &
&        -.75764580D+00, 0.99521570D+00, 0.10000000D+01, -.33373700D+00, 0.75898310D+00, &
&        0.46061090D+00, 0.26060300D+00, -.68391500D+00, -.11987807D+01, 0.10251221D+01, &
&        0.10000000D+01, -.39262920D+00, 0.71473390D+00, 0.55094000D+00, 0.21974460D+00, &
&        -.45460670D+00, -.69381810D+00, 0.11260172D+01, 0.10000000D+01, -.36370980D+00, &
&        0.81845330D+00, 0.41845260D+00, 0.20490270D+00, -.55754130D+00, -.58936520D+00, &
&        0.11451661D+01, 0.10000000D+01, -.42773020D+00, 0.70696910D+00, 0.58958130D+00, &
&        0.25010810D+00, -.46977770D+00, -.71294150D+00, 0.11006569D+01, 0.10000000D+01, &
&        -.38056890D+00, 0.77956250D+00, 0.47307770D+00, 0.22031110D+00, -.54233910D+00, &
&        -.60477160D+00, 0.11136657D+01, 0.10000000D+01, -.37645080D+00, 0.77247890D+00, &
&        0.47693460D+00, 0.21199660D+00, -.51994720D+00, -.58576810D+00, 0.11003964D+01, &
&        0.10000000D+01, -.39278820D+00, 0.77126430D+00, 0.49202280D+00, 0.17868770D+00, &
&        -.41940320D+00, -.45681850D+00, 0.11035048D+01, 0.10000000D+01, -.38567340D+00, &
&        0.74531160D+00, 0.50918190D+00, 0.17361610D+00, -.39704420D+00, -.46306220D+00, &
&        0.10899654D+01, 0.10000000D+01, -.40825500D+00, 0.74553080D+00, 0.53257210D+00, &
&        0.18725910D+00, -.39669640D+00, -.49540030D+00, 0.10844343D+01, 0.10000000D+01, &
&        -.42102600D+00, 0.73859240D+00, 0.55256920D+00, 0.17876650D+00, -.35922730D+00, &
&        -.47048250D+00, 0.10807407D+01, 0.10000000D+01, -.64861120D+00, 0.13138291D+01, &
&        0.10000000D+01, -.16759436D+01, 0.19877108D+01, 0.10000000D+01, -.21756591D+01, &
&        0.24493467D+01, 0.10000000D+01, -.26709549D+01, 0.29387892D+01, 0.10000000D+01, &
&        -.33224095D+01, 0.36003462D+01, 0.10000000D+01, -.30378769D+01, 0.33703735D+01, &
&        0.10000000D+01, -.40317198D+01, 0.43554125D+01, 0.10000000D+01/)
      real(8) :: expp4(75)= &
&     (/ 0.82330000D+01, 0.95260000D+00, 0.30130000D+00, 0.37600000D-01, 0.14000000D-01, &
&        0.94200000D+01, 0.11500000D+01, 0.36880000D+00, 0.70500000D-01, 0.26300000D-01, &
&        0.10400000D+02, 0.13110000D+01, 0.42660000D+00, 0.47000000D-01, 0.14000000D-01, &
&        0.12520000D+02, 0.14910000D+01, 0.48590000D+00, 0.53000000D-01, 0.16000000D-01, &
&        0.13760000D+02, 0.17120000D+01, 0.55870000D+00, 0.59000000D-01, 0.18000000D-01, &
&        0.16420000D+02, 0.19140000D+01, 0.62410000D+00, 0.63000000D-01, 0.19000000D-01, &
&        0.18200000D+02, 0.21410000D+01, 0.70090000D+00, 0.69000000D-01, 0.21000000D-01, &
&        0.19480000D+02, 0.23890000D+01, 0.77950000D+00, 0.74000000D-01, 0.22000000D-01, &
&        0.21390000D+02, 0.26500000D+01, 0.86190000D+00, 0.80000000D-01, 0.23000000D-01, &
&        0.23660000D+02, 0.28930000D+01, 0.94350000D+00, 0.84000000D-01, 0.24000000D-01, &
&        0.25630000D+02, 0.31660000D+01, 0.10230000D+01, 0.86000000D-01, 0.24000000D-01, &
&        0.12020000D+00, 0.35100000D-01, 0.16750000D+01, 0.20300000D+00, 0.57900000D-01, &
&        0.18770000D+01, 0.26230000D+00, 0.79800000D-01, 0.20840000D+01, 0.32240000D+00, &
&        0.10200000D+00, 0.23660000D+01, 0.38330000D+00, 0.11860000D+00, 0.26910000D+01, &
&        0.44460000D+00, 0.13770000D+00, 0.29200000D+01, 0.51690000D+00, 0.16140000D+00/)
      real(8) :: coeffp4(75)= &
&     (/ -.41991900D-01, 0.57768190D+00, 0.52346110D+00, 0.10000000D+01, 0.10000000D+01, &
&        -.43694300D-01, 0.60347470D+00, 0.49611070D+00, 0.10000000D+01, 0.10000000D+01, &
&        -.49261500D-01, 0.60719650D+00, 0.49175820D+00, 0.10000000D+01, 0.10000000D+01, &
&        -.45690800D-01, 0.62033130D+00, 0.47653290D+00, 0.10000000D+01, 0.10000000D+01, &
&        -.48231200D-01, 0.61411610D+00, 0.48383420D+00, 0.10000000D+01, 0.10000000D+01, &
&        -.46139700D-01, 0.61099650D+00, 0.48596350D+00, 0.10000000D+01, 0.10000000D+01, &
&        -.44790100D-01, 0.62603110D+00, 0.46963290D+00, 0.10000000D+01, 0.10000000D+01, &
&        -.47028200D-01, 0.62488410D+00, 0.47225420D+00, 0.10000000D+01, 0.10000000D+01, &
&        -.48041300D-01, 0.62223370D+00, 0.47580420D+00, 0.10000000D+01, 0.10000000D+01, &
&        -.48155800D-01, 0.62584730D+00, 0.47151580D+00, 0.10000000D+01, 0.10000000D+01, &
&        -.48917300D-01, 0.62728540D+00, 0.47161880D+00, 0.10000000D+01, 0.10000000D+01, &
&        0.10000000D+01, 0.10000000D+01, -.85610700D-01, 0.10226850D+01, 0.10000000D+01, &
&        -.10067790D+00, 0.10306256D+01, 0.10000000D+01, -.11371000D+00, 0.10380266D+01, &
&        0.10000000D+01, -.11855220D+00, 0.10414320D+01, 0.10000000D+01, -.11898000D+00, &
&        0.10424471D+01, 0.10000000D+01, -.13306850D+00, 0.10502807D+01, 0.10000000D+01/)
      real(8) :: expd4(50)= &
&     (/ 0.15130000D+02, 0.42050000D+01, 0.13030000D+01, 0.36800000D+00, 0.81200000D-01, &
&        0.20210000D+02, 0.54950000D+01, 0.16990000D+01, 0.48400000D+00, 0.11570000D+00, &
&        0.25700000D+02, 0.65300000D+01, 0.20780000D+01, 0.62430000D+00, 0.15420000D+00, &
&        0.28950000D+02, 0.77080000D+01, 0.24950000D+01, 0.76550000D+00, 0.18890000D+00, &
&        0.32270000D+02, 0.88750000D+01, 0.28900000D+01, 0.87610000D+00, 0.21200000D+00, &
&        0.37080000D+02, 0.10100000D+02, 0.32200000D+01, 0.96280000D+00, 0.22620000D+00, &
&        0.39250000D+02, 0.10780000D+02, 0.34960000D+01, 0.10660000D+01, 0.26060000D+00, &
&        0.42720000D+02, 0.11760000D+02, 0.38170000D+01, 0.11690000D+01, 0.28360000D+00, &
&        0.41340000D+02, 0.11420000D+02, 0.38390000D+01, 0.12300000D+01, 0.31020000D+00, &
&        0.68850000D+02, 0.18320000D+02, 0.59220000D+01, 0.19270000D+01, 0.55280000D+00/)
      real(8) :: coeffd4(50)= &
&     (/ 0.37929100D-01, 0.17383600D+00, 0.42680570D+00, 0.62385390D+00, 0.10000000D+01, &
&        0.34168200D-01, 0.17100060D+00, 0.44058490D+00, 0.61142460D+00, 0.10000000D+01, &
&        0.33103300D-01, 0.17957530D+00, 0.43730620D+00, 0.59848600D+00, 0.10000000D+01, &
&        0.33778700D-01, 0.17803450D+00, 0.43700080D+00, 0.59417950D+00, 0.10000000D+01, &
&        0.34158000D-01, 0.17611050D+00, 0.43942980D+00, 0.59432710D+00, 0.10000000D+01, &
&        0.32900000D-01, 0.17874180D+00, 0.44876570D+00, 0.58763610D+00, 0.10000000D+01, &
&        0.36154100D-01, 0.18967440D+00, 0.45249810D+00, 0.57104270D+00, 0.10000000D+01, &
&        0.37269900D-01, 0.19561030D+00, 0.45612730D+00, 0.56215870D+00, 0.10000000D+01, &
&        0.46542400D-01, 0.22278240D+00, 0.45390590D+00, 0.53147690D+00, 0.10000000D+01, &
&        0.25853200D-01, 0.16511950D+00, 0.44682120D+00, 0.58310800D+00, 0.10000000D+01/)
      real(8) :: exps5(109)= &
&     (/ 0.14570000D+01, 0.94000000D+00, 0.27060000D+00, 0.14570000D+01, 0.94000000D+00, &
&        0.27060000D+00, 0.36600000D-01, 0.15500000D-01, 0.16480000D+01, 0.10030000D+01, &
&        0.31060000D+00, 0.16480000D+01, 0.10030000D+01, 0.31060000D+00, 0.10990000D+00, &
&        0.29200000D-01, 0.17510000D+01, 0.11430000D+01, 0.35810000D+00, 0.17510000D+01, &
&        0.11430000D+01, 0.35810000D+00, 0.10580000D+00, 0.31800000D-01, 0.19760000D+01, &
&        0.11540000D+01, 0.39100000D+00, 0.19760000D+01, 0.11540000D+01, 0.39100000D+00, &
&        0.10010000D+00, 0.33400000D-01, 0.21820000D+01, 0.12090000D+01, 0.41650000D+00, &
&        0.21820000D+01, 0.12090000D+01, 0.41650000D+00, 0.14540000D+00, 0.39200000D-01, &
&        0.23610000D+01, 0.13090000D+01, 0.45000000D+00, 0.23610000D+01, 0.13090000D+01, &
&        0.45000000D+00, 0.16810000D+00, 0.42300000D-01, 0.23420000D+01, 0.16340000D+01, &
&        0.50940000D+00, 0.23420000D+01, 0.16340000D+01, 0.50940000D+00, 0.17060000D+00, &
&        0.43500000D-01, 0.25650000D+01, 0.15080000D+01, 0.51290000D+00, 0.25650000D+01, &
&        0.15080000D+01, 0.51290000D+00, 0.13620000D+00, 0.41700000D-01, 0.26460000D+01, &
&        0.17510000D+01, 0.57130000D+00, 0.26460000D+01, 0.17510000D+01, 0.57130000D+00, &
&        0.14380000D+00, 0.42800000D-01, 0.27870000D+01, 0.19650000D+01, 0.62430000D+00, &
&        0.27870000D+01, 0.19650000D+01, 0.62430000D+00, 0.14960000D+00, 0.43600000D-01, &
&        0.29500000D+01, 0.21490000D+01, 0.66840000D+00, 0.29500000D+01, 0.21490000D+01, &
&        0.66840000D+00, 0.99700000D-01, 0.34700000D-01, 0.50950000D+00, 0.19240000D+00, &
&        0.54400000D-01, 0.49150000D+00, 0.34040000D+00, 0.77400000D-01, 0.54180000D+00, &
&        0.37840000D+00, 0.92600000D-01, 0.58630000D+00, 0.42930000D+00, 0.10780000D+00, &
&        0.69380000D+00, 0.40380000D+00, 0.11650000D+00, 0.72420000D+00, 0.46530000D+00, &
&        0.13360000D+00, 0.76460000D+00, 0.53220000D+00, 0.14910000D+00/)
      real(8) :: coeffs5(109)= &
&     (/ -.10617406D+01, 0.11722125D+01, 0.74370080D+00, 0.50521740D+00, -.57184040D+00, &
&        -.68353080D+00, 0.11136477D+01, 0.10000000D+01, -.95198790D+00, 0.10869857D+01, &
&        0.72031600D+00, 0.79143370D+00, -.94340460D+00, -.13043063D+01, 0.90468690D+00, &
&        0.10000000D+01, -.11617905D+01, 0.12958877D+01, 0.71151250D+00, 0.96831230D+00, &
&        -.11346367D+01, -.11914054D+01, 0.92810630D+00, 0.10000000D+01, -.92067450D+00, &
&        0.10929033D+01, 0.67956210D+00, 0.75060600D+00, -.98108260D+00, -.10225151D+01, &
&        0.10860038D+01, 0.10000000D+01, -.88461440D+00, 0.11033775D+01, 0.62987760D+00, &
&        0.77902870D+00, -.10752788D+01, -.11506014D+01, 0.99691550D+00, 0.10000000D+01, &
&        -.91217600D+00, 0.11477453D+01, 0.60971090D+00, 0.81392590D+00, -.11360084D+01, &
&        -.11611592D+01, 0.10064786D+01, 0.10000000D+01, -.14911782D+01, 0.16749043D+01, &
&        0.65730060D+00, 0.13523997D+01, -.16216301D+01, -.11463770D+01, 0.98591900D+00, &
&        0.10000000D+01, -.10431056D+01, 0.13314786D+01, 0.56130650D+00, 0.87701280D+00, &
&        -.12634660D+01, -.83849870D+00, 0.10637773D+01, 0.10000000D+01, -.13554084D+01, &
&        0.16112233D+01, 0.58938140D+00, 0.11472137D+01, -.14943525D+01, -.85897040D+00, &
&        0.10297241D+01, 0.10000000D+01, -.16102393D+01, 0.18489842D+01, 0.60374920D+00, &
&        0.13540775D+01, -.16780848D+01, -.85593810D+00, 0.10200299D+01, 0.10000000D+01, &
&        -.17910564D+01, 0.20244570D+01, 0.60728390D+00, 0.10141125D+01, -.12413971D+01, &
&        -.49014270D+00, 0.11128375D+01, 0.10000000D+01, -.12713002D+01, 0.18002112D+01, &
&        0.10000000D+01, -.42418681D+01, 0.44784826D+01, 0.10000000D+01, -.42089644D+01, &
&        0.45198368D+01, 0.10000000D+01, -.47537662D+01, 0.51096593D+01, 0.10000000D+01, &
&        -.24115013D+01, 0.29179976D+01, 0.10000000D+01, -.29731048D+01, 0.34827643D+01, &
&        0.10000000D+01, -.36543172D+01, 0.41674919D+01, 0.10000000D+01/)
      real(8) :: expp5(87)= &
&     (/ 0.33000000D+01, 0.59850000D+00, 0.20670000D+00, 0.19470000D+00, 0.31800000D-01, &
&        0.12400000D-01, 0.35520000D+01, 0.69750000D+00, 0.24800000D+00, 0.27350000D+00, &
&        0.57000000D-01, 0.22200000D-01, 0.38840000D+01, 0.76600000D+00, 0.28900000D+00, &
&        0.28960000D+00, 0.62900000D-01, 0.22300000D-01, 0.41920000D+01, 0.87640000D+00, &
&        0.32630000D+00, 0.29720000D+00, 0.72400000D-01, 0.24300000D-01, 0.45190000D+01, &
&        0.94060000D+00, 0.34920000D+00, 0.41060000D+00, 0.75200000D-01, 0.24700000D-01, &
&        0.48950000D+01, 0.10440000D+01, 0.38770000D+00, 0.49950000D+00, 0.78000000D-01, &
&        0.24700000D-01, 0.52780000D+01, 0.11560000D+01, 0.43020000D+00, 0.47670000D+00, &
&        0.89500000D-01, 0.24600000D-01, 0.48590000D+01, 0.12190000D+01, 0.44130000D+00, &
&        0.57250000D+00, 0.83000000D-01, 0.25000000D-01, 0.54400000D+01, 0.13290000D+01, &
&        0.48450000D+00, 0.65950000D+00, 0.86900000D-01, 0.25700000D-01, 0.59990000D+01, &
&        0.14430000D+01, 0.52640000D+00, 0.73680000D+00, 0.89900000D-01, 0.26200000D-01, &
&        0.65530000D+01, 0.15650000D+01, 0.57480000D+00, 0.90850000D+00, 0.83300000D-01, &
&        0.25200000D-01, 0.82700000D+00, 0.12870000D+00, 0.40500000D-01, 0.97550000D+00, &
&        0.15500000D+00, 0.47400000D-01, 0.10470000D+01, 0.19320000D+00, 0.63000000D-01, &
&        0.11110000D+01, 0.23650000D+00, 0.80000000D-01, 0.12310000D+01, 0.27560000D+00, &
&        0.91100000D-01, 0.12900000D+01, 0.31800000D+00, 0.10530000D+00, 0.12110000D+01, &
&        0.38080000D+00, 0.12590000D+00/)
      real(8) :: coeffp5(87)= &
&     (/ -.72941700D-01, 0.63217820D+00, 0.47074260D+00, -.12505520D+00, 0.10438206D+01, &
&        0.10000000D+01, -.88646100D-01, 0.66618570D+00, 0.43907050D+00, -.17874700D+00, &
&        0.10766134D+01, 0.10000000D+01, -.82060100D-01, 0.67564130D+00, 0.41954820D+00, &
&        -.15010800D+00, 0.10686926D+01, 0.10000000D+01, -.94255400D-01, 0.67836750D+00, &
&        0.42466180D+00, -.15274010D+00, 0.10777185D+01, 0.10000000D+01, -.81730300D-01, &
&        0.69951150D+00, 0.39809960D+00, -.12121760D+00, 0.10480477D+01, 0.10000000D+01, &
&        -.90825800D-01, 0.70428990D+00, 0.39731790D+00, -.10819450D+00, 0.10368093D+01, &
&        0.10000000D+01, -.99541900D-01, 0.70815440D+00, 0.39735710D+00, -.97312700D-01, &
&        0.10404862D+01, 0.10000000D+01, -.94575500D-01, 0.74347980D+00, 0.36681440D+00, &
&        -.88086400D-01, 0.10283970D+01, 0.10000000D+01, -.98769900D-01, 0.74335950D+00, &
&        0.36684620D+00, -.83805600D-01, 0.10244841D+01, 0.10000000D+01, -.10349100D+00, &
&        0.74569520D+00, 0.36564940D+00, 0.76328500D-01, 0.97400650D+00, 0.10000000D+01, &
&        -.10791170D+00, 0.74036450D+00, 0.37210080D+00, -.41837100D-01, 0.10087586D+01, &
&        0.10000000D+01, -.10830200D+00, 0.10367049D+01, 0.10000000D+01, -.12264730D+00, &
&        0.10417571D+01, 0.10000000D+01, -.14176780D+00, 0.10554488D+01, 0.10000000D+01, &
&        -.18117070D+00, 0.10791649D+01, 0.10000000D+01, -.19233400D+00, 0.10876382D+01, &
&        0.10000000D+01, -.20923770D+00, 0.11035347D+01, 0.10000000D+01, -.26169240D+00, &
&        0.11570355D+01, 0.10000000D+01/)
      real(8) :: expd5(40)= &
&     (/ 0.15230000D+01, 0.56340000D+00, 0.18340000D+00, 0.56900000D-01, 0.22690000D+01, &
&        0.78550000D+00, 0.26150000D+00, 0.80200000D-01, 0.34660000D+01, 0.99380000D+00, &
&        0.33500000D+00, 0.10240000D+00, 0.29930000D+01, 0.10630000D+01, 0.37210000D+00, &
&        0.11780000D+00, 0.46320000D+01, 0.12790000D+01, 0.44250000D+00, 0.13640000D+00, &
&        0.41950000D+01, 0.13770000D+01, 0.48280000D+00, 0.15010000D+00, 0.36690000D+01, &
&        0.14230000D+01, 0.50910000D+00, 0.16100000D+00, 0.60910000D+01, 0.17190000D+01, &
&        0.60560000D+00, 0.18830000D+00, 0.33910000D+01, 0.15990000D+01, 0.62820000D+00, &
&        0.21080000D+00, 0.51480000D+01, 0.19660000D+01, 0.73600000D+00, 0.24790000D+00/)
      real(8) :: coeffd5(40)= &
&     (/ 0.10748430D+00, 0.45639540D+00, 0.60398550D+00, 0.10000000D+01, 0.59974100D-01, &
&        0.47341580D+00, 0.61171780D+00, 0.10000000D+01, 0.31598300D-01, 0.48343060D+00, &
&        0.61648930D+00, 0.10000000D+01, 0.52706300D-01, 0.50039070D+00, 0.57940240D+00, &
&        0.10000000D+01, 0.26872400D-01, 0.50730890D+00, 0.59113810D+00, 0.10000000D+01, &
&        0.48572900D-01, 0.51052230D+00, 0.57300280D+00, 0.10000000D+01, 0.76005900D-01, &
&        0.51588520D+00, 0.54365850D+00, 0.10000000D+01, 0.37614600D-01, 0.52004790D+00, &
&        0.57060710D+00, 0.10000000D+01, 0.13969380D+00, 0.47444210D+00, 0.51563110D+00, &
&        0.10000000D+01, 0.70307100D-01, 0.51384270D+00, 0.54167580D+00, 0.10000000D+01/)
      real(8) :: exps6(100)= &
&     (/ 0.87090000D+00, 0.53930000D+00, 0.17240000D+00, 0.87090000D+00, 0.53930000D+00, &
&        0.17240000D+00, 0.39300000D-01, 0.15100000D-01, 0.86990000D+00, 0.66760000D+00, &
&        0.19820000D+00, 0.86990000D+00, 0.66760000D+00, 0.19820000D+00, 0.82300000D-01, &
&        0.23100000D-01, 0.91670000D+00, 0.74270000D+00, 0.22370000D+00, 0.91670000D+00, &
&        0.74270000D+00, 0.22370000D+00, 0.79200000D-01, 0.23900000D-01, 0.19500000D+01, &
&        0.11830000D+01, 0.38970000D+00, 0.19500000D+01, 0.11830000D+01, 0.38970000D+00, &
&        0.16560000D+00, 0.42400000D-01, 0.20440000D+01, 0.12670000D+01, 0.41570000D+00, &
&        0.20440000D+01, 0.12670000D+01, 0.41570000D+00, 0.16710000D+00, 0.48200000D-01, &
&        0.21370000D+01, 0.13470000D+01, 0.43660000D+00, 0.21370000D+01, 0.13470000D+01, &
&        0.43660000D+00, 0.18830000D+00, 0.51800000D-01, 0.21850000D+01, 0.14510000D+01, &
&        0.45850000D+00, 0.21850000D+01, 0.14510000D+01, 0.45850000D+00, 0.23140000D+00, &
&        0.56600000D-01, 0.22220000D+01, 0.14960000D+01, 0.47740000D+00, 0.22220000D+01, &
&        0.14960000D+01, 0.47740000D+00, 0.24370000D+00, 0.58300000D-01, 0.23500000D+01, &
&        0.15820000D+01, 0.50180000D+00, 0.23500000D+01, 0.15820000D+01, 0.50180000D+00, &
&        0.25000000D+00, 0.59800000D-01, 0.25470000D+01, 0.16140000D+01, 0.51670000D+00, &
&        0.25470000D+01, 0.16140000D+01, 0.51670000D+00, 0.26510000D+00, 0.58000000D-01, &
&        0.28090000D+01, 0.15950000D+01, 0.53270000D+00, 0.28090000D+01, 0.15950000D+01, &
&        0.53270000D+00, 0.28260000D+00, 0.59800000D-01, 0.52750000D+00, 0.23340000D+00, &
&        0.68610000D-01, 0.53550000D+00, 0.30820000D+00, 0.81830000D-01, 0.51350000D+00, &
&        0.37560000D+00, 0.94400000D-01, 0.57440000D+00, 0.38510000D+00, 0.10500000D+00/)
      real(8) :: coeffs6(100)= &
&     (/ -.11604798D+01, 0.14198515D+01, 0.58729050D+00, 0.82377270D+00, -.10709181D+01, &
&        -.90851220D+00, 0.10774049D+01, 0.10000000D+01, -.22549747D+01, 0.25145786D+01, &
&        0.57751840D+00, 0.20391383D+01, -.23717712D+01, -.12758006D+01, 0.11703346D+01, &
&        0.10000000D+01, -.30240751D+01, 0.32971476D+01, 0.55135420D+00, 0.26910243D+01, &
&        -.30474363D+01, -.11030211D+01, 0.11941863D+01, 0.10000000D+01, -.12260850D+01, &
&        0.15695074D+01, 0.49334120D+00, 0.10778382D+01, -.15010862D+01, -.11982175D+01, &
&        0.12290296D+01, 0.10000000D+01, -.13104779D+01, 0.16579925D+01, 0.48483370D+00, &
&        0.11897315D+01, -.16650462D+01, -.10563335D+01, 0.11840652D+01, 0.10000000D+01, &
&        -.13916151D+01, 0.17510261D+01, 0.46946470D+00, 0.12985088D+01, -.18102429D+01, &
&        -.10844531D+01, 0.12580618D+01, 0.10000000D+01, -.16223730D+01, 0.19938647D+01, &
&        0.45311660D+00, 0.15459752D+01, -.20758927D+01, -.11922396D+01, 0.12272864D+01, &
&        0.10000000D+01, -.16538036D+01, 0.20670297D+01, 0.42320170D+00, 0.16046968D+01, &
&        -.22120386D+01, -.11301209D+01, 0.12654553D+01, 0.10000000D+01, -.16784642D+01, &
&        0.20952553D+01, 0.41629340D+00, 0.16464467D+01, -.22748150D+01, -.10494357D+01, &
&        0.12167791D+01, 0.10000000D+01, -.14739175D+01, 0.19115719D+01, 0.39223190D+00, &
&        0.14388166D+01, -.20911821D+01, -.10921315D+01, 0.13426596D+01, 0.10000000D+01, &
&        -.12021556D+01, 0.16741578D+01, 0.35265930D+00, 0.11608481D+01, -.18642846D+01, &
&        -.10356230D+01, 0.13064399D+01, 0.10000000D+01, -.49116760D+00, 0.60440700D+00, &
&        0.10000000D+01, -.80083500D+00, 0.92201300D+00, 0.10000000D+01, -.43675036D+01, &
&        0.48504656D+01, 0.10000000D+01, -.32278875D+01, 0.37637689D+01, 0.10000000D+01/)
      real(8) :: expp6(78)= &
&     (/ 0.14680000D+01, 0.41340000D+00, 0.15360000D+00, 0.14580000D+00, 0.27900000D-01, &
&        0.11300000D-01, 0.16050000D+01, 0.47900000D+00, 0.18180000D+00, 0.18040000D+00, &
&        0.47600000D-01, 0.19200000D-01, 0.15540000D+01, 0.56220000D+00, 0.22390000D+00, &
&        0.21250000D+00, 0.48300000D-01, 0.17900000D-01, 0.19720000D+01, 0.13540000D+01, &
&        0.41340000D+00, 0.34270000D+00, 0.80400000D-01, 0.27400000D-01, 0.25650000D+01, &
&        0.12290000D+01, 0.42440000D+00, 0.43600000D+00, 0.84000000D-01, 0.28000000D-01, &
&        0.30050000D+01, 0.12280000D+01, 0.44150000D+00, 0.40100000D+00, 0.90000000D-01, &
&        0.28000000D-01, 0.33580000D+01, 0.12710000D+01, 0.46440000D+00, 0.49600000D+00, &
&        0.89000000D-01, 0.28000000D-01, 0.25180000D+01, 0.14600000D+01, 0.49230000D+00, &
&        0.51000000D+00, 0.98000000D-01, 0.29000000D-01, 0.27920000D+01, 0.15410000D+01, &
&        0.52850000D+00, 0.51000000D+00, 0.98000000D-01, 0.29000000D-01, 0.29110000D+01, &
&        0.18360000D+01, 0.59820000D+00, 0.60480000D+00, 0.99600000D-01, 0.29000000D-01, &
&        0.36840000D+01, 0.16660000D+01, 0.59890000D+00, 0.68380000D+00, 0.97700000D-01, &
&        0.27900000D-01, 0.65030000D+00, 0.13680000D+00, 0.42560000D-01, 0.79770000D+00, &
&        0.14980000D+00, 0.44350000D-01, 0.87480000D+00, 0.18430000D+00, 0.59800000D-01, &
&        0.91050000D+00, 0.21940000D+00, 0.74500000D-01/)
      real(8) :: coeffp6(78)= &
&     (/ -.13210330D+00, 0.65724100D+00, 0.47564560D+00, -.16805560D+00, 0.10665746D+01, &
&        0.10000000D+01, -.16264030D+00, 0.69712890D+00, 0.45051070D+00, -.26425370D+00, &
&        0.11347212D+01, 0.10000000D+01, -.18171680D+00, 0.66322260D+00, 0.50297820D+00, &
&        -.19688100D+00, 0.10907542D+01, 0.10000000D+01, -.63234670D+00, 0.10465162D+01, &
&        0.58016700D+00, -.17391780D+00, 0.10841644D+01, 0.10000000D+01, -.29172380D+00, &
&        0.75706490D+00, 0.52400810D+00, -.15130140D+00, 0.10613123D+01, 0.10000000D+01, &
&        -.24055630D+00, 0.73640920D+00, 0.48814870D+00, -.14974990D+00, 0.10707463D+01, &
&        0.10000000D+01, -.23186670D+00, 0.74586830D+00, 0.46326860D+00, -.13113700D+00, &
&        0.10503671D+01, 0.10000000D+01, -.41773450D+00, 0.94349470D+00, 0.46729760D+00, &
&        -.14902790D+00, 0.10603616D+01, 0.10000000D+01, -.38892120D+00, 0.90775160D+00, &
&        0.46914430D+00, -.11706690D+00, 0.10489002D+01, 0.10000000D+01, -.52474380D+00, &
&        0.96718840D+00, 0.54386320D+00, -.10614380D+00, 0.10383102D+01, 0.10000000D+01, &
&        -.28026810D+00, 0.78183980D+00, 0.48047760D+00, -.95207800D-01, 0.10299147D+01, &
&        0.10000000D+01, -.67227100D-01, 0.49790230D+00, 0.10000000D+01, -.69067100D-01, &
&        0.51097100D+00, 0.10000000D+01, -.17931280D+00, 0.10776505D+01, 0.10000000D+01, &
&        -.21641890D+00, 0.11041867D+01, 0.10000000D+01/)
      real(8) :: expd6(33)= &
&     (/ 0.45240000D+00, 0.16020000D+00, 0.53100000D-01, 0.82260000D+00, 0.25850000D+00, &
&        0.76200000D-01, 0.89480000D+00, 0.29890000D+00, 0.93500000D-01, 0.95190000D+00, &
&        0.32700000D+00, 0.10540000D+00, 0.11160000D+01, 0.42670000D+00, 0.13780000D+00, &
&        0.11830000D+01, 0.44920000D+00, 0.14630000D+00, 0.12400000D+01, 0.46470000D+00, &
&        0.15290000D+00, 0.12430000D+01, 0.42710000D+00, 0.13700000D+00, 0.12870000D+01, &
&        0.43350000D+00, 0.13960000D+00, 0.14840000D+01, 0.56050000D+00, 0.19230000D+00, &
&        0.86550000D+01, 0.14150000D+01, 0.44420000D+00/)
      real(8) :: coeffd6(33)= &
&     (/ 0.44680510D+00, 0.65434930D+00, 0.10000000D+01, 0.44904050D+00, 0.67238910D+00, &
&        0.10000000D+01, 0.47310720D+00, 0.63992000D+00, 0.10000000D+01, 0.49852650D+00, &
&        0.61111100D+00, 0.10000000D+01, 0.46898880D+00, 0.62095910D+00, 0.10000000D+01, &
&        0.48384700D+00, 0.60795730D+00, 0.10000000D+01, 0.50870220D+00, 0.58621020D+00, &
&        0.10000000D+01, 0.55981500D+00, 0.55110900D+00, 0.10000000D+01, 0.58442730D+00, &
&        0.52981610D+00, 0.10000000D+01, 0.48999270D+00, 0.49327110D+00, 0.10000000D+01, &
&        0.15602900D-01, 0.62265100D+00, 0.10000000D+01/)
!
      select case(numatomic(iatom))
        case(:10)
!
! Set Na - Ar functions
!
        case(11:18)
! S function
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= exps3(ise3(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs3(ise3(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 2
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps3(ise3(numatomic(iatom))+j+2)
            coeff(locprim(ishell)+j)= coeffs3(ise3(numatomic(iatom))+j+2)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
! P function
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= expp3(ise3(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp3(ise3(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 2
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp3(ise3(numatomic(iatom))+j+2)
            coeff(locprim(ishell)+j)= coeffp3(ise3(numatomic(iatom))+j+2)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
! Set K  - Ca functions
!
        case(19:20)
! S function
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exps4(ise4(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs4(ise4(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,4
            ex(locprim(ishell)+j)= exps4(ise4(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffs4(ise4(numatomic(iatom))+j+3)
          enddo
          mprim(ishell)= 4
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+4
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps4(ise4(numatomic(iatom))+j+7)
            coeff(locprim(ishell)+j)= coeffs4(ise4(numatomic(iatom))+j+7)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
! P function
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= expp4(ipe4(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp4(ipe4(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp4(ipe4(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffp4(ipe4(numatomic(iatom))+j+3)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp4(ipe4(numatomic(iatom))+j+4)
            coeff(locprim(ishell)+j)= coeffp4(ipe4(numatomic(iatom))+j+4)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
! Set Sc - Cu functions
!
        case(21:29)
! S function
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exps4(ise4(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs4(ise4(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,4
            ex(locprim(ishell)+j)= exps4(ise4(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffs4(ise4(numatomic(iatom))+j+3)
          enddo
          mprim(ishell)= 4
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+4
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps4(ise4(numatomic(iatom))+j+7)
            coeff(locprim(ishell)+j)= coeffs4(ise4(numatomic(iatom))+j+7)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
! P function
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= expp4(ipe4(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp4(ipe4(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp4(ipe4(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffp4(ipe4(numatomic(iatom))+j+3)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp4(ipe4(numatomic(iatom))+j+4)
            coeff(locprim(ishell)+j)= coeffp4(ipe4(numatomic(iatom))+j+4)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
! D function
          ishell= ishell+1
          do j= 1,4
            ex(locprim(ishell)+j)= expd4(ide4(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffd4(ide4(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 4
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+4
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1) = locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1) = locbf(ishell)+6
          endif
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expd4(ide4(numatomic(iatom))+j+4)
            coeff(locprim(ishell)+j)= coeffd4(ide4(numatomic(iatom))+j+4)
          enddo
          mprim(ishell)= 1
          mtype(ishell)= 2
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          if(spher) then
            mbf(ishell)= 5
            locbf(ishell+1) = locbf(ishell)+5
          else
            mbf(ishell)= 6
            locbf(ishell+1) = locbf(ishell)+6
          endif
!
! Set Zn  - Kr functions
!
        case(30:36)
! S function
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= exps4(ise4(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs4(ise4(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 2
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps4(ise4(numatomic(iatom))+j+2)
            coeff(locprim(ishell)+j)= coeffs4(ise4(numatomic(iatom))+j+2)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
! P function
          nump= 2
          if(numatomic(iatom) == 30) nump= 1
          ishell= ishell+1
          do j= 1,nump
            ex(locprim(ishell)+j)= expp4(ipe4(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp4(ipe4(numatomic(iatom))+j)
          enddo
          mprim(ishell)= nump
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+nump
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp4(ipe4(numatomic(iatom))+j+nump)
            coeff(locprim(ishell)+j)= coeffp4(ipe4(numatomic(iatom))+j+nump)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
! D function
          if(numatomic(iatom) == 30) then
            ishell= ishell+1
            do j= 1,4
              ex(locprim(ishell)+j)= expd4(ide4(numatomic(iatom))+j)
              coeff(locprim(ishell)+j)= coeffd4(ide4(numatomic(iatom))+j)
            enddo
            mprim(ishell)= 4
            mtype(ishell)= 2
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+4
            if(spher) then
              mbf(ishell)= 5
              locbf(ishell+1) = locbf(ishell)+5
            else
              mbf(ishell)= 6
              locbf(ishell+1) = locbf(ishell)+6
            endif
!
            ishell= ishell+1
            do j= 1,1
              ex(locprim(ishell)+j)= expd4(ide4(numatomic(iatom))+j+4)
              coeff(locprim(ishell)+j)= coeffd4(ide4(numatomic(iatom))+j+4)
            enddo
            mprim(ishell)= 1
            mtype(ishell)= 2
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+1
            if(spher) then
              mbf(ishell)= 5
              locbf(ishell+1) = locbf(ishell)+5
            else
              mbf(ishell)= 6
              locbf(ishell+1) = locbf(ishell)+6
            endif
          endif
!
! Set Rb - Ag functions
!
        case(37:47)
! S function
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exps5(ise5(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs5(ise5(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,4
            ex(locprim(ishell)+j)= exps5(ise5(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffs5(ise5(numatomic(iatom))+j+3)
          enddo
          mprim(ishell)= 4
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+4
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps5(ise5(numatomic(iatom))+j+7)
            coeff(locprim(ishell)+j)= coeffs5(ise5(numatomic(iatom))+j+7)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
! P function
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= expp5(ipe5(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp5(ipe5(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= expp5(ipe5(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffp5(ipe5(numatomic(iatom))+j+3)
          enddo
          mprim(ishell)= 2
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp5(ipe5(numatomic(iatom))+j+5)
            coeff(locprim(ishell)+j)= coeffp5(ipe5(numatomic(iatom))+j+5)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
! D function
          if(numatomic(iatom) >= 39) then
            ishell= ishell+1
            do j= 1,3
              ex(locprim(ishell)+j)= expd5(ide5(numatomic(iatom))+j)
              coeff(locprim(ishell)+j)= coeffd5(ide5(numatomic(iatom))+j)
            enddo
            mprim(ishell)= 3
            mtype(ishell)= 2
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+3
            if(spher) then
              mbf(ishell)= 5
              locbf(ishell+1) = locbf(ishell)+5
            else
              mbf(ishell)= 6
              locbf(ishell+1) = locbf(ishell)+6
            endif
!
            ishell= ishell+1
            do j= 1,1
              ex(locprim(ishell)+j)= expd5(ide5(numatomic(iatom))+j+3)
              coeff(locprim(ishell)+j)= coeffd5(ide5(numatomic(iatom))+j+3)
            enddo
            mprim(ishell)= 1
            mtype(ishell)= 2
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+1
            if(spher) then
              mbf(ishell)= 5
              locbf(ishell+1) = locbf(ishell)+5
            else
              mbf(ishell)= 6
              locbf(ishell+1) = locbf(ishell)+6
            endif
          endif
!
! Set Cd - Xe functions
!
        case(48:54)
! S function
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= exps5(ise5(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs5(ise5(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 2
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps5(ise5(numatomic(iatom))+j+2)
            coeff(locprim(ishell)+j)= coeffs5(ise5(numatomic(iatom))+j+2)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
! P function
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= expp5(ipe5(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp5(ipe5(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 2
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp5(ipe5(numatomic(iatom))+j+2)
            coeff(locprim(ishell)+j)= coeffp5(ipe5(numatomic(iatom))+j+2)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
! D function
          if(numatomic(iatom) == 48) then
            ishell= ishell+1
            do j= 1,3
              ex(locprim(ishell)+j)= expd5(ide5(numatomic(iatom))+j)
              coeff(locprim(ishell)+j)= coeffd5(ide5(numatomic(iatom))+j)
            enddo
            mprim(ishell)= 3
            mtype(ishell)= 2
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+3
            if(spher) then
              mbf(ishell)= 5
              locbf(ishell+1) = locbf(ishell)+5
            else
              mbf(ishell)= 6
              locbf(ishell+1) = locbf(ishell)+6
            endif
!
            ishell= ishell+1
            do j= 1,1
              ex(locprim(ishell)+j)= expd5(ide5(numatomic(iatom))+j+3)
              coeff(locprim(ishell)+j)= coeffd5(ide5(numatomic(iatom))+j+3)
            enddo
            mprim(ishell)= 1
            mtype(ishell)= 2
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+1
            if(spher) then
              mbf(ishell)= 5
              locbf(ishell+1) = locbf(ishell)+5
            else
              mbf(ishell)= 6
              locbf(ishell+1) = locbf(ishell)+6
            endif
          endif
!
! Set Cs - La, Hf - Au  functions
!
        case(55:57,72:79)
! S function
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= exps6(ise6(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs6(ise6(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,4
            ex(locprim(ishell)+j)= exps6(ise6(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffs6(ise6(numatomic(iatom))+j+3)
          enddo
          mprim(ishell)= 4
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+4
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps6(ise6(numatomic(iatom))+j+7)
            coeff(locprim(ishell)+j)= coeffs6(ise6(numatomic(iatom))+j+7)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
! P function
          ishell= ishell+1
          do j= 1,3
            ex(locprim(ishell)+j)= expp6(ipe6(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp6(ipe6(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 3
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+3
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= expp6(ipe6(numatomic(iatom))+j+3)
            coeff(locprim(ishell)+j)= coeffp6(ipe6(numatomic(iatom))+j+3)
          enddo
          mprim(ishell)= 2
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp6(ipe6(numatomic(iatom))+j+5)
            coeff(locprim(ishell)+j)= coeffp6(ipe6(numatomic(iatom))+j+5)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
! D function
          if(numatomic(iatom) >= 57) then
            ishell= ishell+1
            do j= 1,2
              ex(locprim(ishell)+j)= expd6(ide6(numatomic(iatom))+j)
              coeff(locprim(ishell)+j)= coeffd6(ide6(numatomic(iatom))+j)
            enddo
            mprim(ishell)= 2
            mtype(ishell)= 2
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+2
            if(spher) then
              mbf(ishell)= 5
              locbf(ishell+1) = locbf(ishell)+5
            else
              mbf(ishell)= 6
              locbf(ishell+1) = locbf(ishell)+6
            endif
!
            ishell= ishell+1
            do j= 1,1
              ex(locprim(ishell)+j)= expd6(ide6(numatomic(iatom))+j+2)
              coeff(locprim(ishell)+j)= coeffd6(ide6(numatomic(iatom))+j+2)
            enddo
            mprim(ishell)= 1
            mtype(ishell)= 2
            locatom(ishell)= iatom
            locprim(ishell+1)= locprim(ishell)+1
            if(spher) then
              mbf(ishell)= 5
              locbf(ishell+1) = locbf(ishell)+5
            else
              mbf(ishell)= 6
              locbf(ishell+1) = locbf(ishell)+6
            endif
          endif
!
! Set Hg - Bi functions
!
        case(80:83)
! S function
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= exps6(ise6(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffs6(ise6(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 2
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          locbf(ishell+1) = locbf(ishell)+1
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= exps6(ise6(numatomic(iatom))+j+2)
            coeff(locprim(ishell)+j)= coeffs6(ise6(numatomic(iatom))+j+2)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 1
          mtype(ishell)= 0
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+1
! P function
          ishell= ishell+1
          do j= 1,2
            ex(locprim(ishell)+j)= expp6(ipe6(numatomic(iatom))+j)
            coeff(locprim(ishell)+j)= coeffp6(ipe6(numatomic(iatom))+j)
          enddo
          mprim(ishell)= 2
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+2
          locbf(ishell+1) = locbf(ishell)+3
!
          ishell= ishell+1
          do j= 1,1
            ex(locprim(ishell)+j)= expp6(ipe6(numatomic(iatom))+j+2)
            coeff(locprim(ishell)+j)= coeffp6(ipe6(numatomic(iatom))+j+2)
          enddo
          mprim(ishell)= 1
          mbf(ishell)= 3
          mtype(ishell)= 1
          locatom(ishell)= iatom
          locprim(ishell+1)= locprim(ishell)+1
          locbf(ishell+1) = locbf(ishell)+3
        case default
          write(*,'(" Error! This program supports Na - La, Hf - Bi Lanl2dz basis set.")')
          call iabort
      endselect
!
      if(ishell > mxshell) then
        write(*,'(" Error! The number of basis shells exceeds mxshell",i6,".")')mxshell
        call iabort
      endif
      if(locprim(ishell+1) > mxprim ) then
        write(*,'(" Error! The number of primitive basis functions exceeds mxprim", &
&             i6,".")')mxprim
        call iabort
      endif
      if(locbf(ishell+1) > mxao ) then
        write(*,'(" Error! The number of basis functions exceeds mxao",i6,".")')mxao
        call iabort
      endif
      return
end


!----------------------------------------------
  subroutine bshuzmini6_g(iatom,ishell,itype)
!----------------------------------------------
!
! Set 6th row basis functions of minimal Huzinaga set for guess calculation
! itype = 1 : valence functions
!       = 2 : core functions
!
      use modmolecule, only : numatomic
      use modguess, only : ex_g, coeff_g, locprim_g, locbf_g, locatom_g, mprim_g, mbf_g, &
&                          mtype_g, spher_g
      use modparam, only : mxao, mxshell, mxprim
      use modecp, only : flagecp, izcore
!
      implicit none
      integer,intent(in) :: iatom, itype
      integer,intent(inout) :: ishell
      integer :: j
      real(8) :: exps(3,6,55:86), expp(3,2:6,55:86), expd(3,2:5,55:86)
      real(8) :: expf1(4,57:70), expf2(3,71:86)
      real(8) :: coeffs(3,6,55:86), coeffp(3,2:6,55:86), coeffd(3,2:5,55:86)
      real(8) :: coefff1(4,57:70), coefff2(3,71:86)
      data exps/ &
&       1.5525866D+04,2.3490055D+03,5.1623555D+02,6.8750505D+02,6.7191157D+01,2.7992003D+01,&
&       6.0168188D+01,1.0868694D+01,4.7407047D+00,9.8871626D+00,2.3693425D+00,1.1416761D+00,&
&       1.7690784D+00,3.8362031D-01,1.7130634D-01,2.1184878D-01,4.0324210D-02,1.7058140D-02,&
&       1.6098886D+04,2.4362960D+03,5.3562078D+02,7.1309716D+02,6.9806392D+01,2.9031078D+01,&
&       6.2480417D+01,1.1412912D+01,4.9805876D+00,1.0265539D+01,2.5292321D+00,1.2273505D+00,&
&       1.9672783D+00,4.1058204D-01,1.8712746D-01,2.1270737D-01,4.1803880D-02,1.7684080D-02,&
&       1.6625440D+04,2.5192937D+03,5.5467690D+02,7.3806161D+02,7.2542668D+01,3.0332746D+01,&
&       6.5022157D+01,1.1880510D+01,5.0799848D+00,1.0833873D+01,2.6683331D+00,1.2794770D+00,&
&       2.0541754D+00,4.2145344D-01,1.8969742D-01,2.1333194D-01,4.3310200D-02,1.8321290D-02,&
&       1.7261118D+04,2.6142239D+03,5.7525356D+02,7.6572152D+02,7.5195756D+01,3.1381854D+01,&
&       6.7475838D+01,1.2438016D+01,5.3575335D+00,1.1413568D+01,2.7973046D+00,1.3446362D+00,&
&       2.2097198D+00,4.3697325D-01,1.9857828D-01,2.2452750D-01,4.4843190D-02,1.8969780D-02,&
&       1.7886843D+04,2.7083926D+03,5.9592268D+02,7.9303026D+02,7.7889637D+01,3.2526404D+01,&
&       7.0100725D+01,1.2959578D+01,5.5771203D+00,1.1952788D+01,2.9175114D+00,1.3900121D+00,&
&       2.3097738D+00,4.5748947D-01,2.0271971D-01,2.3370798D-01,4.6054820D-02,1.9776780D-02,&
&       1.8526642D+04,2.8048423D+03,6.1704358D+02,8.2124079D+02,8.0575481D+01,3.3535519D+01,&
&       7.2745461D+01,1.3509141D+01,5.8627242D+00,1.2536668D+01,3.0529657D+00,1.4459091D+00,&
&       2.3675661D+00,4.8032361D-01,2.1321496D-01,2.4169741D-01,4.8563610D-02,2.0452860D-02,&
&       1.9159059D+04,2.9008327D+03,6.3827198D+02,8.4958277D+02,8.3341451D+01,3.4625265D+01,&
&       7.5520257D+01,1.4029012D+01,6.0294403D+00,1.3179882D+01,3.1664005D+00,1.5049793D+00,&
&       2.5039828D+00,5.0626814D-01,2.2606303D-01,2.4982112D-01,4.9722400D-02,2.1140300D-02,&
&       1.9842708D+04,3.0021144D+03,6.6028800D+02,8.7904586D+02,8.6285511D+01,3.6009643D+01,&
&       7.8329115D+01,1.4573156D+01,6.2287472D+00,1.3752721D+01,3.3563672D+00,1.5718294D+00,&
&       2.6439751D+00,5.2293987D-01,2.3610458D-01,2.5807912D-01,5.1366010D-02,2.1839110D-02,&
&       2.0507003D+04,3.1028009D+03,6.8235369D+02,9.0788537D+02,8.9668374D+01,3.8112692D+01,&
&       8.0647184D+01,1.5647728D+01,7.4065456D+00,1.4082371D+01,3.9851714D+00,2.1380518D+00,&
&       4.5489800D+00,7.0187288D-01,3.5287489D-01,8.3915121D-01,6.4915490D-02,2.4693370D-02,&
&       2.1161540D+04,3.2022178D+03,7.0445174D+02,9.3554786D+02,9.2934461D+01,3.9928439D+01,&
&       8.3470797D+01,1.6165723D+01,7.4522280D+00,1.4303089D+01,4.0502079D+00,2.1005199D+00,&
&       4.6945382D+00,7.2433139D-01,3.7987327D-01,8.6600235D-01,6.6992650D-02,2.5483510D-02,&
&       2.1835019D+04,3.3056159D+03,7.2731432D+02,9.6508624D+02,9.5681960D+01,4.0815799D+01,&
&       8.6613515D+01,1.6578152D+01,7.8003711D+00,1.5775660D+01,4.0878441D+00,2.1592946D+00,&
&       4.8423886D+00,7.4714359D-01,3.8423134D-01,8.9327635D-01,6.9102530D-02,2.6286090D-02,&
&       2.2512038D+04,3.4087516D+03,7.5021737D+02,9.9666912D+02,9.8727277D+01,4.2083165D+01,&
&       8.9147986D+01,1.7526082D+01,8.1030013D+00,1.5711507D+01,4.2363528D+00,2.2096177D+00,&
&       4.8826712D+00,7.7030946D-01,3.9223241D-01,9.2097321D-01,7.1245110D-02,2.7101120D-02,&
&       2.3180533D+04,3.5116702D+03,7.7332413D+02,1.0292619D+03,1.0170128D+02,4.3328224D+01,&
&       9.2657169D+01,1.7937269D+01,8.4777760D+00,1.6692436D+01,4.3358910D+00,2.2322713D+00,&
&       4.8851960D+00,7.9382901D-01,3.9985838D-01,9.4909291D-01,7.3420410D-02,2.7928580D-02,&
&       2.3932986D+04,3.6231807D+03,7.9744649D+02,1.0580700D+03,1.0560404D+02,4.5715154D+01,&
&       9.5445255D+01,1.8624316D+01,8.6540627D+00,1.7427739D+01,4.4976911D+00,2.3319556D+00,&
&       4.8804329D+00,8.1161512D-01,4.1312407D-01,9.9230000D-01,7.5628420D-02,2.8768490D-02,&
&       2.4656410D+04,3.7325062D+03,8.2163815D+02,1.0914725D+03,1.0854804D+02,4.6629354D+01,&
&       9.8840728D+01,1.9274990D+01,8.9158303D+00,1.7990337D+01,4.7415251D+00,2.4155100D+00,&
&       4.9925787D+00,8.3566168D-01,4.1594385D-01,1.0216999D+00,7.7869140D-02,2.9620850D-02,&
&       2.5385694D+04,3.8436032D+03,8.4615684D+02,1.1234055D+03,1.1169297D+02,4.7233256D+01,&
&       1.0178796D+02,1.9813955D+01,9.1947339D+00,1.8855724D+01,4.8006027D+00,2.4675257D+00,&
&       5.1383398D+00,8.6005928D-01,4.3522801D-01,1.0515290D+00,8.0142580D-02,3.0485640D-02,&
&       2.6213072D+04,3.9645756D+03,8.7202321D+02,1.1596313D+03,1.1482931D+02,4.8944774D+01,&
&       1.0517731D+02,2.0488594D+01,9.4407118D+00,2.0292410D+01,4.6894803D+00,2.5387204D+00,&
&       5.0301304D+00,8.8480792D-01,4.4784708D-01,1.0345315D+00,8.2448720D-02,3.0597940D-02,&
&       2.6969327D+04,4.0796473D+03,8.9744182D+02,1.1911456D+03,1.1787390D+02,4.9501542D+01,&
&       1.0779325D+02,2.1253125D+01,9.6602190D+00,2.0900824D+01,4.8004147D+00,2.5851469D+00,&
&       4.8165887D+00,9.1909860D-01,4.6756457D-01,1.0028075D+00,8.4787570D-02,3.1465920D-02,&
&       2.7707063D+04,4.1922778D+03,9.2272356D+02,1.2259011D+03,1.2142987D+02,5.0889951D+01,&
&       1.1155105D+02,2.1746520D+01,1.0030729D+01,2.1501135D+01,4.9890210D+00,2.6049632D+00,&
&       4.7611306D+00,9.7852208D-01,4.8069074D-01,9.7168133D-01,8.7159140D-02,3.2346040D-02,&
&       2.8483470D+04,4.3106522D+03,9.4880189D+02,1.2601431D+03,1.2499189D+02,5.3264926D+01,&
&       1.1485307D+02,2.2490909D+01,1.0205372D+01,2.1986566D+01,5.1545586D+00,2.6238159D+00,&
&       4.5567679D+00,9.9070497D-01,4.9395055D-01,9.4116795D-01,8.9563420D-02,3.3238310D-02,&
&       2.9263742D+04,4.4298015D+03,9.7511876D+02,1.2932196D+03,1.2860003D+02,5.4807887D+01,&
&       1.1845628D+02,2.3141641D+01,1.0379264D+01,2.2971336D+01,5.3257089D+00,2.7498632D+00,&
&       4.7595836D+00,1.0178146D+00,5.0739077D-01,9.1127985D-01,9.2000410D-02,3.4142710D-02,&
&       3.0105978D+04,4.5546731D+03,1.0023444D+03,1.3299061D+03,1.3189507D+02,5.5352028D+01,&
&       1.2184431D+02,2.3821993D+01,1.0499604D+01,2.3220256D+01,5.6648299D+00,2.8808691D+00,&
&       4.6824502D+00,1.0770750D+00,5.2101139D-01,8.6522667D-01,9.5424350D-02,3.5059250D-02,&
&       3.0907827D+04,4.6769981D+03,1.0294994D+03,1.3673566D+03,1.3548412D+02,5.7389281D+01,&
&       1.2557265D+02,2.4514298D+01,1.0831887D+01,2.4082749D+01,5.7865340D+00,2.9424679D+00,&
&       4.9603751D+00,1.1002150D+00,5.0934516D-01,8.6227728D-01,9.7952040D-02,3.5987930D-02,&
&       3.1715151D+04,4.8004470D+03,1.0568133D+03,1.4024880D+03,1.3919570D+02,5.8595143D+01,&
&       1.2903195D+02,2.5401783D+01,1.1170637D+01,2.4521199D+01,6.1520604D+00,3.0498908D+00,&
&       5.0181161D+00,1.1523707D+00,5.0991299D-01,8.5904808D-01,9.9517590D-02,3.6928750D-02,&
&       3.2538784D+04,4.9255753D+03,1.0846021D+03,1.4392918D+03,1.4287377D+02,6.0413806D+01,&
&       1.3290783D+02,2.5950372D+01,1.1561528D+01,2.5227768D+01,6.3776449D+00,3.1285944D+00,&
&       4.9230485D+00,1.1881975D+00,5.2307149D-01,8.8981019D-01,1.0617371D-01,3.7881710D-02,&
&       3.3404144D+04,5.0549760D+03,1.1129314D+03,1.4756246D+03,1.4640265D+02,6.1389131D+01,&
&       1.3621257D+02,2.6696685D+01,1.1554849D+01,2.5982997D+01,6.5233802D+00,3.2362472D+00,&
&       5.3106531D+00,1.2593063D+00,5.4614633D-01,8.9197217D-01,1.0752428D-01,3.9379900D-02,&
&       3.4131365D+04,5.1723898D+03,1.1401778D+03,1.5117882D+03,1.5097546D+02,6.4320792D+01,&
&       1.4010346D+02,2.7595196D+01,1.1845526D+01,2.6902991D+01,6.6871496D+00,3.3176591D+00,&
&       5.6090779D+00,1.2974407D+00,5.5988532D-01,9.2812701D-01,1.1022919D-01,4.0572400D-02,&
&       3.5075240D+04,5.3112719D+03,1.1700875D+03,1.5518174D+03,1.5431441D+02,6.5481080D+01,&
&       1.4395513D+02,2.8198551D+01,1.2139813D+01,2.7546889D+01,6.8739521D+00,3.4374831D+00,&
&       6.0560593D+00,1.2999385D+00,5.8010673D-01,1.0041909D+00,1.5157282D-01,4.1580370D-02,&
&       3.5963660D+04,5.4454286D+03,1.1996058D+03,1.5912502D+03,1.5816746D+02,6.6985958D+01,&
&       1.4755203D+02,2.9134674D+01,1.2498034D+01,2.8216255D+01,7.1495803D+00,3.5218354D+00,&
&       6.2636009D+00,1.4271684D+00,6.1662978D-01,1.0835409D+00,1.6361768D-01,4.2600710D-02,&
&       3.6867892D+04,5.5815605D+03,1.2295467D+03,1.6313133D+03,1.6182070D+02,6.7853196D+01,&
&       1.5133201D+02,2.9857813D+01,1.2666596D+01,2.9409549D+01,7.2021546D+00,3.5479184D+00,&
&       6.4693577D+00,1.4613166D+00,6.4491060D-01,1.2428201D+00,1.8277504D-01,4.3633420D-02,&
&       3.7776571D+04,5.7185732D+03,1.2597282D+03,1.6696097D+03,1.6583960D+02,6.9717694D+01,&
&       1.5565678D+02,3.0520449D+01,1.2699037D+01,3.0256338D+01,7.4299444D+00,3.6548245D+00,&
&       6.7990670D+00,1.5270767D+00,6.6035701D-01,1.3540229D+00,2.0700628D-01,4.4678490D-02,&
&       3.8634492D+04,5.8523360D+03,1.2897187D+03,1.7105080D+03,1.6955891D+02,7.0548371D+01,&
&       1.5892829D+02,3.1748256D+01,1.4197685D+01,3.0871286D+01,7.9177164D+00,3.9643042D+00,&
&       6.6237042D+00,1.7190412D+00,7.9915287D-01,1.3421310D+00,2.6767369D-01,1.0544732D-01/
      data coeffs/ &
&       6.072420D-02, 3.658416D-01, 6.920470D-01,-1.102127D-01, 7.272034D-01, 3.401429D-01,&
&      -2.753151D-01, 8.901281D-01, 2.648633D-01,-3.688268D-01, 8.325824D-01, 3.887891D-01,&
&      -3.409418D-01, 7.000971D-01, 4.934524D-01,-2.750217D-01, 1.936040D-01, 9.141935D-01,&
&       6.071990D-02, 3.657475D-01, 6.920963D-01,-1.103332D-01, 7.275923D-01, 3.399361D-01,&
&      -2.773775D-01, 8.867966D-01, 2.701325D-01,-3.778720D-01, 8.405296D-01, 3.887798D-01,&
&      -3.632895D-01, 8.040338D-01, 3.967839D-01,-2.013774D-01, 7.154536D-01, 4.176434D-01,&
&       6.093590D-02, 3.661543D-01, 6.914798D-01,-1.105265D-01, 7.265141D-01, 3.408799D-01,&
&      -2.778738D-01, 8.951231D-01, 2.629053D-01,-3.775915D-01, 8.430458D-01, 3.866470D-01,&
&      -3.560968D-01, 8.367243D-01, 3.610625D-01,-1.919227D-01, 7.008800D-01, 4.300542D-01,&
&       6.076380D-02, 3.656823D-01, 6.920432D-01,-1.106185D-01, 7.278710D-01, 3.396050D-01,&
&      -2.798715D-01, 8.931133D-01, 2.662826D-01,-3.759205D-01, 8.430916D-01, 3.850760D-01,&
&      -3.554107D-01, 8.258265D-01, 3.659862D-01,-2.004514D-01, 6.844032D-01, 4.488729D-01,&
&       6.068100D-02, 3.654327D-01, 6.923172D-01,-1.107190D-01, 7.292466D-01, 3.381817D-01,&
&      -2.809137D-01, 8.959656D-01, 2.641582D-01,-3.748941D-01, 8.546478D-01, 3.733135D-01,&
&      -3.516056D-01, 8.231239D-01, 3.686808D-01,-2.047518D-01, 6.691118D-01, 4.637875D-01,&
&       6.058760D-02, 3.651271D-01, 6.926526D-01,-1.107753D-01, 7.317651D-01, 3.357808D-01,&
&      -2.823472D-01, 8.966472D-01, 2.641148D-01,-3.733290D-01, 8.556639D-01, 3.715675D-01,&
&      -3.452861D-01, 8.094339D-01, 3.827644D-01,-2.123201D-01, 6.410192D-01, 4.966707D-01,&
&       6.056190D-02, 3.650056D-01, 6.927569D-01,-1.108554D-01, 7.336528D-01, 3.339623D-01,&
&      -2.828345D-01, 9.021283D-01, 2.592573D-01,-3.692896D-01, 8.617535D-01, 3.618931D-01,&
&      -3.467544D-01, 7.774164D-01, 4.133012D-01,-2.139898D-01, 6.289105D-01, 5.077477D-01,&
&       6.041290D-02, 3.646398D-01, 6.932013D-01,-1.109973D-01, 7.328357D-01, 3.345842D-01,&
&      -2.835070D-01, 9.054729D-01, 2.565598D-01,-3.703420D-01, 8.468583D-01, 3.795108D-01,&
&      -3.547850D-01, 7.671895D-01, 4.223481D-01,-2.158990D-01, 6.094654D-01, 5.271336D-01,&
&       6.035260D-02, 3.644263D-01, 6.934313D-01,-1.113376D-01, 7.253497D-01, 3.414365D-01,&
&      -2.893875D-01, 8.501431D-01, 3.140286D-01,-3.914035D-01, 5.756313D-01, 6.601493D-01,&
&      -4.177300D-03, 2.346292D-01, 7.822717D-01,-8.198940D-02,-3.165455D-01,-6.935623D-01,&
&       6.036780D-02, 3.644456D-01, 6.933749D-01,-1.115875D-01, 7.215131D-01, 3.449564D-01,&
&      -2.897088D-01, 8.635201D-01, 3.018628D-01,-3.962516D-01, 6.657319D-01, 5.798678D-01,&
&      -1.153430D-02, 1.844031D-01, 8.315934D-01, 7.471410D-02, 2.936293D-01, 7.177832D-01,&
&       6.034390D-02, 3.642347D-01, 6.935561D-01,-1.115621D-01, 7.271652D-01, 3.395253D-01,&
&      -2.901632D-01, 8.787749D-01, 2.847502D-01,-3.703324D-01, 6.573689D-01, 5.630353D-01,&
&       3.621900D-03, 2.157088D-01, 7.961194D-01, 7.667290D-02, 2.782983D-01, 7.316191D-01,&
&       6.034970D-02, 3.642008D-01, 6.935559D-01,-1.116446D-01, 7.284610D-01, 3.382378D-01,&
&      -2.922379D-01, 8.565066D-01, 3.113017D-01,-3.883642D-01, 7.041276D-01, 5.332091D-01,&
&      -1.006390D-02, 2.164299D-01, 8.016135D-01, 7.844920D-02, 2.655523D-01, 7.430275D-01,&
&       6.040860D-02, 3.642593D-01, 6.934164D-01,-1.116859D-01, 7.311446D-01, 3.354456D-01,&
&      -2.915066D-01, 8.703011D-01, 2.947321D-01,-3.779939D-01, 7.188710D-01, 5.097373D-01,&
&      -6.747100D-03, 2.293095D-01, 7.884314D-01, 7.982730D-02, 2.501738D-01, 7.569843D-01,&
&       6.026830D-02, 3.639567D-01, 6.938113D-01,-1.120662D-01, 7.216807D-01, 3.446059D-01,&
&      -2.931741D-01, 8.716745D-01, 2.956139D-01,-3.754235D-01, 7.118474D-01, 5.141121D-01,&
&      -1.686790D-02, 2.226920D-01, 7.987453D-01, 7.679450D-02, 2.395752D-01, 7.680281D-01,&
&       6.024080D-02, 3.638374D-01, 6.939176D-01,-1.120302D-01, 7.267395D-01, 3.397656D-01,&
&      -2.924475D-01, 8.725192D-01, 2.946143D-01,-3.786380D-01, 7.019566D-01, 5.282080D-01,&
&      -1.271300D-02, 2.458730D-01, 7.758077D-01, 7.969950D-02, 2.235851D-01, 7.819576D-01,&
&       6.021780D-02, 3.637205D-01, 6.940245D-01,-1.120119D-01, 7.308335D-01, 3.365089D-01,&
&      -2.946531D-01, 8.851695D-01, 2.825866D-01,-3.703201D-01, 7.219045D-01, 5.008053D-01,&
&      -1.984220D-02, 2.196768D-01, 8.031232D-01, 7.470880D-02, 2.034147D-01, 8.021211D-01,&
&       5.999160D-02, 3.631627D-01, 6.947236D-01,-1.120702D-01, 7.324106D-01, 3.342924D-01,&
&      -2.944235D-01, 8.862065D-01, 2.818423D-01,-3.484181D-01, 7.636206D-01, 4.375170D-01,&
&      -4.281840D-02, 2.562323D-01, 7.786912D-01, 8.195200D-02, 3.276736D-01, 6.843155D-01,&
&       5.996350D-02, 3.630082D-01, 6.948626D-01,-1.119877D-01, 7.381808D-01, 3.292139D-01,&
&      -2.972360D-01, 8.867886D-01, 2.843174D-01,-3.510341D-01, 7.977843D-01, 4.052346D-01,&
&      -1.102500D-01, 3.161871D-01, 7.519785D-01, 9.646160D-02, 4.044906D-01, 6.034115D-01,&
&       6.001300D-02, 3.630896D-01, 6.947137D-01,-1.120743D-01, 7.373676D-01, 3.302371D-01,&
&      -2.968041D-01, 8.976579D-01, 2.714252D-01,-3.545572D-01, 8.174897D-01, 3.900778D-01,&
&      -1.740521D-01, 3.866417D-01, 7.160257D-01, 1.133863D-01, 4.626334D-01, 5.371199D-01,&
&       5.998470D-02, 3.629424D-01, 6.948544D-01,-1.122722D-01, 7.356066D-01, 3.311238D-01,&
&      -2.968151D-01, 8.959453D-01, 2.743324D-01,-3.600200D-01, 8.471586D-01, 3.660872D-01,&
&      -3.082678D-01, 4.976985D-01, 6.665086D-01, 1.050188D-01, 4.969138D-01, 5.046213D-01,&
&       5.997260D-02, 3.628568D-01, 6.949255D-01,-1.123781D-01, 7.355024D-01, 3.313340D-01,&
&      -2.962528D-01, 9.005200D-01, 2.697741D-01,-3.550665D-01, 8.440500D-01, 3.649955D-01,&
&      -3.102330D-01, 5.625827D-01, 6.054301D-01, 1.036582D-01, 5.125143D-01, 4.878821D-01,&
&       5.986490D-02, 3.625769D-01, 6.952575D-01,-1.122784D-01, 7.397582D-01, 3.277809D-01,&
&      -2.966599D-01, 9.060556D-01, 2.652479D-01,-3.683901D-01, 8.288352D-01, 3.928901D-01,&
&      -3.721096D-01, 6.395389D-01, 5.653353D-01,-8.187100D-03, 5.441173D-01, 5.030192D-01,&
&       5.985960D-02, 3.624691D-01, 6.953337D-01,-1.123837D-01, 7.396342D-01, 3.273403D-01,&
&      -2.964114D-01, 9.069686D-01, 2.639532D-01,-3.668321D-01, 8.518878D-01, 3.679458D-01,&
&      -3.599398D-01, 7.202836D-01, 4.825747D-01,-1.661545D-01, 5.908108D-01, 5.070257D-01,&
&       5.985970D-02, 3.624256D-01, 6.953573D-01,-1.124334D-01, 7.402836D-01, 3.271179D-01,&
&      -2.968866D-01, 8.998716D-01, 2.724105D-01,-3.765160D-01, 8.367551D-01, 3.937522D-01,&
&      -3.948925D-01, 7.877904D-01, 4.384118D-01,-2.480406D-01, 6.300501D-01, 4.869011D-01,&
&       5.985600D-02, 3.623664D-01, 6.953909D-01,-1.125149D-01, 7.407681D-01, 3.263717D-01,&
&      -2.972133D-01, 9.102609D-01, 2.606326D-01,-3.796176D-01, 8.433751D-01, 3.902769D-01,&
&      -3.845678D-01, 8.484104D-01, 3.848926D-01,-2.042568D-01, 6.180411D-01, 4.942361D-01,&
&       5.979390D-02, 3.622279D-01, 6.955610D-01,-1.124854D-01, 7.439363D-01, 3.235869D-01,&
&      -2.979513D-01, 9.145658D-01, 2.583104D-01,-3.806873D-01, 8.614964D-01, 3.719927D-01,&
&      -3.956055D-01, 8.415402D-01, 3.937369D-01,-1.998916D-01, 6.354779D-01, 4.747992D-01,&
&       6.000100D-02, 3.626063D-01, 6.949969D-01,-1.127989D-01, 7.369186D-01, 3.301527D-01,&
&      -2.971921D-01, 9.065842D-01, 2.669625D-01,-3.793081D-01, 8.769824D-01, 3.550289D-01,&
&      -3.948161D-01, 8.823134D-01, 3.507943D-01,-1.732925D-01, 8.697721D-01, 2.258045D-01,&
&       5.983510D-02, 3.621904D-01, 6.955115D-01,-1.127270D-01, 7.418872D-01, 3.251188D-01,&
&      -2.977604D-01, 9.161297D-01, 2.568361D-01,-3.839922D-01, 8.909138D-01, 3.440137D-01,&
&      -3.675232D-01, 9.261441D-01, 2.848743D-01,-2.345531D-01, 8.702589D-01, 2.826179D-01,&
&       5.978990D-02, 3.620306D-01, 6.956797D-01,-1.127545D-01, 7.427995D-01, 3.242900D-01,&
&      -2.984864D-01, 9.096834D-01, 2.646867D-01,-3.886382D-01, 8.948072D-01, 3.451839D-01,&
&      -3.813910D-01, 9.092481D-01, 3.171021D-01,-2.663030D-01, 9.547685D-01, 2.038896D-01,&
&       5.973960D-02, 3.618671D-01, 6.958550D-01,-1.126817D-01, 7.461081D-01, 3.213951D-01,&
&      -2.989084D-01, 9.149115D-01, 2.600228D-01,-3.820933D-01, 9.275938D-01, 3.053887D-01,&
&      -3.830339D-01, 9.531587D-01, 2.725793D-01,-2.907553D-01, 9.911980D-01, 1.728911D-01,&
&       5.970160D-02, 3.617435D-01, 6.959848D-01,-1.127567D-01, 7.462998D-01, 3.211019D-01,&
&      -2.973679D-01, 9.201903D-01, 2.542612D-01,-3.835086D-01, 9.334748D-01, 3.007817D-01,&
&      -3.767089D-01, 9.732817D-01, 2.500856D-01,-2.865344D-01, 1.010233D+00, 1.589027D-01,&
&       5.975630D-02, 3.617782D-01, 6.958837D-01,-1.126837D-01, 7.496104D-01, 3.182236D-01,&
&      -3.022314D-01, 9.042464D-01, 2.711234D-01,-3.929887D-01, 8.867217D-01, 3.562264D-01,&
&      -4.354600D-01, 9.268309D-01, 3.407903D-01,-3.405358D-01, 8.102372D-01, 3.844879D-01/
      data expp(1:3,2:5,55:80)/ &
&       1.0242833D+03,2.3964889D+02,7.1255548D+01,1.3945864D+02,2.5810810D+01,1.0139452D+01,&
&       4.3242445D+00,1.9364152D+00,8.1450530D-01,5.9458425D-01,2.7499182D-01,1.1155290D-01,&
&       1.0653684D+03,2.4949087D+02,7.4311477D+01,1.4706367D+02,2.6792790D+01,1.0604816D+01,&
&       4.4335440D+00,1.9758409D+00,8.1191629D-01,6.1380042D-01,2.7282586D-01,1.1006559D-01,&
&       1.1007340D+03,2.5810620D+02,7.7064873D+01,1.5729247D+02,2.7442045D+01,1.0932262D+01,&
&       4.5978720D+00,2.0136263D+00,7.7692443D-01,6.3938885D-01,2.7659869D-01,1.0974625D-01,&
&       1.1410638D+03,2.6781001D+02,8.0147542D+01,1.5936103D+02,2.9022085D+01,1.1568460D+01,&
&       4.8235492D+00,2.1015449D+00,7.8594489D-01,6.6808886D-01,2.8892840D-01,1.1363078D-01,&
&       1.1917574D+03,2.7946559D+02,8.3496655D+01,1.6474089D+02,3.0359081D+01,1.2121698D+01,&
&       5.1385128D+00,2.2538269D+00,8.7313773D-01,7.0148489D-01,3.0047369D-01,1.2131011D-01,&
&       1.2345803D+03,2.8969646D+02,8.6648623D+01,1.6858307D+02,3.1866276D+01,1.2721128D+01,&
&       5.5014686D+00,2.4419010D+00,1.0013475D+00,7.4285559D-01,3.2091091D-01,1.2917825D-01,&
&       1.2809289D+03,3.0067027D+02,8.9959488D+01,1.7515156D+02,3.3032232D+01,1.3226298D+01,&
&       5.7636903D+00,2.5431625D+00,1.0364114D+00,7.7440027D-01,3.3432315D-01,1.3759787D-01,&
&       1.3298270D+03,3.1219809D+02,9.3453620D+01,1.8254042D+02,3.4346164D+01,1.3784402D+01,&
&       6.1634636D+00,2.7292360D+00,1.0706707D+00,8.0855734D-01,3.4537441D-01,1.4370986D-01,&
&       1.3845829D+03,3.2459262D+02,9.7074434D+01,1.8869784D+02,3.5858186D+01,1.4439022D+01,&
&       7.0626748D+00,3.4820688D+00,1.6742596D+00,1.9741929D+00,7.7456119D-01,2.7333192D-01,&
&       1.4301271D+03,3.3566767D+02,1.0050588D+02,1.9218592D+02,3.7398973D+01,1.5032371D+01,&
&       7.2245576D+00,3.5923145D+00,1.7459549D+00,2.0373631D+00,7.7355580D-01,2.7298819D-01,&
&       1.4802159D+03,3.4751808D+02,1.0411187D+02,2.0260666D+02,3.8524834D+01,1.5570199D+01,&
&       7.7988359D+00,3.7575148D+00,1.7630710D+00,2.1015281D+00,7.8244623D-01,2.7860506D-01,&
&       1.5272025D+03,3.5881603D+02,1.0760264D+02,2.0687090D+02,4.0194535D+01,1.6221668D+01,&
&       7.8825557D+00,3.7978777D+00,1.8177367D+00,2.1190102D+00,8.0074647D-01,2.8941946D-01,&
&       1.5737315D+03,3.6996920D+02,1.1106992D+02,2.0941284D+02,4.1838408D+01,1.6853075D+01,&
&       8.1634159D+00,3.8567519D+00,1.8073015D+00,2.1201059D+00,8.2864216D-01,2.9950199D-01,&
&       1.6318520D+03,3.8338971D+02,1.1506026D+02,2.1745963D+02,4.3178334D+01,1.7445255D+01,&
&       8.4116514D+00,4.0197899D+00,1.9188907D+00,2.1201431D+00,8.4677323D-01,2.9517372D-01,&
&       1.6854211D+03,3.9599853D+02,1.1891754D+02,2.2782181D+02,4.4528319D+01,1.8046201D+01,&
&       8.9691876D+00,4.2464848D+00,1.9503073D+00,2.1349230D+00,8.6763438D-01,3.1114728D-01,&
&       1.7377854D+03,4.0850276D+02,1.2273231D+02,2.3449301D+02,4.6044797D+01,1.8679254D+01,&
&       9.1334438D+00,4.3362256D+00,2.0173973D+00,2.1972533D+00,8.7979164D-01,3.1464744D-01,&
&       1.7955607D+03,4.2201312D+02,1.2679229D+02,2.4023172D+02,4.7779063D+01,1.9398857D+01,&
&       9.5967927D+00,4.4367903D+00,2.0246813D+00,2.1297889D+00,8.9173211D-01,3.2974496D-01,&
&       1.8523764D+03,4.3542673D+02,1.3087367D+02,2.4845071D+02,4.9377736D+01,2.0077755D+01,&
&       9.8265412D+00,4.5507968D+00,2.1136634D+00,2.0644787D+00,9.3120381D-01,3.4261920D-01,&
&       1.9081438D+03,4.4862494D+02,1.3493024D+02,2.5483949D+02,5.1093731D+01,2.0793577D+01,&
&       1.0057908D+01,4.5799849D+00,2.1300786D+00,1.9903475D+00,8.8952373D-01,3.4184436D-01,&
&       1.9659337D+03,4.6228149D+02,1.3907004D+02,2.6033590D+02,5.2992087D+01,2.1564622D+01,&
&       1.0448376D+01,4.7961135D+00,2.2000558D+00,1.9472206D+00,8.6995199D-01,3.2961937D-01,&
&       2.0244843D+03,4.7614657D+02,1.4333076D+02,2.7029118D+02,5.4486952D+01,2.2248123D+01,&
&       1.0747778D+01,4.8975554D+00,2.2377669D+00,1.8856196D+00,8.8924388D-01,3.4366701D-01,&
&       2.0883111D+03,4.9108568D+02,1.4776879D+02,2.7266734D+02,5.6742347D+01,2.3097886D+01,&
&       1.0934214D+01,4.9651040D+00,2.1875843D+00,1.8723995D+00,8.9080145D-01,3.5473666D-01,&
&       2.1451999D+03,5.0478331D+02,1.5203985D+02,2.8416628D+02,5.8149227D+01,2.3772619D+01,&
&       1.1374593D+01,5.1149536D+00,2.2232980D+00,1.8664367D+00,9.1439782D-01,3.7323659D-01,&
&       2.2068799D+03,5.1935114D+02,1.5651110D+02,2.9310449D+02,5.9918114D+01,2.4556392D+01,&
&       1.1628939D+01,5.2412093D+00,2.2477054D+00,1.8605783D+00,8.9302366D-01,3.7742695D-01,&
&       2.2683556D+03,5.3384887D+02,1.6096667D+02,3.0198242D+02,6.1409206D+01,2.5215656D+01,&
&       1.1983824D+01,5.3684081D+00,2.2381172D+00,1.8782203D+00,9.0272768D-01,3.7578948D-01,&
&       2.3328646D+03,5.4914520D+02,1.6564020D+02,3.0619849D+02,6.3373202D+01,2.6012143D+01,&
&       1.1965861D+01,5.3228262D+00,2.1625624D+00,1.9253031D+00,8.7120263D-01,3.6365398D-01/
      data expp(1:3,2:6,81:86)/ &
&       2.3775987D+03,5.6119485D+02,1.6973120D+02,3.1631847D+02,6.4888544D+01,2.6733530D+01,&
&       1.2328211D+01,5.5114323D+00,2.2169643D+00,1.9932760D+00,8.9758441D-01,3.7280215D-01,&
&       2.2282221D-01,9.3133630D-02,3.7253450D-02,2.4494365D+03,5.7781442D+02,1.7467073D+02,&
&       3.2439605D+02,6.7056357D+01,2.7668571D+01,1.3013204D+01,5.8680654D+00,2.2720421D+00,&
&       2.1325320D+00,9.8157824D-01,3.9467207D-01,2.3589377D-01,9.7028080D-02,3.8178970D-02,&
&       2.5172545D+03,5.9388843D+02,1.7956047D+02,3.3093361D+02,6.9075410D+01,2.8522477D+01,&
&       1.3329817D+01,6.0302971D+00,2.4456258D+00,2.2105424D+00,1.0056903D+00,4.0840048D-01,&
&       2.4893283D-01,9.8865290D-02,3.9115840D-02,2.5859828D+03,6.1013029D+02,1.8448204D+02,&
&       3.4055674D+02,7.1276662D+01,2.9491315D+01,1.4029785D+01,6.4245656D+00,2.6492696D+00,&
&       2.3710859D+00,1.1111472D+00,4.1830074D-01,2.7680626D-01,1.0126194D-01,4.0064070D-02,&
&       2.6553837D+03,6.2633825D+02,1.8939113D+02,3.4649889D+02,7.3495813D+01,3.0402772D+01,&
&       1.4334294D+01,6.5437078D+00,2.6726334D+00,2.4797553D+00,1.1577118D+00,4.2831956D-01,&
&       2.9831651D-01,1.0695344D-01,4.1023650D-02,2.7332176D+03,6.4418252D+02,1.9494224D+02,&
&       3.6084753D+02,7.7080714D+01,3.1547775D+01,5.5835907D+01,1.2265749D+01,5.6389792D+00,&
&       2.5027700D+00,1.1043323D+00,4.1119000D-01,3.8199600D-01,1.6039200D-01,6.6115700D-02/
      data coeffp(1:3,2:5,55:80)/ &
&       7.918520D-02, 4.199333D-01, 6.432514D-01,-3.312280D-02, 3.891083D-01, 6.790324D-01,&
&       3.816688D-01, 5.769763D-01, 1.154047D-01, 3.199069D-01, 5.531867D-01, 2.160885D-01,&
&       7.902710D-02, 4.191952D-01, 6.437580D-01,-3.333750D-02, 3.922373D-01, 6.752270D-01,&
&       4.153937D-01, 5.642885D-01, 9.237200D-02, 4.303242D-01, 5.355877D-01, 1.136744D-01,&
&       7.952990D-02, 4.204338D-01, 6.419900D-01,-3.335480D-02, 4.057066D-01, 6.616068D-01,&
&       4.393940D-01, 5.579740D-01, 7.528930D-02, 4.469577D-01, 5.338706D-01, 9.968110D-02,&
&       7.963170D-02, 4.202625D-01, 6.417642D-01,-3.328070D-02, 3.970831D-01, 6.699493D-01,&
&       4.438528D-01, 5.576090D-01, 7.190060D-02, 4.453960D-01, 5.335010D-01, 1.027921D-01,&
&       7.875160D-02, 4.186510D-01, 6.439389D-01,-3.387000D-02, 3.950470D-01, 6.719295D-01,&
&       4.321463D-01, 5.611356D-01, 8.013830D-02, 4.422667D-01, 5.336879D-01, 1.066328D-01,&
&       7.874540D-02, 4.186020D-01, 6.438205D-01,-3.400750D-02, 3.909916D-01, 6.759948D-01,&
&       4.135184D-01, 5.644878D-01, 9.519120D-02, 4.228918D-01, 5.407912D-01, 1.205496D-01,&
&       7.847820D-02, 4.180326D-01, 6.444278D-01,-3.424610D-02, 3.940065D-01, 6.729218D-01,&
&       4.174654D-01, 5.639300D-01, 9.211610D-02, 4.197609D-01, 5.371351D-01, 1.280269D-01,&
&       7.810220D-02, 4.170192D-01, 6.454850D-01,-3.453410D-02, 3.937909D-01, 6.729263D-01,&
&       3.975439D-01, 5.769811D-01, 1.012375D-01, 4.152969D-01, 5.410361D-01, 1.292299D-01,&
&       7.737600D-02, 4.157355D-01, 6.471840D-01,-3.505430D-02, 3.895085D-01, 6.767807D-01,&
&       3.001791D-01, 5.330359D-01, 2.383721D-01,-1.237310D-02, 5.332838D-01, 5.520593D-01,&
&       7.741310D-02, 4.155803D-01, 6.471221D-01,-3.504490D-02, 3.886697D-01, 6.779961D-01,&
&       3.154012D-01, 5.187296D-01, 2.372441D-01,-1.098600D-03, 5.498008D-01, 5.283374D-01,&
&       7.718470D-02, 4.150012D-01, 6.476882D-01,-3.536200D-02, 3.920956D-01, 6.740818D-01,&
&       2.983771D-01, 5.507163D-01, 2.242048D-01, 8.219600D-03, 5.531276D-01, 5.178321D-01,&
&       7.724440D-02, 4.151054D-01, 6.474102D-01,-3.545860D-02, 3.894833D-01, 6.769116D-01,&
&       3.234649D-01, 5.357630D-01, 2.136097D-01, 1.571900D-02, 5.448934D-01, 5.192350D-01,&
&       7.739850D-02, 4.155441D-01, 6.467612D-01,-3.541270D-02, 3.889577D-01, 6.778885D-01,&
&       3.329288D-01, 5.475342D-01, 1.934593D-01, 1.662510D-02, 5.383539D-01, 5.250268D-01,&
&       7.682150D-02, 4.142441D-01, 6.483076D-01,-3.566390D-02, 3.907248D-01, 6.759383D-01,&
&       3.387219D-01, 5.328755D-01, 2.018587D-01, 1.386750D-02, 5.582060D-01, 5.114888D-01,&
&       7.662550D-02, 4.137828D-01, 6.487444D-01,-3.590850D-02, 3.923215D-01, 6.740286D-01,&
&       3.202135D-01, 5.533706D-01, 2.016160D-01, 2.191590D-02, 5.421133D-01, 5.183720D-01,&
&       7.654450D-02, 4.135681D-01, 6.489045D-01,-3.606840D-02, 3.931550D-01, 6.732472D-01,&
&       3.354143D-01, 5.418041D-01, 1.976589D-01, 2.665140D-02, 5.465593D-01, 5.111040D-01,&
&       7.620340D-02, 4.127937D-01, 6.497893D-01,-3.632450D-02, 3.912159D-01, 6.751724D-01,&
&       3.360259D-01, 5.591485D-01, 1.807147D-01, 4.097320D-02, 5.605793D-01, 4.812806D-01,&
&       7.598140D-02, 4.122427D-01, 6.503501D-01,-3.653250D-02, 3.909150D-01, 6.753456D-01,&
&       3.475344D-01, 5.527717D-01, 1.742289D-01, 5.026440D-02, 5.912507D-01, 4.410394D-01,&
&       7.588460D-02, 4.120017D-01, 6.505248D-01,-3.665070D-02, 3.898915D-01, 6.763348D-01,&
&       3.645967D-01, 5.571217D-01, 1.518756D-01, 1.092912D-01, 6.171303D-01, 3.579635D-01,&
&       7.571230D-02, 4.116367D-01, 6.509095D-01,-3.687730D-02, 3.872042D-01, 6.790686D-01,&
&       3.599774D-01, 5.606850D-01, 1.526327D-01, 1.570695D-01, 6.436159D-01, 2.852549D-01,&
&       7.555970D-02, 4.112017D-01, 6.512966D-01,-3.701880D-02, 3.885717D-01, 6.773764D-01,&
&       3.693555D-01, 5.637240D-01, 1.393835D-01, 1.999751D-01, 6.225831D-01, 2.615926D-01,&
&       7.517190D-02, 4.103016D-01, 6.523726D-01,-3.728680D-02, 3.835103D-01, 6.828280D-01,&
&       3.834679D-01, 5.658388D-01, 1.230106D-01, 2.522785D-01, 6.021502D-01, 2.279055D-01,&
&       7.519290D-02, 4.102334D-01, 6.522788D-01,-3.738070D-02, 3.862631D-01, 6.796648D-01,&
&       3.847966D-01, 5.738330D-01, 1.132512D-01, 2.954146D-01, 5.703553D-01, 2.147306D-01,&
&       7.503160D-02, 4.097815D-01, 6.527002D-01,-3.754070D-02, 3.853595D-01, 6.802771D-01,&
&       3.943199D-01, 5.710945D-01, 1.057293D-01, 3.623975D-01, 5.386728D-01, 1.779594D-01,&
&       7.492740D-02, 4.095379D-01, 6.529009D-01,-3.752870D-02, 3.888529D-01, 6.767782D-01,&
&       4.010601D-01, 5.743307D-01, 9.546350D-02, 4.042684D-01, 5.182477D-01, 1.556140D-01,&
&       7.471910D-02, 4.088872D-01, 6.535490D-01,-3.754060D-02, 3.880999D-01, 6.777274D-01,&
&       4.339413D-01, 5.593747D-01, 7.591210D-02, 4.572396D-01, 5.064070D-01, 1.138995D-01/
      data coeffp(1:3,2:6,81:86)/ &
&       7.539260D-02, 4.099688D-01, 6.518046D-01,-3.740190D-02, 3.911708D-01, 6.743423D-01,&
&       4.360675D-01, 5.574551D-01, 7.543100D-02, 4.847093D-01, 4.987045D-01, 8.957610D-02,&
&       2.145554D-01, 5.125545D-01, 3.817007D-01, 7.493610D-02, 4.089706D-01, 6.530269D-01,&
&      -3.778040D-02, 3.872369D-01, 6.780358D-01, 4.135703D-01, 5.716137D-01, 8.582610D-02,&
&       4.708301D-01, 5.131364D-01, 8.629590D-02, 3.212519D-01, 5.604186D-01, 2.230238D-01,&
&       7.469570D-02, 4.083137D-01, 6.537243D-01,-3.790460D-02, 3.862204D-01, 6.790555D-01,&
&       4.230622D-01, 5.634345D-01, 8.206630D-02, 5.015939D-01, 4.977996D-01, 6.730200D-02,&
&       4.353007D-01, 5.414271D-01, 1.184326D-01, 7.446290D-02, 4.077334D-01, 6.543630D-01,&
&      -3.825610D-02, 3.823956D-01, 6.825233D-01, 4.029453D-01, 5.715225D-01, 9.427720D-02,&
&       4.781188D-01, 5.168473D-01, 7.113460D-02, 4.983952D-01, 5.216655D-01, 7.704690D-02,&
&       7.426460D-02, 4.073604D-01, 6.548009D-01,-3.843200D-02, 3.806310D-01, 6.843942D-01,&
&       4.138476D-01, 5.676362D-01, 8.656710D-02, 4.940629D-01, 5.099957D-01, 5.961950D-02,&
&       5.541128D-01, 4.849933D-01, 5.599090D-02, 7.386300D-02, 4.061085D-01, 6.560630D-01,&
&      -3.866890D-02, 3.674170D-01, 6.977983D-01,-1.144090D-01, 5.688023D-01, 5.191901D-01,&
&       5.662413D-01, 4.638213D-01, 3.027460D-02, 4.321029D-01, 5.050096D-01, 1.595361D-01/
      data expd(1:3,3:4,55:70)/ &
&       1.4405139D+02,4.0505146D+01,1.3086244D+01,7.3715777D+00,2.7248237D+00,1.0266781D+00,&
&       1.5169168D+02,4.2753274D+01,1.3850108D+01,7.7435537D+00,2.8709772D+00,1.0976557D+00,&
&       1.5717832D+02,4.4393780D+01,1.4460783D+01,7.9320490D+00,2.9378166D+00,1.1238460D+00,&
&       1.6413350D+02,4.6508119D+01,1.5198705D+01,8.4807186D+00,3.1599453D+00,1.2103814D+00,&
&       1.7177896D+02,4.8703353D+01,1.5963562D+01,8.9281848D+00,3.3319930D+00,1.2748513D+00,&
&       1.7965660D+02,5.0987916D+01,1.6747778D+01,9.4309935D+00,3.5273865D+00,1.3506401D+00,&
&       1.8722437D+02,5.3217800D+01,1.7525139D+01,9.8543016D+00,3.6872548D+00,1.4117181D+00,&
&       1.9529689D+02,5.5593987D+01,1.8348060D+01,1.0389082D+01,3.8703504D+00,1.4811196D+00,&
&       2.0592120D+02,5.8526401D+01,1.9299158D+01,1.1011484D+01,4.1362487D+00,1.5896167D+00,&
&       2.1333543D+02,6.0761796D+01,2.0097751D+01,1.1465387D+01,4.3293004D+00,1.6672499D+00,&
&       2.2188313D+02,6.3252113D+01,2.0957007D+01,1.2013693D+01,4.5266838D+00,1.7459841D+00,&
&       2.3167736D+02,6.6102671D+01,2.1916854D+01,1.2545087D+01,4.7409523D+00,1.8318939D+00,&
&       2.3721490D+02,6.7889543D+01,2.2617285D+01,1.3093652D+01,4.9213964D+00,1.8661708D+00,&
&       2.4776101D+02,7.0854934D+01,2.3602151D+01,1.3592062D+01,5.1225912D+00,1.9684932D+00,&
&       2.5688245D+02,7.3519651D+01,2.4520879D+01,1.4183386D+01,5.3348920D+00,2.0382571D+00,&
&       2.6549751D+02,7.6107209D+01,2.5437233D+01,1.4703007D+01,5.5153232D+00,2.0915122D+00/
      data expd(1:3,3:5,71:86)/ &
&       2.7718839D+02,7.9463212D+01,2.6610039D+01,1.5600726D+01,5.9803128D+00,2.3344255D+00,&
&       2.0469529D+00,6.8141564D-01,2.0136947D-01,2.8685428D+02,8.2345093D+01,2.7632269D+01,&
&       1.6251487D+01,6.2625999D+00,2.4873801D+00,2.0540301D+00,6.9039011D-01,2.1235875D-01,&
&       2.9767187D+02,8.5493895D+01,2.8722520D+01,1.6876653D+01,6.5142030D+00,2.6052822D+00,&
&       2.0602332D+00,7.0272676D-01,2.1834769D-01,3.0802405D+02,8.8529624D+01,2.9802451D+01,&
&       1.7714492D+01,6.8804872D+00,2.8091248D+00,2.0762160D+00,7.3677302D-01,2.2781597D-01,&
&       3.1855059D+02,9.1648758D+01,3.0906581D+01,1.8298578D+01,7.1624476D+00,2.9489065D+00,&
&       2.0604954D+00,7.4967705D-01,2.4248290D-01,3.3005620D+02,9.5019652D+01,3.2080580D+01,&
&       1.9043432D+01,7.4801362D+00,3.1008998D+00,2.0861668D+00,7.6997534D-01,2.5281533D-01,&
&       3.4065395D+02,9.8218233D+01,3.3213551D+01,1.9801434D+01,7.8062497D+00,3.2420460D+00,&
&       2.1418014D+00,8.0244678D-01,2.5966359D-01,3.5185991D+02,1.0148732D+02,3.4372939D+01,&
&       2.0563355D+01,8.1266279D+00,3.4000049D+00,2.2428150D+00,8.3045525D-01,2.6917025D-01,&
&       3.6341101D+02,1.0488558D+02,3.5572000D+01,2.1143756D+01,8.3920687D+00,3.5488617D+00,&
&       2.3358427D+00,8.6918220D-01,2.8462575D-01,3.7583702D+02,1.0848287D+02,3.6833664D+01,&
&       2.1948085D+01,8.7311154D+00,3.6993476D+00,2.3911943D+00,8.9025734D-01,2.9559019D-01,&
&       3.8360621D+02,1.1109160D+02,3.7869799D+01,2.2556467D+01,9.0361081D+00,3.8880752D+00,&
&       2.5003711D+00,9.4037699D-01,3.2314233D-01,3.9602788D+02,1.1478406D+02,3.9165515D+01,&
&       2.3480294D+01,9.4592198D+00,4.1021059D+00,2.6854228D+00,1.0321544D+00,3.6406946D-01,&
&       4.0774836D+02,1.1836555D+02,4.0481465D+01,2.4322803D+01,9.8293386D+00,4.2764203D+00,&
&       2.7618843D+00,1.0712666D+00,3.8408917D-01,4.2170380D+02,1.2238581D+02,4.1880233D+01,&
&       2.5183871D+01,1.0214589D+01,4.4950523D+00,3.0692218D+00,1.2195410D+00,4.4471254D-01,&
&       4.3427898D+02,1.2615550D+02,4.3240302D+01,2.6081763D+01,1.0651015D+01,4.7470935D+00,&
&       3.1664963D+00,1.2549894D+00,4.7297912D-01,4.5168521D+02,1.3110730D+02,4.4898191D+01,&
&       2.6989804D+01,1.0979775D+01,4.8990999D+00,3.6329003D+00,1.5131756D+00,5.8490621D-01/
      data coeffd(1:3,3:4,55:70)/ &
&       1.118302D-01, 4.768349D-01, 5.954174D-01, 2.537255D-01, 5.788860D-01, 3.325981D-01,&
&       1.105491D-01, 4.750537D-01, 5.972132D-01, 2.643998D-01, 5.858296D-01, 3.112276D-01,&
&       1.120244D-01, 4.782146D-01, 5.922607D-01, 2.829726D-01, 5.906859D-01, 2.872377D-01,&
&       1.118343D-01, 4.779431D-01, 5.918922D-01, 2.785615D-01, 5.871626D-01, 2.949838D-01,&
&       1.113158D-01, 4.777975D-01, 5.917250D-01, 2.809806D-01, 5.869343D-01, 2.928538D-01,&
&       1.106871D-01, 4.774459D-01, 5.919556D-01, 2.808573D-01, 5.852852D-01, 2.946460D-01,&
&       1.104956D-01, 4.776191D-01, 5.913436D-01, 2.853567D-01, 5.852009D-01, 2.902922D-01,&
&       1.099910D-01, 4.771221D-01, 5.916223D-01, 2.859941D-01, 5.863899D-01, 2.888684D-01,&
&       1.076601D-01, 4.744727D-01, 5.956181D-01, 2.799785D-01, 5.796519D-01, 3.014955D-01,&
&       1.080439D-01, 4.753318D-01, 5.939344D-01, 2.828563D-01, 5.769317D-01, 3.010715D-01,&
&       1.076858D-01, 4.752801D-01, 5.938155D-01, 2.839041D-01, 5.768815D-01, 3.001895D-01,&
&       1.065078D-01, 4.736060D-01, 5.958856D-01, 2.845373D-01, 5.744270D-01, 3.019803D-01,&
&       1.085255D-01, 4.770762D-01, 5.904551D-01, 2.858557D-01, 5.819554D-01, 2.947871D-01,&
&       1.071608D-01, 4.757213D-01, 5.925282D-01, 2.891028D-01, 5.765507D-01, 2.960610D-01,&
&       1.068666D-01, 4.757448D-01, 5.923866D-01, 2.893569D-01, 5.783934D-01, 2.946384D-01,&
&       1.069945D-01, 4.761187D-01, 5.915046D-01, 2.922065D-01, 5.811741D-01, 2.897880D-01/
      data coeffd(1:3,3:5,71:86)/ &
&       1.055301D-01, 4.738686D-01, 5.941071D-01, 2.805518D-01, 5.660570D-01, 3.127337D-01,&
&       8.708350D-02, 3.990061D-01, 6.900484D-01, 1.052524D-01, 4.734033D-01, 5.942747D-01,&
&       2.809288D-01, 5.630773D-01, 3.126199D-01, 1.125023D-01, 4.581519D-01, 6.142745D-01,&
&       1.044578D-01, 4.722716D-01, 5.955515D-01, 2.825487D-01, 5.651022D-01, 3.072128D-01,&
&       1.391422D-01, 5.065767D-01, 5.467383D-01, 1.040791D-01, 4.717159D-01, 5.958943D-01,&
&       2.790386D-01, 5.599718D-01, 3.128650D-01, 1.612120D-01, 5.329476D-01, 5.016985D-01,&
&       1.036622D-01, 4.710062D-01, 5.964425D-01, 2.810588D-01, 5.588154D-01, 3.099727D-01,&
&       1.961924D-01, 5.459967D-01, 4.501411D-01, 1.028721D-01, 4.696901D-01, 5.978924D-01,&
&       2.804308D-01, 5.593308D-01, 3.083838D-01, 2.275213D-01, 5.470991D-01, 4.189813D-01,&
&       1.025775D-01, 4.690760D-01, 5.982939D-01, 2.795121D-01, 5.612916D-01, 3.060363D-01,&
&       2.501547D-01, 5.534561D-01, 3.922531D-01, 1.021866D-01, 4.686069D-01, 5.986499D-01,&
&       2.796901D-01, 5.621668D-01, 3.033210D-01, 2.714830D-01, 5.621581D-01, 3.627509D-01,&
&       1.017057D-01, 4.678669D-01, 5.993448D-01, 2.840649D-01, 5.617204D-01, 2.974496D-01,&
&       2.897666D-01, 5.607057D-01, 3.443139D-01, 1.009775D-01, 4.667307D-01, 6.006099D-01,&
&       2.834370D-01, 5.638837D-01, 2.947653D-01, 3.162181D-01, 5.584083D-01, 3.179613D-01,&
&       1.021961D-01, 4.683110D-01, 5.976181D-01, 2.872732D-01, 5.606543D-01, 2.917691D-01,&
&       3.399404D-01, 5.676635D-01, 2.723492D-01, 1.015518D-01, 4.672091D-01, 5.988161D-01,&
&       2.843855D-01, 5.588180D-01, 2.948680D-01, 3.437064D-01, 5.720556D-01, 2.539847D-01,&
&       1.012945D-01, 4.662523D-01, 5.994080D-01, 2.836262D-01, 5.599295D-01, 2.933151D-01,&
&       3.726898D-01, 5.718582D-01, 2.160448D-01, 1.003116D-01, 4.647457D-01, 6.012760D-01,&
&       2.835530D-01, 5.580633D-01, 2.934985D-01, 3.500265D-01, 5.809209D-01, 2.231577D-01,&
&       9.991310D-02, 4.638397D-01, 6.020334D-01, 2.823875D-01, 5.537894D-01, 2.970252D-01,&
&       3.785868D-01, 5.763374D-01, 1.915263D-01, 9.790080D-02, 4.603345D-01, 6.066244D-01,&
&       2.829638D-01, 5.586436D-01, 2.909510D-01, 3.218022D-01, 5.816200D-01, 2.380017D-01/
      data expf1/ &
&       2.3231033D+01,6.8221675D+00,2.1843543D+00,6.2500149D-01,2.6139561D+01,7.7491838D+00,&
&       2.5050383D+00,7.3333938D-01,2.7475121D+01,8.1971451D+00,2.6760726D+00,7.9088718D-01,&
&       2.9610554D+01,8.8796987D+00,2.9115210D+00,8.6225689D-01,3.0839766D+01,9.3015739D+00,&
&       3.0627573D+00,9.0819810D-01,3.3212664D+01,1.0045060D+01,3.3169803D+00,9.7947735D-01,&
&       3.7125837D+01,1.1248423D+01,3.7191113D+00,1.0960099D+00,3.8819868D+01,1.1714379D+01,&
&       3.8626271D+00,1.1414087D+00,4.0680481D+01,1.2337528D+01,4.0635453D+00,1.1838629D+00,&
&       4.2262676D+01,1.2869755D+01,4.2445315D+00,1.2421218D+00,4.4012473D+01,1.3416270D+01,&
&       4.4270768D+00,1.2896098D+00,4.5343394D+01,1.3864352D+01,4.5763128D+00,1.3278260D+00,&
&       4.6858648D+01,1.4376483D+01,4.7433937D+00,1.3719529D+00,4.8314683D+01,1.4868502D+01,&
&       4.9219249D+00,1.4244136D+00/
      data expf2/ &
&       3.1797464D+01,9.1472500D+00,2.4942441D+00,3.4279107D+01,9.9276463D+00,2.7558598D+00,&
&       3.6692390D+01,1.0694725D+01,3.0231809D+00,3.9064396D+01,1.1456039D+01,3.2885779D+00,&
&       4.1648317D+01,1.2273287D+01,3.5704082D+00,4.4179317D+01,1.3087026D+01,3.8529209D+00,&
&       4.6700526D+01,1.3898596D+01,4.1392565D+00,4.9223375D+01,1.4713828D+01,4.4259372D+00,&
&       5.1866259D+01,1.5554932D+01,4.7228707D+00,5.4207840D+01,1.6330801D+01,5.0053841D+00,&
&       5.6851804D+01,1.7181822D+01,5.3079932D+00,5.9277049D+01,1.7991863D+01,5.6059964D+00,&
&       6.1976718D+01,1.8871879D+01,5.9233061D+00,6.4795841D+01,1.9791217D+01,6.2541857D+00,&
&       6.7670974D+01,2.0720618D+01,6.5872235D+00,7.1255348D+01,2.1844942D+01,6.9772755D+00/
      data coefff1/ &
&       8.480260D-02, 3.185478D-01, 5.215605D-01, 4.458382D-01, 8.097360D-02, 3.139003D-01,&
&       5.232838D-01, 4.426836D-01, 8.425260D-02, 3.215852D-01, 5.239577D-01, 4.293941D-01,&
&       8.360330D-02, 3.215007D-01, 5.245325D-01, 4.276097D-01, 8.661580D-02, 3.277359D-01,&
&       5.240850D-01, 4.196829D-01, 8.511070D-02, 3.257303D-01, 5.249144D-01, 4.214426D-01,&
&       7.859510D-02, 3.140544D-01, 5.258441D-01, 4.342082D-01, 8.022360D-02, 3.202909D-01,&
&       5.238402D-01, 4.302543D-01, 8.084400D-02, 3.214826D-01, 5.255148D-01, 4.297361D-01,&
&       8.244560D-02, 3.249725D-01, 5.236025D-01, 4.268708D-01, 8.379220D-02, 3.283186D-01,&
&       5.233255D-01, 4.242354D-01, 8.620390D-02, 3.331165D-01, 5.228664D-01, 4.197267D-01,&
&       8.802120D-02, 3.368047D-01, 5.227706D-01, 4.159762D-01, 9.015330D-02, 3.403088D-01,&
&       5.210745D-01, 4.128335D-01/
      data coefff2/ &
&       1.889543D-01, 5.299306D-01, 5.769141D-01, 1.855064D-01, 5.305358D-01, 5.738195D-01,&
&       1.829896D-01, 5.308347D-01, 5.704735D-01, 1.810851D-01, 5.314735D-01, 5.670319D-01,&
&       1.783847D-01, 5.314959D-01, 5.652308D-01, 1.763139D-01, 5.316512D-01, 5.632001D-01,&
&       1.746611D-01, 5.319165D-01, 5.609526D-01, 1.732586D-01, 5.323738D-01, 5.587198D-01,&
&       1.717270D-01, 5.328346D-01, 5.567749D-01, 1.715987D-01, 5.338727D-01, 5.533129D-01,&
&       1.704893D-01, 5.344428D-01, 5.513373D-01, 1.703522D-01, 5.352027D-01, 5.483777D-01,&
&       1.693912D-01, 5.355429D-01, 5.466911D-01, 1.681614D-01, 5.355027D-01, 5.456886D-01,&
&       1.670327D-01, 5.357835D-01, 5.445086D-01, 1.637813D-01, 5.341357D-01, 5.469714D-01/
!
      if((numatomic(iatom) > 86).or.(numatomic(iatom) < 55)) then
        write(*,'(" Error! This program supports only 6th row elements in bshuzmini6_g.")')
        call iabort
      endif
!
! Set valence basis functions
!
      if(itype == 1) then
!   6S function
        ishell= ishell+1
        do j= 1,3
          ex_g(locprim_g(ishell)+j)= exps(j,6,numatomic(iatom))
          coeff_g(locprim_g(ishell)+j)= coeffs(j,6,numatomic(iatom))
        enddo
        mprim_g(ishell)= 3
        mbf_g(ishell)= 1
        mtype_g(ishell)= 0
        locatom_g(ishell)= iatom
        locprim_g(ishell+1)= locprim_g(ishell)+3
        locbf_g(ishell+1) = locbf_g(ishell)+1
!
        select case(numatomic(iatom))
!
! Set La - Yb functions
!
          case(57:70)
!   4F function
            ishell= ishell+1
            do j= 1,4
              ex_g(locprim_g(ishell)+j)= expf1(j,numatomic(iatom))
              coeff_g(locprim_g(ishell)+j)= coefff1(j,numatomic(iatom))
            enddo
            mprim_g(ishell)= 4
            mtype_g(ishell)= 3
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+4
            if(spher_g) then
              mbf_g(ishell)= 7
              locbf_g(ishell+1) = locbf_g(ishell)+7
            else
              mbf_g(ishell)= 10
              locbf_g(ishell+1) = locbf_g(ishell)+10
            endif
!ishi
   if(numatomic(iatom)==58)then
        ishell= ishell+1
            do j= 1,1
              ex_g(locprim_g(ishell)+j)= 0.05D0
              coeff_g(locprim_g(ishell)+j)= 1.0D0
            enddo
            mprim_g(ishell)= 1
            mtype_g(ishell)= 1
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+1
              mbf_g(ishell)= 3
              locbf_g(ishell+1) = locbf_g(ishell)+3
        ishell= ishell+1
            do j= 1,1
              ex_g(locprim_g(ishell)+j)= 0.3D0
              coeff_g(locprim_g(ishell)+j)= 1.0D0
            enddo
            mprim_g(ishell)= 1
            mtype_g(ishell)= 2
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+1
              mbf_g(ishell)= 5
              locbf_g(ishell+1) = locbf_g(ishell)+5
    endif   
!
! Set Lu - Hg functions
!
          case(71:80)
!   5D function
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= expd(j,5,numatomic(iatom))
              coeff_g(locprim_g(ishell)+j)= coeffd(j,5,numatomic(iatom))
            enddo
            mprim_g(ishell)= 3
            mtype_g(ishell)= 2
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            if(spher_g) then
              mbf_g(ishell)= 5
              locbf_g(ishell+1) = locbf_g(ishell)+5
            else
              mbf_g(ishell)= 6
              locbf_g(ishell+1) = locbf_g(ishell)+6
            endif
!
! Set Tl - Rn functions
!
          case(81:86)
!   6P function
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= expp(j,6,numatomic(iatom))
              coeff_g(locprim_g(ishell)+j)= coeffp(j,6,numatomic(iatom))
            enddo
            mprim_g(ishell)= 3
            mbf_g(ishell)= 3
            mtype_g(ishell)= 1
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            locbf_g(ishell+1) = locbf_g(ishell)+3
        end select
!
! Set valence basis functions
!
      else
! Set 1S functions
        if(.not.flagecp.or.(izcore(iatom) < 2)) then
          ishell= ishell+1
          do j= 1,3
            ex_g(locprim_g(ishell)+j)= exps(j,1,numatomic(iatom))
            coeff_g(locprim_g(ishell)+j)= coeffs(j,1,numatomic(iatom))
          enddo
          mprim_g(ishell)= 3
          mbf_g(ishell)= 1
          mtype_g(ishell)= 0
          locatom_g(ishell)= iatom
          locprim_g(ishell+1)= locprim_g(ishell)+3
          locbf_g(ishell+1) = locbf_g(ishell)+1
        endif
! Set 2SP functions
        if(.not.flagecp.or.(izcore(iatom) < 10)) then
!   S function
          ishell= ishell+1
          do j= 1,3
            ex_g(locprim_g(ishell)+j)= exps(j,2,numatomic(iatom))
            coeff_g(locprim_g(ishell)+j)= coeffs(j,2,numatomic(iatom))
          enddo
          mprim_g(ishell)= 3
          mbf_g(ishell)= 1
          mtype_g(ishell)= 0
          locatom_g(ishell)= iatom
          locprim_g(ishell+1)= locprim_g(ishell)+3
          locbf_g(ishell+1) = locbf_g(ishell)+1
!   P function
          ishell= ishell+1
          do j= 1,3
            ex_g(locprim_g(ishell)+j)= expp(j,2,numatomic(iatom))
            coeff_g(locprim_g(ishell)+j)= coeffp(j,2,numatomic(iatom))
          enddo
          mprim_g(ishell)= 3
          mbf_g(ishell)= 3
          mtype_g(ishell)= 1
          locatom_g(ishell)= iatom
          locprim_g(ishell+1)= locprim_g(ishell)+3
          locbf_g(ishell+1) = locbf_g(ishell)+3
        endif
! Set 3SPD functions
        if(.not.flagecp.or.(izcore(iatom) < 18)) then
!   S function
          ishell= ishell+1
          do j= 1,3
            ex_g(locprim_g(ishell)+j)= exps(j,3,numatomic(iatom))
            coeff_g(locprim_g(ishell)+j)= coeffs(j,3,numatomic(iatom))
          enddo
          mprim_g(ishell)= 3
          mbf_g(ishell)= 1
          mtype_g(ishell)= 0
          locatom_g(ishell)= iatom
          locprim_g(ishell+1)= locprim_g(ishell)+3
          locbf_g(ishell+1) = locbf_g(ishell)+1
!   P function
          ishell= ishell+1
          do j= 1,3
            ex_g(locprim_g(ishell)+j)= expp(j,3,numatomic(iatom))
            coeff_g(locprim_g(ishell)+j)= coeffp(j,3,numatomic(iatom))
          enddo
          mprim_g(ishell)= 3
          mbf_g(ishell)= 3
          mtype_g(ishell)= 1
          locatom_g(ishell)= iatom
          locprim_g(ishell+1)= locprim_g(ishell)+3
          locbf_g(ishell+1) = locbf_g(ishell)+3
        endif
        if(.not.flagecp.or.(izcore(iatom) < 28)) then
!   D function
          ishell= ishell+1
          do j= 1,3
            ex_g(locprim_g(ishell)+j)= expd(j,3,numatomic(iatom))
            coeff_g(locprim_g(ishell)+j)= coeffd(j,3,numatomic(iatom))
          enddo
          mprim_g(ishell)= 3
          mtype_g(ishell)= 2
          locatom_g(ishell)= iatom
          locprim_g(ishell+1)= locprim_g(ishell)+3
          if(spher_g) then
            mbf_g(ishell)= 5
            locbf_g(ishell+1) = locbf_g(ishell)+5
          else
            mbf_g(ishell)= 6
            locbf_g(ishell+1) = locbf_g(ishell)+6
          endif
        endif
! Set 4SPDF functions
        if(.not.flagecp.or.(izcore(iatom) < 36)) then
!   S function
          ishell= ishell+1
          do j= 1,3
            ex_g(locprim_g(ishell)+j)= exps(j,4,numatomic(iatom))
            coeff_g(locprim_g(ishell)+j)= coeffs(j,4,numatomic(iatom))
          enddo
          mprim_g(ishell)= 3
          mbf_g(ishell)= 1
          mtype_g(ishell)= 0
          locatom_g(ishell)= iatom
          locprim_g(ishell+1)= locprim_g(ishell)+3
          locbf_g(ishell+1) = locbf_g(ishell)+1
!   P function
          ishell= ishell+1
          do j= 1,3
            ex_g(locprim_g(ishell)+j)= expp(j,4,numatomic(iatom))
            coeff_g(locprim_g(ishell)+j)= coeffp(j,4,numatomic(iatom))
          enddo
          mprim_g(ishell)= 3
          mbf_g(ishell)= 3
          mtype_g(ishell)= 1
          locatom_g(ishell)= iatom
          locprim_g(ishell+1)= locprim_g(ishell)+3
          locbf_g(ishell+1) = locbf_g(ishell)+3
        endif
!   D function
        if(.not.flagecp.or.(izcore(iatom) < 46)) then
          ishell= ishell+1
          do j= 1,3
            ex_g(locprim_g(ishell)+j)= expd(j,4,numatomic(iatom))
            coeff_g(locprim_g(ishell)+j)= coeffd(j,4,numatomic(iatom))
          enddo
          mprim_g(ishell)= 3
          mtype_g(ishell)= 2
          locatom_g(ishell)= iatom
          locprim_g(ishell+1)= locprim_g(ishell)+3
          if(spher_g) then
            mbf_g(ishell)= 5
            locbf_g(ishell+1) = locbf_g(ishell)+5
          else
            mbf_g(ishell)= 6
            locbf_g(ishell+1) = locbf_g(ishell)+6
          endif
        endif
!   F function
        if(numatomic(iatom) >= 71) then
          if(.not.flagecp.or.(izcore(iatom) < 60)) then
              ishell= ishell+1
              do j= 1,3
                ex_g(locprim_g(ishell)+j)= expf2(j,numatomic(iatom))
                coeff_g(locprim_g(ishell)+j)= coefff2(j,numatomic(iatom))
              enddo
              mprim_g(ishell)= 3
              mtype_g(ishell)= 3
              locatom_g(ishell)= iatom
              locprim_g(ishell+1)= locprim_g(ishell)+3
              if(spher_g) then
                mbf_g(ishell)= 7
                locbf_g(ishell+1) = locbf_g(ishell)+7
              else
                mbf_g(ishell)= 10
                locbf_g(ishell+1) = locbf_g(ishell)+10
              endif
            endif
          endif
! Set 5SPD functions
        if(.not.flagecp.or.((izcore(iatom) /= 54).and.(izcore(iatom) < 68))) then
!   S function
          ishell= ishell+1
          do j= 1,3
            ex_g(locprim_g(ishell)+j)= exps(j,5,numatomic(iatom))
            coeff_g(locprim_g(ishell)+j)= coeffs(j,5,numatomic(iatom))
          enddo
          mprim_g(ishell)= 3
          mbf_g(ishell)= 1
          mtype_g(ishell)= 0
          locatom_g(ishell)= iatom
          locprim_g(ishell+1)= locprim_g(ishell)+3
          locbf_g(ishell+1) = locbf_g(ishell)+1
!   P function
          ishell= ishell+1
          do j= 1,3
            ex_g(locprim_g(ishell)+j)= expp(j,5,numatomic(iatom))
            coeff_g(locprim_g(ishell)+j)= coeffp(j,5,numatomic(iatom))
          enddo
          mprim_g(ishell)= 3
          mbf_g(ishell)= 3
          mtype_g(ishell)= 1
          locatom_g(ishell)= iatom
          locprim_g(ishell+1)= locprim_g(ishell)+3
          locbf_g(ishell+1) = locbf_g(ishell)+3
        endif
!   D function
        if(numatomic(iatom) >= 81) then
          if(.not.flagecp.or.(izcore(iatom) < 78)) then
            ishell= ishell+1
            do j= 1,3
              ex_g(locprim_g(ishell)+j)= expd(j,5,numatomic(iatom))
              coeff_g(locprim_g(ishell)+j)= coeffd(j,5,numatomic(iatom))
            enddo
            mprim_g(ishell)= 3
            mtype_g(ishell)= 2
            locatom_g(ishell)= iatom
            locprim_g(ishell+1)= locprim_g(ishell)+3
            if(spher_g) then
              mbf_g(ishell)= 5
              locbf_g(ishell+1) = locbf_g(ishell)+5
            else
              mbf_g(ishell)= 6
              locbf_g(ishell+1) = locbf_g(ishell)+6
            endif
          endif
        endif
      endif
!
      return
end
